/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a BGP peer configuration for a route server endpoint.
 * </p>
 * <p>
 * A route server peer is a session between a route server endpoint and the device deployed in Amazon Web Services (such
 * as a firewall appliance or other network security function running on an EC2 instance). The device must meet these
 * requirements:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Have an elastic network interface in the VPC
 * </p>
 * </li>
 * <li>
 * <p>
 * Support BGP (Border Gateway Protocol)
 * </p>
 * </li>
 * <li>
 * <p>
 * Can initiate BGP sessions
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RouteServerPeer implements SdkPojo, Serializable, ToCopyableBuilder<RouteServerPeer.Builder, RouteServerPeer> {
    private static final SdkField<String> ROUTE_SERVER_PEER_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("RouteServerPeerId")
            .getter(getter(RouteServerPeer::routeServerPeerId))
            .setter(setter(Builder::routeServerPeerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RouteServerPeerId")
                    .unmarshallLocationName("routeServerPeerId").build()).build();

    private static final SdkField<String> ROUTE_SERVER_ENDPOINT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("RouteServerEndpointId")
            .getter(getter(RouteServerPeer::routeServerEndpointId))
            .setter(setter(Builder::routeServerEndpointId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RouteServerEndpointId")
                    .unmarshallLocationName("routeServerEndpointId").build()).build();

    private static final SdkField<String> ROUTE_SERVER_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("RouteServerId")
            .getter(getter(RouteServerPeer::routeServerId))
            .setter(setter(Builder::routeServerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RouteServerId")
                    .unmarshallLocationName("routeServerId").build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VpcId")
            .getter(getter(RouteServerPeer::vpcId))
            .setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId")
                    .unmarshallLocationName("vpcId").build()).build();

    private static final SdkField<String> SUBNET_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SubnetId")
            .getter(getter(RouteServerPeer::subnetId))
            .setter(setter(Builder::subnetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetId")
                    .unmarshallLocationName("subnetId").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("State")
            .getter(getter(RouteServerPeer::stateAsString))
            .setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State")
                    .unmarshallLocationName("state").build()).build();

    private static final SdkField<String> FAILURE_REASON_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("FailureReason")
            .getter(getter(RouteServerPeer::failureReason))
            .setter(setter(Builder::failureReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureReason")
                    .unmarshallLocationName("failureReason").build()).build();

    private static final SdkField<String> ENDPOINT_ENI_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("EndpointEniId")
            .getter(getter(RouteServerPeer::endpointEniId))
            .setter(setter(Builder::endpointEniId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointEniId")
                    .unmarshallLocationName("endpointEniId").build()).build();

    private static final SdkField<String> ENDPOINT_ENI_ADDRESS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("EndpointEniAddress")
            .getter(getter(RouteServerPeer::endpointEniAddress))
            .setter(setter(Builder::endpointEniAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointEniAddress")
                    .unmarshallLocationName("endpointEniAddress").build()).build();

    private static final SdkField<String> PEER_ADDRESS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("PeerAddress")
            .getter(getter(RouteServerPeer::peerAddress))
            .setter(setter(Builder::peerAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PeerAddress")
                    .unmarshallLocationName("peerAddress").build()).build();

    private static final SdkField<RouteServerBgpOptions> BGP_OPTIONS_FIELD = SdkField
            .<RouteServerBgpOptions> builder(MarshallingType.SDK_POJO)
            .memberName("BgpOptions")
            .getter(getter(RouteServerPeer::bgpOptions))
            .setter(setter(Builder::bgpOptions))
            .constructor(RouteServerBgpOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BgpOptions")
                    .unmarshallLocationName("bgpOptions").build()).build();

    private static final SdkField<RouteServerBgpStatus> BGP_STATUS_FIELD = SdkField
            .<RouteServerBgpStatus> builder(MarshallingType.SDK_POJO)
            .memberName("BgpStatus")
            .getter(getter(RouteServerPeer::bgpStatus))
            .setter(setter(Builder::bgpStatus))
            .constructor(RouteServerBgpStatus::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BgpStatus")
                    .unmarshallLocationName("bgpStatus").build()).build();

    private static final SdkField<RouteServerBfdStatus> BFD_STATUS_FIELD = SdkField
            .<RouteServerBfdStatus> builder(MarshallingType.SDK_POJO)
            .memberName("BfdStatus")
            .getter(getter(RouteServerPeer::bfdStatus))
            .setter(setter(Builder::bfdStatus))
            .constructor(RouteServerBfdStatus::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BfdStatus")
                    .unmarshallLocationName("bfdStatus").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(RouteServerPeer::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ROUTE_SERVER_PEER_ID_FIELD,
            ROUTE_SERVER_ENDPOINT_ID_FIELD, ROUTE_SERVER_ID_FIELD, VPC_ID_FIELD, SUBNET_ID_FIELD, STATE_FIELD,
            FAILURE_REASON_FIELD, ENDPOINT_ENI_ID_FIELD, ENDPOINT_ENI_ADDRESS_FIELD, PEER_ADDRESS_FIELD, BGP_OPTIONS_FIELD,
            BGP_STATUS_FIELD, BFD_STATUS_FIELD, TAGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String routeServerPeerId;

    private final String routeServerEndpointId;

    private final String routeServerId;

    private final String vpcId;

    private final String subnetId;

    private final String state;

    private final String failureReason;

    private final String endpointEniId;

    private final String endpointEniAddress;

    private final String peerAddress;

    private final RouteServerBgpOptions bgpOptions;

    private final RouteServerBgpStatus bgpStatus;

    private final RouteServerBfdStatus bfdStatus;

    private final List<Tag> tags;

    private RouteServerPeer(BuilderImpl builder) {
        this.routeServerPeerId = builder.routeServerPeerId;
        this.routeServerEndpointId = builder.routeServerEndpointId;
        this.routeServerId = builder.routeServerId;
        this.vpcId = builder.vpcId;
        this.subnetId = builder.subnetId;
        this.state = builder.state;
        this.failureReason = builder.failureReason;
        this.endpointEniId = builder.endpointEniId;
        this.endpointEniAddress = builder.endpointEniAddress;
        this.peerAddress = builder.peerAddress;
        this.bgpOptions = builder.bgpOptions;
        this.bgpStatus = builder.bgpStatus;
        this.bfdStatus = builder.bfdStatus;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The unique identifier of the route server peer.
     * </p>
     * 
     * @return The unique identifier of the route server peer.
     */
    public final String routeServerPeerId() {
        return routeServerPeerId;
    }

    /**
     * <p>
     * The ID of the route server endpoint associated with this peer.
     * </p>
     * 
     * @return The ID of the route server endpoint associated with this peer.
     */
    public final String routeServerEndpointId() {
        return routeServerEndpointId;
    }

    /**
     * <p>
     * The ID of the route server associated with this peer.
     * </p>
     * 
     * @return The ID of the route server associated with this peer.
     */
    public final String routeServerId() {
        return routeServerId;
    }

    /**
     * <p>
     * The ID of the VPC containing the route server peer.
     * </p>
     * 
     * @return The ID of the VPC containing the route server peer.
     */
    public final String vpcId() {
        return vpcId;
    }

    /**
     * <p>
     * The ID of the subnet containing the route server peer.
     * </p>
     * 
     * @return The ID of the subnet containing the route server peer.
     */
    public final String subnetId() {
        return subnetId;
    }

    /**
     * <p>
     * The current state of the route server peer.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link RouteServerPeerState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the route server peer.
     * @see RouteServerPeerState
     */
    public final RouteServerPeerState state() {
        return RouteServerPeerState.fromValue(state);
    }

    /**
     * <p>
     * The current state of the route server peer.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link RouteServerPeerState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the route server peer.
     * @see RouteServerPeerState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The reason for any failure in peer creation or operation.
     * </p>
     * 
     * @return The reason for any failure in peer creation or operation.
     */
    public final String failureReason() {
        return failureReason;
    }

    /**
     * <p>
     * The ID of the Elastic network interface for the route server endpoint.
     * </p>
     * 
     * @return The ID of the Elastic network interface for the route server endpoint.
     */
    public final String endpointEniId() {
        return endpointEniId;
    }

    /**
     * <p>
     * The IP address of the Elastic network interface for the route server endpoint.
     * </p>
     * 
     * @return The IP address of the Elastic network interface for the route server endpoint.
     */
    public final String endpointEniAddress() {
        return endpointEniAddress;
    }

    /**
     * <p>
     * The IPv4 address of the peer device.
     * </p>
     * 
     * @return The IPv4 address of the peer device.
     */
    public final String peerAddress() {
        return peerAddress;
    }

    /**
     * <p>
     * The BGP configuration options for this peer, including ASN (Autonomous System Number) and BFD (Bidrectional
     * Forwarding Detection) settings.
     * </p>
     * 
     * @return The BGP configuration options for this peer, including ASN (Autonomous System Number) and BFD
     *         (Bidrectional Forwarding Detection) settings.
     */
    public final RouteServerBgpOptions bgpOptions() {
        return bgpOptions;
    }

    /**
     * <p>
     * The current status of the BGP session with this peer.
     * </p>
     * 
     * @return The current status of the BGP session with this peer.
     */
    public final RouteServerBgpStatus bgpStatus() {
        return bgpStatus;
    }

    /**
     * <p>
     * The current status of the BFD session with this peer.
     * </p>
     * 
     * @return The current status of the BFD session with this peer.
     */
    public final RouteServerBfdStatus bfdStatus() {
        return bfdStatus;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Any tags assigned to the route server peer.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Any tags assigned to the route server peer.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(routeServerPeerId());
        hashCode = 31 * hashCode + Objects.hashCode(routeServerEndpointId());
        hashCode = 31 * hashCode + Objects.hashCode(routeServerId());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(subnetId());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(failureReason());
        hashCode = 31 * hashCode + Objects.hashCode(endpointEniId());
        hashCode = 31 * hashCode + Objects.hashCode(endpointEniAddress());
        hashCode = 31 * hashCode + Objects.hashCode(peerAddress());
        hashCode = 31 * hashCode + Objects.hashCode(bgpOptions());
        hashCode = 31 * hashCode + Objects.hashCode(bgpStatus());
        hashCode = 31 * hashCode + Objects.hashCode(bfdStatus());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RouteServerPeer)) {
            return false;
        }
        RouteServerPeer other = (RouteServerPeer) obj;
        return Objects.equals(routeServerPeerId(), other.routeServerPeerId())
                && Objects.equals(routeServerEndpointId(), other.routeServerEndpointId())
                && Objects.equals(routeServerId(), other.routeServerId()) && Objects.equals(vpcId(), other.vpcId())
                && Objects.equals(subnetId(), other.subnetId()) && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(failureReason(), other.failureReason())
                && Objects.equals(endpointEniId(), other.endpointEniId())
                && Objects.equals(endpointEniAddress(), other.endpointEniAddress())
                && Objects.equals(peerAddress(), other.peerAddress()) && Objects.equals(bgpOptions(), other.bgpOptions())
                && Objects.equals(bgpStatus(), other.bgpStatus()) && Objects.equals(bfdStatus(), other.bfdStatus())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RouteServerPeer").add("RouteServerPeerId", routeServerPeerId())
                .add("RouteServerEndpointId", routeServerEndpointId()).add("RouteServerId", routeServerId())
                .add("VpcId", vpcId()).add("SubnetId", subnetId()).add("State", stateAsString())
                .add("FailureReason", failureReason()).add("EndpointEniId", endpointEniId())
                .add("EndpointEniAddress", endpointEniAddress()).add("PeerAddress", peerAddress())
                .add("BgpOptions", bgpOptions()).add("BgpStatus", bgpStatus()).add("BfdStatus", bfdStatus())
                .add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RouteServerPeerId":
            return Optional.ofNullable(clazz.cast(routeServerPeerId()));
        case "RouteServerEndpointId":
            return Optional.ofNullable(clazz.cast(routeServerEndpointId()));
        case "RouteServerId":
            return Optional.ofNullable(clazz.cast(routeServerId()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "SubnetId":
            return Optional.ofNullable(clazz.cast(subnetId()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "FailureReason":
            return Optional.ofNullable(clazz.cast(failureReason()));
        case "EndpointEniId":
            return Optional.ofNullable(clazz.cast(endpointEniId()));
        case "EndpointEniAddress":
            return Optional.ofNullable(clazz.cast(endpointEniAddress()));
        case "PeerAddress":
            return Optional.ofNullable(clazz.cast(peerAddress()));
        case "BgpOptions":
            return Optional.ofNullable(clazz.cast(bgpOptions()));
        case "BgpStatus":
            return Optional.ofNullable(clazz.cast(bgpStatus()));
        case "BfdStatus":
            return Optional.ofNullable(clazz.cast(bfdStatus()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("RouteServerPeerId", ROUTE_SERVER_PEER_ID_FIELD);
        map.put("RouteServerEndpointId", ROUTE_SERVER_ENDPOINT_ID_FIELD);
        map.put("RouteServerId", ROUTE_SERVER_ID_FIELD);
        map.put("VpcId", VPC_ID_FIELD);
        map.put("SubnetId", SUBNET_ID_FIELD);
        map.put("State", STATE_FIELD);
        map.put("FailureReason", FAILURE_REASON_FIELD);
        map.put("EndpointEniId", ENDPOINT_ENI_ID_FIELD);
        map.put("EndpointEniAddress", ENDPOINT_ENI_ADDRESS_FIELD);
        map.put("PeerAddress", PEER_ADDRESS_FIELD);
        map.put("BgpOptions", BGP_OPTIONS_FIELD);
        map.put("BgpStatus", BGP_STATUS_FIELD);
        map.put("BfdStatus", BFD_STATUS_FIELD);
        map.put("TagSet", TAGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RouteServerPeer, T> g) {
        return obj -> g.apply((RouteServerPeer) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RouteServerPeer> {
        /**
         * <p>
         * The unique identifier of the route server peer.
         * </p>
         * 
         * @param routeServerPeerId
         *        The unique identifier of the route server peer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routeServerPeerId(String routeServerPeerId);

        /**
         * <p>
         * The ID of the route server endpoint associated with this peer.
         * </p>
         * 
         * @param routeServerEndpointId
         *        The ID of the route server endpoint associated with this peer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routeServerEndpointId(String routeServerEndpointId);

        /**
         * <p>
         * The ID of the route server associated with this peer.
         * </p>
         * 
         * @param routeServerId
         *        The ID of the route server associated with this peer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routeServerId(String routeServerId);

        /**
         * <p>
         * The ID of the VPC containing the route server peer.
         * </p>
         * 
         * @param vpcId
         *        The ID of the VPC containing the route server peer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * The ID of the subnet containing the route server peer.
         * </p>
         * 
         * @param subnetId
         *        The ID of the subnet containing the route server peer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetId(String subnetId);

        /**
         * <p>
         * The current state of the route server peer.
         * </p>
         * 
         * @param state
         *        The current state of the route server peer.
         * @see RouteServerPeerState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouteServerPeerState
         */
        Builder state(String state);

        /**
         * <p>
         * The current state of the route server peer.
         * </p>
         * 
         * @param state
         *        The current state of the route server peer.
         * @see RouteServerPeerState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouteServerPeerState
         */
        Builder state(RouteServerPeerState state);

        /**
         * <p>
         * The reason for any failure in peer creation or operation.
         * </p>
         * 
         * @param failureReason
         *        The reason for any failure in peer creation or operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReason(String failureReason);

        /**
         * <p>
         * The ID of the Elastic network interface for the route server endpoint.
         * </p>
         * 
         * @param endpointEniId
         *        The ID of the Elastic network interface for the route server endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointEniId(String endpointEniId);

        /**
         * <p>
         * The IP address of the Elastic network interface for the route server endpoint.
         * </p>
         * 
         * @param endpointEniAddress
         *        The IP address of the Elastic network interface for the route server endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointEniAddress(String endpointEniAddress);

        /**
         * <p>
         * The IPv4 address of the peer device.
         * </p>
         * 
         * @param peerAddress
         *        The IPv4 address of the peer device.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder peerAddress(String peerAddress);

        /**
         * <p>
         * The BGP configuration options for this peer, including ASN (Autonomous System Number) and BFD (Bidrectional
         * Forwarding Detection) settings.
         * </p>
         * 
         * @param bgpOptions
         *        The BGP configuration options for this peer, including ASN (Autonomous System Number) and BFD
         *        (Bidrectional Forwarding Detection) settings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bgpOptions(RouteServerBgpOptions bgpOptions);

        /**
         * <p>
         * The BGP configuration options for this peer, including ASN (Autonomous System Number) and BFD (Bidrectional
         * Forwarding Detection) settings.
         * </p>
         * This is a convenience method that creates an instance of the {@link RouteServerBgpOptions.Builder} avoiding
         * the need to create one manually via {@link RouteServerBgpOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RouteServerBgpOptions.Builder#build()} is called immediately and
         * its result is passed to {@link #bgpOptions(RouteServerBgpOptions)}.
         * 
         * @param bgpOptions
         *        a consumer that will call methods on {@link RouteServerBgpOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #bgpOptions(RouteServerBgpOptions)
         */
        default Builder bgpOptions(Consumer<RouteServerBgpOptions.Builder> bgpOptions) {
            return bgpOptions(RouteServerBgpOptions.builder().applyMutation(bgpOptions).build());
        }

        /**
         * <p>
         * The current status of the BGP session with this peer.
         * </p>
         * 
         * @param bgpStatus
         *        The current status of the BGP session with this peer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bgpStatus(RouteServerBgpStatus bgpStatus);

        /**
         * <p>
         * The current status of the BGP session with this peer.
         * </p>
         * This is a convenience method that creates an instance of the {@link RouteServerBgpStatus.Builder} avoiding
         * the need to create one manually via {@link RouteServerBgpStatus#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RouteServerBgpStatus.Builder#build()} is called immediately and
         * its result is passed to {@link #bgpStatus(RouteServerBgpStatus)}.
         * 
         * @param bgpStatus
         *        a consumer that will call methods on {@link RouteServerBgpStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #bgpStatus(RouteServerBgpStatus)
         */
        default Builder bgpStatus(Consumer<RouteServerBgpStatus.Builder> bgpStatus) {
            return bgpStatus(RouteServerBgpStatus.builder().applyMutation(bgpStatus).build());
        }

        /**
         * <p>
         * The current status of the BFD session with this peer.
         * </p>
         * 
         * @param bfdStatus
         *        The current status of the BFD session with this peer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bfdStatus(RouteServerBfdStatus bfdStatus);

        /**
         * <p>
         * The current status of the BFD session with this peer.
         * </p>
         * This is a convenience method that creates an instance of the {@link RouteServerBfdStatus.Builder} avoiding
         * the need to create one manually via {@link RouteServerBfdStatus#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RouteServerBfdStatus.Builder#build()} is called immediately and
         * its result is passed to {@link #bfdStatus(RouteServerBfdStatus)}.
         * 
         * @param bfdStatus
         *        a consumer that will call methods on {@link RouteServerBfdStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #bfdStatus(RouteServerBfdStatus)
         */
        default Builder bfdStatus(Consumer<RouteServerBfdStatus.Builder> bfdStatus) {
            return bfdStatus(RouteServerBfdStatus.builder().applyMutation(bfdStatus).build());
        }

        /**
         * <p>
         * Any tags assigned to the route server peer.
         * </p>
         * 
         * @param tags
         *        Any tags assigned to the route server peer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Any tags assigned to the route server peer.
         * </p>
         * 
         * @param tags
         *        Any tags assigned to the route server peer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Any tags assigned to the route server peer.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ec2.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ec2.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String routeServerPeerId;

        private String routeServerEndpointId;

        private String routeServerId;

        private String vpcId;

        private String subnetId;

        private String state;

        private String failureReason;

        private String endpointEniId;

        private String endpointEniAddress;

        private String peerAddress;

        private RouteServerBgpOptions bgpOptions;

        private RouteServerBgpStatus bgpStatus;

        private RouteServerBfdStatus bfdStatus;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(RouteServerPeer model) {
            routeServerPeerId(model.routeServerPeerId);
            routeServerEndpointId(model.routeServerEndpointId);
            routeServerId(model.routeServerId);
            vpcId(model.vpcId);
            subnetId(model.subnetId);
            state(model.state);
            failureReason(model.failureReason);
            endpointEniId(model.endpointEniId);
            endpointEniAddress(model.endpointEniAddress);
            peerAddress(model.peerAddress);
            bgpOptions(model.bgpOptions);
            bgpStatus(model.bgpStatus);
            bfdStatus(model.bfdStatus);
            tags(model.tags);
        }

        public final String getRouteServerPeerId() {
            return routeServerPeerId;
        }

        public final void setRouteServerPeerId(String routeServerPeerId) {
            this.routeServerPeerId = routeServerPeerId;
        }

        @Override
        public final Builder routeServerPeerId(String routeServerPeerId) {
            this.routeServerPeerId = routeServerPeerId;
            return this;
        }

        public final String getRouteServerEndpointId() {
            return routeServerEndpointId;
        }

        public final void setRouteServerEndpointId(String routeServerEndpointId) {
            this.routeServerEndpointId = routeServerEndpointId;
        }

        @Override
        public final Builder routeServerEndpointId(String routeServerEndpointId) {
            this.routeServerEndpointId = routeServerEndpointId;
            return this;
        }

        public final String getRouteServerId() {
            return routeServerId;
        }

        public final void setRouteServerId(String routeServerId) {
            this.routeServerId = routeServerId;
        }

        @Override
        public final Builder routeServerId(String routeServerId) {
            this.routeServerId = routeServerId;
            return this;
        }

        public final String getVpcId() {
            return vpcId;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final String getSubnetId() {
            return subnetId;
        }

        public final void setSubnetId(String subnetId) {
            this.subnetId = subnetId;
        }

        @Override
        public final Builder subnetId(String subnetId) {
            this.subnetId = subnetId;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(RouteServerPeerState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getFailureReason() {
            return failureReason;
        }

        public final void setFailureReason(String failureReason) {
            this.failureReason = failureReason;
        }

        @Override
        public final Builder failureReason(String failureReason) {
            this.failureReason = failureReason;
            return this;
        }

        public final String getEndpointEniId() {
            return endpointEniId;
        }

        public final void setEndpointEniId(String endpointEniId) {
            this.endpointEniId = endpointEniId;
        }

        @Override
        public final Builder endpointEniId(String endpointEniId) {
            this.endpointEniId = endpointEniId;
            return this;
        }

        public final String getEndpointEniAddress() {
            return endpointEniAddress;
        }

        public final void setEndpointEniAddress(String endpointEniAddress) {
            this.endpointEniAddress = endpointEniAddress;
        }

        @Override
        public final Builder endpointEniAddress(String endpointEniAddress) {
            this.endpointEniAddress = endpointEniAddress;
            return this;
        }

        public final String getPeerAddress() {
            return peerAddress;
        }

        public final void setPeerAddress(String peerAddress) {
            this.peerAddress = peerAddress;
        }

        @Override
        public final Builder peerAddress(String peerAddress) {
            this.peerAddress = peerAddress;
            return this;
        }

        public final RouteServerBgpOptions.Builder getBgpOptions() {
            return bgpOptions != null ? bgpOptions.toBuilder() : null;
        }

        public final void setBgpOptions(RouteServerBgpOptions.BuilderImpl bgpOptions) {
            this.bgpOptions = bgpOptions != null ? bgpOptions.build() : null;
        }

        @Override
        public final Builder bgpOptions(RouteServerBgpOptions bgpOptions) {
            this.bgpOptions = bgpOptions;
            return this;
        }

        public final RouteServerBgpStatus.Builder getBgpStatus() {
            return bgpStatus != null ? bgpStatus.toBuilder() : null;
        }

        public final void setBgpStatus(RouteServerBgpStatus.BuilderImpl bgpStatus) {
            this.bgpStatus = bgpStatus != null ? bgpStatus.build() : null;
        }

        @Override
        public final Builder bgpStatus(RouteServerBgpStatus bgpStatus) {
            this.bgpStatus = bgpStatus;
            return this;
        }

        public final RouteServerBfdStatus.Builder getBfdStatus() {
            return bfdStatus != null ? bfdStatus.toBuilder() : null;
        }

        public final void setBfdStatus(RouteServerBfdStatus.BuilderImpl bfdStatus) {
            this.bfdStatus = bfdStatus != null ? bfdStatus.build() : null;
        }

        @Override
        public final Builder bfdStatus(RouteServerBfdStatus bfdStatus) {
            this.bfdStatus = bfdStatus;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public RouteServerPeer build() {
            return new RouteServerPeer(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
