/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a route server and its configuration.
 * </p>
 * <p>
 * Amazon VPC Route Server simplifies routing for traffic between workloads that are deployed within a VPC and its
 * internet gateways. With this feature, VPC Route Server dynamically updates VPC and internet gateway route tables with
 * your preferred IPv4 or IPv6 routes to achieve routing fault tolerance for those workloads. This enables you to
 * automatically reroute traffic within a VPC, which increases the manageability of VPC routing and interoperability
 * with third-party workloads.
 * </p>
 * <p>
 * Route server supports the follow route table types:
 * </p>
 * <ul>
 * <li>
 * <p>
 * VPC route tables not associated with subnets
 * </p>
 * </li>
 * <li>
 * <p>
 * Subnet route tables
 * </p>
 * </li>
 * <li>
 * <p>
 * Internet gateway route tables
 * </p>
 * </li>
 * </ul>
 * <p>
 * Route server does not support route tables associated with virtual private gateways. To propagate routes into a
 * transit gateway route table, use <a href="https://docs.aws.amazon.com/vpc/latest/tgw/tgw-connect.html">Transit
 * Gateway Connect</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RouteServer implements SdkPojo, Serializable, ToCopyableBuilder<RouteServer.Builder, RouteServer> {
    private static final SdkField<String> ROUTE_SERVER_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("RouteServerId")
            .getter(getter(RouteServer::routeServerId))
            .setter(setter(Builder::routeServerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RouteServerId")
                    .unmarshallLocationName("routeServerId").build()).build();

    private static final SdkField<Long> AMAZON_SIDE_ASN_FIELD = SdkField
            .<Long> builder(MarshallingType.LONG)
            .memberName("AmazonSideAsn")
            .getter(getter(RouteServer::amazonSideAsn))
            .setter(setter(Builder::amazonSideAsn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AmazonSideAsn")
                    .unmarshallLocationName("amazonSideAsn").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("State")
            .getter(getter(RouteServer::stateAsString))
            .setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State")
                    .unmarshallLocationName("state").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(RouteServer::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> PERSIST_ROUTES_STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("PersistRoutesState")
            .getter(getter(RouteServer::persistRoutesStateAsString))
            .setter(setter(Builder::persistRoutesState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PersistRoutesState")
                    .unmarshallLocationName("persistRoutesState").build()).build();

    private static final SdkField<Long> PERSIST_ROUTES_DURATION_FIELD = SdkField
            .<Long> builder(MarshallingType.LONG)
            .memberName("PersistRoutesDuration")
            .getter(getter(RouteServer::persistRoutesDuration))
            .setter(setter(Builder::persistRoutesDuration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PersistRoutesDuration")
                    .unmarshallLocationName("persistRoutesDuration").build()).build();

    private static final SdkField<Boolean> SNS_NOTIFICATIONS_ENABLED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("SnsNotificationsEnabled")
            .getter(getter(RouteServer::snsNotificationsEnabled))
            .setter(setter(Builder::snsNotificationsEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnsNotificationsEnabled")
                    .unmarshallLocationName("snsNotificationsEnabled").build()).build();

    private static final SdkField<String> SNS_TOPIC_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SnsTopicArn")
            .getter(getter(RouteServer::snsTopicArn))
            .setter(setter(Builder::snsTopicArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnsTopicArn")
                    .unmarshallLocationName("snsTopicArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ROUTE_SERVER_ID_FIELD,
            AMAZON_SIDE_ASN_FIELD, STATE_FIELD, TAGS_FIELD, PERSIST_ROUTES_STATE_FIELD, PERSIST_ROUTES_DURATION_FIELD,
            SNS_NOTIFICATIONS_ENABLED_FIELD, SNS_TOPIC_ARN_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String routeServerId;

    private final Long amazonSideAsn;

    private final String state;

    private final List<Tag> tags;

    private final String persistRoutesState;

    private final Long persistRoutesDuration;

    private final Boolean snsNotificationsEnabled;

    private final String snsTopicArn;

    private RouteServer(BuilderImpl builder) {
        this.routeServerId = builder.routeServerId;
        this.amazonSideAsn = builder.amazonSideAsn;
        this.state = builder.state;
        this.tags = builder.tags;
        this.persistRoutesState = builder.persistRoutesState;
        this.persistRoutesDuration = builder.persistRoutesDuration;
        this.snsNotificationsEnabled = builder.snsNotificationsEnabled;
        this.snsTopicArn = builder.snsTopicArn;
    }

    /**
     * <p>
     * The unique identifier of the route server.
     * </p>
     * 
     * @return The unique identifier of the route server.
     */
    public final String routeServerId() {
        return routeServerId;
    }

    /**
     * <p>
     * The Border Gateway Protocol (BGP) Autonomous System Number (ASN) for the appliance. Valid values are from 1 to
     * 4294967295. We recommend using a private ASN in the 64512–65534 (16-bit ASN) or 4200000000–4294967294 (32-bit
     * ASN) range.
     * </p>
     * 
     * @return The Border Gateway Protocol (BGP) Autonomous System Number (ASN) for the appliance. Valid values are from
     *         1 to 4294967295. We recommend using a private ASN in the 64512–65534 (16-bit ASN) or
     *         4200000000–4294967294 (32-bit ASN) range.
     */
    public final Long amazonSideAsn() {
        return amazonSideAsn;
    }

    /**
     * <p>
     * The current state of the route server.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link RouteServerState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the route server.
     * @see RouteServerState
     */
    public final RouteServerState state() {
        return RouteServerState.fromValue(state);
    }

    /**
     * <p>
     * The current state of the route server.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link RouteServerState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the route server.
     * @see RouteServerState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Any tags assigned to the route server.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Any tags assigned to the route server.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The current state of route persistence for the route server.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #persistRoutesState} will return {@link RouteServerPersistRoutesState#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #persistRoutesStateAsString}.
     * </p>
     * 
     * @return The current state of route persistence for the route server.
     * @see RouteServerPersistRoutesState
     */
    public final RouteServerPersistRoutesState persistRoutesState() {
        return RouteServerPersistRoutesState.fromValue(persistRoutesState);
    }

    /**
     * <p>
     * The current state of route persistence for the route server.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #persistRoutesState} will return {@link RouteServerPersistRoutesState#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #persistRoutesStateAsString}.
     * </p>
     * 
     * @return The current state of route persistence for the route server.
     * @see RouteServerPersistRoutesState
     */
    public final String persistRoutesStateAsString() {
        return persistRoutesState;
    }

    /**
     * <p>
     * The number of minutes a route server will wait after BGP is re-established to unpersist the routes in the FIB and
     * RIB. Value must be in the range of 1-5. The default value is 1. Only valid if <code>persistRoutesState</code> is
     * 'enabled'.
     * </p>
     * <p>
     * If you set the duration to 1 minute, then when your network appliance re-establishes BGP with route server, it
     * has 1 minute to relearn it's adjacent network and advertise those routes to route server before route server
     * resumes normal functionality. In most cases, 1 minute is probably sufficient. If, however, you have concerns that
     * your BGP network may not be capable of fully re-establishing and re-learning everything in 1 minute, you can
     * increase the duration up to 5 minutes.
     * </p>
     * 
     * @return The number of minutes a route server will wait after BGP is re-established to unpersist the routes in the
     *         FIB and RIB. Value must be in the range of 1-5. The default value is 1. Only valid if
     *         <code>persistRoutesState</code> is 'enabled'.</p>
     *         <p>
     *         If you set the duration to 1 minute, then when your network appliance re-establishes BGP with route
     *         server, it has 1 minute to relearn it's adjacent network and advertise those routes to route server
     *         before route server resumes normal functionality. In most cases, 1 minute is probably sufficient. If,
     *         however, you have concerns that your BGP network may not be capable of fully re-establishing and
     *         re-learning everything in 1 minute, you can increase the duration up to 5 minutes.
     */
    public final Long persistRoutesDuration() {
        return persistRoutesDuration;
    }

    /**
     * <p>
     * Indicates whether SNS notifications are enabled for the route server. Enabling SNS notifications persists BGP
     * status changes to an SNS topic provisioned by Amazon Web Services.
     * </p>
     * 
     * @return Indicates whether SNS notifications are enabled for the route server. Enabling SNS notifications persists
     *         BGP status changes to an SNS topic provisioned by Amazon Web Services.
     */
    public final Boolean snsNotificationsEnabled() {
        return snsNotificationsEnabled;
    }

    /**
     * <p>
     * The ARN of the SNS topic where notifications are published.
     * </p>
     * 
     * @return The ARN of the SNS topic where notifications are published.
     */
    public final String snsTopicArn() {
        return snsTopicArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(routeServerId());
        hashCode = 31 * hashCode + Objects.hashCode(amazonSideAsn());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(persistRoutesStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(persistRoutesDuration());
        hashCode = 31 * hashCode + Objects.hashCode(snsNotificationsEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(snsTopicArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RouteServer)) {
            return false;
        }
        RouteServer other = (RouteServer) obj;
        return Objects.equals(routeServerId(), other.routeServerId()) && Objects.equals(amazonSideAsn(), other.amazonSideAsn())
                && Objects.equals(stateAsString(), other.stateAsString()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags())
                && Objects.equals(persistRoutesStateAsString(), other.persistRoutesStateAsString())
                && Objects.equals(persistRoutesDuration(), other.persistRoutesDuration())
                && Objects.equals(snsNotificationsEnabled(), other.snsNotificationsEnabled())
                && Objects.equals(snsTopicArn(), other.snsTopicArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RouteServer").add("RouteServerId", routeServerId()).add("AmazonSideAsn", amazonSideAsn())
                .add("State", stateAsString()).add("Tags", hasTags() ? tags() : null)
                .add("PersistRoutesState", persistRoutesStateAsString()).add("PersistRoutesDuration", persistRoutesDuration())
                .add("SnsNotificationsEnabled", snsNotificationsEnabled()).add("SnsTopicArn", snsTopicArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RouteServerId":
            return Optional.ofNullable(clazz.cast(routeServerId()));
        case "AmazonSideAsn":
            return Optional.ofNullable(clazz.cast(amazonSideAsn()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "PersistRoutesState":
            return Optional.ofNullable(clazz.cast(persistRoutesStateAsString()));
        case "PersistRoutesDuration":
            return Optional.ofNullable(clazz.cast(persistRoutesDuration()));
        case "SnsNotificationsEnabled":
            return Optional.ofNullable(clazz.cast(snsNotificationsEnabled()));
        case "SnsTopicArn":
            return Optional.ofNullable(clazz.cast(snsTopicArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("RouteServerId", ROUTE_SERVER_ID_FIELD);
        map.put("AmazonSideAsn", AMAZON_SIDE_ASN_FIELD);
        map.put("State", STATE_FIELD);
        map.put("TagSet", TAGS_FIELD);
        map.put("PersistRoutesState", PERSIST_ROUTES_STATE_FIELD);
        map.put("PersistRoutesDuration", PERSIST_ROUTES_DURATION_FIELD);
        map.put("SnsNotificationsEnabled", SNS_NOTIFICATIONS_ENABLED_FIELD);
        map.put("SnsTopicArn", SNS_TOPIC_ARN_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RouteServer, T> g) {
        return obj -> g.apply((RouteServer) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RouteServer> {
        /**
         * <p>
         * The unique identifier of the route server.
         * </p>
         * 
         * @param routeServerId
         *        The unique identifier of the route server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routeServerId(String routeServerId);

        /**
         * <p>
         * The Border Gateway Protocol (BGP) Autonomous System Number (ASN) for the appliance. Valid values are from 1
         * to 4294967295. We recommend using a private ASN in the 64512–65534 (16-bit ASN) or 4200000000–4294967294
         * (32-bit ASN) range.
         * </p>
         * 
         * @param amazonSideAsn
         *        The Border Gateway Protocol (BGP) Autonomous System Number (ASN) for the appliance. Valid values are
         *        from 1 to 4294967295. We recommend using a private ASN in the 64512–65534 (16-bit ASN) or
         *        4200000000–4294967294 (32-bit ASN) range.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder amazonSideAsn(Long amazonSideAsn);

        /**
         * <p>
         * The current state of the route server.
         * </p>
         * 
         * @param state
         *        The current state of the route server.
         * @see RouteServerState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouteServerState
         */
        Builder state(String state);

        /**
         * <p>
         * The current state of the route server.
         * </p>
         * 
         * @param state
         *        The current state of the route server.
         * @see RouteServerState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouteServerState
         */
        Builder state(RouteServerState state);

        /**
         * <p>
         * Any tags assigned to the route server.
         * </p>
         * 
         * @param tags
         *        Any tags assigned to the route server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Any tags assigned to the route server.
         * </p>
         * 
         * @param tags
         *        Any tags assigned to the route server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Any tags assigned to the route server.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ec2.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ec2.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The current state of route persistence for the route server.
         * </p>
         * 
         * @param persistRoutesState
         *        The current state of route persistence for the route server.
         * @see RouteServerPersistRoutesState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouteServerPersistRoutesState
         */
        Builder persistRoutesState(String persistRoutesState);

        /**
         * <p>
         * The current state of route persistence for the route server.
         * </p>
         * 
         * @param persistRoutesState
         *        The current state of route persistence for the route server.
         * @see RouteServerPersistRoutesState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouteServerPersistRoutesState
         */
        Builder persistRoutesState(RouteServerPersistRoutesState persistRoutesState);

        /**
         * <p>
         * The number of minutes a route server will wait after BGP is re-established to unpersist the routes in the FIB
         * and RIB. Value must be in the range of 1-5. The default value is 1. Only valid if
         * <code>persistRoutesState</code> is 'enabled'.
         * </p>
         * <p>
         * If you set the duration to 1 minute, then when your network appliance re-establishes BGP with route server,
         * it has 1 minute to relearn it's adjacent network and advertise those routes to route server before route
         * server resumes normal functionality. In most cases, 1 minute is probably sufficient. If, however, you have
         * concerns that your BGP network may not be capable of fully re-establishing and re-learning everything in 1
         * minute, you can increase the duration up to 5 minutes.
         * </p>
         * 
         * @param persistRoutesDuration
         *        The number of minutes a route server will wait after BGP is re-established to unpersist the routes in
         *        the FIB and RIB. Value must be in the range of 1-5. The default value is 1. Only valid if
         *        <code>persistRoutesState</code> is 'enabled'.</p>
         *        <p>
         *        If you set the duration to 1 minute, then when your network appliance re-establishes BGP with route
         *        server, it has 1 minute to relearn it's adjacent network and advertise those routes to route server
         *        before route server resumes normal functionality. In most cases, 1 minute is probably sufficient. If,
         *        however, you have concerns that your BGP network may not be capable of fully re-establishing and
         *        re-learning everything in 1 minute, you can increase the duration up to 5 minutes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder persistRoutesDuration(Long persistRoutesDuration);

        /**
         * <p>
         * Indicates whether SNS notifications are enabled for the route server. Enabling SNS notifications persists BGP
         * status changes to an SNS topic provisioned by Amazon Web Services.
         * </p>
         * 
         * @param snsNotificationsEnabled
         *        Indicates whether SNS notifications are enabled for the route server. Enabling SNS notifications
         *        persists BGP status changes to an SNS topic provisioned by Amazon Web Services.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snsNotificationsEnabled(Boolean snsNotificationsEnabled);

        /**
         * <p>
         * The ARN of the SNS topic where notifications are published.
         * </p>
         * 
         * @param snsTopicArn
         *        The ARN of the SNS topic where notifications are published.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snsTopicArn(String snsTopicArn);
    }

    static final class BuilderImpl implements Builder {
        private String routeServerId;

        private Long amazonSideAsn;

        private String state;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String persistRoutesState;

        private Long persistRoutesDuration;

        private Boolean snsNotificationsEnabled;

        private String snsTopicArn;

        private BuilderImpl() {
        }

        private BuilderImpl(RouteServer model) {
            routeServerId(model.routeServerId);
            amazonSideAsn(model.amazonSideAsn);
            state(model.state);
            tags(model.tags);
            persistRoutesState(model.persistRoutesState);
            persistRoutesDuration(model.persistRoutesDuration);
            snsNotificationsEnabled(model.snsNotificationsEnabled);
            snsTopicArn(model.snsTopicArn);
        }

        public final String getRouteServerId() {
            return routeServerId;
        }

        public final void setRouteServerId(String routeServerId) {
            this.routeServerId = routeServerId;
        }

        @Override
        public final Builder routeServerId(String routeServerId) {
            this.routeServerId = routeServerId;
            return this;
        }

        public final Long getAmazonSideAsn() {
            return amazonSideAsn;
        }

        public final void setAmazonSideAsn(Long amazonSideAsn) {
            this.amazonSideAsn = amazonSideAsn;
        }

        @Override
        public final Builder amazonSideAsn(Long amazonSideAsn) {
            this.amazonSideAsn = amazonSideAsn;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(RouteServerState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getPersistRoutesState() {
            return persistRoutesState;
        }

        public final void setPersistRoutesState(String persistRoutesState) {
            this.persistRoutesState = persistRoutesState;
        }

        @Override
        public final Builder persistRoutesState(String persistRoutesState) {
            this.persistRoutesState = persistRoutesState;
            return this;
        }

        @Override
        public final Builder persistRoutesState(RouteServerPersistRoutesState persistRoutesState) {
            this.persistRoutesState(persistRoutesState == null ? null : persistRoutesState.toString());
            return this;
        }

        public final Long getPersistRoutesDuration() {
            return persistRoutesDuration;
        }

        public final void setPersistRoutesDuration(Long persistRoutesDuration) {
            this.persistRoutesDuration = persistRoutesDuration;
        }

        @Override
        public final Builder persistRoutesDuration(Long persistRoutesDuration) {
            this.persistRoutesDuration = persistRoutesDuration;
            return this;
        }

        public final Boolean getSnsNotificationsEnabled() {
            return snsNotificationsEnabled;
        }

        public final void setSnsNotificationsEnabled(Boolean snsNotificationsEnabled) {
            this.snsNotificationsEnabled = snsNotificationsEnabled;
        }

        @Override
        public final Builder snsNotificationsEnabled(Boolean snsNotificationsEnabled) {
            this.snsNotificationsEnabled = snsNotificationsEnabled;
            return this;
        }

        public final String getSnsTopicArn() {
            return snsTopicArn;
        }

        public final void setSnsTopicArn(String snsTopicArn) {
            this.snsTopicArn = snsTopicArn;
        }

        @Override
        public final Builder snsTopicArn(String snsTopicArn) {
            this.snsTopicArn = snsTopicArn;
            return this;
        }

        @Override
        public RouteServer build() {
            return new RouteServer(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
