/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a network interface.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceNetworkInterfaceSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<InstanceNetworkInterfaceSpecification.Builder, InstanceNetworkInterfaceSpecification> {
    private static final SdkField<Boolean> ASSOCIATE_PUBLIC_IP_ADDRESS_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(InstanceNetworkInterfaceSpecification::associatePublicIpAddress))
            .setter(setter(Builder::associatePublicIpAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociatePublicIpAddress")
                    .unmarshallLocationName("associatePublicIpAddress").build()).build();

    private static final SdkField<Boolean> DELETE_ON_TERMINATION_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(InstanceNetworkInterfaceSpecification::deleteOnTermination))
            .setter(setter(Builder::deleteOnTermination))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeleteOnTermination")
                    .unmarshallLocationName("deleteOnTermination").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceNetworkInterfaceSpecification::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description")
                    .unmarshallLocationName("description").build()).build();

    private static final SdkField<Integer> DEVICE_INDEX_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(InstanceNetworkInterfaceSpecification::deviceIndex))
            .setter(setter(Builder::deviceIndex))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeviceIndex")
                    .unmarshallLocationName("deviceIndex").build()).build();

    private static final SdkField<List<String>> GROUPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(InstanceNetworkInterfaceSpecification::groups))
            .setter(setter(Builder::groups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroupId")
                    .unmarshallLocationName("SecurityGroupId").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("SecurityGroupId")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("SecurityGroupId").unmarshallLocationName("SecurityGroupId")
                                                    .build()).build()).build()).build();

    private static final SdkField<Integer> IPV6_ADDRESS_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(InstanceNetworkInterfaceSpecification::ipv6AddressCount))
            .setter(setter(Builder::ipv6AddressCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Ipv6AddressCount")
                    .unmarshallLocationName("ipv6AddressCount").build()).build();

    private static final SdkField<List<InstanceIpv6Address>> IPV6_ADDRESSES_FIELD = SdkField
            .<List<InstanceIpv6Address>> builder(MarshallingType.LIST)
            .getter(getter(InstanceNetworkInterfaceSpecification::ipv6Addresses))
            .setter(setter(Builder::ipv6Addresses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Ipv6Addresses")
                    .unmarshallLocationName("ipv6AddressesSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<InstanceIpv6Address> builder(MarshallingType.SDK_POJO)
                                            .constructor(InstanceIpv6Address::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> NETWORK_INTERFACE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceNetworkInterfaceSpecification::networkInterfaceId))
            .setter(setter(Builder::networkInterfaceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkInterfaceId")
                    .unmarshallLocationName("networkInterfaceId").build()).build();

    private static final SdkField<String> PRIVATE_IP_ADDRESS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceNetworkInterfaceSpecification::privateIpAddress))
            .setter(setter(Builder::privateIpAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PrivateIpAddress")
                    .unmarshallLocationName("privateIpAddress").build()).build();

    private static final SdkField<List<PrivateIpAddressSpecification>> PRIVATE_IP_ADDRESSES_FIELD = SdkField
            .<List<PrivateIpAddressSpecification>> builder(MarshallingType.LIST)
            .getter(getter(InstanceNetworkInterfaceSpecification::privateIpAddresses))
            .setter(setter(Builder::privateIpAddresses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PrivateIpAddresses")
                    .unmarshallLocationName("privateIpAddressesSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<PrivateIpAddressSpecification> builder(MarshallingType.SDK_POJO)
                                            .constructor(PrivateIpAddressSpecification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<Integer> SECONDARY_PRIVATE_IP_ADDRESS_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(InstanceNetworkInterfaceSpecification::secondaryPrivateIpAddressCount))
            .setter(setter(Builder::secondaryPrivateIpAddressCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecondaryPrivateIpAddressCount")
                    .unmarshallLocationName("secondaryPrivateIpAddressCount").build()).build();

    private static final SdkField<String> SUBNET_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceNetworkInterfaceSpecification::subnetId))
            .setter(setter(Builder::subnetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetId")
                    .unmarshallLocationName("subnetId").build()).build();

    private static final SdkField<String> INTERFACE_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceNetworkInterfaceSpecification::interfaceType))
            .setter(setter(Builder::interfaceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InterfaceType")
                    .unmarshallLocationName("InterfaceType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            ASSOCIATE_PUBLIC_IP_ADDRESS_FIELD, DELETE_ON_TERMINATION_FIELD, DESCRIPTION_FIELD, DEVICE_INDEX_FIELD, GROUPS_FIELD,
            IPV6_ADDRESS_COUNT_FIELD, IPV6_ADDRESSES_FIELD, NETWORK_INTERFACE_ID_FIELD, PRIVATE_IP_ADDRESS_FIELD,
            PRIVATE_IP_ADDRESSES_FIELD, SECONDARY_PRIVATE_IP_ADDRESS_COUNT_FIELD, SUBNET_ID_FIELD, INTERFACE_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean associatePublicIpAddress;

    private final Boolean deleteOnTermination;

    private final String description;

    private final Integer deviceIndex;

    private final List<String> groups;

    private final Integer ipv6AddressCount;

    private final List<InstanceIpv6Address> ipv6Addresses;

    private final String networkInterfaceId;

    private final String privateIpAddress;

    private final List<PrivateIpAddressSpecification> privateIpAddresses;

    private final Integer secondaryPrivateIpAddressCount;

    private final String subnetId;

    private final String interfaceType;

    private InstanceNetworkInterfaceSpecification(BuilderImpl builder) {
        this.associatePublicIpAddress = builder.associatePublicIpAddress;
        this.deleteOnTermination = builder.deleteOnTermination;
        this.description = builder.description;
        this.deviceIndex = builder.deviceIndex;
        this.groups = builder.groups;
        this.ipv6AddressCount = builder.ipv6AddressCount;
        this.ipv6Addresses = builder.ipv6Addresses;
        this.networkInterfaceId = builder.networkInterfaceId;
        this.privateIpAddress = builder.privateIpAddress;
        this.privateIpAddresses = builder.privateIpAddresses;
        this.secondaryPrivateIpAddressCount = builder.secondaryPrivateIpAddressCount;
        this.subnetId = builder.subnetId;
        this.interfaceType = builder.interfaceType;
    }

    /**
     * <p>
     * Indicates whether to assign a public IPv4 address to an instance you launch in a VPC. The public IP address can
     * only be assigned to a network interface for eth0, and can only be assigned to a new network interface, not an
     * existing one. You cannot specify more than one network interface in the request. If launching into a default
     * subnet, the default value is <code>true</code>.
     * </p>
     * 
     * @return Indicates whether to assign a public IPv4 address to an instance you launch in a VPC. The public IP
     *         address can only be assigned to a network interface for eth0, and can only be assigned to a new network
     *         interface, not an existing one. You cannot specify more than one network interface in the request. If
     *         launching into a default subnet, the default value is <code>true</code>.
     */
    public Boolean associatePublicIpAddress() {
        return associatePublicIpAddress;
    }

    /**
     * <p>
     * If set to <code>true</code>, the interface is deleted when the instance is terminated. You can specify
     * <code>true</code> only if creating a new network interface when launching an instance.
     * </p>
     * 
     * @return If set to <code>true</code>, the interface is deleted when the instance is terminated. You can specify
     *         <code>true</code> only if creating a new network interface when launching an instance.
     */
    public Boolean deleteOnTermination() {
        return deleteOnTermination;
    }

    /**
     * <p>
     * The description of the network interface. Applies only if creating a network interface when launching an
     * instance.
     * </p>
     * 
     * @return The description of the network interface. Applies only if creating a network interface when launching an
     *         instance.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The position of the network interface in the attachment order. A primary network interface has a device index of
     * 0.
     * </p>
     * <p>
     * If you specify a network interface when launching an instance, you must specify the device index.
     * </p>
     * 
     * @return The position of the network interface in the attachment order. A primary network interface has a device
     *         index of 0.</p>
     *         <p>
     *         If you specify a network interface when launching an instance, you must specify the device index.
     */
    public Integer deviceIndex() {
        return deviceIndex;
    }

    /**
     * <p>
     * The IDs of the security groups for the network interface. Applies only if creating a network interface when
     * launching an instance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The IDs of the security groups for the network interface. Applies only if creating a network interface
     *         when launching an instance.
     */
    public List<String> groups() {
        return groups;
    }

    /**
     * <p>
     * A number of IPv6 addresses to assign to the network interface. Amazon EC2 chooses the IPv6 addresses from the
     * range of the subnet. You cannot specify this option and the option to assign specific IPv6 addresses in the same
     * request. You can specify this option if you've specified a minimum number of instances to launch.
     * </p>
     * 
     * @return A number of IPv6 addresses to assign to the network interface. Amazon EC2 chooses the IPv6 addresses from
     *         the range of the subnet. You cannot specify this option and the option to assign specific IPv6 addresses
     *         in the same request. You can specify this option if you've specified a minimum number of instances to
     *         launch.
     */
    public Integer ipv6AddressCount() {
        return ipv6AddressCount;
    }

    /**
     * <p>
     * One or more IPv6 addresses to assign to the network interface. You cannot specify this option and the option to
     * assign a number of IPv6 addresses in the same request. You cannot specify this option if you've specified a
     * minimum number of instances to launch.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return One or more IPv6 addresses to assign to the network interface. You cannot specify this option and the
     *         option to assign a number of IPv6 addresses in the same request. You cannot specify this option if you've
     *         specified a minimum number of instances to launch.
     */
    public List<InstanceIpv6Address> ipv6Addresses() {
        return ipv6Addresses;
    }

    /**
     * <p>
     * The ID of the network interface.
     * </p>
     * 
     * @return The ID of the network interface.
     */
    public String networkInterfaceId() {
        return networkInterfaceId;
    }

    /**
     * <p>
     * The private IPv4 address of the network interface. Applies only if creating a network interface when launching an
     * instance. You cannot specify this option if you're launching more than one instance in a <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html">RunInstances</a> request.
     * </p>
     * 
     * @return The private IPv4 address of the network interface. Applies only if creating a network interface when
     *         launching an instance. You cannot specify this option if you're launching more than one instance in a <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html">RunInstances</a>
     *         request.
     */
    public String privateIpAddress() {
        return privateIpAddress;
    }

    /**
     * <p>
     * One or more private IPv4 addresses to assign to the network interface. Only one private IPv4 address can be
     * designated as primary. You cannot specify this option if you're launching more than one instance in a <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html">RunInstances</a> request.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return One or more private IPv4 addresses to assign to the network interface. Only one private IPv4 address can
     *         be designated as primary. You cannot specify this option if you're launching more than one instance in a
     *         <a href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html">RunInstances</a>
     *         request.
     */
    public List<PrivateIpAddressSpecification> privateIpAddresses() {
        return privateIpAddresses;
    }

    /**
     * <p>
     * The number of secondary private IPv4 addresses. You can't specify this option and specify more than one private
     * IP address using the private IP addresses option. You cannot specify this option if you're launching more than
     * one instance in a <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html">RunInstances</a> request.
     * </p>
     * 
     * @return The number of secondary private IPv4 addresses. You can't specify this option and specify more than one
     *         private IP address using the private IP addresses option. You cannot specify this option if you're
     *         launching more than one instance in a <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html">RunInstances</a>
     *         request.
     */
    public Integer secondaryPrivateIpAddressCount() {
        return secondaryPrivateIpAddressCount;
    }

    /**
     * <p>
     * The ID of the subnet associated with the network string. Applies only if creating a network interface when
     * launching an instance.
     * </p>
     * 
     * @return The ID of the subnet associated with the network string. Applies only if creating a network interface
     *         when launching an instance.
     */
    public String subnetId() {
        return subnetId;
    }

    /**
     * <p>
     * The type of network interface. To create an Elastic Fabric Adapter (EFA), specify <code>efa</code>. For more
     * information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/efa.html">Elastic Fabric
     * Adapter</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * <p>
     * If you are not creating an EFA, specify <code>interface</code> or omit this parameter.
     * </p>
     * <p>
     * Valide values: <code>interface</code> | <code>efa</code>
     * </p>
     * 
     * @return The type of network interface. To create an Elastic Fabric Adapter (EFA), specify <code>efa</code>. For
     *         more information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/efa.html">Elastic
     *         Fabric Adapter</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.</p>
     *         <p>
     *         If you are not creating an EFA, specify <code>interface</code> or omit this parameter.
     *         </p>
     *         <p>
     *         Valide values: <code>interface</code> | <code>efa</code>
     */
    public String interfaceType() {
        return interfaceType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(associatePublicIpAddress());
        hashCode = 31 * hashCode + Objects.hashCode(deleteOnTermination());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(deviceIndex());
        hashCode = 31 * hashCode + Objects.hashCode(groups());
        hashCode = 31 * hashCode + Objects.hashCode(ipv6AddressCount());
        hashCode = 31 * hashCode + Objects.hashCode(ipv6Addresses());
        hashCode = 31 * hashCode + Objects.hashCode(networkInterfaceId());
        hashCode = 31 * hashCode + Objects.hashCode(privateIpAddress());
        hashCode = 31 * hashCode + Objects.hashCode(privateIpAddresses());
        hashCode = 31 * hashCode + Objects.hashCode(secondaryPrivateIpAddressCount());
        hashCode = 31 * hashCode + Objects.hashCode(subnetId());
        hashCode = 31 * hashCode + Objects.hashCode(interfaceType());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceNetworkInterfaceSpecification)) {
            return false;
        }
        InstanceNetworkInterfaceSpecification other = (InstanceNetworkInterfaceSpecification) obj;
        return Objects.equals(associatePublicIpAddress(), other.associatePublicIpAddress())
                && Objects.equals(deleteOnTermination(), other.deleteOnTermination())
                && Objects.equals(description(), other.description()) && Objects.equals(deviceIndex(), other.deviceIndex())
                && Objects.equals(groups(), other.groups()) && Objects.equals(ipv6AddressCount(), other.ipv6AddressCount())
                && Objects.equals(ipv6Addresses(), other.ipv6Addresses())
                && Objects.equals(networkInterfaceId(), other.networkInterfaceId())
                && Objects.equals(privateIpAddress(), other.privateIpAddress())
                && Objects.equals(privateIpAddresses(), other.privateIpAddresses())
                && Objects.equals(secondaryPrivateIpAddressCount(), other.secondaryPrivateIpAddressCount())
                && Objects.equals(subnetId(), other.subnetId()) && Objects.equals(interfaceType(), other.interfaceType());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("InstanceNetworkInterfaceSpecification")
                .add("AssociatePublicIpAddress", associatePublicIpAddress()).add("DeleteOnTermination", deleteOnTermination())
                .add("Description", description()).add("DeviceIndex", deviceIndex()).add("Groups", groups())
                .add("Ipv6AddressCount", ipv6AddressCount()).add("Ipv6Addresses", ipv6Addresses())
                .add("NetworkInterfaceId", networkInterfaceId()).add("PrivateIpAddress", privateIpAddress())
                .add("PrivateIpAddresses", privateIpAddresses())
                .add("SecondaryPrivateIpAddressCount", secondaryPrivateIpAddressCount()).add("SubnetId", subnetId())
                .add("InterfaceType", interfaceType()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AssociatePublicIpAddress":
            return Optional.ofNullable(clazz.cast(associatePublicIpAddress()));
        case "DeleteOnTermination":
            return Optional.ofNullable(clazz.cast(deleteOnTermination()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "DeviceIndex":
            return Optional.ofNullable(clazz.cast(deviceIndex()));
        case "Groups":
            return Optional.ofNullable(clazz.cast(groups()));
        case "Ipv6AddressCount":
            return Optional.ofNullable(clazz.cast(ipv6AddressCount()));
        case "Ipv6Addresses":
            return Optional.ofNullable(clazz.cast(ipv6Addresses()));
        case "NetworkInterfaceId":
            return Optional.ofNullable(clazz.cast(networkInterfaceId()));
        case "PrivateIpAddress":
            return Optional.ofNullable(clazz.cast(privateIpAddress()));
        case "PrivateIpAddresses":
            return Optional.ofNullable(clazz.cast(privateIpAddresses()));
        case "SecondaryPrivateIpAddressCount":
            return Optional.ofNullable(clazz.cast(secondaryPrivateIpAddressCount()));
        case "SubnetId":
            return Optional.ofNullable(clazz.cast(subnetId()));
        case "InterfaceType":
            return Optional.ofNullable(clazz.cast(interfaceType()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InstanceNetworkInterfaceSpecification, T> g) {
        return obj -> g.apply((InstanceNetworkInterfaceSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceNetworkInterfaceSpecification> {
        /**
         * <p>
         * Indicates whether to assign a public IPv4 address to an instance you launch in a VPC. The public IP address
         * can only be assigned to a network interface for eth0, and can only be assigned to a new network interface,
         * not an existing one. You cannot specify more than one network interface in the request. If launching into a
         * default subnet, the default value is <code>true</code>.
         * </p>
         * 
         * @param associatePublicIpAddress
         *        Indicates whether to assign a public IPv4 address to an instance you launch in a VPC. The public IP
         *        address can only be assigned to a network interface for eth0, and can only be assigned to a new
         *        network interface, not an existing one. You cannot specify more than one network interface in the
         *        request. If launching into a default subnet, the default value is <code>true</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associatePublicIpAddress(Boolean associatePublicIpAddress);

        /**
         * <p>
         * If set to <code>true</code>, the interface is deleted when the instance is terminated. You can specify
         * <code>true</code> only if creating a new network interface when launching an instance.
         * </p>
         * 
         * @param deleteOnTermination
         *        If set to <code>true</code>, the interface is deleted when the instance is terminated. You can specify
         *        <code>true</code> only if creating a new network interface when launching an instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deleteOnTermination(Boolean deleteOnTermination);

        /**
         * <p>
         * The description of the network interface. Applies only if creating a network interface when launching an
         * instance.
         * </p>
         * 
         * @param description
         *        The description of the network interface. Applies only if creating a network interface when launching
         *        an instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The position of the network interface in the attachment order. A primary network interface has a device index
         * of 0.
         * </p>
         * <p>
         * If you specify a network interface when launching an instance, you must specify the device index.
         * </p>
         * 
         * @param deviceIndex
         *        The position of the network interface in the attachment order. A primary network interface has a
         *        device index of 0.</p>
         *        <p>
         *        If you specify a network interface when launching an instance, you must specify the device index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deviceIndex(Integer deviceIndex);

        /**
         * <p>
         * The IDs of the security groups for the network interface. Applies only if creating a network interface when
         * launching an instance.
         * </p>
         * 
         * @param groups
         *        The IDs of the security groups for the network interface. Applies only if creating a network interface
         *        when launching an instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groups(Collection<String> groups);

        /**
         * <p>
         * The IDs of the security groups for the network interface. Applies only if creating a network interface when
         * launching an instance.
         * </p>
         * 
         * @param groups
         *        The IDs of the security groups for the network interface. Applies only if creating a network interface
         *        when launching an instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groups(String... groups);

        /**
         * <p>
         * A number of IPv6 addresses to assign to the network interface. Amazon EC2 chooses the IPv6 addresses from the
         * range of the subnet. You cannot specify this option and the option to assign specific IPv6 addresses in the
         * same request. You can specify this option if you've specified a minimum number of instances to launch.
         * </p>
         * 
         * @param ipv6AddressCount
         *        A number of IPv6 addresses to assign to the network interface. Amazon EC2 chooses the IPv6 addresses
         *        from the range of the subnet. You cannot specify this option and the option to assign specific IPv6
         *        addresses in the same request. You can specify this option if you've specified a minimum number of
         *        instances to launch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv6AddressCount(Integer ipv6AddressCount);

        /**
         * <p>
         * One or more IPv6 addresses to assign to the network interface. You cannot specify this option and the option
         * to assign a number of IPv6 addresses in the same request. You cannot specify this option if you've specified
         * a minimum number of instances to launch.
         * </p>
         * 
         * @param ipv6Addresses
         *        One or more IPv6 addresses to assign to the network interface. You cannot specify this option and the
         *        option to assign a number of IPv6 addresses in the same request. You cannot specify this option if
         *        you've specified a minimum number of instances to launch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv6Addresses(Collection<InstanceIpv6Address> ipv6Addresses);

        /**
         * <p>
         * One or more IPv6 addresses to assign to the network interface. You cannot specify this option and the option
         * to assign a number of IPv6 addresses in the same request. You cannot specify this option if you've specified
         * a minimum number of instances to launch.
         * </p>
         * 
         * @param ipv6Addresses
         *        One or more IPv6 addresses to assign to the network interface. You cannot specify this option and the
         *        option to assign a number of IPv6 addresses in the same request. You cannot specify this option if
         *        you've specified a minimum number of instances to launch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv6Addresses(InstanceIpv6Address... ipv6Addresses);

        /**
         * <p>
         * One or more IPv6 addresses to assign to the network interface. You cannot specify this option and the option
         * to assign a number of IPv6 addresses in the same request. You cannot specify this option if you've specified
         * a minimum number of instances to launch.
         * </p>
         * This is a convenience that creates an instance of the {@link List<InstanceIpv6Address>.Builder} avoiding the
         * need to create one manually via {@link List<InstanceIpv6Address>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<InstanceIpv6Address>.Builder#build()} is called immediately
         * and its result is passed to {@link #ipv6Addresses(List<InstanceIpv6Address>)}.
         * 
         * @param ipv6Addresses
         *        a consumer that will call methods on {@link List<InstanceIpv6Address>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ipv6Addresses(List<InstanceIpv6Address>)
         */
        Builder ipv6Addresses(Consumer<InstanceIpv6Address.Builder>... ipv6Addresses);

        /**
         * <p>
         * The ID of the network interface.
         * </p>
         * 
         * @param networkInterfaceId
         *        The ID of the network interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaceId(String networkInterfaceId);

        /**
         * <p>
         * The private IPv4 address of the network interface. Applies only if creating a network interface when
         * launching an instance. You cannot specify this option if you're launching more than one instance in a <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html">RunInstances</a> request.
         * </p>
         * 
         * @param privateIpAddress
         *        The private IPv4 address of the network interface. Applies only if creating a network interface when
         *        launching an instance. You cannot specify this option if you're launching more than one instance in a
         *        <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html">RunInstances</a>
         *        request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privateIpAddress(String privateIpAddress);

        /**
         * <p>
         * One or more private IPv4 addresses to assign to the network interface. Only one private IPv4 address can be
         * designated as primary. You cannot specify this option if you're launching more than one instance in a <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html">RunInstances</a> request.
         * </p>
         * 
         * @param privateIpAddresses
         *        One or more private IPv4 addresses to assign to the network interface. Only one private IPv4 address
         *        can be designated as primary. You cannot specify this option if you're launching more than one
         *        instance in a <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html">RunInstances</a>
         *        request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privateIpAddresses(Collection<PrivateIpAddressSpecification> privateIpAddresses);

        /**
         * <p>
         * One or more private IPv4 addresses to assign to the network interface. Only one private IPv4 address can be
         * designated as primary. You cannot specify this option if you're launching more than one instance in a <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html">RunInstances</a> request.
         * </p>
         * 
         * @param privateIpAddresses
         *        One or more private IPv4 addresses to assign to the network interface. Only one private IPv4 address
         *        can be designated as primary. You cannot specify this option if you're launching more than one
         *        instance in a <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html">RunInstances</a>
         *        request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privateIpAddresses(PrivateIpAddressSpecification... privateIpAddresses);

        /**
         * <p>
         * One or more private IPv4 addresses to assign to the network interface. Only one private IPv4 address can be
         * designated as primary. You cannot specify this option if you're launching more than one instance in a <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html">RunInstances</a> request.
         * </p>
         * This is a convenience that creates an instance of the {@link List<PrivateIpAddressSpecification>.Builder}
         * avoiding the need to create one manually via {@link List<PrivateIpAddressSpecification>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<PrivateIpAddressSpecification>.Builder#build()} is called
         * immediately and its result is passed to {@link #privateIpAddresses(List<PrivateIpAddressSpecification>)}.
         * 
         * @param privateIpAddresses
         *        a consumer that will call methods on {@link List<PrivateIpAddressSpecification>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #privateIpAddresses(List<PrivateIpAddressSpecification>)
         */
        Builder privateIpAddresses(Consumer<PrivateIpAddressSpecification.Builder>... privateIpAddresses);

        /**
         * <p>
         * The number of secondary private IPv4 addresses. You can't specify this option and specify more than one
         * private IP address using the private IP addresses option. You cannot specify this option if you're launching
         * more than one instance in a <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html">RunInstances</a> request.
         * </p>
         * 
         * @param secondaryPrivateIpAddressCount
         *        The number of secondary private IPv4 addresses. You can't specify this option and specify more than
         *        one private IP address using the private IP addresses option. You cannot specify this option if you're
         *        launching more than one instance in a <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html">RunInstances</a>
         *        request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secondaryPrivateIpAddressCount(Integer secondaryPrivateIpAddressCount);

        /**
         * <p>
         * The ID of the subnet associated with the network string. Applies only if creating a network interface when
         * launching an instance.
         * </p>
         * 
         * @param subnetId
         *        The ID of the subnet associated with the network string. Applies only if creating a network interface
         *        when launching an instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetId(String subnetId);

        /**
         * <p>
         * The type of network interface. To create an Elastic Fabric Adapter (EFA), specify <code>efa</code>. For more
         * information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/efa.html">Elastic Fabric
         * Adapter</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
         * </p>
         * <p>
         * If you are not creating an EFA, specify <code>interface</code> or omit this parameter.
         * </p>
         * <p>
         * Valide values: <code>interface</code> | <code>efa</code>
         * </p>
         * 
         * @param interfaceType
         *        The type of network interface. To create an Elastic Fabric Adapter (EFA), specify <code>efa</code>.
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/efa.html">Elastic Fabric Adapter</a> in the
         *        <i>Amazon Elastic Compute Cloud User Guide</i>.</p>
         *        <p>
         *        If you are not creating an EFA, specify <code>interface</code> or omit this parameter.
         *        </p>
         *        <p>
         *        Valide values: <code>interface</code> | <code>efa</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder interfaceType(String interfaceType);
    }

    static final class BuilderImpl implements Builder {
        private Boolean associatePublicIpAddress;

        private Boolean deleteOnTermination;

        private String description;

        private Integer deviceIndex;

        private List<String> groups = DefaultSdkAutoConstructList.getInstance();

        private Integer ipv6AddressCount;

        private List<InstanceIpv6Address> ipv6Addresses = DefaultSdkAutoConstructList.getInstance();

        private String networkInterfaceId;

        private String privateIpAddress;

        private List<PrivateIpAddressSpecification> privateIpAddresses = DefaultSdkAutoConstructList.getInstance();

        private Integer secondaryPrivateIpAddressCount;

        private String subnetId;

        private String interfaceType;

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceNetworkInterfaceSpecification model) {
            associatePublicIpAddress(model.associatePublicIpAddress);
            deleteOnTermination(model.deleteOnTermination);
            description(model.description);
            deviceIndex(model.deviceIndex);
            groups(model.groups);
            ipv6AddressCount(model.ipv6AddressCount);
            ipv6Addresses(model.ipv6Addresses);
            networkInterfaceId(model.networkInterfaceId);
            privateIpAddress(model.privateIpAddress);
            privateIpAddresses(model.privateIpAddresses);
            secondaryPrivateIpAddressCount(model.secondaryPrivateIpAddressCount);
            subnetId(model.subnetId);
            interfaceType(model.interfaceType);
        }

        public final Boolean getAssociatePublicIpAddress() {
            return associatePublicIpAddress;
        }

        @Override
        public final Builder associatePublicIpAddress(Boolean associatePublicIpAddress) {
            this.associatePublicIpAddress = associatePublicIpAddress;
            return this;
        }

        public final void setAssociatePublicIpAddress(Boolean associatePublicIpAddress) {
            this.associatePublicIpAddress = associatePublicIpAddress;
        }

        public final Boolean getDeleteOnTermination() {
            return deleteOnTermination;
        }

        @Override
        public final Builder deleteOnTermination(Boolean deleteOnTermination) {
            this.deleteOnTermination = deleteOnTermination;
            return this;
        }

        public final void setDeleteOnTermination(Boolean deleteOnTermination) {
            this.deleteOnTermination = deleteOnTermination;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Integer getDeviceIndex() {
            return deviceIndex;
        }

        @Override
        public final Builder deviceIndex(Integer deviceIndex) {
            this.deviceIndex = deviceIndex;
            return this;
        }

        public final void setDeviceIndex(Integer deviceIndex) {
            this.deviceIndex = deviceIndex;
        }

        public final Collection<String> getGroups() {
            return groups;
        }

        @Override
        public final Builder groups(Collection<String> groups) {
            this.groups = SecurityGroupIdStringListCopier.copy(groups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder groups(String... groups) {
            groups(Arrays.asList(groups));
            return this;
        }

        public final void setGroups(Collection<String> groups) {
            this.groups = SecurityGroupIdStringListCopier.copy(groups);
        }

        public final Integer getIpv6AddressCount() {
            return ipv6AddressCount;
        }

        @Override
        public final Builder ipv6AddressCount(Integer ipv6AddressCount) {
            this.ipv6AddressCount = ipv6AddressCount;
            return this;
        }

        public final void setIpv6AddressCount(Integer ipv6AddressCount) {
            this.ipv6AddressCount = ipv6AddressCount;
        }

        public final Collection<InstanceIpv6Address.Builder> getIpv6Addresses() {
            return ipv6Addresses != null ? ipv6Addresses.stream().map(InstanceIpv6Address::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder ipv6Addresses(Collection<InstanceIpv6Address> ipv6Addresses) {
            this.ipv6Addresses = InstanceIpv6AddressListCopier.copy(ipv6Addresses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ipv6Addresses(InstanceIpv6Address... ipv6Addresses) {
            ipv6Addresses(Arrays.asList(ipv6Addresses));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ipv6Addresses(Consumer<InstanceIpv6Address.Builder>... ipv6Addresses) {
            ipv6Addresses(Stream.of(ipv6Addresses).map(c -> InstanceIpv6Address.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setIpv6Addresses(Collection<InstanceIpv6Address.BuilderImpl> ipv6Addresses) {
            this.ipv6Addresses = InstanceIpv6AddressListCopier.copyFromBuilder(ipv6Addresses);
        }

        public final String getNetworkInterfaceId() {
            return networkInterfaceId;
        }

        @Override
        public final Builder networkInterfaceId(String networkInterfaceId) {
            this.networkInterfaceId = networkInterfaceId;
            return this;
        }

        public final void setNetworkInterfaceId(String networkInterfaceId) {
            this.networkInterfaceId = networkInterfaceId;
        }

        public final String getPrivateIpAddress() {
            return privateIpAddress;
        }

        @Override
        public final Builder privateIpAddress(String privateIpAddress) {
            this.privateIpAddress = privateIpAddress;
            return this;
        }

        public final void setPrivateIpAddress(String privateIpAddress) {
            this.privateIpAddress = privateIpAddress;
        }

        public final Collection<PrivateIpAddressSpecification.Builder> getPrivateIpAddresses() {
            return privateIpAddresses != null ? privateIpAddresses.stream().map(PrivateIpAddressSpecification::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder privateIpAddresses(Collection<PrivateIpAddressSpecification> privateIpAddresses) {
            this.privateIpAddresses = PrivateIpAddressSpecificationListCopier.copy(privateIpAddresses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder privateIpAddresses(PrivateIpAddressSpecification... privateIpAddresses) {
            privateIpAddresses(Arrays.asList(privateIpAddresses));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder privateIpAddresses(Consumer<PrivateIpAddressSpecification.Builder>... privateIpAddresses) {
            privateIpAddresses(Stream.of(privateIpAddresses)
                    .map(c -> PrivateIpAddressSpecification.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setPrivateIpAddresses(Collection<PrivateIpAddressSpecification.BuilderImpl> privateIpAddresses) {
            this.privateIpAddresses = PrivateIpAddressSpecificationListCopier.copyFromBuilder(privateIpAddresses);
        }

        public final Integer getSecondaryPrivateIpAddressCount() {
            return secondaryPrivateIpAddressCount;
        }

        @Override
        public final Builder secondaryPrivateIpAddressCount(Integer secondaryPrivateIpAddressCount) {
            this.secondaryPrivateIpAddressCount = secondaryPrivateIpAddressCount;
            return this;
        }

        public final void setSecondaryPrivateIpAddressCount(Integer secondaryPrivateIpAddressCount) {
            this.secondaryPrivateIpAddressCount = secondaryPrivateIpAddressCount;
        }

        public final String getSubnetId() {
            return subnetId;
        }

        @Override
        public final Builder subnetId(String subnetId) {
            this.subnetId = subnetId;
            return this;
        }

        public final void setSubnetId(String subnetId) {
            this.subnetId = subnetId;
        }

        public final String getInterfaceType() {
            return interfaceType;
        }

        @Override
        public final Builder interfaceType(String interfaceType) {
            this.interfaceType = interfaceType;
            return this;
        }

        public final void setInterfaceType(String interfaceType) {
            this.interfaceType = interfaceType;
        }

        @Override
        public InstanceNetworkInterfaceSpecification build() {
            return new InstanceNetworkInterfaceSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
