/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a Client VPN endpoint.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ClientVpnEndpoint implements SdkPojo, Serializable,
        ToCopyableBuilder<ClientVpnEndpoint.Builder, ClientVpnEndpoint> {
    private static final SdkField<String> CLIENT_VPN_ENDPOINT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(ClientVpnEndpoint::clientVpnEndpointId))
            .setter(setter(Builder::clientVpnEndpointId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientVpnEndpointId")
                    .unmarshallLocationName("clientVpnEndpointId").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(ClientVpnEndpoint::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description")
                    .unmarshallLocationName("description").build()).build();

    private static final SdkField<ClientVpnEndpointStatus> STATUS_FIELD = SdkField
            .<ClientVpnEndpointStatus> builder(MarshallingType.SDK_POJO)
            .getter(getter(ClientVpnEndpoint::status))
            .setter(setter(Builder::status))
            .constructor(ClientVpnEndpointStatus::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status")
                    .unmarshallLocationName("status").build()).build();

    private static final SdkField<String> CREATION_TIME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(ClientVpnEndpoint::creationTime))
            .setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime")
                    .unmarshallLocationName("creationTime").build()).build();

    private static final SdkField<String> DELETION_TIME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(ClientVpnEndpoint::deletionTime))
            .setter(setter(Builder::deletionTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeletionTime")
                    .unmarshallLocationName("deletionTime").build()).build();

    private static final SdkField<String> DNS_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(ClientVpnEndpoint::dnsName))
            .setter(setter(Builder::dnsName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsName")
                    .unmarshallLocationName("dnsName").build()).build();

    private static final SdkField<String> CLIENT_CIDR_BLOCK_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(ClientVpnEndpoint::clientCidrBlock))
            .setter(setter(Builder::clientCidrBlock))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientCidrBlock")
                    .unmarshallLocationName("clientCidrBlock").build()).build();

    private static final SdkField<List<String>> DNS_SERVERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(ClientVpnEndpoint::dnsServers))
            .setter(setter(Builder::dnsServers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsServer")
                    .unmarshallLocationName("dnsServer").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<Boolean> SPLIT_TUNNEL_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(ClientVpnEndpoint::splitTunnel))
            .setter(setter(Builder::splitTunnel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SplitTunnel")
                    .unmarshallLocationName("splitTunnel").build()).build();

    private static final SdkField<String> VPN_PROTOCOL_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(ClientVpnEndpoint::vpnProtocolAsString))
            .setter(setter(Builder::vpnProtocol))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpnProtocol")
                    .unmarshallLocationName("vpnProtocol").build()).build();

    private static final SdkField<String> TRANSPORT_PROTOCOL_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(ClientVpnEndpoint::transportProtocolAsString))
            .setter(setter(Builder::transportProtocol))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TransportProtocol")
                    .unmarshallLocationName("transportProtocol").build()).build();

    private static final SdkField<List<AssociatedTargetNetwork>> ASSOCIATED_TARGET_NETWORKS_FIELD = SdkField
            .<List<AssociatedTargetNetwork>> builder(MarshallingType.LIST)
            .getter(getter(ClientVpnEndpoint::associatedTargetNetworks))
            .setter(setter(Builder::associatedTargetNetworks))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociatedTargetNetwork")
                    .unmarshallLocationName("associatedTargetNetwork").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<AssociatedTargetNetwork> builder(MarshallingType.SDK_POJO)
                                            .constructor(AssociatedTargetNetwork::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> SERVER_CERTIFICATE_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(ClientVpnEndpoint::serverCertificateArn))
            .setter(setter(Builder::serverCertificateArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerCertificateArn")
                    .unmarshallLocationName("serverCertificateArn").build()).build();

    private static final SdkField<List<ClientVpnAuthentication>> AUTHENTICATION_OPTIONS_FIELD = SdkField
            .<List<ClientVpnAuthentication>> builder(MarshallingType.LIST)
            .getter(getter(ClientVpnEndpoint::authenticationOptions))
            .setter(setter(Builder::authenticationOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AuthenticationOptions")
                    .unmarshallLocationName("authenticationOptions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<ClientVpnAuthentication> builder(MarshallingType.SDK_POJO)
                                            .constructor(ClientVpnAuthentication::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<ConnectionLogResponseOptions> CONNECTION_LOG_OPTIONS_FIELD = SdkField
            .<ConnectionLogResponseOptions> builder(MarshallingType.SDK_POJO)
            .getter(getter(ClientVpnEndpoint::connectionLogOptions))
            .setter(setter(Builder::connectionLogOptions))
            .constructor(ConnectionLogResponseOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectionLogOptions")
                    .unmarshallLocationName("connectionLogOptions").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .getter(getter(ClientVpnEndpoint::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CLIENT_VPN_ENDPOINT_ID_FIELD,
            DESCRIPTION_FIELD, STATUS_FIELD, CREATION_TIME_FIELD, DELETION_TIME_FIELD, DNS_NAME_FIELD, CLIENT_CIDR_BLOCK_FIELD,
            DNS_SERVERS_FIELD, SPLIT_TUNNEL_FIELD, VPN_PROTOCOL_FIELD, TRANSPORT_PROTOCOL_FIELD,
            ASSOCIATED_TARGET_NETWORKS_FIELD, SERVER_CERTIFICATE_ARN_FIELD, AUTHENTICATION_OPTIONS_FIELD,
            CONNECTION_LOG_OPTIONS_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String clientVpnEndpointId;

    private final String description;

    private final ClientVpnEndpointStatus status;

    private final String creationTime;

    private final String deletionTime;

    private final String dnsName;

    private final String clientCidrBlock;

    private final List<String> dnsServers;

    private final Boolean splitTunnel;

    private final String vpnProtocol;

    private final String transportProtocol;

    private final List<AssociatedTargetNetwork> associatedTargetNetworks;

    private final String serverCertificateArn;

    private final List<ClientVpnAuthentication> authenticationOptions;

    private final ConnectionLogResponseOptions connectionLogOptions;

    private final List<Tag> tags;

    private ClientVpnEndpoint(BuilderImpl builder) {
        this.clientVpnEndpointId = builder.clientVpnEndpointId;
        this.description = builder.description;
        this.status = builder.status;
        this.creationTime = builder.creationTime;
        this.deletionTime = builder.deletionTime;
        this.dnsName = builder.dnsName;
        this.clientCidrBlock = builder.clientCidrBlock;
        this.dnsServers = builder.dnsServers;
        this.splitTunnel = builder.splitTunnel;
        this.vpnProtocol = builder.vpnProtocol;
        this.transportProtocol = builder.transportProtocol;
        this.associatedTargetNetworks = builder.associatedTargetNetworks;
        this.serverCertificateArn = builder.serverCertificateArn;
        this.authenticationOptions = builder.authenticationOptions;
        this.connectionLogOptions = builder.connectionLogOptions;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The ID of the Client VPN endpoint.
     * </p>
     * 
     * @return The ID of the Client VPN endpoint.
     */
    public String clientVpnEndpointId() {
        return clientVpnEndpointId;
    }

    /**
     * <p>
     * A brief description of the endpoint.
     * </p>
     * 
     * @return A brief description of the endpoint.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The current state of the Client VPN endpoint.
     * </p>
     * 
     * @return The current state of the Client VPN endpoint.
     */
    public ClientVpnEndpointStatus status() {
        return status;
    }

    /**
     * <p>
     * The date and time the Client VPN endpoint was created.
     * </p>
     * 
     * @return The date and time the Client VPN endpoint was created.
     */
    public String creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The date and time the Client VPN endpoint was deleted, if applicable.
     * </p>
     * 
     * @return The date and time the Client VPN endpoint was deleted, if applicable.
     */
    public String deletionTime() {
        return deletionTime;
    }

    /**
     * <p>
     * The DNS name to be used by clients when connecting to the Client VPN endpoint.
     * </p>
     * 
     * @return The DNS name to be used by clients when connecting to the Client VPN endpoint.
     */
    public String dnsName() {
        return dnsName;
    }

    /**
     * <p>
     * The IPv4 address range, in CIDR notation, from which client IP addresses are assigned.
     * </p>
     * 
     * @return The IPv4 address range, in CIDR notation, from which client IP addresses are assigned.
     */
    public String clientCidrBlock() {
        return clientCidrBlock;
    }

    /**
     * <p>
     * Information about the DNS servers to be used for DNS resolution.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Information about the DNS servers to be used for DNS resolution.
     */
    public List<String> dnsServers() {
        return dnsServers;
    }

    /**
     * <p>
     * Indicates whether VPN split tunneling is supported.
     * </p>
     * 
     * @return Indicates whether VPN split tunneling is supported.
     */
    public Boolean splitTunnel() {
        return splitTunnel;
    }

    /**
     * <p>
     * The protocol used by the VPN session.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #vpnProtocol} will
     * return {@link VpnProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #vpnProtocolAsString}.
     * </p>
     * 
     * @return The protocol used by the VPN session.
     * @see VpnProtocol
     */
    public VpnProtocol vpnProtocol() {
        return VpnProtocol.fromValue(vpnProtocol);
    }

    /**
     * <p>
     * The protocol used by the VPN session.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #vpnProtocol} will
     * return {@link VpnProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #vpnProtocolAsString}.
     * </p>
     * 
     * @return The protocol used by the VPN session.
     * @see VpnProtocol
     */
    public String vpnProtocolAsString() {
        return vpnProtocol;
    }

    /**
     * <p>
     * The transport protocol used by the Client VPN endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #transportProtocol}
     * will return {@link TransportProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #transportProtocolAsString}.
     * </p>
     * 
     * @return The transport protocol used by the Client VPN endpoint.
     * @see TransportProtocol
     */
    public TransportProtocol transportProtocol() {
        return TransportProtocol.fromValue(transportProtocol);
    }

    /**
     * <p>
     * The transport protocol used by the Client VPN endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #transportProtocol}
     * will return {@link TransportProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #transportProtocolAsString}.
     * </p>
     * 
     * @return The transport protocol used by the Client VPN endpoint.
     * @see TransportProtocol
     */
    public String transportProtocolAsString() {
        return transportProtocol;
    }

    /**
     * <p>
     * Information about the associated target networks. A target network is a subnet in a VPC.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Information about the associated target networks. A target network is a subnet in a VPC.
     */
    public List<AssociatedTargetNetwork> associatedTargetNetworks() {
        return associatedTargetNetworks;
    }

    /**
     * <p>
     * The ARN of the server certificate.
     * </p>
     * 
     * @return The ARN of the server certificate.
     */
    public String serverCertificateArn() {
        return serverCertificateArn;
    }

    /**
     * <p>
     * Information about the authentication method used by the Client VPN endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Information about the authentication method used by the Client VPN endpoint.
     */
    public List<ClientVpnAuthentication> authenticationOptions() {
        return authenticationOptions;
    }

    /**
     * <p>
     * Information about the client connection logging options for the Client VPN endpoint.
     * </p>
     * 
     * @return Information about the client connection logging options for the Client VPN endpoint.
     */
    public ConnectionLogResponseOptions connectionLogOptions() {
        return connectionLogOptions;
    }

    /**
     * <p>
     * Any tags assigned to the Client VPN endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Any tags assigned to the Client VPN endpoint.
     */
    public List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(clientVpnEndpointId());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(deletionTime());
        hashCode = 31 * hashCode + Objects.hashCode(dnsName());
        hashCode = 31 * hashCode + Objects.hashCode(clientCidrBlock());
        hashCode = 31 * hashCode + Objects.hashCode(dnsServers());
        hashCode = 31 * hashCode + Objects.hashCode(splitTunnel());
        hashCode = 31 * hashCode + Objects.hashCode(vpnProtocolAsString());
        hashCode = 31 * hashCode + Objects.hashCode(transportProtocolAsString());
        hashCode = 31 * hashCode + Objects.hashCode(associatedTargetNetworks());
        hashCode = 31 * hashCode + Objects.hashCode(serverCertificateArn());
        hashCode = 31 * hashCode + Objects.hashCode(authenticationOptions());
        hashCode = 31 * hashCode + Objects.hashCode(connectionLogOptions());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ClientVpnEndpoint)) {
            return false;
        }
        ClientVpnEndpoint other = (ClientVpnEndpoint) obj;
        return Objects.equals(clientVpnEndpointId(), other.clientVpnEndpointId())
                && Objects.equals(description(), other.description()) && Objects.equals(status(), other.status())
                && Objects.equals(creationTime(), other.creationTime()) && Objects.equals(deletionTime(), other.deletionTime())
                && Objects.equals(dnsName(), other.dnsName()) && Objects.equals(clientCidrBlock(), other.clientCidrBlock())
                && Objects.equals(dnsServers(), other.dnsServers()) && Objects.equals(splitTunnel(), other.splitTunnel())
                && Objects.equals(vpnProtocolAsString(), other.vpnProtocolAsString())
                && Objects.equals(transportProtocolAsString(), other.transportProtocolAsString())
                && Objects.equals(associatedTargetNetworks(), other.associatedTargetNetworks())
                && Objects.equals(serverCertificateArn(), other.serverCertificateArn())
                && Objects.equals(authenticationOptions(), other.authenticationOptions())
                && Objects.equals(connectionLogOptions(), other.connectionLogOptions()) && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ClientVpnEndpoint").add("ClientVpnEndpointId", clientVpnEndpointId())
                .add("Description", description()).add("Status", status()).add("CreationTime", creationTime())
                .add("DeletionTime", deletionTime()).add("DnsName", dnsName()).add("ClientCidrBlock", clientCidrBlock())
                .add("DnsServers", dnsServers()).add("SplitTunnel", splitTunnel()).add("VpnProtocol", vpnProtocolAsString())
                .add("TransportProtocol", transportProtocolAsString())
                .add("AssociatedTargetNetworks", associatedTargetNetworks()).add("ServerCertificateArn", serverCertificateArn())
                .add("AuthenticationOptions", authenticationOptions()).add("ConnectionLogOptions", connectionLogOptions())
                .add("Tags", tags()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ClientVpnEndpointId":
            return Optional.ofNullable(clazz.cast(clientVpnEndpointId()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "DeletionTime":
            return Optional.ofNullable(clazz.cast(deletionTime()));
        case "DnsName":
            return Optional.ofNullable(clazz.cast(dnsName()));
        case "ClientCidrBlock":
            return Optional.ofNullable(clazz.cast(clientCidrBlock()));
        case "DnsServers":
            return Optional.ofNullable(clazz.cast(dnsServers()));
        case "SplitTunnel":
            return Optional.ofNullable(clazz.cast(splitTunnel()));
        case "VpnProtocol":
            return Optional.ofNullable(clazz.cast(vpnProtocolAsString()));
        case "TransportProtocol":
            return Optional.ofNullable(clazz.cast(transportProtocolAsString()));
        case "AssociatedTargetNetworks":
            return Optional.ofNullable(clazz.cast(associatedTargetNetworks()));
        case "ServerCertificateArn":
            return Optional.ofNullable(clazz.cast(serverCertificateArn()));
        case "AuthenticationOptions":
            return Optional.ofNullable(clazz.cast(authenticationOptions()));
        case "ConnectionLogOptions":
            return Optional.ofNullable(clazz.cast(connectionLogOptions()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ClientVpnEndpoint, T> g) {
        return obj -> g.apply((ClientVpnEndpoint) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ClientVpnEndpoint> {
        /**
         * <p>
         * The ID of the Client VPN endpoint.
         * </p>
         * 
         * @param clientVpnEndpointId
         *        The ID of the Client VPN endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientVpnEndpointId(String clientVpnEndpointId);

        /**
         * <p>
         * A brief description of the endpoint.
         * </p>
         * 
         * @param description
         *        A brief description of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The current state of the Client VPN endpoint.
         * </p>
         * 
         * @param status
         *        The current state of the Client VPN endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(ClientVpnEndpointStatus status);

        /**
         * <p>
         * The current state of the Client VPN endpoint.
         * </p>
         * This is a convenience that creates an instance of the {@link ClientVpnEndpointStatus.Builder} avoiding the
         * need to create one manually via {@link ClientVpnEndpointStatus#builder()}.
         *
         * When the {@link Consumer} completes, {@link ClientVpnEndpointStatus.Builder#build()} is called immediately
         * and its result is passed to {@link #status(ClientVpnEndpointStatus)}.
         * 
         * @param status
         *        a consumer that will call methods on {@link ClientVpnEndpointStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #status(ClientVpnEndpointStatus)
         */
        default Builder status(Consumer<ClientVpnEndpointStatus.Builder> status) {
            return status(ClientVpnEndpointStatus.builder().applyMutation(status).build());
        }

        /**
         * <p>
         * The date and time the Client VPN endpoint was created.
         * </p>
         * 
         * @param creationTime
         *        The date and time the Client VPN endpoint was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(String creationTime);

        /**
         * <p>
         * The date and time the Client VPN endpoint was deleted, if applicable.
         * </p>
         * 
         * @param deletionTime
         *        The date and time the Client VPN endpoint was deleted, if applicable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deletionTime(String deletionTime);

        /**
         * <p>
         * The DNS name to be used by clients when connecting to the Client VPN endpoint.
         * </p>
         * 
         * @param dnsName
         *        The DNS name to be used by clients when connecting to the Client VPN endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsName(String dnsName);

        /**
         * <p>
         * The IPv4 address range, in CIDR notation, from which client IP addresses are assigned.
         * </p>
         * 
         * @param clientCidrBlock
         *        The IPv4 address range, in CIDR notation, from which client IP addresses are assigned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientCidrBlock(String clientCidrBlock);

        /**
         * <p>
         * Information about the DNS servers to be used for DNS resolution.
         * </p>
         * 
         * @param dnsServers
         *        Information about the DNS servers to be used for DNS resolution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsServers(Collection<String> dnsServers);

        /**
         * <p>
         * Information about the DNS servers to be used for DNS resolution.
         * </p>
         * 
         * @param dnsServers
         *        Information about the DNS servers to be used for DNS resolution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsServers(String... dnsServers);

        /**
         * <p>
         * Indicates whether VPN split tunneling is supported.
         * </p>
         * 
         * @param splitTunnel
         *        Indicates whether VPN split tunneling is supported.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder splitTunnel(Boolean splitTunnel);

        /**
         * <p>
         * The protocol used by the VPN session.
         * </p>
         * 
         * @param vpnProtocol
         *        The protocol used by the VPN session.
         * @see VpnProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpnProtocol
         */
        Builder vpnProtocol(String vpnProtocol);

        /**
         * <p>
         * The protocol used by the VPN session.
         * </p>
         * 
         * @param vpnProtocol
         *        The protocol used by the VPN session.
         * @see VpnProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpnProtocol
         */
        Builder vpnProtocol(VpnProtocol vpnProtocol);

        /**
         * <p>
         * The transport protocol used by the Client VPN endpoint.
         * </p>
         * 
         * @param transportProtocol
         *        The transport protocol used by the Client VPN endpoint.
         * @see TransportProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransportProtocol
         */
        Builder transportProtocol(String transportProtocol);

        /**
         * <p>
         * The transport protocol used by the Client VPN endpoint.
         * </p>
         * 
         * @param transportProtocol
         *        The transport protocol used by the Client VPN endpoint.
         * @see TransportProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransportProtocol
         */
        Builder transportProtocol(TransportProtocol transportProtocol);

        /**
         * <p>
         * Information about the associated target networks. A target network is a subnet in a VPC.
         * </p>
         * 
         * @param associatedTargetNetworks
         *        Information about the associated target networks. A target network is a subnet in a VPC.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associatedTargetNetworks(Collection<AssociatedTargetNetwork> associatedTargetNetworks);

        /**
         * <p>
         * Information about the associated target networks. A target network is a subnet in a VPC.
         * </p>
         * 
         * @param associatedTargetNetworks
         *        Information about the associated target networks. A target network is a subnet in a VPC.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associatedTargetNetworks(AssociatedTargetNetwork... associatedTargetNetworks);

        /**
         * <p>
         * Information about the associated target networks. A target network is a subnet in a VPC.
         * </p>
         * This is a convenience that creates an instance of the {@link List<AssociatedTargetNetwork>.Builder} avoiding
         * the need to create one manually via {@link List<AssociatedTargetNetwork>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AssociatedTargetNetwork>.Builder#build()} is called
         * immediately and its result is passed to {@link #associatedTargetNetworks(List<AssociatedTargetNetwork>)}.
         * 
         * @param associatedTargetNetworks
         *        a consumer that will call methods on {@link List<AssociatedTargetNetwork>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #associatedTargetNetworks(List<AssociatedTargetNetwork>)
         */
        Builder associatedTargetNetworks(Consumer<AssociatedTargetNetwork.Builder>... associatedTargetNetworks);

        /**
         * <p>
         * The ARN of the server certificate.
         * </p>
         * 
         * @param serverCertificateArn
         *        The ARN of the server certificate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverCertificateArn(String serverCertificateArn);

        /**
         * <p>
         * Information about the authentication method used by the Client VPN endpoint.
         * </p>
         * 
         * @param authenticationOptions
         *        Information about the authentication method used by the Client VPN endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authenticationOptions(Collection<ClientVpnAuthentication> authenticationOptions);

        /**
         * <p>
         * Information about the authentication method used by the Client VPN endpoint.
         * </p>
         * 
         * @param authenticationOptions
         *        Information about the authentication method used by the Client VPN endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authenticationOptions(ClientVpnAuthentication... authenticationOptions);

        /**
         * <p>
         * Information about the authentication method used by the Client VPN endpoint.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ClientVpnAuthentication>.Builder} avoiding
         * the need to create one manually via {@link List<ClientVpnAuthentication>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ClientVpnAuthentication>.Builder#build()} is called
         * immediately and its result is passed to {@link #authenticationOptions(List<ClientVpnAuthentication>)}.
         * 
         * @param authenticationOptions
         *        a consumer that will call methods on {@link List<ClientVpnAuthentication>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #authenticationOptions(List<ClientVpnAuthentication>)
         */
        Builder authenticationOptions(Consumer<ClientVpnAuthentication.Builder>... authenticationOptions);

        /**
         * <p>
         * Information about the client connection logging options for the Client VPN endpoint.
         * </p>
         * 
         * @param connectionLogOptions
         *        Information about the client connection logging options for the Client VPN endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionLogOptions(ConnectionLogResponseOptions connectionLogOptions);

        /**
         * <p>
         * Information about the client connection logging options for the Client VPN endpoint.
         * </p>
         * This is a convenience that creates an instance of the {@link ConnectionLogResponseOptions.Builder} avoiding
         * the need to create one manually via {@link ConnectionLogResponseOptions#builder()}.
         *
         * When the {@link Consumer} completes, {@link ConnectionLogResponseOptions.Builder#build()} is called
         * immediately and its result is passed to {@link #connectionLogOptions(ConnectionLogResponseOptions)}.
         * 
         * @param connectionLogOptions
         *        a consumer that will call methods on {@link ConnectionLogResponseOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #connectionLogOptions(ConnectionLogResponseOptions)
         */
        default Builder connectionLogOptions(Consumer<ConnectionLogResponseOptions.Builder> connectionLogOptions) {
            return connectionLogOptions(ConnectionLogResponseOptions.builder().applyMutation(connectionLogOptions).build());
        }

        /**
         * <p>
         * Any tags assigned to the Client VPN endpoint.
         * </p>
         * 
         * @param tags
         *        Any tags assigned to the Client VPN endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Any tags assigned to the Client VPN endpoint.
         * </p>
         * 
         * @param tags
         *        Any tags assigned to the Client VPN endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Any tags assigned to the Client VPN endpoint.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String clientVpnEndpointId;

        private String description;

        private ClientVpnEndpointStatus status;

        private String creationTime;

        private String deletionTime;

        private String dnsName;

        private String clientCidrBlock;

        private List<String> dnsServers = DefaultSdkAutoConstructList.getInstance();

        private Boolean splitTunnel;

        private String vpnProtocol;

        private String transportProtocol;

        private List<AssociatedTargetNetwork> associatedTargetNetworks = DefaultSdkAutoConstructList.getInstance();

        private String serverCertificateArn;

        private List<ClientVpnAuthentication> authenticationOptions = DefaultSdkAutoConstructList.getInstance();

        private ConnectionLogResponseOptions connectionLogOptions;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ClientVpnEndpoint model) {
            clientVpnEndpointId(model.clientVpnEndpointId);
            description(model.description);
            status(model.status);
            creationTime(model.creationTime);
            deletionTime(model.deletionTime);
            dnsName(model.dnsName);
            clientCidrBlock(model.clientCidrBlock);
            dnsServers(model.dnsServers);
            splitTunnel(model.splitTunnel);
            vpnProtocol(model.vpnProtocol);
            transportProtocol(model.transportProtocol);
            associatedTargetNetworks(model.associatedTargetNetworks);
            serverCertificateArn(model.serverCertificateArn);
            authenticationOptions(model.authenticationOptions);
            connectionLogOptions(model.connectionLogOptions);
            tags(model.tags);
        }

        public final String getClientVpnEndpointId() {
            return clientVpnEndpointId;
        }

        @Override
        public final Builder clientVpnEndpointId(String clientVpnEndpointId) {
            this.clientVpnEndpointId = clientVpnEndpointId;
            return this;
        }

        public final void setClientVpnEndpointId(String clientVpnEndpointId) {
            this.clientVpnEndpointId = clientVpnEndpointId;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final ClientVpnEndpointStatus.Builder getStatus() {
            return status != null ? status.toBuilder() : null;
        }

        @Override
        public final Builder status(ClientVpnEndpointStatus status) {
            this.status = status;
            return this;
        }

        public final void setStatus(ClientVpnEndpointStatus.BuilderImpl status) {
            this.status = status != null ? status.build() : null;
        }

        public final String getCreationTime() {
            return creationTime;
        }

        @Override
        public final Builder creationTime(String creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final void setCreationTime(String creationTime) {
            this.creationTime = creationTime;
        }

        public final String getDeletionTime() {
            return deletionTime;
        }

        @Override
        public final Builder deletionTime(String deletionTime) {
            this.deletionTime = deletionTime;
            return this;
        }

        public final void setDeletionTime(String deletionTime) {
            this.deletionTime = deletionTime;
        }

        public final String getDnsName() {
            return dnsName;
        }

        @Override
        public final Builder dnsName(String dnsName) {
            this.dnsName = dnsName;
            return this;
        }

        public final void setDnsName(String dnsName) {
            this.dnsName = dnsName;
        }

        public final String getClientCidrBlock() {
            return clientCidrBlock;
        }

        @Override
        public final Builder clientCidrBlock(String clientCidrBlock) {
            this.clientCidrBlock = clientCidrBlock;
            return this;
        }

        public final void setClientCidrBlock(String clientCidrBlock) {
            this.clientCidrBlock = clientCidrBlock;
        }

        public final Collection<String> getDnsServers() {
            return dnsServers;
        }

        @Override
        public final Builder dnsServers(Collection<String> dnsServers) {
            this.dnsServers = ValueStringListCopier.copy(dnsServers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dnsServers(String... dnsServers) {
            dnsServers(Arrays.asList(dnsServers));
            return this;
        }

        public final void setDnsServers(Collection<String> dnsServers) {
            this.dnsServers = ValueStringListCopier.copy(dnsServers);
        }

        public final Boolean getSplitTunnel() {
            return splitTunnel;
        }

        @Override
        public final Builder splitTunnel(Boolean splitTunnel) {
            this.splitTunnel = splitTunnel;
            return this;
        }

        public final void setSplitTunnel(Boolean splitTunnel) {
            this.splitTunnel = splitTunnel;
        }

        public final String getVpnProtocolAsString() {
            return vpnProtocol;
        }

        @Override
        public final Builder vpnProtocol(String vpnProtocol) {
            this.vpnProtocol = vpnProtocol;
            return this;
        }

        @Override
        public final Builder vpnProtocol(VpnProtocol vpnProtocol) {
            this.vpnProtocol(vpnProtocol == null ? null : vpnProtocol.toString());
            return this;
        }

        public final void setVpnProtocol(String vpnProtocol) {
            this.vpnProtocol = vpnProtocol;
        }

        public final String getTransportProtocolAsString() {
            return transportProtocol;
        }

        @Override
        public final Builder transportProtocol(String transportProtocol) {
            this.transportProtocol = transportProtocol;
            return this;
        }

        @Override
        public final Builder transportProtocol(TransportProtocol transportProtocol) {
            this.transportProtocol(transportProtocol == null ? null : transportProtocol.toString());
            return this;
        }

        public final void setTransportProtocol(String transportProtocol) {
            this.transportProtocol = transportProtocol;
        }

        public final Collection<AssociatedTargetNetwork.Builder> getAssociatedTargetNetworks() {
            return associatedTargetNetworks != null ? associatedTargetNetworks.stream().map(AssociatedTargetNetwork::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder associatedTargetNetworks(Collection<AssociatedTargetNetwork> associatedTargetNetworks) {
            this.associatedTargetNetworks = AssociatedTargetNetworkSetCopier.copy(associatedTargetNetworks);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder associatedTargetNetworks(AssociatedTargetNetwork... associatedTargetNetworks) {
            associatedTargetNetworks(Arrays.asList(associatedTargetNetworks));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder associatedTargetNetworks(Consumer<AssociatedTargetNetwork.Builder>... associatedTargetNetworks) {
            associatedTargetNetworks(Stream.of(associatedTargetNetworks)
                    .map(c -> AssociatedTargetNetwork.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setAssociatedTargetNetworks(Collection<AssociatedTargetNetwork.BuilderImpl> associatedTargetNetworks) {
            this.associatedTargetNetworks = AssociatedTargetNetworkSetCopier.copyFromBuilder(associatedTargetNetworks);
        }

        public final String getServerCertificateArn() {
            return serverCertificateArn;
        }

        @Override
        public final Builder serverCertificateArn(String serverCertificateArn) {
            this.serverCertificateArn = serverCertificateArn;
            return this;
        }

        public final void setServerCertificateArn(String serverCertificateArn) {
            this.serverCertificateArn = serverCertificateArn;
        }

        public final Collection<ClientVpnAuthentication.Builder> getAuthenticationOptions() {
            return authenticationOptions != null ? authenticationOptions.stream().map(ClientVpnAuthentication::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder authenticationOptions(Collection<ClientVpnAuthentication> authenticationOptions) {
            this.authenticationOptions = ClientVpnAuthenticationListCopier.copy(authenticationOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder authenticationOptions(ClientVpnAuthentication... authenticationOptions) {
            authenticationOptions(Arrays.asList(authenticationOptions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder authenticationOptions(Consumer<ClientVpnAuthentication.Builder>... authenticationOptions) {
            authenticationOptions(Stream.of(authenticationOptions)
                    .map(c -> ClientVpnAuthentication.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setAuthenticationOptions(Collection<ClientVpnAuthentication.BuilderImpl> authenticationOptions) {
            this.authenticationOptions = ClientVpnAuthenticationListCopier.copyFromBuilder(authenticationOptions);
        }

        public final ConnectionLogResponseOptions.Builder getConnectionLogOptions() {
            return connectionLogOptions != null ? connectionLogOptions.toBuilder() : null;
        }

        @Override
        public final Builder connectionLogOptions(ConnectionLogResponseOptions connectionLogOptions) {
            this.connectionLogOptions = connectionLogOptions;
            return this;
        }

        public final void setConnectionLogOptions(ConnectionLogResponseOptions.BuilderImpl connectionLogOptions) {
            this.connectionLogOptions = connectionLogOptions != null ? connectionLogOptions.build() : null;
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public ClientVpnEndpoint build() {
            return new ClientVpnEndpoint(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
