/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the parameters for CreateSnapshot.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateSnapshotRequest extends Ec2Request implements
        ToCopyableBuilder<CreateSnapshotRequest.Builder, CreateSnapshotRequest> {
    private static final SdkField<String> DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(CreateSnapshotRequest::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description")
                    .unmarshallLocationName("Description").build()).build();

    private static final SdkField<String> VOLUME_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(CreateSnapshotRequest::volumeId))
            .setter(setter(Builder::volumeId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VolumeId")
                    .unmarshallLocationName("VolumeId").build()).build();

    private static final SdkField<List<TagSpecification>> TAG_SPECIFICATIONS_FIELD = SdkField
            .<List<TagSpecification>> builder(MarshallingType.LIST)
            .getter(getter(CreateSnapshotRequest::tagSpecifications))
            .setter(setter(Builder::tagSpecifications))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSpecification")
                    .unmarshallLocationName("TagSpecification").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<TagSpecification> builder(MarshallingType.SDK_POJO)
                                            .constructor(TagSpecification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DESCRIPTION_FIELD,
            VOLUME_ID_FIELD, TAG_SPECIFICATIONS_FIELD));

    private final String description;

    private final String volumeId;

    private final List<TagSpecification> tagSpecifications;

    private CreateSnapshotRequest(BuilderImpl builder) {
        super(builder);
        this.description = builder.description;
        this.volumeId = builder.volumeId;
        this.tagSpecifications = builder.tagSpecifications;
    }

    /**
     * <p>
     * A description for the snapshot.
     * </p>
     * 
     * @return A description for the snapshot.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The ID of the EBS volume.
     * </p>
     * 
     * @return The ID of the EBS volume.
     */
    public String volumeId() {
        return volumeId;
    }

    /**
     * <p>
     * The tags to apply to the snapshot during creation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The tags to apply to the snapshot during creation.
     */
    public List<TagSpecification> tagSpecifications() {
        return tagSpecifications;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(volumeId());
        hashCode = 31 * hashCode + Objects.hashCode(tagSpecifications());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateSnapshotRequest)) {
            return false;
        }
        CreateSnapshotRequest other = (CreateSnapshotRequest) obj;
        return Objects.equals(description(), other.description()) && Objects.equals(volumeId(), other.volumeId())
                && Objects.equals(tagSpecifications(), other.tagSpecifications());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CreateSnapshotRequest").add("Description", description()).add("VolumeId", volumeId())
                .add("TagSpecifications", tagSpecifications()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "VolumeId":
            return Optional.ofNullable(clazz.cast(volumeId()));
        case "TagSpecifications":
            return Optional.ofNullable(clazz.cast(tagSpecifications()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateSnapshotRequest, T> g) {
        return obj -> g.apply((CreateSnapshotRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends Ec2Request.Builder, SdkPojo, CopyableBuilder<Builder, CreateSnapshotRequest> {
        /**
         * <p>
         * A description for the snapshot.
         * </p>
         * 
         * @param description
         *        A description for the snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The ID of the EBS volume.
         * </p>
         * 
         * @param volumeId
         *        The ID of the EBS volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeId(String volumeId);

        /**
         * <p>
         * The tags to apply to the snapshot during creation.
         * </p>
         * 
         * @param tagSpecifications
         *        The tags to apply to the snapshot during creation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(Collection<TagSpecification> tagSpecifications);

        /**
         * <p>
         * The tags to apply to the snapshot during creation.
         * </p>
         * 
         * @param tagSpecifications
         *        The tags to apply to the snapshot during creation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(TagSpecification... tagSpecifications);

        /**
         * <p>
         * The tags to apply to the snapshot during creation.
         * </p>
         * This is a convenience that creates an instance of the {@link List<TagSpecification>.Builder} avoiding the
         * need to create one manually via {@link List<TagSpecification>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<TagSpecification>.Builder#build()} is called immediately and
         * its result is passed to {@link #tagSpecifications(List<TagSpecification>)}.
         * 
         * @param tagSpecifications
         *        a consumer that will call methods on {@link List<TagSpecification>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tagSpecifications(List<TagSpecification>)
         */
        Builder tagSpecifications(Consumer<TagSpecification.Builder>... tagSpecifications);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends Ec2Request.BuilderImpl implements Builder {
        private String description;

        private String volumeId;

        private List<TagSpecification> tagSpecifications = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateSnapshotRequest model) {
            super(model);
            description(model.description);
            volumeId(model.volumeId);
            tagSpecifications(model.tagSpecifications);
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getVolumeId() {
            return volumeId;
        }

        @Override
        public final Builder volumeId(String volumeId) {
            this.volumeId = volumeId;
            return this;
        }

        public final void setVolumeId(String volumeId) {
            this.volumeId = volumeId;
        }

        public final Collection<TagSpecification.Builder> getTagSpecifications() {
            return tagSpecifications != null ? tagSpecifications.stream().map(TagSpecification::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tagSpecifications(Collection<TagSpecification> tagSpecifications) {
            this.tagSpecifications = TagSpecificationListCopier.copy(tagSpecifications);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSpecifications(TagSpecification... tagSpecifications) {
            tagSpecifications(Arrays.asList(tagSpecifications));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSpecifications(Consumer<TagSpecification.Builder>... tagSpecifications) {
            tagSpecifications(Stream.of(tagSpecifications).map(c -> TagSpecification.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setTagSpecifications(Collection<TagSpecification.BuilderImpl> tagSpecifications) {
            this.tagSpecifications = TagSpecificationListCopier.copyFromBuilder(tagSpecifications);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateSnapshotRequest build() {
            return new CreateSnapshotRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
