/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecr.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about an image scan finding.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ImageScanFinding implements SdkPojo, Serializable,
        ToCopyableBuilder<ImageScanFinding.Builder, ImageScanFinding> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(ImageScanFinding::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(ImageScanFinding::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<String> URI_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("uri")
            .getter(getter(ImageScanFinding::uri)).setter(setter(Builder::uri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("uri").build()).build();

    private static final SdkField<String> SEVERITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("severity").getter(getter(ImageScanFinding::severityAsString)).setter(setter(Builder::severity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("severity").build()).build();

    private static final SdkField<List<Attribute>> ATTRIBUTES_FIELD = SdkField
            .<List<Attribute>> builder(MarshallingType.LIST)
            .memberName("attributes")
            .getter(getter(ImageScanFinding::attributes))
            .setter(setter(Builder::attributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("attributes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Attribute> builder(MarshallingType.SDK_POJO)
                                            .constructor(Attribute::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, DESCRIPTION_FIELD,
            URI_FIELD, SEVERITY_FIELD, ATTRIBUTES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String description;

    private final String uri;

    private final String severity;

    private final List<Attribute> attributes;

    private ImageScanFinding(BuilderImpl builder) {
        this.name = builder.name;
        this.description = builder.description;
        this.uri = builder.uri;
        this.severity = builder.severity;
        this.attributes = builder.attributes;
    }

    /**
     * <p>
     * The name associated with the finding, usually a CVE number.
     * </p>
     * 
     * @return The name associated with the finding, usually a CVE number.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The description of the finding.
     * </p>
     * 
     * @return The description of the finding.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * A link containing additional details about the security vulnerability.
     * </p>
     * 
     * @return A link containing additional details about the security vulnerability.
     */
    public final String uri() {
        return uri;
    }

    /**
     * <p>
     * The finding severity.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #severity} will
     * return {@link FindingSeverity#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #severityAsString}.
     * </p>
     * 
     * @return The finding severity.
     * @see FindingSeverity
     */
    public final FindingSeverity severity() {
        return FindingSeverity.fromValue(severity);
    }

    /**
     * <p>
     * The finding severity.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #severity} will
     * return {@link FindingSeverity#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #severityAsString}.
     * </p>
     * 
     * @return The finding severity.
     * @see FindingSeverity
     */
    public final String severityAsString() {
        return severity;
    }

    /**
     * For responses, this returns true if the service returned a value for the Attributes property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAttributes() {
        return attributes != null && !(attributes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A collection of attributes of the host from which the finding is generated.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttributes} method.
     * </p>
     * 
     * @return A collection of attributes of the host from which the finding is generated.
     */
    public final List<Attribute> attributes() {
        return attributes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(uri());
        hashCode = 31 * hashCode + Objects.hashCode(severityAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasAttributes() ? attributes() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ImageScanFinding)) {
            return false;
        }
        ImageScanFinding other = (ImageScanFinding) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(description(), other.description())
                && Objects.equals(uri(), other.uri()) && Objects.equals(severityAsString(), other.severityAsString())
                && hasAttributes() == other.hasAttributes() && Objects.equals(attributes(), other.attributes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ImageScanFinding").add("Name", name()).add("Description", description()).add("Uri", uri())
                .add("Severity", severityAsString()).add("Attributes", hasAttributes() ? attributes() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "uri":
            return Optional.ofNullable(clazz.cast(uri()));
        case "severity":
            return Optional.ofNullable(clazz.cast(severityAsString()));
        case "attributes":
            return Optional.ofNullable(clazz.cast(attributes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ImageScanFinding, T> g) {
        return obj -> g.apply((ImageScanFinding) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ImageScanFinding> {
        /**
         * <p>
         * The name associated with the finding, usually a CVE number.
         * </p>
         * 
         * @param name
         *        The name associated with the finding, usually a CVE number.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The description of the finding.
         * </p>
         * 
         * @param description
         *        The description of the finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A link containing additional details about the security vulnerability.
         * </p>
         * 
         * @param uri
         *        A link containing additional details about the security vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder uri(String uri);

        /**
         * <p>
         * The finding severity.
         * </p>
         * 
         * @param severity
         *        The finding severity.
         * @see FindingSeverity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FindingSeverity
         */
        Builder severity(String severity);

        /**
         * <p>
         * The finding severity.
         * </p>
         * 
         * @param severity
         *        The finding severity.
         * @see FindingSeverity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FindingSeverity
         */
        Builder severity(FindingSeverity severity);

        /**
         * <p>
         * A collection of attributes of the host from which the finding is generated.
         * </p>
         * 
         * @param attributes
         *        A collection of attributes of the host from which the finding is generated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(Collection<Attribute> attributes);

        /**
         * <p>
         * A collection of attributes of the host from which the finding is generated.
         * </p>
         * 
         * @param attributes
         *        A collection of attributes of the host from which the finding is generated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(Attribute... attributes);

        /**
         * <p>
         * A collection of attributes of the host from which the finding is generated.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Attribute>.Builder} avoiding the need to
         * create one manually via {@link List<Attribute>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Attribute>.Builder#build()} is called immediately and its
         * result is passed to {@link #attributes(List<Attribute>)}.
         * 
         * @param attributes
         *        a consumer that will call methods on {@link List<Attribute>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #attributes(List<Attribute>)
         */
        Builder attributes(Consumer<Attribute.Builder>... attributes);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String description;

        private String uri;

        private String severity;

        private List<Attribute> attributes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ImageScanFinding model) {
            name(model.name);
            description(model.description);
            uri(model.uri);
            severity(model.severity);
            attributes(model.attributes);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        @Transient
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getUri() {
            return uri;
        }

        public final void setUri(String uri) {
            this.uri = uri;
        }

        @Override
        @Transient
        public final Builder uri(String uri) {
            this.uri = uri;
            return this;
        }

        public final String getSeverity() {
            return severity;
        }

        public final void setSeverity(String severity) {
            this.severity = severity;
        }

        @Override
        @Transient
        public final Builder severity(String severity) {
            this.severity = severity;
            return this;
        }

        @Override
        @Transient
        public final Builder severity(FindingSeverity severity) {
            this.severity(severity == null ? null : severity.toString());
            return this;
        }

        public final List<Attribute.Builder> getAttributes() {
            List<Attribute.Builder> result = AttributeListCopier.copyToBuilder(this.attributes);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAttributes(Collection<Attribute.BuilderImpl> attributes) {
            this.attributes = AttributeListCopier.copyFromBuilder(attributes);
        }

        @Override
        @Transient
        public final Builder attributes(Collection<Attribute> attributes) {
            this.attributes = AttributeListCopier.copy(attributes);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder attributes(Attribute... attributes) {
            attributes(Arrays.asList(attributes));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder attributes(Consumer<Attribute.Builder>... attributes) {
            attributes(Stream.of(attributes).map(c -> Attribute.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public ImageScanFinding build() {
            return new ImageScanFinding(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
