/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecr.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.ecr.EcrClient;
import software.amazon.awssdk.services.ecr.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.ecr.model.DescribeImageScanFindingsRequest;
import software.amazon.awssdk.services.ecr.model.DescribeImageScanFindingsResponse;
import software.amazon.awssdk.services.ecr.model.EcrRequest;
import software.amazon.awssdk.services.ecr.model.GetLifecyclePolicyPreviewRequest;
import software.amazon.awssdk.services.ecr.model.GetLifecyclePolicyPreviewResponse;
import software.amazon.awssdk.services.ecr.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultEcrWaiter implements EcrWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final EcrClient client;

    private final AttributeMap managedResources;

    private final Waiter<DescribeImageScanFindingsResponse> imageScanCompleteWaiter;

    private final Waiter<GetLifecyclePolicyPreviewResponse> lifecyclePolicyPreviewCompleteWaiter;

    private DefaultEcrWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = EcrClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.imageScanCompleteWaiter = Waiter.builder(DescribeImageScanFindingsResponse.class)
                .acceptors(imageScanCompleteWaiterAcceptors())
                .overrideConfiguration(imageScanCompleteWaiterConfig(builder.overrideConfiguration)).build();
        this.lifecyclePolicyPreviewCompleteWaiter = Waiter.builder(GetLifecyclePolicyPreviewResponse.class)
                .acceptors(lifecyclePolicyPreviewCompleteWaiterAcceptors())
                .overrideConfiguration(lifecyclePolicyPreviewCompleteWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<DescribeImageScanFindingsResponse> waitUntilImageScanComplete(
            DescribeImageScanFindingsRequest describeImageScanFindingsRequest) {
        return imageScanCompleteWaiter.run(() -> client
                .describeImageScanFindings(applyWaitersUserAgent(describeImageScanFindingsRequest)));
    }

    @Override
    public WaiterResponse<DescribeImageScanFindingsResponse> waitUntilImageScanComplete(
            DescribeImageScanFindingsRequest describeImageScanFindingsRequest, WaiterOverrideConfiguration overrideConfig) {
        return imageScanCompleteWaiter.run(
                () -> client.describeImageScanFindings(applyWaitersUserAgent(describeImageScanFindingsRequest)),
                imageScanCompleteWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetLifecyclePolicyPreviewResponse> waitUntilLifecyclePolicyPreviewComplete(
            GetLifecyclePolicyPreviewRequest getLifecyclePolicyPreviewRequest) {
        return lifecyclePolicyPreviewCompleteWaiter.run(() -> client
                .getLifecyclePolicyPreview(applyWaitersUserAgent(getLifecyclePolicyPreviewRequest)));
    }

    @Override
    public WaiterResponse<GetLifecyclePolicyPreviewResponse> waitUntilLifecyclePolicyPreviewComplete(
            GetLifecyclePolicyPreviewRequest getLifecyclePolicyPreviewRequest, WaiterOverrideConfiguration overrideConfig) {
        return lifecyclePolicyPreviewCompleteWaiter.run(
                () -> client.getLifecyclePolicyPreview(applyWaitersUserAgent(getLifecyclePolicyPreviewRequest)),
                lifecyclePolicyPreviewCompleteWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeImageScanFindingsResponse>> imageScanCompleteWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeImageScanFindingsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("imageScanStatus").field("status").value(), "COMPLETE");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("imageScanStatus").field("status").value(), "FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (imageScanStatus.status=FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetLifecyclePolicyPreviewResponse>> lifecyclePolicyPreviewCompleteWaiterAcceptors() {
        List<WaiterAcceptor<? super GetLifecyclePolicyPreviewResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "COMPLETE");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "FAILED");
        },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration imageScanCompleteWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration lifecyclePolicyPreviewCompleteWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static EcrWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends EcrRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements EcrWaiter.Builder {
        private EcrClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public EcrWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public EcrWaiter.Builder client(EcrClient client) {
            this.client = client;
            return this;
        }

        public EcrWaiter build() {
            return new DefaultEcrWaiter(this);
        }
    }
}
