/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecr;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.ecr.internal.EcrServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.ecr.model.BatchCheckLayerAvailabilityRequest;
import software.amazon.awssdk.services.ecr.model.BatchCheckLayerAvailabilityResponse;
import software.amazon.awssdk.services.ecr.model.BatchDeleteImageRequest;
import software.amazon.awssdk.services.ecr.model.BatchDeleteImageResponse;
import software.amazon.awssdk.services.ecr.model.BatchGetImageRequest;
import software.amazon.awssdk.services.ecr.model.BatchGetImageResponse;
import software.amazon.awssdk.services.ecr.model.BatchGetRepositoryScanningConfigurationRequest;
import software.amazon.awssdk.services.ecr.model.BatchGetRepositoryScanningConfigurationResponse;
import software.amazon.awssdk.services.ecr.model.CompleteLayerUploadRequest;
import software.amazon.awssdk.services.ecr.model.CompleteLayerUploadResponse;
import software.amazon.awssdk.services.ecr.model.CreatePullThroughCacheRuleRequest;
import software.amazon.awssdk.services.ecr.model.CreatePullThroughCacheRuleResponse;
import software.amazon.awssdk.services.ecr.model.CreateRepositoryCreationTemplateRequest;
import software.amazon.awssdk.services.ecr.model.CreateRepositoryCreationTemplateResponse;
import software.amazon.awssdk.services.ecr.model.CreateRepositoryRequest;
import software.amazon.awssdk.services.ecr.model.CreateRepositoryResponse;
import software.amazon.awssdk.services.ecr.model.DeleteLifecyclePolicyRequest;
import software.amazon.awssdk.services.ecr.model.DeleteLifecyclePolicyResponse;
import software.amazon.awssdk.services.ecr.model.DeletePullThroughCacheRuleRequest;
import software.amazon.awssdk.services.ecr.model.DeletePullThroughCacheRuleResponse;
import software.amazon.awssdk.services.ecr.model.DeleteRegistryPolicyRequest;
import software.amazon.awssdk.services.ecr.model.DeleteRegistryPolicyResponse;
import software.amazon.awssdk.services.ecr.model.DeleteRepositoryCreationTemplateRequest;
import software.amazon.awssdk.services.ecr.model.DeleteRepositoryCreationTemplateResponse;
import software.amazon.awssdk.services.ecr.model.DeleteRepositoryPolicyRequest;
import software.amazon.awssdk.services.ecr.model.DeleteRepositoryPolicyResponse;
import software.amazon.awssdk.services.ecr.model.DeleteRepositoryRequest;
import software.amazon.awssdk.services.ecr.model.DeleteRepositoryResponse;
import software.amazon.awssdk.services.ecr.model.DescribeImageReplicationStatusRequest;
import software.amazon.awssdk.services.ecr.model.DescribeImageReplicationStatusResponse;
import software.amazon.awssdk.services.ecr.model.DescribeImageScanFindingsRequest;
import software.amazon.awssdk.services.ecr.model.DescribeImageScanFindingsResponse;
import software.amazon.awssdk.services.ecr.model.DescribeImagesRequest;
import software.amazon.awssdk.services.ecr.model.DescribeImagesResponse;
import software.amazon.awssdk.services.ecr.model.DescribePullThroughCacheRulesRequest;
import software.amazon.awssdk.services.ecr.model.DescribePullThroughCacheRulesResponse;
import software.amazon.awssdk.services.ecr.model.DescribeRegistryRequest;
import software.amazon.awssdk.services.ecr.model.DescribeRegistryResponse;
import software.amazon.awssdk.services.ecr.model.DescribeRepositoriesRequest;
import software.amazon.awssdk.services.ecr.model.DescribeRepositoriesResponse;
import software.amazon.awssdk.services.ecr.model.DescribeRepositoryCreationTemplatesRequest;
import software.amazon.awssdk.services.ecr.model.DescribeRepositoryCreationTemplatesResponse;
import software.amazon.awssdk.services.ecr.model.EcrException;
import software.amazon.awssdk.services.ecr.model.EmptyUploadException;
import software.amazon.awssdk.services.ecr.model.GetAccountSettingRequest;
import software.amazon.awssdk.services.ecr.model.GetAccountSettingResponse;
import software.amazon.awssdk.services.ecr.model.GetAuthorizationTokenRequest;
import software.amazon.awssdk.services.ecr.model.GetAuthorizationTokenResponse;
import software.amazon.awssdk.services.ecr.model.GetDownloadUrlForLayerRequest;
import software.amazon.awssdk.services.ecr.model.GetDownloadUrlForLayerResponse;
import software.amazon.awssdk.services.ecr.model.GetLifecyclePolicyPreviewRequest;
import software.amazon.awssdk.services.ecr.model.GetLifecyclePolicyPreviewResponse;
import software.amazon.awssdk.services.ecr.model.GetLifecyclePolicyRequest;
import software.amazon.awssdk.services.ecr.model.GetLifecyclePolicyResponse;
import software.amazon.awssdk.services.ecr.model.GetRegistryPolicyRequest;
import software.amazon.awssdk.services.ecr.model.GetRegistryPolicyResponse;
import software.amazon.awssdk.services.ecr.model.GetRegistryScanningConfigurationRequest;
import software.amazon.awssdk.services.ecr.model.GetRegistryScanningConfigurationResponse;
import software.amazon.awssdk.services.ecr.model.GetRepositoryPolicyRequest;
import software.amazon.awssdk.services.ecr.model.GetRepositoryPolicyResponse;
import software.amazon.awssdk.services.ecr.model.ImageAlreadyExistsException;
import software.amazon.awssdk.services.ecr.model.ImageDigestDoesNotMatchException;
import software.amazon.awssdk.services.ecr.model.ImageNotFoundException;
import software.amazon.awssdk.services.ecr.model.ImageTagAlreadyExistsException;
import software.amazon.awssdk.services.ecr.model.InitiateLayerUploadRequest;
import software.amazon.awssdk.services.ecr.model.InitiateLayerUploadResponse;
import software.amazon.awssdk.services.ecr.model.InvalidLayerException;
import software.amazon.awssdk.services.ecr.model.InvalidLayerPartException;
import software.amazon.awssdk.services.ecr.model.InvalidParameterException;
import software.amazon.awssdk.services.ecr.model.InvalidTagParameterException;
import software.amazon.awssdk.services.ecr.model.KmsException;
import software.amazon.awssdk.services.ecr.model.LayerAlreadyExistsException;
import software.amazon.awssdk.services.ecr.model.LayerInaccessibleException;
import software.amazon.awssdk.services.ecr.model.LayerPartTooSmallException;
import software.amazon.awssdk.services.ecr.model.LayersNotFoundException;
import software.amazon.awssdk.services.ecr.model.LifecyclePolicyNotFoundException;
import software.amazon.awssdk.services.ecr.model.LifecyclePolicyPreviewInProgressException;
import software.amazon.awssdk.services.ecr.model.LifecyclePolicyPreviewNotFoundException;
import software.amazon.awssdk.services.ecr.model.LimitExceededException;
import software.amazon.awssdk.services.ecr.model.ListImagesRequest;
import software.amazon.awssdk.services.ecr.model.ListImagesResponse;
import software.amazon.awssdk.services.ecr.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.ecr.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.ecr.model.PullThroughCacheRuleAlreadyExistsException;
import software.amazon.awssdk.services.ecr.model.PullThroughCacheRuleNotFoundException;
import software.amazon.awssdk.services.ecr.model.PutAccountSettingRequest;
import software.amazon.awssdk.services.ecr.model.PutAccountSettingResponse;
import software.amazon.awssdk.services.ecr.model.PutImageRequest;
import software.amazon.awssdk.services.ecr.model.PutImageResponse;
import software.amazon.awssdk.services.ecr.model.PutImageScanningConfigurationRequest;
import software.amazon.awssdk.services.ecr.model.PutImageScanningConfigurationResponse;
import software.amazon.awssdk.services.ecr.model.PutImageTagMutabilityRequest;
import software.amazon.awssdk.services.ecr.model.PutImageTagMutabilityResponse;
import software.amazon.awssdk.services.ecr.model.PutLifecyclePolicyRequest;
import software.amazon.awssdk.services.ecr.model.PutLifecyclePolicyResponse;
import software.amazon.awssdk.services.ecr.model.PutRegistryPolicyRequest;
import software.amazon.awssdk.services.ecr.model.PutRegistryPolicyResponse;
import software.amazon.awssdk.services.ecr.model.PutRegistryScanningConfigurationRequest;
import software.amazon.awssdk.services.ecr.model.PutRegistryScanningConfigurationResponse;
import software.amazon.awssdk.services.ecr.model.PutReplicationConfigurationRequest;
import software.amazon.awssdk.services.ecr.model.PutReplicationConfigurationResponse;
import software.amazon.awssdk.services.ecr.model.ReferencedImagesNotFoundException;
import software.amazon.awssdk.services.ecr.model.RegistryPolicyNotFoundException;
import software.amazon.awssdk.services.ecr.model.RepositoryAlreadyExistsException;
import software.amazon.awssdk.services.ecr.model.RepositoryNotEmptyException;
import software.amazon.awssdk.services.ecr.model.RepositoryNotFoundException;
import software.amazon.awssdk.services.ecr.model.RepositoryPolicyNotFoundException;
import software.amazon.awssdk.services.ecr.model.ScanNotFoundException;
import software.amazon.awssdk.services.ecr.model.SecretNotFoundException;
import software.amazon.awssdk.services.ecr.model.ServerException;
import software.amazon.awssdk.services.ecr.model.SetRepositoryPolicyRequest;
import software.amazon.awssdk.services.ecr.model.SetRepositoryPolicyResponse;
import software.amazon.awssdk.services.ecr.model.StartImageScanRequest;
import software.amazon.awssdk.services.ecr.model.StartImageScanResponse;
import software.amazon.awssdk.services.ecr.model.StartLifecyclePolicyPreviewRequest;
import software.amazon.awssdk.services.ecr.model.StartLifecyclePolicyPreviewResponse;
import software.amazon.awssdk.services.ecr.model.TagResourceRequest;
import software.amazon.awssdk.services.ecr.model.TagResourceResponse;
import software.amazon.awssdk.services.ecr.model.TemplateAlreadyExistsException;
import software.amazon.awssdk.services.ecr.model.TemplateNotFoundException;
import software.amazon.awssdk.services.ecr.model.TooManyTagsException;
import software.amazon.awssdk.services.ecr.model.UnableToAccessSecretException;
import software.amazon.awssdk.services.ecr.model.UnableToDecryptSecretValueException;
import software.amazon.awssdk.services.ecr.model.UnableToGetUpstreamImageException;
import software.amazon.awssdk.services.ecr.model.UnableToGetUpstreamLayerException;
import software.amazon.awssdk.services.ecr.model.UnsupportedImageTypeException;
import software.amazon.awssdk.services.ecr.model.UnsupportedUpstreamRegistryException;
import software.amazon.awssdk.services.ecr.model.UntagResourceRequest;
import software.amazon.awssdk.services.ecr.model.UntagResourceResponse;
import software.amazon.awssdk.services.ecr.model.UpdatePullThroughCacheRuleRequest;
import software.amazon.awssdk.services.ecr.model.UpdatePullThroughCacheRuleResponse;
import software.amazon.awssdk.services.ecr.model.UpdateRepositoryCreationTemplateRequest;
import software.amazon.awssdk.services.ecr.model.UpdateRepositoryCreationTemplateResponse;
import software.amazon.awssdk.services.ecr.model.UploadLayerPartRequest;
import software.amazon.awssdk.services.ecr.model.UploadLayerPartResponse;
import software.amazon.awssdk.services.ecr.model.UploadNotFoundException;
import software.amazon.awssdk.services.ecr.model.ValidatePullThroughCacheRuleRequest;
import software.amazon.awssdk.services.ecr.model.ValidatePullThroughCacheRuleResponse;
import software.amazon.awssdk.services.ecr.model.ValidationException;
import software.amazon.awssdk.services.ecr.transform.BatchCheckLayerAvailabilityRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.BatchDeleteImageRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.BatchGetImageRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.BatchGetRepositoryScanningConfigurationRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.CompleteLayerUploadRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.CreatePullThroughCacheRuleRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.CreateRepositoryCreationTemplateRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.CreateRepositoryRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.DeleteLifecyclePolicyRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.DeletePullThroughCacheRuleRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.DeleteRegistryPolicyRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.DeleteRepositoryCreationTemplateRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.DeleteRepositoryPolicyRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.DeleteRepositoryRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.DescribeImageReplicationStatusRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.DescribeImageScanFindingsRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.DescribeImagesRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.DescribePullThroughCacheRulesRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.DescribeRegistryRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.DescribeRepositoriesRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.DescribeRepositoryCreationTemplatesRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.GetAccountSettingRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.GetAuthorizationTokenRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.GetDownloadUrlForLayerRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.GetLifecyclePolicyPreviewRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.GetLifecyclePolicyRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.GetRegistryPolicyRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.GetRegistryScanningConfigurationRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.GetRepositoryPolicyRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.InitiateLayerUploadRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.ListImagesRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.PutAccountSettingRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.PutImageRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.PutImageScanningConfigurationRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.PutImageTagMutabilityRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.PutLifecyclePolicyRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.PutRegistryPolicyRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.PutRegistryScanningConfigurationRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.PutReplicationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.SetRepositoryPolicyRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.StartImageScanRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.StartLifecyclePolicyPreviewRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.UpdatePullThroughCacheRuleRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.UpdateRepositoryCreationTemplateRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.UploadLayerPartRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.ValidatePullThroughCacheRuleRequestMarshaller;
import software.amazon.awssdk.services.ecr.waiters.EcrAsyncWaiter;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link EcrAsyncClient}.
 *
 * @see EcrAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultEcrAsyncClient implements EcrAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultEcrAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final ScheduledExecutorService executorService;

    protected DefaultEcrAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
        this.executorService = clientConfiguration.option(SdkClientOption.SCHEDULED_EXECUTOR_SERVICE);
    }

    /**
     * <p>
     * Checks the availability of one or more image layers in a repository.
     * </p>
     * <p>
     * When an image is pushed to a repository, each image layer is checked to verify if it has been uploaded before. If
     * it has been uploaded, then the image layer is skipped.
     * </p>
     * <note>
     * <p>
     * This operation is used by the Amazon ECR proxy and is not generally used by customers for pulling and pushing
     * images. In most cases, you should use the <code>docker</code> CLI to pull, tag, and push images.
     * </p>
     * </note>
     *
     * @param batchCheckLayerAvailabilityRequest
     * @return A Java Future containing the result of the BatchCheckLayerAvailability operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.BatchCheckLayerAvailability
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/BatchCheckLayerAvailability"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchCheckLayerAvailabilityResponse> batchCheckLayerAvailability(
            BatchCheckLayerAvailabilityRequest batchCheckLayerAvailabilityRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchCheckLayerAvailabilityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchCheckLayerAvailabilityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchCheckLayerAvailability");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchCheckLayerAvailabilityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchCheckLayerAvailabilityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchCheckLayerAvailabilityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchCheckLayerAvailabilityRequest, BatchCheckLayerAvailabilityResponse>()
                            .withOperationName("BatchCheckLayerAvailability").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchCheckLayerAvailabilityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchCheckLayerAvailabilityRequest));
            CompletableFuture<BatchCheckLayerAvailabilityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a list of specified images within a repository. Images are specified with either an <code>imageTag</code>
     * or <code>imageDigest</code>.
     * </p>
     * <p>
     * You can remove a tag from an image by specifying the image's tag in your request. When you remove the last tag
     * from an image, the image is deleted from your repository.
     * </p>
     * <p>
     * You can completely delete an image (and all of its tags) by specifying the image's digest in your request.
     * </p>
     *
     * @param batchDeleteImageRequest
     *        Deletes specified images within a specified repository. Images are specified with either the
     *        <code>imageTag</code> or <code>imageDigest</code>.
     * @return A Java Future containing the result of the BatchDeleteImage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.BatchDeleteImage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/BatchDeleteImage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<BatchDeleteImageResponse> batchDeleteImage(BatchDeleteImageRequest batchDeleteImageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchDeleteImageRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchDeleteImageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDeleteImage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchDeleteImageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchDeleteImageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchDeleteImageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchDeleteImageRequest, BatchDeleteImageResponse>()
                            .withOperationName("BatchDeleteImage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchDeleteImageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchDeleteImageRequest));
            CompletableFuture<BatchDeleteImageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets detailed information for an image. Images are specified with either an <code>imageTag</code> or
     * <code>imageDigest</code>.
     * </p>
     * <p>
     * When an image is pulled, the BatchGetImage API is called once to retrieve the image manifest.
     * </p>
     *
     * @param batchGetImageRequest
     * @return A Java Future containing the result of the BatchGetImage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>LimitExceededException The operation did not succeed because it would have exceeded a service limit
     *         for your account. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/service-quotas.html">Amazon ECR service
     *         quotas</a> in the Amazon Elastic Container Registry User Guide.</li>
     *         <li>UnableToGetUpstreamImageException The image or images were unable to be pulled using the pull through
     *         cache rule. This is usually caused because of an issue with the Secrets Manager secret containing the
     *         credentials for the upstream registry.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.BatchGetImage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/BatchGetImage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetImageResponse> batchGetImage(BatchGetImageRequest batchGetImageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchGetImageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetImageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetImage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetImageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    BatchGetImageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchGetImageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetImageRequest, BatchGetImageResponse>()
                            .withOperationName("BatchGetImage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchGetImageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchGetImageRequest));
            CompletableFuture<BatchGetImageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the scanning configuration for one or more repositories.
     * </p>
     *
     * @param batchGetRepositoryScanningConfigurationRequest
     * @return A Java Future containing the result of the BatchGetRepositoryScanningConfiguration operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.BatchGetRepositoryScanningConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/BatchGetRepositoryScanningConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetRepositoryScanningConfigurationResponse> batchGetRepositoryScanningConfiguration(
            BatchGetRepositoryScanningConfigurationRequest batchGetRepositoryScanningConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchGetRepositoryScanningConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                batchGetRepositoryScanningConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetRepositoryScanningConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetRepositoryScanningConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, BatchGetRepositoryScanningConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchGetRepositoryScanningConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetRepositoryScanningConfigurationRequest, BatchGetRepositoryScanningConfigurationResponse>()
                            .withOperationName("BatchGetRepositoryScanningConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchGetRepositoryScanningConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchGetRepositoryScanningConfigurationRequest));
            CompletableFuture<BatchGetRepositoryScanningConfigurationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Informs Amazon ECR that the image layer upload has completed for a specified registry, repository name, and
     * upload ID. You can optionally provide a <code>sha256</code> digest of the image layer for data validation
     * purposes.
     * </p>
     * <p>
     * When an image is pushed, the CompleteLayerUpload API is called once per each new image layer to verify that the
     * upload has completed.
     * </p>
     * <note>
     * <p>
     * This operation is used by the Amazon ECR proxy and is not generally used by customers for pulling and pushing
     * images. In most cases, you should use the <code>docker</code> CLI to pull, tag, and push images.
     * </p>
     * </note>
     *
     * @param completeLayerUploadRequest
     * @return A Java Future containing the result of the CompleteLayerUpload operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>UploadNotFoundException The upload could not be found, or the specified upload ID is not valid for
     *         this repository.</li>
     *         <li>InvalidLayerException The layer digest calculation performed by Amazon ECR upon receipt of the image
     *         layer does not match the digest specified.</li>
     *         <li>LayerPartTooSmallException Layer parts must be at least 5 MiB in size.</li>
     *         <li>LayerAlreadyExistsException The image layer already exists in the associated repository.</li>
     *         <li>EmptyUploadException The specified layer upload does not contain any layer parts.</li>
     *         <li>KmsException The operation failed due to a KMS exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.CompleteLayerUpload
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/CompleteLayerUpload" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CompleteLayerUploadResponse> completeLayerUpload(
            CompleteLayerUploadRequest completeLayerUploadRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(completeLayerUploadRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, completeLayerUploadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CompleteLayerUpload");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CompleteLayerUploadResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CompleteLayerUploadResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CompleteLayerUploadResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CompleteLayerUploadRequest, CompleteLayerUploadResponse>()
                            .withOperationName("CompleteLayerUpload").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CompleteLayerUploadRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(completeLayerUploadRequest));
            CompletableFuture<CompleteLayerUploadResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a pull through cache rule. A pull through cache rule provides a way to cache images from an upstream
     * registry source in your Amazon ECR private registry. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/pull-through-cache.html">Using pull through cache
     * rules</a> in the <i>Amazon Elastic Container Registry User Guide</i>.
     * </p>
     *
     * @param createPullThroughCacheRuleRequest
     * @return A Java Future containing the result of the CreatePullThroughCacheRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>PullThroughCacheRuleAlreadyExistsException A pull through cache rule with these settings already
     *         exists for the private registry.</li>
     *         <li>UnsupportedUpstreamRegistryException The specified upstream registry isn't supported.</li>
     *         <li>LimitExceededException The operation did not succeed because it would have exceeded a service limit
     *         for your account. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/service-quotas.html">Amazon ECR service
     *         quotas</a> in the Amazon Elastic Container Registry User Guide.</li>
     *         <li>UnableToAccessSecretException The secret is unable to be accessed. Verify the resource permissions
     *         for the secret and try again.</li>
     *         <li>SecretNotFoundException The ARN of the secret specified in the pull through cache rule was not found.
     *         Update the pull through cache rule with a valid secret ARN and try again.</li>
     *         <li>UnableToDecryptSecretValueException The secret is accessible but is unable to be decrypted. Verify
     *         the resource permisisons and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.CreatePullThroughCacheRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/CreatePullThroughCacheRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePullThroughCacheRuleResponse> createPullThroughCacheRule(
            CreatePullThroughCacheRuleRequest createPullThroughCacheRuleRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPullThroughCacheRuleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPullThroughCacheRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePullThroughCacheRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePullThroughCacheRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePullThroughCacheRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePullThroughCacheRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePullThroughCacheRuleRequest, CreatePullThroughCacheRuleResponse>()
                            .withOperationName("CreatePullThroughCacheRule").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreatePullThroughCacheRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPullThroughCacheRuleRequest));
            CompletableFuture<CreatePullThroughCacheRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a repository. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/Repositories.html">Amazon ECR repositories</a> in
     * the <i>Amazon Elastic Container Registry User Guide</i>.
     * </p>
     *
     * @param createRepositoryRequest
     * @return A Java Future containing the result of the CreateRepository operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>InvalidTagParameterException An invalid parameter has been specified. Tag keys can have a maximum
     *         character length of 128 characters, and tag values can have a maximum length of 256 characters.</li>
     *         <li>TooManyTagsException The list of tags on the repository is over the limit. The maximum number of tags
     *         that can be applied to a repository is 50.</li>
     *         <li>RepositoryAlreadyExistsException The specified repository already exists in the specified registry.</li>
     *         <li>LimitExceededException The operation did not succeed because it would have exceeded a service limit
     *         for your account. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/service-quotas.html">Amazon ECR service
     *         quotas</a> in the Amazon Elastic Container Registry User Guide.</li>
     *         <li>KmsException The operation failed due to a KMS exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.CreateRepository
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/CreateRepository" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateRepositoryResponse> createRepository(CreateRepositoryRequest createRepositoryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRepositoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRepositoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRepository");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRepositoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateRepositoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateRepositoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRepositoryRequest, CreateRepositoryResponse>()
                            .withOperationName("CreateRepository").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateRepositoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createRepositoryRequest));
            CompletableFuture<CreateRepositoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a repository creation template. This template is used to define the settings for repositories created by
     * Amazon ECR on your behalf. For example, repositories created through pull through cache actions. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/repository-creation-templates.html">Private
     * repository creation templates</a> in the <i>Amazon Elastic Container Registry User Guide</i>.
     * </p>
     *
     * @param createRepositoryCreationTemplateRequest
     * @return A Java Future containing the result of the CreateRepositoryCreationTemplate operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>LimitExceededException The operation did not succeed because it would have exceeded a service limit
     *         for your account. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/service-quotas.html">Amazon ECR service
     *         quotas</a> in the Amazon Elastic Container Registry User Guide.</li>
     *         <li>TemplateAlreadyExistsException The repository creation template already exists. Specify a unique
     *         prefix and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.CreateRepositoryCreationTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/CreateRepositoryCreationTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateRepositoryCreationTemplateResponse> createRepositoryCreationTemplate(
            CreateRepositoryCreationTemplateRequest createRepositoryCreationTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRepositoryCreationTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createRepositoryCreationTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRepositoryCreationTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRepositoryCreationTemplateResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, CreateRepositoryCreationTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateRepositoryCreationTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRepositoryCreationTemplateRequest, CreateRepositoryCreationTemplateResponse>()
                            .withOperationName("CreateRepositoryCreationTemplate").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateRepositoryCreationTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createRepositoryCreationTemplateRequest));
            CompletableFuture<CreateRepositoryCreationTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the lifecycle policy associated with the specified repository.
     * </p>
     *
     * @param deleteLifecyclePolicyRequest
     * @return A Java Future containing the result of the DeleteLifecyclePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>LifecyclePolicyNotFoundException The lifecycle policy could not be found, and no policy is set to the
     *         repository.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.DeleteLifecyclePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/DeleteLifecyclePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLifecyclePolicyResponse> deleteLifecyclePolicy(
            DeleteLifecyclePolicyRequest deleteLifecyclePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteLifecyclePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLifecyclePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLifecyclePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLifecyclePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteLifecyclePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLifecyclePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLifecyclePolicyRequest, DeleteLifecyclePolicyResponse>()
                            .withOperationName("DeleteLifecyclePolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteLifecyclePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteLifecyclePolicyRequest));
            CompletableFuture<DeleteLifecyclePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a pull through cache rule.
     * </p>
     *
     * @param deletePullThroughCacheRuleRequest
     * @return A Java Future containing the result of the DeletePullThroughCacheRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>PullThroughCacheRuleNotFoundException The pull through cache rule was not found. Specify a valid pull
     *         through cache rule and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.DeletePullThroughCacheRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/DeletePullThroughCacheRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePullThroughCacheRuleResponse> deletePullThroughCacheRule(
            DeletePullThroughCacheRuleRequest deletePullThroughCacheRuleRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePullThroughCacheRuleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePullThroughCacheRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePullThroughCacheRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePullThroughCacheRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePullThroughCacheRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePullThroughCacheRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePullThroughCacheRuleRequest, DeletePullThroughCacheRuleResponse>()
                            .withOperationName("DeletePullThroughCacheRule").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeletePullThroughCacheRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deletePullThroughCacheRuleRequest));
            CompletableFuture<DeletePullThroughCacheRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the registry permissions policy.
     * </p>
     *
     * @param deleteRegistryPolicyRequest
     * @return A Java Future containing the result of the DeleteRegistryPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RegistryPolicyNotFoundException The registry doesn't have an associated registry policy.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.DeleteRegistryPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/DeleteRegistryPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRegistryPolicyResponse> deleteRegistryPolicy(
            DeleteRegistryPolicyRequest deleteRegistryPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRegistryPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRegistryPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRegistryPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRegistryPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteRegistryPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRegistryPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRegistryPolicyRequest, DeleteRegistryPolicyResponse>()
                            .withOperationName("DeleteRegistryPolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteRegistryPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteRegistryPolicyRequest));
            CompletableFuture<DeleteRegistryPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a repository. If the repository isn't empty, you must either delete the contents of the repository or use
     * the <code>force</code> option to delete the repository and have Amazon ECR delete all of its contents on your
     * behalf.
     * </p>
     *
     * @param deleteRepositoryRequest
     * @return A Java Future containing the result of the DeleteRepository operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>RepositoryNotEmptyException The specified repository contains images. To delete a repository that
     *         contains images, you must force the deletion with the <code>force</code> parameter.</li>
     *         <li>KmsException The operation failed due to a KMS exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.DeleteRepository
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/DeleteRepository" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRepositoryResponse> deleteRepository(DeleteRepositoryRequest deleteRepositoryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRepositoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRepositoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRepository");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRepositoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteRepositoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRepositoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRepositoryRequest, DeleteRepositoryResponse>()
                            .withOperationName("DeleteRepository").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteRepositoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteRepositoryRequest));
            CompletableFuture<DeleteRepositoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a repository creation template.
     * </p>
     *
     * @param deleteRepositoryCreationTemplateRequest
     * @return A Java Future containing the result of the DeleteRepositoryCreationTemplate operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>TemplateNotFoundException The specified repository creation template can't be found. Verify the
     *         registry ID and prefix and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.DeleteRepositoryCreationTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/DeleteRepositoryCreationTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRepositoryCreationTemplateResponse> deleteRepositoryCreationTemplate(
            DeleteRepositoryCreationTemplateRequest deleteRepositoryCreationTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRepositoryCreationTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteRepositoryCreationTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRepositoryCreationTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRepositoryCreationTemplateResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteRepositoryCreationTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRepositoryCreationTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRepositoryCreationTemplateRequest, DeleteRepositoryCreationTemplateResponse>()
                            .withOperationName("DeleteRepositoryCreationTemplate").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteRepositoryCreationTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteRepositoryCreationTemplateRequest));
            CompletableFuture<DeleteRepositoryCreationTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the repository policy associated with the specified repository.
     * </p>
     *
     * @param deleteRepositoryPolicyRequest
     * @return A Java Future containing the result of the DeleteRepositoryPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>RepositoryPolicyNotFoundException The specified repository and registry combination does not have an
     *         associated repository policy.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.DeleteRepositoryPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/DeleteRepositoryPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRepositoryPolicyResponse> deleteRepositoryPolicy(
            DeleteRepositoryPolicyRequest deleteRepositoryPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRepositoryPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRepositoryPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRepositoryPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRepositoryPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteRepositoryPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRepositoryPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRepositoryPolicyRequest, DeleteRepositoryPolicyResponse>()
                            .withOperationName("DeleteRepositoryPolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteRepositoryPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteRepositoryPolicyRequest));
            CompletableFuture<DeleteRepositoryPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the replication status for a specified image.
     * </p>
     *
     * @param describeImageReplicationStatusRequest
     * @return A Java Future containing the result of the DescribeImageReplicationStatus operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ImageNotFoundException The image requested does not exist in the specified repository.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.DescribeImageReplicationStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/DescribeImageReplicationStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeImageReplicationStatusResponse> describeImageReplicationStatus(
            DescribeImageReplicationStatusRequest describeImageReplicationStatusRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeImageReplicationStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeImageReplicationStatusRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeImageReplicationStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeImageReplicationStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeImageReplicationStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeImageReplicationStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeImageReplicationStatusRequest, DescribeImageReplicationStatusResponse>()
                            .withOperationName("DescribeImageReplicationStatus").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeImageReplicationStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeImageReplicationStatusRequest));
            CompletableFuture<DescribeImageReplicationStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the scan findings for the specified image.
     * </p>
     *
     * @param describeImageScanFindingsRequest
     * @return A Java Future containing the result of the DescribeImageScanFindings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>ImageNotFoundException The image requested does not exist in the specified repository.</li>
     *         <li>ScanNotFoundException The specified image scan could not be found. Ensure that image scanning is
     *         enabled on the repository and try again.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.DescribeImageScanFindings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/DescribeImageScanFindings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeImageScanFindingsResponse> describeImageScanFindings(
            DescribeImageScanFindingsRequest describeImageScanFindingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeImageScanFindingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeImageScanFindingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeImageScanFindings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeImageScanFindingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeImageScanFindingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeImageScanFindingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeImageScanFindingsRequest, DescribeImageScanFindingsResponse>()
                            .withOperationName("DescribeImageScanFindings").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeImageScanFindingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeImageScanFindingsRequest));
            CompletableFuture<DescribeImageScanFindingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns metadata about the images in a repository.
     * </p>
     * <note>
     * <p>
     * Beginning with Docker version 1.9, the Docker client compresses image layers before pushing them to a V2 Docker
     * registry. The output of the <code>docker images</code> command shows the uncompressed image size, so it may
     * return a larger image size than the image sizes returned by <a>DescribeImages</a>.
     * </p>
     * </note>
     *
     * @param describeImagesRequest
     * @return A Java Future containing the result of the DescribeImages operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>ImageNotFoundException The image requested does not exist in the specified repository.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.DescribeImages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/DescribeImages" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeImagesResponse> describeImages(DescribeImagesRequest describeImagesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeImagesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeImagesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeImages");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeImagesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeImagesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeImagesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeImagesRequest, DescribeImagesResponse>()
                            .withOperationName("DescribeImages").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeImagesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeImagesRequest));
            CompletableFuture<DescribeImagesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the pull through cache rules for a registry.
     * </p>
     *
     * @param describePullThroughCacheRulesRequest
     * @return A Java Future containing the result of the DescribePullThroughCacheRules operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>PullThroughCacheRuleNotFoundException The pull through cache rule was not found. Specify a valid pull
     *         through cache rule and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.DescribePullThroughCacheRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/DescribePullThroughCacheRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribePullThroughCacheRulesResponse> describePullThroughCacheRules(
            DescribePullThroughCacheRulesRequest describePullThroughCacheRulesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describePullThroughCacheRulesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describePullThroughCacheRulesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePullThroughCacheRules");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribePullThroughCacheRulesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribePullThroughCacheRulesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribePullThroughCacheRulesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePullThroughCacheRulesRequest, DescribePullThroughCacheRulesResponse>()
                            .withOperationName("DescribePullThroughCacheRules").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribePullThroughCacheRulesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describePullThroughCacheRulesRequest));
            CompletableFuture<DescribePullThroughCacheRulesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the settings for a registry. The replication configuration for a repository can be created or updated
     * with the <a>PutReplicationConfiguration</a> API action.
     * </p>
     *
     * @param describeRegistryRequest
     * @return A Java Future containing the result of the DescribeRegistry operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.DescribeRegistry
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/DescribeRegistry" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeRegistryResponse> describeRegistry(DescribeRegistryRequest describeRegistryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeRegistryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRegistryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRegistry");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeRegistryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeRegistryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeRegistryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeRegistryRequest, DescribeRegistryResponse>()
                            .withOperationName("DescribeRegistry").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeRegistryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeRegistryRequest));
            CompletableFuture<DescribeRegistryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes image repositories in a registry.
     * </p>
     *
     * @param describeRepositoriesRequest
     * @return A Java Future containing the result of the DescribeRepositories operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.DescribeRepositories
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/DescribeRepositories" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeRepositoriesResponse> describeRepositories(
            DescribeRepositoriesRequest describeRepositoriesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeRepositoriesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRepositoriesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRepositories");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeRepositoriesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeRepositoriesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeRepositoriesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeRepositoriesRequest, DescribeRepositoriesResponse>()
                            .withOperationName("DescribeRepositories").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeRepositoriesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeRepositoriesRequest));
            CompletableFuture<DescribeRepositoriesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns details about the repository creation templates in a registry. The <code>prefixes</code> request
     * parameter can be used to return the details for a specific repository creation template.
     * </p>
     *
     * @param describeRepositoryCreationTemplatesRequest
     * @return A Java Future containing the result of the DescribeRepositoryCreationTemplates operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.DescribeRepositoryCreationTemplates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/DescribeRepositoryCreationTemplates"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeRepositoryCreationTemplatesResponse> describeRepositoryCreationTemplates(
            DescribeRepositoryCreationTemplatesRequest describeRepositoryCreationTemplatesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeRepositoryCreationTemplatesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeRepositoryCreationTemplatesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRepositoryCreationTemplates");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeRepositoryCreationTemplatesResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeRepositoryCreationTemplatesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeRepositoryCreationTemplatesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeRepositoryCreationTemplatesRequest, DescribeRepositoryCreationTemplatesResponse>()
                            .withOperationName("DescribeRepositoryCreationTemplates").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeRepositoryCreationTemplatesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeRepositoryCreationTemplatesRequest));
            CompletableFuture<DescribeRepositoryCreationTemplatesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the account setting value for the specified setting name.
     * </p>
     *
     * @param getAccountSettingRequest
     * @return A Java Future containing the result of the GetAccountSetting operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.GetAccountSetting
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/GetAccountSetting" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetAccountSettingResponse> getAccountSetting(GetAccountSettingRequest getAccountSettingRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAccountSettingRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAccountSettingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccountSetting");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAccountSettingResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAccountSettingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAccountSettingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAccountSettingRequest, GetAccountSettingResponse>()
                            .withOperationName("GetAccountSetting").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetAccountSettingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAccountSettingRequest));
            CompletableFuture<GetAccountSettingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves an authorization token. An authorization token represents your IAM authentication credentials and can
     * be used to access any Amazon ECR registry that your IAM principal has access to. The authorization token is valid
     * for 12 hours.
     * </p>
     * <p>
     * The <code>authorizationToken</code> returned is a base64 encoded string that can be decoded and used in a
     * <code>docker login</code> command to authenticate to a registry. The CLI offers an
     * <code>get-login-password</code> command that simplifies the login process. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/Registries.html#registry_auth">Registry
     * authentication</a> in the <i>Amazon Elastic Container Registry User Guide</i>.
     * </p>
     *
     * @param getAuthorizationTokenRequest
     * @return A Java Future containing the result of the GetAuthorizationToken operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.GetAuthorizationToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/GetAuthorizationToken" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetAuthorizationTokenResponse> getAuthorizationToken(
            GetAuthorizationTokenRequest getAuthorizationTokenRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAuthorizationTokenRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAuthorizationTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAuthorizationToken");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAuthorizationTokenResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAuthorizationTokenResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAuthorizationTokenResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAuthorizationTokenRequest, GetAuthorizationTokenResponse>()
                            .withOperationName("GetAuthorizationToken").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetAuthorizationTokenRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAuthorizationTokenRequest));
            CompletableFuture<GetAuthorizationTokenResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the pre-signed Amazon S3 download URL corresponding to an image layer. You can only get URLs for image
     * layers that are referenced in an image.
     * </p>
     * <p>
     * When an image is pulled, the GetDownloadUrlForLayer API is called once per image layer that is not already
     * cached.
     * </p>
     * <note>
     * <p>
     * This operation is used by the Amazon ECR proxy and is not generally used by customers for pulling and pushing
     * images. In most cases, you should use the <code>docker</code> CLI to pull, tag, and push images.
     * </p>
     * </note>
     *
     * @param getDownloadUrlForLayerRequest
     * @return A Java Future containing the result of the GetDownloadUrlForLayer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>LayersNotFoundException The specified layers could not be found, or the specified layer is not valid
     *         for this repository.</li>
     *         <li>LayerInaccessibleException The specified layer is not available because it is not associated with an
     *         image. Unassociated image layers may be cleaned up at any time.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>UnableToGetUpstreamLayerException There was an issue getting the upstream layer matching the pull
     *         through cache rule.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.GetDownloadUrlForLayer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/GetDownloadUrlForLayer" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDownloadUrlForLayerResponse> getDownloadUrlForLayer(
            GetDownloadUrlForLayerRequest getDownloadUrlForLayerRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDownloadUrlForLayerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDownloadUrlForLayerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDownloadUrlForLayer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDownloadUrlForLayerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDownloadUrlForLayerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDownloadUrlForLayerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDownloadUrlForLayerRequest, GetDownloadUrlForLayerResponse>()
                            .withOperationName("GetDownloadUrlForLayer").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDownloadUrlForLayerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDownloadUrlForLayerRequest));
            CompletableFuture<GetDownloadUrlForLayerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the lifecycle policy for the specified repository.
     * </p>
     *
     * @param getLifecyclePolicyRequest
     * @return A Java Future containing the result of the GetLifecyclePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>LifecyclePolicyNotFoundException The lifecycle policy could not be found, and no policy is set to the
     *         repository.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.GetLifecyclePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/GetLifecyclePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetLifecyclePolicyResponse> getLifecyclePolicy(GetLifecyclePolicyRequest getLifecyclePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getLifecyclePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLifecyclePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLifecyclePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLifecyclePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetLifecyclePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLifecyclePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLifecyclePolicyRequest, GetLifecyclePolicyResponse>()
                            .withOperationName("GetLifecyclePolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetLifecyclePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getLifecyclePolicyRequest));
            CompletableFuture<GetLifecyclePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the results of the lifecycle policy preview request for the specified repository.
     * </p>
     *
     * @param getLifecyclePolicyPreviewRequest
     * @return A Java Future containing the result of the GetLifecyclePolicyPreview operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>LifecyclePolicyPreviewNotFoundException There is no dry run for this repository.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.GetLifecyclePolicyPreview
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/GetLifecyclePolicyPreview" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetLifecyclePolicyPreviewResponse> getLifecyclePolicyPreview(
            GetLifecyclePolicyPreviewRequest getLifecyclePolicyPreviewRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getLifecyclePolicyPreviewRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLifecyclePolicyPreviewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLifecyclePolicyPreview");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLifecyclePolicyPreviewResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetLifecyclePolicyPreviewResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLifecyclePolicyPreviewResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLifecyclePolicyPreviewRequest, GetLifecyclePolicyPreviewResponse>()
                            .withOperationName("GetLifecyclePolicyPreview").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetLifecyclePolicyPreviewRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getLifecyclePolicyPreviewRequest));
            CompletableFuture<GetLifecyclePolicyPreviewResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the permissions policy for a registry.
     * </p>
     *
     * @param getRegistryPolicyRequest
     * @return A Java Future containing the result of the GetRegistryPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RegistryPolicyNotFoundException The registry doesn't have an associated registry policy.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.GetRegistryPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/GetRegistryPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetRegistryPolicyResponse> getRegistryPolicy(GetRegistryPolicyRequest getRegistryPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRegistryPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRegistryPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRegistryPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRegistryPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRegistryPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRegistryPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRegistryPolicyRequest, GetRegistryPolicyResponse>()
                            .withOperationName("GetRegistryPolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetRegistryPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getRegistryPolicyRequest));
            CompletableFuture<GetRegistryPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the scanning configuration for a registry.
     * </p>
     *
     * @param getRegistryScanningConfigurationRequest
     * @return A Java Future containing the result of the GetRegistryScanningConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.GetRegistryScanningConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/GetRegistryScanningConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRegistryScanningConfigurationResponse> getRegistryScanningConfiguration(
            GetRegistryScanningConfigurationRequest getRegistryScanningConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRegistryScanningConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getRegistryScanningConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRegistryScanningConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRegistryScanningConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetRegistryScanningConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRegistryScanningConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRegistryScanningConfigurationRequest, GetRegistryScanningConfigurationResponse>()
                            .withOperationName("GetRegistryScanningConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetRegistryScanningConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getRegistryScanningConfigurationRequest));
            CompletableFuture<GetRegistryScanningConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the repository policy for the specified repository.
     * </p>
     *
     * @param getRepositoryPolicyRequest
     * @return A Java Future containing the result of the GetRepositoryPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>RepositoryPolicyNotFoundException The specified repository and registry combination does not have an
     *         associated repository policy.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.GetRepositoryPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/GetRepositoryPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetRepositoryPolicyResponse> getRepositoryPolicy(
            GetRepositoryPolicyRequest getRepositoryPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRepositoryPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRepositoryPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRepositoryPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRepositoryPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRepositoryPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRepositoryPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRepositoryPolicyRequest, GetRepositoryPolicyResponse>()
                            .withOperationName("GetRepositoryPolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetRepositoryPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getRepositoryPolicyRequest));
            CompletableFuture<GetRepositoryPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Notifies Amazon ECR that you intend to upload an image layer.
     * </p>
     * <p>
     * When an image is pushed, the InitiateLayerUpload API is called once per image layer that has not already been
     * uploaded. Whether or not an image layer has been uploaded is determined by the BatchCheckLayerAvailability API
     * action.
     * </p>
     * <note>
     * <p>
     * This operation is used by the Amazon ECR proxy and is not generally used by customers for pulling and pushing
     * images. In most cases, you should use the <code>docker</code> CLI to pull, tag, and push images.
     * </p>
     * </note>
     *
     * @param initiateLayerUploadRequest
     * @return A Java Future containing the result of the InitiateLayerUpload operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>KmsException The operation failed due to a KMS exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.InitiateLayerUpload
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/InitiateLayerUpload" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<InitiateLayerUploadResponse> initiateLayerUpload(
            InitiateLayerUploadRequest initiateLayerUploadRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(initiateLayerUploadRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, initiateLayerUploadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InitiateLayerUpload");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<InitiateLayerUploadResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, InitiateLayerUploadResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<InitiateLayerUploadResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<InitiateLayerUploadRequest, InitiateLayerUploadResponse>()
                            .withOperationName("InitiateLayerUpload").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new InitiateLayerUploadRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(initiateLayerUploadRequest));
            CompletableFuture<InitiateLayerUploadResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the image IDs for the specified repository.
     * </p>
     * <p>
     * You can filter images based on whether or not they are tagged by using the <code>tagStatus</code> filter and
     * specifying either <code>TAGGED</code>, <code>UNTAGGED</code> or <code>ANY</code>. For example, you can filter
     * your results to return only <code>UNTAGGED</code> images and then pipe that result to a <a>BatchDeleteImage</a>
     * operation to delete them. Or, you can filter your results to return only <code>TAGGED</code> images to list all
     * of the tags in your repository.
     * </p>
     *
     * @param listImagesRequest
     * @return A Java Future containing the result of the ListImages operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.ListImages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/ListImages" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListImagesResponse> listImages(ListImagesRequest listImagesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listImagesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listImagesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListImages");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListImagesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListImagesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListImagesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListImagesRequest, ListImagesResponse>().withOperationName("ListImages")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListImagesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listImagesRequest));
            CompletableFuture<ListImagesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the tags for an Amazon ECR resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Allows you to change the basic scan type version or registry policy scope.
     * </p>
     *
     * @param putAccountSettingRequest
     * @return A Java Future containing the result of the PutAccountSetting operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>LimitExceededException The operation did not succeed because it would have exceeded a service limit
     *         for your account. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/service-quotas.html">Amazon ECR service
     *         quotas</a> in the Amazon Elastic Container Registry User Guide.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.PutAccountSetting
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/PutAccountSetting" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutAccountSettingResponse> putAccountSetting(PutAccountSettingRequest putAccountSettingRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putAccountSettingRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putAccountSettingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAccountSetting");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutAccountSettingResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutAccountSettingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutAccountSettingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutAccountSettingRequest, PutAccountSettingResponse>()
                            .withOperationName("PutAccountSetting").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutAccountSettingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putAccountSettingRequest));
            CompletableFuture<PutAccountSettingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates the image manifest and tags associated with an image.
     * </p>
     * <p>
     * When an image is pushed and all new image layers have been uploaded, the PutImage API is called once to create or
     * update the image manifest and the tags associated with the image.
     * </p>
     * <note>
     * <p>
     * This operation is used by the Amazon ECR proxy and is not generally used by customers for pulling and pushing
     * images. In most cases, you should use the <code>docker</code> CLI to pull, tag, and push images.
     * </p>
     * </note>
     *
     * @param putImageRequest
     * @return A Java Future containing the result of the PutImage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>ImageAlreadyExistsException The specified image has already been pushed, and there were no changes to
     *         the manifest or image tag after the last push.</li>
     *         <li>LayersNotFoundException The specified layers could not be found, or the specified layer is not valid
     *         for this repository.</li>
     *         <li>ReferencedImagesNotFoundException The manifest list is referencing an image that does not exist.</li>
     *         <li>LimitExceededException The operation did not succeed because it would have exceeded a service limit
     *         for your account. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/service-quotas.html">Amazon ECR service
     *         quotas</a> in the Amazon Elastic Container Registry User Guide.</li>
     *         <li>ImageTagAlreadyExistsException The specified image is tagged with a tag that already exists. The
     *         repository is configured for tag immutability.</li>
     *         <li>ImageDigestDoesNotMatchException The specified image digest does not match the digest that Amazon ECR
     *         calculated for the image.</li>
     *         <li>KmsException The operation failed due to a KMS exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.PutImage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/PutImage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutImageResponse> putImage(PutImageRequest putImageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putImageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putImageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutImage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutImageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutImageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutImageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutImageRequest, PutImageResponse>().withOperationName("PutImage")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutImageRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(putImageRequest));
            CompletableFuture<PutImageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <important>
     * <p>
     * The <code>PutImageScanningConfiguration</code> API is being deprecated, in favor of specifying the image scanning
     * configuration at the registry level. For more information, see <a>PutRegistryScanningConfiguration</a>.
     * </p>
     * </important>
     * <p>
     * Updates the image scanning configuration for the specified repository.
     * </p>
     *
     * @param putImageScanningConfigurationRequest
     * @return A Java Future containing the result of the PutImageScanningConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.PutImageScanningConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/PutImageScanningConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutImageScanningConfigurationResponse> putImageScanningConfiguration(
            PutImageScanningConfigurationRequest putImageScanningConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putImageScanningConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putImageScanningConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutImageScanningConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutImageScanningConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutImageScanningConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutImageScanningConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutImageScanningConfigurationRequest, PutImageScanningConfigurationResponse>()
                            .withOperationName("PutImageScanningConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutImageScanningConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putImageScanningConfigurationRequest));
            CompletableFuture<PutImageScanningConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the image tag mutability settings for the specified repository. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/image-tag-mutability.html">Image tag mutability</a>
     * in the <i>Amazon Elastic Container Registry User Guide</i>.
     * </p>
     *
     * @param putImageTagMutabilityRequest
     * @return A Java Future containing the result of the PutImageTagMutability operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.PutImageTagMutability
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/PutImageTagMutability" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutImageTagMutabilityResponse> putImageTagMutability(
            PutImageTagMutabilityRequest putImageTagMutabilityRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putImageTagMutabilityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putImageTagMutabilityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutImageTagMutability");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutImageTagMutabilityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutImageTagMutabilityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutImageTagMutabilityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutImageTagMutabilityRequest, PutImageTagMutabilityResponse>()
                            .withOperationName("PutImageTagMutability").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutImageTagMutabilityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putImageTagMutabilityRequest));
            CompletableFuture<PutImageTagMutabilityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates the lifecycle policy for the specified repository. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/LifecyclePolicies.html">Lifecycle policy
     * template</a>.
     * </p>
     *
     * @param putLifecyclePolicyRequest
     * @return A Java Future containing the result of the PutLifecyclePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.PutLifecyclePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/PutLifecyclePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutLifecyclePolicyResponse> putLifecyclePolicy(PutLifecyclePolicyRequest putLifecyclePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putLifecyclePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putLifecyclePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutLifecyclePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutLifecyclePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutLifecyclePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutLifecyclePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutLifecyclePolicyRequest, PutLifecyclePolicyResponse>()
                            .withOperationName("PutLifecyclePolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutLifecyclePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putLifecyclePolicyRequest));
            CompletableFuture<PutLifecyclePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates the permissions policy for your registry.
     * </p>
     * <p>
     * A registry policy is used to specify permissions for another Amazon Web Services account and is used when
     * configuring cross-account replication. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/registry-permissions.html">Registry permissions</a>
     * in the <i>Amazon Elastic Container Registry User Guide</i>.
     * </p>
     *
     * @param putRegistryPolicyRequest
     * @return A Java Future containing the result of the PutRegistryPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.PutRegistryPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/PutRegistryPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutRegistryPolicyResponse> putRegistryPolicy(PutRegistryPolicyRequest putRegistryPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putRegistryPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putRegistryPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRegistryPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutRegistryPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutRegistryPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutRegistryPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutRegistryPolicyRequest, PutRegistryPolicyResponse>()
                            .withOperationName("PutRegistryPolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutRegistryPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putRegistryPolicyRequest));
            CompletableFuture<PutRegistryPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates the scanning configuration for your private registry.
     * </p>
     *
     * @param putRegistryScanningConfigurationRequest
     * @return A Java Future containing the result of the PutRegistryScanningConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.PutRegistryScanningConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/PutRegistryScanningConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutRegistryScanningConfigurationResponse> putRegistryScanningConfiguration(
            PutRegistryScanningConfigurationRequest putRegistryScanningConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putRegistryScanningConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putRegistryScanningConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRegistryScanningConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutRegistryScanningConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, PutRegistryScanningConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutRegistryScanningConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutRegistryScanningConfigurationRequest, PutRegistryScanningConfigurationResponse>()
                            .withOperationName("PutRegistryScanningConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutRegistryScanningConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putRegistryScanningConfigurationRequest));
            CompletableFuture<PutRegistryScanningConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates the replication configuration for a registry. The existing replication configuration for a
     * repository can be retrieved with the <a>DescribeRegistry</a> API action. The first time the
     * PutReplicationConfiguration API is called, a service-linked IAM role is created in your account for the
     * replication process. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/using-service-linked-roles.html">Using
     * service-linked roles for Amazon ECR</a> in the <i>Amazon Elastic Container Registry User Guide</i>. For more
     * information on the custom role for replication, see <a href=
     * "https://docs.aws.amazon.com/AmazonECR/latest/userguide/replication-creation-templates.html#roles-creatingrole-user-console"
     * >Creating an IAM role for replication</a>.
     * </p>
     * <note>
     * <p>
     * When configuring cross-account replication, the destination account must grant the source account permission to
     * replicate. This permission is controlled using a registry permissions policy. For more information, see
     * <a>PutRegistryPolicy</a>.
     * </p>
     * </note>
     *
     * @param putReplicationConfigurationRequest
     * @return A Java Future containing the result of the PutReplicationConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.PutReplicationConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/PutReplicationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutReplicationConfigurationResponse> putReplicationConfiguration(
            PutReplicationConfigurationRequest putReplicationConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putReplicationConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putReplicationConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutReplicationConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutReplicationConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutReplicationConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutReplicationConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutReplicationConfigurationRequest, PutReplicationConfigurationResponse>()
                            .withOperationName("PutReplicationConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutReplicationConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putReplicationConfigurationRequest));
            CompletableFuture<PutReplicationConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Applies a repository policy to the specified repository to control access permissions. For more information, see
     * <a href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/repository-policies.html">Amazon ECR Repository
     * policies</a> in the <i>Amazon Elastic Container Registry User Guide</i>.
     * </p>
     *
     * @param setRepositoryPolicyRequest
     * @return A Java Future containing the result of the SetRepositoryPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.SetRepositoryPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/SetRepositoryPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SetRepositoryPolicyResponse> setRepositoryPolicy(
            SetRepositoryPolicyRequest setRepositoryPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(setRepositoryPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setRepositoryPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetRepositoryPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SetRepositoryPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SetRepositoryPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SetRepositoryPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SetRepositoryPolicyRequest, SetRepositoryPolicyResponse>()
                            .withOperationName("SetRepositoryPolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SetRepositoryPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(setRepositoryPolicyRequest));
            CompletableFuture<SetRepositoryPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a basic image vulnerability scan.
     * </p>
     * <p>
     * A basic image scan can only be started once per 24 hours on an individual image. This limit includes if an image
     * was scanned on initial push. You can start up to 100,000 basic scans per 24 hours. This limit includes both scans
     * on initial push and scans initiated by the StartImageScan API. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/image-scanning-basic.html">Basic scanning</a> in the
     * <i>Amazon Elastic Container Registry User Guide</i>.
     * </p>
     *
     * @param startImageScanRequest
     * @return A Java Future containing the result of the StartImageScan operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>UnsupportedImageTypeException The image is of a type that cannot be scanned.</li>
     *         <li>LimitExceededException The operation did not succeed because it would have exceeded a service limit
     *         for your account. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/service-quotas.html">Amazon ECR service
     *         quotas</a> in the Amazon Elastic Container Registry User Guide.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>ImageNotFoundException The image requested does not exist in the specified repository.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.StartImageScan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/StartImageScan" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartImageScanResponse> startImageScan(StartImageScanRequest startImageScanRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startImageScanRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startImageScanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartImageScan");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartImageScanResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartImageScanResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartImageScanResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartImageScanRequest, StartImageScanResponse>()
                            .withOperationName("StartImageScan").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartImageScanRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startImageScanRequest));
            CompletableFuture<StartImageScanResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a preview of a lifecycle policy for the specified repository. This allows you to see the results before
     * associating the lifecycle policy with the repository.
     * </p>
     *
     * @param startLifecyclePolicyPreviewRequest
     * @return A Java Future containing the result of the StartLifecyclePolicyPreview operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>LifecyclePolicyNotFoundException The lifecycle policy could not be found, and no policy is set to the
     *         repository.</li>
     *         <li>LifecyclePolicyPreviewInProgressException The previous lifecycle policy preview request has not
     *         completed. Wait and try again.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.StartLifecyclePolicyPreview
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/StartLifecyclePolicyPreview"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartLifecyclePolicyPreviewResponse> startLifecyclePolicyPreview(
            StartLifecyclePolicyPreviewRequest startLifecyclePolicyPreviewRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startLifecyclePolicyPreviewRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startLifecyclePolicyPreviewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartLifecyclePolicyPreview");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartLifecyclePolicyPreviewResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartLifecyclePolicyPreviewResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartLifecyclePolicyPreviewResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartLifecyclePolicyPreviewRequest, StartLifecyclePolicyPreviewResponse>()
                            .withOperationName("StartLifecyclePolicyPreview").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartLifecyclePolicyPreviewRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startLifecyclePolicyPreviewRequest));
            CompletableFuture<StartLifecyclePolicyPreviewResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds specified tags to a resource with the specified ARN. Existing tags on a resource are not changed if they are
     * not specified in the request parameters.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>InvalidTagParameterException An invalid parameter has been specified. Tag keys can have a maximum
     *         character length of 128 characters, and tag values can have a maximum length of 256 characters.</li>
     *         <li>TooManyTagsException The list of tags on the repository is over the limit. The maximum number of tags
     *         that can be applied to a repository is 50.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes specified tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>InvalidTagParameterException An invalid parameter has been specified. Tag keys can have a maximum
     *         character length of 128 characters, and tag values can have a maximum length of 256 characters.</li>
     *         <li>TooManyTagsException The list of tags on the repository is over the limit. The maximum number of tags
     *         that can be applied to a repository is 50.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing pull through cache rule.
     * </p>
     *
     * @param updatePullThroughCacheRuleRequest
     * @return A Java Future containing the result of the UpdatePullThroughCacheRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>UnableToAccessSecretException The secret is unable to be accessed. Verify the resource permissions
     *         for the secret and try again.</li>
     *         <li>PullThroughCacheRuleNotFoundException The pull through cache rule was not found. Specify a valid pull
     *         through cache rule and try again.</li>
     *         <li>SecretNotFoundException The ARN of the secret specified in the pull through cache rule was not found.
     *         Update the pull through cache rule with a valid secret ARN and try again.</li>
     *         <li>UnableToDecryptSecretValueException The secret is accessible but is unable to be decrypted. Verify
     *         the resource permisisons and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.UpdatePullThroughCacheRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/UpdatePullThroughCacheRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePullThroughCacheRuleResponse> updatePullThroughCacheRule(
            UpdatePullThroughCacheRuleRequest updatePullThroughCacheRuleRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePullThroughCacheRuleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePullThroughCacheRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePullThroughCacheRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePullThroughCacheRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePullThroughCacheRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePullThroughCacheRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePullThroughCacheRuleRequest, UpdatePullThroughCacheRuleResponse>()
                            .withOperationName("UpdatePullThroughCacheRule").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdatePullThroughCacheRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updatePullThroughCacheRuleRequest));
            CompletableFuture<UpdatePullThroughCacheRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing repository creation template.
     * </p>
     *
     * @param updateRepositoryCreationTemplateRequest
     * @return A Java Future containing the result of the UpdateRepositoryCreationTemplate operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>TemplateNotFoundException The specified repository creation template can't be found. Verify the
     *         registry ID and prefix and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.UpdateRepositoryCreationTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/UpdateRepositoryCreationTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRepositoryCreationTemplateResponse> updateRepositoryCreationTemplate(
            UpdateRepositoryCreationTemplateRequest updateRepositoryCreationTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateRepositoryCreationTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateRepositoryCreationTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRepositoryCreationTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRepositoryCreationTemplateResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateRepositoryCreationTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateRepositoryCreationTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRepositoryCreationTemplateRequest, UpdateRepositoryCreationTemplateResponse>()
                            .withOperationName("UpdateRepositoryCreationTemplate").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateRepositoryCreationTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateRepositoryCreationTemplateRequest));
            CompletableFuture<UpdateRepositoryCreationTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Uploads an image layer part to Amazon ECR.
     * </p>
     * <p>
     * When an image is pushed, each new image layer is uploaded in parts. The maximum size of each image layer part can
     * be 20971520 bytes (or about 20MB). The UploadLayerPart API is called once per each new image layer part.
     * </p>
     * <note>
     * <p>
     * This operation is used by the Amazon ECR proxy and is not generally used by customers for pulling and pushing
     * images. In most cases, you should use the <code>docker</code> CLI to pull, tag, and push images.
     * </p>
     * </note>
     *
     * @param uploadLayerPartRequest
     * @return A Java Future containing the result of the UploadLayerPart operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>InvalidLayerPartException The layer part size is not valid, or the first byte specified is not
     *         consecutive to the last byte of a previous layer part upload.</li>
     *         <li>RepositoryNotFoundException The specified repository could not be found. Check the spelling of the
     *         specified repository and ensure that you are performing operations on the correct registry.</li>
     *         <li>UploadNotFoundException The upload could not be found, or the specified upload ID is not valid for
     *         this repository.</li>
     *         <li>LimitExceededException The operation did not succeed because it would have exceeded a service limit
     *         for your account. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/service-quotas.html">Amazon ECR service
     *         quotas</a> in the Amazon Elastic Container Registry User Guide.</li>
     *         <li>KmsException The operation failed due to a KMS exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.UploadLayerPart
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/UploadLayerPart" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UploadLayerPartResponse> uploadLayerPart(UploadLayerPartRequest uploadLayerPartRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(uploadLayerPartRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, uploadLayerPartRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UploadLayerPart");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UploadLayerPartResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UploadLayerPartResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UploadLayerPartResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UploadLayerPartRequest, UploadLayerPartResponse>()
                            .withOperationName("UploadLayerPart").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UploadLayerPartRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(uploadLayerPartRequest));
            CompletableFuture<UploadLayerPartResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Validates an existing pull through cache rule for an upstream registry that requires authentication. This will
     * retrieve the contents of the Amazon Web Services Secrets Manager secret, verify the syntax, and then validate
     * that authentication to the upstream registry is successful.
     * </p>
     *
     * @param validatePullThroughCacheRuleRequest
     * @return A Java Future containing the result of the ValidatePullThroughCacheRule operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ValidationException There was an exception validating this request.</li>
     *         <li>PullThroughCacheRuleNotFoundException The pull through cache rule was not found. Specify a valid pull
     *         through cache rule and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EcrAsyncClient.ValidatePullThroughCacheRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/ValidatePullThroughCacheRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ValidatePullThroughCacheRuleResponse> validatePullThroughCacheRule(
            ValidatePullThroughCacheRuleRequest validatePullThroughCacheRuleRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(validatePullThroughCacheRuleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, validatePullThroughCacheRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ValidatePullThroughCacheRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ValidatePullThroughCacheRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ValidatePullThroughCacheRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ValidatePullThroughCacheRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ValidatePullThroughCacheRuleRequest, ValidatePullThroughCacheRuleResponse>()
                            .withOperationName("ValidatePullThroughCacheRule").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ValidatePullThroughCacheRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(validatePullThroughCacheRuleRequest));
            CompletableFuture<ValidatePullThroughCacheRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public EcrAsyncWaiter waiter() {
        return EcrAsyncWaiter.builder().client(this).scheduledExecutorService(executorService).build();
    }

    @Override
    public final EcrServiceClientConfiguration serviceClientConfiguration() {
        return new EcrServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(EcrException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LayerPartTooSmallException")
                                .exceptionBuilderSupplier(LayerPartTooSmallException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException")
                                .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException")
                                .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnableToAccessSecretException")
                                .exceptionBuilderSupplier(UnableToAccessSecretException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EmptyUploadException")
                                .exceptionBuilderSupplier(EmptyUploadException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException")
                                .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TemplateNotFoundException")
                                .exceptionBuilderSupplier(TemplateNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException")
                                .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException")
                                .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                                .exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LayerInaccessibleException")
                                .exceptionBuilderSupplier(LayerInaccessibleException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidLayerPartException")
                                .exceptionBuilderSupplier(InvalidLayerPartException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException")
                                .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                                .exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ScanNotFoundException")
                                .exceptionBuilderSupplier(ScanNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException")
                                .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException")
                                .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ImageNotFoundException")
                                .exceptionBuilderSupplier(ImageNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LayersNotFoundException")
                                .exceptionBuilderSupplier(LayersNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RepositoryNotFoundException")
                                .exceptionBuilderSupplier(RepositoryNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException")
                                .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException")
                                .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KmsException").exceptionBuilderSupplier(KmsException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException")
                                .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                                .exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException")
                                .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UploadNotFoundException")
                                .exceptionBuilderSupplier(UploadNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SecretNotFoundException")
                                .exceptionBuilderSupplier(SecretNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServerException")
                                .exceptionBuilderSupplier(ServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException")
                                .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException")
                                .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTagParameterException")
                                .exceptionBuilderSupplier(InvalidTagParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidLayerException")
                                .exceptionBuilderSupplier(InvalidLayerException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                                .exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                                .exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                                .httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        EcrServiceClientConfigurationBuilder serviceConfigBuilder = new EcrServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
