/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecr;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.ecr.internal.EcrServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.ecr.model.BatchCheckLayerAvailabilityRequest;
import software.amazon.awssdk.services.ecr.model.BatchCheckLayerAvailabilityResponse;
import software.amazon.awssdk.services.ecr.model.BatchDeleteImageRequest;
import software.amazon.awssdk.services.ecr.model.BatchDeleteImageResponse;
import software.amazon.awssdk.services.ecr.model.BatchGetImageRequest;
import software.amazon.awssdk.services.ecr.model.BatchGetImageResponse;
import software.amazon.awssdk.services.ecr.model.BatchGetRepositoryScanningConfigurationRequest;
import software.amazon.awssdk.services.ecr.model.BatchGetRepositoryScanningConfigurationResponse;
import software.amazon.awssdk.services.ecr.model.CompleteLayerUploadRequest;
import software.amazon.awssdk.services.ecr.model.CompleteLayerUploadResponse;
import software.amazon.awssdk.services.ecr.model.CreatePullThroughCacheRuleRequest;
import software.amazon.awssdk.services.ecr.model.CreatePullThroughCacheRuleResponse;
import software.amazon.awssdk.services.ecr.model.CreateRepositoryCreationTemplateRequest;
import software.amazon.awssdk.services.ecr.model.CreateRepositoryCreationTemplateResponse;
import software.amazon.awssdk.services.ecr.model.CreateRepositoryRequest;
import software.amazon.awssdk.services.ecr.model.CreateRepositoryResponse;
import software.amazon.awssdk.services.ecr.model.DeleteLifecyclePolicyRequest;
import software.amazon.awssdk.services.ecr.model.DeleteLifecyclePolicyResponse;
import software.amazon.awssdk.services.ecr.model.DeletePullThroughCacheRuleRequest;
import software.amazon.awssdk.services.ecr.model.DeletePullThroughCacheRuleResponse;
import software.amazon.awssdk.services.ecr.model.DeleteRegistryPolicyRequest;
import software.amazon.awssdk.services.ecr.model.DeleteRegistryPolicyResponse;
import software.amazon.awssdk.services.ecr.model.DeleteRepositoryCreationTemplateRequest;
import software.amazon.awssdk.services.ecr.model.DeleteRepositoryCreationTemplateResponse;
import software.amazon.awssdk.services.ecr.model.DeleteRepositoryPolicyRequest;
import software.amazon.awssdk.services.ecr.model.DeleteRepositoryPolicyResponse;
import software.amazon.awssdk.services.ecr.model.DeleteRepositoryRequest;
import software.amazon.awssdk.services.ecr.model.DeleteRepositoryResponse;
import software.amazon.awssdk.services.ecr.model.DescribeImageReplicationStatusRequest;
import software.amazon.awssdk.services.ecr.model.DescribeImageReplicationStatusResponse;
import software.amazon.awssdk.services.ecr.model.DescribeImageScanFindingsRequest;
import software.amazon.awssdk.services.ecr.model.DescribeImageScanFindingsResponse;
import software.amazon.awssdk.services.ecr.model.DescribeImagesRequest;
import software.amazon.awssdk.services.ecr.model.DescribeImagesResponse;
import software.amazon.awssdk.services.ecr.model.DescribePullThroughCacheRulesRequest;
import software.amazon.awssdk.services.ecr.model.DescribePullThroughCacheRulesResponse;
import software.amazon.awssdk.services.ecr.model.DescribeRegistryRequest;
import software.amazon.awssdk.services.ecr.model.DescribeRegistryResponse;
import software.amazon.awssdk.services.ecr.model.DescribeRepositoriesRequest;
import software.amazon.awssdk.services.ecr.model.DescribeRepositoriesResponse;
import software.amazon.awssdk.services.ecr.model.DescribeRepositoryCreationTemplatesRequest;
import software.amazon.awssdk.services.ecr.model.DescribeRepositoryCreationTemplatesResponse;
import software.amazon.awssdk.services.ecr.model.EcrException;
import software.amazon.awssdk.services.ecr.model.EmptyUploadException;
import software.amazon.awssdk.services.ecr.model.GetAccountSettingRequest;
import software.amazon.awssdk.services.ecr.model.GetAccountSettingResponse;
import software.amazon.awssdk.services.ecr.model.GetAuthorizationTokenRequest;
import software.amazon.awssdk.services.ecr.model.GetAuthorizationTokenResponse;
import software.amazon.awssdk.services.ecr.model.GetDownloadUrlForLayerRequest;
import software.amazon.awssdk.services.ecr.model.GetDownloadUrlForLayerResponse;
import software.amazon.awssdk.services.ecr.model.GetLifecyclePolicyPreviewRequest;
import software.amazon.awssdk.services.ecr.model.GetLifecyclePolicyPreviewResponse;
import software.amazon.awssdk.services.ecr.model.GetLifecyclePolicyRequest;
import software.amazon.awssdk.services.ecr.model.GetLifecyclePolicyResponse;
import software.amazon.awssdk.services.ecr.model.GetRegistryPolicyRequest;
import software.amazon.awssdk.services.ecr.model.GetRegistryPolicyResponse;
import software.amazon.awssdk.services.ecr.model.GetRegistryScanningConfigurationRequest;
import software.amazon.awssdk.services.ecr.model.GetRegistryScanningConfigurationResponse;
import software.amazon.awssdk.services.ecr.model.GetRepositoryPolicyRequest;
import software.amazon.awssdk.services.ecr.model.GetRepositoryPolicyResponse;
import software.amazon.awssdk.services.ecr.model.ImageAlreadyExistsException;
import software.amazon.awssdk.services.ecr.model.ImageDigestDoesNotMatchException;
import software.amazon.awssdk.services.ecr.model.ImageNotFoundException;
import software.amazon.awssdk.services.ecr.model.ImageTagAlreadyExistsException;
import software.amazon.awssdk.services.ecr.model.InitiateLayerUploadRequest;
import software.amazon.awssdk.services.ecr.model.InitiateLayerUploadResponse;
import software.amazon.awssdk.services.ecr.model.InvalidLayerException;
import software.amazon.awssdk.services.ecr.model.InvalidLayerPartException;
import software.amazon.awssdk.services.ecr.model.InvalidParameterException;
import software.amazon.awssdk.services.ecr.model.InvalidTagParameterException;
import software.amazon.awssdk.services.ecr.model.KmsException;
import software.amazon.awssdk.services.ecr.model.LayerAlreadyExistsException;
import software.amazon.awssdk.services.ecr.model.LayerInaccessibleException;
import software.amazon.awssdk.services.ecr.model.LayerPartTooSmallException;
import software.amazon.awssdk.services.ecr.model.LayersNotFoundException;
import software.amazon.awssdk.services.ecr.model.LifecyclePolicyNotFoundException;
import software.amazon.awssdk.services.ecr.model.LifecyclePolicyPreviewInProgressException;
import software.amazon.awssdk.services.ecr.model.LifecyclePolicyPreviewNotFoundException;
import software.amazon.awssdk.services.ecr.model.LimitExceededException;
import software.amazon.awssdk.services.ecr.model.ListImagesRequest;
import software.amazon.awssdk.services.ecr.model.ListImagesResponse;
import software.amazon.awssdk.services.ecr.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.ecr.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.ecr.model.PullThroughCacheRuleAlreadyExistsException;
import software.amazon.awssdk.services.ecr.model.PullThroughCacheRuleNotFoundException;
import software.amazon.awssdk.services.ecr.model.PutAccountSettingRequest;
import software.amazon.awssdk.services.ecr.model.PutAccountSettingResponse;
import software.amazon.awssdk.services.ecr.model.PutImageRequest;
import software.amazon.awssdk.services.ecr.model.PutImageResponse;
import software.amazon.awssdk.services.ecr.model.PutImageScanningConfigurationRequest;
import software.amazon.awssdk.services.ecr.model.PutImageScanningConfigurationResponse;
import software.amazon.awssdk.services.ecr.model.PutImageTagMutabilityRequest;
import software.amazon.awssdk.services.ecr.model.PutImageTagMutabilityResponse;
import software.amazon.awssdk.services.ecr.model.PutLifecyclePolicyRequest;
import software.amazon.awssdk.services.ecr.model.PutLifecyclePolicyResponse;
import software.amazon.awssdk.services.ecr.model.PutRegistryPolicyRequest;
import software.amazon.awssdk.services.ecr.model.PutRegistryPolicyResponse;
import software.amazon.awssdk.services.ecr.model.PutRegistryScanningConfigurationRequest;
import software.amazon.awssdk.services.ecr.model.PutRegistryScanningConfigurationResponse;
import software.amazon.awssdk.services.ecr.model.PutReplicationConfigurationRequest;
import software.amazon.awssdk.services.ecr.model.PutReplicationConfigurationResponse;
import software.amazon.awssdk.services.ecr.model.ReferencedImagesNotFoundException;
import software.amazon.awssdk.services.ecr.model.RegistryPolicyNotFoundException;
import software.amazon.awssdk.services.ecr.model.RepositoryAlreadyExistsException;
import software.amazon.awssdk.services.ecr.model.RepositoryNotEmptyException;
import software.amazon.awssdk.services.ecr.model.RepositoryNotFoundException;
import software.amazon.awssdk.services.ecr.model.RepositoryPolicyNotFoundException;
import software.amazon.awssdk.services.ecr.model.ScanNotFoundException;
import software.amazon.awssdk.services.ecr.model.SecretNotFoundException;
import software.amazon.awssdk.services.ecr.model.ServerException;
import software.amazon.awssdk.services.ecr.model.SetRepositoryPolicyRequest;
import software.amazon.awssdk.services.ecr.model.SetRepositoryPolicyResponse;
import software.amazon.awssdk.services.ecr.model.StartImageScanRequest;
import software.amazon.awssdk.services.ecr.model.StartImageScanResponse;
import software.amazon.awssdk.services.ecr.model.StartLifecyclePolicyPreviewRequest;
import software.amazon.awssdk.services.ecr.model.StartLifecyclePolicyPreviewResponse;
import software.amazon.awssdk.services.ecr.model.TagResourceRequest;
import software.amazon.awssdk.services.ecr.model.TagResourceResponse;
import software.amazon.awssdk.services.ecr.model.TemplateAlreadyExistsException;
import software.amazon.awssdk.services.ecr.model.TemplateNotFoundException;
import software.amazon.awssdk.services.ecr.model.TooManyTagsException;
import software.amazon.awssdk.services.ecr.model.UnableToAccessSecretException;
import software.amazon.awssdk.services.ecr.model.UnableToDecryptSecretValueException;
import software.amazon.awssdk.services.ecr.model.UnableToGetUpstreamImageException;
import software.amazon.awssdk.services.ecr.model.UnableToGetUpstreamLayerException;
import software.amazon.awssdk.services.ecr.model.UnsupportedImageTypeException;
import software.amazon.awssdk.services.ecr.model.UnsupportedUpstreamRegistryException;
import software.amazon.awssdk.services.ecr.model.UntagResourceRequest;
import software.amazon.awssdk.services.ecr.model.UntagResourceResponse;
import software.amazon.awssdk.services.ecr.model.UpdatePullThroughCacheRuleRequest;
import software.amazon.awssdk.services.ecr.model.UpdatePullThroughCacheRuleResponse;
import software.amazon.awssdk.services.ecr.model.UpdateRepositoryCreationTemplateRequest;
import software.amazon.awssdk.services.ecr.model.UpdateRepositoryCreationTemplateResponse;
import software.amazon.awssdk.services.ecr.model.UploadLayerPartRequest;
import software.amazon.awssdk.services.ecr.model.UploadLayerPartResponse;
import software.amazon.awssdk.services.ecr.model.UploadNotFoundException;
import software.amazon.awssdk.services.ecr.model.ValidatePullThroughCacheRuleRequest;
import software.amazon.awssdk.services.ecr.model.ValidatePullThroughCacheRuleResponse;
import software.amazon.awssdk.services.ecr.model.ValidationException;
import software.amazon.awssdk.services.ecr.transform.BatchCheckLayerAvailabilityRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.BatchDeleteImageRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.BatchGetImageRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.BatchGetRepositoryScanningConfigurationRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.CompleteLayerUploadRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.CreatePullThroughCacheRuleRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.CreateRepositoryCreationTemplateRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.CreateRepositoryRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.DeleteLifecyclePolicyRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.DeletePullThroughCacheRuleRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.DeleteRegistryPolicyRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.DeleteRepositoryCreationTemplateRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.DeleteRepositoryPolicyRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.DeleteRepositoryRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.DescribeImageReplicationStatusRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.DescribeImageScanFindingsRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.DescribeImagesRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.DescribePullThroughCacheRulesRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.DescribeRegistryRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.DescribeRepositoriesRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.DescribeRepositoryCreationTemplatesRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.GetAccountSettingRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.GetAuthorizationTokenRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.GetDownloadUrlForLayerRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.GetLifecyclePolicyPreviewRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.GetLifecyclePolicyRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.GetRegistryPolicyRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.GetRegistryScanningConfigurationRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.GetRepositoryPolicyRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.InitiateLayerUploadRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.ListImagesRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.PutAccountSettingRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.PutImageRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.PutImageScanningConfigurationRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.PutImageTagMutabilityRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.PutLifecyclePolicyRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.PutRegistryPolicyRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.PutRegistryScanningConfigurationRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.PutReplicationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.SetRepositoryPolicyRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.StartImageScanRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.StartLifecyclePolicyPreviewRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.UpdatePullThroughCacheRuleRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.UpdateRepositoryCreationTemplateRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.UploadLayerPartRequestMarshaller;
import software.amazon.awssdk.services.ecr.transform.ValidatePullThroughCacheRuleRequestMarshaller;
import software.amazon.awssdk.services.ecr.waiters.EcrWaiter;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link EcrClient}.
 *
 * @see EcrClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultEcrClient implements EcrClient {
    private static final Logger log = Logger.loggerFor(DefaultEcrClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultEcrClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Checks the availability of one or more image layers in a repository.
     * </p>
     * <p>
     * When an image is pushed to a repository, each image layer is checked to verify if it has been uploaded before. If
     * it has been uploaded, then the image layer is skipped.
     * </p>
     * <note>
     * <p>
     * This operation is used by the Amazon ECR proxy and is not generally used by customers for pulling and pushing
     * images. In most cases, you should use the <code>docker</code> CLI to pull, tag, and push images.
     * </p>
     * </note>
     *
     * @param batchCheckLayerAvailabilityRequest
     * @return Result of the BatchCheckLayerAvailability operation returned by the service.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.BatchCheckLayerAvailability
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/BatchCheckLayerAvailability"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchCheckLayerAvailabilityResponse batchCheckLayerAvailability(
            BatchCheckLayerAvailabilityRequest batchCheckLayerAvailabilityRequest) throws RepositoryNotFoundException,
            InvalidParameterException, ServerException, AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchCheckLayerAvailabilityResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchCheckLayerAvailabilityResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchCheckLayerAvailabilityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchCheckLayerAvailabilityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchCheckLayerAvailability");

            return clientHandler
                    .execute(new ClientExecutionParams<BatchCheckLayerAvailabilityRequest, BatchCheckLayerAvailabilityResponse>()
                            .withOperationName("BatchCheckLayerAvailability").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(batchCheckLayerAvailabilityRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new BatchCheckLayerAvailabilityRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a list of specified images within a repository. Images are specified with either an <code>imageTag</code>
     * or <code>imageDigest</code>.
     * </p>
     * <p>
     * You can remove a tag from an image by specifying the image's tag in your request. When you remove the last tag
     * from an image, the image is deleted from your repository.
     * </p>
     * <p>
     * You can completely delete an image (and all of its tags) by specifying the image's digest in your request.
     * </p>
     *
     * @param batchDeleteImageRequest
     *        Deletes specified images within a specified repository. Images are specified with either the
     *        <code>imageTag</code> or <code>imageDigest</code>.
     * @return Result of the BatchDeleteImage operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.BatchDeleteImage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/BatchDeleteImage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public BatchDeleteImageResponse batchDeleteImage(BatchDeleteImageRequest batchDeleteImageRequest) throws ServerException,
            InvalidParameterException, RepositoryNotFoundException, AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchDeleteImageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                BatchDeleteImageResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchDeleteImageRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchDeleteImageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDeleteImage");

            return clientHandler.execute(new ClientExecutionParams<BatchDeleteImageRequest, BatchDeleteImageResponse>()
                    .withOperationName("BatchDeleteImage").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(batchDeleteImageRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchDeleteImageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets detailed information for an image. Images are specified with either an <code>imageTag</code> or
     * <code>imageDigest</code>.
     * </p>
     * <p>
     * When an image is pulled, the BatchGetImage API is called once to retrieve the image manifest.
     * </p>
     *
     * @param batchGetImageRequest
     * @return Result of the BatchGetImage operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws LimitExceededException
     *         The operation did not succeed because it would have exceeded a service limit for your account. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/service-quotas.html">Amazon ECR service
     *         quotas</a> in the Amazon Elastic Container Registry User Guide.
     * @throws UnableToGetUpstreamImageException
     *         The image or images were unable to be pulled using the pull through cache rule. This is usually caused
     *         because of an issue with the Secrets Manager secret containing the credentials for the upstream registry.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.BatchGetImage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/BatchGetImage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public BatchGetImageResponse batchGetImage(BatchGetImageRequest batchGetImageRequest) throws ServerException,
            InvalidParameterException, RepositoryNotFoundException, LimitExceededException, UnableToGetUpstreamImageException,
            AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchGetImageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                BatchGetImageResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchGetImageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetImageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetImage");

            return clientHandler.execute(new ClientExecutionParams<BatchGetImageRequest, BatchGetImageResponse>()
                    .withOperationName("BatchGetImage").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(batchGetImageRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchGetImageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the scanning configuration for one or more repositories.
     * </p>
     *
     * @param batchGetRepositoryScanningConfigurationRequest
     * @return Result of the BatchGetRepositoryScanningConfiguration operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.BatchGetRepositoryScanningConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/BatchGetRepositoryScanningConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchGetRepositoryScanningConfigurationResponse batchGetRepositoryScanningConfiguration(
            BatchGetRepositoryScanningConfigurationRequest batchGetRepositoryScanningConfigurationRequest)
            throws ServerException, InvalidParameterException, RepositoryNotFoundException, ValidationException,
            AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchGetRepositoryScanningConfigurationResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, BatchGetRepositoryScanningConfigurationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchGetRepositoryScanningConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                batchGetRepositoryScanningConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetRepositoryScanningConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<BatchGetRepositoryScanningConfigurationRequest, BatchGetRepositoryScanningConfigurationResponse>()
                            .withOperationName("BatchGetRepositoryScanningConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration)
                            .withInput(batchGetRepositoryScanningConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new BatchGetRepositoryScanningConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Informs Amazon ECR that the image layer upload has completed for a specified registry, repository name, and
     * upload ID. You can optionally provide a <code>sha256</code> digest of the image layer for data validation
     * purposes.
     * </p>
     * <p>
     * When an image is pushed, the CompleteLayerUpload API is called once per each new image layer to verify that the
     * upload has completed.
     * </p>
     * <note>
     * <p>
     * This operation is used by the Amazon ECR proxy and is not generally used by customers for pulling and pushing
     * images. In most cases, you should use the <code>docker</code> CLI to pull, tag, and push images.
     * </p>
     * </note>
     *
     * @param completeLayerUploadRequest
     * @return Result of the CompleteLayerUpload operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws UploadNotFoundException
     *         The upload could not be found, or the specified upload ID is not valid for this repository.
     * @throws InvalidLayerException
     *         The layer digest calculation performed by Amazon ECR upon receipt of the image layer does not match the
     *         digest specified.
     * @throws LayerPartTooSmallException
     *         Layer parts must be at least 5 MiB in size.
     * @throws LayerAlreadyExistsException
     *         The image layer already exists in the associated repository.
     * @throws EmptyUploadException
     *         The specified layer upload does not contain any layer parts.
     * @throws KmsException
     *         The operation failed due to a KMS exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.CompleteLayerUpload
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/CompleteLayerUpload" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompleteLayerUploadResponse completeLayerUpload(CompleteLayerUploadRequest completeLayerUploadRequest)
            throws ServerException, InvalidParameterException, RepositoryNotFoundException, UploadNotFoundException,
            InvalidLayerException, LayerPartTooSmallException, LayerAlreadyExistsException, EmptyUploadException, KmsException,
            AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CompleteLayerUploadResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CompleteLayerUploadResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(completeLayerUploadRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, completeLayerUploadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CompleteLayerUpload");

            return clientHandler.execute(new ClientExecutionParams<CompleteLayerUploadRequest, CompleteLayerUploadResponse>()
                    .withOperationName("CompleteLayerUpload").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(completeLayerUploadRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CompleteLayerUploadRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a pull through cache rule. A pull through cache rule provides a way to cache images from an upstream
     * registry source in your Amazon ECR private registry. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/pull-through-cache.html">Using pull through cache
     * rules</a> in the <i>Amazon Elastic Container Registry User Guide</i>.
     * </p>
     *
     * @param createPullThroughCacheRuleRequest
     * @return Result of the CreatePullThroughCacheRule operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws PullThroughCacheRuleAlreadyExistsException
     *         A pull through cache rule with these settings already exists for the private registry.
     * @throws UnsupportedUpstreamRegistryException
     *         The specified upstream registry isn't supported.
     * @throws LimitExceededException
     *         The operation did not succeed because it would have exceeded a service limit for your account. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/service-quotas.html">Amazon ECR service
     *         quotas</a> in the Amazon Elastic Container Registry User Guide.
     * @throws UnableToAccessSecretException
     *         The secret is unable to be accessed. Verify the resource permissions for the secret and try again.
     * @throws SecretNotFoundException
     *         The ARN of the secret specified in the pull through cache rule was not found. Update the pull through
     *         cache rule with a valid secret ARN and try again.
     * @throws UnableToDecryptSecretValueException
     *         The secret is accessible but is unable to be decrypted. Verify the resource permisisons and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.CreatePullThroughCacheRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/CreatePullThroughCacheRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreatePullThroughCacheRuleResponse createPullThroughCacheRule(
            CreatePullThroughCacheRuleRequest createPullThroughCacheRuleRequest) throws ServerException,
            InvalidParameterException, ValidationException, PullThroughCacheRuleAlreadyExistsException,
            UnsupportedUpstreamRegistryException, LimitExceededException, UnableToAccessSecretException, SecretNotFoundException,
            UnableToDecryptSecretValueException, AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreatePullThroughCacheRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreatePullThroughCacheRuleResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPullThroughCacheRuleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPullThroughCacheRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePullThroughCacheRule");

            return clientHandler
                    .execute(new ClientExecutionParams<CreatePullThroughCacheRuleRequest, CreatePullThroughCacheRuleResponse>()
                            .withOperationName("CreatePullThroughCacheRule").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createPullThroughCacheRuleRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreatePullThroughCacheRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a repository. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/Repositories.html">Amazon ECR repositories</a> in
     * the <i>Amazon Elastic Container Registry User Guide</i>.
     * </p>
     *
     * @param createRepositoryRequest
     * @return Result of the CreateRepository operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws InvalidTagParameterException
     *         An invalid parameter has been specified. Tag keys can have a maximum character length of 128 characters,
     *         and tag values can have a maximum length of 256 characters.
     * @throws TooManyTagsException
     *         The list of tags on the repository is over the limit. The maximum number of tags that can be applied to a
     *         repository is 50.
     * @throws RepositoryAlreadyExistsException
     *         The specified repository already exists in the specified registry.
     * @throws LimitExceededException
     *         The operation did not succeed because it would have exceeded a service limit for your account. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/service-quotas.html">Amazon ECR service
     *         quotas</a> in the Amazon Elastic Container Registry User Guide.
     * @throws KmsException
     *         The operation failed due to a KMS exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.CreateRepository
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/CreateRepository" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateRepositoryResponse createRepository(CreateRepositoryRequest createRepositoryRequest) throws ServerException,
            InvalidParameterException, InvalidTagParameterException, TooManyTagsException, RepositoryAlreadyExistsException,
            LimitExceededException, KmsException, AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateRepositoryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateRepositoryResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRepositoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRepositoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRepository");

            return clientHandler.execute(new ClientExecutionParams<CreateRepositoryRequest, CreateRepositoryResponse>()
                    .withOperationName("CreateRepository").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createRepositoryRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateRepositoryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a repository creation template. This template is used to define the settings for repositories created by
     * Amazon ECR on your behalf. For example, repositories created through pull through cache actions. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/repository-creation-templates.html">Private
     * repository creation templates</a> in the <i>Amazon Elastic Container Registry User Guide</i>.
     * </p>
     *
     * @param createRepositoryCreationTemplateRequest
     * @return Result of the CreateRepositoryCreationTemplate operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws LimitExceededException
     *         The operation did not succeed because it would have exceeded a service limit for your account. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/service-quotas.html">Amazon ECR service
     *         quotas</a> in the Amazon Elastic Container Registry User Guide.
     * @throws TemplateAlreadyExistsException
     *         The repository creation template already exists. Specify a unique prefix and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.CreateRepositoryCreationTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/CreateRepositoryCreationTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateRepositoryCreationTemplateResponse createRepositoryCreationTemplate(
            CreateRepositoryCreationTemplateRequest createRepositoryCreationTemplateRequest) throws ServerException,
            ValidationException, InvalidParameterException, LimitExceededException, TemplateAlreadyExistsException,
            AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateRepositoryCreationTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateRepositoryCreationTemplateResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRepositoryCreationTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createRepositoryCreationTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRepositoryCreationTemplate");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateRepositoryCreationTemplateRequest, CreateRepositoryCreationTemplateResponse>()
                            .withOperationName("CreateRepositoryCreationTemplate").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createRepositoryCreationTemplateRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateRepositoryCreationTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the lifecycle policy associated with the specified repository.
     * </p>
     *
     * @param deleteLifecyclePolicyRequest
     * @return Result of the DeleteLifecyclePolicy operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws LifecyclePolicyNotFoundException
     *         The lifecycle policy could not be found, and no policy is set to the repository.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.DeleteLifecyclePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/DeleteLifecyclePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteLifecyclePolicyResponse deleteLifecyclePolicy(DeleteLifecyclePolicyRequest deleteLifecyclePolicyRequest)
            throws ServerException, InvalidParameterException, RepositoryNotFoundException, LifecyclePolicyNotFoundException,
            ValidationException, AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteLifecyclePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteLifecyclePolicyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteLifecyclePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLifecyclePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLifecyclePolicy");

            return clientHandler.execute(new ClientExecutionParams<DeleteLifecyclePolicyRequest, DeleteLifecyclePolicyResponse>()
                    .withOperationName("DeleteLifecyclePolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteLifecyclePolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteLifecyclePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a pull through cache rule.
     * </p>
     *
     * @param deletePullThroughCacheRuleRequest
     * @return Result of the DeletePullThroughCacheRule operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws PullThroughCacheRuleNotFoundException
     *         The pull through cache rule was not found. Specify a valid pull through cache rule and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.DeletePullThroughCacheRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/DeletePullThroughCacheRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeletePullThroughCacheRuleResponse deletePullThroughCacheRule(
            DeletePullThroughCacheRuleRequest deletePullThroughCacheRuleRequest) throws ServerException,
            InvalidParameterException, ValidationException, PullThroughCacheRuleNotFoundException, AwsServiceException,
            SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeletePullThroughCacheRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeletePullThroughCacheRuleResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePullThroughCacheRuleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePullThroughCacheRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePullThroughCacheRule");

            return clientHandler
                    .execute(new ClientExecutionParams<DeletePullThroughCacheRuleRequest, DeletePullThroughCacheRuleResponse>()
                            .withOperationName("DeletePullThroughCacheRule").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deletePullThroughCacheRuleRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeletePullThroughCacheRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the registry permissions policy.
     * </p>
     *
     * @param deleteRegistryPolicyRequest
     * @return Result of the DeleteRegistryPolicy operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RegistryPolicyNotFoundException
     *         The registry doesn't have an associated registry policy.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.DeleteRegistryPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/DeleteRegistryPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteRegistryPolicyResponse deleteRegistryPolicy(DeleteRegistryPolicyRequest deleteRegistryPolicyRequest)
            throws ServerException, InvalidParameterException, RegistryPolicyNotFoundException, ValidationException,
            AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteRegistryPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteRegistryPolicyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRegistryPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRegistryPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRegistryPolicy");

            return clientHandler.execute(new ClientExecutionParams<DeleteRegistryPolicyRequest, DeleteRegistryPolicyResponse>()
                    .withOperationName("DeleteRegistryPolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteRegistryPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteRegistryPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a repository. If the repository isn't empty, you must either delete the contents of the repository or use
     * the <code>force</code> option to delete the repository and have Amazon ECR delete all of its contents on your
     * behalf.
     * </p>
     *
     * @param deleteRepositoryRequest
     * @return Result of the DeleteRepository operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws RepositoryNotEmptyException
     *         The specified repository contains images. To delete a repository that contains images, you must force the
     *         deletion with the <code>force</code> parameter.
     * @throws KmsException
     *         The operation failed due to a KMS exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.DeleteRepository
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/DeleteRepository" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteRepositoryResponse deleteRepository(DeleteRepositoryRequest deleteRepositoryRequest) throws ServerException,
            InvalidParameterException, RepositoryNotFoundException, RepositoryNotEmptyException, KmsException,
            AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteRepositoryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteRepositoryResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRepositoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRepositoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRepository");

            return clientHandler.execute(new ClientExecutionParams<DeleteRepositoryRequest, DeleteRepositoryResponse>()
                    .withOperationName("DeleteRepository").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteRepositoryRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteRepositoryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a repository creation template.
     * </p>
     *
     * @param deleteRepositoryCreationTemplateRequest
     * @return Result of the DeleteRepositoryCreationTemplate operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws TemplateNotFoundException
     *         The specified repository creation template can't be found. Verify the registry ID and prefix and try
     *         again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.DeleteRepositoryCreationTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/DeleteRepositoryCreationTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteRepositoryCreationTemplateResponse deleteRepositoryCreationTemplate(
            DeleteRepositoryCreationTemplateRequest deleteRepositoryCreationTemplateRequest) throws ServerException,
            ValidationException, InvalidParameterException, TemplateNotFoundException, AwsServiceException, SdkClientException,
            EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteRepositoryCreationTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteRepositoryCreationTemplateResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRepositoryCreationTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteRepositoryCreationTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRepositoryCreationTemplate");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteRepositoryCreationTemplateRequest, DeleteRepositoryCreationTemplateResponse>()
                            .withOperationName("DeleteRepositoryCreationTemplate").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteRepositoryCreationTemplateRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteRepositoryCreationTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the repository policy associated with the specified repository.
     * </p>
     *
     * @param deleteRepositoryPolicyRequest
     * @return Result of the DeleteRepositoryPolicy operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws RepositoryPolicyNotFoundException
     *         The specified repository and registry combination does not have an associated repository policy.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.DeleteRepositoryPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/DeleteRepositoryPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteRepositoryPolicyResponse deleteRepositoryPolicy(DeleteRepositoryPolicyRequest deleteRepositoryPolicyRequest)
            throws ServerException, InvalidParameterException, RepositoryNotFoundException, RepositoryPolicyNotFoundException,
            AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteRepositoryPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteRepositoryPolicyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRepositoryPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRepositoryPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRepositoryPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteRepositoryPolicyRequest, DeleteRepositoryPolicyResponse>()
                            .withOperationName("DeleteRepositoryPolicy").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteRepositoryPolicyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteRepositoryPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the replication status for a specified image.
     * </p>
     *
     * @param describeImageReplicationStatusRequest
     * @return Result of the DescribeImageReplicationStatus operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ImageNotFoundException
     *         The image requested does not exist in the specified repository.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.DescribeImageReplicationStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/DescribeImageReplicationStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeImageReplicationStatusResponse describeImageReplicationStatus(
            DescribeImageReplicationStatusRequest describeImageReplicationStatusRequest) throws ServerException,
            InvalidParameterException, ImageNotFoundException, RepositoryNotFoundException, ValidationException,
            AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeImageReplicationStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeImageReplicationStatusResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeImageReplicationStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeImageReplicationStatusRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeImageReplicationStatus");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeImageReplicationStatusRequest, DescribeImageReplicationStatusResponse>()
                            .withOperationName("DescribeImageReplicationStatus").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeImageReplicationStatusRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeImageReplicationStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the scan findings for the specified image.
     * </p>
     *
     * @param describeImageScanFindingsRequest
     * @return Result of the DescribeImageScanFindings operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws ImageNotFoundException
     *         The image requested does not exist in the specified repository.
     * @throws ScanNotFoundException
     *         The specified image scan could not be found. Ensure that image scanning is enabled on the repository and
     *         try again.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.DescribeImageScanFindings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/DescribeImageScanFindings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeImageScanFindingsResponse describeImageScanFindings(
            DescribeImageScanFindingsRequest describeImageScanFindingsRequest) throws ServerException, InvalidParameterException,
            RepositoryNotFoundException, ImageNotFoundException, ScanNotFoundException, ValidationException, AwsServiceException,
            SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeImageScanFindingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeImageScanFindingsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeImageScanFindingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeImageScanFindingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeImageScanFindings");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeImageScanFindingsRequest, DescribeImageScanFindingsResponse>()
                            .withOperationName("DescribeImageScanFindings").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeImageScanFindingsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeImageScanFindingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns metadata about the images in a repository.
     * </p>
     * <note>
     * <p>
     * Beginning with Docker version 1.9, the Docker client compresses image layers before pushing them to a V2 Docker
     * registry. The output of the <code>docker images</code> command shows the uncompressed image size, so it may
     * return a larger image size than the image sizes returned by <a>DescribeImages</a>.
     * </p>
     * </note>
     *
     * @param describeImagesRequest
     * @return Result of the DescribeImages operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws ImageNotFoundException
     *         The image requested does not exist in the specified repository.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.DescribeImages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/DescribeImages" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeImagesResponse describeImages(DescribeImagesRequest describeImagesRequest) throws ServerException,
            InvalidParameterException, RepositoryNotFoundException, ImageNotFoundException, AwsServiceException,
            SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeImagesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeImagesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeImagesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeImagesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeImages");

            return clientHandler.execute(new ClientExecutionParams<DescribeImagesRequest, DescribeImagesResponse>()
                    .withOperationName("DescribeImages").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeImagesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeImagesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the pull through cache rules for a registry.
     * </p>
     *
     * @param describePullThroughCacheRulesRequest
     * @return Result of the DescribePullThroughCacheRules operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws PullThroughCacheRuleNotFoundException
     *         The pull through cache rule was not found. Specify a valid pull through cache rule and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.DescribePullThroughCacheRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/DescribePullThroughCacheRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribePullThroughCacheRulesResponse describePullThroughCacheRules(
            DescribePullThroughCacheRulesRequest describePullThroughCacheRulesRequest) throws ServerException,
            InvalidParameterException, ValidationException, PullThroughCacheRuleNotFoundException, AwsServiceException,
            SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribePullThroughCacheRulesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribePullThroughCacheRulesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describePullThroughCacheRulesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describePullThroughCacheRulesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePullThroughCacheRules");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribePullThroughCacheRulesRequest, DescribePullThroughCacheRulesResponse>()
                            .withOperationName("DescribePullThroughCacheRules").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describePullThroughCacheRulesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribePullThroughCacheRulesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the settings for a registry. The replication configuration for a repository can be created or updated
     * with the <a>PutReplicationConfiguration</a> API action.
     * </p>
     *
     * @param describeRegistryRequest
     * @return Result of the DescribeRegistry operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.DescribeRegistry
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/DescribeRegistry" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeRegistryResponse describeRegistry(DescribeRegistryRequest describeRegistryRequest) throws ServerException,
            InvalidParameterException, ValidationException, AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeRegistryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeRegistryResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeRegistryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRegistryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRegistry");

            return clientHandler.execute(new ClientExecutionParams<DescribeRegistryRequest, DescribeRegistryResponse>()
                    .withOperationName("DescribeRegistry").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeRegistryRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeRegistryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes image repositories in a registry.
     * </p>
     *
     * @param describeRepositoriesRequest
     * @return Result of the DescribeRepositories operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.DescribeRepositories
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/DescribeRepositories" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeRepositoriesResponse describeRepositories(DescribeRepositoriesRequest describeRepositoriesRequest)
            throws ServerException, InvalidParameterException, RepositoryNotFoundException, AwsServiceException,
            SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeRepositoriesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeRepositoriesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeRepositoriesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRepositoriesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRepositories");

            return clientHandler.execute(new ClientExecutionParams<DescribeRepositoriesRequest, DescribeRepositoriesResponse>()
                    .withOperationName("DescribeRepositories").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeRepositoriesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeRepositoriesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns details about the repository creation templates in a registry. The <code>prefixes</code> request
     * parameter can be used to return the details for a specific repository creation template.
     * </p>
     *
     * @param describeRepositoryCreationTemplatesRequest
     * @return Result of the DescribeRepositoryCreationTemplates operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.DescribeRepositoryCreationTemplates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/DescribeRepositoryCreationTemplates"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeRepositoryCreationTemplatesResponse describeRepositoryCreationTemplates(
            DescribeRepositoryCreationTemplatesRequest describeRepositoryCreationTemplatesRequest) throws ServerException,
            ValidationException, InvalidParameterException, AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeRepositoryCreationTemplatesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeRepositoryCreationTemplatesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeRepositoryCreationTemplatesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeRepositoryCreationTemplatesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRepositoryCreationTemplates");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeRepositoryCreationTemplatesRequest, DescribeRepositoryCreationTemplatesResponse>()
                            .withOperationName("DescribeRepositoryCreationTemplates").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeRepositoryCreationTemplatesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeRepositoryCreationTemplatesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the account setting value for the specified setting name.
     * </p>
     *
     * @param getAccountSettingRequest
     * @return Result of the GetAccountSetting operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.GetAccountSetting
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/GetAccountSetting" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetAccountSettingResponse getAccountSetting(GetAccountSettingRequest getAccountSettingRequest) throws ServerException,
            ValidationException, InvalidParameterException, AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAccountSettingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAccountSettingResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAccountSettingRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAccountSettingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccountSetting");

            return clientHandler.execute(new ClientExecutionParams<GetAccountSettingRequest, GetAccountSettingResponse>()
                    .withOperationName("GetAccountSetting").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getAccountSettingRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAccountSettingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves an authorization token. An authorization token represents your IAM authentication credentials and can
     * be used to access any Amazon ECR registry that your IAM principal has access to. The authorization token is valid
     * for 12 hours.
     * </p>
     * <p>
     * The <code>authorizationToken</code> returned is a base64 encoded string that can be decoded and used in a
     * <code>docker login</code> command to authenticate to a registry. The CLI offers an
     * <code>get-login-password</code> command that simplifies the login process. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/Registries.html#registry_auth">Registry
     * authentication</a> in the <i>Amazon Elastic Container Registry User Guide</i>.
     * </p>
     *
     * @param getAuthorizationTokenRequest
     * @return Result of the GetAuthorizationToken operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.GetAuthorizationToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/GetAuthorizationToken" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetAuthorizationTokenResponse getAuthorizationToken(GetAuthorizationTokenRequest getAuthorizationTokenRequest)
            throws ServerException, InvalidParameterException, AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAuthorizationTokenResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetAuthorizationTokenResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAuthorizationTokenRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAuthorizationTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAuthorizationToken");

            return clientHandler.execute(new ClientExecutionParams<GetAuthorizationTokenRequest, GetAuthorizationTokenResponse>()
                    .withOperationName("GetAuthorizationToken").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getAuthorizationTokenRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAuthorizationTokenRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the pre-signed Amazon S3 download URL corresponding to an image layer. You can only get URLs for image
     * layers that are referenced in an image.
     * </p>
     * <p>
     * When an image is pulled, the GetDownloadUrlForLayer API is called once per image layer that is not already
     * cached.
     * </p>
     * <note>
     * <p>
     * This operation is used by the Amazon ECR proxy and is not generally used by customers for pulling and pushing
     * images. In most cases, you should use the <code>docker</code> CLI to pull, tag, and push images.
     * </p>
     * </note>
     *
     * @param getDownloadUrlForLayerRequest
     * @return Result of the GetDownloadUrlForLayer operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws LayersNotFoundException
     *         The specified layers could not be found, or the specified layer is not valid for this repository.
     * @throws LayerInaccessibleException
     *         The specified layer is not available because it is not associated with an image. Unassociated image
     *         layers may be cleaned up at any time.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws UnableToGetUpstreamLayerException
     *         There was an issue getting the upstream layer matching the pull through cache rule.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.GetDownloadUrlForLayer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/GetDownloadUrlForLayer" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetDownloadUrlForLayerResponse getDownloadUrlForLayer(GetDownloadUrlForLayerRequest getDownloadUrlForLayerRequest)
            throws ServerException, InvalidParameterException, LayersNotFoundException, LayerInaccessibleException,
            RepositoryNotFoundException, UnableToGetUpstreamLayerException, AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDownloadUrlForLayerResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDownloadUrlForLayerResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDownloadUrlForLayerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDownloadUrlForLayerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDownloadUrlForLayer");

            return clientHandler
                    .execute(new ClientExecutionParams<GetDownloadUrlForLayerRequest, GetDownloadUrlForLayerResponse>()
                            .withOperationName("GetDownloadUrlForLayer").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getDownloadUrlForLayerRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetDownloadUrlForLayerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the lifecycle policy for the specified repository.
     * </p>
     *
     * @param getLifecyclePolicyRequest
     * @return Result of the GetLifecyclePolicy operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws LifecyclePolicyNotFoundException
     *         The lifecycle policy could not be found, and no policy is set to the repository.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.GetLifecyclePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/GetLifecyclePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetLifecyclePolicyResponse getLifecyclePolicy(GetLifecyclePolicyRequest getLifecyclePolicyRequest)
            throws ServerException, InvalidParameterException, RepositoryNotFoundException, LifecyclePolicyNotFoundException,
            ValidationException, AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetLifecyclePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetLifecyclePolicyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getLifecyclePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLifecyclePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLifecyclePolicy");

            return clientHandler.execute(new ClientExecutionParams<GetLifecyclePolicyRequest, GetLifecyclePolicyResponse>()
                    .withOperationName("GetLifecyclePolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getLifecyclePolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetLifecyclePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the results of the lifecycle policy preview request for the specified repository.
     * </p>
     *
     * @param getLifecyclePolicyPreviewRequest
     * @return Result of the GetLifecyclePolicyPreview operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws LifecyclePolicyPreviewNotFoundException
     *         There is no dry run for this repository.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.GetLifecyclePolicyPreview
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/GetLifecyclePolicyPreview" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetLifecyclePolicyPreviewResponse getLifecyclePolicyPreview(
            GetLifecyclePolicyPreviewRequest getLifecyclePolicyPreviewRequest) throws ServerException, InvalidParameterException,
            RepositoryNotFoundException, LifecyclePolicyPreviewNotFoundException, ValidationException, AwsServiceException,
            SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetLifecyclePolicyPreviewResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetLifecyclePolicyPreviewResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getLifecyclePolicyPreviewRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLifecyclePolicyPreviewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLifecyclePolicyPreview");

            return clientHandler
                    .execute(new ClientExecutionParams<GetLifecyclePolicyPreviewRequest, GetLifecyclePolicyPreviewResponse>()
                            .withOperationName("GetLifecyclePolicyPreview").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getLifecyclePolicyPreviewRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetLifecyclePolicyPreviewRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the permissions policy for a registry.
     * </p>
     *
     * @param getRegistryPolicyRequest
     * @return Result of the GetRegistryPolicy operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RegistryPolicyNotFoundException
     *         The registry doesn't have an associated registry policy.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.GetRegistryPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/GetRegistryPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetRegistryPolicyResponse getRegistryPolicy(GetRegistryPolicyRequest getRegistryPolicyRequest) throws ServerException,
            InvalidParameterException, RegistryPolicyNotFoundException, ValidationException, AwsServiceException,
            SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRegistryPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetRegistryPolicyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRegistryPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRegistryPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRegistryPolicy");

            return clientHandler.execute(new ClientExecutionParams<GetRegistryPolicyRequest, GetRegistryPolicyResponse>()
                    .withOperationName("GetRegistryPolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getRegistryPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetRegistryPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the scanning configuration for a registry.
     * </p>
     *
     * @param getRegistryScanningConfigurationRequest
     * @return Result of the GetRegistryScanningConfiguration operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.GetRegistryScanningConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/GetRegistryScanningConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetRegistryScanningConfigurationResponse getRegistryScanningConfiguration(
            GetRegistryScanningConfigurationRequest getRegistryScanningConfigurationRequest) throws ServerException,
            InvalidParameterException, ValidationException, AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRegistryScanningConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetRegistryScanningConfigurationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRegistryScanningConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getRegistryScanningConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRegistryScanningConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<GetRegistryScanningConfigurationRequest, GetRegistryScanningConfigurationResponse>()
                            .withOperationName("GetRegistryScanningConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getRegistryScanningConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetRegistryScanningConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the repository policy for the specified repository.
     * </p>
     *
     * @param getRepositoryPolicyRequest
     * @return Result of the GetRepositoryPolicy operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws RepositoryPolicyNotFoundException
     *         The specified repository and registry combination does not have an associated repository policy.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.GetRepositoryPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/GetRepositoryPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetRepositoryPolicyResponse getRepositoryPolicy(GetRepositoryPolicyRequest getRepositoryPolicyRequest)
            throws ServerException, InvalidParameterException, RepositoryNotFoundException, RepositoryPolicyNotFoundException,
            AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRepositoryPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetRepositoryPolicyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRepositoryPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRepositoryPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRepositoryPolicy");

            return clientHandler.execute(new ClientExecutionParams<GetRepositoryPolicyRequest, GetRepositoryPolicyResponse>()
                    .withOperationName("GetRepositoryPolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getRepositoryPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetRepositoryPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Notifies Amazon ECR that you intend to upload an image layer.
     * </p>
     * <p>
     * When an image is pushed, the InitiateLayerUpload API is called once per image layer that has not already been
     * uploaded. Whether or not an image layer has been uploaded is determined by the BatchCheckLayerAvailability API
     * action.
     * </p>
     * <note>
     * <p>
     * This operation is used by the Amazon ECR proxy and is not generally used by customers for pulling and pushing
     * images. In most cases, you should use the <code>docker</code> CLI to pull, tag, and push images.
     * </p>
     * </note>
     *
     * @param initiateLayerUploadRequest
     * @return Result of the InitiateLayerUpload operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws KmsException
     *         The operation failed due to a KMS exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.InitiateLayerUpload
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/InitiateLayerUpload" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public InitiateLayerUploadResponse initiateLayerUpload(InitiateLayerUploadRequest initiateLayerUploadRequest)
            throws ServerException, InvalidParameterException, RepositoryNotFoundException, KmsException, AwsServiceException,
            SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<InitiateLayerUploadResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, InitiateLayerUploadResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(initiateLayerUploadRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, initiateLayerUploadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InitiateLayerUpload");

            return clientHandler.execute(new ClientExecutionParams<InitiateLayerUploadRequest, InitiateLayerUploadResponse>()
                    .withOperationName("InitiateLayerUpload").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(initiateLayerUploadRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new InitiateLayerUploadRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all the image IDs for the specified repository.
     * </p>
     * <p>
     * You can filter images based on whether or not they are tagged by using the <code>tagStatus</code> filter and
     * specifying either <code>TAGGED</code>, <code>UNTAGGED</code> or <code>ANY</code>. For example, you can filter
     * your results to return only <code>UNTAGGED</code> images and then pipe that result to a <a>BatchDeleteImage</a>
     * operation to delete them. Or, you can filter your results to return only <code>TAGGED</code> images to list all
     * of the tags in your repository.
     * </p>
     *
     * @param listImagesRequest
     * @return Result of the ListImages operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.ListImages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/ListImages" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListImagesResponse listImages(ListImagesRequest listImagesRequest) throws ServerException, InvalidParameterException,
            RepositoryNotFoundException, AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListImagesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListImagesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listImagesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listImagesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListImages");

            return clientHandler.execute(new ClientExecutionParams<ListImagesRequest, ListImagesResponse>()
                    .withOperationName("ListImages").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listImagesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListImagesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List the tags for an Amazon ECR resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InvalidParameterException, RepositoryNotFoundException, ServerException, AwsServiceException,
            SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Allows you to change the basic scan type version or registry policy scope.
     * </p>
     *
     * @param putAccountSettingRequest
     * @return Result of the PutAccountSetting operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws LimitExceededException
     *         The operation did not succeed because it would have exceeded a service limit for your account. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/service-quotas.html">Amazon ECR service
     *         quotas</a> in the Amazon Elastic Container Registry User Guide.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.PutAccountSetting
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/PutAccountSetting" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutAccountSettingResponse putAccountSetting(PutAccountSettingRequest putAccountSettingRequest) throws ServerException,
            ValidationException, InvalidParameterException, LimitExceededException, AwsServiceException, SdkClientException,
            EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutAccountSettingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutAccountSettingResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putAccountSettingRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putAccountSettingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAccountSetting");

            return clientHandler.execute(new ClientExecutionParams<PutAccountSettingRequest, PutAccountSettingResponse>()
                    .withOperationName("PutAccountSetting").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putAccountSettingRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutAccountSettingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates the image manifest and tags associated with an image.
     * </p>
     * <p>
     * When an image is pushed and all new image layers have been uploaded, the PutImage API is called once to create or
     * update the image manifest and the tags associated with the image.
     * </p>
     * <note>
     * <p>
     * This operation is used by the Amazon ECR proxy and is not generally used by customers for pulling and pushing
     * images. In most cases, you should use the <code>docker</code> CLI to pull, tag, and push images.
     * </p>
     * </note>
     *
     * @param putImageRequest
     * @return Result of the PutImage operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws ImageAlreadyExistsException
     *         The specified image has already been pushed, and there were no changes to the manifest or image tag after
     *         the last push.
     * @throws LayersNotFoundException
     *         The specified layers could not be found, or the specified layer is not valid for this repository.
     * @throws ReferencedImagesNotFoundException
     *         The manifest list is referencing an image that does not exist.
     * @throws LimitExceededException
     *         The operation did not succeed because it would have exceeded a service limit for your account. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/service-quotas.html">Amazon ECR service
     *         quotas</a> in the Amazon Elastic Container Registry User Guide.
     * @throws ImageTagAlreadyExistsException
     *         The specified image is tagged with a tag that already exists. The repository is configured for tag
     *         immutability.
     * @throws ImageDigestDoesNotMatchException
     *         The specified image digest does not match the digest that Amazon ECR calculated for the image.
     * @throws KmsException
     *         The operation failed due to a KMS exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.PutImage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/PutImage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutImageResponse putImage(PutImageRequest putImageRequest) throws ServerException, InvalidParameterException,
            RepositoryNotFoundException, ImageAlreadyExistsException, LayersNotFoundException, ReferencedImagesNotFoundException,
            LimitExceededException, ImageTagAlreadyExistsException, ImageDigestDoesNotMatchException, KmsException,
            AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutImageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutImageResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putImageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putImageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutImage");

            return clientHandler.execute(new ClientExecutionParams<PutImageRequest, PutImageResponse>()
                    .withOperationName("PutImage").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(putImageRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutImageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <important>
     * <p>
     * The <code>PutImageScanningConfiguration</code> API is being deprecated, in favor of specifying the image scanning
     * configuration at the registry level. For more information, see <a>PutRegistryScanningConfiguration</a>.
     * </p>
     * </important>
     * <p>
     * Updates the image scanning configuration for the specified repository.
     * </p>
     *
     * @param putImageScanningConfigurationRequest
     * @return Result of the PutImageScanningConfiguration operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.PutImageScanningConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/PutImageScanningConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutImageScanningConfigurationResponse putImageScanningConfiguration(
            PutImageScanningConfigurationRequest putImageScanningConfigurationRequest) throws ServerException,
            InvalidParameterException, RepositoryNotFoundException, ValidationException, AwsServiceException, SdkClientException,
            EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutImageScanningConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutImageScanningConfigurationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putImageScanningConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putImageScanningConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutImageScanningConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<PutImageScanningConfigurationRequest, PutImageScanningConfigurationResponse>()
                            .withOperationName("PutImageScanningConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(putImageScanningConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutImageScanningConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the image tag mutability settings for the specified repository. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/image-tag-mutability.html">Image tag mutability</a>
     * in the <i>Amazon Elastic Container Registry User Guide</i>.
     * </p>
     *
     * @param putImageTagMutabilityRequest
     * @return Result of the PutImageTagMutability operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.PutImageTagMutability
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/PutImageTagMutability" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutImageTagMutabilityResponse putImageTagMutability(PutImageTagMutabilityRequest putImageTagMutabilityRequest)
            throws ServerException, InvalidParameterException, RepositoryNotFoundException, AwsServiceException,
            SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutImageTagMutabilityResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutImageTagMutabilityResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putImageTagMutabilityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putImageTagMutabilityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutImageTagMutability");

            return clientHandler.execute(new ClientExecutionParams<PutImageTagMutabilityRequest, PutImageTagMutabilityResponse>()
                    .withOperationName("PutImageTagMutability").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putImageTagMutabilityRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutImageTagMutabilityRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates the lifecycle policy for the specified repository. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/LifecyclePolicies.html">Lifecycle policy
     * template</a>.
     * </p>
     *
     * @param putLifecyclePolicyRequest
     * @return Result of the PutLifecyclePolicy operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.PutLifecyclePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/PutLifecyclePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutLifecyclePolicyResponse putLifecyclePolicy(PutLifecyclePolicyRequest putLifecyclePolicyRequest)
            throws ServerException, InvalidParameterException, RepositoryNotFoundException, ValidationException,
            AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutLifecyclePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutLifecyclePolicyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putLifecyclePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putLifecyclePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutLifecyclePolicy");

            return clientHandler.execute(new ClientExecutionParams<PutLifecyclePolicyRequest, PutLifecyclePolicyResponse>()
                    .withOperationName("PutLifecyclePolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putLifecyclePolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutLifecyclePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates the permissions policy for your registry.
     * </p>
     * <p>
     * A registry policy is used to specify permissions for another Amazon Web Services account and is used when
     * configuring cross-account replication. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/registry-permissions.html">Registry permissions</a>
     * in the <i>Amazon Elastic Container Registry User Guide</i>.
     * </p>
     *
     * @param putRegistryPolicyRequest
     * @return Result of the PutRegistryPolicy operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.PutRegistryPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/PutRegistryPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutRegistryPolicyResponse putRegistryPolicy(PutRegistryPolicyRequest putRegistryPolicyRequest) throws ServerException,
            InvalidParameterException, ValidationException, AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutRegistryPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutRegistryPolicyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putRegistryPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putRegistryPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRegistryPolicy");

            return clientHandler.execute(new ClientExecutionParams<PutRegistryPolicyRequest, PutRegistryPolicyResponse>()
                    .withOperationName("PutRegistryPolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putRegistryPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutRegistryPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates the scanning configuration for your private registry.
     * </p>
     *
     * @param putRegistryScanningConfigurationRequest
     * @return Result of the PutRegistryScanningConfiguration operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.PutRegistryScanningConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/PutRegistryScanningConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutRegistryScanningConfigurationResponse putRegistryScanningConfiguration(
            PutRegistryScanningConfigurationRequest putRegistryScanningConfigurationRequest) throws ServerException,
            InvalidParameterException, ValidationException, AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutRegistryScanningConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutRegistryScanningConfigurationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putRegistryScanningConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putRegistryScanningConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRegistryScanningConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<PutRegistryScanningConfigurationRequest, PutRegistryScanningConfigurationResponse>()
                            .withOperationName("PutRegistryScanningConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(putRegistryScanningConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutRegistryScanningConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates the replication configuration for a registry. The existing replication configuration for a
     * repository can be retrieved with the <a>DescribeRegistry</a> API action. The first time the
     * PutReplicationConfiguration API is called, a service-linked IAM role is created in your account for the
     * replication process. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/using-service-linked-roles.html">Using
     * service-linked roles for Amazon ECR</a> in the <i>Amazon Elastic Container Registry User Guide</i>. For more
     * information on the custom role for replication, see <a href=
     * "https://docs.aws.amazon.com/AmazonECR/latest/userguide/replication-creation-templates.html#roles-creatingrole-user-console"
     * >Creating an IAM role for replication</a>.
     * </p>
     * <note>
     * <p>
     * When configuring cross-account replication, the destination account must grant the source account permission to
     * replicate. This permission is controlled using a registry permissions policy. For more information, see
     * <a>PutRegistryPolicy</a>.
     * </p>
     * </note>
     *
     * @param putReplicationConfigurationRequest
     * @return Result of the PutReplicationConfiguration operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.PutReplicationConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/PutReplicationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutReplicationConfigurationResponse putReplicationConfiguration(
            PutReplicationConfigurationRequest putReplicationConfigurationRequest) throws ServerException,
            InvalidParameterException, ValidationException, AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutReplicationConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutReplicationConfigurationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putReplicationConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putReplicationConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutReplicationConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<PutReplicationConfigurationRequest, PutReplicationConfigurationResponse>()
                            .withOperationName("PutReplicationConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(putReplicationConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutReplicationConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Applies a repository policy to the specified repository to control access permissions. For more information, see
     * <a href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/repository-policies.html">Amazon ECR Repository
     * policies</a> in the <i>Amazon Elastic Container Registry User Guide</i>.
     * </p>
     *
     * @param setRepositoryPolicyRequest
     * @return Result of the SetRepositoryPolicy operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.SetRepositoryPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/SetRepositoryPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SetRepositoryPolicyResponse setRepositoryPolicy(SetRepositoryPolicyRequest setRepositoryPolicyRequest)
            throws ServerException, InvalidParameterException, RepositoryNotFoundException, AwsServiceException,
            SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SetRepositoryPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SetRepositoryPolicyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(setRepositoryPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setRepositoryPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetRepositoryPolicy");

            return clientHandler.execute(new ClientExecutionParams<SetRepositoryPolicyRequest, SetRepositoryPolicyResponse>()
                    .withOperationName("SetRepositoryPolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(setRepositoryPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SetRepositoryPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a basic image vulnerability scan.
     * </p>
     * <p>
     * A basic image scan can only be started once per 24 hours on an individual image. This limit includes if an image
     * was scanned on initial push. You can start up to 100,000 basic scans per 24 hours. This limit includes both scans
     * on initial push and scans initiated by the StartImageScan API. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/image-scanning-basic.html">Basic scanning</a> in the
     * <i>Amazon Elastic Container Registry User Guide</i>.
     * </p>
     *
     * @param startImageScanRequest
     * @return Result of the StartImageScan operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws UnsupportedImageTypeException
     *         The image is of a type that cannot be scanned.
     * @throws LimitExceededException
     *         The operation did not succeed because it would have exceeded a service limit for your account. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/service-quotas.html">Amazon ECR service
     *         quotas</a> in the Amazon Elastic Container Registry User Guide.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws ImageNotFoundException
     *         The image requested does not exist in the specified repository.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.StartImageScan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/StartImageScan" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartImageScanResponse startImageScan(StartImageScanRequest startImageScanRequest) throws ServerException,
            InvalidParameterException, UnsupportedImageTypeException, LimitExceededException, RepositoryNotFoundException,
            ImageNotFoundException, ValidationException, AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartImageScanResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartImageScanResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startImageScanRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startImageScanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartImageScan");

            return clientHandler.execute(new ClientExecutionParams<StartImageScanRequest, StartImageScanResponse>()
                    .withOperationName("StartImageScan").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(startImageScanRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartImageScanRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a preview of a lifecycle policy for the specified repository. This allows you to see the results before
     * associating the lifecycle policy with the repository.
     * </p>
     *
     * @param startLifecyclePolicyPreviewRequest
     * @return Result of the StartLifecyclePolicyPreview operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws LifecyclePolicyNotFoundException
     *         The lifecycle policy could not be found, and no policy is set to the repository.
     * @throws LifecyclePolicyPreviewInProgressException
     *         The previous lifecycle policy preview request has not completed. Wait and try again.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.StartLifecyclePolicyPreview
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/StartLifecyclePolicyPreview"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartLifecyclePolicyPreviewResponse startLifecyclePolicyPreview(
            StartLifecyclePolicyPreviewRequest startLifecyclePolicyPreviewRequest) throws ServerException,
            InvalidParameterException, RepositoryNotFoundException, LifecyclePolicyNotFoundException,
            LifecyclePolicyPreviewInProgressException, ValidationException, AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartLifecyclePolicyPreviewResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartLifecyclePolicyPreviewResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startLifecyclePolicyPreviewRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startLifecyclePolicyPreviewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartLifecyclePolicyPreview");

            return clientHandler
                    .execute(new ClientExecutionParams<StartLifecyclePolicyPreviewRequest, StartLifecyclePolicyPreviewResponse>()
                            .withOperationName("StartLifecyclePolicyPreview").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(startLifecyclePolicyPreviewRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartLifecyclePolicyPreviewRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds specified tags to a resource with the specified ARN. Existing tags on a resource are not changed if they are
     * not specified in the request parameters.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws InvalidTagParameterException
     *         An invalid parameter has been specified. Tag keys can have a maximum character length of 128 characters,
     *         and tag values can have a maximum length of 256 characters.
     * @throws TooManyTagsException
     *         The list of tags on the repository is over the limit. The maximum number of tags that can be applied to a
     *         repository is 50.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InvalidParameterException,
            InvalidTagParameterException, TooManyTagsException, RepositoryNotFoundException, ServerException,
            AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes specified tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws InvalidTagParameterException
     *         An invalid parameter has been specified. Tag keys can have a maximum character length of 128 characters,
     *         and tag values can have a maximum length of 256 characters.
     * @throws TooManyTagsException
     *         The list of tags on the repository is over the limit. The maximum number of tags that can be applied to a
     *         repository is 50.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InvalidParameterException,
            InvalidTagParameterException, TooManyTagsException, RepositoryNotFoundException, ServerException,
            AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing pull through cache rule.
     * </p>
     *
     * @param updatePullThroughCacheRuleRequest
     * @return Result of the UpdatePullThroughCacheRule operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws UnableToAccessSecretException
     *         The secret is unable to be accessed. Verify the resource permissions for the secret and try again.
     * @throws PullThroughCacheRuleNotFoundException
     *         The pull through cache rule was not found. Specify a valid pull through cache rule and try again.
     * @throws SecretNotFoundException
     *         The ARN of the secret specified in the pull through cache rule was not found. Update the pull through
     *         cache rule with a valid secret ARN and try again.
     * @throws UnableToDecryptSecretValueException
     *         The secret is accessible but is unable to be decrypted. Verify the resource permisisons and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.UpdatePullThroughCacheRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/UpdatePullThroughCacheRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdatePullThroughCacheRuleResponse updatePullThroughCacheRule(
            UpdatePullThroughCacheRuleRequest updatePullThroughCacheRuleRequest) throws ServerException,
            InvalidParameterException, ValidationException, UnableToAccessSecretException, PullThroughCacheRuleNotFoundException,
            SecretNotFoundException, UnableToDecryptSecretValueException, AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdatePullThroughCacheRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdatePullThroughCacheRuleResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePullThroughCacheRuleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePullThroughCacheRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePullThroughCacheRule");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdatePullThroughCacheRuleRequest, UpdatePullThroughCacheRuleResponse>()
                            .withOperationName("UpdatePullThroughCacheRule").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updatePullThroughCacheRuleRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdatePullThroughCacheRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing repository creation template.
     * </p>
     *
     * @param updateRepositoryCreationTemplateRequest
     * @return Result of the UpdateRepositoryCreationTemplate operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws TemplateNotFoundException
     *         The specified repository creation template can't be found. Verify the registry ID and prefix and try
     *         again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.UpdateRepositoryCreationTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/UpdateRepositoryCreationTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateRepositoryCreationTemplateResponse updateRepositoryCreationTemplate(
            UpdateRepositoryCreationTemplateRequest updateRepositoryCreationTemplateRequest) throws ServerException,
            ValidationException, InvalidParameterException, TemplateNotFoundException, AwsServiceException, SdkClientException,
            EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateRepositoryCreationTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateRepositoryCreationTemplateResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateRepositoryCreationTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateRepositoryCreationTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRepositoryCreationTemplate");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateRepositoryCreationTemplateRequest, UpdateRepositoryCreationTemplateResponse>()
                            .withOperationName("UpdateRepositoryCreationTemplate").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateRepositoryCreationTemplateRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateRepositoryCreationTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Uploads an image layer part to Amazon ECR.
     * </p>
     * <p>
     * When an image is pushed, each new image layer is uploaded in parts. The maximum size of each image layer part can
     * be 20971520 bytes (or about 20MB). The UploadLayerPart API is called once per each new image layer part.
     * </p>
     * <note>
     * <p>
     * This operation is used by the Amazon ECR proxy and is not generally used by customers for pulling and pushing
     * images. In most cases, you should use the <code>docker</code> CLI to pull, tag, and push images.
     * </p>
     * </note>
     *
     * @param uploadLayerPartRequest
     * @return Result of the UploadLayerPart operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws InvalidLayerPartException
     *         The layer part size is not valid, or the first byte specified is not consecutive to the last byte of a
     *         previous layer part upload.
     * @throws RepositoryNotFoundException
     *         The specified repository could not be found. Check the spelling of the specified repository and ensure
     *         that you are performing operations on the correct registry.
     * @throws UploadNotFoundException
     *         The upload could not be found, or the specified upload ID is not valid for this repository.
     * @throws LimitExceededException
     *         The operation did not succeed because it would have exceeded a service limit for your account. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/service-quotas.html">Amazon ECR service
     *         quotas</a> in the Amazon Elastic Container Registry User Guide.
     * @throws KmsException
     *         The operation failed due to a KMS exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.UploadLayerPart
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/UploadLayerPart" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UploadLayerPartResponse uploadLayerPart(UploadLayerPartRequest uploadLayerPartRequest) throws ServerException,
            InvalidParameterException, InvalidLayerPartException, RepositoryNotFoundException, UploadNotFoundException,
            LimitExceededException, KmsException, AwsServiceException, SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UploadLayerPartResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UploadLayerPartResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(uploadLayerPartRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, uploadLayerPartRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UploadLayerPart");

            return clientHandler.execute(new ClientExecutionParams<UploadLayerPartRequest, UploadLayerPartResponse>()
                    .withOperationName("UploadLayerPart").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(uploadLayerPartRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UploadLayerPartRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Validates an existing pull through cache rule for an upstream registry that requires authentication. This will
     * retrieve the contents of the Amazon Web Services Secrets Manager secret, verify the syntax, and then validate
     * that authentication to the upstream registry is successful.
     * </p>
     *
     * @param validatePullThroughCacheRuleRequest
     * @return Result of the ValidatePullThroughCacheRule operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ValidationException
     *         There was an exception validating this request.
     * @throws PullThroughCacheRuleNotFoundException
     *         The pull through cache rule was not found. Specify a valid pull through cache rule and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrClient.ValidatePullThroughCacheRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-2015-09-21/ValidatePullThroughCacheRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ValidatePullThroughCacheRuleResponse validatePullThroughCacheRule(
            ValidatePullThroughCacheRuleRequest validatePullThroughCacheRuleRequest) throws ServerException,
            InvalidParameterException, ValidationException, PullThroughCacheRuleNotFoundException, AwsServiceException,
            SdkClientException, EcrException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ValidatePullThroughCacheRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ValidatePullThroughCacheRuleResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "LayerPartTooSmallException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerPartTooSmallException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerPartTooSmallException::builder).build());
            case "RepositoryNotEmptyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).build());
            case "LayerAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).build());
            case "UnableToAccessSecretException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToAccessSecretException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToAccessSecretException::builder).build());
            case "EmptyUploadException":
                return Optional.of(ExceptionMetadata.builder().errorCode("EmptyUploadException").httpStatusCode(400)
                        .exceptionBuilderSupplier(EmptyUploadException::builder).build());
            case "TemplateAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateAlreadyExistsException::builder).build());
            case "TemplateNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TemplateNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TemplateNotFoundException::builder).build());
            case "RepositoryAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).build());
            case "RepositoryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).build());
            case "UnsupportedUpstreamRegistryException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedUpstreamRegistryException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedUpstreamRegistryException::builder).build());
            case "LayerInaccessibleException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayerInaccessibleException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayerInaccessibleException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "InvalidLayerPartException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerPartException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerPartException::builder).build());
            case "ImageTagAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).build());
            case "LifecyclePolicyPreviewNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewNotFoundException::builder).build());
            case "ScanNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ScanNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ScanNotFoundException::builder).build());
            case "LifecyclePolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LifecyclePolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LifecyclePolicyNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "RegistryPolicyNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RegistryPolicyNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RegistryPolicyNotFoundException::builder).build());
            case "ImageNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageNotFoundException::builder).build());
            case "LayersNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LayersNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LayersNotFoundException::builder).build());
            case "RepositoryNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RepositoryNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RepositoryNotFoundException::builder).build());
            case "UnsupportedImageTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedImageTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedImageTypeException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "UnableToGetUpstreamImageException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamImageException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamImageException::builder).build());
            case "KmsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("KmsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(KmsException::builder).build());
            case "ReferencedImagesNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).build());
            case "UnableToDecryptSecretValueException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToDecryptSecretValueException")
                        .httpStatusCode(400).exceptionBuilderSupplier(UnableToDecryptSecretValueException::builder).build());
            case "ImageAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).build());
            case "UploadNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UploadNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UploadNotFoundException::builder).build());
            case "TooManyTagsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyTagsException::builder).build());
            case "SecretNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SecretNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SecretNotFoundException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "ImageDigestDoesNotMatchException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).build());
            case "UnableToGetUpstreamLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnableToGetUpstreamLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnableToGetUpstreamLayerException::builder).build());
            case "InvalidTagParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidTagParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidTagParameterException::builder).build());
            case "InvalidLayerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidLayerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidLayerException::builder).build());
            case "PullThroughCacheRuleNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleNotFoundException::builder).build());
            case "LifecyclePolicyPreviewInProgressException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("LifecyclePolicyPreviewInProgressException")
                                .httpStatusCode(400).exceptionBuilderSupplier(LifecyclePolicyPreviewInProgressException::builder)
                                .build());
            case "PullThroughCacheRuleAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PullThroughCacheRuleAlreadyExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PullThroughCacheRuleAlreadyExistsException::builder)
                        .build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(validatePullThroughCacheRuleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, validatePullThroughCacheRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ValidatePullThroughCacheRule");

            return clientHandler
                    .execute(new ClientExecutionParams<ValidatePullThroughCacheRuleRequest, ValidatePullThroughCacheRuleResponse>()
                            .withOperationName("ValidatePullThroughCacheRule").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(validatePullThroughCacheRuleRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ValidatePullThroughCacheRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Create an instance of {@link EcrWaiter} using this client.
     * <p>
     * Waiters created via this method are managed by the SDK and resources will be released when the service client is
     * closed.
     *
     * @return an instance of {@link EcrWaiter}
     */
    @Override
    public EcrWaiter waiter() {
        return EcrWaiter.builder().client(this).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        EcrServiceClientConfigurationBuilder serviceConfigBuilder = new EcrServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(EcrException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON).protocolVersion("1.1");
    }

    @Override
    public final EcrServiceClientConfiguration serviceClientConfiguration() {
        return new EcrServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
