/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecr.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The details of the repository creation template associated with the request.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RepositoryCreationTemplate implements SdkPojo, Serializable,
        ToCopyableBuilder<RepositoryCreationTemplate.Builder, RepositoryCreationTemplate> {
    private static final SdkField<String> PREFIX_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("prefix")
            .getter(getter(RepositoryCreationTemplate::prefix)).setter(setter(Builder::prefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("prefix").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(RepositoryCreationTemplate::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<EncryptionConfigurationForRepositoryCreationTemplate> ENCRYPTION_CONFIGURATION_FIELD = SdkField
            .<EncryptionConfigurationForRepositoryCreationTemplate> builder(MarshallingType.SDK_POJO)
            .memberName("encryptionConfiguration").getter(getter(RepositoryCreationTemplate::encryptionConfiguration))
            .setter(setter(Builder::encryptionConfiguration))
            .constructor(EncryptionConfigurationForRepositoryCreationTemplate::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("encryptionConfiguration").build())
            .build();

    private static final SdkField<List<Tag>> RESOURCE_TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("resourceTags")
            .getter(getter(RepositoryCreationTemplate::resourceTags))
            .setter(setter(Builder::resourceTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceTags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> IMAGE_TAG_MUTABILITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("imageTagMutability").getter(getter(RepositoryCreationTemplate::imageTagMutabilityAsString))
            .setter(setter(Builder::imageTagMutability))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("imageTagMutability").build())
            .build();

    private static final SdkField<String> REPOSITORY_POLICY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("repositoryPolicy").getter(getter(RepositoryCreationTemplate::repositoryPolicy))
            .setter(setter(Builder::repositoryPolicy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("repositoryPolicy").build()).build();

    private static final SdkField<String> LIFECYCLE_POLICY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("lifecyclePolicy").getter(getter(RepositoryCreationTemplate::lifecyclePolicy))
            .setter(setter(Builder::lifecyclePolicy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lifecyclePolicy").build()).build();

    private static final SdkField<List<String>> APPLIED_FOR_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("appliedFor")
            .getter(getter(RepositoryCreationTemplate::appliedForAsStrings))
            .setter(setter(Builder::appliedForWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("appliedFor").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CUSTOM_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("customRoleArn").getter(getter(RepositoryCreationTemplate::customRoleArn))
            .setter(setter(Builder::customRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("customRoleArn").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt").getter(getter(RepositoryCreationTemplate::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<Instant> UPDATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("updatedAt").getter(getter(RepositoryCreationTemplate::updatedAt)).setter(setter(Builder::updatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("updatedAt").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PREFIX_FIELD,
            DESCRIPTION_FIELD, ENCRYPTION_CONFIGURATION_FIELD, RESOURCE_TAGS_FIELD, IMAGE_TAG_MUTABILITY_FIELD,
            REPOSITORY_POLICY_FIELD, LIFECYCLE_POLICY_FIELD, APPLIED_FOR_FIELD, CUSTOM_ROLE_ARN_FIELD, CREATED_AT_FIELD,
            UPDATED_AT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String prefix;

    private final String description;

    private final EncryptionConfigurationForRepositoryCreationTemplate encryptionConfiguration;

    private final List<Tag> resourceTags;

    private final String imageTagMutability;

    private final String repositoryPolicy;

    private final String lifecyclePolicy;

    private final List<String> appliedFor;

    private final String customRoleArn;

    private final Instant createdAt;

    private final Instant updatedAt;

    private RepositoryCreationTemplate(BuilderImpl builder) {
        this.prefix = builder.prefix;
        this.description = builder.description;
        this.encryptionConfiguration = builder.encryptionConfiguration;
        this.resourceTags = builder.resourceTags;
        this.imageTagMutability = builder.imageTagMutability;
        this.repositoryPolicy = builder.repositoryPolicy;
        this.lifecyclePolicy = builder.lifecyclePolicy;
        this.appliedFor = builder.appliedFor;
        this.customRoleArn = builder.customRoleArn;
        this.createdAt = builder.createdAt;
        this.updatedAt = builder.updatedAt;
    }

    /**
     * <p>
     * The repository namespace prefix associated with the repository creation template.
     * </p>
     * 
     * @return The repository namespace prefix associated with the repository creation template.
     */
    public final String prefix() {
        return prefix;
    }

    /**
     * <p>
     * The description associated with the repository creation template.
     * </p>
     * 
     * @return The description associated with the repository creation template.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The encryption configuration associated with the repository creation template.
     * </p>
     * 
     * @return The encryption configuration associated with the repository creation template.
     */
    public final EncryptionConfigurationForRepositoryCreationTemplate encryptionConfiguration() {
        return encryptionConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceTags property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceTags() {
        return resourceTags != null && !(resourceTags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The metadata to apply to the repository to help you categorize and organize. Each tag consists of a key and an
     * optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag
     * values can have a maximum length of 256 characters.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceTags} method.
     * </p>
     * 
     * @return The metadata to apply to the repository to help you categorize and organize. Each tag consists of a key
     *         and an optional value, both of which you define. Tag keys can have a maximum character length of 128
     *         characters, and tag values can have a maximum length of 256 characters.
     */
    public final List<Tag> resourceTags() {
        return resourceTags;
    }

    /**
     * <p>
     * The tag mutability setting for the repository. If this parameter is omitted, the default setting of MUTABLE will
     * be used which will allow image tags to be overwritten. If IMMUTABLE is specified, all image tags within the
     * repository will be immutable which will prevent them from being overwritten.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #imageTagMutability} will return {@link ImageTagMutability#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #imageTagMutabilityAsString}.
     * </p>
     * 
     * @return The tag mutability setting for the repository. If this parameter is omitted, the default setting of
     *         MUTABLE will be used which will allow image tags to be overwritten. If IMMUTABLE is specified, all image
     *         tags within the repository will be immutable which will prevent them from being overwritten.
     * @see ImageTagMutability
     */
    public final ImageTagMutability imageTagMutability() {
        return ImageTagMutability.fromValue(imageTagMutability);
    }

    /**
     * <p>
     * The tag mutability setting for the repository. If this parameter is omitted, the default setting of MUTABLE will
     * be used which will allow image tags to be overwritten. If IMMUTABLE is specified, all image tags within the
     * repository will be immutable which will prevent them from being overwritten.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #imageTagMutability} will return {@link ImageTagMutability#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #imageTagMutabilityAsString}.
     * </p>
     * 
     * @return The tag mutability setting for the repository. If this parameter is omitted, the default setting of
     *         MUTABLE will be used which will allow image tags to be overwritten. If IMMUTABLE is specified, all image
     *         tags within the repository will be immutable which will prevent them from being overwritten.
     * @see ImageTagMutability
     */
    public final String imageTagMutabilityAsString() {
        return imageTagMutability;
    }

    /**
     * <p>
     * The repository policy to apply to repositories created using the template. A repository policy is a permissions
     * policy associated with a repository to control access permissions.
     * </p>
     * 
     * @return The repository policy to apply to repositories created using the template. A repository policy is a
     *         permissions policy associated with a repository to control access permissions.
     */
    public final String repositoryPolicy() {
        return repositoryPolicy;
    }

    /**
     * <p>
     * The lifecycle policy to use for repositories created using the template.
     * </p>
     * 
     * @return The lifecycle policy to use for repositories created using the template.
     */
    public final String lifecyclePolicy() {
        return lifecyclePolicy;
    }

    /**
     * <p>
     * A list of enumerable Strings representing the repository creation scenarios that this template will apply
     * towards. The two supported scenarios are PULL_THROUGH_CACHE and REPLICATION
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAppliedFor} method.
     * </p>
     * 
     * @return A list of enumerable Strings representing the repository creation scenarios that this template will apply
     *         towards. The two supported scenarios are PULL_THROUGH_CACHE and REPLICATION
     */
    public final List<RCTAppliedFor> appliedFor() {
        return RCTAppliedForListCopier.copyStringToEnum(appliedFor);
    }

    /**
     * For responses, this returns true if the service returned a value for the AppliedFor property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAppliedFor() {
        return appliedFor != null && !(appliedFor instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of enumerable Strings representing the repository creation scenarios that this template will apply
     * towards. The two supported scenarios are PULL_THROUGH_CACHE and REPLICATION
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAppliedFor} method.
     * </p>
     * 
     * @return A list of enumerable Strings representing the repository creation scenarios that this template will apply
     *         towards. The two supported scenarios are PULL_THROUGH_CACHE and REPLICATION
     */
    public final List<String> appliedForAsStrings() {
        return appliedFor;
    }

    /**
     * <p>
     * The ARN of the role to be assumed by Amazon ECR. Amazon ECR will assume your supplied role when the customRoleArn
     * is specified. When this field isn't specified, Amazon ECR will use the service-linked role for the repository
     * creation template.
     * </p>
     * 
     * @return The ARN of the role to be assumed by Amazon ECR. Amazon ECR will assume your supplied role when the
     *         customRoleArn is specified. When this field isn't specified, Amazon ECR will use the service-linked role
     *         for the repository creation template.
     */
    public final String customRoleArn() {
        return customRoleArn;
    }

    /**
     * <p>
     * The date and time, in JavaScript date format, when the repository creation template was created.
     * </p>
     * 
     * @return The date and time, in JavaScript date format, when the repository creation template was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The date and time, in JavaScript date format, when the repository creation template was last updated.
     * </p>
     * 
     * @return The date and time, in JavaScript date format, when the repository creation template was last updated.
     */
    public final Instant updatedAt() {
        return updatedAt;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(prefix());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceTags() ? resourceTags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(imageTagMutabilityAsString());
        hashCode = 31 * hashCode + Objects.hashCode(repositoryPolicy());
        hashCode = 31 * hashCode + Objects.hashCode(lifecyclePolicy());
        hashCode = 31 * hashCode + Objects.hashCode(hasAppliedFor() ? appliedForAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(customRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(updatedAt());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RepositoryCreationTemplate)) {
            return false;
        }
        RepositoryCreationTemplate other = (RepositoryCreationTemplate) obj;
        return Objects.equals(prefix(), other.prefix()) && Objects.equals(description(), other.description())
                && Objects.equals(encryptionConfiguration(), other.encryptionConfiguration())
                && hasResourceTags() == other.hasResourceTags() && Objects.equals(resourceTags(), other.resourceTags())
                && Objects.equals(imageTagMutabilityAsString(), other.imageTagMutabilityAsString())
                && Objects.equals(repositoryPolicy(), other.repositoryPolicy())
                && Objects.equals(lifecyclePolicy(), other.lifecyclePolicy()) && hasAppliedFor() == other.hasAppliedFor()
                && Objects.equals(appliedForAsStrings(), other.appliedForAsStrings())
                && Objects.equals(customRoleArn(), other.customRoleArn()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(updatedAt(), other.updatedAt());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RepositoryCreationTemplate").add("Prefix", prefix()).add("Description", description())
                .add("EncryptionConfiguration", encryptionConfiguration())
                .add("ResourceTags", hasResourceTags() ? resourceTags() : null)
                .add("ImageTagMutability", imageTagMutabilityAsString()).add("RepositoryPolicy", repositoryPolicy())
                .add("LifecyclePolicy", lifecyclePolicy()).add("AppliedFor", hasAppliedFor() ? appliedForAsStrings() : null)
                .add("CustomRoleArn", customRoleArn()).add("CreatedAt", createdAt()).add("UpdatedAt", updatedAt()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "prefix":
            return Optional.ofNullable(clazz.cast(prefix()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "encryptionConfiguration":
            return Optional.ofNullable(clazz.cast(encryptionConfiguration()));
        case "resourceTags":
            return Optional.ofNullable(clazz.cast(resourceTags()));
        case "imageTagMutability":
            return Optional.ofNullable(clazz.cast(imageTagMutabilityAsString()));
        case "repositoryPolicy":
            return Optional.ofNullable(clazz.cast(repositoryPolicy()));
        case "lifecyclePolicy":
            return Optional.ofNullable(clazz.cast(lifecyclePolicy()));
        case "appliedFor":
            return Optional.ofNullable(clazz.cast(appliedForAsStrings()));
        case "customRoleArn":
            return Optional.ofNullable(clazz.cast(customRoleArn()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "updatedAt":
            return Optional.ofNullable(clazz.cast(updatedAt()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("prefix", PREFIX_FIELD);
        map.put("description", DESCRIPTION_FIELD);
        map.put("encryptionConfiguration", ENCRYPTION_CONFIGURATION_FIELD);
        map.put("resourceTags", RESOURCE_TAGS_FIELD);
        map.put("imageTagMutability", IMAGE_TAG_MUTABILITY_FIELD);
        map.put("repositoryPolicy", REPOSITORY_POLICY_FIELD);
        map.put("lifecyclePolicy", LIFECYCLE_POLICY_FIELD);
        map.put("appliedFor", APPLIED_FOR_FIELD);
        map.put("customRoleArn", CUSTOM_ROLE_ARN_FIELD);
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("updatedAt", UPDATED_AT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RepositoryCreationTemplate, T> g) {
        return obj -> g.apply((RepositoryCreationTemplate) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RepositoryCreationTemplate> {
        /**
         * <p>
         * The repository namespace prefix associated with the repository creation template.
         * </p>
         * 
         * @param prefix
         *        The repository namespace prefix associated with the repository creation template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder prefix(String prefix);

        /**
         * <p>
         * The description associated with the repository creation template.
         * </p>
         * 
         * @param description
         *        The description associated with the repository creation template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The encryption configuration associated with the repository creation template.
         * </p>
         * 
         * @param encryptionConfiguration
         *        The encryption configuration associated with the repository creation template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionConfiguration(EncryptionConfigurationForRepositoryCreationTemplate encryptionConfiguration);

        /**
         * <p>
         * The encryption configuration associated with the repository creation template.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link EncryptionConfigurationForRepositoryCreationTemplate.Builder} avoiding the need to create one manually
         * via {@link EncryptionConfigurationForRepositoryCreationTemplate#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link EncryptionConfigurationForRepositoryCreationTemplate.Builder#build()} is called immediately and its
         * result is passed to {@link #encryptionConfiguration(EncryptionConfigurationForRepositoryCreationTemplate)}.
         * 
         * @param encryptionConfiguration
         *        a consumer that will call methods on
         *        {@link EncryptionConfigurationForRepositoryCreationTemplate.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #encryptionConfiguration(EncryptionConfigurationForRepositoryCreationTemplate)
         */
        default Builder encryptionConfiguration(
                Consumer<EncryptionConfigurationForRepositoryCreationTemplate.Builder> encryptionConfiguration) {
            return encryptionConfiguration(EncryptionConfigurationForRepositoryCreationTemplate.builder()
                    .applyMutation(encryptionConfiguration).build());
        }

        /**
         * <p>
         * The metadata to apply to the repository to help you categorize and organize. Each tag consists of a key and
         * an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters,
         * and tag values can have a maximum length of 256 characters.
         * </p>
         * 
         * @param resourceTags
         *        The metadata to apply to the repository to help you categorize and organize. Each tag consists of a
         *        key and an optional value, both of which you define. Tag keys can have a maximum character length of
         *        128 characters, and tag values can have a maximum length of 256 characters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTags(Collection<Tag> resourceTags);

        /**
         * <p>
         * The metadata to apply to the repository to help you categorize and organize. Each tag consists of a key and
         * an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters,
         * and tag values can have a maximum length of 256 characters.
         * </p>
         * 
         * @param resourceTags
         *        The metadata to apply to the repository to help you categorize and organize. Each tag consists of a
         *        key and an optional value, both of which you define. Tag keys can have a maximum character length of
         *        128 characters, and tag values can have a maximum length of 256 characters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTags(Tag... resourceTags);

        /**
         * <p>
         * The metadata to apply to the repository to help you categorize and organize. Each tag consists of a key and
         * an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters,
         * and tag values can have a maximum length of 256 characters.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecr.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ecr.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ecr.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #resourceTags(List<Tag>)}.
         * 
         * @param resourceTags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ecr.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceTags(java.util.Collection<Tag>)
         */
        Builder resourceTags(Consumer<Tag.Builder>... resourceTags);

        /**
         * <p>
         * The tag mutability setting for the repository. If this parameter is omitted, the default setting of MUTABLE
         * will be used which will allow image tags to be overwritten. If IMMUTABLE is specified, all image tags within
         * the repository will be immutable which will prevent them from being overwritten.
         * </p>
         * 
         * @param imageTagMutability
         *        The tag mutability setting for the repository. If this parameter is omitted, the default setting of
         *        MUTABLE will be used which will allow image tags to be overwritten. If IMMUTABLE is specified, all
         *        image tags within the repository will be immutable which will prevent them from being overwritten.
         * @see ImageTagMutability
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ImageTagMutability
         */
        Builder imageTagMutability(String imageTagMutability);

        /**
         * <p>
         * The tag mutability setting for the repository. If this parameter is omitted, the default setting of MUTABLE
         * will be used which will allow image tags to be overwritten. If IMMUTABLE is specified, all image tags within
         * the repository will be immutable which will prevent them from being overwritten.
         * </p>
         * 
         * @param imageTagMutability
         *        The tag mutability setting for the repository. If this parameter is omitted, the default setting of
         *        MUTABLE will be used which will allow image tags to be overwritten. If IMMUTABLE is specified, all
         *        image tags within the repository will be immutable which will prevent them from being overwritten.
         * @see ImageTagMutability
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ImageTagMutability
         */
        Builder imageTagMutability(ImageTagMutability imageTagMutability);

        /**
         * <p>
         * The repository policy to apply to repositories created using the template. A repository policy is a
         * permissions policy associated with a repository to control access permissions.
         * </p>
         * 
         * @param repositoryPolicy
         *        The repository policy to apply to repositories created using the template. A repository policy is a
         *        permissions policy associated with a repository to control access permissions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder repositoryPolicy(String repositoryPolicy);

        /**
         * <p>
         * The lifecycle policy to use for repositories created using the template.
         * </p>
         * 
         * @param lifecyclePolicy
         *        The lifecycle policy to use for repositories created using the template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lifecyclePolicy(String lifecyclePolicy);

        /**
         * <p>
         * A list of enumerable Strings representing the repository creation scenarios that this template will apply
         * towards. The two supported scenarios are PULL_THROUGH_CACHE and REPLICATION
         * </p>
         * 
         * @param appliedFor
         *        A list of enumerable Strings representing the repository creation scenarios that this template will
         *        apply towards. The two supported scenarios are PULL_THROUGH_CACHE and REPLICATION
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appliedForWithStrings(Collection<String> appliedFor);

        /**
         * <p>
         * A list of enumerable Strings representing the repository creation scenarios that this template will apply
         * towards. The two supported scenarios are PULL_THROUGH_CACHE and REPLICATION
         * </p>
         * 
         * @param appliedFor
         *        A list of enumerable Strings representing the repository creation scenarios that this template will
         *        apply towards. The two supported scenarios are PULL_THROUGH_CACHE and REPLICATION
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appliedForWithStrings(String... appliedFor);

        /**
         * <p>
         * A list of enumerable Strings representing the repository creation scenarios that this template will apply
         * towards. The two supported scenarios are PULL_THROUGH_CACHE and REPLICATION
         * </p>
         * 
         * @param appliedFor
         *        A list of enumerable Strings representing the repository creation scenarios that this template will
         *        apply towards. The two supported scenarios are PULL_THROUGH_CACHE and REPLICATION
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appliedFor(Collection<RCTAppliedFor> appliedFor);

        /**
         * <p>
         * A list of enumerable Strings representing the repository creation scenarios that this template will apply
         * towards. The two supported scenarios are PULL_THROUGH_CACHE and REPLICATION
         * </p>
         * 
         * @param appliedFor
         *        A list of enumerable Strings representing the repository creation scenarios that this template will
         *        apply towards. The two supported scenarios are PULL_THROUGH_CACHE and REPLICATION
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appliedFor(RCTAppliedFor... appliedFor);

        /**
         * <p>
         * The ARN of the role to be assumed by Amazon ECR. Amazon ECR will assume your supplied role when the
         * customRoleArn is specified. When this field isn't specified, Amazon ECR will use the service-linked role for
         * the repository creation template.
         * </p>
         * 
         * @param customRoleArn
         *        The ARN of the role to be assumed by Amazon ECR. Amazon ECR will assume your supplied role when the
         *        customRoleArn is specified. When this field isn't specified, Amazon ECR will use the service-linked
         *        role for the repository creation template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customRoleArn(String customRoleArn);

        /**
         * <p>
         * The date and time, in JavaScript date format, when the repository creation template was created.
         * </p>
         * 
         * @param createdAt
         *        The date and time, in JavaScript date format, when the repository creation template was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The date and time, in JavaScript date format, when the repository creation template was last updated.
         * </p>
         * 
         * @param updatedAt
         *        The date and time, in JavaScript date format, when the repository creation template was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedAt(Instant updatedAt);
    }

    static final class BuilderImpl implements Builder {
        private String prefix;

        private String description;

        private EncryptionConfigurationForRepositoryCreationTemplate encryptionConfiguration;

        private List<Tag> resourceTags = DefaultSdkAutoConstructList.getInstance();

        private String imageTagMutability;

        private String repositoryPolicy;

        private String lifecyclePolicy;

        private List<String> appliedFor = DefaultSdkAutoConstructList.getInstance();

        private String customRoleArn;

        private Instant createdAt;

        private Instant updatedAt;

        private BuilderImpl() {
        }

        private BuilderImpl(RepositoryCreationTemplate model) {
            prefix(model.prefix);
            description(model.description);
            encryptionConfiguration(model.encryptionConfiguration);
            resourceTags(model.resourceTags);
            imageTagMutability(model.imageTagMutability);
            repositoryPolicy(model.repositoryPolicy);
            lifecyclePolicy(model.lifecyclePolicy);
            appliedForWithStrings(model.appliedFor);
            customRoleArn(model.customRoleArn);
            createdAt(model.createdAt);
            updatedAt(model.updatedAt);
        }

        public final String getPrefix() {
            return prefix;
        }

        public final void setPrefix(String prefix) {
            this.prefix = prefix;
        }

        @Override
        public final Builder prefix(String prefix) {
            this.prefix = prefix;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final EncryptionConfigurationForRepositoryCreationTemplate.Builder getEncryptionConfiguration() {
            return encryptionConfiguration != null ? encryptionConfiguration.toBuilder() : null;
        }

        public final void setEncryptionConfiguration(
                EncryptionConfigurationForRepositoryCreationTemplate.BuilderImpl encryptionConfiguration) {
            this.encryptionConfiguration = encryptionConfiguration != null ? encryptionConfiguration.build() : null;
        }

        @Override
        public final Builder encryptionConfiguration(EncryptionConfigurationForRepositoryCreationTemplate encryptionConfiguration) {
            this.encryptionConfiguration = encryptionConfiguration;
            return this;
        }

        public final List<Tag.Builder> getResourceTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.resourceTags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setResourceTags(Collection<Tag.BuilderImpl> resourceTags) {
            this.resourceTags = TagListCopier.copyFromBuilder(resourceTags);
        }

        @Override
        public final Builder resourceTags(Collection<Tag> resourceTags) {
            this.resourceTags = TagListCopier.copy(resourceTags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTags(Tag... resourceTags) {
            resourceTags(Arrays.asList(resourceTags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTags(Consumer<Tag.Builder>... resourceTags) {
            resourceTags(Stream.of(resourceTags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getImageTagMutability() {
            return imageTagMutability;
        }

        public final void setImageTagMutability(String imageTagMutability) {
            this.imageTagMutability = imageTagMutability;
        }

        @Override
        public final Builder imageTagMutability(String imageTagMutability) {
            this.imageTagMutability = imageTagMutability;
            return this;
        }

        @Override
        public final Builder imageTagMutability(ImageTagMutability imageTagMutability) {
            this.imageTagMutability(imageTagMutability == null ? null : imageTagMutability.toString());
            return this;
        }

        public final String getRepositoryPolicy() {
            return repositoryPolicy;
        }

        public final void setRepositoryPolicy(String repositoryPolicy) {
            this.repositoryPolicy = repositoryPolicy;
        }

        @Override
        public final Builder repositoryPolicy(String repositoryPolicy) {
            this.repositoryPolicy = repositoryPolicy;
            return this;
        }

        public final String getLifecyclePolicy() {
            return lifecyclePolicy;
        }

        public final void setLifecyclePolicy(String lifecyclePolicy) {
            this.lifecyclePolicy = lifecyclePolicy;
        }

        @Override
        public final Builder lifecyclePolicy(String lifecyclePolicy) {
            this.lifecyclePolicy = lifecyclePolicy;
            return this;
        }

        public final Collection<String> getAppliedFor() {
            if (appliedFor instanceof SdkAutoConstructList) {
                return null;
            }
            return appliedFor;
        }

        public final void setAppliedFor(Collection<String> appliedFor) {
            this.appliedFor = RCTAppliedForListCopier.copy(appliedFor);
        }

        @Override
        public final Builder appliedForWithStrings(Collection<String> appliedFor) {
            this.appliedFor = RCTAppliedForListCopier.copy(appliedFor);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder appliedForWithStrings(String... appliedFor) {
            appliedForWithStrings(Arrays.asList(appliedFor));
            return this;
        }

        @Override
        public final Builder appliedFor(Collection<RCTAppliedFor> appliedFor) {
            this.appliedFor = RCTAppliedForListCopier.copyEnumToString(appliedFor);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder appliedFor(RCTAppliedFor... appliedFor) {
            appliedFor(Arrays.asList(appliedFor));
            return this;
        }

        public final String getCustomRoleArn() {
            return customRoleArn;
        }

        public final void setCustomRoleArn(String customRoleArn) {
            this.customRoleArn = customRoleArn;
        }

        @Override
        public final Builder customRoleArn(String customRoleArn) {
            this.customRoleArn = customRoleArn;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getUpdatedAt() {
            return updatedAt;
        }

        public final void setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
        }

        @Override
        public final Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        @Override
        public RepositoryCreationTemplate build() {
            return new RepositoryCreationTemplate(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
