/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecrpublic;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.ecrpublic.internal.EcrPublicServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.ecrpublic.model.BatchCheckLayerAvailabilityRequest;
import software.amazon.awssdk.services.ecrpublic.model.BatchCheckLayerAvailabilityResponse;
import software.amazon.awssdk.services.ecrpublic.model.BatchDeleteImageRequest;
import software.amazon.awssdk.services.ecrpublic.model.BatchDeleteImageResponse;
import software.amazon.awssdk.services.ecrpublic.model.CompleteLayerUploadRequest;
import software.amazon.awssdk.services.ecrpublic.model.CompleteLayerUploadResponse;
import software.amazon.awssdk.services.ecrpublic.model.CreateRepositoryRequest;
import software.amazon.awssdk.services.ecrpublic.model.CreateRepositoryResponse;
import software.amazon.awssdk.services.ecrpublic.model.DeleteRepositoryPolicyRequest;
import software.amazon.awssdk.services.ecrpublic.model.DeleteRepositoryPolicyResponse;
import software.amazon.awssdk.services.ecrpublic.model.DeleteRepositoryRequest;
import software.amazon.awssdk.services.ecrpublic.model.DeleteRepositoryResponse;
import software.amazon.awssdk.services.ecrpublic.model.DescribeImageTagsRequest;
import software.amazon.awssdk.services.ecrpublic.model.DescribeImageTagsResponse;
import software.amazon.awssdk.services.ecrpublic.model.DescribeImagesRequest;
import software.amazon.awssdk.services.ecrpublic.model.DescribeImagesResponse;
import software.amazon.awssdk.services.ecrpublic.model.DescribeRegistriesRequest;
import software.amazon.awssdk.services.ecrpublic.model.DescribeRegistriesResponse;
import software.amazon.awssdk.services.ecrpublic.model.DescribeRepositoriesRequest;
import software.amazon.awssdk.services.ecrpublic.model.DescribeRepositoriesResponse;
import software.amazon.awssdk.services.ecrpublic.model.EcrPublicException;
import software.amazon.awssdk.services.ecrpublic.model.EmptyUploadException;
import software.amazon.awssdk.services.ecrpublic.model.GetAuthorizationTokenRequest;
import software.amazon.awssdk.services.ecrpublic.model.GetAuthorizationTokenResponse;
import software.amazon.awssdk.services.ecrpublic.model.GetRegistryCatalogDataRequest;
import software.amazon.awssdk.services.ecrpublic.model.GetRegistryCatalogDataResponse;
import software.amazon.awssdk.services.ecrpublic.model.GetRepositoryCatalogDataRequest;
import software.amazon.awssdk.services.ecrpublic.model.GetRepositoryCatalogDataResponse;
import software.amazon.awssdk.services.ecrpublic.model.GetRepositoryPolicyRequest;
import software.amazon.awssdk.services.ecrpublic.model.GetRepositoryPolicyResponse;
import software.amazon.awssdk.services.ecrpublic.model.ImageAlreadyExistsException;
import software.amazon.awssdk.services.ecrpublic.model.ImageDigestDoesNotMatchException;
import software.amazon.awssdk.services.ecrpublic.model.ImageNotFoundException;
import software.amazon.awssdk.services.ecrpublic.model.ImageTagAlreadyExistsException;
import software.amazon.awssdk.services.ecrpublic.model.InitiateLayerUploadRequest;
import software.amazon.awssdk.services.ecrpublic.model.InitiateLayerUploadResponse;
import software.amazon.awssdk.services.ecrpublic.model.InvalidLayerException;
import software.amazon.awssdk.services.ecrpublic.model.InvalidLayerPartException;
import software.amazon.awssdk.services.ecrpublic.model.InvalidParameterException;
import software.amazon.awssdk.services.ecrpublic.model.InvalidTagParameterException;
import software.amazon.awssdk.services.ecrpublic.model.LayerAlreadyExistsException;
import software.amazon.awssdk.services.ecrpublic.model.LayerPartTooSmallException;
import software.amazon.awssdk.services.ecrpublic.model.LayersNotFoundException;
import software.amazon.awssdk.services.ecrpublic.model.LimitExceededException;
import software.amazon.awssdk.services.ecrpublic.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.ecrpublic.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.ecrpublic.model.PutImageRequest;
import software.amazon.awssdk.services.ecrpublic.model.PutImageResponse;
import software.amazon.awssdk.services.ecrpublic.model.PutRegistryCatalogDataRequest;
import software.amazon.awssdk.services.ecrpublic.model.PutRegistryCatalogDataResponse;
import software.amazon.awssdk.services.ecrpublic.model.PutRepositoryCatalogDataRequest;
import software.amazon.awssdk.services.ecrpublic.model.PutRepositoryCatalogDataResponse;
import software.amazon.awssdk.services.ecrpublic.model.ReferencedImagesNotFoundException;
import software.amazon.awssdk.services.ecrpublic.model.RegistryNotFoundException;
import software.amazon.awssdk.services.ecrpublic.model.RepositoryAlreadyExistsException;
import software.amazon.awssdk.services.ecrpublic.model.RepositoryCatalogDataNotFoundException;
import software.amazon.awssdk.services.ecrpublic.model.RepositoryNotEmptyException;
import software.amazon.awssdk.services.ecrpublic.model.RepositoryNotFoundException;
import software.amazon.awssdk.services.ecrpublic.model.RepositoryPolicyNotFoundException;
import software.amazon.awssdk.services.ecrpublic.model.ServerException;
import software.amazon.awssdk.services.ecrpublic.model.SetRepositoryPolicyRequest;
import software.amazon.awssdk.services.ecrpublic.model.SetRepositoryPolicyResponse;
import software.amazon.awssdk.services.ecrpublic.model.TagResourceRequest;
import software.amazon.awssdk.services.ecrpublic.model.TagResourceResponse;
import software.amazon.awssdk.services.ecrpublic.model.TooManyTagsException;
import software.amazon.awssdk.services.ecrpublic.model.UnsupportedCommandException;
import software.amazon.awssdk.services.ecrpublic.model.UntagResourceRequest;
import software.amazon.awssdk.services.ecrpublic.model.UntagResourceResponse;
import software.amazon.awssdk.services.ecrpublic.model.UploadLayerPartRequest;
import software.amazon.awssdk.services.ecrpublic.model.UploadLayerPartResponse;
import software.amazon.awssdk.services.ecrpublic.model.UploadNotFoundException;
import software.amazon.awssdk.services.ecrpublic.transform.BatchCheckLayerAvailabilityRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.BatchDeleteImageRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.CompleteLayerUploadRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.CreateRepositoryRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.DeleteRepositoryPolicyRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.DeleteRepositoryRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.DescribeImageTagsRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.DescribeImagesRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.DescribeRegistriesRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.DescribeRepositoriesRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.GetAuthorizationTokenRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.GetRegistryCatalogDataRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.GetRepositoryCatalogDataRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.GetRepositoryPolicyRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.InitiateLayerUploadRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.PutImageRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.PutRegistryCatalogDataRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.PutRepositoryCatalogDataRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.SetRepositoryPolicyRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.UploadLayerPartRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link EcrPublicClient}.
 *
 * @see EcrPublicClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultEcrPublicClient implements EcrPublicClient {
    private static final Logger log = Logger.loggerFor(DefaultEcrPublicClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final EcrPublicServiceClientConfiguration serviceClientConfiguration;

    protected DefaultEcrPublicClient(EcrPublicServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Checks the availability of one or more image layers that are within a repository in a public registry. When an
     * image is pushed to a repository, each image layer is checked to verify if it has been uploaded before. If it has
     * been uploaded, then the image layer is skipped.
     * </p>
     * <note>
     * <p>
     * This operation is used by the Amazon ECR proxy and is not generally used by customers for pulling and pushing
     * images. In most cases, you should use the <code>docker</code> CLI to pull, tag, and push images.
     * </p>
     * </note>
     *
     * @param batchCheckLayerAvailabilityRequest
     * @return Result of the BatchCheckLayerAvailability operation returned by the service.
     * @throws RepositoryNotFoundException
     *         The specified repository can't be found. Check the spelling of the specified repository and ensure that
     *         you're performing operations on the correct registry.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws RegistryNotFoundException
     *         The registry doesn't exist.
     * @throws UnsupportedCommandException
     *         The action isn't supported in this Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrPublicException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrPublicClient.BatchCheckLayerAvailability
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/BatchCheckLayerAvailability"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchCheckLayerAvailabilityResponse batchCheckLayerAvailability(
            BatchCheckLayerAvailabilityRequest batchCheckLayerAvailabilityRequest) throws RepositoryNotFoundException,
            InvalidParameterException, ServerException, RegistryNotFoundException, UnsupportedCommandException,
            AwsServiceException, SdkClientException, EcrPublicException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchCheckLayerAvailabilityResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchCheckLayerAvailabilityResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchCheckLayerAvailabilityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchCheckLayerAvailabilityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchCheckLayerAvailability");

            return clientHandler
                    .execute(new ClientExecutionParams<BatchCheckLayerAvailabilityRequest, BatchCheckLayerAvailabilityResponse>()
                            .withOperationName("BatchCheckLayerAvailability").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(batchCheckLayerAvailabilityRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new BatchCheckLayerAvailabilityRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a list of specified images that are within a repository in a public registry. Images are specified with
     * either an <code>imageTag</code> or <code>imageDigest</code>.
     * </p>
     * <p>
     * You can remove a tag from an image by specifying the image's tag in your request. When you remove the last tag
     * from an image, the image is deleted from your repository.
     * </p>
     * <p>
     * You can completely delete an image (and all of its tags) by specifying the digest of the image in your request.
     * </p>
     *
     * @param batchDeleteImageRequest
     * @return Result of the BatchDeleteImage operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository can't be found. Check the spelling of the specified repository and ensure that
     *         you're performing operations on the correct registry.
     * @throws UnsupportedCommandException
     *         The action isn't supported in this Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrPublicException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrPublicClient.BatchDeleteImage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/BatchDeleteImage" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public BatchDeleteImageResponse batchDeleteImage(BatchDeleteImageRequest batchDeleteImageRequest) throws ServerException,
            InvalidParameterException, RepositoryNotFoundException, UnsupportedCommandException, AwsServiceException,
            SdkClientException, EcrPublicException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchDeleteImageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                BatchDeleteImageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchDeleteImageRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchDeleteImageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDeleteImage");

            return clientHandler.execute(new ClientExecutionParams<BatchDeleteImageRequest, BatchDeleteImageResponse>()
                    .withOperationName("BatchDeleteImage").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(batchDeleteImageRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchDeleteImageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Informs Amazon ECR that the image layer upload is complete for a specified public registry, repository name, and
     * upload ID. You can optionally provide a <code>sha256</code> digest of the image layer for data validation
     * purposes.
     * </p>
     * <p>
     * When an image is pushed, the CompleteLayerUpload API is called once for each new image layer to verify that the
     * upload is complete.
     * </p>
     * <note>
     * <p>
     * This operation is used by the Amazon ECR proxy and is not generally used by customers for pulling and pushing
     * images. In most cases, you should use the <code>docker</code> CLI to pull, tag, and push images.
     * </p>
     * </note>
     *
     * @param completeLayerUploadRequest
     * @return Result of the CompleteLayerUpload operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository can't be found. Check the spelling of the specified repository and ensure that
     *         you're performing operations on the correct registry.
     * @throws UploadNotFoundException
     *         The upload can't be found, or the specified upload ID isn't valid for this repository.
     * @throws InvalidLayerException
     *         The layer digest calculation performed by Amazon ECR when the image layer doesn't match the digest
     *         specified.
     * @throws LayerPartTooSmallException
     *         Layer parts must be at least 5 MiB in size.
     * @throws LayerAlreadyExistsException
     *         The image layer already exists in the associated repository.
     * @throws EmptyUploadException
     *         The specified layer upload doesn't contain any layer parts.
     * @throws RegistryNotFoundException
     *         The registry doesn't exist.
     * @throws UnsupportedCommandException
     *         The action isn't supported in this Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrPublicException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrPublicClient.CompleteLayerUpload
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/CompleteLayerUpload"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompleteLayerUploadResponse completeLayerUpload(CompleteLayerUploadRequest completeLayerUploadRequest)
            throws ServerException, InvalidParameterException, RepositoryNotFoundException, UploadNotFoundException,
            InvalidLayerException, LayerPartTooSmallException, LayerAlreadyExistsException, EmptyUploadException,
            RegistryNotFoundException, UnsupportedCommandException, AwsServiceException, SdkClientException, EcrPublicException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CompleteLayerUploadResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CompleteLayerUploadResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(completeLayerUploadRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, completeLayerUploadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CompleteLayerUpload");

            return clientHandler.execute(new ClientExecutionParams<CompleteLayerUploadRequest, CompleteLayerUploadResponse>()
                    .withOperationName("CompleteLayerUpload").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(completeLayerUploadRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CompleteLayerUploadRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a repository in a public registry. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/Repositories.html">Amazon ECR repositories</a> in
     * the <i>Amazon Elastic Container Registry User Guide</i>.
     * </p>
     *
     * @param createRepositoryRequest
     * @return Result of the CreateRepository operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws InvalidTagParameterException
     *         An invalid parameter has been specified. Tag keys can have a maximum character length of 128 characters,
     *         and tag values can have a maximum length of 256 characters.
     * @throws TooManyTagsException
     *         The list of tags on the repository is over the limit. The maximum number of tags that can be applied to a
     *         repository is 50.
     * @throws RepositoryAlreadyExistsException
     *         The specified repository already exists in the specified registry.
     * @throws LimitExceededException
     *         The operation didn't succeed because it would have exceeded a service limit for your account. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/service-quotas.html">Amazon ECR Service
     *         Quotas</a> in the Amazon Elastic Container Registry User Guide.
     * @throws UnsupportedCommandException
     *         The action isn't supported in this Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrPublicException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrPublicClient.CreateRepository
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/CreateRepository" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateRepositoryResponse createRepository(CreateRepositoryRequest createRepositoryRequest) throws ServerException,
            InvalidParameterException, InvalidTagParameterException, TooManyTagsException, RepositoryAlreadyExistsException,
            LimitExceededException, UnsupportedCommandException, AwsServiceException, SdkClientException, EcrPublicException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateRepositoryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateRepositoryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRepositoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRepositoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRepository");

            return clientHandler.execute(new ClientExecutionParams<CreateRepositoryRequest, CreateRepositoryResponse>()
                    .withOperationName("CreateRepository").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createRepositoryRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateRepositoryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a repository in a public registry. If the repository contains images, you must either manually delete all
     * images in the repository or use the <code>force</code> option. This option deletes all images on your behalf
     * before deleting the repository.
     * </p>
     *
     * @param deleteRepositoryRequest
     * @return Result of the DeleteRepository operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository can't be found. Check the spelling of the specified repository and ensure that
     *         you're performing operations on the correct registry.
     * @throws RepositoryNotEmptyException
     *         The specified repository contains images. To delete a repository that contains images, you must force the
     *         deletion with the <code>force</code> parameter.
     * @throws UnsupportedCommandException
     *         The action isn't supported in this Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrPublicException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrPublicClient.DeleteRepository
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/DeleteRepository" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteRepositoryResponse deleteRepository(DeleteRepositoryRequest deleteRepositoryRequest) throws ServerException,
            InvalidParameterException, RepositoryNotFoundException, RepositoryNotEmptyException, UnsupportedCommandException,
            AwsServiceException, SdkClientException, EcrPublicException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteRepositoryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteRepositoryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRepositoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRepositoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRepository");

            return clientHandler.execute(new ClientExecutionParams<DeleteRepositoryRequest, DeleteRepositoryResponse>()
                    .withOperationName("DeleteRepository").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteRepositoryRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteRepositoryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the repository policy that's associated with the specified repository.
     * </p>
     *
     * @param deleteRepositoryPolicyRequest
     * @return Result of the DeleteRepositoryPolicy operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository can't be found. Check the spelling of the specified repository and ensure that
     *         you're performing operations on the correct registry.
     * @throws RepositoryPolicyNotFoundException
     *         The specified repository and registry combination doesn't have an associated repository policy.
     * @throws UnsupportedCommandException
     *         The action isn't supported in this Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrPublicException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrPublicClient.DeleteRepositoryPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/DeleteRepositoryPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteRepositoryPolicyResponse deleteRepositoryPolicy(DeleteRepositoryPolicyRequest deleteRepositoryPolicyRequest)
            throws ServerException, InvalidParameterException, RepositoryNotFoundException, RepositoryPolicyNotFoundException,
            UnsupportedCommandException, AwsServiceException, SdkClientException, EcrPublicException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteRepositoryPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteRepositoryPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRepositoryPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRepositoryPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRepositoryPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteRepositoryPolicyRequest, DeleteRepositoryPolicyResponse>()
                            .withOperationName("DeleteRepositoryPolicy").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(deleteRepositoryPolicyRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteRepositoryPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the image tag details for a repository in a public registry.
     * </p>
     *
     * @param describeImageTagsRequest
     * @return Result of the DescribeImageTags operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository can't be found. Check the spelling of the specified repository and ensure that
     *         you're performing operations on the correct registry.
     * @throws UnsupportedCommandException
     *         The action isn't supported in this Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrPublicException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrPublicClient.DescribeImageTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/DescribeImageTags" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeImageTagsResponse describeImageTags(DescribeImageTagsRequest describeImageTagsRequest) throws ServerException,
            InvalidParameterException, RepositoryNotFoundException, UnsupportedCommandException, AwsServiceException,
            SdkClientException, EcrPublicException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeImageTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeImageTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeImageTagsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeImageTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeImageTags");

            return clientHandler.execute(new ClientExecutionParams<DescribeImageTagsRequest, DescribeImageTagsResponse>()
                    .withOperationName("DescribeImageTags").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(describeImageTagsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeImageTagsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns metadata that's related to the images in a repository in a public registry.
     * </p>
     * <note>
     * <p>
     * Beginning with Docker version 1.9, the Docker client compresses image layers before pushing them to a V2 Docker
     * registry. The output of the <code>docker images</code> command shows the uncompressed image size. Therefore, it
     * might return a larger image size than the image sizes that are returned by <a>DescribeImages</a>.
     * </p>
     * </note>
     *
     * @param describeImagesRequest
     * @return Result of the DescribeImages operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository can't be found. Check the spelling of the specified repository and ensure that
     *         you're performing operations on the correct registry.
     * @throws ImageNotFoundException
     *         The image requested doesn't exist in the specified repository.
     * @throws UnsupportedCommandException
     *         The action isn't supported in this Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrPublicException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrPublicClient.DescribeImages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/DescribeImages" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeImagesResponse describeImages(DescribeImagesRequest describeImagesRequest) throws ServerException,
            InvalidParameterException, RepositoryNotFoundException, ImageNotFoundException, UnsupportedCommandException,
            AwsServiceException, SdkClientException, EcrPublicException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeImagesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeImagesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeImagesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeImagesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeImages");

            return clientHandler.execute(new ClientExecutionParams<DescribeImagesRequest, DescribeImagesResponse>()
                    .withOperationName("DescribeImages").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(describeImagesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeImagesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns details for a public registry.
     * </p>
     *
     * @param describeRegistriesRequest
     * @return Result of the DescribeRegistries operation returned by the service.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws UnsupportedCommandException
     *         The action isn't supported in this Region.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrPublicException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrPublicClient.DescribeRegistries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/DescribeRegistries" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeRegistriesResponse describeRegistries(DescribeRegistriesRequest describeRegistriesRequest)
            throws InvalidParameterException, UnsupportedCommandException, ServerException, AwsServiceException,
            SdkClientException, EcrPublicException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeRegistriesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeRegistriesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeRegistriesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRegistriesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRegistries");

            return clientHandler.execute(new ClientExecutionParams<DescribeRegistriesRequest, DescribeRegistriesResponse>()
                    .withOperationName("DescribeRegistries").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(describeRegistriesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeRegistriesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes repositories that are in a public registry.
     * </p>
     *
     * @param describeRepositoriesRequest
     * @return Result of the DescribeRepositories operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository can't be found. Check the spelling of the specified repository and ensure that
     *         you're performing operations on the correct registry.
     * @throws UnsupportedCommandException
     *         The action isn't supported in this Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrPublicException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrPublicClient.DescribeRepositories
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/DescribeRepositories"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeRepositoriesResponse describeRepositories(DescribeRepositoriesRequest describeRepositoriesRequest)
            throws ServerException, InvalidParameterException, RepositoryNotFoundException, UnsupportedCommandException,
            AwsServiceException, SdkClientException, EcrPublicException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeRepositoriesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeRepositoriesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeRepositoriesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRepositoriesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRepositories");

            return clientHandler.execute(new ClientExecutionParams<DescribeRepositoriesRequest, DescribeRepositoriesResponse>()
                    .withOperationName("DescribeRepositories").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(describeRepositoriesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeRepositoriesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves an authorization token. An authorization token represents your IAM authentication credentials. You can
     * use it to access any Amazon ECR registry that your IAM principal has access to. The authorization token is valid
     * for 12 hours. This API requires the <code>ecr-public:GetAuthorizationToken</code> and
     * <code>sts:GetServiceBearerToken</code> permissions.
     * </p>
     *
     * @param getAuthorizationTokenRequest
     * @return Result of the GetAuthorizationToken operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws UnsupportedCommandException
     *         The action isn't supported in this Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrPublicException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrPublicClient.GetAuthorizationToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/GetAuthorizationToken"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetAuthorizationTokenResponse getAuthorizationToken(GetAuthorizationTokenRequest getAuthorizationTokenRequest)
            throws ServerException, InvalidParameterException, UnsupportedCommandException, AwsServiceException,
            SdkClientException, EcrPublicException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAuthorizationTokenResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetAuthorizationTokenResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAuthorizationTokenRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAuthorizationTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAuthorizationToken");

            return clientHandler.execute(new ClientExecutionParams<GetAuthorizationTokenRequest, GetAuthorizationTokenResponse>()
                    .withOperationName("GetAuthorizationToken").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getAuthorizationTokenRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAuthorizationTokenRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves catalog metadata for a public registry.
     * </p>
     *
     * @param getRegistryCatalogDataRequest
     * @return Result of the GetRegistryCatalogData operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws UnsupportedCommandException
     *         The action isn't supported in this Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrPublicException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrPublicClient.GetRegistryCatalogData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/GetRegistryCatalogData"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetRegistryCatalogDataResponse getRegistryCatalogData(GetRegistryCatalogDataRequest getRegistryCatalogDataRequest)
            throws ServerException, UnsupportedCommandException, AwsServiceException, SdkClientException, EcrPublicException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRegistryCatalogDataResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetRegistryCatalogDataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRegistryCatalogDataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRegistryCatalogDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRegistryCatalogData");

            return clientHandler
                    .execute(new ClientExecutionParams<GetRegistryCatalogDataRequest, GetRegistryCatalogDataResponse>()
                            .withOperationName("GetRegistryCatalogData").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(getRegistryCatalogDataRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetRegistryCatalogDataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieve catalog metadata for a repository in a public registry. This metadata is displayed publicly in the
     * Amazon ECR Public Gallery.
     * </p>
     *
     * @param getRepositoryCatalogDataRequest
     * @return Result of the GetRepositoryCatalogData operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryCatalogDataNotFoundException
     *         The repository catalog data doesn't exist.
     * @throws RepositoryNotFoundException
     *         The specified repository can't be found. Check the spelling of the specified repository and ensure that
     *         you're performing operations on the correct registry.
     * @throws UnsupportedCommandException
     *         The action isn't supported in this Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrPublicException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrPublicClient.GetRepositoryCatalogData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/GetRepositoryCatalogData"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetRepositoryCatalogDataResponse getRepositoryCatalogData(
            GetRepositoryCatalogDataRequest getRepositoryCatalogDataRequest) throws ServerException, InvalidParameterException,
            RepositoryCatalogDataNotFoundException, RepositoryNotFoundException, UnsupportedCommandException,
            AwsServiceException, SdkClientException, EcrPublicException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRepositoryCatalogDataResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetRepositoryCatalogDataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRepositoryCatalogDataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRepositoryCatalogDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRepositoryCatalogData");

            return clientHandler
                    .execute(new ClientExecutionParams<GetRepositoryCatalogDataRequest, GetRepositoryCatalogDataResponse>()
                            .withOperationName("GetRepositoryCatalogData").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(getRepositoryCatalogDataRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetRepositoryCatalogDataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the repository policy for the specified repository.
     * </p>
     *
     * @param getRepositoryPolicyRequest
     * @return Result of the GetRepositoryPolicy operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository can't be found. Check the spelling of the specified repository and ensure that
     *         you're performing operations on the correct registry.
     * @throws RepositoryPolicyNotFoundException
     *         The specified repository and registry combination doesn't have an associated repository policy.
     * @throws UnsupportedCommandException
     *         The action isn't supported in this Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrPublicException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrPublicClient.GetRepositoryPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/GetRepositoryPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetRepositoryPolicyResponse getRepositoryPolicy(GetRepositoryPolicyRequest getRepositoryPolicyRequest)
            throws ServerException, InvalidParameterException, RepositoryNotFoundException, RepositoryPolicyNotFoundException,
            UnsupportedCommandException, AwsServiceException, SdkClientException, EcrPublicException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRepositoryPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetRepositoryPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRepositoryPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRepositoryPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRepositoryPolicy");

            return clientHandler.execute(new ClientExecutionParams<GetRepositoryPolicyRequest, GetRepositoryPolicyResponse>()
                    .withOperationName("GetRepositoryPolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getRepositoryPolicyRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetRepositoryPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Notifies Amazon ECR that you intend to upload an image layer.
     * </p>
     * <p>
     * When an image is pushed, the InitiateLayerUpload API is called once for each image layer that hasn't already been
     * uploaded. Whether an image layer uploads is determined by the BatchCheckLayerAvailability API action.
     * </p>
     * <note>
     * <p>
     * This operation is used by the Amazon ECR proxy and is not generally used by customers for pulling and pushing
     * images. In most cases, you should use the <code>docker</code> CLI to pull, tag, and push images.
     * </p>
     * </note>
     *
     * @param initiateLayerUploadRequest
     * @return Result of the InitiateLayerUpload operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository can't be found. Check the spelling of the specified repository and ensure that
     *         you're performing operations on the correct registry.
     * @throws RegistryNotFoundException
     *         The registry doesn't exist.
     * @throws UnsupportedCommandException
     *         The action isn't supported in this Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrPublicException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrPublicClient.InitiateLayerUpload
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/InitiateLayerUpload"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public InitiateLayerUploadResponse initiateLayerUpload(InitiateLayerUploadRequest initiateLayerUploadRequest)
            throws ServerException, InvalidParameterException, RepositoryNotFoundException, RegistryNotFoundException,
            UnsupportedCommandException, AwsServiceException, SdkClientException, EcrPublicException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<InitiateLayerUploadResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, InitiateLayerUploadResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(initiateLayerUploadRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, initiateLayerUploadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InitiateLayerUpload");

            return clientHandler.execute(new ClientExecutionParams<InitiateLayerUploadRequest, InitiateLayerUploadResponse>()
                    .withOperationName("InitiateLayerUpload").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(initiateLayerUploadRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new InitiateLayerUploadRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List the tags for an Amazon ECR Public resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository can't be found. Check the spelling of the specified repository and ensure that
     *         you're performing operations on the correct registry.
     * @throws UnsupportedCommandException
     *         The action isn't supported in this Region.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrPublicException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrPublicClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InvalidParameterException, RepositoryNotFoundException, UnsupportedCommandException, ServerException,
            AwsServiceException, SdkClientException, EcrPublicException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listTagsForResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates the image manifest and tags that are associated with an image.
     * </p>
     * <p>
     * When an image is pushed and all new image layers have been uploaded, the PutImage API is called once to create or
     * update the image manifest and the tags that are associated with the image.
     * </p>
     * <note>
     * <p>
     * This operation is used by the Amazon ECR proxy and is not generally used by customers for pulling and pushing
     * images. In most cases, you should use the <code>docker</code> CLI to pull, tag, and push images.
     * </p>
     * </note>
     *
     * @param putImageRequest
     * @return Result of the PutImage operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository can't be found. Check the spelling of the specified repository and ensure that
     *         you're performing operations on the correct registry.
     * @throws ImageAlreadyExistsException
     *         The specified image has already been pushed, and there were no changes to the manifest or image tag after
     *         the last push.
     * @throws LayersNotFoundException
     *         The specified layers can't be found, or the specified layer isn't valid for this repository.
     * @throws ReferencedImagesNotFoundException
     *         The manifest list is referencing an image that doesn't exist.
     * @throws LimitExceededException
     *         The operation didn't succeed because it would have exceeded a service limit for your account. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/service-quotas.html">Amazon ECR Service
     *         Quotas</a> in the Amazon Elastic Container Registry User Guide.
     * @throws ImageTagAlreadyExistsException
     *         The specified image is tagged with a tag that already exists. The repository is configured for tag
     *         immutability.
     * @throws ImageDigestDoesNotMatchException
     *         The specified image digest doesn't match the digest that Amazon ECR calculated for the image.
     * @throws RegistryNotFoundException
     *         The registry doesn't exist.
     * @throws UnsupportedCommandException
     *         The action isn't supported in this Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrPublicException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrPublicClient.PutImage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/PutImage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutImageResponse putImage(PutImageRequest putImageRequest) throws ServerException, InvalidParameterException,
            RepositoryNotFoundException, ImageAlreadyExistsException, LayersNotFoundException, ReferencedImagesNotFoundException,
            LimitExceededException, ImageTagAlreadyExistsException, ImageDigestDoesNotMatchException, RegistryNotFoundException,
            UnsupportedCommandException, AwsServiceException, SdkClientException, EcrPublicException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutImageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutImageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putImageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putImageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutImage");

            return clientHandler.execute(new ClientExecutionParams<PutImageRequest, PutImageResponse>()
                    .withOperationName("PutImage").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(putImageRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutImageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Create or update the catalog data for a public registry.
     * </p>
     *
     * @param putRegistryCatalogDataRequest
     * @return Result of the PutRegistryCatalogData operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws UnsupportedCommandException
     *         The action isn't supported in this Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrPublicException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrPublicClient.PutRegistryCatalogData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/PutRegistryCatalogData"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutRegistryCatalogDataResponse putRegistryCatalogData(PutRegistryCatalogDataRequest putRegistryCatalogDataRequest)
            throws ServerException, InvalidParameterException, UnsupportedCommandException, AwsServiceException,
            SdkClientException, EcrPublicException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutRegistryCatalogDataResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutRegistryCatalogDataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putRegistryCatalogDataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putRegistryCatalogDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRegistryCatalogData");

            return clientHandler
                    .execute(new ClientExecutionParams<PutRegistryCatalogDataRequest, PutRegistryCatalogDataResponse>()
                            .withOperationName("PutRegistryCatalogData").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(putRegistryCatalogDataRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutRegistryCatalogDataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates the catalog data for a repository in a public registry.
     * </p>
     *
     * @param putRepositoryCatalogDataRequest
     * @return Result of the PutRepositoryCatalogData operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository can't be found. Check the spelling of the specified repository and ensure that
     *         you're performing operations on the correct registry.
     * @throws UnsupportedCommandException
     *         The action isn't supported in this Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrPublicException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrPublicClient.PutRepositoryCatalogData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/PutRepositoryCatalogData"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutRepositoryCatalogDataResponse putRepositoryCatalogData(
            PutRepositoryCatalogDataRequest putRepositoryCatalogDataRequest) throws ServerException, InvalidParameterException,
            RepositoryNotFoundException, UnsupportedCommandException, AwsServiceException, SdkClientException, EcrPublicException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutRepositoryCatalogDataResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutRepositoryCatalogDataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putRepositoryCatalogDataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putRepositoryCatalogDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRepositoryCatalogData");

            return clientHandler
                    .execute(new ClientExecutionParams<PutRepositoryCatalogDataRequest, PutRepositoryCatalogDataResponse>()
                            .withOperationName("PutRepositoryCatalogData").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(putRepositoryCatalogDataRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutRepositoryCatalogDataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Applies a repository policy to the specified public repository to control access permissions. For more
     * information, see <a href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/repository-policies.html">Amazon
     * ECR Repository Policies</a> in the <i>Amazon Elastic Container Registry User Guide</i>.
     * </p>
     *
     * @param setRepositoryPolicyRequest
     * @return Result of the SetRepositoryPolicy operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws RepositoryNotFoundException
     *         The specified repository can't be found. Check the spelling of the specified repository and ensure that
     *         you're performing operations on the correct registry.
     * @throws UnsupportedCommandException
     *         The action isn't supported in this Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrPublicException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrPublicClient.SetRepositoryPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/SetRepositoryPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SetRepositoryPolicyResponse setRepositoryPolicy(SetRepositoryPolicyRequest setRepositoryPolicyRequest)
            throws ServerException, InvalidParameterException, RepositoryNotFoundException, UnsupportedCommandException,
            AwsServiceException, SdkClientException, EcrPublicException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SetRepositoryPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SetRepositoryPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(setRepositoryPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setRepositoryPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetRepositoryPolicy");

            return clientHandler.execute(new ClientExecutionParams<SetRepositoryPolicyRequest, SetRepositoryPolicyResponse>()
                    .withOperationName("SetRepositoryPolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(setRepositoryPolicyRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SetRepositoryPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Associates the specified tags to a resource with the specified <code>resourceArn</code>. If existing tags on a
     * resource aren't specified in the request parameters, they aren't changed. When a resource is deleted, the tags
     * associated with that resource are also deleted.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws InvalidTagParameterException
     *         An invalid parameter has been specified. Tag keys can have a maximum character length of 128 characters,
     *         and tag values can have a maximum length of 256 characters.
     * @throws TooManyTagsException
     *         The list of tags on the repository is over the limit. The maximum number of tags that can be applied to a
     *         repository is 50.
     * @throws RepositoryNotFoundException
     *         The specified repository can't be found. Check the spelling of the specified repository and ensure that
     *         you're performing operations on the correct registry.
     * @throws UnsupportedCommandException
     *         The action isn't supported in this Region.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrPublicException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrPublicClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InvalidParameterException,
            InvalidTagParameterException, TooManyTagsException, RepositoryNotFoundException, UnsupportedCommandException,
            ServerException, AwsServiceException, SdkClientException, EcrPublicException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes specified tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws InvalidTagParameterException
     *         An invalid parameter has been specified. Tag keys can have a maximum character length of 128 characters,
     *         and tag values can have a maximum length of 256 characters.
     * @throws TooManyTagsException
     *         The list of tags on the repository is over the limit. The maximum number of tags that can be applied to a
     *         repository is 50.
     * @throws RepositoryNotFoundException
     *         The specified repository can't be found. Check the spelling of the specified repository and ensure that
     *         you're performing operations on the correct registry.
     * @throws UnsupportedCommandException
     *         The action isn't supported in this Region.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrPublicException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrPublicClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InvalidParameterException,
            InvalidTagParameterException, TooManyTagsException, RepositoryNotFoundException, UnsupportedCommandException,
            ServerException, AwsServiceException, SdkClientException, EcrPublicException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(untagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Uploads an image layer part to Amazon ECR.
     * </p>
     * <p>
     * When an image is pushed, each new image layer is uploaded in parts. The maximum size of each image layer part can
     * be 20971520 bytes (about 20MB). The UploadLayerPart API is called once for each new image layer part.
     * </p>
     * <note>
     * <p>
     * This operation is used by the Amazon ECR proxy and is not generally used by customers for pulling and pushing
     * images. In most cases, you should use the <code>docker</code> CLI to pull, tag, and push images.
     * </p>
     * </note>
     *
     * @param uploadLayerPartRequest
     * @return Result of the UploadLayerPart operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server-side issue.
     * @throws InvalidParameterException
     *         The specified parameter is invalid. Review the available parameters for the API request.
     * @throws InvalidLayerPartException
     *         The layer part size isn't valid, or the first byte specified isn't consecutive to the last byte of a
     *         previous layer part upload.
     * @throws RepositoryNotFoundException
     *         The specified repository can't be found. Check the spelling of the specified repository and ensure that
     *         you're performing operations on the correct registry.
     * @throws UploadNotFoundException
     *         The upload can't be found, or the specified upload ID isn't valid for this repository.
     * @throws LimitExceededException
     *         The operation didn't succeed because it would have exceeded a service limit for your account. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/service-quotas.html">Amazon ECR Service
     *         Quotas</a> in the Amazon Elastic Container Registry User Guide.
     * @throws RegistryNotFoundException
     *         The registry doesn't exist.
     * @throws UnsupportedCommandException
     *         The action isn't supported in this Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcrPublicException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcrPublicClient.UploadLayerPart
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/UploadLayerPart" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UploadLayerPartResponse uploadLayerPart(UploadLayerPartRequest uploadLayerPartRequest) throws ServerException,
            InvalidParameterException, InvalidLayerPartException, RepositoryNotFoundException, UploadNotFoundException,
            LimitExceededException, RegistryNotFoundException, UnsupportedCommandException, AwsServiceException,
            SdkClientException, EcrPublicException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UploadLayerPartResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UploadLayerPartResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(uploadLayerPartRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, uploadLayerPartRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UploadLayerPart");

            return clientHandler.execute(new ClientExecutionParams<UploadLayerPartRequest, UploadLayerPartResponse>()
                    .withOperationName("UploadLayerPart").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(uploadLayerPartRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UploadLayerPartRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        EcrPublicServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = EcrPublicServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(EcrPublicException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedCommandException")
                                .exceptionBuilderSupplier(UnsupportedCommandException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LayerPartTooSmallException")
                                .exceptionBuilderSupplier(LayerPartTooSmallException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidLayerPartException")
                                .exceptionBuilderSupplier(InvalidLayerPartException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException")
                                .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServerException")
                                .exceptionBuilderSupplier(ServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException")
                                .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException")
                                .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException")
                                .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RegistryNotFoundException")
                                .exceptionBuilderSupplier(RegistryNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EmptyUploadException")
                                .exceptionBuilderSupplier(EmptyUploadException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTagParameterException")
                                .exceptionBuilderSupplier(InvalidTagParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ImageNotFoundException")
                                .exceptionBuilderSupplier(ImageNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RepositoryCatalogDataNotFoundException")
                                .exceptionBuilderSupplier(RepositoryCatalogDataNotFoundException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LayersNotFoundException")
                                .exceptionBuilderSupplier(LayersNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException")
                                .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidLayerException")
                                .exceptionBuilderSupplier(InvalidLayerException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RepositoryNotFoundException")
                                .exceptionBuilderSupplier(RepositoryNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException")
                                .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException")
                                .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException")
                                .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UploadNotFoundException")
                                .exceptionBuilderSupplier(UploadNotFoundException::builder).httpStatusCode(400).build());
    }

    @Override
    public final EcrPublicServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
