/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecrpublic;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.ecrpublic.internal.EcrPublicServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.ecrpublic.model.BatchCheckLayerAvailabilityRequest;
import software.amazon.awssdk.services.ecrpublic.model.BatchCheckLayerAvailabilityResponse;
import software.amazon.awssdk.services.ecrpublic.model.BatchDeleteImageRequest;
import software.amazon.awssdk.services.ecrpublic.model.BatchDeleteImageResponse;
import software.amazon.awssdk.services.ecrpublic.model.CompleteLayerUploadRequest;
import software.amazon.awssdk.services.ecrpublic.model.CompleteLayerUploadResponse;
import software.amazon.awssdk.services.ecrpublic.model.CreateRepositoryRequest;
import software.amazon.awssdk.services.ecrpublic.model.CreateRepositoryResponse;
import software.amazon.awssdk.services.ecrpublic.model.DeleteRepositoryPolicyRequest;
import software.amazon.awssdk.services.ecrpublic.model.DeleteRepositoryPolicyResponse;
import software.amazon.awssdk.services.ecrpublic.model.DeleteRepositoryRequest;
import software.amazon.awssdk.services.ecrpublic.model.DeleteRepositoryResponse;
import software.amazon.awssdk.services.ecrpublic.model.DescribeImageTagsRequest;
import software.amazon.awssdk.services.ecrpublic.model.DescribeImageTagsResponse;
import software.amazon.awssdk.services.ecrpublic.model.DescribeImagesRequest;
import software.amazon.awssdk.services.ecrpublic.model.DescribeImagesResponse;
import software.amazon.awssdk.services.ecrpublic.model.DescribeRegistriesRequest;
import software.amazon.awssdk.services.ecrpublic.model.DescribeRegistriesResponse;
import software.amazon.awssdk.services.ecrpublic.model.DescribeRepositoriesRequest;
import software.amazon.awssdk.services.ecrpublic.model.DescribeRepositoriesResponse;
import software.amazon.awssdk.services.ecrpublic.model.EcrPublicException;
import software.amazon.awssdk.services.ecrpublic.model.EmptyUploadException;
import software.amazon.awssdk.services.ecrpublic.model.GetAuthorizationTokenRequest;
import software.amazon.awssdk.services.ecrpublic.model.GetAuthorizationTokenResponse;
import software.amazon.awssdk.services.ecrpublic.model.GetRegistryCatalogDataRequest;
import software.amazon.awssdk.services.ecrpublic.model.GetRegistryCatalogDataResponse;
import software.amazon.awssdk.services.ecrpublic.model.GetRepositoryCatalogDataRequest;
import software.amazon.awssdk.services.ecrpublic.model.GetRepositoryCatalogDataResponse;
import software.amazon.awssdk.services.ecrpublic.model.GetRepositoryPolicyRequest;
import software.amazon.awssdk.services.ecrpublic.model.GetRepositoryPolicyResponse;
import software.amazon.awssdk.services.ecrpublic.model.ImageAlreadyExistsException;
import software.amazon.awssdk.services.ecrpublic.model.ImageDigestDoesNotMatchException;
import software.amazon.awssdk.services.ecrpublic.model.ImageNotFoundException;
import software.amazon.awssdk.services.ecrpublic.model.ImageTagAlreadyExistsException;
import software.amazon.awssdk.services.ecrpublic.model.InitiateLayerUploadRequest;
import software.amazon.awssdk.services.ecrpublic.model.InitiateLayerUploadResponse;
import software.amazon.awssdk.services.ecrpublic.model.InvalidLayerException;
import software.amazon.awssdk.services.ecrpublic.model.InvalidLayerPartException;
import software.amazon.awssdk.services.ecrpublic.model.InvalidParameterException;
import software.amazon.awssdk.services.ecrpublic.model.InvalidTagParameterException;
import software.amazon.awssdk.services.ecrpublic.model.LayerAlreadyExistsException;
import software.amazon.awssdk.services.ecrpublic.model.LayerPartTooSmallException;
import software.amazon.awssdk.services.ecrpublic.model.LayersNotFoundException;
import software.amazon.awssdk.services.ecrpublic.model.LimitExceededException;
import software.amazon.awssdk.services.ecrpublic.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.ecrpublic.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.ecrpublic.model.PutImageRequest;
import software.amazon.awssdk.services.ecrpublic.model.PutImageResponse;
import software.amazon.awssdk.services.ecrpublic.model.PutRegistryCatalogDataRequest;
import software.amazon.awssdk.services.ecrpublic.model.PutRegistryCatalogDataResponse;
import software.amazon.awssdk.services.ecrpublic.model.PutRepositoryCatalogDataRequest;
import software.amazon.awssdk.services.ecrpublic.model.PutRepositoryCatalogDataResponse;
import software.amazon.awssdk.services.ecrpublic.model.ReferencedImagesNotFoundException;
import software.amazon.awssdk.services.ecrpublic.model.RegistryNotFoundException;
import software.amazon.awssdk.services.ecrpublic.model.RepositoryAlreadyExistsException;
import software.amazon.awssdk.services.ecrpublic.model.RepositoryCatalogDataNotFoundException;
import software.amazon.awssdk.services.ecrpublic.model.RepositoryNotEmptyException;
import software.amazon.awssdk.services.ecrpublic.model.RepositoryNotFoundException;
import software.amazon.awssdk.services.ecrpublic.model.RepositoryPolicyNotFoundException;
import software.amazon.awssdk.services.ecrpublic.model.ServerException;
import software.amazon.awssdk.services.ecrpublic.model.SetRepositoryPolicyRequest;
import software.amazon.awssdk.services.ecrpublic.model.SetRepositoryPolicyResponse;
import software.amazon.awssdk.services.ecrpublic.model.TagResourceRequest;
import software.amazon.awssdk.services.ecrpublic.model.TagResourceResponse;
import software.amazon.awssdk.services.ecrpublic.model.TooManyTagsException;
import software.amazon.awssdk.services.ecrpublic.model.UnsupportedCommandException;
import software.amazon.awssdk.services.ecrpublic.model.UntagResourceRequest;
import software.amazon.awssdk.services.ecrpublic.model.UntagResourceResponse;
import software.amazon.awssdk.services.ecrpublic.model.UploadLayerPartRequest;
import software.amazon.awssdk.services.ecrpublic.model.UploadLayerPartResponse;
import software.amazon.awssdk.services.ecrpublic.model.UploadNotFoundException;
import software.amazon.awssdk.services.ecrpublic.transform.BatchCheckLayerAvailabilityRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.BatchDeleteImageRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.CompleteLayerUploadRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.CreateRepositoryRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.DeleteRepositoryPolicyRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.DeleteRepositoryRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.DescribeImageTagsRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.DescribeImagesRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.DescribeRegistriesRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.DescribeRepositoriesRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.GetAuthorizationTokenRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.GetRegistryCatalogDataRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.GetRepositoryCatalogDataRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.GetRepositoryPolicyRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.InitiateLayerUploadRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.PutImageRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.PutRegistryCatalogDataRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.PutRepositoryCatalogDataRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.SetRepositoryPolicyRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.ecrpublic.transform.UploadLayerPartRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link EcrPublicAsyncClient}.
 *
 * @see EcrPublicAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultEcrPublicAsyncClient implements EcrPublicAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultEcrPublicAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultEcrPublicAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Checks the availability of one or more image layers that are within a repository in a public registry. When an
     * image is pushed to a repository, each image layer is checked to verify if it has been uploaded before. If it has
     * been uploaded, then the image layer is skipped.
     * </p>
     * <note>
     * <p>
     * This operation is used by the Amazon ECR proxy and is not generally used by customers for pulling and pushing
     * images. In most cases, you should use the <code>docker</code> CLI to pull, tag, and push images.
     * </p>
     * </note>
     *
     * @param batchCheckLayerAvailabilityRequest
     * @return A Java Future containing the result of the BatchCheckLayerAvailability operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNotFoundException The specified repository can't be found. Check the spelling of the
     *         specified repository and ensure that you're performing operations on the correct registry.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>RegistryNotFoundException The registry doesn't exist.</li>
     *         <li>UnsupportedCommandException The action isn't supported in this Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrPublicException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EcrPublicAsyncClient.BatchCheckLayerAvailability
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/BatchCheckLayerAvailability"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchCheckLayerAvailabilityResponse> batchCheckLayerAvailability(
            BatchCheckLayerAvailabilityRequest batchCheckLayerAvailabilityRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchCheckLayerAvailabilityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchCheckLayerAvailabilityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchCheckLayerAvailability");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchCheckLayerAvailabilityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchCheckLayerAvailabilityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchCheckLayerAvailabilityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchCheckLayerAvailabilityRequest, BatchCheckLayerAvailabilityResponse>()
                            .withOperationName("BatchCheckLayerAvailability").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchCheckLayerAvailabilityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchCheckLayerAvailabilityRequest));
            CompletableFuture<BatchCheckLayerAvailabilityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a list of specified images that are within a repository in a public registry. Images are specified with
     * either an <code>imageTag</code> or <code>imageDigest</code>.
     * </p>
     * <p>
     * You can remove a tag from an image by specifying the image's tag in your request. When you remove the last tag
     * from an image, the image is deleted from your repository.
     * </p>
     * <p>
     * You can completely delete an image (and all of its tags) by specifying the digest of the image in your request.
     * </p>
     *
     * @param batchDeleteImageRequest
     * @return A Java Future containing the result of the BatchDeleteImage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository can't be found. Check the spelling of the
     *         specified repository and ensure that you're performing operations on the correct registry.</li>
     *         <li>UnsupportedCommandException The action isn't supported in this Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrPublicException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EcrPublicAsyncClient.BatchDeleteImage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/BatchDeleteImage" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchDeleteImageResponse> batchDeleteImage(BatchDeleteImageRequest batchDeleteImageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchDeleteImageRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchDeleteImageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDeleteImage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchDeleteImageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchDeleteImageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchDeleteImageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchDeleteImageRequest, BatchDeleteImageResponse>()
                            .withOperationName("BatchDeleteImage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchDeleteImageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchDeleteImageRequest));
            CompletableFuture<BatchDeleteImageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Informs Amazon ECR that the image layer upload is complete for a specified public registry, repository name, and
     * upload ID. You can optionally provide a <code>sha256</code> digest of the image layer for data validation
     * purposes.
     * </p>
     * <p>
     * When an image is pushed, the CompleteLayerUpload API is called once for each new image layer to verify that the
     * upload is complete.
     * </p>
     * <note>
     * <p>
     * This operation is used by the Amazon ECR proxy and is not generally used by customers for pulling and pushing
     * images. In most cases, you should use the <code>docker</code> CLI to pull, tag, and push images.
     * </p>
     * </note>
     *
     * @param completeLayerUploadRequest
     * @return A Java Future containing the result of the CompleteLayerUpload operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository can't be found. Check the spelling of the
     *         specified repository and ensure that you're performing operations on the correct registry.</li>
     *         <li>UploadNotFoundException The upload can't be found, or the specified upload ID isn't valid for this
     *         repository.</li>
     *         <li>InvalidLayerException The layer digest calculation performed by Amazon ECR when the image layer
     *         doesn't match the digest specified.</li>
     *         <li>LayerPartTooSmallException Layer parts must be at least 5 MiB in size.</li>
     *         <li>LayerAlreadyExistsException The image layer already exists in the associated repository.</li>
     *         <li>EmptyUploadException The specified layer upload doesn't contain any layer parts.</li>
     *         <li>RegistryNotFoundException The registry doesn't exist.</li>
     *         <li>UnsupportedCommandException The action isn't supported in this Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrPublicException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EcrPublicAsyncClient.CompleteLayerUpload
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/CompleteLayerUpload"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CompleteLayerUploadResponse> completeLayerUpload(
            CompleteLayerUploadRequest completeLayerUploadRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(completeLayerUploadRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, completeLayerUploadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CompleteLayerUpload");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CompleteLayerUploadResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CompleteLayerUploadResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CompleteLayerUploadResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CompleteLayerUploadRequest, CompleteLayerUploadResponse>()
                            .withOperationName("CompleteLayerUpload").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CompleteLayerUploadRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(completeLayerUploadRequest));
            CompletableFuture<CompleteLayerUploadResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a repository in a public registry. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/Repositories.html">Amazon ECR repositories</a> in
     * the <i>Amazon Elastic Container Registry User Guide</i>.
     * </p>
     *
     * @param createRepositoryRequest
     * @return A Java Future containing the result of the CreateRepository operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>InvalidTagParameterException An invalid parameter has been specified. Tag keys can have a maximum
     *         character length of 128 characters, and tag values can have a maximum length of 256 characters.</li>
     *         <li>TooManyTagsException The list of tags on the repository is over the limit. The maximum number of tags
     *         that can be applied to a repository is 50.</li>
     *         <li>RepositoryAlreadyExistsException The specified repository already exists in the specified registry.</li>
     *         <li>LimitExceededException The operation didn't succeed because it would have exceeded a service limit
     *         for your account. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/service-quotas.html">Amazon ECR Service
     *         Quotas</a> in the Amazon Elastic Container Registry User Guide.</li>
     *         <li>UnsupportedCommandException The action isn't supported in this Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrPublicException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EcrPublicAsyncClient.CreateRepository
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/CreateRepository" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateRepositoryResponse> createRepository(CreateRepositoryRequest createRepositoryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRepositoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRepositoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRepository");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRepositoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateRepositoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateRepositoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRepositoryRequest, CreateRepositoryResponse>()
                            .withOperationName("CreateRepository").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateRepositoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createRepositoryRequest));
            CompletableFuture<CreateRepositoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a repository in a public registry. If the repository contains images, you must either manually delete all
     * images in the repository or use the <code>force</code> option. This option deletes all images on your behalf
     * before deleting the repository.
     * </p>
     *
     * @param deleteRepositoryRequest
     * @return A Java Future containing the result of the DeleteRepository operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository can't be found. Check the spelling of the
     *         specified repository and ensure that you're performing operations on the correct registry.</li>
     *         <li>RepositoryNotEmptyException The specified repository contains images. To delete a repository that
     *         contains images, you must force the deletion with the <code>force</code> parameter.</li>
     *         <li>UnsupportedCommandException The action isn't supported in this Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrPublicException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EcrPublicAsyncClient.DeleteRepository
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/DeleteRepository" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRepositoryResponse> deleteRepository(DeleteRepositoryRequest deleteRepositoryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRepositoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRepositoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRepository");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRepositoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteRepositoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRepositoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRepositoryRequest, DeleteRepositoryResponse>()
                            .withOperationName("DeleteRepository").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteRepositoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteRepositoryRequest));
            CompletableFuture<DeleteRepositoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the repository policy that's associated with the specified repository.
     * </p>
     *
     * @param deleteRepositoryPolicyRequest
     * @return A Java Future containing the result of the DeleteRepositoryPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository can't be found. Check the spelling of the
     *         specified repository and ensure that you're performing operations on the correct registry.</li>
     *         <li>RepositoryPolicyNotFoundException The specified repository and registry combination doesn't have an
     *         associated repository policy.</li>
     *         <li>UnsupportedCommandException The action isn't supported in this Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrPublicException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EcrPublicAsyncClient.DeleteRepositoryPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/DeleteRepositoryPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRepositoryPolicyResponse> deleteRepositoryPolicy(
            DeleteRepositoryPolicyRequest deleteRepositoryPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRepositoryPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRepositoryPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRepositoryPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRepositoryPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteRepositoryPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRepositoryPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRepositoryPolicyRequest, DeleteRepositoryPolicyResponse>()
                            .withOperationName("DeleteRepositoryPolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteRepositoryPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteRepositoryPolicyRequest));
            CompletableFuture<DeleteRepositoryPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the image tag details for a repository in a public registry.
     * </p>
     *
     * @param describeImageTagsRequest
     * @return A Java Future containing the result of the DescribeImageTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository can't be found. Check the spelling of the
     *         specified repository and ensure that you're performing operations on the correct registry.</li>
     *         <li>UnsupportedCommandException The action isn't supported in this Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrPublicException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EcrPublicAsyncClient.DescribeImageTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/DescribeImageTags" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeImageTagsResponse> describeImageTags(DescribeImageTagsRequest describeImageTagsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeImageTagsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeImageTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeImageTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeImageTagsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeImageTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeImageTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeImageTagsRequest, DescribeImageTagsResponse>()
                            .withOperationName("DescribeImageTags").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeImageTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeImageTagsRequest));
            CompletableFuture<DescribeImageTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns metadata that's related to the images in a repository in a public registry.
     * </p>
     * <note>
     * <p>
     * Beginning with Docker version 1.9, the Docker client compresses image layers before pushing them to a V2 Docker
     * registry. The output of the <code>docker images</code> command shows the uncompressed image size. Therefore, it
     * might return a larger image size than the image sizes that are returned by <a>DescribeImages</a>.
     * </p>
     * </note>
     *
     * @param describeImagesRequest
     * @return A Java Future containing the result of the DescribeImages operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository can't be found. Check the spelling of the
     *         specified repository and ensure that you're performing operations on the correct registry.</li>
     *         <li>ImageNotFoundException The image requested doesn't exist in the specified repository.</li>
     *         <li>UnsupportedCommandException The action isn't supported in this Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrPublicException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EcrPublicAsyncClient.DescribeImages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/DescribeImages" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeImagesResponse> describeImages(DescribeImagesRequest describeImagesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeImagesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeImagesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeImages");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeImagesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeImagesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeImagesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeImagesRequest, DescribeImagesResponse>()
                            .withOperationName("DescribeImages").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeImagesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeImagesRequest));
            CompletableFuture<DescribeImagesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns details for a public registry.
     * </p>
     *
     * @param describeRegistriesRequest
     * @return A Java Future containing the result of the DescribeRegistries operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>UnsupportedCommandException The action isn't supported in this Region.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrPublicException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EcrPublicAsyncClient.DescribeRegistries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/DescribeRegistries" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeRegistriesResponse> describeRegistries(DescribeRegistriesRequest describeRegistriesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeRegistriesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRegistriesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRegistries");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeRegistriesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeRegistriesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeRegistriesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeRegistriesRequest, DescribeRegistriesResponse>()
                            .withOperationName("DescribeRegistries").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeRegistriesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeRegistriesRequest));
            CompletableFuture<DescribeRegistriesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes repositories that are in a public registry.
     * </p>
     *
     * @param describeRepositoriesRequest
     * @return A Java Future containing the result of the DescribeRepositories operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository can't be found. Check the spelling of the
     *         specified repository and ensure that you're performing operations on the correct registry.</li>
     *         <li>UnsupportedCommandException The action isn't supported in this Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrPublicException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EcrPublicAsyncClient.DescribeRepositories
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/DescribeRepositories"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeRepositoriesResponse> describeRepositories(
            DescribeRepositoriesRequest describeRepositoriesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeRepositoriesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRepositoriesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRepositories");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeRepositoriesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeRepositoriesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeRepositoriesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeRepositoriesRequest, DescribeRepositoriesResponse>()
                            .withOperationName("DescribeRepositories").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeRepositoriesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeRepositoriesRequest));
            CompletableFuture<DescribeRepositoriesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves an authorization token. An authorization token represents your IAM authentication credentials. You can
     * use it to access any Amazon ECR registry that your IAM principal has access to. The authorization token is valid
     * for 12 hours. This API requires the <code>ecr-public:GetAuthorizationToken</code> and
     * <code>sts:GetServiceBearerToken</code> permissions.
     * </p>
     *
     * @param getAuthorizationTokenRequest
     * @return A Java Future containing the result of the GetAuthorizationToken operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>UnsupportedCommandException The action isn't supported in this Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrPublicException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EcrPublicAsyncClient.GetAuthorizationToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/GetAuthorizationToken"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAuthorizationTokenResponse> getAuthorizationToken(
            GetAuthorizationTokenRequest getAuthorizationTokenRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAuthorizationTokenRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAuthorizationTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAuthorizationToken");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAuthorizationTokenResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAuthorizationTokenResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAuthorizationTokenResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAuthorizationTokenRequest, GetAuthorizationTokenResponse>()
                            .withOperationName("GetAuthorizationToken").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetAuthorizationTokenRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAuthorizationTokenRequest));
            CompletableFuture<GetAuthorizationTokenResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves catalog metadata for a public registry.
     * </p>
     *
     * @param getRegistryCatalogDataRequest
     * @return A Java Future containing the result of the GetRegistryCatalogData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>UnsupportedCommandException The action isn't supported in this Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrPublicException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EcrPublicAsyncClient.GetRegistryCatalogData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/GetRegistryCatalogData"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRegistryCatalogDataResponse> getRegistryCatalogData(
            GetRegistryCatalogDataRequest getRegistryCatalogDataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRegistryCatalogDataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRegistryCatalogDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRegistryCatalogData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRegistryCatalogDataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRegistryCatalogDataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRegistryCatalogDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRegistryCatalogDataRequest, GetRegistryCatalogDataResponse>()
                            .withOperationName("GetRegistryCatalogData").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetRegistryCatalogDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getRegistryCatalogDataRequest));
            CompletableFuture<GetRegistryCatalogDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve catalog metadata for a repository in a public registry. This metadata is displayed publicly in the
     * Amazon ECR Public Gallery.
     * </p>
     *
     * @param getRepositoryCatalogDataRequest
     * @return A Java Future containing the result of the GetRepositoryCatalogData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryCatalogDataNotFoundException The repository catalog data doesn't exist.</li>
     *         <li>RepositoryNotFoundException The specified repository can't be found. Check the spelling of the
     *         specified repository and ensure that you're performing operations on the correct registry.</li>
     *         <li>UnsupportedCommandException The action isn't supported in this Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrPublicException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EcrPublicAsyncClient.GetRepositoryCatalogData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/GetRepositoryCatalogData"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRepositoryCatalogDataResponse> getRepositoryCatalogData(
            GetRepositoryCatalogDataRequest getRepositoryCatalogDataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRepositoryCatalogDataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRepositoryCatalogDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRepositoryCatalogData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRepositoryCatalogDataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRepositoryCatalogDataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRepositoryCatalogDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRepositoryCatalogDataRequest, GetRepositoryCatalogDataResponse>()
                            .withOperationName("GetRepositoryCatalogData").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetRepositoryCatalogDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getRepositoryCatalogDataRequest));
            CompletableFuture<GetRepositoryCatalogDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the repository policy for the specified repository.
     * </p>
     *
     * @param getRepositoryPolicyRequest
     * @return A Java Future containing the result of the GetRepositoryPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository can't be found. Check the spelling of the
     *         specified repository and ensure that you're performing operations on the correct registry.</li>
     *         <li>RepositoryPolicyNotFoundException The specified repository and registry combination doesn't have an
     *         associated repository policy.</li>
     *         <li>UnsupportedCommandException The action isn't supported in this Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrPublicException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EcrPublicAsyncClient.GetRepositoryPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/GetRepositoryPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRepositoryPolicyResponse> getRepositoryPolicy(
            GetRepositoryPolicyRequest getRepositoryPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRepositoryPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRepositoryPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRepositoryPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRepositoryPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRepositoryPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRepositoryPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRepositoryPolicyRequest, GetRepositoryPolicyResponse>()
                            .withOperationName("GetRepositoryPolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetRepositoryPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getRepositoryPolicyRequest));
            CompletableFuture<GetRepositoryPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Notifies Amazon ECR that you intend to upload an image layer.
     * </p>
     * <p>
     * When an image is pushed, the InitiateLayerUpload API is called once for each image layer that hasn't already been
     * uploaded. Whether an image layer uploads is determined by the BatchCheckLayerAvailability API action.
     * </p>
     * <note>
     * <p>
     * This operation is used by the Amazon ECR proxy and is not generally used by customers for pulling and pushing
     * images. In most cases, you should use the <code>docker</code> CLI to pull, tag, and push images.
     * </p>
     * </note>
     *
     * @param initiateLayerUploadRequest
     * @return A Java Future containing the result of the InitiateLayerUpload operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository can't be found. Check the spelling of the
     *         specified repository and ensure that you're performing operations on the correct registry.</li>
     *         <li>RegistryNotFoundException The registry doesn't exist.</li>
     *         <li>UnsupportedCommandException The action isn't supported in this Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrPublicException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EcrPublicAsyncClient.InitiateLayerUpload
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/InitiateLayerUpload"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<InitiateLayerUploadResponse> initiateLayerUpload(
            InitiateLayerUploadRequest initiateLayerUploadRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(initiateLayerUploadRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, initiateLayerUploadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InitiateLayerUpload");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<InitiateLayerUploadResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, InitiateLayerUploadResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<InitiateLayerUploadResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<InitiateLayerUploadRequest, InitiateLayerUploadResponse>()
                            .withOperationName("InitiateLayerUpload").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new InitiateLayerUploadRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(initiateLayerUploadRequest));
            CompletableFuture<InitiateLayerUploadResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the tags for an Amazon ECR Public resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository can't be found. Check the spelling of the
     *         specified repository and ensure that you're performing operations on the correct registry.</li>
     *         <li>UnsupportedCommandException The action isn't supported in this Region.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrPublicException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EcrPublicAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates the image manifest and tags that are associated with an image.
     * </p>
     * <p>
     * When an image is pushed and all new image layers have been uploaded, the PutImage API is called once to create or
     * update the image manifest and the tags that are associated with the image.
     * </p>
     * <note>
     * <p>
     * This operation is used by the Amazon ECR proxy and is not generally used by customers for pulling and pushing
     * images. In most cases, you should use the <code>docker</code> CLI to pull, tag, and push images.
     * </p>
     * </note>
     *
     * @param putImageRequest
     * @return A Java Future containing the result of the PutImage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository can't be found. Check the spelling of the
     *         specified repository and ensure that you're performing operations on the correct registry.</li>
     *         <li>ImageAlreadyExistsException The specified image has already been pushed, and there were no changes to
     *         the manifest or image tag after the last push.</li>
     *         <li>LayersNotFoundException The specified layers can't be found, or the specified layer isn't valid for
     *         this repository.</li>
     *         <li>ReferencedImagesNotFoundException The manifest list is referencing an image that doesn't exist.</li>
     *         <li>LimitExceededException The operation didn't succeed because it would have exceeded a service limit
     *         for your account. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/service-quotas.html">Amazon ECR Service
     *         Quotas</a> in the Amazon Elastic Container Registry User Guide.</li>
     *         <li>ImageTagAlreadyExistsException The specified image is tagged with a tag that already exists. The
     *         repository is configured for tag immutability.</li>
     *         <li>ImageDigestDoesNotMatchException The specified image digest doesn't match the digest that Amazon ECR
     *         calculated for the image.</li>
     *         <li>RegistryNotFoundException The registry doesn't exist.</li>
     *         <li>UnsupportedCommandException The action isn't supported in this Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrPublicException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EcrPublicAsyncClient.PutImage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/PutImage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutImageResponse> putImage(PutImageRequest putImageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putImageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putImageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutImage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutImageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutImageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutImageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutImageRequest, PutImageResponse>().withOperationName("PutImage")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutImageRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(putImageRequest));
            CompletableFuture<PutImageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create or update the catalog data for a public registry.
     * </p>
     *
     * @param putRegistryCatalogDataRequest
     * @return A Java Future containing the result of the PutRegistryCatalogData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>UnsupportedCommandException The action isn't supported in this Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrPublicException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EcrPublicAsyncClient.PutRegistryCatalogData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/PutRegistryCatalogData"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutRegistryCatalogDataResponse> putRegistryCatalogData(
            PutRegistryCatalogDataRequest putRegistryCatalogDataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putRegistryCatalogDataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putRegistryCatalogDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRegistryCatalogData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutRegistryCatalogDataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutRegistryCatalogDataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutRegistryCatalogDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutRegistryCatalogDataRequest, PutRegistryCatalogDataResponse>()
                            .withOperationName("PutRegistryCatalogData").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutRegistryCatalogDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putRegistryCatalogDataRequest));
            CompletableFuture<PutRegistryCatalogDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates the catalog data for a repository in a public registry.
     * </p>
     *
     * @param putRepositoryCatalogDataRequest
     * @return A Java Future containing the result of the PutRepositoryCatalogData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository can't be found. Check the spelling of the
     *         specified repository and ensure that you're performing operations on the correct registry.</li>
     *         <li>UnsupportedCommandException The action isn't supported in this Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrPublicException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EcrPublicAsyncClient.PutRepositoryCatalogData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/PutRepositoryCatalogData"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutRepositoryCatalogDataResponse> putRepositoryCatalogData(
            PutRepositoryCatalogDataRequest putRepositoryCatalogDataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putRepositoryCatalogDataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putRepositoryCatalogDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRepositoryCatalogData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutRepositoryCatalogDataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutRepositoryCatalogDataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutRepositoryCatalogDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutRepositoryCatalogDataRequest, PutRepositoryCatalogDataResponse>()
                            .withOperationName("PutRepositoryCatalogData").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutRepositoryCatalogDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putRepositoryCatalogDataRequest));
            CompletableFuture<PutRepositoryCatalogDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Applies a repository policy to the specified public repository to control access permissions. For more
     * information, see <a href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/repository-policies.html">Amazon
     * ECR Repository Policies</a> in the <i>Amazon Elastic Container Registry User Guide</i>.
     * </p>
     *
     * @param setRepositoryPolicyRequest
     * @return A Java Future containing the result of the SetRepositoryPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>RepositoryNotFoundException The specified repository can't be found. Check the spelling of the
     *         specified repository and ensure that you're performing operations on the correct registry.</li>
     *         <li>UnsupportedCommandException The action isn't supported in this Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrPublicException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EcrPublicAsyncClient.SetRepositoryPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/SetRepositoryPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SetRepositoryPolicyResponse> setRepositoryPolicy(
            SetRepositoryPolicyRequest setRepositoryPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(setRepositoryPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setRepositoryPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetRepositoryPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SetRepositoryPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SetRepositoryPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SetRepositoryPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SetRepositoryPolicyRequest, SetRepositoryPolicyResponse>()
                            .withOperationName("SetRepositoryPolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SetRepositoryPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(setRepositoryPolicyRequest));
            CompletableFuture<SetRepositoryPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates the specified tags to a resource with the specified <code>resourceArn</code>. If existing tags on a
     * resource aren't specified in the request parameters, they aren't changed. When a resource is deleted, the tags
     * associated with that resource are also deleted.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>InvalidTagParameterException An invalid parameter has been specified. Tag keys can have a maximum
     *         character length of 128 characters, and tag values can have a maximum length of 256 characters.</li>
     *         <li>TooManyTagsException The list of tags on the repository is over the limit. The maximum number of tags
     *         that can be applied to a repository is 50.</li>
     *         <li>RepositoryNotFoundException The specified repository can't be found. Check the spelling of the
     *         specified repository and ensure that you're performing operations on the correct registry.</li>
     *         <li>UnsupportedCommandException The action isn't supported in this Region.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrPublicException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EcrPublicAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes specified tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>InvalidTagParameterException An invalid parameter has been specified. Tag keys can have a maximum
     *         character length of 128 characters, and tag values can have a maximum length of 256 characters.</li>
     *         <li>TooManyTagsException The list of tags on the repository is over the limit. The maximum number of tags
     *         that can be applied to a repository is 50.</li>
     *         <li>RepositoryNotFoundException The specified repository can't be found. Check the spelling of the
     *         specified repository and ensure that you're performing operations on the correct registry.</li>
     *         <li>UnsupportedCommandException The action isn't supported in this Region.</li>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrPublicException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EcrPublicAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Uploads an image layer part to Amazon ECR.
     * </p>
     * <p>
     * When an image is pushed, each new image layer is uploaded in parts. The maximum size of each image layer part can
     * be 20971520 bytes (about 20MB). The UploadLayerPart API is called once for each new image layer part.
     * </p>
     * <note>
     * <p>
     * This operation is used by the Amazon ECR proxy and is not generally used by customers for pulling and pushing
     * images. In most cases, you should use the <code>docker</code> CLI to pull, tag, and push images.
     * </p>
     * </note>
     *
     * @param uploadLayerPartRequest
     * @return A Java Future containing the result of the UploadLayerPart operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServerException These errors are usually caused by a server-side issue.</li>
     *         <li>InvalidParameterException The specified parameter is invalid. Review the available parameters for the
     *         API request.</li>
     *         <li>InvalidLayerPartException The layer part size isn't valid, or the first byte specified isn't
     *         consecutive to the last byte of a previous layer part upload.</li>
     *         <li>RepositoryNotFoundException The specified repository can't be found. Check the spelling of the
     *         specified repository and ensure that you're performing operations on the correct registry.</li>
     *         <li>UploadNotFoundException The upload can't be found, or the specified upload ID isn't valid for this
     *         repository.</li>
     *         <li>LimitExceededException The operation didn't succeed because it would have exceeded a service limit
     *         for your account. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/service-quotas.html">Amazon ECR Service
     *         Quotas</a> in the Amazon Elastic Container Registry User Guide.</li>
     *         <li>RegistryNotFoundException The registry doesn't exist.</li>
     *         <li>UnsupportedCommandException The action isn't supported in this Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EcrPublicException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample EcrPublicAsyncClient.UploadLayerPart
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecr-public-2020-10-30/UploadLayerPart" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UploadLayerPartResponse> uploadLayerPart(UploadLayerPartRequest uploadLayerPartRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(uploadLayerPartRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, uploadLayerPartRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECR PUBLIC");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UploadLayerPart");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UploadLayerPartResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UploadLayerPartResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UploadLayerPartResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UploadLayerPartRequest, UploadLayerPartResponse>()
                            .withOperationName("UploadLayerPart").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UploadLayerPartRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(uploadLayerPartRequest));
            CompletableFuture<UploadLayerPartResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final EcrPublicServiceClientConfiguration serviceClientConfiguration() {
        return new EcrPublicServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(EcrPublicException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedCommandException")
                                .exceptionBuilderSupplier(UnsupportedCommandException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LayerPartTooSmallException")
                                .exceptionBuilderSupplier(LayerPartTooSmallException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidLayerPartException")
                                .exceptionBuilderSupplier(InvalidLayerPartException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ImageTagAlreadyExistsException")
                                .exceptionBuilderSupplier(ImageTagAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServerException")
                                .exceptionBuilderSupplier(ServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RepositoryNotEmptyException")
                                .exceptionBuilderSupplier(RepositoryNotEmptyException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LayerAlreadyExistsException")
                                .exceptionBuilderSupplier(LayerAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ImageDigestDoesNotMatchException")
                                .exceptionBuilderSupplier(ImageDigestDoesNotMatchException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RegistryNotFoundException")
                                .exceptionBuilderSupplier(RegistryNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EmptyUploadException")
                                .exceptionBuilderSupplier(EmptyUploadException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTagParameterException")
                                .exceptionBuilderSupplier(InvalidTagParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ImageNotFoundException")
                                .exceptionBuilderSupplier(ImageNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RepositoryCatalogDataNotFoundException")
                                .exceptionBuilderSupplier(RepositoryCatalogDataNotFoundException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LayersNotFoundException")
                                .exceptionBuilderSupplier(LayersNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ReferencedImagesNotFoundException")
                                .exceptionBuilderSupplier(ReferencedImagesNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidLayerException")
                                .exceptionBuilderSupplier(InvalidLayerException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RepositoryNotFoundException")
                                .exceptionBuilderSupplier(RepositoryNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RepositoryAlreadyExistsException")
                                .exceptionBuilderSupplier(RepositoryAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RepositoryPolicyNotFoundException")
                                .exceptionBuilderSupplier(RepositoryPolicyNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ImageAlreadyExistsException")
                                .exceptionBuilderSupplier(ImageAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UploadNotFoundException")
                                .exceptionBuilderSupplier(UploadNotFoundException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        EcrPublicServiceClientConfigurationBuilder serviceConfigBuilder = new EcrPublicServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
