/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The managed scaling settings for the Auto Scaling group capacity provider.
 * </p>
 * <p>
 * When managed scaling is enabled, Amazon ECS manages the scale-in and scale-out actions of the Auto Scaling group.
 * Amazon ECS manages a target tracking scaling policy using an Amazon ECS-managed CloudWatch metric with the specified
 * <code>targetCapacity</code> value as the target value for the metric. For more information, see <a href=
 * "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/asg-capacity-providers.html#asg-capacity-providers-managed-scaling"
 * >Using Managed Scaling</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
 * </p>
 * <p>
 * If managed scaling is disabled, the user must manage the scaling of the Auto Scaling group.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ManagedScaling implements SdkPojo, Serializable, ToCopyableBuilder<ManagedScaling.Builder, ManagedScaling> {
    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ManagedScaling::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Integer> TARGET_CAPACITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ManagedScaling::targetCapacity)).setter(setter(Builder::targetCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("targetCapacity").build()).build();

    private static final SdkField<Integer> MINIMUM_SCALING_STEP_SIZE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ManagedScaling::minimumScalingStepSize)).setter(setter(Builder::minimumScalingStepSize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("minimumScalingStepSize").build())
            .build();

    private static final SdkField<Integer> MAXIMUM_SCALING_STEP_SIZE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ManagedScaling::maximumScalingStepSize)).setter(setter(Builder::maximumScalingStepSize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maximumScalingStepSize").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STATUS_FIELD,
            TARGET_CAPACITY_FIELD, MINIMUM_SCALING_STEP_SIZE_FIELD, MAXIMUM_SCALING_STEP_SIZE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String status;

    private final Integer targetCapacity;

    private final Integer minimumScalingStepSize;

    private final Integer maximumScalingStepSize;

    private ManagedScaling(BuilderImpl builder) {
        this.status = builder.status;
        this.targetCapacity = builder.targetCapacity;
        this.minimumScalingStepSize = builder.minimumScalingStepSize;
        this.maximumScalingStepSize = builder.maximumScalingStepSize;
    }

    /**
     * <p>
     * Whether or not to enable managed scaling for the capacity provider.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ManagedScalingStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return Whether or not to enable managed scaling for the capacity provider.
     * @see ManagedScalingStatus
     */
    public ManagedScalingStatus status() {
        return ManagedScalingStatus.fromValue(status);
    }

    /**
     * <p>
     * Whether or not to enable managed scaling for the capacity provider.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ManagedScalingStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return Whether or not to enable managed scaling for the capacity provider.
     * @see ManagedScalingStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The target capacity value for the capacity provider. The specified value must be greater than <code>0</code> and
     * less than or equal to <code>100</code>. A value of <code>100</code> will result in the Amazon EC2 instances in
     * your Auto Scaling group being completely utilized.
     * </p>
     * 
     * @return The target capacity value for the capacity provider. The specified value must be greater than
     *         <code>0</code> and less than or equal to <code>100</code>. A value of <code>100</code> will result in the
     *         Amazon EC2 instances in your Auto Scaling group being completely utilized.
     */
    public Integer targetCapacity() {
        return targetCapacity;
    }

    /**
     * <p>
     * The minimum number of container instances that Amazon ECS will scale in or scale out at one time. If this
     * parameter is omitted, the default value of <code>1</code> is used.
     * </p>
     * 
     * @return The minimum number of container instances that Amazon ECS will scale in or scale out at one time. If this
     *         parameter is omitted, the default value of <code>1</code> is used.
     */
    public Integer minimumScalingStepSize() {
        return minimumScalingStepSize;
    }

    /**
     * <p>
     * The maximum number of container instances that Amazon ECS will scale in or scale out at one time. If this
     * parameter is omitted, the default value of <code>10000</code> is used.
     * </p>
     * 
     * @return The maximum number of container instances that Amazon ECS will scale in or scale out at one time. If this
     *         parameter is omitted, the default value of <code>10000</code> is used.
     */
    public Integer maximumScalingStepSize() {
        return maximumScalingStepSize;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(targetCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(minimumScalingStepSize());
        hashCode = 31 * hashCode + Objects.hashCode(maximumScalingStepSize());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ManagedScaling)) {
            return false;
        }
        ManagedScaling other = (ManagedScaling) obj;
        return Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(targetCapacity(), other.targetCapacity())
                && Objects.equals(minimumScalingStepSize(), other.minimumScalingStepSize())
                && Objects.equals(maximumScalingStepSize(), other.maximumScalingStepSize());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ManagedScaling").add("Status", statusAsString()).add("TargetCapacity", targetCapacity())
                .add("MinimumScalingStepSize", minimumScalingStepSize()).add("MaximumScalingStepSize", maximumScalingStepSize())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "targetCapacity":
            return Optional.ofNullable(clazz.cast(targetCapacity()));
        case "minimumScalingStepSize":
            return Optional.ofNullable(clazz.cast(minimumScalingStepSize()));
        case "maximumScalingStepSize":
            return Optional.ofNullable(clazz.cast(maximumScalingStepSize()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ManagedScaling, T> g) {
        return obj -> g.apply((ManagedScaling) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ManagedScaling> {
        /**
         * <p>
         * Whether or not to enable managed scaling for the capacity provider.
         * </p>
         * 
         * @param status
         *        Whether or not to enable managed scaling for the capacity provider.
         * @see ManagedScalingStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ManagedScalingStatus
         */
        Builder status(String status);

        /**
         * <p>
         * Whether or not to enable managed scaling for the capacity provider.
         * </p>
         * 
         * @param status
         *        Whether or not to enable managed scaling for the capacity provider.
         * @see ManagedScalingStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ManagedScalingStatus
         */
        Builder status(ManagedScalingStatus status);

        /**
         * <p>
         * The target capacity value for the capacity provider. The specified value must be greater than <code>0</code>
         * and less than or equal to <code>100</code>. A value of <code>100</code> will result in the Amazon EC2
         * instances in your Auto Scaling group being completely utilized.
         * </p>
         * 
         * @param targetCapacity
         *        The target capacity value for the capacity provider. The specified value must be greater than
         *        <code>0</code> and less than or equal to <code>100</code>. A value of <code>100</code> will result in
         *        the Amazon EC2 instances in your Auto Scaling group being completely utilized.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetCapacity(Integer targetCapacity);

        /**
         * <p>
         * The minimum number of container instances that Amazon ECS will scale in or scale out at one time. If this
         * parameter is omitted, the default value of <code>1</code> is used.
         * </p>
         * 
         * @param minimumScalingStepSize
         *        The minimum number of container instances that Amazon ECS will scale in or scale out at one time. If
         *        this parameter is omitted, the default value of <code>1</code> is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minimumScalingStepSize(Integer minimumScalingStepSize);

        /**
         * <p>
         * The maximum number of container instances that Amazon ECS will scale in or scale out at one time. If this
         * parameter is omitted, the default value of <code>10000</code> is used.
         * </p>
         * 
         * @param maximumScalingStepSize
         *        The maximum number of container instances that Amazon ECS will scale in or scale out at one time. If
         *        this parameter is omitted, the default value of <code>10000</code> is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maximumScalingStepSize(Integer maximumScalingStepSize);
    }

    static final class BuilderImpl implements Builder {
        private String status;

        private Integer targetCapacity;

        private Integer minimumScalingStepSize;

        private Integer maximumScalingStepSize;

        private BuilderImpl() {
        }

        private BuilderImpl(ManagedScaling model) {
            status(model.status);
            targetCapacity(model.targetCapacity);
            minimumScalingStepSize(model.minimumScalingStepSize);
            maximumScalingStepSize(model.maximumScalingStepSize);
        }

        public final String getStatusAsString() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ManagedScalingStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final Integer getTargetCapacity() {
            return targetCapacity;
        }

        @Override
        public final Builder targetCapacity(Integer targetCapacity) {
            this.targetCapacity = targetCapacity;
            return this;
        }

        public final void setTargetCapacity(Integer targetCapacity) {
            this.targetCapacity = targetCapacity;
        }

        public final Integer getMinimumScalingStepSize() {
            return minimumScalingStepSize;
        }

        @Override
        public final Builder minimumScalingStepSize(Integer minimumScalingStepSize) {
            this.minimumScalingStepSize = minimumScalingStepSize;
            return this;
        }

        public final void setMinimumScalingStepSize(Integer minimumScalingStepSize) {
            this.minimumScalingStepSize = minimumScalingStepSize;
        }

        public final Integer getMaximumScalingStepSize() {
            return maximumScalingStepSize;
        }

        @Override
        public final Builder maximumScalingStepSize(Integer maximumScalingStepSize) {
            this.maximumScalingStepSize = maximumScalingStepSize;
            return this;
        }

        public final void setMaximumScalingStepSize(Integer maximumScalingStepSize) {
            this.maximumScalingStepSize = maximumScalingStepSize;
        }

        @Override
        public ManagedScaling build() {
            return new ManagedScaling(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
