/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An EC2 instance that is running the Amazon ECS agent and has been registered with a cluster.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ContainerInstance implements SdkPojo, Serializable,
        ToCopyableBuilder<ContainerInstance.Builder, ContainerInstance> {
    private static final SdkField<String> CONTAINER_INSTANCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ContainerInstance::containerInstanceArn)).setter(setter(Builder::containerInstanceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerInstanceArn").build())
            .build();

    private static final SdkField<String> EC2_INSTANCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ContainerInstance::ec2InstanceId)).setter(setter(Builder::ec2InstanceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ec2InstanceId").build()).build();

    private static final SdkField<String> CAPACITY_PROVIDER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ContainerInstance::capacityProviderName)).setter(setter(Builder::capacityProviderName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("capacityProviderName").build())
            .build();

    private static final SdkField<Long> VERSION_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .getter(getter(ContainerInstance::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("version").build()).build();

    private static final SdkField<VersionInfo> VERSION_INFO_FIELD = SdkField.<VersionInfo> builder(MarshallingType.SDK_POJO)
            .getter(getter(ContainerInstance::versionInfo)).setter(setter(Builder::versionInfo))
            .constructor(VersionInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("versionInfo").build()).build();

    private static final SdkField<List<Resource>> REMAINING_RESOURCES_FIELD = SdkField
            .<List<Resource>> builder(MarshallingType.LIST)
            .getter(getter(ContainerInstance::remainingResources))
            .setter(setter(Builder::remainingResources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("remainingResources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Resource> builder(MarshallingType.SDK_POJO)
                                            .constructor(Resource::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Resource>> REGISTERED_RESOURCES_FIELD = SdkField
            .<List<Resource>> builder(MarshallingType.LIST)
            .getter(getter(ContainerInstance::registeredResources))
            .setter(setter(Builder::registeredResources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("registeredResources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Resource> builder(MarshallingType.SDK_POJO)
                                            .constructor(Resource::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ContainerInstance::status)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> STATUS_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ContainerInstance::statusReason)).setter(setter(Builder::statusReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusReason").build()).build();

    private static final SdkField<Boolean> AGENT_CONNECTED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(ContainerInstance::agentConnected)).setter(setter(Builder::agentConnected))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("agentConnected").build()).build();

    private static final SdkField<Integer> RUNNING_TASKS_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ContainerInstance::runningTasksCount)).setter(setter(Builder::runningTasksCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("runningTasksCount").build()).build();

    private static final SdkField<Integer> PENDING_TASKS_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ContainerInstance::pendingTasksCount)).setter(setter(Builder::pendingTasksCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("pendingTasksCount").build()).build();

    private static final SdkField<String> AGENT_UPDATE_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ContainerInstance::agentUpdateStatusAsString)).setter(setter(Builder::agentUpdateStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("agentUpdateStatus").build()).build();

    private static final SdkField<List<Attribute>> ATTRIBUTES_FIELD = SdkField
            .<List<Attribute>> builder(MarshallingType.LIST)
            .getter(getter(ContainerInstance::attributes))
            .setter(setter(Builder::attributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("attributes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Attribute> builder(MarshallingType.SDK_POJO)
                                            .constructor(Attribute::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> REGISTERED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(ContainerInstance::registeredAt)).setter(setter(Builder::registeredAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("registeredAt").build()).build();

    private static final SdkField<List<Attachment>> ATTACHMENTS_FIELD = SdkField
            .<List<Attachment>> builder(MarshallingType.LIST)
            .getter(getter(ContainerInstance::attachments))
            .setter(setter(Builder::attachments))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("attachments").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Attachment> builder(MarshallingType.SDK_POJO)
                                            .constructor(Attachment::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .getter(getter(ContainerInstance::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONTAINER_INSTANCE_ARN_FIELD,
            EC2_INSTANCE_ID_FIELD, CAPACITY_PROVIDER_NAME_FIELD, VERSION_FIELD, VERSION_INFO_FIELD, REMAINING_RESOURCES_FIELD,
            REGISTERED_RESOURCES_FIELD, STATUS_FIELD, STATUS_REASON_FIELD, AGENT_CONNECTED_FIELD, RUNNING_TASKS_COUNT_FIELD,
            PENDING_TASKS_COUNT_FIELD, AGENT_UPDATE_STATUS_FIELD, ATTRIBUTES_FIELD, REGISTERED_AT_FIELD, ATTACHMENTS_FIELD,
            TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String containerInstanceArn;

    private final String ec2InstanceId;

    private final String capacityProviderName;

    private final Long version;

    private final VersionInfo versionInfo;

    private final List<Resource> remainingResources;

    private final List<Resource> registeredResources;

    private final String status;

    private final String statusReason;

    private final Boolean agentConnected;

    private final Integer runningTasksCount;

    private final Integer pendingTasksCount;

    private final String agentUpdateStatus;

    private final List<Attribute> attributes;

    private final Instant registeredAt;

    private final List<Attachment> attachments;

    private final List<Tag> tags;

    private ContainerInstance(BuilderImpl builder) {
        this.containerInstanceArn = builder.containerInstanceArn;
        this.ec2InstanceId = builder.ec2InstanceId;
        this.capacityProviderName = builder.capacityProviderName;
        this.version = builder.version;
        this.versionInfo = builder.versionInfo;
        this.remainingResources = builder.remainingResources;
        this.registeredResources = builder.registeredResources;
        this.status = builder.status;
        this.statusReason = builder.statusReason;
        this.agentConnected = builder.agentConnected;
        this.runningTasksCount = builder.runningTasksCount;
        this.pendingTasksCount = builder.pendingTasksCount;
        this.agentUpdateStatus = builder.agentUpdateStatus;
        this.attributes = builder.attributes;
        this.registeredAt = builder.registeredAt;
        this.attachments = builder.attachments;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the container instance. The ARN contains the <code>arn:aws:ecs</code>
     * namespace, followed by the Region of the container instance, the AWS account ID of the container instance owner,
     * the <code>container-instance</code> namespace, and then the container instance ID. For example,
     * <code>arn:aws:ecs:region:aws_account_id:container-instance/container_instance_ID</code>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the container instance. The ARN contains the <code>arn:aws:ecs</code>
     *         namespace, followed by the Region of the container instance, the AWS account ID of the container instance
     *         owner, the <code>container-instance</code> namespace, and then the container instance ID. For example,
     *         <code>arn:aws:ecs:region:aws_account_id:container-instance/container_instance_ID</code>.
     */
    public String containerInstanceArn() {
        return containerInstanceArn;
    }

    /**
     * <p>
     * The EC2 instance ID of the container instance.
     * </p>
     * 
     * @return The EC2 instance ID of the container instance.
     */
    public String ec2InstanceId() {
        return ec2InstanceId;
    }

    /**
     * <p>
     * The capacity provider associated with the container instance.
     * </p>
     * 
     * @return The capacity provider associated with the container instance.
     */
    public String capacityProviderName() {
        return capacityProviderName;
    }

    /**
     * <p>
     * The version counter for the container instance. Every time a container instance experiences a change that
     * triggers a CloudWatch event, the version counter is incremented. If you are replicating your Amazon ECS container
     * instance state with CloudWatch Events, you can compare the version of a container instance reported by the Amazon
     * ECS APIs with the version reported in CloudWatch Events for the container instance (inside the
     * <code>detail</code> object) to verify that the version in your event stream is current.
     * </p>
     * 
     * @return The version counter for the container instance. Every time a container instance experiences a change that
     *         triggers a CloudWatch event, the version counter is incremented. If you are replicating your Amazon ECS
     *         container instance state with CloudWatch Events, you can compare the version of a container instance
     *         reported by the Amazon ECS APIs with the version reported in CloudWatch Events for the container instance
     *         (inside the <code>detail</code> object) to verify that the version in your event stream is current.
     */
    public Long version() {
        return version;
    }

    /**
     * <p>
     * The version information for the Amazon ECS container agent and Docker daemon running on the container instance.
     * </p>
     * 
     * @return The version information for the Amazon ECS container agent and Docker daemon running on the container
     *         instance.
     */
    public VersionInfo versionInfo() {
        return versionInfo;
    }

    /**
     * Returns true if the RemainingResources property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasRemainingResources() {
        return remainingResources != null && !(remainingResources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * For CPU and memory resource types, this parameter describes the remaining CPU and memory that has not already
     * been allocated to tasks and is therefore available for new tasks. For port resource types, this parameter
     * describes the ports that were reserved by the Amazon ECS container agent (at instance registration time) and any
     * task containers that have reserved port mappings on the host (with the <code>host</code> or <code>bridge</code>
     * network mode). Any port that is not specified here is available for new tasks.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRemainingResources()} to see if a value was sent in this field.
     * </p>
     * 
     * @return For CPU and memory resource types, this parameter describes the remaining CPU and memory that has not
     *         already been allocated to tasks and is therefore available for new tasks. For port resource types, this
     *         parameter describes the ports that were reserved by the Amazon ECS container agent (at instance
     *         registration time) and any task containers that have reserved port mappings on the host (with the
     *         <code>host</code> or <code>bridge</code> network mode). Any port that is not specified here is available
     *         for new tasks.
     */
    public List<Resource> remainingResources() {
        return remainingResources;
    }

    /**
     * Returns true if the RegisteredResources property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasRegisteredResources() {
        return registeredResources != null && !(registeredResources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * For CPU and memory resource types, this parameter describes the amount of each resource that was available on the
     * container instance when the container agent registered it with Amazon ECS. This value represents the total amount
     * of CPU and memory that can be allocated on this container instance to tasks. For port resource types, this
     * parameter describes the ports that were reserved by the Amazon ECS container agent when it registered the
     * container instance with Amazon ECS.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRegisteredResources()} to see if a value was sent in this field.
     * </p>
     * 
     * @return For CPU and memory resource types, this parameter describes the amount of each resource that was
     *         available on the container instance when the container agent registered it with Amazon ECS. This value
     *         represents the total amount of CPU and memory that can be allocated on this container instance to tasks.
     *         For port resource types, this parameter describes the ports that were reserved by the Amazon ECS
     *         container agent when it registered the container instance with Amazon ECS.
     */
    public List<Resource> registeredResources() {
        return registeredResources;
    }

    /**
     * <p>
     * The status of the container instance. The valid values are <code>REGISTERING</code>,
     * <code>REGISTRATION_FAILED</code>, <code>ACTIVE</code>, <code>INACTIVE</code>, <code>DEREGISTERING</code>, or
     * <code>DRAINING</code>.
     * </p>
     * <p>
     * If your account has opted in to the <code>awsvpcTrunking</code> account setting, then any newly registered
     * container instance will transition to a <code>REGISTERING</code> status while the trunk elastic network interface
     * is provisioned for the instance. If the registration fails, the instance will transition to a
     * <code>REGISTRATION_FAILED</code> status. You can describe the container instance and see the reason for failure
     * in the <code>statusReason</code> parameter. Once the container instance is terminated, the instance transitions
     * to a <code>DEREGISTERING</code> status while the trunk elastic network interface is deprovisioned. The instance
     * then transitions to an <code>INACTIVE</code> status.
     * </p>
     * <p>
     * The <code>ACTIVE</code> status indicates that the container instance can accept tasks. The <code>DRAINING</code>
     * indicates that new tasks are not placed on the container instance and any service tasks running on the container
     * instance are removed if possible. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-instance-draining.html">Container
     * Instance Draining</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * 
     * @return The status of the container instance. The valid values are <code>REGISTERING</code>,
     *         <code>REGISTRATION_FAILED</code>, <code>ACTIVE</code>, <code>INACTIVE</code>, <code>DEREGISTERING</code>,
     *         or <code>DRAINING</code>.</p>
     *         <p>
     *         If your account has opted in to the <code>awsvpcTrunking</code> account setting, then any newly
     *         registered container instance will transition to a <code>REGISTERING</code> status while the trunk
     *         elastic network interface is provisioned for the instance. If the registration fails, the instance will
     *         transition to a <code>REGISTRATION_FAILED</code> status. You can describe the container instance and see
     *         the reason for failure in the <code>statusReason</code> parameter. Once the container instance is
     *         terminated, the instance transitions to a <code>DEREGISTERING</code> status while the trunk elastic
     *         network interface is deprovisioned. The instance then transitions to an <code>INACTIVE</code> status.
     *         </p>
     *         <p>
     *         The <code>ACTIVE</code> status indicates that the container instance can accept tasks. The
     *         <code>DRAINING</code> indicates that new tasks are not placed on the container instance and any service
     *         tasks running on the container instance are removed if possible. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-instance-draining.html"
     *         >Container Instance Draining</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     */
    public String status() {
        return status;
    }

    /**
     * <p>
     * The reason that the container instance reached its current status.
     * </p>
     * 
     * @return The reason that the container instance reached its current status.
     */
    public String statusReason() {
        return statusReason;
    }

    /**
     * <p>
     * This parameter returns <code>true</code> if the agent is connected to Amazon ECS. Registered instances with an
     * agent that may be unhealthy or stopped return <code>false</code>. Only instances connected to an agent can accept
     * placement requests.
     * </p>
     * 
     * @return This parameter returns <code>true</code> if the agent is connected to Amazon ECS. Registered instances
     *         with an agent that may be unhealthy or stopped return <code>false</code>. Only instances connected to an
     *         agent can accept placement requests.
     */
    public Boolean agentConnected() {
        return agentConnected;
    }

    /**
     * <p>
     * The number of tasks on the container instance that are in the <code>RUNNING</code> status.
     * </p>
     * 
     * @return The number of tasks on the container instance that are in the <code>RUNNING</code> status.
     */
    public Integer runningTasksCount() {
        return runningTasksCount;
    }

    /**
     * <p>
     * The number of tasks on the container instance that are in the <code>PENDING</code> status.
     * </p>
     * 
     * @return The number of tasks on the container instance that are in the <code>PENDING</code> status.
     */
    public Integer pendingTasksCount() {
        return pendingTasksCount;
    }

    /**
     * <p>
     * The status of the most recent agent update. If an update has never been requested, this value is
     * <code>NULL</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #agentUpdateStatus}
     * will return {@link AgentUpdateStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #agentUpdateStatusAsString}.
     * </p>
     * 
     * @return The status of the most recent agent update. If an update has never been requested, this value is
     *         <code>NULL</code>.
     * @see AgentUpdateStatus
     */
    public AgentUpdateStatus agentUpdateStatus() {
        return AgentUpdateStatus.fromValue(agentUpdateStatus);
    }

    /**
     * <p>
     * The status of the most recent agent update. If an update has never been requested, this value is
     * <code>NULL</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #agentUpdateStatus}
     * will return {@link AgentUpdateStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #agentUpdateStatusAsString}.
     * </p>
     * 
     * @return The status of the most recent agent update. If an update has never been requested, this value is
     *         <code>NULL</code>.
     * @see AgentUpdateStatus
     */
    public String agentUpdateStatusAsString() {
        return agentUpdateStatus;
    }

    /**
     * Returns true if the Attributes property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasAttributes() {
        return attributes != null && !(attributes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The attributes set for the container instance, either by the Amazon ECS container agent at instance registration
     * or manually with the <a>PutAttributes</a> operation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAttributes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The attributes set for the container instance, either by the Amazon ECS container agent at instance
     *         registration or manually with the <a>PutAttributes</a> operation.
     */
    public List<Attribute> attributes() {
        return attributes;
    }

    /**
     * <p>
     * The Unix timestamp for when the container instance was registered.
     * </p>
     * 
     * @return The Unix timestamp for when the container instance was registered.
     */
    public Instant registeredAt() {
        return registeredAt;
    }

    /**
     * Returns true if the Attachments property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasAttachments() {
        return attachments != null && !(attachments instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The resources attached to a container instance, such as elastic network interfaces.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAttachments()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The resources attached to a container instance, such as elastic network interfaces.
     */
    public List<Attachment> attachments() {
        return attachments;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The metadata that you apply to the container instance to help you categorize and organize them. Each tag consists
     * of a key and an optional value, both of which you define.
     * </p>
     * <p>
     * The following basic restrictions apply to tags:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Maximum number of tags per resource - 50
     * </p>
     * </li>
     * <li>
     * <p>
     * For each resource, each tag key must be unique, and each tag key can have only one value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Maximum key length - 128 Unicode characters in UTF-8
     * </p>
     * </li>
     * <li>
     * <p>
     * Maximum value length - 256 Unicode characters in UTF-8
     * </p>
     * </li>
     * <li>
     * <p>
     * If your tagging schema is used across multiple services and resources, remember that other services may have
     * restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable
     * in UTF-8, and the following characters: + - = . _ : / @.
     * </p>
     * </li>
     * <li>
     * <p>
     * Tag keys and values are case-sensitive.
     * </p>
     * </li>
     * <li>
     * <p>
     * Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a prefix for
     * either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this
     * prefix. Tags with this prefix do not count against your tags per resource limit.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The metadata that you apply to the container instance to help you categorize and organize them. Each tag
     *         consists of a key and an optional value, both of which you define.</p>
     *         <p>
     *         The following basic restrictions apply to tags:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Maximum number of tags per resource - 50
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For each resource, each tag key must be unique, and each tag key can have only one value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Maximum key length - 128 Unicode characters in UTF-8
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Maximum value length - 256 Unicode characters in UTF-8
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If your tagging schema is used across multiple services and resources, remember that other services may
     *         have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces
     *         representable in UTF-8, and the following characters: + - = . _ : / @.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Tag keys and values are case-sensitive.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a
     *         prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or
     *         values with this prefix. Tags with this prefix do not count against your tags per resource limit.
     *         </p>
     *         </li>
     */
    public List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(containerInstanceArn());
        hashCode = 31 * hashCode + Objects.hashCode(ec2InstanceId());
        hashCode = 31 * hashCode + Objects.hashCode(capacityProviderName());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(versionInfo());
        hashCode = 31 * hashCode + Objects.hashCode(remainingResources());
        hashCode = 31 * hashCode + Objects.hashCode(registeredResources());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(statusReason());
        hashCode = 31 * hashCode + Objects.hashCode(agentConnected());
        hashCode = 31 * hashCode + Objects.hashCode(runningTasksCount());
        hashCode = 31 * hashCode + Objects.hashCode(pendingTasksCount());
        hashCode = 31 * hashCode + Objects.hashCode(agentUpdateStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(attributes());
        hashCode = 31 * hashCode + Objects.hashCode(registeredAt());
        hashCode = 31 * hashCode + Objects.hashCode(attachments());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ContainerInstance)) {
            return false;
        }
        ContainerInstance other = (ContainerInstance) obj;
        return Objects.equals(containerInstanceArn(), other.containerInstanceArn())
                && Objects.equals(ec2InstanceId(), other.ec2InstanceId())
                && Objects.equals(capacityProviderName(), other.capacityProviderName())
                && Objects.equals(version(), other.version()) && Objects.equals(versionInfo(), other.versionInfo())
                && Objects.equals(remainingResources(), other.remainingResources())
                && Objects.equals(registeredResources(), other.registeredResources()) && Objects.equals(status(), other.status())
                && Objects.equals(statusReason(), other.statusReason())
                && Objects.equals(agentConnected(), other.agentConnected())
                && Objects.equals(runningTasksCount(), other.runningTasksCount())
                && Objects.equals(pendingTasksCount(), other.pendingTasksCount())
                && Objects.equals(agentUpdateStatusAsString(), other.agentUpdateStatusAsString())
                && Objects.equals(attributes(), other.attributes()) && Objects.equals(registeredAt(), other.registeredAt())
                && Objects.equals(attachments(), other.attachments()) && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ContainerInstance").add("ContainerInstanceArn", containerInstanceArn())
                .add("Ec2InstanceId", ec2InstanceId()).add("CapacityProviderName", capacityProviderName())
                .add("Version", version()).add("VersionInfo", versionInfo()).add("RemainingResources", remainingResources())
                .add("RegisteredResources", registeredResources()).add("Status", status()).add("StatusReason", statusReason())
                .add("AgentConnected", agentConnected()).add("RunningTasksCount", runningTasksCount())
                .add("PendingTasksCount", pendingTasksCount()).add("AgentUpdateStatus", agentUpdateStatusAsString())
                .add("Attributes", attributes()).add("RegisteredAt", registeredAt()).add("Attachments", attachments())
                .add("Tags", tags()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "containerInstanceArn":
            return Optional.ofNullable(clazz.cast(containerInstanceArn()));
        case "ec2InstanceId":
            return Optional.ofNullable(clazz.cast(ec2InstanceId()));
        case "capacityProviderName":
            return Optional.ofNullable(clazz.cast(capacityProviderName()));
        case "version":
            return Optional.ofNullable(clazz.cast(version()));
        case "versionInfo":
            return Optional.ofNullable(clazz.cast(versionInfo()));
        case "remainingResources":
            return Optional.ofNullable(clazz.cast(remainingResources()));
        case "registeredResources":
            return Optional.ofNullable(clazz.cast(registeredResources()));
        case "status":
            return Optional.ofNullable(clazz.cast(status()));
        case "statusReason":
            return Optional.ofNullable(clazz.cast(statusReason()));
        case "agentConnected":
            return Optional.ofNullable(clazz.cast(agentConnected()));
        case "runningTasksCount":
            return Optional.ofNullable(clazz.cast(runningTasksCount()));
        case "pendingTasksCount":
            return Optional.ofNullable(clazz.cast(pendingTasksCount()));
        case "agentUpdateStatus":
            return Optional.ofNullable(clazz.cast(agentUpdateStatusAsString()));
        case "attributes":
            return Optional.ofNullable(clazz.cast(attributes()));
        case "registeredAt":
            return Optional.ofNullable(clazz.cast(registeredAt()));
        case "attachments":
            return Optional.ofNullable(clazz.cast(attachments()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ContainerInstance, T> g) {
        return obj -> g.apply((ContainerInstance) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ContainerInstance> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the container instance. The ARN contains the <code>arn:aws:ecs</code>
         * namespace, followed by the Region of the container instance, the AWS account ID of the container instance
         * owner, the <code>container-instance</code> namespace, and then the container instance ID. For example,
         * <code>arn:aws:ecs:region:aws_account_id:container-instance/container_instance_ID</code>.
         * </p>
         * 
         * @param containerInstanceArn
         *        The Amazon Resource Name (ARN) of the container instance. The ARN contains the
         *        <code>arn:aws:ecs</code> namespace, followed by the Region of the container instance, the AWS account
         *        ID of the container instance owner, the <code>container-instance</code> namespace, and then the
         *        container instance ID. For example,
         *        <code>arn:aws:ecs:region:aws_account_id:container-instance/container_instance_ID</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerInstanceArn(String containerInstanceArn);

        /**
         * <p>
         * The EC2 instance ID of the container instance.
         * </p>
         * 
         * @param ec2InstanceId
         *        The EC2 instance ID of the container instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ec2InstanceId(String ec2InstanceId);

        /**
         * <p>
         * The capacity provider associated with the container instance.
         * </p>
         * 
         * @param capacityProviderName
         *        The capacity provider associated with the container instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityProviderName(String capacityProviderName);

        /**
         * <p>
         * The version counter for the container instance. Every time a container instance experiences a change that
         * triggers a CloudWatch event, the version counter is incremented. If you are replicating your Amazon ECS
         * container instance state with CloudWatch Events, you can compare the version of a container instance reported
         * by the Amazon ECS APIs with the version reported in CloudWatch Events for the container instance (inside the
         * <code>detail</code> object) to verify that the version in your event stream is current.
         * </p>
         * 
         * @param version
         *        The version counter for the container instance. Every time a container instance experiences a change
         *        that triggers a CloudWatch event, the version counter is incremented. If you are replicating your
         *        Amazon ECS container instance state with CloudWatch Events, you can compare the version of a container
         *        instance reported by the Amazon ECS APIs with the version reported in CloudWatch Events for the
         *        container instance (inside the <code>detail</code> object) to verify that the version in your event
         *        stream is current.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(Long version);

        /**
         * <p>
         * The version information for the Amazon ECS container agent and Docker daemon running on the container
         * instance.
         * </p>
         * 
         * @param versionInfo
         *        The version information for the Amazon ECS container agent and Docker daemon running on the container
         *        instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionInfo(VersionInfo versionInfo);

        /**
         * <p>
         * The version information for the Amazon ECS container agent and Docker daemon running on the container
         * instance.
         * </p>
         * This is a convenience that creates an instance of the {@link VersionInfo.Builder} avoiding the need to create
         * one manually via {@link VersionInfo#builder()}.
         *
         * When the {@link Consumer} completes, {@link VersionInfo.Builder#build()} is called immediately and its result
         * is passed to {@link #versionInfo(VersionInfo)}.
         * 
         * @param versionInfo
         *        a consumer that will call methods on {@link VersionInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #versionInfo(VersionInfo)
         */
        default Builder versionInfo(Consumer<VersionInfo.Builder> versionInfo) {
            return versionInfo(VersionInfo.builder().applyMutation(versionInfo).build());
        }

        /**
         * <p>
         * For CPU and memory resource types, this parameter describes the remaining CPU and memory that has not already
         * been allocated to tasks and is therefore available for new tasks. For port resource types, this parameter
         * describes the ports that were reserved by the Amazon ECS container agent (at instance registration time) and
         * any task containers that have reserved port mappings on the host (with the <code>host</code> or
         * <code>bridge</code> network mode). Any port that is not specified here is available for new tasks.
         * </p>
         * 
         * @param remainingResources
         *        For CPU and memory resource types, this parameter describes the remaining CPU and memory that has not
         *        already been allocated to tasks and is therefore available for new tasks. For port resource types,
         *        this parameter describes the ports that were reserved by the Amazon ECS container agent (at instance
         *        registration time) and any task containers that have reserved port mappings on the host (with the
         *        <code>host</code> or <code>bridge</code> network mode). Any port that is not specified here is
         *        available for new tasks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder remainingResources(Collection<Resource> remainingResources);

        /**
         * <p>
         * For CPU and memory resource types, this parameter describes the remaining CPU and memory that has not already
         * been allocated to tasks and is therefore available for new tasks. For port resource types, this parameter
         * describes the ports that were reserved by the Amazon ECS container agent (at instance registration time) and
         * any task containers that have reserved port mappings on the host (with the <code>host</code> or
         * <code>bridge</code> network mode). Any port that is not specified here is available for new tasks.
         * </p>
         * 
         * @param remainingResources
         *        For CPU and memory resource types, this parameter describes the remaining CPU and memory that has not
         *        already been allocated to tasks and is therefore available for new tasks. For port resource types,
         *        this parameter describes the ports that were reserved by the Amazon ECS container agent (at instance
         *        registration time) and any task containers that have reserved port mappings on the host (with the
         *        <code>host</code> or <code>bridge</code> network mode). Any port that is not specified here is
         *        available for new tasks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder remainingResources(Resource... remainingResources);

        /**
         * <p>
         * For CPU and memory resource types, this parameter describes the remaining CPU and memory that has not already
         * been allocated to tasks and is therefore available for new tasks. For port resource types, this parameter
         * describes the ports that were reserved by the Amazon ECS container agent (at instance registration time) and
         * any task containers that have reserved port mappings on the host (with the <code>host</code> or
         * <code>bridge</code> network mode). Any port that is not specified here is available for new tasks.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Resource>.Builder} avoiding the need to
         * create one manually via {@link List<Resource>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Resource>.Builder#build()} is called immediately and its
         * result is passed to {@link #remainingResources(List<Resource>)}.
         * 
         * @param remainingResources
         *        a consumer that will call methods on {@link List<Resource>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #remainingResources(List<Resource>)
         */
        Builder remainingResources(Consumer<Resource.Builder>... remainingResources);

        /**
         * <p>
         * For CPU and memory resource types, this parameter describes the amount of each resource that was available on
         * the container instance when the container agent registered it with Amazon ECS. This value represents the
         * total amount of CPU and memory that can be allocated on this container instance to tasks. For port resource
         * types, this parameter describes the ports that were reserved by the Amazon ECS container agent when it
         * registered the container instance with Amazon ECS.
         * </p>
         * 
         * @param registeredResources
         *        For CPU and memory resource types, this parameter describes the amount of each resource that was
         *        available on the container instance when the container agent registered it with Amazon ECS. This value
         *        represents the total amount of CPU and memory that can be allocated on this container instance to
         *        tasks. For port resource types, this parameter describes the ports that were reserved by the Amazon
         *        ECS container agent when it registered the container instance with Amazon ECS.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder registeredResources(Collection<Resource> registeredResources);

        /**
         * <p>
         * For CPU and memory resource types, this parameter describes the amount of each resource that was available on
         * the container instance when the container agent registered it with Amazon ECS. This value represents the
         * total amount of CPU and memory that can be allocated on this container instance to tasks. For port resource
         * types, this parameter describes the ports that were reserved by the Amazon ECS container agent when it
         * registered the container instance with Amazon ECS.
         * </p>
         * 
         * @param registeredResources
         *        For CPU and memory resource types, this parameter describes the amount of each resource that was
         *        available on the container instance when the container agent registered it with Amazon ECS. This value
         *        represents the total amount of CPU and memory that can be allocated on this container instance to
         *        tasks. For port resource types, this parameter describes the ports that were reserved by the Amazon
         *        ECS container agent when it registered the container instance with Amazon ECS.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder registeredResources(Resource... registeredResources);

        /**
         * <p>
         * For CPU and memory resource types, this parameter describes the amount of each resource that was available on
         * the container instance when the container agent registered it with Amazon ECS. This value represents the
         * total amount of CPU and memory that can be allocated on this container instance to tasks. For port resource
         * types, this parameter describes the ports that were reserved by the Amazon ECS container agent when it
         * registered the container instance with Amazon ECS.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Resource>.Builder} avoiding the need to
         * create one manually via {@link List<Resource>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Resource>.Builder#build()} is called immediately and its
         * result is passed to {@link #registeredResources(List<Resource>)}.
         * 
         * @param registeredResources
         *        a consumer that will call methods on {@link List<Resource>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #registeredResources(List<Resource>)
         */
        Builder registeredResources(Consumer<Resource.Builder>... registeredResources);

        /**
         * <p>
         * The status of the container instance. The valid values are <code>REGISTERING</code>,
         * <code>REGISTRATION_FAILED</code>, <code>ACTIVE</code>, <code>INACTIVE</code>, <code>DEREGISTERING</code>, or
         * <code>DRAINING</code>.
         * </p>
         * <p>
         * If your account has opted in to the <code>awsvpcTrunking</code> account setting, then any newly registered
         * container instance will transition to a <code>REGISTERING</code> status while the trunk elastic network
         * interface is provisioned for the instance. If the registration fails, the instance will transition to a
         * <code>REGISTRATION_FAILED</code> status. You can describe the container instance and see the reason for
         * failure in the <code>statusReason</code> parameter. Once the container instance is terminated, the instance
         * transitions to a <code>DEREGISTERING</code> status while the trunk elastic network interface is
         * deprovisioned. The instance then transitions to an <code>INACTIVE</code> status.
         * </p>
         * <p>
         * The <code>ACTIVE</code> status indicates that the container instance can accept tasks. The
         * <code>DRAINING</code> indicates that new tasks are not placed on the container instance and any service tasks
         * running on the container instance are removed if possible. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-instance-draining.html">Container
         * Instance Draining</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param status
         *        The status of the container instance. The valid values are <code>REGISTERING</code>,
         *        <code>REGISTRATION_FAILED</code>, <code>ACTIVE</code>, <code>INACTIVE</code>,
         *        <code>DEREGISTERING</code>, or <code>DRAINING</code>.</p>
         *        <p>
         *        If your account has opted in to the <code>awsvpcTrunking</code> account setting, then any newly
         *        registered container instance will transition to a <code>REGISTERING</code> status while the trunk
         *        elastic network interface is provisioned for the instance. If the registration fails, the instance
         *        will transition to a <code>REGISTRATION_FAILED</code> status. You can describe the container instance
         *        and see the reason for failure in the <code>statusReason</code> parameter. Once the container instance
         *        is terminated, the instance transitions to a <code>DEREGISTERING</code> status while the trunk elastic
         *        network interface is deprovisioned. The instance then transitions to an <code>INACTIVE</code> status.
         *        </p>
         *        <p>
         *        The <code>ACTIVE</code> status indicates that the container instance can accept tasks. The
         *        <code>DRAINING</code> indicates that new tasks are not placed on the container instance and any
         *        service tasks running on the container instance are removed if possible. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-instance-draining.html">
         *        Container Instance Draining</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);

        /**
         * <p>
         * The reason that the container instance reached its current status.
         * </p>
         * 
         * @param statusReason
         *        The reason that the container instance reached its current status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReason(String statusReason);

        /**
         * <p>
         * This parameter returns <code>true</code> if the agent is connected to Amazon ECS. Registered instances with
         * an agent that may be unhealthy or stopped return <code>false</code>. Only instances connected to an agent can
         * accept placement requests.
         * </p>
         * 
         * @param agentConnected
         *        This parameter returns <code>true</code> if the agent is connected to Amazon ECS. Registered instances
         *        with an agent that may be unhealthy or stopped return <code>false</code>. Only instances connected to
         *        an agent can accept placement requests.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentConnected(Boolean agentConnected);

        /**
         * <p>
         * The number of tasks on the container instance that are in the <code>RUNNING</code> status.
         * </p>
         * 
         * @param runningTasksCount
         *        The number of tasks on the container instance that are in the <code>RUNNING</code> status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder runningTasksCount(Integer runningTasksCount);

        /**
         * <p>
         * The number of tasks on the container instance that are in the <code>PENDING</code> status.
         * </p>
         * 
         * @param pendingTasksCount
         *        The number of tasks on the container instance that are in the <code>PENDING</code> status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pendingTasksCount(Integer pendingTasksCount);

        /**
         * <p>
         * The status of the most recent agent update. If an update has never been requested, this value is
         * <code>NULL</code>.
         * </p>
         * 
         * @param agentUpdateStatus
         *        The status of the most recent agent update. If an update has never been requested, this value is
         *        <code>NULL</code>.
         * @see AgentUpdateStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AgentUpdateStatus
         */
        Builder agentUpdateStatus(String agentUpdateStatus);

        /**
         * <p>
         * The status of the most recent agent update. If an update has never been requested, this value is
         * <code>NULL</code>.
         * </p>
         * 
         * @param agentUpdateStatus
         *        The status of the most recent agent update. If an update has never been requested, this value is
         *        <code>NULL</code>.
         * @see AgentUpdateStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AgentUpdateStatus
         */
        Builder agentUpdateStatus(AgentUpdateStatus agentUpdateStatus);

        /**
         * <p>
         * The attributes set for the container instance, either by the Amazon ECS container agent at instance
         * registration or manually with the <a>PutAttributes</a> operation.
         * </p>
         * 
         * @param attributes
         *        The attributes set for the container instance, either by the Amazon ECS container agent at instance
         *        registration or manually with the <a>PutAttributes</a> operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(Collection<Attribute> attributes);

        /**
         * <p>
         * The attributes set for the container instance, either by the Amazon ECS container agent at instance
         * registration or manually with the <a>PutAttributes</a> operation.
         * </p>
         * 
         * @param attributes
         *        The attributes set for the container instance, either by the Amazon ECS container agent at instance
         *        registration or manually with the <a>PutAttributes</a> operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(Attribute... attributes);

        /**
         * <p>
         * The attributes set for the container instance, either by the Amazon ECS container agent at instance
         * registration or manually with the <a>PutAttributes</a> operation.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Attribute>.Builder} avoiding the need to
         * create one manually via {@link List<Attribute>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Attribute>.Builder#build()} is called immediately and its
         * result is passed to {@link #attributes(List<Attribute>)}.
         * 
         * @param attributes
         *        a consumer that will call methods on {@link List<Attribute>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #attributes(List<Attribute>)
         */
        Builder attributes(Consumer<Attribute.Builder>... attributes);

        /**
         * <p>
         * The Unix timestamp for when the container instance was registered.
         * </p>
         * 
         * @param registeredAt
         *        The Unix timestamp for when the container instance was registered.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder registeredAt(Instant registeredAt);

        /**
         * <p>
         * The resources attached to a container instance, such as elastic network interfaces.
         * </p>
         * 
         * @param attachments
         *        The resources attached to a container instance, such as elastic network interfaces.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachments(Collection<Attachment> attachments);

        /**
         * <p>
         * The resources attached to a container instance, such as elastic network interfaces.
         * </p>
         * 
         * @param attachments
         *        The resources attached to a container instance, such as elastic network interfaces.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachments(Attachment... attachments);

        /**
         * <p>
         * The resources attached to a container instance, such as elastic network interfaces.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Attachment>.Builder} avoiding the need to
         * create one manually via {@link List<Attachment>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Attachment>.Builder#build()} is called immediately and its
         * result is passed to {@link #attachments(List<Attachment>)}.
         * 
         * @param attachments
         *        a consumer that will call methods on {@link List<Attachment>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #attachments(List<Attachment>)
         */
        Builder attachments(Consumer<Attachment.Builder>... attachments);

        /**
         * <p>
         * The metadata that you apply to the container instance to help you categorize and organize them. Each tag
         * consists of a key and an optional value, both of which you define.
         * </p>
         * <p>
         * The following basic restrictions apply to tags:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Maximum number of tags per resource - 50
         * </p>
         * </li>
         * <li>
         * <p>
         * For each resource, each tag key must be unique, and each tag key can have only one value.
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum key length - 128 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum value length - 256 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * If your tagging schema is used across multiple services and resources, remember that other services may have
         * restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces
         * representable in UTF-8, and the following characters: + - = . _ : / @.
         * </p>
         * </li>
         * <li>
         * <p>
         * Tag keys and values are case-sensitive.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a prefix
         * for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with
         * this prefix. Tags with this prefix do not count against your tags per resource limit.
         * </p>
         * </li>
         * </ul>
         * 
         * @param tags
         *        The metadata that you apply to the container instance to help you categorize and organize them. Each
         *        tag consists of a key and an optional value, both of which you define.</p>
         *        <p>
         *        The following basic restrictions apply to tags:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Maximum number of tags per resource - 50
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For each resource, each tag key must be unique, and each tag key can have only one value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum key length - 128 Unicode characters in UTF-8
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum value length - 256 Unicode characters in UTF-8
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If your tagging schema is used across multiple services and resources, remember that other services
         *        may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and
         *        spaces representable in UTF-8, and the following characters: + - = . _ : / @.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Tag keys and values are case-sensitive.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a
         *        prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or
         *        values with this prefix. Tags with this prefix do not count against your tags per resource limit.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The metadata that you apply to the container instance to help you categorize and organize them. Each tag
         * consists of a key and an optional value, both of which you define.
         * </p>
         * <p>
         * The following basic restrictions apply to tags:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Maximum number of tags per resource - 50
         * </p>
         * </li>
         * <li>
         * <p>
         * For each resource, each tag key must be unique, and each tag key can have only one value.
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum key length - 128 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum value length - 256 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * If your tagging schema is used across multiple services and resources, remember that other services may have
         * restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces
         * representable in UTF-8, and the following characters: + - = . _ : / @.
         * </p>
         * </li>
         * <li>
         * <p>
         * Tag keys and values are case-sensitive.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a prefix
         * for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with
         * this prefix. Tags with this prefix do not count against your tags per resource limit.
         * </p>
         * </li>
         * </ul>
         * 
         * @param tags
         *        The metadata that you apply to the container instance to help you categorize and organize them. Each
         *        tag consists of a key and an optional value, both of which you define.</p>
         *        <p>
         *        The following basic restrictions apply to tags:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Maximum number of tags per resource - 50
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For each resource, each tag key must be unique, and each tag key can have only one value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum key length - 128 Unicode characters in UTF-8
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum value length - 256 Unicode characters in UTF-8
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If your tagging schema is used across multiple services and resources, remember that other services
         *        may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and
         *        spaces representable in UTF-8, and the following characters: + - = . _ : / @.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Tag keys and values are case-sensitive.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a
         *        prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or
         *        values with this prefix. Tags with this prefix do not count against your tags per resource limit.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The metadata that you apply to the container instance to help you categorize and organize them. Each tag
         * consists of a key and an optional value, both of which you define.
         * </p>
         * <p>
         * The following basic restrictions apply to tags:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Maximum number of tags per resource - 50
         * </p>
         * </li>
         * <li>
         * <p>
         * For each resource, each tag key must be unique, and each tag key can have only one value.
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum key length - 128 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum value length - 256 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * If your tagging schema is used across multiple services and resources, remember that other services may have
         * restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces
         * representable in UTF-8, and the following characters: + - = . _ : / @.
         * </p>
         * </li>
         * <li>
         * <p>
         * Tag keys and values are case-sensitive.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a prefix
         * for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with
         * this prefix. Tags with this prefix do not count against your tags per resource limit.
         * </p>
         * </li>
         * </ul>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String containerInstanceArn;

        private String ec2InstanceId;

        private String capacityProviderName;

        private Long version;

        private VersionInfo versionInfo;

        private List<Resource> remainingResources = DefaultSdkAutoConstructList.getInstance();

        private List<Resource> registeredResources = DefaultSdkAutoConstructList.getInstance();

        private String status;

        private String statusReason;

        private Boolean agentConnected;

        private Integer runningTasksCount;

        private Integer pendingTasksCount;

        private String agentUpdateStatus;

        private List<Attribute> attributes = DefaultSdkAutoConstructList.getInstance();

        private Instant registeredAt;

        private List<Attachment> attachments = DefaultSdkAutoConstructList.getInstance();

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ContainerInstance model) {
            containerInstanceArn(model.containerInstanceArn);
            ec2InstanceId(model.ec2InstanceId);
            capacityProviderName(model.capacityProviderName);
            version(model.version);
            versionInfo(model.versionInfo);
            remainingResources(model.remainingResources);
            registeredResources(model.registeredResources);
            status(model.status);
            statusReason(model.statusReason);
            agentConnected(model.agentConnected);
            runningTasksCount(model.runningTasksCount);
            pendingTasksCount(model.pendingTasksCount);
            agentUpdateStatus(model.agentUpdateStatus);
            attributes(model.attributes);
            registeredAt(model.registeredAt);
            attachments(model.attachments);
            tags(model.tags);
        }

        public final String getContainerInstanceArn() {
            return containerInstanceArn;
        }

        @Override
        public final Builder containerInstanceArn(String containerInstanceArn) {
            this.containerInstanceArn = containerInstanceArn;
            return this;
        }

        public final void setContainerInstanceArn(String containerInstanceArn) {
            this.containerInstanceArn = containerInstanceArn;
        }

        public final String getEc2InstanceId() {
            return ec2InstanceId;
        }

        @Override
        public final Builder ec2InstanceId(String ec2InstanceId) {
            this.ec2InstanceId = ec2InstanceId;
            return this;
        }

        public final void setEc2InstanceId(String ec2InstanceId) {
            this.ec2InstanceId = ec2InstanceId;
        }

        public final String getCapacityProviderName() {
            return capacityProviderName;
        }

        @Override
        public final Builder capacityProviderName(String capacityProviderName) {
            this.capacityProviderName = capacityProviderName;
            return this;
        }

        public final void setCapacityProviderName(String capacityProviderName) {
            this.capacityProviderName = capacityProviderName;
        }

        public final Long getVersion() {
            return version;
        }

        @Override
        public final Builder version(Long version) {
            this.version = version;
            return this;
        }

        public final void setVersion(Long version) {
            this.version = version;
        }

        public final VersionInfo.Builder getVersionInfo() {
            return versionInfo != null ? versionInfo.toBuilder() : null;
        }

        @Override
        public final Builder versionInfo(VersionInfo versionInfo) {
            this.versionInfo = versionInfo;
            return this;
        }

        public final void setVersionInfo(VersionInfo.BuilderImpl versionInfo) {
            this.versionInfo = versionInfo != null ? versionInfo.build() : null;
        }

        public final Collection<Resource.Builder> getRemainingResources() {
            return remainingResources != null ? remainingResources.stream().map(Resource::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder remainingResources(Collection<Resource> remainingResources) {
            this.remainingResources = ResourcesCopier.copy(remainingResources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder remainingResources(Resource... remainingResources) {
            remainingResources(Arrays.asList(remainingResources));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder remainingResources(Consumer<Resource.Builder>... remainingResources) {
            remainingResources(Stream.of(remainingResources).map(c -> Resource.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setRemainingResources(Collection<Resource.BuilderImpl> remainingResources) {
            this.remainingResources = ResourcesCopier.copyFromBuilder(remainingResources);
        }

        public final Collection<Resource.Builder> getRegisteredResources() {
            return registeredResources != null ? registeredResources.stream().map(Resource::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder registeredResources(Collection<Resource> registeredResources) {
            this.registeredResources = ResourcesCopier.copy(registeredResources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder registeredResources(Resource... registeredResources) {
            registeredResources(Arrays.asList(registeredResources));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder registeredResources(Consumer<Resource.Builder>... registeredResources) {
            registeredResources(Stream.of(registeredResources).map(c -> Resource.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setRegisteredResources(Collection<Resource.BuilderImpl> registeredResources) {
            this.registeredResources = ResourcesCopier.copyFromBuilder(registeredResources);
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getStatusReason() {
            return statusReason;
        }

        @Override
        public final Builder statusReason(String statusReason) {
            this.statusReason = statusReason;
            return this;
        }

        public final void setStatusReason(String statusReason) {
            this.statusReason = statusReason;
        }

        public final Boolean getAgentConnected() {
            return agentConnected;
        }

        @Override
        public final Builder agentConnected(Boolean agentConnected) {
            this.agentConnected = agentConnected;
            return this;
        }

        public final void setAgentConnected(Boolean agentConnected) {
            this.agentConnected = agentConnected;
        }

        public final Integer getRunningTasksCount() {
            return runningTasksCount;
        }

        @Override
        public final Builder runningTasksCount(Integer runningTasksCount) {
            this.runningTasksCount = runningTasksCount;
            return this;
        }

        public final void setRunningTasksCount(Integer runningTasksCount) {
            this.runningTasksCount = runningTasksCount;
        }

        public final Integer getPendingTasksCount() {
            return pendingTasksCount;
        }

        @Override
        public final Builder pendingTasksCount(Integer pendingTasksCount) {
            this.pendingTasksCount = pendingTasksCount;
            return this;
        }

        public final void setPendingTasksCount(Integer pendingTasksCount) {
            this.pendingTasksCount = pendingTasksCount;
        }

        public final String getAgentUpdateStatus() {
            return agentUpdateStatus;
        }

        @Override
        public final Builder agentUpdateStatus(String agentUpdateStatus) {
            this.agentUpdateStatus = agentUpdateStatus;
            return this;
        }

        @Override
        public final Builder agentUpdateStatus(AgentUpdateStatus agentUpdateStatus) {
            this.agentUpdateStatus(agentUpdateStatus == null ? null : agentUpdateStatus.toString());
            return this;
        }

        public final void setAgentUpdateStatus(String agentUpdateStatus) {
            this.agentUpdateStatus = agentUpdateStatus;
        }

        public final Collection<Attribute.Builder> getAttributes() {
            return attributes != null ? attributes.stream().map(Attribute::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder attributes(Collection<Attribute> attributes) {
            this.attributes = AttributesCopier.copy(attributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attributes(Attribute... attributes) {
            attributes(Arrays.asList(attributes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attributes(Consumer<Attribute.Builder>... attributes) {
            attributes(Stream.of(attributes).map(c -> Attribute.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setAttributes(Collection<Attribute.BuilderImpl> attributes) {
            this.attributes = AttributesCopier.copyFromBuilder(attributes);
        }

        public final Instant getRegisteredAt() {
            return registeredAt;
        }

        @Override
        public final Builder registeredAt(Instant registeredAt) {
            this.registeredAt = registeredAt;
            return this;
        }

        public final void setRegisteredAt(Instant registeredAt) {
            this.registeredAt = registeredAt;
        }

        public final Collection<Attachment.Builder> getAttachments() {
            return attachments != null ? attachments.stream().map(Attachment::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder attachments(Collection<Attachment> attachments) {
            this.attachments = AttachmentsCopier.copy(attachments);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attachments(Attachment... attachments) {
            attachments(Arrays.asList(attachments));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attachments(Consumer<Attachment.Builder>... attachments) {
            attachments(Stream.of(attachments).map(c -> Attachment.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setAttachments(Collection<Attachment.BuilderImpl> attachments) {
            this.attachments = AttachmentsCopier.copyFromBuilder(attachments);
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        @Override
        public ContainerInstance build() {
            return new ContainerInstance(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
