/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Optional deployment parameters that control how many tasks run during a deployment and the ordering of stopping and
 * starting tasks.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DeploymentConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<DeploymentConfiguration.Builder, DeploymentConfiguration> {
    private static final SdkField<DeploymentCircuitBreaker> DEPLOYMENT_CIRCUIT_BREAKER_FIELD = SdkField
            .<DeploymentCircuitBreaker> builder(MarshallingType.SDK_POJO).memberName("deploymentCircuitBreaker")
            .getter(getter(DeploymentConfiguration::deploymentCircuitBreaker)).setter(setter(Builder::deploymentCircuitBreaker))
            .constructor(DeploymentCircuitBreaker::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("deploymentCircuitBreaker").build())
            .build();

    private static final SdkField<Integer> MAXIMUM_PERCENT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("maximumPercent").getter(getter(DeploymentConfiguration::maximumPercent))
            .setter(setter(Builder::maximumPercent))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maximumPercent").build()).build();

    private static final SdkField<Integer> MINIMUM_HEALTHY_PERCENT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("minimumHealthyPercent").getter(getter(DeploymentConfiguration::minimumHealthyPercent))
            .setter(setter(Builder::minimumHealthyPercent))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("minimumHealthyPercent").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            DEPLOYMENT_CIRCUIT_BREAKER_FIELD, MAXIMUM_PERCENT_FIELD, MINIMUM_HEALTHY_PERCENT_FIELD));

    private static final long serialVersionUID = 1L;

    private final DeploymentCircuitBreaker deploymentCircuitBreaker;

    private final Integer maximumPercent;

    private final Integer minimumHealthyPercent;

    private DeploymentConfiguration(BuilderImpl builder) {
        this.deploymentCircuitBreaker = builder.deploymentCircuitBreaker;
        this.maximumPercent = builder.maximumPercent;
        this.minimumHealthyPercent = builder.minimumHealthyPercent;
    }

    /**
     * <note>
     * <p>
     * The deployment circuit breaker can only be used for services using the rolling update (<code>ECS</code>)
     * deployment type.
     * </p>
     * </note>
     * <p>
     * The <b>deployment circuit breaker</b> determines whether a service deployment will fail if the service can't
     * reach a steady state. If deployment circuit breaker is enabled, a service deployment will transition to a failed
     * state and stop launching new tasks. If rollback is enabled, when a service deployment fails, the service is
     * rolled back to the last deployment that completed successfully.
     * </p>
     * 
     * @return <p>
     *         The deployment circuit breaker can only be used for services using the rolling update (<code>ECS</code>)
     *         deployment type.
     *         </p>
     *         </note>
     *         <p>
     *         The <b>deployment circuit breaker</b> determines whether a service deployment will fail if the service
     *         can't reach a steady state. If deployment circuit breaker is enabled, a service deployment will
     *         transition to a failed state and stop launching new tasks. If rollback is enabled, when a service
     *         deployment fails, the service is rolled back to the last deployment that completed successfully.
     */
    public DeploymentCircuitBreaker deploymentCircuitBreaker() {
        return deploymentCircuitBreaker;
    }

    /**
     * <p>
     * If a service is using the rolling update (<code>ECS</code>) deployment type, the <b>maximum percent</b> parameter
     * represents an upper limit on the number of tasks in a service that are allowed in the <code>RUNNING</code> or
     * <code>PENDING</code> state during a deployment, as a percentage of the desired number of tasks (rounded down to
     * the nearest integer), and while any container instances are in the <code>DRAINING</code> state if the service
     * contains tasks using the EC2 launch type. This parameter enables you to define the deployment batch size. For
     * example, if your service has a desired number of four tasks and a maximum percent value of 200%, the scheduler
     * may start four new tasks before stopping the four older tasks (provided that the cluster resources required to do
     * this are available). The default value for maximum percent is 200%.
     * </p>
     * <p>
     * If a service is using the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment types and
     * tasks that use the EC2 launch type, the <b>maximum percent</b> value is set to the default value and is used to
     * define the upper limit on the number of the tasks in the service that remain in the <code>RUNNING</code> state
     * while the container instances are in the <code>DRAINING</code> state. If the tasks in the service use the Fargate
     * launch type, the maximum percent value is not used, although it is returned when describing your service.
     * </p>
     * 
     * @return If a service is using the rolling update (<code>ECS</code>) deployment type, the <b>maximum percent</b>
     *         parameter represents an upper limit on the number of tasks in a service that are allowed in the
     *         <code>RUNNING</code> or <code>PENDING</code> state during a deployment, as a percentage of the desired
     *         number of tasks (rounded down to the nearest integer), and while any container instances are in the
     *         <code>DRAINING</code> state if the service contains tasks using the EC2 launch type. This parameter
     *         enables you to define the deployment batch size. For example, if your service has a desired number of
     *         four tasks and a maximum percent value of 200%, the scheduler may start four new tasks before stopping
     *         the four older tasks (provided that the cluster resources required to do this are available). The default
     *         value for maximum percent is 200%.</p>
     *         <p>
     *         If a service is using the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment types
     *         and tasks that use the EC2 launch type, the <b>maximum percent</b> value is set to the default value and
     *         is used to define the upper limit on the number of the tasks in the service that remain in the
     *         <code>RUNNING</code> state while the container instances are in the <code>DRAINING</code> state. If the
     *         tasks in the service use the Fargate launch type, the maximum percent value is not used, although it is
     *         returned when describing your service.
     */
    public Integer maximumPercent() {
        return maximumPercent;
    }

    /**
     * <p>
     * If a service is using the rolling update (<code>ECS</code>) deployment type, the <b>minimum healthy percent</b>
     * represents a lower limit on the number of tasks in a service that must remain in the <code>RUNNING</code> state
     * during a deployment, as a percentage of the desired number of tasks (rounded up to the nearest integer), and
     * while any container instances are in the <code>DRAINING</code> state if the service contains tasks using the EC2
     * launch type. This parameter enables you to deploy without using additional cluster capacity. For example, if your
     * service has a desired number of four tasks and a minimum healthy percent of 50%, the scheduler may stop two
     * existing tasks to free up cluster capacity before starting two new tasks. Tasks for services that <i>do not</i>
     * use a load balancer are considered healthy if they are in the <code>RUNNING</code> state; tasks for services that
     * <i>do</i> use a load balancer are considered healthy if they are in the <code>RUNNING</code> state and they are
     * reported as healthy by the load balancer. The default value for minimum healthy percent is 100%.
     * </p>
     * <p>
     * If a service is using the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment types and
     * tasks that use the EC2 launch type, the <b>minimum healthy percent</b> value is set to the default value and is
     * used to define the lower limit on the number of the tasks in the service that remain in the <code>RUNNING</code>
     * state while the container instances are in the <code>DRAINING</code> state. If the tasks in the service use the
     * Fargate launch type, the minimum healthy percent value is not used, although it is returned when describing your
     * service.
     * </p>
     * 
     * @return If a service is using the rolling update (<code>ECS</code>) deployment type, the <b>minimum healthy
     *         percent</b> represents a lower limit on the number of tasks in a service that must remain in the
     *         <code>RUNNING</code> state during a deployment, as a percentage of the desired number of tasks (rounded
     *         up to the nearest integer), and while any container instances are in the <code>DRAINING</code> state if
     *         the service contains tasks using the EC2 launch type. This parameter enables you to deploy without using
     *         additional cluster capacity. For example, if your service has a desired number of four tasks and a
     *         minimum healthy percent of 50%, the scheduler may stop two existing tasks to free up cluster capacity
     *         before starting two new tasks. Tasks for services that <i>do not</i> use a load balancer are considered
     *         healthy if they are in the <code>RUNNING</code> state; tasks for services that <i>do</i> use a load
     *         balancer are considered healthy if they are in the <code>RUNNING</code> state and they are reported as
     *         healthy by the load balancer. The default value for minimum healthy percent is 100%.</p>
     *         <p>
     *         If a service is using the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment types
     *         and tasks that use the EC2 launch type, the <b>minimum healthy percent</b> value is set to the default
     *         value and is used to define the lower limit on the number of the tasks in the service that remain in the
     *         <code>RUNNING</code> state while the container instances are in the <code>DRAINING</code> state. If the
     *         tasks in the service use the Fargate launch type, the minimum healthy percent value is not used, although
     *         it is returned when describing your service.
     */
    public Integer minimumHealthyPercent() {
        return minimumHealthyPercent;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(deploymentCircuitBreaker());
        hashCode = 31 * hashCode + Objects.hashCode(maximumPercent());
        hashCode = 31 * hashCode + Objects.hashCode(minimumHealthyPercent());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DeploymentConfiguration)) {
            return false;
        }
        DeploymentConfiguration other = (DeploymentConfiguration) obj;
        return Objects.equals(deploymentCircuitBreaker(), other.deploymentCircuitBreaker())
                && Objects.equals(maximumPercent(), other.maximumPercent())
                && Objects.equals(minimumHealthyPercent(), other.minimumHealthyPercent());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("DeploymentConfiguration").add("DeploymentCircuitBreaker", deploymentCircuitBreaker())
                .add("MaximumPercent", maximumPercent()).add("MinimumHealthyPercent", minimumHealthyPercent()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "deploymentCircuitBreaker":
            return Optional.ofNullable(clazz.cast(deploymentCircuitBreaker()));
        case "maximumPercent":
            return Optional.ofNullable(clazz.cast(maximumPercent()));
        case "minimumHealthyPercent":
            return Optional.ofNullable(clazz.cast(minimumHealthyPercent()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DeploymentConfiguration, T> g) {
        return obj -> g.apply((DeploymentConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DeploymentConfiguration> {
        /**
         * <note>
         * <p>
         * The deployment circuit breaker can only be used for services using the rolling update (<code>ECS</code>)
         * deployment type.
         * </p>
         * </note>
         * <p>
         * The <b>deployment circuit breaker</b> determines whether a service deployment will fail if the service can't
         * reach a steady state. If deployment circuit breaker is enabled, a service deployment will transition to a
         * failed state and stop launching new tasks. If rollback is enabled, when a service deployment fails, the
         * service is rolled back to the last deployment that completed successfully.
         * </p>
         * 
         * @param deploymentCircuitBreaker
         *        <p>
         *        The deployment circuit breaker can only be used for services using the rolling update (
         *        <code>ECS</code>) deployment type.
         *        </p>
         *        </note>
         *        <p>
         *        The <b>deployment circuit breaker</b> determines whether a service deployment will fail if the service
         *        can't reach a steady state. If deployment circuit breaker is enabled, a service deployment will
         *        transition to a failed state and stop launching new tasks. If rollback is enabled, when a service
         *        deployment fails, the service is rolled back to the last deployment that completed successfully.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deploymentCircuitBreaker(DeploymentCircuitBreaker deploymentCircuitBreaker);

        /**
         * <note>
         * <p>
         * The deployment circuit breaker can only be used for services using the rolling update (<code>ECS</code>)
         * deployment type.
         * </p>
         * </note>
         * <p>
         * The <b>deployment circuit breaker</b> determines whether a service deployment will fail if the service can't
         * reach a steady state. If deployment circuit breaker is enabled, a service deployment will transition to a
         * failed state and stop launching new tasks. If rollback is enabled, when a service deployment fails, the
         * service is rolled back to the last deployment that completed successfully.
         * </p>
         * This is a convenience that creates an instance of the {@link DeploymentCircuitBreaker.Builder} avoiding the
         * need to create one manually via {@link DeploymentCircuitBreaker#builder()}.
         *
         * When the {@link Consumer} completes, {@link DeploymentCircuitBreaker.Builder#build()} is called immediately
         * and its result is passed to {@link #deploymentCircuitBreaker(DeploymentCircuitBreaker)}.
         * 
         * @param deploymentCircuitBreaker
         *        a consumer that will call methods on {@link DeploymentCircuitBreaker.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #deploymentCircuitBreaker(DeploymentCircuitBreaker)
         */
        default Builder deploymentCircuitBreaker(Consumer<DeploymentCircuitBreaker.Builder> deploymentCircuitBreaker) {
            return deploymentCircuitBreaker(DeploymentCircuitBreaker.builder().applyMutation(deploymentCircuitBreaker).build());
        }

        /**
         * <p>
         * If a service is using the rolling update (<code>ECS</code>) deployment type, the <b>maximum percent</b>
         * parameter represents an upper limit on the number of tasks in a service that are allowed in the
         * <code>RUNNING</code> or <code>PENDING</code> state during a deployment, as a percentage of the desired number
         * of tasks (rounded down to the nearest integer), and while any container instances are in the
         * <code>DRAINING</code> state if the service contains tasks using the EC2 launch type. This parameter enables
         * you to define the deployment batch size. For example, if your service has a desired number of four tasks and
         * a maximum percent value of 200%, the scheduler may start four new tasks before stopping the four older tasks
         * (provided that the cluster resources required to do this are available). The default value for maximum
         * percent is 200%.
         * </p>
         * <p>
         * If a service is using the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment types and
         * tasks that use the EC2 launch type, the <b>maximum percent</b> value is set to the default value and is used
         * to define the upper limit on the number of the tasks in the service that remain in the <code>RUNNING</code>
         * state while the container instances are in the <code>DRAINING</code> state. If the tasks in the service use
         * the Fargate launch type, the maximum percent value is not used, although it is returned when describing your
         * service.
         * </p>
         * 
         * @param maximumPercent
         *        If a service is using the rolling update (<code>ECS</code>) deployment type, the <b>maximum
         *        percent</b> parameter represents an upper limit on the number of tasks in a service that are allowed
         *        in the <code>RUNNING</code> or <code>PENDING</code> state during a deployment, as a percentage of the
         *        desired number of tasks (rounded down to the nearest integer), and while any container instances are
         *        in the <code>DRAINING</code> state if the service contains tasks using the EC2 launch type. This
         *        parameter enables you to define the deployment batch size. For example, if your service has a desired
         *        number of four tasks and a maximum percent value of 200%, the scheduler may start four new tasks
         *        before stopping the four older tasks (provided that the cluster resources required to do this are
         *        available). The default value for maximum percent is 200%.</p>
         *        <p>
         *        If a service is using the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment
         *        types and tasks that use the EC2 launch type, the <b>maximum percent</b> value is set to the default
         *        value and is used to define the upper limit on the number of the tasks in the service that remain in
         *        the <code>RUNNING</code> state while the container instances are in the <code>DRAINING</code> state.
         *        If the tasks in the service use the Fargate launch type, the maximum percent value is not used,
         *        although it is returned when describing your service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maximumPercent(Integer maximumPercent);

        /**
         * <p>
         * If a service is using the rolling update (<code>ECS</code>) deployment type, the <b>minimum healthy
         * percent</b> represents a lower limit on the number of tasks in a service that must remain in the
         * <code>RUNNING</code> state during a deployment, as a percentage of the desired number of tasks (rounded up to
         * the nearest integer), and while any container instances are in the <code>DRAINING</code> state if the service
         * contains tasks using the EC2 launch type. This parameter enables you to deploy without using additional
         * cluster capacity. For example, if your service has a desired number of four tasks and a minimum healthy
         * percent of 50%, the scheduler may stop two existing tasks to free up cluster capacity before starting two new
         * tasks. Tasks for services that <i>do not</i> use a load balancer are considered healthy if they are in the
         * <code>RUNNING</code> state; tasks for services that <i>do</i> use a load balancer are considered healthy if
         * they are in the <code>RUNNING</code> state and they are reported as healthy by the load balancer. The default
         * value for minimum healthy percent is 100%.
         * </p>
         * <p>
         * If a service is using the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment types and
         * tasks that use the EC2 launch type, the <b>minimum healthy percent</b> value is set to the default value and
         * is used to define the lower limit on the number of the tasks in the service that remain in the
         * <code>RUNNING</code> state while the container instances are in the <code>DRAINING</code> state. If the tasks
         * in the service use the Fargate launch type, the minimum healthy percent value is not used, although it is
         * returned when describing your service.
         * </p>
         * 
         * @param minimumHealthyPercent
         *        If a service is using the rolling update (<code>ECS</code>) deployment type, the <b>minimum healthy
         *        percent</b> represents a lower limit on the number of tasks in a service that must remain in the
         *        <code>RUNNING</code> state during a deployment, as a percentage of the desired number of tasks
         *        (rounded up to the nearest integer), and while any container instances are in the
         *        <code>DRAINING</code> state if the service contains tasks using the EC2 launch type. This parameter
         *        enables you to deploy without using additional cluster capacity. For example, if your service has a
         *        desired number of four tasks and a minimum healthy percent of 50%, the scheduler may stop two existing
         *        tasks to free up cluster capacity before starting two new tasks. Tasks for services that <i>do not</i>
         *        use a load balancer are considered healthy if they are in the <code>RUNNING</code> state; tasks for
         *        services that <i>do</i> use a load balancer are considered healthy if they are in the
         *        <code>RUNNING</code> state and they are reported as healthy by the load balancer. The default value
         *        for minimum healthy percent is 100%.</p>
         *        <p>
         *        If a service is using the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment
         *        types and tasks that use the EC2 launch type, the <b>minimum healthy percent</b> value is set to the
         *        default value and is used to define the lower limit on the number of the tasks in the service that
         *        remain in the <code>RUNNING</code> state while the container instances are in the
         *        <code>DRAINING</code> state. If the tasks in the service use the Fargate launch type, the minimum
         *        healthy percent value is not used, although it is returned when describing your service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minimumHealthyPercent(Integer minimumHealthyPercent);
    }

    static final class BuilderImpl implements Builder {
        private DeploymentCircuitBreaker deploymentCircuitBreaker;

        private Integer maximumPercent;

        private Integer minimumHealthyPercent;

        private BuilderImpl() {
        }

        private BuilderImpl(DeploymentConfiguration model) {
            deploymentCircuitBreaker(model.deploymentCircuitBreaker);
            maximumPercent(model.maximumPercent);
            minimumHealthyPercent(model.minimumHealthyPercent);
        }

        public final DeploymentCircuitBreaker.Builder getDeploymentCircuitBreaker() {
            return deploymentCircuitBreaker != null ? deploymentCircuitBreaker.toBuilder() : null;
        }

        @Override
        public final Builder deploymentCircuitBreaker(DeploymentCircuitBreaker deploymentCircuitBreaker) {
            this.deploymentCircuitBreaker = deploymentCircuitBreaker;
            return this;
        }

        public final void setDeploymentCircuitBreaker(DeploymentCircuitBreaker.BuilderImpl deploymentCircuitBreaker) {
            this.deploymentCircuitBreaker = deploymentCircuitBreaker != null ? deploymentCircuitBreaker.build() : null;
        }

        public final Integer getMaximumPercent() {
            return maximumPercent;
        }

        @Override
        public final Builder maximumPercent(Integer maximumPercent) {
            this.maximumPercent = maximumPercent;
            return this;
        }

        public final void setMaximumPercent(Integer maximumPercent) {
            this.maximumPercent = maximumPercent;
        }

        public final Integer getMinimumHealthyPercent() {
            return minimumHealthyPercent;
        }

        @Override
        public final Builder minimumHealthyPercent(Integer minimumHealthyPercent) {
            this.minimumHealthyPercent = minimumHealthyPercent;
            return this;
        }

        public final void setMinimumHealthyPercent(Integer minimumHealthyPercent) {
            this.minimumHealthyPercent = minimumHealthyPercent;
        }

        @Override
        public DeploymentConfiguration build() {
            return new DeploymentConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
