/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a set of Amazon ECS tasks in either an AWS CodeDeploy or an <code>EXTERNAL</code> deployment. An
 * Amazon ECS task set includes details such as the desired number of tasks, how many tasks are running, and whether the
 * task set serves production traffic.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TaskSet implements SdkPojo, Serializable, ToCopyableBuilder<TaskSet.Builder, TaskSet> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(TaskSet::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<String> TASK_SET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("taskSetArn").getter(getter(TaskSet::taskSetArn)).setter(setter(Builder::taskSetArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("taskSetArn").build()).build();

    private static final SdkField<String> SERVICE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("serviceArn").getter(getter(TaskSet::serviceArn)).setter(setter(Builder::serviceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serviceArn").build()).build();

    private static final SdkField<String> CLUSTER_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("clusterArn").getter(getter(TaskSet::clusterArn)).setter(setter(Builder::clusterArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("clusterArn").build()).build();

    private static final SdkField<String> STARTED_BY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("startedBy").getter(getter(TaskSet::startedBy)).setter(setter(Builder::startedBy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startedBy").build()).build();

    private static final SdkField<String> EXTERNAL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("externalId").getter(getter(TaskSet::externalId)).setter(setter(Builder::externalId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("externalId").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(TaskSet::status)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> TASK_DEFINITION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("taskDefinition").getter(getter(TaskSet::taskDefinition)).setter(setter(Builder::taskDefinition))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("taskDefinition").build()).build();

    private static final SdkField<Integer> COMPUTED_DESIRED_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("computedDesiredCount").getter(getter(TaskSet::computedDesiredCount))
            .setter(setter(Builder::computedDesiredCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("computedDesiredCount").build())
            .build();

    private static final SdkField<Integer> PENDING_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("pendingCount").getter(getter(TaskSet::pendingCount)).setter(setter(Builder::pendingCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("pendingCount").build()).build();

    private static final SdkField<Integer> RUNNING_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("runningCount").getter(getter(TaskSet::runningCount)).setter(setter(Builder::runningCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("runningCount").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt").getter(getter(TaskSet::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<Instant> UPDATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("updatedAt").getter(getter(TaskSet::updatedAt)).setter(setter(Builder::updatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("updatedAt").build()).build();

    private static final SdkField<String> LAUNCH_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("launchType").getter(getter(TaskSet::launchTypeAsString)).setter(setter(Builder::launchType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("launchType").build()).build();

    private static final SdkField<List<CapacityProviderStrategyItem>> CAPACITY_PROVIDER_STRATEGY_FIELD = SdkField
            .<List<CapacityProviderStrategyItem>> builder(MarshallingType.LIST)
            .memberName("capacityProviderStrategy")
            .getter(getter(TaskSet::capacityProviderStrategy))
            .setter(setter(Builder::capacityProviderStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("capacityProviderStrategy").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<CapacityProviderStrategyItem> builder(MarshallingType.SDK_POJO)
                                            .constructor(CapacityProviderStrategyItem::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> PLATFORM_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("platformVersion").getter(getter(TaskSet::platformVersion)).setter(setter(Builder::platformVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("platformVersion").build()).build();

    private static final SdkField<NetworkConfiguration> NETWORK_CONFIGURATION_FIELD = SdkField
            .<NetworkConfiguration> builder(MarshallingType.SDK_POJO).memberName("networkConfiguration")
            .getter(getter(TaskSet::networkConfiguration)).setter(setter(Builder::networkConfiguration))
            .constructor(NetworkConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("networkConfiguration").build())
            .build();

    private static final SdkField<List<LoadBalancer>> LOAD_BALANCERS_FIELD = SdkField
            .<List<LoadBalancer>> builder(MarshallingType.LIST)
            .memberName("loadBalancers")
            .getter(getter(TaskSet::loadBalancers))
            .setter(setter(Builder::loadBalancers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("loadBalancers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LoadBalancer> builder(MarshallingType.SDK_POJO)
                                            .constructor(LoadBalancer::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ServiceRegistry>> SERVICE_REGISTRIES_FIELD = SdkField
            .<List<ServiceRegistry>> builder(MarshallingType.LIST)
            .memberName("serviceRegistries")
            .getter(getter(TaskSet::serviceRegistries))
            .setter(setter(Builder::serviceRegistries))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serviceRegistries").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ServiceRegistry> builder(MarshallingType.SDK_POJO)
                                            .constructor(ServiceRegistry::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Scale> SCALE_FIELD = SdkField.<Scale> builder(MarshallingType.SDK_POJO).memberName("scale")
            .getter(getter(TaskSet::scale)).setter(setter(Builder::scale)).constructor(Scale::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("scale").build()).build();

    private static final SdkField<String> STABILITY_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("stabilityStatus").getter(getter(TaskSet::stabilityStatusAsString))
            .setter(setter(Builder::stabilityStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("stabilityStatus").build()).build();

    private static final SdkField<Instant> STABILITY_STATUS_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("stabilityStatusAt").getter(getter(TaskSet::stabilityStatusAt))
            .setter(setter(Builder::stabilityStatusAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("stabilityStatusAt").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("tags")
            .getter(getter(TaskSet::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, TASK_SET_ARN_FIELD,
            SERVICE_ARN_FIELD, CLUSTER_ARN_FIELD, STARTED_BY_FIELD, EXTERNAL_ID_FIELD, STATUS_FIELD, TASK_DEFINITION_FIELD,
            COMPUTED_DESIRED_COUNT_FIELD, PENDING_COUNT_FIELD, RUNNING_COUNT_FIELD, CREATED_AT_FIELD, UPDATED_AT_FIELD,
            LAUNCH_TYPE_FIELD, CAPACITY_PROVIDER_STRATEGY_FIELD, PLATFORM_VERSION_FIELD, NETWORK_CONFIGURATION_FIELD,
            LOAD_BALANCERS_FIELD, SERVICE_REGISTRIES_FIELD, SCALE_FIELD, STABILITY_STATUS_FIELD, STABILITY_STATUS_AT_FIELD,
            TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String taskSetArn;

    private final String serviceArn;

    private final String clusterArn;

    private final String startedBy;

    private final String externalId;

    private final String status;

    private final String taskDefinition;

    private final Integer computedDesiredCount;

    private final Integer pendingCount;

    private final Integer runningCount;

    private final Instant createdAt;

    private final Instant updatedAt;

    private final String launchType;

    private final List<CapacityProviderStrategyItem> capacityProviderStrategy;

    private final String platformVersion;

    private final NetworkConfiguration networkConfiguration;

    private final List<LoadBalancer> loadBalancers;

    private final List<ServiceRegistry> serviceRegistries;

    private final Scale scale;

    private final String stabilityStatus;

    private final Instant stabilityStatusAt;

    private final List<Tag> tags;

    private TaskSet(BuilderImpl builder) {
        this.id = builder.id;
        this.taskSetArn = builder.taskSetArn;
        this.serviceArn = builder.serviceArn;
        this.clusterArn = builder.clusterArn;
        this.startedBy = builder.startedBy;
        this.externalId = builder.externalId;
        this.status = builder.status;
        this.taskDefinition = builder.taskDefinition;
        this.computedDesiredCount = builder.computedDesiredCount;
        this.pendingCount = builder.pendingCount;
        this.runningCount = builder.runningCount;
        this.createdAt = builder.createdAt;
        this.updatedAt = builder.updatedAt;
        this.launchType = builder.launchType;
        this.capacityProviderStrategy = builder.capacityProviderStrategy;
        this.platformVersion = builder.platformVersion;
        this.networkConfiguration = builder.networkConfiguration;
        this.loadBalancers = builder.loadBalancers;
        this.serviceRegistries = builder.serviceRegistries;
        this.scale = builder.scale;
        this.stabilityStatus = builder.stabilityStatus;
        this.stabilityStatusAt = builder.stabilityStatusAt;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The ID of the task set.
     * </p>
     * 
     * @return The ID of the task set.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the task set.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the task set.
     */
    public final String taskSetArn() {
        return taskSetArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the service the task set exists in.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the service the task set exists in.
     */
    public final String serviceArn() {
        return serviceArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the cluster that the service that hosts the task set exists in.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the cluster that the service that hosts the task set exists in.
     */
    public final String clusterArn() {
        return clusterArn;
    }

    /**
     * <p>
     * The tag specified when a task set is started. If the task set is created by an AWS CodeDeploy deployment, the
     * <code>startedBy</code> parameter is <code>CODE_DEPLOY</code>. For a task set created for an external deployment,
     * the startedBy field isn't used.
     * </p>
     * 
     * @return The tag specified when a task set is started. If the task set is created by an AWS CodeDeploy deployment,
     *         the <code>startedBy</code> parameter is <code>CODE_DEPLOY</code>. For a task set created for an external
     *         deployment, the startedBy field isn't used.
     */
    public final String startedBy() {
        return startedBy;
    }

    /**
     * <p>
     * The external ID associated with the task set.
     * </p>
     * <p>
     * If a task set is created by an AWS CodeDeploy deployment, the <code>externalId</code> parameter contains the AWS
     * CodeDeploy deployment ID.
     * </p>
     * <p>
     * If a task set is created for an external deployment and is associated with a service discovery registry, the
     * <code>externalId</code> parameter contains the <code>ECS_TASK_SET_EXTERNAL_ID</code> AWS Cloud Map attribute.
     * </p>
     * 
     * @return The external ID associated with the task set.</p>
     *         <p>
     *         If a task set is created by an AWS CodeDeploy deployment, the <code>externalId</code> parameter contains
     *         the AWS CodeDeploy deployment ID.
     *         </p>
     *         <p>
     *         If a task set is created for an external deployment and is associated with a service discovery registry,
     *         the <code>externalId</code> parameter contains the <code>ECS_TASK_SET_EXTERNAL_ID</code> AWS Cloud Map
     *         attribute.
     */
    public final String externalId() {
        return externalId;
    }

    /**
     * <p>
     * The status of the task set. The following describes each state:
     * </p>
     * <dl>
     * <dt>PRIMARY</dt>
     * <dd>
     * <p>
     * The task set is serving production traffic.
     * </p>
     * </dd>
     * <dt>ACTIVE</dt>
     * <dd>
     * <p>
     * The task set is not serving production traffic.
     * </p>
     * </dd>
     * <dt>DRAINING</dt>
     * <dd>
     * <p>
     * The tasks in the task set are being stopped and their corresponding targets are being deregistered from their
     * target group.
     * </p>
     * </dd>
     * </dl>
     * 
     * @return The status of the task set. The following describes each state:</p>
     *         <dl>
     *         <dt>PRIMARY</dt>
     *         <dd>
     *         <p>
     *         The task set is serving production traffic.
     *         </p>
     *         </dd>
     *         <dt>ACTIVE</dt>
     *         <dd>
     *         <p>
     *         The task set is not serving production traffic.
     *         </p>
     *         </dd>
     *         <dt>DRAINING</dt>
     *         <dd>
     *         <p>
     *         The tasks in the task set are being stopped and their corresponding targets are being deregistered from
     *         their target group.
     *         </p>
     *         </dd>
     */
    public final String status() {
        return status;
    }

    /**
     * <p>
     * The task definition the task set is using.
     * </p>
     * 
     * @return The task definition the task set is using.
     */
    public final String taskDefinition() {
        return taskDefinition;
    }

    /**
     * <p>
     * The computed desired count for the task set. This is calculated by multiplying the service's
     * <code>desiredCount</code> by the task set's <code>scale</code> percentage. The result is always rounded up. For
     * example, if the computed desired count is 1.2, it rounds up to 2 tasks.
     * </p>
     * 
     * @return The computed desired count for the task set. This is calculated by multiplying the service's
     *         <code>desiredCount</code> by the task set's <code>scale</code> percentage. The result is always rounded
     *         up. For example, if the computed desired count is 1.2, it rounds up to 2 tasks.
     */
    public final Integer computedDesiredCount() {
        return computedDesiredCount;
    }

    /**
     * <p>
     * The number of tasks in the task set that are in the <code>PENDING</code> status during a deployment. A task in
     * the <code>PENDING</code> state is preparing to enter the <code>RUNNING</code> state. A task set enters the
     * <code>PENDING</code> status when it launches for the first time or when it is restarted after being in the
     * <code>STOPPED</code> state.
     * </p>
     * 
     * @return The number of tasks in the task set that are in the <code>PENDING</code> status during a deployment. A
     *         task in the <code>PENDING</code> state is preparing to enter the <code>RUNNING</code> state. A task set
     *         enters the <code>PENDING</code> status when it launches for the first time or when it is restarted after
     *         being in the <code>STOPPED</code> state.
     */
    public final Integer pendingCount() {
        return pendingCount;
    }

    /**
     * <p>
     * The number of tasks in the task set that are in the <code>RUNNING</code> status during a deployment. A task in
     * the <code>RUNNING</code> state is running and ready for use.
     * </p>
     * 
     * @return The number of tasks in the task set that are in the <code>RUNNING</code> status during a deployment. A
     *         task in the <code>RUNNING</code> state is running and ready for use.
     */
    public final Integer runningCount() {
        return runningCount;
    }

    /**
     * <p>
     * The Unix timestamp for when the task set was created.
     * </p>
     * 
     * @return The Unix timestamp for when the task set was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The Unix timestamp for when the task set was last updated.
     * </p>
     * 
     * @return The Unix timestamp for when the task set was last updated.
     */
    public final Instant updatedAt() {
        return updatedAt;
    }

    /**
     * <p>
     * The launch type the tasks in the task set are using. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch types</a>
     * in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #launchType} will
     * return {@link LaunchType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #launchTypeAsString}.
     * </p>
     * 
     * @return The launch type the tasks in the task set are using. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch
     *         types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * @see LaunchType
     */
    public final LaunchType launchType() {
        return LaunchType.fromValue(launchType);
    }

    /**
     * <p>
     * The launch type the tasks in the task set are using. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch types</a>
     * in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #launchType} will
     * return {@link LaunchType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #launchTypeAsString}.
     * </p>
     * 
     * @return The launch type the tasks in the task set are using. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch
     *         types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * @see LaunchType
     */
    public final String launchTypeAsString() {
        return launchType;
    }

    /**
     * Returns true if the CapacityProviderStrategy property was specified by the sender (it may be empty), or false if
     * the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasCapacityProviderStrategy() {
        return capacityProviderStrategy != null && !(capacityProviderStrategy instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The capacity provider strategy associated with the task set.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasCapacityProviderStrategy()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The capacity provider strategy associated with the task set.
     */
    public final List<CapacityProviderStrategyItem> capacityProviderStrategy() {
        return capacityProviderStrategy;
    }

    /**
     * <p>
     * The AWS Fargate platform version on which the tasks in the task set are running. A platform version is only
     * specified for tasks run on AWS Fargate. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html">AWS Fargate platform
     * versions</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * 
     * @return The AWS Fargate platform version on which the tasks in the task set are running. A platform version is
     *         only specified for tasks run on AWS Fargate. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html">AWS Fargate
     *         platform versions</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     */
    public final String platformVersion() {
        return platformVersion;
    }

    /**
     * <p>
     * The network configuration for the task set.
     * </p>
     * 
     * @return The network configuration for the task set.
     */
    public final NetworkConfiguration networkConfiguration() {
        return networkConfiguration;
    }

    /**
     * Returns true if the LoadBalancers property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasLoadBalancers() {
        return loadBalancers != null && !(loadBalancers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Details on a load balancer that is used with a task set.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasLoadBalancers()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Details on a load balancer that is used with a task set.
     */
    public final List<LoadBalancer> loadBalancers() {
        return loadBalancers;
    }

    /**
     * Returns true if the ServiceRegistries property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasServiceRegistries() {
        return serviceRegistries != null && !(serviceRegistries instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The details of the service discovery registries to assign to this task set. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-discovery.html">Service discovery</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasServiceRegistries()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The details of the service discovery registries to assign to this task set. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-discovery.html">Service
     *         discovery</a>.
     */
    public final List<ServiceRegistry> serviceRegistries() {
        return serviceRegistries;
    }

    /**
     * <p>
     * A floating-point percentage of the desired number of tasks to place and keep running in the task set.
     * </p>
     * 
     * @return A floating-point percentage of the desired number of tasks to place and keep running in the task set.
     */
    public final Scale scale() {
        return scale;
    }

    /**
     * <p>
     * The stability status, which indicates whether the task set has reached a steady state. If the following
     * conditions are met, the task set will be in <code>STEADY_STATE</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The task <code>runningCount</code> is equal to the <code>computedDesiredCount</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>pendingCount</code> is <code>0</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * There are no tasks running on container instances in the <code>DRAINING</code> status.
     * </p>
     * </li>
     * <li>
     * <p>
     * All tasks are reporting a healthy status from the load balancers, service discovery, and container health checks.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If any of those conditions are not met, the stability status returns <code>STABILIZING</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stabilityStatus}
     * will return {@link StabilityStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #stabilityStatusAsString}.
     * </p>
     * 
     * @return The stability status, which indicates whether the task set has reached a steady state. If the following
     *         conditions are met, the task set will be in <code>STEADY_STATE</code>:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The task <code>runningCount</code> is equal to the <code>computedDesiredCount</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>pendingCount</code> is <code>0</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There are no tasks running on container instances in the <code>DRAINING</code> status.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         All tasks are reporting a healthy status from the load balancers, service discovery, and container health
     *         checks.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If any of those conditions are not met, the stability status returns <code>STABILIZING</code>.
     * @see StabilityStatus
     */
    public final StabilityStatus stabilityStatus() {
        return StabilityStatus.fromValue(stabilityStatus);
    }

    /**
     * <p>
     * The stability status, which indicates whether the task set has reached a steady state. If the following
     * conditions are met, the task set will be in <code>STEADY_STATE</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The task <code>runningCount</code> is equal to the <code>computedDesiredCount</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>pendingCount</code> is <code>0</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * There are no tasks running on container instances in the <code>DRAINING</code> status.
     * </p>
     * </li>
     * <li>
     * <p>
     * All tasks are reporting a healthy status from the load balancers, service discovery, and container health checks.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If any of those conditions are not met, the stability status returns <code>STABILIZING</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stabilityStatus}
     * will return {@link StabilityStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #stabilityStatusAsString}.
     * </p>
     * 
     * @return The stability status, which indicates whether the task set has reached a steady state. If the following
     *         conditions are met, the task set will be in <code>STEADY_STATE</code>:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The task <code>runningCount</code> is equal to the <code>computedDesiredCount</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>pendingCount</code> is <code>0</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There are no tasks running on container instances in the <code>DRAINING</code> status.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         All tasks are reporting a healthy status from the load balancers, service discovery, and container health
     *         checks.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If any of those conditions are not met, the stability status returns <code>STABILIZING</code>.
     * @see StabilityStatus
     */
    public final String stabilityStatusAsString() {
        return stabilityStatus;
    }

    /**
     * <p>
     * The Unix timestamp for when the task set stability status was retrieved.
     * </p>
     * 
     * @return The Unix timestamp for when the task set stability status was retrieved.
     */
    public final Instant stabilityStatusAt() {
        return stabilityStatusAt;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The metadata that you apply to the task set to help you categorize and organize them. Each tag consists of a key
     * and an optional value, both of which you define.
     * </p>
     * <p>
     * The following basic restrictions apply to tags:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Maximum number of tags per resource - 50
     * </p>
     * </li>
     * <li>
     * <p>
     * For each resource, each tag key must be unique, and each tag key can have only one value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Maximum key length - 128 Unicode characters in UTF-8
     * </p>
     * </li>
     * <li>
     * <p>
     * Maximum value length - 256 Unicode characters in UTF-8
     * </p>
     * </li>
     * <li>
     * <p>
     * If your tagging schema is used across multiple services and resources, remember that other services may have
     * restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable
     * in UTF-8, and the following characters: + - = . _ : / @.
     * </p>
     * </li>
     * <li>
     * <p>
     * Tag keys and values are case-sensitive.
     * </p>
     * </li>
     * <li>
     * <p>
     * Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a prefix for
     * either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this
     * prefix. Tags with this prefix do not count against your tags per resource limit.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The metadata that you apply to the task set to help you categorize and organize them. Each tag consists
     *         of a key and an optional value, both of which you define.</p>
     *         <p>
     *         The following basic restrictions apply to tags:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Maximum number of tags per resource - 50
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For each resource, each tag key must be unique, and each tag key can have only one value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Maximum key length - 128 Unicode characters in UTF-8
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Maximum value length - 256 Unicode characters in UTF-8
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If your tagging schema is used across multiple services and resources, remember that other services may
     *         have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces
     *         representable in UTF-8, and the following characters: + - = . _ : / @.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Tag keys and values are case-sensitive.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a
     *         prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or
     *         values with this prefix. Tags with this prefix do not count against your tags per resource limit.
     *         </p>
     *         </li>
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(taskSetArn());
        hashCode = 31 * hashCode + Objects.hashCode(serviceArn());
        hashCode = 31 * hashCode + Objects.hashCode(clusterArn());
        hashCode = 31 * hashCode + Objects.hashCode(startedBy());
        hashCode = 31 * hashCode + Objects.hashCode(externalId());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(taskDefinition());
        hashCode = 31 * hashCode + Objects.hashCode(computedDesiredCount());
        hashCode = 31 * hashCode + Objects.hashCode(pendingCount());
        hashCode = 31 * hashCode + Objects.hashCode(runningCount());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(updatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(launchTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasCapacityProviderStrategy() ? capacityProviderStrategy() : null);
        hashCode = 31 * hashCode + Objects.hashCode(platformVersion());
        hashCode = 31 * hashCode + Objects.hashCode(networkConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasLoadBalancers() ? loadBalancers() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasServiceRegistries() ? serviceRegistries() : null);
        hashCode = 31 * hashCode + Objects.hashCode(scale());
        hashCode = 31 * hashCode + Objects.hashCode(stabilityStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stabilityStatusAt());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TaskSet)) {
            return false;
        }
        TaskSet other = (TaskSet) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(taskSetArn(), other.taskSetArn())
                && Objects.equals(serviceArn(), other.serviceArn()) && Objects.equals(clusterArn(), other.clusterArn())
                && Objects.equals(startedBy(), other.startedBy()) && Objects.equals(externalId(), other.externalId())
                && Objects.equals(status(), other.status()) && Objects.equals(taskDefinition(), other.taskDefinition())
                && Objects.equals(computedDesiredCount(), other.computedDesiredCount())
                && Objects.equals(pendingCount(), other.pendingCount()) && Objects.equals(runningCount(), other.runningCount())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(updatedAt(), other.updatedAt())
                && Objects.equals(launchTypeAsString(), other.launchTypeAsString())
                && hasCapacityProviderStrategy() == other.hasCapacityProviderStrategy()
                && Objects.equals(capacityProviderStrategy(), other.capacityProviderStrategy())
                && Objects.equals(platformVersion(), other.platformVersion())
                && Objects.equals(networkConfiguration(), other.networkConfiguration())
                && hasLoadBalancers() == other.hasLoadBalancers() && Objects.equals(loadBalancers(), other.loadBalancers())
                && hasServiceRegistries() == other.hasServiceRegistries()
                && Objects.equals(serviceRegistries(), other.serviceRegistries()) && Objects.equals(scale(), other.scale())
                && Objects.equals(stabilityStatusAsString(), other.stabilityStatusAsString())
                && Objects.equals(stabilityStatusAt(), other.stabilityStatusAt()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TaskSet").add("Id", id()).add("TaskSetArn", taskSetArn()).add("ServiceArn", serviceArn())
                .add("ClusterArn", clusterArn()).add("StartedBy", startedBy()).add("ExternalId", externalId())
                .add("Status", status()).add("TaskDefinition", taskDefinition())
                .add("ComputedDesiredCount", computedDesiredCount()).add("PendingCount", pendingCount())
                .add("RunningCount", runningCount()).add("CreatedAt", createdAt()).add("UpdatedAt", updatedAt())
                .add("LaunchType", launchTypeAsString())
                .add("CapacityProviderStrategy", hasCapacityProviderStrategy() ? capacityProviderStrategy() : null)
                .add("PlatformVersion", platformVersion()).add("NetworkConfiguration", networkConfiguration())
                .add("LoadBalancers", hasLoadBalancers() ? loadBalancers() : null)
                .add("ServiceRegistries", hasServiceRegistries() ? serviceRegistries() : null).add("Scale", scale())
                .add("StabilityStatus", stabilityStatusAsString()).add("StabilityStatusAt", stabilityStatusAt())
                .add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "taskSetArn":
            return Optional.ofNullable(clazz.cast(taskSetArn()));
        case "serviceArn":
            return Optional.ofNullable(clazz.cast(serviceArn()));
        case "clusterArn":
            return Optional.ofNullable(clazz.cast(clusterArn()));
        case "startedBy":
            return Optional.ofNullable(clazz.cast(startedBy()));
        case "externalId":
            return Optional.ofNullable(clazz.cast(externalId()));
        case "status":
            return Optional.ofNullable(clazz.cast(status()));
        case "taskDefinition":
            return Optional.ofNullable(clazz.cast(taskDefinition()));
        case "computedDesiredCount":
            return Optional.ofNullable(clazz.cast(computedDesiredCount()));
        case "pendingCount":
            return Optional.ofNullable(clazz.cast(pendingCount()));
        case "runningCount":
            return Optional.ofNullable(clazz.cast(runningCount()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "updatedAt":
            return Optional.ofNullable(clazz.cast(updatedAt()));
        case "launchType":
            return Optional.ofNullable(clazz.cast(launchTypeAsString()));
        case "capacityProviderStrategy":
            return Optional.ofNullable(clazz.cast(capacityProviderStrategy()));
        case "platformVersion":
            return Optional.ofNullable(clazz.cast(platformVersion()));
        case "networkConfiguration":
            return Optional.ofNullable(clazz.cast(networkConfiguration()));
        case "loadBalancers":
            return Optional.ofNullable(clazz.cast(loadBalancers()));
        case "serviceRegistries":
            return Optional.ofNullable(clazz.cast(serviceRegistries()));
        case "scale":
            return Optional.ofNullable(clazz.cast(scale()));
        case "stabilityStatus":
            return Optional.ofNullable(clazz.cast(stabilityStatusAsString()));
        case "stabilityStatusAt":
            return Optional.ofNullable(clazz.cast(stabilityStatusAt()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TaskSet, T> g) {
        return obj -> g.apply((TaskSet) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TaskSet> {
        /**
         * <p>
         * The ID of the task set.
         * </p>
         * 
         * @param id
         *        The ID of the task set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the task set.
         * </p>
         * 
         * @param taskSetArn
         *        The Amazon Resource Name (ARN) of the task set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskSetArn(String taskSetArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the service the task set exists in.
         * </p>
         * 
         * @param serviceArn
         *        The Amazon Resource Name (ARN) of the service the task set exists in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceArn(String serviceArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the cluster that the service that hosts the task set exists in.
         * </p>
         * 
         * @param clusterArn
         *        The Amazon Resource Name (ARN) of the cluster that the service that hosts the task set exists in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterArn(String clusterArn);

        /**
         * <p>
         * The tag specified when a task set is started. If the task set is created by an AWS CodeDeploy deployment, the
         * <code>startedBy</code> parameter is <code>CODE_DEPLOY</code>. For a task set created for an external
         * deployment, the startedBy field isn't used.
         * </p>
         * 
         * @param startedBy
         *        The tag specified when a task set is started. If the task set is created by an AWS CodeDeploy
         *        deployment, the <code>startedBy</code> parameter is <code>CODE_DEPLOY</code>. For a task set created
         *        for an external deployment, the startedBy field isn't used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startedBy(String startedBy);

        /**
         * <p>
         * The external ID associated with the task set.
         * </p>
         * <p>
         * If a task set is created by an AWS CodeDeploy deployment, the <code>externalId</code> parameter contains the
         * AWS CodeDeploy deployment ID.
         * </p>
         * <p>
         * If a task set is created for an external deployment and is associated with a service discovery registry, the
         * <code>externalId</code> parameter contains the <code>ECS_TASK_SET_EXTERNAL_ID</code> AWS Cloud Map attribute.
         * </p>
         * 
         * @param externalId
         *        The external ID associated with the task set.</p>
         *        <p>
         *        If a task set is created by an AWS CodeDeploy deployment, the <code>externalId</code> parameter
         *        contains the AWS CodeDeploy deployment ID.
         *        </p>
         *        <p>
         *        If a task set is created for an external deployment and is associated with a service discovery
         *        registry, the <code>externalId</code> parameter contains the <code>ECS_TASK_SET_EXTERNAL_ID</code> AWS
         *        Cloud Map attribute.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder externalId(String externalId);

        /**
         * <p>
         * The status of the task set. The following describes each state:
         * </p>
         * <dl>
         * <dt>PRIMARY</dt>
         * <dd>
         * <p>
         * The task set is serving production traffic.
         * </p>
         * </dd>
         * <dt>ACTIVE</dt>
         * <dd>
         * <p>
         * The task set is not serving production traffic.
         * </p>
         * </dd>
         * <dt>DRAINING</dt>
         * <dd>
         * <p>
         * The tasks in the task set are being stopped and their corresponding targets are being deregistered from their
         * target group.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param status
         *        The status of the task set. The following describes each state:</p>
         *        <dl>
         *        <dt>PRIMARY</dt>
         *        <dd>
         *        <p>
         *        The task set is serving production traffic.
         *        </p>
         *        </dd>
         *        <dt>ACTIVE</dt>
         *        <dd>
         *        <p>
         *        The task set is not serving production traffic.
         *        </p>
         *        </dd>
         *        <dt>DRAINING</dt>
         *        <dd>
         *        <p>
         *        The tasks in the task set are being stopped and their corresponding targets are being deregistered
         *        from their target group.
         *        </p>
         *        </dd>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);

        /**
         * <p>
         * The task definition the task set is using.
         * </p>
         * 
         * @param taskDefinition
         *        The task definition the task set is using.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskDefinition(String taskDefinition);

        /**
         * <p>
         * The computed desired count for the task set. This is calculated by multiplying the service's
         * <code>desiredCount</code> by the task set's <code>scale</code> percentage. The result is always rounded up.
         * For example, if the computed desired count is 1.2, it rounds up to 2 tasks.
         * </p>
         * 
         * @param computedDesiredCount
         *        The computed desired count for the task set. This is calculated by multiplying the service's
         *        <code>desiredCount</code> by the task set's <code>scale</code> percentage. The result is always
         *        rounded up. For example, if the computed desired count is 1.2, it rounds up to 2 tasks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder computedDesiredCount(Integer computedDesiredCount);

        /**
         * <p>
         * The number of tasks in the task set that are in the <code>PENDING</code> status during a deployment. A task
         * in the <code>PENDING</code> state is preparing to enter the <code>RUNNING</code> state. A task set enters the
         * <code>PENDING</code> status when it launches for the first time or when it is restarted after being in the
         * <code>STOPPED</code> state.
         * </p>
         * 
         * @param pendingCount
         *        The number of tasks in the task set that are in the <code>PENDING</code> status during a deployment. A
         *        task in the <code>PENDING</code> state is preparing to enter the <code>RUNNING</code> state. A task
         *        set enters the <code>PENDING</code> status when it launches for the first time or when it is restarted
         *        after being in the <code>STOPPED</code> state.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pendingCount(Integer pendingCount);

        /**
         * <p>
         * The number of tasks in the task set that are in the <code>RUNNING</code> status during a deployment. A task
         * in the <code>RUNNING</code> state is running and ready for use.
         * </p>
         * 
         * @param runningCount
         *        The number of tasks in the task set that are in the <code>RUNNING</code> status during a deployment. A
         *        task in the <code>RUNNING</code> state is running and ready for use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder runningCount(Integer runningCount);

        /**
         * <p>
         * The Unix timestamp for when the task set was created.
         * </p>
         * 
         * @param createdAt
         *        The Unix timestamp for when the task set was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The Unix timestamp for when the task set was last updated.
         * </p>
         * 
         * @param updatedAt
         *        The Unix timestamp for when the task set was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedAt(Instant updatedAt);

        /**
         * <p>
         * The launch type the tasks in the task set are using. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch
         * types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param launchType
         *        The launch type the tasks in the task set are using. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch
         *        types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @see LaunchType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LaunchType
         */
        Builder launchType(String launchType);

        /**
         * <p>
         * The launch type the tasks in the task set are using. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch
         * types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param launchType
         *        The launch type the tasks in the task set are using. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch
         *        types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @see LaunchType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LaunchType
         */
        Builder launchType(LaunchType launchType);

        /**
         * <p>
         * The capacity provider strategy associated with the task set.
         * </p>
         * 
         * @param capacityProviderStrategy
         *        The capacity provider strategy associated with the task set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityProviderStrategy(Collection<CapacityProviderStrategyItem> capacityProviderStrategy);

        /**
         * <p>
         * The capacity provider strategy associated with the task set.
         * </p>
         * 
         * @param capacityProviderStrategy
         *        The capacity provider strategy associated with the task set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityProviderStrategy(CapacityProviderStrategyItem... capacityProviderStrategy);

        /**
         * <p>
         * The capacity provider strategy associated with the task set.
         * </p>
         * This is a convenience that creates an instance of the {@link List<CapacityProviderStrategyItem>.Builder}
         * avoiding the need to create one manually via {@link List<CapacityProviderStrategyItem>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<CapacityProviderStrategyItem>.Builder#build()} is called
         * immediately and its result is passed to {@link #capacityProviderStrategy(List<CapacityProviderStrategyItem>)}
         * .
         * 
         * @param capacityProviderStrategy
         *        a consumer that will call methods on {@link List<CapacityProviderStrategyItem>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #capacityProviderStrategy(List<CapacityProviderStrategyItem>)
         */
        Builder capacityProviderStrategy(Consumer<CapacityProviderStrategyItem.Builder>... capacityProviderStrategy);

        /**
         * <p>
         * The AWS Fargate platform version on which the tasks in the task set are running. A platform version is only
         * specified for tasks run on AWS Fargate. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html">AWS Fargate
         * platform versions</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param platformVersion
         *        The AWS Fargate platform version on which the tasks in the task set are running. A platform version is
         *        only specified for tasks run on AWS Fargate. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html">AWS Fargate
         *        platform versions</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformVersion(String platformVersion);

        /**
         * <p>
         * The network configuration for the task set.
         * </p>
         * 
         * @param networkConfiguration
         *        The network configuration for the task set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkConfiguration(NetworkConfiguration networkConfiguration);

        /**
         * <p>
         * The network configuration for the task set.
         * </p>
         * This is a convenience that creates an instance of the {@link NetworkConfiguration.Builder} avoiding the need
         * to create one manually via {@link NetworkConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link NetworkConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #networkConfiguration(NetworkConfiguration)}.
         * 
         * @param networkConfiguration
         *        a consumer that will call methods on {@link NetworkConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkConfiguration(NetworkConfiguration)
         */
        default Builder networkConfiguration(Consumer<NetworkConfiguration.Builder> networkConfiguration) {
            return networkConfiguration(NetworkConfiguration.builder().applyMutation(networkConfiguration).build());
        }

        /**
         * <p>
         * Details on a load balancer that is used with a task set.
         * </p>
         * 
         * @param loadBalancers
         *        Details on a load balancer that is used with a task set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loadBalancers(Collection<LoadBalancer> loadBalancers);

        /**
         * <p>
         * Details on a load balancer that is used with a task set.
         * </p>
         * 
         * @param loadBalancers
         *        Details on a load balancer that is used with a task set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loadBalancers(LoadBalancer... loadBalancers);

        /**
         * <p>
         * Details on a load balancer that is used with a task set.
         * </p>
         * This is a convenience that creates an instance of the {@link List<LoadBalancer>.Builder} avoiding the need to
         * create one manually via {@link List<LoadBalancer>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<LoadBalancer>.Builder#build()} is called immediately and its
         * result is passed to {@link #loadBalancers(List<LoadBalancer>)}.
         * 
         * @param loadBalancers
         *        a consumer that will call methods on {@link List<LoadBalancer>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #loadBalancers(List<LoadBalancer>)
         */
        Builder loadBalancers(Consumer<LoadBalancer.Builder>... loadBalancers);

        /**
         * <p>
         * The details of the service discovery registries to assign to this task set. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-discovery.html">Service
         * discovery</a>.
         * </p>
         * 
         * @param serviceRegistries
         *        The details of the service discovery registries to assign to this task set. For more information, see
         *        <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-discovery.html">Service
         *        discovery</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceRegistries(Collection<ServiceRegistry> serviceRegistries);

        /**
         * <p>
         * The details of the service discovery registries to assign to this task set. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-discovery.html">Service
         * discovery</a>.
         * </p>
         * 
         * @param serviceRegistries
         *        The details of the service discovery registries to assign to this task set. For more information, see
         *        <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-discovery.html">Service
         *        discovery</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceRegistries(ServiceRegistry... serviceRegistries);

        /**
         * <p>
         * The details of the service discovery registries to assign to this task set. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-discovery.html">Service
         * discovery</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ServiceRegistry>.Builder} avoiding the need
         * to create one manually via {@link List<ServiceRegistry>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ServiceRegistry>.Builder#build()} is called immediately and
         * its result is passed to {@link #serviceRegistries(List<ServiceRegistry>)}.
         * 
         * @param serviceRegistries
         *        a consumer that will call methods on {@link List<ServiceRegistry>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #serviceRegistries(List<ServiceRegistry>)
         */
        Builder serviceRegistries(Consumer<ServiceRegistry.Builder>... serviceRegistries);

        /**
         * <p>
         * A floating-point percentage of the desired number of tasks to place and keep running in the task set.
         * </p>
         * 
         * @param scale
         *        A floating-point percentage of the desired number of tasks to place and keep running in the task set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scale(Scale scale);

        /**
         * <p>
         * A floating-point percentage of the desired number of tasks to place and keep running in the task set.
         * </p>
         * This is a convenience that creates an instance of the {@link Scale.Builder} avoiding the need to create one
         * manually via {@link Scale#builder()}.
         *
         * When the {@link Consumer} completes, {@link Scale.Builder#build()} is called immediately and its result is
         * passed to {@link #scale(Scale)}.
         * 
         * @param scale
         *        a consumer that will call methods on {@link Scale.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scale(Scale)
         */
        default Builder scale(Consumer<Scale.Builder> scale) {
            return scale(Scale.builder().applyMutation(scale).build());
        }

        /**
         * <p>
         * The stability status, which indicates whether the task set has reached a steady state. If the following
         * conditions are met, the task set will be in <code>STEADY_STATE</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The task <code>runningCount</code> is equal to the <code>computedDesiredCount</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * The <code>pendingCount</code> is <code>0</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * There are no tasks running on container instances in the <code>DRAINING</code> status.
         * </p>
         * </li>
         * <li>
         * <p>
         * All tasks are reporting a healthy status from the load balancers, service discovery, and container health
         * checks.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If any of those conditions are not met, the stability status returns <code>STABILIZING</code>.
         * </p>
         * 
         * @param stabilityStatus
         *        The stability status, which indicates whether the task set has reached a steady state. If the
         *        following conditions are met, the task set will be in <code>STEADY_STATE</code>:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The task <code>runningCount</code> is equal to the <code>computedDesiredCount</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The <code>pendingCount</code> is <code>0</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        There are no tasks running on container instances in the <code>DRAINING</code> status.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        All tasks are reporting a healthy status from the load balancers, service discovery, and container
         *        health checks.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If any of those conditions are not met, the stability status returns <code>STABILIZING</code>.
         * @see StabilityStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StabilityStatus
         */
        Builder stabilityStatus(String stabilityStatus);

        /**
         * <p>
         * The stability status, which indicates whether the task set has reached a steady state. If the following
         * conditions are met, the task set will be in <code>STEADY_STATE</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The task <code>runningCount</code> is equal to the <code>computedDesiredCount</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * The <code>pendingCount</code> is <code>0</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * There are no tasks running on container instances in the <code>DRAINING</code> status.
         * </p>
         * </li>
         * <li>
         * <p>
         * All tasks are reporting a healthy status from the load balancers, service discovery, and container health
         * checks.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If any of those conditions are not met, the stability status returns <code>STABILIZING</code>.
         * </p>
         * 
         * @param stabilityStatus
         *        The stability status, which indicates whether the task set has reached a steady state. If the
         *        following conditions are met, the task set will be in <code>STEADY_STATE</code>:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The task <code>runningCount</code> is equal to the <code>computedDesiredCount</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The <code>pendingCount</code> is <code>0</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        There are no tasks running on container instances in the <code>DRAINING</code> status.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        All tasks are reporting a healthy status from the load balancers, service discovery, and container
         *        health checks.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If any of those conditions are not met, the stability status returns <code>STABILIZING</code>.
         * @see StabilityStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StabilityStatus
         */
        Builder stabilityStatus(StabilityStatus stabilityStatus);

        /**
         * <p>
         * The Unix timestamp for when the task set stability status was retrieved.
         * </p>
         * 
         * @param stabilityStatusAt
         *        The Unix timestamp for when the task set stability status was retrieved.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stabilityStatusAt(Instant stabilityStatusAt);

        /**
         * <p>
         * The metadata that you apply to the task set to help you categorize and organize them. Each tag consists of a
         * key and an optional value, both of which you define.
         * </p>
         * <p>
         * The following basic restrictions apply to tags:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Maximum number of tags per resource - 50
         * </p>
         * </li>
         * <li>
         * <p>
         * For each resource, each tag key must be unique, and each tag key can have only one value.
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum key length - 128 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum value length - 256 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * If your tagging schema is used across multiple services and resources, remember that other services may have
         * restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces
         * representable in UTF-8, and the following characters: + - = . _ : / @.
         * </p>
         * </li>
         * <li>
         * <p>
         * Tag keys and values are case-sensitive.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a prefix
         * for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with
         * this prefix. Tags with this prefix do not count against your tags per resource limit.
         * </p>
         * </li>
         * </ul>
         * 
         * @param tags
         *        The metadata that you apply to the task set to help you categorize and organize them. Each tag
         *        consists of a key and an optional value, both of which you define.</p>
         *        <p>
         *        The following basic restrictions apply to tags:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Maximum number of tags per resource - 50
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For each resource, each tag key must be unique, and each tag key can have only one value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum key length - 128 Unicode characters in UTF-8
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum value length - 256 Unicode characters in UTF-8
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If your tagging schema is used across multiple services and resources, remember that other services
         *        may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and
         *        spaces representable in UTF-8, and the following characters: + - = . _ : / @.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Tag keys and values are case-sensitive.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a
         *        prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or
         *        values with this prefix. Tags with this prefix do not count against your tags per resource limit.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The metadata that you apply to the task set to help you categorize and organize them. Each tag consists of a
         * key and an optional value, both of which you define.
         * </p>
         * <p>
         * The following basic restrictions apply to tags:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Maximum number of tags per resource - 50
         * </p>
         * </li>
         * <li>
         * <p>
         * For each resource, each tag key must be unique, and each tag key can have only one value.
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum key length - 128 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum value length - 256 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * If your tagging schema is used across multiple services and resources, remember that other services may have
         * restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces
         * representable in UTF-8, and the following characters: + - = . _ : / @.
         * </p>
         * </li>
         * <li>
         * <p>
         * Tag keys and values are case-sensitive.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a prefix
         * for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with
         * this prefix. Tags with this prefix do not count against your tags per resource limit.
         * </p>
         * </li>
         * </ul>
         * 
         * @param tags
         *        The metadata that you apply to the task set to help you categorize and organize them. Each tag
         *        consists of a key and an optional value, both of which you define.</p>
         *        <p>
         *        The following basic restrictions apply to tags:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Maximum number of tags per resource - 50
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For each resource, each tag key must be unique, and each tag key can have only one value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum key length - 128 Unicode characters in UTF-8
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum value length - 256 Unicode characters in UTF-8
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If your tagging schema is used across multiple services and resources, remember that other services
         *        may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and
         *        spaces representable in UTF-8, and the following characters: + - = . _ : / @.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Tag keys and values are case-sensitive.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a
         *        prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or
         *        values with this prefix. Tags with this prefix do not count against your tags per resource limit.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The metadata that you apply to the task set to help you categorize and organize them. Each tag consists of a
         * key and an optional value, both of which you define.
         * </p>
         * <p>
         * The following basic restrictions apply to tags:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Maximum number of tags per resource - 50
         * </p>
         * </li>
         * <li>
         * <p>
         * For each resource, each tag key must be unique, and each tag key can have only one value.
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum key length - 128 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum value length - 256 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * If your tagging schema is used across multiple services and resources, remember that other services may have
         * restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces
         * representable in UTF-8, and the following characters: + - = . _ : / @.
         * </p>
         * </li>
         * <li>
         * <p>
         * Tag keys and values are case-sensitive.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a prefix
         * for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with
         * this prefix. Tags with this prefix do not count against your tags per resource limit.
         * </p>
         * </li>
         * </ul>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String taskSetArn;

        private String serviceArn;

        private String clusterArn;

        private String startedBy;

        private String externalId;

        private String status;

        private String taskDefinition;

        private Integer computedDesiredCount;

        private Integer pendingCount;

        private Integer runningCount;

        private Instant createdAt;

        private Instant updatedAt;

        private String launchType;

        private List<CapacityProviderStrategyItem> capacityProviderStrategy = DefaultSdkAutoConstructList.getInstance();

        private String platformVersion;

        private NetworkConfiguration networkConfiguration;

        private List<LoadBalancer> loadBalancers = DefaultSdkAutoConstructList.getInstance();

        private List<ServiceRegistry> serviceRegistries = DefaultSdkAutoConstructList.getInstance();

        private Scale scale;

        private String stabilityStatus;

        private Instant stabilityStatusAt;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(TaskSet model) {
            id(model.id);
            taskSetArn(model.taskSetArn);
            serviceArn(model.serviceArn);
            clusterArn(model.clusterArn);
            startedBy(model.startedBy);
            externalId(model.externalId);
            status(model.status);
            taskDefinition(model.taskDefinition);
            computedDesiredCount(model.computedDesiredCount);
            pendingCount(model.pendingCount);
            runningCount(model.runningCount);
            createdAt(model.createdAt);
            updatedAt(model.updatedAt);
            launchType(model.launchType);
            capacityProviderStrategy(model.capacityProviderStrategy);
            platformVersion(model.platformVersion);
            networkConfiguration(model.networkConfiguration);
            loadBalancers(model.loadBalancers);
            serviceRegistries(model.serviceRegistries);
            scale(model.scale);
            stabilityStatus(model.stabilityStatus);
            stabilityStatusAt(model.stabilityStatusAt);
            tags(model.tags);
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getTaskSetArn() {
            return taskSetArn;
        }

        @Override
        public final Builder taskSetArn(String taskSetArn) {
            this.taskSetArn = taskSetArn;
            return this;
        }

        public final void setTaskSetArn(String taskSetArn) {
            this.taskSetArn = taskSetArn;
        }

        public final String getServiceArn() {
            return serviceArn;
        }

        @Override
        public final Builder serviceArn(String serviceArn) {
            this.serviceArn = serviceArn;
            return this;
        }

        public final void setServiceArn(String serviceArn) {
            this.serviceArn = serviceArn;
        }

        public final String getClusterArn() {
            return clusterArn;
        }

        @Override
        public final Builder clusterArn(String clusterArn) {
            this.clusterArn = clusterArn;
            return this;
        }

        public final void setClusterArn(String clusterArn) {
            this.clusterArn = clusterArn;
        }

        public final String getStartedBy() {
            return startedBy;
        }

        @Override
        public final Builder startedBy(String startedBy) {
            this.startedBy = startedBy;
            return this;
        }

        public final void setStartedBy(String startedBy) {
            this.startedBy = startedBy;
        }

        public final String getExternalId() {
            return externalId;
        }

        @Override
        public final Builder externalId(String externalId) {
            this.externalId = externalId;
            return this;
        }

        public final void setExternalId(String externalId) {
            this.externalId = externalId;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getTaskDefinition() {
            return taskDefinition;
        }

        @Override
        public final Builder taskDefinition(String taskDefinition) {
            this.taskDefinition = taskDefinition;
            return this;
        }

        public final void setTaskDefinition(String taskDefinition) {
            this.taskDefinition = taskDefinition;
        }

        public final Integer getComputedDesiredCount() {
            return computedDesiredCount;
        }

        @Override
        public final Builder computedDesiredCount(Integer computedDesiredCount) {
            this.computedDesiredCount = computedDesiredCount;
            return this;
        }

        public final void setComputedDesiredCount(Integer computedDesiredCount) {
            this.computedDesiredCount = computedDesiredCount;
        }

        public final Integer getPendingCount() {
            return pendingCount;
        }

        @Override
        public final Builder pendingCount(Integer pendingCount) {
            this.pendingCount = pendingCount;
            return this;
        }

        public final void setPendingCount(Integer pendingCount) {
            this.pendingCount = pendingCount;
        }

        public final Integer getRunningCount() {
            return runningCount;
        }

        @Override
        public final Builder runningCount(Integer runningCount) {
            this.runningCount = runningCount;
            return this;
        }

        public final void setRunningCount(Integer runningCount) {
            this.runningCount = runningCount;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        public final Instant getUpdatedAt() {
            return updatedAt;
        }

        @Override
        public final Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        public final void setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
        }

        public final String getLaunchType() {
            return launchType;
        }

        @Override
        public final Builder launchType(String launchType) {
            this.launchType = launchType;
            return this;
        }

        @Override
        public final Builder launchType(LaunchType launchType) {
            this.launchType(launchType == null ? null : launchType.toString());
            return this;
        }

        public final void setLaunchType(String launchType) {
            this.launchType = launchType;
        }

        public final Collection<CapacityProviderStrategyItem.Builder> getCapacityProviderStrategy() {
            if (capacityProviderStrategy instanceof SdkAutoConstructList) {
                return null;
            }
            return capacityProviderStrategy != null ? capacityProviderStrategy.stream()
                    .map(CapacityProviderStrategyItem::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder capacityProviderStrategy(Collection<CapacityProviderStrategyItem> capacityProviderStrategy) {
            this.capacityProviderStrategy = CapacityProviderStrategyCopier.copy(capacityProviderStrategy);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder capacityProviderStrategy(CapacityProviderStrategyItem... capacityProviderStrategy) {
            capacityProviderStrategy(Arrays.asList(capacityProviderStrategy));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder capacityProviderStrategy(Consumer<CapacityProviderStrategyItem.Builder>... capacityProviderStrategy) {
            capacityProviderStrategy(Stream.of(capacityProviderStrategy)
                    .map(c -> CapacityProviderStrategyItem.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setCapacityProviderStrategy(
                Collection<CapacityProviderStrategyItem.BuilderImpl> capacityProviderStrategy) {
            this.capacityProviderStrategy = CapacityProviderStrategyCopier.copyFromBuilder(capacityProviderStrategy);
        }

        public final String getPlatformVersion() {
            return platformVersion;
        }

        @Override
        public final Builder platformVersion(String platformVersion) {
            this.platformVersion = platformVersion;
            return this;
        }

        public final void setPlatformVersion(String platformVersion) {
            this.platformVersion = platformVersion;
        }

        public final NetworkConfiguration.Builder getNetworkConfiguration() {
            return networkConfiguration != null ? networkConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder networkConfiguration(NetworkConfiguration networkConfiguration) {
            this.networkConfiguration = networkConfiguration;
            return this;
        }

        public final void setNetworkConfiguration(NetworkConfiguration.BuilderImpl networkConfiguration) {
            this.networkConfiguration = networkConfiguration != null ? networkConfiguration.build() : null;
        }

        public final Collection<LoadBalancer.Builder> getLoadBalancers() {
            if (loadBalancers instanceof SdkAutoConstructList) {
                return null;
            }
            return loadBalancers != null ? loadBalancers.stream().map(LoadBalancer::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder loadBalancers(Collection<LoadBalancer> loadBalancers) {
            this.loadBalancers = LoadBalancersCopier.copy(loadBalancers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder loadBalancers(LoadBalancer... loadBalancers) {
            loadBalancers(Arrays.asList(loadBalancers));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder loadBalancers(Consumer<LoadBalancer.Builder>... loadBalancers) {
            loadBalancers(Stream.of(loadBalancers).map(c -> LoadBalancer.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setLoadBalancers(Collection<LoadBalancer.BuilderImpl> loadBalancers) {
            this.loadBalancers = LoadBalancersCopier.copyFromBuilder(loadBalancers);
        }

        public final Collection<ServiceRegistry.Builder> getServiceRegistries() {
            if (serviceRegistries instanceof SdkAutoConstructList) {
                return null;
            }
            return serviceRegistries != null ? serviceRegistries.stream().map(ServiceRegistry::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder serviceRegistries(Collection<ServiceRegistry> serviceRegistries) {
            this.serviceRegistries = ServiceRegistriesCopier.copy(serviceRegistries);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder serviceRegistries(ServiceRegistry... serviceRegistries) {
            serviceRegistries(Arrays.asList(serviceRegistries));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder serviceRegistries(Consumer<ServiceRegistry.Builder>... serviceRegistries) {
            serviceRegistries(Stream.of(serviceRegistries).map(c -> ServiceRegistry.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setServiceRegistries(Collection<ServiceRegistry.BuilderImpl> serviceRegistries) {
            this.serviceRegistries = ServiceRegistriesCopier.copyFromBuilder(serviceRegistries);
        }

        public final Scale.Builder getScale() {
            return scale != null ? scale.toBuilder() : null;
        }

        @Override
        public final Builder scale(Scale scale) {
            this.scale = scale;
            return this;
        }

        public final void setScale(Scale.BuilderImpl scale) {
            this.scale = scale != null ? scale.build() : null;
        }

        public final String getStabilityStatus() {
            return stabilityStatus;
        }

        @Override
        public final Builder stabilityStatus(String stabilityStatus) {
            this.stabilityStatus = stabilityStatus;
            return this;
        }

        @Override
        public final Builder stabilityStatus(StabilityStatus stabilityStatus) {
            this.stabilityStatus(stabilityStatus == null ? null : stabilityStatus.toString());
            return this;
        }

        public final void setStabilityStatus(String stabilityStatus) {
            this.stabilityStatus = stabilityStatus;
        }

        public final Instant getStabilityStatusAt() {
            return stabilityStatusAt;
        }

        @Override
        public final Builder stabilityStatusAt(Instant stabilityStatusAt) {
            this.stabilityStatusAt = stabilityStatusAt;
            return this;
        }

        public final void setStabilityStatusAt(Instant stabilityStatusAt) {
            this.stabilityStatusAt = stabilityStatusAt;
        }

        public final Collection<Tag.Builder> getTags() {
            if (tags instanceof SdkAutoConstructList) {
                return null;
            }
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        @Override
        public TaskSet build() {
            return new TaskSet(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
