/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The dependencies defined for container startup and shutdown. A container can contain multiple dependencies. When a
 * dependency is defined for container startup, for container shutdown it is reversed.
 * </p>
 * <p>
 * Your Amazon ECS container instances require at least version 1.26.0 of the container agent to enable container
 * dependencies. However, we recommend using the latest container agent version. For information about checking your
 * agent version and updating to the latest version, see <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html">Updating the Amazon ECS
 * Container Agent</a> in the <i>Amazon Elastic Container Service Developer Guide</i>. If you are using an Amazon
 * ECS-optimized Linux AMI, your instance needs at least version 1.26.0-1 of the <code>ecs-init</code> package. If your
 * container instances are launched from version <code>20190301</code> or later, then they contain the required versions
 * of the container agent and <code>ecs-init</code>. For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html">Amazon ECS-optimized Linux
 * AMI</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
 * </p>
 * <note>
 * <p>
 * For tasks using the Fargate launch type, this parameter requires that the task or service uses platform version 1.3.0
 * or later.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ContainerDependency implements SdkPojo, Serializable,
        ToCopyableBuilder<ContainerDependency.Builder, ContainerDependency> {
    private static final SdkField<String> CONTAINER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("containerName").getter(getter(ContainerDependency::containerName))
            .setter(setter(Builder::containerName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerName").build()).build();

    private static final SdkField<String> CONDITION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("condition").getter(getter(ContainerDependency::conditionAsString)).setter(setter(Builder::condition))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("condition").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONTAINER_NAME_FIELD,
            CONDITION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String containerName;

    private final String condition;

    private ContainerDependency(BuilderImpl builder) {
        this.containerName = builder.containerName;
        this.condition = builder.condition;
    }

    /**
     * <p>
     * The name of a container.
     * </p>
     * 
     * @return The name of a container.
     */
    public final String containerName() {
        return containerName;
    }

    /**
     * <p>
     * The dependency condition of the container. The following are the available conditions and their behavior:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>START</code> - This condition emulates the behavior of links and volumes today. It validates that a
     * dependent container is started before permitting other containers to start.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>COMPLETE</code> - This condition validates that a dependent container runs to completion (exits) before
     * permitting other containers to start. This can be useful for nonessential containers that run a script and then
     * exit. This condition cannot be set on an essential container.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SUCCESS</code> - This condition is the same as <code>COMPLETE</code>, but it also requires that the
     * container exits with a <code>zero</code> status. This condition cannot be set on an essential container.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>HEALTHY</code> - This condition validates that the dependent container passes its Docker health check
     * before permitting other containers to start. This requires that the dependent container has health checks
     * configured. This condition is confirmed only at task startup.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #condition} will
     * return {@link ContainerCondition#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #conditionAsString}.
     * </p>
     * 
     * @return The dependency condition of the container. The following are the available conditions and their
     *         behavior:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>START</code> - This condition emulates the behavior of links and volumes today. It validates that a
     *         dependent container is started before permitting other containers to start.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>COMPLETE</code> - This condition validates that a dependent container runs to completion (exits)
     *         before permitting other containers to start. This can be useful for nonessential containers that run a
     *         script and then exit. This condition cannot be set on an essential container.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SUCCESS</code> - This condition is the same as <code>COMPLETE</code>, but it also requires that the
     *         container exits with a <code>zero</code> status. This condition cannot be set on an essential container.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>HEALTHY</code> - This condition validates that the dependent container passes its Docker health
     *         check before permitting other containers to start. This requires that the dependent container has health
     *         checks configured. This condition is confirmed only at task startup.
     *         </p>
     *         </li>
     * @see ContainerCondition
     */
    public final ContainerCondition condition() {
        return ContainerCondition.fromValue(condition);
    }

    /**
     * <p>
     * The dependency condition of the container. The following are the available conditions and their behavior:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>START</code> - This condition emulates the behavior of links and volumes today. It validates that a
     * dependent container is started before permitting other containers to start.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>COMPLETE</code> - This condition validates that a dependent container runs to completion (exits) before
     * permitting other containers to start. This can be useful for nonessential containers that run a script and then
     * exit. This condition cannot be set on an essential container.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SUCCESS</code> - This condition is the same as <code>COMPLETE</code>, but it also requires that the
     * container exits with a <code>zero</code> status. This condition cannot be set on an essential container.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>HEALTHY</code> - This condition validates that the dependent container passes its Docker health check
     * before permitting other containers to start. This requires that the dependent container has health checks
     * configured. This condition is confirmed only at task startup.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #condition} will
     * return {@link ContainerCondition#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #conditionAsString}.
     * </p>
     * 
     * @return The dependency condition of the container. The following are the available conditions and their
     *         behavior:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>START</code> - This condition emulates the behavior of links and volumes today. It validates that a
     *         dependent container is started before permitting other containers to start.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>COMPLETE</code> - This condition validates that a dependent container runs to completion (exits)
     *         before permitting other containers to start. This can be useful for nonessential containers that run a
     *         script and then exit. This condition cannot be set on an essential container.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SUCCESS</code> - This condition is the same as <code>COMPLETE</code>, but it also requires that the
     *         container exits with a <code>zero</code> status. This condition cannot be set on an essential container.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>HEALTHY</code> - This condition validates that the dependent container passes its Docker health
     *         check before permitting other containers to start. This requires that the dependent container has health
     *         checks configured. This condition is confirmed only at task startup.
     *         </p>
     *         </li>
     * @see ContainerCondition
     */
    public final String conditionAsString() {
        return condition;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(containerName());
        hashCode = 31 * hashCode + Objects.hashCode(conditionAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ContainerDependency)) {
            return false;
        }
        ContainerDependency other = (ContainerDependency) obj;
        return Objects.equals(containerName(), other.containerName())
                && Objects.equals(conditionAsString(), other.conditionAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ContainerDependency").add("ContainerName", containerName())
                .add("Condition", conditionAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "containerName":
            return Optional.ofNullable(clazz.cast(containerName()));
        case "condition":
            return Optional.ofNullable(clazz.cast(conditionAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ContainerDependency, T> g) {
        return obj -> g.apply((ContainerDependency) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ContainerDependency> {
        /**
         * <p>
         * The name of a container.
         * </p>
         * 
         * @param containerName
         *        The name of a container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerName(String containerName);

        /**
         * <p>
         * The dependency condition of the container. The following are the available conditions and their behavior:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>START</code> - This condition emulates the behavior of links and volumes today. It validates that a
         * dependent container is started before permitting other containers to start.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>COMPLETE</code> - This condition validates that a dependent container runs to completion (exits) before
         * permitting other containers to start. This can be useful for nonessential containers that run a script and
         * then exit. This condition cannot be set on an essential container.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SUCCESS</code> - This condition is the same as <code>COMPLETE</code>, but it also requires that the
         * container exits with a <code>zero</code> status. This condition cannot be set on an essential container.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>HEALTHY</code> - This condition validates that the dependent container passes its Docker health check
         * before permitting other containers to start. This requires that the dependent container has health checks
         * configured. This condition is confirmed only at task startup.
         * </p>
         * </li>
         * </ul>
         * 
         * @param condition
         *        The dependency condition of the container. The following are the available conditions and their
         *        behavior:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>START</code> - This condition emulates the behavior of links and volumes today. It validates
         *        that a dependent container is started before permitting other containers to start.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>COMPLETE</code> - This condition validates that a dependent container runs to completion (exits)
         *        before permitting other containers to start. This can be useful for nonessential containers that run a
         *        script and then exit. This condition cannot be set on an essential container.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SUCCESS</code> - This condition is the same as <code>COMPLETE</code>, but it also requires that
         *        the container exits with a <code>zero</code> status. This condition cannot be set on an essential
         *        container.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>HEALTHY</code> - This condition validates that the dependent container passes its Docker health
         *        check before permitting other containers to start. This requires that the dependent container has
         *        health checks configured. This condition is confirmed only at task startup.
         *        </p>
         *        </li>
         * @see ContainerCondition
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerCondition
         */
        Builder condition(String condition);

        /**
         * <p>
         * The dependency condition of the container. The following are the available conditions and their behavior:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>START</code> - This condition emulates the behavior of links and volumes today. It validates that a
         * dependent container is started before permitting other containers to start.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>COMPLETE</code> - This condition validates that a dependent container runs to completion (exits) before
         * permitting other containers to start. This can be useful for nonessential containers that run a script and
         * then exit. This condition cannot be set on an essential container.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SUCCESS</code> - This condition is the same as <code>COMPLETE</code>, but it also requires that the
         * container exits with a <code>zero</code> status. This condition cannot be set on an essential container.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>HEALTHY</code> - This condition validates that the dependent container passes its Docker health check
         * before permitting other containers to start. This requires that the dependent container has health checks
         * configured. This condition is confirmed only at task startup.
         * </p>
         * </li>
         * </ul>
         * 
         * @param condition
         *        The dependency condition of the container. The following are the available conditions and their
         *        behavior:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>START</code> - This condition emulates the behavior of links and volumes today. It validates
         *        that a dependent container is started before permitting other containers to start.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>COMPLETE</code> - This condition validates that a dependent container runs to completion (exits)
         *        before permitting other containers to start. This can be useful for nonessential containers that run a
         *        script and then exit. This condition cannot be set on an essential container.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SUCCESS</code> - This condition is the same as <code>COMPLETE</code>, but it also requires that
         *        the container exits with a <code>zero</code> status. This condition cannot be set on an essential
         *        container.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>HEALTHY</code> - This condition validates that the dependent container passes its Docker health
         *        check before permitting other containers to start. This requires that the dependent container has
         *        health checks configured. This condition is confirmed only at task startup.
         *        </p>
         *        </li>
         * @see ContainerCondition
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerCondition
         */
        Builder condition(ContainerCondition condition);
    }

    static final class BuilderImpl implements Builder {
        private String containerName;

        private String condition;

        private BuilderImpl() {
        }

        private BuilderImpl(ContainerDependency model) {
            containerName(model.containerName);
            condition(model.condition);
        }

        public final String getContainerName() {
            return containerName;
        }

        @Override
        public final Builder containerName(String containerName) {
            this.containerName = containerName;
            return this;
        }

        public final void setContainerName(String containerName) {
            this.containerName = containerName;
        }

        public final String getCondition() {
            return condition;
        }

        @Override
        public final Builder condition(String condition) {
            this.condition = condition;
            return this;
        }

        @Override
        public final Builder condition(ContainerCondition condition) {
            this.condition(condition == null ? null : condition.toString());
            return this;
        }

        public final void setCondition(String condition) {
            this.condition = condition;
        }

        @Override
        public ContainerDependency build() {
            return new ContainerDependency(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
