/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Port mappings allow containers to access ports on the host container instance to send or receive traffic. Port
 * mappings are specified as part of the container definition.
 * </p>
 * <p>
 * If you are using containers in a task with the <code>awsvpc</code> or <code>host</code> network mode, exposed ports
 * should be specified using <code>containerPort</code>. The <code>hostPort</code> can be left blank or it must be the
 * same value as the <code>containerPort</code>.
 * </p>
 * <note>
 * <p>
 * You cannot expose the same container port for multiple protocols. An error will be returned if this is attempted
 * </p>
 * </note>
 * <p>
 * After a task reaches the <code>RUNNING</code> status, manual and automatic host and container port assignments are
 * visible in the <code>networkBindings</code> section of <a>DescribeTasks</a> API responses.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PortMapping implements SdkPojo, Serializable, ToCopyableBuilder<PortMapping.Builder, PortMapping> {
    private static final SdkField<Integer> CONTAINER_PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("containerPort").getter(getter(PortMapping::containerPort)).setter(setter(Builder::containerPort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerPort").build()).build();

    private static final SdkField<Integer> HOST_PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("hostPort").getter(getter(PortMapping::hostPort)).setter(setter(Builder::hostPort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("hostPort").build()).build();

    private static final SdkField<String> PROTOCOL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("protocol").getter(getter(PortMapping::protocolAsString)).setter(setter(Builder::protocol))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("protocol").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONTAINER_PORT_FIELD,
            HOST_PORT_FIELD, PROTOCOL_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer containerPort;

    private final Integer hostPort;

    private final String protocol;

    private PortMapping(BuilderImpl builder) {
        this.containerPort = builder.containerPort;
        this.hostPort = builder.hostPort;
        this.protocol = builder.protocol;
    }

    /**
     * <p>
     * The port number on the container that is bound to the user-specified or automatically assigned host port.
     * </p>
     * <p>
     * If you are using containers in a task with the <code>awsvpc</code> or <code>host</code> network mode, exposed
     * ports should be specified using <code>containerPort</code>.
     * </p>
     * <p>
     * If you are using containers in a task with the <code>bridge</code> network mode and you specify a container port
     * and not a host port, your container automatically receives a host port in the ephemeral port range. For more
     * information, see <code>hostPort</code>. Port mappings that are automatically assigned in this way do not count
     * toward the 100 reserved ports limit of a container instance.
     * </p>
     * 
     * @return The port number on the container that is bound to the user-specified or automatically assigned host
     *         port.</p>
     *         <p>
     *         If you are using containers in a task with the <code>awsvpc</code> or <code>host</code> network mode,
     *         exposed ports should be specified using <code>containerPort</code>.
     *         </p>
     *         <p>
     *         If you are using containers in a task with the <code>bridge</code> network mode and you specify a
     *         container port and not a host port, your container automatically receives a host port in the ephemeral
     *         port range. For more information, see <code>hostPort</code>. Port mappings that are automatically
     *         assigned in this way do not count toward the 100 reserved ports limit of a container instance.
     */
    public final Integer containerPort() {
        return containerPort;
    }

    /**
     * <p>
     * The port number on the container instance to reserve for your container.
     * </p>
     * <p>
     * If you are using containers in a task with the <code>awsvpc</code> or <code>host</code> network mode, the
     * <code>hostPort</code> can either be left blank or set to the same value as the <code>containerPort</code>.
     * </p>
     * <p>
     * If you are using containers in a task with the <code>bridge</code> network mode, you can specify a non-reserved
     * host port for your container port mapping, or you can omit the <code>hostPort</code> (or set it to <code>0</code>
     * ) while specifying a <code>containerPort</code> and your container automatically receives a port in the ephemeral
     * port range for your container instance operating system and Docker version.
     * </p>
     * <p>
     * The default ephemeral port range for Docker version 1.6.0 and later is listed on the instance under
     * <code>/proc/sys/net/ipv4/ip_local_port_range</code>. If this kernel parameter is unavailable, the default
     * ephemeral port range from 49153 through 65535 is used. Do not attempt to specify a host port in the ephemeral
     * port range as these are reserved for automatic assignment. In general, ports below 32768 are outside of the
     * ephemeral port range.
     * </p>
     * <note>
     * <p>
     * The default ephemeral port range from 49153 through 65535 is always used for Docker versions before 1.6.0.
     * </p>
     * </note>
     * <p>
     * The default reserved ports are 22 for SSH, the Docker ports 2375 and 2376, and the Amazon ECS container agent
     * ports 51678-51680. Any host port that was previously specified in a running task is also reserved while the task
     * is running (after a task stops, the host port is released). The current reserved ports are displayed in the
     * <code>remainingResources</code> of <a>DescribeContainerInstances</a> output. A container instance can have up to
     * 100 reserved ports at a time, including the default reserved ports. Automatically assigned ports don't count
     * toward the 100 reserved ports limit.
     * </p>
     * 
     * @return The port number on the container instance to reserve for your container.</p>
     *         <p>
     *         If you are using containers in a task with the <code>awsvpc</code> or <code>host</code> network mode, the
     *         <code>hostPort</code> can either be left blank or set to the same value as the <code>containerPort</code>
     *         .
     *         </p>
     *         <p>
     *         If you are using containers in a task with the <code>bridge</code> network mode, you can specify a
     *         non-reserved host port for your container port mapping, or you can omit the <code>hostPort</code> (or set
     *         it to <code>0</code>) while specifying a <code>containerPort</code> and your container automatically
     *         receives a port in the ephemeral port range for your container instance operating system and Docker
     *         version.
     *         </p>
     *         <p>
     *         The default ephemeral port range for Docker version 1.6.0 and later is listed on the instance under
     *         <code>/proc/sys/net/ipv4/ip_local_port_range</code>. If this kernel parameter is unavailable, the default
     *         ephemeral port range from 49153 through 65535 is used. Do not attempt to specify a host port in the
     *         ephemeral port range as these are reserved for automatic assignment. In general, ports below 32768 are
     *         outside of the ephemeral port range.
     *         </p>
     *         <note>
     *         <p>
     *         The default ephemeral port range from 49153 through 65535 is always used for Docker versions before
     *         1.6.0.
     *         </p>
     *         </note>
     *         <p>
     *         The default reserved ports are 22 for SSH, the Docker ports 2375 and 2376, and the Amazon ECS container
     *         agent ports 51678-51680. Any host port that was previously specified in a running task is also reserved
     *         while the task is running (after a task stops, the host port is released). The current reserved ports are
     *         displayed in the <code>remainingResources</code> of <a>DescribeContainerInstances</a> output. A container
     *         instance can have up to 100 reserved ports at a time, including the default reserved ports. Automatically
     *         assigned ports don't count toward the 100 reserved ports limit.
     */
    public final Integer hostPort() {
        return hostPort;
    }

    /**
     * <p>
     * The protocol used for the port mapping. Valid values are <code>tcp</code> and <code>udp</code>. The default is
     * <code>tcp</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #protocol} will
     * return {@link TransportProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #protocolAsString}.
     * </p>
     * 
     * @return The protocol used for the port mapping. Valid values are <code>tcp</code> and <code>udp</code>. The
     *         default is <code>tcp</code>.
     * @see TransportProtocol
     */
    public final TransportProtocol protocol() {
        return TransportProtocol.fromValue(protocol);
    }

    /**
     * <p>
     * The protocol used for the port mapping. Valid values are <code>tcp</code> and <code>udp</code>. The default is
     * <code>tcp</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #protocol} will
     * return {@link TransportProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #protocolAsString}.
     * </p>
     * 
     * @return The protocol used for the port mapping. Valid values are <code>tcp</code> and <code>udp</code>. The
     *         default is <code>tcp</code>.
     * @see TransportProtocol
     */
    public final String protocolAsString() {
        return protocol;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(containerPort());
        hashCode = 31 * hashCode + Objects.hashCode(hostPort());
        hashCode = 31 * hashCode + Objects.hashCode(protocolAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PortMapping)) {
            return false;
        }
        PortMapping other = (PortMapping) obj;
        return Objects.equals(containerPort(), other.containerPort()) && Objects.equals(hostPort(), other.hostPort())
                && Objects.equals(protocolAsString(), other.protocolAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PortMapping").add("ContainerPort", containerPort()).add("HostPort", hostPort())
                .add("Protocol", protocolAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "containerPort":
            return Optional.ofNullable(clazz.cast(containerPort()));
        case "hostPort":
            return Optional.ofNullable(clazz.cast(hostPort()));
        case "protocol":
            return Optional.ofNullable(clazz.cast(protocolAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PortMapping, T> g) {
        return obj -> g.apply((PortMapping) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PortMapping> {
        /**
         * <p>
         * The port number on the container that is bound to the user-specified or automatically assigned host port.
         * </p>
         * <p>
         * If you are using containers in a task with the <code>awsvpc</code> or <code>host</code> network mode, exposed
         * ports should be specified using <code>containerPort</code>.
         * </p>
         * <p>
         * If you are using containers in a task with the <code>bridge</code> network mode and you specify a container
         * port and not a host port, your container automatically receives a host port in the ephemeral port range. For
         * more information, see <code>hostPort</code>. Port mappings that are automatically assigned in this way do not
         * count toward the 100 reserved ports limit of a container instance.
         * </p>
         * 
         * @param containerPort
         *        The port number on the container that is bound to the user-specified or automatically assigned host
         *        port.</p>
         *        <p>
         *        If you are using containers in a task with the <code>awsvpc</code> or <code>host</code> network mode,
         *        exposed ports should be specified using <code>containerPort</code>.
         *        </p>
         *        <p>
         *        If you are using containers in a task with the <code>bridge</code> network mode and you specify a
         *        container port and not a host port, your container automatically receives a host port in the ephemeral
         *        port range. For more information, see <code>hostPort</code>. Port mappings that are automatically
         *        assigned in this way do not count toward the 100 reserved ports limit of a container instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerPort(Integer containerPort);

        /**
         * <p>
         * The port number on the container instance to reserve for your container.
         * </p>
         * <p>
         * If you are using containers in a task with the <code>awsvpc</code> or <code>host</code> network mode, the
         * <code>hostPort</code> can either be left blank or set to the same value as the <code>containerPort</code>.
         * </p>
         * <p>
         * If you are using containers in a task with the <code>bridge</code> network mode, you can specify a
         * non-reserved host port for your container port mapping, or you can omit the <code>hostPort</code> (or set it
         * to <code>0</code>) while specifying a <code>containerPort</code> and your container automatically receives a
         * port in the ephemeral port range for your container instance operating system and Docker version.
         * </p>
         * <p>
         * The default ephemeral port range for Docker version 1.6.0 and later is listed on the instance under
         * <code>/proc/sys/net/ipv4/ip_local_port_range</code>. If this kernel parameter is unavailable, the default
         * ephemeral port range from 49153 through 65535 is used. Do not attempt to specify a host port in the ephemeral
         * port range as these are reserved for automatic assignment. In general, ports below 32768 are outside of the
         * ephemeral port range.
         * </p>
         * <note>
         * <p>
         * The default ephemeral port range from 49153 through 65535 is always used for Docker versions before 1.6.0.
         * </p>
         * </note>
         * <p>
         * The default reserved ports are 22 for SSH, the Docker ports 2375 and 2376, and the Amazon ECS container agent
         * ports 51678-51680. Any host port that was previously specified in a running task is also reserved while the
         * task is running (after a task stops, the host port is released). The current reserved ports are displayed in
         * the <code>remainingResources</code> of <a>DescribeContainerInstances</a> output. A container instance can
         * have up to 100 reserved ports at a time, including the default reserved ports. Automatically assigned ports
         * don't count toward the 100 reserved ports limit.
         * </p>
         * 
         * @param hostPort
         *        The port number on the container instance to reserve for your container.</p>
         *        <p>
         *        If you are using containers in a task with the <code>awsvpc</code> or <code>host</code> network mode,
         *        the <code>hostPort</code> can either be left blank or set to the same value as the
         *        <code>containerPort</code>.
         *        </p>
         *        <p>
         *        If you are using containers in a task with the <code>bridge</code> network mode, you can specify a
         *        non-reserved host port for your container port mapping, or you can omit the <code>hostPort</code> (or
         *        set it to <code>0</code>) while specifying a <code>containerPort</code> and your container
         *        automatically receives a port in the ephemeral port range for your container instance operating system
         *        and Docker version.
         *        </p>
         *        <p>
         *        The default ephemeral port range for Docker version 1.6.0 and later is listed on the instance under
         *        <code>/proc/sys/net/ipv4/ip_local_port_range</code>. If this kernel parameter is unavailable, the
         *        default ephemeral port range from 49153 through 65535 is used. Do not attempt to specify a host port
         *        in the ephemeral port range as these are reserved for automatic assignment. In general, ports below
         *        32768 are outside of the ephemeral port range.
         *        </p>
         *        <note>
         *        <p>
         *        The default ephemeral port range from 49153 through 65535 is always used for Docker versions before
         *        1.6.0.
         *        </p>
         *        </note>
         *        <p>
         *        The default reserved ports are 22 for SSH, the Docker ports 2375 and 2376, and the Amazon ECS
         *        container agent ports 51678-51680. Any host port that was previously specified in a running task is
         *        also reserved while the task is running (after a task stops, the host port is released). The current
         *        reserved ports are displayed in the <code>remainingResources</code> of
         *        <a>DescribeContainerInstances</a> output. A container instance can have up to 100 reserved ports at a
         *        time, including the default reserved ports. Automatically assigned ports don't count toward the 100
         *        reserved ports limit.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hostPort(Integer hostPort);

        /**
         * <p>
         * The protocol used for the port mapping. Valid values are <code>tcp</code> and <code>udp</code>. The default
         * is <code>tcp</code>.
         * </p>
         * 
         * @param protocol
         *        The protocol used for the port mapping. Valid values are <code>tcp</code> and <code>udp</code>. The
         *        default is <code>tcp</code>.
         * @see TransportProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransportProtocol
         */
        Builder protocol(String protocol);

        /**
         * <p>
         * The protocol used for the port mapping. Valid values are <code>tcp</code> and <code>udp</code>. The default
         * is <code>tcp</code>.
         * </p>
         * 
         * @param protocol
         *        The protocol used for the port mapping. Valid values are <code>tcp</code> and <code>udp</code>. The
         *        default is <code>tcp</code>.
         * @see TransportProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransportProtocol
         */
        Builder protocol(TransportProtocol protocol);
    }

    static final class BuilderImpl implements Builder {
        private Integer containerPort;

        private Integer hostPort;

        private String protocol;

        private BuilderImpl() {
        }

        private BuilderImpl(PortMapping model) {
            containerPort(model.containerPort);
            hostPort(model.hostPort);
            protocol(model.protocol);
        }

        public final Integer getContainerPort() {
            return containerPort;
        }

        @Override
        public final Builder containerPort(Integer containerPort) {
            this.containerPort = containerPort;
            return this;
        }

        public final void setContainerPort(Integer containerPort) {
            this.containerPort = containerPort;
        }

        public final Integer getHostPort() {
            return hostPort;
        }

        @Override
        public final Builder hostPort(Integer hostPort) {
            this.hostPort = hostPort;
            return this;
        }

        public final void setHostPort(Integer hostPort) {
            this.hostPort = hostPort;
        }

        public final String getProtocol() {
            return protocol;
        }

        @Override
        public final Builder protocol(String protocol) {
            this.protocol = protocol;
            return this;
        }

        @Override
        public final Builder protocol(TransportProtocol protocol) {
            this.protocol(protocol == null ? null : protocol.toString());
            return this;
        }

        public final void setProtocol(String protocol) {
            this.protocol = protocol;
        }

        @Override
        public PortMapping build() {
            return new PortMapping(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
