/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An attribute is a name-value pair that's associated with an Amazon ECS object. Use attributes to extend the Amazon
 * ECS data model by adding custom metadata to your resources. For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html#attributes"
 * >Attributes</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Attribute implements SdkPojo, Serializable, ToCopyableBuilder<Attribute.Builder, Attribute> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Attribute::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("value")
            .getter(getter(Attribute::value)).setter(setter(Builder::value))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("value").build()).build();

    private static final SdkField<String> TARGET_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("targetType").getter(getter(Attribute::targetTypeAsString)).setter(setter(Builder::targetType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("targetType").build()).build();

    private static final SdkField<String> TARGET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("targetId").getter(getter(Attribute::targetId)).setter(setter(Builder::targetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("targetId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, VALUE_FIELD,
            TARGET_TYPE_FIELD, TARGET_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String value;

    private final String targetType;

    private final String targetId;

    private Attribute(BuilderImpl builder) {
        this.name = builder.name;
        this.value = builder.value;
        this.targetType = builder.targetType;
        this.targetId = builder.targetId;
    }

    /**
     * <p>
     * The name of the attribute. The <code>name</code> must contain between 1 and 128 characters. The name may contain
     * letters (uppercase and lowercase), numbers, hyphens (-), underscores (_), forward slashes (/), back slashes (\),
     * or periods (.).
     * </p>
     * 
     * @return The name of the attribute. The <code>name</code> must contain between 1 and 128 characters. The name may
     *         contain letters (uppercase and lowercase), numbers, hyphens (-), underscores (_), forward slashes (/),
     *         back slashes (\), or periods (.).
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The value of the attribute. The <code>value</code> must contain between 1 and 128 characters. It can contain
     * letters (uppercase and lowercase), numbers, hyphens (-), underscores (_), periods (.), at signs (@), forward
     * slashes (/), back slashes (\), colons (:), or spaces. The value can't can't start or end with a space.
     * </p>
     * 
     * @return The value of the attribute. The <code>value</code> must contain between 1 and 128 characters. It can
     *         contain letters (uppercase and lowercase), numbers, hyphens (-), underscores (_), periods (.), at signs
     *         (@), forward slashes (/), back slashes (\), colons (:), or spaces. The value can't can't start or end
     *         with a space.
     */
    public final String value() {
        return value;
    }

    /**
     * <p>
     * The type of the target to attach the attribute with. This parameter is required if you use the short form ID for
     * a resource instead of the full ARN.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #targetType} will
     * return {@link TargetType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #targetTypeAsString}.
     * </p>
     * 
     * @return The type of the target to attach the attribute with. This parameter is required if you use the short form
     *         ID for a resource instead of the full ARN.
     * @see TargetType
     */
    public final TargetType targetType() {
        return TargetType.fromValue(targetType);
    }

    /**
     * <p>
     * The type of the target to attach the attribute with. This parameter is required if you use the short form ID for
     * a resource instead of the full ARN.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #targetType} will
     * return {@link TargetType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #targetTypeAsString}.
     * </p>
     * 
     * @return The type of the target to attach the attribute with. This parameter is required if you use the short form
     *         ID for a resource instead of the full ARN.
     * @see TargetType
     */
    public final String targetTypeAsString() {
        return targetType;
    }

    /**
     * <p>
     * The ID of the target. You can specify the short form ID for a resource or the full Amazon Resource Name (ARN).
     * </p>
     * 
     * @return The ID of the target. You can specify the short form ID for a resource or the full Amazon Resource Name
     *         (ARN).
     */
    public final String targetId() {
        return targetId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(value());
        hashCode = 31 * hashCode + Objects.hashCode(targetTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(targetId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Attribute)) {
            return false;
        }
        Attribute other = (Attribute) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(value(), other.value())
                && Objects.equals(targetTypeAsString(), other.targetTypeAsString())
                && Objects.equals(targetId(), other.targetId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Attribute").add("Name", name()).add("Value", value()).add("TargetType", targetTypeAsString())
                .add("TargetId", targetId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "value":
            return Optional.ofNullable(clazz.cast(value()));
        case "targetType":
            return Optional.ofNullable(clazz.cast(targetTypeAsString()));
        case "targetId":
            return Optional.ofNullable(clazz.cast(targetId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Attribute, T> g) {
        return obj -> g.apply((Attribute) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Attribute> {
        /**
         * <p>
         * The name of the attribute. The <code>name</code> must contain between 1 and 128 characters. The name may
         * contain letters (uppercase and lowercase), numbers, hyphens (-), underscores (_), forward slashes (/), back
         * slashes (\), or periods (.).
         * </p>
         * 
         * @param name
         *        The name of the attribute. The <code>name</code> must contain between 1 and 128 characters. The name
         *        may contain letters (uppercase and lowercase), numbers, hyphens (-), underscores (_), forward slashes
         *        (/), back slashes (\), or periods (.).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The value of the attribute. The <code>value</code> must contain between 1 and 128 characters. It can contain
         * letters (uppercase and lowercase), numbers, hyphens (-), underscores (_), periods (.), at signs (@), forward
         * slashes (/), back slashes (\), colons (:), or spaces. The value can't can't start or end with a space.
         * </p>
         * 
         * @param value
         *        The value of the attribute. The <code>value</code> must contain between 1 and 128 characters. It can
         *        contain letters (uppercase and lowercase), numbers, hyphens (-), underscores (_), periods (.), at
         *        signs (@), forward slashes (/), back slashes (\), colons (:), or spaces. The value can't can't start
         *        or end with a space.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder value(String value);

        /**
         * <p>
         * The type of the target to attach the attribute with. This parameter is required if you use the short form ID
         * for a resource instead of the full ARN.
         * </p>
         * 
         * @param targetType
         *        The type of the target to attach the attribute with. This parameter is required if you use the short
         *        form ID for a resource instead of the full ARN.
         * @see TargetType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetType
         */
        Builder targetType(String targetType);

        /**
         * <p>
         * The type of the target to attach the attribute with. This parameter is required if you use the short form ID
         * for a resource instead of the full ARN.
         * </p>
         * 
         * @param targetType
         *        The type of the target to attach the attribute with. This parameter is required if you use the short
         *        form ID for a resource instead of the full ARN.
         * @see TargetType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetType
         */
        Builder targetType(TargetType targetType);

        /**
         * <p>
         * The ID of the target. You can specify the short form ID for a resource or the full Amazon Resource Name
         * (ARN).
         * </p>
         * 
         * @param targetId
         *        The ID of the target. You can specify the short form ID for a resource or the full Amazon Resource
         *        Name (ARN).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetId(String targetId);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String value;

        private String targetType;

        private String targetId;

        private BuilderImpl() {
        }

        private BuilderImpl(Attribute model) {
            name(model.name);
            value(model.value);
            targetType(model.targetType);
            targetId(model.targetId);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getValue() {
            return value;
        }

        public final void setValue(String value) {
            this.value = value;
        }

        @Override
        @Transient
        public final Builder value(String value) {
            this.value = value;
            return this;
        }

        public final String getTargetType() {
            return targetType;
        }

        public final void setTargetType(String targetType) {
            this.targetType = targetType;
        }

        @Override
        @Transient
        public final Builder targetType(String targetType) {
            this.targetType = targetType;
            return this;
        }

        @Override
        @Transient
        public final Builder targetType(TargetType targetType) {
            this.targetType(targetType == null ? null : targetType.toString());
            return this;
        }

        public final String getTargetId() {
            return targetId;
        }

        public final void setTargetId(String targetId) {
            this.targetId = targetId;
        }

        @Override
        @Transient
        public final Builder targetId(String targetId) {
            this.targetId = targetId;
            return this;
        }

        @Override
        public Attribute build() {
            return new Attribute(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
