/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The log configuration for the container. This parameter maps to <code>LogConfig</code> in the <a
 * href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section of the <a
 * href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the <code>--log-driver</code> option to <a
 * href="https://docs.docker.com/engine/reference/commandline/run/"> <code>docker run</code> </a>.
 * </p>
 * <p>
 * By default, containers use the same logging driver that the Docker daemon uses. However, the container might use a
 * different logging driver than the Docker daemon by specifying a log driver configuration in the container definition.
 * For more information about the options for different supported log drivers, see <a
 * href="https://docs.docker.com/engine/admin/logging/overview/">Configure logging drivers</a> in the Docker
 * documentation.
 * </p>
 * <p>
 * Understand the following when specifying a log configuration for your containers.
 * </p>
 * <ul>
 * <li>
 * <p>
 * Amazon ECS currently supports a subset of the logging drivers available to the Docker daemon (shown in the valid
 * values below). Additional log drivers may be available in future releases of the Amazon ECS container agent.
 * </p>
 * </li>
 * <li>
 * <p>
 * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance.
 * </p>
 * </li>
 * <li>
 * <p>
 * For tasks that are hosted on Amazon EC2 instances, the Amazon ECS container agent must register the available logging
 * drivers with the <code>ECS_AVAILABLE_LOGGING_DRIVERS</code> environment variable before containers placed on that
 * instance can use these log configuration options. For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon ECS container agent
 * configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
 * </p>
 * </li>
 * <li>
 * <p>
 * For tasks that are on Fargate, because you don't have access to the underlying infrastructure your tasks are hosted
 * on, any additional software needed must be installed outside of the task. For example, the Fluentd output aggregators
 * or a remote host running Logstash to send Gelf logs to.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LogConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<LogConfiguration.Builder, LogConfiguration> {
    private static final SdkField<String> LOG_DRIVER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("logDriver").getter(getter(LogConfiguration::logDriverAsString)).setter(setter(Builder::logDriver))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("logDriver").build()).build();

    private static final SdkField<Map<String, String>> OPTIONS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("options")
            .getter(getter(LogConfiguration::options))
            .setter(setter(Builder::options))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("options").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<Secret>> SECRET_OPTIONS_FIELD = SdkField
            .<List<Secret>> builder(MarshallingType.LIST)
            .memberName("secretOptions")
            .getter(getter(LogConfiguration::secretOptions))
            .setter(setter(Builder::secretOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("secretOptions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Secret> builder(MarshallingType.SDK_POJO)
                                            .constructor(Secret::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LOG_DRIVER_FIELD,
            OPTIONS_FIELD, SECRET_OPTIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String logDriver;

    private final Map<String, String> options;

    private final List<Secret> secretOptions;

    private LogConfiguration(BuilderImpl builder) {
        this.logDriver = builder.logDriver;
        this.options = builder.options;
        this.secretOptions = builder.secretOptions;
    }

    /**
     * <p>
     * The log driver to use for the container.
     * </p>
     * <p>
     * For tasks on Fargate, the supported log drivers are <code>awslogs</code>, <code>splunk</code>, and
     * <code>awsfirelens</code>.
     * </p>
     * <p>
     * For tasks hosted on Amazon EC2 instances, the supported log drivers are <code>awslogs</code>,
     * <code>fluentd</code>, <code>gelf</code>, <code>json-file</code>, <code>journald</code>, <code>logentries</code>,
     * <code>syslog</code>, <code>splunk</code>, and <code>awsfirelens</code>.
     * </p>
     * <p>
     * For more information about using the <code>awslogs</code> log driver, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html">Using the awslogs log
     * driver</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * For more information about using the <code>awsfirelens</code> log driver, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Custom log routing</a> in
     * the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * If you have a custom driver that isn't listed, you can fork the Amazon ECS container agent project that's <a
     * href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with that driver.
     * We encourage you to submit pull requests for changes that you would like to have included. However, we don't
     * currently provide support for running modified copies of this software.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #logDriver} will
     * return {@link LogDriver#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #logDriverAsString}.
     * </p>
     * 
     * @return The log driver to use for the container.</p>
     *         <p>
     *         For tasks on Fargate, the supported log drivers are <code>awslogs</code>, <code>splunk</code>, and
     *         <code>awsfirelens</code>.
     *         </p>
     *         <p>
     *         For tasks hosted on Amazon EC2 instances, the supported log drivers are <code>awslogs</code>,
     *         <code>fluentd</code>, <code>gelf</code>, <code>json-file</code>, <code>journald</code>,
     *         <code>logentries</code>,<code>syslog</code>, <code>splunk</code>, and <code>awsfirelens</code>.
     *         </p>
     *         <p>
     *         For more information about using the <code>awslogs</code> log driver, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html">Using the awslogs
     *         log driver</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     *         </p>
     *         <p>
     *         For more information about using the <code>awsfirelens</code> log driver, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Custom log
     *         routing</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     *         </p>
     *         <note>
     *         <p>
     *         If you have a custom driver that isn't listed, you can fork the Amazon ECS container agent project that's
     *         <a href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with
     *         that driver. We encourage you to submit pull requests for changes that you would like to have included.
     *         However, we don't currently provide support for running modified copies of this software.
     *         </p>
     * @see LogDriver
     */
    public final LogDriver logDriver() {
        return LogDriver.fromValue(logDriver);
    }

    /**
     * <p>
     * The log driver to use for the container.
     * </p>
     * <p>
     * For tasks on Fargate, the supported log drivers are <code>awslogs</code>, <code>splunk</code>, and
     * <code>awsfirelens</code>.
     * </p>
     * <p>
     * For tasks hosted on Amazon EC2 instances, the supported log drivers are <code>awslogs</code>,
     * <code>fluentd</code>, <code>gelf</code>, <code>json-file</code>, <code>journald</code>, <code>logentries</code>,
     * <code>syslog</code>, <code>splunk</code>, and <code>awsfirelens</code>.
     * </p>
     * <p>
     * For more information about using the <code>awslogs</code> log driver, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html">Using the awslogs log
     * driver</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * For more information about using the <code>awsfirelens</code> log driver, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Custom log routing</a> in
     * the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * If you have a custom driver that isn't listed, you can fork the Amazon ECS container agent project that's <a
     * href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with that driver.
     * We encourage you to submit pull requests for changes that you would like to have included. However, we don't
     * currently provide support for running modified copies of this software.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #logDriver} will
     * return {@link LogDriver#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #logDriverAsString}.
     * </p>
     * 
     * @return The log driver to use for the container.</p>
     *         <p>
     *         For tasks on Fargate, the supported log drivers are <code>awslogs</code>, <code>splunk</code>, and
     *         <code>awsfirelens</code>.
     *         </p>
     *         <p>
     *         For tasks hosted on Amazon EC2 instances, the supported log drivers are <code>awslogs</code>,
     *         <code>fluentd</code>, <code>gelf</code>, <code>json-file</code>, <code>journald</code>,
     *         <code>logentries</code>,<code>syslog</code>, <code>splunk</code>, and <code>awsfirelens</code>.
     *         </p>
     *         <p>
     *         For more information about using the <code>awslogs</code> log driver, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html">Using the awslogs
     *         log driver</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     *         </p>
     *         <p>
     *         For more information about using the <code>awsfirelens</code> log driver, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Custom log
     *         routing</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     *         </p>
     *         <note>
     *         <p>
     *         If you have a custom driver that isn't listed, you can fork the Amazon ECS container agent project that's
     *         <a href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with
     *         that driver. We encourage you to submit pull requests for changes that you would like to have included.
     *         However, we don't currently provide support for running modified copies of this software.
     *         </p>
     * @see LogDriver
     */
    public final String logDriverAsString() {
        return logDriver;
    }

    /**
     * For responses, this returns true if the service returned a value for the Options property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasOptions() {
        return options != null && !(options instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The configuration options to send to the log driver. This parameter requires version 1.19 of the Docker Remote
     * API or greater on your container instance. To check the Docker Remote API version on your container instance, log
     * in to your container instance and run the following command:
     * <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOptions} method.
     * </p>
     * 
     * @return The configuration options to send to the log driver. This parameter requires version 1.19 of the Docker
     *         Remote API or greater on your container instance. To check the Docker Remote API version on your
     *         container instance, log in to your container instance and run the following command:
     *         <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
     */
    public final Map<String, String> options() {
        return options;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecretOptions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSecretOptions() {
        return secretOptions != null && !(secretOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The secrets to pass to the log configuration. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html">Specifying
     * Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecretOptions} method.
     * </p>
     * 
     * @return The secrets to pass to the log configuration. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html"
     *         >Specifying Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     */
    public final List<Secret> secretOptions() {
        return secretOptions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(logDriverAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasOptions() ? options() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSecretOptions() ? secretOptions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LogConfiguration)) {
            return false;
        }
        LogConfiguration other = (LogConfiguration) obj;
        return Objects.equals(logDriverAsString(), other.logDriverAsString()) && hasOptions() == other.hasOptions()
                && Objects.equals(options(), other.options()) && hasSecretOptions() == other.hasSecretOptions()
                && Objects.equals(secretOptions(), other.secretOptions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LogConfiguration").add("LogDriver", logDriverAsString())
                .add("Options", hasOptions() ? options() : null)
                .add("SecretOptions", hasSecretOptions() ? secretOptions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "logDriver":
            return Optional.ofNullable(clazz.cast(logDriverAsString()));
        case "options":
            return Optional.ofNullable(clazz.cast(options()));
        case "secretOptions":
            return Optional.ofNullable(clazz.cast(secretOptions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LogConfiguration, T> g) {
        return obj -> g.apply((LogConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LogConfiguration> {
        /**
         * <p>
         * The log driver to use for the container.
         * </p>
         * <p>
         * For tasks on Fargate, the supported log drivers are <code>awslogs</code>, <code>splunk</code>, and
         * <code>awsfirelens</code>.
         * </p>
         * <p>
         * For tasks hosted on Amazon EC2 instances, the supported log drivers are <code>awslogs</code>,
         * <code>fluentd</code>, <code>gelf</code>, <code>json-file</code>, <code>journald</code>,
         * <code>logentries</code>,<code>syslog</code>, <code>splunk</code>, and <code>awsfirelens</code>.
         * </p>
         * <p>
         * For more information about using the <code>awslogs</code> log driver, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html">Using the awslogs log
         * driver</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <p>
         * For more information about using the <code>awsfirelens</code> log driver, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Custom log routing</a>
         * in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <note>
         * <p>
         * If you have a custom driver that isn't listed, you can fork the Amazon ECS container agent project that's <a
         * href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with that
         * driver. We encourage you to submit pull requests for changes that you would like to have included. However,
         * we don't currently provide support for running modified copies of this software.
         * </p>
         * </note>
         * 
         * @param logDriver
         *        The log driver to use for the container.</p>
         *        <p>
         *        For tasks on Fargate, the supported log drivers are <code>awslogs</code>, <code>splunk</code>, and
         *        <code>awsfirelens</code>.
         *        </p>
         *        <p>
         *        For tasks hosted on Amazon EC2 instances, the supported log drivers are <code>awslogs</code>,
         *        <code>fluentd</code>, <code>gelf</code>, <code>json-file</code>, <code>journald</code>,
         *        <code>logentries</code>,<code>syslog</code>, <code>splunk</code>, and <code>awsfirelens</code>.
         *        </p>
         *        <p>
         *        For more information about using the <code>awslogs</code> log driver, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html">Using the
         *        awslogs log driver</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         *        </p>
         *        <p>
         *        For more information about using the <code>awsfirelens</code> log driver, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Custom log
         *        routing</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         *        </p>
         *        <note>
         *        <p>
         *        If you have a custom driver that isn't listed, you can fork the Amazon ECS container agent project
         *        that's <a href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to
         *        work with that driver. We encourage you to submit pull requests for changes that you would like to
         *        have included. However, we don't currently provide support for running modified copies of this
         *        software.
         *        </p>
         * @see LogDriver
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogDriver
         */
        Builder logDriver(String logDriver);

        /**
         * <p>
         * The log driver to use for the container.
         * </p>
         * <p>
         * For tasks on Fargate, the supported log drivers are <code>awslogs</code>, <code>splunk</code>, and
         * <code>awsfirelens</code>.
         * </p>
         * <p>
         * For tasks hosted on Amazon EC2 instances, the supported log drivers are <code>awslogs</code>,
         * <code>fluentd</code>, <code>gelf</code>, <code>json-file</code>, <code>journald</code>,
         * <code>logentries</code>,<code>syslog</code>, <code>splunk</code>, and <code>awsfirelens</code>.
         * </p>
         * <p>
         * For more information about using the <code>awslogs</code> log driver, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html">Using the awslogs log
         * driver</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <p>
         * For more information about using the <code>awsfirelens</code> log driver, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Custom log routing</a>
         * in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <note>
         * <p>
         * If you have a custom driver that isn't listed, you can fork the Amazon ECS container agent project that's <a
         * href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with that
         * driver. We encourage you to submit pull requests for changes that you would like to have included. However,
         * we don't currently provide support for running modified copies of this software.
         * </p>
         * </note>
         * 
         * @param logDriver
         *        The log driver to use for the container.</p>
         *        <p>
         *        For tasks on Fargate, the supported log drivers are <code>awslogs</code>, <code>splunk</code>, and
         *        <code>awsfirelens</code>.
         *        </p>
         *        <p>
         *        For tasks hosted on Amazon EC2 instances, the supported log drivers are <code>awslogs</code>,
         *        <code>fluentd</code>, <code>gelf</code>, <code>json-file</code>, <code>journald</code>,
         *        <code>logentries</code>,<code>syslog</code>, <code>splunk</code>, and <code>awsfirelens</code>.
         *        </p>
         *        <p>
         *        For more information about using the <code>awslogs</code> log driver, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html">Using the
         *        awslogs log driver</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         *        </p>
         *        <p>
         *        For more information about using the <code>awsfirelens</code> log driver, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Custom log
         *        routing</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         *        </p>
         *        <note>
         *        <p>
         *        If you have a custom driver that isn't listed, you can fork the Amazon ECS container agent project
         *        that's <a href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to
         *        work with that driver. We encourage you to submit pull requests for changes that you would like to
         *        have included. However, we don't currently provide support for running modified copies of this
         *        software.
         *        </p>
         * @see LogDriver
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogDriver
         */
        Builder logDriver(LogDriver logDriver);

        /**
         * <p>
         * The configuration options to send to the log driver. This parameter requires version 1.19 of the Docker
         * Remote API or greater on your container instance. To check the Docker Remote API version on your container
         * instance, log in to your container instance and run the following command:
         * <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
         * </p>
         * 
         * @param options
         *        The configuration options to send to the log driver. This parameter requires version 1.19 of the
         *        Docker Remote API or greater on your container instance. To check the Docker Remote API version on
         *        your container instance, log in to your container instance and run the following command:
         *        <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder options(Map<String, String> options);

        /**
         * <p>
         * The secrets to pass to the log configuration. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html">Specifying
         * Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param secretOptions
         *        The secrets to pass to the log configuration. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html"
         *        >Specifying Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretOptions(Collection<Secret> secretOptions);

        /**
         * <p>
         * The secrets to pass to the log configuration. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html">Specifying
         * Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param secretOptions
         *        The secrets to pass to the log configuration. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html"
         *        >Specifying Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretOptions(Secret... secretOptions);

        /**
         * <p>
         * The secrets to pass to the log configuration. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html">Specifying
         * Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<Secret>.Builder} avoiding the need
         * to create one manually via {@link List<Secret>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Secret>.Builder#build()} is called immediately and its
         * result is passed to {@link #secretOptions(List<Secret>)}.
         * 
         * @param secretOptions
         *        a consumer that will call methods on {@link List<Secret>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #secretOptions(List<Secret>)
         */
        Builder secretOptions(Consumer<Secret.Builder>... secretOptions);
    }

    static final class BuilderImpl implements Builder {
        private String logDriver;

        private Map<String, String> options = DefaultSdkAutoConstructMap.getInstance();

        private List<Secret> secretOptions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(LogConfiguration model) {
            logDriver(model.logDriver);
            options(model.options);
            secretOptions(model.secretOptions);
        }

        public final String getLogDriver() {
            return logDriver;
        }

        public final void setLogDriver(String logDriver) {
            this.logDriver = logDriver;
        }

        @Override
        public final Builder logDriver(String logDriver) {
            this.logDriver = logDriver;
            return this;
        }

        @Override
        public final Builder logDriver(LogDriver logDriver) {
            this.logDriver(logDriver == null ? null : logDriver.toString());
            return this;
        }

        public final Map<String, String> getOptions() {
            if (options instanceof SdkAutoConstructMap) {
                return null;
            }
            return options;
        }

        public final void setOptions(Map<String, String> options) {
            this.options = LogConfigurationOptionsMapCopier.copy(options);
        }

        @Override
        public final Builder options(Map<String, String> options) {
            this.options = LogConfigurationOptionsMapCopier.copy(options);
            return this;
        }

        public final List<Secret.Builder> getSecretOptions() {
            List<Secret.Builder> result = SecretListCopier.copyToBuilder(this.secretOptions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSecretOptions(Collection<Secret.BuilderImpl> secretOptions) {
            this.secretOptions = SecretListCopier.copyFromBuilder(secretOptions);
        }

        @Override
        public final Builder secretOptions(Collection<Secret> secretOptions) {
            this.secretOptions = SecretListCopier.copy(secretOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secretOptions(Secret... secretOptions) {
            secretOptions(Arrays.asList(secretOptions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secretOptions(Consumer<Secret.Builder>... secretOptions) {
            secretOptions(Stream.of(secretOptions).map(c -> Secret.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public LogConfiguration build() {
            return new LogConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
