/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The details of an Amazon ECS service deployment. This is used only when a service uses the <code>ECS</code>
 * deployment controller type.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Deployment implements SdkPojo, Serializable, ToCopyableBuilder<Deployment.Builder, Deployment> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(Deployment::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(Deployment::status)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> TASK_DEFINITION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("taskDefinition").getter(getter(Deployment::taskDefinition)).setter(setter(Builder::taskDefinition))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("taskDefinition").build()).build();

    private static final SdkField<Integer> DESIRED_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("desiredCount").getter(getter(Deployment::desiredCount)).setter(setter(Builder::desiredCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("desiredCount").build()).build();

    private static final SdkField<Integer> PENDING_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("pendingCount").getter(getter(Deployment::pendingCount)).setter(setter(Builder::pendingCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("pendingCount").build()).build();

    private static final SdkField<Integer> RUNNING_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("runningCount").getter(getter(Deployment::runningCount)).setter(setter(Builder::runningCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("runningCount").build()).build();

    private static final SdkField<Integer> FAILED_TASKS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("failedTasks").getter(getter(Deployment::failedTasks)).setter(setter(Builder::failedTasks))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("failedTasks").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt").getter(getter(Deployment::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<Instant> UPDATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("updatedAt").getter(getter(Deployment::updatedAt)).setter(setter(Builder::updatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("updatedAt").build()).build();

    private static final SdkField<List<CapacityProviderStrategyItem>> CAPACITY_PROVIDER_STRATEGY_FIELD = SdkField
            .<List<CapacityProviderStrategyItem>> builder(MarshallingType.LIST)
            .memberName("capacityProviderStrategy")
            .getter(getter(Deployment::capacityProviderStrategy))
            .setter(setter(Builder::capacityProviderStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("capacityProviderStrategy").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<CapacityProviderStrategyItem> builder(MarshallingType.SDK_POJO)
                                            .constructor(CapacityProviderStrategyItem::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> LAUNCH_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("launchType").getter(getter(Deployment::launchTypeAsString)).setter(setter(Builder::launchType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("launchType").build()).build();

    private static final SdkField<String> PLATFORM_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("platformVersion").getter(getter(Deployment::platformVersion)).setter(setter(Builder::platformVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("platformVersion").build()).build();

    private static final SdkField<String> PLATFORM_FAMILY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("platformFamily").getter(getter(Deployment::platformFamily)).setter(setter(Builder::platformFamily))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("platformFamily").build()).build();

    private static final SdkField<NetworkConfiguration> NETWORK_CONFIGURATION_FIELD = SdkField
            .<NetworkConfiguration> builder(MarshallingType.SDK_POJO).memberName("networkConfiguration")
            .getter(getter(Deployment::networkConfiguration)).setter(setter(Builder::networkConfiguration))
            .constructor(NetworkConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("networkConfiguration").build())
            .build();

    private static final SdkField<String> ROLLOUT_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("rolloutState").getter(getter(Deployment::rolloutStateAsString)).setter(setter(Builder::rolloutState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("rolloutState").build()).build();

    private static final SdkField<String> ROLLOUT_STATE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("rolloutStateReason").getter(getter(Deployment::rolloutStateReason))
            .setter(setter(Builder::rolloutStateReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("rolloutStateReason").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, STATUS_FIELD,
            TASK_DEFINITION_FIELD, DESIRED_COUNT_FIELD, PENDING_COUNT_FIELD, RUNNING_COUNT_FIELD, FAILED_TASKS_FIELD,
            CREATED_AT_FIELD, UPDATED_AT_FIELD, CAPACITY_PROVIDER_STRATEGY_FIELD, LAUNCH_TYPE_FIELD, PLATFORM_VERSION_FIELD,
            PLATFORM_FAMILY_FIELD, NETWORK_CONFIGURATION_FIELD, ROLLOUT_STATE_FIELD, ROLLOUT_STATE_REASON_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String status;

    private final String taskDefinition;

    private final Integer desiredCount;

    private final Integer pendingCount;

    private final Integer runningCount;

    private final Integer failedTasks;

    private final Instant createdAt;

    private final Instant updatedAt;

    private final List<CapacityProviderStrategyItem> capacityProviderStrategy;

    private final String launchType;

    private final String platformVersion;

    private final String platformFamily;

    private final NetworkConfiguration networkConfiguration;

    private final String rolloutState;

    private final String rolloutStateReason;

    private Deployment(BuilderImpl builder) {
        this.id = builder.id;
        this.status = builder.status;
        this.taskDefinition = builder.taskDefinition;
        this.desiredCount = builder.desiredCount;
        this.pendingCount = builder.pendingCount;
        this.runningCount = builder.runningCount;
        this.failedTasks = builder.failedTasks;
        this.createdAt = builder.createdAt;
        this.updatedAt = builder.updatedAt;
        this.capacityProviderStrategy = builder.capacityProviderStrategy;
        this.launchType = builder.launchType;
        this.platformVersion = builder.platformVersion;
        this.platformFamily = builder.platformFamily;
        this.networkConfiguration = builder.networkConfiguration;
        this.rolloutState = builder.rolloutState;
        this.rolloutStateReason = builder.rolloutStateReason;
    }

    /**
     * <p>
     * The ID of the deployment.
     * </p>
     * 
     * @return The ID of the deployment.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The status of the deployment. The following describes each state.
     * </p>
     * <dl>
     * <dt>PRIMARY</dt>
     * <dd>
     * <p>
     * The most recent deployment of a service.
     * </p>
     * </dd>
     * <dt>ACTIVE</dt>
     * <dd>
     * <p>
     * A service deployment that still has running tasks, but are in the process of being replaced with a new
     * <code>PRIMARY</code> deployment.
     * </p>
     * </dd>
     * <dt>INACTIVE</dt>
     * <dd>
     * <p>
     * A deployment that has been completely replaced.
     * </p>
     * </dd>
     * </dl>
     * 
     * @return The status of the deployment. The following describes each state.</p>
     *         <dl>
     *         <dt>PRIMARY</dt>
     *         <dd>
     *         <p>
     *         The most recent deployment of a service.
     *         </p>
     *         </dd>
     *         <dt>ACTIVE</dt>
     *         <dd>
     *         <p>
     *         A service deployment that still has running tasks, but are in the process of being replaced with a new
     *         <code>PRIMARY</code> deployment.
     *         </p>
     *         </dd>
     *         <dt>INACTIVE</dt>
     *         <dd>
     *         <p>
     *         A deployment that has been completely replaced.
     *         </p>
     *         </dd>
     */
    public final String status() {
        return status;
    }

    /**
     * <p>
     * The most recent task definition that was specified for the tasks in the service to use.
     * </p>
     * 
     * @return The most recent task definition that was specified for the tasks in the service to use.
     */
    public final String taskDefinition() {
        return taskDefinition;
    }

    /**
     * <p>
     * The most recent desired count of tasks that was specified for the service to deploy or maintain.
     * </p>
     * 
     * @return The most recent desired count of tasks that was specified for the service to deploy or maintain.
     */
    public final Integer desiredCount() {
        return desiredCount;
    }

    /**
     * <p>
     * The number of tasks in the deployment that are in the <code>PENDING</code> status.
     * </p>
     * 
     * @return The number of tasks in the deployment that are in the <code>PENDING</code> status.
     */
    public final Integer pendingCount() {
        return pendingCount;
    }

    /**
     * <p>
     * The number of tasks in the deployment that are in the <code>RUNNING</code> status.
     * </p>
     * 
     * @return The number of tasks in the deployment that are in the <code>RUNNING</code> status.
     */
    public final Integer runningCount() {
        return runningCount;
    }

    /**
     * <p>
     * The number of consecutively failed tasks in the deployment. A task is considered a failure if the service
     * scheduler can't launch the task, the task doesn't transition to a <code>RUNNING</code> state, or if it fails any
     * of its defined health checks and is stopped.
     * </p>
     * <note>
     * <p>
     * Once a service deployment has one or more successfully running tasks, the failed task count resets to zero and
     * stops being evaluated.
     * </p>
     * </note>
     * 
     * @return The number of consecutively failed tasks in the deployment. A task is considered a failure if the service
     *         scheduler can't launch the task, the task doesn't transition to a <code>RUNNING</code> state, or if it
     *         fails any of its defined health checks and is stopped.</p> <note>
     *         <p>
     *         Once a service deployment has one or more successfully running tasks, the failed task count resets to
     *         zero and stops being evaluated.
     *         </p>
     */
    public final Integer failedTasks() {
        return failedTasks;
    }

    /**
     * <p>
     * The Unix timestamp for the time when the service deployment was created.
     * </p>
     * 
     * @return The Unix timestamp for the time when the service deployment was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The Unix timestamp for the time when the service deployment was last updated.
     * </p>
     * 
     * @return The Unix timestamp for the time when the service deployment was last updated.
     */
    public final Instant updatedAt() {
        return updatedAt;
    }

    /**
     * For responses, this returns true if the service returned a value for the CapacityProviderStrategy property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasCapacityProviderStrategy() {
        return capacityProviderStrategy != null && !(capacityProviderStrategy instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The capacity provider strategy that the deployment is using.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCapacityProviderStrategy} method.
     * </p>
     * 
     * @return The capacity provider strategy that the deployment is using.
     */
    public final List<CapacityProviderStrategyItem> capacityProviderStrategy() {
        return capacityProviderStrategy;
    }

    /**
     * <p>
     * The launch type the tasks in the service are using. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS Launch Types</a>
     * in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #launchType} will
     * return {@link LaunchType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #launchTypeAsString}.
     * </p>
     * 
     * @return The launch type the tasks in the service are using. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS Launch
     *         Types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * @see LaunchType
     */
    public final LaunchType launchType() {
        return LaunchType.fromValue(launchType);
    }

    /**
     * <p>
     * The launch type the tasks in the service are using. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS Launch Types</a>
     * in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #launchType} will
     * return {@link LaunchType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #launchTypeAsString}.
     * </p>
     * 
     * @return The launch type the tasks in the service are using. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS Launch
     *         Types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * @see LaunchType
     */
    public final String launchTypeAsString() {
        return launchType;
    }

    /**
     * <p>
     * The platform version that your tasks in the service run on. A platform version is only specified for tasks using
     * the Fargate launch type. If one isn't specified, the <code>LATEST</code> platform version is used. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html">Fargate Platform
     * Versions</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * 
     * @return The platform version that your tasks in the service run on. A platform version is only specified for
     *         tasks using the Fargate launch type. If one isn't specified, the <code>LATEST</code> platform version is
     *         used. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html">Fargate
     *         Platform Versions</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     */
    public final String platformVersion() {
        return platformVersion;
    }

    /**
     * <p>
     * The operating system that your tasks in the service, or tasks are running on. A platform family is specified only
     * for tasks using the Fargate launch type.
     * </p>
     * <p>
     * All tasks that run as part of this service must use the same <code>platformFamily</code> value as the service,
     * for example, <code> LINUX.</code>.
     * </p>
     * 
     * @return The operating system that your tasks in the service, or tasks are running on. A platform family is
     *         specified only for tasks using the Fargate launch type. </p>
     *         <p>
     *         All tasks that run as part of this service must use the same <code>platformFamily</code> value as the
     *         service, for example, <code> LINUX.</code>.
     */
    public final String platformFamily() {
        return platformFamily;
    }

    /**
     * <p>
     * The VPC subnet and security group configuration for tasks that receive their own elastic network interface by
     * using the <code>awsvpc</code> networking mode.
     * </p>
     * 
     * @return The VPC subnet and security group configuration for tasks that receive their own elastic network
     *         interface by using the <code>awsvpc</code> networking mode.
     */
    public final NetworkConfiguration networkConfiguration() {
        return networkConfiguration;
    }

    /**
     * <note>
     * <p>
     * The <code>rolloutState</code> of a service is only returned for services that use the rolling update (
     * <code>ECS</code>) deployment type that aren't behind a Classic Load Balancer.
     * </p>
     * </note>
     * <p>
     * The rollout state of the deployment. When a service deployment is started, it begins in an
     * <code>IN_PROGRESS</code> state. When the service reaches a steady state, the deployment transitions to a
     * <code>COMPLETED</code> state. If the service fails to reach a steady state and circuit breaker is enabled, the
     * deployment transitions to a <code>FAILED</code> state. A deployment in <code>FAILED</code> state doesn't launch
     * any new tasks. For more information, see <a>DeploymentCircuitBreaker</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #rolloutState} will
     * return {@link DeploymentRolloutState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #rolloutStateAsString}.
     * </p>
     * 
     * @return <p>
     *         The <code>rolloutState</code> of a service is only returned for services that use the rolling update (
     *         <code>ECS</code>) deployment type that aren't behind a Classic Load Balancer.
     *         </p>
     *         </note>
     *         <p>
     *         The rollout state of the deployment. When a service deployment is started, it begins in an
     *         <code>IN_PROGRESS</code> state. When the service reaches a steady state, the deployment transitions to a
     *         <code>COMPLETED</code> state. If the service fails to reach a steady state and circuit breaker is
     *         enabled, the deployment transitions to a <code>FAILED</code> state. A deployment in <code>FAILED</code>
     *         state doesn't launch any new tasks. For more information, see <a>DeploymentCircuitBreaker</a>.
     * @see DeploymentRolloutState
     */
    public final DeploymentRolloutState rolloutState() {
        return DeploymentRolloutState.fromValue(rolloutState);
    }

    /**
     * <note>
     * <p>
     * The <code>rolloutState</code> of a service is only returned for services that use the rolling update (
     * <code>ECS</code>) deployment type that aren't behind a Classic Load Balancer.
     * </p>
     * </note>
     * <p>
     * The rollout state of the deployment. When a service deployment is started, it begins in an
     * <code>IN_PROGRESS</code> state. When the service reaches a steady state, the deployment transitions to a
     * <code>COMPLETED</code> state. If the service fails to reach a steady state and circuit breaker is enabled, the
     * deployment transitions to a <code>FAILED</code> state. A deployment in <code>FAILED</code> state doesn't launch
     * any new tasks. For more information, see <a>DeploymentCircuitBreaker</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #rolloutState} will
     * return {@link DeploymentRolloutState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #rolloutStateAsString}.
     * </p>
     * 
     * @return <p>
     *         The <code>rolloutState</code> of a service is only returned for services that use the rolling update (
     *         <code>ECS</code>) deployment type that aren't behind a Classic Load Balancer.
     *         </p>
     *         </note>
     *         <p>
     *         The rollout state of the deployment. When a service deployment is started, it begins in an
     *         <code>IN_PROGRESS</code> state. When the service reaches a steady state, the deployment transitions to a
     *         <code>COMPLETED</code> state. If the service fails to reach a steady state and circuit breaker is
     *         enabled, the deployment transitions to a <code>FAILED</code> state. A deployment in <code>FAILED</code>
     *         state doesn't launch any new tasks. For more information, see <a>DeploymentCircuitBreaker</a>.
     * @see DeploymentRolloutState
     */
    public final String rolloutStateAsString() {
        return rolloutState;
    }

    /**
     * <p>
     * A description of the rollout state of a deployment.
     * </p>
     * 
     * @return A description of the rollout state of a deployment.
     */
    public final String rolloutStateReason() {
        return rolloutStateReason;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(taskDefinition());
        hashCode = 31 * hashCode + Objects.hashCode(desiredCount());
        hashCode = 31 * hashCode + Objects.hashCode(pendingCount());
        hashCode = 31 * hashCode + Objects.hashCode(runningCount());
        hashCode = 31 * hashCode + Objects.hashCode(failedTasks());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(updatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(hasCapacityProviderStrategy() ? capacityProviderStrategy() : null);
        hashCode = 31 * hashCode + Objects.hashCode(launchTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(platformVersion());
        hashCode = 31 * hashCode + Objects.hashCode(platformFamily());
        hashCode = 31 * hashCode + Objects.hashCode(networkConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(rolloutStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(rolloutStateReason());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Deployment)) {
            return false;
        }
        Deployment other = (Deployment) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(status(), other.status())
                && Objects.equals(taskDefinition(), other.taskDefinition())
                && Objects.equals(desiredCount(), other.desiredCount()) && Objects.equals(pendingCount(), other.pendingCount())
                && Objects.equals(runningCount(), other.runningCount()) && Objects.equals(failedTasks(), other.failedTasks())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(updatedAt(), other.updatedAt())
                && hasCapacityProviderStrategy() == other.hasCapacityProviderStrategy()
                && Objects.equals(capacityProviderStrategy(), other.capacityProviderStrategy())
                && Objects.equals(launchTypeAsString(), other.launchTypeAsString())
                && Objects.equals(platformVersion(), other.platformVersion())
                && Objects.equals(platformFamily(), other.platformFamily())
                && Objects.equals(networkConfiguration(), other.networkConfiguration())
                && Objects.equals(rolloutStateAsString(), other.rolloutStateAsString())
                && Objects.equals(rolloutStateReason(), other.rolloutStateReason());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Deployment").add("Id", id()).add("Status", status()).add("TaskDefinition", taskDefinition())
                .add("DesiredCount", desiredCount()).add("PendingCount", pendingCount()).add("RunningCount", runningCount())
                .add("FailedTasks", failedTasks()).add("CreatedAt", createdAt()).add("UpdatedAt", updatedAt())
                .add("CapacityProviderStrategy", hasCapacityProviderStrategy() ? capacityProviderStrategy() : null)
                .add("LaunchType", launchTypeAsString()).add("PlatformVersion", platformVersion())
                .add("PlatformFamily", platformFamily()).add("NetworkConfiguration", networkConfiguration())
                .add("RolloutState", rolloutStateAsString()).add("RolloutStateReason", rolloutStateReason()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "status":
            return Optional.ofNullable(clazz.cast(status()));
        case "taskDefinition":
            return Optional.ofNullable(clazz.cast(taskDefinition()));
        case "desiredCount":
            return Optional.ofNullable(clazz.cast(desiredCount()));
        case "pendingCount":
            return Optional.ofNullable(clazz.cast(pendingCount()));
        case "runningCount":
            return Optional.ofNullable(clazz.cast(runningCount()));
        case "failedTasks":
            return Optional.ofNullable(clazz.cast(failedTasks()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "updatedAt":
            return Optional.ofNullable(clazz.cast(updatedAt()));
        case "capacityProviderStrategy":
            return Optional.ofNullable(clazz.cast(capacityProviderStrategy()));
        case "launchType":
            return Optional.ofNullable(clazz.cast(launchTypeAsString()));
        case "platformVersion":
            return Optional.ofNullable(clazz.cast(platformVersion()));
        case "platformFamily":
            return Optional.ofNullable(clazz.cast(platformFamily()));
        case "networkConfiguration":
            return Optional.ofNullable(clazz.cast(networkConfiguration()));
        case "rolloutState":
            return Optional.ofNullable(clazz.cast(rolloutStateAsString()));
        case "rolloutStateReason":
            return Optional.ofNullable(clazz.cast(rolloutStateReason()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Deployment, T> g) {
        return obj -> g.apply((Deployment) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Deployment> {
        /**
         * <p>
         * The ID of the deployment.
         * </p>
         * 
         * @param id
         *        The ID of the deployment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The status of the deployment. The following describes each state.
         * </p>
         * <dl>
         * <dt>PRIMARY</dt>
         * <dd>
         * <p>
         * The most recent deployment of a service.
         * </p>
         * </dd>
         * <dt>ACTIVE</dt>
         * <dd>
         * <p>
         * A service deployment that still has running tasks, but are in the process of being replaced with a new
         * <code>PRIMARY</code> deployment.
         * </p>
         * </dd>
         * <dt>INACTIVE</dt>
         * <dd>
         * <p>
         * A deployment that has been completely replaced.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param status
         *        The status of the deployment. The following describes each state.</p>
         *        <dl>
         *        <dt>PRIMARY</dt>
         *        <dd>
         *        <p>
         *        The most recent deployment of a service.
         *        </p>
         *        </dd>
         *        <dt>ACTIVE</dt>
         *        <dd>
         *        <p>
         *        A service deployment that still has running tasks, but are in the process of being replaced with a new
         *        <code>PRIMARY</code> deployment.
         *        </p>
         *        </dd>
         *        <dt>INACTIVE</dt>
         *        <dd>
         *        <p>
         *        A deployment that has been completely replaced.
         *        </p>
         *        </dd>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);

        /**
         * <p>
         * The most recent task definition that was specified for the tasks in the service to use.
         * </p>
         * 
         * @param taskDefinition
         *        The most recent task definition that was specified for the tasks in the service to use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskDefinition(String taskDefinition);

        /**
         * <p>
         * The most recent desired count of tasks that was specified for the service to deploy or maintain.
         * </p>
         * 
         * @param desiredCount
         *        The most recent desired count of tasks that was specified for the service to deploy or maintain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder desiredCount(Integer desiredCount);

        /**
         * <p>
         * The number of tasks in the deployment that are in the <code>PENDING</code> status.
         * </p>
         * 
         * @param pendingCount
         *        The number of tasks in the deployment that are in the <code>PENDING</code> status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pendingCount(Integer pendingCount);

        /**
         * <p>
         * The number of tasks in the deployment that are in the <code>RUNNING</code> status.
         * </p>
         * 
         * @param runningCount
         *        The number of tasks in the deployment that are in the <code>RUNNING</code> status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder runningCount(Integer runningCount);

        /**
         * <p>
         * The number of consecutively failed tasks in the deployment. A task is considered a failure if the service
         * scheduler can't launch the task, the task doesn't transition to a <code>RUNNING</code> state, or if it fails
         * any of its defined health checks and is stopped.
         * </p>
         * <note>
         * <p>
         * Once a service deployment has one or more successfully running tasks, the failed task count resets to zero
         * and stops being evaluated.
         * </p>
         * </note>
         * 
         * @param failedTasks
         *        The number of consecutively failed tasks in the deployment. A task is considered a failure if the
         *        service scheduler can't launch the task, the task doesn't transition to a <code>RUNNING</code> state,
         *        or if it fails any of its defined health checks and is stopped.</p> <note>
         *        <p>
         *        Once a service deployment has one or more successfully running tasks, the failed task count resets to
         *        zero and stops being evaluated.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failedTasks(Integer failedTasks);

        /**
         * <p>
         * The Unix timestamp for the time when the service deployment was created.
         * </p>
         * 
         * @param createdAt
         *        The Unix timestamp for the time when the service deployment was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The Unix timestamp for the time when the service deployment was last updated.
         * </p>
         * 
         * @param updatedAt
         *        The Unix timestamp for the time when the service deployment was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedAt(Instant updatedAt);

        /**
         * <p>
         * The capacity provider strategy that the deployment is using.
         * </p>
         * 
         * @param capacityProviderStrategy
         *        The capacity provider strategy that the deployment is using.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityProviderStrategy(Collection<CapacityProviderStrategyItem> capacityProviderStrategy);

        /**
         * <p>
         * The capacity provider strategy that the deployment is using.
         * </p>
         * 
         * @param capacityProviderStrategy
         *        The capacity provider strategy that the deployment is using.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityProviderStrategy(CapacityProviderStrategyItem... capacityProviderStrategy);

        /**
         * <p>
         * The capacity provider strategy that the deployment is using.
         * </p>
         * This is a convenience method that creates an instance of the {@link List
         * <CapacityProviderStrategyItem>.Builder} avoiding the need to create one manually via {@link List
         * <CapacityProviderStrategyItem>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<CapacityProviderStrategyItem>.Builder#build()} is called
         * immediately and its result is passed to {@link #capacityProviderStrategy(List<CapacityProviderStrategyItem>)}
         * .
         * 
         * @param capacityProviderStrategy
         *        a consumer that will call methods on {@link List<CapacityProviderStrategyItem>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #capacityProviderStrategy(List<CapacityProviderStrategyItem>)
         */
        Builder capacityProviderStrategy(Consumer<CapacityProviderStrategyItem.Builder>... capacityProviderStrategy);

        /**
         * <p>
         * The launch type the tasks in the service are using. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS Launch
         * Types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param launchType
         *        The launch type the tasks in the service are using. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS Launch
         *        Types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @see LaunchType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LaunchType
         */
        Builder launchType(String launchType);

        /**
         * <p>
         * The launch type the tasks in the service are using. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS Launch
         * Types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param launchType
         *        The launch type the tasks in the service are using. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS Launch
         *        Types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @see LaunchType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LaunchType
         */
        Builder launchType(LaunchType launchType);

        /**
         * <p>
         * The platform version that your tasks in the service run on. A platform version is only specified for tasks
         * using the Fargate launch type. If one isn't specified, the <code>LATEST</code> platform version is used. For
         * more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html">Fargate Platform
         * Versions</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param platformVersion
         *        The platform version that your tasks in the service run on. A platform version is only specified for
         *        tasks using the Fargate launch type. If one isn't specified, the <code>LATEST</code> platform version
         *        is used. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html">Fargate
         *        Platform Versions</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformVersion(String platformVersion);

        /**
         * <p>
         * The operating system that your tasks in the service, or tasks are running on. A platform family is specified
         * only for tasks using the Fargate launch type.
         * </p>
         * <p>
         * All tasks that run as part of this service must use the same <code>platformFamily</code> value as the
         * service, for example, <code> LINUX.</code>.
         * </p>
         * 
         * @param platformFamily
         *        The operating system that your tasks in the service, or tasks are running on. A platform family is
         *        specified only for tasks using the Fargate launch type. </p>
         *        <p>
         *        All tasks that run as part of this service must use the same <code>platformFamily</code> value as the
         *        service, for example, <code> LINUX.</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformFamily(String platformFamily);

        /**
         * <p>
         * The VPC subnet and security group configuration for tasks that receive their own elastic network interface by
         * using the <code>awsvpc</code> networking mode.
         * </p>
         * 
         * @param networkConfiguration
         *        The VPC subnet and security group configuration for tasks that receive their own elastic network
         *        interface by using the <code>awsvpc</code> networking mode.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkConfiguration(NetworkConfiguration networkConfiguration);

        /**
         * <p>
         * The VPC subnet and security group configuration for tasks that receive their own elastic network interface by
         * using the <code>awsvpc</code> networking mode.
         * </p>
         * This is a convenience method that creates an instance of the {@link NetworkConfiguration.Builder} avoiding
         * the need to create one manually via {@link NetworkConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link NetworkConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #networkConfiguration(NetworkConfiguration)}.
         * 
         * @param networkConfiguration
         *        a consumer that will call methods on {@link NetworkConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkConfiguration(NetworkConfiguration)
         */
        default Builder networkConfiguration(Consumer<NetworkConfiguration.Builder> networkConfiguration) {
            return networkConfiguration(NetworkConfiguration.builder().applyMutation(networkConfiguration).build());
        }

        /**
         * <note>
         * <p>
         * The <code>rolloutState</code> of a service is only returned for services that use the rolling update (
         * <code>ECS</code>) deployment type that aren't behind a Classic Load Balancer.
         * </p>
         * </note>
         * <p>
         * The rollout state of the deployment. When a service deployment is started, it begins in an
         * <code>IN_PROGRESS</code> state. When the service reaches a steady state, the deployment transitions to a
         * <code>COMPLETED</code> state. If the service fails to reach a steady state and circuit breaker is enabled,
         * the deployment transitions to a <code>FAILED</code> state. A deployment in <code>FAILED</code> state doesn't
         * launch any new tasks. For more information, see <a>DeploymentCircuitBreaker</a>.
         * </p>
         * 
         * @param rolloutState
         *        <p>
         *        The <code>rolloutState</code> of a service is only returned for services that use the rolling update (
         *        <code>ECS</code>) deployment type that aren't behind a Classic Load Balancer.
         *        </p>
         *        </note>
         *        <p>
         *        The rollout state of the deployment. When a service deployment is started, it begins in an
         *        <code>IN_PROGRESS</code> state. When the service reaches a steady state, the deployment transitions to
         *        a <code>COMPLETED</code> state. If the service fails to reach a steady state and circuit breaker is
         *        enabled, the deployment transitions to a <code>FAILED</code> state. A deployment in
         *        <code>FAILED</code> state doesn't launch any new tasks. For more information, see
         *        <a>DeploymentCircuitBreaker</a>.
         * @see DeploymentRolloutState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeploymentRolloutState
         */
        Builder rolloutState(String rolloutState);

        /**
         * <note>
         * <p>
         * The <code>rolloutState</code> of a service is only returned for services that use the rolling update (
         * <code>ECS</code>) deployment type that aren't behind a Classic Load Balancer.
         * </p>
         * </note>
         * <p>
         * The rollout state of the deployment. When a service deployment is started, it begins in an
         * <code>IN_PROGRESS</code> state. When the service reaches a steady state, the deployment transitions to a
         * <code>COMPLETED</code> state. If the service fails to reach a steady state and circuit breaker is enabled,
         * the deployment transitions to a <code>FAILED</code> state. A deployment in <code>FAILED</code> state doesn't
         * launch any new tasks. For more information, see <a>DeploymentCircuitBreaker</a>.
         * </p>
         * 
         * @param rolloutState
         *        <p>
         *        The <code>rolloutState</code> of a service is only returned for services that use the rolling update (
         *        <code>ECS</code>) deployment type that aren't behind a Classic Load Balancer.
         *        </p>
         *        </note>
         *        <p>
         *        The rollout state of the deployment. When a service deployment is started, it begins in an
         *        <code>IN_PROGRESS</code> state. When the service reaches a steady state, the deployment transitions to
         *        a <code>COMPLETED</code> state. If the service fails to reach a steady state and circuit breaker is
         *        enabled, the deployment transitions to a <code>FAILED</code> state. A deployment in
         *        <code>FAILED</code> state doesn't launch any new tasks. For more information, see
         *        <a>DeploymentCircuitBreaker</a>.
         * @see DeploymentRolloutState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeploymentRolloutState
         */
        Builder rolloutState(DeploymentRolloutState rolloutState);

        /**
         * <p>
         * A description of the rollout state of a deployment.
         * </p>
         * 
         * @param rolloutStateReason
         *        A description of the rollout state of a deployment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rolloutStateReason(String rolloutStateReason);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String status;

        private String taskDefinition;

        private Integer desiredCount;

        private Integer pendingCount;

        private Integer runningCount;

        private Integer failedTasks;

        private Instant createdAt;

        private Instant updatedAt;

        private List<CapacityProviderStrategyItem> capacityProviderStrategy = DefaultSdkAutoConstructList.getInstance();

        private String launchType;

        private String platformVersion;

        private String platformFamily;

        private NetworkConfiguration networkConfiguration;

        private String rolloutState;

        private String rolloutStateReason;

        private BuilderImpl() {
        }

        private BuilderImpl(Deployment model) {
            id(model.id);
            status(model.status);
            taskDefinition(model.taskDefinition);
            desiredCount(model.desiredCount);
            pendingCount(model.pendingCount);
            runningCount(model.runningCount);
            failedTasks(model.failedTasks);
            createdAt(model.createdAt);
            updatedAt(model.updatedAt);
            capacityProviderStrategy(model.capacityProviderStrategy);
            launchType(model.launchType);
            platformVersion(model.platformVersion);
            platformFamily(model.platformFamily);
            networkConfiguration(model.networkConfiguration);
            rolloutState(model.rolloutState);
            rolloutStateReason(model.rolloutStateReason);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final String getTaskDefinition() {
            return taskDefinition;
        }

        public final void setTaskDefinition(String taskDefinition) {
            this.taskDefinition = taskDefinition;
        }

        @Override
        public final Builder taskDefinition(String taskDefinition) {
            this.taskDefinition = taskDefinition;
            return this;
        }

        public final Integer getDesiredCount() {
            return desiredCount;
        }

        public final void setDesiredCount(Integer desiredCount) {
            this.desiredCount = desiredCount;
        }

        @Override
        public final Builder desiredCount(Integer desiredCount) {
            this.desiredCount = desiredCount;
            return this;
        }

        public final Integer getPendingCount() {
            return pendingCount;
        }

        public final void setPendingCount(Integer pendingCount) {
            this.pendingCount = pendingCount;
        }

        @Override
        public final Builder pendingCount(Integer pendingCount) {
            this.pendingCount = pendingCount;
            return this;
        }

        public final Integer getRunningCount() {
            return runningCount;
        }

        public final void setRunningCount(Integer runningCount) {
            this.runningCount = runningCount;
        }

        @Override
        public final Builder runningCount(Integer runningCount) {
            this.runningCount = runningCount;
            return this;
        }

        public final Integer getFailedTasks() {
            return failedTasks;
        }

        public final void setFailedTasks(Integer failedTasks) {
            this.failedTasks = failedTasks;
        }

        @Override
        public final Builder failedTasks(Integer failedTasks) {
            this.failedTasks = failedTasks;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getUpdatedAt() {
            return updatedAt;
        }

        public final void setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
        }

        @Override
        public final Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        public final List<CapacityProviderStrategyItem.Builder> getCapacityProviderStrategy() {
            List<CapacityProviderStrategyItem.Builder> result = CapacityProviderStrategyCopier
                    .copyToBuilder(this.capacityProviderStrategy);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setCapacityProviderStrategy(
                Collection<CapacityProviderStrategyItem.BuilderImpl> capacityProviderStrategy) {
            this.capacityProviderStrategy = CapacityProviderStrategyCopier.copyFromBuilder(capacityProviderStrategy);
        }

        @Override
        public final Builder capacityProviderStrategy(Collection<CapacityProviderStrategyItem> capacityProviderStrategy) {
            this.capacityProviderStrategy = CapacityProviderStrategyCopier.copy(capacityProviderStrategy);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder capacityProviderStrategy(CapacityProviderStrategyItem... capacityProviderStrategy) {
            capacityProviderStrategy(Arrays.asList(capacityProviderStrategy));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder capacityProviderStrategy(Consumer<CapacityProviderStrategyItem.Builder>... capacityProviderStrategy) {
            capacityProviderStrategy(Stream.of(capacityProviderStrategy)
                    .map(c -> CapacityProviderStrategyItem.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getLaunchType() {
            return launchType;
        }

        public final void setLaunchType(String launchType) {
            this.launchType = launchType;
        }

        @Override
        public final Builder launchType(String launchType) {
            this.launchType = launchType;
            return this;
        }

        @Override
        public final Builder launchType(LaunchType launchType) {
            this.launchType(launchType == null ? null : launchType.toString());
            return this;
        }

        public final String getPlatformVersion() {
            return platformVersion;
        }

        public final void setPlatformVersion(String platformVersion) {
            this.platformVersion = platformVersion;
        }

        @Override
        public final Builder platformVersion(String platformVersion) {
            this.platformVersion = platformVersion;
            return this;
        }

        public final String getPlatformFamily() {
            return platformFamily;
        }

        public final void setPlatformFamily(String platformFamily) {
            this.platformFamily = platformFamily;
        }

        @Override
        public final Builder platformFamily(String platformFamily) {
            this.platformFamily = platformFamily;
            return this;
        }

        public final NetworkConfiguration.Builder getNetworkConfiguration() {
            return networkConfiguration != null ? networkConfiguration.toBuilder() : null;
        }

        public final void setNetworkConfiguration(NetworkConfiguration.BuilderImpl networkConfiguration) {
            this.networkConfiguration = networkConfiguration != null ? networkConfiguration.build() : null;
        }

        @Override
        public final Builder networkConfiguration(NetworkConfiguration networkConfiguration) {
            this.networkConfiguration = networkConfiguration;
            return this;
        }

        public final String getRolloutState() {
            return rolloutState;
        }

        public final void setRolloutState(String rolloutState) {
            this.rolloutState = rolloutState;
        }

        @Override
        public final Builder rolloutState(String rolloutState) {
            this.rolloutState = rolloutState;
            return this;
        }

        @Override
        public final Builder rolloutState(DeploymentRolloutState rolloutState) {
            this.rolloutState(rolloutState == null ? null : rolloutState.toString());
            return this;
        }

        public final String getRolloutStateReason() {
            return rolloutStateReason;
        }

        public final void setRolloutStateReason(String rolloutStateReason) {
            this.rolloutStateReason = rolloutStateReason;
        }

        @Override
        public final Builder rolloutStateReason(String rolloutStateReason) {
            this.rolloutStateReason = rolloutStateReason;
            return this;
        }

        @Override
        public Deployment build() {
            return new Deployment(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
