/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A Docker container that's part of a task.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Container implements SdkPojo, Serializable, ToCopyableBuilder<Container.Builder, Container> {
    private static final SdkField<String> CONTAINER_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("containerArn").getter(getter(Container::containerArn)).setter(setter(Builder::containerArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerArn").build()).build();

    private static final SdkField<String> TASK_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("taskArn").getter(getter(Container::taskArn)).setter(setter(Builder::taskArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("taskArn").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Container::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> IMAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("image")
            .getter(getter(Container::image)).setter(setter(Builder::image))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("image").build()).build();

    private static final SdkField<String> IMAGE_DIGEST_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("imageDigest").getter(getter(Container::imageDigest)).setter(setter(Builder::imageDigest))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("imageDigest").build()).build();

    private static final SdkField<String> RUNTIME_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("runtimeId").getter(getter(Container::runtimeId)).setter(setter(Builder::runtimeId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("runtimeId").build()).build();

    private static final SdkField<String> LAST_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("lastStatus").getter(getter(Container::lastStatus)).setter(setter(Builder::lastStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastStatus").build()).build();

    private static final SdkField<Integer> EXIT_CODE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("exitCode").getter(getter(Container::exitCode)).setter(setter(Builder::exitCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("exitCode").build()).build();

    private static final SdkField<String> REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("reason")
            .getter(getter(Container::reason)).setter(setter(Builder::reason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("reason").build()).build();

    private static final SdkField<List<NetworkBinding>> NETWORK_BINDINGS_FIELD = SdkField
            .<List<NetworkBinding>> builder(MarshallingType.LIST)
            .memberName("networkBindings")
            .getter(getter(Container::networkBindings))
            .setter(setter(Builder::networkBindings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("networkBindings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<NetworkBinding> builder(MarshallingType.SDK_POJO)
                                            .constructor(NetworkBinding::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<NetworkInterface>> NETWORK_INTERFACES_FIELD = SdkField
            .<List<NetworkInterface>> builder(MarshallingType.LIST)
            .memberName("networkInterfaces")
            .getter(getter(Container::networkInterfaces))
            .setter(setter(Builder::networkInterfaces))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("networkInterfaces").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<NetworkInterface> builder(MarshallingType.SDK_POJO)
                                            .constructor(NetworkInterface::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> HEALTH_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("healthStatus").getter(getter(Container::healthStatusAsString)).setter(setter(Builder::healthStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("healthStatus").build()).build();

    private static final SdkField<List<ManagedAgent>> MANAGED_AGENTS_FIELD = SdkField
            .<List<ManagedAgent>> builder(MarshallingType.LIST)
            .memberName("managedAgents")
            .getter(getter(Container::managedAgents))
            .setter(setter(Builder::managedAgents))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("managedAgents").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ManagedAgent> builder(MarshallingType.SDK_POJO)
                                            .constructor(ManagedAgent::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CPU_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("cpu")
            .getter(getter(Container::cpu)).setter(setter(Builder::cpu))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cpu").build()).build();

    private static final SdkField<String> MEMORY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("memory")
            .getter(getter(Container::memory)).setter(setter(Builder::memory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("memory").build()).build();

    private static final SdkField<String> MEMORY_RESERVATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("memoryReservation").getter(getter(Container::memoryReservation))
            .setter(setter(Builder::memoryReservation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("memoryReservation").build()).build();

    private static final SdkField<List<String>> GPU_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("gpuIds")
            .getter(getter(Container::gpuIds))
            .setter(setter(Builder::gpuIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("gpuIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONTAINER_ARN_FIELD,
            TASK_ARN_FIELD, NAME_FIELD, IMAGE_FIELD, IMAGE_DIGEST_FIELD, RUNTIME_ID_FIELD, LAST_STATUS_FIELD, EXIT_CODE_FIELD,
            REASON_FIELD, NETWORK_BINDINGS_FIELD, NETWORK_INTERFACES_FIELD, HEALTH_STATUS_FIELD, MANAGED_AGENTS_FIELD, CPU_FIELD,
            MEMORY_FIELD, MEMORY_RESERVATION_FIELD, GPU_IDS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String containerArn;

    private final String taskArn;

    private final String name;

    private final String image;

    private final String imageDigest;

    private final String runtimeId;

    private final String lastStatus;

    private final Integer exitCode;

    private final String reason;

    private final List<NetworkBinding> networkBindings;

    private final List<NetworkInterface> networkInterfaces;

    private final String healthStatus;

    private final List<ManagedAgent> managedAgents;

    private final String cpu;

    private final String memory;

    private final String memoryReservation;

    private final List<String> gpuIds;

    private Container(BuilderImpl builder) {
        this.containerArn = builder.containerArn;
        this.taskArn = builder.taskArn;
        this.name = builder.name;
        this.image = builder.image;
        this.imageDigest = builder.imageDigest;
        this.runtimeId = builder.runtimeId;
        this.lastStatus = builder.lastStatus;
        this.exitCode = builder.exitCode;
        this.reason = builder.reason;
        this.networkBindings = builder.networkBindings;
        this.networkInterfaces = builder.networkInterfaces;
        this.healthStatus = builder.healthStatus;
        this.managedAgents = builder.managedAgents;
        this.cpu = builder.cpu;
        this.memory = builder.memory;
        this.memoryReservation = builder.memoryReservation;
        this.gpuIds = builder.gpuIds;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the container.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the container.
     */
    public final String containerArn() {
        return containerArn;
    }

    /**
     * <p>
     * The ARN of the task.
     * </p>
     * 
     * @return The ARN of the task.
     */
    public final String taskArn() {
        return taskArn;
    }

    /**
     * <p>
     * The name of the container.
     * </p>
     * 
     * @return The name of the container.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The image used for the container.
     * </p>
     * 
     * @return The image used for the container.
     */
    public final String image() {
        return image;
    }

    /**
     * <p>
     * The container image manifest digest.
     * </p>
     * <note>
     * <p>
     * The <code>imageDigest</code> is only returned if the container is using an image hosted in Amazon ECR, otherwise
     * it is omitted.
     * </p>
     * </note>
     * 
     * @return The container image manifest digest.</p> <note>
     *         <p>
     *         The <code>imageDigest</code> is only returned if the container is using an image hosted in Amazon ECR,
     *         otherwise it is omitted.
     *         </p>
     */
    public final String imageDigest() {
        return imageDigest;
    }

    /**
     * <p>
     * The ID of the Docker container.
     * </p>
     * 
     * @return The ID of the Docker container.
     */
    public final String runtimeId() {
        return runtimeId;
    }

    /**
     * <p>
     * The last known status of the container.
     * </p>
     * 
     * @return The last known status of the container.
     */
    public final String lastStatus() {
        return lastStatus;
    }

    /**
     * <p>
     * The exit code returned from the container.
     * </p>
     * 
     * @return The exit code returned from the container.
     */
    public final Integer exitCode() {
        return exitCode;
    }

    /**
     * <p>
     * A short (255 max characters) human-readable string to provide additional details about a running or stopped
     * container.
     * </p>
     * 
     * @return A short (255 max characters) human-readable string to provide additional details about a running or
     *         stopped container.
     */
    public final String reason() {
        return reason;
    }

    /**
     * For responses, this returns true if the service returned a value for the NetworkBindings property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasNetworkBindings() {
        return networkBindings != null && !(networkBindings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The network bindings associated with the container.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNetworkBindings} method.
     * </p>
     * 
     * @return The network bindings associated with the container.
     */
    public final List<NetworkBinding> networkBindings() {
        return networkBindings;
    }

    /**
     * For responses, this returns true if the service returned a value for the NetworkInterfaces property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasNetworkInterfaces() {
        return networkInterfaces != null && !(networkInterfaces instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The network interfaces associated with the container.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNetworkInterfaces} method.
     * </p>
     * 
     * @return The network interfaces associated with the container.
     */
    public final List<NetworkInterface> networkInterfaces() {
        return networkInterfaces;
    }

    /**
     * <p>
     * The health status of the container. If health checks aren't configured for this container in its task definition,
     * then it reports the health status as <code>UNKNOWN</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #healthStatus} will
     * return {@link HealthStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #healthStatusAsString}.
     * </p>
     * 
     * @return The health status of the container. If health checks aren't configured for this container in its task
     *         definition, then it reports the health status as <code>UNKNOWN</code>.
     * @see HealthStatus
     */
    public final HealthStatus healthStatus() {
        return HealthStatus.fromValue(healthStatus);
    }

    /**
     * <p>
     * The health status of the container. If health checks aren't configured for this container in its task definition,
     * then it reports the health status as <code>UNKNOWN</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #healthStatus} will
     * return {@link HealthStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #healthStatusAsString}.
     * </p>
     * 
     * @return The health status of the container. If health checks aren't configured for this container in its task
     *         definition, then it reports the health status as <code>UNKNOWN</code>.
     * @see HealthStatus
     */
    public final String healthStatusAsString() {
        return healthStatus;
    }

    /**
     * For responses, this returns true if the service returned a value for the ManagedAgents property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasManagedAgents() {
        return managedAgents != null && !(managedAgents instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The details of any Amazon ECS managed agents associated with the container.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasManagedAgents} method.
     * </p>
     * 
     * @return The details of any Amazon ECS managed agents associated with the container.
     */
    public final List<ManagedAgent> managedAgents() {
        return managedAgents;
    }

    /**
     * <p>
     * The number of CPU units set for the container. The value is <code>0</code> if no value was specified in the
     * container definition when the task definition was registered.
     * </p>
     * 
     * @return The number of CPU units set for the container. The value is <code>0</code> if no value was specified in
     *         the container definition when the task definition was registered.
     */
    public final String cpu() {
        return cpu;
    }

    /**
     * <p>
     * The hard limit (in MiB) of memory set for the container.
     * </p>
     * 
     * @return The hard limit (in MiB) of memory set for the container.
     */
    public final String memory() {
        return memory;
    }

    /**
     * <p>
     * The soft limit (in MiB) of memory set for the container.
     * </p>
     * 
     * @return The soft limit (in MiB) of memory set for the container.
     */
    public final String memoryReservation() {
        return memoryReservation;
    }

    /**
     * For responses, this returns true if the service returned a value for the GpuIds property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasGpuIds() {
        return gpuIds != null && !(gpuIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IDs of each GPU assigned to the container.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasGpuIds} method.
     * </p>
     * 
     * @return The IDs of each GPU assigned to the container.
     */
    public final List<String> gpuIds() {
        return gpuIds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(containerArn());
        hashCode = 31 * hashCode + Objects.hashCode(taskArn());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(image());
        hashCode = 31 * hashCode + Objects.hashCode(imageDigest());
        hashCode = 31 * hashCode + Objects.hashCode(runtimeId());
        hashCode = 31 * hashCode + Objects.hashCode(lastStatus());
        hashCode = 31 * hashCode + Objects.hashCode(exitCode());
        hashCode = 31 * hashCode + Objects.hashCode(reason());
        hashCode = 31 * hashCode + Objects.hashCode(hasNetworkBindings() ? networkBindings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasNetworkInterfaces() ? networkInterfaces() : null);
        hashCode = 31 * hashCode + Objects.hashCode(healthStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasManagedAgents() ? managedAgents() : null);
        hashCode = 31 * hashCode + Objects.hashCode(cpu());
        hashCode = 31 * hashCode + Objects.hashCode(memory());
        hashCode = 31 * hashCode + Objects.hashCode(memoryReservation());
        hashCode = 31 * hashCode + Objects.hashCode(hasGpuIds() ? gpuIds() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Container)) {
            return false;
        }
        Container other = (Container) obj;
        return Objects.equals(containerArn(), other.containerArn()) && Objects.equals(taskArn(), other.taskArn())
                && Objects.equals(name(), other.name()) && Objects.equals(image(), other.image())
                && Objects.equals(imageDigest(), other.imageDigest()) && Objects.equals(runtimeId(), other.runtimeId())
                && Objects.equals(lastStatus(), other.lastStatus()) && Objects.equals(exitCode(), other.exitCode())
                && Objects.equals(reason(), other.reason()) && hasNetworkBindings() == other.hasNetworkBindings()
                && Objects.equals(networkBindings(), other.networkBindings())
                && hasNetworkInterfaces() == other.hasNetworkInterfaces()
                && Objects.equals(networkInterfaces(), other.networkInterfaces())
                && Objects.equals(healthStatusAsString(), other.healthStatusAsString())
                && hasManagedAgents() == other.hasManagedAgents() && Objects.equals(managedAgents(), other.managedAgents())
                && Objects.equals(cpu(), other.cpu()) && Objects.equals(memory(), other.memory())
                && Objects.equals(memoryReservation(), other.memoryReservation()) && hasGpuIds() == other.hasGpuIds()
                && Objects.equals(gpuIds(), other.gpuIds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Container").add("ContainerArn", containerArn()).add("TaskArn", taskArn()).add("Name", name())
                .add("Image", image()).add("ImageDigest", imageDigest()).add("RuntimeId", runtimeId())
                .add("LastStatus", lastStatus()).add("ExitCode", exitCode()).add("Reason", reason())
                .add("NetworkBindings", hasNetworkBindings() ? networkBindings() : null)
                .add("NetworkInterfaces", hasNetworkInterfaces() ? networkInterfaces() : null)
                .add("HealthStatus", healthStatusAsString()).add("ManagedAgents", hasManagedAgents() ? managedAgents() : null)
                .add("Cpu", cpu()).add("Memory", memory()).add("MemoryReservation", memoryReservation())
                .add("GpuIds", hasGpuIds() ? gpuIds() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "containerArn":
            return Optional.ofNullable(clazz.cast(containerArn()));
        case "taskArn":
            return Optional.ofNullable(clazz.cast(taskArn()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "image":
            return Optional.ofNullable(clazz.cast(image()));
        case "imageDigest":
            return Optional.ofNullable(clazz.cast(imageDigest()));
        case "runtimeId":
            return Optional.ofNullable(clazz.cast(runtimeId()));
        case "lastStatus":
            return Optional.ofNullable(clazz.cast(lastStatus()));
        case "exitCode":
            return Optional.ofNullable(clazz.cast(exitCode()));
        case "reason":
            return Optional.ofNullable(clazz.cast(reason()));
        case "networkBindings":
            return Optional.ofNullable(clazz.cast(networkBindings()));
        case "networkInterfaces":
            return Optional.ofNullable(clazz.cast(networkInterfaces()));
        case "healthStatus":
            return Optional.ofNullable(clazz.cast(healthStatusAsString()));
        case "managedAgents":
            return Optional.ofNullable(clazz.cast(managedAgents()));
        case "cpu":
            return Optional.ofNullable(clazz.cast(cpu()));
        case "memory":
            return Optional.ofNullable(clazz.cast(memory()));
        case "memoryReservation":
            return Optional.ofNullable(clazz.cast(memoryReservation()));
        case "gpuIds":
            return Optional.ofNullable(clazz.cast(gpuIds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Container, T> g) {
        return obj -> g.apply((Container) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Container> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the container.
         * </p>
         * 
         * @param containerArn
         *        The Amazon Resource Name (ARN) of the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerArn(String containerArn);

        /**
         * <p>
         * The ARN of the task.
         * </p>
         * 
         * @param taskArn
         *        The ARN of the task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskArn(String taskArn);

        /**
         * <p>
         * The name of the container.
         * </p>
         * 
         * @param name
         *        The name of the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The image used for the container.
         * </p>
         * 
         * @param image
         *        The image used for the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder image(String image);

        /**
         * <p>
         * The container image manifest digest.
         * </p>
         * <note>
         * <p>
         * The <code>imageDigest</code> is only returned if the container is using an image hosted in Amazon ECR,
         * otherwise it is omitted.
         * </p>
         * </note>
         * 
         * @param imageDigest
         *        The container image manifest digest.</p> <note>
         *        <p>
         *        The <code>imageDigest</code> is only returned if the container is using an image hosted in Amazon ECR,
         *        otherwise it is omitted.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageDigest(String imageDigest);

        /**
         * <p>
         * The ID of the Docker container.
         * </p>
         * 
         * @param runtimeId
         *        The ID of the Docker container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder runtimeId(String runtimeId);

        /**
         * <p>
         * The last known status of the container.
         * </p>
         * 
         * @param lastStatus
         *        The last known status of the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastStatus(String lastStatus);

        /**
         * <p>
         * The exit code returned from the container.
         * </p>
         * 
         * @param exitCode
         *        The exit code returned from the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exitCode(Integer exitCode);

        /**
         * <p>
         * A short (255 max characters) human-readable string to provide additional details about a running or stopped
         * container.
         * </p>
         * 
         * @param reason
         *        A short (255 max characters) human-readable string to provide additional details about a running or
         *        stopped container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reason(String reason);

        /**
         * <p>
         * The network bindings associated with the container.
         * </p>
         * 
         * @param networkBindings
         *        The network bindings associated with the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkBindings(Collection<NetworkBinding> networkBindings);

        /**
         * <p>
         * The network bindings associated with the container.
         * </p>
         * 
         * @param networkBindings
         *        The network bindings associated with the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkBindings(NetworkBinding... networkBindings);

        /**
         * <p>
         * The network bindings associated with the container.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.NetworkBinding.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ecs.model.NetworkBinding#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecs.model.NetworkBinding.Builder#build()} is called immediately and
         * its result is passed to {@link #networkBindings(List<NetworkBinding>)}.
         * 
         * @param networkBindings
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecs.model.NetworkBinding.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkBindings(java.util.Collection<NetworkBinding>)
         */
        Builder networkBindings(Consumer<NetworkBinding.Builder>... networkBindings);

        /**
         * <p>
         * The network interfaces associated with the container.
         * </p>
         * 
         * @param networkInterfaces
         *        The network interfaces associated with the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaces(Collection<NetworkInterface> networkInterfaces);

        /**
         * <p>
         * The network interfaces associated with the container.
         * </p>
         * 
         * @param networkInterfaces
         *        The network interfaces associated with the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaces(NetworkInterface... networkInterfaces);

        /**
         * <p>
         * The network interfaces associated with the container.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.NetworkInterface.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ecs.model.NetworkInterface#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecs.model.NetworkInterface.Builder#build()} is called immediately and
         * its result is passed to {@link #networkInterfaces(List<NetworkInterface>)}.
         * 
         * @param networkInterfaces
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecs.model.NetworkInterface.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkInterfaces(java.util.Collection<NetworkInterface>)
         */
        Builder networkInterfaces(Consumer<NetworkInterface.Builder>... networkInterfaces);

        /**
         * <p>
         * The health status of the container. If health checks aren't configured for this container in its task
         * definition, then it reports the health status as <code>UNKNOWN</code>.
         * </p>
         * 
         * @param healthStatus
         *        The health status of the container. If health checks aren't configured for this container in its task
         *        definition, then it reports the health status as <code>UNKNOWN</code>.
         * @see HealthStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HealthStatus
         */
        Builder healthStatus(String healthStatus);

        /**
         * <p>
         * The health status of the container. If health checks aren't configured for this container in its task
         * definition, then it reports the health status as <code>UNKNOWN</code>.
         * </p>
         * 
         * @param healthStatus
         *        The health status of the container. If health checks aren't configured for this container in its task
         *        definition, then it reports the health status as <code>UNKNOWN</code>.
         * @see HealthStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HealthStatus
         */
        Builder healthStatus(HealthStatus healthStatus);

        /**
         * <p>
         * The details of any Amazon ECS managed agents associated with the container.
         * </p>
         * 
         * @param managedAgents
         *        The details of any Amazon ECS managed agents associated with the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder managedAgents(Collection<ManagedAgent> managedAgents);

        /**
         * <p>
         * The details of any Amazon ECS managed agents associated with the container.
         * </p>
         * 
         * @param managedAgents
         *        The details of any Amazon ECS managed agents associated with the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder managedAgents(ManagedAgent... managedAgents);

        /**
         * <p>
         * The details of any Amazon ECS managed agents associated with the container.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.ManagedAgent.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ecs.model.ManagedAgent#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecs.model.ManagedAgent.Builder#build()} is called immediately and its
         * result is passed to {@link #managedAgents(List<ManagedAgent>)}.
         * 
         * @param managedAgents
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecs.model.ManagedAgent.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #managedAgents(java.util.Collection<ManagedAgent>)
         */
        Builder managedAgents(Consumer<ManagedAgent.Builder>... managedAgents);

        /**
         * <p>
         * The number of CPU units set for the container. The value is <code>0</code> if no value was specified in the
         * container definition when the task definition was registered.
         * </p>
         * 
         * @param cpu
         *        The number of CPU units set for the container. The value is <code>0</code> if no value was specified
         *        in the container definition when the task definition was registered.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cpu(String cpu);

        /**
         * <p>
         * The hard limit (in MiB) of memory set for the container.
         * </p>
         * 
         * @param memory
         *        The hard limit (in MiB) of memory set for the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memory(String memory);

        /**
         * <p>
         * The soft limit (in MiB) of memory set for the container.
         * </p>
         * 
         * @param memoryReservation
         *        The soft limit (in MiB) of memory set for the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memoryReservation(String memoryReservation);

        /**
         * <p>
         * The IDs of each GPU assigned to the container.
         * </p>
         * 
         * @param gpuIds
         *        The IDs of each GPU assigned to the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gpuIds(Collection<String> gpuIds);

        /**
         * <p>
         * The IDs of each GPU assigned to the container.
         * </p>
         * 
         * @param gpuIds
         *        The IDs of each GPU assigned to the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gpuIds(String... gpuIds);
    }

    static final class BuilderImpl implements Builder {
        private String containerArn;

        private String taskArn;

        private String name;

        private String image;

        private String imageDigest;

        private String runtimeId;

        private String lastStatus;

        private Integer exitCode;

        private String reason;

        private List<NetworkBinding> networkBindings = DefaultSdkAutoConstructList.getInstance();

        private List<NetworkInterface> networkInterfaces = DefaultSdkAutoConstructList.getInstance();

        private String healthStatus;

        private List<ManagedAgent> managedAgents = DefaultSdkAutoConstructList.getInstance();

        private String cpu;

        private String memory;

        private String memoryReservation;

        private List<String> gpuIds = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Container model) {
            containerArn(model.containerArn);
            taskArn(model.taskArn);
            name(model.name);
            image(model.image);
            imageDigest(model.imageDigest);
            runtimeId(model.runtimeId);
            lastStatus(model.lastStatus);
            exitCode(model.exitCode);
            reason(model.reason);
            networkBindings(model.networkBindings);
            networkInterfaces(model.networkInterfaces);
            healthStatus(model.healthStatus);
            managedAgents(model.managedAgents);
            cpu(model.cpu);
            memory(model.memory);
            memoryReservation(model.memoryReservation);
            gpuIds(model.gpuIds);
        }

        public final String getContainerArn() {
            return containerArn;
        }

        public final void setContainerArn(String containerArn) {
            this.containerArn = containerArn;
        }

        @Override
        public final Builder containerArn(String containerArn) {
            this.containerArn = containerArn;
            return this;
        }

        public final String getTaskArn() {
            return taskArn;
        }

        public final void setTaskArn(String taskArn) {
            this.taskArn = taskArn;
        }

        @Override
        public final Builder taskArn(String taskArn) {
            this.taskArn = taskArn;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getImage() {
            return image;
        }

        public final void setImage(String image) {
            this.image = image;
        }

        @Override
        public final Builder image(String image) {
            this.image = image;
            return this;
        }

        public final String getImageDigest() {
            return imageDigest;
        }

        public final void setImageDigest(String imageDigest) {
            this.imageDigest = imageDigest;
        }

        @Override
        public final Builder imageDigest(String imageDigest) {
            this.imageDigest = imageDigest;
            return this;
        }

        public final String getRuntimeId() {
            return runtimeId;
        }

        public final void setRuntimeId(String runtimeId) {
            this.runtimeId = runtimeId;
        }

        @Override
        public final Builder runtimeId(String runtimeId) {
            this.runtimeId = runtimeId;
            return this;
        }

        public final String getLastStatus() {
            return lastStatus;
        }

        public final void setLastStatus(String lastStatus) {
            this.lastStatus = lastStatus;
        }

        @Override
        public final Builder lastStatus(String lastStatus) {
            this.lastStatus = lastStatus;
            return this;
        }

        public final Integer getExitCode() {
            return exitCode;
        }

        public final void setExitCode(Integer exitCode) {
            this.exitCode = exitCode;
        }

        @Override
        public final Builder exitCode(Integer exitCode) {
            this.exitCode = exitCode;
            return this;
        }

        public final String getReason() {
            return reason;
        }

        public final void setReason(String reason) {
            this.reason = reason;
        }

        @Override
        public final Builder reason(String reason) {
            this.reason = reason;
            return this;
        }

        public final List<NetworkBinding.Builder> getNetworkBindings() {
            List<NetworkBinding.Builder> result = NetworkBindingsCopier.copyToBuilder(this.networkBindings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setNetworkBindings(Collection<NetworkBinding.BuilderImpl> networkBindings) {
            this.networkBindings = NetworkBindingsCopier.copyFromBuilder(networkBindings);
        }

        @Override
        public final Builder networkBindings(Collection<NetworkBinding> networkBindings) {
            this.networkBindings = NetworkBindingsCopier.copy(networkBindings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkBindings(NetworkBinding... networkBindings) {
            networkBindings(Arrays.asList(networkBindings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkBindings(Consumer<NetworkBinding.Builder>... networkBindings) {
            networkBindings(Stream.of(networkBindings).map(c -> NetworkBinding.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<NetworkInterface.Builder> getNetworkInterfaces() {
            List<NetworkInterface.Builder> result = NetworkInterfacesCopier.copyToBuilder(this.networkInterfaces);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setNetworkInterfaces(Collection<NetworkInterface.BuilderImpl> networkInterfaces) {
            this.networkInterfaces = NetworkInterfacesCopier.copyFromBuilder(networkInterfaces);
        }

        @Override
        public final Builder networkInterfaces(Collection<NetworkInterface> networkInterfaces) {
            this.networkInterfaces = NetworkInterfacesCopier.copy(networkInterfaces);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkInterfaces(NetworkInterface... networkInterfaces) {
            networkInterfaces(Arrays.asList(networkInterfaces));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkInterfaces(Consumer<NetworkInterface.Builder>... networkInterfaces) {
            networkInterfaces(Stream.of(networkInterfaces).map(c -> NetworkInterface.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getHealthStatus() {
            return healthStatus;
        }

        public final void setHealthStatus(String healthStatus) {
            this.healthStatus = healthStatus;
        }

        @Override
        public final Builder healthStatus(String healthStatus) {
            this.healthStatus = healthStatus;
            return this;
        }

        @Override
        public final Builder healthStatus(HealthStatus healthStatus) {
            this.healthStatus(healthStatus == null ? null : healthStatus.toString());
            return this;
        }

        public final List<ManagedAgent.Builder> getManagedAgents() {
            List<ManagedAgent.Builder> result = ManagedAgentsCopier.copyToBuilder(this.managedAgents);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setManagedAgents(Collection<ManagedAgent.BuilderImpl> managedAgents) {
            this.managedAgents = ManagedAgentsCopier.copyFromBuilder(managedAgents);
        }

        @Override
        public final Builder managedAgents(Collection<ManagedAgent> managedAgents) {
            this.managedAgents = ManagedAgentsCopier.copy(managedAgents);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder managedAgents(ManagedAgent... managedAgents) {
            managedAgents(Arrays.asList(managedAgents));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder managedAgents(Consumer<ManagedAgent.Builder>... managedAgents) {
            managedAgents(Stream.of(managedAgents).map(c -> ManagedAgent.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getCpu() {
            return cpu;
        }

        public final void setCpu(String cpu) {
            this.cpu = cpu;
        }

        @Override
        public final Builder cpu(String cpu) {
            this.cpu = cpu;
            return this;
        }

        public final String getMemory() {
            return memory;
        }

        public final void setMemory(String memory) {
            this.memory = memory;
        }

        @Override
        public final Builder memory(String memory) {
            this.memory = memory;
            return this;
        }

        public final String getMemoryReservation() {
            return memoryReservation;
        }

        public final void setMemoryReservation(String memoryReservation) {
            this.memoryReservation = memoryReservation;
        }

        @Override
        public final Builder memoryReservation(String memoryReservation) {
            this.memoryReservation = memoryReservation;
            return this;
        }

        public final Collection<String> getGpuIds() {
            if (gpuIds instanceof SdkAutoConstructList) {
                return null;
            }
            return gpuIds;
        }

        public final void setGpuIds(Collection<String> gpuIds) {
            this.gpuIds = GpuIdsCopier.copy(gpuIds);
        }

        @Override
        public final Builder gpuIds(Collection<String> gpuIds) {
            this.gpuIds = GpuIdsCopier.copy(gpuIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder gpuIds(String... gpuIds) {
            gpuIds(Arrays.asList(gpuIds));
            return this;
        }

        @Override
        public Container build() {
            return new Container(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
