/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object representing a container health check. Health check parameters that are specified in a container definition
 * override any Docker health checks that exist in the container image (such as those specified in a parent image or
 * from the image's Dockerfile).
 * </p>
 * <note>
 * <p>
 * The Amazon ECS container agent only monitors and reports on the health checks specified in the task definition.
 * Amazon ECS does not monitor Docker health checks that are embedded in a container image and not specified in the
 * container definition. Health check parameters that are specified in a container definition override any Docker health
 * checks that exist in the container image.
 * </p>
 * </note>
 * <p>
 * You can view the health status of both individual containers and a task with the DescribeTasks API operation or when
 * viewing the task details in the console.
 * </p>
 * <p>
 * The following describes the possible <code>healthStatus</code> values for a container:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>HEALTHY</code>-The container health check has passed successfully.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>UNHEALTHY</code>-The container health check has failed.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>UNKNOWN</code>-The container health check is being evaluated or there's no container health check defined.
 * </p>
 * </li>
 * </ul>
 * <p>
 * The following describes the possible <code>healthStatus</code> values for a task. The container health check status
 * of nonessential containers do not have an effect on the health status of a task.
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>HEALTHY</code>-All essential containers within the task have passed their health checks.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>UNHEALTHY</code>-One or more essential containers have failed their health check.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>UNKNOWN</code>-The essential containers within the task are still having their health checks evaluated or there
 * are no container health checks defined.
 * </p>
 * </li>
 * </ul>
 * <p>
 * If a task is run manually, and not as part of a service, the task will continue its lifecycle regardless of its
 * health status. For tasks that are part of a service, if the task reports as unhealthy then the task will be stopped
 * and the service scheduler will replace it.
 * </p>
 * <p>
 * The following are notes about container health check support:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Container health checks require version 1.17.0 or greater of the Amazon ECS container agent. For more information,
 * see <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html">Updating the Amazon
 * ECS container agent</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * Container health checks are supported for Fargate tasks if you're using platform version <code>1.1.0</code> or
 * greater. For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html">Fargate platform
 * versions</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * Container health checks aren't supported for tasks that are part of a service that's configured to use a Classic Load
 * Balancer.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class HealthCheck implements SdkPojo, Serializable, ToCopyableBuilder<HealthCheck.Builder, HealthCheck> {
    private static final SdkField<List<String>> COMMAND_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("command")
            .getter(getter(HealthCheck::command))
            .setter(setter(Builder::command))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("command").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> INTERVAL_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("interval").getter(getter(HealthCheck::interval)).setter(setter(Builder::interval))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("interval").build()).build();

    private static final SdkField<Integer> TIMEOUT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("timeout").getter(getter(HealthCheck::timeout)).setter(setter(Builder::timeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("timeout").build()).build();

    private static final SdkField<Integer> RETRIES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("retries").getter(getter(HealthCheck::retries)).setter(setter(Builder::retries))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("retries").build()).build();

    private static final SdkField<Integer> START_PERIOD_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("startPeriod").getter(getter(HealthCheck::startPeriod)).setter(setter(Builder::startPeriod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startPeriod").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COMMAND_FIELD, INTERVAL_FIELD,
            TIMEOUT_FIELD, RETRIES_FIELD, START_PERIOD_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<String> command;

    private final Integer interval;

    private final Integer timeout;

    private final Integer retries;

    private final Integer startPeriod;

    private HealthCheck(BuilderImpl builder) {
        this.command = builder.command;
        this.interval = builder.interval;
        this.timeout = builder.timeout;
        this.retries = builder.retries;
        this.startPeriod = builder.startPeriod;
    }

    /**
     * For responses, this returns true if the service returned a value for the Command property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasCommand() {
        return command != null && !(command instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A string array representing the command that the container runs to determine if it is healthy. The string array
     * must start with <code>CMD</code> to run the command arguments directly, or <code>CMD-SHELL</code> to run the
     * command with the container's default shell.
     * </p>
     * <p>
     * When you use the Amazon Web Services Management Console JSON panel, the Command Line Interface, or the APIs,
     * enclose the list of commands in brackets.
     * </p>
     * <p>
     * <code>[ "CMD-SHELL", "curl -f http://localhost/ || exit 1" ]</code>
     * </p>
     * <p>
     * You don't need to include the brackets when you use the Amazon Web Services Management Console.
     * </p>
     * <p>
     * <code> "CMD-SHELL", "curl -f http://localhost/ || exit 1" </code>
     * </p>
     * <p>
     * An exit code of 0 indicates success, and non-zero exit code indicates failure. For more information, see
     * <code>HealthCheck</code> in the <a
     * href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section of the
     * <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCommand} method.
     * </p>
     * 
     * @return A string array representing the command that the container runs to determine if it is healthy. The string
     *         array must start with <code>CMD</code> to run the command arguments directly, or <code>CMD-SHELL</code>
     *         to run the command with the container's default shell. </p>
     *         <p>
     *         When you use the Amazon Web Services Management Console JSON panel, the Command Line Interface, or the
     *         APIs, enclose the list of commands in brackets.
     *         </p>
     *         <p>
     *         <code>[ "CMD-SHELL", "curl -f http://localhost/ || exit 1" ]</code>
     *         </p>
     *         <p>
     *         You don't need to include the brackets when you use the Amazon Web Services Management Console.
     *         </p>
     *         <p>
     *         <code> "CMD-SHELL", "curl -f http://localhost/ || exit 1" </code>
     *         </p>
     *         <p>
     *         An exit code of 0 indicates success, and non-zero exit code indicates failure. For more information, see
     *         <code>HealthCheck</code> in the <a
     *         href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section
     *         of the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a>.
     */
    public final List<String> command() {
        return command;
    }

    /**
     * <p>
     * The time period in seconds between each health check execution. You may specify between 5 and 300 seconds. The
     * default value is 30 seconds.
     * </p>
     * 
     * @return The time period in seconds between each health check execution. You may specify between 5 and 300
     *         seconds. The default value is 30 seconds.
     */
    public final Integer interval() {
        return interval;
    }

    /**
     * <p>
     * The time period in seconds to wait for a health check to succeed before it is considered a failure. You may
     * specify between 2 and 60 seconds. The default value is 5.
     * </p>
     * 
     * @return The time period in seconds to wait for a health check to succeed before it is considered a failure. You
     *         may specify between 2 and 60 seconds. The default value is 5.
     */
    public final Integer timeout() {
        return timeout;
    }

    /**
     * <p>
     * The number of times to retry a failed health check before the container is considered unhealthy. You may specify
     * between 1 and 10 retries. The default value is 3.
     * </p>
     * 
     * @return The number of times to retry a failed health check before the container is considered unhealthy. You may
     *         specify between 1 and 10 retries. The default value is 3.
     */
    public final Integer retries() {
        return retries;
    }

    /**
     * <p>
     * The optional grace period to provide containers time to bootstrap before failed health checks count towards the
     * maximum number of retries. You can specify between 0 and 300 seconds. By default, the <code>startPeriod</code> is
     * disabled.
     * </p>
     * <note>
     * <p>
     * If a health check succeeds within the <code>startPeriod</code>, then the container is considered healthy and any
     * subsequent failures count toward the maximum number of retries.
     * </p>
     * </note>
     * 
     * @return The optional grace period to provide containers time to bootstrap before failed health checks count
     *         towards the maximum number of retries. You can specify between 0 and 300 seconds. By default, the
     *         <code>startPeriod</code> is disabled.</p> <note>
     *         <p>
     *         If a health check succeeds within the <code>startPeriod</code>, then the container is considered healthy
     *         and any subsequent failures count toward the maximum number of retries.
     *         </p>
     */
    public final Integer startPeriod() {
        return startPeriod;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasCommand() ? command() : null);
        hashCode = 31 * hashCode + Objects.hashCode(interval());
        hashCode = 31 * hashCode + Objects.hashCode(timeout());
        hashCode = 31 * hashCode + Objects.hashCode(retries());
        hashCode = 31 * hashCode + Objects.hashCode(startPeriod());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof HealthCheck)) {
            return false;
        }
        HealthCheck other = (HealthCheck) obj;
        return hasCommand() == other.hasCommand() && Objects.equals(command(), other.command())
                && Objects.equals(interval(), other.interval()) && Objects.equals(timeout(), other.timeout())
                && Objects.equals(retries(), other.retries()) && Objects.equals(startPeriod(), other.startPeriod());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("HealthCheck").add("Command", hasCommand() ? command() : null).add("Interval", interval())
                .add("Timeout", timeout()).add("Retries", retries()).add("StartPeriod", startPeriod()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "command":
            return Optional.ofNullable(clazz.cast(command()));
        case "interval":
            return Optional.ofNullable(clazz.cast(interval()));
        case "timeout":
            return Optional.ofNullable(clazz.cast(timeout()));
        case "retries":
            return Optional.ofNullable(clazz.cast(retries()));
        case "startPeriod":
            return Optional.ofNullable(clazz.cast(startPeriod()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<HealthCheck, T> g) {
        return obj -> g.apply((HealthCheck) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, HealthCheck> {
        /**
         * <p>
         * A string array representing the command that the container runs to determine if it is healthy. The string
         * array must start with <code>CMD</code> to run the command arguments directly, or <code>CMD-SHELL</code> to
         * run the command with the container's default shell.
         * </p>
         * <p>
         * When you use the Amazon Web Services Management Console JSON panel, the Command Line Interface, or the APIs,
         * enclose the list of commands in brackets.
         * </p>
         * <p>
         * <code>[ "CMD-SHELL", "curl -f http://localhost/ || exit 1" ]</code>
         * </p>
         * <p>
         * You don't need to include the brackets when you use the Amazon Web Services Management Console.
         * </p>
         * <p>
         * <code> "CMD-SHELL", "curl -f http://localhost/ || exit 1" </code>
         * </p>
         * <p>
         * An exit code of 0 indicates success, and non-zero exit code indicates failure. For more information, see
         * <code>HealthCheck</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section of
         * the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a>.
         * </p>
         * 
         * @param command
         *        A string array representing the command that the container runs to determine if it is healthy. The
         *        string array must start with <code>CMD</code> to run the command arguments directly, or
         *        <code>CMD-SHELL</code> to run the command with the container's default shell. </p>
         *        <p>
         *        When you use the Amazon Web Services Management Console JSON panel, the Command Line Interface, or the
         *        APIs, enclose the list of commands in brackets.
         *        </p>
         *        <p>
         *        <code>[ "CMD-SHELL", "curl -f http://localhost/ || exit 1" ]</code>
         *        </p>
         *        <p>
         *        You don't need to include the brackets when you use the Amazon Web Services Management Console.
         *        </p>
         *        <p>
         *        <code> "CMD-SHELL", "curl -f http://localhost/ || exit 1" </code>
         *        </p>
         *        <p>
         *        An exit code of 0 indicates success, and non-zero exit code indicates failure. For more information,
         *        see <code>HealthCheck</code> in the <a
         *        href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a>
         *        section of the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(Collection<String> command);

        /**
         * <p>
         * A string array representing the command that the container runs to determine if it is healthy. The string
         * array must start with <code>CMD</code> to run the command arguments directly, or <code>CMD-SHELL</code> to
         * run the command with the container's default shell.
         * </p>
         * <p>
         * When you use the Amazon Web Services Management Console JSON panel, the Command Line Interface, or the APIs,
         * enclose the list of commands in brackets.
         * </p>
         * <p>
         * <code>[ "CMD-SHELL", "curl -f http://localhost/ || exit 1" ]</code>
         * </p>
         * <p>
         * You don't need to include the brackets when you use the Amazon Web Services Management Console.
         * </p>
         * <p>
         * <code> "CMD-SHELL", "curl -f http://localhost/ || exit 1" </code>
         * </p>
         * <p>
         * An exit code of 0 indicates success, and non-zero exit code indicates failure. For more information, see
         * <code>HealthCheck</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section of
         * the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a>.
         * </p>
         * 
         * @param command
         *        A string array representing the command that the container runs to determine if it is healthy. The
         *        string array must start with <code>CMD</code> to run the command arguments directly, or
         *        <code>CMD-SHELL</code> to run the command with the container's default shell. </p>
         *        <p>
         *        When you use the Amazon Web Services Management Console JSON panel, the Command Line Interface, or the
         *        APIs, enclose the list of commands in brackets.
         *        </p>
         *        <p>
         *        <code>[ "CMD-SHELL", "curl -f http://localhost/ || exit 1" ]</code>
         *        </p>
         *        <p>
         *        You don't need to include the brackets when you use the Amazon Web Services Management Console.
         *        </p>
         *        <p>
         *        <code> "CMD-SHELL", "curl -f http://localhost/ || exit 1" </code>
         *        </p>
         *        <p>
         *        An exit code of 0 indicates success, and non-zero exit code indicates failure. For more information,
         *        see <code>HealthCheck</code> in the <a
         *        href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a>
         *        section of the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(String... command);

        /**
         * <p>
         * The time period in seconds between each health check execution. You may specify between 5 and 300 seconds.
         * The default value is 30 seconds.
         * </p>
         * 
         * @param interval
         *        The time period in seconds between each health check execution. You may specify between 5 and 300
         *        seconds. The default value is 30 seconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder interval(Integer interval);

        /**
         * <p>
         * The time period in seconds to wait for a health check to succeed before it is considered a failure. You may
         * specify between 2 and 60 seconds. The default value is 5.
         * </p>
         * 
         * @param timeout
         *        The time period in seconds to wait for a health check to succeed before it is considered a failure.
         *        You may specify between 2 and 60 seconds. The default value is 5.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeout(Integer timeout);

        /**
         * <p>
         * The number of times to retry a failed health check before the container is considered unhealthy. You may
         * specify between 1 and 10 retries. The default value is 3.
         * </p>
         * 
         * @param retries
         *        The number of times to retry a failed health check before the container is considered unhealthy. You
         *        may specify between 1 and 10 retries. The default value is 3.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder retries(Integer retries);

        /**
         * <p>
         * The optional grace period to provide containers time to bootstrap before failed health checks count towards
         * the maximum number of retries. You can specify between 0 and 300 seconds. By default, the
         * <code>startPeriod</code> is disabled.
         * </p>
         * <note>
         * <p>
         * If a health check succeeds within the <code>startPeriod</code>, then the container is considered healthy and
         * any subsequent failures count toward the maximum number of retries.
         * </p>
         * </note>
         * 
         * @param startPeriod
         *        The optional grace period to provide containers time to bootstrap before failed health checks count
         *        towards the maximum number of retries. You can specify between 0 and 300 seconds. By default, the
         *        <code>startPeriod</code> is disabled.</p> <note>
         *        <p>
         *        If a health check succeeds within the <code>startPeriod</code>, then the container is considered
         *        healthy and any subsequent failures count toward the maximum number of retries.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startPeriod(Integer startPeriod);
    }

    static final class BuilderImpl implements Builder {
        private List<String> command = DefaultSdkAutoConstructList.getInstance();

        private Integer interval;

        private Integer timeout;

        private Integer retries;

        private Integer startPeriod;

        private BuilderImpl() {
        }

        private BuilderImpl(HealthCheck model) {
            command(model.command);
            interval(model.interval);
            timeout(model.timeout);
            retries(model.retries);
            startPeriod(model.startPeriod);
        }

        public final Collection<String> getCommand() {
            if (command instanceof SdkAutoConstructList) {
                return null;
            }
            return command;
        }

        public final void setCommand(Collection<String> command) {
            this.command = StringListCopier.copy(command);
        }

        @Override
        public final Builder command(Collection<String> command) {
            this.command = StringListCopier.copy(command);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder command(String... command) {
            command(Arrays.asList(command));
            return this;
        }

        public final Integer getInterval() {
            return interval;
        }

        public final void setInterval(Integer interval) {
            this.interval = interval;
        }

        @Override
        public final Builder interval(Integer interval) {
            this.interval = interval;
            return this;
        }

        public final Integer getTimeout() {
            return timeout;
        }

        public final void setTimeout(Integer timeout) {
            this.timeout = timeout;
        }

        @Override
        public final Builder timeout(Integer timeout) {
            this.timeout = timeout;
            return this;
        }

        public final Integer getRetries() {
            return retries;
        }

        public final void setRetries(Integer retries) {
            this.retries = retries;
        }

        @Override
        public final Builder retries(Integer retries) {
            this.retries = retries;
            return this;
        }

        public final Integer getStartPeriod() {
            return startPeriod;
        }

        public final void setStartPeriod(Integer startPeriod) {
            this.startPeriod = startPeriod;
        }

        @Override
        public final Builder startPeriod(Integer startPeriod) {
            this.startPeriod = startPeriod;
            return this;
        }

        @Override
        public HealthCheck build() {
            return new HealthCheck(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
