/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This parameter is specified when you're using an Amazon Elastic File System file system for task storage. For more
 * information, see <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/efs-volumes.html">Amazon EFS
 * volumes</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EFSVolumeConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<EFSVolumeConfiguration.Builder, EFSVolumeConfiguration> {
    private static final SdkField<String> FILE_SYSTEM_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("fileSystemId").getter(getter(EFSVolumeConfiguration::fileSystemId))
            .setter(setter(Builder::fileSystemId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("fileSystemId").build()).build();

    private static final SdkField<String> ROOT_DIRECTORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("rootDirectory").getter(getter(EFSVolumeConfiguration::rootDirectory))
            .setter(setter(Builder::rootDirectory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("rootDirectory").build()).build();

    private static final SdkField<String> TRANSIT_ENCRYPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("transitEncryption").getter(getter(EFSVolumeConfiguration::transitEncryptionAsString))
            .setter(setter(Builder::transitEncryption))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("transitEncryption").build()).build();

    private static final SdkField<Integer> TRANSIT_ENCRYPTION_PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("transitEncryptionPort").getter(getter(EFSVolumeConfiguration::transitEncryptionPort))
            .setter(setter(Builder::transitEncryptionPort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("transitEncryptionPort").build())
            .build();

    private static final SdkField<EFSAuthorizationConfig> AUTHORIZATION_CONFIG_FIELD = SdkField
            .<EFSAuthorizationConfig> builder(MarshallingType.SDK_POJO).memberName("authorizationConfig")
            .getter(getter(EFSVolumeConfiguration::authorizationConfig)).setter(setter(Builder::authorizationConfig))
            .constructor(EFSAuthorizationConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authorizationConfig").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FILE_SYSTEM_ID_FIELD,
            ROOT_DIRECTORY_FIELD, TRANSIT_ENCRYPTION_FIELD, TRANSIT_ENCRYPTION_PORT_FIELD, AUTHORIZATION_CONFIG_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("fileSystemId", FILE_SYSTEM_ID_FIELD);
                    put("rootDirectory", ROOT_DIRECTORY_FIELD);
                    put("transitEncryption", TRANSIT_ENCRYPTION_FIELD);
                    put("transitEncryptionPort", TRANSIT_ENCRYPTION_PORT_FIELD);
                    put("authorizationConfig", AUTHORIZATION_CONFIG_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String fileSystemId;

    private final String rootDirectory;

    private final String transitEncryption;

    private final Integer transitEncryptionPort;

    private final EFSAuthorizationConfig authorizationConfig;

    private EFSVolumeConfiguration(BuilderImpl builder) {
        this.fileSystemId = builder.fileSystemId;
        this.rootDirectory = builder.rootDirectory;
        this.transitEncryption = builder.transitEncryption;
        this.transitEncryptionPort = builder.transitEncryptionPort;
        this.authorizationConfig = builder.authorizationConfig;
    }

    /**
     * <p>
     * The Amazon EFS file system ID to use.
     * </p>
     * 
     * @return The Amazon EFS file system ID to use.
     */
    public final String fileSystemId() {
        return fileSystemId;
    }

    /**
     * <p>
     * The directory within the Amazon EFS file system to mount as the root directory inside the host. If this parameter
     * is omitted, the root of the Amazon EFS volume will be used. Specifying <code>/</code> will have the same effect
     * as omitting this parameter.
     * </p>
     * <important>
     * <p>
     * If an EFS access point is specified in the <code>authorizationConfig</code>, the root directory parameter must
     * either be omitted or set to <code>/</code> which will enforce the path set on the EFS access point.
     * </p>
     * </important>
     * 
     * @return The directory within the Amazon EFS file system to mount as the root directory inside the host. If this
     *         parameter is omitted, the root of the Amazon EFS volume will be used. Specifying <code>/</code> will have
     *         the same effect as omitting this parameter.</p> <important>
     *         <p>
     *         If an EFS access point is specified in the <code>authorizationConfig</code>, the root directory parameter
     *         must either be omitted or set to <code>/</code> which will enforce the path set on the EFS access point.
     *         </p>
     */
    public final String rootDirectory() {
        return rootDirectory;
    }

    /**
     * <p>
     * Determines whether to use encryption for Amazon EFS data in transit between the Amazon ECS host and the Amazon
     * EFS server. Transit encryption must be turned on if Amazon EFS IAM authorization is used. If this parameter is
     * omitted, the default value of <code>DISABLED</code> is used. For more information, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/encryption-in-transit.html">Encrypting data in transit</a> in the
     * <i>Amazon Elastic File System User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #transitEncryption}
     * will return {@link EFSTransitEncryption#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #transitEncryptionAsString}.
     * </p>
     * 
     * @return Determines whether to use encryption for Amazon EFS data in transit between the Amazon ECS host and the
     *         Amazon EFS server. Transit encryption must be turned on if Amazon EFS IAM authorization is used. If this
     *         parameter is omitted, the default value of <code>DISABLED</code> is used. For more information, see <a
     *         href="https://docs.aws.amazon.com/efs/latest/ug/encryption-in-transit.html">Encrypting data in
     *         transit</a> in the <i>Amazon Elastic File System User Guide</i>.
     * @see EFSTransitEncryption
     */
    public final EFSTransitEncryption transitEncryption() {
        return EFSTransitEncryption.fromValue(transitEncryption);
    }

    /**
     * <p>
     * Determines whether to use encryption for Amazon EFS data in transit between the Amazon ECS host and the Amazon
     * EFS server. Transit encryption must be turned on if Amazon EFS IAM authorization is used. If this parameter is
     * omitted, the default value of <code>DISABLED</code> is used. For more information, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/encryption-in-transit.html">Encrypting data in transit</a> in the
     * <i>Amazon Elastic File System User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #transitEncryption}
     * will return {@link EFSTransitEncryption#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #transitEncryptionAsString}.
     * </p>
     * 
     * @return Determines whether to use encryption for Amazon EFS data in transit between the Amazon ECS host and the
     *         Amazon EFS server. Transit encryption must be turned on if Amazon EFS IAM authorization is used. If this
     *         parameter is omitted, the default value of <code>DISABLED</code> is used. For more information, see <a
     *         href="https://docs.aws.amazon.com/efs/latest/ug/encryption-in-transit.html">Encrypting data in
     *         transit</a> in the <i>Amazon Elastic File System User Guide</i>.
     * @see EFSTransitEncryption
     */
    public final String transitEncryptionAsString() {
        return transitEncryption;
    }

    /**
     * <p>
     * The port to use when sending encrypted data between the Amazon ECS host and the Amazon EFS server. If you do not
     * specify a transit encryption port, it will use the port selection strategy that the Amazon EFS mount helper uses.
     * For more information, see <a href="https://docs.aws.amazon.com/efs/latest/ug/efs-mount-helper.html">EFS mount
     * helper</a> in the <i>Amazon Elastic File System User Guide</i>.
     * </p>
     * 
     * @return The port to use when sending encrypted data between the Amazon ECS host and the Amazon EFS server. If you
     *         do not specify a transit encryption port, it will use the port selection strategy that the Amazon EFS
     *         mount helper uses. For more information, see <a
     *         href="https://docs.aws.amazon.com/efs/latest/ug/efs-mount-helper.html">EFS mount helper</a> in the
     *         <i>Amazon Elastic File System User Guide</i>.
     */
    public final Integer transitEncryptionPort() {
        return transitEncryptionPort;
    }

    /**
     * <p>
     * The authorization configuration details for the Amazon EFS file system.
     * </p>
     * 
     * @return The authorization configuration details for the Amazon EFS file system.
     */
    public final EFSAuthorizationConfig authorizationConfig() {
        return authorizationConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(fileSystemId());
        hashCode = 31 * hashCode + Objects.hashCode(rootDirectory());
        hashCode = 31 * hashCode + Objects.hashCode(transitEncryptionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(transitEncryptionPort());
        hashCode = 31 * hashCode + Objects.hashCode(authorizationConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EFSVolumeConfiguration)) {
            return false;
        }
        EFSVolumeConfiguration other = (EFSVolumeConfiguration) obj;
        return Objects.equals(fileSystemId(), other.fileSystemId()) && Objects.equals(rootDirectory(), other.rootDirectory())
                && Objects.equals(transitEncryptionAsString(), other.transitEncryptionAsString())
                && Objects.equals(transitEncryptionPort(), other.transitEncryptionPort())
                && Objects.equals(authorizationConfig(), other.authorizationConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EFSVolumeConfiguration").add("FileSystemId", fileSystemId())
                .add("RootDirectory", rootDirectory()).add("TransitEncryption", transitEncryptionAsString())
                .add("TransitEncryptionPort", transitEncryptionPort()).add("AuthorizationConfig", authorizationConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "fileSystemId":
            return Optional.ofNullable(clazz.cast(fileSystemId()));
        case "rootDirectory":
            return Optional.ofNullable(clazz.cast(rootDirectory()));
        case "transitEncryption":
            return Optional.ofNullable(clazz.cast(transitEncryptionAsString()));
        case "transitEncryptionPort":
            return Optional.ofNullable(clazz.cast(transitEncryptionPort()));
        case "authorizationConfig":
            return Optional.ofNullable(clazz.cast(authorizationConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<EFSVolumeConfiguration, T> g) {
        return obj -> g.apply((EFSVolumeConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EFSVolumeConfiguration> {
        /**
         * <p>
         * The Amazon EFS file system ID to use.
         * </p>
         * 
         * @param fileSystemId
         *        The Amazon EFS file system ID to use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileSystemId(String fileSystemId);

        /**
         * <p>
         * The directory within the Amazon EFS file system to mount as the root directory inside the host. If this
         * parameter is omitted, the root of the Amazon EFS volume will be used. Specifying <code>/</code> will have the
         * same effect as omitting this parameter.
         * </p>
         * <important>
         * <p>
         * If an EFS access point is specified in the <code>authorizationConfig</code>, the root directory parameter
         * must either be omitted or set to <code>/</code> which will enforce the path set on the EFS access point.
         * </p>
         * </important>
         * 
         * @param rootDirectory
         *        The directory within the Amazon EFS file system to mount as the root directory inside the host. If
         *        this parameter is omitted, the root of the Amazon EFS volume will be used. Specifying <code>/</code>
         *        will have the same effect as omitting this parameter.</p> <important>
         *        <p>
         *        If an EFS access point is specified in the <code>authorizationConfig</code>, the root directory
         *        parameter must either be omitted or set to <code>/</code> which will enforce the path set on the EFS
         *        access point.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rootDirectory(String rootDirectory);

        /**
         * <p>
         * Determines whether to use encryption for Amazon EFS data in transit between the Amazon ECS host and the
         * Amazon EFS server. Transit encryption must be turned on if Amazon EFS IAM authorization is used. If this
         * parameter is omitted, the default value of <code>DISABLED</code> is used. For more information, see <a
         * href="https://docs.aws.amazon.com/efs/latest/ug/encryption-in-transit.html">Encrypting data in transit</a> in
         * the <i>Amazon Elastic File System User Guide</i>.
         * </p>
         * 
         * @param transitEncryption
         *        Determines whether to use encryption for Amazon EFS data in transit between the Amazon ECS host and
         *        the Amazon EFS server. Transit encryption must be turned on if Amazon EFS IAM authorization is used.
         *        If this parameter is omitted, the default value of <code>DISABLED</code> is used. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/efs/latest/ug/encryption-in-transit.html">Encrypting data in
         *        transit</a> in the <i>Amazon Elastic File System User Guide</i>.
         * @see EFSTransitEncryption
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EFSTransitEncryption
         */
        Builder transitEncryption(String transitEncryption);

        /**
         * <p>
         * Determines whether to use encryption for Amazon EFS data in transit between the Amazon ECS host and the
         * Amazon EFS server. Transit encryption must be turned on if Amazon EFS IAM authorization is used. If this
         * parameter is omitted, the default value of <code>DISABLED</code> is used. For more information, see <a
         * href="https://docs.aws.amazon.com/efs/latest/ug/encryption-in-transit.html">Encrypting data in transit</a> in
         * the <i>Amazon Elastic File System User Guide</i>.
         * </p>
         * 
         * @param transitEncryption
         *        Determines whether to use encryption for Amazon EFS data in transit between the Amazon ECS host and
         *        the Amazon EFS server. Transit encryption must be turned on if Amazon EFS IAM authorization is used.
         *        If this parameter is omitted, the default value of <code>DISABLED</code> is used. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/efs/latest/ug/encryption-in-transit.html">Encrypting data in
         *        transit</a> in the <i>Amazon Elastic File System User Guide</i>.
         * @see EFSTransitEncryption
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EFSTransitEncryption
         */
        Builder transitEncryption(EFSTransitEncryption transitEncryption);

        /**
         * <p>
         * The port to use when sending encrypted data between the Amazon ECS host and the Amazon EFS server. If you do
         * not specify a transit encryption port, it will use the port selection strategy that the Amazon EFS mount
         * helper uses. For more information, see <a
         * href="https://docs.aws.amazon.com/efs/latest/ug/efs-mount-helper.html">EFS mount helper</a> in the <i>Amazon
         * Elastic File System User Guide</i>.
         * </p>
         * 
         * @param transitEncryptionPort
         *        The port to use when sending encrypted data between the Amazon ECS host and the Amazon EFS server. If
         *        you do not specify a transit encryption port, it will use the port selection strategy that the Amazon
         *        EFS mount helper uses. For more information, see <a
         *        href="https://docs.aws.amazon.com/efs/latest/ug/efs-mount-helper.html">EFS mount helper</a> in the
         *        <i>Amazon Elastic File System User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transitEncryptionPort(Integer transitEncryptionPort);

        /**
         * <p>
         * The authorization configuration details for the Amazon EFS file system.
         * </p>
         * 
         * @param authorizationConfig
         *        The authorization configuration details for the Amazon EFS file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authorizationConfig(EFSAuthorizationConfig authorizationConfig);

        /**
         * <p>
         * The authorization configuration details for the Amazon EFS file system.
         * </p>
         * This is a convenience method that creates an instance of the {@link EFSAuthorizationConfig.Builder} avoiding
         * the need to create one manually via {@link EFSAuthorizationConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EFSAuthorizationConfig.Builder#build()} is called immediately and
         * its result is passed to {@link #authorizationConfig(EFSAuthorizationConfig)}.
         * 
         * @param authorizationConfig
         *        a consumer that will call methods on {@link EFSAuthorizationConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #authorizationConfig(EFSAuthorizationConfig)
         */
        default Builder authorizationConfig(Consumer<EFSAuthorizationConfig.Builder> authorizationConfig) {
            return authorizationConfig(EFSAuthorizationConfig.builder().applyMutation(authorizationConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String fileSystemId;

        private String rootDirectory;

        private String transitEncryption;

        private Integer transitEncryptionPort;

        private EFSAuthorizationConfig authorizationConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(EFSVolumeConfiguration model) {
            fileSystemId(model.fileSystemId);
            rootDirectory(model.rootDirectory);
            transitEncryption(model.transitEncryption);
            transitEncryptionPort(model.transitEncryptionPort);
            authorizationConfig(model.authorizationConfig);
        }

        public final String getFileSystemId() {
            return fileSystemId;
        }

        public final void setFileSystemId(String fileSystemId) {
            this.fileSystemId = fileSystemId;
        }

        @Override
        public final Builder fileSystemId(String fileSystemId) {
            this.fileSystemId = fileSystemId;
            return this;
        }

        public final String getRootDirectory() {
            return rootDirectory;
        }

        public final void setRootDirectory(String rootDirectory) {
            this.rootDirectory = rootDirectory;
        }

        @Override
        public final Builder rootDirectory(String rootDirectory) {
            this.rootDirectory = rootDirectory;
            return this;
        }

        public final String getTransitEncryption() {
            return transitEncryption;
        }

        public final void setTransitEncryption(String transitEncryption) {
            this.transitEncryption = transitEncryption;
        }

        @Override
        public final Builder transitEncryption(String transitEncryption) {
            this.transitEncryption = transitEncryption;
            return this;
        }

        @Override
        public final Builder transitEncryption(EFSTransitEncryption transitEncryption) {
            this.transitEncryption(transitEncryption == null ? null : transitEncryption.toString());
            return this;
        }

        public final Integer getTransitEncryptionPort() {
            return transitEncryptionPort;
        }

        public final void setTransitEncryptionPort(Integer transitEncryptionPort) {
            this.transitEncryptionPort = transitEncryptionPort;
        }

        @Override
        public final Builder transitEncryptionPort(Integer transitEncryptionPort) {
            this.transitEncryptionPort = transitEncryptionPort;
            return this;
        }

        public final EFSAuthorizationConfig.Builder getAuthorizationConfig() {
            return authorizationConfig != null ? authorizationConfig.toBuilder() : null;
        }

        public final void setAuthorizationConfig(EFSAuthorizationConfig.BuilderImpl authorizationConfig) {
            this.authorizationConfig = authorizationConfig != null ? authorizationConfig.build() : null;
        }

        @Override
        public final Builder authorizationConfig(EFSAuthorizationConfig authorizationConfig) {
            this.authorizationConfig = authorizationConfig;
            return this;
        }

        @Override
        public EFSVolumeConfiguration build() {
            return new EFSVolumeConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
