/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The details of a capacity provider strategy. A capacity provider strategy can be set when using the <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_RunTask.html">RunTask</a>or <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_CreateCluster.html">CreateCluster</a> APIs or as
 * the default capacity provider strategy for a cluster with the <code>CreateCluster</code> API.
 * </p>
 * <p>
 * Only capacity providers that are already associated with a cluster and have an <code>ACTIVE</code> or
 * <code>UPDATING</code> status can be used in a capacity provider strategy. The <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutClusterCapacityProviders.html"
 * >PutClusterCapacityProviders</a> API is used to associate a capacity provider with a cluster.
 * </p>
 * <p>
 * If specifying a capacity provider that uses an Auto Scaling group, the capacity provider must already be created. New
 * Auto Scaling group capacity providers can be created with the <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_CreateClusterCapacityProvider.html"
 * >CreateClusterCapacityProvider</a> API operation.
 * </p>
 * <p>
 * To use a Fargate capacity provider, specify either the <code>FARGATE</code> or <code>FARGATE_SPOT</code> capacity
 * providers. The Fargate capacity providers are available to all accounts and only need to be associated with a cluster
 * to be used in a capacity provider strategy.
 * </p>
 * <p>
 * With <code>FARGATE_SPOT</code>, you can run interruption tolerant tasks at a rate that's discounted compared to the
 * <code>FARGATE</code> price. <code>FARGATE_SPOT</code> runs tasks on spare compute capacity. When Amazon Web Services
 * needs the capacity back, your tasks are interrupted with a two-minute warning. <code>FARGATE_SPOT</code> supports
 * Linux tasks with the X86_64 architecture on platform version 1.3.0 or later. <code>FARGATE_SPOT</code> supports Linux
 * tasks with the ARM64 architecture on platform version 1.4.0 or later.
 * </p>
 * <p>
 * A capacity provider strategy may contain a maximum of 6 capacity providers.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CapacityProviderStrategyItem implements SdkPojo, Serializable,
        ToCopyableBuilder<CapacityProviderStrategyItem.Builder, CapacityProviderStrategyItem> {
    private static final SdkField<String> CAPACITY_PROVIDER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("capacityProvider").getter(getter(CapacityProviderStrategyItem::capacityProvider))
            .setter(setter(Builder::capacityProvider))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("capacityProvider").build()).build();

    private static final SdkField<Integer> WEIGHT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("weight").getter(getter(CapacityProviderStrategyItem::weight)).setter(setter(Builder::weight))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("weight").build()).build();

    private static final SdkField<Integer> BASE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("base")
            .getter(getter(CapacityProviderStrategyItem::base)).setter(setter(Builder::base))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("base").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CAPACITY_PROVIDER_FIELD,
            WEIGHT_FIELD, BASE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("capacityProvider", CAPACITY_PROVIDER_FIELD);
                    put("weight", WEIGHT_FIELD);
                    put("base", BASE_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String capacityProvider;

    private final Integer weight;

    private final Integer base;

    private CapacityProviderStrategyItem(BuilderImpl builder) {
        this.capacityProvider = builder.capacityProvider;
        this.weight = builder.weight;
        this.base = builder.base;
    }

    /**
     * <p>
     * The short name of the capacity provider.
     * </p>
     * 
     * @return The short name of the capacity provider.
     */
    public final String capacityProvider() {
        return capacityProvider;
    }

    /**
     * <p>
     * The <i>weight</i> value designates the relative percentage of the total number of tasks launched that should use
     * the specified capacity provider. The <code>weight</code> value is taken into consideration after the
     * <code>base</code> value, if defined, is satisfied.
     * </p>
     * <p>
     * If no <code>weight</code> value is specified, the default value of <code>0</code> is used. When multiple capacity
     * providers are specified within a capacity provider strategy, at least one of the capacity providers must have a
     * weight value greater than zero and any capacity providers with a weight of <code>0</code> can't be used to place
     * tasks. If you specify multiple capacity providers in a strategy that all have a weight of <code>0</code>, any
     * <code>RunTask</code> or <code>CreateService</code> actions using the capacity provider strategy will fail.
     * </p>
     * <p>
     * An example scenario for using weights is defining a strategy that contains two capacity providers and both have a
     * weight of <code>1</code>, then when the <code>base</code> is satisfied, the tasks will be split evenly across the
     * two capacity providers. Using that same logic, if you specify a weight of <code>1</code> for
     * <i>capacityProviderA</i> and a weight of <code>4</code> for <i>capacityProviderB</i>, then for every one task
     * that's run using <i>capacityProviderA</i>, four tasks would use <i>capacityProviderB</i>.
     * </p>
     * 
     * @return The <i>weight</i> value designates the relative percentage of the total number of tasks launched that
     *         should use the specified capacity provider. The <code>weight</code> value is taken into consideration
     *         after the <code>base</code> value, if defined, is satisfied.</p>
     *         <p>
     *         If no <code>weight</code> value is specified, the default value of <code>0</code> is used. When multiple
     *         capacity providers are specified within a capacity provider strategy, at least one of the capacity
     *         providers must have a weight value greater than zero and any capacity providers with a weight of
     *         <code>0</code> can't be used to place tasks. If you specify multiple capacity providers in a strategy
     *         that all have a weight of <code>0</code>, any <code>RunTask</code> or <code>CreateService</code> actions
     *         using the capacity provider strategy will fail.
     *         </p>
     *         <p>
     *         An example scenario for using weights is defining a strategy that contains two capacity providers and
     *         both have a weight of <code>1</code>, then when the <code>base</code> is satisfied, the tasks will be
     *         split evenly across the two capacity providers. Using that same logic, if you specify a weight of
     *         <code>1</code> for <i>capacityProviderA</i> and a weight of <code>4</code> for <i>capacityProviderB</i>,
     *         then for every one task that's run using <i>capacityProviderA</i>, four tasks would use
     *         <i>capacityProviderB</i>.
     */
    public final Integer weight() {
        return weight;
    }

    /**
     * <p>
     * The <i>base</i> value designates how many tasks, at a minimum, to run on the specified capacity provider. Only
     * one capacity provider in a capacity provider strategy can have a <i>base</i> defined. If no value is specified,
     * the default value of <code>0</code> is used.
     * </p>
     * 
     * @return The <i>base</i> value designates how many tasks, at a minimum, to run on the specified capacity provider.
     *         Only one capacity provider in a capacity provider strategy can have a <i>base</i> defined. If no value is
     *         specified, the default value of <code>0</code> is used.
     */
    public final Integer base() {
        return base;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(capacityProvider());
        hashCode = 31 * hashCode + Objects.hashCode(weight());
        hashCode = 31 * hashCode + Objects.hashCode(base());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CapacityProviderStrategyItem)) {
            return false;
        }
        CapacityProviderStrategyItem other = (CapacityProviderStrategyItem) obj;
        return Objects.equals(capacityProvider(), other.capacityProvider()) && Objects.equals(weight(), other.weight())
                && Objects.equals(base(), other.base());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CapacityProviderStrategyItem").add("CapacityProvider", capacityProvider())
                .add("Weight", weight()).add("Base", base()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "capacityProvider":
            return Optional.ofNullable(clazz.cast(capacityProvider()));
        case "weight":
            return Optional.ofNullable(clazz.cast(weight()));
        case "base":
            return Optional.ofNullable(clazz.cast(base()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<CapacityProviderStrategyItem, T> g) {
        return obj -> g.apply((CapacityProviderStrategyItem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CapacityProviderStrategyItem> {
        /**
         * <p>
         * The short name of the capacity provider.
         * </p>
         * 
         * @param capacityProvider
         *        The short name of the capacity provider.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityProvider(String capacityProvider);

        /**
         * <p>
         * The <i>weight</i> value designates the relative percentage of the total number of tasks launched that should
         * use the specified capacity provider. The <code>weight</code> value is taken into consideration after the
         * <code>base</code> value, if defined, is satisfied.
         * </p>
         * <p>
         * If no <code>weight</code> value is specified, the default value of <code>0</code> is used. When multiple
         * capacity providers are specified within a capacity provider strategy, at least one of the capacity providers
         * must have a weight value greater than zero and any capacity providers with a weight of <code>0</code> can't
         * be used to place tasks. If you specify multiple capacity providers in a strategy that all have a weight of
         * <code>0</code>, any <code>RunTask</code> or <code>CreateService</code> actions using the capacity provider
         * strategy will fail.
         * </p>
         * <p>
         * An example scenario for using weights is defining a strategy that contains two capacity providers and both
         * have a weight of <code>1</code>, then when the <code>base</code> is satisfied, the tasks will be split evenly
         * across the two capacity providers. Using that same logic, if you specify a weight of <code>1</code> for
         * <i>capacityProviderA</i> and a weight of <code>4</code> for <i>capacityProviderB</i>, then for every one task
         * that's run using <i>capacityProviderA</i>, four tasks would use <i>capacityProviderB</i>.
         * </p>
         * 
         * @param weight
         *        The <i>weight</i> value designates the relative percentage of the total number of tasks launched that
         *        should use the specified capacity provider. The <code>weight</code> value is taken into consideration
         *        after the <code>base</code> value, if defined, is satisfied.</p>
         *        <p>
         *        If no <code>weight</code> value is specified, the default value of <code>0</code> is used. When
         *        multiple capacity providers are specified within a capacity provider strategy, at least one of the
         *        capacity providers must have a weight value greater than zero and any capacity providers with a weight
         *        of <code>0</code> can't be used to place tasks. If you specify multiple capacity providers in a
         *        strategy that all have a weight of <code>0</code>, any <code>RunTask</code> or
         *        <code>CreateService</code> actions using the capacity provider strategy will fail.
         *        </p>
         *        <p>
         *        An example scenario for using weights is defining a strategy that contains two capacity providers and
         *        both have a weight of <code>1</code>, then when the <code>base</code> is satisfied, the tasks will be
         *        split evenly across the two capacity providers. Using that same logic, if you specify a weight of
         *        <code>1</code> for <i>capacityProviderA</i> and a weight of <code>4</code> for
         *        <i>capacityProviderB</i>, then for every one task that's run using <i>capacityProviderA</i>, four
         *        tasks would use <i>capacityProviderB</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder weight(Integer weight);

        /**
         * <p>
         * The <i>base</i> value designates how many tasks, at a minimum, to run on the specified capacity provider.
         * Only one capacity provider in a capacity provider strategy can have a <i>base</i> defined. If no value is
         * specified, the default value of <code>0</code> is used.
         * </p>
         * 
         * @param base
         *        The <i>base</i> value designates how many tasks, at a minimum, to run on the specified capacity
         *        provider. Only one capacity provider in a capacity provider strategy can have a <i>base</i> defined.
         *        If no value is specified, the default value of <code>0</code> is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder base(Integer base);
    }

    static final class BuilderImpl implements Builder {
        private String capacityProvider;

        private Integer weight;

        private Integer base;

        private BuilderImpl() {
        }

        private BuilderImpl(CapacityProviderStrategyItem model) {
            capacityProvider(model.capacityProvider);
            weight(model.weight);
            base(model.base);
        }

        public final String getCapacityProvider() {
            return capacityProvider;
        }

        public final void setCapacityProvider(String capacityProvider) {
            this.capacityProvider = capacityProvider;
        }

        @Override
        public final Builder capacityProvider(String capacityProvider) {
            this.capacityProvider = capacityProvider;
            return this;
        }

        public final Integer getWeight() {
            return weight;
        }

        public final void setWeight(Integer weight) {
            this.weight = weight;
        }

        @Override
        public final Builder weight(Integer weight) {
            this.weight = weight;
            return this;
        }

        public final Integer getBase() {
            return base;
        }

        public final void setBase(Integer base) {
            this.base = base;
        }

        @Override
        public final Builder base(Integer base) {
            this.base = base;
            return this;
        }

        @Override
        public CapacityProviderStrategyItem build() {
            return new CapacityProviderStrategyItem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
