/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * One of the methods which provide a way for you to quickly identify when a deployment has failed, and then to
 * optionally roll back the failure to the last working deployment.
 * </p>
 * <p>
 * When the alarms are generated, Amazon ECS sets the service deployment to failed. Set the rollback parameter to have
 * Amazon ECS to roll back your service to the last completed deployment after a failure.
 * </p>
 * <p>
 * You can only use the <code>DeploymentAlarms</code> method to detect failures when the
 * <code>DeploymentController</code> is set to <code>ECS</code> (rolling update).
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html">Rolling update</a> in the
 * <i> <i>Amazon Elastic Container Service Developer Guide</i> </i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DeploymentAlarms implements SdkPojo, Serializable,
        ToCopyableBuilder<DeploymentAlarms.Builder, DeploymentAlarms> {
    private static final SdkField<List<String>> ALARM_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("alarmNames")
            .getter(getter(DeploymentAlarms::alarmNames))
            .setter(setter(Builder::alarmNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("alarmNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> ROLLBACK_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("rollback").getter(getter(DeploymentAlarms::rollback)).setter(setter(Builder::rollback))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("rollback").build()).build();

    private static final SdkField<Boolean> ENABLE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("enable").getter(getter(DeploymentAlarms::enable)).setter(setter(Builder::enable))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("enable").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ALARM_NAMES_FIELD,
            ROLLBACK_FIELD, ENABLE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("alarmNames", ALARM_NAMES_FIELD);
                    put("rollback", ROLLBACK_FIELD);
                    put("enable", ENABLE_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final List<String> alarmNames;

    private final Boolean rollback;

    private final Boolean enable;

    private DeploymentAlarms(BuilderImpl builder) {
        this.alarmNames = builder.alarmNames;
        this.rollback = builder.rollback;
        this.enable = builder.enable;
    }

    /**
     * For responses, this returns true if the service returned a value for the AlarmNames property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAlarmNames() {
        return alarmNames != null && !(alarmNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more CloudWatch alarm names. Use a "," to separate the alarms.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAlarmNames} method.
     * </p>
     * 
     * @return One or more CloudWatch alarm names. Use a "," to separate the alarms.
     */
    public final List<String> alarmNames() {
        return alarmNames;
    }

    /**
     * <p>
     * Determines whether to configure Amazon ECS to roll back the service if a service deployment fails. If rollback is
     * used, when a service deployment fails, the service is rolled back to the last deployment that completed
     * successfully.
     * </p>
     * 
     * @return Determines whether to configure Amazon ECS to roll back the service if a service deployment fails. If
     *         rollback is used, when a service deployment fails, the service is rolled back to the last deployment that
     *         completed successfully.
     */
    public final Boolean rollback() {
        return rollback;
    }

    /**
     * <p>
     * Determines whether to use the CloudWatch alarm option in the service deployment process.
     * </p>
     * 
     * @return Determines whether to use the CloudWatch alarm option in the service deployment process.
     */
    public final Boolean enable() {
        return enable;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasAlarmNames() ? alarmNames() : null);
        hashCode = 31 * hashCode + Objects.hashCode(rollback());
        hashCode = 31 * hashCode + Objects.hashCode(enable());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DeploymentAlarms)) {
            return false;
        }
        DeploymentAlarms other = (DeploymentAlarms) obj;
        return hasAlarmNames() == other.hasAlarmNames() && Objects.equals(alarmNames(), other.alarmNames())
                && Objects.equals(rollback(), other.rollback()) && Objects.equals(enable(), other.enable());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DeploymentAlarms").add("AlarmNames", hasAlarmNames() ? alarmNames() : null)
                .add("Rollback", rollback()).add("Enable", enable()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "alarmNames":
            return Optional.ofNullable(clazz.cast(alarmNames()));
        case "rollback":
            return Optional.ofNullable(clazz.cast(rollback()));
        case "enable":
            return Optional.ofNullable(clazz.cast(enable()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<DeploymentAlarms, T> g) {
        return obj -> g.apply((DeploymentAlarms) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DeploymentAlarms> {
        /**
         * <p>
         * One or more CloudWatch alarm names. Use a "," to separate the alarms.
         * </p>
         * 
         * @param alarmNames
         *        One or more CloudWatch alarm names. Use a "," to separate the alarms.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarmNames(Collection<String> alarmNames);

        /**
         * <p>
         * One or more CloudWatch alarm names. Use a "," to separate the alarms.
         * </p>
         * 
         * @param alarmNames
         *        One or more CloudWatch alarm names. Use a "," to separate the alarms.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarmNames(String... alarmNames);

        /**
         * <p>
         * Determines whether to configure Amazon ECS to roll back the service if a service deployment fails. If
         * rollback is used, when a service deployment fails, the service is rolled back to the last deployment that
         * completed successfully.
         * </p>
         * 
         * @param rollback
         *        Determines whether to configure Amazon ECS to roll back the service if a service deployment fails. If
         *        rollback is used, when a service deployment fails, the service is rolled back to the last deployment
         *        that completed successfully.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rollback(Boolean rollback);

        /**
         * <p>
         * Determines whether to use the CloudWatch alarm option in the service deployment process.
         * </p>
         * 
         * @param enable
         *        Determines whether to use the CloudWatch alarm option in the service deployment process.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enable(Boolean enable);
    }

    static final class BuilderImpl implements Builder {
        private List<String> alarmNames = DefaultSdkAutoConstructList.getInstance();

        private Boolean rollback;

        private Boolean enable;

        private BuilderImpl() {
        }

        private BuilderImpl(DeploymentAlarms model) {
            alarmNames(model.alarmNames);
            rollback(model.rollback);
            enable(model.enable);
        }

        public final Collection<String> getAlarmNames() {
            if (alarmNames instanceof SdkAutoConstructList) {
                return null;
            }
            return alarmNames;
        }

        public final void setAlarmNames(Collection<String> alarmNames) {
            this.alarmNames = StringListCopier.copy(alarmNames);
        }

        @Override
        public final Builder alarmNames(Collection<String> alarmNames) {
            this.alarmNames = StringListCopier.copy(alarmNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder alarmNames(String... alarmNames) {
            alarmNames(Arrays.asList(alarmNames));
            return this;
        }

        public final Boolean getRollback() {
            return rollback;
        }

        public final void setRollback(Boolean rollback) {
            this.rollback = rollback;
        }

        @Override
        public final Builder rollback(Boolean rollback) {
            this.rollback = rollback;
            return this;
        }

        public final Boolean getEnable() {
            return enable;
        }

        public final void setEnable(Boolean enable) {
            this.enable = enable;
        }

        @Override
        public final Builder enable(Boolean enable) {
            this.enable = enable;
            return this;
        }

        @Override
        public DeploymentAlarms build() {
            return new DeploymentAlarms(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
