/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object representing a constraint on task placement. For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html">Task placement
 * constraints</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
 * </p>
 * <note>
 * <p>
 * If you're using the Fargate launch type, task placement constraints aren't supported.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PlacementConstraint implements SdkPojo, Serializable,
        ToCopyableBuilder<PlacementConstraint.Builder, PlacementConstraint> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(PlacementConstraint::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> EXPRESSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("expression").getter(getter(PlacementConstraint::expression)).setter(setter(Builder::expression))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("expression").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD, EXPRESSION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("type", TYPE_FIELD);
                    put("expression", EXPRESSION_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String type;

    private final String expression;

    private PlacementConstraint(BuilderImpl builder) {
        this.type = builder.type;
        this.expression = builder.expression;
    }

    /**
     * <p>
     * The type of constraint. Use <code>distinctInstance</code> to ensure that each task in a particular group is
     * running on a different container instance. Use <code>memberOf</code> to restrict the selection to a group of
     * valid candidates.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link PlacementConstraintType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of constraint. Use <code>distinctInstance</code> to ensure that each task in a particular group
     *         is running on a different container instance. Use <code>memberOf</code> to restrict the selection to a
     *         group of valid candidates.
     * @see PlacementConstraintType
     */
    public final PlacementConstraintType type() {
        return PlacementConstraintType.fromValue(type);
    }

    /**
     * <p>
     * The type of constraint. Use <code>distinctInstance</code> to ensure that each task in a particular group is
     * running on a different container instance. Use <code>memberOf</code> to restrict the selection to a group of
     * valid candidates.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link PlacementConstraintType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of constraint. Use <code>distinctInstance</code> to ensure that each task in a particular group
     *         is running on a different container instance. Use <code>memberOf</code> to restrict the selection to a
     *         group of valid candidates.
     * @see PlacementConstraintType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * A cluster query language expression to apply to the constraint. The expression can have a maximum length of 2000
     * characters. You can't specify an expression if the constraint type is <code>distinctInstance</code>. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-query-language.html">Cluster query
     * language</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * 
     * @return A cluster query language expression to apply to the constraint. The expression can have a maximum length
     *         of 2000 characters. You can't specify an expression if the constraint type is
     *         <code>distinctInstance</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-query-language.html">Cluster
     *         query language</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     */
    public final String expression() {
        return expression;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(expression());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PlacementConstraint)) {
            return false;
        }
        PlacementConstraint other = (PlacementConstraint) obj;
        return Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(expression(), other.expression());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PlacementConstraint").add("Type", typeAsString()).add("Expression", expression()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "expression":
            return Optional.ofNullable(clazz.cast(expression()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<PlacementConstraint, T> g) {
        return obj -> g.apply((PlacementConstraint) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PlacementConstraint> {
        /**
         * <p>
         * The type of constraint. Use <code>distinctInstance</code> to ensure that each task in a particular group is
         * running on a different container instance. Use <code>memberOf</code> to restrict the selection to a group of
         * valid candidates.
         * </p>
         * 
         * @param type
         *        The type of constraint. Use <code>distinctInstance</code> to ensure that each task in a particular
         *        group is running on a different container instance. Use <code>memberOf</code> to restrict the
         *        selection to a group of valid candidates.
         * @see PlacementConstraintType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PlacementConstraintType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of constraint. Use <code>distinctInstance</code> to ensure that each task in a particular group is
         * running on a different container instance. Use <code>memberOf</code> to restrict the selection to a group of
         * valid candidates.
         * </p>
         * 
         * @param type
         *        The type of constraint. Use <code>distinctInstance</code> to ensure that each task in a particular
         *        group is running on a different container instance. Use <code>memberOf</code> to restrict the
         *        selection to a group of valid candidates.
         * @see PlacementConstraintType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PlacementConstraintType
         */
        Builder type(PlacementConstraintType type);

        /**
         * <p>
         * A cluster query language expression to apply to the constraint. The expression can have a maximum length of
         * 2000 characters. You can't specify an expression if the constraint type is <code>distinctInstance</code>. For
         * more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-query-language.html">Cluster query
         * language</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param expression
         *        A cluster query language expression to apply to the constraint. The expression can have a maximum
         *        length of 2000 characters. You can't specify an expression if the constraint type is
         *        <code>distinctInstance</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-query-language.html">Cluster
         *        query language</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expression(String expression);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private String expression;

        private BuilderImpl() {
        }

        private BuilderImpl(PlacementConstraint model) {
            type(model.type);
            expression(model.expression);
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(PlacementConstraintType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getExpression() {
            return expression;
        }

        public final void setExpression(String expression) {
            this.expression = expression;
        }

        @Override
        public final Builder expression(String expression) {
            this.expression = expression;
            return this;
        }

        @Override
        public PlacementConstraint build() {
            return new PlacementConstraint(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
