/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * You can enable a restart policy for each container defined in your task definition, to overcome transient failures
 * faster and maintain task availability. When you enable a restart policy for a container, Amazon ECS can restart the
 * container if it exits, without needing to replace the task. For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-restart-policy.html">Restart individual
 * containers in Amazon ECS tasks with container restart policies</a> in the <i>Amazon Elastic Container Service
 * Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ContainerRestartPolicy implements SdkPojo, Serializable,
        ToCopyableBuilder<ContainerRestartPolicy.Builder, ContainerRestartPolicy> {
    private static final SdkField<Boolean> ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("enabled").getter(getter(ContainerRestartPolicy::enabled)).setter(setter(Builder::enabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("enabled").build()).build();

    private static final SdkField<List<Integer>> IGNORED_EXIT_CODES_FIELD = SdkField
            .<List<Integer>> builder(MarshallingType.LIST)
            .memberName("ignoredExitCodes")
            .getter(getter(ContainerRestartPolicy::ignoredExitCodes))
            .setter(setter(Builder::ignoredExitCodes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ignoredExitCodes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Integer> builder(MarshallingType.INTEGER)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> RESTART_ATTEMPT_PERIOD_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("restartAttemptPeriod").getter(getter(ContainerRestartPolicy::restartAttemptPeriod))
            .setter(setter(Builder::restartAttemptPeriod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("restartAttemptPeriod").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENABLED_FIELD,
            IGNORED_EXIT_CODES_FIELD, RESTART_ATTEMPT_PERIOD_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("enabled", ENABLED_FIELD);
                    put("ignoredExitCodes", IGNORED_EXIT_CODES_FIELD);
                    put("restartAttemptPeriod", RESTART_ATTEMPT_PERIOD_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final Boolean enabled;

    private final List<Integer> ignoredExitCodes;

    private final Integer restartAttemptPeriod;

    private ContainerRestartPolicy(BuilderImpl builder) {
        this.enabled = builder.enabled;
        this.ignoredExitCodes = builder.ignoredExitCodes;
        this.restartAttemptPeriod = builder.restartAttemptPeriod;
    }

    /**
     * <p>
     * Specifies whether a restart policy is enabled for the container.
     * </p>
     * 
     * @return Specifies whether a restart policy is enabled for the container.
     */
    public final Boolean enabled() {
        return enabled;
    }

    /**
     * For responses, this returns true if the service returned a value for the IgnoredExitCodes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasIgnoredExitCodes() {
        return ignoredExitCodes != null && !(ignoredExitCodes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of exit codes that Amazon ECS will ignore and not attempt a restart on. You can specify a maximum of 50
     * container exit codes. By default, Amazon ECS does not ignore any exit codes.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIgnoredExitCodes} method.
     * </p>
     * 
     * @return A list of exit codes that Amazon ECS will ignore and not attempt a restart on. You can specify a maximum
     *         of 50 container exit codes. By default, Amazon ECS does not ignore any exit codes.
     */
    public final List<Integer> ignoredExitCodes() {
        return ignoredExitCodes;
    }

    /**
     * <p>
     * A period of time (in seconds) that the container must run for before a restart can be attempted. A container can
     * be restarted only once every <code>restartAttemptPeriod</code> seconds. If a container isn't able to run for this
     * time period and exits early, it will not be restarted. You can set a minimum <code>restartAttemptPeriod</code> of
     * 60 seconds and a maximum <code>restartAttemptPeriod</code> of 1800 seconds. By default, a container must run for
     * 300 seconds before it can be restarted.
     * </p>
     * 
     * @return A period of time (in seconds) that the container must run for before a restart can be attempted. A
     *         container can be restarted only once every <code>restartAttemptPeriod</code> seconds. If a container
     *         isn't able to run for this time period and exits early, it will not be restarted. You can set a minimum
     *         <code>restartAttemptPeriod</code> of 60 seconds and a maximum <code>restartAttemptPeriod</code> of 1800
     *         seconds. By default, a container must run for 300 seconds before it can be restarted.
     */
    public final Integer restartAttemptPeriod() {
        return restartAttemptPeriod;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(enabled());
        hashCode = 31 * hashCode + Objects.hashCode(hasIgnoredExitCodes() ? ignoredExitCodes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(restartAttemptPeriod());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ContainerRestartPolicy)) {
            return false;
        }
        ContainerRestartPolicy other = (ContainerRestartPolicy) obj;
        return Objects.equals(enabled(), other.enabled()) && hasIgnoredExitCodes() == other.hasIgnoredExitCodes()
                && Objects.equals(ignoredExitCodes(), other.ignoredExitCodes())
                && Objects.equals(restartAttemptPeriod(), other.restartAttemptPeriod());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ContainerRestartPolicy").add("Enabled", enabled())
                .add("IgnoredExitCodes", hasIgnoredExitCodes() ? ignoredExitCodes() : null)
                .add("RestartAttemptPeriod", restartAttemptPeriod()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "enabled":
            return Optional.ofNullable(clazz.cast(enabled()));
        case "ignoredExitCodes":
            return Optional.ofNullable(clazz.cast(ignoredExitCodes()));
        case "restartAttemptPeriod":
            return Optional.ofNullable(clazz.cast(restartAttemptPeriod()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<ContainerRestartPolicy, T> g) {
        return obj -> g.apply((ContainerRestartPolicy) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ContainerRestartPolicy> {
        /**
         * <p>
         * Specifies whether a restart policy is enabled for the container.
         * </p>
         * 
         * @param enabled
         *        Specifies whether a restart policy is enabled for the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabled(Boolean enabled);

        /**
         * <p>
         * A list of exit codes that Amazon ECS will ignore and not attempt a restart on. You can specify a maximum of
         * 50 container exit codes. By default, Amazon ECS does not ignore any exit codes.
         * </p>
         * 
         * @param ignoredExitCodes
         *        A list of exit codes that Amazon ECS will ignore and not attempt a restart on. You can specify a
         *        maximum of 50 container exit codes. By default, Amazon ECS does not ignore any exit codes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ignoredExitCodes(Collection<Integer> ignoredExitCodes);

        /**
         * <p>
         * A list of exit codes that Amazon ECS will ignore and not attempt a restart on. You can specify a maximum of
         * 50 container exit codes. By default, Amazon ECS does not ignore any exit codes.
         * </p>
         * 
         * @param ignoredExitCodes
         *        A list of exit codes that Amazon ECS will ignore and not attempt a restart on. You can specify a
         *        maximum of 50 container exit codes. By default, Amazon ECS does not ignore any exit codes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ignoredExitCodes(Integer... ignoredExitCodes);

        /**
         * <p>
         * A period of time (in seconds) that the container must run for before a restart can be attempted. A container
         * can be restarted only once every <code>restartAttemptPeriod</code> seconds. If a container isn't able to run
         * for this time period and exits early, it will not be restarted. You can set a minimum
         * <code>restartAttemptPeriod</code> of 60 seconds and a maximum <code>restartAttemptPeriod</code> of 1800
         * seconds. By default, a container must run for 300 seconds before it can be restarted.
         * </p>
         * 
         * @param restartAttemptPeriod
         *        A period of time (in seconds) that the container must run for before a restart can be attempted. A
         *        container can be restarted only once every <code>restartAttemptPeriod</code> seconds. If a container
         *        isn't able to run for this time period and exits early, it will not be restarted. You can set a
         *        minimum <code>restartAttemptPeriod</code> of 60 seconds and a maximum
         *        <code>restartAttemptPeriod</code> of 1800 seconds. By default, a container must run for 300 seconds
         *        before it can be restarted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder restartAttemptPeriod(Integer restartAttemptPeriod);
    }

    static final class BuilderImpl implements Builder {
        private Boolean enabled;

        private List<Integer> ignoredExitCodes = DefaultSdkAutoConstructList.getInstance();

        private Integer restartAttemptPeriod;

        private BuilderImpl() {
        }

        private BuilderImpl(ContainerRestartPolicy model) {
            enabled(model.enabled);
            ignoredExitCodes(model.ignoredExitCodes);
            restartAttemptPeriod(model.restartAttemptPeriod);
        }

        public final Boolean getEnabled() {
            return enabled;
        }

        public final void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        @Override
        public final Builder enabled(Boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        public final Collection<Integer> getIgnoredExitCodes() {
            if (ignoredExitCodes instanceof SdkAutoConstructList) {
                return null;
            }
            return ignoredExitCodes;
        }

        public final void setIgnoredExitCodes(Collection<Integer> ignoredExitCodes) {
            this.ignoredExitCodes = IntegerListCopier.copy(ignoredExitCodes);
        }

        @Override
        public final Builder ignoredExitCodes(Collection<Integer> ignoredExitCodes) {
            this.ignoredExitCodes = IntegerListCopier.copy(ignoredExitCodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ignoredExitCodes(Integer... ignoredExitCodes) {
            ignoredExitCodes(Arrays.asList(ignoredExitCodes));
            return this;
        }

        public final Integer getRestartAttemptPeriod() {
            return restartAttemptPeriod;
        }

        public final void setRestartAttemptPeriod(Integer restartAttemptPeriod) {
            this.restartAttemptPeriod = restartAttemptPeriod;
        }

        @Override
        public final Builder restartAttemptPeriod(Integer restartAttemptPeriod) {
            this.restartAttemptPeriod = restartAttemptPeriod;
            return this;
        }

        @Override
        public ContainerRestartPolicy build() {
            return new ContainerRestartPolicy(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
