/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class ListServiceDeploymentsRequest extends EcsRequest implements
        ToCopyableBuilder<ListServiceDeploymentsRequest.Builder, ListServiceDeploymentsRequest> {
    private static final SdkField<String> SERVICE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("service")
            .getter(getter(ListServiceDeploymentsRequest::service)).setter(setter(Builder::service))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("service").build()).build();

    private static final SdkField<String> CLUSTER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("cluster")
            .getter(getter(ListServiceDeploymentsRequest::cluster)).setter(setter(Builder::cluster))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cluster").build()).build();

    private static final SdkField<List<String>> STATUS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("status")
            .getter(getter(ListServiceDeploymentsRequest::statusAsStrings))
            .setter(setter(Builder::statusWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<CreatedAt> CREATED_AT_FIELD = SdkField.<CreatedAt> builder(MarshallingType.SDK_POJO)
            .memberName("createdAt").getter(getter(ListServiceDeploymentsRequest::createdAt)).setter(setter(Builder::createdAt))
            .constructor(CreatedAt::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<String> NEXT_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("nextToken").getter(getter(ListServiceDeploymentsRequest::nextToken)).setter(setter(Builder::nextToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("nextToken").build()).build();

    private static final SdkField<Integer> MAX_RESULTS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("maxResults").getter(getter(ListServiceDeploymentsRequest::maxResults))
            .setter(setter(Builder::maxResults))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maxResults").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SERVICE_FIELD, CLUSTER_FIELD,
            STATUS_FIELD, CREATED_AT_FIELD, NEXT_TOKEN_FIELD, MAX_RESULTS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String service;

    private final String cluster;

    private final List<String> status;

    private final CreatedAt createdAt;

    private final String nextToken;

    private final Integer maxResults;

    private ListServiceDeploymentsRequest(BuilderImpl builder) {
        super(builder);
        this.service = builder.service;
        this.cluster = builder.cluster;
        this.status = builder.status;
        this.createdAt = builder.createdAt;
        this.nextToken = builder.nextToken;
        this.maxResults = builder.maxResults;
    }

    /**
     * <p>
     * The ARN or name of the service
     * </p>
     * 
     * @return The ARN or name of the service
     */
    public final String service() {
        return service;
    }

    /**
     * <p>
     * The cluster that hosts the service. This can either be the cluster name or ARN. Starting April 15, 2023, Amazon
     * Web Services will not onboard new customers to Amazon Elastic Inference (EI), and will help current customers
     * migrate their workloads to options that offer better price and performanceIf you don't specify a cluster,
     * <code>default</code> is used.
     * </p>
     * 
     * @return The cluster that hosts the service. This can either be the cluster name or ARN. Starting April 15, 2023,
     *         Amazon Web Services will not onboard new customers to Amazon Elastic Inference (EI), and will help
     *         current customers migrate their workloads to options that offer better price and performanceIf you don't
     *         specify a cluster, <code>default</code> is used.
     */
    public final String cluster() {
        return cluster;
    }

    /**
     * <p>
     * An optional filter you can use to narrow the results. If you do not specify a status, then all status values are
     * included in the result.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStatus} method.
     * </p>
     * 
     * @return An optional filter you can use to narrow the results. If you do not specify a status, then all status
     *         values are included in the result.
     */
    public final List<ServiceDeploymentStatus> status() {
        return ServiceDeploymentStatusListCopier.copyStringToEnum(status);
    }

    /**
     * For responses, this returns true if the service returned a value for the Status property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasStatus() {
        return status != null && !(status instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An optional filter you can use to narrow the results. If you do not specify a status, then all status values are
     * included in the result.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStatus} method.
     * </p>
     * 
     * @return An optional filter you can use to narrow the results. If you do not specify a status, then all status
     *         values are included in the result.
     */
    public final List<String> statusAsStrings() {
        return status;
    }

    /**
     * <p>
     * An optional filter you can use to narrow the results by the service creation date. If you do not specify a value,
     * the result includes all services created before the current time. The format is yyyy-MM-dd HH:mm:ss.SSSSSS.
     * </p>
     * 
     * @return An optional filter you can use to narrow the results by the service creation date. If you do not specify
     *         a value, the result includes all services created before the current time. The format is yyyy-MM-dd
     *         HH:mm:ss.SSSSSS.
     */
    public final CreatedAt createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The <code>nextToken</code> value returned from a <code>ListServiceDeployments</code> request indicating that more
     * results are available to fulfill the request and further calls are needed. If you provided
     * <code>maxResults</code>, it's possible the number of results is fewer than <code>maxResults</code>.
     * </p>
     * 
     * @return The <code>nextToken</code> value returned from a <code>ListServiceDeployments</code> request indicating
     *         that more results are available to fulfill the request and further calls are needed. If you provided
     *         <code>maxResults</code>, it's possible the number of results is fewer than <code>maxResults</code>.
     */
    public final String nextToken() {
        return nextToken;
    }

    /**
     * <p>
     * The maximum number of service deployment results that <code>ListServiceDeployments</code> returned in paginated
     * output. When this parameter is used, <code>ListServiceDeployments</code> only returns <code>maxResults</code>
     * results in a single page along with a <code>nextToken</code> response element. The remaining results of the
     * initial request can be seen by sending another <code>ListServiceDeployments</code> request with the returned
     * <code>nextToken</code> value. This value can be between 1 and 100. If this parameter isn't used, then
     * <code>ListServiceDeployments</code> returns up to 20 results and a <code>nextToken</code> value if applicable.
     * </p>
     * 
     * @return The maximum number of service deployment results that <code>ListServiceDeployments</code> returned in
     *         paginated output. When this parameter is used, <code>ListServiceDeployments</code> only returns
     *         <code>maxResults</code> results in a single page along with a <code>nextToken</code> response element.
     *         The remaining results of the initial request can be seen by sending another
     *         <code>ListServiceDeployments</code> request with the returned <code>nextToken</code> value. This value
     *         can be between 1 and 100. If this parameter isn't used, then <code>ListServiceDeployments</code> returns
     *         up to 20 results and a <code>nextToken</code> value if applicable.
     */
    public final Integer maxResults() {
        return maxResults;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(service());
        hashCode = 31 * hashCode + Objects.hashCode(cluster());
        hashCode = 31 * hashCode + Objects.hashCode(hasStatus() ? statusAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(nextToken());
        hashCode = 31 * hashCode + Objects.hashCode(maxResults());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ListServiceDeploymentsRequest)) {
            return false;
        }
        ListServiceDeploymentsRequest other = (ListServiceDeploymentsRequest) obj;
        return Objects.equals(service(), other.service()) && Objects.equals(cluster(), other.cluster())
                && hasStatus() == other.hasStatus() && Objects.equals(statusAsStrings(), other.statusAsStrings())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(nextToken(), other.nextToken())
                && Objects.equals(maxResults(), other.maxResults());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ListServiceDeploymentsRequest").add("Service", service()).add("Cluster", cluster())
                .add("Status", hasStatus() ? statusAsStrings() : null).add("CreatedAt", createdAt())
                .add("NextToken", nextToken()).add("MaxResults", maxResults()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "service":
            return Optional.ofNullable(clazz.cast(service()));
        case "cluster":
            return Optional.ofNullable(clazz.cast(cluster()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsStrings()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "nextToken":
            return Optional.ofNullable(clazz.cast(nextToken()));
        case "maxResults":
            return Optional.ofNullable(clazz.cast(maxResults()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("service", SERVICE_FIELD);
        map.put("cluster", CLUSTER_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("nextToken", NEXT_TOKEN_FIELD);
        map.put("maxResults", MAX_RESULTS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ListServiceDeploymentsRequest, T> g) {
        return obj -> g.apply((ListServiceDeploymentsRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends EcsRequest.Builder, SdkPojo, CopyableBuilder<Builder, ListServiceDeploymentsRequest> {
        /**
         * <p>
         * The ARN or name of the service
         * </p>
         * 
         * @param service
         *        The ARN or name of the service
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder service(String service);

        /**
         * <p>
         * The cluster that hosts the service. This can either be the cluster name or ARN. Starting April 15, 2023,
         * Amazon Web Services will not onboard new customers to Amazon Elastic Inference (EI), and will help current
         * customers migrate their workloads to options that offer better price and performanceIf you don't specify a
         * cluster, <code>default</code> is used.
         * </p>
         * 
         * @param cluster
         *        The cluster that hosts the service. This can either be the cluster name or ARN. Starting April 15,
         *        2023, Amazon Web Services will not onboard new customers to Amazon Elastic Inference (EI), and will
         *        help current customers migrate their workloads to options that offer better price and performanceIf
         *        you don't specify a cluster, <code>default</code> is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cluster(String cluster);

        /**
         * <p>
         * An optional filter you can use to narrow the results. If you do not specify a status, then all status values
         * are included in the result.
         * </p>
         * 
         * @param status
         *        An optional filter you can use to narrow the results. If you do not specify a status, then all status
         *        values are included in the result.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusWithStrings(Collection<String> status);

        /**
         * <p>
         * An optional filter you can use to narrow the results. If you do not specify a status, then all status values
         * are included in the result.
         * </p>
         * 
         * @param status
         *        An optional filter you can use to narrow the results. If you do not specify a status, then all status
         *        values are included in the result.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusWithStrings(String... status);

        /**
         * <p>
         * An optional filter you can use to narrow the results. If you do not specify a status, then all status values
         * are included in the result.
         * </p>
         * 
         * @param status
         *        An optional filter you can use to narrow the results. If you do not specify a status, then all status
         *        values are included in the result.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(Collection<ServiceDeploymentStatus> status);

        /**
         * <p>
         * An optional filter you can use to narrow the results. If you do not specify a status, then all status values
         * are included in the result.
         * </p>
         * 
         * @param status
         *        An optional filter you can use to narrow the results. If you do not specify a status, then all status
         *        values are included in the result.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(ServiceDeploymentStatus... status);

        /**
         * <p>
         * An optional filter you can use to narrow the results by the service creation date. If you do not specify a
         * value, the result includes all services created before the current time. The format is yyyy-MM-dd
         * HH:mm:ss.SSSSSS.
         * </p>
         * 
         * @param createdAt
         *        An optional filter you can use to narrow the results by the service creation date. If you do not
         *        specify a value, the result includes all services created before the current time. The format is
         *        yyyy-MM-dd HH:mm:ss.SSSSSS.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(CreatedAt createdAt);

        /**
         * <p>
         * An optional filter you can use to narrow the results by the service creation date. If you do not specify a
         * value, the result includes all services created before the current time. The format is yyyy-MM-dd
         * HH:mm:ss.SSSSSS.
         * </p>
         * This is a convenience method that creates an instance of the {@link CreatedAt.Builder} avoiding the need to
         * create one manually via {@link CreatedAt#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CreatedAt.Builder#build()} is called immediately and its result
         * is passed to {@link #createdAt(CreatedAt)}.
         * 
         * @param createdAt
         *        a consumer that will call methods on {@link CreatedAt.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #createdAt(CreatedAt)
         */
        default Builder createdAt(Consumer<CreatedAt.Builder> createdAt) {
            return createdAt(CreatedAt.builder().applyMutation(createdAt).build());
        }

        /**
         * <p>
         * The <code>nextToken</code> value returned from a <code>ListServiceDeployments</code> request indicating that
         * more results are available to fulfill the request and further calls are needed. If you provided
         * <code>maxResults</code>, it's possible the number of results is fewer than <code>maxResults</code>.
         * </p>
         * 
         * @param nextToken
         *        The <code>nextToken</code> value returned from a <code>ListServiceDeployments</code> request
         *        indicating that more results are available to fulfill the request and further calls are needed. If you
         *        provided <code>maxResults</code>, it's possible the number of results is fewer than
         *        <code>maxResults</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nextToken(String nextToken);

        /**
         * <p>
         * The maximum number of service deployment results that <code>ListServiceDeployments</code> returned in
         * paginated output. When this parameter is used, <code>ListServiceDeployments</code> only returns
         * <code>maxResults</code> results in a single page along with a <code>nextToken</code> response element. The
         * remaining results of the initial request can be seen by sending another <code>ListServiceDeployments</code>
         * request with the returned <code>nextToken</code> value. This value can be between 1 and 100. If this
         * parameter isn't used, then <code>ListServiceDeployments</code> returns up to 20 results and a
         * <code>nextToken</code> value if applicable.
         * </p>
         * 
         * @param maxResults
         *        The maximum number of service deployment results that <code>ListServiceDeployments</code> returned in
         *        paginated output. When this parameter is used, <code>ListServiceDeployments</code> only returns
         *        <code>maxResults</code> results in a single page along with a <code>nextToken</code> response element.
         *        The remaining results of the initial request can be seen by sending another
         *        <code>ListServiceDeployments</code> request with the returned <code>nextToken</code> value. This value
         *        can be between 1 and 100. If this parameter isn't used, then <code>ListServiceDeployments</code>
         *        returns up to 20 results and a <code>nextToken</code> value if applicable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxResults(Integer maxResults);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends EcsRequest.BuilderImpl implements Builder {
        private String service;

        private String cluster;

        private List<String> status = DefaultSdkAutoConstructList.getInstance();

        private CreatedAt createdAt;

        private String nextToken;

        private Integer maxResults;

        private BuilderImpl() {
        }

        private BuilderImpl(ListServiceDeploymentsRequest model) {
            super(model);
            service(model.service);
            cluster(model.cluster);
            statusWithStrings(model.status);
            createdAt(model.createdAt);
            nextToken(model.nextToken);
            maxResults(model.maxResults);
        }

        public final String getService() {
            return service;
        }

        public final void setService(String service) {
            this.service = service;
        }

        @Override
        public final Builder service(String service) {
            this.service = service;
            return this;
        }

        public final String getCluster() {
            return cluster;
        }

        public final void setCluster(String cluster) {
            this.cluster = cluster;
        }

        @Override
        public final Builder cluster(String cluster) {
            this.cluster = cluster;
            return this;
        }

        public final Collection<String> getStatus() {
            if (status instanceof SdkAutoConstructList) {
                return null;
            }
            return status;
        }

        public final void setStatus(Collection<String> status) {
            this.status = ServiceDeploymentStatusListCopier.copy(status);
        }

        @Override
        public final Builder statusWithStrings(Collection<String> status) {
            this.status = ServiceDeploymentStatusListCopier.copy(status);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder statusWithStrings(String... status) {
            statusWithStrings(Arrays.asList(status));
            return this;
        }

        @Override
        public final Builder status(Collection<ServiceDeploymentStatus> status) {
            this.status = ServiceDeploymentStatusListCopier.copyEnumToString(status);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder status(ServiceDeploymentStatus... status) {
            status(Arrays.asList(status));
            return this;
        }

        public final CreatedAt.Builder getCreatedAt() {
            return createdAt != null ? createdAt.toBuilder() : null;
        }

        public final void setCreatedAt(CreatedAt.BuilderImpl createdAt) {
            this.createdAt = createdAt != null ? createdAt.build() : null;
        }

        @Override
        public final Builder createdAt(CreatedAt createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getNextToken() {
            return nextToken;
        }

        public final void setNextToken(String nextToken) {
            this.nextToken = nextToken;
        }

        @Override
        public final Builder nextToken(String nextToken) {
            this.nextToken = nextToken;
            return this;
        }

        public final Integer getMaxResults() {
            return maxResults;
        }

        public final void setMaxResults(Integer maxResults) {
            this.maxResults = maxResults;
        }

        @Override
        public final Builder maxResults(Integer maxResults) {
            this.maxResults = maxResults;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public ListServiceDeploymentsRequest build() {
            return new ListServiceDeploymentsRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
