/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The details of a task definition which describes the container and volume definitions of an Amazon Elastic Container
 * Service task. You can specify which Docker images to use, the required resources, and other configurations related to
 * launching the task definition through an Amazon ECS service or task.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TaskDefinition implements SdkPojo, Serializable, ToCopyableBuilder<TaskDefinition.Builder, TaskDefinition> {
    private static final SdkField<String> TASK_DEFINITION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("taskDefinitionArn").getter(getter(TaskDefinition::taskDefinitionArn))
            .setter(setter(Builder::taskDefinitionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("taskDefinitionArn").build()).build();

    private static final SdkField<List<ContainerDefinition>> CONTAINER_DEFINITIONS_FIELD = SdkField
            .<List<ContainerDefinition>> builder(MarshallingType.LIST)
            .memberName("containerDefinitions")
            .getter(getter(TaskDefinition::containerDefinitions))
            .setter(setter(Builder::containerDefinitions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerDefinitions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ContainerDefinition> builder(MarshallingType.SDK_POJO)
                                            .constructor(ContainerDefinition::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> FAMILY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("family")
            .getter(getter(TaskDefinition::family)).setter(setter(Builder::family))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("family").build()).build();

    private static final SdkField<String> TASK_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("taskRoleArn").getter(getter(TaskDefinition::taskRoleArn)).setter(setter(Builder::taskRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("taskRoleArn").build()).build();

    private static final SdkField<String> EXECUTION_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("executionRoleArn").getter(getter(TaskDefinition::executionRoleArn))
            .setter(setter(Builder::executionRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("executionRoleArn").build()).build();

    private static final SdkField<String> NETWORK_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("networkMode").getter(getter(TaskDefinition::networkModeAsString)).setter(setter(Builder::networkMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("networkMode").build()).build();

    private static final SdkField<Integer> REVISION_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("revision").getter(getter(TaskDefinition::revision)).setter(setter(Builder::revision))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("revision").build()).build();

    private static final SdkField<List<Volume>> VOLUMES_FIELD = SdkField
            .<List<Volume>> builder(MarshallingType.LIST)
            .memberName("volumes")
            .getter(getter(TaskDefinition::volumes))
            .setter(setter(Builder::volumes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("volumes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Volume> builder(MarshallingType.SDK_POJO)
                                            .constructor(Volume::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(TaskDefinition::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<List<Attribute>> REQUIRES_ATTRIBUTES_FIELD = SdkField
            .<List<Attribute>> builder(MarshallingType.LIST)
            .memberName("requiresAttributes")
            .getter(getter(TaskDefinition::requiresAttributes))
            .setter(setter(Builder::requiresAttributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("requiresAttributes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Attribute> builder(MarshallingType.SDK_POJO)
                                            .constructor(Attribute::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<TaskDefinitionPlacementConstraint>> PLACEMENT_CONSTRAINTS_FIELD = SdkField
            .<List<TaskDefinitionPlacementConstraint>> builder(MarshallingType.LIST)
            .memberName("placementConstraints")
            .getter(getter(TaskDefinition::placementConstraints))
            .setter(setter(Builder::placementConstraints))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("placementConstraints").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<TaskDefinitionPlacementConstraint> builder(MarshallingType.SDK_POJO)
                                            .constructor(TaskDefinitionPlacementConstraint::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> COMPATIBILITIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("compatibilities")
            .getter(getter(TaskDefinition::compatibilitiesAsStrings))
            .setter(setter(Builder::compatibilitiesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("compatibilities").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<RuntimePlatform> RUNTIME_PLATFORM_FIELD = SdkField
            .<RuntimePlatform> builder(MarshallingType.SDK_POJO).memberName("runtimePlatform")
            .getter(getter(TaskDefinition::runtimePlatform)).setter(setter(Builder::runtimePlatform))
            .constructor(RuntimePlatform::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("runtimePlatform").build()).build();

    private static final SdkField<List<String>> REQUIRES_COMPATIBILITIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("requiresCompatibilities")
            .getter(getter(TaskDefinition::requiresCompatibilitiesAsStrings))
            .setter(setter(Builder::requiresCompatibilitiesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("requiresCompatibilities").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CPU_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("cpu")
            .getter(getter(TaskDefinition::cpu)).setter(setter(Builder::cpu))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cpu").build()).build();

    private static final SdkField<String> MEMORY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("memory")
            .getter(getter(TaskDefinition::memory)).setter(setter(Builder::memory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("memory").build()).build();

    private static final SdkField<List<InferenceAccelerator>> INFERENCE_ACCELERATORS_FIELD = SdkField
            .<List<InferenceAccelerator>> builder(MarshallingType.LIST)
            .memberName("inferenceAccelerators")
            .getter(getter(TaskDefinition::inferenceAccelerators))
            .setter(setter(Builder::inferenceAccelerators))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("inferenceAccelerators").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<InferenceAccelerator> builder(MarshallingType.SDK_POJO)
                                            .constructor(InferenceAccelerator::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> PID_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("pidMode").getter(getter(TaskDefinition::pidModeAsString)).setter(setter(Builder::pidMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("pidMode").build()).build();

    private static final SdkField<String> IPC_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ipcMode").getter(getter(TaskDefinition::ipcModeAsString)).setter(setter(Builder::ipcMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ipcMode").build()).build();

    private static final SdkField<ProxyConfiguration> PROXY_CONFIGURATION_FIELD = SdkField
            .<ProxyConfiguration> builder(MarshallingType.SDK_POJO).memberName("proxyConfiguration")
            .getter(getter(TaskDefinition::proxyConfiguration)).setter(setter(Builder::proxyConfiguration))
            .constructor(ProxyConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("proxyConfiguration").build())
            .build();

    private static final SdkField<Instant> REGISTERED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("registeredAt").getter(getter(TaskDefinition::registeredAt)).setter(setter(Builder::registeredAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("registeredAt").build()).build();

    private static final SdkField<Instant> DEREGISTERED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("deregisteredAt").getter(getter(TaskDefinition::deregisteredAt)).setter(setter(Builder::deregisteredAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("deregisteredAt").build()).build();

    private static final SdkField<String> REGISTERED_BY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("registeredBy").getter(getter(TaskDefinition::registeredBy)).setter(setter(Builder::registeredBy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("registeredBy").build()).build();

    private static final SdkField<EphemeralStorage> EPHEMERAL_STORAGE_FIELD = SdkField
            .<EphemeralStorage> builder(MarshallingType.SDK_POJO).memberName("ephemeralStorage")
            .getter(getter(TaskDefinition::ephemeralStorage)).setter(setter(Builder::ephemeralStorage))
            .constructor(EphemeralStorage::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ephemeralStorage").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TASK_DEFINITION_ARN_FIELD,
            CONTAINER_DEFINITIONS_FIELD, FAMILY_FIELD, TASK_ROLE_ARN_FIELD, EXECUTION_ROLE_ARN_FIELD, NETWORK_MODE_FIELD,
            REVISION_FIELD, VOLUMES_FIELD, STATUS_FIELD, REQUIRES_ATTRIBUTES_FIELD, PLACEMENT_CONSTRAINTS_FIELD,
            COMPATIBILITIES_FIELD, RUNTIME_PLATFORM_FIELD, REQUIRES_COMPATIBILITIES_FIELD, CPU_FIELD, MEMORY_FIELD,
            INFERENCE_ACCELERATORS_FIELD, PID_MODE_FIELD, IPC_MODE_FIELD, PROXY_CONFIGURATION_FIELD, REGISTERED_AT_FIELD,
            DEREGISTERED_AT_FIELD, REGISTERED_BY_FIELD, EPHEMERAL_STORAGE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String taskDefinitionArn;

    private final List<ContainerDefinition> containerDefinitions;

    private final String family;

    private final String taskRoleArn;

    private final String executionRoleArn;

    private final String networkMode;

    private final Integer revision;

    private final List<Volume> volumes;

    private final String status;

    private final List<Attribute> requiresAttributes;

    private final List<TaskDefinitionPlacementConstraint> placementConstraints;

    private final List<String> compatibilities;

    private final RuntimePlatform runtimePlatform;

    private final List<String> requiresCompatibilities;

    private final String cpu;

    private final String memory;

    private final List<InferenceAccelerator> inferenceAccelerators;

    private final String pidMode;

    private final String ipcMode;

    private final ProxyConfiguration proxyConfiguration;

    private final Instant registeredAt;

    private final Instant deregisteredAt;

    private final String registeredBy;

    private final EphemeralStorage ephemeralStorage;

    private TaskDefinition(BuilderImpl builder) {
        this.taskDefinitionArn = builder.taskDefinitionArn;
        this.containerDefinitions = builder.containerDefinitions;
        this.family = builder.family;
        this.taskRoleArn = builder.taskRoleArn;
        this.executionRoleArn = builder.executionRoleArn;
        this.networkMode = builder.networkMode;
        this.revision = builder.revision;
        this.volumes = builder.volumes;
        this.status = builder.status;
        this.requiresAttributes = builder.requiresAttributes;
        this.placementConstraints = builder.placementConstraints;
        this.compatibilities = builder.compatibilities;
        this.runtimePlatform = builder.runtimePlatform;
        this.requiresCompatibilities = builder.requiresCompatibilities;
        this.cpu = builder.cpu;
        this.memory = builder.memory;
        this.inferenceAccelerators = builder.inferenceAccelerators;
        this.pidMode = builder.pidMode;
        this.ipcMode = builder.ipcMode;
        this.proxyConfiguration = builder.proxyConfiguration;
        this.registeredAt = builder.registeredAt;
        this.deregisteredAt = builder.deregisteredAt;
        this.registeredBy = builder.registeredBy;
        this.ephemeralStorage = builder.ephemeralStorage;
    }

    /**
     * <p>
     * The full Amazon Resource Name (ARN) of the task definition.
     * </p>
     * 
     * @return The full Amazon Resource Name (ARN) of the task definition.
     */
    public final String taskDefinitionArn() {
        return taskDefinitionArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the ContainerDefinitions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasContainerDefinitions() {
        return containerDefinitions != null && !(containerDefinitions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of container definitions in JSON format that describe the different containers that make up your task. For
     * more information about container definition parameters and defaults, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_defintions.html">Amazon ECS Task
     * Definitions</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasContainerDefinitions} method.
     * </p>
     * 
     * @return A list of container definitions in JSON format that describe the different containers that make up your
     *         task. For more information about container definition parameters and defaults, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_defintions.html">Amazon ECS Task
     *         Definitions</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     */
    public final List<ContainerDefinition> containerDefinitions() {
        return containerDefinitions;
    }

    /**
     * <p>
     * The name of a family that this task definition is registered to. Up to 255 characters are allowed. Letters (both
     * uppercase and lowercase letters), numbers, hyphens (-), and underscores (_) are allowed.
     * </p>
     * <p>
     * A family groups multiple versions of a task definition. Amazon ECS gives the first task definition that you
     * registered to a family a revision number of 1. Amazon ECS gives sequential revision numbers to each task
     * definition that you add.
     * </p>
     * 
     * @return The name of a family that this task definition is registered to. Up to 255 characters are allowed.
     *         Letters (both uppercase and lowercase letters), numbers, hyphens (-), and underscores (_) are
     *         allowed.</p>
     *         <p>
     *         A family groups multiple versions of a task definition. Amazon ECS gives the first task definition that
     *         you registered to a family a revision number of 1. Amazon ECS gives sequential revision numbers to each
     *         task definition that you add.
     */
    public final String family() {
        return family;
    }

    /**
     * <p>
     * The short name or full Amazon Resource Name (ARN) of the Identity and Access Management role that grants
     * containers in the task permission to call Amazon Web Services APIs on your behalf. For informationabout the
     * required IAM roles for Amazon ECS, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/security-ecs-iam-role-overview.html">IAM roles
     * for Amazon ECS</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * 
     * @return The short name or full Amazon Resource Name (ARN) of the Identity and Access Management role that grants
     *         containers in the task permission to call Amazon Web Services APIs on your behalf. For informationabout
     *         the required IAM roles for Amazon ECS, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/security-ecs-iam-role-overview.html"
     *         >IAM roles for Amazon ECS</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     */
    public final String taskRoleArn() {
        return taskRoleArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the task execution role that grants the Amazon ECS container agent permission
     * to make Amazon Web Services API calls on your behalf. For informationabout the required IAM roles for Amazon ECS,
     * see <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/security-ecs-iam-role-overview.html">IAM
     * roles for Amazon ECS</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the task execution role that grants the Amazon ECS container agent
     *         permission to make Amazon Web Services API calls on your behalf. For informationabout the required IAM
     *         roles for Amazon ECS, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/security-ecs-iam-role-overview.html"
     *         >IAM roles for Amazon ECS</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     */
    public final String executionRoleArn() {
        return executionRoleArn;
    }

    /**
     * <p>
     * The Docker networking mode to use for the containers in the task. The valid values are <code>none</code>,
     * <code>bridge</code>, <code>awsvpc</code>, and <code>host</code>. If no network mode is specified, the default is
     * <code>bridge</code>.
     * </p>
     * <p>
     * For Amazon ECS tasks on Fargate, the <code>awsvpc</code> network mode is required. For Amazon ECS tasks on Amazon
     * EC2 Linux instances, any network mode can be used. For Amazon ECS tasks on Amazon EC2 Windows instances,
     * <code>&lt;default&gt;</code> or <code>awsvpc</code> can be used. If the network mode is set to <code>none</code>,
     * you cannot specify port mappings in your container definitions, and the tasks containers do not have external
     * connectivity. The <code>host</code> and <code>awsvpc</code> network modes offer the highest networking
     * performance for containers because they use the EC2 network stack instead of the virtualized network stack
     * provided by the <code>bridge</code> mode.
     * </p>
     * <p>
     * With the <code>host</code> and <code>awsvpc</code> network modes, exposed container ports are mapped directly to
     * the corresponding host port (for the <code>host</code> network mode) or the attached elastic network interface
     * port (for the <code>awsvpc</code> network mode), so you cannot take advantage of dynamic host port mappings.
     * </p>
     * <important>
     * <p>
     * When using the <code>host</code> network mode, you should not run containers using the root user (UID 0). It is
     * considered best practice to use a non-root user.
     * </p>
     * </important>
     * <p>
     * If the network mode is <code>awsvpc</code>, the task is allocated an elastic network interface, and you must
     * specify a <a href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_NetworkConfiguration.html">
     * NetworkConfiguration</a> value when you create a service or run a task with the task definition. For more
     * information, see <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html">Task
     * Networking</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * If the network mode is <code>host</code>, you cannot run multiple instantiations of the same task on a single
     * container instance when port mappings are used.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #networkMode} will
     * return {@link NetworkMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #networkModeAsString}.
     * </p>
     * 
     * @return The Docker networking mode to use for the containers in the task. The valid values are <code>none</code>,
     *         <code>bridge</code>, <code>awsvpc</code>, and <code>host</code>. If no network mode is specified, the
     *         default is <code>bridge</code>.</p>
     *         <p>
     *         For Amazon ECS tasks on Fargate, the <code>awsvpc</code> network mode is required. For Amazon ECS tasks
     *         on Amazon EC2 Linux instances, any network mode can be used. For Amazon ECS tasks on Amazon EC2 Windows
     *         instances, <code>&lt;default&gt;</code> or <code>awsvpc</code> can be used. If the network mode is set to
     *         <code>none</code>, you cannot specify port mappings in your container definitions, and the tasks
     *         containers do not have external connectivity. The <code>host</code> and <code>awsvpc</code> network modes
     *         offer the highest networking performance for containers because they use the EC2 network stack instead of
     *         the virtualized network stack provided by the <code>bridge</code> mode.
     *         </p>
     *         <p>
     *         With the <code>host</code> and <code>awsvpc</code> network modes, exposed container ports are mapped
     *         directly to the corresponding host port (for the <code>host</code> network mode) or the attached elastic
     *         network interface port (for the <code>awsvpc</code> network mode), so you cannot take advantage of
     *         dynamic host port mappings.
     *         </p>
     *         <important>
     *         <p>
     *         When using the <code>host</code> network mode, you should not run containers using the root user (UID 0).
     *         It is considered best practice to use a non-root user.
     *         </p>
     *         </important>
     *         <p>
     *         If the network mode is <code>awsvpc</code>, the task is allocated an elastic network interface, and you
     *         must specify a <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_NetworkConfiguration.html"
     *         >NetworkConfiguration</a> value when you create a service or run a task with the task definition. For
     *         more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html">Task
     *         Networking</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     *         </p>
     *         <p>
     *         If the network mode is <code>host</code>, you cannot run multiple instantiations of the same task on a
     *         single container instance when port mappings are used.
     * @see NetworkMode
     */
    public final NetworkMode networkMode() {
        return NetworkMode.fromValue(networkMode);
    }

    /**
     * <p>
     * The Docker networking mode to use for the containers in the task. The valid values are <code>none</code>,
     * <code>bridge</code>, <code>awsvpc</code>, and <code>host</code>. If no network mode is specified, the default is
     * <code>bridge</code>.
     * </p>
     * <p>
     * For Amazon ECS tasks on Fargate, the <code>awsvpc</code> network mode is required. For Amazon ECS tasks on Amazon
     * EC2 Linux instances, any network mode can be used. For Amazon ECS tasks on Amazon EC2 Windows instances,
     * <code>&lt;default&gt;</code> or <code>awsvpc</code> can be used. If the network mode is set to <code>none</code>,
     * you cannot specify port mappings in your container definitions, and the tasks containers do not have external
     * connectivity. The <code>host</code> and <code>awsvpc</code> network modes offer the highest networking
     * performance for containers because they use the EC2 network stack instead of the virtualized network stack
     * provided by the <code>bridge</code> mode.
     * </p>
     * <p>
     * With the <code>host</code> and <code>awsvpc</code> network modes, exposed container ports are mapped directly to
     * the corresponding host port (for the <code>host</code> network mode) or the attached elastic network interface
     * port (for the <code>awsvpc</code> network mode), so you cannot take advantage of dynamic host port mappings.
     * </p>
     * <important>
     * <p>
     * When using the <code>host</code> network mode, you should not run containers using the root user (UID 0). It is
     * considered best practice to use a non-root user.
     * </p>
     * </important>
     * <p>
     * If the network mode is <code>awsvpc</code>, the task is allocated an elastic network interface, and you must
     * specify a <a href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_NetworkConfiguration.html">
     * NetworkConfiguration</a> value when you create a service or run a task with the task definition. For more
     * information, see <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html">Task
     * Networking</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * If the network mode is <code>host</code>, you cannot run multiple instantiations of the same task on a single
     * container instance when port mappings are used.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #networkMode} will
     * return {@link NetworkMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #networkModeAsString}.
     * </p>
     * 
     * @return The Docker networking mode to use for the containers in the task. The valid values are <code>none</code>,
     *         <code>bridge</code>, <code>awsvpc</code>, and <code>host</code>. If no network mode is specified, the
     *         default is <code>bridge</code>.</p>
     *         <p>
     *         For Amazon ECS tasks on Fargate, the <code>awsvpc</code> network mode is required. For Amazon ECS tasks
     *         on Amazon EC2 Linux instances, any network mode can be used. For Amazon ECS tasks on Amazon EC2 Windows
     *         instances, <code>&lt;default&gt;</code> or <code>awsvpc</code> can be used. If the network mode is set to
     *         <code>none</code>, you cannot specify port mappings in your container definitions, and the tasks
     *         containers do not have external connectivity. The <code>host</code> and <code>awsvpc</code> network modes
     *         offer the highest networking performance for containers because they use the EC2 network stack instead of
     *         the virtualized network stack provided by the <code>bridge</code> mode.
     *         </p>
     *         <p>
     *         With the <code>host</code> and <code>awsvpc</code> network modes, exposed container ports are mapped
     *         directly to the corresponding host port (for the <code>host</code> network mode) or the attached elastic
     *         network interface port (for the <code>awsvpc</code> network mode), so you cannot take advantage of
     *         dynamic host port mappings.
     *         </p>
     *         <important>
     *         <p>
     *         When using the <code>host</code> network mode, you should not run containers using the root user (UID 0).
     *         It is considered best practice to use a non-root user.
     *         </p>
     *         </important>
     *         <p>
     *         If the network mode is <code>awsvpc</code>, the task is allocated an elastic network interface, and you
     *         must specify a <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_NetworkConfiguration.html"
     *         >NetworkConfiguration</a> value when you create a service or run a task with the task definition. For
     *         more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html">Task
     *         Networking</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     *         </p>
     *         <p>
     *         If the network mode is <code>host</code>, you cannot run multiple instantiations of the same task on a
     *         single container instance when port mappings are used.
     * @see NetworkMode
     */
    public final String networkModeAsString() {
        return networkMode;
    }

    /**
     * <p>
     * The revision of the task in a particular family. The revision is a version number of a task definition in a
     * family. When you register a task definition for the first time, the revision is <code>1</code>. Each time that
     * you register a new revision of a task definition in the same family, the revision value always increases by one.
     * This is even if you deregistered previous revisions in this family.
     * </p>
     * 
     * @return The revision of the task in a particular family. The revision is a version number of a task definition in
     *         a family. When you register a task definition for the first time, the revision is <code>1</code>. Each
     *         time that you register a new revision of a task definition in the same family, the revision value always
     *         increases by one. This is even if you deregistered previous revisions in this family.
     */
    public final Integer revision() {
        return revision;
    }

    /**
     * For responses, this returns true if the service returned a value for the Volumes property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasVolumes() {
        return volumes != null && !(volumes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of data volume definitions for the task. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_data_volumes.html">Using data volumes in
     * tasks</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * The <code>host</code> and <code>sourcePath</code> parameters aren't supported for tasks run on Fargate.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVolumes} method.
     * </p>
     * 
     * @return The list of data volume definitions for the task. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_data_volumes.html">Using data
     *         volumes in tasks</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.</p> <note>
     *         <p>
     *         The <code>host</code> and <code>sourcePath</code> parameters aren't supported for tasks run on Fargate.
     *         </p>
     */
    public final List<Volume> volumes() {
        return volumes;
    }

    /**
     * <p>
     * The status of the task definition.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link TaskDefinitionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the task definition.
     * @see TaskDefinitionStatus
     */
    public final TaskDefinitionStatus status() {
        return TaskDefinitionStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the task definition.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link TaskDefinitionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the task definition.
     * @see TaskDefinitionStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * For responses, this returns true if the service returned a value for the RequiresAttributes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRequiresAttributes() {
        return requiresAttributes != null && !(requiresAttributes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The container instance attributes required by your task. When an Amazon EC2 instance is registered to your
     * cluster, the Amazon ECS container agent assigns some standard attributes to the instance. You can apply custom
     * attributes. These are specified as key-value pairs using the Amazon ECS console or the <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAttributes.html">PutAttributes</a> API.
     * These attributes are used when determining task placement for tasks hosted on Amazon EC2 instances. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html#attributes"
     * >Attributes</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * This parameter isn't supported for tasks run on Fargate.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRequiresAttributes} method.
     * </p>
     * 
     * @return The container instance attributes required by your task. When an Amazon EC2 instance is registered to
     *         your cluster, the Amazon ECS container agent assigns some standard attributes to the instance. You can
     *         apply custom attributes. These are specified as key-value pairs using the Amazon ECS console or the <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAttributes.html">PutAttributes</a>
     *         API. These attributes are used when determining task placement for tasks hosted on Amazon EC2 instances.
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html#attributes"
     *         >Attributes</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.</p> <note>
     *         <p>
     *         This parameter isn't supported for tasks run on Fargate.
     *         </p>
     */
    public final List<Attribute> requiresAttributes() {
        return requiresAttributes;
    }

    /**
     * For responses, this returns true if the service returned a value for the PlacementConstraints property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPlacementConstraints() {
        return placementConstraints != null && !(placementConstraints instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of placement constraint objects to use for tasks.
     * </p>
     * <note>
     * <p>
     * This parameter isn't supported for tasks run on Fargate.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPlacementConstraints} method.
     * </p>
     * 
     * @return An array of placement constraint objects to use for tasks.</p> <note>
     *         <p>
     *         This parameter isn't supported for tasks run on Fargate.
     *         </p>
     */
    public final List<TaskDefinitionPlacementConstraint> placementConstraints() {
        return placementConstraints;
    }

    /**
     * <p>
     * Amazon ECS validates the task definition parameters with those supported by the launch type. For more
     * information, see <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon
     * ECS launch types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCompatibilities} method.
     * </p>
     * 
     * @return Amazon ECS validates the task definition parameters with those supported by the launch type. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch
     *         types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     */
    public final List<Compatibility> compatibilities() {
        return CompatibilityListCopier.copyStringToEnum(compatibilities);
    }

    /**
     * For responses, this returns true if the service returned a value for the Compatibilities property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCompatibilities() {
        return compatibilities != null && !(compatibilities instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Amazon ECS validates the task definition parameters with those supported by the launch type. For more
     * information, see <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon
     * ECS launch types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCompatibilities} method.
     * </p>
     * 
     * @return Amazon ECS validates the task definition parameters with those supported by the launch type. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch
     *         types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     */
    public final List<String> compatibilitiesAsStrings() {
        return compatibilities;
    }

    /**
     * <p>
     * The operating system that your task definitions are running on. A platform family is specified only for tasks
     * using the Fargate launch type.
     * </p>
     * <p>
     * When you specify a task in a service, this value must match the <code>runtimePlatform</code> value of the
     * service.
     * </p>
     * 
     * @return The operating system that your task definitions are running on. A platform family is specified only for
     *         tasks using the Fargate launch type. </p>
     *         <p>
     *         When you specify a task in a service, this value must match the <code>runtimePlatform</code> value of the
     *         service.
     */
    public final RuntimePlatform runtimePlatform() {
        return runtimePlatform;
    }

    /**
     * <p>
     * The task launch types the task definition was validated against. The valid values are <code>EC2</code>,
     * <code>FARGATE</code>, and <code>EXTERNAL</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch types</a>
     * in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRequiresCompatibilities} method.
     * </p>
     * 
     * @return The task launch types the task definition was validated against. The valid values are <code>EC2</code>,
     *         <code>FARGATE</code>, and <code>EXTERNAL</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch
     *         types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     */
    public final List<Compatibility> requiresCompatibilities() {
        return CompatibilityListCopier.copyStringToEnum(requiresCompatibilities);
    }

    /**
     * For responses, this returns true if the service returned a value for the RequiresCompatibilities property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasRequiresCompatibilities() {
        return requiresCompatibilities != null && !(requiresCompatibilities instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The task launch types the task definition was validated against. The valid values are <code>EC2</code>,
     * <code>FARGATE</code>, and <code>EXTERNAL</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch types</a>
     * in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRequiresCompatibilities} method.
     * </p>
     * 
     * @return The task launch types the task definition was validated against. The valid values are <code>EC2</code>,
     *         <code>FARGATE</code>, and <code>EXTERNAL</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch
     *         types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     */
    public final List<String> requiresCompatibilitiesAsStrings() {
        return requiresCompatibilities;
    }

    /**
     * <p>
     * The number of <code>cpu</code> units used by the task. If you use the EC2 launch type, this field is optional.
     * Any value can be used. If you use the Fargate launch type, this field is required. You must use one of the
     * following values. The value that you choose determines your range of valid values for the <code>memory</code>
     * parameter.
     * </p>
     * <p>
     * If you use the EC2 launch type, this field is optional. Supported values are between <code>128</code> CPU units (
     * <code>0.125</code> vCPUs) and <code>10240</code> CPU units (<code>10</code> vCPUs).
     * </p>
     * <p>
     * The CPU units cannot be less than 1 vCPU when you use Windows containers on Fargate.
     * </p>
     * <ul>
     * <li>
     * <p>
     * 256 (.25 vCPU) - Available <code>memory</code> values: 512 (0.5 GB), 1024 (1 GB), 2048 (2 GB)
     * </p>
     * </li>
     * <li>
     * <p>
     * 512 (.5 vCPU) - Available <code>memory</code> values: 1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB)
     * </p>
     * </li>
     * <li>
     * <p>
     * 1024 (1 vCPU) - Available <code>memory</code> values: 2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB), 6144 (6
     * GB), 7168 (7 GB), 8192 (8 GB)
     * </p>
     * </li>
     * <li>
     * <p>
     * 2048 (2 vCPU) - Available <code>memory</code> values: 4096 (4 GB) and 16384 (16 GB) in increments of 1024 (1 GB)
     * </p>
     * </li>
     * <li>
     * <p>
     * 4096 (4 vCPU) - Available <code>memory</code> values: 8192 (8 GB) and 30720 (30 GB) in increments of 1024 (1 GB)
     * </p>
     * </li>
     * <li>
     * <p>
     * 8192 (8 vCPU) - Available <code>memory</code> values: 16 GB and 60 GB in 4 GB increments
     * </p>
     * <p>
     * This option requires Linux platform <code>1.4.0</code> or later.
     * </p>
     * </li>
     * <li>
     * <p>
     * 16384 (16vCPU) - Available <code>memory</code> values: 32GB and 120 GB in 8 GB increments
     * </p>
     * <p>
     * This option requires Linux platform <code>1.4.0</code> or later.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The number of <code>cpu</code> units used by the task. If you use the EC2 launch type, this field is
     *         optional. Any value can be used. If you use the Fargate launch type, this field is required. You must use
     *         one of the following values. The value that you choose determines your range of valid values for the
     *         <code>memory</code> parameter.</p>
     *         <p>
     *         If you use the EC2 launch type, this field is optional. Supported values are between <code>128</code> CPU
     *         units (<code>0.125</code> vCPUs) and <code>10240</code> CPU units (<code>10</code> vCPUs).
     *         </p>
     *         <p>
     *         The CPU units cannot be less than 1 vCPU when you use Windows containers on Fargate.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         256 (.25 vCPU) - Available <code>memory</code> values: 512 (0.5 GB), 1024 (1 GB), 2048 (2 GB)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         512 (.5 vCPU) - Available <code>memory</code> values: 1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         1024 (1 vCPU) - Available <code>memory</code> values: 2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB),
     *         6144 (6 GB), 7168 (7 GB), 8192 (8 GB)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         2048 (2 vCPU) - Available <code>memory</code> values: 4096 (4 GB) and 16384 (16 GB) in increments of 1024
     *         (1 GB)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4096 (4 vCPU) - Available <code>memory</code> values: 8192 (8 GB) and 30720 (30 GB) in increments of 1024
     *         (1 GB)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         8192 (8 vCPU) - Available <code>memory</code> values: 16 GB and 60 GB in 4 GB increments
     *         </p>
     *         <p>
     *         This option requires Linux platform <code>1.4.0</code> or later.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         16384 (16vCPU) - Available <code>memory</code> values: 32GB and 120 GB in 8 GB increments
     *         </p>
     *         <p>
     *         This option requires Linux platform <code>1.4.0</code> or later.
     *         </p>
     *         </li>
     */
    public final String cpu() {
        return cpu;
    }

    /**
     * <p>
     * The amount (in MiB) of memory used by the task.
     * </p>
     * <p>
     * If your tasks runs on Amazon EC2 instances, you must specify either a task-level memory value or a
     * container-level memory value. This field is optional and any value can be used. If a task-level memory value is
     * specified, the container-level memory value is optional. For more information regarding container-level memory
     * and memory reservation, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html"
     * >ContainerDefinition</a>.
     * </p>
     * <p>
     * If your tasks runs on Fargate, this field is required. You must use one of the following values. The value you
     * choose determines your range of valid values for the <code>cpu</code> parameter.
     * </p>
     * <ul>
     * <li>
     * <p>
     * 512 (0.5 GB), 1024 (1 GB), 2048 (2 GB) - Available <code>cpu</code> values: 256 (.25 vCPU)
     * </p>
     * </li>
     * <li>
     * <p>
     * 1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB) - Available <code>cpu</code> values: 512 (.5 vCPU)
     * </p>
     * </li>
     * <li>
     * <p>
     * 2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB), 6144 (6 GB), 7168 (7 GB), 8192 (8 GB) - Available
     * <code>cpu</code> values: 1024 (1 vCPU)
     * </p>
     * </li>
     * <li>
     * <p>
     * Between 4096 (4 GB) and 16384 (16 GB) in increments of 1024 (1 GB) - Available <code>cpu</code> values: 2048 (2
     * vCPU)
     * </p>
     * </li>
     * <li>
     * <p>
     * Between 8192 (8 GB) and 30720 (30 GB) in increments of 1024 (1 GB) - Available <code>cpu</code> values: 4096 (4
     * vCPU)
     * </p>
     * </li>
     * <li>
     * <p>
     * Between 16 GB and 60 GB in 4 GB increments - Available <code>cpu</code> values: 8192 (8 vCPU)
     * </p>
     * <p>
     * This option requires Linux platform <code>1.4.0</code> or later.
     * </p>
     * </li>
     * <li>
     * <p>
     * Between 32GB and 120 GB in 8 GB increments - Available <code>cpu</code> values: 16384 (16 vCPU)
     * </p>
     * <p>
     * This option requires Linux platform <code>1.4.0</code> or later.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The amount (in MiB) of memory used by the task.</p>
     *         <p>
     *         If your tasks runs on Amazon EC2 instances, you must specify either a task-level memory value or a
     *         container-level memory value. This field is optional and any value can be used. If a task-level memory
     *         value is specified, the container-level memory value is optional. For more information regarding
     *         container-level memory and memory reservation, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html"
     *         >ContainerDefinition</a>.
     *         </p>
     *         <p>
     *         If your tasks runs on Fargate, this field is required. You must use one of the following values. The
     *         value you choose determines your range of valid values for the <code>cpu</code> parameter.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         512 (0.5 GB), 1024 (1 GB), 2048 (2 GB) - Available <code>cpu</code> values: 256 (.25 vCPU)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB) - Available <code>cpu</code> values: 512 (.5 vCPU)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB), 6144 (6 GB), 7168 (7 GB), 8192 (8 GB) - Available
     *         <code>cpu</code> values: 1024 (1 vCPU)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Between 4096 (4 GB) and 16384 (16 GB) in increments of 1024 (1 GB) - Available <code>cpu</code> values:
     *         2048 (2 vCPU)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Between 8192 (8 GB) and 30720 (30 GB) in increments of 1024 (1 GB) - Available <code>cpu</code> values:
     *         4096 (4 vCPU)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Between 16 GB and 60 GB in 4 GB increments - Available <code>cpu</code> values: 8192 (8 vCPU)
     *         </p>
     *         <p>
     *         This option requires Linux platform <code>1.4.0</code> or later.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Between 32GB and 120 GB in 8 GB increments - Available <code>cpu</code> values: 16384 (16 vCPU)
     *         </p>
     *         <p>
     *         This option requires Linux platform <code>1.4.0</code> or later.
     *         </p>
     *         </li>
     */
    public final String memory() {
        return memory;
    }

    /**
     * For responses, this returns true if the service returned a value for the InferenceAccelerators property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasInferenceAccelerators() {
        return inferenceAccelerators != null && !(inferenceAccelerators instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Elastic Inference accelerator that's associated with the task.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInferenceAccelerators} method.
     * </p>
     * 
     * @return The Elastic Inference accelerator that's associated with the task.
     */
    public final List<InferenceAccelerator> inferenceAccelerators() {
        return inferenceAccelerators;
    }

    /**
     * <p>
     * The process namespace to use for the containers in the task. The valid values are <code>host</code> or
     * <code>task</code>. On Fargate for Linux containers, the only valid value is <code>task</code>. For example,
     * monitoring sidecars might need <code>pidMode</code> to access information about other containers running in the
     * same task.
     * </p>
     * <p>
     * If <code>host</code> is specified, all containers within the tasks that specified the <code>host</code> PID mode
     * on the same container instance share the same process namespace with the host Amazon EC2 instance.
     * </p>
     * <p>
     * If <code>task</code> is specified, all containers within the specified task share the same process namespace.
     * </p>
     * <p>
     * If no value is specified, the default is a private namespace for each container.
     * </p>
     * <p>
     * If the <code>host</code> PID mode is used, there's a heightened risk of undesired process namespace exposure.
     * </p>
     * <note>
     * <p>
     * This parameter is not supported for Windows containers.
     * </p>
     * </note> <note>
     * <p>
     * This parameter is only supported for tasks that are hosted on Fargate if the tasks are using platform version
     * <code>1.4.0</code> or later (Linux). This isn't supported for Windows containers on Fargate.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #pidMode} will
     * return {@link PidMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #pidModeAsString}.
     * </p>
     * 
     * @return The process namespace to use for the containers in the task. The valid values are <code>host</code> or
     *         <code>task</code>. On Fargate for Linux containers, the only valid value is <code>task</code>. For
     *         example, monitoring sidecars might need <code>pidMode</code> to access information about other containers
     *         running in the same task.</p>
     *         <p>
     *         If <code>host</code> is specified, all containers within the tasks that specified the <code>host</code>
     *         PID mode on the same container instance share the same process namespace with the host Amazon EC2
     *         instance.
     *         </p>
     *         <p>
     *         If <code>task</code> is specified, all containers within the specified task share the same process
     *         namespace.
     *         </p>
     *         <p>
     *         If no value is specified, the default is a private namespace for each container.
     *         </p>
     *         <p>
     *         If the <code>host</code> PID mode is used, there's a heightened risk of undesired process namespace
     *         exposure.
     *         </p>
     *         <note>
     *         <p>
     *         This parameter is not supported for Windows containers.
     *         </p>
     *         </note> <note>
     *         <p>
     *         This parameter is only supported for tasks that are hosted on Fargate if the tasks are using platform
     *         version <code>1.4.0</code> or later (Linux). This isn't supported for Windows containers on Fargate.
     *         </p>
     * @see PidMode
     */
    public final PidMode pidMode() {
        return PidMode.fromValue(pidMode);
    }

    /**
     * <p>
     * The process namespace to use for the containers in the task. The valid values are <code>host</code> or
     * <code>task</code>. On Fargate for Linux containers, the only valid value is <code>task</code>. For example,
     * monitoring sidecars might need <code>pidMode</code> to access information about other containers running in the
     * same task.
     * </p>
     * <p>
     * If <code>host</code> is specified, all containers within the tasks that specified the <code>host</code> PID mode
     * on the same container instance share the same process namespace with the host Amazon EC2 instance.
     * </p>
     * <p>
     * If <code>task</code> is specified, all containers within the specified task share the same process namespace.
     * </p>
     * <p>
     * If no value is specified, the default is a private namespace for each container.
     * </p>
     * <p>
     * If the <code>host</code> PID mode is used, there's a heightened risk of undesired process namespace exposure.
     * </p>
     * <note>
     * <p>
     * This parameter is not supported for Windows containers.
     * </p>
     * </note> <note>
     * <p>
     * This parameter is only supported for tasks that are hosted on Fargate if the tasks are using platform version
     * <code>1.4.0</code> or later (Linux). This isn't supported for Windows containers on Fargate.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #pidMode} will
     * return {@link PidMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #pidModeAsString}.
     * </p>
     * 
     * @return The process namespace to use for the containers in the task. The valid values are <code>host</code> or
     *         <code>task</code>. On Fargate for Linux containers, the only valid value is <code>task</code>. For
     *         example, monitoring sidecars might need <code>pidMode</code> to access information about other containers
     *         running in the same task.</p>
     *         <p>
     *         If <code>host</code> is specified, all containers within the tasks that specified the <code>host</code>
     *         PID mode on the same container instance share the same process namespace with the host Amazon EC2
     *         instance.
     *         </p>
     *         <p>
     *         If <code>task</code> is specified, all containers within the specified task share the same process
     *         namespace.
     *         </p>
     *         <p>
     *         If no value is specified, the default is a private namespace for each container.
     *         </p>
     *         <p>
     *         If the <code>host</code> PID mode is used, there's a heightened risk of undesired process namespace
     *         exposure.
     *         </p>
     *         <note>
     *         <p>
     *         This parameter is not supported for Windows containers.
     *         </p>
     *         </note> <note>
     *         <p>
     *         This parameter is only supported for tasks that are hosted on Fargate if the tasks are using platform
     *         version <code>1.4.0</code> or later (Linux). This isn't supported for Windows containers on Fargate.
     *         </p>
     * @see PidMode
     */
    public final String pidModeAsString() {
        return pidMode;
    }

    /**
     * <p>
     * The IPC resource namespace to use for the containers in the task. The valid values are <code>host</code>,
     * <code>task</code>, or <code>none</code>. If <code>host</code> is specified, then all containers within the tasks
     * that specified the <code>host</code> IPC mode on the same container instance share the same IPC resources with
     * the host Amazon EC2 instance. If <code>task</code> is specified, all containers within the specified task share
     * the same IPC resources. If <code>none</code> is specified, then IPC resources within the containers of a task are
     * private and not shared with other containers in a task or on the container instance. If no value is specified,
     * then the IPC resource namespace sharing depends on the Docker daemon setting on the container instance.
     * </p>
     * <p>
     * If the <code>host</code> IPC mode is used, be aware that there is a heightened risk of undesired IPC namespace
     * expose.
     * </p>
     * <p>
     * If you are setting namespaced kernel parameters using <code>systemControls</code> for the containers in the task,
     * the following will apply to your IPC resource namespace. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html">System
     * Controls</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For tasks that use the <code>host</code> IPC mode, IPC namespace related <code>systemControls</code> are not
     * supported.
     * </p>
     * </li>
     * <li>
     * <p>
     * For tasks that use the <code>task</code> IPC mode, IPC namespace related <code>systemControls</code> will apply
     * to all containers within a task.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * This parameter is not supported for Windows containers or tasks run on Fargate.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ipcMode} will
     * return {@link IpcMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ipcModeAsString}.
     * </p>
     * 
     * @return The IPC resource namespace to use for the containers in the task. The valid values are <code>host</code>,
     *         <code>task</code>, or <code>none</code>. If <code>host</code> is specified, then all containers within
     *         the tasks that specified the <code>host</code> IPC mode on the same container instance share the same IPC
     *         resources with the host Amazon EC2 instance. If <code>task</code> is specified, all containers within the
     *         specified task share the same IPC resources. If <code>none</code> is specified, then IPC resources within
     *         the containers of a task are private and not shared with other containers in a task or on the container
     *         instance. If no value is specified, then the IPC resource namespace sharing depends on the Docker daemon
     *         setting on the container instance.</p>
     *         <p>
     *         If the <code>host</code> IPC mode is used, be aware that there is a heightened risk of undesired IPC
     *         namespace expose.
     *         </p>
     *         <p>
     *         If you are setting namespaced kernel parameters using <code>systemControls</code> for the containers in
     *         the task, the following will apply to your IPC resource namespace. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html">System
     *         Controls</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For tasks that use the <code>host</code> IPC mode, IPC namespace related <code>systemControls</code> are
     *         not supported.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For tasks that use the <code>task</code> IPC mode, IPC namespace related <code>systemControls</code> will
     *         apply to all containers within a task.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         This parameter is not supported for Windows containers or tasks run on Fargate.
     *         </p>
     * @see IpcMode
     */
    public final IpcMode ipcMode() {
        return IpcMode.fromValue(ipcMode);
    }

    /**
     * <p>
     * The IPC resource namespace to use for the containers in the task. The valid values are <code>host</code>,
     * <code>task</code>, or <code>none</code>. If <code>host</code> is specified, then all containers within the tasks
     * that specified the <code>host</code> IPC mode on the same container instance share the same IPC resources with
     * the host Amazon EC2 instance. If <code>task</code> is specified, all containers within the specified task share
     * the same IPC resources. If <code>none</code> is specified, then IPC resources within the containers of a task are
     * private and not shared with other containers in a task or on the container instance. If no value is specified,
     * then the IPC resource namespace sharing depends on the Docker daemon setting on the container instance.
     * </p>
     * <p>
     * If the <code>host</code> IPC mode is used, be aware that there is a heightened risk of undesired IPC namespace
     * expose.
     * </p>
     * <p>
     * If you are setting namespaced kernel parameters using <code>systemControls</code> for the containers in the task,
     * the following will apply to your IPC resource namespace. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html">System
     * Controls</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For tasks that use the <code>host</code> IPC mode, IPC namespace related <code>systemControls</code> are not
     * supported.
     * </p>
     * </li>
     * <li>
     * <p>
     * For tasks that use the <code>task</code> IPC mode, IPC namespace related <code>systemControls</code> will apply
     * to all containers within a task.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * This parameter is not supported for Windows containers or tasks run on Fargate.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ipcMode} will
     * return {@link IpcMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ipcModeAsString}.
     * </p>
     * 
     * @return The IPC resource namespace to use for the containers in the task. The valid values are <code>host</code>,
     *         <code>task</code>, or <code>none</code>. If <code>host</code> is specified, then all containers within
     *         the tasks that specified the <code>host</code> IPC mode on the same container instance share the same IPC
     *         resources with the host Amazon EC2 instance. If <code>task</code> is specified, all containers within the
     *         specified task share the same IPC resources. If <code>none</code> is specified, then IPC resources within
     *         the containers of a task are private and not shared with other containers in a task or on the container
     *         instance. If no value is specified, then the IPC resource namespace sharing depends on the Docker daemon
     *         setting on the container instance.</p>
     *         <p>
     *         If the <code>host</code> IPC mode is used, be aware that there is a heightened risk of undesired IPC
     *         namespace expose.
     *         </p>
     *         <p>
     *         If you are setting namespaced kernel parameters using <code>systemControls</code> for the containers in
     *         the task, the following will apply to your IPC resource namespace. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html">System
     *         Controls</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For tasks that use the <code>host</code> IPC mode, IPC namespace related <code>systemControls</code> are
     *         not supported.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For tasks that use the <code>task</code> IPC mode, IPC namespace related <code>systemControls</code> will
     *         apply to all containers within a task.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         This parameter is not supported for Windows containers or tasks run on Fargate.
     *         </p>
     * @see IpcMode
     */
    public final String ipcModeAsString() {
        return ipcMode;
    }

    /**
     * <p>
     * The configuration details for the App Mesh proxy.
     * </p>
     * <p>
     * Your Amazon ECS container instances require at least version 1.26.0 of the container agent and at least version
     * 1.26.0-1 of the <code>ecs-init</code> package to use a proxy configuration. If your container instances are
     * launched from the Amazon ECS optimized AMI version <code>20190301</code> or later, they contain the required
     * versions of the container agent and <code>ecs-init</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html">Amazon ECS-optimized
     * Linux AMI</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * 
     * @return The configuration details for the App Mesh proxy.</p>
     *         <p>
     *         Your Amazon ECS container instances require at least version 1.26.0 of the container agent and at least
     *         version 1.26.0-1 of the <code>ecs-init</code> package to use a proxy configuration. If your container
     *         instances are launched from the Amazon ECS optimized AMI version <code>20190301</code> or later, they
     *         contain the required versions of the container agent and <code>ecs-init</code>. For more information, see
     *         <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html">Amazon
     *         ECS-optimized Linux AMI</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     */
    public final ProxyConfiguration proxyConfiguration() {
        return proxyConfiguration;
    }

    /**
     * <p>
     * The Unix timestamp for the time when the task definition was registered.
     * </p>
     * 
     * @return The Unix timestamp for the time when the task definition was registered.
     */
    public final Instant registeredAt() {
        return registeredAt;
    }

    /**
     * <p>
     * The Unix timestamp for the time when the task definition was deregistered.
     * </p>
     * 
     * @return The Unix timestamp for the time when the task definition was deregistered.
     */
    public final Instant deregisteredAt() {
        return deregisteredAt;
    }

    /**
     * <p>
     * The principal that registered the task definition.
     * </p>
     * 
     * @return The principal that registered the task definition.
     */
    public final String registeredBy() {
        return registeredBy;
    }

    /**
     * <p>
     * The ephemeral storage settings to use for tasks run with the task definition.
     * </p>
     * 
     * @return The ephemeral storage settings to use for tasks run with the task definition.
     */
    public final EphemeralStorage ephemeralStorage() {
        return ephemeralStorage;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(taskDefinitionArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasContainerDefinitions() ? containerDefinitions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(family());
        hashCode = 31 * hashCode + Objects.hashCode(taskRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(executionRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(networkModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(revision());
        hashCode = 31 * hashCode + Objects.hashCode(hasVolumes() ? volumes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasRequiresAttributes() ? requiresAttributes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasPlacementConstraints() ? placementConstraints() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasCompatibilities() ? compatibilitiesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(runtimePlatform());
        hashCode = 31 * hashCode + Objects.hashCode(hasRequiresCompatibilities() ? requiresCompatibilitiesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(cpu());
        hashCode = 31 * hashCode + Objects.hashCode(memory());
        hashCode = 31 * hashCode + Objects.hashCode(hasInferenceAccelerators() ? inferenceAccelerators() : null);
        hashCode = 31 * hashCode + Objects.hashCode(pidModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(ipcModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(proxyConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(registeredAt());
        hashCode = 31 * hashCode + Objects.hashCode(deregisteredAt());
        hashCode = 31 * hashCode + Objects.hashCode(registeredBy());
        hashCode = 31 * hashCode + Objects.hashCode(ephemeralStorage());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TaskDefinition)) {
            return false;
        }
        TaskDefinition other = (TaskDefinition) obj;
        return Objects.equals(taskDefinitionArn(), other.taskDefinitionArn())
                && hasContainerDefinitions() == other.hasContainerDefinitions()
                && Objects.equals(containerDefinitions(), other.containerDefinitions())
                && Objects.equals(family(), other.family()) && Objects.equals(taskRoleArn(), other.taskRoleArn())
                && Objects.equals(executionRoleArn(), other.executionRoleArn())
                && Objects.equals(networkModeAsString(), other.networkModeAsString())
                && Objects.equals(revision(), other.revision()) && hasVolumes() == other.hasVolumes()
                && Objects.equals(volumes(), other.volumes()) && Objects.equals(statusAsString(), other.statusAsString())
                && hasRequiresAttributes() == other.hasRequiresAttributes()
                && Objects.equals(requiresAttributes(), other.requiresAttributes())
                && hasPlacementConstraints() == other.hasPlacementConstraints()
                && Objects.equals(placementConstraints(), other.placementConstraints())
                && hasCompatibilities() == other.hasCompatibilities()
                && Objects.equals(compatibilitiesAsStrings(), other.compatibilitiesAsStrings())
                && Objects.equals(runtimePlatform(), other.runtimePlatform())
                && hasRequiresCompatibilities() == other.hasRequiresCompatibilities()
                && Objects.equals(requiresCompatibilitiesAsStrings(), other.requiresCompatibilitiesAsStrings())
                && Objects.equals(cpu(), other.cpu()) && Objects.equals(memory(), other.memory())
                && hasInferenceAccelerators() == other.hasInferenceAccelerators()
                && Objects.equals(inferenceAccelerators(), other.inferenceAccelerators())
                && Objects.equals(pidModeAsString(), other.pidModeAsString())
                && Objects.equals(ipcModeAsString(), other.ipcModeAsString())
                && Objects.equals(proxyConfiguration(), other.proxyConfiguration())
                && Objects.equals(registeredAt(), other.registeredAt())
                && Objects.equals(deregisteredAt(), other.deregisteredAt())
                && Objects.equals(registeredBy(), other.registeredBy())
                && Objects.equals(ephemeralStorage(), other.ephemeralStorage());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TaskDefinition").add("TaskDefinitionArn", taskDefinitionArn())
                .add("ContainerDefinitions", hasContainerDefinitions() ? containerDefinitions() : null).add("Family", family())
                .add("TaskRoleArn", taskRoleArn()).add("ExecutionRoleArn", executionRoleArn())
                .add("NetworkMode", networkModeAsString()).add("Revision", revision())
                .add("Volumes", hasVolumes() ? volumes() : null).add("Status", statusAsString())
                .add("RequiresAttributes", hasRequiresAttributes() ? requiresAttributes() : null)
                .add("PlacementConstraints", hasPlacementConstraints() ? placementConstraints() : null)
                .add("Compatibilities", hasCompatibilities() ? compatibilitiesAsStrings() : null)
                .add("RuntimePlatform", runtimePlatform())
                .add("RequiresCompatibilities", hasRequiresCompatibilities() ? requiresCompatibilitiesAsStrings() : null)
                .add("Cpu", cpu()).add("Memory", memory())
                .add("InferenceAccelerators", hasInferenceAccelerators() ? inferenceAccelerators() : null)
                .add("PidMode", pidModeAsString()).add("IpcMode", ipcModeAsString())
                .add("ProxyConfiguration", proxyConfiguration()).add("RegisteredAt", registeredAt())
                .add("DeregisteredAt", deregisteredAt()).add("RegisteredBy", registeredBy())
                .add("EphemeralStorage", ephemeralStorage()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "taskDefinitionArn":
            return Optional.ofNullable(clazz.cast(taskDefinitionArn()));
        case "containerDefinitions":
            return Optional.ofNullable(clazz.cast(containerDefinitions()));
        case "family":
            return Optional.ofNullable(clazz.cast(family()));
        case "taskRoleArn":
            return Optional.ofNullable(clazz.cast(taskRoleArn()));
        case "executionRoleArn":
            return Optional.ofNullable(clazz.cast(executionRoleArn()));
        case "networkMode":
            return Optional.ofNullable(clazz.cast(networkModeAsString()));
        case "revision":
            return Optional.ofNullable(clazz.cast(revision()));
        case "volumes":
            return Optional.ofNullable(clazz.cast(volumes()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "requiresAttributes":
            return Optional.ofNullable(clazz.cast(requiresAttributes()));
        case "placementConstraints":
            return Optional.ofNullable(clazz.cast(placementConstraints()));
        case "compatibilities":
            return Optional.ofNullable(clazz.cast(compatibilitiesAsStrings()));
        case "runtimePlatform":
            return Optional.ofNullable(clazz.cast(runtimePlatform()));
        case "requiresCompatibilities":
            return Optional.ofNullable(clazz.cast(requiresCompatibilitiesAsStrings()));
        case "cpu":
            return Optional.ofNullable(clazz.cast(cpu()));
        case "memory":
            return Optional.ofNullable(clazz.cast(memory()));
        case "inferenceAccelerators":
            return Optional.ofNullable(clazz.cast(inferenceAccelerators()));
        case "pidMode":
            return Optional.ofNullable(clazz.cast(pidModeAsString()));
        case "ipcMode":
            return Optional.ofNullable(clazz.cast(ipcModeAsString()));
        case "proxyConfiguration":
            return Optional.ofNullable(clazz.cast(proxyConfiguration()));
        case "registeredAt":
            return Optional.ofNullable(clazz.cast(registeredAt()));
        case "deregisteredAt":
            return Optional.ofNullable(clazz.cast(deregisteredAt()));
        case "registeredBy":
            return Optional.ofNullable(clazz.cast(registeredBy()));
        case "ephemeralStorage":
            return Optional.ofNullable(clazz.cast(ephemeralStorage()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("taskDefinitionArn", TASK_DEFINITION_ARN_FIELD);
        map.put("containerDefinitions", CONTAINER_DEFINITIONS_FIELD);
        map.put("family", FAMILY_FIELD);
        map.put("taskRoleArn", TASK_ROLE_ARN_FIELD);
        map.put("executionRoleArn", EXECUTION_ROLE_ARN_FIELD);
        map.put("networkMode", NETWORK_MODE_FIELD);
        map.put("revision", REVISION_FIELD);
        map.put("volumes", VOLUMES_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("requiresAttributes", REQUIRES_ATTRIBUTES_FIELD);
        map.put("placementConstraints", PLACEMENT_CONSTRAINTS_FIELD);
        map.put("compatibilities", COMPATIBILITIES_FIELD);
        map.put("runtimePlatform", RUNTIME_PLATFORM_FIELD);
        map.put("requiresCompatibilities", REQUIRES_COMPATIBILITIES_FIELD);
        map.put("cpu", CPU_FIELD);
        map.put("memory", MEMORY_FIELD);
        map.put("inferenceAccelerators", INFERENCE_ACCELERATORS_FIELD);
        map.put("pidMode", PID_MODE_FIELD);
        map.put("ipcMode", IPC_MODE_FIELD);
        map.put("proxyConfiguration", PROXY_CONFIGURATION_FIELD);
        map.put("registeredAt", REGISTERED_AT_FIELD);
        map.put("deregisteredAt", DEREGISTERED_AT_FIELD);
        map.put("registeredBy", REGISTERED_BY_FIELD);
        map.put("ephemeralStorage", EPHEMERAL_STORAGE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<TaskDefinition, T> g) {
        return obj -> g.apply((TaskDefinition) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TaskDefinition> {
        /**
         * <p>
         * The full Amazon Resource Name (ARN) of the task definition.
         * </p>
         * 
         * @param taskDefinitionArn
         *        The full Amazon Resource Name (ARN) of the task definition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskDefinitionArn(String taskDefinitionArn);

        /**
         * <p>
         * A list of container definitions in JSON format that describe the different containers that make up your task.
         * For more information about container definition parameters and defaults, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_defintions.html">Amazon ECS Task
         * Definitions</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param containerDefinitions
         *        A list of container definitions in JSON format that describe the different containers that make up
         *        your task. For more information about container definition parameters and defaults, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_defintions.html">Amazon ECS
         *        Task Definitions</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerDefinitions(Collection<ContainerDefinition> containerDefinitions);

        /**
         * <p>
         * A list of container definitions in JSON format that describe the different containers that make up your task.
         * For more information about container definition parameters and defaults, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_defintions.html">Amazon ECS Task
         * Definitions</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param containerDefinitions
         *        A list of container definitions in JSON format that describe the different containers that make up
         *        your task. For more information about container definition parameters and defaults, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_defintions.html">Amazon ECS
         *        Task Definitions</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerDefinitions(ContainerDefinition... containerDefinitions);

        /**
         * <p>
         * A list of container definitions in JSON format that describe the different containers that make up your task.
         * For more information about container definition parameters and defaults, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_defintions.html">Amazon ECS Task
         * Definitions</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.ContainerDefinition.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ecs.model.ContainerDefinition#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecs.model.ContainerDefinition.Builder#build()} is called immediately
         * and its result is passed to {@link #containerDefinitions(List<ContainerDefinition>)}.
         * 
         * @param containerDefinitions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecs.model.ContainerDefinition.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #containerDefinitions(java.util.Collection<ContainerDefinition>)
         */
        Builder containerDefinitions(Consumer<ContainerDefinition.Builder>... containerDefinitions);

        /**
         * <p>
         * The name of a family that this task definition is registered to. Up to 255 characters are allowed. Letters
         * (both uppercase and lowercase letters), numbers, hyphens (-), and underscores (_) are allowed.
         * </p>
         * <p>
         * A family groups multiple versions of a task definition. Amazon ECS gives the first task definition that you
         * registered to a family a revision number of 1. Amazon ECS gives sequential revision numbers to each task
         * definition that you add.
         * </p>
         * 
         * @param family
         *        The name of a family that this task definition is registered to. Up to 255 characters are allowed.
         *        Letters (both uppercase and lowercase letters), numbers, hyphens (-), and underscores (_) are
         *        allowed.</p>
         *        <p>
         *        A family groups multiple versions of a task definition. Amazon ECS gives the first task definition
         *        that you registered to a family a revision number of 1. Amazon ECS gives sequential revision numbers
         *        to each task definition that you add.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder family(String family);

        /**
         * <p>
         * The short name or full Amazon Resource Name (ARN) of the Identity and Access Management role that grants
         * containers in the task permission to call Amazon Web Services APIs on your behalf. For informationabout the
         * required IAM roles for Amazon ECS, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/security-ecs-iam-role-overview.html">IAM
         * roles for Amazon ECS</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param taskRoleArn
         *        The short name or full Amazon Resource Name (ARN) of the Identity and Access Management role that
         *        grants containers in the task permission to call Amazon Web Services APIs on your behalf. For
         *        informationabout the required IAM roles for Amazon ECS, see <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/security-ecs-iam-role-overview.html">IAM
         *        roles for Amazon ECS</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskRoleArn(String taskRoleArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the task execution role that grants the Amazon ECS container agent
         * permission to make Amazon Web Services API calls on your behalf. For informationabout the required IAM roles
         * for Amazon ECS, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/security-ecs-iam-role-overview.html">IAM
         * roles for Amazon ECS</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param executionRoleArn
         *        The Amazon Resource Name (ARN) of the task execution role that grants the Amazon ECS container agent
         *        permission to make Amazon Web Services API calls on your behalf. For informationabout the required IAM
         *        roles for Amazon ECS, see <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/security-ecs-iam-role-overview.html">IAM
         *        roles for Amazon ECS</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionRoleArn(String executionRoleArn);

        /**
         * <p>
         * The Docker networking mode to use for the containers in the task. The valid values are <code>none</code>,
         * <code>bridge</code>, <code>awsvpc</code>, and <code>host</code>. If no network mode is specified, the default
         * is <code>bridge</code>.
         * </p>
         * <p>
         * For Amazon ECS tasks on Fargate, the <code>awsvpc</code> network mode is required. For Amazon ECS tasks on
         * Amazon EC2 Linux instances, any network mode can be used. For Amazon ECS tasks on Amazon EC2 Windows
         * instances, <code>&lt;default&gt;</code> or <code>awsvpc</code> can be used. If the network mode is set to
         * <code>none</code>, you cannot specify port mappings in your container definitions, and the tasks containers
         * do not have external connectivity. The <code>host</code> and <code>awsvpc</code> network modes offer the
         * highest networking performance for containers because they use the EC2 network stack instead of the
         * virtualized network stack provided by the <code>bridge</code> mode.
         * </p>
         * <p>
         * With the <code>host</code> and <code>awsvpc</code> network modes, exposed container ports are mapped directly
         * to the corresponding host port (for the <code>host</code> network mode) or the attached elastic network
         * interface port (for the <code>awsvpc</code> network mode), so you cannot take advantage of dynamic host port
         * mappings.
         * </p>
         * <important>
         * <p>
         * When using the <code>host</code> network mode, you should not run containers using the root user (UID 0). It
         * is considered best practice to use a non-root user.
         * </p>
         * </important>
         * <p>
         * If the network mode is <code>awsvpc</code>, the task is allocated an elastic network interface, and you must
         * specify a <a href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_NetworkConfiguration.html">
         * NetworkConfiguration</a> value when you create a service or run a task with the task definition. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html">Task Networking</a>
         * in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <p>
         * If the network mode is <code>host</code>, you cannot run multiple instantiations of the same task on a single
         * container instance when port mappings are used.
         * </p>
         * 
         * @param networkMode
         *        The Docker networking mode to use for the containers in the task. The valid values are
         *        <code>none</code>, <code>bridge</code>, <code>awsvpc</code>, and <code>host</code>. If no network mode
         *        is specified, the default is <code>bridge</code>.</p>
         *        <p>
         *        For Amazon ECS tasks on Fargate, the <code>awsvpc</code> network mode is required. For Amazon ECS
         *        tasks on Amazon EC2 Linux instances, any network mode can be used. For Amazon ECS tasks on Amazon EC2
         *        Windows instances, <code>&lt;default&gt;</code> or <code>awsvpc</code> can be used. If the network
         *        mode is set to <code>none</code>, you cannot specify port mappings in your container definitions, and
         *        the tasks containers do not have external connectivity. The <code>host</code> and <code>awsvpc</code>
         *        network modes offer the highest networking performance for containers because they use the EC2 network
         *        stack instead of the virtualized network stack provided by the <code>bridge</code> mode.
         *        </p>
         *        <p>
         *        With the <code>host</code> and <code>awsvpc</code> network modes, exposed container ports are mapped
         *        directly to the corresponding host port (for the <code>host</code> network mode) or the attached
         *        elastic network interface port (for the <code>awsvpc</code> network mode), so you cannot take
         *        advantage of dynamic host port mappings.
         *        </p>
         *        <important>
         *        <p>
         *        When using the <code>host</code> network mode, you should not run containers using the root user (UID
         *        0). It is considered best practice to use a non-root user.
         *        </p>
         *        </important>
         *        <p>
         *        If the network mode is <code>awsvpc</code>, the task is allocated an elastic network interface, and
         *        you must specify a <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_NetworkConfiguration.html"
         *        >NetworkConfiguration</a> value when you create a service or run a task with the task definition. For
         *        more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html">Task
         *        Networking</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         *        </p>
         *        <p>
         *        If the network mode is <code>host</code>, you cannot run multiple instantiations of the same task on a
         *        single container instance when port mappings are used.
         * @see NetworkMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NetworkMode
         */
        Builder networkMode(String networkMode);

        /**
         * <p>
         * The Docker networking mode to use for the containers in the task. The valid values are <code>none</code>,
         * <code>bridge</code>, <code>awsvpc</code>, and <code>host</code>. If no network mode is specified, the default
         * is <code>bridge</code>.
         * </p>
         * <p>
         * For Amazon ECS tasks on Fargate, the <code>awsvpc</code> network mode is required. For Amazon ECS tasks on
         * Amazon EC2 Linux instances, any network mode can be used. For Amazon ECS tasks on Amazon EC2 Windows
         * instances, <code>&lt;default&gt;</code> or <code>awsvpc</code> can be used. If the network mode is set to
         * <code>none</code>, you cannot specify port mappings in your container definitions, and the tasks containers
         * do not have external connectivity. The <code>host</code> and <code>awsvpc</code> network modes offer the
         * highest networking performance for containers because they use the EC2 network stack instead of the
         * virtualized network stack provided by the <code>bridge</code> mode.
         * </p>
         * <p>
         * With the <code>host</code> and <code>awsvpc</code> network modes, exposed container ports are mapped directly
         * to the corresponding host port (for the <code>host</code> network mode) or the attached elastic network
         * interface port (for the <code>awsvpc</code> network mode), so you cannot take advantage of dynamic host port
         * mappings.
         * </p>
         * <important>
         * <p>
         * When using the <code>host</code> network mode, you should not run containers using the root user (UID 0). It
         * is considered best practice to use a non-root user.
         * </p>
         * </important>
         * <p>
         * If the network mode is <code>awsvpc</code>, the task is allocated an elastic network interface, and you must
         * specify a <a href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_NetworkConfiguration.html">
         * NetworkConfiguration</a> value when you create a service or run a task with the task definition. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html">Task Networking</a>
         * in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <p>
         * If the network mode is <code>host</code>, you cannot run multiple instantiations of the same task on a single
         * container instance when port mappings are used.
         * </p>
         * 
         * @param networkMode
         *        The Docker networking mode to use for the containers in the task. The valid values are
         *        <code>none</code>, <code>bridge</code>, <code>awsvpc</code>, and <code>host</code>. If no network mode
         *        is specified, the default is <code>bridge</code>.</p>
         *        <p>
         *        For Amazon ECS tasks on Fargate, the <code>awsvpc</code> network mode is required. For Amazon ECS
         *        tasks on Amazon EC2 Linux instances, any network mode can be used. For Amazon ECS tasks on Amazon EC2
         *        Windows instances, <code>&lt;default&gt;</code> or <code>awsvpc</code> can be used. If the network
         *        mode is set to <code>none</code>, you cannot specify port mappings in your container definitions, and
         *        the tasks containers do not have external connectivity. The <code>host</code> and <code>awsvpc</code>
         *        network modes offer the highest networking performance for containers because they use the EC2 network
         *        stack instead of the virtualized network stack provided by the <code>bridge</code> mode.
         *        </p>
         *        <p>
         *        With the <code>host</code> and <code>awsvpc</code> network modes, exposed container ports are mapped
         *        directly to the corresponding host port (for the <code>host</code> network mode) or the attached
         *        elastic network interface port (for the <code>awsvpc</code> network mode), so you cannot take
         *        advantage of dynamic host port mappings.
         *        </p>
         *        <important>
         *        <p>
         *        When using the <code>host</code> network mode, you should not run containers using the root user (UID
         *        0). It is considered best practice to use a non-root user.
         *        </p>
         *        </important>
         *        <p>
         *        If the network mode is <code>awsvpc</code>, the task is allocated an elastic network interface, and
         *        you must specify a <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_NetworkConfiguration.html"
         *        >NetworkConfiguration</a> value when you create a service or run a task with the task definition. For
         *        more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html">Task
         *        Networking</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         *        </p>
         *        <p>
         *        If the network mode is <code>host</code>, you cannot run multiple instantiations of the same task on a
         *        single container instance when port mappings are used.
         * @see NetworkMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NetworkMode
         */
        Builder networkMode(NetworkMode networkMode);

        /**
         * <p>
         * The revision of the task in a particular family. The revision is a version number of a task definition in a
         * family. When you register a task definition for the first time, the revision is <code>1</code>. Each time
         * that you register a new revision of a task definition in the same family, the revision value always increases
         * by one. This is even if you deregistered previous revisions in this family.
         * </p>
         * 
         * @param revision
         *        The revision of the task in a particular family. The revision is a version number of a task definition
         *        in a family. When you register a task definition for the first time, the revision is <code>1</code>.
         *        Each time that you register a new revision of a task definition in the same family, the revision value
         *        always increases by one. This is even if you deregistered previous revisions in this family.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder revision(Integer revision);

        /**
         * <p>
         * The list of data volume definitions for the task. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_data_volumes.html">Using data volumes
         * in tasks</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <note>
         * <p>
         * The <code>host</code> and <code>sourcePath</code> parameters aren't supported for tasks run on Fargate.
         * </p>
         * </note>
         * 
         * @param volumes
         *        The list of data volume definitions for the task. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_data_volumes.html">Using data
         *        volumes in tasks</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.</p> <note>
         *        <p>
         *        The <code>host</code> and <code>sourcePath</code> parameters aren't supported for tasks run on
         *        Fargate.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumes(Collection<Volume> volumes);

        /**
         * <p>
         * The list of data volume definitions for the task. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_data_volumes.html">Using data volumes
         * in tasks</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <note>
         * <p>
         * The <code>host</code> and <code>sourcePath</code> parameters aren't supported for tasks run on Fargate.
         * </p>
         * </note>
         * 
         * @param volumes
         *        The list of data volume definitions for the task. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_data_volumes.html">Using data
         *        volumes in tasks</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.</p> <note>
         *        <p>
         *        The <code>host</code> and <code>sourcePath</code> parameters aren't supported for tasks run on
         *        Fargate.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumes(Volume... volumes);

        /**
         * <p>
         * The list of data volume definitions for the task. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_data_volumes.html">Using data volumes
         * in tasks</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <note>
         * <p>
         * The <code>host</code> and <code>sourcePath</code> parameters aren't supported for tasks run on Fargate.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.Volume.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ecs.model.Volume#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ecs.model.Volume.Builder#build()}
         * is called immediately and its result is passed to {@link #volumes(List<Volume>)}.
         * 
         * @param volumes
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ecs.model.Volume.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #volumes(java.util.Collection<Volume>)
         */
        Builder volumes(Consumer<Volume.Builder>... volumes);

        /**
         * <p>
         * The status of the task definition.
         * </p>
         * 
         * @param status
         *        The status of the task definition.
         * @see TaskDefinitionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TaskDefinitionStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the task definition.
         * </p>
         * 
         * @param status
         *        The status of the task definition.
         * @see TaskDefinitionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TaskDefinitionStatus
         */
        Builder status(TaskDefinitionStatus status);

        /**
         * <p>
         * The container instance attributes required by your task. When an Amazon EC2 instance is registered to your
         * cluster, the Amazon ECS container agent assigns some standard attributes to the instance. You can apply
         * custom attributes. These are specified as key-value pairs using the Amazon ECS console or the <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAttributes.html">PutAttributes</a>
         * API. These attributes are used when determining task placement for tasks hosted on Amazon EC2 instances. For
         * more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html#attributes"
         * >Attributes</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <note>
         * <p>
         * This parameter isn't supported for tasks run on Fargate.
         * </p>
         * </note>
         * 
         * @param requiresAttributes
         *        The container instance attributes required by your task. When an Amazon EC2 instance is registered to
         *        your cluster, the Amazon ECS container agent assigns some standard attributes to the instance. You can
         *        apply custom attributes. These are specified as key-value pairs using the Amazon ECS console or the <a
         *        href
         *        ="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAttributes.html">PutAttributes</a>
         *        API. These attributes are used when determining task placement for tasks hosted on Amazon EC2
         *        instances. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html#attributes"
         *        >Attributes</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.</p> <note>
         *        <p>
         *        This parameter isn't supported for tasks run on Fargate.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requiresAttributes(Collection<Attribute> requiresAttributes);

        /**
         * <p>
         * The container instance attributes required by your task. When an Amazon EC2 instance is registered to your
         * cluster, the Amazon ECS container agent assigns some standard attributes to the instance. You can apply
         * custom attributes. These are specified as key-value pairs using the Amazon ECS console or the <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAttributes.html">PutAttributes</a>
         * API. These attributes are used when determining task placement for tasks hosted on Amazon EC2 instances. For
         * more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html#attributes"
         * >Attributes</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <note>
         * <p>
         * This parameter isn't supported for tasks run on Fargate.
         * </p>
         * </note>
         * 
         * @param requiresAttributes
         *        The container instance attributes required by your task. When an Amazon EC2 instance is registered to
         *        your cluster, the Amazon ECS container agent assigns some standard attributes to the instance. You can
         *        apply custom attributes. These are specified as key-value pairs using the Amazon ECS console or the <a
         *        href
         *        ="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAttributes.html">PutAttributes</a>
         *        API. These attributes are used when determining task placement for tasks hosted on Amazon EC2
         *        instances. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html#attributes"
         *        >Attributes</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.</p> <note>
         *        <p>
         *        This parameter isn't supported for tasks run on Fargate.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requiresAttributes(Attribute... requiresAttributes);

        /**
         * <p>
         * The container instance attributes required by your task. When an Amazon EC2 instance is registered to your
         * cluster, the Amazon ECS container agent assigns some standard attributes to the instance. You can apply
         * custom attributes. These are specified as key-value pairs using the Amazon ECS console or the <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAttributes.html">PutAttributes</a>
         * API. These attributes are used when determining task placement for tasks hosted on Amazon EC2 instances. For
         * more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html#attributes"
         * >Attributes</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <note>
         * <p>
         * This parameter isn't supported for tasks run on Fargate.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.Attribute.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.ecs.model.Attribute#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecs.model.Attribute.Builder#build()} is called immediately and its
         * result is passed to {@link #requiresAttributes(List<Attribute>)}.
         * 
         * @param requiresAttributes
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecs.model.Attribute.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #requiresAttributes(java.util.Collection<Attribute>)
         */
        Builder requiresAttributes(Consumer<Attribute.Builder>... requiresAttributes);

        /**
         * <p>
         * An array of placement constraint objects to use for tasks.
         * </p>
         * <note>
         * <p>
         * This parameter isn't supported for tasks run on Fargate.
         * </p>
         * </note>
         * 
         * @param placementConstraints
         *        An array of placement constraint objects to use for tasks.</p> <note>
         *        <p>
         *        This parameter isn't supported for tasks run on Fargate.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder placementConstraints(Collection<TaskDefinitionPlacementConstraint> placementConstraints);

        /**
         * <p>
         * An array of placement constraint objects to use for tasks.
         * </p>
         * <note>
         * <p>
         * This parameter isn't supported for tasks run on Fargate.
         * </p>
         * </note>
         * 
         * @param placementConstraints
         *        An array of placement constraint objects to use for tasks.</p> <note>
         *        <p>
         *        This parameter isn't supported for tasks run on Fargate.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder placementConstraints(TaskDefinitionPlacementConstraint... placementConstraints);

        /**
         * <p>
         * An array of placement constraint objects to use for tasks.
         * </p>
         * <note>
         * <p>
         * This parameter isn't supported for tasks run on Fargate.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.TaskDefinitionPlacementConstraint.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.ecs.model.TaskDefinitionPlacementConstraint#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecs.model.TaskDefinitionPlacementConstraint.Builder#build()} is called
         * immediately and its result is passed to {@link
         * #placementConstraints(List<TaskDefinitionPlacementConstraint>)}.
         * 
         * @param placementConstraints
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecs.model.TaskDefinitionPlacementConstraint.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #placementConstraints(java.util.Collection<TaskDefinitionPlacementConstraint>)
         */
        Builder placementConstraints(Consumer<TaskDefinitionPlacementConstraint.Builder>... placementConstraints);

        /**
         * <p>
         * Amazon ECS validates the task definition parameters with those supported by the launch type. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch
         * types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param compatibilities
         *        Amazon ECS validates the task definition parameters with those supported by the launch type. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch
         *        types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder compatibilitiesWithStrings(Collection<String> compatibilities);

        /**
         * <p>
         * Amazon ECS validates the task definition parameters with those supported by the launch type. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch
         * types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param compatibilities
         *        Amazon ECS validates the task definition parameters with those supported by the launch type. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch
         *        types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder compatibilitiesWithStrings(String... compatibilities);

        /**
         * <p>
         * Amazon ECS validates the task definition parameters with those supported by the launch type. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch
         * types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param compatibilities
         *        Amazon ECS validates the task definition parameters with those supported by the launch type. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch
         *        types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder compatibilities(Collection<Compatibility> compatibilities);

        /**
         * <p>
         * Amazon ECS validates the task definition parameters with those supported by the launch type. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch
         * types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param compatibilities
         *        Amazon ECS validates the task definition parameters with those supported by the launch type. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch
         *        types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder compatibilities(Compatibility... compatibilities);

        /**
         * <p>
         * The operating system that your task definitions are running on. A platform family is specified only for tasks
         * using the Fargate launch type.
         * </p>
         * <p>
         * When you specify a task in a service, this value must match the <code>runtimePlatform</code> value of the
         * service.
         * </p>
         * 
         * @param runtimePlatform
         *        The operating system that your task definitions are running on. A platform family is specified only
         *        for tasks using the Fargate launch type. </p>
         *        <p>
         *        When you specify a task in a service, this value must match the <code>runtimePlatform</code> value of
         *        the service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder runtimePlatform(RuntimePlatform runtimePlatform);

        /**
         * <p>
         * The operating system that your task definitions are running on. A platform family is specified only for tasks
         * using the Fargate launch type.
         * </p>
         * <p>
         * When you specify a task in a service, this value must match the <code>runtimePlatform</code> value of the
         * service.
         * </p>
         * This is a convenience method that creates an instance of the {@link RuntimePlatform.Builder} avoiding the
         * need to create one manually via {@link RuntimePlatform#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RuntimePlatform.Builder#build()} is called immediately and its
         * result is passed to {@link #runtimePlatform(RuntimePlatform)}.
         * 
         * @param runtimePlatform
         *        a consumer that will call methods on {@link RuntimePlatform.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #runtimePlatform(RuntimePlatform)
         */
        default Builder runtimePlatform(Consumer<RuntimePlatform.Builder> runtimePlatform) {
            return runtimePlatform(RuntimePlatform.builder().applyMutation(runtimePlatform).build());
        }

        /**
         * <p>
         * The task launch types the task definition was validated against. The valid values are <code>EC2</code>,
         * <code>FARGATE</code>, and <code>EXTERNAL</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch
         * types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param requiresCompatibilities
         *        The task launch types the task definition was validated against. The valid values are <code>EC2</code>
         *        , <code>FARGATE</code>, and <code>EXTERNAL</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch
         *        types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requiresCompatibilitiesWithStrings(Collection<String> requiresCompatibilities);

        /**
         * <p>
         * The task launch types the task definition was validated against. The valid values are <code>EC2</code>,
         * <code>FARGATE</code>, and <code>EXTERNAL</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch
         * types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param requiresCompatibilities
         *        The task launch types the task definition was validated against. The valid values are <code>EC2</code>
         *        , <code>FARGATE</code>, and <code>EXTERNAL</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch
         *        types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requiresCompatibilitiesWithStrings(String... requiresCompatibilities);

        /**
         * <p>
         * The task launch types the task definition was validated against. The valid values are <code>EC2</code>,
         * <code>FARGATE</code>, and <code>EXTERNAL</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch
         * types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param requiresCompatibilities
         *        The task launch types the task definition was validated against. The valid values are <code>EC2</code>
         *        , <code>FARGATE</code>, and <code>EXTERNAL</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch
         *        types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requiresCompatibilities(Collection<Compatibility> requiresCompatibilities);

        /**
         * <p>
         * The task launch types the task definition was validated against. The valid values are <code>EC2</code>,
         * <code>FARGATE</code>, and <code>EXTERNAL</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch
         * types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param requiresCompatibilities
         *        The task launch types the task definition was validated against. The valid values are <code>EC2</code>
         *        , <code>FARGATE</code>, and <code>EXTERNAL</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html">Amazon ECS launch
         *        types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requiresCompatibilities(Compatibility... requiresCompatibilities);

        /**
         * <p>
         * The number of <code>cpu</code> units used by the task. If you use the EC2 launch type, this field is
         * optional. Any value can be used. If you use the Fargate launch type, this field is required. You must use one
         * of the following values. The value that you choose determines your range of valid values for the
         * <code>memory</code> parameter.
         * </p>
         * <p>
         * If you use the EC2 launch type, this field is optional. Supported values are between <code>128</code> CPU
         * units (<code>0.125</code> vCPUs) and <code>10240</code> CPU units (<code>10</code> vCPUs).
         * </p>
         * <p>
         * The CPU units cannot be less than 1 vCPU when you use Windows containers on Fargate.
         * </p>
         * <ul>
         * <li>
         * <p>
         * 256 (.25 vCPU) - Available <code>memory</code> values: 512 (0.5 GB), 1024 (1 GB), 2048 (2 GB)
         * </p>
         * </li>
         * <li>
         * <p>
         * 512 (.5 vCPU) - Available <code>memory</code> values: 1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB)
         * </p>
         * </li>
         * <li>
         * <p>
         * 1024 (1 vCPU) - Available <code>memory</code> values: 2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB),
         * 6144 (6 GB), 7168 (7 GB), 8192 (8 GB)
         * </p>
         * </li>
         * <li>
         * <p>
         * 2048 (2 vCPU) - Available <code>memory</code> values: 4096 (4 GB) and 16384 (16 GB) in increments of 1024 (1
         * GB)
         * </p>
         * </li>
         * <li>
         * <p>
         * 4096 (4 vCPU) - Available <code>memory</code> values: 8192 (8 GB) and 30720 (30 GB) in increments of 1024 (1
         * GB)
         * </p>
         * </li>
         * <li>
         * <p>
         * 8192 (8 vCPU) - Available <code>memory</code> values: 16 GB and 60 GB in 4 GB increments
         * </p>
         * <p>
         * This option requires Linux platform <code>1.4.0</code> or later.
         * </p>
         * </li>
         * <li>
         * <p>
         * 16384 (16vCPU) - Available <code>memory</code> values: 32GB and 120 GB in 8 GB increments
         * </p>
         * <p>
         * This option requires Linux platform <code>1.4.0</code> or later.
         * </p>
         * </li>
         * </ul>
         * 
         * @param cpu
         *        The number of <code>cpu</code> units used by the task. If you use the EC2 launch type, this field is
         *        optional. Any value can be used. If you use the Fargate launch type, this field is required. You must
         *        use one of the following values. The value that you choose determines your range of valid values for
         *        the <code>memory</code> parameter.</p>
         *        <p>
         *        If you use the EC2 launch type, this field is optional. Supported values are between <code>128</code>
         *        CPU units (<code>0.125</code> vCPUs) and <code>10240</code> CPU units (<code>10</code> vCPUs).
         *        </p>
         *        <p>
         *        The CPU units cannot be less than 1 vCPU when you use Windows containers on Fargate.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        256 (.25 vCPU) - Available <code>memory</code> values: 512 (0.5 GB), 1024 (1 GB), 2048 (2 GB)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        512 (.5 vCPU) - Available <code>memory</code> values: 1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4
         *        GB)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        1024 (1 vCPU) - Available <code>memory</code> values: 2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5
         *        GB), 6144 (6 GB), 7168 (7 GB), 8192 (8 GB)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        2048 (2 vCPU) - Available <code>memory</code> values: 4096 (4 GB) and 16384 (16 GB) in increments of
         *        1024 (1 GB)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        4096 (4 vCPU) - Available <code>memory</code> values: 8192 (8 GB) and 30720 (30 GB) in increments of
         *        1024 (1 GB)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        8192 (8 vCPU) - Available <code>memory</code> values: 16 GB and 60 GB in 4 GB increments
         *        </p>
         *        <p>
         *        This option requires Linux platform <code>1.4.0</code> or later.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        16384 (16vCPU) - Available <code>memory</code> values: 32GB and 120 GB in 8 GB increments
         *        </p>
         *        <p>
         *        This option requires Linux platform <code>1.4.0</code> or later.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cpu(String cpu);

        /**
         * <p>
         * The amount (in MiB) of memory used by the task.
         * </p>
         * <p>
         * If your tasks runs on Amazon EC2 instances, you must specify either a task-level memory value or a
         * container-level memory value. This field is optional and any value can be used. If a task-level memory value
         * is specified, the container-level memory value is optional. For more information regarding container-level
         * memory and memory reservation, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html"
         * >ContainerDefinition</a>.
         * </p>
         * <p>
         * If your tasks runs on Fargate, this field is required. You must use one of the following values. The value
         * you choose determines your range of valid values for the <code>cpu</code> parameter.
         * </p>
         * <ul>
         * <li>
         * <p>
         * 512 (0.5 GB), 1024 (1 GB), 2048 (2 GB) - Available <code>cpu</code> values: 256 (.25 vCPU)
         * </p>
         * </li>
         * <li>
         * <p>
         * 1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB) - Available <code>cpu</code> values: 512 (.5 vCPU)
         * </p>
         * </li>
         * <li>
         * <p>
         * 2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB), 6144 (6 GB), 7168 (7 GB), 8192 (8 GB) - Available
         * <code>cpu</code> values: 1024 (1 vCPU)
         * </p>
         * </li>
         * <li>
         * <p>
         * Between 4096 (4 GB) and 16384 (16 GB) in increments of 1024 (1 GB) - Available <code>cpu</code> values: 2048
         * (2 vCPU)
         * </p>
         * </li>
         * <li>
         * <p>
         * Between 8192 (8 GB) and 30720 (30 GB) in increments of 1024 (1 GB) - Available <code>cpu</code> values: 4096
         * (4 vCPU)
         * </p>
         * </li>
         * <li>
         * <p>
         * Between 16 GB and 60 GB in 4 GB increments - Available <code>cpu</code> values: 8192 (8 vCPU)
         * </p>
         * <p>
         * This option requires Linux platform <code>1.4.0</code> or later.
         * </p>
         * </li>
         * <li>
         * <p>
         * Between 32GB and 120 GB in 8 GB increments - Available <code>cpu</code> values: 16384 (16 vCPU)
         * </p>
         * <p>
         * This option requires Linux platform <code>1.4.0</code> or later.
         * </p>
         * </li>
         * </ul>
         * 
         * @param memory
         *        The amount (in MiB) of memory used by the task.</p>
         *        <p>
         *        If your tasks runs on Amazon EC2 instances, you must specify either a task-level memory value or a
         *        container-level memory value. This field is optional and any value can be used. If a task-level memory
         *        value is specified, the container-level memory value is optional. For more information regarding
         *        container-level memory and memory reservation, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html"
         *        >ContainerDefinition</a>.
         *        </p>
         *        <p>
         *        If your tasks runs on Fargate, this field is required. You must use one of the following values. The
         *        value you choose determines your range of valid values for the <code>cpu</code> parameter.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        512 (0.5 GB), 1024 (1 GB), 2048 (2 GB) - Available <code>cpu</code> values: 256 (.25 vCPU)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB) - Available <code>cpu</code> values: 512 (.5 vCPU)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB), 6144 (6 GB), 7168 (7 GB), 8192 (8 GB) - Available
         *        <code>cpu</code> values: 1024 (1 vCPU)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Between 4096 (4 GB) and 16384 (16 GB) in increments of 1024 (1 GB) - Available <code>cpu</code>
         *        values: 2048 (2 vCPU)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Between 8192 (8 GB) and 30720 (30 GB) in increments of 1024 (1 GB) - Available <code>cpu</code>
         *        values: 4096 (4 vCPU)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Between 16 GB and 60 GB in 4 GB increments - Available <code>cpu</code> values: 8192 (8 vCPU)
         *        </p>
         *        <p>
         *        This option requires Linux platform <code>1.4.0</code> or later.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Between 32GB and 120 GB in 8 GB increments - Available <code>cpu</code> values: 16384 (16 vCPU)
         *        </p>
         *        <p>
         *        This option requires Linux platform <code>1.4.0</code> or later.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memory(String memory);

        /**
         * <p>
         * The Elastic Inference accelerator that's associated with the task.
         * </p>
         * 
         * @param inferenceAccelerators
         *        The Elastic Inference accelerator that's associated with the task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inferenceAccelerators(Collection<InferenceAccelerator> inferenceAccelerators);

        /**
         * <p>
         * The Elastic Inference accelerator that's associated with the task.
         * </p>
         * 
         * @param inferenceAccelerators
         *        The Elastic Inference accelerator that's associated with the task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inferenceAccelerators(InferenceAccelerator... inferenceAccelerators);

        /**
         * <p>
         * The Elastic Inference accelerator that's associated with the task.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.InferenceAccelerator.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.ecs.model.InferenceAccelerator#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecs.model.InferenceAccelerator.Builder#build()} is called immediately
         * and its result is passed to {@link #inferenceAccelerators(List<InferenceAccelerator>)}.
         * 
         * @param inferenceAccelerators
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecs.model.InferenceAccelerator.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #inferenceAccelerators(java.util.Collection<InferenceAccelerator>)
         */
        Builder inferenceAccelerators(Consumer<InferenceAccelerator.Builder>... inferenceAccelerators);

        /**
         * <p>
         * The process namespace to use for the containers in the task. The valid values are <code>host</code> or
         * <code>task</code>. On Fargate for Linux containers, the only valid value is <code>task</code>. For example,
         * monitoring sidecars might need <code>pidMode</code> to access information about other containers running in
         * the same task.
         * </p>
         * <p>
         * If <code>host</code> is specified, all containers within the tasks that specified the <code>host</code> PID
         * mode on the same container instance share the same process namespace with the host Amazon EC2 instance.
         * </p>
         * <p>
         * If <code>task</code> is specified, all containers within the specified task share the same process namespace.
         * </p>
         * <p>
         * If no value is specified, the default is a private namespace for each container.
         * </p>
         * <p>
         * If the <code>host</code> PID mode is used, there's a heightened risk of undesired process namespace exposure.
         * </p>
         * <note>
         * <p>
         * This parameter is not supported for Windows containers.
         * </p>
         * </note> <note>
         * <p>
         * This parameter is only supported for tasks that are hosted on Fargate if the tasks are using platform version
         * <code>1.4.0</code> or later (Linux). This isn't supported for Windows containers on Fargate.
         * </p>
         * </note>
         * 
         * @param pidMode
         *        The process namespace to use for the containers in the task. The valid values are <code>host</code> or
         *        <code>task</code>. On Fargate for Linux containers, the only valid value is <code>task</code>. For
         *        example, monitoring sidecars might need <code>pidMode</code> to access information about other
         *        containers running in the same task.</p>
         *        <p>
         *        If <code>host</code> is specified, all containers within the tasks that specified the
         *        <code>host</code> PID mode on the same container instance share the same process namespace with the
         *        host Amazon EC2 instance.
         *        </p>
         *        <p>
         *        If <code>task</code> is specified, all containers within the specified task share the same process
         *        namespace.
         *        </p>
         *        <p>
         *        If no value is specified, the default is a private namespace for each container.
         *        </p>
         *        <p>
         *        If the <code>host</code> PID mode is used, there's a heightened risk of undesired process namespace
         *        exposure.
         *        </p>
         *        <note>
         *        <p>
         *        This parameter is not supported for Windows containers.
         *        </p>
         *        </note> <note>
         *        <p>
         *        This parameter is only supported for tasks that are hosted on Fargate if the tasks are using platform
         *        version <code>1.4.0</code> or later (Linux). This isn't supported for Windows containers on Fargate.
         *        </p>
         * @see PidMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PidMode
         */
        Builder pidMode(String pidMode);

        /**
         * <p>
         * The process namespace to use for the containers in the task. The valid values are <code>host</code> or
         * <code>task</code>. On Fargate for Linux containers, the only valid value is <code>task</code>. For example,
         * monitoring sidecars might need <code>pidMode</code> to access information about other containers running in
         * the same task.
         * </p>
         * <p>
         * If <code>host</code> is specified, all containers within the tasks that specified the <code>host</code> PID
         * mode on the same container instance share the same process namespace with the host Amazon EC2 instance.
         * </p>
         * <p>
         * If <code>task</code> is specified, all containers within the specified task share the same process namespace.
         * </p>
         * <p>
         * If no value is specified, the default is a private namespace for each container.
         * </p>
         * <p>
         * If the <code>host</code> PID mode is used, there's a heightened risk of undesired process namespace exposure.
         * </p>
         * <note>
         * <p>
         * This parameter is not supported for Windows containers.
         * </p>
         * </note> <note>
         * <p>
         * This parameter is only supported for tasks that are hosted on Fargate if the tasks are using platform version
         * <code>1.4.0</code> or later (Linux). This isn't supported for Windows containers on Fargate.
         * </p>
         * </note>
         * 
         * @param pidMode
         *        The process namespace to use for the containers in the task. The valid values are <code>host</code> or
         *        <code>task</code>. On Fargate for Linux containers, the only valid value is <code>task</code>. For
         *        example, monitoring sidecars might need <code>pidMode</code> to access information about other
         *        containers running in the same task.</p>
         *        <p>
         *        If <code>host</code> is specified, all containers within the tasks that specified the
         *        <code>host</code> PID mode on the same container instance share the same process namespace with the
         *        host Amazon EC2 instance.
         *        </p>
         *        <p>
         *        If <code>task</code> is specified, all containers within the specified task share the same process
         *        namespace.
         *        </p>
         *        <p>
         *        If no value is specified, the default is a private namespace for each container.
         *        </p>
         *        <p>
         *        If the <code>host</code> PID mode is used, there's a heightened risk of undesired process namespace
         *        exposure.
         *        </p>
         *        <note>
         *        <p>
         *        This parameter is not supported for Windows containers.
         *        </p>
         *        </note> <note>
         *        <p>
         *        This parameter is only supported for tasks that are hosted on Fargate if the tasks are using platform
         *        version <code>1.4.0</code> or later (Linux). This isn't supported for Windows containers on Fargate.
         *        </p>
         * @see PidMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PidMode
         */
        Builder pidMode(PidMode pidMode);

        /**
         * <p>
         * The IPC resource namespace to use for the containers in the task. The valid values are <code>host</code>,
         * <code>task</code>, or <code>none</code>. If <code>host</code> is specified, then all containers within the
         * tasks that specified the <code>host</code> IPC mode on the same container instance share the same IPC
         * resources with the host Amazon EC2 instance. If <code>task</code> is specified, all containers within the
         * specified task share the same IPC resources. If <code>none</code> is specified, then IPC resources within the
         * containers of a task are private and not shared with other containers in a task or on the container instance.
         * If no value is specified, then the IPC resource namespace sharing depends on the Docker daemon setting on the
         * container instance.
         * </p>
         * <p>
         * If the <code>host</code> IPC mode is used, be aware that there is a heightened risk of undesired IPC
         * namespace expose.
         * </p>
         * <p>
         * If you are setting namespaced kernel parameters using <code>systemControls</code> for the containers in the
         * task, the following will apply to your IPC resource namespace. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html">System
         * Controls</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For tasks that use the <code>host</code> IPC mode, IPC namespace related <code>systemControls</code> are not
         * supported.
         * </p>
         * </li>
         * <li>
         * <p>
         * For tasks that use the <code>task</code> IPC mode, IPC namespace related <code>systemControls</code> will
         * apply to all containers within a task.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * This parameter is not supported for Windows containers or tasks run on Fargate.
         * </p>
         * </note>
         * 
         * @param ipcMode
         *        The IPC resource namespace to use for the containers in the task. The valid values are
         *        <code>host</code>, <code>task</code>, or <code>none</code>. If <code>host</code> is specified, then
         *        all containers within the tasks that specified the <code>host</code> IPC mode on the same container
         *        instance share the same IPC resources with the host Amazon EC2 instance. If <code>task</code> is
         *        specified, all containers within the specified task share the same IPC resources. If <code>none</code>
         *        is specified, then IPC resources within the containers of a task are private and not shared with other
         *        containers in a task or on the container instance. If no value is specified, then the IPC resource
         *        namespace sharing depends on the Docker daemon setting on the container instance.</p>
         *        <p>
         *        If the <code>host</code> IPC mode is used, be aware that there is a heightened risk of undesired IPC
         *        namespace expose.
         *        </p>
         *        <p>
         *        If you are setting namespaced kernel parameters using <code>systemControls</code> for the containers
         *        in the task, the following will apply to your IPC resource namespace. For more information, see <a
         *        href
         *        ="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html">System
         *        Controls</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For tasks that use the <code>host</code> IPC mode, IPC namespace related <code>systemControls</code>
         *        are not supported.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For tasks that use the <code>task</code> IPC mode, IPC namespace related <code>systemControls</code>
         *        will apply to all containers within a task.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        This parameter is not supported for Windows containers or tasks run on Fargate.
         *        </p>
         * @see IpcMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpcMode
         */
        Builder ipcMode(String ipcMode);

        /**
         * <p>
         * The IPC resource namespace to use for the containers in the task. The valid values are <code>host</code>,
         * <code>task</code>, or <code>none</code>. If <code>host</code> is specified, then all containers within the
         * tasks that specified the <code>host</code> IPC mode on the same container instance share the same IPC
         * resources with the host Amazon EC2 instance. If <code>task</code> is specified, all containers within the
         * specified task share the same IPC resources. If <code>none</code> is specified, then IPC resources within the
         * containers of a task are private and not shared with other containers in a task or on the container instance.
         * If no value is specified, then the IPC resource namespace sharing depends on the Docker daemon setting on the
         * container instance.
         * </p>
         * <p>
         * If the <code>host</code> IPC mode is used, be aware that there is a heightened risk of undesired IPC
         * namespace expose.
         * </p>
         * <p>
         * If you are setting namespaced kernel parameters using <code>systemControls</code> for the containers in the
         * task, the following will apply to your IPC resource namespace. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html">System
         * Controls</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For tasks that use the <code>host</code> IPC mode, IPC namespace related <code>systemControls</code> are not
         * supported.
         * </p>
         * </li>
         * <li>
         * <p>
         * For tasks that use the <code>task</code> IPC mode, IPC namespace related <code>systemControls</code> will
         * apply to all containers within a task.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * This parameter is not supported for Windows containers or tasks run on Fargate.
         * </p>
         * </note>
         * 
         * @param ipcMode
         *        The IPC resource namespace to use for the containers in the task. The valid values are
         *        <code>host</code>, <code>task</code>, or <code>none</code>. If <code>host</code> is specified, then
         *        all containers within the tasks that specified the <code>host</code> IPC mode on the same container
         *        instance share the same IPC resources with the host Amazon EC2 instance. If <code>task</code> is
         *        specified, all containers within the specified task share the same IPC resources. If <code>none</code>
         *        is specified, then IPC resources within the containers of a task are private and not shared with other
         *        containers in a task or on the container instance. If no value is specified, then the IPC resource
         *        namespace sharing depends on the Docker daemon setting on the container instance.</p>
         *        <p>
         *        If the <code>host</code> IPC mode is used, be aware that there is a heightened risk of undesired IPC
         *        namespace expose.
         *        </p>
         *        <p>
         *        If you are setting namespaced kernel parameters using <code>systemControls</code> for the containers
         *        in the task, the following will apply to your IPC resource namespace. For more information, see <a
         *        href
         *        ="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html">System
         *        Controls</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For tasks that use the <code>host</code> IPC mode, IPC namespace related <code>systemControls</code>
         *        are not supported.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For tasks that use the <code>task</code> IPC mode, IPC namespace related <code>systemControls</code>
         *        will apply to all containers within a task.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        This parameter is not supported for Windows containers or tasks run on Fargate.
         *        </p>
         * @see IpcMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpcMode
         */
        Builder ipcMode(IpcMode ipcMode);

        /**
         * <p>
         * The configuration details for the App Mesh proxy.
         * </p>
         * <p>
         * Your Amazon ECS container instances require at least version 1.26.0 of the container agent and at least
         * version 1.26.0-1 of the <code>ecs-init</code> package to use a proxy configuration. If your container
         * instances are launched from the Amazon ECS optimized AMI version <code>20190301</code> or later, they contain
         * the required versions of the container agent and <code>ecs-init</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html">Amazon
         * ECS-optimized Linux AMI</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param proxyConfiguration
         *        The configuration details for the App Mesh proxy.</p>
         *        <p>
         *        Your Amazon ECS container instances require at least version 1.26.0 of the container agent and at
         *        least version 1.26.0-1 of the <code>ecs-init</code> package to use a proxy configuration. If your
         *        container instances are launched from the Amazon ECS optimized AMI version <code>20190301</code> or
         *        later, they contain the required versions of the container agent and <code>ecs-init</code>. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html">Amazon
         *        ECS-optimized Linux AMI</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder proxyConfiguration(ProxyConfiguration proxyConfiguration);

        /**
         * <p>
         * The configuration details for the App Mesh proxy.
         * </p>
         * <p>
         * Your Amazon ECS container instances require at least version 1.26.0 of the container agent and at least
         * version 1.26.0-1 of the <code>ecs-init</code> package to use a proxy configuration. If your container
         * instances are launched from the Amazon ECS optimized AMI version <code>20190301</code> or later, they contain
         * the required versions of the container agent and <code>ecs-init</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html">Amazon
         * ECS-optimized Linux AMI</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link ProxyConfiguration.Builder} avoiding the
         * need to create one manually via {@link ProxyConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ProxyConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #proxyConfiguration(ProxyConfiguration)}.
         * 
         * @param proxyConfiguration
         *        a consumer that will call methods on {@link ProxyConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #proxyConfiguration(ProxyConfiguration)
         */
        default Builder proxyConfiguration(Consumer<ProxyConfiguration.Builder> proxyConfiguration) {
            return proxyConfiguration(ProxyConfiguration.builder().applyMutation(proxyConfiguration).build());
        }

        /**
         * <p>
         * The Unix timestamp for the time when the task definition was registered.
         * </p>
         * 
         * @param registeredAt
         *        The Unix timestamp for the time when the task definition was registered.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder registeredAt(Instant registeredAt);

        /**
         * <p>
         * The Unix timestamp for the time when the task definition was deregistered.
         * </p>
         * 
         * @param deregisteredAt
         *        The Unix timestamp for the time when the task definition was deregistered.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deregisteredAt(Instant deregisteredAt);

        /**
         * <p>
         * The principal that registered the task definition.
         * </p>
         * 
         * @param registeredBy
         *        The principal that registered the task definition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder registeredBy(String registeredBy);

        /**
         * <p>
         * The ephemeral storage settings to use for tasks run with the task definition.
         * </p>
         * 
         * @param ephemeralStorage
         *        The ephemeral storage settings to use for tasks run with the task definition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ephemeralStorage(EphemeralStorage ephemeralStorage);

        /**
         * <p>
         * The ephemeral storage settings to use for tasks run with the task definition.
         * </p>
         * This is a convenience method that creates an instance of the {@link EphemeralStorage.Builder} avoiding the
         * need to create one manually via {@link EphemeralStorage#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EphemeralStorage.Builder#build()} is called immediately and its
         * result is passed to {@link #ephemeralStorage(EphemeralStorage)}.
         * 
         * @param ephemeralStorage
         *        a consumer that will call methods on {@link EphemeralStorage.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ephemeralStorage(EphemeralStorage)
         */
        default Builder ephemeralStorage(Consumer<EphemeralStorage.Builder> ephemeralStorage) {
            return ephemeralStorage(EphemeralStorage.builder().applyMutation(ephemeralStorage).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String taskDefinitionArn;

        private List<ContainerDefinition> containerDefinitions = DefaultSdkAutoConstructList.getInstance();

        private String family;

        private String taskRoleArn;

        private String executionRoleArn;

        private String networkMode;

        private Integer revision;

        private List<Volume> volumes = DefaultSdkAutoConstructList.getInstance();

        private String status;

        private List<Attribute> requiresAttributes = DefaultSdkAutoConstructList.getInstance();

        private List<TaskDefinitionPlacementConstraint> placementConstraints = DefaultSdkAutoConstructList.getInstance();

        private List<String> compatibilities = DefaultSdkAutoConstructList.getInstance();

        private RuntimePlatform runtimePlatform;

        private List<String> requiresCompatibilities = DefaultSdkAutoConstructList.getInstance();

        private String cpu;

        private String memory;

        private List<InferenceAccelerator> inferenceAccelerators = DefaultSdkAutoConstructList.getInstance();

        private String pidMode;

        private String ipcMode;

        private ProxyConfiguration proxyConfiguration;

        private Instant registeredAt;

        private Instant deregisteredAt;

        private String registeredBy;

        private EphemeralStorage ephemeralStorage;

        private BuilderImpl() {
        }

        private BuilderImpl(TaskDefinition model) {
            taskDefinitionArn(model.taskDefinitionArn);
            containerDefinitions(model.containerDefinitions);
            family(model.family);
            taskRoleArn(model.taskRoleArn);
            executionRoleArn(model.executionRoleArn);
            networkMode(model.networkMode);
            revision(model.revision);
            volumes(model.volumes);
            status(model.status);
            requiresAttributes(model.requiresAttributes);
            placementConstraints(model.placementConstraints);
            compatibilitiesWithStrings(model.compatibilities);
            runtimePlatform(model.runtimePlatform);
            requiresCompatibilitiesWithStrings(model.requiresCompatibilities);
            cpu(model.cpu);
            memory(model.memory);
            inferenceAccelerators(model.inferenceAccelerators);
            pidMode(model.pidMode);
            ipcMode(model.ipcMode);
            proxyConfiguration(model.proxyConfiguration);
            registeredAt(model.registeredAt);
            deregisteredAt(model.deregisteredAt);
            registeredBy(model.registeredBy);
            ephemeralStorage(model.ephemeralStorage);
        }

        public final String getTaskDefinitionArn() {
            return taskDefinitionArn;
        }

        public final void setTaskDefinitionArn(String taskDefinitionArn) {
            this.taskDefinitionArn = taskDefinitionArn;
        }

        @Override
        public final Builder taskDefinitionArn(String taskDefinitionArn) {
            this.taskDefinitionArn = taskDefinitionArn;
            return this;
        }

        public final List<ContainerDefinition.Builder> getContainerDefinitions() {
            List<ContainerDefinition.Builder> result = ContainerDefinitionsCopier.copyToBuilder(this.containerDefinitions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setContainerDefinitions(Collection<ContainerDefinition.BuilderImpl> containerDefinitions) {
            this.containerDefinitions = ContainerDefinitionsCopier.copyFromBuilder(containerDefinitions);
        }

        @Override
        public final Builder containerDefinitions(Collection<ContainerDefinition> containerDefinitions) {
            this.containerDefinitions = ContainerDefinitionsCopier.copy(containerDefinitions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder containerDefinitions(ContainerDefinition... containerDefinitions) {
            containerDefinitions(Arrays.asList(containerDefinitions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder containerDefinitions(Consumer<ContainerDefinition.Builder>... containerDefinitions) {
            containerDefinitions(Stream.of(containerDefinitions).map(c -> ContainerDefinition.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getFamily() {
            return family;
        }

        public final void setFamily(String family) {
            this.family = family;
        }

        @Override
        public final Builder family(String family) {
            this.family = family;
            return this;
        }

        public final String getTaskRoleArn() {
            return taskRoleArn;
        }

        public final void setTaskRoleArn(String taskRoleArn) {
            this.taskRoleArn = taskRoleArn;
        }

        @Override
        public final Builder taskRoleArn(String taskRoleArn) {
            this.taskRoleArn = taskRoleArn;
            return this;
        }

        public final String getExecutionRoleArn() {
            return executionRoleArn;
        }

        public final void setExecutionRoleArn(String executionRoleArn) {
            this.executionRoleArn = executionRoleArn;
        }

        @Override
        public final Builder executionRoleArn(String executionRoleArn) {
            this.executionRoleArn = executionRoleArn;
            return this;
        }

        public final String getNetworkMode() {
            return networkMode;
        }

        public final void setNetworkMode(String networkMode) {
            this.networkMode = networkMode;
        }

        @Override
        public final Builder networkMode(String networkMode) {
            this.networkMode = networkMode;
            return this;
        }

        @Override
        public final Builder networkMode(NetworkMode networkMode) {
            this.networkMode(networkMode == null ? null : networkMode.toString());
            return this;
        }

        public final Integer getRevision() {
            return revision;
        }

        public final void setRevision(Integer revision) {
            this.revision = revision;
        }

        @Override
        public final Builder revision(Integer revision) {
            this.revision = revision;
            return this;
        }

        public final List<Volume.Builder> getVolumes() {
            List<Volume.Builder> result = VolumeListCopier.copyToBuilder(this.volumes);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setVolumes(Collection<Volume.BuilderImpl> volumes) {
            this.volumes = VolumeListCopier.copyFromBuilder(volumes);
        }

        @Override
        public final Builder volumes(Collection<Volume> volumes) {
            this.volumes = VolumeListCopier.copy(volumes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder volumes(Volume... volumes) {
            volumes(Arrays.asList(volumes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder volumes(Consumer<Volume.Builder>... volumes) {
            volumes(Stream.of(volumes).map(c -> Volume.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(TaskDefinitionStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final List<Attribute.Builder> getRequiresAttributes() {
            List<Attribute.Builder> result = RequiresAttributesCopier.copyToBuilder(this.requiresAttributes);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRequiresAttributes(Collection<Attribute.BuilderImpl> requiresAttributes) {
            this.requiresAttributes = RequiresAttributesCopier.copyFromBuilder(requiresAttributes);
        }

        @Override
        public final Builder requiresAttributes(Collection<Attribute> requiresAttributes) {
            this.requiresAttributes = RequiresAttributesCopier.copy(requiresAttributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder requiresAttributes(Attribute... requiresAttributes) {
            requiresAttributes(Arrays.asList(requiresAttributes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder requiresAttributes(Consumer<Attribute.Builder>... requiresAttributes) {
            requiresAttributes(Stream.of(requiresAttributes).map(c -> Attribute.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<TaskDefinitionPlacementConstraint.Builder> getPlacementConstraints() {
            List<TaskDefinitionPlacementConstraint.Builder> result = TaskDefinitionPlacementConstraintsCopier
                    .copyToBuilder(this.placementConstraints);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPlacementConstraints(Collection<TaskDefinitionPlacementConstraint.BuilderImpl> placementConstraints) {
            this.placementConstraints = TaskDefinitionPlacementConstraintsCopier.copyFromBuilder(placementConstraints);
        }

        @Override
        public final Builder placementConstraints(Collection<TaskDefinitionPlacementConstraint> placementConstraints) {
            this.placementConstraints = TaskDefinitionPlacementConstraintsCopier.copy(placementConstraints);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder placementConstraints(TaskDefinitionPlacementConstraint... placementConstraints) {
            placementConstraints(Arrays.asList(placementConstraints));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder placementConstraints(Consumer<TaskDefinitionPlacementConstraint.Builder>... placementConstraints) {
            placementConstraints(Stream.of(placementConstraints)
                    .map(c -> TaskDefinitionPlacementConstraint.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Collection<String> getCompatibilities() {
            if (compatibilities instanceof SdkAutoConstructList) {
                return null;
            }
            return compatibilities;
        }

        public final void setCompatibilities(Collection<String> compatibilities) {
            this.compatibilities = CompatibilityListCopier.copy(compatibilities);
        }

        @Override
        public final Builder compatibilitiesWithStrings(Collection<String> compatibilities) {
            this.compatibilities = CompatibilityListCopier.copy(compatibilities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder compatibilitiesWithStrings(String... compatibilities) {
            compatibilitiesWithStrings(Arrays.asList(compatibilities));
            return this;
        }

        @Override
        public final Builder compatibilities(Collection<Compatibility> compatibilities) {
            this.compatibilities = CompatibilityListCopier.copyEnumToString(compatibilities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder compatibilities(Compatibility... compatibilities) {
            compatibilities(Arrays.asList(compatibilities));
            return this;
        }

        public final RuntimePlatform.Builder getRuntimePlatform() {
            return runtimePlatform != null ? runtimePlatform.toBuilder() : null;
        }

        public final void setRuntimePlatform(RuntimePlatform.BuilderImpl runtimePlatform) {
            this.runtimePlatform = runtimePlatform != null ? runtimePlatform.build() : null;
        }

        @Override
        public final Builder runtimePlatform(RuntimePlatform runtimePlatform) {
            this.runtimePlatform = runtimePlatform;
            return this;
        }

        public final Collection<String> getRequiresCompatibilities() {
            if (requiresCompatibilities instanceof SdkAutoConstructList) {
                return null;
            }
            return requiresCompatibilities;
        }

        public final void setRequiresCompatibilities(Collection<String> requiresCompatibilities) {
            this.requiresCompatibilities = CompatibilityListCopier.copy(requiresCompatibilities);
        }

        @Override
        public final Builder requiresCompatibilitiesWithStrings(Collection<String> requiresCompatibilities) {
            this.requiresCompatibilities = CompatibilityListCopier.copy(requiresCompatibilities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder requiresCompatibilitiesWithStrings(String... requiresCompatibilities) {
            requiresCompatibilitiesWithStrings(Arrays.asList(requiresCompatibilities));
            return this;
        }

        @Override
        public final Builder requiresCompatibilities(Collection<Compatibility> requiresCompatibilities) {
            this.requiresCompatibilities = CompatibilityListCopier.copyEnumToString(requiresCompatibilities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder requiresCompatibilities(Compatibility... requiresCompatibilities) {
            requiresCompatibilities(Arrays.asList(requiresCompatibilities));
            return this;
        }

        public final String getCpu() {
            return cpu;
        }

        public final void setCpu(String cpu) {
            this.cpu = cpu;
        }

        @Override
        public final Builder cpu(String cpu) {
            this.cpu = cpu;
            return this;
        }

        public final String getMemory() {
            return memory;
        }

        public final void setMemory(String memory) {
            this.memory = memory;
        }

        @Override
        public final Builder memory(String memory) {
            this.memory = memory;
            return this;
        }

        public final List<InferenceAccelerator.Builder> getInferenceAccelerators() {
            List<InferenceAccelerator.Builder> result = InferenceAcceleratorsCopier.copyToBuilder(this.inferenceAccelerators);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setInferenceAccelerators(Collection<InferenceAccelerator.BuilderImpl> inferenceAccelerators) {
            this.inferenceAccelerators = InferenceAcceleratorsCopier.copyFromBuilder(inferenceAccelerators);
        }

        @Override
        public final Builder inferenceAccelerators(Collection<InferenceAccelerator> inferenceAccelerators) {
            this.inferenceAccelerators = InferenceAcceleratorsCopier.copy(inferenceAccelerators);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inferenceAccelerators(InferenceAccelerator... inferenceAccelerators) {
            inferenceAccelerators(Arrays.asList(inferenceAccelerators));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inferenceAccelerators(Consumer<InferenceAccelerator.Builder>... inferenceAccelerators) {
            inferenceAccelerators(Stream.of(inferenceAccelerators)
                    .map(c -> InferenceAccelerator.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getPidMode() {
            return pidMode;
        }

        public final void setPidMode(String pidMode) {
            this.pidMode = pidMode;
        }

        @Override
        public final Builder pidMode(String pidMode) {
            this.pidMode = pidMode;
            return this;
        }

        @Override
        public final Builder pidMode(PidMode pidMode) {
            this.pidMode(pidMode == null ? null : pidMode.toString());
            return this;
        }

        public final String getIpcMode() {
            return ipcMode;
        }

        public final void setIpcMode(String ipcMode) {
            this.ipcMode = ipcMode;
        }

        @Override
        public final Builder ipcMode(String ipcMode) {
            this.ipcMode = ipcMode;
            return this;
        }

        @Override
        public final Builder ipcMode(IpcMode ipcMode) {
            this.ipcMode(ipcMode == null ? null : ipcMode.toString());
            return this;
        }

        public final ProxyConfiguration.Builder getProxyConfiguration() {
            return proxyConfiguration != null ? proxyConfiguration.toBuilder() : null;
        }

        public final void setProxyConfiguration(ProxyConfiguration.BuilderImpl proxyConfiguration) {
            this.proxyConfiguration = proxyConfiguration != null ? proxyConfiguration.build() : null;
        }

        @Override
        public final Builder proxyConfiguration(ProxyConfiguration proxyConfiguration) {
            this.proxyConfiguration = proxyConfiguration;
            return this;
        }

        public final Instant getRegisteredAt() {
            return registeredAt;
        }

        public final void setRegisteredAt(Instant registeredAt) {
            this.registeredAt = registeredAt;
        }

        @Override
        public final Builder registeredAt(Instant registeredAt) {
            this.registeredAt = registeredAt;
            return this;
        }

        public final Instant getDeregisteredAt() {
            return deregisteredAt;
        }

        public final void setDeregisteredAt(Instant deregisteredAt) {
            this.deregisteredAt = deregisteredAt;
        }

        @Override
        public final Builder deregisteredAt(Instant deregisteredAt) {
            this.deregisteredAt = deregisteredAt;
            return this;
        }

        public final String getRegisteredBy() {
            return registeredBy;
        }

        public final void setRegisteredBy(String registeredBy) {
            this.registeredBy = registeredBy;
        }

        @Override
        public final Builder registeredBy(String registeredBy) {
            this.registeredBy = registeredBy;
            return this;
        }

        public final EphemeralStorage.Builder getEphemeralStorage() {
            return ephemeralStorage != null ? ephemeralStorage.toBuilder() : null;
        }

        public final void setEphemeralStorage(EphemeralStorage.BuilderImpl ephemeralStorage) {
            this.ephemeralStorage = ephemeralStorage != null ? ephemeralStorage.build() : null;
        }

        @Override
        public final Builder ephemeralStorage(EphemeralStorage ephemeralStorage) {
            this.ephemeralStorage = ephemeralStorage;
            return this;
        }

        @Override
        public TaskDefinition build() {
            return new TaskDefinition(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
