/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the service deployment.
 * </p>
 * <p>
 * Service deployments provide a comprehensive view of your deployments. For information about service deployments, see
 * <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-deployment.html">View service history
 * using Amazon ECS service deployments</a> in the <i> <i>Amazon Elastic Container Service Developer Guide</i> </i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ServiceDeployment implements SdkPojo, Serializable,
        ToCopyableBuilder<ServiceDeployment.Builder, ServiceDeployment> {
    private static final SdkField<String> SERVICE_DEPLOYMENT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("serviceDeploymentArn").getter(getter(ServiceDeployment::serviceDeploymentArn))
            .setter(setter(Builder::serviceDeploymentArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serviceDeploymentArn").build())
            .build();

    private static final SdkField<String> SERVICE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("serviceArn").getter(getter(ServiceDeployment::serviceArn)).setter(setter(Builder::serviceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serviceArn").build()).build();

    private static final SdkField<String> CLUSTER_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("clusterArn").getter(getter(ServiceDeployment::clusterArn)).setter(setter(Builder::clusterArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("clusterArn").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt").getter(getter(ServiceDeployment::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<Instant> STARTED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("startedAt").getter(getter(ServiceDeployment::startedAt)).setter(setter(Builder::startedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startedAt").build()).build();

    private static final SdkField<Instant> FINISHED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("finishedAt").getter(getter(ServiceDeployment::finishedAt)).setter(setter(Builder::finishedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("finishedAt").build()).build();

    private static final SdkField<Instant> STOPPED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("stoppedAt").getter(getter(ServiceDeployment::stoppedAt)).setter(setter(Builder::stoppedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("stoppedAt").build()).build();

    private static final SdkField<Instant> UPDATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("updatedAt").getter(getter(ServiceDeployment::updatedAt)).setter(setter(Builder::updatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("updatedAt").build()).build();

    private static final SdkField<List<ServiceRevisionSummary>> SOURCE_SERVICE_REVISIONS_FIELD = SdkField
            .<List<ServiceRevisionSummary>> builder(MarshallingType.LIST)
            .memberName("sourceServiceRevisions")
            .getter(getter(ServiceDeployment::sourceServiceRevisions))
            .setter(setter(Builder::sourceServiceRevisions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceServiceRevisions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ServiceRevisionSummary> builder(MarshallingType.SDK_POJO)
                                            .constructor(ServiceRevisionSummary::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ServiceRevisionSummary> TARGET_SERVICE_REVISION_FIELD = SdkField
            .<ServiceRevisionSummary> builder(MarshallingType.SDK_POJO).memberName("targetServiceRevision")
            .getter(getter(ServiceDeployment::targetServiceRevision)).setter(setter(Builder::targetServiceRevision))
            .constructor(ServiceRevisionSummary::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("targetServiceRevision").build())
            .build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(ServiceDeployment::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> STATUS_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("statusReason").getter(getter(ServiceDeployment::statusReason)).setter(setter(Builder::statusReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusReason").build()).build();

    private static final SdkField<DeploymentConfiguration> DEPLOYMENT_CONFIGURATION_FIELD = SdkField
            .<DeploymentConfiguration> builder(MarshallingType.SDK_POJO).memberName("deploymentConfiguration")
            .getter(getter(ServiceDeployment::deploymentConfiguration)).setter(setter(Builder::deploymentConfiguration))
            .constructor(DeploymentConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("deploymentConfiguration").build())
            .build();

    private static final SdkField<Rollback> ROLLBACK_FIELD = SdkField.<Rollback> builder(MarshallingType.SDK_POJO)
            .memberName("rollback").getter(getter(ServiceDeployment::rollback)).setter(setter(Builder::rollback))
            .constructor(Rollback::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("rollback").build()).build();

    private static final SdkField<ServiceDeploymentCircuitBreaker> DEPLOYMENT_CIRCUIT_BREAKER_FIELD = SdkField
            .<ServiceDeploymentCircuitBreaker> builder(MarshallingType.SDK_POJO).memberName("deploymentCircuitBreaker")
            .getter(getter(ServiceDeployment::deploymentCircuitBreaker)).setter(setter(Builder::deploymentCircuitBreaker))
            .constructor(ServiceDeploymentCircuitBreaker::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("deploymentCircuitBreaker").build())
            .build();

    private static final SdkField<ServiceDeploymentAlarms> ALARMS_FIELD = SdkField
            .<ServiceDeploymentAlarms> builder(MarshallingType.SDK_POJO).memberName("alarms")
            .getter(getter(ServiceDeployment::alarms)).setter(setter(Builder::alarms))
            .constructor(ServiceDeploymentAlarms::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("alarms").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SERVICE_DEPLOYMENT_ARN_FIELD,
            SERVICE_ARN_FIELD, CLUSTER_ARN_FIELD, CREATED_AT_FIELD, STARTED_AT_FIELD, FINISHED_AT_FIELD, STOPPED_AT_FIELD,
            UPDATED_AT_FIELD, SOURCE_SERVICE_REVISIONS_FIELD, TARGET_SERVICE_REVISION_FIELD, STATUS_FIELD, STATUS_REASON_FIELD,
            DEPLOYMENT_CONFIGURATION_FIELD, ROLLBACK_FIELD, DEPLOYMENT_CIRCUIT_BREAKER_FIELD, ALARMS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String serviceDeploymentArn;

    private final String serviceArn;

    private final String clusterArn;

    private final Instant createdAt;

    private final Instant startedAt;

    private final Instant finishedAt;

    private final Instant stoppedAt;

    private final Instant updatedAt;

    private final List<ServiceRevisionSummary> sourceServiceRevisions;

    private final ServiceRevisionSummary targetServiceRevision;

    private final String status;

    private final String statusReason;

    private final DeploymentConfiguration deploymentConfiguration;

    private final Rollback rollback;

    private final ServiceDeploymentCircuitBreaker deploymentCircuitBreaker;

    private final ServiceDeploymentAlarms alarms;

    private ServiceDeployment(BuilderImpl builder) {
        this.serviceDeploymentArn = builder.serviceDeploymentArn;
        this.serviceArn = builder.serviceArn;
        this.clusterArn = builder.clusterArn;
        this.createdAt = builder.createdAt;
        this.startedAt = builder.startedAt;
        this.finishedAt = builder.finishedAt;
        this.stoppedAt = builder.stoppedAt;
        this.updatedAt = builder.updatedAt;
        this.sourceServiceRevisions = builder.sourceServiceRevisions;
        this.targetServiceRevision = builder.targetServiceRevision;
        this.status = builder.status;
        this.statusReason = builder.statusReason;
        this.deploymentConfiguration = builder.deploymentConfiguration;
        this.rollback = builder.rollback;
        this.deploymentCircuitBreaker = builder.deploymentCircuitBreaker;
        this.alarms = builder.alarms;
    }

    /**
     * <p>
     * The ARN of the service deployment.
     * </p>
     * 
     * @return The ARN of the service deployment.
     */
    public final String serviceDeploymentArn() {
        return serviceDeploymentArn;
    }

    /**
     * <p>
     * The ARN of the service for this service deployment.
     * </p>
     * 
     * @return The ARN of the service for this service deployment.
     */
    public final String serviceArn() {
        return serviceArn;
    }

    /**
     * <p>
     * The ARN of the cluster that hosts the service.
     * </p>
     * 
     * @return The ARN of the cluster that hosts the service.
     */
    public final String clusterArn() {
        return clusterArn;
    }

    /**
     * <p>
     * The time the service deployment was created. The format is yyyy-MM-dd HH:mm:ss.SSSSSS.
     * </p>
     * 
     * @return The time the service deployment was created. The format is yyyy-MM-dd HH:mm:ss.SSSSSS.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The time the service deployment statred. The format is yyyy-MM-dd HH:mm:ss.SSSSSS.
     * </p>
     * 
     * @return The time the service deployment statred. The format is yyyy-MM-dd HH:mm:ss.SSSSSS.
     */
    public final Instant startedAt() {
        return startedAt;
    }

    /**
     * <p>
     * The time the service deployment finished. The format is yyyy-MM-dd HH:mm:ss.SSSSSS.
     * </p>
     * 
     * @return The time the service deployment finished. The format is yyyy-MM-dd HH:mm:ss.SSSSSS.
     */
    public final Instant finishedAt() {
        return finishedAt;
    }

    /**
     * <p>
     * The time the service deployment stopped. The format is yyyy-MM-dd HH:mm:ss.SSSSSS.
     * </p>
     * <p>
     * The service deployment stops when any of the following actions happen:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A user manually stops the deployment
     * </p>
     * </li>
     * <li>
     * <p>
     * The rollback option is not in use for the failure detection mechanism (the circuit breaker or alarm-based) and
     * the service fails.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The time the service deployment stopped. The format is yyyy-MM-dd HH:mm:ss.SSSSSS.</p>
     *         <p>
     *         The service deployment stops when any of the following actions happen:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A user manually stops the deployment
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The rollback option is not in use for the failure detection mechanism (the circuit breaker or
     *         alarm-based) and the service fails.
     *         </p>
     *         </li>
     */
    public final Instant stoppedAt() {
        return stoppedAt;
    }

    /**
     * <p>
     * The time that the service deployment was last updated. The format is yyyy-MM-dd HH:mm:ss.SSSSSS.
     * </p>
     * 
     * @return The time that the service deployment was last updated. The format is yyyy-MM-dd HH:mm:ss.SSSSSS.
     */
    public final Instant updatedAt() {
        return updatedAt;
    }

    /**
     * For responses, this returns true if the service returned a value for the SourceServiceRevisions property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasSourceServiceRevisions() {
        return sourceServiceRevisions != null && !(sourceServiceRevisions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The currently deployed workload configuration.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSourceServiceRevisions} method.
     * </p>
     * 
     * @return The currently deployed workload configuration.
     */
    public final List<ServiceRevisionSummary> sourceServiceRevisions() {
        return sourceServiceRevisions;
    }

    /**
     * <p>
     * The workload configuration being deployed.
     * </p>
     * 
     * @return The workload configuration being deployed.
     */
    public final ServiceRevisionSummary targetServiceRevision() {
        return targetServiceRevision;
    }

    /**
     * <p>
     * The service deployment state.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ServiceDeploymentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The service deployment state.
     * @see ServiceDeploymentStatus
     */
    public final ServiceDeploymentStatus status() {
        return ServiceDeploymentStatus.fromValue(status);
    }

    /**
     * <p>
     * The service deployment state.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ServiceDeploymentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The service deployment state.
     * @see ServiceDeploymentStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Information about why the service deployment is in the current status. For example, the circuit breaker detected
     * a failure.
     * </p>
     * 
     * @return Information about why the service deployment is in the current status. For example, the circuit breaker
     *         detected a failure.
     */
    public final String statusReason() {
        return statusReason;
    }

    /**
     * Returns the value of the DeploymentConfiguration property for this object.
     * 
     * @return The value of the DeploymentConfiguration property for this object.
     */
    public final DeploymentConfiguration deploymentConfiguration() {
        return deploymentConfiguration;
    }

    /**
     * <p>
     * The rollback options the service deployment uses when the deployment fails.
     * </p>
     * 
     * @return The rollback options the service deployment uses when the deployment fails.
     */
    public final Rollback rollback() {
        return rollback;
    }

    /**
     * <p>
     * The circuit breaker configuration that determines a service deployment failed.
     * </p>
     * 
     * @return The circuit breaker configuration that determines a service deployment failed.
     */
    public final ServiceDeploymentCircuitBreaker deploymentCircuitBreaker() {
        return deploymentCircuitBreaker;
    }

    /**
     * <p>
     * The CloudWatch alarms that determine when a service deployment fails.
     * </p>
     * 
     * @return The CloudWatch alarms that determine when a service deployment fails.
     */
    public final ServiceDeploymentAlarms alarms() {
        return alarms;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(serviceDeploymentArn());
        hashCode = 31 * hashCode + Objects.hashCode(serviceArn());
        hashCode = 31 * hashCode + Objects.hashCode(clusterArn());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(startedAt());
        hashCode = 31 * hashCode + Objects.hashCode(finishedAt());
        hashCode = 31 * hashCode + Objects.hashCode(stoppedAt());
        hashCode = 31 * hashCode + Objects.hashCode(updatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(hasSourceServiceRevisions() ? sourceServiceRevisions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(targetServiceRevision());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusReason());
        hashCode = 31 * hashCode + Objects.hashCode(deploymentConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(rollback());
        hashCode = 31 * hashCode + Objects.hashCode(deploymentCircuitBreaker());
        hashCode = 31 * hashCode + Objects.hashCode(alarms());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ServiceDeployment)) {
            return false;
        }
        ServiceDeployment other = (ServiceDeployment) obj;
        return Objects.equals(serviceDeploymentArn(), other.serviceDeploymentArn())
                && Objects.equals(serviceArn(), other.serviceArn()) && Objects.equals(clusterArn(), other.clusterArn())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(startedAt(), other.startedAt())
                && Objects.equals(finishedAt(), other.finishedAt()) && Objects.equals(stoppedAt(), other.stoppedAt())
                && Objects.equals(updatedAt(), other.updatedAt())
                && hasSourceServiceRevisions() == other.hasSourceServiceRevisions()
                && Objects.equals(sourceServiceRevisions(), other.sourceServiceRevisions())
                && Objects.equals(targetServiceRevision(), other.targetServiceRevision())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusReason(), other.statusReason())
                && Objects.equals(deploymentConfiguration(), other.deploymentConfiguration())
                && Objects.equals(rollback(), other.rollback())
                && Objects.equals(deploymentCircuitBreaker(), other.deploymentCircuitBreaker())
                && Objects.equals(alarms(), other.alarms());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ServiceDeployment").add("ServiceDeploymentArn", serviceDeploymentArn())
                .add("ServiceArn", serviceArn()).add("ClusterArn", clusterArn()).add("CreatedAt", createdAt())
                .add("StartedAt", startedAt()).add("FinishedAt", finishedAt()).add("StoppedAt", stoppedAt())
                .add("UpdatedAt", updatedAt())
                .add("SourceServiceRevisions", hasSourceServiceRevisions() ? sourceServiceRevisions() : null)
                .add("TargetServiceRevision", targetServiceRevision()).add("Status", statusAsString())
                .add("StatusReason", statusReason()).add("DeploymentConfiguration", deploymentConfiguration())
                .add("Rollback", rollback()).add("DeploymentCircuitBreaker", deploymentCircuitBreaker()).add("Alarms", alarms())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "serviceDeploymentArn":
            return Optional.ofNullable(clazz.cast(serviceDeploymentArn()));
        case "serviceArn":
            return Optional.ofNullable(clazz.cast(serviceArn()));
        case "clusterArn":
            return Optional.ofNullable(clazz.cast(clusterArn()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "startedAt":
            return Optional.ofNullable(clazz.cast(startedAt()));
        case "finishedAt":
            return Optional.ofNullable(clazz.cast(finishedAt()));
        case "stoppedAt":
            return Optional.ofNullable(clazz.cast(stoppedAt()));
        case "updatedAt":
            return Optional.ofNullable(clazz.cast(updatedAt()));
        case "sourceServiceRevisions":
            return Optional.ofNullable(clazz.cast(sourceServiceRevisions()));
        case "targetServiceRevision":
            return Optional.ofNullable(clazz.cast(targetServiceRevision()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "statusReason":
            return Optional.ofNullable(clazz.cast(statusReason()));
        case "deploymentConfiguration":
            return Optional.ofNullable(clazz.cast(deploymentConfiguration()));
        case "rollback":
            return Optional.ofNullable(clazz.cast(rollback()));
        case "deploymentCircuitBreaker":
            return Optional.ofNullable(clazz.cast(deploymentCircuitBreaker()));
        case "alarms":
            return Optional.ofNullable(clazz.cast(alarms()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("serviceDeploymentArn", SERVICE_DEPLOYMENT_ARN_FIELD);
        map.put("serviceArn", SERVICE_ARN_FIELD);
        map.put("clusterArn", CLUSTER_ARN_FIELD);
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("startedAt", STARTED_AT_FIELD);
        map.put("finishedAt", FINISHED_AT_FIELD);
        map.put("stoppedAt", STOPPED_AT_FIELD);
        map.put("updatedAt", UPDATED_AT_FIELD);
        map.put("sourceServiceRevisions", SOURCE_SERVICE_REVISIONS_FIELD);
        map.put("targetServiceRevision", TARGET_SERVICE_REVISION_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("statusReason", STATUS_REASON_FIELD);
        map.put("deploymentConfiguration", DEPLOYMENT_CONFIGURATION_FIELD);
        map.put("rollback", ROLLBACK_FIELD);
        map.put("deploymentCircuitBreaker", DEPLOYMENT_CIRCUIT_BREAKER_FIELD);
        map.put("alarms", ALARMS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ServiceDeployment, T> g) {
        return obj -> g.apply((ServiceDeployment) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ServiceDeployment> {
        /**
         * <p>
         * The ARN of the service deployment.
         * </p>
         * 
         * @param serviceDeploymentArn
         *        The ARN of the service deployment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceDeploymentArn(String serviceDeploymentArn);

        /**
         * <p>
         * The ARN of the service for this service deployment.
         * </p>
         * 
         * @param serviceArn
         *        The ARN of the service for this service deployment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceArn(String serviceArn);

        /**
         * <p>
         * The ARN of the cluster that hosts the service.
         * </p>
         * 
         * @param clusterArn
         *        The ARN of the cluster that hosts the service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterArn(String clusterArn);

        /**
         * <p>
         * The time the service deployment was created. The format is yyyy-MM-dd HH:mm:ss.SSSSSS.
         * </p>
         * 
         * @param createdAt
         *        The time the service deployment was created. The format is yyyy-MM-dd HH:mm:ss.SSSSSS.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The time the service deployment statred. The format is yyyy-MM-dd HH:mm:ss.SSSSSS.
         * </p>
         * 
         * @param startedAt
         *        The time the service deployment statred. The format is yyyy-MM-dd HH:mm:ss.SSSSSS.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startedAt(Instant startedAt);

        /**
         * <p>
         * The time the service deployment finished. The format is yyyy-MM-dd HH:mm:ss.SSSSSS.
         * </p>
         * 
         * @param finishedAt
         *        The time the service deployment finished. The format is yyyy-MM-dd HH:mm:ss.SSSSSS.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder finishedAt(Instant finishedAt);

        /**
         * <p>
         * The time the service deployment stopped. The format is yyyy-MM-dd HH:mm:ss.SSSSSS.
         * </p>
         * <p>
         * The service deployment stops when any of the following actions happen:
         * </p>
         * <ul>
         * <li>
         * <p>
         * A user manually stops the deployment
         * </p>
         * </li>
         * <li>
         * <p>
         * The rollback option is not in use for the failure detection mechanism (the circuit breaker or alarm-based)
         * and the service fails.
         * </p>
         * </li>
         * </ul>
         * 
         * @param stoppedAt
         *        The time the service deployment stopped. The format is yyyy-MM-dd HH:mm:ss.SSSSSS.</p>
         *        <p>
         *        The service deployment stops when any of the following actions happen:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A user manually stops the deployment
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The rollback option is not in use for the failure detection mechanism (the circuit breaker or
         *        alarm-based) and the service fails.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stoppedAt(Instant stoppedAt);

        /**
         * <p>
         * The time that the service deployment was last updated. The format is yyyy-MM-dd HH:mm:ss.SSSSSS.
         * </p>
         * 
         * @param updatedAt
         *        The time that the service deployment was last updated. The format is yyyy-MM-dd HH:mm:ss.SSSSSS.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedAt(Instant updatedAt);

        /**
         * <p>
         * The currently deployed workload configuration.
         * </p>
         * 
         * @param sourceServiceRevisions
         *        The currently deployed workload configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceServiceRevisions(Collection<ServiceRevisionSummary> sourceServiceRevisions);

        /**
         * <p>
         * The currently deployed workload configuration.
         * </p>
         * 
         * @param sourceServiceRevisions
         *        The currently deployed workload configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceServiceRevisions(ServiceRevisionSummary... sourceServiceRevisions);

        /**
         * <p>
         * The currently deployed workload configuration.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.ServiceRevisionSummary.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.ecs.model.ServiceRevisionSummary#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecs.model.ServiceRevisionSummary.Builder#build()} is called
         * immediately and its result is passed to {@link #sourceServiceRevisions(List<ServiceRevisionSummary>)}.
         * 
         * @param sourceServiceRevisions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecs.model.ServiceRevisionSummary.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sourceServiceRevisions(java.util.Collection<ServiceRevisionSummary>)
         */
        Builder sourceServiceRevisions(Consumer<ServiceRevisionSummary.Builder>... sourceServiceRevisions);

        /**
         * <p>
         * The workload configuration being deployed.
         * </p>
         * 
         * @param targetServiceRevision
         *        The workload configuration being deployed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetServiceRevision(ServiceRevisionSummary targetServiceRevision);

        /**
         * <p>
         * The workload configuration being deployed.
         * </p>
         * This is a convenience method that creates an instance of the {@link ServiceRevisionSummary.Builder} avoiding
         * the need to create one manually via {@link ServiceRevisionSummary#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ServiceRevisionSummary.Builder#build()} is called immediately and
         * its result is passed to {@link #targetServiceRevision(ServiceRevisionSummary)}.
         * 
         * @param targetServiceRevision
         *        a consumer that will call methods on {@link ServiceRevisionSummary.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targetServiceRevision(ServiceRevisionSummary)
         */
        default Builder targetServiceRevision(Consumer<ServiceRevisionSummary.Builder> targetServiceRevision) {
            return targetServiceRevision(ServiceRevisionSummary.builder().applyMutation(targetServiceRevision).build());
        }

        /**
         * <p>
         * The service deployment state.
         * </p>
         * 
         * @param status
         *        The service deployment state.
         * @see ServiceDeploymentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ServiceDeploymentStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The service deployment state.
         * </p>
         * 
         * @param status
         *        The service deployment state.
         * @see ServiceDeploymentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ServiceDeploymentStatus
         */
        Builder status(ServiceDeploymentStatus status);

        /**
         * <p>
         * Information about why the service deployment is in the current status. For example, the circuit breaker
         * detected a failure.
         * </p>
         * 
         * @param statusReason
         *        Information about why the service deployment is in the current status. For example, the circuit
         *        breaker detected a failure.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReason(String statusReason);

        /**
         * Sets the value of the DeploymentConfiguration property for this object.
         *
         * @param deploymentConfiguration
         *        The new value for the DeploymentConfiguration property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deploymentConfiguration(DeploymentConfiguration deploymentConfiguration);

        /**
         * Sets the value of the DeploymentConfiguration property for this object.
         *
         * This is a convenience method that creates an instance of the {@link DeploymentConfiguration.Builder} avoiding
         * the need to create one manually via {@link DeploymentConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DeploymentConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #deploymentConfiguration(DeploymentConfiguration)}.
         * 
         * @param deploymentConfiguration
         *        a consumer that will call methods on {@link DeploymentConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #deploymentConfiguration(DeploymentConfiguration)
         */
        default Builder deploymentConfiguration(Consumer<DeploymentConfiguration.Builder> deploymentConfiguration) {
            return deploymentConfiguration(DeploymentConfiguration.builder().applyMutation(deploymentConfiguration).build());
        }

        /**
         * <p>
         * The rollback options the service deployment uses when the deployment fails.
         * </p>
         * 
         * @param rollback
         *        The rollback options the service deployment uses when the deployment fails.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rollback(Rollback rollback);

        /**
         * <p>
         * The rollback options the service deployment uses when the deployment fails.
         * </p>
         * This is a convenience method that creates an instance of the {@link Rollback.Builder} avoiding the need to
         * create one manually via {@link Rollback#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Rollback.Builder#build()} is called immediately and its result is
         * passed to {@link #rollback(Rollback)}.
         * 
         * @param rollback
         *        a consumer that will call methods on {@link Rollback.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #rollback(Rollback)
         */
        default Builder rollback(Consumer<Rollback.Builder> rollback) {
            return rollback(Rollback.builder().applyMutation(rollback).build());
        }

        /**
         * <p>
         * The circuit breaker configuration that determines a service deployment failed.
         * </p>
         * 
         * @param deploymentCircuitBreaker
         *        The circuit breaker configuration that determines a service deployment failed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deploymentCircuitBreaker(ServiceDeploymentCircuitBreaker deploymentCircuitBreaker);

        /**
         * <p>
         * The circuit breaker configuration that determines a service deployment failed.
         * </p>
         * This is a convenience method that creates an instance of the {@link ServiceDeploymentCircuitBreaker.Builder}
         * avoiding the need to create one manually via {@link ServiceDeploymentCircuitBreaker#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ServiceDeploymentCircuitBreaker.Builder#build()} is called
         * immediately and its result is passed to {@link #deploymentCircuitBreaker(ServiceDeploymentCircuitBreaker)}.
         * 
         * @param deploymentCircuitBreaker
         *        a consumer that will call methods on {@link ServiceDeploymentCircuitBreaker.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #deploymentCircuitBreaker(ServiceDeploymentCircuitBreaker)
         */
        default Builder deploymentCircuitBreaker(Consumer<ServiceDeploymentCircuitBreaker.Builder> deploymentCircuitBreaker) {
            return deploymentCircuitBreaker(ServiceDeploymentCircuitBreaker.builder().applyMutation(deploymentCircuitBreaker)
                    .build());
        }

        /**
         * <p>
         * The CloudWatch alarms that determine when a service deployment fails.
         * </p>
         * 
         * @param alarms
         *        The CloudWatch alarms that determine when a service deployment fails.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarms(ServiceDeploymentAlarms alarms);

        /**
         * <p>
         * The CloudWatch alarms that determine when a service deployment fails.
         * </p>
         * This is a convenience method that creates an instance of the {@link ServiceDeploymentAlarms.Builder} avoiding
         * the need to create one manually via {@link ServiceDeploymentAlarms#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ServiceDeploymentAlarms.Builder#build()} is called immediately
         * and its result is passed to {@link #alarms(ServiceDeploymentAlarms)}.
         * 
         * @param alarms
         *        a consumer that will call methods on {@link ServiceDeploymentAlarms.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #alarms(ServiceDeploymentAlarms)
         */
        default Builder alarms(Consumer<ServiceDeploymentAlarms.Builder> alarms) {
            return alarms(ServiceDeploymentAlarms.builder().applyMutation(alarms).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String serviceDeploymentArn;

        private String serviceArn;

        private String clusterArn;

        private Instant createdAt;

        private Instant startedAt;

        private Instant finishedAt;

        private Instant stoppedAt;

        private Instant updatedAt;

        private List<ServiceRevisionSummary> sourceServiceRevisions = DefaultSdkAutoConstructList.getInstance();

        private ServiceRevisionSummary targetServiceRevision;

        private String status;

        private String statusReason;

        private DeploymentConfiguration deploymentConfiguration;

        private Rollback rollback;

        private ServiceDeploymentCircuitBreaker deploymentCircuitBreaker;

        private ServiceDeploymentAlarms alarms;

        private BuilderImpl() {
        }

        private BuilderImpl(ServiceDeployment model) {
            serviceDeploymentArn(model.serviceDeploymentArn);
            serviceArn(model.serviceArn);
            clusterArn(model.clusterArn);
            createdAt(model.createdAt);
            startedAt(model.startedAt);
            finishedAt(model.finishedAt);
            stoppedAt(model.stoppedAt);
            updatedAt(model.updatedAt);
            sourceServiceRevisions(model.sourceServiceRevisions);
            targetServiceRevision(model.targetServiceRevision);
            status(model.status);
            statusReason(model.statusReason);
            deploymentConfiguration(model.deploymentConfiguration);
            rollback(model.rollback);
            deploymentCircuitBreaker(model.deploymentCircuitBreaker);
            alarms(model.alarms);
        }

        public final String getServiceDeploymentArn() {
            return serviceDeploymentArn;
        }

        public final void setServiceDeploymentArn(String serviceDeploymentArn) {
            this.serviceDeploymentArn = serviceDeploymentArn;
        }

        @Override
        public final Builder serviceDeploymentArn(String serviceDeploymentArn) {
            this.serviceDeploymentArn = serviceDeploymentArn;
            return this;
        }

        public final String getServiceArn() {
            return serviceArn;
        }

        public final void setServiceArn(String serviceArn) {
            this.serviceArn = serviceArn;
        }

        @Override
        public final Builder serviceArn(String serviceArn) {
            this.serviceArn = serviceArn;
            return this;
        }

        public final String getClusterArn() {
            return clusterArn;
        }

        public final void setClusterArn(String clusterArn) {
            this.clusterArn = clusterArn;
        }

        @Override
        public final Builder clusterArn(String clusterArn) {
            this.clusterArn = clusterArn;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getStartedAt() {
            return startedAt;
        }

        public final void setStartedAt(Instant startedAt) {
            this.startedAt = startedAt;
        }

        @Override
        public final Builder startedAt(Instant startedAt) {
            this.startedAt = startedAt;
            return this;
        }

        public final Instant getFinishedAt() {
            return finishedAt;
        }

        public final void setFinishedAt(Instant finishedAt) {
            this.finishedAt = finishedAt;
        }

        @Override
        public final Builder finishedAt(Instant finishedAt) {
            this.finishedAt = finishedAt;
            return this;
        }

        public final Instant getStoppedAt() {
            return stoppedAt;
        }

        public final void setStoppedAt(Instant stoppedAt) {
            this.stoppedAt = stoppedAt;
        }

        @Override
        public final Builder stoppedAt(Instant stoppedAt) {
            this.stoppedAt = stoppedAt;
            return this;
        }

        public final Instant getUpdatedAt() {
            return updatedAt;
        }

        public final void setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
        }

        @Override
        public final Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        public final List<ServiceRevisionSummary.Builder> getSourceServiceRevisions() {
            List<ServiceRevisionSummary.Builder> result = ServiceRevisionsSummaryListCopier
                    .copyToBuilder(this.sourceServiceRevisions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSourceServiceRevisions(Collection<ServiceRevisionSummary.BuilderImpl> sourceServiceRevisions) {
            this.sourceServiceRevisions = ServiceRevisionsSummaryListCopier.copyFromBuilder(sourceServiceRevisions);
        }

        @Override
        public final Builder sourceServiceRevisions(Collection<ServiceRevisionSummary> sourceServiceRevisions) {
            this.sourceServiceRevisions = ServiceRevisionsSummaryListCopier.copy(sourceServiceRevisions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sourceServiceRevisions(ServiceRevisionSummary... sourceServiceRevisions) {
            sourceServiceRevisions(Arrays.asList(sourceServiceRevisions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sourceServiceRevisions(Consumer<ServiceRevisionSummary.Builder>... sourceServiceRevisions) {
            sourceServiceRevisions(Stream.of(sourceServiceRevisions)
                    .map(c -> ServiceRevisionSummary.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final ServiceRevisionSummary.Builder getTargetServiceRevision() {
            return targetServiceRevision != null ? targetServiceRevision.toBuilder() : null;
        }

        public final void setTargetServiceRevision(ServiceRevisionSummary.BuilderImpl targetServiceRevision) {
            this.targetServiceRevision = targetServiceRevision != null ? targetServiceRevision.build() : null;
        }

        @Override
        public final Builder targetServiceRevision(ServiceRevisionSummary targetServiceRevision) {
            this.targetServiceRevision = targetServiceRevision;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ServiceDeploymentStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getStatusReason() {
            return statusReason;
        }

        public final void setStatusReason(String statusReason) {
            this.statusReason = statusReason;
        }

        @Override
        public final Builder statusReason(String statusReason) {
            this.statusReason = statusReason;
            return this;
        }

        public final DeploymentConfiguration.Builder getDeploymentConfiguration() {
            return deploymentConfiguration != null ? deploymentConfiguration.toBuilder() : null;
        }

        public final void setDeploymentConfiguration(DeploymentConfiguration.BuilderImpl deploymentConfiguration) {
            this.deploymentConfiguration = deploymentConfiguration != null ? deploymentConfiguration.build() : null;
        }

        @Override
        public final Builder deploymentConfiguration(DeploymentConfiguration deploymentConfiguration) {
            this.deploymentConfiguration = deploymentConfiguration;
            return this;
        }

        public final Rollback.Builder getRollback() {
            return rollback != null ? rollback.toBuilder() : null;
        }

        public final void setRollback(Rollback.BuilderImpl rollback) {
            this.rollback = rollback != null ? rollback.build() : null;
        }

        @Override
        public final Builder rollback(Rollback rollback) {
            this.rollback = rollback;
            return this;
        }

        public final ServiceDeploymentCircuitBreaker.Builder getDeploymentCircuitBreaker() {
            return deploymentCircuitBreaker != null ? deploymentCircuitBreaker.toBuilder() : null;
        }

        public final void setDeploymentCircuitBreaker(ServiceDeploymentCircuitBreaker.BuilderImpl deploymentCircuitBreaker) {
            this.deploymentCircuitBreaker = deploymentCircuitBreaker != null ? deploymentCircuitBreaker.build() : null;
        }

        @Override
        public final Builder deploymentCircuitBreaker(ServiceDeploymentCircuitBreaker deploymentCircuitBreaker) {
            this.deploymentCircuitBreaker = deploymentCircuitBreaker;
            return this;
        }

        public final ServiceDeploymentAlarms.Builder getAlarms() {
            return alarms != null ? alarms.toBuilder() : null;
        }

        public final void setAlarms(ServiceDeploymentAlarms.BuilderImpl alarms) {
            this.alarms = alarms != null ? alarms.build() : null;
        }

        @Override
        public final Builder alarms(ServiceDeploymentAlarms alarms) {
            this.alarms = alarms;
            return this;
        }

        @Override
        public ServiceDeployment build() {
            return new ServiceDeployment(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
