/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A Docker container that is part of a task.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Container implements SdkPojo, Serializable, ToCopyableBuilder<Container.Builder, Container> {
    private static final SdkField<String> CONTAINER_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Container::containerArn)).setter(setter(Builder::containerArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerArn").build()).build();

    private static final SdkField<String> TASK_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Container::taskArn)).setter(setter(Builder::taskArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("taskArn").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Container::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> LAST_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Container::lastStatus)).setter(setter(Builder::lastStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastStatus").build()).build();

    private static final SdkField<Integer> EXIT_CODE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(Container::exitCode)).setter(setter(Builder::exitCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("exitCode").build()).build();

    private static final SdkField<String> REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Container::reason)).setter(setter(Builder::reason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("reason").build()).build();

    private static final SdkField<List<NetworkBinding>> NETWORK_BINDINGS_FIELD = SdkField
            .<List<NetworkBinding>> builder(MarshallingType.LIST)
            .getter(getter(Container::networkBindings))
            .setter(setter(Builder::networkBindings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("networkBindings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<NetworkBinding> builder(MarshallingType.SDK_POJO)
                                            .constructor(NetworkBinding::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<NetworkInterface>> NETWORK_INTERFACES_FIELD = SdkField
            .<List<NetworkInterface>> builder(MarshallingType.LIST)
            .getter(getter(Container::networkInterfaces))
            .setter(setter(Builder::networkInterfaces))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("networkInterfaces").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<NetworkInterface> builder(MarshallingType.SDK_POJO)
                                            .constructor(NetworkInterface::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> HEALTH_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Container::healthStatusAsString)).setter(setter(Builder::healthStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("healthStatus").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONTAINER_ARN_FIELD,
            TASK_ARN_FIELD, NAME_FIELD, LAST_STATUS_FIELD, EXIT_CODE_FIELD, REASON_FIELD, NETWORK_BINDINGS_FIELD,
            NETWORK_INTERFACES_FIELD, HEALTH_STATUS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String containerArn;

    private final String taskArn;

    private final String name;

    private final String lastStatus;

    private final Integer exitCode;

    private final String reason;

    private final List<NetworkBinding> networkBindings;

    private final List<NetworkInterface> networkInterfaces;

    private final String healthStatus;

    private Container(BuilderImpl builder) {
        this.containerArn = builder.containerArn;
        this.taskArn = builder.taskArn;
        this.name = builder.name;
        this.lastStatus = builder.lastStatus;
        this.exitCode = builder.exitCode;
        this.reason = builder.reason;
        this.networkBindings = builder.networkBindings;
        this.networkInterfaces = builder.networkInterfaces;
        this.healthStatus = builder.healthStatus;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the container.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the container.
     */
    public String containerArn() {
        return containerArn;
    }

    /**
     * <p>
     * The ARN of the task.
     * </p>
     * 
     * @return The ARN of the task.
     */
    public String taskArn() {
        return taskArn;
    }

    /**
     * <p>
     * The name of the container.
     * </p>
     * 
     * @return The name of the container.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The last known status of the container.
     * </p>
     * 
     * @return The last known status of the container.
     */
    public String lastStatus() {
        return lastStatus;
    }

    /**
     * <p>
     * The exit code returned from the container.
     * </p>
     * 
     * @return The exit code returned from the container.
     */
    public Integer exitCode() {
        return exitCode;
    }

    /**
     * <p>
     * A short (255 max characters) human-readable string to provide additional details about a running or stopped
     * container.
     * </p>
     * 
     * @return A short (255 max characters) human-readable string to provide additional details about a running or
     *         stopped container.
     */
    public String reason() {
        return reason;
    }

    /**
     * <p>
     * The network bindings associated with the container.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The network bindings associated with the container.
     */
    public List<NetworkBinding> networkBindings() {
        return networkBindings;
    }

    /**
     * <p>
     * The network interfaces associated with the container.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The network interfaces associated with the container.
     */
    public List<NetworkInterface> networkInterfaces() {
        return networkInterfaces;
    }

    /**
     * <p>
     * The health status of the container. If health checks are not configured for this container in its task
     * definition, then it reports the health status as <code>UNKNOWN</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #healthStatus} will
     * return {@link HealthStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #healthStatusAsString}.
     * </p>
     * 
     * @return The health status of the container. If health checks are not configured for this container in its task
     *         definition, then it reports the health status as <code>UNKNOWN</code>.
     * @see HealthStatus
     */
    public HealthStatus healthStatus() {
        return HealthStatus.fromValue(healthStatus);
    }

    /**
     * <p>
     * The health status of the container. If health checks are not configured for this container in its task
     * definition, then it reports the health status as <code>UNKNOWN</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #healthStatus} will
     * return {@link HealthStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #healthStatusAsString}.
     * </p>
     * 
     * @return The health status of the container. If health checks are not configured for this container in its task
     *         definition, then it reports the health status as <code>UNKNOWN</code>.
     * @see HealthStatus
     */
    public String healthStatusAsString() {
        return healthStatus;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(containerArn());
        hashCode = 31 * hashCode + Objects.hashCode(taskArn());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(lastStatus());
        hashCode = 31 * hashCode + Objects.hashCode(exitCode());
        hashCode = 31 * hashCode + Objects.hashCode(reason());
        hashCode = 31 * hashCode + Objects.hashCode(networkBindings());
        hashCode = 31 * hashCode + Objects.hashCode(networkInterfaces());
        hashCode = 31 * hashCode + Objects.hashCode(healthStatusAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Container)) {
            return false;
        }
        Container other = (Container) obj;
        return Objects.equals(containerArn(), other.containerArn()) && Objects.equals(taskArn(), other.taskArn())
                && Objects.equals(name(), other.name()) && Objects.equals(lastStatus(), other.lastStatus())
                && Objects.equals(exitCode(), other.exitCode()) && Objects.equals(reason(), other.reason())
                && Objects.equals(networkBindings(), other.networkBindings())
                && Objects.equals(networkInterfaces(), other.networkInterfaces())
                && Objects.equals(healthStatusAsString(), other.healthStatusAsString());
    }

    @Override
    public String toString() {
        return ToString.builder("Container").add("ContainerArn", containerArn()).add("TaskArn", taskArn()).add("Name", name())
                .add("LastStatus", lastStatus()).add("ExitCode", exitCode()).add("Reason", reason())
                .add("NetworkBindings", networkBindings()).add("NetworkInterfaces", networkInterfaces())
                .add("HealthStatus", healthStatusAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "containerArn":
            return Optional.ofNullable(clazz.cast(containerArn()));
        case "taskArn":
            return Optional.ofNullable(clazz.cast(taskArn()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "lastStatus":
            return Optional.ofNullable(clazz.cast(lastStatus()));
        case "exitCode":
            return Optional.ofNullable(clazz.cast(exitCode()));
        case "reason":
            return Optional.ofNullable(clazz.cast(reason()));
        case "networkBindings":
            return Optional.ofNullable(clazz.cast(networkBindings()));
        case "networkInterfaces":
            return Optional.ofNullable(clazz.cast(networkInterfaces()));
        case "healthStatus":
            return Optional.ofNullable(clazz.cast(healthStatusAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Container, T> g) {
        return obj -> g.apply((Container) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Container> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the container.
         * </p>
         * 
         * @param containerArn
         *        The Amazon Resource Name (ARN) of the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerArn(String containerArn);

        /**
         * <p>
         * The ARN of the task.
         * </p>
         * 
         * @param taskArn
         *        The ARN of the task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskArn(String taskArn);

        /**
         * <p>
         * The name of the container.
         * </p>
         * 
         * @param name
         *        The name of the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The last known status of the container.
         * </p>
         * 
         * @param lastStatus
         *        The last known status of the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastStatus(String lastStatus);

        /**
         * <p>
         * The exit code returned from the container.
         * </p>
         * 
         * @param exitCode
         *        The exit code returned from the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exitCode(Integer exitCode);

        /**
         * <p>
         * A short (255 max characters) human-readable string to provide additional details about a running or stopped
         * container.
         * </p>
         * 
         * @param reason
         *        A short (255 max characters) human-readable string to provide additional details about a running or
         *        stopped container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reason(String reason);

        /**
         * <p>
         * The network bindings associated with the container.
         * </p>
         * 
         * @param networkBindings
         *        The network bindings associated with the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkBindings(Collection<NetworkBinding> networkBindings);

        /**
         * <p>
         * The network bindings associated with the container.
         * </p>
         * 
         * @param networkBindings
         *        The network bindings associated with the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkBindings(NetworkBinding... networkBindings);

        /**
         * <p>
         * The network bindings associated with the container.
         * </p>
         * This is a convenience that creates an instance of the {@link List<NetworkBinding>.Builder} avoiding the need
         * to create one manually via {@link List<NetworkBinding>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<NetworkBinding>.Builder#build()} is called immediately and
         * its result is passed to {@link #networkBindings(List<NetworkBinding>)}.
         * 
         * @param networkBindings
         *        a consumer that will call methods on {@link List<NetworkBinding>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkBindings(List<NetworkBinding>)
         */
        Builder networkBindings(Consumer<NetworkBinding.Builder>... networkBindings);

        /**
         * <p>
         * The network interfaces associated with the container.
         * </p>
         * 
         * @param networkInterfaces
         *        The network interfaces associated with the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaces(Collection<NetworkInterface> networkInterfaces);

        /**
         * <p>
         * The network interfaces associated with the container.
         * </p>
         * 
         * @param networkInterfaces
         *        The network interfaces associated with the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaces(NetworkInterface... networkInterfaces);

        /**
         * <p>
         * The network interfaces associated with the container.
         * </p>
         * This is a convenience that creates an instance of the {@link List<NetworkInterface>.Builder} avoiding the
         * need to create one manually via {@link List<NetworkInterface>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<NetworkInterface>.Builder#build()} is called immediately and
         * its result is passed to {@link #networkInterfaces(List<NetworkInterface>)}.
         * 
         * @param networkInterfaces
         *        a consumer that will call methods on {@link List<NetworkInterface>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkInterfaces(List<NetworkInterface>)
         */
        Builder networkInterfaces(Consumer<NetworkInterface.Builder>... networkInterfaces);

        /**
         * <p>
         * The health status of the container. If health checks are not configured for this container in its task
         * definition, then it reports the health status as <code>UNKNOWN</code>.
         * </p>
         * 
         * @param healthStatus
         *        The health status of the container. If health checks are not configured for this container in its task
         *        definition, then it reports the health status as <code>UNKNOWN</code>.
         * @see HealthStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HealthStatus
         */
        Builder healthStatus(String healthStatus);

        /**
         * <p>
         * The health status of the container. If health checks are not configured for this container in its task
         * definition, then it reports the health status as <code>UNKNOWN</code>.
         * </p>
         * 
         * @param healthStatus
         *        The health status of the container. If health checks are not configured for this container in its task
         *        definition, then it reports the health status as <code>UNKNOWN</code>.
         * @see HealthStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HealthStatus
         */
        Builder healthStatus(HealthStatus healthStatus);
    }

    static final class BuilderImpl implements Builder {
        private String containerArn;

        private String taskArn;

        private String name;

        private String lastStatus;

        private Integer exitCode;

        private String reason;

        private List<NetworkBinding> networkBindings = DefaultSdkAutoConstructList.getInstance();

        private List<NetworkInterface> networkInterfaces = DefaultSdkAutoConstructList.getInstance();

        private String healthStatus;

        private BuilderImpl() {
        }

        private BuilderImpl(Container model) {
            containerArn(model.containerArn);
            taskArn(model.taskArn);
            name(model.name);
            lastStatus(model.lastStatus);
            exitCode(model.exitCode);
            reason(model.reason);
            networkBindings(model.networkBindings);
            networkInterfaces(model.networkInterfaces);
            healthStatus(model.healthStatus);
        }

        public final String getContainerArn() {
            return containerArn;
        }

        @Override
        public final Builder containerArn(String containerArn) {
            this.containerArn = containerArn;
            return this;
        }

        public final void setContainerArn(String containerArn) {
            this.containerArn = containerArn;
        }

        public final String getTaskArn() {
            return taskArn;
        }

        @Override
        public final Builder taskArn(String taskArn) {
            this.taskArn = taskArn;
            return this;
        }

        public final void setTaskArn(String taskArn) {
            this.taskArn = taskArn;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getLastStatus() {
            return lastStatus;
        }

        @Override
        public final Builder lastStatus(String lastStatus) {
            this.lastStatus = lastStatus;
            return this;
        }

        public final void setLastStatus(String lastStatus) {
            this.lastStatus = lastStatus;
        }

        public final Integer getExitCode() {
            return exitCode;
        }

        @Override
        public final Builder exitCode(Integer exitCode) {
            this.exitCode = exitCode;
            return this;
        }

        public final void setExitCode(Integer exitCode) {
            this.exitCode = exitCode;
        }

        public final String getReason() {
            return reason;
        }

        @Override
        public final Builder reason(String reason) {
            this.reason = reason;
            return this;
        }

        public final void setReason(String reason) {
            this.reason = reason;
        }

        public final Collection<NetworkBinding.Builder> getNetworkBindings() {
            return networkBindings != null ? networkBindings.stream().map(NetworkBinding::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder networkBindings(Collection<NetworkBinding> networkBindings) {
            this.networkBindings = NetworkBindingsCopier.copy(networkBindings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkBindings(NetworkBinding... networkBindings) {
            networkBindings(Arrays.asList(networkBindings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkBindings(Consumer<NetworkBinding.Builder>... networkBindings) {
            networkBindings(Stream.of(networkBindings).map(c -> NetworkBinding.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setNetworkBindings(Collection<NetworkBinding.BuilderImpl> networkBindings) {
            this.networkBindings = NetworkBindingsCopier.copyFromBuilder(networkBindings);
        }

        public final Collection<NetworkInterface.Builder> getNetworkInterfaces() {
            return networkInterfaces != null ? networkInterfaces.stream().map(NetworkInterface::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder networkInterfaces(Collection<NetworkInterface> networkInterfaces) {
            this.networkInterfaces = NetworkInterfacesCopier.copy(networkInterfaces);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkInterfaces(NetworkInterface... networkInterfaces) {
            networkInterfaces(Arrays.asList(networkInterfaces));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkInterfaces(Consumer<NetworkInterface.Builder>... networkInterfaces) {
            networkInterfaces(Stream.of(networkInterfaces).map(c -> NetworkInterface.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setNetworkInterfaces(Collection<NetworkInterface.BuilderImpl> networkInterfaces) {
            this.networkInterfaces = NetworkInterfacesCopier.copyFromBuilder(networkInterfaces);
        }

        public final String getHealthStatusAsString() {
            return healthStatus;
        }

        @Override
        public final Builder healthStatus(String healthStatus) {
            this.healthStatus = healthStatus;
            return this;
        }

        @Override
        public final Builder healthStatus(HealthStatus healthStatus) {
            this.healthStatus(healthStatus.toString());
            return this;
        }

        public final void setHealthStatus(String healthStatus) {
            this.healthStatus = healthStatus;
        }

        @Override
        public Container build() {
            return new Container(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
