/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the resources available for a container instance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Resource implements SdkPojo, Serializable, ToCopyableBuilder<Resource.Builder, Resource> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Resource::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Resource::type)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<Double> DOUBLE_VALUE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .getter(getter(Resource::doubleValue)).setter(setter(Builder::doubleValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("doubleValue").build()).build();

    private static final SdkField<Long> LONG_VALUE_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .getter(getter(Resource::longValue)).setter(setter(Builder::longValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("longValue").build()).build();

    private static final SdkField<Integer> INTEGER_VALUE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(Resource::integerValue)).setter(setter(Builder::integerValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("integerValue").build()).build();

    private static final SdkField<List<String>> STRING_SET_VALUE_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(Resource::stringSetValue))
            .setter(setter(Builder::stringSetValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("stringSetValue").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, TYPE_FIELD,
            DOUBLE_VALUE_FIELD, LONG_VALUE_FIELD, INTEGER_VALUE_FIELD, STRING_SET_VALUE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String type;

    private final Double doubleValue;

    private final Long longValue;

    private final Integer integerValue;

    private final List<String> stringSetValue;

    private Resource(BuilderImpl builder) {
        this.name = builder.name;
        this.type = builder.type;
        this.doubleValue = builder.doubleValue;
        this.longValue = builder.longValue;
        this.integerValue = builder.integerValue;
        this.stringSetValue = builder.stringSetValue;
    }

    /**
     * <p>
     * The name of the resource, such as <code>CPU</code>, <code>MEMORY</code>, <code>PORTS</code>,
     * <code>PORTS_UDP</code>, or a user-defined resource.
     * </p>
     * 
     * @return The name of the resource, such as <code>CPU</code>, <code>MEMORY</code>, <code>PORTS</code>,
     *         <code>PORTS_UDP</code>, or a user-defined resource.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The type of the resource, such as <code>INTEGER</code>, <code>DOUBLE</code>, <code>LONG</code>, or
     * <code>STRINGSET</code>.
     * </p>
     * 
     * @return The type of the resource, such as <code>INTEGER</code>, <code>DOUBLE</code>, <code>LONG</code>, or
     *         <code>STRINGSET</code>.
     */
    public String type() {
        return type;
    }

    /**
     * <p>
     * When the <code>doubleValue</code> type is set, the value of the resource must be a double precision
     * floating-point type.
     * </p>
     * 
     * @return When the <code>doubleValue</code> type is set, the value of the resource must be a double precision
     *         floating-point type.
     */
    public Double doubleValue() {
        return doubleValue;
    }

    /**
     * <p>
     * When the <code>longValue</code> type is set, the value of the resource must be an extended precision
     * floating-point type.
     * </p>
     * 
     * @return When the <code>longValue</code> type is set, the value of the resource must be an extended precision
     *         floating-point type.
     */
    public Long longValue() {
        return longValue;
    }

    /**
     * <p>
     * When the <code>integerValue</code> type is set, the value of the resource must be an integer.
     * </p>
     * 
     * @return When the <code>integerValue</code> type is set, the value of the resource must be an integer.
     */
    public Integer integerValue() {
        return integerValue;
    }

    /**
     * <p>
     * When the <code>stringSetValue</code> type is set, the value of the resource must be a string type.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return When the <code>stringSetValue</code> type is set, the value of the resource must be a string type.
     */
    public List<String> stringSetValue() {
        return stringSetValue;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(type());
        hashCode = 31 * hashCode + Objects.hashCode(doubleValue());
        hashCode = 31 * hashCode + Objects.hashCode(longValue());
        hashCode = 31 * hashCode + Objects.hashCode(integerValue());
        hashCode = 31 * hashCode + Objects.hashCode(stringSetValue());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Resource)) {
            return false;
        }
        Resource other = (Resource) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(type(), other.type())
                && Objects.equals(doubleValue(), other.doubleValue()) && Objects.equals(longValue(), other.longValue())
                && Objects.equals(integerValue(), other.integerValue())
                && Objects.equals(stringSetValue(), other.stringSetValue());
    }

    @Override
    public String toString() {
        return ToString.builder("Resource").add("Name", name()).add("Type", type()).add("DoubleValue", doubleValue())
                .add("LongValue", longValue()).add("IntegerValue", integerValue()).add("StringSetValue", stringSetValue())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "type":
            return Optional.ofNullable(clazz.cast(type()));
        case "doubleValue":
            return Optional.ofNullable(clazz.cast(doubleValue()));
        case "longValue":
            return Optional.ofNullable(clazz.cast(longValue()));
        case "integerValue":
            return Optional.ofNullable(clazz.cast(integerValue()));
        case "stringSetValue":
            return Optional.ofNullable(clazz.cast(stringSetValue()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Resource, T> g) {
        return obj -> g.apply((Resource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Resource> {
        /**
         * <p>
         * The name of the resource, such as <code>CPU</code>, <code>MEMORY</code>, <code>PORTS</code>,
         * <code>PORTS_UDP</code>, or a user-defined resource.
         * </p>
         * 
         * @param name
         *        The name of the resource, such as <code>CPU</code>, <code>MEMORY</code>, <code>PORTS</code>,
         *        <code>PORTS_UDP</code>, or a user-defined resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The type of the resource, such as <code>INTEGER</code>, <code>DOUBLE</code>, <code>LONG</code>, or
         * <code>STRINGSET</code>.
         * </p>
         * 
         * @param type
         *        The type of the resource, such as <code>INTEGER</code>, <code>DOUBLE</code>, <code>LONG</code>, or
         *        <code>STRINGSET</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder type(String type);

        /**
         * <p>
         * When the <code>doubleValue</code> type is set, the value of the resource must be a double precision
         * floating-point type.
         * </p>
         * 
         * @param doubleValue
         *        When the <code>doubleValue</code> type is set, the value of the resource must be a double precision
         *        floating-point type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder doubleValue(Double doubleValue);

        /**
         * <p>
         * When the <code>longValue</code> type is set, the value of the resource must be an extended precision
         * floating-point type.
         * </p>
         * 
         * @param longValue
         *        When the <code>longValue</code> type is set, the value of the resource must be an extended precision
         *        floating-point type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder longValue(Long longValue);

        /**
         * <p>
         * When the <code>integerValue</code> type is set, the value of the resource must be an integer.
         * </p>
         * 
         * @param integerValue
         *        When the <code>integerValue</code> type is set, the value of the resource must be an integer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder integerValue(Integer integerValue);

        /**
         * <p>
         * When the <code>stringSetValue</code> type is set, the value of the resource must be a string type.
         * </p>
         * 
         * @param stringSetValue
         *        When the <code>stringSetValue</code> type is set, the value of the resource must be a string type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stringSetValue(Collection<String> stringSetValue);

        /**
         * <p>
         * When the <code>stringSetValue</code> type is set, the value of the resource must be a string type.
         * </p>
         * 
         * @param stringSetValue
         *        When the <code>stringSetValue</code> type is set, the value of the resource must be a string type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stringSetValue(String... stringSetValue);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String type;

        private Double doubleValue;

        private Long longValue;

        private Integer integerValue;

        private List<String> stringSetValue = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Resource model) {
            name(model.name);
            type(model.type);
            doubleValue(model.doubleValue);
            longValue(model.longValue);
            integerValue(model.integerValue);
            stringSetValue(model.stringSetValue);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getType() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final Double getDoubleValue() {
            return doubleValue;
        }

        @Override
        public final Builder doubleValue(Double doubleValue) {
            this.doubleValue = doubleValue;
            return this;
        }

        public final void setDoubleValue(Double doubleValue) {
            this.doubleValue = doubleValue;
        }

        public final Long getLongValue() {
            return longValue;
        }

        @Override
        public final Builder longValue(Long longValue) {
            this.longValue = longValue;
            return this;
        }

        public final void setLongValue(Long longValue) {
            this.longValue = longValue;
        }

        public final Integer getIntegerValue() {
            return integerValue;
        }

        @Override
        public final Builder integerValue(Integer integerValue) {
            this.integerValue = integerValue;
            return this;
        }

        public final void setIntegerValue(Integer integerValue) {
            this.integerValue = integerValue;
        }

        public final Collection<String> getStringSetValue() {
            return stringSetValue;
        }

        @Override
        public final Builder stringSetValue(Collection<String> stringSetValue) {
            this.stringSetValue = StringListCopier.copy(stringSetValue);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stringSetValue(String... stringSetValue) {
            stringSetValue(Arrays.asList(stringSetValue));
            return this;
        }

        public final void setStringSetValue(Collection<String> stringSetValue) {
            this.stringSetValue = StringListCopier.copy(stringSetValue);
        }

        @Override
        public Resource build() {
            return new Resource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
