/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class StartTaskRequest extends EcsRequest implements ToCopyableBuilder<StartTaskRequest.Builder, StartTaskRequest> {
    private static final SdkField<String> CLUSTER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StartTaskRequest::cluster)).setter(setter(Builder::cluster))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cluster").build()).build();

    private static final SdkField<String> TASK_DEFINITION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StartTaskRequest::taskDefinition)).setter(setter(Builder::taskDefinition))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("taskDefinition").build()).build();

    private static final SdkField<TaskOverride> OVERRIDES_FIELD = SdkField.<TaskOverride> builder(MarshallingType.SDK_POJO)
            .getter(getter(StartTaskRequest::overrides)).setter(setter(Builder::overrides)).constructor(TaskOverride::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("overrides").build()).build();

    private static final SdkField<List<String>> CONTAINER_INSTANCES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(StartTaskRequest::containerInstances))
            .setter(setter(Builder::containerInstances))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerInstances").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STARTED_BY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StartTaskRequest::startedBy)).setter(setter(Builder::startedBy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startedBy").build()).build();

    private static final SdkField<String> GROUP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StartTaskRequest::group)).setter(setter(Builder::group))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("group").build()).build();

    private static final SdkField<NetworkConfiguration> NETWORK_CONFIGURATION_FIELD = SdkField
            .<NetworkConfiguration> builder(MarshallingType.SDK_POJO).getter(getter(StartTaskRequest::networkConfiguration))
            .setter(setter(Builder::networkConfiguration)).constructor(NetworkConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("networkConfiguration").build())
            .build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .getter(getter(StartTaskRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> ENABLE_ECS_MANAGED_TAGS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(StartTaskRequest::enableECSManagedTags)).setter(setter(Builder::enableECSManagedTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("enableECSManagedTags").build())
            .build();

    private static final SdkField<String> PROPAGATE_TAGS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StartTaskRequest::propagateTagsAsString)).setter(setter(Builder::propagateTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("propagateTags").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CLUSTER_FIELD,
            TASK_DEFINITION_FIELD, OVERRIDES_FIELD, CONTAINER_INSTANCES_FIELD, STARTED_BY_FIELD, GROUP_FIELD,
            NETWORK_CONFIGURATION_FIELD, TAGS_FIELD, ENABLE_ECS_MANAGED_TAGS_FIELD, PROPAGATE_TAGS_FIELD));

    private final String cluster;

    private final String taskDefinition;

    private final TaskOverride overrides;

    private final List<String> containerInstances;

    private final String startedBy;

    private final String group;

    private final NetworkConfiguration networkConfiguration;

    private final List<Tag> tags;

    private final Boolean enableECSManagedTags;

    private final String propagateTags;

    private StartTaskRequest(BuilderImpl builder) {
        super(builder);
        this.cluster = builder.cluster;
        this.taskDefinition = builder.taskDefinition;
        this.overrides = builder.overrides;
        this.containerInstances = builder.containerInstances;
        this.startedBy = builder.startedBy;
        this.group = builder.group;
        this.networkConfiguration = builder.networkConfiguration;
        this.tags = builder.tags;
        this.enableECSManagedTags = builder.enableECSManagedTags;
        this.propagateTags = builder.propagateTags;
    }

    /**
     * <p>
     * The short name or full Amazon Resource Name (ARN) of the cluster on which to start your task. If you do not
     * specify a cluster, the default cluster is assumed.
     * </p>
     * 
     * @return The short name or full Amazon Resource Name (ARN) of the cluster on which to start your task. If you do
     *         not specify a cluster, the default cluster is assumed.
     */
    public String cluster() {
        return cluster;
    }

    /**
     * <p>
     * The <code>family</code> and <code>revision</code> (<code>family:revision</code>) or full ARN of the task
     * definition to start. If a <code>revision</code> is not specified, the latest <code>ACTIVE</code> revision is
     * used.
     * </p>
     * 
     * @return The <code>family</code> and <code>revision</code> (<code>family:revision</code>) or full ARN of the task
     *         definition to start. If a <code>revision</code> is not specified, the latest <code>ACTIVE</code> revision
     *         is used.
     */
    public String taskDefinition() {
        return taskDefinition;
    }

    /**
     * <p>
     * A list of container overrides in JSON format that specify the name of a container in the specified task
     * definition and the overrides it should receive. You can override the default command for a container (that is
     * specified in the task definition or Docker image) with a <code>command</code> override. You can also override
     * existing environment variables (that are specified in the task definition or Docker image) on a container or add
     * new environment variables to it with an <code>environment</code> override.
     * </p>
     * <note>
     * <p>
     * A total of 8192 characters are allowed for overrides. This limit includes the JSON formatting characters of the
     * override structure.
     * </p>
     * </note>
     * 
     * @return A list of container overrides in JSON format that specify the name of a container in the specified task
     *         definition and the overrides it should receive. You can override the default command for a container
     *         (that is specified in the task definition or Docker image) with a <code>command</code> override. You can
     *         also override existing environment variables (that are specified in the task definition or Docker image)
     *         on a container or add new environment variables to it with an <code>environment</code> override.</p>
     *         <note>
     *         <p>
     *         A total of 8192 characters are allowed for overrides. This limit includes the JSON formatting characters
     *         of the override structure.
     *         </p>
     */
    public TaskOverride overrides() {
        return overrides;
    }

    /**
     * <p>
     * The container instance IDs or full ARN entries for the container instances on which you would like to place your
     * task. You can specify up to 10 container instances.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The container instance IDs or full ARN entries for the container instances on which you would like to
     *         place your task. You can specify up to 10 container instances.
     */
    public List<String> containerInstances() {
        return containerInstances;
    }

    /**
     * <p>
     * An optional tag specified when a task is started. For example, if you automatically trigger a task to run a batch
     * process job, you could apply a unique identifier for that job to your task with the <code>startedBy</code>
     * parameter. You can then identify which tasks belong to that job by filtering the results of a <a>ListTasks</a>
     * call with the <code>startedBy</code> value. Up to 36 letters (uppercase and lowercase), numbers, hyphens, and
     * underscores are allowed.
     * </p>
     * <p>
     * If a task is started by an Amazon ECS service, then the <code>startedBy</code> parameter contains the deployment
     * ID of the service that starts it.
     * </p>
     * 
     * @return An optional tag specified when a task is started. For example, if you automatically trigger a task to run
     *         a batch process job, you could apply a unique identifier for that job to your task with the
     *         <code>startedBy</code> parameter. You can then identify which tasks belong to that job by filtering the
     *         results of a <a>ListTasks</a> call with the <code>startedBy</code> value. Up to 36 letters (uppercase and
     *         lowercase), numbers, hyphens, and underscores are allowed.</p>
     *         <p>
     *         If a task is started by an Amazon ECS service, then the <code>startedBy</code> parameter contains the
     *         deployment ID of the service that starts it.
     */
    public String startedBy() {
        return startedBy;
    }

    /**
     * <p>
     * The name of the task group to associate with the task. The default value is the family name of the task
     * definition (for example, family:my-family-name).
     * </p>
     * 
     * @return The name of the task group to associate with the task. The default value is the family name of the task
     *         definition (for example, family:my-family-name).
     */
    public String group() {
        return group;
    }

    /**
     * <p>
     * The VPC subnet and security group configuration for tasks that receive their own elastic network interface by
     * using the <code>awsvpc</code> networking mode.
     * </p>
     * 
     * @return The VPC subnet and security group configuration for tasks that receive their own elastic network
     *         interface by using the <code>awsvpc</code> networking mode.
     */
    public NetworkConfiguration networkConfiguration() {
        return networkConfiguration;
    }

    /**
     * <p>
     * The metadata that you apply to the task to help you categorize and organize them. Each tag consists of a key and
     * an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and
     * tag values can have a maximum length of 256 characters.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The metadata that you apply to the task to help you categorize and organize them. Each tag consists of a
     *         key and an optional value, both of which you define. Tag keys can have a maximum character length of 128
     *         characters, and tag values can have a maximum length of 256 characters.
     */
    public List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * Specifies whether to enable Amazon ECS managed tags for the task. For more information, see <a
     * href="http://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html">Tagging Your Amazon ECS
     * Resources</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * 
     * @return Specifies whether to enable Amazon ECS managed tags for the task. For more information, see <a
     *         href="http://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html">Tagging Your Amazon
     *         ECS Resources</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     */
    public Boolean enableECSManagedTags() {
        return enableECSManagedTags;
    }

    /**
     * <p>
     * Specifies whether to propagate the tags from the task definition or the service to the task. If no value is
     * specified, the tags are not propagated.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #propagateTags}
     * will return {@link PropagateTags#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #propagateTagsAsString}.
     * </p>
     * 
     * @return Specifies whether to propagate the tags from the task definition or the service to the task. If no value
     *         is specified, the tags are not propagated.
     * @see PropagateTags
     */
    public PropagateTags propagateTags() {
        return PropagateTags.fromValue(propagateTags);
    }

    /**
     * <p>
     * Specifies whether to propagate the tags from the task definition or the service to the task. If no value is
     * specified, the tags are not propagated.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #propagateTags}
     * will return {@link PropagateTags#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #propagateTagsAsString}.
     * </p>
     * 
     * @return Specifies whether to propagate the tags from the task definition or the service to the task. If no value
     *         is specified, the tags are not propagated.
     * @see PropagateTags
     */
    public String propagateTagsAsString() {
        return propagateTags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(cluster());
        hashCode = 31 * hashCode + Objects.hashCode(taskDefinition());
        hashCode = 31 * hashCode + Objects.hashCode(overrides());
        hashCode = 31 * hashCode + Objects.hashCode(containerInstances());
        hashCode = 31 * hashCode + Objects.hashCode(startedBy());
        hashCode = 31 * hashCode + Objects.hashCode(group());
        hashCode = 31 * hashCode + Objects.hashCode(networkConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(enableECSManagedTags());
        hashCode = 31 * hashCode + Objects.hashCode(propagateTagsAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StartTaskRequest)) {
            return false;
        }
        StartTaskRequest other = (StartTaskRequest) obj;
        return Objects.equals(cluster(), other.cluster()) && Objects.equals(taskDefinition(), other.taskDefinition())
                && Objects.equals(overrides(), other.overrides())
                && Objects.equals(containerInstances(), other.containerInstances())
                && Objects.equals(startedBy(), other.startedBy()) && Objects.equals(group(), other.group())
                && Objects.equals(networkConfiguration(), other.networkConfiguration()) && Objects.equals(tags(), other.tags())
                && Objects.equals(enableECSManagedTags(), other.enableECSManagedTags())
                && Objects.equals(propagateTagsAsString(), other.propagateTagsAsString());
    }

    @Override
    public String toString() {
        return ToString.builder("StartTaskRequest").add("Cluster", cluster()).add("TaskDefinition", taskDefinition())
                .add("Overrides", overrides()).add("ContainerInstances", containerInstances()).add("StartedBy", startedBy())
                .add("Group", group()).add("NetworkConfiguration", networkConfiguration()).add("Tags", tags())
                .add("EnableECSManagedTags", enableECSManagedTags()).add("PropagateTags", propagateTagsAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "cluster":
            return Optional.ofNullable(clazz.cast(cluster()));
        case "taskDefinition":
            return Optional.ofNullable(clazz.cast(taskDefinition()));
        case "overrides":
            return Optional.ofNullable(clazz.cast(overrides()));
        case "containerInstances":
            return Optional.ofNullable(clazz.cast(containerInstances()));
        case "startedBy":
            return Optional.ofNullable(clazz.cast(startedBy()));
        case "group":
            return Optional.ofNullable(clazz.cast(group()));
        case "networkConfiguration":
            return Optional.ofNullable(clazz.cast(networkConfiguration()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "enableECSManagedTags":
            return Optional.ofNullable(clazz.cast(enableECSManagedTags()));
        case "propagateTags":
            return Optional.ofNullable(clazz.cast(propagateTagsAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StartTaskRequest, T> g) {
        return obj -> g.apply((StartTaskRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends EcsRequest.Builder, SdkPojo, CopyableBuilder<Builder, StartTaskRequest> {
        /**
         * <p>
         * The short name or full Amazon Resource Name (ARN) of the cluster on which to start your task. If you do not
         * specify a cluster, the default cluster is assumed.
         * </p>
         * 
         * @param cluster
         *        The short name or full Amazon Resource Name (ARN) of the cluster on which to start your task. If you
         *        do not specify a cluster, the default cluster is assumed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cluster(String cluster);

        /**
         * <p>
         * The <code>family</code> and <code>revision</code> (<code>family:revision</code>) or full ARN of the task
         * definition to start. If a <code>revision</code> is not specified, the latest <code>ACTIVE</code> revision is
         * used.
         * </p>
         * 
         * @param taskDefinition
         *        The <code>family</code> and <code>revision</code> (<code>family:revision</code>) or full ARN of the
         *        task definition to start. If a <code>revision</code> is not specified, the latest <code>ACTIVE</code>
         *        revision is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskDefinition(String taskDefinition);

        /**
         * <p>
         * A list of container overrides in JSON format that specify the name of a container in the specified task
         * definition and the overrides it should receive. You can override the default command for a container (that is
         * specified in the task definition or Docker image) with a <code>command</code> override. You can also override
         * existing environment variables (that are specified in the task definition or Docker image) on a container or
         * add new environment variables to it with an <code>environment</code> override.
         * </p>
         * <note>
         * <p>
         * A total of 8192 characters are allowed for overrides. This limit includes the JSON formatting characters of
         * the override structure.
         * </p>
         * </note>
         * 
         * @param overrides
         *        A list of container overrides in JSON format that specify the name of a container in the specified
         *        task definition and the overrides it should receive. You can override the default command for a
         *        container (that is specified in the task definition or Docker image) with a <code>command</code>
         *        override. You can also override existing environment variables (that are specified in the task
         *        definition or Docker image) on a container or add new environment variables to it with an
         *        <code>environment</code> override.</p> <note>
         *        <p>
         *        A total of 8192 characters are allowed for overrides. This limit includes the JSON formatting
         *        characters of the override structure.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder overrides(TaskOverride overrides);

        /**
         * <p>
         * A list of container overrides in JSON format that specify the name of a container in the specified task
         * definition and the overrides it should receive. You can override the default command for a container (that is
         * specified in the task definition or Docker image) with a <code>command</code> override. You can also override
         * existing environment variables (that are specified in the task definition or Docker image) on a container or
         * add new environment variables to it with an <code>environment</code> override.
         * </p>
         * <note>
         * <p>
         * A total of 8192 characters are allowed for overrides. This limit includes the JSON formatting characters of
         * the override structure.
         * </p>
         * </note> This is a convenience that creates an instance of the {@link TaskOverride.Builder} avoiding the need
         * to create one manually via {@link TaskOverride#builder()}.
         *
         * When the {@link Consumer} completes, {@link TaskOverride.Builder#build()} is called immediately and its
         * result is passed to {@link #overrides(TaskOverride)}.
         * 
         * @param overrides
         *        a consumer that will call methods on {@link TaskOverride.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #overrides(TaskOverride)
         */
        default Builder overrides(Consumer<TaskOverride.Builder> overrides) {
            return overrides(TaskOverride.builder().applyMutation(overrides).build());
        }

        /**
         * <p>
         * The container instance IDs or full ARN entries for the container instances on which you would like to place
         * your task. You can specify up to 10 container instances.
         * </p>
         * 
         * @param containerInstances
         *        The container instance IDs or full ARN entries for the container instances on which you would like to
         *        place your task. You can specify up to 10 container instances.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerInstances(Collection<String> containerInstances);

        /**
         * <p>
         * The container instance IDs or full ARN entries for the container instances on which you would like to place
         * your task. You can specify up to 10 container instances.
         * </p>
         * 
         * @param containerInstances
         *        The container instance IDs or full ARN entries for the container instances on which you would like to
         *        place your task. You can specify up to 10 container instances.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerInstances(String... containerInstances);

        /**
         * <p>
         * An optional tag specified when a task is started. For example, if you automatically trigger a task to run a
         * batch process job, you could apply a unique identifier for that job to your task with the
         * <code>startedBy</code> parameter. You can then identify which tasks belong to that job by filtering the
         * results of a <a>ListTasks</a> call with the <code>startedBy</code> value. Up to 36 letters (uppercase and
         * lowercase), numbers, hyphens, and underscores are allowed.
         * </p>
         * <p>
         * If a task is started by an Amazon ECS service, then the <code>startedBy</code> parameter contains the
         * deployment ID of the service that starts it.
         * </p>
         * 
         * @param startedBy
         *        An optional tag specified when a task is started. For example, if you automatically trigger a task to
         *        run a batch process job, you could apply a unique identifier for that job to your task with the
         *        <code>startedBy</code> parameter. You can then identify which tasks belong to that job by filtering
         *        the results of a <a>ListTasks</a> call with the <code>startedBy</code> value. Up to 36 letters
         *        (uppercase and lowercase), numbers, hyphens, and underscores are allowed.</p>
         *        <p>
         *        If a task is started by an Amazon ECS service, then the <code>startedBy</code> parameter contains the
         *        deployment ID of the service that starts it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startedBy(String startedBy);

        /**
         * <p>
         * The name of the task group to associate with the task. The default value is the family name of the task
         * definition (for example, family:my-family-name).
         * </p>
         * 
         * @param group
         *        The name of the task group to associate with the task. The default value is the family name of the
         *        task definition (for example, family:my-family-name).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder group(String group);

        /**
         * <p>
         * The VPC subnet and security group configuration for tasks that receive their own elastic network interface by
         * using the <code>awsvpc</code> networking mode.
         * </p>
         * 
         * @param networkConfiguration
         *        The VPC subnet and security group configuration for tasks that receive their own elastic network
         *        interface by using the <code>awsvpc</code> networking mode.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkConfiguration(NetworkConfiguration networkConfiguration);

        /**
         * <p>
         * The VPC subnet and security group configuration for tasks that receive their own elastic network interface by
         * using the <code>awsvpc</code> networking mode.
         * </p>
         * This is a convenience that creates an instance of the {@link NetworkConfiguration.Builder} avoiding the need
         * to create one manually via {@link NetworkConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link NetworkConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #networkConfiguration(NetworkConfiguration)}.
         * 
         * @param networkConfiguration
         *        a consumer that will call methods on {@link NetworkConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkConfiguration(NetworkConfiguration)
         */
        default Builder networkConfiguration(Consumer<NetworkConfiguration.Builder> networkConfiguration) {
            return networkConfiguration(NetworkConfiguration.builder().applyMutation(networkConfiguration).build());
        }

        /**
         * <p>
         * The metadata that you apply to the task to help you categorize and organize them. Each tag consists of a key
         * and an optional value, both of which you define. Tag keys can have a maximum character length of 128
         * characters, and tag values can have a maximum length of 256 characters.
         * </p>
         * 
         * @param tags
         *        The metadata that you apply to the task to help you categorize and organize them. Each tag consists of
         *        a key and an optional value, both of which you define. Tag keys can have a maximum character length of
         *        128 characters, and tag values can have a maximum length of 256 characters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The metadata that you apply to the task to help you categorize and organize them. Each tag consists of a key
         * and an optional value, both of which you define. Tag keys can have a maximum character length of 128
         * characters, and tag values can have a maximum length of 256 characters.
         * </p>
         * 
         * @param tags
         *        The metadata that you apply to the task to help you categorize and organize them. Each tag consists of
         *        a key and an optional value, both of which you define. Tag keys can have a maximum character length of
         *        128 characters, and tag values can have a maximum length of 256 characters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The metadata that you apply to the task to help you categorize and organize them. Each tag consists of a key
         * and an optional value, both of which you define. Tag keys can have a maximum character length of 128
         * characters, and tag values can have a maximum length of 256 characters.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * Specifies whether to enable Amazon ECS managed tags for the task. For more information, see <a
         * href="http://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html">Tagging Your Amazon ECS
         * Resources</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param enableECSManagedTags
         *        Specifies whether to enable Amazon ECS managed tags for the task. For more information, see <a
         *        href="http://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html">Tagging Your
         *        Amazon ECS Resources</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableECSManagedTags(Boolean enableECSManagedTags);

        /**
         * <p>
         * Specifies whether to propagate the tags from the task definition or the service to the task. If no value is
         * specified, the tags are not propagated.
         * </p>
         * 
         * @param propagateTags
         *        Specifies whether to propagate the tags from the task definition or the service to the task. If no
         *        value is specified, the tags are not propagated.
         * @see PropagateTags
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PropagateTags
         */
        Builder propagateTags(String propagateTags);

        /**
         * <p>
         * Specifies whether to propagate the tags from the task definition or the service to the task. If no value is
         * specified, the tags are not propagated.
         * </p>
         * 
         * @param propagateTags
         *        Specifies whether to propagate the tags from the task definition or the service to the task. If no
         *        value is specified, the tags are not propagated.
         * @see PropagateTags
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PropagateTags
         */
        Builder propagateTags(PropagateTags propagateTags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends EcsRequest.BuilderImpl implements Builder {
        private String cluster;

        private String taskDefinition;

        private TaskOverride overrides;

        private List<String> containerInstances = DefaultSdkAutoConstructList.getInstance();

        private String startedBy;

        private String group;

        private NetworkConfiguration networkConfiguration;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private Boolean enableECSManagedTags;

        private String propagateTags;

        private BuilderImpl() {
        }

        private BuilderImpl(StartTaskRequest model) {
            super(model);
            cluster(model.cluster);
            taskDefinition(model.taskDefinition);
            overrides(model.overrides);
            containerInstances(model.containerInstances);
            startedBy(model.startedBy);
            group(model.group);
            networkConfiguration(model.networkConfiguration);
            tags(model.tags);
            enableECSManagedTags(model.enableECSManagedTags);
            propagateTags(model.propagateTags);
        }

        public final String getCluster() {
            return cluster;
        }

        @Override
        public final Builder cluster(String cluster) {
            this.cluster = cluster;
            return this;
        }

        public final void setCluster(String cluster) {
            this.cluster = cluster;
        }

        public final String getTaskDefinition() {
            return taskDefinition;
        }

        @Override
        public final Builder taskDefinition(String taskDefinition) {
            this.taskDefinition = taskDefinition;
            return this;
        }

        public final void setTaskDefinition(String taskDefinition) {
            this.taskDefinition = taskDefinition;
        }

        public final TaskOverride.Builder getOverrides() {
            return overrides != null ? overrides.toBuilder() : null;
        }

        @Override
        public final Builder overrides(TaskOverride overrides) {
            this.overrides = overrides;
            return this;
        }

        public final void setOverrides(TaskOverride.BuilderImpl overrides) {
            this.overrides = overrides != null ? overrides.build() : null;
        }

        public final Collection<String> getContainerInstances() {
            return containerInstances;
        }

        @Override
        public final Builder containerInstances(Collection<String> containerInstances) {
            this.containerInstances = StringListCopier.copy(containerInstances);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder containerInstances(String... containerInstances) {
            containerInstances(Arrays.asList(containerInstances));
            return this;
        }

        public final void setContainerInstances(Collection<String> containerInstances) {
            this.containerInstances = StringListCopier.copy(containerInstances);
        }

        public final String getStartedBy() {
            return startedBy;
        }

        @Override
        public final Builder startedBy(String startedBy) {
            this.startedBy = startedBy;
            return this;
        }

        public final void setStartedBy(String startedBy) {
            this.startedBy = startedBy;
        }

        public final String getGroup() {
            return group;
        }

        @Override
        public final Builder group(String group) {
            this.group = group;
            return this;
        }

        public final void setGroup(String group) {
            this.group = group;
        }

        public final NetworkConfiguration.Builder getNetworkConfiguration() {
            return networkConfiguration != null ? networkConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder networkConfiguration(NetworkConfiguration networkConfiguration) {
            this.networkConfiguration = networkConfiguration;
            return this;
        }

        public final void setNetworkConfiguration(NetworkConfiguration.BuilderImpl networkConfiguration) {
            this.networkConfiguration = networkConfiguration != null ? networkConfiguration.build() : null;
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        public final Boolean getEnableECSManagedTags() {
            return enableECSManagedTags;
        }

        @Override
        public final Builder enableECSManagedTags(Boolean enableECSManagedTags) {
            this.enableECSManagedTags = enableECSManagedTags;
            return this;
        }

        public final void setEnableECSManagedTags(Boolean enableECSManagedTags) {
            this.enableECSManagedTags = enableECSManagedTags;
        }

        public final String getPropagateTagsAsString() {
            return propagateTags;
        }

        @Override
        public final Builder propagateTags(String propagateTags) {
            this.propagateTags = propagateTags;
            return this;
        }

        @Override
        public final Builder propagateTags(PropagateTags propagateTags) {
            this.propagateTags(propagateTags.toString());
            return this;
        }

        public final void setPropagateTags(String propagateTags) {
            this.propagateTags = propagateTags;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public StartTaskRequest build() {
            return new StartTaskRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
