/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The log configuration for the container. This parameter maps to <code>LogConfig</code> in the docker container create
 * command and the <code>--log-driver</code> option to docker run.
 * </p>
 * <p>
 * By default, containers use the same logging driver that the Docker daemon uses. However, the container might use a
 * different logging driver than the Docker daemon by specifying a log driver configuration in the container definition.
 * </p>
 * <p>
 * Understand the following when specifying a log configuration for your containers.
 * </p>
 * <ul>
 * <li>
 * <p>
 * Amazon ECS currently supports a subset of the logging drivers available to the Docker daemon. Additional log drivers
 * may be available in future releases of the Amazon ECS container agent.
 * </p>
 * <p>
 * For tasks on Fargate, the supported log drivers are <code>awslogs</code>, <code>splunk</code>, and
 * <code>awsfirelens</code>.
 * </p>
 * <p>
 * For tasks hosted on Amazon EC2 instances, the supported log drivers are <code>awslogs</code>, <code>fluentd</code>,
 * <code>gelf</code>, <code>json-file</code>, <code>journald</code>,<code>syslog</code>, <code>splunk</code>, and
 * <code>awsfirelens</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance.
 * </p>
 * </li>
 * <li>
 * <p>
 * For tasks that are hosted on Amazon EC2 instances, the Amazon ECS container agent must register the available logging
 * drivers with the <code>ECS_AVAILABLE_LOGGING_DRIVERS</code> environment variable before containers placed on that
 * instance can use these log configuration options. For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon ECS container agent
 * configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
 * </p>
 * </li>
 * <li>
 * <p>
 * For tasks that are on Fargate, because you don't have access to the underlying infrastructure your tasks are hosted
 * on, any additional software needed must be installed outside of the task. For example, the Fluentd output aggregators
 * or a remote host running Logstash to send Gelf logs to.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LogConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<LogConfiguration.Builder, LogConfiguration> {
    private static final SdkField<String> LOG_DRIVER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("logDriver").getter(getter(LogConfiguration::logDriverAsString)).setter(setter(Builder::logDriver))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("logDriver").build()).build();

    private static final SdkField<Map<String, String>> OPTIONS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("options")
            .getter(getter(LogConfiguration::options))
            .setter(setter(Builder::options))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("options").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<Secret>> SECRET_OPTIONS_FIELD = SdkField
            .<List<Secret>> builder(MarshallingType.LIST)
            .memberName("secretOptions")
            .getter(getter(LogConfiguration::secretOptions))
            .setter(setter(Builder::secretOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("secretOptions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Secret> builder(MarshallingType.SDK_POJO)
                                            .constructor(Secret::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LOG_DRIVER_FIELD,
            OPTIONS_FIELD, SECRET_OPTIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String logDriver;

    private final Map<String, String> options;

    private final List<Secret> secretOptions;

    private LogConfiguration(BuilderImpl builder) {
        this.logDriver = builder.logDriver;
        this.options = builder.options;
        this.secretOptions = builder.secretOptions;
    }

    /**
     * <p>
     * The log driver to use for the container.
     * </p>
     * <p>
     * For tasks on Fargate, the supported log drivers are <code>awslogs</code>, <code>splunk</code>, and
     * <code>awsfirelens</code>.
     * </p>
     * <p>
     * For tasks hosted on Amazon EC2 instances, the supported log drivers are <code>awslogs</code>,
     * <code>fluentd</code>, <code>gelf</code>, <code>json-file</code>, <code>journald</code>, <code>syslog</code>,
     * <code>splunk</code>, and <code>awsfirelens</code>.
     * </p>
     * <p>
     * For more information about using the <code>awslogs</code> log driver, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html">Send Amazon ECS logs to
     * CloudWatch</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * For more information about using the <code>awsfirelens</code> log driver, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Send Amazon ECS logs to an
     * Amazon Web Services service or Amazon Web Services Partner</a>.
     * </p>
     * <note>
     * <p>
     * If you have a custom driver that isn't listed, you can fork the Amazon ECS container agent project that's <a
     * href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with that driver.
     * We encourage you to submit pull requests for changes that you would like to have included. However, we don't
     * currently provide support for running modified copies of this software.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #logDriver} will
     * return {@link LogDriver#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #logDriverAsString}.
     * </p>
     * 
     * @return The log driver to use for the container.</p>
     *         <p>
     *         For tasks on Fargate, the supported log drivers are <code>awslogs</code>, <code>splunk</code>, and
     *         <code>awsfirelens</code>.
     *         </p>
     *         <p>
     *         For tasks hosted on Amazon EC2 instances, the supported log drivers are <code>awslogs</code>,
     *         <code>fluentd</code>, <code>gelf</code>, <code>json-file</code>, <code>journald</code>,
     *         <code>syslog</code>, <code>splunk</code>, and <code>awsfirelens</code>.
     *         </p>
     *         <p>
     *         For more information about using the <code>awslogs</code> log driver, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html">Send Amazon ECS
     *         logs to CloudWatch</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     *         </p>
     *         <p>
     *         For more information about using the <code>awsfirelens</code> log driver, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Send Amazon ECS
     *         logs to an Amazon Web Services service or Amazon Web Services Partner</a>.
     *         </p>
     *         <note>
     *         <p>
     *         If you have a custom driver that isn't listed, you can fork the Amazon ECS container agent project that's
     *         <a href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with
     *         that driver. We encourage you to submit pull requests for changes that you would like to have included.
     *         However, we don't currently provide support for running modified copies of this software.
     *         </p>
     * @see LogDriver
     */
    public final LogDriver logDriver() {
        return LogDriver.fromValue(logDriver);
    }

    /**
     * <p>
     * The log driver to use for the container.
     * </p>
     * <p>
     * For tasks on Fargate, the supported log drivers are <code>awslogs</code>, <code>splunk</code>, and
     * <code>awsfirelens</code>.
     * </p>
     * <p>
     * For tasks hosted on Amazon EC2 instances, the supported log drivers are <code>awslogs</code>,
     * <code>fluentd</code>, <code>gelf</code>, <code>json-file</code>, <code>journald</code>, <code>syslog</code>,
     * <code>splunk</code>, and <code>awsfirelens</code>.
     * </p>
     * <p>
     * For more information about using the <code>awslogs</code> log driver, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html">Send Amazon ECS logs to
     * CloudWatch</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * For more information about using the <code>awsfirelens</code> log driver, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Send Amazon ECS logs to an
     * Amazon Web Services service or Amazon Web Services Partner</a>.
     * </p>
     * <note>
     * <p>
     * If you have a custom driver that isn't listed, you can fork the Amazon ECS container agent project that's <a
     * href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with that driver.
     * We encourage you to submit pull requests for changes that you would like to have included. However, we don't
     * currently provide support for running modified copies of this software.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #logDriver} will
     * return {@link LogDriver#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #logDriverAsString}.
     * </p>
     * 
     * @return The log driver to use for the container.</p>
     *         <p>
     *         For tasks on Fargate, the supported log drivers are <code>awslogs</code>, <code>splunk</code>, and
     *         <code>awsfirelens</code>.
     *         </p>
     *         <p>
     *         For tasks hosted on Amazon EC2 instances, the supported log drivers are <code>awslogs</code>,
     *         <code>fluentd</code>, <code>gelf</code>, <code>json-file</code>, <code>journald</code>,
     *         <code>syslog</code>, <code>splunk</code>, and <code>awsfirelens</code>.
     *         </p>
     *         <p>
     *         For more information about using the <code>awslogs</code> log driver, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html">Send Amazon ECS
     *         logs to CloudWatch</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     *         </p>
     *         <p>
     *         For more information about using the <code>awsfirelens</code> log driver, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Send Amazon ECS
     *         logs to an Amazon Web Services service or Amazon Web Services Partner</a>.
     *         </p>
     *         <note>
     *         <p>
     *         If you have a custom driver that isn't listed, you can fork the Amazon ECS container agent project that's
     *         <a href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with
     *         that driver. We encourage you to submit pull requests for changes that you would like to have included.
     *         However, we don't currently provide support for running modified copies of this software.
     *         </p>
     * @see LogDriver
     */
    public final String logDriverAsString() {
        return logDriver;
    }

    /**
     * For responses, this returns true if the service returned a value for the Options property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasOptions() {
        return options != null && !(options instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The configuration options to send to the log driver.
     * </p>
     * <p>
     * The options you can specify depend on the log driver. Some of the options you can specify when you use the
     * <code>awslogs</code> log driver to route logs to Amazon CloudWatch include the following:
     * </p>
     * <dl>
     * <dt>awslogs-create-group</dt>
     * <dd>
     * <p>
     * Required: No
     * </p>
     * <p>
     * Specify whether you want the log group to be created automatically. If this option isn't specified, it defaults
     * to <code>false</code>.
     * </p>
     * <note>
     * <p>
     * Your IAM policy must include the <code>logs:CreateLogGroup</code> permission before you attempt to use
     * <code>awslogs-create-group</code>.
     * </p>
     * </note></dd>
     * <dt>awslogs-region</dt>
     * <dd>
     * <p>
     * Required: Yes
     * </p>
     * <p>
     * Specify the Amazon Web Services Region that the <code>awslogs</code> log driver is to send your Docker logs to.
     * You can choose to send all of your logs from clusters in different Regions to a single region in CloudWatch Logs.
     * This is so that they're all visible in one location. Otherwise, you can separate them by Region for more
     * granularity. Make sure that the specified log group exists in the Region that you specify with this option.
     * </p>
     * </dd>
     * <dt>awslogs-group</dt>
     * <dd>
     * <p>
     * Required: Yes
     * </p>
     * <p>
     * Make sure to specify a log group that the <code>awslogs</code> log driver sends its log streams to.
     * </p>
     * </dd>
     * <dt>awslogs-stream-prefix</dt>
     * <dd>
     * <p>
     * Required: Yes, when using the Fargate launch type.Optional for the EC2 launch type, required for the Fargate
     * launch type.
     * </p>
     * <p>
     * Use the <code>awslogs-stream-prefix</code> option to associate a log stream with the specified prefix, the
     * container name, and the ID of the Amazon ECS task that the container belongs to. If you specify a prefix with
     * this option, then the log stream takes the format <code>prefix-name/container-name/ecs-task-id</code>.
     * </p>
     * <p>
     * If you don't specify a prefix with this option, then the log stream is named after the container ID that's
     * assigned by the Docker daemon on the container instance. Because it's difficult to trace logs back to the
     * container that sent them with just the Docker container ID (which is only available on the container instance),
     * we recommend that you specify a prefix with this option.
     * </p>
     * <p>
     * For Amazon ECS services, you can use the service name as the prefix. Doing so, you can trace log streams to the
     * service that the container belongs to, the name of the container that sent them, and the ID of the task that the
     * container belongs to.
     * </p>
     * <p>
     * You must specify a stream-prefix for your logs to have your logs appear in the Log pane when using the Amazon ECS
     * console.
     * </p>
     * </dd>
     * <dt>awslogs-datetime-format</dt>
     * <dd>
     * <p>
     * Required: No
     * </p>
     * <p>
     * This option defines a multiline start pattern in Python <code>strftime</code> format. A log message consists of a
     * line that matches the pattern and any following lines that don’t match the pattern. The matched line is the
     * delimiter between log messages.
     * </p>
     * <p>
     * One example of a use case for using this format is for parsing output such as a stack dump, which might otherwise
     * be logged in multiple entries. The correct pattern allows it to be captured in a single entry.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.docker.com/config/containers/logging/awslogs/#awslogs-datetime-format"
     * >awslogs-datetime-format</a>.
     * </p>
     * <p>
     * You cannot configure both the <code>awslogs-datetime-format</code> and <code>awslogs-multiline-pattern</code>
     * options.
     * </p>
     * <note>
     * <p>
     * Multiline logging performs regular expression parsing and matching of all log messages. This might have a
     * negative impact on logging performance.
     * </p>
     * </note></dd>
     * <dt>awslogs-multiline-pattern</dt>
     * <dd>
     * <p>
     * Required: No
     * </p>
     * <p>
     * This option defines a multiline start pattern that uses a regular expression. A log message consists of a line
     * that matches the pattern and any following lines that don’t match the pattern. The matched line is the delimiter
     * between log messages.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.docker.com/config/containers/logging/awslogs/#awslogs-multiline-pattern"
     * >awslogs-multiline-pattern</a>.
     * </p>
     * <p>
     * This option is ignored if <code>awslogs-datetime-format</code> is also configured.
     * </p>
     * <p>
     * You cannot configure both the <code>awslogs-datetime-format</code> and <code>awslogs-multiline-pattern</code>
     * options.
     * </p>
     * <note>
     * <p>
     * Multiline logging performs regular expression parsing and matching of all log messages. This might have a
     * negative impact on logging performance.
     * </p>
     * </note></dd>
     * <dt>mode</dt>
     * <dd>
     * <p>
     * Required: No
     * </p>
     * <p>
     * Valid values: <code>non-blocking</code> | <code>blocking</code>
     * </p>
     * <p>
     * This option defines the delivery mode of log messages from the container to CloudWatch Logs. The delivery mode
     * you choose affects application availability when the flow of logs from container to CloudWatch is interrupted.
     * </p>
     * <p>
     * If you use the <code>blocking</code> mode and the flow of logs to CloudWatch is interrupted, calls from container
     * code to write to the <code>stdout</code> and <code>stderr</code> streams will block. The logging thread of the
     * application will block as a result. This may cause the application to become unresponsive and lead to container
     * healthcheck failure.
     * </p>
     * <p>
     * If you use the <code>non-blocking</code> mode, the container's logs are instead stored in an in-memory
     * intermediate buffer configured with the <code>max-buffer-size</code> option. This prevents the application from
     * becoming unresponsive when logs cannot be sent to CloudWatch. We recommend using this mode if you want to ensure
     * service availability and are okay with some log loss. For more information, see <a href=
     * "http://aws.amazon.com/blogs/containers/preventing-log-loss-with-non-blocking-mode-in-the-awslogs-container-log-driver/"
     * >Preventing log loss with non-blocking mode in the <code>awslogs</code> container log driver</a>.
     * </p>
     * </dd>
     * <dt>max-buffer-size</dt>
     * <dd>
     * <p>
     * Required: No
     * </p>
     * <p>
     * Default value: <code>1m</code>
     * </p>
     * <p>
     * When <code>non-blocking</code> mode is used, the <code>max-buffer-size</code> log option controls the size of the
     * buffer that's used for intermediate message storage. Make sure to specify an adequate buffer size based on your
     * application. When the buffer fills up, further logs cannot be stored. Logs that cannot be stored are lost.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * To route logs using the <code>splunk</code> log router, you need to specify a <code>splunk-token</code> and a
     * <code>splunk-url</code>.
     * </p>
     * <p>
     * When you use the <code>awsfirelens</code> log router to route logs to an Amazon Web Services Service or Amazon
     * Web Services Partner Network destination for log storage and analytics, you can set the
     * <code>log-driver-buffer-limit</code> option to limit the number of events that are buffered in memory, before
     * being sent to the log router container. It can help to resolve potential log loss issue because high throughput
     * might result in memory running out for the buffer inside of Docker.
     * </p>
     * <p>
     * Other options you can specify when using <code>awsfirelens</code> to route logs depend on the destination. When
     * you export logs to Amazon Data Firehose, you can specify the Amazon Web Services Region with <code>region</code>
     * and a name for the log stream with <code>delivery_stream</code>.
     * </p>
     * <p>
     * When you export logs to Amazon Kinesis Data Streams, you can specify an Amazon Web Services Region with
     * <code>region</code> and a data stream name with <code>stream</code>.
     * </p>
     * <p>
     * When you export logs to Amazon OpenSearch Service, you can specify options like <code>Name</code>,
     * <code>Host</code> (OpenSearch Service endpoint without protocol), <code>Port</code>, <code>Index</code>,
     * <code>Type</code>, <code>Aws_auth</code>, <code>Aws_region</code>, <code>Suppress_Type_Name</code>, and
     * <code>tls</code>. For more information, see <a
     * href="http://aws.amazon.com/blogs/containers/under-the-hood-firelens-for-amazon-ecs-tasks/">Under the hood:
     * FireLens for Amazon ECS Tasks</a>.
     * </p>
     * <p>
     * When you export logs to Amazon S3, you can specify the bucket using the <code>bucket</code> option. You can also
     * specify <code>region</code>, <code>total_file_size</code>, <code>upload_timeout</code>, and
     * <code>use_put_object</code> as options.
     * </p>
     * <p>
     * This parameter requires version 1.19 of the Docker Remote API or greater on your container instance. To check the
     * Docker Remote API version on your container instance, log in to your container instance and run the following
     * command: <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOptions} method.
     * </p>
     * 
     * @return The configuration options to send to the log driver.</p>
     *         <p>
     *         The options you can specify depend on the log driver. Some of the options you can specify when you use
     *         the <code>awslogs</code> log driver to route logs to Amazon CloudWatch include the following:
     *         </p>
     *         <dl>
     *         <dt>awslogs-create-group</dt>
     *         <dd>
     *         <p>
     *         Required: No
     *         </p>
     *         <p>
     *         Specify whether you want the log group to be created automatically. If this option isn't specified, it
     *         defaults to <code>false</code>.
     *         </p>
     *         <note>
     *         <p>
     *         Your IAM policy must include the <code>logs:CreateLogGroup</code> permission before you attempt to use
     *         <code>awslogs-create-group</code>.
     *         </p>
     *         </note></dd>
     *         <dt>awslogs-region</dt>
     *         <dd>
     *         <p>
     *         Required: Yes
     *         </p>
     *         <p>
     *         Specify the Amazon Web Services Region that the <code>awslogs</code> log driver is to send your Docker
     *         logs to. You can choose to send all of your logs from clusters in different Regions to a single region in
     *         CloudWatch Logs. This is so that they're all visible in one location. Otherwise, you can separate them by
     *         Region for more granularity. Make sure that the specified log group exists in the Region that you specify
     *         with this option.
     *         </p>
     *         </dd>
     *         <dt>awslogs-group</dt>
     *         <dd>
     *         <p>
     *         Required: Yes
     *         </p>
     *         <p>
     *         Make sure to specify a log group that the <code>awslogs</code> log driver sends its log streams to.
     *         </p>
     *         </dd>
     *         <dt>awslogs-stream-prefix</dt>
     *         <dd>
     *         <p>
     *         Required: Yes, when using the Fargate launch type.Optional for the EC2 launch type, required for the
     *         Fargate launch type.
     *         </p>
     *         <p>
     *         Use the <code>awslogs-stream-prefix</code> option to associate a log stream with the specified prefix,
     *         the container name, and the ID of the Amazon ECS task that the container belongs to. If you specify a
     *         prefix with this option, then the log stream takes the format
     *         <code>prefix-name/container-name/ecs-task-id</code>.
     *         </p>
     *         <p>
     *         If you don't specify a prefix with this option, then the log stream is named after the container ID
     *         that's assigned by the Docker daemon on the container instance. Because it's difficult to trace logs back
     *         to the container that sent them with just the Docker container ID (which is only available on the
     *         container instance), we recommend that you specify a prefix with this option.
     *         </p>
     *         <p>
     *         For Amazon ECS services, you can use the service name as the prefix. Doing so, you can trace log streams
     *         to the service that the container belongs to, the name of the container that sent them, and the ID of the
     *         task that the container belongs to.
     *         </p>
     *         <p>
     *         You must specify a stream-prefix for your logs to have your logs appear in the Log pane when using the
     *         Amazon ECS console.
     *         </p>
     *         </dd>
     *         <dt>awslogs-datetime-format</dt>
     *         <dd>
     *         <p>
     *         Required: No
     *         </p>
     *         <p>
     *         This option defines a multiline start pattern in Python <code>strftime</code> format. A log message
     *         consists of a line that matches the pattern and any following lines that don’t match the pattern. The
     *         matched line is the delimiter between log messages.
     *         </p>
     *         <p>
     *         One example of a use case for using this format is for parsing output such as a stack dump, which might
     *         otherwise be logged in multiple entries. The correct pattern allows it to be captured in a single entry.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.docker.com/config/containers/logging/awslogs/#awslogs-datetime-format"
     *         >awslogs-datetime-format</a>.
     *         </p>
     *         <p>
     *         You cannot configure both the <code>awslogs-datetime-format</code> and
     *         <code>awslogs-multiline-pattern</code> options.
     *         </p>
     *         <note>
     *         <p>
     *         Multiline logging performs regular expression parsing and matching of all log messages. This might have a
     *         negative impact on logging performance.
     *         </p>
     *         </note></dd>
     *         <dt>awslogs-multiline-pattern</dt>
     *         <dd>
     *         <p>
     *         Required: No
     *         </p>
     *         <p>
     *         This option defines a multiline start pattern that uses a regular expression. A log message consists of a
     *         line that matches the pattern and any following lines that don’t match the pattern. The matched line is
     *         the delimiter between log messages.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.docker.com/config/containers/logging/awslogs/#awslogs-multiline-pattern"
     *         >awslogs-multiline-pattern</a>.
     *         </p>
     *         <p>
     *         This option is ignored if <code>awslogs-datetime-format</code> is also configured.
     *         </p>
     *         <p>
     *         You cannot configure both the <code>awslogs-datetime-format</code> and
     *         <code>awslogs-multiline-pattern</code> options.
     *         </p>
     *         <note>
     *         <p>
     *         Multiline logging performs regular expression parsing and matching of all log messages. This might have a
     *         negative impact on logging performance.
     *         </p>
     *         </note></dd>
     *         <dt>mode</dt>
     *         <dd>
     *         <p>
     *         Required: No
     *         </p>
     *         <p>
     *         Valid values: <code>non-blocking</code> | <code>blocking</code>
     *         </p>
     *         <p>
     *         This option defines the delivery mode of log messages from the container to CloudWatch Logs. The delivery
     *         mode you choose affects application availability when the flow of logs from container to CloudWatch is
     *         interrupted.
     *         </p>
     *         <p>
     *         If you use the <code>blocking</code> mode and the flow of logs to CloudWatch is interrupted, calls from
     *         container code to write to the <code>stdout</code> and <code>stderr</code> streams will block. The
     *         logging thread of the application will block as a result. This may cause the application to become
     *         unresponsive and lead to container healthcheck failure.
     *         </p>
     *         <p>
     *         If you use the <code>non-blocking</code> mode, the container's logs are instead stored in an in-memory
     *         intermediate buffer configured with the <code>max-buffer-size</code> option. This prevents the
     *         application from becoming unresponsive when logs cannot be sent to CloudWatch. We recommend using this
     *         mode if you want to ensure service availability and are okay with some log loss. For more information,
     *         see <a href=
     *         "http://aws.amazon.com/blogs/containers/preventing-log-loss-with-non-blocking-mode-in-the-awslogs-container-log-driver/"
     *         >Preventing log loss with non-blocking mode in the <code>awslogs</code> container log driver</a>.
     *         </p>
     *         </dd>
     *         <dt>max-buffer-size</dt>
     *         <dd>
     *         <p>
     *         Required: No
     *         </p>
     *         <p>
     *         Default value: <code>1m</code>
     *         </p>
     *         <p>
     *         When <code>non-blocking</code> mode is used, the <code>max-buffer-size</code> log option controls the
     *         size of the buffer that's used for intermediate message storage. Make sure to specify an adequate buffer
     *         size based on your application. When the buffer fills up, further logs cannot be stored. Logs that cannot
     *         be stored are lost.
     *         </p>
     *         </dd>
     *         </dl>
     *         <p>
     *         To route logs using the <code>splunk</code> log router, you need to specify a <code>splunk-token</code>
     *         and a <code>splunk-url</code>.
     *         </p>
     *         <p>
     *         When you use the <code>awsfirelens</code> log router to route logs to an Amazon Web Services Service or
     *         Amazon Web Services Partner Network destination for log storage and analytics, you can set the
     *         <code>log-driver-buffer-limit</code> option to limit the number of events that are buffered in memory,
     *         before being sent to the log router container. It can help to resolve potential log loss issue because
     *         high throughput might result in memory running out for the buffer inside of Docker.
     *         </p>
     *         <p>
     *         Other options you can specify when using <code>awsfirelens</code> to route logs depend on the
     *         destination. When you export logs to Amazon Data Firehose, you can specify the Amazon Web Services Region
     *         with <code>region</code> and a name for the log stream with <code>delivery_stream</code>.
     *         </p>
     *         <p>
     *         When you export logs to Amazon Kinesis Data Streams, you can specify an Amazon Web Services Region with
     *         <code>region</code> and a data stream name with <code>stream</code>.
     *         </p>
     *         <p>
     *         When you export logs to Amazon OpenSearch Service, you can specify options like <code>Name</code>,
     *         <code>Host</code> (OpenSearch Service endpoint without protocol), <code>Port</code>, <code>Index</code>,
     *         <code>Type</code>, <code>Aws_auth</code>, <code>Aws_region</code>, <code>Suppress_Type_Name</code>, and
     *         <code>tls</code>. For more information, see <a
     *         href="http://aws.amazon.com/blogs/containers/under-the-hood-firelens-for-amazon-ecs-tasks/">Under the
     *         hood: FireLens for Amazon ECS Tasks</a>.
     *         </p>
     *         <p>
     *         When you export logs to Amazon S3, you can specify the bucket using the <code>bucket</code> option. You
     *         can also specify <code>region</code>, <code>total_file_size</code>, <code>upload_timeout</code>, and
     *         <code>use_put_object</code> as options.
     *         </p>
     *         <p>
     *         This parameter requires version 1.19 of the Docker Remote API or greater on your container instance. To
     *         check the Docker Remote API version on your container instance, log in to your container instance and run
     *         the following command: <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
     */
    public final Map<String, String> options() {
        return options;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecretOptions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSecretOptions() {
        return secretOptions != null && !(secretOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The secrets to pass to the log configuration. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html">Specifying
     * sensitive data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecretOptions} method.
     * </p>
     * 
     * @return The secrets to pass to the log configuration. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html"
     *         >Specifying sensitive data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     */
    public final List<Secret> secretOptions() {
        return secretOptions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(logDriverAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasOptions() ? options() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSecretOptions() ? secretOptions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LogConfiguration)) {
            return false;
        }
        LogConfiguration other = (LogConfiguration) obj;
        return Objects.equals(logDriverAsString(), other.logDriverAsString()) && hasOptions() == other.hasOptions()
                && Objects.equals(options(), other.options()) && hasSecretOptions() == other.hasSecretOptions()
                && Objects.equals(secretOptions(), other.secretOptions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LogConfiguration").add("LogDriver", logDriverAsString())
                .add("Options", hasOptions() ? options() : null)
                .add("SecretOptions", hasSecretOptions() ? secretOptions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "logDriver":
            return Optional.ofNullable(clazz.cast(logDriverAsString()));
        case "options":
            return Optional.ofNullable(clazz.cast(options()));
        case "secretOptions":
            return Optional.ofNullable(clazz.cast(secretOptions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("logDriver", LOG_DRIVER_FIELD);
        map.put("options", OPTIONS_FIELD);
        map.put("secretOptions", SECRET_OPTIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<LogConfiguration, T> g) {
        return obj -> g.apply((LogConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LogConfiguration> {
        /**
         * <p>
         * The log driver to use for the container.
         * </p>
         * <p>
         * For tasks on Fargate, the supported log drivers are <code>awslogs</code>, <code>splunk</code>, and
         * <code>awsfirelens</code>.
         * </p>
         * <p>
         * For tasks hosted on Amazon EC2 instances, the supported log drivers are <code>awslogs</code>,
         * <code>fluentd</code>, <code>gelf</code>, <code>json-file</code>, <code>journald</code>, <code>syslog</code>,
         * <code>splunk</code>, and <code>awsfirelens</code>.
         * </p>
         * <p>
         * For more information about using the <code>awslogs</code> log driver, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html">Send Amazon ECS logs to
         * CloudWatch</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <p>
         * For more information about using the <code>awsfirelens</code> log driver, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Send Amazon ECS logs
         * to an Amazon Web Services service or Amazon Web Services Partner</a>.
         * </p>
         * <note>
         * <p>
         * If you have a custom driver that isn't listed, you can fork the Amazon ECS container agent project that's <a
         * href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with that
         * driver. We encourage you to submit pull requests for changes that you would like to have included. However,
         * we don't currently provide support for running modified copies of this software.
         * </p>
         * </note>
         * 
         * @param logDriver
         *        The log driver to use for the container.</p>
         *        <p>
         *        For tasks on Fargate, the supported log drivers are <code>awslogs</code>, <code>splunk</code>, and
         *        <code>awsfirelens</code>.
         *        </p>
         *        <p>
         *        For tasks hosted on Amazon EC2 instances, the supported log drivers are <code>awslogs</code>,
         *        <code>fluentd</code>, <code>gelf</code>, <code>json-file</code>, <code>journald</code>,
         *        <code>syslog</code>, <code>splunk</code>, and <code>awsfirelens</code>.
         *        </p>
         *        <p>
         *        For more information about using the <code>awslogs</code> log driver, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html">Send Amazon ECS
         *        logs to CloudWatch</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         *        </p>
         *        <p>
         *        For more information about using the <code>awsfirelens</code> log driver, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Send Amazon ECS
         *        logs to an Amazon Web Services service or Amazon Web Services Partner</a>.
         *        </p>
         *        <note>
         *        <p>
         *        If you have a custom driver that isn't listed, you can fork the Amazon ECS container agent project
         *        that's <a href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to
         *        work with that driver. We encourage you to submit pull requests for changes that you would like to
         *        have included. However, we don't currently provide support for running modified copies of this
         *        software.
         *        </p>
         * @see LogDriver
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogDriver
         */
        Builder logDriver(String logDriver);

        /**
         * <p>
         * The log driver to use for the container.
         * </p>
         * <p>
         * For tasks on Fargate, the supported log drivers are <code>awslogs</code>, <code>splunk</code>, and
         * <code>awsfirelens</code>.
         * </p>
         * <p>
         * For tasks hosted on Amazon EC2 instances, the supported log drivers are <code>awslogs</code>,
         * <code>fluentd</code>, <code>gelf</code>, <code>json-file</code>, <code>journald</code>, <code>syslog</code>,
         * <code>splunk</code>, and <code>awsfirelens</code>.
         * </p>
         * <p>
         * For more information about using the <code>awslogs</code> log driver, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html">Send Amazon ECS logs to
         * CloudWatch</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <p>
         * For more information about using the <code>awsfirelens</code> log driver, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Send Amazon ECS logs
         * to an Amazon Web Services service or Amazon Web Services Partner</a>.
         * </p>
         * <note>
         * <p>
         * If you have a custom driver that isn't listed, you can fork the Amazon ECS container agent project that's <a
         * href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with that
         * driver. We encourage you to submit pull requests for changes that you would like to have included. However,
         * we don't currently provide support for running modified copies of this software.
         * </p>
         * </note>
         * 
         * @param logDriver
         *        The log driver to use for the container.</p>
         *        <p>
         *        For tasks on Fargate, the supported log drivers are <code>awslogs</code>, <code>splunk</code>, and
         *        <code>awsfirelens</code>.
         *        </p>
         *        <p>
         *        For tasks hosted on Amazon EC2 instances, the supported log drivers are <code>awslogs</code>,
         *        <code>fluentd</code>, <code>gelf</code>, <code>json-file</code>, <code>journald</code>,
         *        <code>syslog</code>, <code>splunk</code>, and <code>awsfirelens</code>.
         *        </p>
         *        <p>
         *        For more information about using the <code>awslogs</code> log driver, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html">Send Amazon ECS
         *        logs to CloudWatch</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         *        </p>
         *        <p>
         *        For more information about using the <code>awsfirelens</code> log driver, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Send Amazon ECS
         *        logs to an Amazon Web Services service or Amazon Web Services Partner</a>.
         *        </p>
         *        <note>
         *        <p>
         *        If you have a custom driver that isn't listed, you can fork the Amazon ECS container agent project
         *        that's <a href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to
         *        work with that driver. We encourage you to submit pull requests for changes that you would like to
         *        have included. However, we don't currently provide support for running modified copies of this
         *        software.
         *        </p>
         * @see LogDriver
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogDriver
         */
        Builder logDriver(LogDriver logDriver);

        /**
         * <p>
         * The configuration options to send to the log driver.
         * </p>
         * <p>
         * The options you can specify depend on the log driver. Some of the options you can specify when you use the
         * <code>awslogs</code> log driver to route logs to Amazon CloudWatch include the following:
         * </p>
         * <dl>
         * <dt>awslogs-create-group</dt>
         * <dd>
         * <p>
         * Required: No
         * </p>
         * <p>
         * Specify whether you want the log group to be created automatically. If this option isn't specified, it
         * defaults to <code>false</code>.
         * </p>
         * <note>
         * <p>
         * Your IAM policy must include the <code>logs:CreateLogGroup</code> permission before you attempt to use
         * <code>awslogs-create-group</code>.
         * </p>
         * </note></dd>
         * <dt>awslogs-region</dt>
         * <dd>
         * <p>
         * Required: Yes
         * </p>
         * <p>
         * Specify the Amazon Web Services Region that the <code>awslogs</code> log driver is to send your Docker logs
         * to. You can choose to send all of your logs from clusters in different Regions to a single region in
         * CloudWatch Logs. This is so that they're all visible in one location. Otherwise, you can separate them by
         * Region for more granularity. Make sure that the specified log group exists in the Region that you specify
         * with this option.
         * </p>
         * </dd>
         * <dt>awslogs-group</dt>
         * <dd>
         * <p>
         * Required: Yes
         * </p>
         * <p>
         * Make sure to specify a log group that the <code>awslogs</code> log driver sends its log streams to.
         * </p>
         * </dd>
         * <dt>awslogs-stream-prefix</dt>
         * <dd>
         * <p>
         * Required: Yes, when using the Fargate launch type.Optional for the EC2 launch type, required for the Fargate
         * launch type.
         * </p>
         * <p>
         * Use the <code>awslogs-stream-prefix</code> option to associate a log stream with the specified prefix, the
         * container name, and the ID of the Amazon ECS task that the container belongs to. If you specify a prefix with
         * this option, then the log stream takes the format <code>prefix-name/container-name/ecs-task-id</code>.
         * </p>
         * <p>
         * If you don't specify a prefix with this option, then the log stream is named after the container ID that's
         * assigned by the Docker daemon on the container instance. Because it's difficult to trace logs back to the
         * container that sent them with just the Docker container ID (which is only available on the container
         * instance), we recommend that you specify a prefix with this option.
         * </p>
         * <p>
         * For Amazon ECS services, you can use the service name as the prefix. Doing so, you can trace log streams to
         * the service that the container belongs to, the name of the container that sent them, and the ID of the task
         * that the container belongs to.
         * </p>
         * <p>
         * You must specify a stream-prefix for your logs to have your logs appear in the Log pane when using the Amazon
         * ECS console.
         * </p>
         * </dd>
         * <dt>awslogs-datetime-format</dt>
         * <dd>
         * <p>
         * Required: No
         * </p>
         * <p>
         * This option defines a multiline start pattern in Python <code>strftime</code> format. A log message consists
         * of a line that matches the pattern and any following lines that don’t match the pattern. The matched line is
         * the delimiter between log messages.
         * </p>
         * <p>
         * One example of a use case for using this format is for parsing output such as a stack dump, which might
         * otherwise be logged in multiple entries. The correct pattern allows it to be captured in a single entry.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.docker.com/config/containers/logging/awslogs/#awslogs-datetime-format"
         * >awslogs-datetime-format</a>.
         * </p>
         * <p>
         * You cannot configure both the <code>awslogs-datetime-format</code> and <code>awslogs-multiline-pattern</code>
         * options.
         * </p>
         * <note>
         * <p>
         * Multiline logging performs regular expression parsing and matching of all log messages. This might have a
         * negative impact on logging performance.
         * </p>
         * </note></dd>
         * <dt>awslogs-multiline-pattern</dt>
         * <dd>
         * <p>
         * Required: No
         * </p>
         * <p>
         * This option defines a multiline start pattern that uses a regular expression. A log message consists of a
         * line that matches the pattern and any following lines that don’t match the pattern. The matched line is the
         * delimiter between log messages.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.docker.com/config/containers/logging/awslogs/#awslogs-multiline-pattern"
         * >awslogs-multiline-pattern</a>.
         * </p>
         * <p>
         * This option is ignored if <code>awslogs-datetime-format</code> is also configured.
         * </p>
         * <p>
         * You cannot configure both the <code>awslogs-datetime-format</code> and <code>awslogs-multiline-pattern</code>
         * options.
         * </p>
         * <note>
         * <p>
         * Multiline logging performs regular expression parsing and matching of all log messages. This might have a
         * negative impact on logging performance.
         * </p>
         * </note></dd>
         * <dt>mode</dt>
         * <dd>
         * <p>
         * Required: No
         * </p>
         * <p>
         * Valid values: <code>non-blocking</code> | <code>blocking</code>
         * </p>
         * <p>
         * This option defines the delivery mode of log messages from the container to CloudWatch Logs. The delivery
         * mode you choose affects application availability when the flow of logs from container to CloudWatch is
         * interrupted.
         * </p>
         * <p>
         * If you use the <code>blocking</code> mode and the flow of logs to CloudWatch is interrupted, calls from
         * container code to write to the <code>stdout</code> and <code>stderr</code> streams will block. The logging
         * thread of the application will block as a result. This may cause the application to become unresponsive and
         * lead to container healthcheck failure.
         * </p>
         * <p>
         * If you use the <code>non-blocking</code> mode, the container's logs are instead stored in an in-memory
         * intermediate buffer configured with the <code>max-buffer-size</code> option. This prevents the application
         * from becoming unresponsive when logs cannot be sent to CloudWatch. We recommend using this mode if you want
         * to ensure service availability and are okay with some log loss. For more information, see <a href=
         * "http://aws.amazon.com/blogs/containers/preventing-log-loss-with-non-blocking-mode-in-the-awslogs-container-log-driver/"
         * >Preventing log loss with non-blocking mode in the <code>awslogs</code> container log driver</a>.
         * </p>
         * </dd>
         * <dt>max-buffer-size</dt>
         * <dd>
         * <p>
         * Required: No
         * </p>
         * <p>
         * Default value: <code>1m</code>
         * </p>
         * <p>
         * When <code>non-blocking</code> mode is used, the <code>max-buffer-size</code> log option controls the size of
         * the buffer that's used for intermediate message storage. Make sure to specify an adequate buffer size based
         * on your application. When the buffer fills up, further logs cannot be stored. Logs that cannot be stored are
         * lost.
         * </p>
         * </dd>
         * </dl>
         * <p>
         * To route logs using the <code>splunk</code> log router, you need to specify a <code>splunk-token</code> and a
         * <code>splunk-url</code>.
         * </p>
         * <p>
         * When you use the <code>awsfirelens</code> log router to route logs to an Amazon Web Services Service or
         * Amazon Web Services Partner Network destination for log storage and analytics, you can set the
         * <code>log-driver-buffer-limit</code> option to limit the number of events that are buffered in memory, before
         * being sent to the log router container. It can help to resolve potential log loss issue because high
         * throughput might result in memory running out for the buffer inside of Docker.
         * </p>
         * <p>
         * Other options you can specify when using <code>awsfirelens</code> to route logs depend on the destination.
         * When you export logs to Amazon Data Firehose, you can specify the Amazon Web Services Region with
         * <code>region</code> and a name for the log stream with <code>delivery_stream</code>.
         * </p>
         * <p>
         * When you export logs to Amazon Kinesis Data Streams, you can specify an Amazon Web Services Region with
         * <code>region</code> and a data stream name with <code>stream</code>.
         * </p>
         * <p>
         * When you export logs to Amazon OpenSearch Service, you can specify options like <code>Name</code>,
         * <code>Host</code> (OpenSearch Service endpoint without protocol), <code>Port</code>, <code>Index</code>,
         * <code>Type</code>, <code>Aws_auth</code>, <code>Aws_region</code>, <code>Suppress_Type_Name</code>, and
         * <code>tls</code>. For more information, see <a
         * href="http://aws.amazon.com/blogs/containers/under-the-hood-firelens-for-amazon-ecs-tasks/">Under the hood:
         * FireLens for Amazon ECS Tasks</a>.
         * </p>
         * <p>
         * When you export logs to Amazon S3, you can specify the bucket using the <code>bucket</code> option. You can
         * also specify <code>region</code>, <code>total_file_size</code>, <code>upload_timeout</code>, and
         * <code>use_put_object</code> as options.
         * </p>
         * <p>
         * This parameter requires version 1.19 of the Docker Remote API or greater on your container instance. To check
         * the Docker Remote API version on your container instance, log in to your container instance and run the
         * following command: <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
         * </p>
         * 
         * @param options
         *        The configuration options to send to the log driver.</p>
         *        <p>
         *        The options you can specify depend on the log driver. Some of the options you can specify when you use
         *        the <code>awslogs</code> log driver to route logs to Amazon CloudWatch include the following:
         *        </p>
         *        <dl>
         *        <dt>awslogs-create-group</dt>
         *        <dd>
         *        <p>
         *        Required: No
         *        </p>
         *        <p>
         *        Specify whether you want the log group to be created automatically. If this option isn't specified, it
         *        defaults to <code>false</code>.
         *        </p>
         *        <note>
         *        <p>
         *        Your IAM policy must include the <code>logs:CreateLogGroup</code> permission before you attempt to use
         *        <code>awslogs-create-group</code>.
         *        </p>
         *        </note></dd>
         *        <dt>awslogs-region</dt>
         *        <dd>
         *        <p>
         *        Required: Yes
         *        </p>
         *        <p>
         *        Specify the Amazon Web Services Region that the <code>awslogs</code> log driver is to send your Docker
         *        logs to. You can choose to send all of your logs from clusters in different Regions to a single region
         *        in CloudWatch Logs. This is so that they're all visible in one location. Otherwise, you can separate
         *        them by Region for more granularity. Make sure that the specified log group exists in the Region that
         *        you specify with this option.
         *        </p>
         *        </dd>
         *        <dt>awslogs-group</dt>
         *        <dd>
         *        <p>
         *        Required: Yes
         *        </p>
         *        <p>
         *        Make sure to specify a log group that the <code>awslogs</code> log driver sends its log streams to.
         *        </p>
         *        </dd>
         *        <dt>awslogs-stream-prefix</dt>
         *        <dd>
         *        <p>
         *        Required: Yes, when using the Fargate launch type.Optional for the EC2 launch type, required for the
         *        Fargate launch type.
         *        </p>
         *        <p>
         *        Use the <code>awslogs-stream-prefix</code> option to associate a log stream with the specified prefix,
         *        the container name, and the ID of the Amazon ECS task that the container belongs to. If you specify a
         *        prefix with this option, then the log stream takes the format
         *        <code>prefix-name/container-name/ecs-task-id</code>.
         *        </p>
         *        <p>
         *        If you don't specify a prefix with this option, then the log stream is named after the container ID
         *        that's assigned by the Docker daemon on the container instance. Because it's difficult to trace logs
         *        back to the container that sent them with just the Docker container ID (which is only available on the
         *        container instance), we recommend that you specify a prefix with this option.
         *        </p>
         *        <p>
         *        For Amazon ECS services, you can use the service name as the prefix. Doing so, you can trace log
         *        streams to the service that the container belongs to, the name of the container that sent them, and
         *        the ID of the task that the container belongs to.
         *        </p>
         *        <p>
         *        You must specify a stream-prefix for your logs to have your logs appear in the Log pane when using the
         *        Amazon ECS console.
         *        </p>
         *        </dd>
         *        <dt>awslogs-datetime-format</dt>
         *        <dd>
         *        <p>
         *        Required: No
         *        </p>
         *        <p>
         *        This option defines a multiline start pattern in Python <code>strftime</code> format. A log message
         *        consists of a line that matches the pattern and any following lines that don’t match the pattern. The
         *        matched line is the delimiter between log messages.
         *        </p>
         *        <p>
         *        One example of a use case for using this format is for parsing output such as a stack dump, which
         *        might otherwise be logged in multiple entries. The correct pattern allows it to be captured in a
         *        single entry.
         *        </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.docker.com/config/containers/logging/awslogs/#awslogs-datetime-format"
         *        >awslogs-datetime-format</a>.
         *        </p>
         *        <p>
         *        You cannot configure both the <code>awslogs-datetime-format</code> and
         *        <code>awslogs-multiline-pattern</code> options.
         *        </p>
         *        <note>
         *        <p>
         *        Multiline logging performs regular expression parsing and matching of all log messages. This might
         *        have a negative impact on logging performance.
         *        </p>
         *        </note></dd>
         *        <dt>awslogs-multiline-pattern</dt>
         *        <dd>
         *        <p>
         *        Required: No
         *        </p>
         *        <p>
         *        This option defines a multiline start pattern that uses a regular expression. A log message consists
         *        of a line that matches the pattern and any following lines that don’t match the pattern. The matched
         *        line is the delimiter between log messages.
         *        </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.docker.com/config/containers/logging/awslogs/#awslogs-multiline-pattern"
         *        >awslogs-multiline-pattern</a>.
         *        </p>
         *        <p>
         *        This option is ignored if <code>awslogs-datetime-format</code> is also configured.
         *        </p>
         *        <p>
         *        You cannot configure both the <code>awslogs-datetime-format</code> and
         *        <code>awslogs-multiline-pattern</code> options.
         *        </p>
         *        <note>
         *        <p>
         *        Multiline logging performs regular expression parsing and matching of all log messages. This might
         *        have a negative impact on logging performance.
         *        </p>
         *        </note></dd>
         *        <dt>mode</dt>
         *        <dd>
         *        <p>
         *        Required: No
         *        </p>
         *        <p>
         *        Valid values: <code>non-blocking</code> | <code>blocking</code>
         *        </p>
         *        <p>
         *        This option defines the delivery mode of log messages from the container to CloudWatch Logs. The
         *        delivery mode you choose affects application availability when the flow of logs from container to
         *        CloudWatch is interrupted.
         *        </p>
         *        <p>
         *        If you use the <code>blocking</code> mode and the flow of logs to CloudWatch is interrupted, calls
         *        from container code to write to the <code>stdout</code> and <code>stderr</code> streams will block.
         *        The logging thread of the application will block as a result. This may cause the application to become
         *        unresponsive and lead to container healthcheck failure.
         *        </p>
         *        <p>
         *        If you use the <code>non-blocking</code> mode, the container's logs are instead stored in an in-memory
         *        intermediate buffer configured with the <code>max-buffer-size</code> option. This prevents the
         *        application from becoming unresponsive when logs cannot be sent to CloudWatch. We recommend using this
         *        mode if you want to ensure service availability and are okay with some log loss. For more information,
         *        see <a href=
         *        "http://aws.amazon.com/blogs/containers/preventing-log-loss-with-non-blocking-mode-in-the-awslogs-container-log-driver/"
         *        >Preventing log loss with non-blocking mode in the <code>awslogs</code> container log driver</a>.
         *        </p>
         *        </dd>
         *        <dt>max-buffer-size</dt>
         *        <dd>
         *        <p>
         *        Required: No
         *        </p>
         *        <p>
         *        Default value: <code>1m</code>
         *        </p>
         *        <p>
         *        When <code>non-blocking</code> mode is used, the <code>max-buffer-size</code> log option controls the
         *        size of the buffer that's used for intermediate message storage. Make sure to specify an adequate
         *        buffer size based on your application. When the buffer fills up, further logs cannot be stored. Logs
         *        that cannot be stored are lost.
         *        </p>
         *        </dd>
         *        </dl>
         *        <p>
         *        To route logs using the <code>splunk</code> log router, you need to specify a
         *        <code>splunk-token</code> and a <code>splunk-url</code>.
         *        </p>
         *        <p>
         *        When you use the <code>awsfirelens</code> log router to route logs to an Amazon Web Services Service
         *        or Amazon Web Services Partner Network destination for log storage and analytics, you can set the
         *        <code>log-driver-buffer-limit</code> option to limit the number of events that are buffered in memory,
         *        before being sent to the log router container. It can help to resolve potential log loss issue because
         *        high throughput might result in memory running out for the buffer inside of Docker.
         *        </p>
         *        <p>
         *        Other options you can specify when using <code>awsfirelens</code> to route logs depend on the
         *        destination. When you export logs to Amazon Data Firehose, you can specify the Amazon Web Services
         *        Region with <code>region</code> and a name for the log stream with <code>delivery_stream</code>.
         *        </p>
         *        <p>
         *        When you export logs to Amazon Kinesis Data Streams, you can specify an Amazon Web Services Region
         *        with <code>region</code> and a data stream name with <code>stream</code>.
         *        </p>
         *        <p>
         *        When you export logs to Amazon OpenSearch Service, you can specify options like <code>Name</code>,
         *        <code>Host</code> (OpenSearch Service endpoint without protocol), <code>Port</code>,
         *        <code>Index</code>, <code>Type</code>, <code>Aws_auth</code>, <code>Aws_region</code>,
         *        <code>Suppress_Type_Name</code>, and <code>tls</code>. For more information, see <a
         *        href="http://aws.amazon.com/blogs/containers/under-the-hood-firelens-for-amazon-ecs-tasks/">Under the
         *        hood: FireLens for Amazon ECS Tasks</a>.
         *        </p>
         *        <p>
         *        When you export logs to Amazon S3, you can specify the bucket using the <code>bucket</code> option.
         *        You can also specify <code>region</code>, <code>total_file_size</code>, <code>upload_timeout</code>,
         *        and <code>use_put_object</code> as options.
         *        </p>
         *        <p>
         *        This parameter requires version 1.19 of the Docker Remote API or greater on your container instance.
         *        To check the Docker Remote API version on your container instance, log in to your container instance
         *        and run the following command: <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder options(Map<String, String> options);

        /**
         * <p>
         * The secrets to pass to the log configuration. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html">Specifying
         * sensitive data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param secretOptions
         *        The secrets to pass to the log configuration. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html"
         *        >Specifying sensitive data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretOptions(Collection<Secret> secretOptions);

        /**
         * <p>
         * The secrets to pass to the log configuration. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html">Specifying
         * sensitive data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param secretOptions
         *        The secrets to pass to the log configuration. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html"
         *        >Specifying sensitive data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretOptions(Secret... secretOptions);

        /**
         * <p>
         * The secrets to pass to the log configuration. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html">Specifying
         * sensitive data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.Secret.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ecs.model.Secret#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ecs.model.Secret.Builder#build()}
         * is called immediately and its result is passed to {@link #secretOptions(List<Secret>)}.
         * 
         * @param secretOptions
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ecs.model.Secret.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #secretOptions(java.util.Collection<Secret>)
         */
        Builder secretOptions(Consumer<Secret.Builder>... secretOptions);
    }

    static final class BuilderImpl implements Builder {
        private String logDriver;

        private Map<String, String> options = DefaultSdkAutoConstructMap.getInstance();

        private List<Secret> secretOptions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(LogConfiguration model) {
            logDriver(model.logDriver);
            options(model.options);
            secretOptions(model.secretOptions);
        }

        public final String getLogDriver() {
            return logDriver;
        }

        public final void setLogDriver(String logDriver) {
            this.logDriver = logDriver;
        }

        @Override
        public final Builder logDriver(String logDriver) {
            this.logDriver = logDriver;
            return this;
        }

        @Override
        public final Builder logDriver(LogDriver logDriver) {
            this.logDriver(logDriver == null ? null : logDriver.toString());
            return this;
        }

        public final Map<String, String> getOptions() {
            if (options instanceof SdkAutoConstructMap) {
                return null;
            }
            return options;
        }

        public final void setOptions(Map<String, String> options) {
            this.options = LogConfigurationOptionsMapCopier.copy(options);
        }

        @Override
        public final Builder options(Map<String, String> options) {
            this.options = LogConfigurationOptionsMapCopier.copy(options);
            return this;
        }

        public final List<Secret.Builder> getSecretOptions() {
            List<Secret.Builder> result = SecretListCopier.copyToBuilder(this.secretOptions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSecretOptions(Collection<Secret.BuilderImpl> secretOptions) {
            this.secretOptions = SecretListCopier.copyFromBuilder(secretOptions);
        }

        @Override
        public final Builder secretOptions(Collection<Secret> secretOptions) {
            this.secretOptions = SecretListCopier.copy(secretOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secretOptions(Secret... secretOptions) {
            secretOptions(Arrays.asList(secretOptions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secretOptions(Consumer<Secret.Builder>... secretOptions) {
            secretOptions(Stream.of(secretOptions).map(c -> Secret.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public LogConfiguration build() {
            return new LogConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
