/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A list of namespaced kernel parameters to set in the container. This parameter maps to <code>Sysctls</code> in the
 * docker container create command and the <code>--sysctl</code> option to docker run. For example, you can configure
 * <code>net.ipv4.tcp_keepalive_time</code> setting to maintain longer lived connections.
 * </p>
 * <p>
 * We don't recommend that you specify network-related <code>systemControls</code> parameters for multiple containers in
 * a single task that also uses either the <code>awsvpc</code> or <code>host</code> network mode. Doing this has the
 * following disadvantages:
 * </p>
 * <ul>
 * <li>
 * <p>
 * For tasks that use the <code>awsvpc</code> network mode including Fargate, if you set <code>systemControls</code> for
 * any container, it applies to all containers in the task. If you set different <code>systemControls</code> for
 * multiple containers in a single task, the container that's started last determines which <code>systemControls</code>
 * take effect.
 * </p>
 * </li>
 * <li>
 * <p>
 * For tasks that use the <code>host</code> network mode, the network namespace <code>systemControls</code> aren't
 * supported.
 * </p>
 * </li>
 * </ul>
 * <p>
 * If you're setting an IPC resource namespace to use for the containers in the task, the following conditions apply to
 * your system controls. For more information, see <a href=
 * "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html#task_definition_ipcmode"
 * >IPC mode</a>.
 * </p>
 * <ul>
 * <li>
 * <p>
 * For tasks that use the <code>host</code> IPC mode, IPC namespace <code>systemControls</code> aren't supported.
 * </p>
 * </li>
 * <li>
 * <p>
 * For tasks that use the <code>task</code> IPC mode, IPC namespace <code>systemControls</code> values apply to all
 * containers within a task.
 * </p>
 * </li>
 * </ul>
 * <note>
 * <p>
 * This parameter is not supported for Windows containers.
 * </p>
 * </note> <note>
 * <p>
 * This parameter is only supported for tasks that are hosted on Fargate if the tasks are using platform version
 * <code>1.4.0</code> or later (Linux). This isn't supported for Windows containers on Fargate.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SystemControl implements SdkPojo, Serializable, ToCopyableBuilder<SystemControl.Builder, SystemControl> {
    private static final SdkField<String> NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("namespace").getter(getter(SystemControl::namespace)).setter(setter(Builder::namespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("namespace").build()).build();

    private static final SdkField<String> VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("value")
            .getter(getter(SystemControl::value)).setter(setter(Builder::value))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("value").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAMESPACE_FIELD, VALUE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String namespace;

    private final String value;

    private SystemControl(BuilderImpl builder) {
        this.namespace = builder.namespace;
        this.value = builder.value;
    }

    /**
     * <p>
     * The namespaced kernel parameter to set a <code>value</code> for.
     * </p>
     * 
     * @return The namespaced kernel parameter to set a <code>value</code> for.
     */
    public final String namespace() {
        return namespace;
    }

    /**
     * <p>
     * The namespaced kernel parameter to set a <code>value</code> for.
     * </p>
     * <p>
     * Valid IPC namespace values:
     * <code>"kernel.msgmax" | "kernel.msgmnb" | "kernel.msgmni" | "kernel.sem" | "kernel.shmall" | "kernel.shmmax" | "kernel.shmmni" | "kernel.shm_rmid_forced"</code>
     * , and <code>Sysctls</code> that start with <code>"fs.mqueue.*"</code>
     * </p>
     * <p>
     * Valid network namespace values: <code>Sysctls</code> that start with <code>"net.*"</code>
     * </p>
     * <p>
     * All of these values are supported by Fargate.
     * </p>
     * 
     * @return The namespaced kernel parameter to set a <code>value</code> for.</p>
     *         <p>
     *         Valid IPC namespace values:
     *         <code>"kernel.msgmax" | "kernel.msgmnb" | "kernel.msgmni" | "kernel.sem" | "kernel.shmall" | "kernel.shmmax" | "kernel.shmmni" | "kernel.shm_rmid_forced"</code>
     *         , and <code>Sysctls</code> that start with <code>"fs.mqueue.*"</code>
     *         </p>
     *         <p>
     *         Valid network namespace values: <code>Sysctls</code> that start with <code>"net.*"</code>
     *         </p>
     *         <p>
     *         All of these values are supported by Fargate.
     */
    public final String value() {
        return value;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(namespace());
        hashCode = 31 * hashCode + Objects.hashCode(value());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SystemControl)) {
            return false;
        }
        SystemControl other = (SystemControl) obj;
        return Objects.equals(namespace(), other.namespace()) && Objects.equals(value(), other.value());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SystemControl").add("Namespace", namespace()).add("Value", value()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "namespace":
            return Optional.ofNullable(clazz.cast(namespace()));
        case "value":
            return Optional.ofNullable(clazz.cast(value()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("namespace", NAMESPACE_FIELD);
        map.put("value", VALUE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<SystemControl, T> g) {
        return obj -> g.apply((SystemControl) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SystemControl> {
        /**
         * <p>
         * The namespaced kernel parameter to set a <code>value</code> for.
         * </p>
         * 
         * @param namespace
         *        The namespaced kernel parameter to set a <code>value</code> for.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(String namespace);

        /**
         * <p>
         * The namespaced kernel parameter to set a <code>value</code> for.
         * </p>
         * <p>
         * Valid IPC namespace values:
         * <code>"kernel.msgmax" | "kernel.msgmnb" | "kernel.msgmni" | "kernel.sem" | "kernel.shmall" | "kernel.shmmax" | "kernel.shmmni" | "kernel.shm_rmid_forced"</code>
         * , and <code>Sysctls</code> that start with <code>"fs.mqueue.*"</code>
         * </p>
         * <p>
         * Valid network namespace values: <code>Sysctls</code> that start with <code>"net.*"</code>
         * </p>
         * <p>
         * All of these values are supported by Fargate.
         * </p>
         * 
         * @param value
         *        The namespaced kernel parameter to set a <code>value</code> for.</p>
         *        <p>
         *        Valid IPC namespace values:
         *        <code>"kernel.msgmax" | "kernel.msgmnb" | "kernel.msgmni" | "kernel.sem" | "kernel.shmall" | "kernel.shmmax" | "kernel.shmmni" | "kernel.shm_rmid_forced"</code>
         *        , and <code>Sysctls</code> that start with <code>"fs.mqueue.*"</code>
         *        </p>
         *        <p>
         *        Valid network namespace values: <code>Sysctls</code> that start with <code>"net.*"</code>
         *        </p>
         *        <p>
         *        All of these values are supported by Fargate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder value(String value);
    }

    static final class BuilderImpl implements Builder {
        private String namespace;

        private String value;

        private BuilderImpl() {
        }

        private BuilderImpl(SystemControl model) {
            namespace(model.namespace);
            value(model.value);
        }

        public final String getNamespace() {
            return namespace;
        }

        public final void setNamespace(String namespace) {
            this.namespace = namespace;
        }

        @Override
        public final Builder namespace(String namespace) {
            this.namespace = namespace;
            return this;
        }

        public final String getValue() {
            return value;
        }

        public final void setValue(String value) {
            this.value = value;
        }

        @Override
        public final Builder value(String value) {
            this.value = value;
            return this;
        }

        @Override
        public SystemControl build() {
            return new SystemControl(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
