/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.ecs.internal.EcsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.ecs.model.AccessDeniedException;
import software.amazon.awssdk.services.ecs.model.AttributeLimitExceededException;
import software.amazon.awssdk.services.ecs.model.BlockedException;
import software.amazon.awssdk.services.ecs.model.ClientException;
import software.amazon.awssdk.services.ecs.model.ClusterContainsContainerInstancesException;
import software.amazon.awssdk.services.ecs.model.ClusterContainsServicesException;
import software.amazon.awssdk.services.ecs.model.ClusterContainsTasksException;
import software.amazon.awssdk.services.ecs.model.ClusterNotFoundException;
import software.amazon.awssdk.services.ecs.model.ConflictException;
import software.amazon.awssdk.services.ecs.model.CreateCapacityProviderRequest;
import software.amazon.awssdk.services.ecs.model.CreateCapacityProviderResponse;
import software.amazon.awssdk.services.ecs.model.CreateClusterRequest;
import software.amazon.awssdk.services.ecs.model.CreateClusterResponse;
import software.amazon.awssdk.services.ecs.model.CreateServiceRequest;
import software.amazon.awssdk.services.ecs.model.CreateServiceResponse;
import software.amazon.awssdk.services.ecs.model.CreateTaskSetRequest;
import software.amazon.awssdk.services.ecs.model.CreateTaskSetResponse;
import software.amazon.awssdk.services.ecs.model.DeleteAccountSettingRequest;
import software.amazon.awssdk.services.ecs.model.DeleteAccountSettingResponse;
import software.amazon.awssdk.services.ecs.model.DeleteAttributesRequest;
import software.amazon.awssdk.services.ecs.model.DeleteAttributesResponse;
import software.amazon.awssdk.services.ecs.model.DeleteCapacityProviderRequest;
import software.amazon.awssdk.services.ecs.model.DeleteCapacityProviderResponse;
import software.amazon.awssdk.services.ecs.model.DeleteClusterRequest;
import software.amazon.awssdk.services.ecs.model.DeleteClusterResponse;
import software.amazon.awssdk.services.ecs.model.DeleteServiceRequest;
import software.amazon.awssdk.services.ecs.model.DeleteServiceResponse;
import software.amazon.awssdk.services.ecs.model.DeleteTaskDefinitionsRequest;
import software.amazon.awssdk.services.ecs.model.DeleteTaskDefinitionsResponse;
import software.amazon.awssdk.services.ecs.model.DeleteTaskSetRequest;
import software.amazon.awssdk.services.ecs.model.DeleteTaskSetResponse;
import software.amazon.awssdk.services.ecs.model.DeregisterContainerInstanceRequest;
import software.amazon.awssdk.services.ecs.model.DeregisterContainerInstanceResponse;
import software.amazon.awssdk.services.ecs.model.DeregisterTaskDefinitionRequest;
import software.amazon.awssdk.services.ecs.model.DeregisterTaskDefinitionResponse;
import software.amazon.awssdk.services.ecs.model.DescribeCapacityProvidersRequest;
import software.amazon.awssdk.services.ecs.model.DescribeCapacityProvidersResponse;
import software.amazon.awssdk.services.ecs.model.DescribeClustersRequest;
import software.amazon.awssdk.services.ecs.model.DescribeClustersResponse;
import software.amazon.awssdk.services.ecs.model.DescribeContainerInstancesRequest;
import software.amazon.awssdk.services.ecs.model.DescribeContainerInstancesResponse;
import software.amazon.awssdk.services.ecs.model.DescribeServiceDeploymentsRequest;
import software.amazon.awssdk.services.ecs.model.DescribeServiceDeploymentsResponse;
import software.amazon.awssdk.services.ecs.model.DescribeServiceRevisionsRequest;
import software.amazon.awssdk.services.ecs.model.DescribeServiceRevisionsResponse;
import software.amazon.awssdk.services.ecs.model.DescribeServicesRequest;
import software.amazon.awssdk.services.ecs.model.DescribeServicesResponse;
import software.amazon.awssdk.services.ecs.model.DescribeTaskDefinitionRequest;
import software.amazon.awssdk.services.ecs.model.DescribeTaskDefinitionResponse;
import software.amazon.awssdk.services.ecs.model.DescribeTaskSetsRequest;
import software.amazon.awssdk.services.ecs.model.DescribeTaskSetsResponse;
import software.amazon.awssdk.services.ecs.model.DescribeTasksRequest;
import software.amazon.awssdk.services.ecs.model.DescribeTasksResponse;
import software.amazon.awssdk.services.ecs.model.DiscoverPollEndpointRequest;
import software.amazon.awssdk.services.ecs.model.DiscoverPollEndpointResponse;
import software.amazon.awssdk.services.ecs.model.EcsException;
import software.amazon.awssdk.services.ecs.model.ExecuteCommandRequest;
import software.amazon.awssdk.services.ecs.model.ExecuteCommandResponse;
import software.amazon.awssdk.services.ecs.model.GetTaskProtectionRequest;
import software.amazon.awssdk.services.ecs.model.GetTaskProtectionResponse;
import software.amazon.awssdk.services.ecs.model.InvalidParameterException;
import software.amazon.awssdk.services.ecs.model.LimitExceededException;
import software.amazon.awssdk.services.ecs.model.ListAccountSettingsRequest;
import software.amazon.awssdk.services.ecs.model.ListAccountSettingsResponse;
import software.amazon.awssdk.services.ecs.model.ListAttributesRequest;
import software.amazon.awssdk.services.ecs.model.ListAttributesResponse;
import software.amazon.awssdk.services.ecs.model.ListClustersRequest;
import software.amazon.awssdk.services.ecs.model.ListClustersResponse;
import software.amazon.awssdk.services.ecs.model.ListContainerInstancesRequest;
import software.amazon.awssdk.services.ecs.model.ListContainerInstancesResponse;
import software.amazon.awssdk.services.ecs.model.ListServiceDeploymentsRequest;
import software.amazon.awssdk.services.ecs.model.ListServiceDeploymentsResponse;
import software.amazon.awssdk.services.ecs.model.ListServicesByNamespaceRequest;
import software.amazon.awssdk.services.ecs.model.ListServicesByNamespaceResponse;
import software.amazon.awssdk.services.ecs.model.ListServicesRequest;
import software.amazon.awssdk.services.ecs.model.ListServicesResponse;
import software.amazon.awssdk.services.ecs.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.ecs.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.ecs.model.ListTaskDefinitionFamiliesRequest;
import software.amazon.awssdk.services.ecs.model.ListTaskDefinitionFamiliesResponse;
import software.amazon.awssdk.services.ecs.model.ListTaskDefinitionsRequest;
import software.amazon.awssdk.services.ecs.model.ListTaskDefinitionsResponse;
import software.amazon.awssdk.services.ecs.model.ListTasksRequest;
import software.amazon.awssdk.services.ecs.model.ListTasksResponse;
import software.amazon.awssdk.services.ecs.model.MissingVersionException;
import software.amazon.awssdk.services.ecs.model.NamespaceNotFoundException;
import software.amazon.awssdk.services.ecs.model.NoUpdateAvailableException;
import software.amazon.awssdk.services.ecs.model.PlatformTaskDefinitionIncompatibilityException;
import software.amazon.awssdk.services.ecs.model.PlatformUnknownException;
import software.amazon.awssdk.services.ecs.model.PutAccountSettingDefaultRequest;
import software.amazon.awssdk.services.ecs.model.PutAccountSettingDefaultResponse;
import software.amazon.awssdk.services.ecs.model.PutAccountSettingRequest;
import software.amazon.awssdk.services.ecs.model.PutAccountSettingResponse;
import software.amazon.awssdk.services.ecs.model.PutAttributesRequest;
import software.amazon.awssdk.services.ecs.model.PutAttributesResponse;
import software.amazon.awssdk.services.ecs.model.PutClusterCapacityProvidersRequest;
import software.amazon.awssdk.services.ecs.model.PutClusterCapacityProvidersResponse;
import software.amazon.awssdk.services.ecs.model.RegisterContainerInstanceRequest;
import software.amazon.awssdk.services.ecs.model.RegisterContainerInstanceResponse;
import software.amazon.awssdk.services.ecs.model.RegisterTaskDefinitionRequest;
import software.amazon.awssdk.services.ecs.model.RegisterTaskDefinitionResponse;
import software.amazon.awssdk.services.ecs.model.ResourceInUseException;
import software.amazon.awssdk.services.ecs.model.ResourceNotFoundException;
import software.amazon.awssdk.services.ecs.model.RunTaskRequest;
import software.amazon.awssdk.services.ecs.model.RunTaskResponse;
import software.amazon.awssdk.services.ecs.model.ServerException;
import software.amazon.awssdk.services.ecs.model.ServiceDeploymentNotFoundException;
import software.amazon.awssdk.services.ecs.model.ServiceNotActiveException;
import software.amazon.awssdk.services.ecs.model.ServiceNotFoundException;
import software.amazon.awssdk.services.ecs.model.StartTaskRequest;
import software.amazon.awssdk.services.ecs.model.StartTaskResponse;
import software.amazon.awssdk.services.ecs.model.StopServiceDeploymentRequest;
import software.amazon.awssdk.services.ecs.model.StopServiceDeploymentResponse;
import software.amazon.awssdk.services.ecs.model.StopTaskRequest;
import software.amazon.awssdk.services.ecs.model.StopTaskResponse;
import software.amazon.awssdk.services.ecs.model.SubmitAttachmentStateChangesRequest;
import software.amazon.awssdk.services.ecs.model.SubmitAttachmentStateChangesResponse;
import software.amazon.awssdk.services.ecs.model.SubmitContainerStateChangeRequest;
import software.amazon.awssdk.services.ecs.model.SubmitContainerStateChangeResponse;
import software.amazon.awssdk.services.ecs.model.SubmitTaskStateChangeRequest;
import software.amazon.awssdk.services.ecs.model.SubmitTaskStateChangeResponse;
import software.amazon.awssdk.services.ecs.model.TagResourceRequest;
import software.amazon.awssdk.services.ecs.model.TagResourceResponse;
import software.amazon.awssdk.services.ecs.model.TargetNotConnectedException;
import software.amazon.awssdk.services.ecs.model.TargetNotFoundException;
import software.amazon.awssdk.services.ecs.model.TaskSetNotFoundException;
import software.amazon.awssdk.services.ecs.model.UnsupportedFeatureException;
import software.amazon.awssdk.services.ecs.model.UntagResourceRequest;
import software.amazon.awssdk.services.ecs.model.UntagResourceResponse;
import software.amazon.awssdk.services.ecs.model.UpdateCapacityProviderRequest;
import software.amazon.awssdk.services.ecs.model.UpdateCapacityProviderResponse;
import software.amazon.awssdk.services.ecs.model.UpdateClusterRequest;
import software.amazon.awssdk.services.ecs.model.UpdateClusterResponse;
import software.amazon.awssdk.services.ecs.model.UpdateClusterSettingsRequest;
import software.amazon.awssdk.services.ecs.model.UpdateClusterSettingsResponse;
import software.amazon.awssdk.services.ecs.model.UpdateContainerAgentRequest;
import software.amazon.awssdk.services.ecs.model.UpdateContainerAgentResponse;
import software.amazon.awssdk.services.ecs.model.UpdateContainerInstancesStateRequest;
import software.amazon.awssdk.services.ecs.model.UpdateContainerInstancesStateResponse;
import software.amazon.awssdk.services.ecs.model.UpdateInProgressException;
import software.amazon.awssdk.services.ecs.model.UpdateServicePrimaryTaskSetRequest;
import software.amazon.awssdk.services.ecs.model.UpdateServicePrimaryTaskSetResponse;
import software.amazon.awssdk.services.ecs.model.UpdateServiceRequest;
import software.amazon.awssdk.services.ecs.model.UpdateServiceResponse;
import software.amazon.awssdk.services.ecs.model.UpdateTaskProtectionRequest;
import software.amazon.awssdk.services.ecs.model.UpdateTaskProtectionResponse;
import software.amazon.awssdk.services.ecs.model.UpdateTaskSetRequest;
import software.amazon.awssdk.services.ecs.model.UpdateTaskSetResponse;
import software.amazon.awssdk.services.ecs.transform.CreateCapacityProviderRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.CreateClusterRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.CreateServiceRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.CreateTaskSetRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.DeleteAccountSettingRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.DeleteAttributesRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.DeleteCapacityProviderRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.DeleteClusterRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.DeleteServiceRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.DeleteTaskDefinitionsRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.DeleteTaskSetRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.DeregisterContainerInstanceRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.DeregisterTaskDefinitionRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.DescribeCapacityProvidersRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.DescribeClustersRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.DescribeContainerInstancesRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.DescribeServiceDeploymentsRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.DescribeServiceRevisionsRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.DescribeServicesRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.DescribeTaskDefinitionRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.DescribeTaskSetsRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.DescribeTasksRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.DiscoverPollEndpointRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.ExecuteCommandRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.GetTaskProtectionRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.ListAccountSettingsRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.ListAttributesRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.ListClustersRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.ListContainerInstancesRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.ListServiceDeploymentsRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.ListServicesByNamespaceRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.ListServicesRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.ListTaskDefinitionFamiliesRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.ListTaskDefinitionsRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.ListTasksRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.PutAccountSettingDefaultRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.PutAccountSettingRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.PutAttributesRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.PutClusterCapacityProvidersRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.RegisterContainerInstanceRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.RegisterTaskDefinitionRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.RunTaskRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.StartTaskRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.StopServiceDeploymentRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.StopTaskRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.SubmitAttachmentStateChangesRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.SubmitContainerStateChangeRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.SubmitTaskStateChangeRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.UpdateCapacityProviderRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.UpdateClusterRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.UpdateClusterSettingsRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.UpdateContainerAgentRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.UpdateContainerInstancesStateRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.UpdateServicePrimaryTaskSetRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.UpdateServiceRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.UpdateTaskProtectionRequestMarshaller;
import software.amazon.awssdk.services.ecs.transform.UpdateTaskSetRequestMarshaller;
import software.amazon.awssdk.services.ecs.waiters.EcsWaiter;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link EcsClient}.
 *
 * @see EcsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultEcsClient implements EcsClient {
    private static final Logger log = Logger.loggerFor(DefaultEcsClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultEcsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a new capacity provider. Capacity providers are associated with an Amazon ECS cluster and are used in
     * capacity provider strategies to facilitate cluster auto scaling.
     * </p>
     * <p>
     * Only capacity providers that use an Auto Scaling group can be created. Amazon ECS tasks on Fargate use the
     * <code>FARGATE</code> and <code>FARGATE_SPOT</code> capacity providers. These providers are available to all
     * accounts in the Amazon Web Services Regions that Fargate supports.
     * </p>
     *
     * @param createCapacityProviderRequest
     * @return Result of the CreateCapacityProvider operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws LimitExceededException
     *         The limit for the resource was exceeded.
     * @throws UpdateInProgressException
     *         There's already a current Amazon ECS container agent update in progress on the container instance that's
     *         specified. If the container agent becomes disconnected while it's in a transitional stage, such as
     *         <code>PENDING</code> or <code>STAGING</code>, the update process can get stuck in that state. However,
     *         when the agent reconnects, it resumes where it stopped previously.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.CreateCapacityProvider
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/CreateCapacityProvider" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateCapacityProviderResponse createCapacityProvider(CreateCapacityProviderRequest createCapacityProviderRequest)
            throws ServerException, ClientException, InvalidParameterException, LimitExceededException,
            UpdateInProgressException, AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateCapacityProviderResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateCapacityProviderResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createCapacityProviderRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCapacityProviderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCapacityProvider");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateCapacityProviderRequest, CreateCapacityProviderResponse>()
                            .withOperationName("CreateCapacityProvider").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createCapacityProviderRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateCapacityProviderRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new Amazon ECS cluster. By default, your account receives a <code>default</code> cluster when you
     * launch your first container instance. However, you can create your own cluster with a unique name.
     * </p>
     * <note>
     * <p>
     * When you call the <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_CreateCluster.html">CreateCluster</a> API
     * operation, Amazon ECS attempts to create the Amazon ECS service-linked role for your account. This is so that it
     * can manage required resources in other Amazon Web Services services on your behalf. However, if the user that
     * makes the call doesn't have permissions to create the service-linked role, it isn't created. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using-service-linked-roles.html">Using
     * service-linked roles for Amazon ECS</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * </note>
     *
     * @param createClusterRequest
     * @return Result of the CreateCluster operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws NamespaceNotFoundException
     *         The specified namespace wasn't found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.CreateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/CreateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateClusterResponse createCluster(CreateClusterRequest createClusterRequest) throws ServerException,
            ClientException, InvalidParameterException, NamespaceNotFoundException, AwsServiceException, SdkClientException,
            EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateClusterResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createClusterRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCluster");

            return clientHandler.execute(new ClientExecutionParams<CreateClusterRequest, CreateClusterResponse>()
                    .withOperationName("CreateCluster").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createClusterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateClusterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Runs and maintains your desired number of tasks from a specified task definition. If the number of tasks running
     * in a service drops below the <code>desiredCount</code>, Amazon ECS runs another copy of the task in the specified
     * cluster. To update an existing service, use <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_UpdateService.html">UpdateService</a>.
     * </p>
     * <note>
     * <p>
     * On March 21, 2024, a change was made to resolve the task definition revision before authorization. When a task
     * definition revision is not specified, authorization will occur using the latest revision of a task definition.
     * </p>
     * </note> <note>
     * <p>
     * Amazon Elastic Inference (EI) is no longer available to customers.
     * </p>
     * </note>
     * <p>
     * In addition to maintaining the desired count of tasks in your service, you can optionally run your service behind
     * one or more load balancers. The load balancers distribute traffic across the tasks that are associated with the
     * service. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-load-balancing.html">Service load
     * balancing</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * You can attach Amazon EBS volumes to Amazon ECS tasks by configuring the volume when creating or updating a
     * service. <code>volumeConfigurations</code> is only supported for REPLICA service and not DAEMON service. For more
     * infomation, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ebs-volumes.html#ebs-volume-types">Amazon EBS
     * volumes</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * Tasks for services that don't use a load balancer are considered healthy if they're in the <code>RUNNING</code>
     * state. Tasks for services that use a load balancer are considered healthy if they're in the <code>RUNNING</code>
     * state and are reported as healthy by the load balancer.
     * </p>
     * <p>
     * There are two service scheduler strategies available:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>REPLICA</code> - The replica scheduling strategy places and maintains your desired number of tasks across
     * your cluster. By default, the service scheduler spreads tasks across Availability Zones. You can use task
     * placement strategies and constraints to customize task placement decisions. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs_services.html">Service scheduler
     * concepts</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DAEMON</code> - The daemon scheduling strategy deploys exactly one task on each active container instance
     * that meets all of the task placement constraints that you specify in your cluster. The service scheduler also
     * evaluates the task placement constraints for running tasks. It also stops tasks that don't meet the placement
     * constraints. When using this strategy, you don't need to specify a desired number of tasks, a task placement
     * strategy, or use Service Auto Scaling policies. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs_services.html">Service scheduler
     * concepts</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can optionally specify a deployment configuration for your service. The deployment is initiated by changing
     * properties. For example, the deployment might be initiated by the task definition or by your desired count of a
     * service. You can use <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_UpdateService.html">UpdateService</a>. The
     * default value for a replica service for <code>minimumHealthyPercent</code> is 100%. The default value for a
     * daemon service for <code>minimumHealthyPercent</code> is 0%.
     * </p>
     * <p>
     * If a service uses the <code>ECS</code> deployment controller, the minimum healthy percent represents a lower
     * limit on the number of tasks in a service that must remain in the <code>RUNNING</code> state during a deployment.
     * Specifically, it represents it as a percentage of your desired number of tasks (rounded up to the nearest
     * integer). This happens when any of your container instances are in the <code>DRAINING</code> state if the service
     * contains tasks using the EC2 launch type. Using this parameter, you can deploy without using additional cluster
     * capacity. For example, if you set your service to have desired number of four tasks and a minimum healthy percent
     * of 50%, the scheduler might stop two existing tasks to free up cluster capacity before starting two new tasks. If
     * they're in the <code>RUNNING</code> state, tasks for services that don't use a load balancer are considered
     * healthy . If they're in the <code>RUNNING</code> state and reported as healthy by the load balancer, tasks for
     * services that <i>do</i> use a load balancer are considered healthy . The default value for minimum healthy
     * percent is 100%.
     * </p>
     * <p>
     * If a service uses the <code>ECS</code> deployment controller, the <b>maximum percent</b> parameter represents an
     * upper limit on the number of tasks in a service that are allowed in the <code>RUNNING</code> or
     * <code>PENDING</code> state during a deployment. Specifically, it represents it as a percentage of the desired
     * number of tasks (rounded down to the nearest integer). This happens when any of your container instances are in
     * the <code>DRAINING</code> state if the service contains tasks using the EC2 launch type. Using this parameter,
     * you can define the deployment batch size. For example, if your service has a desired number of four tasks and a
     * maximum percent value of 200%, the scheduler may start four new tasks before stopping the four older tasks
     * (provided that the cluster resources required to do this are available). The default value for maximum percent is
     * 200%.
     * </p>
     * <p>
     * If a service uses either the <code>CODE_DEPLOY</code> or <code>EXTERNAL</code> deployment controller types and
     * tasks that use the EC2 launch type, the <b>minimum healthy percent</b> and <b>maximum percent</b> values are used
     * only to define the lower and upper limit on the number of the tasks in the service that remain in the
     * <code>RUNNING</code> state. This is while the container instances are in the <code>DRAINING</code> state. If the
     * tasks in the service use the Fargate launch type, the minimum healthy percent and maximum percent values aren't
     * used. This is the case even if they're currently visible when describing your service.
     * </p>
     * <p>
     * When creating a service that uses the <code>EXTERNAL</code> deployment controller, you can specify only
     * parameters that aren't controlled at the task set level. The only required parameter is the service name. You
     * control your services using the <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_CreateTaskSet.html">CreateTaskSet</a>. For
     * more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html">Amazon ECS deployment
     * types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * When the service scheduler launches new tasks, it determines task placement. For information about task placement
     * and task placement strategies, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement.html">Amazon ECS task
     * placement</a> in the <i>Amazon Elastic Container Service Developer Guide</i>
     * </p>
     *
     * @param createServiceRequest
     * @return Result of the CreateService operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws UnsupportedFeatureException
     *         The specified task isn't supported in this Region.
     * @throws PlatformUnknownException
     *         The specified platform version doesn't exist.
     * @throws PlatformTaskDefinitionIncompatibilityException
     *         The specified platform version doesn't satisfy the required capabilities of the task definition.
     * @throws AccessDeniedException
     *         You don't have authorization to perform the requested action.
     * @throws NamespaceNotFoundException
     *         The specified namespace wasn't found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.CreateService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/CreateService" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateServiceResponse createService(CreateServiceRequest createServiceRequest) throws ServerException,
            ClientException, InvalidParameterException, ClusterNotFoundException, UnsupportedFeatureException,
            PlatformUnknownException, PlatformTaskDefinitionIncompatibilityException, AccessDeniedException,
            NamespaceNotFoundException, AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateServiceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createServiceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateService");

            return clientHandler.execute(new ClientExecutionParams<CreateServiceRequest, CreateServiceResponse>()
                    .withOperationName("CreateService").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createServiceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateServiceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Create a task set in the specified cluster and service. This is used when a service uses the
     * <code>EXTERNAL</code> deployment controller type. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html">Amazon ECS deployment
     * types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * On March 21, 2024, a change was made to resolve the task definition revision before authorization. When a task
     * definition revision is not specified, authorization will occur using the latest revision of a task definition.
     * </p>
     * </note>
     * <p>
     * For information about the maximum number of task sets and other quotas, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-quotas.html">Amazon ECS service
     * quotas</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     *
     * @param createTaskSetRequest
     * @return Result of the CreateTaskSet operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws UnsupportedFeatureException
     *         The specified task isn't supported in this Region.
     * @throws PlatformUnknownException
     *         The specified platform version doesn't exist.
     * @throws PlatformTaskDefinitionIncompatibilityException
     *         The specified platform version doesn't satisfy the required capabilities of the task definition.
     * @throws AccessDeniedException
     *         You don't have authorization to perform the requested action.
     * @throws ServiceNotFoundException
     *         The specified service wasn't found. You can view your available services with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListServices.html">ListServices</a>.
     *         Amazon ECS services are cluster specific and Region specific.
     * @throws ServiceNotActiveException
     *         The specified service isn't active. You can't update a service that's inactive. If you have previously
     *         deleted a service, you can re-create it with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_CreateService.html"
     *         >CreateService</a>.
     * @throws NamespaceNotFoundException
     *         The specified namespace wasn't found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.CreateTaskSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/CreateTaskSet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateTaskSetResponse createTaskSet(CreateTaskSetRequest createTaskSetRequest) throws ServerException,
            ClientException, InvalidParameterException, ClusterNotFoundException, UnsupportedFeatureException,
            PlatformUnknownException, PlatformTaskDefinitionIncompatibilityException, AccessDeniedException,
            ServiceNotFoundException, ServiceNotActiveException, NamespaceNotFoundException, AwsServiceException,
            SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateTaskSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateTaskSetResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createTaskSetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTaskSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTaskSet");

            return clientHandler.execute(new ClientExecutionParams<CreateTaskSetRequest, CreateTaskSetResponse>()
                    .withOperationName("CreateTaskSet").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createTaskSetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateTaskSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disables an account setting for a specified user, role, or the root user for an account.
     * </p>
     *
     * @param deleteAccountSettingRequest
     * @return Result of the DeleteAccountSetting operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.DeleteAccountSetting
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/DeleteAccountSetting" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteAccountSettingResponse deleteAccountSetting(DeleteAccountSettingRequest deleteAccountSettingRequest)
            throws ServerException, ClientException, InvalidParameterException, AwsServiceException, SdkClientException,
            EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAccountSettingResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteAccountSettingResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAccountSettingRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAccountSettingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAccountSetting");

            return clientHandler.execute(new ClientExecutionParams<DeleteAccountSettingRequest, DeleteAccountSettingResponse>()
                    .withOperationName("DeleteAccountSetting").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteAccountSettingRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteAccountSettingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes one or more custom attributes from an Amazon ECS resource.
     * </p>
     *
     * @param deleteAttributesRequest
     * @return Result of the DeleteAttributes operation returned by the service.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws TargetNotFoundException
     *         The specified target wasn't found. You can view your available container instances with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListContainerInstances.html"
     *         >ListContainerInstances</a>. Amazon ECS container instances are cluster-specific and Region-specific.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.DeleteAttributes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/DeleteAttributes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteAttributesResponse deleteAttributes(DeleteAttributesRequest deleteAttributesRequest)
            throws ClusterNotFoundException, TargetNotFoundException, InvalidParameterException, AwsServiceException,
            SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAttributesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteAttributesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAttributesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAttributesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAttributes");

            return clientHandler.execute(new ClientExecutionParams<DeleteAttributesRequest, DeleteAttributesResponse>()
                    .withOperationName("DeleteAttributes").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteAttributesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteAttributesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified capacity provider.
     * </p>
     * <note>
     * <p>
     * The <code>FARGATE</code> and <code>FARGATE_SPOT</code> capacity providers are reserved and can't be deleted. You
     * can disassociate them from a cluster using either <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutClusterCapacityProviders.html"
     * >PutClusterCapacityProviders</a> or by deleting the cluster.
     * </p>
     * </note>
     * <p>
     * Prior to a capacity provider being deleted, the capacity provider must be removed from the capacity provider
     * strategy from all services. The <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_UpdateService.html">UpdateService</a> API can
     * be used to remove a capacity provider from a service's capacity provider strategy. When updating a service, the
     * <code>forceNewDeployment</code> option can be used to ensure that any tasks using the Amazon EC2 instance
     * capacity provided by the capacity provider are transitioned to use the capacity from the remaining capacity
     * providers. Only capacity providers that aren't associated with a cluster can be deleted. To remove a capacity
     * provider from a cluster, you can either use <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutClusterCapacityProviders.html"
     * >PutClusterCapacityProviders</a> or delete the cluster.
     * </p>
     *
     * @param deleteCapacityProviderRequest
     * @return Result of the DeleteCapacityProvider operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.DeleteCapacityProvider
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/DeleteCapacityProvider" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteCapacityProviderResponse deleteCapacityProvider(DeleteCapacityProviderRequest deleteCapacityProviderRequest)
            throws ServerException, ClientException, InvalidParameterException, AwsServiceException, SdkClientException,
            EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteCapacityProviderResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteCapacityProviderResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteCapacityProviderRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCapacityProviderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCapacityProvider");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteCapacityProviderRequest, DeleteCapacityProviderResponse>()
                            .withOperationName("DeleteCapacityProvider").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteCapacityProviderRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteCapacityProviderRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified cluster. The cluster transitions to the <code>INACTIVE</code> state. Clusters with an
     * <code>INACTIVE</code> status might remain discoverable in your account for a period of time. However, this
     * behavior is subject to change in the future. We don't recommend that you rely on <code>INACTIVE</code> clusters
     * persisting.
     * </p>
     * <p>
     * You must deregister all container instances from this cluster before you may delete it. You can list the
     * container instances in a cluster with <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListContainerInstances.html"
     * >ListContainerInstances</a> and deregister them with <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DeregisterContainerInstance.html"
     * >DeregisterContainerInstance</a>.
     * </p>
     *
     * @param deleteClusterRequest
     * @return Result of the DeleteCluster operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws ClusterContainsContainerInstancesException
     *         You can't delete a cluster that has registered container instances. First, deregister the container
     *         instances before you can delete the cluster. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DeregisterContainerInstance.html"
     *         >DeregisterContainerInstance</a>.
     * @throws ClusterContainsServicesException
     *         You can't delete a cluster that contains services. First, update the service to reduce its desired task
     *         count to 0, and then delete the service. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_UpdateService.html">UpdateService</a>
     *         and <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DeleteService.html">DeleteService
     *         </a>.
     * @throws ClusterContainsTasksException
     *         You can't delete a cluster that has active tasks.
     * @throws UpdateInProgressException
     *         There's already a current Amazon ECS container agent update in progress on the container instance that's
     *         specified. If the container agent becomes disconnected while it's in a transitional stage, such as
     *         <code>PENDING</code> or <code>STAGING</code>, the update process can get stuck in that state. However,
     *         when the agent reconnects, it resumes where it stopped previously.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.DeleteCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/DeleteCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteClusterResponse deleteCluster(DeleteClusterRequest deleteClusterRequest) throws ServerException,
            ClientException, InvalidParameterException, ClusterNotFoundException, ClusterContainsContainerInstancesException,
            ClusterContainsServicesException, ClusterContainsTasksException, UpdateInProgressException, AwsServiceException,
            SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteClusterResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteClusterRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCluster");

            return clientHandler.execute(new ClientExecutionParams<DeleteClusterRequest, DeleteClusterResponse>()
                    .withOperationName("DeleteCluster").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteClusterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteClusterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a specified service within a cluster. You can delete a service if you have no running tasks in it and the
     * desired task count is zero. If the service is actively maintaining tasks, you can't delete it, and you must
     * update the service to a desired task count of zero. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_UpdateService.html">UpdateService</a>.
     * </p>
     * <note>
     * <p>
     * When you delete a service, if there are still running tasks that require cleanup, the service status moves from
     * <code>ACTIVE</code> to <code>DRAINING</code>, and the service is no longer visible in the console or in the <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListServices.html">ListServices</a> API
     * operation. After all tasks have transitioned to either <code>STOPPING</code> or <code>STOPPED</code> status, the
     * service status moves from <code>DRAINING</code> to <code>INACTIVE</code>. Services in the <code>DRAINING</code>
     * or <code>INACTIVE</code> status can still be viewed with the <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeServices.html">DescribeServices</a>
     * API operation. However, in the future, <code>INACTIVE</code> services may be cleaned up and purged from Amazon
     * ECS record keeping, and <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeServices.html">DescribeServices</a>
     * calls on those services return a <code>ServiceNotFoundException</code> error.
     * </p>
     * </note> <important>
     * <p>
     * If you attempt to create a new service with the same name as an existing service in either <code>ACTIVE</code> or
     * <code>DRAINING</code> status, you receive an error.
     * </p>
     * </important>
     *
     * @param deleteServiceRequest
     * @return Result of the DeleteService operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws ServiceNotFoundException
     *         The specified service wasn't found. You can view your available services with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListServices.html">ListServices</a>.
     *         Amazon ECS services are cluster specific and Region specific.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.DeleteService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/DeleteService" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteServiceResponse deleteService(DeleteServiceRequest deleteServiceRequest) throws ServerException,
            ClientException, InvalidParameterException, ClusterNotFoundException, ServiceNotFoundException, AwsServiceException,
            SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteServiceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteServiceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteService");

            return clientHandler.execute(new ClientExecutionParams<DeleteServiceRequest, DeleteServiceResponse>()
                    .withOperationName("DeleteService").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteServiceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteServiceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes one or more task definitions.
     * </p>
     * <p>
     * You must deregister a task definition revision before you delete it. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DeregisterTaskDefinition.html"
     * >DeregisterTaskDefinition</a>.
     * </p>
     * <p>
     * When you delete a task definition revision, it is immediately transitions from the <code>INACTIVE</code> to
     * <code>DELETE_IN_PROGRESS</code>. Existing tasks and services that reference a <code>DELETE_IN_PROGRESS</code>
     * task definition revision continue to run without disruption. Existing services that reference a
     * <code>DELETE_IN_PROGRESS</code> task definition revision can still scale up or down by modifying the service's
     * desired count.
     * </p>
     * <p>
     * You can't use a <code>DELETE_IN_PROGRESS</code> task definition revision to run new tasks or create new services.
     * You also can't update an existing service to reference a <code>DELETE_IN_PROGRESS</code> task definition
     * revision.
     * </p>
     * <p>
     * A task definition revision will stay in <code>DELETE_IN_PROGRESS</code> status until all the associated tasks and
     * services have been terminated.
     * </p>
     * <p>
     * When you delete all <code>INACTIVE</code> task definition revisions, the task definition name is not displayed in
     * the console and not returned in the API. If a task definition revisions are in the
     * <code>DELETE_IN_PROGRESS</code> state, the task definition name is displayed in the console and returned in the
     * API. The task definition name is retained by Amazon ECS and the revision is incremented the next time you create
     * a task definition with that name.
     * </p>
     *
     * @param deleteTaskDefinitionsRequest
     * @return Result of the DeleteTaskDefinitions operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have authorization to perform the requested action.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.DeleteTaskDefinitions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/DeleteTaskDefinitions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteTaskDefinitionsResponse deleteTaskDefinitions(DeleteTaskDefinitionsRequest deleteTaskDefinitionsRequest)
            throws AccessDeniedException, ClientException, InvalidParameterException, ServerException, AwsServiceException,
            SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteTaskDefinitionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteTaskDefinitionsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTaskDefinitionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTaskDefinitionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTaskDefinitions");

            return clientHandler.execute(new ClientExecutionParams<DeleteTaskDefinitionsRequest, DeleteTaskDefinitionsResponse>()
                    .withOperationName("DeleteTaskDefinitions").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteTaskDefinitionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteTaskDefinitionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a specified task set within a service. This is used when a service uses the <code>EXTERNAL</code>
     * deployment controller type. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html">Amazon ECS deployment
     * types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     *
     * @param deleteTaskSetRequest
     * @return Result of the DeleteTaskSet operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws UnsupportedFeatureException
     *         The specified task isn't supported in this Region.
     * @throws AccessDeniedException
     *         You don't have authorization to perform the requested action.
     * @throws ServiceNotFoundException
     *         The specified service wasn't found. You can view your available services with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListServices.html">ListServices</a>.
     *         Amazon ECS services are cluster specific and Region specific.
     * @throws ServiceNotActiveException
     *         The specified service isn't active. You can't update a service that's inactive. If you have previously
     *         deleted a service, you can re-create it with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_CreateService.html"
     *         >CreateService</a>.
     * @throws TaskSetNotFoundException
     *         The specified task set wasn't found. You can view your available task sets with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTaskSets.html"
     *         >DescribeTaskSets</a>. Task sets are specific to each cluster, service and Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.DeleteTaskSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/DeleteTaskSet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteTaskSetResponse deleteTaskSet(DeleteTaskSetRequest deleteTaskSetRequest) throws ServerException,
            ClientException, InvalidParameterException, ClusterNotFoundException, UnsupportedFeatureException,
            AccessDeniedException, ServiceNotFoundException, ServiceNotActiveException, TaskSetNotFoundException,
            AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteTaskSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteTaskSetResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTaskSetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTaskSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTaskSet");

            return clientHandler.execute(new ClientExecutionParams<DeleteTaskSetRequest, DeleteTaskSetResponse>()
                    .withOperationName("DeleteTaskSet").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteTaskSetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteTaskSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deregisters an Amazon ECS container instance from the specified cluster. This instance is no longer available to
     * run tasks.
     * </p>
     * <p>
     * If you intend to use the container instance for some other purpose after deregistration, we recommend that you
     * stop all of the tasks running on the container instance before deregistration. That prevents any orphaned tasks
     * from consuming resources.
     * </p>
     * <p>
     * Deregistering a container instance removes the instance from a cluster, but it doesn't terminate the EC2
     * instance. If you are finished using the instance, be sure to terminate it in the Amazon EC2 console to stop
     * billing.
     * </p>
     * <note>
     * <p>
     * If you terminate a running container instance, Amazon ECS automatically deregisters the instance from your
     * cluster (stopped container instances or instances with disconnected agents aren't automatically deregistered when
     * terminated).
     * </p>
     * </note>
     *
     * @param deregisterContainerInstanceRequest
     * @return Result of the DeregisterContainerInstance operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.DeregisterContainerInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/DeregisterContainerInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeregisterContainerInstanceResponse deregisterContainerInstance(
            DeregisterContainerInstanceRequest deregisterContainerInstanceRequest) throws ServerException, ClientException,
            InvalidParameterException, ClusterNotFoundException, AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeregisterContainerInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeregisterContainerInstanceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deregisterContainerInstanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deregisterContainerInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterContainerInstance");

            return clientHandler
                    .execute(new ClientExecutionParams<DeregisterContainerInstanceRequest, DeregisterContainerInstanceResponse>()
                            .withOperationName("DeregisterContainerInstance").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deregisterContainerInstanceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeregisterContainerInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deregisters the specified task definition by family and revision. Upon deregistration, the task definition is
     * marked as <code>INACTIVE</code>. Existing tasks and services that reference an <code>INACTIVE</code> task
     * definition continue to run without disruption. Existing services that reference an <code>INACTIVE</code> task
     * definition can still scale up or down by modifying the service's desired count. If you want to delete a task
     * definition revision, you must first deregister the task definition revision.
     * </p>
     * <p>
     * You can't use an <code>INACTIVE</code> task definition to run new tasks or create new services, and you can't
     * update an existing service to reference an <code>INACTIVE</code> task definition. However, there may be up to a
     * 10-minute window following deregistration where these restrictions have not yet taken effect.
     * </p>
     * <note>
     * <p>
     * At this time, <code>INACTIVE</code> task definitions remain discoverable in your account indefinitely. However,
     * this behavior is subject to change in the future. We don't recommend that you rely on <code>INACTIVE</code> task
     * definitions persisting beyond the lifecycle of any associated tasks and services.
     * </p>
     * </note>
     * <p>
     * You must deregister a task definition revision before you delete it. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DeleteTaskDefinitions.html"
     * >DeleteTaskDefinitions</a>.
     * </p>
     *
     * @param deregisterTaskDefinitionRequest
     * @return Result of the DeregisterTaskDefinition operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.DeregisterTaskDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/DeregisterTaskDefinition" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeregisterTaskDefinitionResponse deregisterTaskDefinition(
            DeregisterTaskDefinitionRequest deregisterTaskDefinitionRequest) throws ServerException, ClientException,
            InvalidParameterException, AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeregisterTaskDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeregisterTaskDefinitionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deregisterTaskDefinitionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deregisterTaskDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterTaskDefinition");

            return clientHandler
                    .execute(new ClientExecutionParams<DeregisterTaskDefinitionRequest, DeregisterTaskDefinitionResponse>()
                            .withOperationName("DeregisterTaskDefinition").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deregisterTaskDefinitionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeregisterTaskDefinitionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes one or more of your capacity providers.
     * </p>
     *
     * @param describeCapacityProvidersRequest
     * @return Result of the DescribeCapacityProviders operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.DescribeCapacityProviders
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/DescribeCapacityProviders" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeCapacityProvidersResponse describeCapacityProviders(
            DescribeCapacityProvidersRequest describeCapacityProvidersRequest) throws ServerException, ClientException,
            InvalidParameterException, AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeCapacityProvidersResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeCapacityProvidersResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeCapacityProvidersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeCapacityProvidersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeCapacityProviders");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeCapacityProvidersRequest, DescribeCapacityProvidersResponse>()
                            .withOperationName("DescribeCapacityProviders").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeCapacityProvidersRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeCapacityProvidersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes one or more of your clusters.
     * </p>
     * <p>
     * For CLI examples, see <a
     * href="https://github.com/aws/aws-cli/blob/develop/awscli/examples/ecs/describe-clusters.rst"
     * >describe-clusters.rst</a> on GitHub.
     * </p>
     *
     * @param describeClustersRequest
     * @return Result of the DescribeClusters operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.DescribeClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/DescribeClusters" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeClustersResponse describeClusters(DescribeClustersRequest describeClustersRequest) throws ServerException,
            ClientException, InvalidParameterException, AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeClustersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeClustersResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeClustersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeClustersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeClusters");

            return clientHandler.execute(new ClientExecutionParams<DescribeClustersRequest, DescribeClustersResponse>()
                    .withOperationName("DescribeClusters").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeClustersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeClustersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes one or more container instances. Returns metadata about each container instance requested.
     * </p>
     *
     * @param describeContainerInstancesRequest
     * @return Result of the DescribeContainerInstances operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.DescribeContainerInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/DescribeContainerInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeContainerInstancesResponse describeContainerInstances(
            DescribeContainerInstancesRequest describeContainerInstancesRequest) throws ServerException, ClientException,
            InvalidParameterException, ClusterNotFoundException, AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeContainerInstancesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeContainerInstancesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeContainerInstancesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeContainerInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeContainerInstances");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeContainerInstancesRequest, DescribeContainerInstancesResponse>()
                            .withOperationName("DescribeContainerInstances").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeContainerInstancesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeContainerInstancesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes one or more of your service deployments.
     * </p>
     * <p>
     * A service deployment happens when you release a software update for the service. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-deployment.html">View service history
     * using Amazon ECS service deployments</a>.
     * </p>
     *
     * @param describeServiceDeploymentsRequest
     * @return Result of the DescribeServiceDeployments operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have authorization to perform the requested action.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ServiceNotFoundException
     *         The specified service wasn't found. You can view your available services with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListServices.html">ListServices</a>.
     *         Amazon ECS services are cluster specific and Region specific.
     * @throws UnsupportedFeatureException
     *         The specified task isn't supported in this Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.DescribeServiceDeployments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/DescribeServiceDeployments"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeServiceDeploymentsResponse describeServiceDeployments(
            DescribeServiceDeploymentsRequest describeServiceDeploymentsRequest) throws AccessDeniedException, ClientException,
            ClusterNotFoundException, InvalidParameterException, ServerException, ServiceNotFoundException,
            UnsupportedFeatureException, AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeServiceDeploymentsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeServiceDeploymentsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeServiceDeploymentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeServiceDeploymentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeServiceDeployments");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeServiceDeploymentsRequest, DescribeServiceDeploymentsResponse>()
                            .withOperationName("DescribeServiceDeployments").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeServiceDeploymentsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeServiceDeploymentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes one or more service revisions.
     * </p>
     * <p>
     * A service revision is a version of the service that includes the values for the Amazon ECS resources (for
     * example, task definition) and the environment resources (for example, load balancers, subnets, and security
     * groups). For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-revision.html">Amazon ECS service
     * revisions</a>.
     * </p>
     * <p>
     * You can't describe a service revision that was created before October 25, 2024.
     * </p>
     *
     * @param describeServiceRevisionsRequest
     * @return Result of the DescribeServiceRevisions operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have authorization to perform the requested action.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ServiceNotFoundException
     *         The specified service wasn't found. You can view your available services with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListServices.html">ListServices</a>.
     *         Amazon ECS services are cluster specific and Region specific.
     * @throws UnsupportedFeatureException
     *         The specified task isn't supported in this Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.DescribeServiceRevisions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/DescribeServiceRevisions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeServiceRevisionsResponse describeServiceRevisions(
            DescribeServiceRevisionsRequest describeServiceRevisionsRequest) throws AccessDeniedException, ClientException,
            ClusterNotFoundException, InvalidParameterException, ServerException, ServiceNotFoundException,
            UnsupportedFeatureException, AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeServiceRevisionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeServiceRevisionsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeServiceRevisionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeServiceRevisionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeServiceRevisions");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeServiceRevisionsRequest, DescribeServiceRevisionsResponse>()
                            .withOperationName("DescribeServiceRevisions").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeServiceRevisionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeServiceRevisionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the specified services running in your cluster.
     * </p>
     *
     * @param describeServicesRequest
     * @return Result of the DescribeServices operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.DescribeServices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/DescribeServices" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeServicesResponse describeServices(DescribeServicesRequest describeServicesRequest) throws ServerException,
            ClientException, InvalidParameterException, ClusterNotFoundException, AwsServiceException, SdkClientException,
            EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeServicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeServicesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeServicesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeServicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeServices");

            return clientHandler.execute(new ClientExecutionParams<DescribeServicesRequest, DescribeServicesResponse>()
                    .withOperationName("DescribeServices").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeServicesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeServicesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes a task definition. You can specify a <code>family</code> and <code>revision</code> to find information
     * about a specific task definition, or you can simply specify the family to find the latest <code>ACTIVE</code>
     * revision in that family.
     * </p>
     * <note>
     * <p>
     * You can only describe <code>INACTIVE</code> task definitions while an active task or service references them.
     * </p>
     * </note>
     *
     * @param describeTaskDefinitionRequest
     * @return Result of the DescribeTaskDefinition operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.DescribeTaskDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/DescribeTaskDefinition" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeTaskDefinitionResponse describeTaskDefinition(DescribeTaskDefinitionRequest describeTaskDefinitionRequest)
            throws ServerException, ClientException, InvalidParameterException, AwsServiceException, SdkClientException,
            EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeTaskDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeTaskDefinitionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeTaskDefinitionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTaskDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTaskDefinition");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeTaskDefinitionRequest, DescribeTaskDefinitionResponse>()
                            .withOperationName("DescribeTaskDefinition").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeTaskDefinitionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeTaskDefinitionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the task sets in the specified cluster and service. This is used when a service uses the
     * <code>EXTERNAL</code> deployment controller type. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html">Amazon ECS Deployment
     * Types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     *
     * @param describeTaskSetsRequest
     * @return Result of the DescribeTaskSets operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws UnsupportedFeatureException
     *         The specified task isn't supported in this Region.
     * @throws AccessDeniedException
     *         You don't have authorization to perform the requested action.
     * @throws ServiceNotFoundException
     *         The specified service wasn't found. You can view your available services with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListServices.html">ListServices</a>.
     *         Amazon ECS services are cluster specific and Region specific.
     * @throws ServiceNotActiveException
     *         The specified service isn't active. You can't update a service that's inactive. If you have previously
     *         deleted a service, you can re-create it with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_CreateService.html"
     *         >CreateService</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.DescribeTaskSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/DescribeTaskSets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeTaskSetsResponse describeTaskSets(DescribeTaskSetsRequest describeTaskSetsRequest) throws ServerException,
            ClientException, InvalidParameterException, ClusterNotFoundException, UnsupportedFeatureException,
            AccessDeniedException, ServiceNotFoundException, ServiceNotActiveException, AwsServiceException, SdkClientException,
            EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeTaskSetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeTaskSetsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeTaskSetsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTaskSetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTaskSets");

            return clientHandler.execute(new ClientExecutionParams<DescribeTaskSetsRequest, DescribeTaskSetsResponse>()
                    .withOperationName("DescribeTaskSets").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeTaskSetsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeTaskSetsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes a specified task or tasks.
     * </p>
     * <p>
     * Currently, stopped tasks appear in the returned results for at least one hour.
     * </p>
     * <p>
     * If you have tasks with tags, and then delete the cluster, the tagged tasks are returned in the response. If you
     * create a new cluster with the same name as the deleted cluster, the tagged tasks are not included in the
     * response.
     * </p>
     *
     * @param describeTasksRequest
     * @return Result of the DescribeTasks operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.DescribeTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/DescribeTasks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeTasksResponse describeTasks(DescribeTasksRequest describeTasksRequest) throws ServerException,
            ClientException, InvalidParameterException, ClusterNotFoundException, AwsServiceException, SdkClientException,
            EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeTasksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeTasksResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeTasksRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTasksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTasks");

            return clientHandler.execute(new ClientExecutionParams<DescribeTasksRequest, DescribeTasksResponse>()
                    .withOperationName("DescribeTasks").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeTasksRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeTasksRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <note>
     * <p>
     * This action is only used by the Amazon ECS agent, and it is not intended for use outside of the agent.
     * </p>
     * </note>
     * <p>
     * Returns an endpoint for the Amazon ECS agent to poll for updates.
     * </p>
     *
     * @param discoverPollEndpointRequest
     * @return Result of the DiscoverPollEndpoint operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.DiscoverPollEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/DiscoverPollEndpoint" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DiscoverPollEndpointResponse discoverPollEndpoint(DiscoverPollEndpointRequest discoverPollEndpointRequest)
            throws ServerException, ClientException, AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DiscoverPollEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DiscoverPollEndpointResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(discoverPollEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, discoverPollEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DiscoverPollEndpoint");

            return clientHandler.execute(new ClientExecutionParams<DiscoverPollEndpointRequest, DiscoverPollEndpointResponse>()
                    .withOperationName("DiscoverPollEndpoint").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(discoverPollEndpointRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DiscoverPollEndpointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Runs a command remotely on a container within a task.
     * </p>
     * <p>
     * If you use a condition key in your IAM policy to refine the conditions for the policy statement, for example
     * limit the actions to a specific cluster, you receive an <code>AccessDeniedException</code> when there is a
     * mismatch between the condition key value and the corresponding parameter value.
     * </p>
     * <p>
     * For information about required permissions and considerations, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-exec.html">Using Amazon ECS Exec for
     * debugging</a> in the <i>Amazon ECS Developer Guide</i>.
     * </p>
     *
     * @param executeCommandRequest
     * @return Result of the ExecuteCommand operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws AccessDeniedException
     *         You don't have authorization to perform the requested action.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws TargetNotConnectedException
     *         The execute command cannot run. This error can be caused by any of the following configuration issues:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Incorrect IAM permissions
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The SSM agent is not installed or is not running
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is an interface Amazon VPC endpoint for Amazon ECS, but there is not one for Systems Manager
     *         Session Manager
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For information about how to troubleshoot the issues, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-exec.html">Troubleshooting issues
     *         with ECS Exec</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.ExecuteCommand
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/ExecuteCommand" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ExecuteCommandResponse executeCommand(ExecuteCommandRequest executeCommandRequest) throws ServerException,
            ClientException, InvalidParameterException, AccessDeniedException, ClusterNotFoundException,
            TargetNotConnectedException, AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ExecuteCommandResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ExecuteCommandResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(executeCommandRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, executeCommandRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExecuteCommand");

            return clientHandler.execute(new ClientExecutionParams<ExecuteCommandRequest, ExecuteCommandResponse>()
                    .withOperationName("ExecuteCommand").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(executeCommandRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ExecuteCommandRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the protection status of tasks in an Amazon ECS service.
     * </p>
     *
     * @param getTaskProtectionRequest
     * @return Result of the GetTaskProtection operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have authorization to perform the requested action.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ResourceNotFoundException
     *         The specified resource wasn't found.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws UnsupportedFeatureException
     *         The specified task isn't supported in this Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.GetTaskProtection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/GetTaskProtection" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetTaskProtectionResponse getTaskProtection(GetTaskProtectionRequest getTaskProtectionRequest)
            throws AccessDeniedException, ClientException, ClusterNotFoundException, InvalidParameterException,
            ResourceNotFoundException, ServerException, UnsupportedFeatureException, AwsServiceException, SdkClientException,
            EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTaskProtectionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTaskProtectionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTaskProtectionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTaskProtectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTaskProtection");

            return clientHandler.execute(new ClientExecutionParams<GetTaskProtectionRequest, GetTaskProtectionResponse>()
                    .withOperationName("GetTaskProtection").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getTaskProtectionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetTaskProtectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the account settings for a specified principal.
     * </p>
     *
     * @param listAccountSettingsRequest
     * @return Result of the ListAccountSettings operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.ListAccountSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/ListAccountSettings" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListAccountSettingsResponse listAccountSettings(ListAccountSettingsRequest listAccountSettingsRequest)
            throws ServerException, ClientException, InvalidParameterException, AwsServiceException, SdkClientException,
            EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAccountSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListAccountSettingsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAccountSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAccountSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAccountSettings");

            return clientHandler.execute(new ClientExecutionParams<ListAccountSettingsRequest, ListAccountSettingsResponse>()
                    .withOperationName("ListAccountSettings").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listAccountSettingsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAccountSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the attributes for Amazon ECS resources within a specified target type and cluster. When you specify a
     * target type and cluster, <code>ListAttributes</code> returns a list of attribute objects, one for each attribute
     * on each resource. You can filter the list of results to a single attribute name to only return results that have
     * that name. You can also filter the results by attribute name and value. You can do this, for example, to see
     * which container instances in a cluster are running a Linux AMI (<code>ecs.os-type=linux</code>).
     * </p>
     *
     * @param listAttributesRequest
     * @return Result of the ListAttributes operation returned by the service.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.ListAttributes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/ListAttributes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListAttributesResponse listAttributes(ListAttributesRequest listAttributesRequest) throws ClusterNotFoundException,
            InvalidParameterException, AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAttributesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAttributesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAttributesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAttributesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAttributes");

            return clientHandler.execute(new ClientExecutionParams<ListAttributesRequest, ListAttributesResponse>()
                    .withOperationName("ListAttributes").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listAttributesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAttributesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of existing clusters.
     * </p>
     *
     * @param listClustersRequest
     * @return Result of the ListClusters operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.ListClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/ListClusters" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListClustersResponse listClusters(ListClustersRequest listClustersRequest) throws ServerException, ClientException,
            InvalidParameterException, AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListClustersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListClustersResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listClustersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listClustersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListClusters");

            return clientHandler.execute(new ClientExecutionParams<ListClustersRequest, ListClustersResponse>()
                    .withOperationName("ListClusters").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listClustersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListClustersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of container instances in a specified cluster. You can filter the results of a
     * <code>ListContainerInstances</code> operation with cluster query language statements inside the
     * <code>filter</code> parameter. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-query-language.html">Cluster Query
     * Language</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     *
     * @param listContainerInstancesRequest
     * @return Result of the ListContainerInstances operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.ListContainerInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/ListContainerInstances" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListContainerInstancesResponse listContainerInstances(ListContainerInstancesRequest listContainerInstancesRequest)
            throws ServerException, ClientException, InvalidParameterException, ClusterNotFoundException, AwsServiceException,
            SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListContainerInstancesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListContainerInstancesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listContainerInstancesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listContainerInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListContainerInstances");

            return clientHandler
                    .execute(new ClientExecutionParams<ListContainerInstancesRequest, ListContainerInstancesResponse>()
                            .withOperationName("ListContainerInstances").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listContainerInstancesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListContainerInstancesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation lists all the service deployments that meet the specified filter criteria.
     * </p>
     * <p>
     * A service deployment happens when you release a software update for the service. You route traffic from the
     * running service revisions to the new service revison and control the number of running tasks.
     * </p>
     * <p>
     * This API returns the values that you use for the request parameters in <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeServiceRevisions.html"
     * >DescribeServiceRevisions</a>.
     * </p>
     *
     * @param listServiceDeploymentsRequest
     * @return Result of the ListServiceDeployments operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have authorization to perform the requested action.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ServiceNotFoundException
     *         The specified service wasn't found. You can view your available services with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListServices.html">ListServices</a>.
     *         Amazon ECS services are cluster specific and Region specific.
     * @throws UnsupportedFeatureException
     *         The specified task isn't supported in this Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.ListServiceDeployments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/ListServiceDeployments" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListServiceDeploymentsResponse listServiceDeployments(ListServiceDeploymentsRequest listServiceDeploymentsRequest)
            throws AccessDeniedException, ClientException, InvalidParameterException, ServerException, ServiceNotFoundException,
            UnsupportedFeatureException, AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListServiceDeploymentsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListServiceDeploymentsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listServiceDeploymentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServiceDeploymentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServiceDeployments");

            return clientHandler
                    .execute(new ClientExecutionParams<ListServiceDeploymentsRequest, ListServiceDeploymentsResponse>()
                            .withOperationName("ListServiceDeployments").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listServiceDeploymentsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListServiceDeploymentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of services. You can filter the results by cluster, launch type, and scheduling strategy.
     * </p>
     *
     * @param listServicesRequest
     * @return Result of the ListServices operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.ListServices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/ListServices" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListServicesResponse listServices(ListServicesRequest listServicesRequest) throws ServerException, ClientException,
            InvalidParameterException, ClusterNotFoundException, AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListServicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListServicesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listServicesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServices");

            return clientHandler.execute(new ClientExecutionParams<ListServicesRequest, ListServicesResponse>()
                    .withOperationName("ListServices").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listServicesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListServicesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation lists all of the services that are associated with a Cloud Map namespace. This list might include
     * services in different clusters. In contrast, <code>ListServices</code> can only list services in one cluster at a
     * time. If you need to filter the list of services in a single cluster by various parameters, use
     * <code>ListServices</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-connect.html">Service Connect</a> in
     * the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     *
     * @param listServicesByNamespaceRequest
     * @return Result of the ListServicesByNamespace operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws NamespaceNotFoundException
     *         The specified namespace wasn't found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.ListServicesByNamespace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/ListServicesByNamespace" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListServicesByNamespaceResponse listServicesByNamespace(ListServicesByNamespaceRequest listServicesByNamespaceRequest)
            throws ServerException, ClientException, InvalidParameterException, NamespaceNotFoundException, AwsServiceException,
            SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListServicesByNamespaceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListServicesByNamespaceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listServicesByNamespaceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServicesByNamespaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServicesByNamespace");

            return clientHandler
                    .execute(new ClientExecutionParams<ListServicesByNamespaceRequest, ListServicesByNamespaceResponse>()
                            .withOperationName("ListServicesByNamespace").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listServicesByNamespaceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListServicesByNamespaceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List the tags for an Amazon ECS resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ServerException, ClientException, ClusterNotFoundException, InvalidParameterException, AwsServiceException,
            SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of task definition families that are registered to your account. This list includes task
     * definition families that no longer have any <code>ACTIVE</code> task definition revisions.
     * </p>
     * <p>
     * You can filter out task definition families that don't contain any <code>ACTIVE</code> task definition revisions
     * by setting the <code>status</code> parameter to <code>ACTIVE</code>. You can also filter the results with the
     * <code>familyPrefix</code> parameter.
     * </p>
     *
     * @param listTaskDefinitionFamiliesRequest
     * @return Result of the ListTaskDefinitionFamilies operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.ListTaskDefinitionFamilies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/ListTaskDefinitionFamilies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTaskDefinitionFamiliesResponse listTaskDefinitionFamilies(
            ListTaskDefinitionFamiliesRequest listTaskDefinitionFamiliesRequest) throws ServerException, ClientException,
            InvalidParameterException, AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTaskDefinitionFamiliesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTaskDefinitionFamiliesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTaskDefinitionFamiliesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTaskDefinitionFamiliesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTaskDefinitionFamilies");

            return clientHandler
                    .execute(new ClientExecutionParams<ListTaskDefinitionFamiliesRequest, ListTaskDefinitionFamiliesResponse>()
                            .withOperationName("ListTaskDefinitionFamilies").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listTaskDefinitionFamiliesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListTaskDefinitionFamiliesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of task definitions that are registered to your account. You can filter the results by family name
     * with the <code>familyPrefix</code> parameter or by status with the <code>status</code> parameter.
     * </p>
     *
     * @param listTaskDefinitionsRequest
     * @return Result of the ListTaskDefinitions operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.ListTaskDefinitions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/ListTaskDefinitions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTaskDefinitionsResponse listTaskDefinitions(ListTaskDefinitionsRequest listTaskDefinitionsRequest)
            throws ServerException, ClientException, InvalidParameterException, AwsServiceException, SdkClientException,
            EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTaskDefinitionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTaskDefinitionsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTaskDefinitionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTaskDefinitionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTaskDefinitions");

            return clientHandler.execute(new ClientExecutionParams<ListTaskDefinitionsRequest, ListTaskDefinitionsResponse>()
                    .withOperationName("ListTaskDefinitions").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTaskDefinitionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTaskDefinitionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of tasks. You can filter the results by cluster, task definition family, container instance,
     * launch type, what IAM principal started the task, or by the desired status of the task.
     * </p>
     * <p>
     * Recently stopped tasks might appear in the returned results.
     * </p>
     *
     * @param listTasksRequest
     * @return Result of the ListTasks operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws ServiceNotFoundException
     *         The specified service wasn't found. You can view your available services with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListServices.html">ListServices</a>.
     *         Amazon ECS services are cluster specific and Region specific.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.ListTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/ListTasks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTasksResponse listTasks(ListTasksRequest listTasksRequest) throws ServerException, ClientException,
            InvalidParameterException, ClusterNotFoundException, ServiceNotFoundException, AwsServiceException,
            SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTasksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTasksResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTasksRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTasksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTasks");

            return clientHandler.execute(new ClientExecutionParams<ListTasksRequest, ListTasksResponse>()
                    .withOperationName("ListTasks").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listTasksRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTasksRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Modifies an account setting. Account settings are set on a per-Region basis.
     * </p>
     * <p>
     * If you change the root user account setting, the default settings are reset for users and roles that do not have
     * specified individual account settings. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-account-settings.html">Account Settings</a>
     * in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     *
     * @param putAccountSettingRequest
     * @return Result of the PutAccountSetting operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.PutAccountSetting
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/PutAccountSetting" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutAccountSettingResponse putAccountSetting(PutAccountSettingRequest putAccountSettingRequest) throws ServerException,
            ClientException, InvalidParameterException, AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutAccountSettingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutAccountSettingResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putAccountSettingRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putAccountSettingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAccountSetting");

            return clientHandler.execute(new ClientExecutionParams<PutAccountSettingRequest, PutAccountSettingResponse>()
                    .withOperationName("PutAccountSetting").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putAccountSettingRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutAccountSettingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Modifies an account setting for all users on an account for whom no individual account setting has been
     * specified. Account settings are set on a per-Region basis.
     * </p>
     *
     * @param putAccountSettingDefaultRequest
     * @return Result of the PutAccountSettingDefault operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.PutAccountSettingDefault
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/PutAccountSettingDefault" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutAccountSettingDefaultResponse putAccountSettingDefault(
            PutAccountSettingDefaultRequest putAccountSettingDefaultRequest) throws ServerException, ClientException,
            InvalidParameterException, AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutAccountSettingDefaultResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutAccountSettingDefaultResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putAccountSettingDefaultRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putAccountSettingDefaultRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAccountSettingDefault");

            return clientHandler
                    .execute(new ClientExecutionParams<PutAccountSettingDefaultRequest, PutAccountSettingDefaultResponse>()
                            .withOperationName("PutAccountSettingDefault").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(putAccountSettingDefaultRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutAccountSettingDefaultRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Create or update an attribute on an Amazon ECS resource. If the attribute doesn't exist, it's created. If the
     * attribute exists, its value is replaced with the specified value. To delete an attribute, use <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DeleteAttributes.html">DeleteAttributes</a>.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html#attributes"
     * >Attributes</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     *
     * @param putAttributesRequest
     * @return Result of the PutAttributes operation returned by the service.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws TargetNotFoundException
     *         The specified target wasn't found. You can view your available container instances with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListContainerInstances.html"
     *         >ListContainerInstances</a>. Amazon ECS container instances are cluster-specific and Region-specific.
     * @throws AttributeLimitExceededException
     *         You can apply up to 10 custom attributes for each resource. You can view the attributes of a resource
     *         with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListAttributes.html">ListAttributes
     *         </a>. You can remove existing attributes on a resource with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DeleteAttributes.html"
     *         >DeleteAttributes</a>.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.PutAttributes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/PutAttributes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutAttributesResponse putAttributes(PutAttributesRequest putAttributesRequest) throws ClusterNotFoundException,
            TargetNotFoundException, AttributeLimitExceededException, InvalidParameterException, AwsServiceException,
            SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutAttributesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutAttributesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putAttributesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putAttributesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAttributes");

            return clientHandler.execute(new ClientExecutionParams<PutAttributesRequest, PutAttributesResponse>()
                    .withOperationName("PutAttributes").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putAttributesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutAttributesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Modifies the available capacity providers and the default capacity provider strategy for a cluster.
     * </p>
     * <p>
     * You must specify both the available capacity providers and a default capacity provider strategy for the cluster.
     * If the specified cluster has existing capacity providers associated with it, you must specify all existing
     * capacity providers in addition to any new ones you want to add. Any existing capacity providers that are
     * associated with a cluster that are omitted from a <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutClusterCapacityProviders.html"
     * >PutClusterCapacityProviders</a> API call will be disassociated with the cluster. You can only disassociate an
     * existing capacity provider from a cluster if it's not being used by any existing tasks.
     * </p>
     * <p>
     * When creating a service or running a task on a cluster, if no capacity provider or launch type is specified, then
     * the cluster's default capacity provider strategy is used. We recommend that you define a default capacity
     * provider strategy for your cluster. However, you must specify an empty array (<code>[]</code>) to bypass defining
     * a default strategy.
     * </p>
     *
     * @param putClusterCapacityProvidersRequest
     * @return Result of the PutClusterCapacityProviders operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws ResourceInUseException
     *         The specified resource is in-use and can't be removed.
     * @throws UpdateInProgressException
     *         There's already a current Amazon ECS container agent update in progress on the container instance that's
     *         specified. If the container agent becomes disconnected while it's in a transitional stage, such as
     *         <code>PENDING</code> or <code>STAGING</code>, the update process can get stuck in that state. However,
     *         when the agent reconnects, it resumes where it stopped previously.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.PutClusterCapacityProviders
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/PutClusterCapacityProviders"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutClusterCapacityProvidersResponse putClusterCapacityProviders(
            PutClusterCapacityProvidersRequest putClusterCapacityProvidersRequest) throws ServerException, ClientException,
            InvalidParameterException, ClusterNotFoundException, ResourceInUseException, UpdateInProgressException,
            AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutClusterCapacityProvidersResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutClusterCapacityProvidersResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putClusterCapacityProvidersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putClusterCapacityProvidersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutClusterCapacityProviders");

            return clientHandler
                    .execute(new ClientExecutionParams<PutClusterCapacityProvidersRequest, PutClusterCapacityProvidersResponse>()
                            .withOperationName("PutClusterCapacityProviders").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(putClusterCapacityProvidersRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutClusterCapacityProvidersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <note>
     * <p>
     * This action is only used by the Amazon ECS agent, and it is not intended for use outside of the agent.
     * </p>
     * </note>
     * <p>
     * Registers an EC2 instance into the specified cluster. This instance becomes available to place containers on.
     * </p>
     *
     * @param registerContainerInstanceRequest
     * @return Result of the RegisterContainerInstance operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.RegisterContainerInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/RegisterContainerInstance" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public RegisterContainerInstanceResponse registerContainerInstance(
            RegisterContainerInstanceRequest registerContainerInstanceRequest) throws ServerException, ClientException,
            InvalidParameterException, AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RegisterContainerInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RegisterContainerInstanceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(registerContainerInstanceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerContainerInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterContainerInstance");

            return clientHandler
                    .execute(new ClientExecutionParams<RegisterContainerInstanceRequest, RegisterContainerInstanceResponse>()
                            .withOperationName("RegisterContainerInstance").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(registerContainerInstanceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RegisterContainerInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Registers a new task definition from the supplied <code>family</code> and <code>containerDefinitions</code>.
     * Optionally, you can add data volumes to your containers with the <code>volumes</code> parameter. For more
     * information about task definition parameters and defaults, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_defintions.html">Amazon ECS Task
     * Definitions</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * You can specify a role for your task with the <code>taskRoleArn</code> parameter. When you specify a role for a
     * task, its containers can then use the latest versions of the CLI or SDKs to make API requests to the Amazon Web
     * Services services that are specified in the policy that's associated with the role. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-iam-roles.html">IAM Roles for Tasks</a> in
     * the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * You can specify a Docker networking mode for the containers in your task definition with the
     * <code>networkMode</code> parameter. If you specify the <code>awsvpc</code> network mode, the task is allocated an
     * elastic network interface, and you must specify a <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_NetworkConfiguration.html"
     * >NetworkConfiguration</a> when you create a service or run a task with the task definition. For more information,
     * see <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html">Task
     * Networking</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     *
     * @param registerTaskDefinitionRequest
     * @return Result of the RegisterTaskDefinition operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.RegisterTaskDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/RegisterTaskDefinition" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public RegisterTaskDefinitionResponse registerTaskDefinition(RegisterTaskDefinitionRequest registerTaskDefinitionRequest)
            throws ServerException, ClientException, InvalidParameterException, AwsServiceException, SdkClientException,
            EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RegisterTaskDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RegisterTaskDefinitionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(registerTaskDefinitionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerTaskDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterTaskDefinition");

            return clientHandler
                    .execute(new ClientExecutionParams<RegisterTaskDefinitionRequest, RegisterTaskDefinitionResponse>()
                            .withOperationName("RegisterTaskDefinition").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(registerTaskDefinitionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RegisterTaskDefinitionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a new task using the specified task definition.
     * </p>
     * <note>
     * <p>
     * On March 21, 2024, a change was made to resolve the task definition revision before authorization. When a task
     * definition revision is not specified, authorization will occur using the latest revision of a task definition.
     * </p>
     * </note> <note>
     * <p>
     * Amazon Elastic Inference (EI) is no longer available to customers.
     * </p>
     * </note>
     * <p>
     * You can allow Amazon ECS to place tasks for you, or you can customize how Amazon ECS places tasks using placement
     * constraints and placement strategies. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/scheduling_tasks.html">Scheduling Tasks</a> in
     * the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * Alternatively, you can use <code>StartTask</code> to use your own scheduler or place tasks manually on specific
     * container instances.
     * </p>
     * <p>
     * You can attach Amazon EBS volumes to Amazon ECS tasks by configuring the volume when creating or updating a
     * service. For more infomation, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ebs-volumes.html#ebs-volume-types">Amazon EBS
     * volumes</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * The Amazon ECS API follows an eventual consistency model. This is because of the distributed nature of the system
     * supporting the API. This means that the result of an API command you run that affects your Amazon ECS resources
     * might not be immediately visible to all subsequent commands you run. Keep this in mind when you carry out an API
     * command that immediately follows a previous API command.
     * </p>
     * <p>
     * To manage eventual consistency, you can do the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Confirm the state of the resource before you run a command to modify it. Run the DescribeTasks command using an
     * exponential backoff algorithm to ensure that you allow enough time for the previous command to propagate through
     * the system. To do this, run the DescribeTasks command repeatedly, starting with a couple of seconds of wait time
     * and increasing gradually up to five minutes of wait time.
     * </p>
     * </li>
     * <li>
     * <p>
     * Add wait time between subsequent commands, even if the DescribeTasks command returns an accurate response. Apply
     * an exponential backoff algorithm starting with a couple of seconds of wait time, and increase gradually up to
     * about five minutes of wait time.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you get a <code>ConflictException</code> error, the <code>RunTask</code> request could not be processed due to
     * conflicts. The provided <code>clientToken</code> is already in use with a different <code>RunTask</code> request.
     * The <code>resourceIds</code> are the existing task ARNs which are already associated with the
     * <code>clientToken</code>.
     * </p>
     * <p>
     * To fix this issue:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Run <code>RunTask</code> with a unique <code>clientToken</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Run <code>RunTask</code> with the <code>clientToken</code> and the original set of parameters
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you get a <code>ClientException</code>error, the <code>RunTask</code> could not be processed because you use
     * managed scaling and there is a capacity error because the quota of tasks in the <code>PROVISIONING</code> per
     * cluster has been reached. For information about the service quotas, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-quotas.html">Amazon ECS service
     * quotas</a>.
     * </p>
     *
     * @param runTaskRequest
     * @return Result of the RunTask operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws UnsupportedFeatureException
     *         The specified task isn't supported in this Region.
     * @throws PlatformUnknownException
     *         The specified platform version doesn't exist.
     * @throws PlatformTaskDefinitionIncompatibilityException
     *         The specified platform version doesn't satisfy the required capabilities of the task definition.
     * @throws AccessDeniedException
     *         You don't have authorization to perform the requested action.
     * @throws BlockedException
     *         Your Amazon Web Services account was blocked. For more information, contact <a
     *         href="http://aws.amazon.com/contact-us/"> Amazon Web ServicesSupport</a>.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.RunTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/RunTask" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public RunTaskResponse runTask(RunTaskRequest runTaskRequest) throws ServerException, ClientException,
            InvalidParameterException, ClusterNotFoundException, UnsupportedFeatureException, PlatformUnknownException,
            PlatformTaskDefinitionIncompatibilityException, AccessDeniedException, BlockedException, ConflictException,
            AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RunTaskResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RunTaskResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(runTaskRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, runTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RunTask");

            return clientHandler.execute(new ClientExecutionParams<RunTaskRequest, RunTaskResponse>()
                    .withOperationName("RunTask").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(runTaskRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RunTaskRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a new task from the specified task definition on the specified container instance or instances.
     * </p>
     * <note>
     * <p>
     * On March 21, 2024, a change was made to resolve the task definition revision before authorization. When a task
     * definition revision is not specified, authorization will occur using the latest revision of a task definition.
     * </p>
     * </note> <note>
     * <p>
     * Amazon Elastic Inference (EI) is no longer available to customers.
     * </p>
     * </note>
     * <p>
     * Alternatively, you can use<code>RunTask</code> to place tasks for you. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/scheduling_tasks.html">Scheduling Tasks</a> in
     * the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * You can attach Amazon EBS volumes to Amazon ECS tasks by configuring the volume when creating or updating a
     * service. For more infomation, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ebs-volumes.html#ebs-volume-types">Amazon EBS
     * volumes</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     *
     * @param startTaskRequest
     * @return Result of the StartTask operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws UnsupportedFeatureException
     *         The specified task isn't supported in this Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.StartTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/StartTask" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartTaskResponse startTask(StartTaskRequest startTaskRequest) throws ServerException, ClientException,
            InvalidParameterException, ClusterNotFoundException, UnsupportedFeatureException, AwsServiceException,
            SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartTaskResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartTaskResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startTaskRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartTask");

            return clientHandler.execute(new ClientExecutionParams<StartTaskRequest, StartTaskResponse>()
                    .withOperationName("StartTask").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(startTaskRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartTaskRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops an ongoing service deployment.
     * </p>
     * <p>
     * The following stop types are avaiable:
     * </p>
     * <ul>
     * <li>
     * <p>
     * ROLLBACK - This option rolls back the service deployment to the previous service revision.
     * </p>
     * <p>
     * You can use this option even if you didn't configure the service deployment for the rollback option.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/stop-service-deployment.html">Stopping Amazon
     * ECS service deployments</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     *
     * @param stopServiceDeploymentRequest
     * @return Result of the StopServiceDeployment operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have authorization to perform the requested action.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ServiceDeploymentNotFoundException
     *         The service deploy ARN that you specified in the <code>StopServiceDeployment</code> doesn't exist. You
     *         can use <code>ListServiceDeployments</code> to retrieve the service deployment ARNs.
     * @throws UnsupportedFeatureException
     *         The specified task isn't supported in this Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.StopServiceDeployment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/StopServiceDeployment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopServiceDeploymentResponse stopServiceDeployment(StopServiceDeploymentRequest stopServiceDeploymentRequest)
            throws AccessDeniedException, ClientException, ConflictException, InvalidParameterException, ServerException,
            ServiceDeploymentNotFoundException, UnsupportedFeatureException, AwsServiceException, SdkClientException,
            EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopServiceDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StopServiceDeploymentResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopServiceDeploymentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopServiceDeploymentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopServiceDeployment");

            return clientHandler.execute(new ClientExecutionParams<StopServiceDeploymentRequest, StopServiceDeploymentResponse>()
                    .withOperationName("StopServiceDeployment").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(stopServiceDeploymentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopServiceDeploymentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops a running task. Any tags associated with the task will be deleted.
     * </p>
     * <p>
     * When you call <code>StopTask</code> on a task, the equivalent of <code>docker stop</code> is issued to the
     * containers running in the task. This results in a <code>SIGTERM</code> value and a default 30-second timeout,
     * after which the <code>SIGKILL</code> value is sent and the containers are forcibly stopped. If the container
     * handles the <code>SIGTERM</code> value gracefully and exits within 30 seconds from receiving it, no
     * <code>SIGKILL</code> value is sent.
     * </p>
     * <p>
     * For Windows containers, POSIX signals do not work and runtime stops the container by sending a
     * <code>CTRL_SHUTDOWN_EVENT</code>. For more information, see <a
     * href="https://github.com/moby/moby/issues/25982">Unable to react to graceful shutdown of (Windows) container
     * #25982</a> on GitHub.
     * </p>
     * <note>
     * <p>
     * The default 30-second timeout can be configured on the Amazon ECS container agent with the
     * <code>ECS_CONTAINER_STOP_TIMEOUT</code> variable. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon ECS Container
     * Agent Configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * </note>
     *
     * @param stopTaskRequest
     * @return Result of the StopTask operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.StopTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/StopTask" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopTaskResponse stopTask(StopTaskRequest stopTaskRequest) throws ServerException, ClientException,
            InvalidParameterException, ClusterNotFoundException, AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopTaskResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopTaskResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopTaskRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopTask");

            return clientHandler.execute(new ClientExecutionParams<StopTaskRequest, StopTaskResponse>()
                    .withOperationName("StopTask").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(stopTaskRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopTaskRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <note>
     * <p>
     * This action is only used by the Amazon ECS agent, and it is not intended for use outside of the agent.
     * </p>
     * </note>
     * <p>
     * Sent to acknowledge that an attachment changed states.
     * </p>
     *
     * @param submitAttachmentStateChangesRequest
     * @return Result of the SubmitAttachmentStateChanges operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws AccessDeniedException
     *         You don't have authorization to perform the requested action.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.SubmitAttachmentStateChanges
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/SubmitAttachmentStateChanges"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SubmitAttachmentStateChangesResponse submitAttachmentStateChanges(
            SubmitAttachmentStateChangesRequest submitAttachmentStateChangesRequest) throws ServerException, ClientException,
            AccessDeniedException, InvalidParameterException, AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SubmitAttachmentStateChangesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SubmitAttachmentStateChangesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(submitAttachmentStateChangesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, submitAttachmentStateChangesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SubmitAttachmentStateChanges");

            return clientHandler
                    .execute(new ClientExecutionParams<SubmitAttachmentStateChangesRequest, SubmitAttachmentStateChangesResponse>()
                            .withOperationName("SubmitAttachmentStateChanges").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(submitAttachmentStateChangesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new SubmitAttachmentStateChangesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <note>
     * <p>
     * This action is only used by the Amazon ECS agent, and it is not intended for use outside of the agent.
     * </p>
     * </note>
     * <p>
     * Sent to acknowledge that a container changed states.
     * </p>
     *
     * @param submitContainerStateChangeRequest
     * @return Result of the SubmitContainerStateChange operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws AccessDeniedException
     *         You don't have authorization to perform the requested action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.SubmitContainerStateChange
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/SubmitContainerStateChange"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SubmitContainerStateChangeResponse submitContainerStateChange(
            SubmitContainerStateChangeRequest submitContainerStateChangeRequest) throws ServerException, ClientException,
            AccessDeniedException, AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SubmitContainerStateChangeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SubmitContainerStateChangeResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(submitContainerStateChangeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, submitContainerStateChangeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SubmitContainerStateChange");

            return clientHandler
                    .execute(new ClientExecutionParams<SubmitContainerStateChangeRequest, SubmitContainerStateChangeResponse>()
                            .withOperationName("SubmitContainerStateChange").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(submitContainerStateChangeRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new SubmitContainerStateChangeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <note>
     * <p>
     * This action is only used by the Amazon ECS agent, and it is not intended for use outside of the agent.
     * </p>
     * </note>
     * <p>
     * Sent to acknowledge that a task changed states.
     * </p>
     *
     * @param submitTaskStateChangeRequest
     * @return Result of the SubmitTaskStateChange operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws AccessDeniedException
     *         You don't have authorization to perform the requested action.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.SubmitTaskStateChange
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/SubmitTaskStateChange" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SubmitTaskStateChangeResponse submitTaskStateChange(SubmitTaskStateChangeRequest submitTaskStateChangeRequest)
            throws ServerException, ClientException, AccessDeniedException, InvalidParameterException, AwsServiceException,
            SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SubmitTaskStateChangeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SubmitTaskStateChangeResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(submitTaskStateChangeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, submitTaskStateChangeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SubmitTaskStateChange");

            return clientHandler.execute(new ClientExecutionParams<SubmitTaskStateChangeRequest, SubmitTaskStateChangeResponse>()
                    .withOperationName("SubmitTaskStateChange").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(submitTaskStateChangeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SubmitTaskStateChangeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Associates the specified tags to a resource with the specified <code>resourceArn</code>. If existing tags on a
     * resource aren't specified in the request parameters, they aren't changed. When a resource is deleted, the tags
     * that are associated with that resource are deleted as well.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws ResourceNotFoundException
     *         The specified resource wasn't found.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ServerException, ClientException,
            ClusterNotFoundException, ResourceNotFoundException, InvalidParameterException, AwsServiceException,
            SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes specified tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws ResourceNotFoundException
     *         The specified resource wasn't found.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ServerException,
            ClientException, ClusterNotFoundException, ResourceNotFoundException, InvalidParameterException, AwsServiceException,
            SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Modifies the parameters for a capacity provider.
     * </p>
     *
     * @param updateCapacityProviderRequest
     * @return Result of the UpdateCapacityProvider operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.UpdateCapacityProvider
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/UpdateCapacityProvider" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateCapacityProviderResponse updateCapacityProvider(UpdateCapacityProviderRequest updateCapacityProviderRequest)
            throws ServerException, ClientException, InvalidParameterException, AwsServiceException, SdkClientException,
            EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateCapacityProviderResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateCapacityProviderResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateCapacityProviderRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateCapacityProviderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateCapacityProvider");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateCapacityProviderRequest, UpdateCapacityProviderResponse>()
                            .withOperationName("UpdateCapacityProvider").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateCapacityProviderRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateCapacityProviderRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the cluster.
     * </p>
     *
     * @param updateClusterRequest
     * @return Result of the UpdateCluster operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws NamespaceNotFoundException
     *         The specified namespace wasn't found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.UpdateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/UpdateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateClusterResponse updateCluster(UpdateClusterRequest updateClusterRequest) throws ServerException,
            ClientException, ClusterNotFoundException, InvalidParameterException, NamespaceNotFoundException,
            AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateClusterResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateClusterRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateCluster");

            return clientHandler.execute(new ClientExecutionParams<UpdateClusterRequest, UpdateClusterResponse>()
                    .withOperationName("UpdateCluster").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateClusterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateClusterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Modifies the settings to use for a cluster.
     * </p>
     *
     * @param updateClusterSettingsRequest
     * @return Result of the UpdateClusterSettings operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.UpdateClusterSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/UpdateClusterSettings" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateClusterSettingsResponse updateClusterSettings(UpdateClusterSettingsRequest updateClusterSettingsRequest)
            throws ServerException, ClientException, ClusterNotFoundException, InvalidParameterException, AwsServiceException,
            SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateClusterSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateClusterSettingsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateClusterSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateClusterSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateClusterSettings");

            return clientHandler.execute(new ClientExecutionParams<UpdateClusterSettingsRequest, UpdateClusterSettingsResponse>()
                    .withOperationName("UpdateClusterSettings").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateClusterSettingsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateClusterSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the Amazon ECS container agent on a specified container instance. Updating the Amazon ECS container agent
     * doesn't interrupt running tasks or services on the container instance. The process for updating the agent differs
     * depending on whether your container instance was launched with the Amazon ECS-optimized AMI or another operating
     * system.
     * </p>
     * <note>
     * <p>
     * The <code>UpdateContainerAgent</code> API isn't supported for container instances using the Amazon ECS-optimized
     * Amazon Linux 2 (arm64) AMI. To update the container agent, you can update the <code>ecs-init</code> package. This
     * updates the agent. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/agent-update-ecs-ami.html">Updating the Amazon
     * ECS container agent</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * </note> <note>
     * <p>
     * Agent updates with the <code>UpdateContainerAgent</code> API operation do not apply to Windows container
     * instances. We recommend that you launch new container instances to update the agent version in your Windows
     * clusters.
     * </p>
     * </note>
     * <p>
     * The <code>UpdateContainerAgent</code> API requires an Amazon ECS-optimized AMI or Amazon Linux AMI with the
     * <code>ecs-init</code> service installed and running. For help updating the Amazon ECS container agent on other
     * operating systems, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html#manually_update_agent"
     * >Manually updating the Amazon ECS container agent</a> in the <i>Amazon Elastic Container Service Developer
     * Guide</i>.
     * </p>
     *
     * @param updateContainerAgentRequest
     * @return Result of the UpdateContainerAgent operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws UpdateInProgressException
     *         There's already a current Amazon ECS container agent update in progress on the container instance that's
     *         specified. If the container agent becomes disconnected while it's in a transitional stage, such as
     *         <code>PENDING</code> or <code>STAGING</code>, the update process can get stuck in that state. However,
     *         when the agent reconnects, it resumes where it stopped previously.
     * @throws NoUpdateAvailableException
     *         There's no update available for this Amazon ECS container agent. This might be because the agent is
     *         already running the latest version or because it's so old that there's no update path to the current
     *         version.
     * @throws MissingVersionException
     *         Amazon ECS can't determine the current version of the Amazon ECS container agent on the container
     *         instance and doesn't have enough information to proceed with an update. This could be because the agent
     *         running on the container instance is a previous or custom version that doesn't use our version
     *         information.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.UpdateContainerAgent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/UpdateContainerAgent" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateContainerAgentResponse updateContainerAgent(UpdateContainerAgentRequest updateContainerAgentRequest)
            throws ServerException, ClientException, InvalidParameterException, ClusterNotFoundException,
            UpdateInProgressException, NoUpdateAvailableException, MissingVersionException, AwsServiceException,
            SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateContainerAgentResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateContainerAgentResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateContainerAgentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateContainerAgentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateContainerAgent");

            return clientHandler.execute(new ClientExecutionParams<UpdateContainerAgentRequest, UpdateContainerAgentResponse>()
                    .withOperationName("UpdateContainerAgent").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateContainerAgentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateContainerAgentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Modifies the status of an Amazon ECS container instance.
     * </p>
     * <p>
     * Once a container instance has reached an <code>ACTIVE</code> state, you can change the status of a container
     * instance to <code>DRAINING</code> to manually remove an instance from a cluster, for example to perform system
     * updates, update the Docker daemon, or scale down the cluster size.
     * </p>
     * <important>
     * <p>
     * A container instance can't be changed to <code>DRAINING</code> until it has reached an <code>ACTIVE</code>
     * status. If the instance is in any other status, an error will be received.
     * </p>
     * </important>
     * <p>
     * When you set a container instance to <code>DRAINING</code>, Amazon ECS prevents new tasks from being scheduled
     * for placement on the container instance and replacement service tasks are started on other container instances in
     * the cluster if the resources are available. Service tasks on the container instance that are in the
     * <code>PENDING</code> state are stopped immediately.
     * </p>
     * <p>
     * Service tasks on the container instance that are in the <code>RUNNING</code> state are stopped and replaced
     * according to the service's deployment configuration parameters, <code>minimumHealthyPercent</code> and
     * <code>maximumPercent</code>. You can change the deployment configuration of your service using <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_UpdateService.html">UpdateService</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If <code>minimumHealthyPercent</code> is below 100%, the scheduler can ignore <code>desiredCount</code>
     * temporarily during task replacement. For example, <code>desiredCount</code> is four tasks, a minimum of 50%
     * allows the scheduler to stop two existing tasks before starting two new tasks. If the minimum is 100%, the
     * service scheduler can't remove existing tasks until the replacement tasks are considered healthy. Tasks for
     * services that do not use a load balancer are considered healthy if they're in the <code>RUNNING</code> state.
     * Tasks for services that use a load balancer are considered healthy if they're in the <code>RUNNING</code> state
     * and are reported as healthy by the load balancer.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>maximumPercent</code> parameter represents an upper limit on the number of running tasks during task
     * replacement. You can use this to define the replacement batch size. For example, if <code>desiredCount</code> is
     * four tasks, a maximum of 200% starts four new tasks before stopping the four tasks to be drained, provided that
     * the cluster resources required to do this are available. If the maximum is 100%, then replacement tasks can't
     * start until the draining tasks have stopped.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Any <code>PENDING</code> or <code>RUNNING</code> tasks that do not belong to a service aren't affected. You must
     * wait for them to finish or stop them manually.
     * </p>
     * <p>
     * A container instance has completed draining when it has no more <code>RUNNING</code> tasks. You can verify this
     * using <a href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListTasks.html">ListTasks</a>.
     * </p>
     * <p>
     * When a container instance has been drained, you can set a container instance to <code>ACTIVE</code> status and
     * once it has reached that status the Amazon ECS scheduler can begin scheduling tasks on the instance again.
     * </p>
     *
     * @param updateContainerInstancesStateRequest
     * @return Result of the UpdateContainerInstancesState operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.UpdateContainerInstancesState
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/UpdateContainerInstancesState"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateContainerInstancesStateResponse updateContainerInstancesState(
            UpdateContainerInstancesStateRequest updateContainerInstancesStateRequest) throws ServerException, ClientException,
            InvalidParameterException, ClusterNotFoundException, AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateContainerInstancesStateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateContainerInstancesStateResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateContainerInstancesStateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateContainerInstancesStateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateContainerInstancesState");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateContainerInstancesStateRequest, UpdateContainerInstancesStateResponse>()
                            .withOperationName("UpdateContainerInstancesState").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateContainerInstancesStateRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateContainerInstancesStateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Modifies the parameters of a service.
     * </p>
     * <note>
     * <p>
     * On March 21, 2024, a change was made to resolve the task definition revision before authorization. When a task
     * definition revision is not specified, authorization will occur using the latest revision of a task definition.
     * </p>
     * </note>
     * <p>
     * For services using the rolling update (<code>ECS</code>) you can update the desired count, deployment
     * configuration, network configuration, load balancers, service registries, enable ECS managed tags option,
     * propagate tags option, task placement constraints and strategies, and task definition. When you update any of
     * these parameters, Amazon ECS starts new tasks with the new configuration.
     * </p>
     * <p>
     * You can attach Amazon EBS volumes to Amazon ECS tasks by configuring the volume when starting or running a task,
     * or when creating or updating a service. For more infomation, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ebs-volumes.html#ebs-volume-types">Amazon EBS
     * volumes</a> in the <i>Amazon Elastic Container Service Developer Guide</i>. You can update your volume
     * configurations and trigger a new deployment. <code>volumeConfigurations</code> is only supported for REPLICA
     * service and not DAEMON service. If you leave <code>volumeConfigurations</code> <code>null</code>, it doesn't
     * trigger a new deployment. For more infomation on volumes, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ebs-volumes.html#ebs-volume-types">Amazon EBS
     * volumes</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * For services using the blue/green (<code>CODE_DEPLOY</code>) deployment controller, only the desired count,
     * deployment configuration, health check grace period, task placement constraints and strategies, enable ECS
     * managed tags option, and propagate tags can be updated using this API. If the network configuration, platform
     * version, task definition, or load balancer need to be updated, create a new CodeDeploy deployment. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_CreateDeployment.html">CreateDeployment</a>
     * in the <i>CodeDeploy API Reference</i>.
     * </p>
     * <p>
     * For services using an external deployment controller, you can update only the desired count, task placement
     * constraints and strategies, health check grace period, enable ECS managed tags option, and propagate tags option,
     * using this API. If the launch type, load balancer, network configuration, platform version, or task definition
     * need to be updated, create a new task set For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_CreateTaskSet.html">CreateTaskSet</a>.
     * </p>
     * <p>
     * You can add to or subtract from the number of instantiations of a task definition in a service by specifying the
     * cluster that the service is running in and a new <code>desiredCount</code> parameter.
     * </p>
     * <p>
     * You can attach Amazon EBS volumes to Amazon ECS tasks by configuring the volume when starting or running a task,
     * or when creating or updating a service. For more infomation, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ebs-volumes.html#ebs-volume-types">Amazon EBS
     * volumes</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * If you have updated the container image of your application, you can create a new task definition with that image
     * and deploy it to your service. The service scheduler uses the minimum healthy percent and maximum percent
     * parameters (in the service's deployment configuration) to determine the deployment strategy.
     * </p>
     * <note>
     * <p>
     * If your updated Docker image uses the same tag as what is in the existing task definition for your service (for
     * example, <code>my_image:latest</code>), you don't need to create a new revision of your task definition. You can
     * update the service using the <code>forceNewDeployment</code> option. The new tasks launched by the deployment
     * pull the current image/tag combination from your repository when they start.
     * </p>
     * </note>
     * <p>
     * You can also update the deployment configuration of a service. When a deployment is triggered by updating the
     * task definition of a service, the service scheduler uses the deployment configuration parameters,
     * <code>minimumHealthyPercent</code> and <code>maximumPercent</code>, to determine the deployment strategy.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If <code>minimumHealthyPercent</code> is below 100%, the scheduler can ignore <code>desiredCount</code>
     * temporarily during a deployment. For example, if <code>desiredCount</code> is four tasks, a minimum of 50% allows
     * the scheduler to stop two existing tasks before starting two new tasks. Tasks for services that don't use a load
     * balancer are considered healthy if they're in the <code>RUNNING</code> state. Tasks for services that use a load
     * balancer are considered healthy if they're in the <code>RUNNING</code> state and are reported as healthy by the
     * load balancer.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>maximumPercent</code> parameter represents an upper limit on the number of running tasks during a
     * deployment. You can use it to define the deployment batch size. For example, if <code>desiredCount</code> is four
     * tasks, a maximum of 200% starts four new tasks before stopping the four older tasks (provided that the cluster
     * resources required to do this are available).
     * </p>
     * </li>
     * </ul>
     * <p>
     * When <a href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_UpdateService.html">UpdateService</a>
     * stops a task during a deployment, the equivalent of <code>docker stop</code> is issued to the containers running
     * in the task. This results in a <code>SIGTERM</code> and a 30-second timeout. After this, <code>SIGKILL</code> is
     * sent and the containers are forcibly stopped. If the container handles the <code>SIGTERM</code> gracefully and
     * exits within 30 seconds from receiving it, no <code>SIGKILL</code> is sent.
     * </p>
     * <p>
     * When the service scheduler launches new tasks, it determines task placement in your cluster with the following
     * logic.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Determine which of the container instances in your cluster can support your service's task definition. For
     * example, they have the required CPU, memory, ports, and container instance attributes.
     * </p>
     * </li>
     * <li>
     * <p>
     * By default, the service scheduler attempts to balance tasks across Availability Zones in this manner even though
     * you can choose a different placement strategy.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Sort the valid container instances by the fewest number of running tasks for this service in the same
     * Availability Zone as the instance. For example, if zone A has one running service task and zones B and C each
     * have zero, valid container instances in either zone B or C are considered optimal for placement.
     * </p>
     * </li>
     * <li>
     * <p>
     * Place the new service task on a valid container instance in an optimal Availability Zone (based on the previous
     * steps), favoring container instances with the fewest number of running tasks for this service.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * When the service scheduler stops running tasks, it attempts to maintain balance across the Availability Zones in
     * your cluster using the following logic:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Sort the container instances by the largest number of running tasks for this service in the same Availability
     * Zone as the instance. For example, if zone A has one running service task and zones B and C each have two,
     * container instances in either zone B or C are considered optimal for termination.
     * </p>
     * </li>
     * <li>
     * <p>
     * Stop the task on a container instance in an optimal Availability Zone (based on the previous steps), favoring
     * container instances with the largest number of running tasks for this service.
     * </p>
     * </li>
     * </ul>
     *
     * @param updateServiceRequest
     * @return Result of the UpdateService operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws ServiceNotFoundException
     *         The specified service wasn't found. You can view your available services with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListServices.html">ListServices</a>.
     *         Amazon ECS services are cluster specific and Region specific.
     * @throws ServiceNotActiveException
     *         The specified service isn't active. You can't update a service that's inactive. If you have previously
     *         deleted a service, you can re-create it with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_CreateService.html"
     *         >CreateService</a>.
     * @throws PlatformUnknownException
     *         The specified platform version doesn't exist.
     * @throws PlatformTaskDefinitionIncompatibilityException
     *         The specified platform version doesn't satisfy the required capabilities of the task definition.
     * @throws AccessDeniedException
     *         You don't have authorization to perform the requested action.
     * @throws NamespaceNotFoundException
     *         The specified namespace wasn't found.
     * @throws UnsupportedFeatureException
     *         The specified task isn't supported in this Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.UpdateService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/UpdateService" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateServiceResponse updateService(UpdateServiceRequest updateServiceRequest) throws ServerException,
            ClientException, InvalidParameterException, ClusterNotFoundException, ServiceNotFoundException,
            ServiceNotActiveException, PlatformUnknownException, PlatformTaskDefinitionIncompatibilityException,
            AccessDeniedException, NamespaceNotFoundException, UnsupportedFeatureException, AwsServiceException,
            SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateServiceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateServiceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateService");

            return clientHandler.execute(new ClientExecutionParams<UpdateServiceRequest, UpdateServiceResponse>()
                    .withOperationName("UpdateService").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateServiceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateServiceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Modifies which task set in a service is the primary task set. Any parameters that are updated on the primary task
     * set in a service will transition to the service. This is used when a service uses the <code>EXTERNAL</code>
     * deployment controller type. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html">Amazon ECS Deployment
     * Types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     *
     * @param updateServicePrimaryTaskSetRequest
     * @return Result of the UpdateServicePrimaryTaskSet operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws UnsupportedFeatureException
     *         The specified task isn't supported in this Region.
     * @throws ServiceNotFoundException
     *         The specified service wasn't found. You can view your available services with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListServices.html">ListServices</a>.
     *         Amazon ECS services are cluster specific and Region specific.
     * @throws ServiceNotActiveException
     *         The specified service isn't active. You can't update a service that's inactive. If you have previously
     *         deleted a service, you can re-create it with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_CreateService.html"
     *         >CreateService</a>.
     * @throws TaskSetNotFoundException
     *         The specified task set wasn't found. You can view your available task sets with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTaskSets.html"
     *         >DescribeTaskSets</a>. Task sets are specific to each cluster, service and Region.
     * @throws AccessDeniedException
     *         You don't have authorization to perform the requested action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.UpdateServicePrimaryTaskSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/UpdateServicePrimaryTaskSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateServicePrimaryTaskSetResponse updateServicePrimaryTaskSet(
            UpdateServicePrimaryTaskSetRequest updateServicePrimaryTaskSetRequest) throws ServerException, ClientException,
            InvalidParameterException, ClusterNotFoundException, UnsupportedFeatureException, ServiceNotFoundException,
            ServiceNotActiveException, TaskSetNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateServicePrimaryTaskSetResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateServicePrimaryTaskSetResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateServicePrimaryTaskSetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateServicePrimaryTaskSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateServicePrimaryTaskSet");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateServicePrimaryTaskSetRequest, UpdateServicePrimaryTaskSetResponse>()
                            .withOperationName("UpdateServicePrimaryTaskSet").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateServicePrimaryTaskSetRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateServicePrimaryTaskSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the protection status of a task. You can set <code>protectionEnabled</code> to <code>true</code> to
     * protect your task from termination during scale-in events from <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-auto-scaling.html">Service
     * Autoscaling</a> or <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html">deployments</a>.
     * </p>
     * <p>
     * Task-protection, by default, expires after 2 hours at which point Amazon ECS clears the
     * <code>protectionEnabled</code> property making the task eligible for termination by a subsequent scale-in event.
     * </p>
     * <p>
     * You can specify a custom expiration period for task protection from 1 minute to up to 2,880 minutes (48 hours).
     * To specify the custom expiration period, set the <code>expiresInMinutes</code> property. The
     * <code>expiresInMinutes</code> property is always reset when you invoke this operation for a task that already has
     * <code>protectionEnabled</code> set to <code>true</code>. You can keep extending the protection expiration period
     * of a task by invoking this operation repeatedly.
     * </p>
     * <p>
     * To learn more about Amazon ECS task protection, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-scale-in-protection.html">Task scale-in
     * protection</a> in the <i> <i>Amazon Elastic Container Service Developer Guide</i> </i>.
     * </p>
     * <note>
     * <p>
     * This operation is only supported for tasks belonging to an Amazon ECS service. Invoking this operation for a
     * standalone task will result in an <code>TASK_NOT_VALID</code> failure. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/api_failures_messages.html">API failure
     * reasons</a>.
     * </p>
     * </note> <important>
     * <p>
     * If you prefer to set task protection from within the container, we recommend using the <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-scale-in-protection-endpoint.html">Task
     * scale-in protection endpoint</a>.
     * </p>
     * </important>
     *
     * @param updateTaskProtectionRequest
     * @return Result of the UpdateTaskProtection operation returned by the service.
     * @throws AccessDeniedException
     *         You don't have authorization to perform the requested action.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ResourceNotFoundException
     *         The specified resource wasn't found.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws UnsupportedFeatureException
     *         The specified task isn't supported in this Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.UpdateTaskProtection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/UpdateTaskProtection" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateTaskProtectionResponse updateTaskProtection(UpdateTaskProtectionRequest updateTaskProtectionRequest)
            throws AccessDeniedException, ClientException, ClusterNotFoundException, InvalidParameterException,
            ResourceNotFoundException, ServerException, UnsupportedFeatureException, AwsServiceException, SdkClientException,
            EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateTaskProtectionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateTaskProtectionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateTaskProtectionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTaskProtectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTaskProtection");

            return clientHandler.execute(new ClientExecutionParams<UpdateTaskProtectionRequest, UpdateTaskProtectionResponse>()
                    .withOperationName("UpdateTaskProtection").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateTaskProtectionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateTaskProtectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Modifies a task set. This is used when a service uses the <code>EXTERNAL</code> deployment controller type. For
     * more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html">Amazon ECS Deployment
     * Types</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     *
     * @param updateTaskSetRequest
     * @return Result of the UpdateTaskSet operation returned by the service.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws ClientException
     *         These errors are usually caused by a client action. This client action might be using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource. Or, it might be
     *         specifying an identifier that isn't valid.
     * @throws InvalidParameterException
     *         The specified parameter isn't valid. Review the available parameters for the API request.</p>
     *         <p>
     *         For more information about service event errors, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-event-messages-list.html"
     *         >Amazon ECS service event messages</a>.
     * @throws ClusterNotFoundException
     *         The specified cluster wasn't found. You can view your available clusters with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListClusters.html">ListClusters</a>.
     *         Amazon ECS clusters are Region specific.
     * @throws UnsupportedFeatureException
     *         The specified task isn't supported in this Region.
     * @throws AccessDeniedException
     *         You don't have authorization to perform the requested action.
     * @throws ServiceNotFoundException
     *         The specified service wasn't found. You can view your available services with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListServices.html">ListServices</a>.
     *         Amazon ECS services are cluster specific and Region specific.
     * @throws ServiceNotActiveException
     *         The specified service isn't active. You can't update a service that's inactive. If you have previously
     *         deleted a service, you can re-create it with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_CreateService.html"
     *         >CreateService</a>.
     * @throws TaskSetNotFoundException
     *         The specified task set wasn't found. You can view your available task sets with <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTaskSets.html"
     *         >DescribeTaskSets</a>. Task sets are specific to each cluster, service and Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EcsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EcsClient.UpdateTaskSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ecs-2014-11-13/UpdateTaskSet" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateTaskSetResponse updateTaskSet(UpdateTaskSetRequest updateTaskSetRequest) throws ServerException,
            ClientException, InvalidParameterException, ClusterNotFoundException, UnsupportedFeatureException,
            AccessDeniedException, ServiceNotFoundException, ServiceNotActiveException, TaskSetNotFoundException,
            AwsServiceException, SdkClientException, EcsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateTaskSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateTaskSetResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "BlockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BlockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BlockedException::builder).build());
            case "TargetNotConnectedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotConnectedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotConnectedException::builder).build());
            case "ClusterContainsServicesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsServicesException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsServicesException::builder).build());
            case "UnsupportedFeatureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedFeatureException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedFeatureException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            case "ClusterNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterNotFoundException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClientException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClientException::builder).build());
            case "PlatformUnknownException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformUnknownException").httpStatusCode(400)
                        .exceptionBuilderSupplier(PlatformUnknownException::builder).build());
            case "UpdateInProgressException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateInProgressException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateInProgressException::builder).build());
            case "NamespaceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NamespaceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NamespaceNotFoundException::builder).build());
            case "ClusterContainsTasksException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsTasksException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ClusterContainsTasksException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "NoUpdateAvailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NoUpdateAvailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NoUpdateAvailableException::builder).build());
            case "PlatformTaskDefinitionIncompatibilityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("PlatformTaskDefinitionIncompatibilityException")
                        .httpStatusCode(400).exceptionBuilderSupplier(PlatformTaskDefinitionIncompatibilityException::builder)
                        .build());
            case "MissingVersionException":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVersionException").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVersionException::builder).build());
            case "ClusterContainsContainerInstancesException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ClusterContainsContainerInstancesException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ClusterContainsContainerInstancesException::builder)
                        .build());
            case "ServiceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotFoundException::builder).build());
            case "TaskSetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TaskSetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TaskSetNotFoundException::builder).build());
            case "AttributeLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AttributeLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AttributeLimitExceededException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ServiceNotActiveException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceNotActiveException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceNotActiveException::builder).build());
            case "ServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServerException::builder).build());
            case "TargetNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TargetNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TargetNotFoundException::builder).build());
            case "ServiceDeploymentNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceDeploymentNotFoundException")
                        .httpStatusCode(400).exceptionBuilderSupplier(ServiceDeploymentNotFoundException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateTaskSetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTaskSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ECS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTaskSet");

            return clientHandler.execute(new ClientExecutionParams<UpdateTaskSetRequest, UpdateTaskSetResponse>()
                    .withOperationName("UpdateTaskSet").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateTaskSetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateTaskSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Create an instance of {@link EcsWaiter} using this client.
     * <p>
     * Waiters created via this method are managed by the SDK and resources will be released when the service client is
     * closed.
     *
     * @return an instance of {@link EcsWaiter}
     */
    @Override
    public EcsWaiter waiter() {
        return EcsWaiter.builder().client(this).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        EcsServiceClientConfigurationBuilder serviceConfigBuilder = new EcsServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(EcsException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON).protocolVersion("1.1");
    }

    @Override
    public final EcsServiceClientConfiguration serviceClientConfiguration() {
        return new EcsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
