/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the service revision.
 * </p>
 * <p>
 * A service revision contains a record of the workload configuration Amazon ECS is attempting to deploy. Whenever you
 * create or deploy a service, Amazon ECS automatically creates and captures the configuration that you're trying to
 * deploy in the service revision. For information about service revisions, see <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-revision.html">Amazon ECS service
 * revisions</a> in the <i> <i>Amazon Elastic Container Service Developer Guide</i> </i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ServiceRevision implements SdkPojo, Serializable, ToCopyableBuilder<ServiceRevision.Builder, ServiceRevision> {
    private static final SdkField<String> SERVICE_REVISION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("serviceRevisionArn").getter(getter(ServiceRevision::serviceRevisionArn))
            .setter(setter(Builder::serviceRevisionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serviceRevisionArn").build())
            .build();

    private static final SdkField<String> SERVICE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("serviceArn").getter(getter(ServiceRevision::serviceArn)).setter(setter(Builder::serviceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serviceArn").build()).build();

    private static final SdkField<String> CLUSTER_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("clusterArn").getter(getter(ServiceRevision::clusterArn)).setter(setter(Builder::clusterArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("clusterArn").build()).build();

    private static final SdkField<String> TASK_DEFINITION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("taskDefinition").getter(getter(ServiceRevision::taskDefinition)).setter(setter(Builder::taskDefinition))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("taskDefinition").build()).build();

    private static final SdkField<List<CapacityProviderStrategyItem>> CAPACITY_PROVIDER_STRATEGY_FIELD = SdkField
            .<List<CapacityProviderStrategyItem>> builder(MarshallingType.LIST)
            .memberName("capacityProviderStrategy")
            .getter(getter(ServiceRevision::capacityProviderStrategy))
            .setter(setter(Builder::capacityProviderStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("capacityProviderStrategy").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<CapacityProviderStrategyItem> builder(MarshallingType.SDK_POJO)
                                            .constructor(CapacityProviderStrategyItem::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> LAUNCH_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("launchType").getter(getter(ServiceRevision::launchTypeAsString)).setter(setter(Builder::launchType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("launchType").build()).build();

    private static final SdkField<String> PLATFORM_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("platformVersion").getter(getter(ServiceRevision::platformVersion))
            .setter(setter(Builder::platformVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("platformVersion").build()).build();

    private static final SdkField<String> PLATFORM_FAMILY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("platformFamily").getter(getter(ServiceRevision::platformFamily)).setter(setter(Builder::platformFamily))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("platformFamily").build()).build();

    private static final SdkField<List<LoadBalancer>> LOAD_BALANCERS_FIELD = SdkField
            .<List<LoadBalancer>> builder(MarshallingType.LIST)
            .memberName("loadBalancers")
            .getter(getter(ServiceRevision::loadBalancers))
            .setter(setter(Builder::loadBalancers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("loadBalancers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LoadBalancer> builder(MarshallingType.SDK_POJO)
                                            .constructor(LoadBalancer::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ServiceRegistry>> SERVICE_REGISTRIES_FIELD = SdkField
            .<List<ServiceRegistry>> builder(MarshallingType.LIST)
            .memberName("serviceRegistries")
            .getter(getter(ServiceRevision::serviceRegistries))
            .setter(setter(Builder::serviceRegistries))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serviceRegistries").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ServiceRegistry> builder(MarshallingType.SDK_POJO)
                                            .constructor(ServiceRegistry::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<NetworkConfiguration> NETWORK_CONFIGURATION_FIELD = SdkField
            .<NetworkConfiguration> builder(MarshallingType.SDK_POJO).memberName("networkConfiguration")
            .getter(getter(ServiceRevision::networkConfiguration)).setter(setter(Builder::networkConfiguration))
            .constructor(NetworkConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("networkConfiguration").build())
            .build();

    private static final SdkField<List<ContainerImage>> CONTAINER_IMAGES_FIELD = SdkField
            .<List<ContainerImage>> builder(MarshallingType.LIST)
            .memberName("containerImages")
            .getter(getter(ServiceRevision::containerImages))
            .setter(setter(Builder::containerImages))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerImages").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ContainerImage> builder(MarshallingType.SDK_POJO)
                                            .constructor(ContainerImage::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> GUARD_DUTY_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("guardDutyEnabled").getter(getter(ServiceRevision::guardDutyEnabled))
            .setter(setter(Builder::guardDutyEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("guardDutyEnabled").build()).build();

    private static final SdkField<ServiceConnectConfiguration> SERVICE_CONNECT_CONFIGURATION_FIELD = SdkField
            .<ServiceConnectConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("serviceConnectConfiguration")
            .getter(getter(ServiceRevision::serviceConnectConfiguration))
            .setter(setter(Builder::serviceConnectConfiguration))
            .constructor(ServiceConnectConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serviceConnectConfiguration")
                    .build()).build();

    private static final SdkField<List<ServiceVolumeConfiguration>> VOLUME_CONFIGURATIONS_FIELD = SdkField
            .<List<ServiceVolumeConfiguration>> builder(MarshallingType.LIST)
            .memberName("volumeConfigurations")
            .getter(getter(ServiceRevision::volumeConfigurations))
            .setter(setter(Builder::volumeConfigurations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("volumeConfigurations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ServiceVolumeConfiguration> builder(MarshallingType.SDK_POJO)
                                            .constructor(ServiceVolumeConfiguration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<DeploymentEphemeralStorage> FARGATE_EPHEMERAL_STORAGE_FIELD = SdkField
            .<DeploymentEphemeralStorage> builder(MarshallingType.SDK_POJO).memberName("fargateEphemeralStorage")
            .getter(getter(ServiceRevision::fargateEphemeralStorage)).setter(setter(Builder::fargateEphemeralStorage))
            .constructor(DeploymentEphemeralStorage::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("fargateEphemeralStorage").build())
            .build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt").getter(getter(ServiceRevision::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<List<VpcLatticeConfiguration>> VPC_LATTICE_CONFIGURATIONS_FIELD = SdkField
            .<List<VpcLatticeConfiguration>> builder(MarshallingType.LIST)
            .memberName("vpcLatticeConfigurations")
            .getter(getter(ServiceRevision::vpcLatticeConfigurations))
            .setter(setter(Builder::vpcLatticeConfigurations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("vpcLatticeConfigurations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<VpcLatticeConfiguration> builder(MarshallingType.SDK_POJO)
                                            .constructor(VpcLatticeConfiguration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SERVICE_REVISION_ARN_FIELD,
            SERVICE_ARN_FIELD, CLUSTER_ARN_FIELD, TASK_DEFINITION_FIELD, CAPACITY_PROVIDER_STRATEGY_FIELD, LAUNCH_TYPE_FIELD,
            PLATFORM_VERSION_FIELD, PLATFORM_FAMILY_FIELD, LOAD_BALANCERS_FIELD, SERVICE_REGISTRIES_FIELD,
            NETWORK_CONFIGURATION_FIELD, CONTAINER_IMAGES_FIELD, GUARD_DUTY_ENABLED_FIELD, SERVICE_CONNECT_CONFIGURATION_FIELD,
            VOLUME_CONFIGURATIONS_FIELD, FARGATE_EPHEMERAL_STORAGE_FIELD, CREATED_AT_FIELD, VPC_LATTICE_CONFIGURATIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String serviceRevisionArn;

    private final String serviceArn;

    private final String clusterArn;

    private final String taskDefinition;

    private final List<CapacityProviderStrategyItem> capacityProviderStrategy;

    private final String launchType;

    private final String platformVersion;

    private final String platformFamily;

    private final List<LoadBalancer> loadBalancers;

    private final List<ServiceRegistry> serviceRegistries;

    private final NetworkConfiguration networkConfiguration;

    private final List<ContainerImage> containerImages;

    private final Boolean guardDutyEnabled;

    private final ServiceConnectConfiguration serviceConnectConfiguration;

    private final List<ServiceVolumeConfiguration> volumeConfigurations;

    private final DeploymentEphemeralStorage fargateEphemeralStorage;

    private final Instant createdAt;

    private final List<VpcLatticeConfiguration> vpcLatticeConfigurations;

    private ServiceRevision(BuilderImpl builder) {
        this.serviceRevisionArn = builder.serviceRevisionArn;
        this.serviceArn = builder.serviceArn;
        this.clusterArn = builder.clusterArn;
        this.taskDefinition = builder.taskDefinition;
        this.capacityProviderStrategy = builder.capacityProviderStrategy;
        this.launchType = builder.launchType;
        this.platformVersion = builder.platformVersion;
        this.platformFamily = builder.platformFamily;
        this.loadBalancers = builder.loadBalancers;
        this.serviceRegistries = builder.serviceRegistries;
        this.networkConfiguration = builder.networkConfiguration;
        this.containerImages = builder.containerImages;
        this.guardDutyEnabled = builder.guardDutyEnabled;
        this.serviceConnectConfiguration = builder.serviceConnectConfiguration;
        this.volumeConfigurations = builder.volumeConfigurations;
        this.fargateEphemeralStorage = builder.fargateEphemeralStorage;
        this.createdAt = builder.createdAt;
        this.vpcLatticeConfigurations = builder.vpcLatticeConfigurations;
    }

    /**
     * <p>
     * The ARN of the service revision.
     * </p>
     * 
     * @return The ARN of the service revision.
     */
    public final String serviceRevisionArn() {
        return serviceRevisionArn;
    }

    /**
     * <p>
     * The ARN of the service for the service revision.
     * </p>
     * 
     * @return The ARN of the service for the service revision.
     */
    public final String serviceArn() {
        return serviceArn;
    }

    /**
     * <p>
     * The ARN of the cluster that hosts the service.
     * </p>
     * 
     * @return The ARN of the cluster that hosts the service.
     */
    public final String clusterArn() {
        return clusterArn;
    }

    /**
     * <p>
     * The task definition the service revision uses.
     * </p>
     * 
     * @return The task definition the service revision uses.
     */
    public final String taskDefinition() {
        return taskDefinition;
    }

    /**
     * For responses, this returns true if the service returned a value for the CapacityProviderStrategy property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasCapacityProviderStrategy() {
        return capacityProviderStrategy != null && !(capacityProviderStrategy instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The capacity provider strategy the service revision uses.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCapacityProviderStrategy} method.
     * </p>
     * 
     * @return The capacity provider strategy the service revision uses.
     */
    public final List<CapacityProviderStrategyItem> capacityProviderStrategy() {
        return capacityProviderStrategy;
    }

    /**
     * <p>
     * The launch type the service revision uses.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #launchType} will
     * return {@link LaunchType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #launchTypeAsString}.
     * </p>
     * 
     * @return The launch type the service revision uses.
     * @see LaunchType
     */
    public final LaunchType launchType() {
        return LaunchType.fromValue(launchType);
    }

    /**
     * <p>
     * The launch type the service revision uses.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #launchType} will
     * return {@link LaunchType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #launchTypeAsString}.
     * </p>
     * 
     * @return The launch type the service revision uses.
     * @see LaunchType
     */
    public final String launchTypeAsString() {
        return launchType;
    }

    /**
     * <p>
     * For the Fargate launch type, the platform version the service revision uses.
     * </p>
     * 
     * @return For the Fargate launch type, the platform version the service revision uses.
     */
    public final String platformVersion() {
        return platformVersion;
    }

    /**
     * <p>
     * The platform family the service revision uses.
     * </p>
     * 
     * @return The platform family the service revision uses.
     */
    public final String platformFamily() {
        return platformFamily;
    }

    /**
     * For responses, this returns true if the service returned a value for the LoadBalancers property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLoadBalancers() {
        return loadBalancers != null && !(loadBalancers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The load balancers the service revision uses.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLoadBalancers} method.
     * </p>
     * 
     * @return The load balancers the service revision uses.
     */
    public final List<LoadBalancer> loadBalancers() {
        return loadBalancers;
    }

    /**
     * For responses, this returns true if the service returned a value for the ServiceRegistries property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasServiceRegistries() {
        return serviceRegistries != null && !(serviceRegistries instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The service registries (for Service Discovery) the service revision uses.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasServiceRegistries} method.
     * </p>
     * 
     * @return The service registries (for Service Discovery) the service revision uses.
     */
    public final List<ServiceRegistry> serviceRegistries() {
        return serviceRegistries;
    }

    /**
     * Returns the value of the NetworkConfiguration property for this object.
     * 
     * @return The value of the NetworkConfiguration property for this object.
     */
    public final NetworkConfiguration networkConfiguration() {
        return networkConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the ContainerImages property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasContainerImages() {
        return containerImages != null && !(containerImages instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The container images the service revision uses.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasContainerImages} method.
     * </p>
     * 
     * @return The container images the service revision uses.
     */
    public final List<ContainerImage> containerImages() {
        return containerImages;
    }

    /**
     * <p>
     * Indicates whether Runtime Monitoring is turned on.
     * </p>
     * 
     * @return Indicates whether Runtime Monitoring is turned on.
     */
    public final Boolean guardDutyEnabled() {
        return guardDutyEnabled;
    }

    /**
     * Returns the value of the ServiceConnectConfiguration property for this object.
     * 
     * @return The value of the ServiceConnectConfiguration property for this object.
     */
    public final ServiceConnectConfiguration serviceConnectConfiguration() {
        return serviceConnectConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the VolumeConfigurations property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasVolumeConfigurations() {
        return volumeConfigurations != null && !(volumeConfigurations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The volumes that are configured at deployment that the service revision uses.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVolumeConfigurations} method.
     * </p>
     * 
     * @return The volumes that are configured at deployment that the service revision uses.
     */
    public final List<ServiceVolumeConfiguration> volumeConfigurations() {
        return volumeConfigurations;
    }

    /**
     * Returns the value of the FargateEphemeralStorage property for this object.
     * 
     * @return The value of the FargateEphemeralStorage property for this object.
     */
    public final DeploymentEphemeralStorage fargateEphemeralStorage() {
        return fargateEphemeralStorage;
    }

    /**
     * <p>
     * The time that the service revision was created. The format is yyyy-mm-dd HH:mm:ss.SSSSS.
     * </p>
     * 
     * @return The time that the service revision was created. The format is yyyy-mm-dd HH:mm:ss.SSSSS.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * For responses, this returns true if the service returned a value for the VpcLatticeConfigurations property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasVpcLatticeConfigurations() {
        return vpcLatticeConfigurations != null && !(vpcLatticeConfigurations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The VPC Lattice configuration for the service revision.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVpcLatticeConfigurations} method.
     * </p>
     * 
     * @return The VPC Lattice configuration for the service revision.
     */
    public final List<VpcLatticeConfiguration> vpcLatticeConfigurations() {
        return vpcLatticeConfigurations;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(serviceRevisionArn());
        hashCode = 31 * hashCode + Objects.hashCode(serviceArn());
        hashCode = 31 * hashCode + Objects.hashCode(clusterArn());
        hashCode = 31 * hashCode + Objects.hashCode(taskDefinition());
        hashCode = 31 * hashCode + Objects.hashCode(hasCapacityProviderStrategy() ? capacityProviderStrategy() : null);
        hashCode = 31 * hashCode + Objects.hashCode(launchTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(platformVersion());
        hashCode = 31 * hashCode + Objects.hashCode(platformFamily());
        hashCode = 31 * hashCode + Objects.hashCode(hasLoadBalancers() ? loadBalancers() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasServiceRegistries() ? serviceRegistries() : null);
        hashCode = 31 * hashCode + Objects.hashCode(networkConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasContainerImages() ? containerImages() : null);
        hashCode = 31 * hashCode + Objects.hashCode(guardDutyEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(serviceConnectConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasVolumeConfigurations() ? volumeConfigurations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(fargateEphemeralStorage());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(hasVpcLatticeConfigurations() ? vpcLatticeConfigurations() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ServiceRevision)) {
            return false;
        }
        ServiceRevision other = (ServiceRevision) obj;
        return Objects.equals(serviceRevisionArn(), other.serviceRevisionArn())
                && Objects.equals(serviceArn(), other.serviceArn()) && Objects.equals(clusterArn(), other.clusterArn())
                && Objects.equals(taskDefinition(), other.taskDefinition())
                && hasCapacityProviderStrategy() == other.hasCapacityProviderStrategy()
                && Objects.equals(capacityProviderStrategy(), other.capacityProviderStrategy())
                && Objects.equals(launchTypeAsString(), other.launchTypeAsString())
                && Objects.equals(platformVersion(), other.platformVersion())
                && Objects.equals(platformFamily(), other.platformFamily()) && hasLoadBalancers() == other.hasLoadBalancers()
                && Objects.equals(loadBalancers(), other.loadBalancers())
                && hasServiceRegistries() == other.hasServiceRegistries()
                && Objects.equals(serviceRegistries(), other.serviceRegistries())
                && Objects.equals(networkConfiguration(), other.networkConfiguration())
                && hasContainerImages() == other.hasContainerImages()
                && Objects.equals(containerImages(), other.containerImages())
                && Objects.equals(guardDutyEnabled(), other.guardDutyEnabled())
                && Objects.equals(serviceConnectConfiguration(), other.serviceConnectConfiguration())
                && hasVolumeConfigurations() == other.hasVolumeConfigurations()
                && Objects.equals(volumeConfigurations(), other.volumeConfigurations())
                && Objects.equals(fargateEphemeralStorage(), other.fargateEphemeralStorage())
                && Objects.equals(createdAt(), other.createdAt())
                && hasVpcLatticeConfigurations() == other.hasVpcLatticeConfigurations()
                && Objects.equals(vpcLatticeConfigurations(), other.vpcLatticeConfigurations());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ServiceRevision").add("ServiceRevisionArn", serviceRevisionArn())
                .add("ServiceArn", serviceArn()).add("ClusterArn", clusterArn()).add("TaskDefinition", taskDefinition())
                .add("CapacityProviderStrategy", hasCapacityProviderStrategy() ? capacityProviderStrategy() : null)
                .add("LaunchType", launchTypeAsString()).add("PlatformVersion", platformVersion())
                .add("PlatformFamily", platformFamily()).add("LoadBalancers", hasLoadBalancers() ? loadBalancers() : null)
                .add("ServiceRegistries", hasServiceRegistries() ? serviceRegistries() : null)
                .add("NetworkConfiguration", networkConfiguration())
                .add("ContainerImages", hasContainerImages() ? containerImages() : null)
                .add("GuardDutyEnabled", guardDutyEnabled()).add("ServiceConnectConfiguration", serviceConnectConfiguration())
                .add("VolumeConfigurations", hasVolumeConfigurations() ? volumeConfigurations() : null)
                .add("FargateEphemeralStorage", fargateEphemeralStorage()).add("CreatedAt", createdAt())
                .add("VpcLatticeConfigurations", hasVpcLatticeConfigurations() ? vpcLatticeConfigurations() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "serviceRevisionArn":
            return Optional.ofNullable(clazz.cast(serviceRevisionArn()));
        case "serviceArn":
            return Optional.ofNullable(clazz.cast(serviceArn()));
        case "clusterArn":
            return Optional.ofNullable(clazz.cast(clusterArn()));
        case "taskDefinition":
            return Optional.ofNullable(clazz.cast(taskDefinition()));
        case "capacityProviderStrategy":
            return Optional.ofNullable(clazz.cast(capacityProviderStrategy()));
        case "launchType":
            return Optional.ofNullable(clazz.cast(launchTypeAsString()));
        case "platformVersion":
            return Optional.ofNullable(clazz.cast(platformVersion()));
        case "platformFamily":
            return Optional.ofNullable(clazz.cast(platformFamily()));
        case "loadBalancers":
            return Optional.ofNullable(clazz.cast(loadBalancers()));
        case "serviceRegistries":
            return Optional.ofNullable(clazz.cast(serviceRegistries()));
        case "networkConfiguration":
            return Optional.ofNullable(clazz.cast(networkConfiguration()));
        case "containerImages":
            return Optional.ofNullable(clazz.cast(containerImages()));
        case "guardDutyEnabled":
            return Optional.ofNullable(clazz.cast(guardDutyEnabled()));
        case "serviceConnectConfiguration":
            return Optional.ofNullable(clazz.cast(serviceConnectConfiguration()));
        case "volumeConfigurations":
            return Optional.ofNullable(clazz.cast(volumeConfigurations()));
        case "fargateEphemeralStorage":
            return Optional.ofNullable(clazz.cast(fargateEphemeralStorage()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "vpcLatticeConfigurations":
            return Optional.ofNullable(clazz.cast(vpcLatticeConfigurations()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("serviceRevisionArn", SERVICE_REVISION_ARN_FIELD);
        map.put("serviceArn", SERVICE_ARN_FIELD);
        map.put("clusterArn", CLUSTER_ARN_FIELD);
        map.put("taskDefinition", TASK_DEFINITION_FIELD);
        map.put("capacityProviderStrategy", CAPACITY_PROVIDER_STRATEGY_FIELD);
        map.put("launchType", LAUNCH_TYPE_FIELD);
        map.put("platformVersion", PLATFORM_VERSION_FIELD);
        map.put("platformFamily", PLATFORM_FAMILY_FIELD);
        map.put("loadBalancers", LOAD_BALANCERS_FIELD);
        map.put("serviceRegistries", SERVICE_REGISTRIES_FIELD);
        map.put("networkConfiguration", NETWORK_CONFIGURATION_FIELD);
        map.put("containerImages", CONTAINER_IMAGES_FIELD);
        map.put("guardDutyEnabled", GUARD_DUTY_ENABLED_FIELD);
        map.put("serviceConnectConfiguration", SERVICE_CONNECT_CONFIGURATION_FIELD);
        map.put("volumeConfigurations", VOLUME_CONFIGURATIONS_FIELD);
        map.put("fargateEphemeralStorage", FARGATE_EPHEMERAL_STORAGE_FIELD);
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("vpcLatticeConfigurations", VPC_LATTICE_CONFIGURATIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ServiceRevision, T> g) {
        return obj -> g.apply((ServiceRevision) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ServiceRevision> {
        /**
         * <p>
         * The ARN of the service revision.
         * </p>
         * 
         * @param serviceRevisionArn
         *        The ARN of the service revision.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceRevisionArn(String serviceRevisionArn);

        /**
         * <p>
         * The ARN of the service for the service revision.
         * </p>
         * 
         * @param serviceArn
         *        The ARN of the service for the service revision.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceArn(String serviceArn);

        /**
         * <p>
         * The ARN of the cluster that hosts the service.
         * </p>
         * 
         * @param clusterArn
         *        The ARN of the cluster that hosts the service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterArn(String clusterArn);

        /**
         * <p>
         * The task definition the service revision uses.
         * </p>
         * 
         * @param taskDefinition
         *        The task definition the service revision uses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskDefinition(String taskDefinition);

        /**
         * <p>
         * The capacity provider strategy the service revision uses.
         * </p>
         * 
         * @param capacityProviderStrategy
         *        The capacity provider strategy the service revision uses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityProviderStrategy(Collection<CapacityProviderStrategyItem> capacityProviderStrategy);

        /**
         * <p>
         * The capacity provider strategy the service revision uses.
         * </p>
         * 
         * @param capacityProviderStrategy
         *        The capacity provider strategy the service revision uses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityProviderStrategy(CapacityProviderStrategyItem... capacityProviderStrategy);

        /**
         * <p>
         * The capacity provider strategy the service revision uses.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.CapacityProviderStrategyItem.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.ecs.model.CapacityProviderStrategyItem#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecs.model.CapacityProviderStrategyItem.Builder#build()} is called
         * immediately and its result is passed to {@link #capacityProviderStrategy(List<CapacityProviderStrategyItem>)}.
         * 
         * @param capacityProviderStrategy
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecs.model.CapacityProviderStrategyItem.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #capacityProviderStrategy(java.util.Collection<CapacityProviderStrategyItem>)
         */
        Builder capacityProviderStrategy(Consumer<CapacityProviderStrategyItem.Builder>... capacityProviderStrategy);

        /**
         * <p>
         * The launch type the service revision uses.
         * </p>
         * 
         * @param launchType
         *        The launch type the service revision uses.
         * @see LaunchType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LaunchType
         */
        Builder launchType(String launchType);

        /**
         * <p>
         * The launch type the service revision uses.
         * </p>
         * 
         * @param launchType
         *        The launch type the service revision uses.
         * @see LaunchType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LaunchType
         */
        Builder launchType(LaunchType launchType);

        /**
         * <p>
         * For the Fargate launch type, the platform version the service revision uses.
         * </p>
         * 
         * @param platformVersion
         *        For the Fargate launch type, the platform version the service revision uses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformVersion(String platformVersion);

        /**
         * <p>
         * The platform family the service revision uses.
         * </p>
         * 
         * @param platformFamily
         *        The platform family the service revision uses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformFamily(String platformFamily);

        /**
         * <p>
         * The load balancers the service revision uses.
         * </p>
         * 
         * @param loadBalancers
         *        The load balancers the service revision uses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loadBalancers(Collection<LoadBalancer> loadBalancers);

        /**
         * <p>
         * The load balancers the service revision uses.
         * </p>
         * 
         * @param loadBalancers
         *        The load balancers the service revision uses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loadBalancers(LoadBalancer... loadBalancers);

        /**
         * <p>
         * The load balancers the service revision uses.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.LoadBalancer.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ecs.model.LoadBalancer#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecs.model.LoadBalancer.Builder#build()} is called immediately and its
         * result is passed to {@link #loadBalancers(List<LoadBalancer>)}.
         * 
         * @param loadBalancers
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecs.model.LoadBalancer.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #loadBalancers(java.util.Collection<LoadBalancer>)
         */
        Builder loadBalancers(Consumer<LoadBalancer.Builder>... loadBalancers);

        /**
         * <p>
         * The service registries (for Service Discovery) the service revision uses.
         * </p>
         * 
         * @param serviceRegistries
         *        The service registries (for Service Discovery) the service revision uses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceRegistries(Collection<ServiceRegistry> serviceRegistries);

        /**
         * <p>
         * The service registries (for Service Discovery) the service revision uses.
         * </p>
         * 
         * @param serviceRegistries
         *        The service registries (for Service Discovery) the service revision uses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceRegistries(ServiceRegistry... serviceRegistries);

        /**
         * <p>
         * The service registries (for Service Discovery) the service revision uses.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.ServiceRegistry.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ecs.model.ServiceRegistry#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecs.model.ServiceRegistry.Builder#build()} is called immediately and
         * its result is passed to {@link #serviceRegistries(List<ServiceRegistry>)}.
         * 
         * @param serviceRegistries
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecs.model.ServiceRegistry.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #serviceRegistries(java.util.Collection<ServiceRegistry>)
         */
        Builder serviceRegistries(Consumer<ServiceRegistry.Builder>... serviceRegistries);

        /**
         * Sets the value of the NetworkConfiguration property for this object.
         *
         * @param networkConfiguration
         *        The new value for the NetworkConfiguration property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkConfiguration(NetworkConfiguration networkConfiguration);

        /**
         * Sets the value of the NetworkConfiguration property for this object.
         *
         * This is a convenience method that creates an instance of the {@link NetworkConfiguration.Builder} avoiding
         * the need to create one manually via {@link NetworkConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link NetworkConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #networkConfiguration(NetworkConfiguration)}.
         * 
         * @param networkConfiguration
         *        a consumer that will call methods on {@link NetworkConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkConfiguration(NetworkConfiguration)
         */
        default Builder networkConfiguration(Consumer<NetworkConfiguration.Builder> networkConfiguration) {
            return networkConfiguration(NetworkConfiguration.builder().applyMutation(networkConfiguration).build());
        }

        /**
         * <p>
         * The container images the service revision uses.
         * </p>
         * 
         * @param containerImages
         *        The container images the service revision uses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerImages(Collection<ContainerImage> containerImages);

        /**
         * <p>
         * The container images the service revision uses.
         * </p>
         * 
         * @param containerImages
         *        The container images the service revision uses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerImages(ContainerImage... containerImages);

        /**
         * <p>
         * The container images the service revision uses.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.ContainerImage.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ecs.model.ContainerImage#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecs.model.ContainerImage.Builder#build()} is called immediately and
         * its result is passed to {@link #containerImages(List<ContainerImage>)}.
         * 
         * @param containerImages
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecs.model.ContainerImage.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #containerImages(java.util.Collection<ContainerImage>)
         */
        Builder containerImages(Consumer<ContainerImage.Builder>... containerImages);

        /**
         * <p>
         * Indicates whether Runtime Monitoring is turned on.
         * </p>
         * 
         * @param guardDutyEnabled
         *        Indicates whether Runtime Monitoring is turned on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder guardDutyEnabled(Boolean guardDutyEnabled);

        /**
         * Sets the value of the ServiceConnectConfiguration property for this object.
         *
         * @param serviceConnectConfiguration
         *        The new value for the ServiceConnectConfiguration property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceConnectConfiguration(ServiceConnectConfiguration serviceConnectConfiguration);

        /**
         * Sets the value of the ServiceConnectConfiguration property for this object.
         *
         * This is a convenience method that creates an instance of the {@link ServiceConnectConfiguration.Builder}
         * avoiding the need to create one manually via {@link ServiceConnectConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ServiceConnectConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #serviceConnectConfiguration(ServiceConnectConfiguration)}.
         * 
         * @param serviceConnectConfiguration
         *        a consumer that will call methods on {@link ServiceConnectConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #serviceConnectConfiguration(ServiceConnectConfiguration)
         */
        default Builder serviceConnectConfiguration(Consumer<ServiceConnectConfiguration.Builder> serviceConnectConfiguration) {
            return serviceConnectConfiguration(ServiceConnectConfiguration.builder().applyMutation(serviceConnectConfiguration)
                    .build());
        }

        /**
         * <p>
         * The volumes that are configured at deployment that the service revision uses.
         * </p>
         * 
         * @param volumeConfigurations
         *        The volumes that are configured at deployment that the service revision uses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeConfigurations(Collection<ServiceVolumeConfiguration> volumeConfigurations);

        /**
         * <p>
         * The volumes that are configured at deployment that the service revision uses.
         * </p>
         * 
         * @param volumeConfigurations
         *        The volumes that are configured at deployment that the service revision uses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeConfigurations(ServiceVolumeConfiguration... volumeConfigurations);

        /**
         * <p>
         * The volumes that are configured at deployment that the service revision uses.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.ServiceVolumeConfiguration.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.ecs.model.ServiceVolumeConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecs.model.ServiceVolumeConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #volumeConfigurations(List<ServiceVolumeConfiguration>)}.
         * 
         * @param volumeConfigurations
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecs.model.ServiceVolumeConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #volumeConfigurations(java.util.Collection<ServiceVolumeConfiguration>)
         */
        Builder volumeConfigurations(Consumer<ServiceVolumeConfiguration.Builder>... volumeConfigurations);

        /**
         * Sets the value of the FargateEphemeralStorage property for this object.
         *
         * @param fargateEphemeralStorage
         *        The new value for the FargateEphemeralStorage property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fargateEphemeralStorage(DeploymentEphemeralStorage fargateEphemeralStorage);

        /**
         * Sets the value of the FargateEphemeralStorage property for this object.
         *
         * This is a convenience method that creates an instance of the {@link DeploymentEphemeralStorage.Builder}
         * avoiding the need to create one manually via {@link DeploymentEphemeralStorage#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DeploymentEphemeralStorage.Builder#build()} is called immediately
         * and its result is passed to {@link #fargateEphemeralStorage(DeploymentEphemeralStorage)}.
         * 
         * @param fargateEphemeralStorage
         *        a consumer that will call methods on {@link DeploymentEphemeralStorage.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #fargateEphemeralStorage(DeploymentEphemeralStorage)
         */
        default Builder fargateEphemeralStorage(Consumer<DeploymentEphemeralStorage.Builder> fargateEphemeralStorage) {
            return fargateEphemeralStorage(DeploymentEphemeralStorage.builder().applyMutation(fargateEphemeralStorage).build());
        }

        /**
         * <p>
         * The time that the service revision was created. The format is yyyy-mm-dd HH:mm:ss.SSSSS.
         * </p>
         * 
         * @param createdAt
         *        The time that the service revision was created. The format is yyyy-mm-dd HH:mm:ss.SSSSS.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The VPC Lattice configuration for the service revision.
         * </p>
         * 
         * @param vpcLatticeConfigurations
         *        The VPC Lattice configuration for the service revision.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcLatticeConfigurations(Collection<VpcLatticeConfiguration> vpcLatticeConfigurations);

        /**
         * <p>
         * The VPC Lattice configuration for the service revision.
         * </p>
         * 
         * @param vpcLatticeConfigurations
         *        The VPC Lattice configuration for the service revision.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcLatticeConfigurations(VpcLatticeConfiguration... vpcLatticeConfigurations);

        /**
         * <p>
         * The VPC Lattice configuration for the service revision.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.VpcLatticeConfiguration.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.ecs.model.VpcLatticeConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecs.model.VpcLatticeConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #vpcLatticeConfigurations(List<VpcLatticeConfiguration>)}.
         * 
         * @param vpcLatticeConfigurations
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecs.model.VpcLatticeConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcLatticeConfigurations(java.util.Collection<VpcLatticeConfiguration>)
         */
        Builder vpcLatticeConfigurations(Consumer<VpcLatticeConfiguration.Builder>... vpcLatticeConfigurations);
    }

    static final class BuilderImpl implements Builder {
        private String serviceRevisionArn;

        private String serviceArn;

        private String clusterArn;

        private String taskDefinition;

        private List<CapacityProviderStrategyItem> capacityProviderStrategy = DefaultSdkAutoConstructList.getInstance();

        private String launchType;

        private String platformVersion;

        private String platformFamily;

        private List<LoadBalancer> loadBalancers = DefaultSdkAutoConstructList.getInstance();

        private List<ServiceRegistry> serviceRegistries = DefaultSdkAutoConstructList.getInstance();

        private NetworkConfiguration networkConfiguration;

        private List<ContainerImage> containerImages = DefaultSdkAutoConstructList.getInstance();

        private Boolean guardDutyEnabled;

        private ServiceConnectConfiguration serviceConnectConfiguration;

        private List<ServiceVolumeConfiguration> volumeConfigurations = DefaultSdkAutoConstructList.getInstance();

        private DeploymentEphemeralStorage fargateEphemeralStorage;

        private Instant createdAt;

        private List<VpcLatticeConfiguration> vpcLatticeConfigurations = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ServiceRevision model) {
            serviceRevisionArn(model.serviceRevisionArn);
            serviceArn(model.serviceArn);
            clusterArn(model.clusterArn);
            taskDefinition(model.taskDefinition);
            capacityProviderStrategy(model.capacityProviderStrategy);
            launchType(model.launchType);
            platformVersion(model.platformVersion);
            platformFamily(model.platformFamily);
            loadBalancers(model.loadBalancers);
            serviceRegistries(model.serviceRegistries);
            networkConfiguration(model.networkConfiguration);
            containerImages(model.containerImages);
            guardDutyEnabled(model.guardDutyEnabled);
            serviceConnectConfiguration(model.serviceConnectConfiguration);
            volumeConfigurations(model.volumeConfigurations);
            fargateEphemeralStorage(model.fargateEphemeralStorage);
            createdAt(model.createdAt);
            vpcLatticeConfigurations(model.vpcLatticeConfigurations);
        }

        public final String getServiceRevisionArn() {
            return serviceRevisionArn;
        }

        public final void setServiceRevisionArn(String serviceRevisionArn) {
            this.serviceRevisionArn = serviceRevisionArn;
        }

        @Override
        public final Builder serviceRevisionArn(String serviceRevisionArn) {
            this.serviceRevisionArn = serviceRevisionArn;
            return this;
        }

        public final String getServiceArn() {
            return serviceArn;
        }

        public final void setServiceArn(String serviceArn) {
            this.serviceArn = serviceArn;
        }

        @Override
        public final Builder serviceArn(String serviceArn) {
            this.serviceArn = serviceArn;
            return this;
        }

        public final String getClusterArn() {
            return clusterArn;
        }

        public final void setClusterArn(String clusterArn) {
            this.clusterArn = clusterArn;
        }

        @Override
        public final Builder clusterArn(String clusterArn) {
            this.clusterArn = clusterArn;
            return this;
        }

        public final String getTaskDefinition() {
            return taskDefinition;
        }

        public final void setTaskDefinition(String taskDefinition) {
            this.taskDefinition = taskDefinition;
        }

        @Override
        public final Builder taskDefinition(String taskDefinition) {
            this.taskDefinition = taskDefinition;
            return this;
        }

        public final List<CapacityProviderStrategyItem.Builder> getCapacityProviderStrategy() {
            List<CapacityProviderStrategyItem.Builder> result = CapacityProviderStrategyCopier
                    .copyToBuilder(this.capacityProviderStrategy);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setCapacityProviderStrategy(
                Collection<CapacityProviderStrategyItem.BuilderImpl> capacityProviderStrategy) {
            this.capacityProviderStrategy = CapacityProviderStrategyCopier.copyFromBuilder(capacityProviderStrategy);
        }

        @Override
        public final Builder capacityProviderStrategy(Collection<CapacityProviderStrategyItem> capacityProviderStrategy) {
            this.capacityProviderStrategy = CapacityProviderStrategyCopier.copy(capacityProviderStrategy);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder capacityProviderStrategy(CapacityProviderStrategyItem... capacityProviderStrategy) {
            capacityProviderStrategy(Arrays.asList(capacityProviderStrategy));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder capacityProviderStrategy(Consumer<CapacityProviderStrategyItem.Builder>... capacityProviderStrategy) {
            capacityProviderStrategy(Stream.of(capacityProviderStrategy)
                    .map(c -> CapacityProviderStrategyItem.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getLaunchType() {
            return launchType;
        }

        public final void setLaunchType(String launchType) {
            this.launchType = launchType;
        }

        @Override
        public final Builder launchType(String launchType) {
            this.launchType = launchType;
            return this;
        }

        @Override
        public final Builder launchType(LaunchType launchType) {
            this.launchType(launchType == null ? null : launchType.toString());
            return this;
        }

        public final String getPlatformVersion() {
            return platformVersion;
        }

        public final void setPlatformVersion(String platformVersion) {
            this.platformVersion = platformVersion;
        }

        @Override
        public final Builder platformVersion(String platformVersion) {
            this.platformVersion = platformVersion;
            return this;
        }

        public final String getPlatformFamily() {
            return platformFamily;
        }

        public final void setPlatformFamily(String platformFamily) {
            this.platformFamily = platformFamily;
        }

        @Override
        public final Builder platformFamily(String platformFamily) {
            this.platformFamily = platformFamily;
            return this;
        }

        public final List<LoadBalancer.Builder> getLoadBalancers() {
            List<LoadBalancer.Builder> result = LoadBalancersCopier.copyToBuilder(this.loadBalancers);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setLoadBalancers(Collection<LoadBalancer.BuilderImpl> loadBalancers) {
            this.loadBalancers = LoadBalancersCopier.copyFromBuilder(loadBalancers);
        }

        @Override
        public final Builder loadBalancers(Collection<LoadBalancer> loadBalancers) {
            this.loadBalancers = LoadBalancersCopier.copy(loadBalancers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder loadBalancers(LoadBalancer... loadBalancers) {
            loadBalancers(Arrays.asList(loadBalancers));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder loadBalancers(Consumer<LoadBalancer.Builder>... loadBalancers) {
            loadBalancers(Stream.of(loadBalancers).map(c -> LoadBalancer.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<ServiceRegistry.Builder> getServiceRegistries() {
            List<ServiceRegistry.Builder> result = ServiceRegistriesCopier.copyToBuilder(this.serviceRegistries);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setServiceRegistries(Collection<ServiceRegistry.BuilderImpl> serviceRegistries) {
            this.serviceRegistries = ServiceRegistriesCopier.copyFromBuilder(serviceRegistries);
        }

        @Override
        public final Builder serviceRegistries(Collection<ServiceRegistry> serviceRegistries) {
            this.serviceRegistries = ServiceRegistriesCopier.copy(serviceRegistries);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder serviceRegistries(ServiceRegistry... serviceRegistries) {
            serviceRegistries(Arrays.asList(serviceRegistries));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder serviceRegistries(Consumer<ServiceRegistry.Builder>... serviceRegistries) {
            serviceRegistries(Stream.of(serviceRegistries).map(c -> ServiceRegistry.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final NetworkConfiguration.Builder getNetworkConfiguration() {
            return networkConfiguration != null ? networkConfiguration.toBuilder() : null;
        }

        public final void setNetworkConfiguration(NetworkConfiguration.BuilderImpl networkConfiguration) {
            this.networkConfiguration = networkConfiguration != null ? networkConfiguration.build() : null;
        }

        @Override
        public final Builder networkConfiguration(NetworkConfiguration networkConfiguration) {
            this.networkConfiguration = networkConfiguration;
            return this;
        }

        public final List<ContainerImage.Builder> getContainerImages() {
            List<ContainerImage.Builder> result = ContainerImagesCopier.copyToBuilder(this.containerImages);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setContainerImages(Collection<ContainerImage.BuilderImpl> containerImages) {
            this.containerImages = ContainerImagesCopier.copyFromBuilder(containerImages);
        }

        @Override
        public final Builder containerImages(Collection<ContainerImage> containerImages) {
            this.containerImages = ContainerImagesCopier.copy(containerImages);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder containerImages(ContainerImage... containerImages) {
            containerImages(Arrays.asList(containerImages));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder containerImages(Consumer<ContainerImage.Builder>... containerImages) {
            containerImages(Stream.of(containerImages).map(c -> ContainerImage.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Boolean getGuardDutyEnabled() {
            return guardDutyEnabled;
        }

        public final void setGuardDutyEnabled(Boolean guardDutyEnabled) {
            this.guardDutyEnabled = guardDutyEnabled;
        }

        @Override
        public final Builder guardDutyEnabled(Boolean guardDutyEnabled) {
            this.guardDutyEnabled = guardDutyEnabled;
            return this;
        }

        public final ServiceConnectConfiguration.Builder getServiceConnectConfiguration() {
            return serviceConnectConfiguration != null ? serviceConnectConfiguration.toBuilder() : null;
        }

        public final void setServiceConnectConfiguration(ServiceConnectConfiguration.BuilderImpl serviceConnectConfiguration) {
            this.serviceConnectConfiguration = serviceConnectConfiguration != null ? serviceConnectConfiguration.build() : null;
        }

        @Override
        public final Builder serviceConnectConfiguration(ServiceConnectConfiguration serviceConnectConfiguration) {
            this.serviceConnectConfiguration = serviceConnectConfiguration;
            return this;
        }

        public final List<ServiceVolumeConfiguration.Builder> getVolumeConfigurations() {
            List<ServiceVolumeConfiguration.Builder> result = ServiceVolumeConfigurationsCopier
                    .copyToBuilder(this.volumeConfigurations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setVolumeConfigurations(Collection<ServiceVolumeConfiguration.BuilderImpl> volumeConfigurations) {
            this.volumeConfigurations = ServiceVolumeConfigurationsCopier.copyFromBuilder(volumeConfigurations);
        }

        @Override
        public final Builder volumeConfigurations(Collection<ServiceVolumeConfiguration> volumeConfigurations) {
            this.volumeConfigurations = ServiceVolumeConfigurationsCopier.copy(volumeConfigurations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder volumeConfigurations(ServiceVolumeConfiguration... volumeConfigurations) {
            volumeConfigurations(Arrays.asList(volumeConfigurations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder volumeConfigurations(Consumer<ServiceVolumeConfiguration.Builder>... volumeConfigurations) {
            volumeConfigurations(Stream.of(volumeConfigurations)
                    .map(c -> ServiceVolumeConfiguration.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final DeploymentEphemeralStorage.Builder getFargateEphemeralStorage() {
            return fargateEphemeralStorage != null ? fargateEphemeralStorage.toBuilder() : null;
        }

        public final void setFargateEphemeralStorage(DeploymentEphemeralStorage.BuilderImpl fargateEphemeralStorage) {
            this.fargateEphemeralStorage = fargateEphemeralStorage != null ? fargateEphemeralStorage.build() : null;
        }

        @Override
        public final Builder fargateEphemeralStorage(DeploymentEphemeralStorage fargateEphemeralStorage) {
            this.fargateEphemeralStorage = fargateEphemeralStorage;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final List<VpcLatticeConfiguration.Builder> getVpcLatticeConfigurations() {
            List<VpcLatticeConfiguration.Builder> result = VpcLatticeConfigurationsCopier
                    .copyToBuilder(this.vpcLatticeConfigurations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setVpcLatticeConfigurations(Collection<VpcLatticeConfiguration.BuilderImpl> vpcLatticeConfigurations) {
            this.vpcLatticeConfigurations = VpcLatticeConfigurationsCopier.copyFromBuilder(vpcLatticeConfigurations);
        }

        @Override
        public final Builder vpcLatticeConfigurations(Collection<VpcLatticeConfiguration> vpcLatticeConfigurations) {
            this.vpcLatticeConfigurations = VpcLatticeConfigurationsCopier.copy(vpcLatticeConfigurations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder vpcLatticeConfigurations(VpcLatticeConfiguration... vpcLatticeConfigurations) {
            vpcLatticeConfigurations(Arrays.asList(vpcLatticeConfigurations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder vpcLatticeConfigurations(Consumer<VpcLatticeConfiguration.Builder>... vpcLatticeConfigurations) {
            vpcLatticeConfigurations(Stream.of(vpcLatticeConfigurations)
                    .map(c -> VpcLatticeConfiguration.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public ServiceRevision build() {
            return new ServiceRevision(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
