/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Container definitions are used in task definitions to describe the different containers that are launched as part of
 * a task.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ContainerDefinition implements SdkPojo, Serializable,
        ToCopyableBuilder<ContainerDefinition.Builder, ContainerDefinition> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(ContainerDefinition::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> IMAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("image")
            .getter(getter(ContainerDefinition::image)).setter(setter(Builder::image))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("image").build()).build();

    private static final SdkField<RepositoryCredentials> REPOSITORY_CREDENTIALS_FIELD = SdkField
            .<RepositoryCredentials> builder(MarshallingType.SDK_POJO).memberName("repositoryCredentials")
            .getter(getter(ContainerDefinition::repositoryCredentials)).setter(setter(Builder::repositoryCredentials))
            .constructor(RepositoryCredentials::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("repositoryCredentials").build())
            .build();

    private static final SdkField<Integer> CPU_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("cpu")
            .getter(getter(ContainerDefinition::cpu)).setter(setter(Builder::cpu))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cpu").build()).build();

    private static final SdkField<Integer> MEMORY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("memory").getter(getter(ContainerDefinition::memory)).setter(setter(Builder::memory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("memory").build()).build();

    private static final SdkField<Integer> MEMORY_RESERVATION_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("memoryReservation").getter(getter(ContainerDefinition::memoryReservation))
            .setter(setter(Builder::memoryReservation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("memoryReservation").build()).build();

    private static final SdkField<List<String>> LINKS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("links")
            .getter(getter(ContainerDefinition::links))
            .setter(setter(Builder::links))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("links").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<PortMapping>> PORT_MAPPINGS_FIELD = SdkField
            .<List<PortMapping>> builder(MarshallingType.LIST)
            .memberName("portMappings")
            .getter(getter(ContainerDefinition::portMappings))
            .setter(setter(Builder::portMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("portMappings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PortMapping> builder(MarshallingType.SDK_POJO)
                                            .constructor(PortMapping::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> ESSENTIAL_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("essential").getter(getter(ContainerDefinition::essential)).setter(setter(Builder::essential))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("essential").build()).build();

    private static final SdkField<ContainerRestartPolicy> RESTART_POLICY_FIELD = SdkField
            .<ContainerRestartPolicy> builder(MarshallingType.SDK_POJO).memberName("restartPolicy")
            .getter(getter(ContainerDefinition::restartPolicy)).setter(setter(Builder::restartPolicy))
            .constructor(ContainerRestartPolicy::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("restartPolicy").build()).build();

    private static final SdkField<List<String>> ENTRY_POINT_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("entryPoint")
            .getter(getter(ContainerDefinition::entryPoint))
            .setter(setter(Builder::entryPoint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("entryPoint").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> COMMAND_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("command")
            .getter(getter(ContainerDefinition::command))
            .setter(setter(Builder::command))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("command").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<KeyValuePair>> ENVIRONMENT_FIELD = SdkField
            .<List<KeyValuePair>> builder(MarshallingType.LIST)
            .memberName("environment")
            .getter(getter(ContainerDefinition::environment))
            .setter(setter(Builder::environment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("environment").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<KeyValuePair> builder(MarshallingType.SDK_POJO)
                                            .constructor(KeyValuePair::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<EnvironmentFile>> ENVIRONMENT_FILES_FIELD = SdkField
            .<List<EnvironmentFile>> builder(MarshallingType.LIST)
            .memberName("environmentFiles")
            .getter(getter(ContainerDefinition::environmentFiles))
            .setter(setter(Builder::environmentFiles))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("environmentFiles").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EnvironmentFile> builder(MarshallingType.SDK_POJO)
                                            .constructor(EnvironmentFile::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<MountPoint>> MOUNT_POINTS_FIELD = SdkField
            .<List<MountPoint>> builder(MarshallingType.LIST)
            .memberName("mountPoints")
            .getter(getter(ContainerDefinition::mountPoints))
            .setter(setter(Builder::mountPoints))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("mountPoints").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MountPoint> builder(MarshallingType.SDK_POJO)
                                            .constructor(MountPoint::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<VolumeFrom>> VOLUMES_FROM_FIELD = SdkField
            .<List<VolumeFrom>> builder(MarshallingType.LIST)
            .memberName("volumesFrom")
            .getter(getter(ContainerDefinition::volumesFrom))
            .setter(setter(Builder::volumesFrom))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("volumesFrom").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<VolumeFrom> builder(MarshallingType.SDK_POJO)
                                            .constructor(VolumeFrom::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<LinuxParameters> LINUX_PARAMETERS_FIELD = SdkField
            .<LinuxParameters> builder(MarshallingType.SDK_POJO).memberName("linuxParameters")
            .getter(getter(ContainerDefinition::linuxParameters)).setter(setter(Builder::linuxParameters))
            .constructor(LinuxParameters::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("linuxParameters").build()).build();

    private static final SdkField<List<Secret>> SECRETS_FIELD = SdkField
            .<List<Secret>> builder(MarshallingType.LIST)
            .memberName("secrets")
            .getter(getter(ContainerDefinition::secrets))
            .setter(setter(Builder::secrets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("secrets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Secret> builder(MarshallingType.SDK_POJO)
                                            .constructor(Secret::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ContainerDependency>> DEPENDS_ON_FIELD = SdkField
            .<List<ContainerDependency>> builder(MarshallingType.LIST)
            .memberName("dependsOn")
            .getter(getter(ContainerDefinition::dependsOn))
            .setter(setter(Builder::dependsOn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dependsOn").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ContainerDependency> builder(MarshallingType.SDK_POJO)
                                            .constructor(ContainerDependency::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> START_TIMEOUT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("startTimeout").getter(getter(ContainerDefinition::startTimeout)).setter(setter(Builder::startTimeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTimeout").build()).build();

    private static final SdkField<Integer> STOP_TIMEOUT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("stopTimeout").getter(getter(ContainerDefinition::stopTimeout)).setter(setter(Builder::stopTimeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("stopTimeout").build()).build();

    private static final SdkField<String> VERSION_CONSISTENCY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("versionConsistency").getter(getter(ContainerDefinition::versionConsistencyAsString))
            .setter(setter(Builder::versionConsistency))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("versionConsistency").build())
            .build();

    private static final SdkField<String> HOSTNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("hostname").getter(getter(ContainerDefinition::hostname)).setter(setter(Builder::hostname))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("hostname").build()).build();

    private static final SdkField<String> USER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("user")
            .getter(getter(ContainerDefinition::user)).setter(setter(Builder::user))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("user").build()).build();

    private static final SdkField<String> WORKING_DIRECTORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("workingDirectory").getter(getter(ContainerDefinition::workingDirectory))
            .setter(setter(Builder::workingDirectory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("workingDirectory").build()).build();

    private static final SdkField<Boolean> DISABLE_NETWORKING_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("disableNetworking").getter(getter(ContainerDefinition::disableNetworking))
            .setter(setter(Builder::disableNetworking))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("disableNetworking").build()).build();

    private static final SdkField<Boolean> PRIVILEGED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("privileged").getter(getter(ContainerDefinition::privileged)).setter(setter(Builder::privileged))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("privileged").build()).build();

    private static final SdkField<Boolean> READONLY_ROOT_FILESYSTEM_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("readonlyRootFilesystem").getter(getter(ContainerDefinition::readonlyRootFilesystem))
            .setter(setter(Builder::readonlyRootFilesystem))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("readonlyRootFilesystem").build())
            .build();

    private static final SdkField<List<String>> DNS_SERVERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("dnsServers")
            .getter(getter(ContainerDefinition::dnsServers))
            .setter(setter(Builder::dnsServers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dnsServers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> DNS_SEARCH_DOMAINS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("dnsSearchDomains")
            .getter(getter(ContainerDefinition::dnsSearchDomains))
            .setter(setter(Builder::dnsSearchDomains))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dnsSearchDomains").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<HostEntry>> EXTRA_HOSTS_FIELD = SdkField
            .<List<HostEntry>> builder(MarshallingType.LIST)
            .memberName("extraHosts")
            .getter(getter(ContainerDefinition::extraHosts))
            .setter(setter(Builder::extraHosts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("extraHosts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<HostEntry> builder(MarshallingType.SDK_POJO)
                                            .constructor(HostEntry::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> DOCKER_SECURITY_OPTIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("dockerSecurityOptions")
            .getter(getter(ContainerDefinition::dockerSecurityOptions))
            .setter(setter(Builder::dockerSecurityOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dockerSecurityOptions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> INTERACTIVE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("interactive").getter(getter(ContainerDefinition::interactive)).setter(setter(Builder::interactive))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("interactive").build()).build();

    private static final SdkField<Boolean> PSEUDO_TERMINAL_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("pseudoTerminal").getter(getter(ContainerDefinition::pseudoTerminal))
            .setter(setter(Builder::pseudoTerminal))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("pseudoTerminal").build()).build();

    private static final SdkField<Map<String, String>> DOCKER_LABELS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("dockerLabels")
            .getter(getter(ContainerDefinition::dockerLabels))
            .setter(setter(Builder::dockerLabels))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dockerLabels").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<Ulimit>> ULIMITS_FIELD = SdkField
            .<List<Ulimit>> builder(MarshallingType.LIST)
            .memberName("ulimits")
            .getter(getter(ContainerDefinition::ulimits))
            .setter(setter(Builder::ulimits))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ulimits").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Ulimit> builder(MarshallingType.SDK_POJO)
                                            .constructor(Ulimit::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<LogConfiguration> LOG_CONFIGURATION_FIELD = SdkField
            .<LogConfiguration> builder(MarshallingType.SDK_POJO).memberName("logConfiguration")
            .getter(getter(ContainerDefinition::logConfiguration)).setter(setter(Builder::logConfiguration))
            .constructor(LogConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("logConfiguration").build()).build();

    private static final SdkField<HealthCheck> HEALTH_CHECK_FIELD = SdkField.<HealthCheck> builder(MarshallingType.SDK_POJO)
            .memberName("healthCheck").getter(getter(ContainerDefinition::healthCheck)).setter(setter(Builder::healthCheck))
            .constructor(HealthCheck::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("healthCheck").build()).build();

    private static final SdkField<List<SystemControl>> SYSTEM_CONTROLS_FIELD = SdkField
            .<List<SystemControl>> builder(MarshallingType.LIST)
            .memberName("systemControls")
            .getter(getter(ContainerDefinition::systemControls))
            .setter(setter(Builder::systemControls))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("systemControls").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SystemControl> builder(MarshallingType.SDK_POJO)
                                            .constructor(SystemControl::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ResourceRequirement>> RESOURCE_REQUIREMENTS_FIELD = SdkField
            .<List<ResourceRequirement>> builder(MarshallingType.LIST)
            .memberName("resourceRequirements")
            .getter(getter(ContainerDefinition::resourceRequirements))
            .setter(setter(Builder::resourceRequirements))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceRequirements").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ResourceRequirement> builder(MarshallingType.SDK_POJO)
                                            .constructor(ResourceRequirement::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<FirelensConfiguration> FIRELENS_CONFIGURATION_FIELD = SdkField
            .<FirelensConfiguration> builder(MarshallingType.SDK_POJO).memberName("firelensConfiguration")
            .getter(getter(ContainerDefinition::firelensConfiguration)).setter(setter(Builder::firelensConfiguration))
            .constructor(FirelensConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("firelensConfiguration").build())
            .build();

    private static final SdkField<List<String>> CREDENTIAL_SPECS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("credentialSpecs")
            .getter(getter(ContainerDefinition::credentialSpecs))
            .setter(setter(Builder::credentialSpecs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("credentialSpecs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, IMAGE_FIELD,
            REPOSITORY_CREDENTIALS_FIELD, CPU_FIELD, MEMORY_FIELD, MEMORY_RESERVATION_FIELD, LINKS_FIELD, PORT_MAPPINGS_FIELD,
            ESSENTIAL_FIELD, RESTART_POLICY_FIELD, ENTRY_POINT_FIELD, COMMAND_FIELD, ENVIRONMENT_FIELD, ENVIRONMENT_FILES_FIELD,
            MOUNT_POINTS_FIELD, VOLUMES_FROM_FIELD, LINUX_PARAMETERS_FIELD, SECRETS_FIELD, DEPENDS_ON_FIELD, START_TIMEOUT_FIELD,
            STOP_TIMEOUT_FIELD, VERSION_CONSISTENCY_FIELD, HOSTNAME_FIELD, USER_FIELD, WORKING_DIRECTORY_FIELD,
            DISABLE_NETWORKING_FIELD, PRIVILEGED_FIELD, READONLY_ROOT_FILESYSTEM_FIELD, DNS_SERVERS_FIELD,
            DNS_SEARCH_DOMAINS_FIELD, EXTRA_HOSTS_FIELD, DOCKER_SECURITY_OPTIONS_FIELD, INTERACTIVE_FIELD, PSEUDO_TERMINAL_FIELD,
            DOCKER_LABELS_FIELD, ULIMITS_FIELD, LOG_CONFIGURATION_FIELD, HEALTH_CHECK_FIELD, SYSTEM_CONTROLS_FIELD,
            RESOURCE_REQUIREMENTS_FIELD, FIRELENS_CONFIGURATION_FIELD, CREDENTIAL_SPECS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String image;

    private final RepositoryCredentials repositoryCredentials;

    private final Integer cpu;

    private final Integer memory;

    private final Integer memoryReservation;

    private final List<String> links;

    private final List<PortMapping> portMappings;

    private final Boolean essential;

    private final ContainerRestartPolicy restartPolicy;

    private final List<String> entryPoint;

    private final List<String> command;

    private final List<KeyValuePair> environment;

    private final List<EnvironmentFile> environmentFiles;

    private final List<MountPoint> mountPoints;

    private final List<VolumeFrom> volumesFrom;

    private final LinuxParameters linuxParameters;

    private final List<Secret> secrets;

    private final List<ContainerDependency> dependsOn;

    private final Integer startTimeout;

    private final Integer stopTimeout;

    private final String versionConsistency;

    private final String hostname;

    private final String user;

    private final String workingDirectory;

    private final Boolean disableNetworking;

    private final Boolean privileged;

    private final Boolean readonlyRootFilesystem;

    private final List<String> dnsServers;

    private final List<String> dnsSearchDomains;

    private final List<HostEntry> extraHosts;

    private final List<String> dockerSecurityOptions;

    private final Boolean interactive;

    private final Boolean pseudoTerminal;

    private final Map<String, String> dockerLabels;

    private final List<Ulimit> ulimits;

    private final LogConfiguration logConfiguration;

    private final HealthCheck healthCheck;

    private final List<SystemControl> systemControls;

    private final List<ResourceRequirement> resourceRequirements;

    private final FirelensConfiguration firelensConfiguration;

    private final List<String> credentialSpecs;

    private ContainerDefinition(BuilderImpl builder) {
        this.name = builder.name;
        this.image = builder.image;
        this.repositoryCredentials = builder.repositoryCredentials;
        this.cpu = builder.cpu;
        this.memory = builder.memory;
        this.memoryReservation = builder.memoryReservation;
        this.links = builder.links;
        this.portMappings = builder.portMappings;
        this.essential = builder.essential;
        this.restartPolicy = builder.restartPolicy;
        this.entryPoint = builder.entryPoint;
        this.command = builder.command;
        this.environment = builder.environment;
        this.environmentFiles = builder.environmentFiles;
        this.mountPoints = builder.mountPoints;
        this.volumesFrom = builder.volumesFrom;
        this.linuxParameters = builder.linuxParameters;
        this.secrets = builder.secrets;
        this.dependsOn = builder.dependsOn;
        this.startTimeout = builder.startTimeout;
        this.stopTimeout = builder.stopTimeout;
        this.versionConsistency = builder.versionConsistency;
        this.hostname = builder.hostname;
        this.user = builder.user;
        this.workingDirectory = builder.workingDirectory;
        this.disableNetworking = builder.disableNetworking;
        this.privileged = builder.privileged;
        this.readonlyRootFilesystem = builder.readonlyRootFilesystem;
        this.dnsServers = builder.dnsServers;
        this.dnsSearchDomains = builder.dnsSearchDomains;
        this.extraHosts = builder.extraHosts;
        this.dockerSecurityOptions = builder.dockerSecurityOptions;
        this.interactive = builder.interactive;
        this.pseudoTerminal = builder.pseudoTerminal;
        this.dockerLabels = builder.dockerLabels;
        this.ulimits = builder.ulimits;
        this.logConfiguration = builder.logConfiguration;
        this.healthCheck = builder.healthCheck;
        this.systemControls = builder.systemControls;
        this.resourceRequirements = builder.resourceRequirements;
        this.firelensConfiguration = builder.firelensConfiguration;
        this.credentialSpecs = builder.credentialSpecs;
    }

    /**
     * <p>
     * The name of a container. If you're linking multiple containers together in a task definition, the
     * <code>name</code> of one container can be entered in the <code>links</code> of another container to connect the
     * containers. Up to 255 letters (uppercase and lowercase), numbers, underscores, and hyphens are allowed. This
     * parameter maps to <code>name</code> in the docker container create command and the <code>--name</code> option to
     * docker run.
     * </p>
     * 
     * @return The name of a container. If you're linking multiple containers together in a task definition, the
     *         <code>name</code> of one container can be entered in the <code>links</code> of another container to
     *         connect the containers. Up to 255 letters (uppercase and lowercase), numbers, underscores, and hyphens
     *         are allowed. This parameter maps to <code>name</code> in the docker container create command and the
     *         <code>--name</code> option to docker run.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The image used to start a container. This string is passed directly to the Docker daemon. By default, images in
     * the Docker Hub registry are available. Other repositories are specified with either
     * <code> <i>repository-url</i>/<i>image</i>:<i>tag</i> </code> or
     * <code> <i>repository-url</i>/<i>image</i>@<i>digest</i> </code>. For images using tags
     * (repository-url/image:tag), up to 255 characters total are allowed, including letters (uppercase and lowercase),
     * numbers, hyphens, underscores, colons, periods, forward slashes, and number signs (#). For images using digests
     * (repository-url/image@digest), the 255 character limit applies only to the repository URL and image name
     * (everything before the @ sign). The only supported hash function is sha256, and the hash value after sha256: must
     * be exactly 64 characters (only letters A-F, a-f, and numbers 0-9 are allowed). This parameter maps to
     * <code>Image</code> in the docker container create command and the <code>IMAGE</code> parameter of docker run.
     * </p>
     * <ul>
     * <li>
     * <p>
     * When a new task starts, the Amazon ECS container agent pulls the latest version of the specified image and tag
     * for the container to use. However, subsequent updates to a repository image aren't propagated to already running
     * tasks.
     * </p>
     * </li>
     * <li>
     * <p>
     * Images in Amazon ECR repositories can be specified by either using the full <code>registry/repository:tag</code>
     * or <code>registry/repository@digest</code>. For example,
     * <code>012345678910.dkr.ecr.&lt;region-name&gt;.amazonaws.com/&lt;repository-name&gt;:latest</code> or
     * <code>012345678910.dkr.ecr.&lt;region-name&gt;.amazonaws.com/&lt;repository-name&gt;@sha256:94afd1f2e64d908bc90dbca0035a5b567EXAMPLE</code>
     * .
     * </p>
     * </li>
     * <li>
     * <p>
     * Images in official repositories on Docker Hub use a single name (for example, <code>ubuntu</code> or
     * <code>mongo</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * Images in other repositories on Docker Hub are qualified with an organization name (for example,
     * <code>amazon/amazon-ecs-agent</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * Images in other online repositories are qualified further by a domain name (for example,
     * <code>quay.io/assemblyline/ubuntu</code>).
     * </p>
     * </li>
     * </ul>
     * 
     * @return The image used to start a container. This string is passed directly to the Docker daemon. By default,
     *         images in the Docker Hub registry are available. Other repositories are specified with either
     *         <code> <i>repository-url</i>/<i>image</i>:<i>tag</i> </code> or
     *         <code> <i>repository-url</i>/<i>image</i>@<i>digest</i> </code>. For images using tags
     *         (repository-url/image:tag), up to 255 characters total are allowed, including letters (uppercase and
     *         lowercase), numbers, hyphens, underscores, colons, periods, forward slashes, and number signs (#). For
     *         images using digests (repository-url/image@digest), the 255 character limit applies only to the
     *         repository URL and image name (everything before the @ sign). The only supported hash function is sha256,
     *         and the hash value after sha256: must be exactly 64 characters (only letters A-F, a-f, and numbers 0-9
     *         are allowed). This parameter maps to <code>Image</code> in the docker container create command and the
     *         <code>IMAGE</code> parameter of docker run.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         When a new task starts, the Amazon ECS container agent pulls the latest version of the specified image
     *         and tag for the container to use. However, subsequent updates to a repository image aren't propagated to
     *         already running tasks.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Images in Amazon ECR repositories can be specified by either using the full
     *         <code>registry/repository:tag</code> or <code>registry/repository@digest</code>. For example,
     *         <code>012345678910.dkr.ecr.&lt;region-name&gt;.amazonaws.com/&lt;repository-name&gt;:latest</code> or
     *         <code>012345678910.dkr.ecr.&lt;region-name&gt;.amazonaws.com/&lt;repository-name&gt;@sha256:94afd1f2e64d908bc90dbca0035a5b567EXAMPLE</code>
     *         .
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Images in official repositories on Docker Hub use a single name (for example, <code>ubuntu</code> or
     *         <code>mongo</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Images in other repositories on Docker Hub are qualified with an organization name (for example,
     *         <code>amazon/amazon-ecs-agent</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Images in other online repositories are qualified further by a domain name (for example,
     *         <code>quay.io/assemblyline/ubuntu</code>).
     *         </p>
     *         </li>
     */
    public final String image() {
        return image;
    }

    /**
     * <p>
     * The private repository authentication credentials to use.
     * </p>
     * 
     * @return The private repository authentication credentials to use.
     */
    public final RepositoryCredentials repositoryCredentials() {
        return repositoryCredentials;
    }

    /**
     * <p>
     * The number of <code>cpu</code> units reserved for the container. This parameter maps to <code>CpuShares</code> in
     * the docker container create commandand the <code>--cpu-shares</code> option to docker run.
     * </p>
     * <p>
     * This field is optional for tasks using the Fargate launch type, and the only requirement is that the total amount
     * of CPU reserved for all containers within a task be lower than the task-level <code>cpu</code> value.
     * </p>
     * <note>
     * <p>
     * You can determine the number of CPU units that are available per EC2 instance type by multiplying the vCPUs
     * listed for that instance type on the <a href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2 Instances</a>
     * detail page by 1,024.
     * </p>
     * </note>
     * <p>
     * Linux containers share unallocated CPU units with other containers on the container instance with the same ratio
     * as their allocated amount. For example, if you run a single-container task on a single-core instance type with
     * 512 CPU units specified for that container, and that's the only task running on the container instance, that
     * container could use the full 1,024 CPU unit share at any given time. However, if you launched another copy of the
     * same task on that container instance, each task is guaranteed a minimum of 512 CPU units when needed. Moreover,
     * each container could float to higher CPU usage if the other container was not using it. If both tasks were 100%
     * active all of the time, they would be limited to 512 CPU units.
     * </p>
     * <p>
     * On Linux container instances, the Docker daemon on the container instance uses the CPU value to calculate the
     * relative CPU share ratios for running containers. The minimum valid CPU share value that the Linux kernel allows
     * is 2, and the maximum valid CPU share value that the Linux kernel allows is 262144. However, the CPU parameter
     * isn't required, and you can use CPU values below 2 or above 262144 in your container definitions. For CPU values
     * below 2 (including null) or above 262144, the behavior varies based on your Amazon ECS container agent version:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Agent versions less than or equal to 1.1.0:</b> Null and zero CPU values are passed to Docker as 0, which
     * Docker then converts to 1,024 CPU shares. CPU values of 1 are passed to Docker as 1, which the Linux kernel
     * converts to two CPU shares.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Agent versions greater than or equal to 1.2.0:</b> Null, zero, and CPU values of 1 are passed to Docker as 2.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Agent versions greater than or equal to 1.84.0:</b> CPU values greater than 256 vCPU are passed to Docker as
     * 256, which is equivalent to 262144 CPU shares.
     * </p>
     * </li>
     * </ul>
     * <p>
     * On Windows container instances, the CPU limit is enforced as an absolute limit, or a quota. Windows containers
     * only have access to the specified amount of CPU that's described in the task definition. A null or zero CPU value
     * is passed to Docker as <code>0</code>, which Windows interprets as 1% of one CPU.
     * </p>
     * 
     * @return The number of <code>cpu</code> units reserved for the container. This parameter maps to
     *         <code>CpuShares</code> in the docker container create commandand the <code>--cpu-shares</code> option to
     *         docker run.</p>
     *         <p>
     *         This field is optional for tasks using the Fargate launch type, and the only requirement is that the
     *         total amount of CPU reserved for all containers within a task be lower than the task-level
     *         <code>cpu</code> value.
     *         </p>
     *         <note>
     *         <p>
     *         You can determine the number of CPU units that are available per EC2 instance type by multiplying the
     *         vCPUs listed for that instance type on the <a href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2
     *         Instances</a> detail page by 1,024.
     *         </p>
     *         </note>
     *         <p>
     *         Linux containers share unallocated CPU units with other containers on the container instance with the
     *         same ratio as their allocated amount. For example, if you run a single-container task on a single-core
     *         instance type with 512 CPU units specified for that container, and that's the only task running on the
     *         container instance, that container could use the full 1,024 CPU unit share at any given time. However, if
     *         you launched another copy of the same task on that container instance, each task is guaranteed a minimum
     *         of 512 CPU units when needed. Moreover, each container could float to higher CPU usage if the other
     *         container was not using it. If both tasks were 100% active all of the time, they would be limited to 512
     *         CPU units.
     *         </p>
     *         <p>
     *         On Linux container instances, the Docker daemon on the container instance uses the CPU value to calculate
     *         the relative CPU share ratios for running containers. The minimum valid CPU share value that the Linux
     *         kernel allows is 2, and the maximum valid CPU share value that the Linux kernel allows is 262144.
     *         However, the CPU parameter isn't required, and you can use CPU values below 2 or above 262144 in your
     *         container definitions. For CPU values below 2 (including null) or above 262144, the behavior varies based
     *         on your Amazon ECS container agent version:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>Agent versions less than or equal to 1.1.0:</b> Null and zero CPU values are passed to Docker as 0,
     *         which Docker then converts to 1,024 CPU shares. CPU values of 1 are passed to Docker as 1, which the
     *         Linux kernel converts to two CPU shares.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Agent versions greater than or equal to 1.2.0:</b> Null, zero, and CPU values of 1 are passed to
     *         Docker as 2.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Agent versions greater than or equal to 1.84.0:</b> CPU values greater than 256 vCPU are passed to
     *         Docker as 256, which is equivalent to 262144 CPU shares.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         On Windows container instances, the CPU limit is enforced as an absolute limit, or a quota. Windows
     *         containers only have access to the specified amount of CPU that's described in the task definition. A
     *         null or zero CPU value is passed to Docker as <code>0</code>, which Windows interprets as 1% of one CPU.
     */
    public final Integer cpu() {
        return cpu;
    }

    /**
     * <p>
     * The amount (in MiB) of memory to present to the container. If your container attempts to exceed the memory
     * specified here, the container is killed. The total amount of memory reserved for all containers within a task
     * must be lower than the task <code>memory</code> value, if one is specified. This parameter maps to
     * <code>Memory</code> in the docker container create command and the <code>--memory</code> option to docker run.
     * </p>
     * <p>
     * If using the Fargate launch type, this parameter is optional.
     * </p>
     * <p>
     * If using the EC2 launch type, you must specify either a task-level memory value or a container-level memory
     * value. If you specify both a container-level <code>memory</code> and <code>memoryReservation</code> value,
     * <code>memory</code> must be greater than <code>memoryReservation</code>. If you specify
     * <code>memoryReservation</code>, then that value is subtracted from the available memory resources for the
     * container instance where the container is placed. Otherwise, the value of <code>memory</code> is used.
     * </p>
     * <p>
     * The Docker 20.10.0 or later daemon reserves a minimum of 6 MiB of memory for a container. So, don't specify less
     * than 6 MiB of memory for your containers.
     * </p>
     * <p>
     * The Docker 19.03.13-ce or earlier daemon reserves a minimum of 4 MiB of memory for a container. So, don't specify
     * less than 4 MiB of memory for your containers.
     * </p>
     * 
     * @return The amount (in MiB) of memory to present to the container. If your container attempts to exceed the
     *         memory specified here, the container is killed. The total amount of memory reserved for all containers
     *         within a task must be lower than the task <code>memory</code> value, if one is specified. This parameter
     *         maps to <code>Memory</code> in the docker container create command and the <code>--memory</code> option
     *         to docker run.</p>
     *         <p>
     *         If using the Fargate launch type, this parameter is optional.
     *         </p>
     *         <p>
     *         If using the EC2 launch type, you must specify either a task-level memory value or a container-level
     *         memory value. If you specify both a container-level <code>memory</code> and
     *         <code>memoryReservation</code> value, <code>memory</code> must be greater than
     *         <code>memoryReservation</code>. If you specify <code>memoryReservation</code>, then that value is
     *         subtracted from the available memory resources for the container instance where the container is placed.
     *         Otherwise, the value of <code>memory</code> is used.
     *         </p>
     *         <p>
     *         The Docker 20.10.0 or later daemon reserves a minimum of 6 MiB of memory for a container. So, don't
     *         specify less than 6 MiB of memory for your containers.
     *         </p>
     *         <p>
     *         The Docker 19.03.13-ce or earlier daemon reserves a minimum of 4 MiB of memory for a container. So, don't
     *         specify less than 4 MiB of memory for your containers.
     */
    public final Integer memory() {
        return memory;
    }

    /**
     * <p>
     * The soft limit (in MiB) of memory to reserve for the container. When system memory is under heavy contention,
     * Docker attempts to keep the container memory to this soft limit. However, your container can consume more memory
     * when it needs to, up to either the hard limit specified with the <code>memory</code> parameter (if applicable),
     * or all of the available memory on the container instance, whichever comes first. This parameter maps to
     * <code>MemoryReservation</code> in the docker container create command and the <code>--memory-reservation</code>
     * option to docker run.
     * </p>
     * <p>
     * If a task-level memory value is not specified, you must specify a non-zero integer for one or both of
     * <code>memory</code> or <code>memoryReservation</code> in a container definition. If you specify both,
     * <code>memory</code> must be greater than <code>memoryReservation</code>. If you specify
     * <code>memoryReservation</code>, then that value is subtracted from the available memory resources for the
     * container instance where the container is placed. Otherwise, the value of <code>memory</code> is used.
     * </p>
     * <p>
     * For example, if your container normally uses 128 MiB of memory, but occasionally bursts to 256 MiB of memory for
     * short periods of time, you can set a <code>memoryReservation</code> of 128 MiB, and a <code>memory</code> hard
     * limit of 300 MiB. This configuration would allow the container to only reserve 128 MiB of memory from the
     * remaining resources on the container instance, but also allow the container to consume more memory resources when
     * needed.
     * </p>
     * <p>
     * The Docker 20.10.0 or later daemon reserves a minimum of 6 MiB of memory for a container. So, don't specify less
     * than 6 MiB of memory for your containers.
     * </p>
     * <p>
     * The Docker 19.03.13-ce or earlier daemon reserves a minimum of 4 MiB of memory for a container. So, don't specify
     * less than 4 MiB of memory for your containers.
     * </p>
     * 
     * @return The soft limit (in MiB) of memory to reserve for the container. When system memory is under heavy
     *         contention, Docker attempts to keep the container memory to this soft limit. However, your container can
     *         consume more memory when it needs to, up to either the hard limit specified with the <code>memory</code>
     *         parameter (if applicable), or all of the available memory on the container instance, whichever comes
     *         first. This parameter maps to <code>MemoryReservation</code> in the docker container create command and
     *         the <code>--memory-reservation</code> option to docker run.</p>
     *         <p>
     *         If a task-level memory value is not specified, you must specify a non-zero integer for one or both of
     *         <code>memory</code> or <code>memoryReservation</code> in a container definition. If you specify both,
     *         <code>memory</code> must be greater than <code>memoryReservation</code>. If you specify
     *         <code>memoryReservation</code>, then that value is subtracted from the available memory resources for the
     *         container instance where the container is placed. Otherwise, the value of <code>memory</code> is used.
     *         </p>
     *         <p>
     *         For example, if your container normally uses 128 MiB of memory, but occasionally bursts to 256 MiB of
     *         memory for short periods of time, you can set a <code>memoryReservation</code> of 128 MiB, and a
     *         <code>memory</code> hard limit of 300 MiB. This configuration would allow the container to only reserve
     *         128 MiB of memory from the remaining resources on the container instance, but also allow the container to
     *         consume more memory resources when needed.
     *         </p>
     *         <p>
     *         The Docker 20.10.0 or later daemon reserves a minimum of 6 MiB of memory for a container. So, don't
     *         specify less than 6 MiB of memory for your containers.
     *         </p>
     *         <p>
     *         The Docker 19.03.13-ce or earlier daemon reserves a minimum of 4 MiB of memory for a container. So, don't
     *         specify less than 4 MiB of memory for your containers.
     */
    public final Integer memoryReservation() {
        return memoryReservation;
    }

    /**
     * For responses, this returns true if the service returned a value for the Links property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasLinks() {
        return links != null && !(links instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The <code>links</code> parameter allows containers to communicate with each other without the need for port
     * mappings. This parameter is only supported if the network mode of a task definition is <code>bridge</code>. The
     * <code>name:internalName</code> construct is analogous to <code>name:alias</code> in Docker links. Up to 255
     * letters (uppercase and lowercase), numbers, underscores, and hyphens are allowed.. This parameter maps to
     * <code>Links</code> in the docker container create command and the <code>--link</code> option to docker run.
     * </p>
     * <note>
     * <p>
     * This parameter is not supported for Windows containers.
     * </p>
     * </note> <important>
     * <p>
     * Containers that are collocated on a single container instance may be able to communicate with each other without
     * requiring links or host port mappings. Network isolation is achieved on the container instance using security
     * groups and VPC settings.
     * </p>
     * </important>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLinks} method.
     * </p>
     * 
     * @return The <code>links</code> parameter allows containers to communicate with each other without the need for
     *         port mappings. This parameter is only supported if the network mode of a task definition is
     *         <code>bridge</code>. The <code>name:internalName</code> construct is analogous to <code>name:alias</code>
     *         in Docker links. Up to 255 letters (uppercase and lowercase), numbers, underscores, and hyphens are
     *         allowed.. This parameter maps to <code>Links</code> in the docker container create command and the
     *         <code>--link</code> option to docker run.</p> <note>
     *         <p>
     *         This parameter is not supported for Windows containers.
     *         </p>
     *         </note> <important>
     *         <p>
     *         Containers that are collocated on a single container instance may be able to communicate with each other
     *         without requiring links or host port mappings. Network isolation is achieved on the container instance
     *         using security groups and VPC settings.
     *         </p>
     */
    public final List<String> links() {
        return links;
    }

    /**
     * For responses, this returns true if the service returned a value for the PortMappings property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPortMappings() {
        return portMappings != null && !(portMappings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of port mappings for the container. Port mappings allow containers to access ports on the host container
     * instance to send or receive traffic.
     * </p>
     * <p>
     * For task definitions that use the <code>awsvpc</code> network mode, only specify the <code>containerPort</code>.
     * The <code>hostPort</code> can be left blank or it must be the same value as the <code>containerPort</code>.
     * </p>
     * <p>
     * Port mappings on Windows use the <code>NetNAT</code> gateway address rather than <code>localhost</code>. There's
     * no loopback for port mappings on Windows, so you can't access a container's mapped port from the host itself.
     * </p>
     * <p>
     * This parameter maps to <code>PortBindings</code> in the the docker container create command and the
     * <code>--publish</code> option to docker run. If the network mode of a task definition is set to <code>none</code>
     * , then you can't specify port mappings. If the network mode of a task definition is set to <code>host</code>,
     * then host ports must either be undefined or they must match the container port in the port mapping.
     * </p>
     * <note>
     * <p>
     * After a task reaches the <code>RUNNING</code> status, manual and automatic host and container port assignments
     * are visible in the <b>Network Bindings</b> section of a container description for a selected task in the Amazon
     * ECS console. The assignments are also visible in the <code>networkBindings</code> section <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTasks.html">DescribeTasks</a>
     * responses.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPortMappings} method.
     * </p>
     * 
     * @return The list of port mappings for the container. Port mappings allow containers to access ports on the host
     *         container instance to send or receive traffic.</p>
     *         <p>
     *         For task definitions that use the <code>awsvpc</code> network mode, only specify the
     *         <code>containerPort</code>. The <code>hostPort</code> can be left blank or it must be the same value as
     *         the <code>containerPort</code>.
     *         </p>
     *         <p>
     *         Port mappings on Windows use the <code>NetNAT</code> gateway address rather than <code>localhost</code>.
     *         There's no loopback for port mappings on Windows, so you can't access a container's mapped port from the
     *         host itself.
     *         </p>
     *         <p>
     *         This parameter maps to <code>PortBindings</code> in the the docker container create command and the
     *         <code>--publish</code> option to docker run. If the network mode of a task definition is set to
     *         <code>none</code>, then you can't specify port mappings. If the network mode of a task definition is set
     *         to <code>host</code>, then host ports must either be undefined or they must match the container port in
     *         the port mapping.
     *         </p>
     *         <note>
     *         <p>
     *         After a task reaches the <code>RUNNING</code> status, manual and automatic host and container port
     *         assignments are visible in the <b>Network Bindings</b> section of a container description for a selected
     *         task in the Amazon ECS console. The assignments are also visible in the <code>networkBindings</code>
     *         section <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTasks.html">DescribeTasks</a>
     *         responses.
     *         </p>
     */
    public final List<PortMapping> portMappings() {
        return portMappings;
    }

    /**
     * <p>
     * If the <code>essential</code> parameter of a container is marked as <code>true</code>, and that container fails
     * or stops for any reason, all other containers that are part of the task are stopped. If the
     * <code>essential</code> parameter of a container is marked as <code>false</code>, its failure doesn't affect the
     * rest of the containers in a task. If this parameter is omitted, a container is assumed to be essential.
     * </p>
     * <p>
     * All tasks must have at least one essential container. If you have an application that's composed of multiple
     * containers, group containers that are used for a common purpose into components, and separate the different
     * components into multiple task definitions. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/application_architecture.html">Application
     * Architecture</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * 
     * @return If the <code>essential</code> parameter of a container is marked as <code>true</code>, and that container
     *         fails or stops for any reason, all other containers that are part of the task are stopped. If the
     *         <code>essential</code> parameter of a container is marked as <code>false</code>, its failure doesn't
     *         affect the rest of the containers in a task. If this parameter is omitted, a container is assumed to be
     *         essential.</p>
     *         <p>
     *         All tasks must have at least one essential container. If you have an application that's composed of
     *         multiple containers, group containers that are used for a common purpose into components, and separate
     *         the different components into multiple task definitions. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/application_architecture.html"
     *         >Application Architecture</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     */
    public final Boolean essential() {
        return essential;
    }

    /**
     * <p>
     * The restart policy for a container. When you set up a restart policy, Amazon ECS can restart the container
     * without needing to replace the task. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-restart-policy.html">Restart
     * individual containers in Amazon ECS tasks with container restart policies</a> in the <i>Amazon Elastic Container
     * Service Developer Guide</i>.
     * </p>
     * 
     * @return The restart policy for a container. When you set up a restart policy, Amazon ECS can restart the
     *         container without needing to replace the task. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-restart-policy.html">Restart
     *         individual containers in Amazon ECS tasks with container restart policies</a> in the <i>Amazon Elastic
     *         Container Service Developer Guide</i>.
     */
    public final ContainerRestartPolicy restartPolicy() {
        return restartPolicy;
    }

    /**
     * For responses, this returns true if the service returned a value for the EntryPoint property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasEntryPoint() {
        return entryPoint != null && !(entryPoint instanceof SdkAutoConstructList);
    }

    /**
     * <important>
     * <p>
     * Early versions of the Amazon ECS container agent don't properly handle <code>entryPoint</code> parameters. If you
     * have problems using <code>entryPoint</code>, update your container agent or enter your commands and arguments as
     * <code>command</code> array items instead.
     * </p>
     * </important>
     * <p>
     * The entry point that's passed to the container. This parameter maps to <code>Entrypoint</code> in the docker
     * container create command and the <code>--entrypoint</code> option to docker run.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEntryPoint} method.
     * </p>
     * 
     * @return <p>
     *         Early versions of the Amazon ECS container agent don't properly handle <code>entryPoint</code>
     *         parameters. If you have problems using <code>entryPoint</code>, update your container agent or enter your
     *         commands and arguments as <code>command</code> array items instead.
     *         </p>
     *         </important>
     *         <p>
     *         The entry point that's passed to the container. This parameter maps to <code>Entrypoint</code> in the
     *         docker container create command and the <code>--entrypoint</code> option to docker run.
     */
    public final List<String> entryPoint() {
        return entryPoint;
    }

    /**
     * For responses, this returns true if the service returned a value for the Command property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasCommand() {
        return command != null && !(command instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The command that's passed to the container. This parameter maps to <code>Cmd</code> in the docker container
     * create command and the <code>COMMAND</code> parameter to docker run. If there are multiple arguments, each
     * argument is a separated string in the array.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCommand} method.
     * </p>
     * 
     * @return The command that's passed to the container. This parameter maps to <code>Cmd</code> in the docker
     *         container create command and the <code>COMMAND</code> parameter to docker run. If there are multiple
     *         arguments, each argument is a separated string in the array.
     */
    public final List<String> command() {
        return command;
    }

    /**
     * For responses, this returns true if the service returned a value for the Environment property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEnvironment() {
        return environment != null && !(environment instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The environment variables to pass to a container. This parameter maps to <code>Env</code> in the docker container
     * create command and the <code>--env</code> option to docker run.
     * </p>
     * <important>
     * <p>
     * We don't recommend that you use plaintext environment variables for sensitive information, such as credential
     * data.
     * </p>
     * </important>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnvironment} method.
     * </p>
     * 
     * @return The environment variables to pass to a container. This parameter maps to <code>Env</code> in the docker
     *         container create command and the <code>--env</code> option to docker run.</p> <important>
     *         <p>
     *         We don't recommend that you use plaintext environment variables for sensitive information, such as
     *         credential data.
     *         </p>
     */
    public final List<KeyValuePair> environment() {
        return environment;
    }

    /**
     * For responses, this returns true if the service returned a value for the EnvironmentFiles property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEnvironmentFiles() {
        return environmentFiles != null && !(environmentFiles instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of files containing the environment variables to pass to a container. This parameter maps to the
     * <code>--env-file</code> option to docker run.
     * </p>
     * <p>
     * You can specify up to ten environment files. The file must have a <code>.env</code> file extension. Each line in
     * an environment file contains an environment variable in <code>VARIABLE=VALUE</code> format. Lines beginning with
     * <code>#</code> are treated as comments and are ignored.
     * </p>
     * <p>
     * If there are environment variables specified using the <code>environment</code> parameter in a container
     * definition, they take precedence over the variables contained within an environment file. If multiple environment
     * files are specified that contain the same variable, they're processed from the top down. We recommend that you
     * use unique variable names. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/taskdef-envfiles.html">Specifying Environment
     * Variables</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnvironmentFiles} method.
     * </p>
     * 
     * @return A list of files containing the environment variables to pass to a container. This parameter maps to the
     *         <code>--env-file</code> option to docker run.</p>
     *         <p>
     *         You can specify up to ten environment files. The file must have a <code>.env</code> file extension. Each
     *         line in an environment file contains an environment variable in <code>VARIABLE=VALUE</code> format. Lines
     *         beginning with <code>#</code> are treated as comments and are ignored.
     *         </p>
     *         <p>
     *         If there are environment variables specified using the <code>environment</code> parameter in a container
     *         definition, they take precedence over the variables contained within an environment file. If multiple
     *         environment files are specified that contain the same variable, they're processed from the top down. We
     *         recommend that you use unique variable names. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/taskdef-envfiles.html">Specifying
     *         Environment Variables</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     */
    public final List<EnvironmentFile> environmentFiles() {
        return environmentFiles;
    }

    /**
     * For responses, this returns true if the service returned a value for the MountPoints property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasMountPoints() {
        return mountPoints != null && !(mountPoints instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The mount points for data volumes in your container.
     * </p>
     * <p>
     * This parameter maps to <code>Volumes</code> in the docker container create command and the <code>--volume</code>
     * option to docker run.
     * </p>
     * <p>
     * Windows containers can mount whole directories on the same drive as <code>$env:ProgramData</code>. Windows
     * containers can't mount directories on a different drive, and mount point can't be across drives.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMountPoints} method.
     * </p>
     * 
     * @return The mount points for data volumes in your container.</p>
     *         <p>
     *         This parameter maps to <code>Volumes</code> in the docker container create command and the
     *         <code>--volume</code> option to docker run.
     *         </p>
     *         <p>
     *         Windows containers can mount whole directories on the same drive as <code>$env:ProgramData</code>.
     *         Windows containers can't mount directories on a different drive, and mount point can't be across drives.
     */
    public final List<MountPoint> mountPoints() {
        return mountPoints;
    }

    /**
     * For responses, this returns true if the service returned a value for the VolumesFrom property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasVolumesFrom() {
        return volumesFrom != null && !(volumesFrom instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Data volumes to mount from another container. This parameter maps to <code>VolumesFrom</code> in the docker
     * container create command and the <code>--volumes-from</code> option to docker run.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVolumesFrom} method.
     * </p>
     * 
     * @return Data volumes to mount from another container. This parameter maps to <code>VolumesFrom</code> in the
     *         docker container create command and the <code>--volumes-from</code> option to docker run.
     */
    public final List<VolumeFrom> volumesFrom() {
        return volumesFrom;
    }

    /**
     * <p>
     * Linux-specific modifications that are applied to the default Docker container configuration, such as Linux kernel
     * capabilities. For more information see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html"
     * >KernelCapabilities</a>.
     * </p>
     * <note>
     * <p>
     * This parameter is not supported for Windows containers.
     * </p>
     * </note>
     * 
     * @return Linux-specific modifications that are applied to the default Docker container configuration, such as
     *         Linux kernel capabilities. For more information see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html"
     *         >KernelCapabilities</a>.</p> <note>
     *         <p>
     *         This parameter is not supported for Windows containers.
     *         </p>
     */
    public final LinuxParameters linuxParameters() {
        return linuxParameters;
    }

    /**
     * For responses, this returns true if the service returned a value for the Secrets property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSecrets() {
        return secrets != null && !(secrets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The secrets to pass to the container. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html">Specifying
     * Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecrets} method.
     * </p>
     * 
     * @return The secrets to pass to the container. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html"
     *         >Specifying Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     */
    public final List<Secret> secrets() {
        return secrets;
    }

    /**
     * For responses, this returns true if the service returned a value for the DependsOn property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDependsOn() {
        return dependsOn != null && !(dependsOn instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The dependencies defined for container startup and shutdown. A container can contain multiple dependencies on
     * other containers in a task definition. When a dependency is defined for container startup, for container shutdown
     * it is reversed.
     * </p>
     * <p>
     * For tasks using the EC2 launch type, the container instances require at least version 1.26.0 of the container
     * agent to turn on container dependencies. However, we recommend using the latest container agent version. For
     * information about checking your agent version and updating to the latest version, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html">Updating the Amazon ECS
     * Container Agent</a> in the <i>Amazon Elastic Container Service Developer Guide</i>. If you're using an Amazon
     * ECS-optimized Linux AMI, your instance needs at least version 1.26.0-1 of the <code>ecs-init</code> package. If
     * your container instances are launched from version <code>20190301</code> or later, then they contain the required
     * versions of the container agent and <code>ecs-init</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html">Amazon ECS-optimized
     * Linux AMI</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * For tasks using the Fargate launch type, the task or service requires the following platforms:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Linux platform version <code>1.3.0</code> or later.
     * </p>
     * </li>
     * <li>
     * <p>
     * Windows platform version <code>1.0.0</code> or later.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDependsOn} method.
     * </p>
     * 
     * @return The dependencies defined for container startup and shutdown. A container can contain multiple
     *         dependencies on other containers in a task definition. When a dependency is defined for container
     *         startup, for container shutdown it is reversed.</p>
     *         <p>
     *         For tasks using the EC2 launch type, the container instances require at least version 1.26.0 of the
     *         container agent to turn on container dependencies. However, we recommend using the latest container agent
     *         version. For information about checking your agent version and updating to the latest version, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html">Updating the
     *         Amazon ECS Container Agent</a> in the <i>Amazon Elastic Container Service Developer Guide</i>. If you're
     *         using an Amazon ECS-optimized Linux AMI, your instance needs at least version 1.26.0-1 of the
     *         <code>ecs-init</code> package. If your container instances are launched from version
     *         <code>20190301</code> or later, then they contain the required versions of the container agent and
     *         <code>ecs-init</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html">Amazon
     *         ECS-optimized Linux AMI</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     *         </p>
     *         <p>
     *         For tasks using the Fargate launch type, the task or service requires the following platforms:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Linux platform version <code>1.3.0</code> or later.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Windows platform version <code>1.0.0</code> or later.
     *         </p>
     *         </li>
     */
    public final List<ContainerDependency> dependsOn() {
        return dependsOn;
    }

    /**
     * <p>
     * Time duration (in seconds) to wait before giving up on resolving dependencies for a container. For example, you
     * specify two containers in a task definition with containerA having a dependency on containerB reaching a
     * <code>COMPLETE</code>, <code>SUCCESS</code>, or <code>HEALTHY</code> status. If a <code>startTimeout</code> value
     * is specified for containerB and it doesn't reach the desired status within that time then containerA gives up and
     * not start. This results in the task transitioning to a <code>STOPPED</code> state.
     * </p>
     * <note>
     * <p>
     * When the <code>ECS_CONTAINER_START_TIMEOUT</code> container agent configuration variable is used, it's enforced
     * independently from this start timeout value.
     * </p>
     * </note>
     * <p>
     * For tasks using the Fargate launch type, the task or service requires the following platforms:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Linux platform version <code>1.3.0</code> or later.
     * </p>
     * </li>
     * <li>
     * <p>
     * Windows platform version <code>1.0.0</code> or later.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For tasks using the EC2 launch type, your container instances require at least version <code>1.26.0</code> of the
     * container agent to use a container start timeout value. However, we recommend using the latest container agent
     * version. For information about checking your agent version and updating to the latest version, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html">Updating the Amazon ECS
     * Container Agent</a> in the <i>Amazon Elastic Container Service Developer Guide</i>. If you're using an Amazon
     * ECS-optimized Linux AMI, your instance needs at least version <code>1.26.0-1</code> of the <code>ecs-init</code>
     * package. If your container instances are launched from version <code>20190301</code> or later, then they contain
     * the required versions of the container agent and <code>ecs-init</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html">Amazon ECS-optimized
     * Linux AMI</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * The valid values for Fargate are 2-120 seconds.
     * </p>
     * 
     * @return Time duration (in seconds) to wait before giving up on resolving dependencies for a container. For
     *         example, you specify two containers in a task definition with containerA having a dependency on
     *         containerB reaching a <code>COMPLETE</code>, <code>SUCCESS</code>, or <code>HEALTHY</code> status. If a
     *         <code>startTimeout</code> value is specified for containerB and it doesn't reach the desired status
     *         within that time then containerA gives up and not start. This results in the task transitioning to a
     *         <code>STOPPED</code> state.</p> <note>
     *         <p>
     *         When the <code>ECS_CONTAINER_START_TIMEOUT</code> container agent configuration variable is used, it's
     *         enforced independently from this start timeout value.
     *         </p>
     *         </note>
     *         <p>
     *         For tasks using the Fargate launch type, the task or service requires the following platforms:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Linux platform version <code>1.3.0</code> or later.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Windows platform version <code>1.0.0</code> or later.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For tasks using the EC2 launch type, your container instances require at least version
     *         <code>1.26.0</code> of the container agent to use a container start timeout value. However, we recommend
     *         using the latest container agent version. For information about checking your agent version and updating
     *         to the latest version, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html">Updating the
     *         Amazon ECS Container Agent</a> in the <i>Amazon Elastic Container Service Developer Guide</i>. If you're
     *         using an Amazon ECS-optimized Linux AMI, your instance needs at least version <code>1.26.0-1</code> of
     *         the <code>ecs-init</code> package. If your container instances are launched from version
     *         <code>20190301</code> or later, then they contain the required versions of the container agent and
     *         <code>ecs-init</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html">Amazon
     *         ECS-optimized Linux AMI</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     *         </p>
     *         <p>
     *         The valid values for Fargate are 2-120 seconds.
     */
    public final Integer startTimeout() {
        return startTimeout;
    }

    /**
     * <p>
     * Time duration (in seconds) to wait before the container is forcefully killed if it doesn't exit normally on its
     * own.
     * </p>
     * <p>
     * For tasks using the Fargate launch type, the task or service requires the following platforms:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Linux platform version <code>1.3.0</code> or later.
     * </p>
     * </li>
     * <li>
     * <p>
     * Windows platform version <code>1.0.0</code> or later.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For tasks that use the Fargate launch type, the max stop timeout value is 120 seconds and if the parameter is not
     * specified, the default value of 30 seconds is used.
     * </p>
     * <p>
     * For tasks that use the EC2 launch type, if the <code>stopTimeout</code> parameter isn't specified, the value set
     * for the Amazon ECS container agent configuration variable <code>ECS_CONTAINER_STOP_TIMEOUT</code> is used. If
     * neither the <code>stopTimeout</code> parameter or the <code>ECS_CONTAINER_STOP_TIMEOUT</code> agent configuration
     * variable are set, then the default values of 30 seconds for Linux containers and 30 seconds on Windows containers
     * are used. Your container instances require at least version 1.26.0 of the container agent to use a container stop
     * timeout value. However, we recommend using the latest container agent version. For information about checking
     * your agent version and updating to the latest version, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html">Updating the Amazon ECS
     * Container Agent</a> in the <i>Amazon Elastic Container Service Developer Guide</i>. If you're using an Amazon
     * ECS-optimized Linux AMI, your instance needs at least version 1.26.0-1 of the <code>ecs-init</code> package. If
     * your container instances are launched from version <code>20190301</code> or later, then they contain the required
     * versions of the container agent and <code>ecs-init</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html">Amazon ECS-optimized
     * Linux AMI</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * The valid values for Fargate are 2-120 seconds.
     * </p>
     * 
     * @return Time duration (in seconds) to wait before the container is forcefully killed if it doesn't exit normally
     *         on its own.</p>
     *         <p>
     *         For tasks using the Fargate launch type, the task or service requires the following platforms:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Linux platform version <code>1.3.0</code> or later.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Windows platform version <code>1.0.0</code> or later.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For tasks that use the Fargate launch type, the max stop timeout value is 120 seconds and if the
     *         parameter is not specified, the default value of 30 seconds is used.
     *         </p>
     *         <p>
     *         For tasks that use the EC2 launch type, if the <code>stopTimeout</code> parameter isn't specified, the
     *         value set for the Amazon ECS container agent configuration variable
     *         <code>ECS_CONTAINER_STOP_TIMEOUT</code> is used. If neither the <code>stopTimeout</code> parameter or the
     *         <code>ECS_CONTAINER_STOP_TIMEOUT</code> agent configuration variable are set, then the default values of
     *         30 seconds for Linux containers and 30 seconds on Windows containers are used. Your container instances
     *         require at least version 1.26.0 of the container agent to use a container stop timeout value. However, we
     *         recommend using the latest container agent version. For information about checking your agent version and
     *         updating to the latest version, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html">Updating the
     *         Amazon ECS Container Agent</a> in the <i>Amazon Elastic Container Service Developer Guide</i>. If you're
     *         using an Amazon ECS-optimized Linux AMI, your instance needs at least version 1.26.0-1 of the
     *         <code>ecs-init</code> package. If your container instances are launched from version
     *         <code>20190301</code> or later, then they contain the required versions of the container agent and
     *         <code>ecs-init</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html">Amazon
     *         ECS-optimized Linux AMI</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     *         </p>
     *         <p>
     *         The valid values for Fargate are 2-120 seconds.
     */
    public final Integer stopTimeout() {
        return stopTimeout;
    }

    /**
     * <p>
     * Specifies whether Amazon ECS will resolve the container image tag provided in the container definition to an
     * image digest. By default, the value is <code>enabled</code>. If you set the value for a container as
     * <code>disabled</code>, Amazon ECS will not resolve the provided container image tag to a digest and will use the
     * original image URI specified in the container definition for deployment. For more information about container
     * image resolution, see <a href=
     * "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html#deployment-container-image-stability"
     * >Container image resolution</a> in the <i>Amazon ECS Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #versionConsistency} will return {@link VersionConsistency#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #versionConsistencyAsString}.
     * </p>
     * 
     * @return Specifies whether Amazon ECS will resolve the container image tag provided in the container definition to
     *         an image digest. By default, the value is <code>enabled</code>. If you set the value for a container as
     *         <code>disabled</code>, Amazon ECS will not resolve the provided container image tag to a digest and will
     *         use the original image URI specified in the container definition for deployment. For more information
     *         about container image resolution, see <a href=
     *         "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html#deployment-container-image-stability"
     *         >Container image resolution</a> in the <i>Amazon ECS Developer Guide</i>.
     * @see VersionConsistency
     */
    public final VersionConsistency versionConsistency() {
        return VersionConsistency.fromValue(versionConsistency);
    }

    /**
     * <p>
     * Specifies whether Amazon ECS will resolve the container image tag provided in the container definition to an
     * image digest. By default, the value is <code>enabled</code>. If you set the value for a container as
     * <code>disabled</code>, Amazon ECS will not resolve the provided container image tag to a digest and will use the
     * original image URI specified in the container definition for deployment. For more information about container
     * image resolution, see <a href=
     * "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html#deployment-container-image-stability"
     * >Container image resolution</a> in the <i>Amazon ECS Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #versionConsistency} will return {@link VersionConsistency#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #versionConsistencyAsString}.
     * </p>
     * 
     * @return Specifies whether Amazon ECS will resolve the container image tag provided in the container definition to
     *         an image digest. By default, the value is <code>enabled</code>. If you set the value for a container as
     *         <code>disabled</code>, Amazon ECS will not resolve the provided container image tag to a digest and will
     *         use the original image URI specified in the container definition for deployment. For more information
     *         about container image resolution, see <a href=
     *         "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html#deployment-container-image-stability"
     *         >Container image resolution</a> in the <i>Amazon ECS Developer Guide</i>.
     * @see VersionConsistency
     */
    public final String versionConsistencyAsString() {
        return versionConsistency;
    }

    /**
     * <p>
     * The hostname to use for your container. This parameter maps to <code>Hostname</code> in the docker container
     * create command and the <code>--hostname</code> option to docker run.
     * </p>
     * <note>
     * <p>
     * The <code>hostname</code> parameter is not supported if you're using the <code>awsvpc</code> network mode.
     * </p>
     * </note>
     * 
     * @return The hostname to use for your container. This parameter maps to <code>Hostname</code> in the docker
     *         container create command and the <code>--hostname</code> option to docker run.</p> <note>
     *         <p>
     *         The <code>hostname</code> parameter is not supported if you're using the <code>awsvpc</code> network
     *         mode.
     *         </p>
     */
    public final String hostname() {
        return hostname;
    }

    /**
     * <p>
     * The user to use inside the container. This parameter maps to <code>User</code> in the docker container create
     * command and the <code>--user</code> option to docker run.
     * </p>
     * <important>
     * <p>
     * When running tasks using the <code>host</code> network mode, don't run containers using the root user (UID 0). We
     * recommend using a non-root user for better security.
     * </p>
     * </important>
     * <p>
     * You can specify the <code>user</code> using the following formats. If specifying a UID or GID, you must specify
     * it as a positive integer.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>user</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>user:group</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>uid</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>uid:gid</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>user:gid</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>uid:group</code>
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * This parameter is not supported for Windows containers.
     * </p>
     * </note>
     * 
     * @return The user to use inside the container. This parameter maps to <code>User</code> in the docker container
     *         create command and the <code>--user</code> option to docker run.</p> <important>
     *         <p>
     *         When running tasks using the <code>host</code> network mode, don't run containers using the root user
     *         (UID 0). We recommend using a non-root user for better security.
     *         </p>
     *         </important>
     *         <p>
     *         You can specify the <code>user</code> using the following formats. If specifying a UID or GID, you must
     *         specify it as a positive integer.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>user</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>user:group</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>uid</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>uid:gid</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>user:gid</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>uid:group</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         This parameter is not supported for Windows containers.
     *         </p>
     */
    public final String user() {
        return user;
    }

    /**
     * <p>
     * The working directory to run commands inside the container in. This parameter maps to <code>WorkingDir</code> in
     * the docker container create command and the <code>--workdir</code> option to docker run.
     * </p>
     * 
     * @return The working directory to run commands inside the container in. This parameter maps to
     *         <code>WorkingDir</code> in the docker container create command and the <code>--workdir</code> option to
     *         docker run.
     */
    public final String workingDirectory() {
        return workingDirectory;
    }

    /**
     * <p>
     * When this parameter is true, networking is off within the container. This parameter maps to
     * <code>NetworkDisabled</code> in the docker container create command.
     * </p>
     * <note>
     * <p>
     * This parameter is not supported for Windows containers.
     * </p>
     * </note>
     * 
     * @return When this parameter is true, networking is off within the container. This parameter maps to
     *         <code>NetworkDisabled</code> in the docker container create command.</p> <note>
     *         <p>
     *         This parameter is not supported for Windows containers.
     *         </p>
     */
    public final Boolean disableNetworking() {
        return disableNetworking;
    }

    /**
     * <p>
     * When this parameter is true, the container is given elevated privileges on the host container instance (similar
     * to the <code>root</code> user). This parameter maps to <code>Privileged</code> in the docker container create
     * command and the <code>--privileged</code> option to docker run
     * </p>
     * <note>
     * <p>
     * This parameter is not supported for Windows containers or tasks run on Fargate.
     * </p>
     * </note>
     * 
     * @return When this parameter is true, the container is given elevated privileges on the host container instance
     *         (similar to the <code>root</code> user). This parameter maps to <code>Privileged</code> in the docker
     *         container create command and the <code>--privileged</code> option to docker run</p> <note>
     *         <p>
     *         This parameter is not supported for Windows containers or tasks run on Fargate.
     *         </p>
     */
    public final Boolean privileged() {
        return privileged;
    }

    /**
     * <p>
     * When this parameter is true, the container is given read-only access to its root file system. This parameter maps
     * to <code>ReadonlyRootfs</code> in the docker container create command and the <code>--read-only</code> option to
     * docker run.
     * </p>
     * <note>
     * <p>
     * This parameter is not supported for Windows containers.
     * </p>
     * </note>
     * 
     * @return When this parameter is true, the container is given read-only access to its root file system. This
     *         parameter maps to <code>ReadonlyRootfs</code> in the docker container create command and the
     *         <code>--read-only</code> option to docker run.</p> <note>
     *         <p>
     *         This parameter is not supported for Windows containers.
     *         </p>
     */
    public final Boolean readonlyRootFilesystem() {
        return readonlyRootFilesystem;
    }

    /**
     * For responses, this returns true if the service returned a value for the DnsServers property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDnsServers() {
        return dnsServers != null && !(dnsServers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of DNS servers that are presented to the container. This parameter maps to <code>Dns</code> in the docker
     * container create command and the <code>--dns</code> option to docker run.
     * </p>
     * <note>
     * <p>
     * This parameter is not supported for Windows containers.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDnsServers} method.
     * </p>
     * 
     * @return A list of DNS servers that are presented to the container. This parameter maps to <code>Dns</code> in the
     *         docker container create command and the <code>--dns</code> option to docker run.</p> <note>
     *         <p>
     *         This parameter is not supported for Windows containers.
     *         </p>
     */
    public final List<String> dnsServers() {
        return dnsServers;
    }

    /**
     * For responses, this returns true if the service returned a value for the DnsSearchDomains property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDnsSearchDomains() {
        return dnsSearchDomains != null && !(dnsSearchDomains instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of DNS search domains that are presented to the container. This parameter maps to <code>DnsSearch</code>
     * in the docker container create command and the <code>--dns-search</code> option to docker run.
     * </p>
     * <note>
     * <p>
     * This parameter is not supported for Windows containers.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDnsSearchDomains} method.
     * </p>
     * 
     * @return A list of DNS search domains that are presented to the container. This parameter maps to
     *         <code>DnsSearch</code> in the docker container create command and the <code>--dns-search</code> option to
     *         docker run.</p> <note>
     *         <p>
     *         This parameter is not supported for Windows containers.
     *         </p>
     */
    public final List<String> dnsSearchDomains() {
        return dnsSearchDomains;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExtraHosts property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasExtraHosts() {
        return extraHosts != null && !(extraHosts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of hostnames and IP address mappings to append to the <code>/etc/hosts</code> file on the container. This
     * parameter maps to <code>ExtraHosts</code> in the docker container create command and the <code>--add-host</code>
     * option to docker run.
     * </p>
     * <note>
     * <p>
     * This parameter isn't supported for Windows containers or tasks that use the <code>awsvpc</code> network mode.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExtraHosts} method.
     * </p>
     * 
     * @return A list of hostnames and IP address mappings to append to the <code>/etc/hosts</code> file on the
     *         container. This parameter maps to <code>ExtraHosts</code> in the docker container create command and the
     *         <code>--add-host</code> option to docker run.</p> <note>
     *         <p>
     *         This parameter isn't supported for Windows containers or tasks that use the <code>awsvpc</code> network
     *         mode.
     *         </p>
     */
    public final List<HostEntry> extraHosts() {
        return extraHosts;
    }

    /**
     * For responses, this returns true if the service returned a value for the DockerSecurityOptions property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasDockerSecurityOptions() {
        return dockerSecurityOptions != null && !(dockerSecurityOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of strings to provide custom configuration for multiple security systems. This field isn't valid for
     * containers in tasks using the Fargate launch type.
     * </p>
     * <p>
     * For Linux tasks on EC2, this parameter can be used to reference custom labels for SELinux and AppArmor
     * multi-level security systems.
     * </p>
     * <p>
     * For any tasks on EC2, this parameter can be used to reference a credential spec file that configures a container
     * for Active Directory authentication. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/windows-gmsa.html">Using gMSAs for Windows
     * Containers</a> and <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/linux-gmsa.html">Using
     * gMSAs for Linux Containers</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * This parameter maps to <code>SecurityOpt</code> in the docker container create command and the
     * <code>--security-opt</code> option to docker run.
     * </p>
     * <note>
     * <p>
     * The Amazon ECS container agent running on a container instance must register with the
     * <code>ECS_SELINUX_CAPABLE=true</code> or <code>ECS_APPARMOR_CAPABLE=true</code> environment variables before
     * containers placed on that instance can use these security options. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon ECS Container
     * Agent Configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * </note>
     * <p>
     * Valid values: "no-new-privileges" | "apparmor:PROFILE" | "label:value" | "credentialspec:CredentialSpecFilePath"
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDockerSecurityOptions} method.
     * </p>
     * 
     * @return A list of strings to provide custom configuration for multiple security systems. This field isn't valid
     *         for containers in tasks using the Fargate launch type.</p>
     *         <p>
     *         For Linux tasks on EC2, this parameter can be used to reference custom labels for SELinux and AppArmor
     *         multi-level security systems.
     *         </p>
     *         <p>
     *         For any tasks on EC2, this parameter can be used to reference a credential spec file that configures a
     *         container for Active Directory authentication. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/windows-gmsa.html">Using gMSAs for
     *         Windows Containers</a> and <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/linux-gmsa.html">Using gMSAs for Linux
     *         Containers</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     *         </p>
     *         <p>
     *         This parameter maps to <code>SecurityOpt</code> in the docker container create command and the
     *         <code>--security-opt</code> option to docker run.
     *         </p>
     *         <note>
     *         <p>
     *         The Amazon ECS container agent running on a container instance must register with the
     *         <code>ECS_SELINUX_CAPABLE=true</code> or <code>ECS_APPARMOR_CAPABLE=true</code> environment variables
     *         before containers placed on that instance can use these security options. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon ECS
     *         Container Agent Configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     *         </p>
     *         </note>
     *         <p>
     *         Valid values: "no-new-privileges" | "apparmor:PROFILE" | "label:value" |
     *         "credentialspec:CredentialSpecFilePath"
     */
    public final List<String> dockerSecurityOptions() {
        return dockerSecurityOptions;
    }

    /**
     * <p>
     * When this parameter is <code>true</code>, you can deploy containerized applications that require
     * <code>stdin</code> or a <code>tty</code> to be allocated. This parameter maps to <code>OpenStdin</code> in the
     * docker container create command and the <code>--interactive</code> option to docker run.
     * </p>
     * 
     * @return When this parameter is <code>true</code>, you can deploy containerized applications that require
     *         <code>stdin</code> or a <code>tty</code> to be allocated. This parameter maps to <code>OpenStdin</code>
     *         in the docker container create command and the <code>--interactive</code> option to docker run.
     */
    public final Boolean interactive() {
        return interactive;
    }

    /**
     * <p>
     * When this parameter is <code>true</code>, a TTY is allocated. This parameter maps to <code>Tty</code> in the
     * docker container create command and the <code>--tty</code> option to docker run.
     * </p>
     * 
     * @return When this parameter is <code>true</code>, a TTY is allocated. This parameter maps to <code>Tty</code> in
     *         the docker container create command and the <code>--tty</code> option to docker run.
     */
    public final Boolean pseudoTerminal() {
        return pseudoTerminal;
    }

    /**
     * For responses, this returns true if the service returned a value for the DockerLabels property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDockerLabels() {
        return dockerLabels != null && !(dockerLabels instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A key/value map of labels to add to the container. This parameter maps to <code>Labels</code> in the docker
     * container create command and the <code>--label</code> option to docker run. This parameter requires version 1.18
     * of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your
     * container instance, log in to your container instance and run the following command:
     * <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDockerLabels} method.
     * </p>
     * 
     * @return A key/value map of labels to add to the container. This parameter maps to <code>Labels</code> in the
     *         docker container create command and the <code>--label</code> option to docker run. This parameter
     *         requires version 1.18 of the Docker Remote API or greater on your container instance. To check the Docker
     *         Remote API version on your container instance, log in to your container instance and run the following
     *         command: <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
     */
    public final Map<String, String> dockerLabels() {
        return dockerLabels;
    }

    /**
     * For responses, this returns true if the service returned a value for the Ulimits property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasUlimits() {
        return ulimits != null && !(ulimits instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of <code>ulimits</code> to set in the container. If a <code>ulimit</code> value is specified in a task
     * definition, it overrides the default values set by Docker. This parameter maps to <code>Ulimits</code> in the
     * docker container create command and the <code>--ulimit</code> option to docker run. Valid naming values are
     * displayed in the <a href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_Ulimit.html">Ulimit</a>
     * data type.
     * </p>
     * <p>
     * Amazon ECS tasks hosted on Fargate use the default resource limit values set by the operating system with the
     * exception of the <code>nofile</code> resource limit parameter which Fargate overrides. The <code>nofile</code>
     * resource limit sets a restriction on the number of open files that a container can use. The default
     * <code>nofile</code> soft limit is <code> 65535</code> and the default hard limit is <code>65535</code>.
     * </p>
     * <p>
     * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the
     * Docker Remote API version on your container instance, log in to your container instance and run the following
     * command: <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
     * </p>
     * <note>
     * <p>
     * This parameter is not supported for Windows containers.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUlimits} method.
     * </p>
     * 
     * @return A list of <code>ulimits</code> to set in the container. If a <code>ulimit</code> value is specified in a
     *         task definition, it overrides the default values set by Docker. This parameter maps to
     *         <code>Ulimits</code> in the docker container create command and the <code>--ulimit</code> option to
     *         docker run. Valid naming values are displayed in the <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_Ulimit.html">Ulimit</a> data
     *         type.</p>
     *         <p>
     *         Amazon ECS tasks hosted on Fargate use the default resource limit values set by the operating system with
     *         the exception of the <code>nofile</code> resource limit parameter which Fargate overrides. The
     *         <code>nofile</code> resource limit sets a restriction on the number of open files that a container can
     *         use. The default <code>nofile</code> soft limit is <code> 65535</code> and the default hard limit is
     *         <code>65535</code>.
     *         </p>
     *         <p>
     *         This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To
     *         check the Docker Remote API version on your container instance, log in to your container instance and run
     *         the following command: <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
     *         </p>
     *         <note>
     *         <p>
     *         This parameter is not supported for Windows containers.
     *         </p>
     */
    public final List<Ulimit> ulimits() {
        return ulimits;
    }

    /**
     * <p>
     * The log configuration specification for the container.
     * </p>
     * <p>
     * This parameter maps to <code>LogConfig</code> in the docker container create command and the
     * <code>--log-driver</code> option to docker run. By default, containers use the same logging driver that the
     * Docker daemon uses. However the container can use a different logging driver than the Docker daemon by specifying
     * a log driver with this parameter in the container definition. To use a different logging driver for a container,
     * the log system must be configured properly on the container instance (or on a different log server for remote
     * logging options).
     * </p>
     * <note>
     * <p>
     * Amazon ECS currently supports a subset of the logging drivers available to the Docker daemon (shown in the <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_LogConfiguration.html">LogConfiguration</a>
     * data type). Additional log drivers may be available in future releases of the Amazon ECS container agent.
     * </p>
     * </note>
     * <p>
     * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the
     * Docker Remote API version on your container instance, log in to your container instance and run the following
     * command: <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
     * </p>
     * <note>
     * <p>
     * The Amazon ECS container agent running on a container instance must register the logging drivers available on
     * that instance with the <code>ECS_AVAILABLE_LOGGING_DRIVERS</code> environment variable before containers placed
     * on that instance can use these log configuration options. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon ECS Container
     * Agent Configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * </note>
     * 
     * @return The log configuration specification for the container.</p>
     *         <p>
     *         This parameter maps to <code>LogConfig</code> in the docker container create command and the
     *         <code>--log-driver</code> option to docker run. By default, containers use the same logging driver that
     *         the Docker daemon uses. However the container can use a different logging driver than the Docker daemon
     *         by specifying a log driver with this parameter in the container definition. To use a different logging
     *         driver for a container, the log system must be configured properly on the container instance (or on a
     *         different log server for remote logging options).
     *         </p>
     *         <note>
     *         <p>
     *         Amazon ECS currently supports a subset of the logging drivers available to the Docker daemon (shown in
     *         the <a href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_LogConfiguration.html">
     *         LogConfiguration</a> data type). Additional log drivers may be available in future releases of the Amazon
     *         ECS container agent.
     *         </p>
     *         </note>
     *         <p>
     *         This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To
     *         check the Docker Remote API version on your container instance, log in to your container instance and run
     *         the following command: <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
     *         </p>
     *         <note>
     *         <p>
     *         The Amazon ECS container agent running on a container instance must register the logging drivers
     *         available on that instance with the <code>ECS_AVAILABLE_LOGGING_DRIVERS</code> environment variable
     *         before containers placed on that instance can use these log configuration options. For more information,
     *         see <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon
     *         ECS Container Agent Configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     *         </p>
     */
    public final LogConfiguration logConfiguration() {
        return logConfiguration;
    }

    /**
     * <p>
     * The container health check command and associated configuration parameters for the container. This parameter maps
     * to <code>HealthCheck</code> in the docker container create command and the <code>HEALTHCHECK</code> parameter of
     * docker run.
     * </p>
     * 
     * @return The container health check command and associated configuration parameters for the container. This
     *         parameter maps to <code>HealthCheck</code> in the docker container create command and the
     *         <code>HEALTHCHECK</code> parameter of docker run.
     */
    public final HealthCheck healthCheck() {
        return healthCheck;
    }

    /**
     * For responses, this returns true if the service returned a value for the SystemControls property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSystemControls() {
        return systemControls != null && !(systemControls instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of namespaced kernel parameters to set in the container. This parameter maps to <code>Sysctls</code> in
     * the docker container create command and the <code>--sysctl</code> option to docker run. For example, you can
     * configure <code>net.ipv4.tcp_keepalive_time</code> setting to maintain longer lived connections.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSystemControls} method.
     * </p>
     * 
     * @return A list of namespaced kernel parameters to set in the container. This parameter maps to
     *         <code>Sysctls</code> in the docker container create command and the <code>--sysctl</code> option to
     *         docker run. For example, you can configure <code>net.ipv4.tcp_keepalive_time</code> setting to maintain
     *         longer lived connections.
     */
    public final List<SystemControl> systemControls() {
        return systemControls;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceRequirements property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceRequirements() {
        return resourceRequirements != null && !(resourceRequirements instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The type and amount of a resource to assign to a container. The only supported resource is a GPU.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceRequirements} method.
     * </p>
     * 
     * @return The type and amount of a resource to assign to a container. The only supported resource is a GPU.
     */
    public final List<ResourceRequirement> resourceRequirements() {
        return resourceRequirements;
    }

    /**
     * <p>
     * The FireLens configuration for the container. This is used to specify and configure a log router for container
     * logs. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Custom Log Routing</a> in
     * the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * 
     * @return The FireLens configuration for the container. This is used to specify and configure a log router for
     *         container logs. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Custom Log
     *         Routing</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     */
    public final FirelensConfiguration firelensConfiguration() {
        return firelensConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the CredentialSpecs property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCredentialSpecs() {
        return credentialSpecs != null && !(credentialSpecs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of ARNs in SSM or Amazon S3 to a credential spec (<code>CredSpec</code>) file that configures the
     * container for Active Directory authentication. We recommend that you use this parameter instead of the
     * <code>dockerSecurityOptions</code>. The maximum number of ARNs is 1.
     * </p>
     * <p>
     * There are two formats for each ARN.
     * </p>
     * <dl>
     * <dt>credentialspecdomainless:MyARN</dt>
     * <dd>
     * <p>
     * You use <code>credentialspecdomainless:MyARN</code> to provide a <code>CredSpec</code> with an additional section
     * for a secret in Secrets Manager. You provide the login credentials to the domain in the secret.
     * </p>
     * <p>
     * Each task that runs on any container instance can join different domains.
     * </p>
     * <p>
     * You can use this format without joining the container instance to a domain.
     * </p>
     * </dd>
     * <dt>credentialspec:MyARN</dt>
     * <dd>
     * <p>
     * You use <code>credentialspec:MyARN</code> to provide a <code>CredSpec</code> for a single domain.
     * </p>
     * <p>
     * You must join the container instance to the domain before you start any tasks that use this task definition.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * In both formats, replace <code>MyARN</code> with the ARN in SSM or Amazon S3.
     * </p>
     * <p>
     * If you provide a <code>credentialspecdomainless:MyARN</code>, the <code>credspec</code> must provide a ARN in
     * Secrets Manager for a secret containing the username, password, and the domain to connect to. For better
     * security, the instance isn't joined to the domain for domainless authentication. Other applications on the
     * instance can't use the domainless credentials. You can use this parameter to run tasks on the same instance, even
     * it the tasks need to join different domains. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/windows-gmsa.html">Using gMSAs for Windows
     * Containers</a> and <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/linux-gmsa.html">Using
     * gMSAs for Linux Containers</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCredentialSpecs} method.
     * </p>
     * 
     * @return A list of ARNs in SSM or Amazon S3 to a credential spec (<code>CredSpec</code>) file that configures the
     *         container for Active Directory authentication. We recommend that you use this parameter instead of the
     *         <code>dockerSecurityOptions</code>. The maximum number of ARNs is 1.</p>
     *         <p>
     *         There are two formats for each ARN.
     *         </p>
     *         <dl>
     *         <dt>credentialspecdomainless:MyARN</dt>
     *         <dd>
     *         <p>
     *         You use <code>credentialspecdomainless:MyARN</code> to provide a <code>CredSpec</code> with an additional
     *         section for a secret in Secrets Manager. You provide the login credentials to the domain in the secret.
     *         </p>
     *         <p>
     *         Each task that runs on any container instance can join different domains.
     *         </p>
     *         <p>
     *         You can use this format without joining the container instance to a domain.
     *         </p>
     *         </dd>
     *         <dt>credentialspec:MyARN</dt>
     *         <dd>
     *         <p>
     *         You use <code>credentialspec:MyARN</code> to provide a <code>CredSpec</code> for a single domain.
     *         </p>
     *         <p>
     *         You must join the container instance to the domain before you start any tasks that use this task
     *         definition.
     *         </p>
     *         </dd>
     *         </dl>
     *         <p>
     *         In both formats, replace <code>MyARN</code> with the ARN in SSM or Amazon S3.
     *         </p>
     *         <p>
     *         If you provide a <code>credentialspecdomainless:MyARN</code>, the <code>credspec</code> must provide a
     *         ARN in Secrets Manager for a secret containing the username, password, and the domain to connect to. For
     *         better security, the instance isn't joined to the domain for domainless authentication. Other
     *         applications on the instance can't use the domainless credentials. You can use this parameter to run
     *         tasks on the same instance, even it the tasks need to join different domains. For more information, see
     *         <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/windows-gmsa.html">Using gMSAs for
     *         Windows Containers</a> and <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/linux-gmsa.html">Using gMSAs for Linux
     *         Containers</a>.
     */
    public final List<String> credentialSpecs() {
        return credentialSpecs;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(image());
        hashCode = 31 * hashCode + Objects.hashCode(repositoryCredentials());
        hashCode = 31 * hashCode + Objects.hashCode(cpu());
        hashCode = 31 * hashCode + Objects.hashCode(memory());
        hashCode = 31 * hashCode + Objects.hashCode(memoryReservation());
        hashCode = 31 * hashCode + Objects.hashCode(hasLinks() ? links() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasPortMappings() ? portMappings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(essential());
        hashCode = 31 * hashCode + Objects.hashCode(restartPolicy());
        hashCode = 31 * hashCode + Objects.hashCode(hasEntryPoint() ? entryPoint() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasCommand() ? command() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasEnvironment() ? environment() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasEnvironmentFiles() ? environmentFiles() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasMountPoints() ? mountPoints() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasVolumesFrom() ? volumesFrom() : null);
        hashCode = 31 * hashCode + Objects.hashCode(linuxParameters());
        hashCode = 31 * hashCode + Objects.hashCode(hasSecrets() ? secrets() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDependsOn() ? dependsOn() : null);
        hashCode = 31 * hashCode + Objects.hashCode(startTimeout());
        hashCode = 31 * hashCode + Objects.hashCode(stopTimeout());
        hashCode = 31 * hashCode + Objects.hashCode(versionConsistencyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hostname());
        hashCode = 31 * hashCode + Objects.hashCode(user());
        hashCode = 31 * hashCode + Objects.hashCode(workingDirectory());
        hashCode = 31 * hashCode + Objects.hashCode(disableNetworking());
        hashCode = 31 * hashCode + Objects.hashCode(privileged());
        hashCode = 31 * hashCode + Objects.hashCode(readonlyRootFilesystem());
        hashCode = 31 * hashCode + Objects.hashCode(hasDnsServers() ? dnsServers() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDnsSearchDomains() ? dnsSearchDomains() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasExtraHosts() ? extraHosts() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDockerSecurityOptions() ? dockerSecurityOptions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(interactive());
        hashCode = 31 * hashCode + Objects.hashCode(pseudoTerminal());
        hashCode = 31 * hashCode + Objects.hashCode(hasDockerLabels() ? dockerLabels() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasUlimits() ? ulimits() : null);
        hashCode = 31 * hashCode + Objects.hashCode(logConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(healthCheck());
        hashCode = 31 * hashCode + Objects.hashCode(hasSystemControls() ? systemControls() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceRequirements() ? resourceRequirements() : null);
        hashCode = 31 * hashCode + Objects.hashCode(firelensConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasCredentialSpecs() ? credentialSpecs() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ContainerDefinition)) {
            return false;
        }
        ContainerDefinition other = (ContainerDefinition) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(image(), other.image())
                && Objects.equals(repositoryCredentials(), other.repositoryCredentials()) && Objects.equals(cpu(), other.cpu())
                && Objects.equals(memory(), other.memory()) && Objects.equals(memoryReservation(), other.memoryReservation())
                && hasLinks() == other.hasLinks() && Objects.equals(links(), other.links())
                && hasPortMappings() == other.hasPortMappings() && Objects.equals(portMappings(), other.portMappings())
                && Objects.equals(essential(), other.essential()) && Objects.equals(restartPolicy(), other.restartPolicy())
                && hasEntryPoint() == other.hasEntryPoint() && Objects.equals(entryPoint(), other.entryPoint())
                && hasCommand() == other.hasCommand() && Objects.equals(command(), other.command())
                && hasEnvironment() == other.hasEnvironment() && Objects.equals(environment(), other.environment())
                && hasEnvironmentFiles() == other.hasEnvironmentFiles()
                && Objects.equals(environmentFiles(), other.environmentFiles()) && hasMountPoints() == other.hasMountPoints()
                && Objects.equals(mountPoints(), other.mountPoints()) && hasVolumesFrom() == other.hasVolumesFrom()
                && Objects.equals(volumesFrom(), other.volumesFrom())
                && Objects.equals(linuxParameters(), other.linuxParameters()) && hasSecrets() == other.hasSecrets()
                && Objects.equals(secrets(), other.secrets()) && hasDependsOn() == other.hasDependsOn()
                && Objects.equals(dependsOn(), other.dependsOn()) && Objects.equals(startTimeout(), other.startTimeout())
                && Objects.equals(stopTimeout(), other.stopTimeout())
                && Objects.equals(versionConsistencyAsString(), other.versionConsistencyAsString())
                && Objects.equals(hostname(), other.hostname()) && Objects.equals(user(), other.user())
                && Objects.equals(workingDirectory(), other.workingDirectory())
                && Objects.equals(disableNetworking(), other.disableNetworking())
                && Objects.equals(privileged(), other.privileged())
                && Objects.equals(readonlyRootFilesystem(), other.readonlyRootFilesystem())
                && hasDnsServers() == other.hasDnsServers() && Objects.equals(dnsServers(), other.dnsServers())
                && hasDnsSearchDomains() == other.hasDnsSearchDomains()
                && Objects.equals(dnsSearchDomains(), other.dnsSearchDomains()) && hasExtraHosts() == other.hasExtraHosts()
                && Objects.equals(extraHosts(), other.extraHosts())
                && hasDockerSecurityOptions() == other.hasDockerSecurityOptions()
                && Objects.equals(dockerSecurityOptions(), other.dockerSecurityOptions())
                && Objects.equals(interactive(), other.interactive()) && Objects.equals(pseudoTerminal(), other.pseudoTerminal())
                && hasDockerLabels() == other.hasDockerLabels() && Objects.equals(dockerLabels(), other.dockerLabels())
                && hasUlimits() == other.hasUlimits() && Objects.equals(ulimits(), other.ulimits())
                && Objects.equals(logConfiguration(), other.logConfiguration())
                && Objects.equals(healthCheck(), other.healthCheck()) && hasSystemControls() == other.hasSystemControls()
                && Objects.equals(systemControls(), other.systemControls())
                && hasResourceRequirements() == other.hasResourceRequirements()
                && Objects.equals(resourceRequirements(), other.resourceRequirements())
                && Objects.equals(firelensConfiguration(), other.firelensConfiguration())
                && hasCredentialSpecs() == other.hasCredentialSpecs()
                && Objects.equals(credentialSpecs(), other.credentialSpecs());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ContainerDefinition").add("Name", name()).add("Image", image())
                .add("RepositoryCredentials", repositoryCredentials()).add("Cpu", cpu()).add("Memory", memory())
                .add("MemoryReservation", memoryReservation()).add("Links", hasLinks() ? links() : null)
                .add("PortMappings", hasPortMappings() ? portMappings() : null).add("Essential", essential())
                .add("RestartPolicy", restartPolicy()).add("EntryPoint", hasEntryPoint() ? entryPoint() : null)
                .add("Command", hasCommand() ? command() : null).add("Environment", hasEnvironment() ? environment() : null)
                .add("EnvironmentFiles", hasEnvironmentFiles() ? environmentFiles() : null)
                .add("MountPoints", hasMountPoints() ? mountPoints() : null)
                .add("VolumesFrom", hasVolumesFrom() ? volumesFrom() : null).add("LinuxParameters", linuxParameters())
                .add("Secrets", hasSecrets() ? secrets() : null).add("DependsOn", hasDependsOn() ? dependsOn() : null)
                .add("StartTimeout", startTimeout()).add("StopTimeout", stopTimeout())
                .add("VersionConsistency", versionConsistencyAsString()).add("Hostname", hostname()).add("User", user())
                .add("WorkingDirectory", workingDirectory()).add("DisableNetworking", disableNetworking())
                .add("Privileged", privileged()).add("ReadonlyRootFilesystem", readonlyRootFilesystem())
                .add("DnsServers", hasDnsServers() ? dnsServers() : null)
                .add("DnsSearchDomains", hasDnsSearchDomains() ? dnsSearchDomains() : null)
                .add("ExtraHosts", hasExtraHosts() ? extraHosts() : null)
                .add("DockerSecurityOptions", hasDockerSecurityOptions() ? dockerSecurityOptions() : null)
                .add("Interactive", interactive()).add("PseudoTerminal", pseudoTerminal())
                .add("DockerLabels", hasDockerLabels() ? dockerLabels() : null).add("Ulimits", hasUlimits() ? ulimits() : null)
                .add("LogConfiguration", logConfiguration()).add("HealthCheck", healthCheck())
                .add("SystemControls", hasSystemControls() ? systemControls() : null)
                .add("ResourceRequirements", hasResourceRequirements() ? resourceRequirements() : null)
                .add("FirelensConfiguration", firelensConfiguration())
                .add("CredentialSpecs", hasCredentialSpecs() ? credentialSpecs() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "image":
            return Optional.ofNullable(clazz.cast(image()));
        case "repositoryCredentials":
            return Optional.ofNullable(clazz.cast(repositoryCredentials()));
        case "cpu":
            return Optional.ofNullable(clazz.cast(cpu()));
        case "memory":
            return Optional.ofNullable(clazz.cast(memory()));
        case "memoryReservation":
            return Optional.ofNullable(clazz.cast(memoryReservation()));
        case "links":
            return Optional.ofNullable(clazz.cast(links()));
        case "portMappings":
            return Optional.ofNullable(clazz.cast(portMappings()));
        case "essential":
            return Optional.ofNullable(clazz.cast(essential()));
        case "restartPolicy":
            return Optional.ofNullable(clazz.cast(restartPolicy()));
        case "entryPoint":
            return Optional.ofNullable(clazz.cast(entryPoint()));
        case "command":
            return Optional.ofNullable(clazz.cast(command()));
        case "environment":
            return Optional.ofNullable(clazz.cast(environment()));
        case "environmentFiles":
            return Optional.ofNullable(clazz.cast(environmentFiles()));
        case "mountPoints":
            return Optional.ofNullable(clazz.cast(mountPoints()));
        case "volumesFrom":
            return Optional.ofNullable(clazz.cast(volumesFrom()));
        case "linuxParameters":
            return Optional.ofNullable(clazz.cast(linuxParameters()));
        case "secrets":
            return Optional.ofNullable(clazz.cast(secrets()));
        case "dependsOn":
            return Optional.ofNullable(clazz.cast(dependsOn()));
        case "startTimeout":
            return Optional.ofNullable(clazz.cast(startTimeout()));
        case "stopTimeout":
            return Optional.ofNullable(clazz.cast(stopTimeout()));
        case "versionConsistency":
            return Optional.ofNullable(clazz.cast(versionConsistencyAsString()));
        case "hostname":
            return Optional.ofNullable(clazz.cast(hostname()));
        case "user":
            return Optional.ofNullable(clazz.cast(user()));
        case "workingDirectory":
            return Optional.ofNullable(clazz.cast(workingDirectory()));
        case "disableNetworking":
            return Optional.ofNullable(clazz.cast(disableNetworking()));
        case "privileged":
            return Optional.ofNullable(clazz.cast(privileged()));
        case "readonlyRootFilesystem":
            return Optional.ofNullable(clazz.cast(readonlyRootFilesystem()));
        case "dnsServers":
            return Optional.ofNullable(clazz.cast(dnsServers()));
        case "dnsSearchDomains":
            return Optional.ofNullable(clazz.cast(dnsSearchDomains()));
        case "extraHosts":
            return Optional.ofNullable(clazz.cast(extraHosts()));
        case "dockerSecurityOptions":
            return Optional.ofNullable(clazz.cast(dockerSecurityOptions()));
        case "interactive":
            return Optional.ofNullable(clazz.cast(interactive()));
        case "pseudoTerminal":
            return Optional.ofNullable(clazz.cast(pseudoTerminal()));
        case "dockerLabels":
            return Optional.ofNullable(clazz.cast(dockerLabels()));
        case "ulimits":
            return Optional.ofNullable(clazz.cast(ulimits()));
        case "logConfiguration":
            return Optional.ofNullable(clazz.cast(logConfiguration()));
        case "healthCheck":
            return Optional.ofNullable(clazz.cast(healthCheck()));
        case "systemControls":
            return Optional.ofNullable(clazz.cast(systemControls()));
        case "resourceRequirements":
            return Optional.ofNullable(clazz.cast(resourceRequirements()));
        case "firelensConfiguration":
            return Optional.ofNullable(clazz.cast(firelensConfiguration()));
        case "credentialSpecs":
            return Optional.ofNullable(clazz.cast(credentialSpecs()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("name", NAME_FIELD);
        map.put("image", IMAGE_FIELD);
        map.put("repositoryCredentials", REPOSITORY_CREDENTIALS_FIELD);
        map.put("cpu", CPU_FIELD);
        map.put("memory", MEMORY_FIELD);
        map.put("memoryReservation", MEMORY_RESERVATION_FIELD);
        map.put("links", LINKS_FIELD);
        map.put("portMappings", PORT_MAPPINGS_FIELD);
        map.put("essential", ESSENTIAL_FIELD);
        map.put("restartPolicy", RESTART_POLICY_FIELD);
        map.put("entryPoint", ENTRY_POINT_FIELD);
        map.put("command", COMMAND_FIELD);
        map.put("environment", ENVIRONMENT_FIELD);
        map.put("environmentFiles", ENVIRONMENT_FILES_FIELD);
        map.put("mountPoints", MOUNT_POINTS_FIELD);
        map.put("volumesFrom", VOLUMES_FROM_FIELD);
        map.put("linuxParameters", LINUX_PARAMETERS_FIELD);
        map.put("secrets", SECRETS_FIELD);
        map.put("dependsOn", DEPENDS_ON_FIELD);
        map.put("startTimeout", START_TIMEOUT_FIELD);
        map.put("stopTimeout", STOP_TIMEOUT_FIELD);
        map.put("versionConsistency", VERSION_CONSISTENCY_FIELD);
        map.put("hostname", HOSTNAME_FIELD);
        map.put("user", USER_FIELD);
        map.put("workingDirectory", WORKING_DIRECTORY_FIELD);
        map.put("disableNetworking", DISABLE_NETWORKING_FIELD);
        map.put("privileged", PRIVILEGED_FIELD);
        map.put("readonlyRootFilesystem", READONLY_ROOT_FILESYSTEM_FIELD);
        map.put("dnsServers", DNS_SERVERS_FIELD);
        map.put("dnsSearchDomains", DNS_SEARCH_DOMAINS_FIELD);
        map.put("extraHosts", EXTRA_HOSTS_FIELD);
        map.put("dockerSecurityOptions", DOCKER_SECURITY_OPTIONS_FIELD);
        map.put("interactive", INTERACTIVE_FIELD);
        map.put("pseudoTerminal", PSEUDO_TERMINAL_FIELD);
        map.put("dockerLabels", DOCKER_LABELS_FIELD);
        map.put("ulimits", ULIMITS_FIELD);
        map.put("logConfiguration", LOG_CONFIGURATION_FIELD);
        map.put("healthCheck", HEALTH_CHECK_FIELD);
        map.put("systemControls", SYSTEM_CONTROLS_FIELD);
        map.put("resourceRequirements", RESOURCE_REQUIREMENTS_FIELD);
        map.put("firelensConfiguration", FIRELENS_CONFIGURATION_FIELD);
        map.put("credentialSpecs", CREDENTIAL_SPECS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ContainerDefinition, T> g) {
        return obj -> g.apply((ContainerDefinition) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ContainerDefinition> {
        /**
         * <p>
         * The name of a container. If you're linking multiple containers together in a task definition, the
         * <code>name</code> of one container can be entered in the <code>links</code> of another container to connect
         * the containers. Up to 255 letters (uppercase and lowercase), numbers, underscores, and hyphens are allowed.
         * This parameter maps to <code>name</code> in the docker container create command and the <code>--name</code>
         * option to docker run.
         * </p>
         * 
         * @param name
         *        The name of a container. If you're linking multiple containers together in a task definition, the
         *        <code>name</code> of one container can be entered in the <code>links</code> of another container to
         *        connect the containers. Up to 255 letters (uppercase and lowercase), numbers, underscores, and hyphens
         *        are allowed. This parameter maps to <code>name</code> in the docker container create command and the
         *        <code>--name</code> option to docker run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The image used to start a container. This string is passed directly to the Docker daemon. By default, images
         * in the Docker Hub registry are available. Other repositories are specified with either
         * <code> <i>repository-url</i>/<i>image</i>:<i>tag</i> </code> or
         * <code> <i>repository-url</i>/<i>image</i>@<i>digest</i> </code>. For images using tags
         * (repository-url/image:tag), up to 255 characters total are allowed, including letters (uppercase and
         * lowercase), numbers, hyphens, underscores, colons, periods, forward slashes, and number signs (#). For images
         * using digests (repository-url/image@digest), the 255 character limit applies only to the repository URL and
         * image name (everything before the @ sign). The only supported hash function is sha256, and the hash value
         * after sha256: must be exactly 64 characters (only letters A-F, a-f, and numbers 0-9 are allowed). This
         * parameter maps to <code>Image</code> in the docker container create command and the <code>IMAGE</code>
         * parameter of docker run.
         * </p>
         * <ul>
         * <li>
         * <p>
         * When a new task starts, the Amazon ECS container agent pulls the latest version of the specified image and
         * tag for the container to use. However, subsequent updates to a repository image aren't propagated to already
         * running tasks.
         * </p>
         * </li>
         * <li>
         * <p>
         * Images in Amazon ECR repositories can be specified by either using the full
         * <code>registry/repository:tag</code> or <code>registry/repository@digest</code>. For example,
         * <code>012345678910.dkr.ecr.&lt;region-name&gt;.amazonaws.com/&lt;repository-name&gt;:latest</code> or
         * <code>012345678910.dkr.ecr.&lt;region-name&gt;.amazonaws.com/&lt;repository-name&gt;@sha256:94afd1f2e64d908bc90dbca0035a5b567EXAMPLE</code>
         * .
         * </p>
         * </li>
         * <li>
         * <p>
         * Images in official repositories on Docker Hub use a single name (for example, <code>ubuntu</code> or
         * <code>mongo</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * Images in other repositories on Docker Hub are qualified with an organization name (for example,
         * <code>amazon/amazon-ecs-agent</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * Images in other online repositories are qualified further by a domain name (for example,
         * <code>quay.io/assemblyline/ubuntu</code>).
         * </p>
         * </li>
         * </ul>
         * 
         * @param image
         *        The image used to start a container. This string is passed directly to the Docker daemon. By default,
         *        images in the Docker Hub registry are available. Other repositories are specified with either
         *        <code> <i>repository-url</i>/<i>image</i>:<i>tag</i> </code> or
         *        <code> <i>repository-url</i>/<i>image</i>@<i>digest</i> </code>. For images using tags
         *        (repository-url/image:tag), up to 255 characters total are allowed, including letters (uppercase and
         *        lowercase), numbers, hyphens, underscores, colons, periods, forward slashes, and number signs (#). For
         *        images using digests (repository-url/image@digest), the 255 character limit applies only to the
         *        repository URL and image name (everything before the @ sign). The only supported hash function is
         *        sha256, and the hash value after sha256: must be exactly 64 characters (only letters A-F, a-f, and
         *        numbers 0-9 are allowed). This parameter maps to <code>Image</code> in the docker container create
         *        command and the <code>IMAGE</code> parameter of docker run.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        When a new task starts, the Amazon ECS container agent pulls the latest version of the specified image
         *        and tag for the container to use. However, subsequent updates to a repository image aren't propagated
         *        to already running tasks.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Images in Amazon ECR repositories can be specified by either using the full
         *        <code>registry/repository:tag</code> or <code>registry/repository@digest</code>. For example,
         *        <code>012345678910.dkr.ecr.&lt;region-name&gt;.amazonaws.com/&lt;repository-name&gt;:latest</code> or
         *        <code>012345678910.dkr.ecr.&lt;region-name&gt;.amazonaws.com/&lt;repository-name&gt;@sha256:94afd1f2e64d908bc90dbca0035a5b567EXAMPLE</code>
         *        .
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Images in official repositories on Docker Hub use a single name (for example, <code>ubuntu</code> or
         *        <code>mongo</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Images in other repositories on Docker Hub are qualified with an organization name (for example,
         *        <code>amazon/amazon-ecs-agent</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Images in other online repositories are qualified further by a domain name (for example,
         *        <code>quay.io/assemblyline/ubuntu</code>).
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder image(String image);

        /**
         * <p>
         * The private repository authentication credentials to use.
         * </p>
         * 
         * @param repositoryCredentials
         *        The private repository authentication credentials to use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder repositoryCredentials(RepositoryCredentials repositoryCredentials);

        /**
         * <p>
         * The private repository authentication credentials to use.
         * </p>
         * This is a convenience method that creates an instance of the {@link RepositoryCredentials.Builder} avoiding
         * the need to create one manually via {@link RepositoryCredentials#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RepositoryCredentials.Builder#build()} is called immediately and
         * its result is passed to {@link #repositoryCredentials(RepositoryCredentials)}.
         * 
         * @param repositoryCredentials
         *        a consumer that will call methods on {@link RepositoryCredentials.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #repositoryCredentials(RepositoryCredentials)
         */
        default Builder repositoryCredentials(Consumer<RepositoryCredentials.Builder> repositoryCredentials) {
            return repositoryCredentials(RepositoryCredentials.builder().applyMutation(repositoryCredentials).build());
        }

        /**
         * <p>
         * The number of <code>cpu</code> units reserved for the container. This parameter maps to
         * <code>CpuShares</code> in the docker container create commandand the <code>--cpu-shares</code> option to
         * docker run.
         * </p>
         * <p>
         * This field is optional for tasks using the Fargate launch type, and the only requirement is that the total
         * amount of CPU reserved for all containers within a task be lower than the task-level <code>cpu</code> value.
         * </p>
         * <note>
         * <p>
         * You can determine the number of CPU units that are available per EC2 instance type by multiplying the vCPUs
         * listed for that instance type on the <a href="http://aws.amazon.com/ec2/instance-types/">Amazon EC2
         * Instances</a> detail page by 1,024.
         * </p>
         * </note>
         * <p>
         * Linux containers share unallocated CPU units with other containers on the container instance with the same
         * ratio as their allocated amount. For example, if you run a single-container task on a single-core instance
         * type with 512 CPU units specified for that container, and that's the only task running on the container
         * instance, that container could use the full 1,024 CPU unit share at any given time. However, if you launched
         * another copy of the same task on that container instance, each task is guaranteed a minimum of 512 CPU units
         * when needed. Moreover, each container could float to higher CPU usage if the other container was not using
         * it. If both tasks were 100% active all of the time, they would be limited to 512 CPU units.
         * </p>
         * <p>
         * On Linux container instances, the Docker daemon on the container instance uses the CPU value to calculate the
         * relative CPU share ratios for running containers. The minimum valid CPU share value that the Linux kernel
         * allows is 2, and the maximum valid CPU share value that the Linux kernel allows is 262144. However, the CPU
         * parameter isn't required, and you can use CPU values below 2 or above 262144 in your container definitions.
         * For CPU values below 2 (including null) or above 262144, the behavior varies based on your Amazon ECS
         * container agent version:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>Agent versions less than or equal to 1.1.0:</b> Null and zero CPU values are passed to Docker as 0, which
         * Docker then converts to 1,024 CPU shares. CPU values of 1 are passed to Docker as 1, which the Linux kernel
         * converts to two CPU shares.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Agent versions greater than or equal to 1.2.0:</b> Null, zero, and CPU values of 1 are passed to Docker as
         * 2.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Agent versions greater than or equal to 1.84.0:</b> CPU values greater than 256 vCPU are passed to Docker
         * as 256, which is equivalent to 262144 CPU shares.
         * </p>
         * </li>
         * </ul>
         * <p>
         * On Windows container instances, the CPU limit is enforced as an absolute limit, or a quota. Windows
         * containers only have access to the specified amount of CPU that's described in the task definition. A null or
         * zero CPU value is passed to Docker as <code>0</code>, which Windows interprets as 1% of one CPU.
         * </p>
         * 
         * @param cpu
         *        The number of <code>cpu</code> units reserved for the container. This parameter maps to
         *        <code>CpuShares</code> in the docker container create commandand the <code>--cpu-shares</code> option
         *        to docker run.</p>
         *        <p>
         *        This field is optional for tasks using the Fargate launch type, and the only requirement is that the
         *        total amount of CPU reserved for all containers within a task be lower than the task-level
         *        <code>cpu</code> value.
         *        </p>
         *        <note>
         *        <p>
         *        You can determine the number of CPU units that are available per EC2 instance type by multiplying the
         *        vCPUs listed for that instance type on the <a href="http://aws.amazon.com/ec2/instance-types/">Amazon
         *        EC2 Instances</a> detail page by 1,024.
         *        </p>
         *        </note>
         *        <p>
         *        Linux containers share unallocated CPU units with other containers on the container instance with the
         *        same ratio as their allocated amount. For example, if you run a single-container task on a single-core
         *        instance type with 512 CPU units specified for that container, and that's the only task running on the
         *        container instance, that container could use the full 1,024 CPU unit share at any given time. However,
         *        if you launched another copy of the same task on that container instance, each task is guaranteed a
         *        minimum of 512 CPU units when needed. Moreover, each container could float to higher CPU usage if the
         *        other container was not using it. If both tasks were 100% active all of the time, they would be
         *        limited to 512 CPU units.
         *        </p>
         *        <p>
         *        On Linux container instances, the Docker daemon on the container instance uses the CPU value to
         *        calculate the relative CPU share ratios for running containers. The minimum valid CPU share value that
         *        the Linux kernel allows is 2, and the maximum valid CPU share value that the Linux kernel allows is
         *        262144. However, the CPU parameter isn't required, and you can use CPU values below 2 or above 262144
         *        in your container definitions. For CPU values below 2 (including null) or above 262144, the behavior
         *        varies based on your Amazon ECS container agent version:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>Agent versions less than or equal to 1.1.0:</b> Null and zero CPU values are passed to Docker as 0,
         *        which Docker then converts to 1,024 CPU shares. CPU values of 1 are passed to Docker as 1, which the
         *        Linux kernel converts to two CPU shares.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Agent versions greater than or equal to 1.2.0:</b> Null, zero, and CPU values of 1 are passed to
         *        Docker as 2.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Agent versions greater than or equal to 1.84.0:</b> CPU values greater than 256 vCPU are passed to
         *        Docker as 256, which is equivalent to 262144 CPU shares.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        On Windows container instances, the CPU limit is enforced as an absolute limit, or a quota. Windows
         *        containers only have access to the specified amount of CPU that's described in the task definition. A
         *        null or zero CPU value is passed to Docker as <code>0</code>, which Windows interprets as 1% of one
         *        CPU.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cpu(Integer cpu);

        /**
         * <p>
         * The amount (in MiB) of memory to present to the container. If your container attempts to exceed the memory
         * specified here, the container is killed. The total amount of memory reserved for all containers within a task
         * must be lower than the task <code>memory</code> value, if one is specified. This parameter maps to
         * <code>Memory</code> in the docker container create command and the <code>--memory</code> option to docker
         * run.
         * </p>
         * <p>
         * If using the Fargate launch type, this parameter is optional.
         * </p>
         * <p>
         * If using the EC2 launch type, you must specify either a task-level memory value or a container-level memory
         * value. If you specify both a container-level <code>memory</code> and <code>memoryReservation</code> value,
         * <code>memory</code> must be greater than <code>memoryReservation</code>. If you specify
         * <code>memoryReservation</code>, then that value is subtracted from the available memory resources for the
         * container instance where the container is placed. Otherwise, the value of <code>memory</code> is used.
         * </p>
         * <p>
         * The Docker 20.10.0 or later daemon reserves a minimum of 6 MiB of memory for a container. So, don't specify
         * less than 6 MiB of memory for your containers.
         * </p>
         * <p>
         * The Docker 19.03.13-ce or earlier daemon reserves a minimum of 4 MiB of memory for a container. So, don't
         * specify less than 4 MiB of memory for your containers.
         * </p>
         * 
         * @param memory
         *        The amount (in MiB) of memory to present to the container. If your container attempts to exceed the
         *        memory specified here, the container is killed. The total amount of memory reserved for all containers
         *        within a task must be lower than the task <code>memory</code> value, if one is specified. This
         *        parameter maps to <code>Memory</code> in the docker container create command and the
         *        <code>--memory</code> option to docker run.</p>
         *        <p>
         *        If using the Fargate launch type, this parameter is optional.
         *        </p>
         *        <p>
         *        If using the EC2 launch type, you must specify either a task-level memory value or a container-level
         *        memory value. If you specify both a container-level <code>memory</code> and
         *        <code>memoryReservation</code> value, <code>memory</code> must be greater than
         *        <code>memoryReservation</code>. If you specify <code>memoryReservation</code>, then that value is
         *        subtracted from the available memory resources for the container instance where the container is
         *        placed. Otherwise, the value of <code>memory</code> is used.
         *        </p>
         *        <p>
         *        The Docker 20.10.0 or later daemon reserves a minimum of 6 MiB of memory for a container. So, don't
         *        specify less than 6 MiB of memory for your containers.
         *        </p>
         *        <p>
         *        The Docker 19.03.13-ce or earlier daemon reserves a minimum of 4 MiB of memory for a container. So,
         *        don't specify less than 4 MiB of memory for your containers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memory(Integer memory);

        /**
         * <p>
         * The soft limit (in MiB) of memory to reserve for the container. When system memory is under heavy contention,
         * Docker attempts to keep the container memory to this soft limit. However, your container can consume more
         * memory when it needs to, up to either the hard limit specified with the <code>memory</code> parameter (if
         * applicable), or all of the available memory on the container instance, whichever comes first. This parameter
         * maps to <code>MemoryReservation</code> in the docker container create command and the
         * <code>--memory-reservation</code> option to docker run.
         * </p>
         * <p>
         * If a task-level memory value is not specified, you must specify a non-zero integer for one or both of
         * <code>memory</code> or <code>memoryReservation</code> in a container definition. If you specify both,
         * <code>memory</code> must be greater than <code>memoryReservation</code>. If you specify
         * <code>memoryReservation</code>, then that value is subtracted from the available memory resources for the
         * container instance where the container is placed. Otherwise, the value of <code>memory</code> is used.
         * </p>
         * <p>
         * For example, if your container normally uses 128 MiB of memory, but occasionally bursts to 256 MiB of memory
         * for short periods of time, you can set a <code>memoryReservation</code> of 128 MiB, and a <code>memory</code>
         * hard limit of 300 MiB. This configuration would allow the container to only reserve 128 MiB of memory from
         * the remaining resources on the container instance, but also allow the container to consume more memory
         * resources when needed.
         * </p>
         * <p>
         * The Docker 20.10.0 or later daemon reserves a minimum of 6 MiB of memory for a container. So, don't specify
         * less than 6 MiB of memory for your containers.
         * </p>
         * <p>
         * The Docker 19.03.13-ce or earlier daemon reserves a minimum of 4 MiB of memory for a container. So, don't
         * specify less than 4 MiB of memory for your containers.
         * </p>
         * 
         * @param memoryReservation
         *        The soft limit (in MiB) of memory to reserve for the container. When system memory is under heavy
         *        contention, Docker attempts to keep the container memory to this soft limit. However, your container
         *        can consume more memory when it needs to, up to either the hard limit specified with the
         *        <code>memory</code> parameter (if applicable), or all of the available memory on the container
         *        instance, whichever comes first. This parameter maps to <code>MemoryReservation</code> in the docker
         *        container create command and the <code>--memory-reservation</code> option to docker run.</p>
         *        <p>
         *        If a task-level memory value is not specified, you must specify a non-zero integer for one or both of
         *        <code>memory</code> or <code>memoryReservation</code> in a container definition. If you specify both,
         *        <code>memory</code> must be greater than <code>memoryReservation</code>. If you specify
         *        <code>memoryReservation</code>, then that value is subtracted from the available memory resources for
         *        the container instance where the container is placed. Otherwise, the value of <code>memory</code> is
         *        used.
         *        </p>
         *        <p>
         *        For example, if your container normally uses 128 MiB of memory, but occasionally bursts to 256 MiB of
         *        memory for short periods of time, you can set a <code>memoryReservation</code> of 128 MiB, and a
         *        <code>memory</code> hard limit of 300 MiB. This configuration would allow the container to only
         *        reserve 128 MiB of memory from the remaining resources on the container instance, but also allow the
         *        container to consume more memory resources when needed.
         *        </p>
         *        <p>
         *        The Docker 20.10.0 or later daemon reserves a minimum of 6 MiB of memory for a container. So, don't
         *        specify less than 6 MiB of memory for your containers.
         *        </p>
         *        <p>
         *        The Docker 19.03.13-ce or earlier daemon reserves a minimum of 4 MiB of memory for a container. So,
         *        don't specify less than 4 MiB of memory for your containers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memoryReservation(Integer memoryReservation);

        /**
         * <p>
         * The <code>links</code> parameter allows containers to communicate with each other without the need for port
         * mappings. This parameter is only supported if the network mode of a task definition is <code>bridge</code>.
         * The <code>name:internalName</code> construct is analogous to <code>name:alias</code> in Docker links. Up to
         * 255 letters (uppercase and lowercase), numbers, underscores, and hyphens are allowed.. This parameter maps to
         * <code>Links</code> in the docker container create command and the <code>--link</code> option to docker run.
         * </p>
         * <note>
         * <p>
         * This parameter is not supported for Windows containers.
         * </p>
         * </note> <important>
         * <p>
         * Containers that are collocated on a single container instance may be able to communicate with each other
         * without requiring links or host port mappings. Network isolation is achieved on the container instance using
         * security groups and VPC settings.
         * </p>
         * </important>
         * 
         * @param links
         *        The <code>links</code> parameter allows containers to communicate with each other without the need for
         *        port mappings. This parameter is only supported if the network mode of a task definition is
         *        <code>bridge</code>. The <code>name:internalName</code> construct is analogous to
         *        <code>name:alias</code> in Docker links. Up to 255 letters (uppercase and lowercase), numbers,
         *        underscores, and hyphens are allowed.. This parameter maps to <code>Links</code> in the docker
         *        container create command and the <code>--link</code> option to docker run.</p> <note>
         *        <p>
         *        This parameter is not supported for Windows containers.
         *        </p>
         *        </note> <important>
         *        <p>
         *        Containers that are collocated on a single container instance may be able to communicate with each
         *        other without requiring links or host port mappings. Network isolation is achieved on the container
         *        instance using security groups and VPC settings.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder links(Collection<String> links);

        /**
         * <p>
         * The <code>links</code> parameter allows containers to communicate with each other without the need for port
         * mappings. This parameter is only supported if the network mode of a task definition is <code>bridge</code>.
         * The <code>name:internalName</code> construct is analogous to <code>name:alias</code> in Docker links. Up to
         * 255 letters (uppercase and lowercase), numbers, underscores, and hyphens are allowed.. This parameter maps to
         * <code>Links</code> in the docker container create command and the <code>--link</code> option to docker run.
         * </p>
         * <note>
         * <p>
         * This parameter is not supported for Windows containers.
         * </p>
         * </note> <important>
         * <p>
         * Containers that are collocated on a single container instance may be able to communicate with each other
         * without requiring links or host port mappings. Network isolation is achieved on the container instance using
         * security groups and VPC settings.
         * </p>
         * </important>
         * 
         * @param links
         *        The <code>links</code> parameter allows containers to communicate with each other without the need for
         *        port mappings. This parameter is only supported if the network mode of a task definition is
         *        <code>bridge</code>. The <code>name:internalName</code> construct is analogous to
         *        <code>name:alias</code> in Docker links. Up to 255 letters (uppercase and lowercase), numbers,
         *        underscores, and hyphens are allowed.. This parameter maps to <code>Links</code> in the docker
         *        container create command and the <code>--link</code> option to docker run.</p> <note>
         *        <p>
         *        This parameter is not supported for Windows containers.
         *        </p>
         *        </note> <important>
         *        <p>
         *        Containers that are collocated on a single container instance may be able to communicate with each
         *        other without requiring links or host port mappings. Network isolation is achieved on the container
         *        instance using security groups and VPC settings.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder links(String... links);

        /**
         * <p>
         * The list of port mappings for the container. Port mappings allow containers to access ports on the host
         * container instance to send or receive traffic.
         * </p>
         * <p>
         * For task definitions that use the <code>awsvpc</code> network mode, only specify the
         * <code>containerPort</code>. The <code>hostPort</code> can be left blank or it must be the same value as the
         * <code>containerPort</code>.
         * </p>
         * <p>
         * Port mappings on Windows use the <code>NetNAT</code> gateway address rather than <code>localhost</code>.
         * There's no loopback for port mappings on Windows, so you can't access a container's mapped port from the host
         * itself.
         * </p>
         * <p>
         * This parameter maps to <code>PortBindings</code> in the the docker container create command and the
         * <code>--publish</code> option to docker run. If the network mode of a task definition is set to
         * <code>none</code>, then you can't specify port mappings. If the network mode of a task definition is set to
         * <code>host</code>, then host ports must either be undefined or they must match the container port in the port
         * mapping.
         * </p>
         * <note>
         * <p>
         * After a task reaches the <code>RUNNING</code> status, manual and automatic host and container port
         * assignments are visible in the <b>Network Bindings</b> section of a container description for a selected task
         * in the Amazon ECS console. The assignments are also visible in the <code>networkBindings</code> section <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTasks.html">DescribeTasks</a>
         * responses.
         * </p>
         * </note>
         * 
         * @param portMappings
         *        The list of port mappings for the container. Port mappings allow containers to access ports on the
         *        host container instance to send or receive traffic.</p>
         *        <p>
         *        For task definitions that use the <code>awsvpc</code> network mode, only specify the
         *        <code>containerPort</code>. The <code>hostPort</code> can be left blank or it must be the same value
         *        as the <code>containerPort</code>.
         *        </p>
         *        <p>
         *        Port mappings on Windows use the <code>NetNAT</code> gateway address rather than
         *        <code>localhost</code>. There's no loopback for port mappings on Windows, so you can't access a
         *        container's mapped port from the host itself.
         *        </p>
         *        <p>
         *        This parameter maps to <code>PortBindings</code> in the the docker container create command and the
         *        <code>--publish</code> option to docker run. If the network mode of a task definition is set to
         *        <code>none</code>, then you can't specify port mappings. If the network mode of a task definition is
         *        set to <code>host</code>, then host ports must either be undefined or they must match the container
         *        port in the port mapping.
         *        </p>
         *        <note>
         *        <p>
         *        After a task reaches the <code>RUNNING</code> status, manual and automatic host and container port
         *        assignments are visible in the <b>Network Bindings</b> section of a container description for a
         *        selected task in the Amazon ECS console. The assignments are also visible in the
         *        <code>networkBindings</code> section <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTasks.html"
         *        >DescribeTasks</a> responses.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder portMappings(Collection<PortMapping> portMappings);

        /**
         * <p>
         * The list of port mappings for the container. Port mappings allow containers to access ports on the host
         * container instance to send or receive traffic.
         * </p>
         * <p>
         * For task definitions that use the <code>awsvpc</code> network mode, only specify the
         * <code>containerPort</code>. The <code>hostPort</code> can be left blank or it must be the same value as the
         * <code>containerPort</code>.
         * </p>
         * <p>
         * Port mappings on Windows use the <code>NetNAT</code> gateway address rather than <code>localhost</code>.
         * There's no loopback for port mappings on Windows, so you can't access a container's mapped port from the host
         * itself.
         * </p>
         * <p>
         * This parameter maps to <code>PortBindings</code> in the the docker container create command and the
         * <code>--publish</code> option to docker run. If the network mode of a task definition is set to
         * <code>none</code>, then you can't specify port mappings. If the network mode of a task definition is set to
         * <code>host</code>, then host ports must either be undefined or they must match the container port in the port
         * mapping.
         * </p>
         * <note>
         * <p>
         * After a task reaches the <code>RUNNING</code> status, manual and automatic host and container port
         * assignments are visible in the <b>Network Bindings</b> section of a container description for a selected task
         * in the Amazon ECS console. The assignments are also visible in the <code>networkBindings</code> section <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTasks.html">DescribeTasks</a>
         * responses.
         * </p>
         * </note>
         * 
         * @param portMappings
         *        The list of port mappings for the container. Port mappings allow containers to access ports on the
         *        host container instance to send or receive traffic.</p>
         *        <p>
         *        For task definitions that use the <code>awsvpc</code> network mode, only specify the
         *        <code>containerPort</code>. The <code>hostPort</code> can be left blank or it must be the same value
         *        as the <code>containerPort</code>.
         *        </p>
         *        <p>
         *        Port mappings on Windows use the <code>NetNAT</code> gateway address rather than
         *        <code>localhost</code>. There's no loopback for port mappings on Windows, so you can't access a
         *        container's mapped port from the host itself.
         *        </p>
         *        <p>
         *        This parameter maps to <code>PortBindings</code> in the the docker container create command and the
         *        <code>--publish</code> option to docker run. If the network mode of a task definition is set to
         *        <code>none</code>, then you can't specify port mappings. If the network mode of a task definition is
         *        set to <code>host</code>, then host ports must either be undefined or they must match the container
         *        port in the port mapping.
         *        </p>
         *        <note>
         *        <p>
         *        After a task reaches the <code>RUNNING</code> status, manual and automatic host and container port
         *        assignments are visible in the <b>Network Bindings</b> section of a container description for a
         *        selected task in the Amazon ECS console. The assignments are also visible in the
         *        <code>networkBindings</code> section <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTasks.html"
         *        >DescribeTasks</a> responses.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder portMappings(PortMapping... portMappings);

        /**
         * <p>
         * The list of port mappings for the container. Port mappings allow containers to access ports on the host
         * container instance to send or receive traffic.
         * </p>
         * <p>
         * For task definitions that use the <code>awsvpc</code> network mode, only specify the
         * <code>containerPort</code>. The <code>hostPort</code> can be left blank or it must be the same value as the
         * <code>containerPort</code>.
         * </p>
         * <p>
         * Port mappings on Windows use the <code>NetNAT</code> gateway address rather than <code>localhost</code>.
         * There's no loopback for port mappings on Windows, so you can't access a container's mapped port from the host
         * itself.
         * </p>
         * <p>
         * This parameter maps to <code>PortBindings</code> in the the docker container create command and the
         * <code>--publish</code> option to docker run. If the network mode of a task definition is set to
         * <code>none</code>, then you can't specify port mappings. If the network mode of a task definition is set to
         * <code>host</code>, then host ports must either be undefined or they must match the container port in the port
         * mapping.
         * </p>
         * <note>
         * <p>
         * After a task reaches the <code>RUNNING</code> status, manual and automatic host and container port
         * assignments are visible in the <b>Network Bindings</b> section of a container description for a selected task
         * in the Amazon ECS console. The assignments are also visible in the <code>networkBindings</code> section <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTasks.html">DescribeTasks</a>
         * responses.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.PortMapping.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ecs.model.PortMapping#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecs.model.PortMapping.Builder#build()} is called immediately and its
         * result is passed to {@link #portMappings(List<PortMapping>)}.
         * 
         * @param portMappings
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecs.model.PortMapping.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #portMappings(java.util.Collection<PortMapping>)
         */
        Builder portMappings(Consumer<PortMapping.Builder>... portMappings);

        /**
         * <p>
         * If the <code>essential</code> parameter of a container is marked as <code>true</code>, and that container
         * fails or stops for any reason, all other containers that are part of the task are stopped. If the
         * <code>essential</code> parameter of a container is marked as <code>false</code>, its failure doesn't affect
         * the rest of the containers in a task. If this parameter is omitted, a container is assumed to be essential.
         * </p>
         * <p>
         * All tasks must have at least one essential container. If you have an application that's composed of multiple
         * containers, group containers that are used for a common purpose into components, and separate the different
         * components into multiple task definitions. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/application_architecture.html">Application
         * Architecture</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param essential
         *        If the <code>essential</code> parameter of a container is marked as <code>true</code>, and that
         *        container fails or stops for any reason, all other containers that are part of the task are stopped.
         *        If the <code>essential</code> parameter of a container is marked as <code>false</code>, its failure
         *        doesn't affect the rest of the containers in a task. If this parameter is omitted, a container is
         *        assumed to be essential.</p>
         *        <p>
         *        All tasks must have at least one essential container. If you have an application that's composed of
         *        multiple containers, group containers that are used for a common purpose into components, and separate
         *        the different components into multiple task definitions. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/application_architecture.html"
         *        >Application Architecture</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder essential(Boolean essential);

        /**
         * <p>
         * The restart policy for a container. When you set up a restart policy, Amazon ECS can restart the container
         * without needing to replace the task. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-restart-policy.html">Restart
         * individual containers in Amazon ECS tasks with container restart policies</a> in the <i>Amazon Elastic
         * Container Service Developer Guide</i>.
         * </p>
         * 
         * @param restartPolicy
         *        The restart policy for a container. When you set up a restart policy, Amazon ECS can restart the
         *        container without needing to replace the task. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-restart-policy.html"
         *        >Restart individual containers in Amazon ECS tasks with container restart policies</a> in the
         *        <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder restartPolicy(ContainerRestartPolicy restartPolicy);

        /**
         * <p>
         * The restart policy for a container. When you set up a restart policy, Amazon ECS can restart the container
         * without needing to replace the task. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-restart-policy.html">Restart
         * individual containers in Amazon ECS tasks with container restart policies</a> in the <i>Amazon Elastic
         * Container Service Developer Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link ContainerRestartPolicy.Builder} avoiding
         * the need to create one manually via {@link ContainerRestartPolicy#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ContainerRestartPolicy.Builder#build()} is called immediately and
         * its result is passed to {@link #restartPolicy(ContainerRestartPolicy)}.
         * 
         * @param restartPolicy
         *        a consumer that will call methods on {@link ContainerRestartPolicy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #restartPolicy(ContainerRestartPolicy)
         */
        default Builder restartPolicy(Consumer<ContainerRestartPolicy.Builder> restartPolicy) {
            return restartPolicy(ContainerRestartPolicy.builder().applyMutation(restartPolicy).build());
        }

        /**
         * <important>
         * <p>
         * Early versions of the Amazon ECS container agent don't properly handle <code>entryPoint</code> parameters. If
         * you have problems using <code>entryPoint</code>, update your container agent or enter your commands and
         * arguments as <code>command</code> array items instead.
         * </p>
         * </important>
         * <p>
         * The entry point that's passed to the container. This parameter maps to <code>Entrypoint</code> in the docker
         * container create command and the <code>--entrypoint</code> option to docker run.
         * </p>
         * 
         * @param entryPoint
         *        <p>
         *        Early versions of the Amazon ECS container agent don't properly handle <code>entryPoint</code>
         *        parameters. If you have problems using <code>entryPoint</code>, update your container agent or enter
         *        your commands and arguments as <code>command</code> array items instead.
         *        </p>
         *        </important>
         *        <p>
         *        The entry point that's passed to the container. This parameter maps to <code>Entrypoint</code> in the
         *        docker container create command and the <code>--entrypoint</code> option to docker run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entryPoint(Collection<String> entryPoint);

        /**
         * <important>
         * <p>
         * Early versions of the Amazon ECS container agent don't properly handle <code>entryPoint</code> parameters. If
         * you have problems using <code>entryPoint</code>, update your container agent or enter your commands and
         * arguments as <code>command</code> array items instead.
         * </p>
         * </important>
         * <p>
         * The entry point that's passed to the container. This parameter maps to <code>Entrypoint</code> in the docker
         * container create command and the <code>--entrypoint</code> option to docker run.
         * </p>
         * 
         * @param entryPoint
         *        <p>
         *        Early versions of the Amazon ECS container agent don't properly handle <code>entryPoint</code>
         *        parameters. If you have problems using <code>entryPoint</code>, update your container agent or enter
         *        your commands and arguments as <code>command</code> array items instead.
         *        </p>
         *        </important>
         *        <p>
         *        The entry point that's passed to the container. This parameter maps to <code>Entrypoint</code> in the
         *        docker container create command and the <code>--entrypoint</code> option to docker run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entryPoint(String... entryPoint);

        /**
         * <p>
         * The command that's passed to the container. This parameter maps to <code>Cmd</code> in the docker container
         * create command and the <code>COMMAND</code> parameter to docker run. If there are multiple arguments, each
         * argument is a separated string in the array.
         * </p>
         * 
         * @param command
         *        The command that's passed to the container. This parameter maps to <code>Cmd</code> in the docker
         *        container create command and the <code>COMMAND</code> parameter to docker run. If there are multiple
         *        arguments, each argument is a separated string in the array.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(Collection<String> command);

        /**
         * <p>
         * The command that's passed to the container. This parameter maps to <code>Cmd</code> in the docker container
         * create command and the <code>COMMAND</code> parameter to docker run. If there are multiple arguments, each
         * argument is a separated string in the array.
         * </p>
         * 
         * @param command
         *        The command that's passed to the container. This parameter maps to <code>Cmd</code> in the docker
         *        container create command and the <code>COMMAND</code> parameter to docker run. If there are multiple
         *        arguments, each argument is a separated string in the array.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(String... command);

        /**
         * <p>
         * The environment variables to pass to a container. This parameter maps to <code>Env</code> in the docker
         * container create command and the <code>--env</code> option to docker run.
         * </p>
         * <important>
         * <p>
         * We don't recommend that you use plaintext environment variables for sensitive information, such as credential
         * data.
         * </p>
         * </important>
         * 
         * @param environment
         *        The environment variables to pass to a container. This parameter maps to <code>Env</code> in the
         *        docker container create command and the <code>--env</code> option to docker run.</p> <important>
         *        <p>
         *        We don't recommend that you use plaintext environment variables for sensitive information, such as
         *        credential data.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(Collection<KeyValuePair> environment);

        /**
         * <p>
         * The environment variables to pass to a container. This parameter maps to <code>Env</code> in the docker
         * container create command and the <code>--env</code> option to docker run.
         * </p>
         * <important>
         * <p>
         * We don't recommend that you use plaintext environment variables for sensitive information, such as credential
         * data.
         * </p>
         * </important>
         * 
         * @param environment
         *        The environment variables to pass to a container. This parameter maps to <code>Env</code> in the
         *        docker container create command and the <code>--env</code> option to docker run.</p> <important>
         *        <p>
         *        We don't recommend that you use plaintext environment variables for sensitive information, such as
         *        credential data.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(KeyValuePair... environment);

        /**
         * <p>
         * The environment variables to pass to a container. This parameter maps to <code>Env</code> in the docker
         * container create command and the <code>--env</code> option to docker run.
         * </p>
         * <important>
         * <p>
         * We don't recommend that you use plaintext environment variables for sensitive information, such as credential
         * data.
         * </p>
         * </important> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.KeyValuePair.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ecs.model.KeyValuePair#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecs.model.KeyValuePair.Builder#build()} is called immediately and its
         * result is passed to {@link #environment(List<KeyValuePair>)}.
         * 
         * @param environment
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecs.model.KeyValuePair.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #environment(java.util.Collection<KeyValuePair>)
         */
        Builder environment(Consumer<KeyValuePair.Builder>... environment);

        /**
         * <p>
         * A list of files containing the environment variables to pass to a container. This parameter maps to the
         * <code>--env-file</code> option to docker run.
         * </p>
         * <p>
         * You can specify up to ten environment files. The file must have a <code>.env</code> file extension. Each line
         * in an environment file contains an environment variable in <code>VARIABLE=VALUE</code> format. Lines
         * beginning with <code>#</code> are treated as comments and are ignored.
         * </p>
         * <p>
         * If there are environment variables specified using the <code>environment</code> parameter in a container
         * definition, they take precedence over the variables contained within an environment file. If multiple
         * environment files are specified that contain the same variable, they're processed from the top down. We
         * recommend that you use unique variable names. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/taskdef-envfiles.html">Specifying
         * Environment Variables</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param environmentFiles
         *        A list of files containing the environment variables to pass to a container. This parameter maps to
         *        the <code>--env-file</code> option to docker run.</p>
         *        <p>
         *        You can specify up to ten environment files. The file must have a <code>.env</code> file extension.
         *        Each line in an environment file contains an environment variable in <code>VARIABLE=VALUE</code>
         *        format. Lines beginning with <code>#</code> are treated as comments and are ignored.
         *        </p>
         *        <p>
         *        If there are environment variables specified using the <code>environment</code> parameter in a
         *        container definition, they take precedence over the variables contained within an environment file. If
         *        multiple environment files are specified that contain the same variable, they're processed from the
         *        top down. We recommend that you use unique variable names. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/taskdef-envfiles.html">Specifying
         *        Environment Variables</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentFiles(Collection<EnvironmentFile> environmentFiles);

        /**
         * <p>
         * A list of files containing the environment variables to pass to a container. This parameter maps to the
         * <code>--env-file</code> option to docker run.
         * </p>
         * <p>
         * You can specify up to ten environment files. The file must have a <code>.env</code> file extension. Each line
         * in an environment file contains an environment variable in <code>VARIABLE=VALUE</code> format. Lines
         * beginning with <code>#</code> are treated as comments and are ignored.
         * </p>
         * <p>
         * If there are environment variables specified using the <code>environment</code> parameter in a container
         * definition, they take precedence over the variables contained within an environment file. If multiple
         * environment files are specified that contain the same variable, they're processed from the top down. We
         * recommend that you use unique variable names. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/taskdef-envfiles.html">Specifying
         * Environment Variables</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param environmentFiles
         *        A list of files containing the environment variables to pass to a container. This parameter maps to
         *        the <code>--env-file</code> option to docker run.</p>
         *        <p>
         *        You can specify up to ten environment files. The file must have a <code>.env</code> file extension.
         *        Each line in an environment file contains an environment variable in <code>VARIABLE=VALUE</code>
         *        format. Lines beginning with <code>#</code> are treated as comments and are ignored.
         *        </p>
         *        <p>
         *        If there are environment variables specified using the <code>environment</code> parameter in a
         *        container definition, they take precedence over the variables contained within an environment file. If
         *        multiple environment files are specified that contain the same variable, they're processed from the
         *        top down. We recommend that you use unique variable names. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/taskdef-envfiles.html">Specifying
         *        Environment Variables</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentFiles(EnvironmentFile... environmentFiles);

        /**
         * <p>
         * A list of files containing the environment variables to pass to a container. This parameter maps to the
         * <code>--env-file</code> option to docker run.
         * </p>
         * <p>
         * You can specify up to ten environment files. The file must have a <code>.env</code> file extension. Each line
         * in an environment file contains an environment variable in <code>VARIABLE=VALUE</code> format. Lines
         * beginning with <code>#</code> are treated as comments and are ignored.
         * </p>
         * <p>
         * If there are environment variables specified using the <code>environment</code> parameter in a container
         * definition, they take precedence over the variables contained within an environment file. If multiple
         * environment files are specified that contain the same variable, they're processed from the top down. We
         * recommend that you use unique variable names. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/taskdef-envfiles.html">Specifying
         * Environment Variables</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.EnvironmentFile.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ecs.model.EnvironmentFile#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecs.model.EnvironmentFile.Builder#build()} is called immediately and
         * its result is passed to {@link #environmentFiles(List<EnvironmentFile>)}.
         * 
         * @param environmentFiles
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecs.model.EnvironmentFile.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #environmentFiles(java.util.Collection<EnvironmentFile>)
         */
        Builder environmentFiles(Consumer<EnvironmentFile.Builder>... environmentFiles);

        /**
         * <p>
         * The mount points for data volumes in your container.
         * </p>
         * <p>
         * This parameter maps to <code>Volumes</code> in the docker container create command and the
         * <code>--volume</code> option to docker run.
         * </p>
         * <p>
         * Windows containers can mount whole directories on the same drive as <code>$env:ProgramData</code>. Windows
         * containers can't mount directories on a different drive, and mount point can't be across drives.
         * </p>
         * 
         * @param mountPoints
         *        The mount points for data volumes in your container.</p>
         *        <p>
         *        This parameter maps to <code>Volumes</code> in the docker container create command and the
         *        <code>--volume</code> option to docker run.
         *        </p>
         *        <p>
         *        Windows containers can mount whole directories on the same drive as <code>$env:ProgramData</code>.
         *        Windows containers can't mount directories on a different drive, and mount point can't be across
         *        drives.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mountPoints(Collection<MountPoint> mountPoints);

        /**
         * <p>
         * The mount points for data volumes in your container.
         * </p>
         * <p>
         * This parameter maps to <code>Volumes</code> in the docker container create command and the
         * <code>--volume</code> option to docker run.
         * </p>
         * <p>
         * Windows containers can mount whole directories on the same drive as <code>$env:ProgramData</code>. Windows
         * containers can't mount directories on a different drive, and mount point can't be across drives.
         * </p>
         * 
         * @param mountPoints
         *        The mount points for data volumes in your container.</p>
         *        <p>
         *        This parameter maps to <code>Volumes</code> in the docker container create command and the
         *        <code>--volume</code> option to docker run.
         *        </p>
         *        <p>
         *        Windows containers can mount whole directories on the same drive as <code>$env:ProgramData</code>.
         *        Windows containers can't mount directories on a different drive, and mount point can't be across
         *        drives.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mountPoints(MountPoint... mountPoints);

        /**
         * <p>
         * The mount points for data volumes in your container.
         * </p>
         * <p>
         * This parameter maps to <code>Volumes</code> in the docker container create command and the
         * <code>--volume</code> option to docker run.
         * </p>
         * <p>
         * Windows containers can mount whole directories on the same drive as <code>$env:ProgramData</code>. Windows
         * containers can't mount directories on a different drive, and mount point can't be across drives.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.MountPoint.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.ecs.model.MountPoint#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecs.model.MountPoint.Builder#build()} is called immediately and its
         * result is passed to {@link #mountPoints(List<MountPoint>)}.
         * 
         * @param mountPoints
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecs.model.MountPoint.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #mountPoints(java.util.Collection<MountPoint>)
         */
        Builder mountPoints(Consumer<MountPoint.Builder>... mountPoints);

        /**
         * <p>
         * Data volumes to mount from another container. This parameter maps to <code>VolumesFrom</code> in the docker
         * container create command and the <code>--volumes-from</code> option to docker run.
         * </p>
         * 
         * @param volumesFrom
         *        Data volumes to mount from another container. This parameter maps to <code>VolumesFrom</code> in the
         *        docker container create command and the <code>--volumes-from</code> option to docker run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumesFrom(Collection<VolumeFrom> volumesFrom);

        /**
         * <p>
         * Data volumes to mount from another container. This parameter maps to <code>VolumesFrom</code> in the docker
         * container create command and the <code>--volumes-from</code> option to docker run.
         * </p>
         * 
         * @param volumesFrom
         *        Data volumes to mount from another container. This parameter maps to <code>VolumesFrom</code> in the
         *        docker container create command and the <code>--volumes-from</code> option to docker run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumesFrom(VolumeFrom... volumesFrom);

        /**
         * <p>
         * Data volumes to mount from another container. This parameter maps to <code>VolumesFrom</code> in the docker
         * container create command and the <code>--volumes-from</code> option to docker run.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.VolumeFrom.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.ecs.model.VolumeFrom#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecs.model.VolumeFrom.Builder#build()} is called immediately and its
         * result is passed to {@link #volumesFrom(List<VolumeFrom>)}.
         * 
         * @param volumesFrom
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecs.model.VolumeFrom.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #volumesFrom(java.util.Collection<VolumeFrom>)
         */
        Builder volumesFrom(Consumer<VolumeFrom.Builder>... volumesFrom);

        /**
         * <p>
         * Linux-specific modifications that are applied to the default Docker container configuration, such as Linux
         * kernel capabilities. For more information see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html"
         * >KernelCapabilities</a>.
         * </p>
         * <note>
         * <p>
         * This parameter is not supported for Windows containers.
         * </p>
         * </note>
         * 
         * @param linuxParameters
         *        Linux-specific modifications that are applied to the default Docker container configuration, such as
         *        Linux kernel capabilities. For more information see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html"
         *        >KernelCapabilities</a>.</p> <note>
         *        <p>
         *        This parameter is not supported for Windows containers.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder linuxParameters(LinuxParameters linuxParameters);

        /**
         * <p>
         * Linux-specific modifications that are applied to the default Docker container configuration, such as Linux
         * kernel capabilities. For more information see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html"
         * >KernelCapabilities</a>.
         * </p>
         * <note>
         * <p>
         * This parameter is not supported for Windows containers.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link LinuxParameters.Builder} avoiding
         * the need to create one manually via {@link LinuxParameters#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LinuxParameters.Builder#build()} is called immediately and its
         * result is passed to {@link #linuxParameters(LinuxParameters)}.
         * 
         * @param linuxParameters
         *        a consumer that will call methods on {@link LinuxParameters.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #linuxParameters(LinuxParameters)
         */
        default Builder linuxParameters(Consumer<LinuxParameters.Builder> linuxParameters) {
            return linuxParameters(LinuxParameters.builder().applyMutation(linuxParameters).build());
        }

        /**
         * <p>
         * The secrets to pass to the container. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html">Specifying
         * Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param secrets
         *        The secrets to pass to the container. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html"
         *        >Specifying Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secrets(Collection<Secret> secrets);

        /**
         * <p>
         * The secrets to pass to the container. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html">Specifying
         * Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param secrets
         *        The secrets to pass to the container. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html"
         *        >Specifying Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secrets(Secret... secrets);

        /**
         * <p>
         * The secrets to pass to the container. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html">Specifying
         * Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.Secret.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ecs.model.Secret#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ecs.model.Secret.Builder#build()}
         * is called immediately and its result is passed to {@link #secrets(List<Secret>)}.
         * 
         * @param secrets
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ecs.model.Secret.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #secrets(java.util.Collection<Secret>)
         */
        Builder secrets(Consumer<Secret.Builder>... secrets);

        /**
         * <p>
         * The dependencies defined for container startup and shutdown. A container can contain multiple dependencies on
         * other containers in a task definition. When a dependency is defined for container startup, for container
         * shutdown it is reversed.
         * </p>
         * <p>
         * For tasks using the EC2 launch type, the container instances require at least version 1.26.0 of the container
         * agent to turn on container dependencies. However, we recommend using the latest container agent version. For
         * information about checking your agent version and updating to the latest version, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html">Updating the Amazon
         * ECS Container Agent</a> in the <i>Amazon Elastic Container Service Developer Guide</i>. If you're using an
         * Amazon ECS-optimized Linux AMI, your instance needs at least version 1.26.0-1 of the <code>ecs-init</code>
         * package. If your container instances are launched from version <code>20190301</code> or later, then they
         * contain the required versions of the container agent and <code>ecs-init</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html">Amazon
         * ECS-optimized Linux AMI</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <p>
         * For tasks using the Fargate launch type, the task or service requires the following platforms:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Linux platform version <code>1.3.0</code> or later.
         * </p>
         * </li>
         * <li>
         * <p>
         * Windows platform version <code>1.0.0</code> or later.
         * </p>
         * </li>
         * </ul>
         * 
         * @param dependsOn
         *        The dependencies defined for container startup and shutdown. A container can contain multiple
         *        dependencies on other containers in a task definition. When a dependency is defined for container
         *        startup, for container shutdown it is reversed.</p>
         *        <p>
         *        For tasks using the EC2 launch type, the container instances require at least version 1.26.0 of the
         *        container agent to turn on container dependencies. However, we recommend using the latest container
         *        agent version. For information about checking your agent version and updating to the latest version,
         *        see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html">Updating the
         *        Amazon ECS Container Agent</a> in the <i>Amazon Elastic Container Service Developer Guide</i>. If
         *        you're using an Amazon ECS-optimized Linux AMI, your instance needs at least version 1.26.0-1 of the
         *        <code>ecs-init</code> package. If your container instances are launched from version
         *        <code>20190301</code> or later, then they contain the required versions of the container agent and
         *        <code>ecs-init</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html">Amazon
         *        ECS-optimized Linux AMI</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         *        </p>
         *        <p>
         *        For tasks using the Fargate launch type, the task or service requires the following platforms:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Linux platform version <code>1.3.0</code> or later.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Windows platform version <code>1.0.0</code> or later.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dependsOn(Collection<ContainerDependency> dependsOn);

        /**
         * <p>
         * The dependencies defined for container startup and shutdown. A container can contain multiple dependencies on
         * other containers in a task definition. When a dependency is defined for container startup, for container
         * shutdown it is reversed.
         * </p>
         * <p>
         * For tasks using the EC2 launch type, the container instances require at least version 1.26.0 of the container
         * agent to turn on container dependencies. However, we recommend using the latest container agent version. For
         * information about checking your agent version and updating to the latest version, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html">Updating the Amazon
         * ECS Container Agent</a> in the <i>Amazon Elastic Container Service Developer Guide</i>. If you're using an
         * Amazon ECS-optimized Linux AMI, your instance needs at least version 1.26.0-1 of the <code>ecs-init</code>
         * package. If your container instances are launched from version <code>20190301</code> or later, then they
         * contain the required versions of the container agent and <code>ecs-init</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html">Amazon
         * ECS-optimized Linux AMI</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <p>
         * For tasks using the Fargate launch type, the task or service requires the following platforms:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Linux platform version <code>1.3.0</code> or later.
         * </p>
         * </li>
         * <li>
         * <p>
         * Windows platform version <code>1.0.0</code> or later.
         * </p>
         * </li>
         * </ul>
         * 
         * @param dependsOn
         *        The dependencies defined for container startup and shutdown. A container can contain multiple
         *        dependencies on other containers in a task definition. When a dependency is defined for container
         *        startup, for container shutdown it is reversed.</p>
         *        <p>
         *        For tasks using the EC2 launch type, the container instances require at least version 1.26.0 of the
         *        container agent to turn on container dependencies. However, we recommend using the latest container
         *        agent version. For information about checking your agent version and updating to the latest version,
         *        see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html">Updating the
         *        Amazon ECS Container Agent</a> in the <i>Amazon Elastic Container Service Developer Guide</i>. If
         *        you're using an Amazon ECS-optimized Linux AMI, your instance needs at least version 1.26.0-1 of the
         *        <code>ecs-init</code> package. If your container instances are launched from version
         *        <code>20190301</code> or later, then they contain the required versions of the container agent and
         *        <code>ecs-init</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html">Amazon
         *        ECS-optimized Linux AMI</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         *        </p>
         *        <p>
         *        For tasks using the Fargate launch type, the task or service requires the following platforms:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Linux platform version <code>1.3.0</code> or later.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Windows platform version <code>1.0.0</code> or later.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dependsOn(ContainerDependency... dependsOn);

        /**
         * <p>
         * The dependencies defined for container startup and shutdown. A container can contain multiple dependencies on
         * other containers in a task definition. When a dependency is defined for container startup, for container
         * shutdown it is reversed.
         * </p>
         * <p>
         * For tasks using the EC2 launch type, the container instances require at least version 1.26.0 of the container
         * agent to turn on container dependencies. However, we recommend using the latest container agent version. For
         * information about checking your agent version and updating to the latest version, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html">Updating the Amazon
         * ECS Container Agent</a> in the <i>Amazon Elastic Container Service Developer Guide</i>. If you're using an
         * Amazon ECS-optimized Linux AMI, your instance needs at least version 1.26.0-1 of the <code>ecs-init</code>
         * package. If your container instances are launched from version <code>20190301</code> or later, then they
         * contain the required versions of the container agent and <code>ecs-init</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html">Amazon
         * ECS-optimized Linux AMI</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <p>
         * For tasks using the Fargate launch type, the task or service requires the following platforms:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Linux platform version <code>1.3.0</code> or later.
         * </p>
         * </li>
         * <li>
         * <p>
         * Windows platform version <code>1.0.0</code> or later.
         * </p>
         * </li>
         * </ul>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.ContainerDependency.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ecs.model.ContainerDependency#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecs.model.ContainerDependency.Builder#build()} is called immediately
         * and its result is passed to {@link #dependsOn(List<ContainerDependency>)}.
         * 
         * @param dependsOn
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecs.model.ContainerDependency.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dependsOn(java.util.Collection<ContainerDependency>)
         */
        Builder dependsOn(Consumer<ContainerDependency.Builder>... dependsOn);

        /**
         * <p>
         * Time duration (in seconds) to wait before giving up on resolving dependencies for a container. For example,
         * you specify two containers in a task definition with containerA having a dependency on containerB reaching a
         * <code>COMPLETE</code>, <code>SUCCESS</code>, or <code>HEALTHY</code> status. If a <code>startTimeout</code>
         * value is specified for containerB and it doesn't reach the desired status within that time then containerA
         * gives up and not start. This results in the task transitioning to a <code>STOPPED</code> state.
         * </p>
         * <note>
         * <p>
         * When the <code>ECS_CONTAINER_START_TIMEOUT</code> container agent configuration variable is used, it's
         * enforced independently from this start timeout value.
         * </p>
         * </note>
         * <p>
         * For tasks using the Fargate launch type, the task or service requires the following platforms:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Linux platform version <code>1.3.0</code> or later.
         * </p>
         * </li>
         * <li>
         * <p>
         * Windows platform version <code>1.0.0</code> or later.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For tasks using the EC2 launch type, your container instances require at least version <code>1.26.0</code> of
         * the container agent to use a container start timeout value. However, we recommend using the latest container
         * agent version. For information about checking your agent version and updating to the latest version, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html">Updating the Amazon
         * ECS Container Agent</a> in the <i>Amazon Elastic Container Service Developer Guide</i>. If you're using an
         * Amazon ECS-optimized Linux AMI, your instance needs at least version <code>1.26.0-1</code> of the
         * <code>ecs-init</code> package. If your container instances are launched from version <code>20190301</code> or
         * later, then they contain the required versions of the container agent and <code>ecs-init</code>. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html">Amazon
         * ECS-optimized Linux AMI</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <p>
         * The valid values for Fargate are 2-120 seconds.
         * </p>
         * 
         * @param startTimeout
         *        Time duration (in seconds) to wait before giving up on resolving dependencies for a container. For
         *        example, you specify two containers in a task definition with containerA having a dependency on
         *        containerB reaching a <code>COMPLETE</code>, <code>SUCCESS</code>, or <code>HEALTHY</code> status. If
         *        a <code>startTimeout</code> value is specified for containerB and it doesn't reach the desired status
         *        within that time then containerA gives up and not start. This results in the task transitioning to a
         *        <code>STOPPED</code> state.</p> <note>
         *        <p>
         *        When the <code>ECS_CONTAINER_START_TIMEOUT</code> container agent configuration variable is used, it's
         *        enforced independently from this start timeout value.
         *        </p>
         *        </note>
         *        <p>
         *        For tasks using the Fargate launch type, the task or service requires the following platforms:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Linux platform version <code>1.3.0</code> or later.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Windows platform version <code>1.0.0</code> or later.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For tasks using the EC2 launch type, your container instances require at least version
         *        <code>1.26.0</code> of the container agent to use a container start timeout value. However, we
         *        recommend using the latest container agent version. For information about checking your agent version
         *        and updating to the latest version, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html">Updating the
         *        Amazon ECS Container Agent</a> in the <i>Amazon Elastic Container Service Developer Guide</i>. If
         *        you're using an Amazon ECS-optimized Linux AMI, your instance needs at least version
         *        <code>1.26.0-1</code> of the <code>ecs-init</code> package. If your container instances are launched
         *        from version <code>20190301</code> or later, then they contain the required versions of the container
         *        agent and <code>ecs-init</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html">Amazon
         *        ECS-optimized Linux AMI</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         *        </p>
         *        <p>
         *        The valid values for Fargate are 2-120 seconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTimeout(Integer startTimeout);

        /**
         * <p>
         * Time duration (in seconds) to wait before the container is forcefully killed if it doesn't exit normally on
         * its own.
         * </p>
         * <p>
         * For tasks using the Fargate launch type, the task or service requires the following platforms:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Linux platform version <code>1.3.0</code> or later.
         * </p>
         * </li>
         * <li>
         * <p>
         * Windows platform version <code>1.0.0</code> or later.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For tasks that use the Fargate launch type, the max stop timeout value is 120 seconds and if the parameter is
         * not specified, the default value of 30 seconds is used.
         * </p>
         * <p>
         * For tasks that use the EC2 launch type, if the <code>stopTimeout</code> parameter isn't specified, the value
         * set for the Amazon ECS container agent configuration variable <code>ECS_CONTAINER_STOP_TIMEOUT</code> is
         * used. If neither the <code>stopTimeout</code> parameter or the <code>ECS_CONTAINER_STOP_TIMEOUT</code> agent
         * configuration variable are set, then the default values of 30 seconds for Linux containers and 30 seconds on
         * Windows containers are used. Your container instances require at least version 1.26.0 of the container agent
         * to use a container stop timeout value. However, we recommend using the latest container agent version. For
         * information about checking your agent version and updating to the latest version, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html">Updating the Amazon
         * ECS Container Agent</a> in the <i>Amazon Elastic Container Service Developer Guide</i>. If you're using an
         * Amazon ECS-optimized Linux AMI, your instance needs at least version 1.26.0-1 of the <code>ecs-init</code>
         * package. If your container instances are launched from version <code>20190301</code> or later, then they
         * contain the required versions of the container agent and <code>ecs-init</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html">Amazon
         * ECS-optimized Linux AMI</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <p>
         * The valid values for Fargate are 2-120 seconds.
         * </p>
         * 
         * @param stopTimeout
         *        Time duration (in seconds) to wait before the container is forcefully killed if it doesn't exit
         *        normally on its own.</p>
         *        <p>
         *        For tasks using the Fargate launch type, the task or service requires the following platforms:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Linux platform version <code>1.3.0</code> or later.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Windows platform version <code>1.0.0</code> or later.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For tasks that use the Fargate launch type, the max stop timeout value is 120 seconds and if the
         *        parameter is not specified, the default value of 30 seconds is used.
         *        </p>
         *        <p>
         *        For tasks that use the EC2 launch type, if the <code>stopTimeout</code> parameter isn't specified, the
         *        value set for the Amazon ECS container agent configuration variable
         *        <code>ECS_CONTAINER_STOP_TIMEOUT</code> is used. If neither the <code>stopTimeout</code> parameter or
         *        the <code>ECS_CONTAINER_STOP_TIMEOUT</code> agent configuration variable are set, then the default
         *        values of 30 seconds for Linux containers and 30 seconds on Windows containers are used. Your
         *        container instances require at least version 1.26.0 of the container agent to use a container stop
         *        timeout value. However, we recommend using the latest container agent version. For information about
         *        checking your agent version and updating to the latest version, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html">Updating the
         *        Amazon ECS Container Agent</a> in the <i>Amazon Elastic Container Service Developer Guide</i>. If
         *        you're using an Amazon ECS-optimized Linux AMI, your instance needs at least version 1.26.0-1 of the
         *        <code>ecs-init</code> package. If your container instances are launched from version
         *        <code>20190301</code> or later, then they contain the required versions of the container agent and
         *        <code>ecs-init</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html">Amazon
         *        ECS-optimized Linux AMI</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         *        </p>
         *        <p>
         *        The valid values for Fargate are 2-120 seconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stopTimeout(Integer stopTimeout);

        /**
         * <p>
         * Specifies whether Amazon ECS will resolve the container image tag provided in the container definition to an
         * image digest. By default, the value is <code>enabled</code>. If you set the value for a container as
         * <code>disabled</code>, Amazon ECS will not resolve the provided container image tag to a digest and will use
         * the original image URI specified in the container definition for deployment. For more information about
         * container image resolution, see <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html#deployment-container-image-stability"
         * >Container image resolution</a> in the <i>Amazon ECS Developer Guide</i>.
         * </p>
         * 
         * @param versionConsistency
         *        Specifies whether Amazon ECS will resolve the container image tag provided in the container definition
         *        to an image digest. By default, the value is <code>enabled</code>. If you set the value for a
         *        container as <code>disabled</code>, Amazon ECS will not resolve the provided container image tag to a
         *        digest and will use the original image URI specified in the container definition for deployment. For
         *        more information about container image resolution, see <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html#deployment-container-image-stability"
         *        >Container image resolution</a> in the <i>Amazon ECS Developer Guide</i>.
         * @see VersionConsistency
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VersionConsistency
         */
        Builder versionConsistency(String versionConsistency);

        /**
         * <p>
         * Specifies whether Amazon ECS will resolve the container image tag provided in the container definition to an
         * image digest. By default, the value is <code>enabled</code>. If you set the value for a container as
         * <code>disabled</code>, Amazon ECS will not resolve the provided container image tag to a digest and will use
         * the original image URI specified in the container definition for deployment. For more information about
         * container image resolution, see <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html#deployment-container-image-stability"
         * >Container image resolution</a> in the <i>Amazon ECS Developer Guide</i>.
         * </p>
         * 
         * @param versionConsistency
         *        Specifies whether Amazon ECS will resolve the container image tag provided in the container definition
         *        to an image digest. By default, the value is <code>enabled</code>. If you set the value for a
         *        container as <code>disabled</code>, Amazon ECS will not resolve the provided container image tag to a
         *        digest and will use the original image URI specified in the container definition for deployment. For
         *        more information about container image resolution, see <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html#deployment-container-image-stability"
         *        >Container image resolution</a> in the <i>Amazon ECS Developer Guide</i>.
         * @see VersionConsistency
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VersionConsistency
         */
        Builder versionConsistency(VersionConsistency versionConsistency);

        /**
         * <p>
         * The hostname to use for your container. This parameter maps to <code>Hostname</code> in the docker container
         * create command and the <code>--hostname</code> option to docker run.
         * </p>
         * <note>
         * <p>
         * The <code>hostname</code> parameter is not supported if you're using the <code>awsvpc</code> network mode.
         * </p>
         * </note>
         * 
         * @param hostname
         *        The hostname to use for your container. This parameter maps to <code>Hostname</code> in the docker
         *        container create command and the <code>--hostname</code> option to docker run.</p> <note>
         *        <p>
         *        The <code>hostname</code> parameter is not supported if you're using the <code>awsvpc</code> network
         *        mode.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hostname(String hostname);

        /**
         * <p>
         * The user to use inside the container. This parameter maps to <code>User</code> in the docker container create
         * command and the <code>--user</code> option to docker run.
         * </p>
         * <important>
         * <p>
         * When running tasks using the <code>host</code> network mode, don't run containers using the root user (UID
         * 0). We recommend using a non-root user for better security.
         * </p>
         * </important>
         * <p>
         * You can specify the <code>user</code> using the following formats. If specifying a UID or GID, you must
         * specify it as a positive integer.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>user</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>user:group</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>uid</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>uid:gid</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>user:gid</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>uid:group</code>
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * This parameter is not supported for Windows containers.
         * </p>
         * </note>
         * 
         * @param user
         *        The user to use inside the container. This parameter maps to <code>User</code> in the docker container
         *        create command and the <code>--user</code> option to docker run.</p> <important>
         *        <p>
         *        When running tasks using the <code>host</code> network mode, don't run containers using the root user
         *        (UID 0). We recommend using a non-root user for better security.
         *        </p>
         *        </important>
         *        <p>
         *        You can specify the <code>user</code> using the following formats. If specifying a UID or GID, you
         *        must specify it as a positive integer.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>user</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>user:group</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>uid</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>uid:gid</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>user:gid</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>uid:group</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        This parameter is not supported for Windows containers.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder user(String user);

        /**
         * <p>
         * The working directory to run commands inside the container in. This parameter maps to <code>WorkingDir</code>
         * in the docker container create command and the <code>--workdir</code> option to docker run.
         * </p>
         * 
         * @param workingDirectory
         *        The working directory to run commands inside the container in. This parameter maps to
         *        <code>WorkingDir</code> in the docker container create command and the <code>--workdir</code> option
         *        to docker run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workingDirectory(String workingDirectory);

        /**
         * <p>
         * When this parameter is true, networking is off within the container. This parameter maps to
         * <code>NetworkDisabled</code> in the docker container create command.
         * </p>
         * <note>
         * <p>
         * This parameter is not supported for Windows containers.
         * </p>
         * </note>
         * 
         * @param disableNetworking
         *        When this parameter is true, networking is off within the container. This parameter maps to
         *        <code>NetworkDisabled</code> in the docker container create command.</p> <note>
         *        <p>
         *        This parameter is not supported for Windows containers.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder disableNetworking(Boolean disableNetworking);

        /**
         * <p>
         * When this parameter is true, the container is given elevated privileges on the host container instance
         * (similar to the <code>root</code> user). This parameter maps to <code>Privileged</code> in the docker
         * container create command and the <code>--privileged</code> option to docker run
         * </p>
         * <note>
         * <p>
         * This parameter is not supported for Windows containers or tasks run on Fargate.
         * </p>
         * </note>
         * 
         * @param privileged
         *        When this parameter is true, the container is given elevated privileges on the host container instance
         *        (similar to the <code>root</code> user). This parameter maps to <code>Privileged</code> in the docker
         *        container create command and the <code>--privileged</code> option to docker run</p> <note>
         *        <p>
         *        This parameter is not supported for Windows containers or tasks run on Fargate.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privileged(Boolean privileged);

        /**
         * <p>
         * When this parameter is true, the container is given read-only access to its root file system. This parameter
         * maps to <code>ReadonlyRootfs</code> in the docker container create command and the <code>--read-only</code>
         * option to docker run.
         * </p>
         * <note>
         * <p>
         * This parameter is not supported for Windows containers.
         * </p>
         * </note>
         * 
         * @param readonlyRootFilesystem
         *        When this parameter is true, the container is given read-only access to its root file system. This
         *        parameter maps to <code>ReadonlyRootfs</code> in the docker container create command and the
         *        <code>--read-only</code> option to docker run.</p> <note>
         *        <p>
         *        This parameter is not supported for Windows containers.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder readonlyRootFilesystem(Boolean readonlyRootFilesystem);

        /**
         * <p>
         * A list of DNS servers that are presented to the container. This parameter maps to <code>Dns</code> in the
         * docker container create command and the <code>--dns</code> option to docker run.
         * </p>
         * <note>
         * <p>
         * This parameter is not supported for Windows containers.
         * </p>
         * </note>
         * 
         * @param dnsServers
         *        A list of DNS servers that are presented to the container. This parameter maps to <code>Dns</code> in
         *        the docker container create command and the <code>--dns</code> option to docker run.</p> <note>
         *        <p>
         *        This parameter is not supported for Windows containers.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsServers(Collection<String> dnsServers);

        /**
         * <p>
         * A list of DNS servers that are presented to the container. This parameter maps to <code>Dns</code> in the
         * docker container create command and the <code>--dns</code> option to docker run.
         * </p>
         * <note>
         * <p>
         * This parameter is not supported for Windows containers.
         * </p>
         * </note>
         * 
         * @param dnsServers
         *        A list of DNS servers that are presented to the container. This parameter maps to <code>Dns</code> in
         *        the docker container create command and the <code>--dns</code> option to docker run.</p> <note>
         *        <p>
         *        This parameter is not supported for Windows containers.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsServers(String... dnsServers);

        /**
         * <p>
         * A list of DNS search domains that are presented to the container. This parameter maps to
         * <code>DnsSearch</code> in the docker container create command and the <code>--dns-search</code> option to
         * docker run.
         * </p>
         * <note>
         * <p>
         * This parameter is not supported for Windows containers.
         * </p>
         * </note>
         * 
         * @param dnsSearchDomains
         *        A list of DNS search domains that are presented to the container. This parameter maps to
         *        <code>DnsSearch</code> in the docker container create command and the <code>--dns-search</code> option
         *        to docker run.</p> <note>
         *        <p>
         *        This parameter is not supported for Windows containers.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsSearchDomains(Collection<String> dnsSearchDomains);

        /**
         * <p>
         * A list of DNS search domains that are presented to the container. This parameter maps to
         * <code>DnsSearch</code> in the docker container create command and the <code>--dns-search</code> option to
         * docker run.
         * </p>
         * <note>
         * <p>
         * This parameter is not supported for Windows containers.
         * </p>
         * </note>
         * 
         * @param dnsSearchDomains
         *        A list of DNS search domains that are presented to the container. This parameter maps to
         *        <code>DnsSearch</code> in the docker container create command and the <code>--dns-search</code> option
         *        to docker run.</p> <note>
         *        <p>
         *        This parameter is not supported for Windows containers.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsSearchDomains(String... dnsSearchDomains);

        /**
         * <p>
         * A list of hostnames and IP address mappings to append to the <code>/etc/hosts</code> file on the container.
         * This parameter maps to <code>ExtraHosts</code> in the docker container create command and the
         * <code>--add-host</code> option to docker run.
         * </p>
         * <note>
         * <p>
         * This parameter isn't supported for Windows containers or tasks that use the <code>awsvpc</code> network mode.
         * </p>
         * </note>
         * 
         * @param extraHosts
         *        A list of hostnames and IP address mappings to append to the <code>/etc/hosts</code> file on the
         *        container. This parameter maps to <code>ExtraHosts</code> in the docker container create command and
         *        the <code>--add-host</code> option to docker run.</p> <note>
         *        <p>
         *        This parameter isn't supported for Windows containers or tasks that use the <code>awsvpc</code>
         *        network mode.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder extraHosts(Collection<HostEntry> extraHosts);

        /**
         * <p>
         * A list of hostnames and IP address mappings to append to the <code>/etc/hosts</code> file on the container.
         * This parameter maps to <code>ExtraHosts</code> in the docker container create command and the
         * <code>--add-host</code> option to docker run.
         * </p>
         * <note>
         * <p>
         * This parameter isn't supported for Windows containers or tasks that use the <code>awsvpc</code> network mode.
         * </p>
         * </note>
         * 
         * @param extraHosts
         *        A list of hostnames and IP address mappings to append to the <code>/etc/hosts</code> file on the
         *        container. This parameter maps to <code>ExtraHosts</code> in the docker container create command and
         *        the <code>--add-host</code> option to docker run.</p> <note>
         *        <p>
         *        This parameter isn't supported for Windows containers or tasks that use the <code>awsvpc</code>
         *        network mode.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder extraHosts(HostEntry... extraHosts);

        /**
         * <p>
         * A list of hostnames and IP address mappings to append to the <code>/etc/hosts</code> file on the container.
         * This parameter maps to <code>ExtraHosts</code> in the docker container create command and the
         * <code>--add-host</code> option to docker run.
         * </p>
         * <note>
         * <p>
         * This parameter isn't supported for Windows containers or tasks that use the <code>awsvpc</code> network mode.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.HostEntry.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.ecs.model.HostEntry#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecs.model.HostEntry.Builder#build()} is called immediately and its
         * result is passed to {@link #extraHosts(List<HostEntry>)}.
         * 
         * @param extraHosts
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecs.model.HostEntry.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #extraHosts(java.util.Collection<HostEntry>)
         */
        Builder extraHosts(Consumer<HostEntry.Builder>... extraHosts);

        /**
         * <p>
         * A list of strings to provide custom configuration for multiple security systems. This field isn't valid for
         * containers in tasks using the Fargate launch type.
         * </p>
         * <p>
         * For Linux tasks on EC2, this parameter can be used to reference custom labels for SELinux and AppArmor
         * multi-level security systems.
         * </p>
         * <p>
         * For any tasks on EC2, this parameter can be used to reference a credential spec file that configures a
         * container for Active Directory authentication. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/windows-gmsa.html">Using gMSAs for Windows
         * Containers</a> and <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/linux-gmsa.html">Using gMSAs for Linux
         * Containers</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <p>
         * This parameter maps to <code>SecurityOpt</code> in the docker container create command and the
         * <code>--security-opt</code> option to docker run.
         * </p>
         * <note>
         * <p>
         * The Amazon ECS container agent running on a container instance must register with the
         * <code>ECS_SELINUX_CAPABLE=true</code> or <code>ECS_APPARMOR_CAPABLE=true</code> environment variables before
         * containers placed on that instance can use these security options. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon ECS Container
         * Agent Configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * </note>
         * <p>
         * Valid values: "no-new-privileges" | "apparmor:PROFILE" | "label:value" |
         * "credentialspec:CredentialSpecFilePath"
         * </p>
         * 
         * @param dockerSecurityOptions
         *        A list of strings to provide custom configuration for multiple security systems. This field isn't
         *        valid for containers in tasks using the Fargate launch type.</p>
         *        <p>
         *        For Linux tasks on EC2, this parameter can be used to reference custom labels for SELinux and AppArmor
         *        multi-level security systems.
         *        </p>
         *        <p>
         *        For any tasks on EC2, this parameter can be used to reference a credential spec file that configures a
         *        container for Active Directory authentication. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/windows-gmsa.html">Using gMSAs for
         *        Windows Containers</a> and <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/linux-gmsa.html">Using gMSAs for
         *        Linux Containers</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         *        </p>
         *        <p>
         *        This parameter maps to <code>SecurityOpt</code> in the docker container create command and the
         *        <code>--security-opt</code> option to docker run.
         *        </p>
         *        <note>
         *        <p>
         *        The Amazon ECS container agent running on a container instance must register with the
         *        <code>ECS_SELINUX_CAPABLE=true</code> or <code>ECS_APPARMOR_CAPABLE=true</code> environment variables
         *        before containers placed on that instance can use these security options. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon ECS
         *        Container Agent Configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         *        </p>
         *        </note>
         *        <p>
         *        Valid values: "no-new-privileges" | "apparmor:PROFILE" | "label:value" |
         *        "credentialspec:CredentialSpecFilePath"
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dockerSecurityOptions(Collection<String> dockerSecurityOptions);

        /**
         * <p>
         * A list of strings to provide custom configuration for multiple security systems. This field isn't valid for
         * containers in tasks using the Fargate launch type.
         * </p>
         * <p>
         * For Linux tasks on EC2, this parameter can be used to reference custom labels for SELinux and AppArmor
         * multi-level security systems.
         * </p>
         * <p>
         * For any tasks on EC2, this parameter can be used to reference a credential spec file that configures a
         * container for Active Directory authentication. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/windows-gmsa.html">Using gMSAs for Windows
         * Containers</a> and <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/linux-gmsa.html">Using gMSAs for Linux
         * Containers</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <p>
         * This parameter maps to <code>SecurityOpt</code> in the docker container create command and the
         * <code>--security-opt</code> option to docker run.
         * </p>
         * <note>
         * <p>
         * The Amazon ECS container agent running on a container instance must register with the
         * <code>ECS_SELINUX_CAPABLE=true</code> or <code>ECS_APPARMOR_CAPABLE=true</code> environment variables before
         * containers placed on that instance can use these security options. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon ECS Container
         * Agent Configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * </note>
         * <p>
         * Valid values: "no-new-privileges" | "apparmor:PROFILE" | "label:value" |
         * "credentialspec:CredentialSpecFilePath"
         * </p>
         * 
         * @param dockerSecurityOptions
         *        A list of strings to provide custom configuration for multiple security systems. This field isn't
         *        valid for containers in tasks using the Fargate launch type.</p>
         *        <p>
         *        For Linux tasks on EC2, this parameter can be used to reference custom labels for SELinux and AppArmor
         *        multi-level security systems.
         *        </p>
         *        <p>
         *        For any tasks on EC2, this parameter can be used to reference a credential spec file that configures a
         *        container for Active Directory authentication. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/windows-gmsa.html">Using gMSAs for
         *        Windows Containers</a> and <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/linux-gmsa.html">Using gMSAs for
         *        Linux Containers</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         *        </p>
         *        <p>
         *        This parameter maps to <code>SecurityOpt</code> in the docker container create command and the
         *        <code>--security-opt</code> option to docker run.
         *        </p>
         *        <note>
         *        <p>
         *        The Amazon ECS container agent running on a container instance must register with the
         *        <code>ECS_SELINUX_CAPABLE=true</code> or <code>ECS_APPARMOR_CAPABLE=true</code> environment variables
         *        before containers placed on that instance can use these security options. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon ECS
         *        Container Agent Configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         *        </p>
         *        </note>
         *        <p>
         *        Valid values: "no-new-privileges" | "apparmor:PROFILE" | "label:value" |
         *        "credentialspec:CredentialSpecFilePath"
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dockerSecurityOptions(String... dockerSecurityOptions);

        /**
         * <p>
         * When this parameter is <code>true</code>, you can deploy containerized applications that require
         * <code>stdin</code> or a <code>tty</code> to be allocated. This parameter maps to <code>OpenStdin</code> in
         * the docker container create command and the <code>--interactive</code> option to docker run.
         * </p>
         * 
         * @param interactive
         *        When this parameter is <code>true</code>, you can deploy containerized applications that require
         *        <code>stdin</code> or a <code>tty</code> to be allocated. This parameter maps to
         *        <code>OpenStdin</code> in the docker container create command and the <code>--interactive</code>
         *        option to docker run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder interactive(Boolean interactive);

        /**
         * <p>
         * When this parameter is <code>true</code>, a TTY is allocated. This parameter maps to <code>Tty</code> in the
         * docker container create command and the <code>--tty</code> option to docker run.
         * </p>
         * 
         * @param pseudoTerminal
         *        When this parameter is <code>true</code>, a TTY is allocated. This parameter maps to <code>Tty</code>
         *        in the docker container create command and the <code>--tty</code> option to docker run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pseudoTerminal(Boolean pseudoTerminal);

        /**
         * <p>
         * A key/value map of labels to add to the container. This parameter maps to <code>Labels</code> in the docker
         * container create command and the <code>--label</code> option to docker run. This parameter requires version
         * 1.18 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version
         * on your container instance, log in to your container instance and run the following command:
         * <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
         * </p>
         * 
         * @param dockerLabels
         *        A key/value map of labels to add to the container. This parameter maps to <code>Labels</code> in the
         *        docker container create command and the <code>--label</code> option to docker run. This parameter
         *        requires version 1.18 of the Docker Remote API or greater on your container instance. To check the
         *        Docker Remote API version on your container instance, log in to your container instance and run the
         *        following command: <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dockerLabels(Map<String, String> dockerLabels);

        /**
         * <p>
         * A list of <code>ulimits</code> to set in the container. If a <code>ulimit</code> value is specified in a task
         * definition, it overrides the default values set by Docker. This parameter maps to <code>Ulimits</code> in the
         * docker container create command and the <code>--ulimit</code> option to docker run. Valid naming values are
         * displayed in the <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_Ulimit.html">Ulimit</a> data type.
         * </p>
         * <p>
         * Amazon ECS tasks hosted on Fargate use the default resource limit values set by the operating system with the
         * exception of the <code>nofile</code> resource limit parameter which Fargate overrides. The
         * <code>nofile</code> resource limit sets a restriction on the number of open files that a container can use.
         * The default <code>nofile</code> soft limit is <code> 65535</code> and the default hard limit is
         * <code>65535</code>.
         * </p>
         * <p>
         * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check
         * the Docker Remote API version on your container instance, log in to your container instance and run the
         * following command: <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
         * </p>
         * <note>
         * <p>
         * This parameter is not supported for Windows containers.
         * </p>
         * </note>
         * 
         * @param ulimits
         *        A list of <code>ulimits</code> to set in the container. If a <code>ulimit</code> value is specified in
         *        a task definition, it overrides the default values set by Docker. This parameter maps to
         *        <code>Ulimits</code> in the docker container create command and the <code>--ulimit</code> option to
         *        docker run. Valid naming values are displayed in the <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_Ulimit.html">Ulimit</a> data
         *        type.</p>
         *        <p>
         *        Amazon ECS tasks hosted on Fargate use the default resource limit values set by the operating system
         *        with the exception of the <code>nofile</code> resource limit parameter which Fargate overrides. The
         *        <code>nofile</code> resource limit sets a restriction on the number of open files that a container can
         *        use. The default <code>nofile</code> soft limit is <code> 65535</code> and the default hard limit is
         *        <code>65535</code>.
         *        </p>
         *        <p>
         *        This parameter requires version 1.18 of the Docker Remote API or greater on your container instance.
         *        To check the Docker Remote API version on your container instance, log in to your container instance
         *        and run the following command: <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
         *        </p>
         *        <note>
         *        <p>
         *        This parameter is not supported for Windows containers.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ulimits(Collection<Ulimit> ulimits);

        /**
         * <p>
         * A list of <code>ulimits</code> to set in the container. If a <code>ulimit</code> value is specified in a task
         * definition, it overrides the default values set by Docker. This parameter maps to <code>Ulimits</code> in the
         * docker container create command and the <code>--ulimit</code> option to docker run. Valid naming values are
         * displayed in the <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_Ulimit.html">Ulimit</a> data type.
         * </p>
         * <p>
         * Amazon ECS tasks hosted on Fargate use the default resource limit values set by the operating system with the
         * exception of the <code>nofile</code> resource limit parameter which Fargate overrides. The
         * <code>nofile</code> resource limit sets a restriction on the number of open files that a container can use.
         * The default <code>nofile</code> soft limit is <code> 65535</code> and the default hard limit is
         * <code>65535</code>.
         * </p>
         * <p>
         * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check
         * the Docker Remote API version on your container instance, log in to your container instance and run the
         * following command: <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
         * </p>
         * <note>
         * <p>
         * This parameter is not supported for Windows containers.
         * </p>
         * </note>
         * 
         * @param ulimits
         *        A list of <code>ulimits</code> to set in the container. If a <code>ulimit</code> value is specified in
         *        a task definition, it overrides the default values set by Docker. This parameter maps to
         *        <code>Ulimits</code> in the docker container create command and the <code>--ulimit</code> option to
         *        docker run. Valid naming values are displayed in the <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_Ulimit.html">Ulimit</a> data
         *        type.</p>
         *        <p>
         *        Amazon ECS tasks hosted on Fargate use the default resource limit values set by the operating system
         *        with the exception of the <code>nofile</code> resource limit parameter which Fargate overrides. The
         *        <code>nofile</code> resource limit sets a restriction on the number of open files that a container can
         *        use. The default <code>nofile</code> soft limit is <code> 65535</code> and the default hard limit is
         *        <code>65535</code>.
         *        </p>
         *        <p>
         *        This parameter requires version 1.18 of the Docker Remote API or greater on your container instance.
         *        To check the Docker Remote API version on your container instance, log in to your container instance
         *        and run the following command: <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
         *        </p>
         *        <note>
         *        <p>
         *        This parameter is not supported for Windows containers.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ulimits(Ulimit... ulimits);

        /**
         * <p>
         * A list of <code>ulimits</code> to set in the container. If a <code>ulimit</code> value is specified in a task
         * definition, it overrides the default values set by Docker. This parameter maps to <code>Ulimits</code> in the
         * docker container create command and the <code>--ulimit</code> option to docker run. Valid naming values are
         * displayed in the <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_Ulimit.html">Ulimit</a> data type.
         * </p>
         * <p>
         * Amazon ECS tasks hosted on Fargate use the default resource limit values set by the operating system with the
         * exception of the <code>nofile</code> resource limit parameter which Fargate overrides. The
         * <code>nofile</code> resource limit sets a restriction on the number of open files that a container can use.
         * The default <code>nofile</code> soft limit is <code> 65535</code> and the default hard limit is
         * <code>65535</code>.
         * </p>
         * <p>
         * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check
         * the Docker Remote API version on your container instance, log in to your container instance and run the
         * following command: <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
         * </p>
         * <note>
         * <p>
         * This parameter is not supported for Windows containers.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.Ulimit.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ecs.model.Ulimit#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ecs.model.Ulimit.Builder#build()}
         * is called immediately and its result is passed to {@link #ulimits(List<Ulimit>)}.
         * 
         * @param ulimits
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ecs.model.Ulimit.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ulimits(java.util.Collection<Ulimit>)
         */
        Builder ulimits(Consumer<Ulimit.Builder>... ulimits);

        /**
         * <p>
         * The log configuration specification for the container.
         * </p>
         * <p>
         * This parameter maps to <code>LogConfig</code> in the docker container create command and the
         * <code>--log-driver</code> option to docker run. By default, containers use the same logging driver that the
         * Docker daemon uses. However the container can use a different logging driver than the Docker daemon by
         * specifying a log driver with this parameter in the container definition. To use a different logging driver
         * for a container, the log system must be configured properly on the container instance (or on a different log
         * server for remote logging options).
         * </p>
         * <note>
         * <p>
         * Amazon ECS currently supports a subset of the logging drivers available to the Docker daemon (shown in the <a
         * href
         * ="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_LogConfiguration.html">LogConfiguration</a>
         * data type). Additional log drivers may be available in future releases of the Amazon ECS container agent.
         * </p>
         * </note>
         * <p>
         * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check
         * the Docker Remote API version on your container instance, log in to your container instance and run the
         * following command: <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
         * </p>
         * <note>
         * <p>
         * The Amazon ECS container agent running on a container instance must register the logging drivers available on
         * that instance with the <code>ECS_AVAILABLE_LOGGING_DRIVERS</code> environment variable before containers
         * placed on that instance can use these log configuration options. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon ECS Container
         * Agent Configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * </note>
         * 
         * @param logConfiguration
         *        The log configuration specification for the container.</p>
         *        <p>
         *        This parameter maps to <code>LogConfig</code> in the docker container create command and the
         *        <code>--log-driver</code> option to docker run. By default, containers use the same logging driver
         *        that the Docker daemon uses. However the container can use a different logging driver than the Docker
         *        daemon by specifying a log driver with this parameter in the container definition. To use a different
         *        logging driver for a container, the log system must be configured properly on the container instance
         *        (or on a different log server for remote logging options).
         *        </p>
         *        <note>
         *        <p>
         *        Amazon ECS currently supports a subset of the logging drivers available to the Docker daemon (shown in
         *        the <a href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_LogConfiguration.html">
         *        LogConfiguration</a> data type). Additional log drivers may be available in future releases of the
         *        Amazon ECS container agent.
         *        </p>
         *        </note>
         *        <p>
         *        This parameter requires version 1.18 of the Docker Remote API or greater on your container instance.
         *        To check the Docker Remote API version on your container instance, log in to your container instance
         *        and run the following command: <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
         *        </p>
         *        <note>
         *        <p>
         *        The Amazon ECS container agent running on a container instance must register the logging drivers
         *        available on that instance with the <code>ECS_AVAILABLE_LOGGING_DRIVERS</code> environment variable
         *        before containers placed on that instance can use these log configuration options. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon ECS
         *        Container Agent Configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logConfiguration(LogConfiguration logConfiguration);

        /**
         * <p>
         * The log configuration specification for the container.
         * </p>
         * <p>
         * This parameter maps to <code>LogConfig</code> in the docker container create command and the
         * <code>--log-driver</code> option to docker run. By default, containers use the same logging driver that the
         * Docker daemon uses. However the container can use a different logging driver than the Docker daemon by
         * specifying a log driver with this parameter in the container definition. To use a different logging driver
         * for a container, the log system must be configured properly on the container instance (or on a different log
         * server for remote logging options).
         * </p>
         * <note>
         * <p>
         * Amazon ECS currently supports a subset of the logging drivers available to the Docker daemon (shown in the <a
         * href
         * ="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_LogConfiguration.html">LogConfiguration</a>
         * data type). Additional log drivers may be available in future releases of the Amazon ECS container agent.
         * </p>
         * </note>
         * <p>
         * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check
         * the Docker Remote API version on your container instance, log in to your container instance and run the
         * following command: <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
         * </p>
         * <note>
         * <p>
         * The Amazon ECS container agent running on a container instance must register the logging drivers available on
         * that instance with the <code>ECS_AVAILABLE_LOGGING_DRIVERS</code> environment variable before containers
         * placed on that instance can use these log configuration options. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon ECS Container
         * Agent Configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link LogConfiguration.Builder}
         * avoiding the need to create one manually via {@link LogConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LogConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #logConfiguration(LogConfiguration)}.
         * 
         * @param logConfiguration
         *        a consumer that will call methods on {@link LogConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #logConfiguration(LogConfiguration)
         */
        default Builder logConfiguration(Consumer<LogConfiguration.Builder> logConfiguration) {
            return logConfiguration(LogConfiguration.builder().applyMutation(logConfiguration).build());
        }

        /**
         * <p>
         * The container health check command and associated configuration parameters for the container. This parameter
         * maps to <code>HealthCheck</code> in the docker container create command and the <code>HEALTHCHECK</code>
         * parameter of docker run.
         * </p>
         * 
         * @param healthCheck
         *        The container health check command and associated configuration parameters for the container. This
         *        parameter maps to <code>HealthCheck</code> in the docker container create command and the
         *        <code>HEALTHCHECK</code> parameter of docker run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder healthCheck(HealthCheck healthCheck);

        /**
         * <p>
         * The container health check command and associated configuration parameters for the container. This parameter
         * maps to <code>HealthCheck</code> in the docker container create command and the <code>HEALTHCHECK</code>
         * parameter of docker run.
         * </p>
         * This is a convenience method that creates an instance of the {@link HealthCheck.Builder} avoiding the need to
         * create one manually via {@link HealthCheck#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link HealthCheck.Builder#build()} is called immediately and its result
         * is passed to {@link #healthCheck(HealthCheck)}.
         * 
         * @param healthCheck
         *        a consumer that will call methods on {@link HealthCheck.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #healthCheck(HealthCheck)
         */
        default Builder healthCheck(Consumer<HealthCheck.Builder> healthCheck) {
            return healthCheck(HealthCheck.builder().applyMutation(healthCheck).build());
        }

        /**
         * <p>
         * A list of namespaced kernel parameters to set in the container. This parameter maps to <code>Sysctls</code>
         * in the docker container create command and the <code>--sysctl</code> option to docker run. For example, you
         * can configure <code>net.ipv4.tcp_keepalive_time</code> setting to maintain longer lived connections.
         * </p>
         * 
         * @param systemControls
         *        A list of namespaced kernel parameters to set in the container. This parameter maps to
         *        <code>Sysctls</code> in the docker container create command and the <code>--sysctl</code> option to
         *        docker run. For example, you can configure <code>net.ipv4.tcp_keepalive_time</code> setting to
         *        maintain longer lived connections.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder systemControls(Collection<SystemControl> systemControls);

        /**
         * <p>
         * A list of namespaced kernel parameters to set in the container. This parameter maps to <code>Sysctls</code>
         * in the docker container create command and the <code>--sysctl</code> option to docker run. For example, you
         * can configure <code>net.ipv4.tcp_keepalive_time</code> setting to maintain longer lived connections.
         * </p>
         * 
         * @param systemControls
         *        A list of namespaced kernel parameters to set in the container. This parameter maps to
         *        <code>Sysctls</code> in the docker container create command and the <code>--sysctl</code> option to
         *        docker run. For example, you can configure <code>net.ipv4.tcp_keepalive_time</code> setting to
         *        maintain longer lived connections.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder systemControls(SystemControl... systemControls);

        /**
         * <p>
         * A list of namespaced kernel parameters to set in the container. This parameter maps to <code>Sysctls</code>
         * in the docker container create command and the <code>--sysctl</code> option to docker run. For example, you
         * can configure <code>net.ipv4.tcp_keepalive_time</code> setting to maintain longer lived connections.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.SystemControl.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ecs.model.SystemControl#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecs.model.SystemControl.Builder#build()} is called immediately and its
         * result is passed to {@link #systemControls(List<SystemControl>)}.
         * 
         * @param systemControls
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecs.model.SystemControl.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #systemControls(java.util.Collection<SystemControl>)
         */
        Builder systemControls(Consumer<SystemControl.Builder>... systemControls);

        /**
         * <p>
         * The type and amount of a resource to assign to a container. The only supported resource is a GPU.
         * </p>
         * 
         * @param resourceRequirements
         *        The type and amount of a resource to assign to a container. The only supported resource is a GPU.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceRequirements(Collection<ResourceRequirement> resourceRequirements);

        /**
         * <p>
         * The type and amount of a resource to assign to a container. The only supported resource is a GPU.
         * </p>
         * 
         * @param resourceRequirements
         *        The type and amount of a resource to assign to a container. The only supported resource is a GPU.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceRequirements(ResourceRequirement... resourceRequirements);

        /**
         * <p>
         * The type and amount of a resource to assign to a container. The only supported resource is a GPU.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.ResourceRequirement.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ecs.model.ResourceRequirement#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecs.model.ResourceRequirement.Builder#build()} is called immediately
         * and its result is passed to {@link #resourceRequirements(List<ResourceRequirement>)}.
         * 
         * @param resourceRequirements
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecs.model.ResourceRequirement.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceRequirements(java.util.Collection<ResourceRequirement>)
         */
        Builder resourceRequirements(Consumer<ResourceRequirement.Builder>... resourceRequirements);

        /**
         * <p>
         * The FireLens configuration for the container. This is used to specify and configure a log router for
         * container logs. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Custom Log Routing</a>
         * in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param firelensConfiguration
         *        The FireLens configuration for the container. This is used to specify and configure a log router for
         *        container logs. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Custom Log
         *        Routing</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder firelensConfiguration(FirelensConfiguration firelensConfiguration);

        /**
         * <p>
         * The FireLens configuration for the container. This is used to specify and configure a log router for
         * container logs. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Custom Log Routing</a>
         * in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link FirelensConfiguration.Builder} avoiding
         * the need to create one manually via {@link FirelensConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link FirelensConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #firelensConfiguration(FirelensConfiguration)}.
         * 
         * @param firelensConfiguration
         *        a consumer that will call methods on {@link FirelensConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #firelensConfiguration(FirelensConfiguration)
         */
        default Builder firelensConfiguration(Consumer<FirelensConfiguration.Builder> firelensConfiguration) {
            return firelensConfiguration(FirelensConfiguration.builder().applyMutation(firelensConfiguration).build());
        }

        /**
         * <p>
         * A list of ARNs in SSM or Amazon S3 to a credential spec (<code>CredSpec</code>) file that configures the
         * container for Active Directory authentication. We recommend that you use this parameter instead of the
         * <code>dockerSecurityOptions</code>. The maximum number of ARNs is 1.
         * </p>
         * <p>
         * There are two formats for each ARN.
         * </p>
         * <dl>
         * <dt>credentialspecdomainless:MyARN</dt>
         * <dd>
         * <p>
         * You use <code>credentialspecdomainless:MyARN</code> to provide a <code>CredSpec</code> with an additional
         * section for a secret in Secrets Manager. You provide the login credentials to the domain in the secret.
         * </p>
         * <p>
         * Each task that runs on any container instance can join different domains.
         * </p>
         * <p>
         * You can use this format without joining the container instance to a domain.
         * </p>
         * </dd>
         * <dt>credentialspec:MyARN</dt>
         * <dd>
         * <p>
         * You use <code>credentialspec:MyARN</code> to provide a <code>CredSpec</code> for a single domain.
         * </p>
         * <p>
         * You must join the container instance to the domain before you start any tasks that use this task definition.
         * </p>
         * </dd>
         * </dl>
         * <p>
         * In both formats, replace <code>MyARN</code> with the ARN in SSM or Amazon S3.
         * </p>
         * <p>
         * If you provide a <code>credentialspecdomainless:MyARN</code>, the <code>credspec</code> must provide a ARN in
         * Secrets Manager for a secret containing the username, password, and the domain to connect to. For better
         * security, the instance isn't joined to the domain for domainless authentication. Other applications on the
         * instance can't use the domainless credentials. You can use this parameter to run tasks on the same instance,
         * even it the tasks need to join different domains. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/windows-gmsa.html">Using gMSAs for Windows
         * Containers</a> and <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/linux-gmsa.html">Using gMSAs for Linux
         * Containers</a>.
         * </p>
         * 
         * @param credentialSpecs
         *        A list of ARNs in SSM or Amazon S3 to a credential spec (<code>CredSpec</code>) file that configures
         *        the container for Active Directory authentication. We recommend that you use this parameter instead of
         *        the <code>dockerSecurityOptions</code>. The maximum number of ARNs is 1.</p>
         *        <p>
         *        There are two formats for each ARN.
         *        </p>
         *        <dl>
         *        <dt>credentialspecdomainless:MyARN</dt>
         *        <dd>
         *        <p>
         *        You use <code>credentialspecdomainless:MyARN</code> to provide a <code>CredSpec</code> with an
         *        additional section for a secret in Secrets Manager. You provide the login credentials to the domain in
         *        the secret.
         *        </p>
         *        <p>
         *        Each task that runs on any container instance can join different domains.
         *        </p>
         *        <p>
         *        You can use this format without joining the container instance to a domain.
         *        </p>
         *        </dd>
         *        <dt>credentialspec:MyARN</dt>
         *        <dd>
         *        <p>
         *        You use <code>credentialspec:MyARN</code> to provide a <code>CredSpec</code> for a single domain.
         *        </p>
         *        <p>
         *        You must join the container instance to the domain before you start any tasks that use this task
         *        definition.
         *        </p>
         *        </dd>
         *        </dl>
         *        <p>
         *        In both formats, replace <code>MyARN</code> with the ARN in SSM or Amazon S3.
         *        </p>
         *        <p>
         *        If you provide a <code>credentialspecdomainless:MyARN</code>, the <code>credspec</code> must provide a
         *        ARN in Secrets Manager for a secret containing the username, password, and the domain to connect to.
         *        For better security, the instance isn't joined to the domain for domainless authentication. Other
         *        applications on the instance can't use the domainless credentials. You can use this parameter to run
         *        tasks on the same instance, even it the tasks need to join different domains. For more information,
         *        see <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/windows-gmsa.html">Using
         *        gMSAs for Windows Containers</a> and <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/linux-gmsa.html">Using gMSAs for
         *        Linux Containers</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder credentialSpecs(Collection<String> credentialSpecs);

        /**
         * <p>
         * A list of ARNs in SSM or Amazon S3 to a credential spec (<code>CredSpec</code>) file that configures the
         * container for Active Directory authentication. We recommend that you use this parameter instead of the
         * <code>dockerSecurityOptions</code>. The maximum number of ARNs is 1.
         * </p>
         * <p>
         * There are two formats for each ARN.
         * </p>
         * <dl>
         * <dt>credentialspecdomainless:MyARN</dt>
         * <dd>
         * <p>
         * You use <code>credentialspecdomainless:MyARN</code> to provide a <code>CredSpec</code> with an additional
         * section for a secret in Secrets Manager. You provide the login credentials to the domain in the secret.
         * </p>
         * <p>
         * Each task that runs on any container instance can join different domains.
         * </p>
         * <p>
         * You can use this format without joining the container instance to a domain.
         * </p>
         * </dd>
         * <dt>credentialspec:MyARN</dt>
         * <dd>
         * <p>
         * You use <code>credentialspec:MyARN</code> to provide a <code>CredSpec</code> for a single domain.
         * </p>
         * <p>
         * You must join the container instance to the domain before you start any tasks that use this task definition.
         * </p>
         * </dd>
         * </dl>
         * <p>
         * In both formats, replace <code>MyARN</code> with the ARN in SSM or Amazon S3.
         * </p>
         * <p>
         * If you provide a <code>credentialspecdomainless:MyARN</code>, the <code>credspec</code> must provide a ARN in
         * Secrets Manager for a secret containing the username, password, and the domain to connect to. For better
         * security, the instance isn't joined to the domain for domainless authentication. Other applications on the
         * instance can't use the domainless credentials. You can use this parameter to run tasks on the same instance,
         * even it the tasks need to join different domains. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/windows-gmsa.html">Using gMSAs for Windows
         * Containers</a> and <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/linux-gmsa.html">Using gMSAs for Linux
         * Containers</a>.
         * </p>
         * 
         * @param credentialSpecs
         *        A list of ARNs in SSM or Amazon S3 to a credential spec (<code>CredSpec</code>) file that configures
         *        the container for Active Directory authentication. We recommend that you use this parameter instead of
         *        the <code>dockerSecurityOptions</code>. The maximum number of ARNs is 1.</p>
         *        <p>
         *        There are two formats for each ARN.
         *        </p>
         *        <dl>
         *        <dt>credentialspecdomainless:MyARN</dt>
         *        <dd>
         *        <p>
         *        You use <code>credentialspecdomainless:MyARN</code> to provide a <code>CredSpec</code> with an
         *        additional section for a secret in Secrets Manager. You provide the login credentials to the domain in
         *        the secret.
         *        </p>
         *        <p>
         *        Each task that runs on any container instance can join different domains.
         *        </p>
         *        <p>
         *        You can use this format without joining the container instance to a domain.
         *        </p>
         *        </dd>
         *        <dt>credentialspec:MyARN</dt>
         *        <dd>
         *        <p>
         *        You use <code>credentialspec:MyARN</code> to provide a <code>CredSpec</code> for a single domain.
         *        </p>
         *        <p>
         *        You must join the container instance to the domain before you start any tasks that use this task
         *        definition.
         *        </p>
         *        </dd>
         *        </dl>
         *        <p>
         *        In both formats, replace <code>MyARN</code> with the ARN in SSM or Amazon S3.
         *        </p>
         *        <p>
         *        If you provide a <code>credentialspecdomainless:MyARN</code>, the <code>credspec</code> must provide a
         *        ARN in Secrets Manager for a secret containing the username, password, and the domain to connect to.
         *        For better security, the instance isn't joined to the domain for domainless authentication. Other
         *        applications on the instance can't use the domainless credentials. You can use this parameter to run
         *        tasks on the same instance, even it the tasks need to join different domains. For more information,
         *        see <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/windows-gmsa.html">Using
         *        gMSAs for Windows Containers</a> and <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/linux-gmsa.html">Using gMSAs for
         *        Linux Containers</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder credentialSpecs(String... credentialSpecs);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String image;

        private RepositoryCredentials repositoryCredentials;

        private Integer cpu;

        private Integer memory;

        private Integer memoryReservation;

        private List<String> links = DefaultSdkAutoConstructList.getInstance();

        private List<PortMapping> portMappings = DefaultSdkAutoConstructList.getInstance();

        private Boolean essential;

        private ContainerRestartPolicy restartPolicy;

        private List<String> entryPoint = DefaultSdkAutoConstructList.getInstance();

        private List<String> command = DefaultSdkAutoConstructList.getInstance();

        private List<KeyValuePair> environment = DefaultSdkAutoConstructList.getInstance();

        private List<EnvironmentFile> environmentFiles = DefaultSdkAutoConstructList.getInstance();

        private List<MountPoint> mountPoints = DefaultSdkAutoConstructList.getInstance();

        private List<VolumeFrom> volumesFrom = DefaultSdkAutoConstructList.getInstance();

        private LinuxParameters linuxParameters;

        private List<Secret> secrets = DefaultSdkAutoConstructList.getInstance();

        private List<ContainerDependency> dependsOn = DefaultSdkAutoConstructList.getInstance();

        private Integer startTimeout;

        private Integer stopTimeout;

        private String versionConsistency;

        private String hostname;

        private String user;

        private String workingDirectory;

        private Boolean disableNetworking;

        private Boolean privileged;

        private Boolean readonlyRootFilesystem;

        private List<String> dnsServers = DefaultSdkAutoConstructList.getInstance();

        private List<String> dnsSearchDomains = DefaultSdkAutoConstructList.getInstance();

        private List<HostEntry> extraHosts = DefaultSdkAutoConstructList.getInstance();

        private List<String> dockerSecurityOptions = DefaultSdkAutoConstructList.getInstance();

        private Boolean interactive;

        private Boolean pseudoTerminal;

        private Map<String, String> dockerLabels = DefaultSdkAutoConstructMap.getInstance();

        private List<Ulimit> ulimits = DefaultSdkAutoConstructList.getInstance();

        private LogConfiguration logConfiguration;

        private HealthCheck healthCheck;

        private List<SystemControl> systemControls = DefaultSdkAutoConstructList.getInstance();

        private List<ResourceRequirement> resourceRequirements = DefaultSdkAutoConstructList.getInstance();

        private FirelensConfiguration firelensConfiguration;

        private List<String> credentialSpecs = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ContainerDefinition model) {
            name(model.name);
            image(model.image);
            repositoryCredentials(model.repositoryCredentials);
            cpu(model.cpu);
            memory(model.memory);
            memoryReservation(model.memoryReservation);
            links(model.links);
            portMappings(model.portMappings);
            essential(model.essential);
            restartPolicy(model.restartPolicy);
            entryPoint(model.entryPoint);
            command(model.command);
            environment(model.environment);
            environmentFiles(model.environmentFiles);
            mountPoints(model.mountPoints);
            volumesFrom(model.volumesFrom);
            linuxParameters(model.linuxParameters);
            secrets(model.secrets);
            dependsOn(model.dependsOn);
            startTimeout(model.startTimeout);
            stopTimeout(model.stopTimeout);
            versionConsistency(model.versionConsistency);
            hostname(model.hostname);
            user(model.user);
            workingDirectory(model.workingDirectory);
            disableNetworking(model.disableNetworking);
            privileged(model.privileged);
            readonlyRootFilesystem(model.readonlyRootFilesystem);
            dnsServers(model.dnsServers);
            dnsSearchDomains(model.dnsSearchDomains);
            extraHosts(model.extraHosts);
            dockerSecurityOptions(model.dockerSecurityOptions);
            interactive(model.interactive);
            pseudoTerminal(model.pseudoTerminal);
            dockerLabels(model.dockerLabels);
            ulimits(model.ulimits);
            logConfiguration(model.logConfiguration);
            healthCheck(model.healthCheck);
            systemControls(model.systemControls);
            resourceRequirements(model.resourceRequirements);
            firelensConfiguration(model.firelensConfiguration);
            credentialSpecs(model.credentialSpecs);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getImage() {
            return image;
        }

        public final void setImage(String image) {
            this.image = image;
        }

        @Override
        public final Builder image(String image) {
            this.image = image;
            return this;
        }

        public final RepositoryCredentials.Builder getRepositoryCredentials() {
            return repositoryCredentials != null ? repositoryCredentials.toBuilder() : null;
        }

        public final void setRepositoryCredentials(RepositoryCredentials.BuilderImpl repositoryCredentials) {
            this.repositoryCredentials = repositoryCredentials != null ? repositoryCredentials.build() : null;
        }

        @Override
        public final Builder repositoryCredentials(RepositoryCredentials repositoryCredentials) {
            this.repositoryCredentials = repositoryCredentials;
            return this;
        }

        public final Integer getCpu() {
            return cpu;
        }

        public final void setCpu(Integer cpu) {
            this.cpu = cpu;
        }

        @Override
        public final Builder cpu(Integer cpu) {
            this.cpu = cpu;
            return this;
        }

        public final Integer getMemory() {
            return memory;
        }

        public final void setMemory(Integer memory) {
            this.memory = memory;
        }

        @Override
        public final Builder memory(Integer memory) {
            this.memory = memory;
            return this;
        }

        public final Integer getMemoryReservation() {
            return memoryReservation;
        }

        public final void setMemoryReservation(Integer memoryReservation) {
            this.memoryReservation = memoryReservation;
        }

        @Override
        public final Builder memoryReservation(Integer memoryReservation) {
            this.memoryReservation = memoryReservation;
            return this;
        }

        public final Collection<String> getLinks() {
            if (links instanceof SdkAutoConstructList) {
                return null;
            }
            return links;
        }

        public final void setLinks(Collection<String> links) {
            this.links = StringListCopier.copy(links);
        }

        @Override
        public final Builder links(Collection<String> links) {
            this.links = StringListCopier.copy(links);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder links(String... links) {
            links(Arrays.asList(links));
            return this;
        }

        public final List<PortMapping.Builder> getPortMappings() {
            List<PortMapping.Builder> result = PortMappingListCopier.copyToBuilder(this.portMappings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPortMappings(Collection<PortMapping.BuilderImpl> portMappings) {
            this.portMappings = PortMappingListCopier.copyFromBuilder(portMappings);
        }

        @Override
        public final Builder portMappings(Collection<PortMapping> portMappings) {
            this.portMappings = PortMappingListCopier.copy(portMappings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder portMappings(PortMapping... portMappings) {
            portMappings(Arrays.asList(portMappings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder portMappings(Consumer<PortMapping.Builder>... portMappings) {
            portMappings(Stream.of(portMappings).map(c -> PortMapping.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Boolean getEssential() {
            return essential;
        }

        public final void setEssential(Boolean essential) {
            this.essential = essential;
        }

        @Override
        public final Builder essential(Boolean essential) {
            this.essential = essential;
            return this;
        }

        public final ContainerRestartPolicy.Builder getRestartPolicy() {
            return restartPolicy != null ? restartPolicy.toBuilder() : null;
        }

        public final void setRestartPolicy(ContainerRestartPolicy.BuilderImpl restartPolicy) {
            this.restartPolicy = restartPolicy != null ? restartPolicy.build() : null;
        }

        @Override
        public final Builder restartPolicy(ContainerRestartPolicy restartPolicy) {
            this.restartPolicy = restartPolicy;
            return this;
        }

        public final Collection<String> getEntryPoint() {
            if (entryPoint instanceof SdkAutoConstructList) {
                return null;
            }
            return entryPoint;
        }

        public final void setEntryPoint(Collection<String> entryPoint) {
            this.entryPoint = StringListCopier.copy(entryPoint);
        }

        @Override
        public final Builder entryPoint(Collection<String> entryPoint) {
            this.entryPoint = StringListCopier.copy(entryPoint);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder entryPoint(String... entryPoint) {
            entryPoint(Arrays.asList(entryPoint));
            return this;
        }

        public final Collection<String> getCommand() {
            if (command instanceof SdkAutoConstructList) {
                return null;
            }
            return command;
        }

        public final void setCommand(Collection<String> command) {
            this.command = StringListCopier.copy(command);
        }

        @Override
        public final Builder command(Collection<String> command) {
            this.command = StringListCopier.copy(command);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder command(String... command) {
            command(Arrays.asList(command));
            return this;
        }

        public final List<KeyValuePair.Builder> getEnvironment() {
            List<KeyValuePair.Builder> result = EnvironmentVariablesCopier.copyToBuilder(this.environment);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEnvironment(Collection<KeyValuePair.BuilderImpl> environment) {
            this.environment = EnvironmentVariablesCopier.copyFromBuilder(environment);
        }

        @Override
        public final Builder environment(Collection<KeyValuePair> environment) {
            this.environment = EnvironmentVariablesCopier.copy(environment);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environment(KeyValuePair... environment) {
            environment(Arrays.asList(environment));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environment(Consumer<KeyValuePair.Builder>... environment) {
            environment(Stream.of(environment).map(c -> KeyValuePair.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<EnvironmentFile.Builder> getEnvironmentFiles() {
            List<EnvironmentFile.Builder> result = EnvironmentFilesCopier.copyToBuilder(this.environmentFiles);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEnvironmentFiles(Collection<EnvironmentFile.BuilderImpl> environmentFiles) {
            this.environmentFiles = EnvironmentFilesCopier.copyFromBuilder(environmentFiles);
        }

        @Override
        public final Builder environmentFiles(Collection<EnvironmentFile> environmentFiles) {
            this.environmentFiles = EnvironmentFilesCopier.copy(environmentFiles);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environmentFiles(EnvironmentFile... environmentFiles) {
            environmentFiles(Arrays.asList(environmentFiles));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environmentFiles(Consumer<EnvironmentFile.Builder>... environmentFiles) {
            environmentFiles(Stream.of(environmentFiles).map(c -> EnvironmentFile.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<MountPoint.Builder> getMountPoints() {
            List<MountPoint.Builder> result = MountPointListCopier.copyToBuilder(this.mountPoints);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setMountPoints(Collection<MountPoint.BuilderImpl> mountPoints) {
            this.mountPoints = MountPointListCopier.copyFromBuilder(mountPoints);
        }

        @Override
        public final Builder mountPoints(Collection<MountPoint> mountPoints) {
            this.mountPoints = MountPointListCopier.copy(mountPoints);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder mountPoints(MountPoint... mountPoints) {
            mountPoints(Arrays.asList(mountPoints));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder mountPoints(Consumer<MountPoint.Builder>... mountPoints) {
            mountPoints(Stream.of(mountPoints).map(c -> MountPoint.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<VolumeFrom.Builder> getVolumesFrom() {
            List<VolumeFrom.Builder> result = VolumeFromListCopier.copyToBuilder(this.volumesFrom);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setVolumesFrom(Collection<VolumeFrom.BuilderImpl> volumesFrom) {
            this.volumesFrom = VolumeFromListCopier.copyFromBuilder(volumesFrom);
        }

        @Override
        public final Builder volumesFrom(Collection<VolumeFrom> volumesFrom) {
            this.volumesFrom = VolumeFromListCopier.copy(volumesFrom);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder volumesFrom(VolumeFrom... volumesFrom) {
            volumesFrom(Arrays.asList(volumesFrom));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder volumesFrom(Consumer<VolumeFrom.Builder>... volumesFrom) {
            volumesFrom(Stream.of(volumesFrom).map(c -> VolumeFrom.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final LinuxParameters.Builder getLinuxParameters() {
            return linuxParameters != null ? linuxParameters.toBuilder() : null;
        }

        public final void setLinuxParameters(LinuxParameters.BuilderImpl linuxParameters) {
            this.linuxParameters = linuxParameters != null ? linuxParameters.build() : null;
        }

        @Override
        public final Builder linuxParameters(LinuxParameters linuxParameters) {
            this.linuxParameters = linuxParameters;
            return this;
        }

        public final List<Secret.Builder> getSecrets() {
            List<Secret.Builder> result = SecretListCopier.copyToBuilder(this.secrets);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSecrets(Collection<Secret.BuilderImpl> secrets) {
            this.secrets = SecretListCopier.copyFromBuilder(secrets);
        }

        @Override
        public final Builder secrets(Collection<Secret> secrets) {
            this.secrets = SecretListCopier.copy(secrets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secrets(Secret... secrets) {
            secrets(Arrays.asList(secrets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secrets(Consumer<Secret.Builder>... secrets) {
            secrets(Stream.of(secrets).map(c -> Secret.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<ContainerDependency.Builder> getDependsOn() {
            List<ContainerDependency.Builder> result = ContainerDependenciesCopier.copyToBuilder(this.dependsOn);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDependsOn(Collection<ContainerDependency.BuilderImpl> dependsOn) {
            this.dependsOn = ContainerDependenciesCopier.copyFromBuilder(dependsOn);
        }

        @Override
        public final Builder dependsOn(Collection<ContainerDependency> dependsOn) {
            this.dependsOn = ContainerDependenciesCopier.copy(dependsOn);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dependsOn(ContainerDependency... dependsOn) {
            dependsOn(Arrays.asList(dependsOn));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dependsOn(Consumer<ContainerDependency.Builder>... dependsOn) {
            dependsOn(Stream.of(dependsOn).map(c -> ContainerDependency.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Integer getStartTimeout() {
            return startTimeout;
        }

        public final void setStartTimeout(Integer startTimeout) {
            this.startTimeout = startTimeout;
        }

        @Override
        public final Builder startTimeout(Integer startTimeout) {
            this.startTimeout = startTimeout;
            return this;
        }

        public final Integer getStopTimeout() {
            return stopTimeout;
        }

        public final void setStopTimeout(Integer stopTimeout) {
            this.stopTimeout = stopTimeout;
        }

        @Override
        public final Builder stopTimeout(Integer stopTimeout) {
            this.stopTimeout = stopTimeout;
            return this;
        }

        public final String getVersionConsistency() {
            return versionConsistency;
        }

        public final void setVersionConsistency(String versionConsistency) {
            this.versionConsistency = versionConsistency;
        }

        @Override
        public final Builder versionConsistency(String versionConsistency) {
            this.versionConsistency = versionConsistency;
            return this;
        }

        @Override
        public final Builder versionConsistency(VersionConsistency versionConsistency) {
            this.versionConsistency(versionConsistency == null ? null : versionConsistency.toString());
            return this;
        }

        public final String getHostname() {
            return hostname;
        }

        public final void setHostname(String hostname) {
            this.hostname = hostname;
        }

        @Override
        public final Builder hostname(String hostname) {
            this.hostname = hostname;
            return this;
        }

        public final String getUser() {
            return user;
        }

        public final void setUser(String user) {
            this.user = user;
        }

        @Override
        public final Builder user(String user) {
            this.user = user;
            return this;
        }

        public final String getWorkingDirectory() {
            return workingDirectory;
        }

        public final void setWorkingDirectory(String workingDirectory) {
            this.workingDirectory = workingDirectory;
        }

        @Override
        public final Builder workingDirectory(String workingDirectory) {
            this.workingDirectory = workingDirectory;
            return this;
        }

        public final Boolean getDisableNetworking() {
            return disableNetworking;
        }

        public final void setDisableNetworking(Boolean disableNetworking) {
            this.disableNetworking = disableNetworking;
        }

        @Override
        public final Builder disableNetworking(Boolean disableNetworking) {
            this.disableNetworking = disableNetworking;
            return this;
        }

        public final Boolean getPrivileged() {
            return privileged;
        }

        public final void setPrivileged(Boolean privileged) {
            this.privileged = privileged;
        }

        @Override
        public final Builder privileged(Boolean privileged) {
            this.privileged = privileged;
            return this;
        }

        public final Boolean getReadonlyRootFilesystem() {
            return readonlyRootFilesystem;
        }

        public final void setReadonlyRootFilesystem(Boolean readonlyRootFilesystem) {
            this.readonlyRootFilesystem = readonlyRootFilesystem;
        }

        @Override
        public final Builder readonlyRootFilesystem(Boolean readonlyRootFilesystem) {
            this.readonlyRootFilesystem = readonlyRootFilesystem;
            return this;
        }

        public final Collection<String> getDnsServers() {
            if (dnsServers instanceof SdkAutoConstructList) {
                return null;
            }
            return dnsServers;
        }

        public final void setDnsServers(Collection<String> dnsServers) {
            this.dnsServers = StringListCopier.copy(dnsServers);
        }

        @Override
        public final Builder dnsServers(Collection<String> dnsServers) {
            this.dnsServers = StringListCopier.copy(dnsServers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dnsServers(String... dnsServers) {
            dnsServers(Arrays.asList(dnsServers));
            return this;
        }

        public final Collection<String> getDnsSearchDomains() {
            if (dnsSearchDomains instanceof SdkAutoConstructList) {
                return null;
            }
            return dnsSearchDomains;
        }

        public final void setDnsSearchDomains(Collection<String> dnsSearchDomains) {
            this.dnsSearchDomains = StringListCopier.copy(dnsSearchDomains);
        }

        @Override
        public final Builder dnsSearchDomains(Collection<String> dnsSearchDomains) {
            this.dnsSearchDomains = StringListCopier.copy(dnsSearchDomains);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dnsSearchDomains(String... dnsSearchDomains) {
            dnsSearchDomains(Arrays.asList(dnsSearchDomains));
            return this;
        }

        public final List<HostEntry.Builder> getExtraHosts() {
            List<HostEntry.Builder> result = HostEntryListCopier.copyToBuilder(this.extraHosts);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setExtraHosts(Collection<HostEntry.BuilderImpl> extraHosts) {
            this.extraHosts = HostEntryListCopier.copyFromBuilder(extraHosts);
        }

        @Override
        public final Builder extraHosts(Collection<HostEntry> extraHosts) {
            this.extraHosts = HostEntryListCopier.copy(extraHosts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder extraHosts(HostEntry... extraHosts) {
            extraHosts(Arrays.asList(extraHosts));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder extraHosts(Consumer<HostEntry.Builder>... extraHosts) {
            extraHosts(Stream.of(extraHosts).map(c -> HostEntry.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Collection<String> getDockerSecurityOptions() {
            if (dockerSecurityOptions instanceof SdkAutoConstructList) {
                return null;
            }
            return dockerSecurityOptions;
        }

        public final void setDockerSecurityOptions(Collection<String> dockerSecurityOptions) {
            this.dockerSecurityOptions = StringListCopier.copy(dockerSecurityOptions);
        }

        @Override
        public final Builder dockerSecurityOptions(Collection<String> dockerSecurityOptions) {
            this.dockerSecurityOptions = StringListCopier.copy(dockerSecurityOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dockerSecurityOptions(String... dockerSecurityOptions) {
            dockerSecurityOptions(Arrays.asList(dockerSecurityOptions));
            return this;
        }

        public final Boolean getInteractive() {
            return interactive;
        }

        public final void setInteractive(Boolean interactive) {
            this.interactive = interactive;
        }

        @Override
        public final Builder interactive(Boolean interactive) {
            this.interactive = interactive;
            return this;
        }

        public final Boolean getPseudoTerminal() {
            return pseudoTerminal;
        }

        public final void setPseudoTerminal(Boolean pseudoTerminal) {
            this.pseudoTerminal = pseudoTerminal;
        }

        @Override
        public final Builder pseudoTerminal(Boolean pseudoTerminal) {
            this.pseudoTerminal = pseudoTerminal;
            return this;
        }

        public final Map<String, String> getDockerLabels() {
            if (dockerLabels instanceof SdkAutoConstructMap) {
                return null;
            }
            return dockerLabels;
        }

        public final void setDockerLabels(Map<String, String> dockerLabels) {
            this.dockerLabels = DockerLabelsMapCopier.copy(dockerLabels);
        }

        @Override
        public final Builder dockerLabels(Map<String, String> dockerLabels) {
            this.dockerLabels = DockerLabelsMapCopier.copy(dockerLabels);
            return this;
        }

        public final List<Ulimit.Builder> getUlimits() {
            List<Ulimit.Builder> result = UlimitListCopier.copyToBuilder(this.ulimits);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setUlimits(Collection<Ulimit.BuilderImpl> ulimits) {
            this.ulimits = UlimitListCopier.copyFromBuilder(ulimits);
        }

        @Override
        public final Builder ulimits(Collection<Ulimit> ulimits) {
            this.ulimits = UlimitListCopier.copy(ulimits);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ulimits(Ulimit... ulimits) {
            ulimits(Arrays.asList(ulimits));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ulimits(Consumer<Ulimit.Builder>... ulimits) {
            ulimits(Stream.of(ulimits).map(c -> Ulimit.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final LogConfiguration.Builder getLogConfiguration() {
            return logConfiguration != null ? logConfiguration.toBuilder() : null;
        }

        public final void setLogConfiguration(LogConfiguration.BuilderImpl logConfiguration) {
            this.logConfiguration = logConfiguration != null ? logConfiguration.build() : null;
        }

        @Override
        public final Builder logConfiguration(LogConfiguration logConfiguration) {
            this.logConfiguration = logConfiguration;
            return this;
        }

        public final HealthCheck.Builder getHealthCheck() {
            return healthCheck != null ? healthCheck.toBuilder() : null;
        }

        public final void setHealthCheck(HealthCheck.BuilderImpl healthCheck) {
            this.healthCheck = healthCheck != null ? healthCheck.build() : null;
        }

        @Override
        public final Builder healthCheck(HealthCheck healthCheck) {
            this.healthCheck = healthCheck;
            return this;
        }

        public final List<SystemControl.Builder> getSystemControls() {
            List<SystemControl.Builder> result = SystemControlsCopier.copyToBuilder(this.systemControls);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSystemControls(Collection<SystemControl.BuilderImpl> systemControls) {
            this.systemControls = SystemControlsCopier.copyFromBuilder(systemControls);
        }

        @Override
        public final Builder systemControls(Collection<SystemControl> systemControls) {
            this.systemControls = SystemControlsCopier.copy(systemControls);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder systemControls(SystemControl... systemControls) {
            systemControls(Arrays.asList(systemControls));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder systemControls(Consumer<SystemControl.Builder>... systemControls) {
            systemControls(Stream.of(systemControls).map(c -> SystemControl.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<ResourceRequirement.Builder> getResourceRequirements() {
            List<ResourceRequirement.Builder> result = ResourceRequirementsCopier.copyToBuilder(this.resourceRequirements);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setResourceRequirements(Collection<ResourceRequirement.BuilderImpl> resourceRequirements) {
            this.resourceRequirements = ResourceRequirementsCopier.copyFromBuilder(resourceRequirements);
        }

        @Override
        public final Builder resourceRequirements(Collection<ResourceRequirement> resourceRequirements) {
            this.resourceRequirements = ResourceRequirementsCopier.copy(resourceRequirements);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceRequirements(ResourceRequirement... resourceRequirements) {
            resourceRequirements(Arrays.asList(resourceRequirements));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceRequirements(Consumer<ResourceRequirement.Builder>... resourceRequirements) {
            resourceRequirements(Stream.of(resourceRequirements).map(c -> ResourceRequirement.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final FirelensConfiguration.Builder getFirelensConfiguration() {
            return firelensConfiguration != null ? firelensConfiguration.toBuilder() : null;
        }

        public final void setFirelensConfiguration(FirelensConfiguration.BuilderImpl firelensConfiguration) {
            this.firelensConfiguration = firelensConfiguration != null ? firelensConfiguration.build() : null;
        }

        @Override
        public final Builder firelensConfiguration(FirelensConfiguration firelensConfiguration) {
            this.firelensConfiguration = firelensConfiguration;
            return this;
        }

        public final Collection<String> getCredentialSpecs() {
            if (credentialSpecs instanceof SdkAutoConstructList) {
                return null;
            }
            return credentialSpecs;
        }

        public final void setCredentialSpecs(Collection<String> credentialSpecs) {
            this.credentialSpecs = StringListCopier.copy(credentialSpecs);
        }

        @Override
        public final Builder credentialSpecs(Collection<String> credentialSpecs) {
            this.credentialSpecs = StringListCopier.copy(credentialSpecs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder credentialSpecs(String... credentialSpecs) {
            credentialSpecs(Arrays.asList(credentialSpecs));
            return this;
        }

        @Override
        public ContainerDefinition build() {
            return new ContainerDefinition(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
