/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The deployment controller to use for the service.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DeploymentController implements SdkPojo, Serializable,
        ToCopyableBuilder<DeploymentController.Builder, DeploymentController> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(DeploymentController::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String type;

    private DeploymentController(BuilderImpl builder) {
        this.type = builder.type;
    }

    /**
     * <p>
     * The deployment controller type to use.
     * </p>
     * <p>
     * The deployment controller is the mechanism that determines how tasks are deployed for your service. The valid
     * options are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * ECS
     * </p>
     * <p>
     * When you create a service which uses the <code>ECS</code> deployment controller, you can choose between the
     * following deployment strategies:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ROLLING</code>: When you create a service which uses the <i>rolling update</i> (<code>ROLLING</code>)
     * deployment strategy, the Amazon ECS service scheduler replaces the currently running tasks with new tasks. The
     * number of tasks that Amazon ECS adds or removes from the service during a rolling update is controlled by the
     * service deployment configuration.
     * </p>
     * <p>
     * Rolling update deployments are best suited for the following scenarios:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Gradual service updates: You need to update your service incrementally without taking the entire service offline
     * at once.
     * </p>
     * </li>
     * <li>
     * <p>
     * Limited resource requirements: You want to avoid the additional resource costs of running two complete
     * environments simultaneously (as required by blue/green deployments).
     * </p>
     * </li>
     * <li>
     * <p>
     * Acceptable deployment time: Your application can tolerate a longer deployment process, as rolling updates replace
     * tasks one by one.
     * </p>
     * </li>
     * <li>
     * <p>
     * No need for instant roll back: Your service can tolerate a rollback process that takes minutes rather than
     * seconds.
     * </p>
     * </li>
     * <li>
     * <p>
     * Simple deployment process: You prefer a straightforward deployment approach without the complexity of managing
     * multiple environments, target groups, and listeners.
     * </p>
     * </li>
     * <li>
     * <p>
     * No load balancer requirement: Your service doesn't use or require a load balancer, Application Load Balancer,
     * Network Load Balancer, or Service Connect (which are required for blue/green deployments).
     * </p>
     * </li>
     * <li>
     * <p>
     * Stateful applications: Your application maintains state that makes it difficult to run two parallel environments.
     * </p>
     * </li>
     * <li>
     * <p>
     * Cost sensitivity: You want to minimize deployment costs by not running duplicate environments during deployment.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Rolling updates are the default deployment strategy for services and provide a balance between deployment safety
     * and resource efficiency for many common application scenarios.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BLUE_GREEN</code>: A <i>blue/green</i> deployment strategy (<code>BLUE_GREEN</code>) is a release
     * methodology that reduces downtime and risk by running two identical production environments called blue and
     * green. With Amazon ECS blue/green deployments, you can validate new service revisions before directing production
     * traffic to them. This approach provides a safer way to deploy changes with the ability to quickly roll back if
     * needed.
     * </p>
     * <p>
     * Amazon ECS blue/green deployments are best suited for the following scenarios:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Service validation: When you need to validate new service revisions before directing production traffic to them
     * </p>
     * </li>
     * <li>
     * <p>
     * Zero downtime: When your service requires zero-downtime deployments
     * </p>
     * </li>
     * <li>
     * <p>
     * Instant roll back: When you need the ability to quickly roll back if issues are detected
     * </p>
     * </li>
     * <li>
     * <p>
     * Load balancer requirement: When your service uses Application Load Balancer, Network Load Balancer, or Service
     * Connect
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * External
     * </p>
     * <p>
     * Use a third-party deployment controller.
     * </p>
     * </li>
     * <li>
     * <p>
     * Blue/green deployment (powered by CodeDeploy)
     * </p>
     * <p>
     * CodeDeploy installs an updated version of the application as a new replacement task set and reroutes production
     * traffic from the original application task set to the replacement task set. The original task set is terminated
     * after a successful deployment. Use this deployment controller to verify a new deployment of a service before
     * sending production traffic to it.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link DeploymentControllerType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The deployment controller type to use.</p>
     *         <p>
     *         The deployment controller is the mechanism that determines how tasks are deployed for your service. The
     *         valid options are:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         ECS
     *         </p>
     *         <p>
     *         When you create a service which uses the <code>ECS</code> deployment controller, you can choose between
     *         the following deployment strategies:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ROLLING</code>: When you create a service which uses the <i>rolling update</i> (
     *         <code>ROLLING</code>) deployment strategy, the Amazon ECS service scheduler replaces the currently
     *         running tasks with new tasks. The number of tasks that Amazon ECS adds or removes from the service during
     *         a rolling update is controlled by the service deployment configuration.
     *         </p>
     *         <p>
     *         Rolling update deployments are best suited for the following scenarios:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Gradual service updates: You need to update your service incrementally without taking the entire service
     *         offline at once.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Limited resource requirements: You want to avoid the additional resource costs of running two complete
     *         environments simultaneously (as required by blue/green deployments).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Acceptable deployment time: Your application can tolerate a longer deployment process, as rolling updates
     *         replace tasks one by one.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         No need for instant roll back: Your service can tolerate a rollback process that takes minutes rather
     *         than seconds.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Simple deployment process: You prefer a straightforward deployment approach without the complexity of
     *         managing multiple environments, target groups, and listeners.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         No load balancer requirement: Your service doesn't use or require a load balancer, Application Load
     *         Balancer, Network Load Balancer, or Service Connect (which are required for blue/green deployments).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Stateful applications: Your application maintains state that makes it difficult to run two parallel
     *         environments.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Cost sensitivity: You want to minimize deployment costs by not running duplicate environments during
     *         deployment.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Rolling updates are the default deployment strategy for services and provide a balance between deployment
     *         safety and resource efficiency for many common application scenarios.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BLUE_GREEN</code>: A <i>blue/green</i> deployment strategy (<code>BLUE_GREEN</code>) is a release
     *         methodology that reduces downtime and risk by running two identical production environments called blue
     *         and green. With Amazon ECS blue/green deployments, you can validate new service revisions before
     *         directing production traffic to them. This approach provides a safer way to deploy changes with the
     *         ability to quickly roll back if needed.
     *         </p>
     *         <p>
     *         Amazon ECS blue/green deployments are best suited for the following scenarios:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Service validation: When you need to validate new service revisions before directing production traffic
     *         to them
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Zero downtime: When your service requires zero-downtime deployments
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Instant roll back: When you need the ability to quickly roll back if issues are detected
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Load balancer requirement: When your service uses Application Load Balancer, Network Load Balancer, or
     *         Service Connect
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         External
     *         </p>
     *         <p>
     *         Use a third-party deployment controller.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Blue/green deployment (powered by CodeDeploy)
     *         </p>
     *         <p>
     *         CodeDeploy installs an updated version of the application as a new replacement task set and reroutes
     *         production traffic from the original application task set to the replacement task set. The original task
     *         set is terminated after a successful deployment. Use this deployment controller to verify a new
     *         deployment of a service before sending production traffic to it.
     *         </p>
     *         </li>
     * @see DeploymentControllerType
     */
    public final DeploymentControllerType type() {
        return DeploymentControllerType.fromValue(type);
    }

    /**
     * <p>
     * The deployment controller type to use.
     * </p>
     * <p>
     * The deployment controller is the mechanism that determines how tasks are deployed for your service. The valid
     * options are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * ECS
     * </p>
     * <p>
     * When you create a service which uses the <code>ECS</code> deployment controller, you can choose between the
     * following deployment strategies:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ROLLING</code>: When you create a service which uses the <i>rolling update</i> (<code>ROLLING</code>)
     * deployment strategy, the Amazon ECS service scheduler replaces the currently running tasks with new tasks. The
     * number of tasks that Amazon ECS adds or removes from the service during a rolling update is controlled by the
     * service deployment configuration.
     * </p>
     * <p>
     * Rolling update deployments are best suited for the following scenarios:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Gradual service updates: You need to update your service incrementally without taking the entire service offline
     * at once.
     * </p>
     * </li>
     * <li>
     * <p>
     * Limited resource requirements: You want to avoid the additional resource costs of running two complete
     * environments simultaneously (as required by blue/green deployments).
     * </p>
     * </li>
     * <li>
     * <p>
     * Acceptable deployment time: Your application can tolerate a longer deployment process, as rolling updates replace
     * tasks one by one.
     * </p>
     * </li>
     * <li>
     * <p>
     * No need for instant roll back: Your service can tolerate a rollback process that takes minutes rather than
     * seconds.
     * </p>
     * </li>
     * <li>
     * <p>
     * Simple deployment process: You prefer a straightforward deployment approach without the complexity of managing
     * multiple environments, target groups, and listeners.
     * </p>
     * </li>
     * <li>
     * <p>
     * No load balancer requirement: Your service doesn't use or require a load balancer, Application Load Balancer,
     * Network Load Balancer, or Service Connect (which are required for blue/green deployments).
     * </p>
     * </li>
     * <li>
     * <p>
     * Stateful applications: Your application maintains state that makes it difficult to run two parallel environments.
     * </p>
     * </li>
     * <li>
     * <p>
     * Cost sensitivity: You want to minimize deployment costs by not running duplicate environments during deployment.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Rolling updates are the default deployment strategy for services and provide a balance between deployment safety
     * and resource efficiency for many common application scenarios.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BLUE_GREEN</code>: A <i>blue/green</i> deployment strategy (<code>BLUE_GREEN</code>) is a release
     * methodology that reduces downtime and risk by running two identical production environments called blue and
     * green. With Amazon ECS blue/green deployments, you can validate new service revisions before directing production
     * traffic to them. This approach provides a safer way to deploy changes with the ability to quickly roll back if
     * needed.
     * </p>
     * <p>
     * Amazon ECS blue/green deployments are best suited for the following scenarios:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Service validation: When you need to validate new service revisions before directing production traffic to them
     * </p>
     * </li>
     * <li>
     * <p>
     * Zero downtime: When your service requires zero-downtime deployments
     * </p>
     * </li>
     * <li>
     * <p>
     * Instant roll back: When you need the ability to quickly roll back if issues are detected
     * </p>
     * </li>
     * <li>
     * <p>
     * Load balancer requirement: When your service uses Application Load Balancer, Network Load Balancer, or Service
     * Connect
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * External
     * </p>
     * <p>
     * Use a third-party deployment controller.
     * </p>
     * </li>
     * <li>
     * <p>
     * Blue/green deployment (powered by CodeDeploy)
     * </p>
     * <p>
     * CodeDeploy installs an updated version of the application as a new replacement task set and reroutes production
     * traffic from the original application task set to the replacement task set. The original task set is terminated
     * after a successful deployment. Use this deployment controller to verify a new deployment of a service before
     * sending production traffic to it.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link DeploymentControllerType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The deployment controller type to use.</p>
     *         <p>
     *         The deployment controller is the mechanism that determines how tasks are deployed for your service. The
     *         valid options are:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         ECS
     *         </p>
     *         <p>
     *         When you create a service which uses the <code>ECS</code> deployment controller, you can choose between
     *         the following deployment strategies:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ROLLING</code>: When you create a service which uses the <i>rolling update</i> (
     *         <code>ROLLING</code>) deployment strategy, the Amazon ECS service scheduler replaces the currently
     *         running tasks with new tasks. The number of tasks that Amazon ECS adds or removes from the service during
     *         a rolling update is controlled by the service deployment configuration.
     *         </p>
     *         <p>
     *         Rolling update deployments are best suited for the following scenarios:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Gradual service updates: You need to update your service incrementally without taking the entire service
     *         offline at once.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Limited resource requirements: You want to avoid the additional resource costs of running two complete
     *         environments simultaneously (as required by blue/green deployments).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Acceptable deployment time: Your application can tolerate a longer deployment process, as rolling updates
     *         replace tasks one by one.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         No need for instant roll back: Your service can tolerate a rollback process that takes minutes rather
     *         than seconds.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Simple deployment process: You prefer a straightforward deployment approach without the complexity of
     *         managing multiple environments, target groups, and listeners.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         No load balancer requirement: Your service doesn't use or require a load balancer, Application Load
     *         Balancer, Network Load Balancer, or Service Connect (which are required for blue/green deployments).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Stateful applications: Your application maintains state that makes it difficult to run two parallel
     *         environments.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Cost sensitivity: You want to minimize deployment costs by not running duplicate environments during
     *         deployment.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Rolling updates are the default deployment strategy for services and provide a balance between deployment
     *         safety and resource efficiency for many common application scenarios.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BLUE_GREEN</code>: A <i>blue/green</i> deployment strategy (<code>BLUE_GREEN</code>) is a release
     *         methodology that reduces downtime and risk by running two identical production environments called blue
     *         and green. With Amazon ECS blue/green deployments, you can validate new service revisions before
     *         directing production traffic to them. This approach provides a safer way to deploy changes with the
     *         ability to quickly roll back if needed.
     *         </p>
     *         <p>
     *         Amazon ECS blue/green deployments are best suited for the following scenarios:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Service validation: When you need to validate new service revisions before directing production traffic
     *         to them
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Zero downtime: When your service requires zero-downtime deployments
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Instant roll back: When you need the ability to quickly roll back if issues are detected
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Load balancer requirement: When your service uses Application Load Balancer, Network Load Balancer, or
     *         Service Connect
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         External
     *         </p>
     *         <p>
     *         Use a third-party deployment controller.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Blue/green deployment (powered by CodeDeploy)
     *         </p>
     *         <p>
     *         CodeDeploy installs an updated version of the application as a new replacement task set and reroutes
     *         production traffic from the original application task set to the replacement task set. The original task
     *         set is terminated after a successful deployment. Use this deployment controller to verify a new
     *         deployment of a service before sending production traffic to it.
     *         </p>
     *         </li>
     * @see DeploymentControllerType
     */
    public final String typeAsString() {
        return type;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DeploymentController)) {
            return false;
        }
        DeploymentController other = (DeploymentController) obj;
        return Objects.equals(typeAsString(), other.typeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DeploymentController").add("Type", typeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("type", TYPE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<DeploymentController, T> g) {
        return obj -> g.apply((DeploymentController) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DeploymentController> {
        /**
         * <p>
         * The deployment controller type to use.
         * </p>
         * <p>
         * The deployment controller is the mechanism that determines how tasks are deployed for your service. The valid
         * options are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * ECS
         * </p>
         * <p>
         * When you create a service which uses the <code>ECS</code> deployment controller, you can choose between the
         * following deployment strategies:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ROLLING</code>: When you create a service which uses the <i>rolling update</i> (<code>ROLLING</code>)
         * deployment strategy, the Amazon ECS service scheduler replaces the currently running tasks with new tasks.
         * The number of tasks that Amazon ECS adds or removes from the service during a rolling update is controlled by
         * the service deployment configuration.
         * </p>
         * <p>
         * Rolling update deployments are best suited for the following scenarios:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Gradual service updates: You need to update your service incrementally without taking the entire service
         * offline at once.
         * </p>
         * </li>
         * <li>
         * <p>
         * Limited resource requirements: You want to avoid the additional resource costs of running two complete
         * environments simultaneously (as required by blue/green deployments).
         * </p>
         * </li>
         * <li>
         * <p>
         * Acceptable deployment time: Your application can tolerate a longer deployment process, as rolling updates
         * replace tasks one by one.
         * </p>
         * </li>
         * <li>
         * <p>
         * No need for instant roll back: Your service can tolerate a rollback process that takes minutes rather than
         * seconds.
         * </p>
         * </li>
         * <li>
         * <p>
         * Simple deployment process: You prefer a straightforward deployment approach without the complexity of
         * managing multiple environments, target groups, and listeners.
         * </p>
         * </li>
         * <li>
         * <p>
         * No load balancer requirement: Your service doesn't use or require a load balancer, Application Load Balancer,
         * Network Load Balancer, or Service Connect (which are required for blue/green deployments).
         * </p>
         * </li>
         * <li>
         * <p>
         * Stateful applications: Your application maintains state that makes it difficult to run two parallel
         * environments.
         * </p>
         * </li>
         * <li>
         * <p>
         * Cost sensitivity: You want to minimize deployment costs by not running duplicate environments during
         * deployment.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Rolling updates are the default deployment strategy for services and provide a balance between deployment
         * safety and resource efficiency for many common application scenarios.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BLUE_GREEN</code>: A <i>blue/green</i> deployment strategy (<code>BLUE_GREEN</code>) is a release
         * methodology that reduces downtime and risk by running two identical production environments called blue and
         * green. With Amazon ECS blue/green deployments, you can validate new service revisions before directing
         * production traffic to them. This approach provides a safer way to deploy changes with the ability to quickly
         * roll back if needed.
         * </p>
         * <p>
         * Amazon ECS blue/green deployments are best suited for the following scenarios:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Service validation: When you need to validate new service revisions before directing production traffic to
         * them
         * </p>
         * </li>
         * <li>
         * <p>
         * Zero downtime: When your service requires zero-downtime deployments
         * </p>
         * </li>
         * <li>
         * <p>
         * Instant roll back: When you need the ability to quickly roll back if issues are detected
         * </p>
         * </li>
         * <li>
         * <p>
         * Load balancer requirement: When your service uses Application Load Balancer, Network Load Balancer, or
         * Service Connect
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * External
         * </p>
         * <p>
         * Use a third-party deployment controller.
         * </p>
         * </li>
         * <li>
         * <p>
         * Blue/green deployment (powered by CodeDeploy)
         * </p>
         * <p>
         * CodeDeploy installs an updated version of the application as a new replacement task set and reroutes
         * production traffic from the original application task set to the replacement task set. The original task set
         * is terminated after a successful deployment. Use this deployment controller to verify a new deployment of a
         * service before sending production traffic to it.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The deployment controller type to use.</p>
         *        <p>
         *        The deployment controller is the mechanism that determines how tasks are deployed for your service.
         *        The valid options are:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        ECS
         *        </p>
         *        <p>
         *        When you create a service which uses the <code>ECS</code> deployment controller, you can choose
         *        between the following deployment strategies:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ROLLING</code>: When you create a service which uses the <i>rolling update</i> (
         *        <code>ROLLING</code>) deployment strategy, the Amazon ECS service scheduler replaces the currently
         *        running tasks with new tasks. The number of tasks that Amazon ECS adds or removes from the service
         *        during a rolling update is controlled by the service deployment configuration.
         *        </p>
         *        <p>
         *        Rolling update deployments are best suited for the following scenarios:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Gradual service updates: You need to update your service incrementally without taking the entire
         *        service offline at once.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Limited resource requirements: You want to avoid the additional resource costs of running two complete
         *        environments simultaneously (as required by blue/green deployments).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Acceptable deployment time: Your application can tolerate a longer deployment process, as rolling
         *        updates replace tasks one by one.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        No need for instant roll back: Your service can tolerate a rollback process that takes minutes rather
         *        than seconds.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Simple deployment process: You prefer a straightforward deployment approach without the complexity of
         *        managing multiple environments, target groups, and listeners.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        No load balancer requirement: Your service doesn't use or require a load balancer, Application Load
         *        Balancer, Network Load Balancer, or Service Connect (which are required for blue/green deployments).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Stateful applications: Your application maintains state that makes it difficult to run two parallel
         *        environments.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Cost sensitivity: You want to minimize deployment costs by not running duplicate environments during
         *        deployment.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Rolling updates are the default deployment strategy for services and provide a balance between
         *        deployment safety and resource efficiency for many common application scenarios.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BLUE_GREEN</code>: A <i>blue/green</i> deployment strategy (<code>BLUE_GREEN</code>) is a
         *        release methodology that reduces downtime and risk by running two identical production environments
         *        called blue and green. With Amazon ECS blue/green deployments, you can validate new service revisions
         *        before directing production traffic to them. This approach provides a safer way to deploy changes with
         *        the ability to quickly roll back if needed.
         *        </p>
         *        <p>
         *        Amazon ECS blue/green deployments are best suited for the following scenarios:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Service validation: When you need to validate new service revisions before directing production
         *        traffic to them
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Zero downtime: When your service requires zero-downtime deployments
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Instant roll back: When you need the ability to quickly roll back if issues are detected
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Load balancer requirement: When your service uses Application Load Balancer, Network Load Balancer, or
         *        Service Connect
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        External
         *        </p>
         *        <p>
         *        Use a third-party deployment controller.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Blue/green deployment (powered by CodeDeploy)
         *        </p>
         *        <p>
         *        CodeDeploy installs an updated version of the application as a new replacement task set and reroutes
         *        production traffic from the original application task set to the replacement task set. The original
         *        task set is terminated after a successful deployment. Use this deployment controller to verify a new
         *        deployment of a service before sending production traffic to it.
         *        </p>
         *        </li>
         * @see DeploymentControllerType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeploymentControllerType
         */
        Builder type(String type);

        /**
         * <p>
         * The deployment controller type to use.
         * </p>
         * <p>
         * The deployment controller is the mechanism that determines how tasks are deployed for your service. The valid
         * options are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * ECS
         * </p>
         * <p>
         * When you create a service which uses the <code>ECS</code> deployment controller, you can choose between the
         * following deployment strategies:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ROLLING</code>: When you create a service which uses the <i>rolling update</i> (<code>ROLLING</code>)
         * deployment strategy, the Amazon ECS service scheduler replaces the currently running tasks with new tasks.
         * The number of tasks that Amazon ECS adds or removes from the service during a rolling update is controlled by
         * the service deployment configuration.
         * </p>
         * <p>
         * Rolling update deployments are best suited for the following scenarios:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Gradual service updates: You need to update your service incrementally without taking the entire service
         * offline at once.
         * </p>
         * </li>
         * <li>
         * <p>
         * Limited resource requirements: You want to avoid the additional resource costs of running two complete
         * environments simultaneously (as required by blue/green deployments).
         * </p>
         * </li>
         * <li>
         * <p>
         * Acceptable deployment time: Your application can tolerate a longer deployment process, as rolling updates
         * replace tasks one by one.
         * </p>
         * </li>
         * <li>
         * <p>
         * No need for instant roll back: Your service can tolerate a rollback process that takes minutes rather than
         * seconds.
         * </p>
         * </li>
         * <li>
         * <p>
         * Simple deployment process: You prefer a straightforward deployment approach without the complexity of
         * managing multiple environments, target groups, and listeners.
         * </p>
         * </li>
         * <li>
         * <p>
         * No load balancer requirement: Your service doesn't use or require a load balancer, Application Load Balancer,
         * Network Load Balancer, or Service Connect (which are required for blue/green deployments).
         * </p>
         * </li>
         * <li>
         * <p>
         * Stateful applications: Your application maintains state that makes it difficult to run two parallel
         * environments.
         * </p>
         * </li>
         * <li>
         * <p>
         * Cost sensitivity: You want to minimize deployment costs by not running duplicate environments during
         * deployment.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Rolling updates are the default deployment strategy for services and provide a balance between deployment
         * safety and resource efficiency for many common application scenarios.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BLUE_GREEN</code>: A <i>blue/green</i> deployment strategy (<code>BLUE_GREEN</code>) is a release
         * methodology that reduces downtime and risk by running two identical production environments called blue and
         * green. With Amazon ECS blue/green deployments, you can validate new service revisions before directing
         * production traffic to them. This approach provides a safer way to deploy changes with the ability to quickly
         * roll back if needed.
         * </p>
         * <p>
         * Amazon ECS blue/green deployments are best suited for the following scenarios:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Service validation: When you need to validate new service revisions before directing production traffic to
         * them
         * </p>
         * </li>
         * <li>
         * <p>
         * Zero downtime: When your service requires zero-downtime deployments
         * </p>
         * </li>
         * <li>
         * <p>
         * Instant roll back: When you need the ability to quickly roll back if issues are detected
         * </p>
         * </li>
         * <li>
         * <p>
         * Load balancer requirement: When your service uses Application Load Balancer, Network Load Balancer, or
         * Service Connect
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * External
         * </p>
         * <p>
         * Use a third-party deployment controller.
         * </p>
         * </li>
         * <li>
         * <p>
         * Blue/green deployment (powered by CodeDeploy)
         * </p>
         * <p>
         * CodeDeploy installs an updated version of the application as a new replacement task set and reroutes
         * production traffic from the original application task set to the replacement task set. The original task set
         * is terminated after a successful deployment. Use this deployment controller to verify a new deployment of a
         * service before sending production traffic to it.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The deployment controller type to use.</p>
         *        <p>
         *        The deployment controller is the mechanism that determines how tasks are deployed for your service.
         *        The valid options are:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        ECS
         *        </p>
         *        <p>
         *        When you create a service which uses the <code>ECS</code> deployment controller, you can choose
         *        between the following deployment strategies:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ROLLING</code>: When you create a service which uses the <i>rolling update</i> (
         *        <code>ROLLING</code>) deployment strategy, the Amazon ECS service scheduler replaces the currently
         *        running tasks with new tasks. The number of tasks that Amazon ECS adds or removes from the service
         *        during a rolling update is controlled by the service deployment configuration.
         *        </p>
         *        <p>
         *        Rolling update deployments are best suited for the following scenarios:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Gradual service updates: You need to update your service incrementally without taking the entire
         *        service offline at once.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Limited resource requirements: You want to avoid the additional resource costs of running two complete
         *        environments simultaneously (as required by blue/green deployments).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Acceptable deployment time: Your application can tolerate a longer deployment process, as rolling
         *        updates replace tasks one by one.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        No need for instant roll back: Your service can tolerate a rollback process that takes minutes rather
         *        than seconds.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Simple deployment process: You prefer a straightforward deployment approach without the complexity of
         *        managing multiple environments, target groups, and listeners.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        No load balancer requirement: Your service doesn't use or require a load balancer, Application Load
         *        Balancer, Network Load Balancer, or Service Connect (which are required for blue/green deployments).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Stateful applications: Your application maintains state that makes it difficult to run two parallel
         *        environments.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Cost sensitivity: You want to minimize deployment costs by not running duplicate environments during
         *        deployment.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Rolling updates are the default deployment strategy for services and provide a balance between
         *        deployment safety and resource efficiency for many common application scenarios.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BLUE_GREEN</code>: A <i>blue/green</i> deployment strategy (<code>BLUE_GREEN</code>) is a
         *        release methodology that reduces downtime and risk by running two identical production environments
         *        called blue and green. With Amazon ECS blue/green deployments, you can validate new service revisions
         *        before directing production traffic to them. This approach provides a safer way to deploy changes with
         *        the ability to quickly roll back if needed.
         *        </p>
         *        <p>
         *        Amazon ECS blue/green deployments are best suited for the following scenarios:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Service validation: When you need to validate new service revisions before directing production
         *        traffic to them
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Zero downtime: When your service requires zero-downtime deployments
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Instant roll back: When you need the ability to quickly roll back if issues are detected
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Load balancer requirement: When your service uses Application Load Balancer, Network Load Balancer, or
         *        Service Connect
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        External
         *        </p>
         *        <p>
         *        Use a third-party deployment controller.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Blue/green deployment (powered by CodeDeploy)
         *        </p>
         *        <p>
         *        CodeDeploy installs an updated version of the application as a new replacement task set and reroutes
         *        production traffic from the original application task set to the replacement task set. The original
         *        task set is terminated after a successful deployment. Use this deployment controller to verify a new
         *        deployment of a service before sending production traffic to it.
         *        </p>
         *        </li>
         * @see DeploymentControllerType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeploymentControllerType
         */
        Builder type(DeploymentControllerType type);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private BuilderImpl() {
        }

        private BuilderImpl(DeploymentController model) {
            type(model.type);
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(DeploymentControllerType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        @Override
        public DeploymentController build() {
            return new DeploymentController(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
