/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The CloudWatch alarms used to determine a service deployment failed.
 * </p>
 * <p>
 * Amazon ECS considers the service deployment as failed when any of the alarms move to the <code>ALARM</code> state.
 * For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-alarm-failure.html">How CloudWatch
 * alarms detect Amazon ECS deployment failures</a> in the Amazon ECS Developer Guide.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ServiceDeploymentAlarms implements SdkPojo, Serializable,
        ToCopyableBuilder<ServiceDeploymentAlarms.Builder, ServiceDeploymentAlarms> {
    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(ServiceDeploymentAlarms::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<List<String>> ALARM_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("alarmNames")
            .getter(getter(ServiceDeploymentAlarms::alarmNames))
            .setter(setter(Builder::alarmNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("alarmNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> TRIGGERED_ALARM_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("triggeredAlarmNames")
            .getter(getter(ServiceDeploymentAlarms::triggeredAlarmNames))
            .setter(setter(Builder::triggeredAlarmNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("triggeredAlarmNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STATUS_FIELD,
            ALARM_NAMES_FIELD, TRIGGERED_ALARM_NAMES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String status;

    private final List<String> alarmNames;

    private final List<String> triggeredAlarmNames;

    private ServiceDeploymentAlarms(BuilderImpl builder) {
        this.status = builder.status;
        this.alarmNames = builder.alarmNames;
        this.triggeredAlarmNames = builder.triggeredAlarmNames;
    }

    /**
     * <p>
     * The status of the alarms check. Amazon ECS is not using alarms for service deployment failures when the status is
     * <code>DISABLED</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ServiceDeploymentRollbackMonitorsStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the alarms check. Amazon ECS is not using alarms for service deployment failures when the
     *         status is <code>DISABLED</code>.
     * @see ServiceDeploymentRollbackMonitorsStatus
     */
    public final ServiceDeploymentRollbackMonitorsStatus status() {
        return ServiceDeploymentRollbackMonitorsStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the alarms check. Amazon ECS is not using alarms for service deployment failures when the status is
     * <code>DISABLED</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ServiceDeploymentRollbackMonitorsStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the alarms check. Amazon ECS is not using alarms for service deployment failures when the
     *         status is <code>DISABLED</code>.
     * @see ServiceDeploymentRollbackMonitorsStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * For responses, this returns true if the service returned a value for the AlarmNames property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAlarmNames() {
        return alarmNames != null && !(alarmNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The name of the CloudWatch alarms that determine when a service deployment failed. A "," separates the alarms.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAlarmNames} method.
     * </p>
     * 
     * @return The name of the CloudWatch alarms that determine when a service deployment failed. A "," separates the
     *         alarms.
     */
    public final List<String> alarmNames() {
        return alarmNames;
    }

    /**
     * For responses, this returns true if the service returned a value for the TriggeredAlarmNames property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTriggeredAlarmNames() {
        return triggeredAlarmNames != null && !(triggeredAlarmNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more CloudWatch alarm names that have been triggered during the service deployment. A "," separates the
     * alarm names.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTriggeredAlarmNames} method.
     * </p>
     * 
     * @return One or more CloudWatch alarm names that have been triggered during the service deployment. A ","
     *         separates the alarm names.
     */
    public final List<String> triggeredAlarmNames() {
        return triggeredAlarmNames;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasAlarmNames() ? alarmNames() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTriggeredAlarmNames() ? triggeredAlarmNames() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ServiceDeploymentAlarms)) {
            return false;
        }
        ServiceDeploymentAlarms other = (ServiceDeploymentAlarms) obj;
        return Objects.equals(statusAsString(), other.statusAsString()) && hasAlarmNames() == other.hasAlarmNames()
                && Objects.equals(alarmNames(), other.alarmNames()) && hasTriggeredAlarmNames() == other.hasTriggeredAlarmNames()
                && Objects.equals(triggeredAlarmNames(), other.triggeredAlarmNames());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ServiceDeploymentAlarms").add("Status", statusAsString())
                .add("AlarmNames", hasAlarmNames() ? alarmNames() : null)
                .add("TriggeredAlarmNames", hasTriggeredAlarmNames() ? triggeredAlarmNames() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "alarmNames":
            return Optional.ofNullable(clazz.cast(alarmNames()));
        case "triggeredAlarmNames":
            return Optional.ofNullable(clazz.cast(triggeredAlarmNames()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("status", STATUS_FIELD);
        map.put("alarmNames", ALARM_NAMES_FIELD);
        map.put("triggeredAlarmNames", TRIGGERED_ALARM_NAMES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ServiceDeploymentAlarms, T> g) {
        return obj -> g.apply((ServiceDeploymentAlarms) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ServiceDeploymentAlarms> {
        /**
         * <p>
         * The status of the alarms check. Amazon ECS is not using alarms for service deployment failures when the
         * status is <code>DISABLED</code>.
         * </p>
         * 
         * @param status
         *        The status of the alarms check. Amazon ECS is not using alarms for service deployment failures when
         *        the status is <code>DISABLED</code>.
         * @see ServiceDeploymentRollbackMonitorsStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ServiceDeploymentRollbackMonitorsStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the alarms check. Amazon ECS is not using alarms for service deployment failures when the
         * status is <code>DISABLED</code>.
         * </p>
         * 
         * @param status
         *        The status of the alarms check. Amazon ECS is not using alarms for service deployment failures when
         *        the status is <code>DISABLED</code>.
         * @see ServiceDeploymentRollbackMonitorsStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ServiceDeploymentRollbackMonitorsStatus
         */
        Builder status(ServiceDeploymentRollbackMonitorsStatus status);

        /**
         * <p>
         * The name of the CloudWatch alarms that determine when a service deployment failed. A "," separates the
         * alarms.
         * </p>
         * 
         * @param alarmNames
         *        The name of the CloudWatch alarms that determine when a service deployment failed. A "," separates the
         *        alarms.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarmNames(Collection<String> alarmNames);

        /**
         * <p>
         * The name of the CloudWatch alarms that determine when a service deployment failed. A "," separates the
         * alarms.
         * </p>
         * 
         * @param alarmNames
         *        The name of the CloudWatch alarms that determine when a service deployment failed. A "," separates the
         *        alarms.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarmNames(String... alarmNames);

        /**
         * <p>
         * One or more CloudWatch alarm names that have been triggered during the service deployment. A "," separates
         * the alarm names.
         * </p>
         * 
         * @param triggeredAlarmNames
         *        One or more CloudWatch alarm names that have been triggered during the service deployment. A ","
         *        separates the alarm names.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder triggeredAlarmNames(Collection<String> triggeredAlarmNames);

        /**
         * <p>
         * One or more CloudWatch alarm names that have been triggered during the service deployment. A "," separates
         * the alarm names.
         * </p>
         * 
         * @param triggeredAlarmNames
         *        One or more CloudWatch alarm names that have been triggered during the service deployment. A ","
         *        separates the alarm names.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder triggeredAlarmNames(String... triggeredAlarmNames);
    }

    static final class BuilderImpl implements Builder {
        private String status;

        private List<String> alarmNames = DefaultSdkAutoConstructList.getInstance();

        private List<String> triggeredAlarmNames = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ServiceDeploymentAlarms model) {
            status(model.status);
            alarmNames(model.alarmNames);
            triggeredAlarmNames(model.triggeredAlarmNames);
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ServiceDeploymentRollbackMonitorsStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Collection<String> getAlarmNames() {
            if (alarmNames instanceof SdkAutoConstructList) {
                return null;
            }
            return alarmNames;
        }

        public final void setAlarmNames(Collection<String> alarmNames) {
            this.alarmNames = StringListCopier.copy(alarmNames);
        }

        @Override
        public final Builder alarmNames(Collection<String> alarmNames) {
            this.alarmNames = StringListCopier.copy(alarmNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder alarmNames(String... alarmNames) {
            alarmNames(Arrays.asList(alarmNames));
            return this;
        }

        public final Collection<String> getTriggeredAlarmNames() {
            if (triggeredAlarmNames instanceof SdkAutoConstructList) {
                return null;
            }
            return triggeredAlarmNames;
        }

        public final void setTriggeredAlarmNames(Collection<String> triggeredAlarmNames) {
            this.triggeredAlarmNames = StringListCopier.copy(triggeredAlarmNames);
        }

        @Override
        public final Builder triggeredAlarmNames(Collection<String> triggeredAlarmNames) {
            this.triggeredAlarmNames = StringListCopier.copy(triggeredAlarmNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder triggeredAlarmNames(String... triggeredAlarmNames) {
            triggeredAlarmNames(Arrays.asList(triggeredAlarmNames));
            return this;
        }

        @Override
        public ServiceDeploymentAlarms build() {
            return new ServiceDeploymentAlarms(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
