/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Log configuration options to send to a custom log driver for the container.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LogConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<LogConfiguration.Builder, LogConfiguration> {
    private static final SdkField<String> LOG_DRIVER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LogConfiguration::logDriverAsString)).setter(setter(Builder::logDriver))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("logDriver").build()).build();

    private static final SdkField<Map<String, String>> OPTIONS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(LogConfiguration::options))
            .setter(setter(Builder::options))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("options").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<Secret>> SECRET_OPTIONS_FIELD = SdkField
            .<List<Secret>> builder(MarshallingType.LIST)
            .getter(getter(LogConfiguration::secretOptions))
            .setter(setter(Builder::secretOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("secretOptions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Secret> builder(MarshallingType.SDK_POJO)
                                            .constructor(Secret::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LOG_DRIVER_FIELD,
            OPTIONS_FIELD, SECRET_OPTIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String logDriver;

    private final Map<String, String> options;

    private final List<Secret> secretOptions;

    private LogConfiguration(BuilderImpl builder) {
        this.logDriver = builder.logDriver;
        this.options = builder.options;
        this.secretOptions = builder.secretOptions;
    }

    /**
     * <p>
     * The log driver to use for the container. The valid values listed for this parameter are log drivers that the
     * Amazon ECS container agent can communicate with by default.
     * </p>
     * <p>
     * For tasks using the Fargate launch type, the supported log drivers are <code>awslogs</code>, <code>splunk</code>,
     * and <code>awsfirelens</code>.
     * </p>
     * <p>
     * For tasks using the EC2 launch type, the supported log drivers are <code>awslogs</code>, <code>fluentd</code>,
     * <code>gelf</code>, <code>json-file</code>, <code>journald</code>, <code>logentries</code>, <code>syslog</code>,
     * <code>splunk</code>, and <code>awsfirelens</code>.
     * </p>
     * <p>
     * For more information about using the <code>awslogs</code> log driver, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html">Using the awslogs Log
     * Driver</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * For more information about using the <code>awsfirelens</code> log driver, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Custom Log Routing</a> in
     * the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * If you have a custom driver that is not listed above that you would like to work with the Amazon ECS container
     * agent, you can fork the Amazon ECS container agent project that is <a
     * href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with that driver.
     * We encourage you to submit pull requests for changes that you would like to have included. However, Amazon Web
     * Services does not currently support running modified copies of this software.
     * </p>
     * </note>
     * <p>
     * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the
     * Docker Remote API version on your container instance, log in to your container instance and run the following
     * command: <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #logDriver} will
     * return {@link LogDriver#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #logDriverAsString}.
     * </p>
     * 
     * @return The log driver to use for the container. The valid values listed for this parameter are log drivers that
     *         the Amazon ECS container agent can communicate with by default.</p>
     *         <p>
     *         For tasks using the Fargate launch type, the supported log drivers are <code>awslogs</code>,
     *         <code>splunk</code>, and <code>awsfirelens</code>.
     *         </p>
     *         <p>
     *         For tasks using the EC2 launch type, the supported log drivers are <code>awslogs</code>,
     *         <code>fluentd</code>, <code>gelf</code>, <code>json-file</code>, <code>journald</code>,
     *         <code>logentries</code>, <code>syslog</code>, <code>splunk</code>, and <code>awsfirelens</code>.
     *         </p>
     *         <p>
     *         For more information about using the <code>awslogs</code> log driver, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html">Using the awslogs
     *         Log Driver</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     *         </p>
     *         <p>
     *         For more information about using the <code>awsfirelens</code> log driver, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Custom Log
     *         Routing</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     *         </p>
     *         <note>
     *         <p>
     *         If you have a custom driver that is not listed above that you would like to work with the Amazon ECS
     *         container agent, you can fork the Amazon ECS container agent project that is <a
     *         href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with that
     *         driver. We encourage you to submit pull requests for changes that you would like to have included.
     *         However, Amazon Web Services does not currently support running modified copies of this software.
     *         </p>
     *         </note>
     *         <p>
     *         This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To
     *         check the Docker Remote API version on your container instance, log in to your container instance and run
     *         the following command: <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
     * @see LogDriver
     */
    public LogDriver logDriver() {
        return LogDriver.fromValue(logDriver);
    }

    /**
     * <p>
     * The log driver to use for the container. The valid values listed for this parameter are log drivers that the
     * Amazon ECS container agent can communicate with by default.
     * </p>
     * <p>
     * For tasks using the Fargate launch type, the supported log drivers are <code>awslogs</code>, <code>splunk</code>,
     * and <code>awsfirelens</code>.
     * </p>
     * <p>
     * For tasks using the EC2 launch type, the supported log drivers are <code>awslogs</code>, <code>fluentd</code>,
     * <code>gelf</code>, <code>json-file</code>, <code>journald</code>, <code>logentries</code>, <code>syslog</code>,
     * <code>splunk</code>, and <code>awsfirelens</code>.
     * </p>
     * <p>
     * For more information about using the <code>awslogs</code> log driver, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html">Using the awslogs Log
     * Driver</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * For more information about using the <code>awsfirelens</code> log driver, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Custom Log Routing</a> in
     * the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * If you have a custom driver that is not listed above that you would like to work with the Amazon ECS container
     * agent, you can fork the Amazon ECS container agent project that is <a
     * href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with that driver.
     * We encourage you to submit pull requests for changes that you would like to have included. However, Amazon Web
     * Services does not currently support running modified copies of this software.
     * </p>
     * </note>
     * <p>
     * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the
     * Docker Remote API version on your container instance, log in to your container instance and run the following
     * command: <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #logDriver} will
     * return {@link LogDriver#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #logDriverAsString}.
     * </p>
     * 
     * @return The log driver to use for the container. The valid values listed for this parameter are log drivers that
     *         the Amazon ECS container agent can communicate with by default.</p>
     *         <p>
     *         For tasks using the Fargate launch type, the supported log drivers are <code>awslogs</code>,
     *         <code>splunk</code>, and <code>awsfirelens</code>.
     *         </p>
     *         <p>
     *         For tasks using the EC2 launch type, the supported log drivers are <code>awslogs</code>,
     *         <code>fluentd</code>, <code>gelf</code>, <code>json-file</code>, <code>journald</code>,
     *         <code>logentries</code>, <code>syslog</code>, <code>splunk</code>, and <code>awsfirelens</code>.
     *         </p>
     *         <p>
     *         For more information about using the <code>awslogs</code> log driver, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html">Using the awslogs
     *         Log Driver</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     *         </p>
     *         <p>
     *         For more information about using the <code>awsfirelens</code> log driver, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Custom Log
     *         Routing</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     *         </p>
     *         <note>
     *         <p>
     *         If you have a custom driver that is not listed above that you would like to work with the Amazon ECS
     *         container agent, you can fork the Amazon ECS container agent project that is <a
     *         href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with that
     *         driver. We encourage you to submit pull requests for changes that you would like to have included.
     *         However, Amazon Web Services does not currently support running modified copies of this software.
     *         </p>
     *         </note>
     *         <p>
     *         This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To
     *         check the Docker Remote API version on your container instance, log in to your container instance and run
     *         the following command: <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
     * @see LogDriver
     */
    public String logDriverAsString() {
        return logDriver;
    }

    /**
     * <p>
     * The configuration options to send to the log driver. This parameter requires version 1.19 of the Docker Remote
     * API or greater on your container instance. To check the Docker Remote API version on your container instance, log
     * in to your container instance and run the following command:
     * <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The configuration options to send to the log driver. This parameter requires version 1.19 of the Docker
     *         Remote API or greater on your container instance. To check the Docker Remote API version on your
     *         container instance, log in to your container instance and run the following command:
     *         <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
     */
    public Map<String, String> options() {
        return options;
    }

    /**
     * <p>
     * The secrets to pass to the log configuration. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html">Specifying
     * Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The secrets to pass to the log configuration. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html"
     *         >Specifying Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     */
    public List<Secret> secretOptions() {
        return secretOptions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(logDriverAsString());
        hashCode = 31 * hashCode + Objects.hashCode(options());
        hashCode = 31 * hashCode + Objects.hashCode(secretOptions());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LogConfiguration)) {
            return false;
        }
        LogConfiguration other = (LogConfiguration) obj;
        return Objects.equals(logDriverAsString(), other.logDriverAsString()) && Objects.equals(options(), other.options())
                && Objects.equals(secretOptions(), other.secretOptions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("LogConfiguration").add("LogDriver", logDriverAsString()).add("Options", options())
                .add("SecretOptions", secretOptions()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "logDriver":
            return Optional.ofNullable(clazz.cast(logDriverAsString()));
        case "options":
            return Optional.ofNullable(clazz.cast(options()));
        case "secretOptions":
            return Optional.ofNullable(clazz.cast(secretOptions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LogConfiguration, T> g) {
        return obj -> g.apply((LogConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LogConfiguration> {
        /**
         * <p>
         * The log driver to use for the container. The valid values listed for this parameter are log drivers that the
         * Amazon ECS container agent can communicate with by default.
         * </p>
         * <p>
         * For tasks using the Fargate launch type, the supported log drivers are <code>awslogs</code>,
         * <code>splunk</code>, and <code>awsfirelens</code>.
         * </p>
         * <p>
         * For tasks using the EC2 launch type, the supported log drivers are <code>awslogs</code>, <code>fluentd</code>, <code>gelf</code>, <code>json-file</code>, <code>journald</code>, <code>logentries</code>,
         * <code>syslog</code>, <code>splunk</code>, and <code>awsfirelens</code>.
         * </p>
         * <p>
         * For more information about using the <code>awslogs</code> log driver, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html">Using the awslogs Log
         * Driver</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <p>
         * For more information about using the <code>awsfirelens</code> log driver, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Custom Log Routing</a>
         * in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <note>
         * <p>
         * If you have a custom driver that is not listed above that you would like to work with the Amazon ECS
         * container agent, you can fork the Amazon ECS container agent project that is <a
         * href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with that
         * driver. We encourage you to submit pull requests for changes that you would like to have included. However,
         * Amazon Web Services does not currently support running modified copies of this software.
         * </p>
         * </note>
         * <p>
         * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check
         * the Docker Remote API version on your container instance, log in to your container instance and run the
         * following command: <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
         * </p>
         * 
         * @param logDriver
         *        The log driver to use for the container. The valid values listed for this parameter are log drivers
         *        that the Amazon ECS container agent can communicate with by default.</p>
         *        <p>
         *        For tasks using the Fargate launch type, the supported log drivers are <code>awslogs</code>,
         *        <code>splunk</code>, and <code>awsfirelens</code>.
         *        </p>
         *        <p>
         *        For tasks using the EC2 launch type, the supported log drivers are <code>awslogs</code>,
         *        <code>fluentd</code>, <code>gelf</code>, <code>json-file</code>, <code>journald</code>,
         *        <code>logentries</code>, <code>syslog</code>, <code>splunk</code>, and <code>awsfirelens</code>.
         *        </p>
         *        <p>
         *        For more information about using the <code>awslogs</code> log driver, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html">Using the
         *        awslogs Log Driver</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         *        </p>
         *        <p>
         *        For more information about using the <code>awsfirelens</code> log driver, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Custom Log
         *        Routing</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         *        </p>
         *        <note>
         *        <p>
         *        If you have a custom driver that is not listed above that you would like to work with the Amazon ECS
         *        container agent, you can fork the Amazon ECS container agent project that is <a
         *        href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with
         *        that driver. We encourage you to submit pull requests for changes that you would like to have
         *        included. However, Amazon Web Services does not currently support running modified copies of this
         *        software.
         *        </p>
         *        </note>
         *        <p>
         *        This parameter requires version 1.18 of the Docker Remote API or greater on your container instance.
         *        To check the Docker Remote API version on your container instance, log in to your container instance
         *        and run the following command: <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
         * @see LogDriver
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogDriver
         */
        Builder logDriver(String logDriver);

        /**
         * <p>
         * The log driver to use for the container. The valid values listed for this parameter are log drivers that the
         * Amazon ECS container agent can communicate with by default.
         * </p>
         * <p>
         * For tasks using the Fargate launch type, the supported log drivers are <code>awslogs</code>,
         * <code>splunk</code>, and <code>awsfirelens</code>.
         * </p>
         * <p>
         * For tasks using the EC2 launch type, the supported log drivers are <code>awslogs</code>, <code>fluentd</code>, <code>gelf</code>, <code>json-file</code>, <code>journald</code>, <code>logentries</code>,
         * <code>syslog</code>, <code>splunk</code>, and <code>awsfirelens</code>.
         * </p>
         * <p>
         * For more information about using the <code>awslogs</code> log driver, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html">Using the awslogs Log
         * Driver</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <p>
         * For more information about using the <code>awsfirelens</code> log driver, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Custom Log Routing</a>
         * in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * <note>
         * <p>
         * If you have a custom driver that is not listed above that you would like to work with the Amazon ECS
         * container agent, you can fork the Amazon ECS container agent project that is <a
         * href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with that
         * driver. We encourage you to submit pull requests for changes that you would like to have included. However,
         * Amazon Web Services does not currently support running modified copies of this software.
         * </p>
         * </note>
         * <p>
         * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check
         * the Docker Remote API version on your container instance, log in to your container instance and run the
         * following command: <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
         * </p>
         * 
         * @param logDriver
         *        The log driver to use for the container. The valid values listed for this parameter are log drivers
         *        that the Amazon ECS container agent can communicate with by default.</p>
         *        <p>
         *        For tasks using the Fargate launch type, the supported log drivers are <code>awslogs</code>,
         *        <code>splunk</code>, and <code>awsfirelens</code>.
         *        </p>
         *        <p>
         *        For tasks using the EC2 launch type, the supported log drivers are <code>awslogs</code>,
         *        <code>fluentd</code>, <code>gelf</code>, <code>json-file</code>, <code>journald</code>,
         *        <code>logentries</code>, <code>syslog</code>, <code>splunk</code>, and <code>awsfirelens</code>.
         *        </p>
         *        <p>
         *        For more information about using the <code>awslogs</code> log driver, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html">Using the
         *        awslogs Log Driver</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         *        </p>
         *        <p>
         *        For more information about using the <code>awsfirelens</code> log driver, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html">Custom Log
         *        Routing</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         *        </p>
         *        <note>
         *        <p>
         *        If you have a custom driver that is not listed above that you would like to work with the Amazon ECS
         *        container agent, you can fork the Amazon ECS container agent project that is <a
         *        href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with
         *        that driver. We encourage you to submit pull requests for changes that you would like to have
         *        included. However, Amazon Web Services does not currently support running modified copies of this
         *        software.
         *        </p>
         *        </note>
         *        <p>
         *        This parameter requires version 1.18 of the Docker Remote API or greater on your container instance.
         *        To check the Docker Remote API version on your container instance, log in to your container instance
         *        and run the following command: <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
         * @see LogDriver
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogDriver
         */
        Builder logDriver(LogDriver logDriver);

        /**
         * <p>
         * The configuration options to send to the log driver. This parameter requires version 1.19 of the Docker
         * Remote API or greater on your container instance. To check the Docker Remote API version on your container
         * instance, log in to your container instance and run the following command:
         * <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
         * </p>
         * 
         * @param options
         *        The configuration options to send to the log driver. This parameter requires version 1.19 of the
         *        Docker Remote API or greater on your container instance. To check the Docker Remote API version on
         *        your container instance, log in to your container instance and run the following command:
         *        <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder options(Map<String, String> options);

        /**
         * <p>
         * The secrets to pass to the log configuration. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html">Specifying
         * Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param secretOptions
         *        The secrets to pass to the log configuration. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html"
         *        >Specifying Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretOptions(Collection<Secret> secretOptions);

        /**
         * <p>
         * The secrets to pass to the log configuration. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html">Specifying
         * Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param secretOptions
         *        The secrets to pass to the log configuration. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html"
         *        >Specifying Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretOptions(Secret... secretOptions);

        /**
         * <p>
         * The secrets to pass to the log configuration. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html">Specifying
         * Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Secret>.Builder} avoiding the need to
         * create one manually via {@link List<Secret>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Secret>.Builder#build()} is called immediately and its
         * result is passed to {@link #secretOptions(List<Secret>)}.
         * 
         * @param secretOptions
         *        a consumer that will call methods on {@link List<Secret>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #secretOptions(List<Secret>)
         */
        Builder secretOptions(Consumer<Secret.Builder>... secretOptions);
    }

    static final class BuilderImpl implements Builder {
        private String logDriver;

        private Map<String, String> options = DefaultSdkAutoConstructMap.getInstance();

        private List<Secret> secretOptions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(LogConfiguration model) {
            logDriver(model.logDriver);
            options(model.options);
            secretOptions(model.secretOptions);
        }

        public final String getLogDriverAsString() {
            return logDriver;
        }

        @Override
        public final Builder logDriver(String logDriver) {
            this.logDriver = logDriver;
            return this;
        }

        @Override
        public final Builder logDriver(LogDriver logDriver) {
            this.logDriver(logDriver == null ? null : logDriver.toString());
            return this;
        }

        public final void setLogDriver(String logDriver) {
            this.logDriver = logDriver;
        }

        public final Map<String, String> getOptions() {
            return options;
        }

        @Override
        public final Builder options(Map<String, String> options) {
            this.options = LogConfigurationOptionsMapCopier.copy(options);
            return this;
        }

        public final void setOptions(Map<String, String> options) {
            this.options = LogConfigurationOptionsMapCopier.copy(options);
        }

        public final Collection<Secret.Builder> getSecretOptions() {
            return secretOptions != null ? secretOptions.stream().map(Secret::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder secretOptions(Collection<Secret> secretOptions) {
            this.secretOptions = SecretListCopier.copy(secretOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secretOptions(Secret... secretOptions) {
            secretOptions(Arrays.asList(secretOptions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secretOptions(Consumer<Secret.Builder>... secretOptions) {
            secretOptions(Stream.of(secretOptions).map(c -> Secret.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setSecretOptions(Collection<Secret.BuilderImpl> secretOptions) {
            this.secretOptions = SecretListCopier.copyFromBuilder(secretOptions);
        }

        @Override
        public LogConfiguration build() {
            return new LogConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
