/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Details on the network bindings between a container and its host container instance. After a task reaches the
 * <code>RUNNING</code> status, manual and automatic host and container port assignments are visible in the
 * <code>networkBindings</code> section of <a>DescribeTasks</a> API responses.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NetworkBinding implements SdkPojo, Serializable, ToCopyableBuilder<NetworkBinding.Builder, NetworkBinding> {
    private static final SdkField<String> BIND_IP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(NetworkBinding::bindIP)).setter(setter(Builder::bindIP))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bindIP").build()).build();

    private static final SdkField<Integer> CONTAINER_PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(NetworkBinding::containerPort)).setter(setter(Builder::containerPort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerPort").build()).build();

    private static final SdkField<Integer> HOST_PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(NetworkBinding::hostPort)).setter(setter(Builder::hostPort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("hostPort").build()).build();

    private static final SdkField<String> PROTOCOL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(NetworkBinding::protocolAsString)).setter(setter(Builder::protocol))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("protocol").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BIND_IP_FIELD,
            CONTAINER_PORT_FIELD, HOST_PORT_FIELD, PROTOCOL_FIELD));

    private static final long serialVersionUID = 1L;

    private final String bindIP;

    private final Integer containerPort;

    private final Integer hostPort;

    private final String protocol;

    private NetworkBinding(BuilderImpl builder) {
        this.bindIP = builder.bindIP;
        this.containerPort = builder.containerPort;
        this.hostPort = builder.hostPort;
        this.protocol = builder.protocol;
    }

    /**
     * <p>
     * The IP address that the container is bound to on the container instance.
     * </p>
     * 
     * @return The IP address that the container is bound to on the container instance.
     */
    public String bindIP() {
        return bindIP;
    }

    /**
     * <p>
     * The port number on the container that is used with the network binding.
     * </p>
     * 
     * @return The port number on the container that is used with the network binding.
     */
    public Integer containerPort() {
        return containerPort;
    }

    /**
     * <p>
     * The port number on the host that is used with the network binding.
     * </p>
     * 
     * @return The port number on the host that is used with the network binding.
     */
    public Integer hostPort() {
        return hostPort;
    }

    /**
     * <p>
     * The protocol used for the network binding.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #protocol} will
     * return {@link TransportProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #protocolAsString}.
     * </p>
     * 
     * @return The protocol used for the network binding.
     * @see TransportProtocol
     */
    public TransportProtocol protocol() {
        return TransportProtocol.fromValue(protocol);
    }

    /**
     * <p>
     * The protocol used for the network binding.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #protocol} will
     * return {@link TransportProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #protocolAsString}.
     * </p>
     * 
     * @return The protocol used for the network binding.
     * @see TransportProtocol
     */
    public String protocolAsString() {
        return protocol;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bindIP());
        hashCode = 31 * hashCode + Objects.hashCode(containerPort());
        hashCode = 31 * hashCode + Objects.hashCode(hostPort());
        hashCode = 31 * hashCode + Objects.hashCode(protocolAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NetworkBinding)) {
            return false;
        }
        NetworkBinding other = (NetworkBinding) obj;
        return Objects.equals(bindIP(), other.bindIP()) && Objects.equals(containerPort(), other.containerPort())
                && Objects.equals(hostPort(), other.hostPort()) && Objects.equals(protocolAsString(), other.protocolAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("NetworkBinding").add("BindIP", bindIP()).add("ContainerPort", containerPort())
                .add("HostPort", hostPort()).add("Protocol", protocolAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "bindIP":
            return Optional.ofNullable(clazz.cast(bindIP()));
        case "containerPort":
            return Optional.ofNullable(clazz.cast(containerPort()));
        case "hostPort":
            return Optional.ofNullable(clazz.cast(hostPort()));
        case "protocol":
            return Optional.ofNullable(clazz.cast(protocolAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<NetworkBinding, T> g) {
        return obj -> g.apply((NetworkBinding) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, NetworkBinding> {
        /**
         * <p>
         * The IP address that the container is bound to on the container instance.
         * </p>
         * 
         * @param bindIP
         *        The IP address that the container is bound to on the container instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bindIP(String bindIP);

        /**
         * <p>
         * The port number on the container that is used with the network binding.
         * </p>
         * 
         * @param containerPort
         *        The port number on the container that is used with the network binding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerPort(Integer containerPort);

        /**
         * <p>
         * The port number on the host that is used with the network binding.
         * </p>
         * 
         * @param hostPort
         *        The port number on the host that is used with the network binding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hostPort(Integer hostPort);

        /**
         * <p>
         * The protocol used for the network binding.
         * </p>
         * 
         * @param protocol
         *        The protocol used for the network binding.
         * @see TransportProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransportProtocol
         */
        Builder protocol(String protocol);

        /**
         * <p>
         * The protocol used for the network binding.
         * </p>
         * 
         * @param protocol
         *        The protocol used for the network binding.
         * @see TransportProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransportProtocol
         */
        Builder protocol(TransportProtocol protocol);
    }

    static final class BuilderImpl implements Builder {
        private String bindIP;

        private Integer containerPort;

        private Integer hostPort;

        private String protocol;

        private BuilderImpl() {
        }

        private BuilderImpl(NetworkBinding model) {
            bindIP(model.bindIP);
            containerPort(model.containerPort);
            hostPort(model.hostPort);
            protocol(model.protocol);
        }

        public final String getBindIP() {
            return bindIP;
        }

        @Override
        public final Builder bindIP(String bindIP) {
            this.bindIP = bindIP;
            return this;
        }

        public final void setBindIP(String bindIP) {
            this.bindIP = bindIP;
        }

        public final Integer getContainerPort() {
            return containerPort;
        }

        @Override
        public final Builder containerPort(Integer containerPort) {
            this.containerPort = containerPort;
            return this;
        }

        public final void setContainerPort(Integer containerPort) {
            this.containerPort = containerPort;
        }

        public final Integer getHostPort() {
            return hostPort;
        }

        @Override
        public final Builder hostPort(Integer hostPort) {
            this.hostPort = hostPort;
            return this;
        }

        public final void setHostPort(Integer hostPort) {
            this.hostPort = hostPort;
        }

        public final String getProtocolAsString() {
            return protocol;
        }

        @Override
        public final Builder protocol(String protocol) {
            this.protocol = protocol;
            return this;
        }

        @Override
        public final Builder protocol(TransportProtocol protocol) {
            this.protocol(protocol == null ? null : protocol.toString());
            return this;
        }

        public final void setProtocol(String protocol) {
            this.protocol = protocol;
        }

        @Override
        public NetworkBinding build() {
            return new NetworkBinding(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
