/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.efs;

import java.util.concurrent.CompletableFuture;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.AmazonServiceException;
import software.amazon.awssdk.core.client.AsyncClientHandler;
import software.amazon.awssdk.core.client.ClientExecutionParams;
import software.amazon.awssdk.core.client.SdkAsyncClientHandler;
import software.amazon.awssdk.core.config.AsyncClientConfiguration;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.protocol.json.JsonClientMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorResponseMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorShapeMetadata;
import software.amazon.awssdk.core.protocol.json.JsonOperationMetadata;
import software.amazon.awssdk.core.protocol.json.SdkJsonProtocolFactory;
import software.amazon.awssdk.services.efs.model.BadRequestException;
import software.amazon.awssdk.services.efs.model.CreateFileSystemRequest;
import software.amazon.awssdk.services.efs.model.CreateFileSystemResponse;
import software.amazon.awssdk.services.efs.model.CreateMountTargetRequest;
import software.amazon.awssdk.services.efs.model.CreateMountTargetResponse;
import software.amazon.awssdk.services.efs.model.CreateTagsRequest;
import software.amazon.awssdk.services.efs.model.CreateTagsResponse;
import software.amazon.awssdk.services.efs.model.DeleteFileSystemRequest;
import software.amazon.awssdk.services.efs.model.DeleteFileSystemResponse;
import software.amazon.awssdk.services.efs.model.DeleteMountTargetRequest;
import software.amazon.awssdk.services.efs.model.DeleteMountTargetResponse;
import software.amazon.awssdk.services.efs.model.DeleteTagsRequest;
import software.amazon.awssdk.services.efs.model.DeleteTagsResponse;
import software.amazon.awssdk.services.efs.model.DependencyTimeoutException;
import software.amazon.awssdk.services.efs.model.DescribeFileSystemsRequest;
import software.amazon.awssdk.services.efs.model.DescribeFileSystemsResponse;
import software.amazon.awssdk.services.efs.model.DescribeMountTargetSecurityGroupsRequest;
import software.amazon.awssdk.services.efs.model.DescribeMountTargetSecurityGroupsResponse;
import software.amazon.awssdk.services.efs.model.DescribeMountTargetsRequest;
import software.amazon.awssdk.services.efs.model.DescribeMountTargetsResponse;
import software.amazon.awssdk.services.efs.model.DescribeTagsRequest;
import software.amazon.awssdk.services.efs.model.DescribeTagsResponse;
import software.amazon.awssdk.services.efs.model.FileSystemAlreadyExistsException;
import software.amazon.awssdk.services.efs.model.FileSystemInUseException;
import software.amazon.awssdk.services.efs.model.FileSystemLimitExceededException;
import software.amazon.awssdk.services.efs.model.FileSystemNotFoundException;
import software.amazon.awssdk.services.efs.model.IncorrectFileSystemLifeCycleStateException;
import software.amazon.awssdk.services.efs.model.IncorrectMountTargetStateException;
import software.amazon.awssdk.services.efs.model.InternalServerErrorException;
import software.amazon.awssdk.services.efs.model.IpAddressInUseException;
import software.amazon.awssdk.services.efs.model.ModifyMountTargetSecurityGroupsRequest;
import software.amazon.awssdk.services.efs.model.ModifyMountTargetSecurityGroupsResponse;
import software.amazon.awssdk.services.efs.model.MountTargetConflictException;
import software.amazon.awssdk.services.efs.model.MountTargetNotFoundException;
import software.amazon.awssdk.services.efs.model.NetworkInterfaceLimitExceededException;
import software.amazon.awssdk.services.efs.model.NoFreeAddressesInSubnetException;
import software.amazon.awssdk.services.efs.model.SecurityGroupLimitExceededException;
import software.amazon.awssdk.services.efs.model.SecurityGroupNotFoundException;
import software.amazon.awssdk.services.efs.model.SubnetNotFoundException;
import software.amazon.awssdk.services.efs.model.UnsupportedAvailabilityZoneException;
import software.amazon.awssdk.services.efs.transform.CreateFileSystemRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.CreateFileSystemResponseUnmarshaller;
import software.amazon.awssdk.services.efs.transform.CreateMountTargetRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.CreateMountTargetResponseUnmarshaller;
import software.amazon.awssdk.services.efs.transform.CreateTagsRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.CreateTagsResponseUnmarshaller;
import software.amazon.awssdk.services.efs.transform.DeleteFileSystemRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DeleteFileSystemResponseUnmarshaller;
import software.amazon.awssdk.services.efs.transform.DeleteMountTargetRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DeleteMountTargetResponseUnmarshaller;
import software.amazon.awssdk.services.efs.transform.DeleteTagsRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DeleteTagsResponseUnmarshaller;
import software.amazon.awssdk.services.efs.transform.DescribeFileSystemsRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DescribeFileSystemsResponseUnmarshaller;
import software.amazon.awssdk.services.efs.transform.DescribeMountTargetSecurityGroupsRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DescribeMountTargetSecurityGroupsResponseUnmarshaller;
import software.amazon.awssdk.services.efs.transform.DescribeMountTargetsRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DescribeMountTargetsResponseUnmarshaller;
import software.amazon.awssdk.services.efs.transform.DescribeTagsRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DescribeTagsResponseUnmarshaller;
import software.amazon.awssdk.services.efs.transform.ModifyMountTargetSecurityGroupsRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.ModifyMountTargetSecurityGroupsResponseUnmarshaller;

/**
 * Internal implementation of {@link EFSAsyncClient}.
 *
 * @see EFSAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultEFSAsyncClient implements EFSAsyncClient {
    private final AsyncClientHandler clientHandler;

    private final SdkJsonProtocolFactory protocolFactory;

    protected DefaultEFSAsyncClient(AsyncClientConfiguration clientConfiguration) {
        this.clientHandler = new SdkAsyncClientHandler(clientConfiguration, null);
        this.protocolFactory = init();
    }

    /**
     * <p>
     * Creates a new, empty file system. The operation requires a creation token in the request that Amazon EFS uses to
     * ensure idempotent creation (calling the operation with same creation token has no effect). If a file system does
     * not currently exist that is owned by the caller's AWS account with the specified creation token, this operation
     * does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a new, empty file system. The file system will have an Amazon EFS assigned ID, and an initial lifecycle
     * state <code>creating</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Returns with the description of the created file system.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Otherwise, this operation returns a <code>FileSystemAlreadyExists</code> error with the ID of the existing file
     * system.
     * </p>
     * <note>
     * <p>
     * For basic use cases, you can use a randomly generated UUID for the creation token.
     * </p>
     * </note>
     * <p>
     * The idempotent operation allows you to retry a <code>CreateFileSystem</code> call without risk of creating an
     * extra file system. This can happen when an initial call fails in a way that leaves it uncertain whether or not a
     * file system was actually created. An example might be that a transport level timeout occurred or your connection
     * was reset. As long as you use the same creation token, if the initial call had succeeded in creating a file
     * system, the client can learn of its existence from the <code>FileSystemAlreadyExists</code> error.
     * </p>
     * <note>
     * <p>
     * The <code>CreateFileSystem</code> call returns while the file system's lifecycle state is still
     * <code>creating</code>. You can check the file system creation status by calling the <a>DescribeFileSystems</a>
     * operation, which among other things returns the file system state.
     * </p>
     * </note>
     * <p>
     * This operation also takes an optional <code>PerformanceMode</code> parameter that you choose for your file
     * system. We recommend <code>generalPurpose</code> performance mode for most file systems. File systems using the
     * <code>maxIO</code> performance mode can scale to higher levels of aggregate throughput and operations per second
     * with a tradeoff of slightly higher latencies for most file operations. The performance mode can't be changed
     * after the file system has been created. For more information, see <a
     * href="http://docs.aws.amazon.com/efs/latest/ug/performance.html#performancemodes.html">Amazon EFS: Performance
     * Modes</a>.
     * </p>
     * <p>
     * After the file system is fully created, Amazon EFS sets its lifecycle state to <code>available</code>, at which
     * point you can create one or more mount targets for the file system in your VPC. For more information, see
     * <a>CreateMountTarget</a>. You mount your Amazon EFS file system on an EC2 instances in your VPC via the mount
     * target. For more information, see <a href="http://docs.aws.amazon.com/efs/latest/ug/how-it-works.html">Amazon
     * EFS: How it Works</a>.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:CreateFileSystem</code> action.
     * </p>
     *
     * @param createFileSystemRequest
     * @return A Java Future containing the result of the CreateFileSystem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemAlreadyExistsException Returned if the file system you are trying to create already exists,
     *         with the creation token you provided.</li>
     *         <li>FileSystemLimitExceededException Returned if the AWS account has already created maximum number of
     *         file systems allowed per account.</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EFSException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EFSAsyncClient.CreateFileSystem
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/CreateFileSystem"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateFileSystemResponse> createFileSystem(CreateFileSystemRequest createFileSystemRequest) {

        HttpResponseHandler<CreateFileSystemResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new CreateFileSystemResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<CreateFileSystemRequest, CreateFileSystemResponse>()
                .withMarshaller(new CreateFileSystemRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createFileSystemRequest));
    }

    /**
     * <p>
     * Creates a mount target for a file system. You can then mount the file system on EC2 instances via the mount
     * target.
     * </p>
     * <p>
     * You can create one mount target in each Availability Zone in your VPC. All EC2 instances in a VPC within a given
     * Availability Zone share a single mount target for a given file system. If you have multiple subnets in an
     * Availability Zone, you create a mount target in one of the subnets. EC2 instances do not need to be in the same
     * subnet as the mount target in order to access their file system. For more information, see <a
     * href="http://docs.aws.amazon.com/efs/latest/ug/how-it-works.html">Amazon EFS: How it Works</a>.
     * </p>
     * <p>
     * In the request, you also specify a file system ID for which you are creating the mount target and the file
     * system's lifecycle state must be <code>available</code>. For more information, see <a>DescribeFileSystems</a>.
     * </p>
     * <p>
     * In the request, you also provide a subnet ID, which determines the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * VPC in which Amazon EFS creates the mount target
     * </p>
     * </li>
     * <li>
     * <p>
     * Availability Zone in which Amazon EFS creates the mount target
     * </p>
     * </li>
     * <li>
     * <p>
     * IP address range from which Amazon EFS selects the IP address of the mount target (if you don't specify an IP
     * address in the request)
     * </p>
     * </li>
     * </ul>
     * <p>
     * After creating the mount target, Amazon EFS returns a response that includes, a <code>MountTargetId</code> and an
     * <code>IpAddress</code>. You use this IP address when mounting the file system in an EC2 instance. You can also
     * use the mount target's DNS name when mounting the file system. The EC2 instance on which you mount the file
     * system via the mount target can resolve the mount target's DNS name to its IP address. For more information, see
     * <a href="http://docs.aws.amazon.com/efs/latest/ug/how-it-works.html#how-it-works-implementation">How it Works:
     * Implementation Overview</a>.
     * </p>
     * <p>
     * Note that you can create mount targets for a file system in only one VPC, and there can be only one mount target
     * per Availability Zone. That is, if the file system already has one or more mount targets created for it, the
     * subnet specified in the request to add another mount target must meet the following requirements:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Must belong to the same VPC as the subnets of the existing mount targets
     * </p>
     * </li>
     * <li>
     * <p>
     * Must not be in the same Availability Zone as any of the subnets of the existing mount targets
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the request satisfies the requirements, Amazon EFS does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a new mount target in the specified subnet.
     * </p>
     * </li>
     * <li>
     * <p>
     * Also creates a new network interface in the subnet as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the request provides an <code>IpAddress</code>, Amazon EFS assigns that IP address to the network interface.
     * Otherwise, Amazon EFS assigns a free address in the subnet (in the same way that the Amazon EC2
     * <code>CreateNetworkInterface</code> call does when a request does not specify a primary private IP address).
     * </p>
     * </li>
     * <li>
     * <p>
     * If the request provides <code>SecurityGroups</code>, this network interface is associated with those security
     * groups. Otherwise, it belongs to the default security group for the subnet's VPC.
     * </p>
     * </li>
     * <li>
     * <p>
     * Assigns the description <code>Mount target <i>fsmt-id</i> for file system <i>fs-id</i> </code> where
     * <code> <i>fsmt-id</i> </code> is the mount target ID, and <code> <i>fs-id</i> </code> is the
     * <code>FileSystemId</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Sets the <code>requesterManaged</code> property of the network interface to <code>true</code>, and the
     * <code>requesterId</code> value to <code>EFS</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Each Amazon EFS mount target has one corresponding requester-managed EC2 network interface. After the network
     * interface is created, Amazon EFS sets the <code>NetworkInterfaceId</code> field in the mount target's description
     * to the network interface ID, and the <code>IpAddress</code> field to its address. If network interface creation
     * fails, the entire <code>CreateMountTarget</code> operation fails.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The <code>CreateMountTarget</code> call returns only after creating the network interface, but while the mount
     * target state is still <code>creating</code>, you can check the mount target creation status by calling the
     * <a>DescribeMountTargets</a> operation, which among other things returns the mount target state.
     * </p>
     * </note>
     * <p>
     * We recommend you create a mount target in each of the Availability Zones. There are cost considerations for using
     * a file system in an Availability Zone through a mount target created in another Availability Zone. For more
     * information, see <a href="http://aws.amazon.com/efs/">Amazon EFS</a>. In addition, by always using a mount target
     * local to the instance's Availability Zone, you eliminate a partial failure scenario. If the Availability Zone in
     * which your mount target is created goes down, then you won't be able to access your file system through that
     * mount target.
     * </p>
     * <p>
     * This operation requires permissions for the following action on the file system:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>elasticfilesystem:CreateMountTarget</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * This operation also requires permissions for the following Amazon EC2 actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ec2:DescribeSubnets</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:DescribeNetworkInterfaces</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:CreateNetworkInterface</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param createMountTargetRequest
     * @return A Java Future containing the result of the CreateMountTarget operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> does not exist in the
     *         requester's AWS account.</li>
     *         <li>IncorrectFileSystemLifeCycleStateException Returned if the file system's life cycle state is not
     *         "created".</li>
     *         <li>MountTargetConflictException Returned if the mount target would violate one of the specified
     *         restrictions based on the file system's existing mount targets.</li>
     *         <li>SubnetNotFoundException Returned if there is no subnet with ID <code>SubnetId</code> provided in the
     *         request.</li>
     *         <li>NoFreeAddressesInSubnetException Returned if <code>IpAddress</code> was not specified in the request
     *         and there are no free IP addresses in the subnet.</li>
     *         <li>IpAddressInUseException Returned if the request specified an <code>IpAddress</code> that is already
     *         in use in the subnet.</li>
     *         <li>NetworkInterfaceLimitExceededException The calling account has reached the ENI limit for the specific
     *         AWS region. Client should try to delete some ENIs or get its account limit raised. For more information,
     *         see <a href="http://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/VPC_Appendix_Limits.html">Amazon VPC
     *         Limits</a> in the Amazon Virtual Private Cloud User Guide (see the Network interfaces per VPC entry in
     *         the table).</li>
     *         <li>SecurityGroupLimitExceededException Returned if the size of <code>SecurityGroups</code> specified in
     *         the request is greater than five.</li>
     *         <li>SecurityGroupNotFoundException Returned if one of the specified security groups does not exist in the
     *         subnet's VPC.</li>
     *         <li>UnsupportedAvailabilityZoneException</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EFSException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EFSAsyncClient.CreateMountTarget
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/CreateMountTarget"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMountTargetResponse> createMountTarget(CreateMountTargetRequest createMountTargetRequest) {

        HttpResponseHandler<CreateMountTargetResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new CreateMountTargetResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<CreateMountTargetRequest, CreateMountTargetResponse>()
                .withMarshaller(new CreateMountTargetRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createMountTargetRequest));
    }

    /**
     * <p>
     * Creates or overwrites tags associated with a file system. Each tag is a key-value pair. If a tag key specified in
     * the request already exists on the file system, this operation overwrites its value with the value provided in the
     * request. If you add the <code>Name</code> tag to your file system, Amazon EFS returns it in the response to the
     * <a>DescribeFileSystems</a> operation.
     * </p>
     * <p>
     * This operation requires permission for the <code>elasticfilesystem:CreateTags</code> action.
     * </p>
     *
     * @param createTagsRequest
     * @return A Java Future containing the result of the CreateTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> does not exist in the
     *         requester's AWS account.</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EFSException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EFSAsyncClient.CreateTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/CreateTags" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTagsResponse> createTags(CreateTagsRequest createTagsRequest) {

        HttpResponseHandler<CreateTagsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new CreateTagsResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<CreateTagsRequest, CreateTagsResponse>()
                .withMarshaller(new CreateTagsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createTagsRequest));
    }

    /**
     * <p>
     * Deletes a file system, permanently severing access to its contents. Upon return, the file system no longer exists
     * and you can't access any contents of the deleted file system.
     * </p>
     * <p>
     * You can't delete a file system that is in use. That is, if the file system has any mount targets, you must first
     * delete them. For more information, see <a>DescribeMountTargets</a> and <a>DeleteMountTarget</a>.
     * </p>
     * <note>
     * <p>
     * The <code>DeleteFileSystem</code> call returns while the file system state is still <code>deleting</code>. You
     * can check the file system deletion status by calling the <a>DescribeFileSystems</a> operation, which returns a
     * list of file systems in your account. If you pass file system ID or creation token for the deleted file system,
     * the <a>DescribeFileSystems</a> returns a <code>404 FileSystemNotFound</code> error.
     * </p>
     * </note>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DeleteFileSystem</code> action.
     * </p>
     *
     * @param deleteFileSystemRequest
     * @return A Java Future containing the result of the DeleteFileSystem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> does not exist in the
     *         requester's AWS account.</li>
     *         <li>FileSystemInUseException Returned if a file system has mount targets.</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EFSException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EFSAsyncClient.DeleteFileSystem
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DeleteFileSystem"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteFileSystemResponse> deleteFileSystem(DeleteFileSystemRequest deleteFileSystemRequest) {

        HttpResponseHandler<DeleteFileSystemResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DeleteFileSystemResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DeleteFileSystemRequest, DeleteFileSystemResponse>()
                .withMarshaller(new DeleteFileSystemRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteFileSystemRequest));
    }

    /**
     * <p>
     * Deletes the specified mount target.
     * </p>
     * <p>
     * This operation forcibly breaks any mounts of the file system via the mount target that is being deleted, which
     * might disrupt instances or applications using those mounts. To avoid applications getting cut off abruptly, you
     * might consider unmounting any mounts of the mount target, if feasible. The operation also deletes the associated
     * network interface. Uncommitted writes may be lost, but breaking a mount target using this operation does not
     * corrupt the file system itself. The file system you created remains. You can mount an EC2 instance in your VPC
     * via another mount target.
     * </p>
     * <p>
     * This operation requires permissions for the following action on the file system:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>elasticfilesystem:DeleteMountTarget</code>
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The <code>DeleteMountTarget</code> call returns while the mount target state is still <code>deleting</code>. You
     * can check the mount target deletion by calling the <a>DescribeMountTargets</a> operation, which returns a list of
     * mount target descriptions for the given file system.
     * </p>
     * </note>
     * <p>
     * The operation also requires permissions for the following Amazon EC2 action on the mount target's network
     * interface:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ec2:DeleteNetworkInterface</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteMountTargetRequest
     * @return A Java Future containing the result of the DeleteMountTarget operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>DependencyTimeoutException The service timed out trying to fulfill the request, and the client should
     *         try the call again.</li>
     *         <li>MountTargetNotFoundException Returned if there is no mount target with the specified ID found in the
     *         caller's account.</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EFSException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EFSAsyncClient.DeleteMountTarget
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DeleteMountTarget"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMountTargetResponse> deleteMountTarget(DeleteMountTargetRequest deleteMountTargetRequest) {

        HttpResponseHandler<DeleteMountTargetResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DeleteMountTargetResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DeleteMountTargetRequest, DeleteMountTargetResponse>()
                .withMarshaller(new DeleteMountTargetRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteMountTargetRequest));
    }

    /**
     * <p>
     * Deletes the specified tags from a file system. If the <code>DeleteTags</code> request includes a tag key that
     * does not exist, Amazon EFS ignores it and doesn't cause an error. For more information about tags and related
     * restrictions, see <a href="http://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html">Tag
     * Restrictions</a> in the <i>AWS Billing and Cost Management User Guide</i>.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DeleteTags</code> action.
     * </p>
     *
     * @param deleteTagsRequest
     * @return A Java Future containing the result of the DeleteTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> does not exist in the
     *         requester's AWS account.</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EFSException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EFSAsyncClient.DeleteTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DeleteTags" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTagsResponse> deleteTags(DeleteTagsRequest deleteTagsRequest) {

        HttpResponseHandler<DeleteTagsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DeleteTagsResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DeleteTagsRequest, DeleteTagsResponse>()
                .withMarshaller(new DeleteTagsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteTagsRequest));
    }

    /**
     * <p>
     * Returns the description of a specific Amazon EFS file system if either the file system <code>CreationToken</code>
     * or the <code>FileSystemId</code> is provided. Otherwise, it returns descriptions of all file systems owned by the
     * caller's AWS account in the AWS Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all file system descriptions, you can optionally specify the <code>MaxItems</code> parameter to
     * limit the number of descriptions in a response. If more file system descriptions remain, Amazon EFS returns a
     * <code>NextMarker</code>, an opaque token, in the response. In this case, you should send a subsequent request
     * with the <code>Marker</code> request parameter set to the value of <code>NextMarker</code>.
     * </p>
     * <p>
     * To retrieve a list of your file system descriptions, this operation is used in an iterative process, where
     * <code>DescribeFileSystems</code> is called first without the <code>Marker</code> and then the operation continues
     * to call it with the <code>Marker</code> parameter set to the value of the <code>NextMarker</code> from the
     * previous response until the response has no <code>NextMarker</code>.
     * </p>
     * <p>
     * The implementation may return fewer than <code>MaxItems</code> file system descriptions while still including a
     * <code>NextMarker</code> value.
     * </p>
     * <p>
     * The order of file systems returned in the response of one <code>DescribeFileSystems</code> call and the order of
     * file systems returned across the responses of a multi-call iteration is unspecified.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeFileSystems</code> action.
     * </p>
     *
     * @param describeFileSystemsRequest
     * @return A Java Future containing the result of the DescribeFileSystems operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> does not exist in the
     *         requester's AWS account.</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EFSException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EFSAsyncClient.DescribeFileSystems
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeFileSystems"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFileSystemsResponse> describeFileSystems(
            DescribeFileSystemsRequest describeFileSystemsRequest) {

        HttpResponseHandler<DescribeFileSystemsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DescribeFileSystemsResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DescribeFileSystemsRequest, DescribeFileSystemsResponse>()
                .withMarshaller(new DescribeFileSystemsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeFileSystemsRequest));
    }

    /**
     * <p>
     * Returns the security groups currently in effect for a mount target. This operation requires that the network
     * interface of the mount target has been created and the lifecycle state of the mount target is not
     * <code>deleted</code>.
     * </p>
     * <p>
     * This operation requires permissions for the following actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>elasticfilesystem:DescribeMountTargetSecurityGroups</code> action on the mount target's file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:DescribeNetworkInterfaceAttribute</code> action on the mount target's network interface.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeMountTargetSecurityGroupsRequest
     * @return A Java Future containing the result of the DescribeMountTargetSecurityGroups operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>MountTargetNotFoundException Returned if there is no mount target with the specified ID found in the
     *         caller's account.</li>
     *         <li>IncorrectMountTargetStateException Returned if the mount target is not in the correct state for the
     *         operation.</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EFSException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EFSAsyncClient.DescribeMountTargetSecurityGroups
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeMountTargetSecurityGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeMountTargetSecurityGroupsResponse> describeMountTargetSecurityGroups(
            DescribeMountTargetSecurityGroupsRequest describeMountTargetSecurityGroupsRequest) {

        HttpResponseHandler<DescribeMountTargetSecurityGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DescribeMountTargetSecurityGroupsResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler
                .execute(new ClientExecutionParams<DescribeMountTargetSecurityGroupsRequest, DescribeMountTargetSecurityGroupsResponse>()
                        .withMarshaller(new DescribeMountTargetSecurityGroupsRequestMarshaller(protocolFactory))
                        .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                        .withInput(describeMountTargetSecurityGroupsRequest));
    }

    /**
     * <p>
     * Returns the descriptions of all the current mount targets, or a specific mount target, for a file system. When
     * requesting all of the current mount targets, the order of mount targets returned in the response is unspecified.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeMountTargets</code> action, on either
     * the file system ID that you specify in <code>FileSystemId</code>, or on the file system of the mount target that
     * you specify in <code>MountTargetId</code>.
     * </p>
     *
     * @param describeMountTargetsRequest
     * @return A Java Future containing the result of the DescribeMountTargets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> does not exist in the
     *         requester's AWS account.</li>
     *         <li>MountTargetNotFoundException Returned if there is no mount target with the specified ID found in the
     *         caller's account.</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EFSException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EFSAsyncClient.DescribeMountTargets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeMountTargets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeMountTargetsResponse> describeMountTargets(
            DescribeMountTargetsRequest describeMountTargetsRequest) {

        HttpResponseHandler<DescribeMountTargetsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DescribeMountTargetsResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DescribeMountTargetsRequest, DescribeMountTargetsResponse>()
                .withMarshaller(new DescribeMountTargetsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeMountTargetsRequest));
    }

    /**
     * <p>
     * Returns the tags associated with a file system. The order of tags returned in the response of one
     * <code>DescribeTags</code> call and the order of tags returned across the responses of a multi-call iteration
     * (when using pagination) is unspecified.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeTags</code> action.
     * </p>
     *
     * @param describeTagsRequest
     * @return A Java Future containing the result of the DescribeTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> does not exist in the
     *         requester's AWS account.</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EFSException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EFSAsyncClient.DescribeTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeTags" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeTagsResponse> describeTags(DescribeTagsRequest describeTagsRequest) {

        HttpResponseHandler<DescribeTagsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DescribeTagsResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DescribeTagsRequest, DescribeTagsResponse>()
                .withMarshaller(new DescribeTagsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeTagsRequest));
    }

    /**
     * <p>
     * Modifies the set of security groups in effect for a mount target.
     * </p>
     * <p>
     * When you create a mount target, Amazon EFS also creates a new network interface. For more information, see
     * <a>CreateMountTarget</a>. This operation replaces the security groups in effect for the network interface
     * associated with a mount target, with the <code>SecurityGroups</code> provided in the request. This operation
     * requires that the network interface of the mount target has been created and the lifecycle state of the mount
     * target is not <code>deleted</code>.
     * </p>
     * <p>
     * The operation requires permissions for the following actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>elasticfilesystem:ModifyMountTargetSecurityGroups</code> action on the mount target's file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:ModifyNetworkInterfaceAttribute</code> action on the mount target's network interface.
     * </p>
     * </li>
     * </ul>
     *
     * @param modifyMountTargetSecurityGroupsRequest
     * @return A Java Future containing the result of the ModifyMountTargetSecurityGroups operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>MountTargetNotFoundException Returned if there is no mount target with the specified ID found in the
     *         caller's account.</li>
     *         <li>IncorrectMountTargetStateException Returned if the mount target is not in the correct state for the
     *         operation.</li>
     *         <li>SecurityGroupLimitExceededException Returned if the size of <code>SecurityGroups</code> specified in
     *         the request is greater than five.</li>
     *         <li>SecurityGroupNotFoundException Returned if one of the specified security groups does not exist in the
     *         subnet's VPC.</li>
     *         <li>SdkBaseException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EFSException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EFSAsyncClient.ModifyMountTargetSecurityGroups
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/ModifyMountTargetSecurityGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ModifyMountTargetSecurityGroupsResponse> modifyMountTargetSecurityGroups(
            ModifyMountTargetSecurityGroupsRequest modifyMountTargetSecurityGroupsRequest) {

        HttpResponseHandler<ModifyMountTargetSecurityGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new ModifyMountTargetSecurityGroupsResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler
                .execute(new ClientExecutionParams<ModifyMountTargetSecurityGroupsRequest, ModifyMountTargetSecurityGroupsResponse>()
                        .withMarshaller(new ModifyMountTargetSecurityGroupsRequestMarshaller(protocolFactory))
                        .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                        .withInput(modifyMountTargetSecurityGroupsRequest));
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private software.amazon.awssdk.core.protocol.json.SdkJsonProtocolFactory init() {
        return new SdkJsonProtocolFactory(new JsonClientMetadata()
                .withProtocolVersion("1.1")
                .withSupportsCbor(false)
                .withSupportsIon(false)
                .withBaseServiceExceptionClass(software.amazon.awssdk.services.efs.model.EFSException.class)
                .withContentTypeOverride("")
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("MountTargetNotFound").withModeledClass(
                                MountTargetNotFoundException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("SubnetNotFound").withModeledClass(
                                SubnetNotFoundException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("IncorrectMountTargetState").withModeledClass(
                                IncorrectMountTargetStateException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("UnsupportedAvailabilityZone").withModeledClass(
                                UnsupportedAvailabilityZoneException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("IpAddressInUse").withModeledClass(
                                IpAddressInUseException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("MountTargetConflict").withModeledClass(
                                MountTargetConflictException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("SecurityGroupNotFound").withModeledClass(
                                SecurityGroupNotFoundException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("FileSystemInUse").withModeledClass(
                                FileSystemInUseException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("FileSystemAlreadyExists").withModeledClass(
                                FileSystemAlreadyExistsException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("SecurityGroupLimitExceeded").withModeledClass(
                                SecurityGroupLimitExceededException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("NetworkInterfaceLimitExceeded").withModeledClass(
                                NetworkInterfaceLimitExceededException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("IncorrectFileSystemLifeCycleState").withModeledClass(
                                IncorrectFileSystemLifeCycleStateException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("DependencyTimeout").withModeledClass(
                                DependencyTimeoutException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("NoFreeAddressesInSubnet").withModeledClass(
                                NoFreeAddressesInSubnetException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("FileSystemLimitExceeded").withModeledClass(
                                FileSystemLimitExceededException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("FileSystemNotFound").withModeledClass(
                                FileSystemNotFoundException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("BadRequest").withModeledClass(BadRequestException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InternalServerError").withModeledClass(
                                InternalServerErrorException.class)));
    }

    private HttpResponseHandler<AmazonServiceException> createErrorResponseHandler() {
        return protocolFactory.createErrorResponseHandler(new JsonErrorResponseMetadata());
    }
}
