/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.efs.model;

import java.time.Instant;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.efs.transform.FileSystemDescriptionMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Description of the file system.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class FileSystemDescription implements StructuredPojo,
        ToCopyableBuilder<FileSystemDescription.Builder, FileSystemDescription> {
    private final String ownerId;

    private final String creationToken;

    private final String fileSystemId;

    private final Instant creationTime;

    private final String lifeCycleState;

    private final String name;

    private final Integer numberOfMountTargets;

    private final FileSystemSize sizeInBytes;

    private final String performanceMode;

    private final Boolean encrypted;

    private final String kmsKeyId;

    private FileSystemDescription(BuilderImpl builder) {
        this.ownerId = builder.ownerId;
        this.creationToken = builder.creationToken;
        this.fileSystemId = builder.fileSystemId;
        this.creationTime = builder.creationTime;
        this.lifeCycleState = builder.lifeCycleState;
        this.name = builder.name;
        this.numberOfMountTargets = builder.numberOfMountTargets;
        this.sizeInBytes = builder.sizeInBytes;
        this.performanceMode = builder.performanceMode;
        this.encrypted = builder.encrypted;
        this.kmsKeyId = builder.kmsKeyId;
    }

    /**
     * <p>
     * AWS account that created the file system. If the file system was created by an IAM user, the parent account to
     * which the user belongs is the owner.
     * </p>
     * 
     * @return AWS account that created the file system. If the file system was created by an IAM user, the parent
     *         account to which the user belongs is the owner.
     */
    public String ownerId() {
        return ownerId;
    }

    /**
     * <p>
     * Opaque string specified in the request.
     * </p>
     * 
     * @return Opaque string specified in the request.
     */
    public String creationToken() {
        return creationToken;
    }

    /**
     * <p>
     * ID of the file system, assigned by Amazon EFS.
     * </p>
     * 
     * @return ID of the file system, assigned by Amazon EFS.
     */
    public String fileSystemId() {
        return fileSystemId;
    }

    /**
     * <p>
     * Time that the file system was created, in seconds (since 1970-01-01T00:00:00Z).
     * </p>
     * 
     * @return Time that the file system was created, in seconds (since 1970-01-01T00:00:00Z).
     */
    public Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * Lifecycle phase of the file system.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lifeCycleState}
     * will return {@link LifeCycleState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #lifeCycleStateString}.
     * </p>
     * 
     * @return Lifecycle phase of the file system.
     * @see LifeCycleState
     */
    public LifeCycleState lifeCycleState() {
        return LifeCycleState.fromValue(lifeCycleState);
    }

    /**
     * <p>
     * Lifecycle phase of the file system.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lifeCycleState}
     * will return {@link LifeCycleState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #lifeCycleStateString}.
     * </p>
     * 
     * @return Lifecycle phase of the file system.
     * @see LifeCycleState
     */
    public String lifeCycleStateString() {
        return lifeCycleState;
    }

    /**
     * <p>
     * You can add tags to a file system, including a <code>Name</code> tag. For more information, see
     * <a>CreateTags</a>. If the file system has a <code>Name</code> tag, Amazon EFS returns the value in this field.
     * </p>
     * 
     * @return You can add tags to a file system, including a <code>Name</code> tag. For more information, see
     *         <a>CreateTags</a>. If the file system has a <code>Name</code> tag, Amazon EFS returns the value in this
     *         field.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * Current number of mount targets that the file system has. For more information, see <a>CreateMountTarget</a>.
     * </p>
     * 
     * @return Current number of mount targets that the file system has. For more information, see
     *         <a>CreateMountTarget</a>.
     */
    public Integer numberOfMountTargets() {
        return numberOfMountTargets;
    }

    /**
     * <p>
     * Latest known metered size (in bytes) of data stored in the file system, in bytes, in its <code>Value</code>
     * field, and the time at which that size was determined in its <code>Timestamp</code> field. The
     * <code>Timestamp</code> value is the integer number of seconds since 1970-01-01T00:00:00Z. Note that the value
     * does not represent the size of a consistent snapshot of the file system, but it is eventually consistent when
     * there are no writes to the file system. That is, the value will represent actual size only if the file system is
     * not modified for a period longer than a couple of hours. Otherwise, the value is not the exact size the file
     * system was at any instant in time.
     * </p>
     * 
     * @return Latest known metered size (in bytes) of data stored in the file system, in bytes, in its
     *         <code>Value</code> field, and the time at which that size was determined in its <code>Timestamp</code>
     *         field. The <code>Timestamp</code> value is the integer number of seconds since 1970-01-01T00:00:00Z. Note
     *         that the value does not represent the size of a consistent snapshot of the file system, but it is
     *         eventually consistent when there are no writes to the file system. That is, the value will represent
     *         actual size only if the file system is not modified for a period longer than a couple of hours.
     *         Otherwise, the value is not the exact size the file system was at any instant in time.
     */
    public FileSystemSize sizeInBytes() {
        return sizeInBytes;
    }

    /**
     * <p>
     * The <code>PerformanceMode</code> of the file system.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #performanceMode}
     * will return {@link PerformanceMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #performanceModeString}.
     * </p>
     * 
     * @return The <code>PerformanceMode</code> of the file system.
     * @see PerformanceMode
     */
    public PerformanceMode performanceMode() {
        return PerformanceMode.fromValue(performanceMode);
    }

    /**
     * <p>
     * The <code>PerformanceMode</code> of the file system.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #performanceMode}
     * will return {@link PerformanceMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #performanceModeString}.
     * </p>
     * 
     * @return The <code>PerformanceMode</code> of the file system.
     * @see PerformanceMode
     */
    public String performanceModeString() {
        return performanceMode;
    }

    /**
     * <p>
     * A boolean value that, if true, indicates that the file system is encrypted.
     * </p>
     * 
     * @return A boolean value that, if true, indicates that the file system is encrypted.
     */
    public Boolean encrypted() {
        return encrypted;
    }

    /**
     * <p>
     * The id of an AWS Key Management Service (AWS KMS) customer master key (CMK) that was used to protect the
     * encrypted file system.
     * </p>
     * 
     * @return The id of an AWS Key Management Service (AWS KMS) customer master key (CMK) that was used to protect the
     *         encrypted file system.
     */
    public String kmsKeyId() {
        return kmsKeyId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ownerId());
        hashCode = 31 * hashCode + Objects.hashCode(creationToken());
        hashCode = 31 * hashCode + Objects.hashCode(fileSystemId());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(lifeCycleStateString());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(numberOfMountTargets());
        hashCode = 31 * hashCode + Objects.hashCode(sizeInBytes());
        hashCode = 31 * hashCode + Objects.hashCode(performanceModeString());
        hashCode = 31 * hashCode + Objects.hashCode(encrypted());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FileSystemDescription)) {
            return false;
        }
        FileSystemDescription other = (FileSystemDescription) obj;
        return Objects.equals(ownerId(), other.ownerId()) && Objects.equals(creationToken(), other.creationToken())
                && Objects.equals(fileSystemId(), other.fileSystemId()) && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(lifeCycleStateString(), other.lifeCycleStateString()) && Objects.equals(name(), other.name())
                && Objects.equals(numberOfMountTargets(), other.numberOfMountTargets())
                && Objects.equals(sizeInBytes(), other.sizeInBytes())
                && Objects.equals(performanceModeString(), other.performanceModeString())
                && Objects.equals(encrypted(), other.encrypted()) && Objects.equals(kmsKeyId(), other.kmsKeyId());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (ownerId() != null) {
            sb.append("OwnerId: ").append(ownerId()).append(",");
        }
        if (creationToken() != null) {
            sb.append("CreationToken: ").append(creationToken()).append(",");
        }
        if (fileSystemId() != null) {
            sb.append("FileSystemId: ").append(fileSystemId()).append(",");
        }
        if (creationTime() != null) {
            sb.append("CreationTime: ").append(creationTime()).append(",");
        }
        if (lifeCycleStateString() != null) {
            sb.append("LifeCycleState: ").append(lifeCycleStateString()).append(",");
        }
        if (name() != null) {
            sb.append("Name: ").append(name()).append(",");
        }
        if (numberOfMountTargets() != null) {
            sb.append("NumberOfMountTargets: ").append(numberOfMountTargets()).append(",");
        }
        if (sizeInBytes() != null) {
            sb.append("SizeInBytes: ").append(sizeInBytes()).append(",");
        }
        if (performanceModeString() != null) {
            sb.append("PerformanceMode: ").append(performanceModeString()).append(",");
        }
        if (encrypted() != null) {
            sb.append("Encrypted: ").append(encrypted()).append(",");
        }
        if (kmsKeyId() != null) {
            sb.append("KmsKeyId: ").append(kmsKeyId()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "OwnerId":
            return Optional.of(clazz.cast(ownerId()));
        case "CreationToken":
            return Optional.of(clazz.cast(creationToken()));
        case "FileSystemId":
            return Optional.of(clazz.cast(fileSystemId()));
        case "CreationTime":
            return Optional.of(clazz.cast(creationTime()));
        case "LifeCycleState":
            return Optional.of(clazz.cast(lifeCycleStateString()));
        case "Name":
            return Optional.of(clazz.cast(name()));
        case "NumberOfMountTargets":
            return Optional.of(clazz.cast(numberOfMountTargets()));
        case "SizeInBytes":
            return Optional.of(clazz.cast(sizeInBytes()));
        case "PerformanceMode":
            return Optional.of(clazz.cast(performanceModeString()));
        case "Encrypted":
            return Optional.of(clazz.cast(encrypted()));
        case "KmsKeyId":
            return Optional.of(clazz.cast(kmsKeyId()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        FileSystemDescriptionMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, FileSystemDescription> {
        /**
         * <p>
         * AWS account that created the file system. If the file system was created by an IAM user, the parent account
         * to which the user belongs is the owner.
         * </p>
         * 
         * @param ownerId
         *        AWS account that created the file system. If the file system was created by an IAM user, the parent
         *        account to which the user belongs is the owner.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerId(String ownerId);

        /**
         * <p>
         * Opaque string specified in the request.
         * </p>
         * 
         * @param creationToken
         *        Opaque string specified in the request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationToken(String creationToken);

        /**
         * <p>
         * ID of the file system, assigned by Amazon EFS.
         * </p>
         * 
         * @param fileSystemId
         *        ID of the file system, assigned by Amazon EFS.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileSystemId(String fileSystemId);

        /**
         * <p>
         * Time that the file system was created, in seconds (since 1970-01-01T00:00:00Z).
         * </p>
         * 
         * @param creationTime
         *        Time that the file system was created, in seconds (since 1970-01-01T00:00:00Z).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * Lifecycle phase of the file system.
         * </p>
         * 
         * @param lifeCycleState
         *        Lifecycle phase of the file system.
         * @see LifeCycleState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LifeCycleState
         */
        Builder lifeCycleState(String lifeCycleState);

        /**
         * <p>
         * Lifecycle phase of the file system.
         * </p>
         * 
         * @param lifeCycleState
         *        Lifecycle phase of the file system.
         * @see LifeCycleState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LifeCycleState
         */
        Builder lifeCycleState(LifeCycleState lifeCycleState);

        /**
         * <p>
         * You can add tags to a file system, including a <code>Name</code> tag. For more information, see
         * <a>CreateTags</a>. If the file system has a <code>Name</code> tag, Amazon EFS returns the value in this
         * field.
         * </p>
         * 
         * @param name
         *        You can add tags to a file system, including a <code>Name</code> tag. For more information, see
         *        <a>CreateTags</a>. If the file system has a <code>Name</code> tag, Amazon EFS returns the value in
         *        this field.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Current number of mount targets that the file system has. For more information, see <a>CreateMountTarget</a>.
         * </p>
         * 
         * @param numberOfMountTargets
         *        Current number of mount targets that the file system has. For more information, see
         *        <a>CreateMountTarget</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numberOfMountTargets(Integer numberOfMountTargets);

        /**
         * <p>
         * Latest known metered size (in bytes) of data stored in the file system, in bytes, in its <code>Value</code>
         * field, and the time at which that size was determined in its <code>Timestamp</code> field. The
         * <code>Timestamp</code> value is the integer number of seconds since 1970-01-01T00:00:00Z. Note that the value
         * does not represent the size of a consistent snapshot of the file system, but it is eventually consistent when
         * there are no writes to the file system. That is, the value will represent actual size only if the file system
         * is not modified for a period longer than a couple of hours. Otherwise, the value is not the exact size the
         * file system was at any instant in time.
         * </p>
         * 
         * @param sizeInBytes
         *        Latest known metered size (in bytes) of data stored in the file system, in bytes, in its
         *        <code>Value</code> field, and the time at which that size was determined in its <code>Timestamp</code>
         *        field. The <code>Timestamp</code> value is the integer number of seconds since 1970-01-01T00:00:00Z.
         *        Note that the value does not represent the size of a consistent snapshot of the file system, but it is
         *        eventually consistent when there are no writes to the file system. That is, the value will represent
         *        actual size only if the file system is not modified for a period longer than a couple of hours.
         *        Otherwise, the value is not the exact size the file system was at any instant in time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sizeInBytes(FileSystemSize sizeInBytes);

        /**
         * <p>
         * Latest known metered size (in bytes) of data stored in the file system, in bytes, in its <code>Value</code>
         * field, and the time at which that size was determined in its <code>Timestamp</code> field. The
         * <code>Timestamp</code> value is the integer number of seconds since 1970-01-01T00:00:00Z. Note that the value
         * does not represent the size of a consistent snapshot of the file system, but it is eventually consistent when
         * there are no writes to the file system. That is, the value will represent actual size only if the file system
         * is not modified for a period longer than a couple of hours. Otherwise, the value is not the exact size the
         * file system was at any instant in time.
         * </p>
         * This is a convenience that creates an instance of the {@link FileSystemSize.Builder} avoiding the need to
         * create one manually via {@link FileSystemSize#builder()}.
         *
         * When the {@link Consumer} completes, {@link FileSystemSize.Builder#build()} is called immediately and its
         * result is passed to {@link #sizeInBytes(FileSystemSize)}.
         * 
         * @param sizeInBytes
         *        a consumer that will call methods on {@link FileSystemSize.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sizeInBytes(FileSystemSize)
         */
        default Builder sizeInBytes(Consumer<FileSystemSize.Builder> sizeInBytes) {
            return sizeInBytes(FileSystemSize.builder().apply(sizeInBytes).build());
        }

        /**
         * <p>
         * The <code>PerformanceMode</code> of the file system.
         * </p>
         * 
         * @param performanceMode
         *        The <code>PerformanceMode</code> of the file system.
         * @see PerformanceMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PerformanceMode
         */
        Builder performanceMode(String performanceMode);

        /**
         * <p>
         * The <code>PerformanceMode</code> of the file system.
         * </p>
         * 
         * @param performanceMode
         *        The <code>PerformanceMode</code> of the file system.
         * @see PerformanceMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PerformanceMode
         */
        Builder performanceMode(PerformanceMode performanceMode);

        /**
         * <p>
         * A boolean value that, if true, indicates that the file system is encrypted.
         * </p>
         * 
         * @param encrypted
         *        A boolean value that, if true, indicates that the file system is encrypted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encrypted(Boolean encrypted);

        /**
         * <p>
         * The id of an AWS Key Management Service (AWS KMS) customer master key (CMK) that was used to protect the
         * encrypted file system.
         * </p>
         * 
         * @param kmsKeyId
         *        The id of an AWS Key Management Service (AWS KMS) customer master key (CMK) that was used to protect
         *        the encrypted file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);
    }

    static final class BuilderImpl implements Builder {
        private String ownerId;

        private String creationToken;

        private String fileSystemId;

        private Instant creationTime;

        private String lifeCycleState;

        private String name;

        private Integer numberOfMountTargets;

        private FileSystemSize sizeInBytes;

        private String performanceMode;

        private Boolean encrypted;

        private String kmsKeyId;

        private BuilderImpl() {
        }

        private BuilderImpl(FileSystemDescription model) {
            ownerId(model.ownerId);
            creationToken(model.creationToken);
            fileSystemId(model.fileSystemId);
            creationTime(model.creationTime);
            lifeCycleState(model.lifeCycleState);
            name(model.name);
            numberOfMountTargets(model.numberOfMountTargets);
            sizeInBytes(model.sizeInBytes);
            performanceMode(model.performanceMode);
            encrypted(model.encrypted);
            kmsKeyId(model.kmsKeyId);
        }

        public final String getOwnerId() {
            return ownerId;
        }

        @Override
        public final Builder ownerId(String ownerId) {
            this.ownerId = ownerId;
            return this;
        }

        public final void setOwnerId(String ownerId) {
            this.ownerId = ownerId;
        }

        public final String getCreationToken() {
            return creationToken;
        }

        @Override
        public final Builder creationToken(String creationToken) {
            this.creationToken = creationToken;
            return this;
        }

        public final void setCreationToken(String creationToken) {
            this.creationToken = creationToken;
        }

        public final String getFileSystemId() {
            return fileSystemId;
        }

        @Override
        public final Builder fileSystemId(String fileSystemId) {
            this.fileSystemId = fileSystemId;
            return this;
        }

        public final void setFileSystemId(String fileSystemId) {
            this.fileSystemId = fileSystemId;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        public final String getLifeCycleState() {
            return lifeCycleState;
        }

        @Override
        public final Builder lifeCycleState(String lifeCycleState) {
            this.lifeCycleState = lifeCycleState;
            return this;
        }

        @Override
        public final Builder lifeCycleState(LifeCycleState lifeCycleState) {
            this.lifeCycleState(lifeCycleState.toString());
            return this;
        }

        public final void setLifeCycleState(String lifeCycleState) {
            this.lifeCycleState = lifeCycleState;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final Integer getNumberOfMountTargets() {
            return numberOfMountTargets;
        }

        @Override
        public final Builder numberOfMountTargets(Integer numberOfMountTargets) {
            this.numberOfMountTargets = numberOfMountTargets;
            return this;
        }

        public final void setNumberOfMountTargets(Integer numberOfMountTargets) {
            this.numberOfMountTargets = numberOfMountTargets;
        }

        public final FileSystemSize.Builder getSizeInBytes() {
            return sizeInBytes != null ? sizeInBytes.toBuilder() : null;
        }

        @Override
        public final Builder sizeInBytes(FileSystemSize sizeInBytes) {
            this.sizeInBytes = sizeInBytes;
            return this;
        }

        public final void setSizeInBytes(FileSystemSize.BuilderImpl sizeInBytes) {
            this.sizeInBytes = sizeInBytes != null ? sizeInBytes.build() : null;
        }

        public final String getPerformanceMode() {
            return performanceMode;
        }

        @Override
        public final Builder performanceMode(String performanceMode) {
            this.performanceMode = performanceMode;
            return this;
        }

        @Override
        public final Builder performanceMode(PerformanceMode performanceMode) {
            this.performanceMode(performanceMode.toString());
            return this;
        }

        public final void setPerformanceMode(String performanceMode) {
            this.performanceMode = performanceMode;
        }

        public final Boolean getEncrypted() {
            return encrypted;
        }

        @Override
        public final Builder encrypted(Boolean encrypted) {
            this.encrypted = encrypted;
            return this;
        }

        public final void setEncrypted(Boolean encrypted) {
            this.encrypted = encrypted;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        public FileSystemDescription build() {
            return new FileSystemDescription(this);
        }
    }
}
