/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.efs.model;

import java.time.Instant;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.efs.transform.FileSystemSizeMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Latest known metered size (in bytes) of data stored in the file system, in its <code>Value</code> field, and the time
 * at which that size was determined in its <code>Timestamp</code> field. Note that the value does not represent the
 * size of a consistent snapshot of the file system, but it is eventually consistent when there are no writes to the
 * file system. That is, the value will represent the actual size only if the file system is not modified for a period
 * longer than a couple of hours. Otherwise, the value is not necessarily the exact size the file system was at any
 * instant in time.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class FileSystemSize implements StructuredPojo, ToCopyableBuilder<FileSystemSize.Builder, FileSystemSize> {
    private final Long value;

    private final Instant timestamp;

    private FileSystemSize(BuilderImpl builder) {
        this.value = builder.value;
        this.timestamp = builder.timestamp;
    }

    /**
     * <p>
     * Latest known metered size (in bytes) of data stored in the file system.
     * </p>
     * 
     * @return Latest known metered size (in bytes) of data stored in the file system.
     */
    public Long value() {
        return value;
    }

    /**
     * <p>
     * Time at which the size of data, returned in the <code>Value</code> field, was determined. The value is the
     * integer number of seconds since 1970-01-01T00:00:00Z.
     * </p>
     * 
     * @return Time at which the size of data, returned in the <code>Value</code> field, was determined. The value is
     *         the integer number of seconds since 1970-01-01T00:00:00Z.
     */
    public Instant timestamp() {
        return timestamp;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(value());
        hashCode = 31 * hashCode + Objects.hashCode(timestamp());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FileSystemSize)) {
            return false;
        }
        FileSystemSize other = (FileSystemSize) obj;
        return Objects.equals(value(), other.value()) && Objects.equals(timestamp(), other.timestamp());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (value() != null) {
            sb.append("Value: ").append(value()).append(",");
        }
        if (timestamp() != null) {
            sb.append("Timestamp: ").append(timestamp()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Value":
            return Optional.of(clazz.cast(value()));
        case "Timestamp":
            return Optional.of(clazz.cast(timestamp()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        FileSystemSizeMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, FileSystemSize> {
        /**
         * <p>
         * Latest known metered size (in bytes) of data stored in the file system.
         * </p>
         * 
         * @param value
         *        Latest known metered size (in bytes) of data stored in the file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder value(Long value);

        /**
         * <p>
         * Time at which the size of data, returned in the <code>Value</code> field, was determined. The value is the
         * integer number of seconds since 1970-01-01T00:00:00Z.
         * </p>
         * 
         * @param timestamp
         *        Time at which the size of data, returned in the <code>Value</code> field, was determined. The value is
         *        the integer number of seconds since 1970-01-01T00:00:00Z.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamp(Instant timestamp);
    }

    static final class BuilderImpl implements Builder {
        private Long value;

        private Instant timestamp;

        private BuilderImpl() {
        }

        private BuilderImpl(FileSystemSize model) {
            value(model.value);
            timestamp(model.timestamp);
        }

        public final Long getValue() {
            return value;
        }

        @Override
        public final Builder value(Long value) {
            this.value = value;
            return this;
        }

        public final void setValue(Long value) {
            this.value = value;
        }

        public final Instant getTimestamp() {
            return timestamp;
        }

        @Override
        public final Builder timestamp(Instant timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public final void setTimestamp(Instant timestamp) {
            this.timestamp = timestamp;
        }

        @Override
        public FileSystemSize build() {
            return new FileSystemSize(this);
        }
    }
}
