/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.efs;

import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.efs.model.BadRequestException;
import software.amazon.awssdk.services.efs.model.CreateFileSystemRequest;
import software.amazon.awssdk.services.efs.model.CreateFileSystemResponse;
import software.amazon.awssdk.services.efs.model.CreateMountTargetRequest;
import software.amazon.awssdk.services.efs.model.CreateMountTargetResponse;
import software.amazon.awssdk.services.efs.model.CreateTagsRequest;
import software.amazon.awssdk.services.efs.model.CreateTagsResponse;
import software.amazon.awssdk.services.efs.model.DeleteFileSystemRequest;
import software.amazon.awssdk.services.efs.model.DeleteFileSystemResponse;
import software.amazon.awssdk.services.efs.model.DeleteMountTargetRequest;
import software.amazon.awssdk.services.efs.model.DeleteMountTargetResponse;
import software.amazon.awssdk.services.efs.model.DeleteTagsRequest;
import software.amazon.awssdk.services.efs.model.DeleteTagsResponse;
import software.amazon.awssdk.services.efs.model.DependencyTimeoutException;
import software.amazon.awssdk.services.efs.model.DescribeFileSystemsRequest;
import software.amazon.awssdk.services.efs.model.DescribeFileSystemsResponse;
import software.amazon.awssdk.services.efs.model.DescribeLifecycleConfigurationRequest;
import software.amazon.awssdk.services.efs.model.DescribeLifecycleConfigurationResponse;
import software.amazon.awssdk.services.efs.model.DescribeMountTargetSecurityGroupsRequest;
import software.amazon.awssdk.services.efs.model.DescribeMountTargetSecurityGroupsResponse;
import software.amazon.awssdk.services.efs.model.DescribeMountTargetsRequest;
import software.amazon.awssdk.services.efs.model.DescribeMountTargetsResponse;
import software.amazon.awssdk.services.efs.model.DescribeTagsRequest;
import software.amazon.awssdk.services.efs.model.DescribeTagsResponse;
import software.amazon.awssdk.services.efs.model.EfsException;
import software.amazon.awssdk.services.efs.model.FileSystemAlreadyExistsException;
import software.amazon.awssdk.services.efs.model.FileSystemInUseException;
import software.amazon.awssdk.services.efs.model.FileSystemLimitExceededException;
import software.amazon.awssdk.services.efs.model.FileSystemNotFoundException;
import software.amazon.awssdk.services.efs.model.IncorrectFileSystemLifeCycleStateException;
import software.amazon.awssdk.services.efs.model.IncorrectMountTargetStateException;
import software.amazon.awssdk.services.efs.model.InsufficientThroughputCapacityException;
import software.amazon.awssdk.services.efs.model.InternalServerErrorException;
import software.amazon.awssdk.services.efs.model.IpAddressInUseException;
import software.amazon.awssdk.services.efs.model.ModifyMountTargetSecurityGroupsRequest;
import software.amazon.awssdk.services.efs.model.ModifyMountTargetSecurityGroupsResponse;
import software.amazon.awssdk.services.efs.model.MountTargetConflictException;
import software.amazon.awssdk.services.efs.model.MountTargetNotFoundException;
import software.amazon.awssdk.services.efs.model.NetworkInterfaceLimitExceededException;
import software.amazon.awssdk.services.efs.model.NoFreeAddressesInSubnetException;
import software.amazon.awssdk.services.efs.model.PutLifecycleConfigurationRequest;
import software.amazon.awssdk.services.efs.model.PutLifecycleConfigurationResponse;
import software.amazon.awssdk.services.efs.model.SecurityGroupLimitExceededException;
import software.amazon.awssdk.services.efs.model.SecurityGroupNotFoundException;
import software.amazon.awssdk.services.efs.model.SubnetNotFoundException;
import software.amazon.awssdk.services.efs.model.ThroughputLimitExceededException;
import software.amazon.awssdk.services.efs.model.TooManyRequestsException;
import software.amazon.awssdk.services.efs.model.UnsupportedAvailabilityZoneException;
import software.amazon.awssdk.services.efs.model.UpdateFileSystemRequest;
import software.amazon.awssdk.services.efs.model.UpdateFileSystemResponse;
import software.amazon.awssdk.services.efs.transform.CreateFileSystemRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.CreateMountTargetRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.CreateTagsRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DeleteFileSystemRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DeleteMountTargetRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DeleteTagsRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DescribeFileSystemsRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DescribeLifecycleConfigurationRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DescribeMountTargetSecurityGroupsRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DescribeMountTargetsRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DescribeTagsRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.ModifyMountTargetSecurityGroupsRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.PutLifecycleConfigurationRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.UpdateFileSystemRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link EfsAsyncClient}.
 *
 * @see EfsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultEfsAsyncClient implements EfsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultEfsAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultEfsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a new, empty file system. The operation requires a creation token in the request that Amazon EFS uses to
     * ensure idempotent creation (calling the operation with same creation token has no effect). If a file system does
     * not currently exist that is owned by the caller's AWS account with the specified creation token, this operation
     * does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a new, empty file system. The file system will have an Amazon EFS assigned ID, and an initial lifecycle
     * state <code>creating</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Returns with the description of the created file system.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Otherwise, this operation returns a <code>FileSystemAlreadyExists</code> error with the ID of the existing file
     * system.
     * </p>
     * <note>
     * <p>
     * For basic use cases, you can use a randomly generated UUID for the creation token.
     * </p>
     * </note>
     * <p>
     * The idempotent operation allows you to retry a <code>CreateFileSystem</code> call without risk of creating an
     * extra file system. This can happen when an initial call fails in a way that leaves it uncertain whether or not a
     * file system was actually created. An example might be that a transport level timeout occurred or your connection
     * was reset. As long as you use the same creation token, if the initial call had succeeded in creating a file
     * system, the client can learn of its existence from the <code>FileSystemAlreadyExists</code> error.
     * </p>
     * <note>
     * <p>
     * The <code>CreateFileSystem</code> call returns while the file system's lifecycle state is still
     * <code>creating</code>. You can check the file system creation status by calling the <a>DescribeFileSystems</a>
     * operation, which among other things returns the file system state.
     * </p>
     * </note>
     * <p>
     * This operation also takes an optional <code>PerformanceMode</code> parameter that you choose for your file
     * system. We recommend <code>generalPurpose</code> performance mode for most file systems. File systems using the
     * <code>maxIO</code> performance mode can scale to higher levels of aggregate throughput and operations per second
     * with a tradeoff of slightly higher latencies for most file operations. The performance mode can't be changed
     * after the file system has been created. For more information, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/performance.html#performancemodes.html">Amazon EFS: Performance
     * Modes</a>.
     * </p>
     * <p>
     * After the file system is fully created, Amazon EFS sets its lifecycle state to <code>available</code>, at which
     * point you can create one or more mount targets for the file system in your VPC. For more information, see
     * <a>CreateMountTarget</a>. You mount your Amazon EFS file system on an EC2 instances in your VPC by using the
     * mount target. For more information, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/how-it-works.html">Amazon EFS: How it Works</a>.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:CreateFileSystem</code> action.
     * </p>
     *
     * @param createFileSystemRequest
     * @return A Java Future containing the result of the CreateFileSystem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemAlreadyExistsException Returned if the file system you are trying to create already exists,
     *         with the creation token you provided.</li>
     *         <li>FileSystemLimitExceededException Returned if the AWS account has already created the maximum number
     *         of file systems allowed per account.</li>
     *         <li>InsufficientThroughputCapacityException Returned if there's not enough capacity to provision
     *         additional throughput. This value might be returned when you try to create a file system in provisioned
     *         throughput mode, when you attempt to increase the provisioned throughput of an existing file system, or
     *         when you attempt to change an existing file system from bursting to provisioned throughput mode.</li>
     *         <li>ThroughputLimitExceededException Returned if the throughput mode or amount of provisioned throughput
     *         can't be changed because the throughput limit of 1024 MiB/s has been reached.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.CreateFileSystem
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/CreateFileSystem"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateFileSystemResponse> createFileSystem(CreateFileSystemRequest createFileSystemRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateFileSystemResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateFileSystemResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateFileSystemResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateFileSystemRequest, CreateFileSystemResponse>()
                            .withOperationName("CreateFileSystem")
                            .withMarshaller(new CreateFileSystemRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createFileSystemRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a mount target for a file system. You can then mount the file system on EC2 instances by using the mount
     * target.
     * </p>
     * <p>
     * You can create one mount target in each Availability Zone in your VPC. All EC2 instances in a VPC within a given
     * Availability Zone share a single mount target for a given file system. If you have multiple subnets in an
     * Availability Zone, you create a mount target in one of the subnets. EC2 instances do not need to be in the same
     * subnet as the mount target in order to access their file system. For more information, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/how-it-works.html">Amazon EFS: How it Works</a>.
     * </p>
     * <p>
     * In the request, you also specify a file system ID for which you are creating the mount target and the file
     * system's lifecycle state must be <code>available</code>. For more information, see <a>DescribeFileSystems</a>.
     * </p>
     * <p>
     * In the request, you also provide a subnet ID, which determines the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * VPC in which Amazon EFS creates the mount target
     * </p>
     * </li>
     * <li>
     * <p>
     * Availability Zone in which Amazon EFS creates the mount target
     * </p>
     * </li>
     * <li>
     * <p>
     * IP address range from which Amazon EFS selects the IP address of the mount target (if you don't specify an IP
     * address in the request)
     * </p>
     * </li>
     * </ul>
     * <p>
     * After creating the mount target, Amazon EFS returns a response that includes, a <code>MountTargetId</code> and an
     * <code>IpAddress</code>. You use this IP address when mounting the file system in an EC2 instance. You can also
     * use the mount target's DNS name when mounting the file system. The EC2 instance on which you mount the file
     * system by using the mount target can resolve the mount target's DNS name to its IP address. For more information,
     * see <a href="https://docs.aws.amazon.com/efs/latest/ug/how-it-works.html#how-it-works-implementation">How it
     * Works: Implementation Overview</a>.
     * </p>
     * <p>
     * Note that you can create mount targets for a file system in only one VPC, and there can be only one mount target
     * per Availability Zone. That is, if the file system already has one or more mount targets created for it, the
     * subnet specified in the request to add another mount target must meet the following requirements:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Must belong to the same VPC as the subnets of the existing mount targets
     * </p>
     * </li>
     * <li>
     * <p>
     * Must not be in the same Availability Zone as any of the subnets of the existing mount targets
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the request satisfies the requirements, Amazon EFS does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a new mount target in the specified subnet.
     * </p>
     * </li>
     * <li>
     * <p>
     * Also creates a new network interface in the subnet as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the request provides an <code>IpAddress</code>, Amazon EFS assigns that IP address to the network interface.
     * Otherwise, Amazon EFS assigns a free address in the subnet (in the same way that the Amazon EC2
     * <code>CreateNetworkInterface</code> call does when a request does not specify a primary private IP address).
     * </p>
     * </li>
     * <li>
     * <p>
     * If the request provides <code>SecurityGroups</code>, this network interface is associated with those security
     * groups. Otherwise, it belongs to the default security group for the subnet's VPC.
     * </p>
     * </li>
     * <li>
     * <p>
     * Assigns the description <code>Mount target <i>fsmt-id</i> for file system <i>fs-id</i> </code> where
     * <code> <i>fsmt-id</i> </code> is the mount target ID, and <code> <i>fs-id</i> </code> is the
     * <code>FileSystemId</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Sets the <code>requesterManaged</code> property of the network interface to <code>true</code>, and the
     * <code>requesterId</code> value to <code>EFS</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Each Amazon EFS mount target has one corresponding requester-managed EC2 network interface. After the network
     * interface is created, Amazon EFS sets the <code>NetworkInterfaceId</code> field in the mount target's description
     * to the network interface ID, and the <code>IpAddress</code> field to its address. If network interface creation
     * fails, the entire <code>CreateMountTarget</code> operation fails.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The <code>CreateMountTarget</code> call returns only after creating the network interface, but while the mount
     * target state is still <code>creating</code>, you can check the mount target creation status by calling the
     * <a>DescribeMountTargets</a> operation, which among other things returns the mount target state.
     * </p>
     * </note>
     * <p>
     * We recommend that you create a mount target in each of the Availability Zones. There are cost considerations for
     * using a file system in an Availability Zone through a mount target created in another Availability Zone. For more
     * information, see <a href="http://aws.amazon.com/efs/">Amazon EFS</a>. In addition, by always using a mount target
     * local to the instance's Availability Zone, you eliminate a partial failure scenario. If the Availability Zone in
     * which your mount target is created goes down, then you can't access your file system through that mount target.
     * </p>
     * <p>
     * This operation requires permissions for the following action on the file system:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>elasticfilesystem:CreateMountTarget</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * This operation also requires permissions for the following Amazon EC2 actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ec2:DescribeSubnets</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:DescribeNetworkInterfaces</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:CreateNetworkInterface</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param createMountTargetRequest
     * @return A Java Future containing the result of the CreateMountTarget operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's AWS account.</li>
     *         <li>IncorrectFileSystemLifeCycleStateException Returned if the file system's lifecycle state is not
     *         "available".</li>
     *         <li>MountTargetConflictException Returned if the mount target would violate one of the specified
     *         restrictions based on the file system's existing mount targets.</li>
     *         <li>SubnetNotFoundException Returned if there is no subnet with ID <code>SubnetId</code> provided in the
     *         request.</li>
     *         <li>NoFreeAddressesInSubnetException Returned if <code>IpAddress</code> was not specified in the request
     *         and there are no free IP addresses in the subnet.</li>
     *         <li>IpAddressInUseException Returned if the request specified an <code>IpAddress</code> that is already
     *         in use in the subnet.</li>
     *         <li>NetworkInterfaceLimitExceededException The calling account has reached the limit for elastic network
     *         interfaces for the specific AWS Region. The client should try to delete some elastic network interfaces
     *         or get the account limit raised. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/VPC_Appendix_Limits.html">Amazon VPC
     *         Limits</a> in the <i>Amazon VPC User Guide </i> (see the Network interfaces per VPC entry in the table).</li>
     *         <li>SecurityGroupLimitExceededException Returned if the size of <code>SecurityGroups</code> specified in
     *         the request is greater than five.</li>
     *         <li>SecurityGroupNotFoundException Returned if one of the specified security groups doesn't exist in the
     *         subnet's VPC.</li>
     *         <li>UnsupportedAvailabilityZoneException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.CreateMountTarget
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/CreateMountTarget"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMountTargetResponse> createMountTarget(CreateMountTargetRequest createMountTargetRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMountTargetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateMountTargetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateMountTargetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMountTargetRequest, CreateMountTargetResponse>()
                            .withOperationName("CreateMountTarget")
                            .withMarshaller(new CreateMountTargetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createMountTargetRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or overwrites tags associated with a file system. Each tag is a key-value pair. If a tag key specified in
     * the request already exists on the file system, this operation overwrites its value with the value provided in the
     * request. If you add the <code>Name</code> tag to your file system, Amazon EFS returns it in the response to the
     * <a>DescribeFileSystems</a> operation.
     * </p>
     * <p>
     * This operation requires permission for the <code>elasticfilesystem:CreateTags</code> action.
     * </p>
     *
     * @param createTagsRequest
     * @return A Java Future containing the result of the CreateTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's AWS account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.CreateTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/CreateTags" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTagsResponse> createTags(CreateTagsRequest createTagsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTagsRequest, CreateTagsResponse>().withOperationName("CreateTags")
                            .withMarshaller(new CreateTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createTagsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a file system, permanently severing access to its contents. Upon return, the file system no longer exists
     * and you can't access any contents of the deleted file system.
     * </p>
     * <p>
     * You can't delete a file system that is in use. That is, if the file system has any mount targets, you must first
     * delete them. For more information, see <a>DescribeMountTargets</a> and <a>DeleteMountTarget</a>.
     * </p>
     * <note>
     * <p>
     * The <code>DeleteFileSystem</code> call returns while the file system state is still <code>deleting</code>. You
     * can check the file system deletion status by calling the <a>DescribeFileSystems</a> operation, which returns a
     * list of file systems in your account. If you pass file system ID or creation token for the deleted file system,
     * the <a>DescribeFileSystems</a> returns a <code>404 FileSystemNotFound</code> error.
     * </p>
     * </note>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DeleteFileSystem</code> action.
     * </p>
     *
     * @param deleteFileSystemRequest
     * @return A Java Future containing the result of the DeleteFileSystem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's AWS account.</li>
     *         <li>FileSystemInUseException Returned if a file system has mount targets.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DeleteFileSystem
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DeleteFileSystem"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteFileSystemResponse> deleteFileSystem(DeleteFileSystemRequest deleteFileSystemRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteFileSystemResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteFileSystemResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteFileSystemResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteFileSystemRequest, DeleteFileSystemResponse>()
                            .withOperationName("DeleteFileSystem")
                            .withMarshaller(new DeleteFileSystemRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteFileSystemRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified mount target.
     * </p>
     * <p>
     * This operation forcibly breaks any mounts of the file system by using the mount target that is being deleted,
     * which might disrupt instances or applications using those mounts. To avoid applications getting cut off abruptly,
     * you might consider unmounting any mounts of the mount target, if feasible. The operation also deletes the
     * associated network interface. Uncommitted writes might be lost, but breaking a mount target using this operation
     * does not corrupt the file system itself. The file system you created remains. You can mount an EC2 instance in
     * your VPC by using another mount target.
     * </p>
     * <p>
     * This operation requires permissions for the following action on the file system:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>elasticfilesystem:DeleteMountTarget</code>
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The <code>DeleteMountTarget</code> call returns while the mount target state is still <code>deleting</code>. You
     * can check the mount target deletion by calling the <a>DescribeMountTargets</a> operation, which returns a list of
     * mount target descriptions for the given file system.
     * </p>
     * </note>
     * <p>
     * The operation also requires permissions for the following Amazon EC2 action on the mount target's network
     * interface:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ec2:DeleteNetworkInterface</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteMountTargetRequest
     * @return A Java Future containing the result of the DeleteMountTarget operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>DependencyTimeoutException The service timed out trying to fulfill the request, and the client should
     *         try the call again.</li>
     *         <li>MountTargetNotFoundException Returned if there is no mount target with the specified ID found in the
     *         caller's account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DeleteMountTarget
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DeleteMountTarget"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMountTargetResponse> deleteMountTarget(DeleteMountTargetRequest deleteMountTargetRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMountTargetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteMountTargetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteMountTargetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMountTargetRequest, DeleteMountTargetResponse>()
                            .withOperationName("DeleteMountTarget")
                            .withMarshaller(new DeleteMountTargetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteMountTargetRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified tags from a file system. If the <code>DeleteTags</code> request includes a tag key that
     * doesn't exist, Amazon EFS ignores it and doesn't cause an error. For more information about tags and related
     * restrictions, see <a href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html">Tag
     * Restrictions</a> in the <i>AWS Billing and Cost Management User Guide</i>.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DeleteTags</code> action.
     * </p>
     *
     * @param deleteTagsRequest
     * @return A Java Future containing the result of the DeleteTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's AWS account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DeleteTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DeleteTags" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTagsResponse> deleteTags(DeleteTagsRequest deleteTagsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTagsRequest, DeleteTagsResponse>().withOperationName("DeleteTags")
                            .withMarshaller(new DeleteTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteTagsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the description of a specific Amazon EFS file system if either the file system <code>CreationToken</code>
     * or the <code>FileSystemId</code> is provided. Otherwise, it returns descriptions of all file systems owned by the
     * caller's AWS account in the AWS Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all file system descriptions, you can optionally specify the <code>MaxItems</code> parameter to
     * limit the number of descriptions in a response. Currently, this number is automatically set to 10. If more file
     * system descriptions remain, Amazon EFS returns a <code>NextMarker</code>, an opaque token, in the response. In
     * this case, you should send a subsequent request with the <code>Marker</code> request parameter set to the value
     * of <code>NextMarker</code>.
     * </p>
     * <p>
     * To retrieve a list of your file system descriptions, this operation is used in an iterative process, where
     * <code>DescribeFileSystems</code> is called first without the <code>Marker</code> and then the operation continues
     * to call it with the <code>Marker</code> parameter set to the value of the <code>NextMarker</code> from the
     * previous response until the response has no <code>NextMarker</code>.
     * </p>
     * <p>
     * The order of file systems returned in the response of one <code>DescribeFileSystems</code> call and the order of
     * file systems returned across the responses of a multi-call iteration is unspecified.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeFileSystems</code> action.
     * </p>
     *
     * @param describeFileSystemsRequest
     * @return A Java Future containing the result of the DescribeFileSystems operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's AWS account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeFileSystems
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeFileSystems"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFileSystemsResponse> describeFileSystems(
            DescribeFileSystemsRequest describeFileSystemsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeFileSystemsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeFileSystemsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeFileSystemsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeFileSystemsRequest, DescribeFileSystemsResponse>()
                            .withOperationName("DescribeFileSystems")
                            .withMarshaller(new DescribeFileSystemsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeFileSystemsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the current <code>LifecycleConfiguration</code> object for the specified Amazon EFS file system. EFS
     * lifecycle management uses the <code>LifecycleConfiguration</code> object to identify which files to move to the
     * EFS Infrequent Access (IA) storage class. For a file system without a <code>LifecycleConfiguration</code> object,
     * the call returns an empty array in the response.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeLifecycleConfiguration</code>
     * operation.
     * </p>
     *
     * @param describeLifecycleConfigurationRequest
     * @return A Java Future containing the result of the DescribeLifecycleConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's AWS account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeLifecycleConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeLifecycleConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLifecycleConfigurationResponse> describeLifecycleConfiguration(
            DescribeLifecycleConfigurationRequest describeLifecycleConfigurationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeLifecycleConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeLifecycleConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeLifecycleConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLifecycleConfigurationRequest, DescribeLifecycleConfigurationResponse>()
                            .withOperationName("DescribeLifecycleConfiguration")
                            .withMarshaller(new DescribeLifecycleConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeLifecycleConfigurationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the security groups currently in effect for a mount target. This operation requires that the network
     * interface of the mount target has been created and the lifecycle state of the mount target is not
     * <code>deleted</code>.
     * </p>
     * <p>
     * This operation requires permissions for the following actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>elasticfilesystem:DescribeMountTargetSecurityGroups</code> action on the mount target's file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:DescribeNetworkInterfaceAttribute</code> action on the mount target's network interface.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeMountTargetSecurityGroupsRequest
     * @return A Java Future containing the result of the DescribeMountTargetSecurityGroups operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>MountTargetNotFoundException Returned if there is no mount target with the specified ID found in the
     *         caller's account.</li>
     *         <li>IncorrectMountTargetStateException Returned if the mount target is not in the correct state for the
     *         operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeMountTargetSecurityGroups
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeMountTargetSecurityGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeMountTargetSecurityGroupsResponse> describeMountTargetSecurityGroups(
            DescribeMountTargetSecurityGroupsRequest describeMountTargetSecurityGroupsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeMountTargetSecurityGroupsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeMountTargetSecurityGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeMountTargetSecurityGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeMountTargetSecurityGroupsRequest, DescribeMountTargetSecurityGroupsResponse>()
                            .withOperationName("DescribeMountTargetSecurityGroups")
                            .withMarshaller(new DescribeMountTargetSecurityGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeMountTargetSecurityGroupsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the descriptions of all the current mount targets, or a specific mount target, for a file system. When
     * requesting all of the current mount targets, the order of mount targets returned in the response is unspecified.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeMountTargets</code> action, on either
     * the file system ID that you specify in <code>FileSystemId</code>, or on the file system of the mount target that
     * you specify in <code>MountTargetId</code>.
     * </p>
     *
     * @param describeMountTargetsRequest
     * @return A Java Future containing the result of the DescribeMountTargets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's AWS account.</li>
     *         <li>MountTargetNotFoundException Returned if there is no mount target with the specified ID found in the
     *         caller's account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeMountTargets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeMountTargets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeMountTargetsResponse> describeMountTargets(
            DescribeMountTargetsRequest describeMountTargetsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeMountTargetsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeMountTargetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeMountTargetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeMountTargetsRequest, DescribeMountTargetsResponse>()
                            .withOperationName("DescribeMountTargets")
                            .withMarshaller(new DescribeMountTargetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeMountTargetsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the tags associated with a file system. The order of tags returned in the response of one
     * <code>DescribeTags</code> call and the order of tags returned across the responses of a multiple-call iteration
     * (when using pagination) is unspecified.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeTags</code> action.
     * </p>
     *
     * @param describeTagsRequest
     * @return A Java Future containing the result of the DescribeTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's AWS account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeTags" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeTagsResponse> describeTags(DescribeTagsRequest describeTagsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeTagsRequest, DescribeTagsResponse>()
                            .withOperationName("DescribeTags").withMarshaller(new DescribeTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeTagsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies the set of security groups in effect for a mount target.
     * </p>
     * <p>
     * When you create a mount target, Amazon EFS also creates a new network interface. For more information, see
     * <a>CreateMountTarget</a>. This operation replaces the security groups in effect for the network interface
     * associated with a mount target, with the <code>SecurityGroups</code> provided in the request. This operation
     * requires that the network interface of the mount target has been created and the lifecycle state of the mount
     * target is not <code>deleted</code>.
     * </p>
     * <p>
     * The operation requires permissions for the following actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>elasticfilesystem:ModifyMountTargetSecurityGroups</code> action on the mount target's file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:ModifyNetworkInterfaceAttribute</code> action on the mount target's network interface.
     * </p>
     * </li>
     * </ul>
     *
     * @param modifyMountTargetSecurityGroupsRequest
     * @return A Java Future containing the result of the ModifyMountTargetSecurityGroups operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>MountTargetNotFoundException Returned if there is no mount target with the specified ID found in the
     *         caller's account.</li>
     *         <li>IncorrectMountTargetStateException Returned if the mount target is not in the correct state for the
     *         operation.</li>
     *         <li>SecurityGroupLimitExceededException Returned if the size of <code>SecurityGroups</code> specified in
     *         the request is greater than five.</li>
     *         <li>SecurityGroupNotFoundException Returned if one of the specified security groups doesn't exist in the
     *         subnet's VPC.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.ModifyMountTargetSecurityGroups
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/ModifyMountTargetSecurityGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ModifyMountTargetSecurityGroupsResponse> modifyMountTargetSecurityGroups(
            ModifyMountTargetSecurityGroupsRequest modifyMountTargetSecurityGroupsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ModifyMountTargetSecurityGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ModifyMountTargetSecurityGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ModifyMountTargetSecurityGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ModifyMountTargetSecurityGroupsRequest, ModifyMountTargetSecurityGroupsResponse>()
                            .withOperationName("ModifyMountTargetSecurityGroups")
                            .withMarshaller(new ModifyMountTargetSecurityGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(modifyMountTargetSecurityGroupsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables lifecycle management by creating a new <code>LifecycleConfiguration</code> object. A
     * <code>LifecycleConfiguration</code> object defines when files in an Amazon EFS file system are automatically
     * transitioned to the lower-cost EFS Infrequent Access (IA) storage class. A <code>LifecycleConfiguration</code>
     * applies to all files in a file system.
     * </p>
     * <p>
     * Each Amazon EFS file system supports one lifecycle configuration, which applies to all files in the file system.
     * If a <code>LifecycleConfiguration</code> object already exists for the specified file system, a
     * <code>PutLifecycleConfiguration</code> call modifies the existing configuration. A
     * <code>PutLifecycleConfiguration</code> call with an empty <code>LifecyclePolicies</code> array in the request
     * body deletes any existing <code>LifecycleConfiguration</code> and disables lifecycle management.
     * </p>
     * <p>
     * In the request, specify the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The ID for the file system for which you are enabling, disabling, or modifying lifecycle management.
     * </p>
     * </li>
     * <li>
     * <p>
     * A <code>LifecyclePolicies</code> array of <code>LifecyclePolicy</code> objects that define when files are moved
     * to the IA storage class. The array can contain only one <code>LifecyclePolicy</code> item.
     * </p>
     * </li>
     * </ul>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:PutLifecycleConfiguration</code> operation.
     * </p>
     * <p>
     * To apply a <code>LifecycleConfiguration</code> object to an encrypted file system, you need the same AWS Key
     * Management Service (AWS KMS) permissions as when you created the encrypted file system.
     * </p>
     *
     * @param putLifecycleConfigurationRequest
     * @return A Java Future containing the result of the PutLifecycleConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's AWS account.</li>
     *         <li>IncorrectFileSystemLifeCycleStateException Returned if the file system's lifecycle state is not
     *         "available".</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.PutLifecycleConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/PutLifecycleConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutLifecycleConfigurationResponse> putLifecycleConfiguration(
            PutLifecycleConfigurationRequest putLifecycleConfigurationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutLifecycleConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutLifecycleConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutLifecycleConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutLifecycleConfigurationRequest, PutLifecycleConfigurationResponse>()
                            .withOperationName("PutLifecycleConfiguration")
                            .withMarshaller(new PutLifecycleConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putLifecycleConfigurationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the throughput mode or the amount of provisioned throughput of an existing file system.
     * </p>
     *
     * @param updateFileSystemRequest
     * @return A Java Future containing the result of the UpdateFileSystem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's AWS account.</li>
     *         <li>IncorrectFileSystemLifeCycleStateException Returned if the file system's lifecycle state is not
     *         "available".</li>
     *         <li>InsufficientThroughputCapacityException Returned if there's not enough capacity to provision
     *         additional throughput. This value might be returned when you try to create a file system in provisioned
     *         throughput mode, when you attempt to increase the provisioned throughput of an existing file system, or
     *         when you attempt to change an existing file system from bursting to provisioned throughput mode.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>ThroughputLimitExceededException Returned if the throughput mode or amount of provisioned throughput
     *         can't be changed because the throughput limit of 1024 MiB/s has been reached.</li>
     *         <li>TooManyRequestsException Returned if you don’t wait at least 24 hours before changing the throughput
     *         mode, or decreasing the Provisioned Throughput value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.UpdateFileSystem
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/UpdateFileSystem"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFileSystemResponse> updateFileSystem(UpdateFileSystemRequest updateFileSystemRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFileSystemResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateFileSystemResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateFileSystemResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFileSystemRequest, UpdateFileSystemResponse>()
                            .withOperationName("UpdateFileSystem")
                            .withMarshaller(new UpdateFileSystemRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateFileSystemRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(EfsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedAvailabilityZone")
                                .exceptionBuilderSupplier(UnsupportedAvailabilityZoneException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IpAddressInUse")
                                .exceptionBuilderSupplier(IpAddressInUseException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MountTargetConflict")
                                .exceptionBuilderSupplier(MountTargetConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SecurityGroupNotFound")
                                .exceptionBuilderSupplier(SecurityGroupNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FileSystemInUse")
                                .exceptionBuilderSupplier(FileSystemInUseException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FileSystemAlreadyExists")
                                .exceptionBuilderSupplier(FileSystemAlreadyExistsException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NetworkInterfaceLimitExceeded")
                                .exceptionBuilderSupplier(NetworkInterfaceLimitExceededException::builder).httpStatusCode(409)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IncorrectFileSystemLifeCycleState")
                                .exceptionBuilderSupplier(IncorrectFileSystemLifeCycleStateException::builder)
                                .httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoFreeAddressesInSubnet")
                                .exceptionBuilderSupplier(NoFreeAddressesInSubnetException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FileSystemNotFound")
                                .exceptionBuilderSupplier(FileSystemNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InsufficientThroughputCapacity")
                                .exceptionBuilderSupplier(InsufficientThroughputCapacityException::builder).httpStatusCode(503)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MountTargetNotFound")
                                .exceptionBuilderSupplier(MountTargetNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SubnetNotFound")
                                .exceptionBuilderSupplier(SubnetNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IncorrectMountTargetState")
                                .exceptionBuilderSupplier(IncorrectMountTargetStateException::builder).httpStatusCode(409)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SecurityGroupLimitExceeded")
                                .exceptionBuilderSupplier(SecurityGroupLimitExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThroughputLimitExceeded")
                                .exceptionBuilderSupplier(ThroughputLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DependencyTimeout")
                                .exceptionBuilderSupplier(DependencyTimeoutException::builder).httpStatusCode(504).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequests")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FileSystemLimitExceeded")
                                .exceptionBuilderSupplier(FileSystemLimitExceededException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequest")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerError")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
