/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.efs.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a policy used by EFS lifecycle management and EFS intelligent tiering that specifies when to transition
 * files into and out of the file system's Infrequent Access (IA) storage class. For more information, see <a
 * href="https://docs.aws.amazon.com/efs/latest/ug/lifecycle-management-efs.html">EFS Intelligent‐Tiering and EFS
 * Lifecycle Management</a>.
 * </p>
 * <note>
 * <p>
 * When using the <code>put-lifecycle-configuration</code> CLI command or the <code>PutLifecycleConfiguration</code> API
 * action, Amazon EFS requires that each <code>LifecyclePolicy</code> object have only a single transition. This means
 * that in a request body, <code>LifecyclePolicies</code> needs to be structured as an array of
 * <code>LifecyclePolicy</code> objects, one object for each transition, <code>TransitionToIA</code>,
 * <code>TransitionToPrimaryStorageClass</code>. For more information, see the request examples in
 * <a>PutLifecycleConfiguration</a>.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LifecyclePolicy implements SdkPojo, Serializable, ToCopyableBuilder<LifecyclePolicy.Builder, LifecyclePolicy> {
    private static final SdkField<String> TRANSITION_TO_IA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TransitionToIA").getter(getter(LifecyclePolicy::transitionToIAAsString))
            .setter(setter(Builder::transitionToIA))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TransitionToIA").build()).build();

    private static final SdkField<String> TRANSITION_TO_PRIMARY_STORAGE_CLASS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("TransitionToPrimaryStorageClass")
            .getter(getter(LifecyclePolicy::transitionToPrimaryStorageClassAsString))
            .setter(setter(Builder::transitionToPrimaryStorageClass))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TransitionToPrimaryStorageClass")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TRANSITION_TO_IA_FIELD,
            TRANSITION_TO_PRIMARY_STORAGE_CLASS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String transitionToIA;

    private final String transitionToPrimaryStorageClass;

    private LifecyclePolicy(BuilderImpl builder) {
        this.transitionToIA = builder.transitionToIA;
        this.transitionToPrimaryStorageClass = builder.transitionToPrimaryStorageClass;
    }

    /**
     * <p>
     * Describes the period of time that a file is not accessed, after which it transitions to IA storage. Metadata
     * operations such as listing the contents of a directory don't count as file access events.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #transitionToIA}
     * will return {@link TransitionToIARules#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #transitionToIAAsString}.
     * </p>
     * 
     * @return Describes the period of time that a file is not accessed, after which it transitions to IA storage.
     *         Metadata operations such as listing the contents of a directory don't count as file access events.
     * @see TransitionToIARules
     */
    public final TransitionToIARules transitionToIA() {
        return TransitionToIARules.fromValue(transitionToIA);
    }

    /**
     * <p>
     * Describes the period of time that a file is not accessed, after which it transitions to IA storage. Metadata
     * operations such as listing the contents of a directory don't count as file access events.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #transitionToIA}
     * will return {@link TransitionToIARules#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #transitionToIAAsString}.
     * </p>
     * 
     * @return Describes the period of time that a file is not accessed, after which it transitions to IA storage.
     *         Metadata operations such as listing the contents of a directory don't count as file access events.
     * @see TransitionToIARules
     */
    public final String transitionToIAAsString() {
        return transitionToIA;
    }

    /**
     * <p>
     * Describes when to transition a file from IA storage to primary storage. Metadata operations such as listing the
     * contents of a directory don't count as file access events.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #transitionToPrimaryStorageClass} will return
     * {@link TransitionToPrimaryStorageClassRules#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #transitionToPrimaryStorageClassAsString}.
     * </p>
     * 
     * @return Describes when to transition a file from IA storage to primary storage. Metadata operations such as
     *         listing the contents of a directory don't count as file access events.
     * @see TransitionToPrimaryStorageClassRules
     */
    public final TransitionToPrimaryStorageClassRules transitionToPrimaryStorageClass() {
        return TransitionToPrimaryStorageClassRules.fromValue(transitionToPrimaryStorageClass);
    }

    /**
     * <p>
     * Describes when to transition a file from IA storage to primary storage. Metadata operations such as listing the
     * contents of a directory don't count as file access events.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #transitionToPrimaryStorageClass} will return
     * {@link TransitionToPrimaryStorageClassRules#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #transitionToPrimaryStorageClassAsString}.
     * </p>
     * 
     * @return Describes when to transition a file from IA storage to primary storage. Metadata operations such as
     *         listing the contents of a directory don't count as file access events.
     * @see TransitionToPrimaryStorageClassRules
     */
    public final String transitionToPrimaryStorageClassAsString() {
        return transitionToPrimaryStorageClass;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(transitionToIAAsString());
        hashCode = 31 * hashCode + Objects.hashCode(transitionToPrimaryStorageClassAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LifecyclePolicy)) {
            return false;
        }
        LifecyclePolicy other = (LifecyclePolicy) obj;
        return Objects.equals(transitionToIAAsString(), other.transitionToIAAsString())
                && Objects.equals(transitionToPrimaryStorageClassAsString(), other.transitionToPrimaryStorageClassAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LifecyclePolicy").add("TransitionToIA", transitionToIAAsString())
                .add("TransitionToPrimaryStorageClass", transitionToPrimaryStorageClassAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TransitionToIA":
            return Optional.ofNullable(clazz.cast(transitionToIAAsString()));
        case "TransitionToPrimaryStorageClass":
            return Optional.ofNullable(clazz.cast(transitionToPrimaryStorageClassAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LifecyclePolicy, T> g) {
        return obj -> g.apply((LifecyclePolicy) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LifecyclePolicy> {
        /**
         * <p>
         * Describes the period of time that a file is not accessed, after which it transitions to IA storage. Metadata
         * operations such as listing the contents of a directory don't count as file access events.
         * </p>
         * 
         * @param transitionToIA
         *        Describes the period of time that a file is not accessed, after which it transitions to IA storage.
         *        Metadata operations such as listing the contents of a directory don't count as file access events.
         * @see TransitionToIARules
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransitionToIARules
         */
        Builder transitionToIA(String transitionToIA);

        /**
         * <p>
         * Describes the period of time that a file is not accessed, after which it transitions to IA storage. Metadata
         * operations such as listing the contents of a directory don't count as file access events.
         * </p>
         * 
         * @param transitionToIA
         *        Describes the period of time that a file is not accessed, after which it transitions to IA storage.
         *        Metadata operations such as listing the contents of a directory don't count as file access events.
         * @see TransitionToIARules
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransitionToIARules
         */
        Builder transitionToIA(TransitionToIARules transitionToIA);

        /**
         * <p>
         * Describes when to transition a file from IA storage to primary storage. Metadata operations such as listing
         * the contents of a directory don't count as file access events.
         * </p>
         * 
         * @param transitionToPrimaryStorageClass
         *        Describes when to transition a file from IA storage to primary storage. Metadata operations such as
         *        listing the contents of a directory don't count as file access events.
         * @see TransitionToPrimaryStorageClassRules
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransitionToPrimaryStorageClassRules
         */
        Builder transitionToPrimaryStorageClass(String transitionToPrimaryStorageClass);

        /**
         * <p>
         * Describes when to transition a file from IA storage to primary storage. Metadata operations such as listing
         * the contents of a directory don't count as file access events.
         * </p>
         * 
         * @param transitionToPrimaryStorageClass
         *        Describes when to transition a file from IA storage to primary storage. Metadata operations such as
         *        listing the contents of a directory don't count as file access events.
         * @see TransitionToPrimaryStorageClassRules
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransitionToPrimaryStorageClassRules
         */
        Builder transitionToPrimaryStorageClass(TransitionToPrimaryStorageClassRules transitionToPrimaryStorageClass);
    }

    static final class BuilderImpl implements Builder {
        private String transitionToIA;

        private String transitionToPrimaryStorageClass;

        private BuilderImpl() {
        }

        private BuilderImpl(LifecyclePolicy model) {
            transitionToIA(model.transitionToIA);
            transitionToPrimaryStorageClass(model.transitionToPrimaryStorageClass);
        }

        public final String getTransitionToIA() {
            return transitionToIA;
        }

        public final void setTransitionToIA(String transitionToIA) {
            this.transitionToIA = transitionToIA;
        }

        @Override
        @Transient
        public final Builder transitionToIA(String transitionToIA) {
            this.transitionToIA = transitionToIA;
            return this;
        }

        @Override
        @Transient
        public final Builder transitionToIA(TransitionToIARules transitionToIA) {
            this.transitionToIA(transitionToIA == null ? null : transitionToIA.toString());
            return this;
        }

        public final String getTransitionToPrimaryStorageClass() {
            return transitionToPrimaryStorageClass;
        }

        public final void setTransitionToPrimaryStorageClass(String transitionToPrimaryStorageClass) {
            this.transitionToPrimaryStorageClass = transitionToPrimaryStorageClass;
        }

        @Override
        @Transient
        public final Builder transitionToPrimaryStorageClass(String transitionToPrimaryStorageClass) {
            this.transitionToPrimaryStorageClass = transitionToPrimaryStorageClass;
            return this;
        }

        @Override
        @Transient
        public final Builder transitionToPrimaryStorageClass(TransitionToPrimaryStorageClassRules transitionToPrimaryStorageClass) {
            this.transitionToPrimaryStorageClass(transitionToPrimaryStorageClass == null ? null : transitionToPrimaryStorageClass
                    .toString());
            return this;
        }

        @Override
        public LifecyclePolicy build() {
            return new LifecyclePolicy(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
