/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.efs;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.efs.model.AccessPointAlreadyExistsException;
import software.amazon.awssdk.services.efs.model.AccessPointLimitExceededException;
import software.amazon.awssdk.services.efs.model.AccessPointNotFoundException;
import software.amazon.awssdk.services.efs.model.AvailabilityZonesMismatchException;
import software.amazon.awssdk.services.efs.model.BadRequestException;
import software.amazon.awssdk.services.efs.model.CreateAccessPointRequest;
import software.amazon.awssdk.services.efs.model.CreateAccessPointResponse;
import software.amazon.awssdk.services.efs.model.CreateFileSystemRequest;
import software.amazon.awssdk.services.efs.model.CreateFileSystemResponse;
import software.amazon.awssdk.services.efs.model.CreateMountTargetRequest;
import software.amazon.awssdk.services.efs.model.CreateMountTargetResponse;
import software.amazon.awssdk.services.efs.model.CreateReplicationConfigurationRequest;
import software.amazon.awssdk.services.efs.model.CreateReplicationConfigurationResponse;
import software.amazon.awssdk.services.efs.model.CreateTagsRequest;
import software.amazon.awssdk.services.efs.model.CreateTagsResponse;
import software.amazon.awssdk.services.efs.model.DeleteAccessPointRequest;
import software.amazon.awssdk.services.efs.model.DeleteAccessPointResponse;
import software.amazon.awssdk.services.efs.model.DeleteFileSystemPolicyRequest;
import software.amazon.awssdk.services.efs.model.DeleteFileSystemPolicyResponse;
import software.amazon.awssdk.services.efs.model.DeleteFileSystemRequest;
import software.amazon.awssdk.services.efs.model.DeleteFileSystemResponse;
import software.amazon.awssdk.services.efs.model.DeleteMountTargetRequest;
import software.amazon.awssdk.services.efs.model.DeleteMountTargetResponse;
import software.amazon.awssdk.services.efs.model.DeleteReplicationConfigurationRequest;
import software.amazon.awssdk.services.efs.model.DeleteReplicationConfigurationResponse;
import software.amazon.awssdk.services.efs.model.DeleteTagsRequest;
import software.amazon.awssdk.services.efs.model.DeleteTagsResponse;
import software.amazon.awssdk.services.efs.model.DependencyTimeoutException;
import software.amazon.awssdk.services.efs.model.DescribeAccessPointsRequest;
import software.amazon.awssdk.services.efs.model.DescribeAccessPointsResponse;
import software.amazon.awssdk.services.efs.model.DescribeAccountPreferencesRequest;
import software.amazon.awssdk.services.efs.model.DescribeAccountPreferencesResponse;
import software.amazon.awssdk.services.efs.model.DescribeBackupPolicyRequest;
import software.amazon.awssdk.services.efs.model.DescribeBackupPolicyResponse;
import software.amazon.awssdk.services.efs.model.DescribeFileSystemPolicyRequest;
import software.amazon.awssdk.services.efs.model.DescribeFileSystemPolicyResponse;
import software.amazon.awssdk.services.efs.model.DescribeFileSystemsRequest;
import software.amazon.awssdk.services.efs.model.DescribeFileSystemsResponse;
import software.amazon.awssdk.services.efs.model.DescribeLifecycleConfigurationRequest;
import software.amazon.awssdk.services.efs.model.DescribeLifecycleConfigurationResponse;
import software.amazon.awssdk.services.efs.model.DescribeMountTargetSecurityGroupsRequest;
import software.amazon.awssdk.services.efs.model.DescribeMountTargetSecurityGroupsResponse;
import software.amazon.awssdk.services.efs.model.DescribeMountTargetsRequest;
import software.amazon.awssdk.services.efs.model.DescribeMountTargetsResponse;
import software.amazon.awssdk.services.efs.model.DescribeReplicationConfigurationsRequest;
import software.amazon.awssdk.services.efs.model.DescribeReplicationConfigurationsResponse;
import software.amazon.awssdk.services.efs.model.DescribeTagsRequest;
import software.amazon.awssdk.services.efs.model.DescribeTagsResponse;
import software.amazon.awssdk.services.efs.model.EfsException;
import software.amazon.awssdk.services.efs.model.EfsRequest;
import software.amazon.awssdk.services.efs.model.FileSystemAlreadyExistsException;
import software.amazon.awssdk.services.efs.model.FileSystemInUseException;
import software.amazon.awssdk.services.efs.model.FileSystemLimitExceededException;
import software.amazon.awssdk.services.efs.model.FileSystemNotFoundException;
import software.amazon.awssdk.services.efs.model.IncorrectFileSystemLifeCycleStateException;
import software.amazon.awssdk.services.efs.model.IncorrectMountTargetStateException;
import software.amazon.awssdk.services.efs.model.InsufficientThroughputCapacityException;
import software.amazon.awssdk.services.efs.model.InternalServerErrorException;
import software.amazon.awssdk.services.efs.model.InvalidPolicyException;
import software.amazon.awssdk.services.efs.model.IpAddressInUseException;
import software.amazon.awssdk.services.efs.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.efs.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.efs.model.ModifyMountTargetSecurityGroupsRequest;
import software.amazon.awssdk.services.efs.model.ModifyMountTargetSecurityGroupsResponse;
import software.amazon.awssdk.services.efs.model.MountTargetConflictException;
import software.amazon.awssdk.services.efs.model.MountTargetNotFoundException;
import software.amazon.awssdk.services.efs.model.NetworkInterfaceLimitExceededException;
import software.amazon.awssdk.services.efs.model.NoFreeAddressesInSubnetException;
import software.amazon.awssdk.services.efs.model.PolicyNotFoundException;
import software.amazon.awssdk.services.efs.model.PutAccountPreferencesRequest;
import software.amazon.awssdk.services.efs.model.PutAccountPreferencesResponse;
import software.amazon.awssdk.services.efs.model.PutBackupPolicyRequest;
import software.amazon.awssdk.services.efs.model.PutBackupPolicyResponse;
import software.amazon.awssdk.services.efs.model.PutFileSystemPolicyRequest;
import software.amazon.awssdk.services.efs.model.PutFileSystemPolicyResponse;
import software.amazon.awssdk.services.efs.model.PutLifecycleConfigurationRequest;
import software.amazon.awssdk.services.efs.model.PutLifecycleConfigurationResponse;
import software.amazon.awssdk.services.efs.model.ReplicationNotFoundException;
import software.amazon.awssdk.services.efs.model.SecurityGroupLimitExceededException;
import software.amazon.awssdk.services.efs.model.SecurityGroupNotFoundException;
import software.amazon.awssdk.services.efs.model.SubnetNotFoundException;
import software.amazon.awssdk.services.efs.model.TagResourceRequest;
import software.amazon.awssdk.services.efs.model.TagResourceResponse;
import software.amazon.awssdk.services.efs.model.ThrottlingException;
import software.amazon.awssdk.services.efs.model.ThroughputLimitExceededException;
import software.amazon.awssdk.services.efs.model.TooManyRequestsException;
import software.amazon.awssdk.services.efs.model.UnsupportedAvailabilityZoneException;
import software.amazon.awssdk.services.efs.model.UntagResourceRequest;
import software.amazon.awssdk.services.efs.model.UntagResourceResponse;
import software.amazon.awssdk.services.efs.model.UpdateFileSystemRequest;
import software.amazon.awssdk.services.efs.model.UpdateFileSystemResponse;
import software.amazon.awssdk.services.efs.model.ValidationException;
import software.amazon.awssdk.services.efs.paginators.DescribeAccessPointsIterable;
import software.amazon.awssdk.services.efs.paginators.DescribeFileSystemsIterable;
import software.amazon.awssdk.services.efs.paginators.DescribeTagsIterable;
import software.amazon.awssdk.services.efs.paginators.ListTagsForResourceIterable;
import software.amazon.awssdk.services.efs.transform.CreateAccessPointRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.CreateFileSystemRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.CreateMountTargetRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.CreateReplicationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.CreateTagsRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DeleteAccessPointRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DeleteFileSystemPolicyRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DeleteFileSystemRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DeleteMountTargetRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DeleteReplicationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DeleteTagsRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DescribeAccessPointsRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DescribeAccountPreferencesRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DescribeBackupPolicyRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DescribeFileSystemPolicyRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DescribeFileSystemsRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DescribeLifecycleConfigurationRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DescribeMountTargetSecurityGroupsRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DescribeMountTargetsRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DescribeReplicationConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.DescribeTagsRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.ModifyMountTargetSecurityGroupsRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.PutAccountPreferencesRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.PutBackupPolicyRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.PutFileSystemPolicyRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.PutLifecycleConfigurationRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.efs.transform.UpdateFileSystemRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link EfsClient}.
 *
 * @see EfsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultEfsClient implements EfsClient {
    private static final Logger log = Logger.loggerFor(DefaultEfsClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultEfsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates an EFS access point. An access point is an application-specific view into an EFS file system that applies
     * an operating system user and group, and a file system path, to any file system request made through the access
     * point. The operating system user and group override any identity information provided by the NFS client. The file
     * system path is exposed as the access point's root directory. Applications using the access point can only access
     * data in the application's own directory and any subdirectories. To learn more, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/efs-access-points.html">Mounting a file system using EFS access
     * points</a>.
     * </p>
     * <note>
     * <p>
     * If multiple requests to create access points on the same file system are sent in quick succession, and the file
     * system is near the limit of 120 access points, you may experience a throttling response for these requests. This
     * is to ensure that the file system does not exceed the stated access point limit.
     * </p>
     * </note>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:CreateAccessPoint</code> action.
     * </p>
     *
     * @param createAccessPointRequest
     * @return Result of the CreateAccessPoint operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws AccessPointAlreadyExistsException
     *         Returned if the access point that you are trying to create already exists, with the creation token you
     *         provided in the request.
     * @throws IncorrectFileSystemLifeCycleStateException
     *         Returned if the file system's lifecycle state is not "available".
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws FileSystemNotFoundException
     *         Returned if the specified <code>FileSystemId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws AccessPointLimitExceededException
     *         Returned if the Amazon Web Services account has already created the maximum number of access points
     *         allowed per file system. For more informaton, see <a href=
     *         "https://docs.aws.amazon.com/efs/latest/ug/limits.html#limits-efs-resources-per-account-per-region"
     *         >https://docs.aws.amazon.com/efs/latest/ug/limits.html#limits-efs-resources-per-account-per-region</a>.
     * @throws ThrottlingException
     *         Returned when the <code>CreateAccessPoint</code> API action is called too quickly and the number of
     *         Access Points on the file system is nearing the <a href=
     *         "https://docs.aws.amazon.com/efs/latest/ug/limits.html#limits-efs-resources-per-account-per-region">limit
     *         of 120</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.CreateAccessPoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/CreateAccessPoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateAccessPointResponse createAccessPoint(CreateAccessPointRequest createAccessPointRequest)
            throws BadRequestException, AccessPointAlreadyExistsException, IncorrectFileSystemLifeCycleStateException,
            InternalServerErrorException, FileSystemNotFoundException, AccessPointLimitExceededException, ThrottlingException,
            AwsServiceException, SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAccessPointResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateAccessPointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAccessPointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAccessPoint");

            return clientHandler.execute(new ClientExecutionParams<CreateAccessPointRequest, CreateAccessPointResponse>()
                    .withOperationName("CreateAccessPoint").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createAccessPointRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateAccessPointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new, empty file system. The operation requires a creation token in the request that Amazon EFS uses to
     * ensure idempotent creation (calling the operation with same creation token has no effect). If a file system does
     * not currently exist that is owned by the caller's Amazon Web Services account with the specified creation token,
     * this operation does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a new, empty file system. The file system will have an Amazon EFS assigned ID, and an initial lifecycle
     * state <code>creating</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Returns with the description of the created file system.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Otherwise, this operation returns a <code>FileSystemAlreadyExists</code> error with the ID of the existing file
     * system.
     * </p>
     * <note>
     * <p>
     * For basic use cases, you can use a randomly generated UUID for the creation token.
     * </p>
     * </note>
     * <p>
     * The idempotent operation allows you to retry a <code>CreateFileSystem</code> call without risk of creating an
     * extra file system. This can happen when an initial call fails in a way that leaves it uncertain whether or not a
     * file system was actually created. An example might be that a transport level timeout occurred or your connection
     * was reset. As long as you use the same creation token, if the initial call had succeeded in creating a file
     * system, the client can learn of its existence from the <code>FileSystemAlreadyExists</code> error.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/creating-using-create-fs.html#creating-using-create-fs-part1"
     * >Creating a file system</a> in the <i>Amazon EFS User Guide</i>.
     * </p>
     * <note>
     * <p>
     * The <code>CreateFileSystem</code> call returns while the file system's lifecycle state is still
     * <code>creating</code>. You can check the file system creation status by calling the <a>DescribeFileSystems</a>
     * operation, which among other things returns the file system state.
     * </p>
     * </note>
     * <p>
     * This operation accepts an optional <code>PerformanceMode</code> parameter that you choose for your file system.
     * We recommend <code>generalPurpose</code> performance mode for most file systems. File systems using the
     * <code>maxIO</code> performance mode can scale to higher levels of aggregate throughput and operations per second
     * with a tradeoff of slightly higher latencies for most file operations. The performance mode can't be changed
     * after the file system has been created. For more information, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/performance.html#performancemodes.html">Amazon EFS performance
     * modes</a>.
     * </p>
     * <p>
     * You can set the throughput mode for the file system using the <code>ThroughputMode</code> parameter.
     * </p>
     * <p>
     * After the file system is fully created, Amazon EFS sets its lifecycle state to <code>available</code>, at which
     * point you can create one or more mount targets for the file system in your VPC. For more information, see
     * <a>CreateMountTarget</a>. You mount your Amazon EFS file system on an EC2 instances in your VPC by using the
     * mount target. For more information, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/how-it-works.html">Amazon EFS: How it Works</a>.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:CreateFileSystem</code> action.
     * </p>
     *
     * @param createFileSystemRequest
     * @return Result of the CreateFileSystem operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws FileSystemAlreadyExistsException
     *         Returned if the file system you are trying to create already exists, with the creation token you
     *         provided.
     * @throws FileSystemLimitExceededException
     *         Returned if the Amazon Web Services account has already created the maximum number of file systems
     *         allowed per account.
     * @throws InsufficientThroughputCapacityException
     *         Returned if there's not enough capacity to provision additional throughput. This value might be returned
     *         when you try to create a file system in provisioned throughput mode, when you attempt to increase the
     *         provisioned throughput of an existing file system, or when you attempt to change an existing file system
     *         from Bursting Throughput to Provisioned Throughput mode. Try again later.
     * @throws ThroughputLimitExceededException
     *         Returned if the throughput mode or amount of provisioned throughput can't be changed because the
     *         throughput limit of 1024 MiB/s has been reached.
     * @throws UnsupportedAvailabilityZoneException
     *         Returned if the requested Amazon EFS functionality is not available in the specified Availability Zone.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.CreateFileSystem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/CreateFileSystem"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateFileSystemResponse createFileSystem(CreateFileSystemRequest createFileSystemRequest) throws BadRequestException,
            InternalServerErrorException, FileSystemAlreadyExistsException, FileSystemLimitExceededException,
            InsufficientThroughputCapacityException, ThroughputLimitExceededException, UnsupportedAvailabilityZoneException,
            AwsServiceException, SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateFileSystemResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateFileSystemResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFileSystemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFileSystem");

            return clientHandler.execute(new ClientExecutionParams<CreateFileSystemRequest, CreateFileSystemResponse>()
                    .withOperationName("CreateFileSystem").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createFileSystemRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateFileSystemRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a mount target for a file system. You can then mount the file system on EC2 instances by using the mount
     * target.
     * </p>
     * <p>
     * You can create one mount target in each Availability Zone in your VPC. All EC2 instances in a VPC within a given
     * Availability Zone share a single mount target for a given file system. If you have multiple subnets in an
     * Availability Zone, you create a mount target in one of the subnets. EC2 instances do not need to be in the same
     * subnet as the mount target in order to access their file system.
     * </p>
     * <p>
     * You can create only one mount target for an EFS file system using One Zone storage classes. You must create that
     * mount target in the same Availability Zone in which the file system is located. Use the
     * <code>AvailabilityZoneName</code> and <code>AvailabiltyZoneId</code> properties in the <a>DescribeFileSystems</a>
     * response object to get this information. Use the <code>subnetId</code> associated with the file system's
     * Availability Zone when creating the mount target.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/efs/latest/ug/how-it-works.html">Amazon EFS: How
     * it Works</a>.
     * </p>
     * <p>
     * To create a mount target for a file system, the file system's lifecycle state must be <code>available</code>. For
     * more information, see <a>DescribeFileSystems</a>.
     * </p>
     * <p>
     * In the request, provide the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The file system ID for which you are creating the mount target.
     * </p>
     * </li>
     * <li>
     * <p>
     * A subnet ID, which determines the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The VPC in which Amazon EFS creates the mount target
     * </p>
     * </li>
     * <li>
     * <p>
     * The Availability Zone in which Amazon EFS creates the mount target
     * </p>
     * </li>
     * <li>
     * <p>
     * The IP address range from which Amazon EFS selects the IP address of the mount target (if you don't specify an IP
     * address in the request)
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * After creating the mount target, Amazon EFS returns a response that includes, a <code>MountTargetId</code> and an
     * <code>IpAddress</code>. You use this IP address when mounting the file system in an EC2 instance. You can also
     * use the mount target's DNS name when mounting the file system. The EC2 instance on which you mount the file
     * system by using the mount target can resolve the mount target's DNS name to its IP address. For more information,
     * see <a href="https://docs.aws.amazon.com/efs/latest/ug/how-it-works.html#how-it-works-implementation">How it
     * Works: Implementation Overview</a>.
     * </p>
     * <p>
     * Note that you can create mount targets for a file system in only one VPC, and there can be only one mount target
     * per Availability Zone. That is, if the file system already has one or more mount targets created for it, the
     * subnet specified in the request to add another mount target must meet the following requirements:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Must belong to the same VPC as the subnets of the existing mount targets
     * </p>
     * </li>
     * <li>
     * <p>
     * Must not be in the same Availability Zone as any of the subnets of the existing mount targets
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the request satisfies the requirements, Amazon EFS does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a new mount target in the specified subnet.
     * </p>
     * </li>
     * <li>
     * <p>
     * Also creates a new network interface in the subnet as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the request provides an <code>IpAddress</code>, Amazon EFS assigns that IP address to the network interface.
     * Otherwise, Amazon EFS assigns a free address in the subnet (in the same way that the Amazon EC2
     * <code>CreateNetworkInterface</code> call does when a request does not specify a primary private IP address).
     * </p>
     * </li>
     * <li>
     * <p>
     * If the request provides <code>SecurityGroups</code>, this network interface is associated with those security
     * groups. Otherwise, it belongs to the default security group for the subnet's VPC.
     * </p>
     * </li>
     * <li>
     * <p>
     * Assigns the description <code>Mount target <i>fsmt-id</i> for file system <i>fs-id</i> </code> where
     * <code> <i>fsmt-id</i> </code> is the mount target ID, and <code> <i>fs-id</i> </code> is the
     * <code>FileSystemId</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Sets the <code>requesterManaged</code> property of the network interface to <code>true</code>, and the
     * <code>requesterId</code> value to <code>EFS</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Each Amazon EFS mount target has one corresponding requester-managed EC2 network interface. After the network
     * interface is created, Amazon EFS sets the <code>NetworkInterfaceId</code> field in the mount target's description
     * to the network interface ID, and the <code>IpAddress</code> field to its address. If network interface creation
     * fails, the entire <code>CreateMountTarget</code> operation fails.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The <code>CreateMountTarget</code> call returns only after creating the network interface, but while the mount
     * target state is still <code>creating</code>, you can check the mount target creation status by calling the
     * <a>DescribeMountTargets</a> operation, which among other things returns the mount target state.
     * </p>
     * </note>
     * <p>
     * We recommend that you create a mount target in each of the Availability Zones. There are cost considerations for
     * using a file system in an Availability Zone through a mount target created in another Availability Zone. For more
     * information, see <a href="http://aws.amazon.com/efs/">Amazon EFS</a>. In addition, by always using a mount target
     * local to the instance's Availability Zone, you eliminate a partial failure scenario. If the Availability Zone in
     * which your mount target is created goes down, then you can't access your file system through that mount target.
     * </p>
     * <p>
     * This operation requires permissions for the following action on the file system:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>elasticfilesystem:CreateMountTarget</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * This operation also requires permissions for the following Amazon EC2 actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ec2:DescribeSubnets</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:DescribeNetworkInterfaces</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:CreateNetworkInterface</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param createMountTargetRequest
     * @return Result of the CreateMountTarget operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws FileSystemNotFoundException
     *         Returned if the specified <code>FileSystemId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws IncorrectFileSystemLifeCycleStateException
     *         Returned if the file system's lifecycle state is not "available".
     * @throws MountTargetConflictException
     *         Returned if the mount target would violate one of the specified restrictions based on the file system's
     *         existing mount targets.
     * @throws SubnetNotFoundException
     *         Returned if there is no subnet with ID <code>SubnetId</code> provided in the request.
     * @throws NoFreeAddressesInSubnetException
     *         Returned if <code>IpAddress</code> was not specified in the request and there are no free IP addresses in
     *         the subnet.
     * @throws IpAddressInUseException
     *         Returned if the request specified an <code>IpAddress</code> that is already in use in the subnet.
     * @throws NetworkInterfaceLimitExceededException
     *         The calling account has reached the limit for elastic network interfaces for the specific Amazon Web
     *         Services Region. Either delete some network interfaces or request that the account quota be raised. For
     *         more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/VPC_Appendix_Limits.html">Amazon VPC
     *         Quotas</a> in the <i>Amazon VPC User Guide</i> (see the <b>Network interfaces per Region</b> entry in the
     *         <b>Network interfaces</b> table).
     * @throws SecurityGroupLimitExceededException
     *         Returned if the size of <code>SecurityGroups</code> specified in the request is greater than five.
     * @throws SecurityGroupNotFoundException
     *         Returned if one of the specified security groups doesn't exist in the subnet's virtual private cloud
     *         (VPC).
     * @throws UnsupportedAvailabilityZoneException
     *         Returned if the requested Amazon EFS functionality is not available in the specified Availability Zone.
     * @throws AvailabilityZonesMismatchException
     *         Returned if the Availability Zone that was specified for a mount target is different from the
     *         Availability Zone that was specified for One Zone storage. For more information, see <a
     *         href="https://docs.aws.amazon.com/efs/latest/ug/availability-durability.html">Regional and One Zone
     *         storage redundancy</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.CreateMountTarget
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/CreateMountTarget"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateMountTargetResponse createMountTarget(CreateMountTargetRequest createMountTargetRequest)
            throws BadRequestException, InternalServerErrorException, FileSystemNotFoundException,
            IncorrectFileSystemLifeCycleStateException, MountTargetConflictException, SubnetNotFoundException,
            NoFreeAddressesInSubnetException, IpAddressInUseException, NetworkInterfaceLimitExceededException,
            SecurityGroupLimitExceededException, SecurityGroupNotFoundException, UnsupportedAvailabilityZoneException,
            AvailabilityZonesMismatchException, AwsServiceException, SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateMountTargetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateMountTargetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMountTargetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMountTarget");

            return clientHandler.execute(new ClientExecutionParams<CreateMountTargetRequest, CreateMountTargetResponse>()
                    .withOperationName("CreateMountTarget").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createMountTargetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateMountTargetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a replication configuration that replicates an existing EFS file system to a new, read-only file system.
     * For more information, see <a href="https://docs.aws.amazon.com/efs/latest/ug/efs-replication.html">Amazon EFS
     * replication</a> in the <i>Amazon EFS User Guide</i>. The replication configuration specifies the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Source file system</b> - An existing EFS file system that you want replicated. The source file system cannot
     * be a destination file system in an existing replication configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Destination file system configuration</b> - The configuration of the destination file system to which the
     * source file system will be replicated. There can only be one destination file system in a replication
     * configuration. The destination file system configuration consists of the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Amazon Web Services Region</b> - The Amazon Web Services Region in which the destination file system is
     * created. Amazon EFS replication is available in all Amazon Web Services Regions that Amazon EFS is available in,
     * except Africa (Cape Town), Asia Pacific (Hong Kong), Asia Pacific (Jakarta), Europe (Milan), and Middle East
     * (Bahrain).
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Availability Zone</b> - If you want the destination file system to use EFS One Zone availability and
     * durability, you must specify the Availability Zone to create the file system in. For more information about EFS
     * storage classes, see <a href="https://docs.aws.amazon.com/efs/latest/ug/storage-classes.html"> Amazon EFS storage
     * classes</a> in the <i>Amazon EFS User Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Encryption</b> - All destination file systems are created with encryption at rest enabled. You can specify the
     * Key Management Service (KMS) key that is used to encrypt the destination file system. If you don't specify a KMS
     * key, your service-managed KMS key for Amazon EFS is used.
     * </p>
     * <note>
     * <p>
     * After the file system is created, you cannot change the KMS key.
     * </p>
     * </note></li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * The following properties are set by default:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Performance mode</b> - The destination file system's performance mode matches that of the source file system,
     * unless the destination file system uses EFS One Zone storage. In that case, the General Purpose performance mode
     * is used. The performance mode cannot be changed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Throughput mode</b> - The destination file system uses the Bursting Throughput mode by default. After the file
     * system is created, you can modify the throughput mode.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The following properties are turned off by default:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Lifecycle management</b> - EFS lifecycle management and EFS Intelligent-Tiering are not enabled on the
     * destination file system. After the destination file system is created, you can enable EFS lifecycle management
     * and EFS Intelligent-Tiering.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Automatic backups</b> - Automatic daily backups not enabled on the destination file system. After the file
     * system is created, you can change this setting.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/efs/latest/ug/efs-replication.html">Amazon EFS
     * replication</a> in the <i>Amazon EFS User Guide</i>.
     * </p>
     *
     * @param createReplicationConfigurationRequest
     * @return Result of the CreateReplicationConfiguration operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws IncorrectFileSystemLifeCycleStateException
     *         Returned if the file system's lifecycle state is not "available".
     * @throws ValidationException
     *         Returned if the Backup service is not available in the Amazon Web Services Region in which the request
     *         was made.
     * @throws ReplicationNotFoundException
     *         Returned if the specified file system does not have a replication configuration.
     * @throws FileSystemNotFoundException
     *         Returned if the specified <code>FileSystemId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws UnsupportedAvailabilityZoneException
     *         Returned if the requested Amazon EFS functionality is not available in the specified Availability Zone.
     * @throws FileSystemLimitExceededException
     *         Returned if the Amazon Web Services account has already created the maximum number of file systems
     *         allowed per account.
     * @throws InsufficientThroughputCapacityException
     *         Returned if there's not enough capacity to provision additional throughput. This value might be returned
     *         when you try to create a file system in provisioned throughput mode, when you attempt to increase the
     *         provisioned throughput of an existing file system, or when you attempt to change an existing file system
     *         from Bursting Throughput to Provisioned Throughput mode. Try again later.
     * @throws ThroughputLimitExceededException
     *         Returned if the throughput mode or amount of provisioned throughput can't be changed because the
     *         throughput limit of 1024 MiB/s has been reached.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.CreateReplicationConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/CreateReplicationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateReplicationConfigurationResponse createReplicationConfiguration(
            CreateReplicationConfigurationRequest createReplicationConfigurationRequest) throws BadRequestException,
            IncorrectFileSystemLifeCycleStateException, ValidationException, ReplicationNotFoundException,
            FileSystemNotFoundException, UnsupportedAvailabilityZoneException, FileSystemLimitExceededException,
            InsufficientThroughputCapacityException, ThroughputLimitExceededException, InternalServerErrorException,
            AwsServiceException, SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateReplicationConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateReplicationConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createReplicationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateReplicationConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateReplicationConfigurationRequest, CreateReplicationConfigurationResponse>()
                            .withOperationName("CreateReplicationConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createReplicationConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateReplicationConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <note>
     * <p>
     * DEPRECATED - <code>CreateTags</code> is deprecated and not maintained. To create tags for EFS resources, use the
     * API action.
     * </p>
     * </note>
     * <p>
     * Creates or overwrites tags associated with a file system. Each tag is a key-value pair. If a tag key specified in
     * the request already exists on the file system, this operation overwrites its value with the value provided in the
     * request. If you add the <code>Name</code> tag to your file system, Amazon EFS returns it in the response to the
     * <a>DescribeFileSystems</a> operation.
     * </p>
     * <p>
     * This operation requires permission for the <code>elasticfilesystem:CreateTags</code> action.
     * </p>
     *
     * @param createTagsRequest
     * @return Result of the CreateTags operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws FileSystemNotFoundException
     *         Returned if the specified <code>FileSystemId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.CreateTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/CreateTags" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateTagsResponse createTags(CreateTagsRequest createTagsRequest) throws BadRequestException,
            InternalServerErrorException, FileSystemNotFoundException, AwsServiceException, SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTags");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateTagsRequest, CreateTagsResponse>().withOperationName("CreateTags")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createTagsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateTagsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified access point. After deletion is complete, new clients can no longer connect to the access
     * points. Clients connected to the access point at the time of deletion will continue to function until they
     * terminate their connection.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DeleteAccessPoint</code> action.
     * </p>
     *
     * @param deleteAccessPointRequest
     * @return Result of the DeleteAccessPoint operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws AccessPointNotFoundException
     *         Returned if the specified <code>AccessPointId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.DeleteAccessPoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DeleteAccessPoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteAccessPointResponse deleteAccessPoint(DeleteAccessPointRequest deleteAccessPointRequest)
            throws BadRequestException, InternalServerErrorException, AccessPointNotFoundException, AwsServiceException,
            SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAccessPointResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteAccessPointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAccessPointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAccessPoint");

            return clientHandler.execute(new ClientExecutionParams<DeleteAccessPointRequest, DeleteAccessPointResponse>()
                    .withOperationName("DeleteAccessPoint").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteAccessPointRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteAccessPointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a file system, permanently severing access to its contents. Upon return, the file system no longer exists
     * and you can't access any contents of the deleted file system.
     * </p>
     * <p>
     * You need to manually delete mount targets attached to a file system before you can delete an EFS file system.
     * This step is performed for you when you use the Amazon Web Services console to delete a file system.
     * </p>
     * <note>
     * <p>
     * You cannot delete a file system that is part of an EFS Replication configuration. You need to delete the
     * replication configuration first.
     * </p>
     * </note>
     * <p>
     * You can't delete a file system that is in use. That is, if the file system has any mount targets, you must first
     * delete them. For more information, see <a>DescribeMountTargets</a> and <a>DeleteMountTarget</a>.
     * </p>
     * <note>
     * <p>
     * The <code>DeleteFileSystem</code> call returns while the file system state is still <code>deleting</code>. You
     * can check the file system deletion status by calling the <a>DescribeFileSystems</a> operation, which returns a
     * list of file systems in your account. If you pass file system ID or creation token for the deleted file system,
     * the <a>DescribeFileSystems</a> returns a <code>404 FileSystemNotFound</code> error.
     * </p>
     * </note>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DeleteFileSystem</code> action.
     * </p>
     *
     * @param deleteFileSystemRequest
     * @return Result of the DeleteFileSystem operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws FileSystemNotFoundException
     *         Returned if the specified <code>FileSystemId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws FileSystemInUseException
     *         Returned if a file system has mount targets.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.DeleteFileSystem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DeleteFileSystem"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteFileSystemResponse deleteFileSystem(DeleteFileSystemRequest deleteFileSystemRequest) throws BadRequestException,
            InternalServerErrorException, FileSystemNotFoundException, FileSystemInUseException, AwsServiceException,
            SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteFileSystemResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteFileSystemResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFileSystemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFileSystem");

            return clientHandler.execute(new ClientExecutionParams<DeleteFileSystemRequest, DeleteFileSystemResponse>()
                    .withOperationName("DeleteFileSystem").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteFileSystemRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteFileSystemRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the <code>FileSystemPolicy</code> for the specified file system. The default
     * <code>FileSystemPolicy</code> goes into effect once the existing policy is deleted. For more information about
     * the default file system policy, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/res-based-policies-efs.html">Using Resource-based Policies with
     * EFS</a>.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DeleteFileSystemPolicy</code> action.
     * </p>
     *
     * @param deleteFileSystemPolicyRequest
     * @return Result of the DeleteFileSystemPolicy operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws FileSystemNotFoundException
     *         Returned if the specified <code>FileSystemId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws IncorrectFileSystemLifeCycleStateException
     *         Returned if the file system's lifecycle state is not "available".
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.DeleteFileSystemPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DeleteFileSystemPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteFileSystemPolicyResponse deleteFileSystemPolicy(DeleteFileSystemPolicyRequest deleteFileSystemPolicyRequest)
            throws BadRequestException, InternalServerErrorException, FileSystemNotFoundException,
            IncorrectFileSystemLifeCycleStateException, AwsServiceException, SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteFileSystemPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteFileSystemPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFileSystemPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFileSystemPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteFileSystemPolicyRequest, DeleteFileSystemPolicyResponse>()
                            .withOperationName("DeleteFileSystemPolicy").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteFileSystemPolicyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteFileSystemPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified mount target.
     * </p>
     * <p>
     * This operation forcibly breaks any mounts of the file system by using the mount target that is being deleted,
     * which might disrupt instances or applications using those mounts. To avoid applications getting cut off abruptly,
     * you might consider unmounting any mounts of the mount target, if feasible. The operation also deletes the
     * associated network interface. Uncommitted writes might be lost, but breaking a mount target using this operation
     * does not corrupt the file system itself. The file system you created remains. You can mount an EC2 instance in
     * your VPC by using another mount target.
     * </p>
     * <p>
     * This operation requires permissions for the following action on the file system:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>elasticfilesystem:DeleteMountTarget</code>
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The <code>DeleteMountTarget</code> call returns while the mount target state is still <code>deleting</code>. You
     * can check the mount target deletion by calling the <a>DescribeMountTargets</a> operation, which returns a list of
     * mount target descriptions for the given file system.
     * </p>
     * </note>
     * <p>
     * The operation also requires permissions for the following Amazon EC2 action on the mount target's network
     * interface:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ec2:DeleteNetworkInterface</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteMountTargetRequest
     * @return Result of the DeleteMountTarget operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws DependencyTimeoutException
     *         The service timed out trying to fulfill the request, and the client should try the call again.
     * @throws MountTargetNotFoundException
     *         Returned if there is no mount target with the specified ID found in the caller's Amazon Web Services
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.DeleteMountTarget
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DeleteMountTarget"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteMountTargetResponse deleteMountTarget(DeleteMountTargetRequest deleteMountTargetRequest)
            throws BadRequestException, InternalServerErrorException, DependencyTimeoutException, MountTargetNotFoundException,
            AwsServiceException, SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteMountTargetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteMountTargetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMountTargetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMountTarget");

            return clientHandler.execute(new ClientExecutionParams<DeleteMountTargetRequest, DeleteMountTargetResponse>()
                    .withOperationName("DeleteMountTarget").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteMountTargetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteMountTargetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an existing replication configuration. To delete a replication configuration, you must make the request
     * from the Amazon Web Services Region in which the destination file system is located. Deleting a replication
     * configuration ends the replication process. After a replication configuration is deleted, the destination file
     * system is no longer read-only. You can write to the destination file system after its status becomes
     * <code>Writeable</code>.
     * </p>
     *
     * @param deleteReplicationConfigurationRequest
     * @return Result of the DeleteReplicationConfiguration operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws FileSystemNotFoundException
     *         Returned if the specified <code>FileSystemId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws ReplicationNotFoundException
     *         Returned if the specified file system does not have a replication configuration.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.DeleteReplicationConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DeleteReplicationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteReplicationConfigurationResponse deleteReplicationConfiguration(
            DeleteReplicationConfigurationRequest deleteReplicationConfigurationRequest) throws BadRequestException,
            InternalServerErrorException, FileSystemNotFoundException, ReplicationNotFoundException, AwsServiceException,
            SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteReplicationConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteReplicationConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteReplicationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteReplicationConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteReplicationConfigurationRequest, DeleteReplicationConfigurationResponse>()
                            .withOperationName("DeleteReplicationConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteReplicationConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteReplicationConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <note>
     * <p>
     * DEPRECATED - <code>DeleteTags</code> is deprecated and not maintained. To remove tags from EFS resources, use the
     * API action.
     * </p>
     * </note>
     * <p>
     * Deletes the specified tags from a file system. If the <code>DeleteTags</code> request includes a tag key that
     * doesn't exist, Amazon EFS ignores it and doesn't cause an error. For more information about tags and related
     * restrictions, see <a href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html">Tag
     * restrictions</a> in the <i>Billing and Cost Management User Guide</i>.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DeleteTags</code> action.
     * </p>
     *
     * @param deleteTagsRequest
     * @return Result of the DeleteTags operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws FileSystemNotFoundException
     *         Returned if the specified <code>FileSystemId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.DeleteTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DeleteTags" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteTagsResponse deleteTags(DeleteTagsRequest deleteTagsRequest) throws BadRequestException,
            InternalServerErrorException, FileSystemNotFoundException, AwsServiceException, SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTags");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteTagsRequest, DeleteTagsResponse>().withOperationName("DeleteTags")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteTagsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteTagsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the description of a specific Amazon EFS access point if the <code>AccessPointId</code> is provided. If
     * you provide an EFS <code>FileSystemId</code>, it returns descriptions of all access points for that file system.
     * You can provide either an <code>AccessPointId</code> or a <code>FileSystemId</code> in the request, but not both.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeAccessPoints</code> action.
     * </p>
     *
     * @param describeAccessPointsRequest
     * @return Result of the DescribeAccessPoints operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws FileSystemNotFoundException
     *         Returned if the specified <code>FileSystemId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws AccessPointNotFoundException
     *         Returned if the specified <code>AccessPointId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.DescribeAccessPoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeAccessPoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAccessPointsResponse describeAccessPoints(DescribeAccessPointsRequest describeAccessPointsRequest)
            throws BadRequestException, InternalServerErrorException, FileSystemNotFoundException, AccessPointNotFoundException,
            AwsServiceException, SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAccessPointsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeAccessPointsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAccessPointsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAccessPoints");

            return clientHandler.execute(new ClientExecutionParams<DescribeAccessPointsRequest, DescribeAccessPointsResponse>()
                    .withOperationName("DescribeAccessPoints").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeAccessPointsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeAccessPointsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the description of a specific Amazon EFS access point if the <code>AccessPointId</code> is provided. If
     * you provide an EFS <code>FileSystemId</code>, it returns descriptions of all access points for that file system.
     * You can provide either an <code>AccessPointId</code> or a <code>FileSystemId</code> in the request, but not both.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeAccessPoints</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeAccessPoints(software.amazon.awssdk.services.efs.model.DescribeAccessPointsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeAccessPointsIterable responses = client.describeAccessPointsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.efs.paginators.DescribeAccessPointsIterable responses = client
     *             .describeAccessPointsPaginator(request);
     *     for (software.amazon.awssdk.services.efs.model.DescribeAccessPointsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeAccessPointsIterable responses = client.describeAccessPointsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeAccessPoints(software.amazon.awssdk.services.efs.model.DescribeAccessPointsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeAccessPointsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws FileSystemNotFoundException
     *         Returned if the specified <code>FileSystemId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws AccessPointNotFoundException
     *         Returned if the specified <code>AccessPointId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.DescribeAccessPoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeAccessPoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAccessPointsIterable describeAccessPointsPaginator(DescribeAccessPointsRequest describeAccessPointsRequest)
            throws BadRequestException, InternalServerErrorException, FileSystemNotFoundException, AccessPointNotFoundException,
            AwsServiceException, SdkClientException, EfsException {
        return new DescribeAccessPointsIterable(this, applyPaginatorUserAgent(describeAccessPointsRequest));
    }

    /**
     * <p>
     * Returns the account preferences settings for the Amazon Web Services account associated with the user making the
     * request, in the current Amazon Web Services Region. For more information, see <a
     * href="efs/latest/ug/manage-efs-resource-ids.html">Managing Amazon EFS resource IDs</a>.
     * </p>
     *
     * @param describeAccountPreferencesRequest
     * @return Result of the DescribeAccountPreferences operation returned by the service.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.DescribeAccountPreferences
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeAccountPreferences"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAccountPreferencesResponse describeAccountPreferences(
            DescribeAccountPreferencesRequest describeAccountPreferencesRequest) throws InternalServerErrorException,
            AwsServiceException, SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAccountPreferencesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeAccountPreferencesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAccountPreferencesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAccountPreferences");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeAccountPreferencesRequest, DescribeAccountPreferencesResponse>()
                            .withOperationName("DescribeAccountPreferences").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeAccountPreferencesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeAccountPreferencesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the backup policy for the specified EFS file system.
     * </p>
     *
     * @param describeBackupPolicyRequest
     * @return Result of the DescribeBackupPolicy operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws FileSystemNotFoundException
     *         Returned if the specified <code>FileSystemId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws PolicyNotFoundException
     *         Returned if the default file system policy is in effect for the EFS file system specified.
     * @throws ValidationException
     *         Returned if the Backup service is not available in the Amazon Web Services Region in which the request
     *         was made.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.DescribeBackupPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeBackupPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeBackupPolicyResponse describeBackupPolicy(DescribeBackupPolicyRequest describeBackupPolicyRequest)
            throws BadRequestException, FileSystemNotFoundException, InternalServerErrorException, PolicyNotFoundException,
            ValidationException, AwsServiceException, SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeBackupPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeBackupPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBackupPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBackupPolicy");

            return clientHandler.execute(new ClientExecutionParams<DescribeBackupPolicyRequest, DescribeBackupPolicyResponse>()
                    .withOperationName("DescribeBackupPolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeBackupPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeBackupPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the <code>FileSystemPolicy</code> for the specified EFS file system.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeFileSystemPolicy</code> action.
     * </p>
     *
     * @param describeFileSystemPolicyRequest
     * @return Result of the DescribeFileSystemPolicy operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws FileSystemNotFoundException
     *         Returned if the specified <code>FileSystemId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws PolicyNotFoundException
     *         Returned if the default file system policy is in effect for the EFS file system specified.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.DescribeFileSystemPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeFileSystemPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeFileSystemPolicyResponse describeFileSystemPolicy(
            DescribeFileSystemPolicyRequest describeFileSystemPolicyRequest) throws BadRequestException,
            InternalServerErrorException, FileSystemNotFoundException, PolicyNotFoundException, AwsServiceException,
            SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeFileSystemPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeFileSystemPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFileSystemPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFileSystemPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeFileSystemPolicyRequest, DescribeFileSystemPolicyResponse>()
                            .withOperationName("DescribeFileSystemPolicy").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeFileSystemPolicyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeFileSystemPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the description of a specific Amazon EFS file system if either the file system <code>CreationToken</code>
     * or the <code>FileSystemId</code> is provided. Otherwise, it returns descriptions of all file systems owned by the
     * caller's Amazon Web Services account in the Amazon Web Services Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all file system descriptions, you can optionally specify the <code>MaxItems</code> parameter to
     * limit the number of descriptions in a response. Currently, this number is automatically set to 10. If more file
     * system descriptions remain, Amazon EFS returns a <code>NextMarker</code>, an opaque token, in the response. In
     * this case, you should send a subsequent request with the <code>Marker</code> request parameter set to the value
     * of <code>NextMarker</code>.
     * </p>
     * <p>
     * To retrieve a list of your file system descriptions, this operation is used in an iterative process, where
     * <code>DescribeFileSystems</code> is called first without the <code>Marker</code> and then the operation continues
     * to call it with the <code>Marker</code> parameter set to the value of the <code>NextMarker</code> from the
     * previous response until the response has no <code>NextMarker</code>.
     * </p>
     * <p>
     * The order of file systems returned in the response of one <code>DescribeFileSystems</code> call and the order of
     * file systems returned across the responses of a multi-call iteration is unspecified.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeFileSystems</code> action.
     * </p>
     *
     * @param describeFileSystemsRequest
     * @return Result of the DescribeFileSystems operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws FileSystemNotFoundException
     *         Returned if the specified <code>FileSystemId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.DescribeFileSystems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeFileSystems"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeFileSystemsResponse describeFileSystems(DescribeFileSystemsRequest describeFileSystemsRequest)
            throws BadRequestException, InternalServerErrorException, FileSystemNotFoundException, AwsServiceException,
            SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeFileSystemsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeFileSystemsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFileSystemsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFileSystems");

            return clientHandler.execute(new ClientExecutionParams<DescribeFileSystemsRequest, DescribeFileSystemsResponse>()
                    .withOperationName("DescribeFileSystems").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeFileSystemsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeFileSystemsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the description of a specific Amazon EFS file system if either the file system <code>CreationToken</code>
     * or the <code>FileSystemId</code> is provided. Otherwise, it returns descriptions of all file systems owned by the
     * caller's Amazon Web Services account in the Amazon Web Services Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all file system descriptions, you can optionally specify the <code>MaxItems</code> parameter to
     * limit the number of descriptions in a response. Currently, this number is automatically set to 10. If more file
     * system descriptions remain, Amazon EFS returns a <code>NextMarker</code>, an opaque token, in the response. In
     * this case, you should send a subsequent request with the <code>Marker</code> request parameter set to the value
     * of <code>NextMarker</code>.
     * </p>
     * <p>
     * To retrieve a list of your file system descriptions, this operation is used in an iterative process, where
     * <code>DescribeFileSystems</code> is called first without the <code>Marker</code> and then the operation continues
     * to call it with the <code>Marker</code> parameter set to the value of the <code>NextMarker</code> from the
     * previous response until the response has no <code>NextMarker</code>.
     * </p>
     * <p>
     * The order of file systems returned in the response of one <code>DescribeFileSystems</code> call and the order of
     * file systems returned across the responses of a multi-call iteration is unspecified.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeFileSystems</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeFileSystems(software.amazon.awssdk.services.efs.model.DescribeFileSystemsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeFileSystemsIterable responses = client.describeFileSystemsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.efs.paginators.DescribeFileSystemsIterable responses = client
     *             .describeFileSystemsPaginator(request);
     *     for (software.amazon.awssdk.services.efs.model.DescribeFileSystemsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeFileSystemsIterable responses = client.describeFileSystemsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxItems won't limit the number of results you get with the paginator.
     * It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeFileSystems(software.amazon.awssdk.services.efs.model.DescribeFileSystemsRequest)} operation.</b>
     * </p>
     *
     * @param describeFileSystemsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws FileSystemNotFoundException
     *         Returned if the specified <code>FileSystemId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.DescribeFileSystems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeFileSystems"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeFileSystemsIterable describeFileSystemsPaginator(DescribeFileSystemsRequest describeFileSystemsRequest)
            throws BadRequestException, InternalServerErrorException, FileSystemNotFoundException, AwsServiceException,
            SdkClientException, EfsException {
        return new DescribeFileSystemsIterable(this, applyPaginatorUserAgent(describeFileSystemsRequest));
    }

    /**
     * <p>
     * Returns the current <code>LifecycleConfiguration</code> object for the specified Amazon EFS file system. EFS
     * lifecycle management uses the <code>LifecycleConfiguration</code> object to identify which files to move to the
     * EFS Infrequent Access (IA) storage class. For a file system without a <code>LifecycleConfiguration</code> object,
     * the call returns an empty array in the response.
     * </p>
     * <p>
     * When EFS Intelligent-Tiering is enabled, <code>TransitionToPrimaryStorageClass</code> has a value of
     * <code>AFTER_1_ACCESS</code>.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeLifecycleConfiguration</code>
     * operation.
     * </p>
     *
     * @param describeLifecycleConfigurationRequest
     * @return Result of the DescribeLifecycleConfiguration operation returned by the service.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws FileSystemNotFoundException
     *         Returned if the specified <code>FileSystemId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.DescribeLifecycleConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeLifecycleConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeLifecycleConfigurationResponse describeLifecycleConfiguration(
            DescribeLifecycleConfigurationRequest describeLifecycleConfigurationRequest) throws InternalServerErrorException,
            BadRequestException, FileSystemNotFoundException, AwsServiceException, SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeLifecycleConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeLifecycleConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeLifecycleConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLifecycleConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeLifecycleConfigurationRequest, DescribeLifecycleConfigurationResponse>()
                            .withOperationName("DescribeLifecycleConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeLifecycleConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeLifecycleConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the security groups currently in effect for a mount target. This operation requires that the network
     * interface of the mount target has been created and the lifecycle state of the mount target is not
     * <code>deleted</code>.
     * </p>
     * <p>
     * This operation requires permissions for the following actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>elasticfilesystem:DescribeMountTargetSecurityGroups</code> action on the mount target's file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:DescribeNetworkInterfaceAttribute</code> action on the mount target's network interface.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeMountTargetSecurityGroupsRequest
     * @return Result of the DescribeMountTargetSecurityGroups operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws MountTargetNotFoundException
     *         Returned if there is no mount target with the specified ID found in the caller's Amazon Web Services
     *         account.
     * @throws IncorrectMountTargetStateException
     *         Returned if the mount target is not in the correct state for the operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.DescribeMountTargetSecurityGroups
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeMountTargetSecurityGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeMountTargetSecurityGroupsResponse describeMountTargetSecurityGroups(
            DescribeMountTargetSecurityGroupsRequest describeMountTargetSecurityGroupsRequest) throws BadRequestException,
            InternalServerErrorException, MountTargetNotFoundException, IncorrectMountTargetStateException, AwsServiceException,
            SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeMountTargetSecurityGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeMountTargetSecurityGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeMountTargetSecurityGroupsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeMountTargetSecurityGroups");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeMountTargetSecurityGroupsRequest, DescribeMountTargetSecurityGroupsResponse>()
                            .withOperationName("DescribeMountTargetSecurityGroups").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeMountTargetSecurityGroupsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeMountTargetSecurityGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the descriptions of all the current mount targets, or a specific mount target, for a file system. When
     * requesting all of the current mount targets, the order of mount targets returned in the response is unspecified.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeMountTargets</code> action, on either
     * the file system ID that you specify in <code>FileSystemId</code>, or on the file system of the mount target that
     * you specify in <code>MountTargetId</code>.
     * </p>
     *
     * @param describeMountTargetsRequest
     * @return Result of the DescribeMountTargets operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws FileSystemNotFoundException
     *         Returned if the specified <code>FileSystemId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws MountTargetNotFoundException
     *         Returned if there is no mount target with the specified ID found in the caller's Amazon Web Services
     *         account.
     * @throws AccessPointNotFoundException
     *         Returned if the specified <code>AccessPointId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.DescribeMountTargets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeMountTargets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeMountTargetsResponse describeMountTargets(DescribeMountTargetsRequest describeMountTargetsRequest)
            throws BadRequestException, InternalServerErrorException, FileSystemNotFoundException, MountTargetNotFoundException,
            AccessPointNotFoundException, AwsServiceException, SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeMountTargetsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeMountTargetsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeMountTargetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeMountTargets");

            return clientHandler.execute(new ClientExecutionParams<DescribeMountTargetsRequest, DescribeMountTargetsResponse>()
                    .withOperationName("DescribeMountTargets").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeMountTargetsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeMountTargetsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the replication configuration for a specific file system. If a file system is not specified, all of the
     * replication configurations for the Amazon Web Services account in an Amazon Web Services Region are retrieved.
     * </p>
     *
     * @param describeReplicationConfigurationsRequest
     * @return Result of the DescribeReplicationConfigurations operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws FileSystemNotFoundException
     *         Returned if the specified <code>FileSystemId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws ReplicationNotFoundException
     *         Returned if the specified file system does not have a replication configuration.
     * @throws ValidationException
     *         Returned if the Backup service is not available in the Amazon Web Services Region in which the request
     *         was made.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.DescribeReplicationConfigurations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeReplicationConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeReplicationConfigurationsResponse describeReplicationConfigurations(
            DescribeReplicationConfigurationsRequest describeReplicationConfigurationsRequest) throws BadRequestException,
            FileSystemNotFoundException, InternalServerErrorException, ReplicationNotFoundException, ValidationException,
            AwsServiceException, SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeReplicationConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeReplicationConfigurationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeReplicationConfigurationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeReplicationConfigurations");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeReplicationConfigurationsRequest, DescribeReplicationConfigurationsResponse>()
                            .withOperationName("DescribeReplicationConfigurations").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeReplicationConfigurationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeReplicationConfigurationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <note>
     * <p>
     * DEPRECATED - The <code>DescribeTags</code> action is deprecated and not maintained. To view tags associated with
     * EFS resources, use the <code>ListTagsForResource</code> API action.
     * </p>
     * </note>
     * <p>
     * Returns the tags associated with a file system. The order of tags returned in the response of one
     * <code>DescribeTags</code> call and the order of tags returned across the responses of a multiple-call iteration
     * (when using pagination) is unspecified.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeTags</code> action.
     * </p>
     *
     * @param describeTagsRequest
     * @return Result of the DescribeTags operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws FileSystemNotFoundException
     *         Returned if the specified <code>FileSystemId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.DescribeTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeTags"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeTagsResponse describeTags(DescribeTagsRequest describeTagsRequest) throws BadRequestException,
            InternalServerErrorException, FileSystemNotFoundException, AwsServiceException, SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTags");

            return clientHandler.execute(new ClientExecutionParams<DescribeTagsRequest, DescribeTagsResponse>()
                    .withOperationName("DescribeTags").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeTagsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeTagsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <note>
     * <p>
     * DEPRECATED - The <code>DescribeTags</code> action is deprecated and not maintained. To view tags associated with
     * EFS resources, use the <code>ListTagsForResource</code> API action.
     * </p>
     * </note>
     * <p>
     * Returns the tags associated with a file system. The order of tags returned in the response of one
     * <code>DescribeTags</code> call and the order of tags returned across the responses of a multiple-call iteration
     * (when using pagination) is unspecified.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeTags</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #describeTags(software.amazon.awssdk.services.efs.model.DescribeTagsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeTagsIterable responses = client.describeTagsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.efs.paginators.DescribeTagsIterable responses = client.describeTagsPaginator(request);
     *     for (software.amazon.awssdk.services.efs.model.DescribeTagsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeTagsIterable responses = client.describeTagsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxItems won't limit the number of results you get with the paginator.
     * It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeTags(software.amazon.awssdk.services.efs.model.DescribeTagsRequest)} operation.</b>
     * </p>
     *
     * @param describeTagsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws FileSystemNotFoundException
     *         Returned if the specified <code>FileSystemId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.DescribeTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeTags"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeTagsIterable describeTagsPaginator(DescribeTagsRequest describeTagsRequest) throws BadRequestException,
            InternalServerErrorException, FileSystemNotFoundException, AwsServiceException, SdkClientException, EfsException {
        return new DescribeTagsIterable(this, applyPaginatorUserAgent(describeTagsRequest));
    }

    /**
     * <p>
     * Lists all tags for a top-level EFS resource. You must provide the ID of the resource that you want to retrieve
     * the tags for.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeAccessPoints</code> action.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws FileSystemNotFoundException
     *         Returned if the specified <code>FileSystemId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws AccessPointNotFoundException
     *         Returned if the specified <code>AccessPointId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws BadRequestException, InternalServerErrorException, FileSystemNotFoundException, AccessPointNotFoundException,
            AwsServiceException, SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all tags for a top-level EFS resource. You must provide the ID of the resource that you want to retrieve
     * the tags for.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeAccessPoints</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTagsForResource(software.amazon.awssdk.services.efs.model.ListTagsForResourceRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.ListTagsForResourceIterable responses = client.listTagsForResourcePaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.efs.paginators.ListTagsForResourceIterable responses = client
     *             .listTagsForResourcePaginator(request);
     *     for (software.amazon.awssdk.services.efs.model.ListTagsForResourceResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.ListTagsForResourceIterable responses = client.listTagsForResourcePaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTagsForResource(software.amazon.awssdk.services.efs.model.ListTagsForResourceRequest)} operation.</b>
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws FileSystemNotFoundException
     *         Returned if the specified <code>FileSystemId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws AccessPointNotFoundException
     *         Returned if the specified <code>AccessPointId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceIterable listTagsForResourcePaginator(ListTagsForResourceRequest listTagsForResourceRequest)
            throws BadRequestException, InternalServerErrorException, FileSystemNotFoundException, AccessPointNotFoundException,
            AwsServiceException, SdkClientException, EfsException {
        return new ListTagsForResourceIterable(this, applyPaginatorUserAgent(listTagsForResourceRequest));
    }

    /**
     * <p>
     * Modifies the set of security groups in effect for a mount target.
     * </p>
     * <p>
     * When you create a mount target, Amazon EFS also creates a new network interface. For more information, see
     * <a>CreateMountTarget</a>. This operation replaces the security groups in effect for the network interface
     * associated with a mount target, with the <code>SecurityGroups</code> provided in the request. This operation
     * requires that the network interface of the mount target has been created and the lifecycle state of the mount
     * target is not <code>deleted</code>.
     * </p>
     * <p>
     * The operation requires permissions for the following actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>elasticfilesystem:ModifyMountTargetSecurityGroups</code> action on the mount target's file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:ModifyNetworkInterfaceAttribute</code> action on the mount target's network interface.
     * </p>
     * </li>
     * </ul>
     *
     * @param modifyMountTargetSecurityGroupsRequest
     * @return Result of the ModifyMountTargetSecurityGroups operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws MountTargetNotFoundException
     *         Returned if there is no mount target with the specified ID found in the caller's Amazon Web Services
     *         account.
     * @throws IncorrectMountTargetStateException
     *         Returned if the mount target is not in the correct state for the operation.
     * @throws SecurityGroupLimitExceededException
     *         Returned if the size of <code>SecurityGroups</code> specified in the request is greater than five.
     * @throws SecurityGroupNotFoundException
     *         Returned if one of the specified security groups doesn't exist in the subnet's virtual private cloud
     *         (VPC).
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.ModifyMountTargetSecurityGroups
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/ModifyMountTargetSecurityGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ModifyMountTargetSecurityGroupsResponse modifyMountTargetSecurityGroups(
            ModifyMountTargetSecurityGroupsRequest modifyMountTargetSecurityGroupsRequest) throws BadRequestException,
            InternalServerErrorException, MountTargetNotFoundException, IncorrectMountTargetStateException,
            SecurityGroupLimitExceededException, SecurityGroupNotFoundException, AwsServiceException, SdkClientException,
            EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ModifyMountTargetSecurityGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ModifyMountTargetSecurityGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                modifyMountTargetSecurityGroupsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ModifyMountTargetSecurityGroups");

            return clientHandler
                    .execute(new ClientExecutionParams<ModifyMountTargetSecurityGroupsRequest, ModifyMountTargetSecurityGroupsResponse>()
                            .withOperationName("ModifyMountTargetSecurityGroups").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(modifyMountTargetSecurityGroupsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ModifyMountTargetSecurityGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Use this operation to set the account preference in the current Amazon Web Services Region to use long 17
     * character (63 bit) or short 8 character (32 bit) resource IDs for new EFS file system and mount target resources.
     * All existing resource IDs are not affected by any changes you make. You can set the ID preference during the
     * opt-in period as EFS transitions to long resource IDs. For more information, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/manage-efs-resource-ids.html">Managing Amazon EFS resource
     * IDs</a>.
     * </p>
     * <note>
     * <p>
     * Starting in October, 2021, you will receive an error if you try to set the account preference to use the short 8
     * character format resource ID. Contact Amazon Web Services support if you receive an error and must use short IDs
     * for file system and mount target resources.
     * </p>
     * </note>
     *
     * @param putAccountPreferencesRequest
     * @return Result of the PutAccountPreferences operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.PutAccountPreferences
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/PutAccountPreferences"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutAccountPreferencesResponse putAccountPreferences(PutAccountPreferencesRequest putAccountPreferencesRequest)
            throws BadRequestException, InternalServerErrorException, AwsServiceException, SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutAccountPreferencesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutAccountPreferencesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putAccountPreferencesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAccountPreferences");

            return clientHandler.execute(new ClientExecutionParams<PutAccountPreferencesRequest, PutAccountPreferencesResponse>()
                    .withOperationName("PutAccountPreferences").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putAccountPreferencesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutAccountPreferencesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the file system's backup policy. Use this action to start or stop automatic backups of the file system.
     * </p>
     *
     * @param putBackupPolicyRequest
     * @return Result of the PutBackupPolicy operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws FileSystemNotFoundException
     *         Returned if the specified <code>FileSystemId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws IncorrectFileSystemLifeCycleStateException
     *         Returned if the file system's lifecycle state is not "available".
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws ValidationException
     *         Returned if the Backup service is not available in the Amazon Web Services Region in which the request
     *         was made.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.PutBackupPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/PutBackupPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutBackupPolicyResponse putBackupPolicy(PutBackupPolicyRequest putBackupPolicyRequest) throws BadRequestException,
            FileSystemNotFoundException, IncorrectFileSystemLifeCycleStateException, InternalServerErrorException,
            ValidationException, AwsServiceException, SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutBackupPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutBackupPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putBackupPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutBackupPolicy");

            return clientHandler.execute(new ClientExecutionParams<PutBackupPolicyRequest, PutBackupPolicyResponse>()
                    .withOperationName("PutBackupPolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putBackupPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutBackupPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Applies an Amazon EFS <code>FileSystemPolicy</code> to an Amazon EFS file system. A file system policy is an IAM
     * resource-based policy and can contain multiple policy statements. A file system always has exactly one file
     * system policy, which can be the default policy or an explicit policy set or updated using this API operation. EFS
     * file system policies have a 20,000 character limit. When an explicit policy is set, it overrides the default
     * policy. For more information about the default file system policy, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/iam-access-control-nfs-efs.html#default-filesystempolicy">Default
     * EFS File System Policy</a>.
     * </p>
     * <note>
     * <p>
     * EFS file system policies have a 20,000 character limit.
     * </p>
     * </note>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:PutFileSystemPolicy</code> action.
     * </p>
     *
     * @param putFileSystemPolicyRequest
     * @return Result of the PutFileSystemPolicy operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws FileSystemNotFoundException
     *         Returned if the specified <code>FileSystemId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws InvalidPolicyException
     *         Returned if the <code>FileSystemPolicy</code> is malformed or contains an error such as a parameter value
     *         that is not valid or a missing required parameter. Returned in the case of a policy lockout safety check
     *         error.
     * @throws IncorrectFileSystemLifeCycleStateException
     *         Returned if the file system's lifecycle state is not "available".
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.PutFileSystemPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/PutFileSystemPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutFileSystemPolicyResponse putFileSystemPolicy(PutFileSystemPolicyRequest putFileSystemPolicyRequest)
            throws BadRequestException, InternalServerErrorException, FileSystemNotFoundException, InvalidPolicyException,
            IncorrectFileSystemLifeCycleStateException, AwsServiceException, SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutFileSystemPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutFileSystemPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putFileSystemPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutFileSystemPolicy");

            return clientHandler.execute(new ClientExecutionParams<PutFileSystemPolicyRequest, PutFileSystemPolicyResponse>()
                    .withOperationName("PutFileSystemPolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putFileSystemPolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutFileSystemPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Use this action to manage EFS lifecycle management and EFS Intelligent-Tiering. A
     * <code>LifecycleConfiguration</code> consists of one or more <code>LifecyclePolicy</code> objects that define the
     * following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>EFS Lifecycle management</b> - When Amazon EFS automatically transitions files in a file system into the
     * lower-cost EFS Infrequent Access (IA) storage class.
     * </p>
     * <p>
     * To enable EFS Lifecycle management, set the value of <code>TransitionToIA</code> to one of the available options.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>EFS Intelligent-Tiering</b> - When Amazon EFS automatically transitions files from IA back into the file
     * system's primary storage class (EFS Standard or EFS One Zone Standard).
     * </p>
     * <p>
     * To enable EFS Intelligent-Tiering, set the value of <code>TransitionToPrimaryStorageClass</code> to
     * <code>AFTER_1_ACCESS</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/efs/latest/ug/lifecycle-management-efs.html">EFS
     * Lifecycle Management</a>.
     * </p>
     * <p>
     * Each Amazon EFS file system supports one lifecycle configuration, which applies to all files in the file system.
     * If a <code>LifecycleConfiguration</code> object already exists for the specified file system, a
     * <code>PutLifecycleConfiguration</code> call modifies the existing configuration. A
     * <code>PutLifecycleConfiguration</code> call with an empty <code>LifecyclePolicies</code> array in the request
     * body deletes any existing <code>LifecycleConfiguration</code> and turns off lifecycle management and EFS
     * Intelligent-Tiering for the file system.
     * </p>
     * <p>
     * In the request, specify the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The ID for the file system for which you are enabling, disabling, or modifying lifecycle management and EFS
     * Intelligent-Tiering.
     * </p>
     * </li>
     * <li>
     * <p>
     * A <code>LifecyclePolicies</code> array of <code>LifecyclePolicy</code> objects that define when files are moved
     * into IA storage, and when they are moved back to Standard storage.
     * </p>
     * <note>
     * <p>
     * Amazon EFS requires that each <code>LifecyclePolicy</code> object have only have a single transition, so the
     * <code>LifecyclePolicies</code> array needs to be structured with separate <code>LifecyclePolicy</code> objects.
     * See the example requests in the following section for more information.
     * </p>
     * </note></li>
     * </ul>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:PutLifecycleConfiguration</code> operation.
     * </p>
     * <p>
     * To apply a <code>LifecycleConfiguration</code> object to an encrypted file system, you need the same Key
     * Management Service permissions as when you created the encrypted file system.
     * </p>
     *
     * @param putLifecycleConfigurationRequest
     * @return Result of the PutLifecycleConfiguration operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws FileSystemNotFoundException
     *         Returned if the specified <code>FileSystemId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws IncorrectFileSystemLifeCycleStateException
     *         Returned if the file system's lifecycle state is not "available".
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.PutLifecycleConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/PutLifecycleConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutLifecycleConfigurationResponse putLifecycleConfiguration(
            PutLifecycleConfigurationRequest putLifecycleConfigurationRequest) throws BadRequestException,
            InternalServerErrorException, FileSystemNotFoundException, IncorrectFileSystemLifeCycleStateException,
            AwsServiceException, SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutLifecycleConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutLifecycleConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putLifecycleConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutLifecycleConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<PutLifecycleConfigurationRequest, PutLifecycleConfigurationResponse>()
                            .withOperationName("PutLifecycleConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(putLifecycleConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutLifecycleConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a tag for an EFS resource. You can create tags for EFS file systems and access points using this API
     * operation.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:TagResource</code> action.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws FileSystemNotFoundException
     *         Returned if the specified <code>FileSystemId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws AccessPointNotFoundException
     *         Returned if the specified <code>AccessPointId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws BadRequestException,
            InternalServerErrorException, FileSystemNotFoundException, AccessPointNotFoundException, AwsServiceException,
            SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from an EFS resource. You can remove tags from EFS file systems and access points using this API
     * operation.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:UntagResource</code> action.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws FileSystemNotFoundException
     *         Returned if the specified <code>FileSystemId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws AccessPointNotFoundException
     *         Returned if the specified <code>AccessPointId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws BadRequestException,
            InternalServerErrorException, FileSystemNotFoundException, AccessPointNotFoundException, AwsServiceException,
            SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the throughput mode or the amount of provisioned throughput of an existing file system.
     * </p>
     *
     * @param updateFileSystemRequest
     * @return Result of the UpdateFileSystem operation returned by the service.
     * @throws BadRequestException
     *         Returned if the request is malformed or contains an error such as an invalid parameter value or a missing
     *         required parameter.
     * @throws FileSystemNotFoundException
     *         Returned if the specified <code>FileSystemId</code> value doesn't exist in the requester's Amazon Web
     *         Services account.
     * @throws IncorrectFileSystemLifeCycleStateException
     *         Returned if the file system's lifecycle state is not "available".
     * @throws InsufficientThroughputCapacityException
     *         Returned if there's not enough capacity to provision additional throughput. This value might be returned
     *         when you try to create a file system in provisioned throughput mode, when you attempt to increase the
     *         provisioned throughput of an existing file system, or when you attempt to change an existing file system
     *         from Bursting Throughput to Provisioned Throughput mode. Try again later.
     * @throws InternalServerErrorException
     *         Returned if an error occurred on the server side.
     * @throws ThroughputLimitExceededException
     *         Returned if the throughput mode or amount of provisioned throughput can't be changed because the
     *         throughput limit of 1024 MiB/s has been reached.
     * @throws TooManyRequestsException
     *         Returned if you don’t wait at least 24 hours before either changing the throughput mode, or decreasing
     *         the Provisioned Throughput value.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EfsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EfsClient.UpdateFileSystem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/UpdateFileSystem"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateFileSystemResponse updateFileSystem(UpdateFileSystemRequest updateFileSystemRequest) throws BadRequestException,
            FileSystemNotFoundException, IncorrectFileSystemLifeCycleStateException, InsufficientThroughputCapacityException,
            InternalServerErrorException, ThroughputLimitExceededException, TooManyRequestsException, AwsServiceException,
            SdkClientException, EfsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateFileSystemResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateFileSystemResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFileSystemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "EFS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFileSystem");

            return clientHandler.execute(new ClientExecutionParams<UpdateFileSystemRequest, UpdateFileSystemResponse>()
                    .withOperationName("UpdateFileSystem").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateFileSystemRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateFileSystemRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(EfsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IpAddressInUse")
                                .exceptionBuilderSupplier(IpAddressInUseException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FileSystemInUse")
                                .exceptionBuilderSupplier(FileSystemInUseException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FileSystemAlreadyExists")
                                .exceptionBuilderSupplier(FileSystemAlreadyExistsException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessPointAlreadyExists")
                                .exceptionBuilderSupplier(AccessPointAlreadyExistsException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IncorrectFileSystemLifeCycleState")
                                .exceptionBuilderSupplier(IncorrectFileSystemLifeCycleStateException::builder)
                                .httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessPointLimitExceeded")
                                .exceptionBuilderSupplier(AccessPointLimitExceededException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoFreeAddressesInSubnet")
                                .exceptionBuilderSupplier(NoFreeAddressesInSubnetException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FileSystemNotFound")
                                .exceptionBuilderSupplier(FileSystemNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InsufficientThroughputCapacity")
                                .exceptionBuilderSupplier(InsufficientThroughputCapacityException::builder).httpStatusCode(503)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SecurityGroupLimitExceeded")
                                .exceptionBuilderSupplier(SecurityGroupLimitExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PolicyNotFound")
                                .exceptionBuilderSupplier(PolicyNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FileSystemLimitExceeded")
                                .exceptionBuilderSupplier(FileSystemLimitExceededException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedAvailabilityZone")
                                .exceptionBuilderSupplier(UnsupportedAvailabilityZoneException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessPointNotFound")
                                .exceptionBuilderSupplier(AccessPointNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MountTargetConflict")
                                .exceptionBuilderSupplier(MountTargetConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPolicyException")
                                .exceptionBuilderSupplier(InvalidPolicyException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SecurityGroupNotFound")
                                .exceptionBuilderSupplier(SecurityGroupNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NetworkInterfaceLimitExceeded")
                                .exceptionBuilderSupplier(NetworkInterfaceLimitExceededException::builder).httpStatusCode(409)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ReplicationNotFound")
                                .exceptionBuilderSupplier(ReplicationNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MountTargetNotFound")
                                .exceptionBuilderSupplier(MountTargetNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SubnetNotFound")
                                .exceptionBuilderSupplier(SubnetNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IncorrectMountTargetState")
                                .exceptionBuilderSupplier(IncorrectMountTargetStateException::builder).httpStatusCode(409)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThroughputLimitExceeded")
                                .exceptionBuilderSupplier(ThroughputLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DependencyTimeout")
                                .exceptionBuilderSupplier(DependencyTimeoutException::builder).httpStatusCode(504).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AvailabilityZonesMismatch")
                                .exceptionBuilderSupplier(AvailabilityZonesMismatchException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequests")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequest")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerError")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends EfsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
