/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.efs;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.services.efs.model.CreateAccessPointRequest;
import software.amazon.awssdk.services.efs.model.CreateAccessPointResponse;
import software.amazon.awssdk.services.efs.model.CreateFileSystemRequest;
import software.amazon.awssdk.services.efs.model.CreateFileSystemResponse;
import software.amazon.awssdk.services.efs.model.CreateMountTargetRequest;
import software.amazon.awssdk.services.efs.model.CreateMountTargetResponse;
import software.amazon.awssdk.services.efs.model.CreateReplicationConfigurationRequest;
import software.amazon.awssdk.services.efs.model.CreateReplicationConfigurationResponse;
import software.amazon.awssdk.services.efs.model.CreateTagsRequest;
import software.amazon.awssdk.services.efs.model.CreateTagsResponse;
import software.amazon.awssdk.services.efs.model.DeleteAccessPointRequest;
import software.amazon.awssdk.services.efs.model.DeleteAccessPointResponse;
import software.amazon.awssdk.services.efs.model.DeleteFileSystemPolicyRequest;
import software.amazon.awssdk.services.efs.model.DeleteFileSystemPolicyResponse;
import software.amazon.awssdk.services.efs.model.DeleteFileSystemRequest;
import software.amazon.awssdk.services.efs.model.DeleteFileSystemResponse;
import software.amazon.awssdk.services.efs.model.DeleteMountTargetRequest;
import software.amazon.awssdk.services.efs.model.DeleteMountTargetResponse;
import software.amazon.awssdk.services.efs.model.DeleteReplicationConfigurationRequest;
import software.amazon.awssdk.services.efs.model.DeleteReplicationConfigurationResponse;
import software.amazon.awssdk.services.efs.model.DeleteTagsRequest;
import software.amazon.awssdk.services.efs.model.DeleteTagsResponse;
import software.amazon.awssdk.services.efs.model.DescribeAccessPointsRequest;
import software.amazon.awssdk.services.efs.model.DescribeAccessPointsResponse;
import software.amazon.awssdk.services.efs.model.DescribeAccountPreferencesRequest;
import software.amazon.awssdk.services.efs.model.DescribeAccountPreferencesResponse;
import software.amazon.awssdk.services.efs.model.DescribeBackupPolicyRequest;
import software.amazon.awssdk.services.efs.model.DescribeBackupPolicyResponse;
import software.amazon.awssdk.services.efs.model.DescribeFileSystemPolicyRequest;
import software.amazon.awssdk.services.efs.model.DescribeFileSystemPolicyResponse;
import software.amazon.awssdk.services.efs.model.DescribeFileSystemsRequest;
import software.amazon.awssdk.services.efs.model.DescribeFileSystemsResponse;
import software.amazon.awssdk.services.efs.model.DescribeLifecycleConfigurationRequest;
import software.amazon.awssdk.services.efs.model.DescribeLifecycleConfigurationResponse;
import software.amazon.awssdk.services.efs.model.DescribeMountTargetSecurityGroupsRequest;
import software.amazon.awssdk.services.efs.model.DescribeMountTargetSecurityGroupsResponse;
import software.amazon.awssdk.services.efs.model.DescribeMountTargetsRequest;
import software.amazon.awssdk.services.efs.model.DescribeMountTargetsResponse;
import software.amazon.awssdk.services.efs.model.DescribeReplicationConfigurationsRequest;
import software.amazon.awssdk.services.efs.model.DescribeReplicationConfigurationsResponse;
import software.amazon.awssdk.services.efs.model.DescribeTagsRequest;
import software.amazon.awssdk.services.efs.model.DescribeTagsResponse;
import software.amazon.awssdk.services.efs.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.efs.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.efs.model.ModifyMountTargetSecurityGroupsRequest;
import software.amazon.awssdk.services.efs.model.ModifyMountTargetSecurityGroupsResponse;
import software.amazon.awssdk.services.efs.model.PutAccountPreferencesRequest;
import software.amazon.awssdk.services.efs.model.PutAccountPreferencesResponse;
import software.amazon.awssdk.services.efs.model.PutBackupPolicyRequest;
import software.amazon.awssdk.services.efs.model.PutBackupPolicyResponse;
import software.amazon.awssdk.services.efs.model.PutFileSystemPolicyRequest;
import software.amazon.awssdk.services.efs.model.PutFileSystemPolicyResponse;
import software.amazon.awssdk.services.efs.model.PutLifecycleConfigurationRequest;
import software.amazon.awssdk.services.efs.model.PutLifecycleConfigurationResponse;
import software.amazon.awssdk.services.efs.model.TagResourceRequest;
import software.amazon.awssdk.services.efs.model.TagResourceResponse;
import software.amazon.awssdk.services.efs.model.UntagResourceRequest;
import software.amazon.awssdk.services.efs.model.UntagResourceResponse;
import software.amazon.awssdk.services.efs.model.UpdateFileSystemProtectionRequest;
import software.amazon.awssdk.services.efs.model.UpdateFileSystemProtectionResponse;
import software.amazon.awssdk.services.efs.model.UpdateFileSystemRequest;
import software.amazon.awssdk.services.efs.model.UpdateFileSystemResponse;
import software.amazon.awssdk.services.efs.paginators.DescribeAccessPointsPublisher;
import software.amazon.awssdk.services.efs.paginators.DescribeFileSystemsPublisher;
import software.amazon.awssdk.services.efs.paginators.DescribeMountTargetsPublisher;
import software.amazon.awssdk.services.efs.paginators.DescribeReplicationConfigurationsPublisher;
import software.amazon.awssdk.services.efs.paginators.DescribeTagsPublisher;
import software.amazon.awssdk.services.efs.paginators.ListTagsForResourcePublisher;

/**
 * Service client for accessing EFS asynchronously. This can be created using the static {@link #builder()} method.The
 * asynchronous client performs non-blocking I/O when configured with any {@link SdkAsyncHttpClient} supported in the
 * SDK. However, full non-blocking is not guaranteed as the async client may perform blocking calls in some cases such
 * as credentials retrieval and endpoint discovery as part of the async API call.
 *
 * <fullname>Amazon Elastic File System</fullname>
 * <p>
 * Amazon Elastic File System (Amazon EFS) provides simple, scalable file storage for use with Amazon EC2 Linux and Mac
 * instances in the Amazon Web Services Cloud. With Amazon EFS, storage capacity is elastic, growing and shrinking
 * automatically as you add and remove files, so that your applications have the storage they need, when they need it.
 * For more information, see the <a href="https://docs.aws.amazon.com/efs/latest/ug/api-reference.html">Amazon Elastic
 * File System API Reference</a> and the <a href="https://docs.aws.amazon.com/efs/latest/ug/whatisefs.html">Amazon
 * Elastic File System User Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface EfsAsyncClient extends AwsClient {
    String SERVICE_NAME = "elasticfilesystem";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "elasticfilesystem";

    /**
     * <p>
     * Creates an EFS access point. An access point is an application-specific view into an EFS file system that applies
     * an operating system user and group, and a file system path, to any file system request made through the access
     * point. The operating system user and group override any identity information provided by the NFS client. The file
     * system path is exposed as the access point's root directory. Applications using the access point can only access
     * data in the application's own directory and any subdirectories. To learn more, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/efs-access-points.html">Mounting a file system using EFS access
     * points</a>.
     * </p>
     * <note>
     * <p>
     * If multiple requests to create access points on the same file system are sent in quick succession, and the file
     * system is near the limit of 1,000 access points, you may experience a throttling response for these requests.
     * This is to ensure that the file system does not exceed the stated access point limit.
     * </p>
     * </note>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:CreateAccessPoint</code> action.
     * </p>
     * <p>
     * Access points can be tagged on creation. If tags are specified in the creation action, IAM performs additional
     * authorization on the <code>elasticfilesystem:TagResource</code> action to verify if users have permissions to
     * create tags. Therefore, you must grant explicit permissions to use the <code>elasticfilesystem:TagResource</code>
     * action. For more information, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/using-tags-efs.html#supported-iam-actions-tagging.html">Granting
     * permissions to tag resources during creation</a>.
     * </p>
     *
     * @param createAccessPointRequest
     * @return A Java Future containing the result of the CreateAccessPoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>AccessPointAlreadyExistsException Returned if the access point that you are trying to create already
     *         exists, with the creation token you provided in the request.</li>
     *         <li>IncorrectFileSystemLifeCycleStateException Returned if the file system's lifecycle state is not
     *         "available".</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>AccessPointLimitExceededException Returned if the Amazon Web Services account has already created the
     *         maximum number of access points allowed per file system. For more informaton, see <a href=
     *         "https://docs.aws.amazon.com/efs/latest/ug/limits.html#limits-efs-resources-per-account-per-region"
     *         >https://docs.aws.amazon.com/efs/latest/ug/limits.html#limits-efs-resources-per-account-per-region</a>.</li>
     *         <li>ThrottlingException Returned when the <code>CreateAccessPoint</code> API action is called too quickly
     *         and the number of Access Points on the file system is nearing the <a href=
     *         "https://docs.aws.amazon.com/efs/latest/ug/limits.html#limits-efs-resources-per-account-per-region">limit
     *         of 120</a>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.CreateAccessPoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/CreateAccessPoint"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<CreateAccessPointResponse> createAccessPoint(CreateAccessPointRequest createAccessPointRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates an EFS access point. An access point is an application-specific view into an EFS file system that applies
     * an operating system user and group, and a file system path, to any file system request made through the access
     * point. The operating system user and group override any identity information provided by the NFS client. The file
     * system path is exposed as the access point's root directory. Applications using the access point can only access
     * data in the application's own directory and any subdirectories. To learn more, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/efs-access-points.html">Mounting a file system using EFS access
     * points</a>.
     * </p>
     * <note>
     * <p>
     * If multiple requests to create access points on the same file system are sent in quick succession, and the file
     * system is near the limit of 1,000 access points, you may experience a throttling response for these requests.
     * This is to ensure that the file system does not exceed the stated access point limit.
     * </p>
     * </note>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:CreateAccessPoint</code> action.
     * </p>
     * <p>
     * Access points can be tagged on creation. If tags are specified in the creation action, IAM performs additional
     * authorization on the <code>elasticfilesystem:TagResource</code> action to verify if users have permissions to
     * create tags. Therefore, you must grant explicit permissions to use the <code>elasticfilesystem:TagResource</code>
     * action. For more information, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/using-tags-efs.html#supported-iam-actions-tagging.html">Granting
     * permissions to tag resources during creation</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateAccessPointRequest.Builder} avoiding the need
     * to create one manually via {@link CreateAccessPointRequest#builder()}
     * </p>
     *
     * @param createAccessPointRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.CreateAccessPointRequest.Builder} to create a request.
     * @return A Java Future containing the result of the CreateAccessPoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>AccessPointAlreadyExistsException Returned if the access point that you are trying to create already
     *         exists, with the creation token you provided in the request.</li>
     *         <li>IncorrectFileSystemLifeCycleStateException Returned if the file system's lifecycle state is not
     *         "available".</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>AccessPointLimitExceededException Returned if the Amazon Web Services account has already created the
     *         maximum number of access points allowed per file system. For more informaton, see <a href=
     *         "https://docs.aws.amazon.com/efs/latest/ug/limits.html#limits-efs-resources-per-account-per-region"
     *         >https://docs.aws.amazon.com/efs/latest/ug/limits.html#limits-efs-resources-per-account-per-region</a>.</li>
     *         <li>ThrottlingException Returned when the <code>CreateAccessPoint</code> API action is called too quickly
     *         and the number of Access Points on the file system is nearing the <a href=
     *         "https://docs.aws.amazon.com/efs/latest/ug/limits.html#limits-efs-resources-per-account-per-region">limit
     *         of 120</a>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.CreateAccessPoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/CreateAccessPoint"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<CreateAccessPointResponse> createAccessPoint(
            Consumer<CreateAccessPointRequest.Builder> createAccessPointRequest) {
        return createAccessPoint(CreateAccessPointRequest.builder().applyMutation(createAccessPointRequest).build());
    }

    /**
     * <p>
     * Creates a new, empty file system. The operation requires a creation token in the request that Amazon EFS uses to
     * ensure idempotent creation (calling the operation with same creation token has no effect). If a file system does
     * not currently exist that is owned by the caller's Amazon Web Services account with the specified creation token,
     * this operation does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a new, empty file system. The file system will have an Amazon EFS assigned ID, and an initial lifecycle
     * state <code>creating</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Returns with the description of the created file system.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Otherwise, this operation returns a <code>FileSystemAlreadyExists</code> error with the ID of the existing file
     * system.
     * </p>
     * <note>
     * <p>
     * For basic use cases, you can use a randomly generated UUID for the creation token.
     * </p>
     * </note>
     * <p>
     * The idempotent operation allows you to retry a <code>CreateFileSystem</code> call without risk of creating an
     * extra file system. This can happen when an initial call fails in a way that leaves it uncertain whether or not a
     * file system was actually created. An example might be that a transport level timeout occurred or your connection
     * was reset. As long as you use the same creation token, if the initial call had succeeded in creating a file
     * system, the client can learn of its existence from the <code>FileSystemAlreadyExists</code> error.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/creating-using-create-fs.html#creating-using-create-fs-part1"
     * >Creating a file system</a> in the <i>Amazon EFS User Guide</i>.
     * </p>
     * <note>
     * <p>
     * The <code>CreateFileSystem</code> call returns while the file system's lifecycle state is still
     * <code>creating</code>. You can check the file system creation status by calling the <a>DescribeFileSystems</a>
     * operation, which among other things returns the file system state.
     * </p>
     * </note>
     * <p>
     * This operation accepts an optional <code>PerformanceMode</code> parameter that you choose for your file system.
     * We recommend <code>generalPurpose</code> performance mode for all file systems. File systems using the
     * <code>maxIO</code> mode is a previous generation performance type that is designed for highly parallelized
     * workloads that can tolerate higher latencies than the General Purpose mode. Max I/O mode is not supported for One
     * Zone file systems or file systems that use Elastic throughput.
     * </p>
     * <important>
     * <p>
     * Due to the higher per-operation latencies with Max I/O, we recommend using General Purpose performance mode for
     * all file systems.
     * </p>
     * </important>
     * <p>
     * The performance mode can't be changed after the file system has been created. For more information, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/performance.html#performancemodes.html">Amazon EFS performance
     * modes</a>.
     * </p>
     * <p>
     * You can set the throughput mode for the file system using the <code>ThroughputMode</code> parameter.
     * </p>
     * <p>
     * After the file system is fully created, Amazon EFS sets its lifecycle state to <code>available</code>, at which
     * point you can create one or more mount targets for the file system in your VPC. For more information, see
     * <a>CreateMountTarget</a>. You mount your Amazon EFS file system on an EC2 instances in your VPC by using the
     * mount target. For more information, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/how-it-works.html">Amazon EFS: How it Works</a>.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:CreateFileSystem</code> action.
     * </p>
     * <p>
     * File systems can be tagged on creation. If tags are specified in the creation action, IAM performs additional
     * authorization on the <code>elasticfilesystem:TagResource</code> action to verify if users have permissions to
     * create tags. Therefore, you must grant explicit permissions to use the <code>elasticfilesystem:TagResource</code>
     * action. For more information, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/using-tags-efs.html#supported-iam-actions-tagging.html">Granting
     * permissions to tag resources during creation</a>.
     * </p>
     *
     * @param createFileSystemRequest
     * @return A Java Future containing the result of the CreateFileSystem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemAlreadyExistsException Returned if the file system you are trying to create already exists,
     *         with the creation token you provided.</li>
     *         <li>FileSystemLimitExceededException Returned if the Amazon Web Services account has already created the
     *         maximum number of file systems allowed per account.</li>
     *         <li>InsufficientThroughputCapacityException Returned if there's not enough capacity to provision
     *         additional throughput. This value might be returned when you try to create a file system in provisioned
     *         throughput mode, when you attempt to increase the provisioned throughput of an existing file system, or
     *         when you attempt to change an existing file system from Bursting Throughput to Provisioned Throughput
     *         mode. Try again later.</li>
     *         <li>ThroughputLimitExceededException Returned if the throughput mode or amount of provisioned throughput
     *         can't be changed because the throughput limit of 1024 MiB/s has been reached.</li>
     *         <li>UnsupportedAvailabilityZoneException Returned if the requested Amazon EFS functionality is not
     *         available in the specified Availability Zone.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.CreateFileSystem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/CreateFileSystem"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<CreateFileSystemResponse> createFileSystem(CreateFileSystemRequest createFileSystemRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a new, empty file system. The operation requires a creation token in the request that Amazon EFS uses to
     * ensure idempotent creation (calling the operation with same creation token has no effect). If a file system does
     * not currently exist that is owned by the caller's Amazon Web Services account with the specified creation token,
     * this operation does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a new, empty file system. The file system will have an Amazon EFS assigned ID, and an initial lifecycle
     * state <code>creating</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Returns with the description of the created file system.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Otherwise, this operation returns a <code>FileSystemAlreadyExists</code> error with the ID of the existing file
     * system.
     * </p>
     * <note>
     * <p>
     * For basic use cases, you can use a randomly generated UUID for the creation token.
     * </p>
     * </note>
     * <p>
     * The idempotent operation allows you to retry a <code>CreateFileSystem</code> call without risk of creating an
     * extra file system. This can happen when an initial call fails in a way that leaves it uncertain whether or not a
     * file system was actually created. An example might be that a transport level timeout occurred or your connection
     * was reset. As long as you use the same creation token, if the initial call had succeeded in creating a file
     * system, the client can learn of its existence from the <code>FileSystemAlreadyExists</code> error.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/creating-using-create-fs.html#creating-using-create-fs-part1"
     * >Creating a file system</a> in the <i>Amazon EFS User Guide</i>.
     * </p>
     * <note>
     * <p>
     * The <code>CreateFileSystem</code> call returns while the file system's lifecycle state is still
     * <code>creating</code>. You can check the file system creation status by calling the <a>DescribeFileSystems</a>
     * operation, which among other things returns the file system state.
     * </p>
     * </note>
     * <p>
     * This operation accepts an optional <code>PerformanceMode</code> parameter that you choose for your file system.
     * We recommend <code>generalPurpose</code> performance mode for all file systems. File systems using the
     * <code>maxIO</code> mode is a previous generation performance type that is designed for highly parallelized
     * workloads that can tolerate higher latencies than the General Purpose mode. Max I/O mode is not supported for One
     * Zone file systems or file systems that use Elastic throughput.
     * </p>
     * <important>
     * <p>
     * Due to the higher per-operation latencies with Max I/O, we recommend using General Purpose performance mode for
     * all file systems.
     * </p>
     * </important>
     * <p>
     * The performance mode can't be changed after the file system has been created. For more information, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/performance.html#performancemodes.html">Amazon EFS performance
     * modes</a>.
     * </p>
     * <p>
     * You can set the throughput mode for the file system using the <code>ThroughputMode</code> parameter.
     * </p>
     * <p>
     * After the file system is fully created, Amazon EFS sets its lifecycle state to <code>available</code>, at which
     * point you can create one or more mount targets for the file system in your VPC. For more information, see
     * <a>CreateMountTarget</a>. You mount your Amazon EFS file system on an EC2 instances in your VPC by using the
     * mount target. For more information, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/how-it-works.html">Amazon EFS: How it Works</a>.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:CreateFileSystem</code> action.
     * </p>
     * <p>
     * File systems can be tagged on creation. If tags are specified in the creation action, IAM performs additional
     * authorization on the <code>elasticfilesystem:TagResource</code> action to verify if users have permissions to
     * create tags. Therefore, you must grant explicit permissions to use the <code>elasticfilesystem:TagResource</code>
     * action. For more information, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/using-tags-efs.html#supported-iam-actions-tagging.html">Granting
     * permissions to tag resources during creation</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateFileSystemRequest.Builder} avoiding the need
     * to create one manually via {@link CreateFileSystemRequest#builder()}
     * </p>
     *
     * @param createFileSystemRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.CreateFileSystemRequest.Builder} to create a request.
     * @return A Java Future containing the result of the CreateFileSystem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemAlreadyExistsException Returned if the file system you are trying to create already exists,
     *         with the creation token you provided.</li>
     *         <li>FileSystemLimitExceededException Returned if the Amazon Web Services account has already created the
     *         maximum number of file systems allowed per account.</li>
     *         <li>InsufficientThroughputCapacityException Returned if there's not enough capacity to provision
     *         additional throughput. This value might be returned when you try to create a file system in provisioned
     *         throughput mode, when you attempt to increase the provisioned throughput of an existing file system, or
     *         when you attempt to change an existing file system from Bursting Throughput to Provisioned Throughput
     *         mode. Try again later.</li>
     *         <li>ThroughputLimitExceededException Returned if the throughput mode or amount of provisioned throughput
     *         can't be changed because the throughput limit of 1024 MiB/s has been reached.</li>
     *         <li>UnsupportedAvailabilityZoneException Returned if the requested Amazon EFS functionality is not
     *         available in the specified Availability Zone.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.CreateFileSystem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/CreateFileSystem"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<CreateFileSystemResponse> createFileSystem(
            Consumer<CreateFileSystemRequest.Builder> createFileSystemRequest) {
        return createFileSystem(CreateFileSystemRequest.builder().applyMutation(createFileSystemRequest).build());
    }

    /**
     * <p>
     * Creates a mount target for a file system. You can then mount the file system on EC2 instances by using the mount
     * target.
     * </p>
     * <p>
     * You can create one mount target in each Availability Zone in your VPC. All EC2 instances in a VPC within a given
     * Availability Zone share a single mount target for a given file system. If you have multiple subnets in an
     * Availability Zone, you create a mount target in one of the subnets. EC2 instances do not need to be in the same
     * subnet as the mount target in order to access their file system.
     * </p>
     * <p>
     * You can create only one mount target for a One Zone file system. You must create that mount target in the same
     * Availability Zone in which the file system is located. Use the <code>AvailabilityZoneName</code> and
     * <code>AvailabiltyZoneId</code> properties in the <a>DescribeFileSystems</a> response object to get this
     * information. Use the <code>subnetId</code> associated with the file system's Availability Zone when creating the
     * mount target.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/efs/latest/ug/how-it-works.html">Amazon EFS: How
     * it Works</a>.
     * </p>
     * <p>
     * To create a mount target for a file system, the file system's lifecycle state must be <code>available</code>. For
     * more information, see <a>DescribeFileSystems</a>.
     * </p>
     * <p>
     * In the request, provide the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The file system ID for which you are creating the mount target.
     * </p>
     * </li>
     * <li>
     * <p>
     * A subnet ID, which determines the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The VPC in which Amazon EFS creates the mount target
     * </p>
     * </li>
     * <li>
     * <p>
     * The Availability Zone in which Amazon EFS creates the mount target
     * </p>
     * </li>
     * <li>
     * <p>
     * The IP address range from which Amazon EFS selects the IP address of the mount target (if you don't specify an IP
     * address in the request)
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * After creating the mount target, Amazon EFS returns a response that includes, a <code>MountTargetId</code> and an
     * <code>IpAddress</code>. You use this IP address when mounting the file system in an EC2 instance. You can also
     * use the mount target's DNS name when mounting the file system. The EC2 instance on which you mount the file
     * system by using the mount target can resolve the mount target's DNS name to its IP address. For more information,
     * see <a href="https://docs.aws.amazon.com/efs/latest/ug/how-it-works.html#how-it-works-implementation">How it
     * Works: Implementation Overview</a>.
     * </p>
     * <p>
     * Note that you can create mount targets for a file system in only one VPC, and there can be only one mount target
     * per Availability Zone. That is, if the file system already has one or more mount targets created for it, the
     * subnet specified in the request to add another mount target must meet the following requirements:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Must belong to the same VPC as the subnets of the existing mount targets
     * </p>
     * </li>
     * <li>
     * <p>
     * Must not be in the same Availability Zone as any of the subnets of the existing mount targets
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the request satisfies the requirements, Amazon EFS does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a new mount target in the specified subnet.
     * </p>
     * </li>
     * <li>
     * <p>
     * Also creates a new network interface in the subnet as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the request provides an <code>IpAddress</code>, Amazon EFS assigns that IP address to the network interface.
     * Otherwise, Amazon EFS assigns a free address in the subnet (in the same way that the Amazon EC2
     * <code>CreateNetworkInterface</code> call does when a request does not specify a primary private IP address).
     * </p>
     * </li>
     * <li>
     * <p>
     * If the request provides <code>SecurityGroups</code>, this network interface is associated with those security
     * groups. Otherwise, it belongs to the default security group for the subnet's VPC.
     * </p>
     * </li>
     * <li>
     * <p>
     * Assigns the description <code>Mount target <i>fsmt-id</i> for file system <i>fs-id</i> </code> where
     * <code> <i>fsmt-id</i> </code> is the mount target ID, and <code> <i>fs-id</i> </code> is the
     * <code>FileSystemId</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Sets the <code>requesterManaged</code> property of the network interface to <code>true</code>, and the
     * <code>requesterId</code> value to <code>EFS</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Each Amazon EFS mount target has one corresponding requester-managed EC2 network interface. After the network
     * interface is created, Amazon EFS sets the <code>NetworkInterfaceId</code> field in the mount target's description
     * to the network interface ID, and the <code>IpAddress</code> field to its address. If network interface creation
     * fails, the entire <code>CreateMountTarget</code> operation fails.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The <code>CreateMountTarget</code> call returns only after creating the network interface, but while the mount
     * target state is still <code>creating</code>, you can check the mount target creation status by calling the
     * <a>DescribeMountTargets</a> operation, which among other things returns the mount target state.
     * </p>
     * </note>
     * <p>
     * We recommend that you create a mount target in each of the Availability Zones. There are cost considerations for
     * using a file system in an Availability Zone through a mount target created in another Availability Zone. For more
     * information, see <a href="http://aws.amazon.com/efs/">Amazon EFS</a>. In addition, by always using a mount target
     * local to the instance's Availability Zone, you eliminate a partial failure scenario. If the Availability Zone in
     * which your mount target is created goes down, then you can't access your file system through that mount target.
     * </p>
     * <p>
     * This operation requires permissions for the following action on the file system:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>elasticfilesystem:CreateMountTarget</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * This operation also requires permissions for the following Amazon EC2 actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ec2:DescribeSubnets</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:DescribeNetworkInterfaces</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:CreateNetworkInterface</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param createMountTargetRequest
     * @return A Java Future containing the result of the CreateMountTarget operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>IncorrectFileSystemLifeCycleStateException Returned if the file system's lifecycle state is not
     *         "available".</li>
     *         <li>MountTargetConflictException Returned if the mount target would violate one of the specified
     *         restrictions based on the file system's existing mount targets.</li>
     *         <li>SubnetNotFoundException Returned if there is no subnet with ID <code>SubnetId</code> provided in the
     *         request.</li>
     *         <li>NoFreeAddressesInSubnetException Returned if <code>IpAddress</code> was not specified in the request
     *         and there are no free IP addresses in the subnet.</li>
     *         <li>IpAddressInUseException Returned if the request specified an <code>IpAddress</code> that is already
     *         in use in the subnet.</li>
     *         <li>NetworkInterfaceLimitExceededException The calling account has reached the limit for elastic network
     *         interfaces for the specific Amazon Web Services Region. Either delete some network interfaces or request
     *         that the account quota be raised. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/VPC_Appendix_Limits.html">Amazon VPC
     *         Quotas</a> in the <i>Amazon VPC User Guide</i> (see the <b>Network interfaces per Region</b> entry in the
     *         <b>Network interfaces</b> table).</li>
     *         <li>SecurityGroupLimitExceededException Returned if the size of <code>SecurityGroups</code> specified in
     *         the request is greater than five.</li>
     *         <li>SecurityGroupNotFoundException Returned if one of the specified security groups doesn't exist in the
     *         subnet's virtual private cloud (VPC).</li>
     *         <li>UnsupportedAvailabilityZoneException Returned if the requested Amazon EFS functionality is not
     *         available in the specified Availability Zone.</li>
     *         <li>AvailabilityZonesMismatchException Returned if the Availability Zone that was specified for a mount
     *         target is different from the Availability Zone that was specified for One Zone storage. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/efs/latest/ug/availability-durability.html">Regional and One Zone
     *         storage redundancy</a>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.CreateMountTarget
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/CreateMountTarget"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<CreateMountTargetResponse> createMountTarget(CreateMountTargetRequest createMountTargetRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a mount target for a file system. You can then mount the file system on EC2 instances by using the mount
     * target.
     * </p>
     * <p>
     * You can create one mount target in each Availability Zone in your VPC. All EC2 instances in a VPC within a given
     * Availability Zone share a single mount target for a given file system. If you have multiple subnets in an
     * Availability Zone, you create a mount target in one of the subnets. EC2 instances do not need to be in the same
     * subnet as the mount target in order to access their file system.
     * </p>
     * <p>
     * You can create only one mount target for a One Zone file system. You must create that mount target in the same
     * Availability Zone in which the file system is located. Use the <code>AvailabilityZoneName</code> and
     * <code>AvailabiltyZoneId</code> properties in the <a>DescribeFileSystems</a> response object to get this
     * information. Use the <code>subnetId</code> associated with the file system's Availability Zone when creating the
     * mount target.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/efs/latest/ug/how-it-works.html">Amazon EFS: How
     * it Works</a>.
     * </p>
     * <p>
     * To create a mount target for a file system, the file system's lifecycle state must be <code>available</code>. For
     * more information, see <a>DescribeFileSystems</a>.
     * </p>
     * <p>
     * In the request, provide the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The file system ID for which you are creating the mount target.
     * </p>
     * </li>
     * <li>
     * <p>
     * A subnet ID, which determines the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The VPC in which Amazon EFS creates the mount target
     * </p>
     * </li>
     * <li>
     * <p>
     * The Availability Zone in which Amazon EFS creates the mount target
     * </p>
     * </li>
     * <li>
     * <p>
     * The IP address range from which Amazon EFS selects the IP address of the mount target (if you don't specify an IP
     * address in the request)
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * After creating the mount target, Amazon EFS returns a response that includes, a <code>MountTargetId</code> and an
     * <code>IpAddress</code>. You use this IP address when mounting the file system in an EC2 instance. You can also
     * use the mount target's DNS name when mounting the file system. The EC2 instance on which you mount the file
     * system by using the mount target can resolve the mount target's DNS name to its IP address. For more information,
     * see <a href="https://docs.aws.amazon.com/efs/latest/ug/how-it-works.html#how-it-works-implementation">How it
     * Works: Implementation Overview</a>.
     * </p>
     * <p>
     * Note that you can create mount targets for a file system in only one VPC, and there can be only one mount target
     * per Availability Zone. That is, if the file system already has one or more mount targets created for it, the
     * subnet specified in the request to add another mount target must meet the following requirements:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Must belong to the same VPC as the subnets of the existing mount targets
     * </p>
     * </li>
     * <li>
     * <p>
     * Must not be in the same Availability Zone as any of the subnets of the existing mount targets
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the request satisfies the requirements, Amazon EFS does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a new mount target in the specified subnet.
     * </p>
     * </li>
     * <li>
     * <p>
     * Also creates a new network interface in the subnet as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the request provides an <code>IpAddress</code>, Amazon EFS assigns that IP address to the network interface.
     * Otherwise, Amazon EFS assigns a free address in the subnet (in the same way that the Amazon EC2
     * <code>CreateNetworkInterface</code> call does when a request does not specify a primary private IP address).
     * </p>
     * </li>
     * <li>
     * <p>
     * If the request provides <code>SecurityGroups</code>, this network interface is associated with those security
     * groups. Otherwise, it belongs to the default security group for the subnet's VPC.
     * </p>
     * </li>
     * <li>
     * <p>
     * Assigns the description <code>Mount target <i>fsmt-id</i> for file system <i>fs-id</i> </code> where
     * <code> <i>fsmt-id</i> </code> is the mount target ID, and <code> <i>fs-id</i> </code> is the
     * <code>FileSystemId</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Sets the <code>requesterManaged</code> property of the network interface to <code>true</code>, and the
     * <code>requesterId</code> value to <code>EFS</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Each Amazon EFS mount target has one corresponding requester-managed EC2 network interface. After the network
     * interface is created, Amazon EFS sets the <code>NetworkInterfaceId</code> field in the mount target's description
     * to the network interface ID, and the <code>IpAddress</code> field to its address. If network interface creation
     * fails, the entire <code>CreateMountTarget</code> operation fails.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The <code>CreateMountTarget</code> call returns only after creating the network interface, but while the mount
     * target state is still <code>creating</code>, you can check the mount target creation status by calling the
     * <a>DescribeMountTargets</a> operation, which among other things returns the mount target state.
     * </p>
     * </note>
     * <p>
     * We recommend that you create a mount target in each of the Availability Zones. There are cost considerations for
     * using a file system in an Availability Zone through a mount target created in another Availability Zone. For more
     * information, see <a href="http://aws.amazon.com/efs/">Amazon EFS</a>. In addition, by always using a mount target
     * local to the instance's Availability Zone, you eliminate a partial failure scenario. If the Availability Zone in
     * which your mount target is created goes down, then you can't access your file system through that mount target.
     * </p>
     * <p>
     * This operation requires permissions for the following action on the file system:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>elasticfilesystem:CreateMountTarget</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * This operation also requires permissions for the following Amazon EC2 actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ec2:DescribeSubnets</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:DescribeNetworkInterfaces</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:CreateNetworkInterface</code>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateMountTargetRequest.Builder} avoiding the need
     * to create one manually via {@link CreateMountTargetRequest#builder()}
     * </p>
     *
     * @param createMountTargetRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.CreateMountTargetRequest.Builder} to create a request.
     * @return A Java Future containing the result of the CreateMountTarget operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>IncorrectFileSystemLifeCycleStateException Returned if the file system's lifecycle state is not
     *         "available".</li>
     *         <li>MountTargetConflictException Returned if the mount target would violate one of the specified
     *         restrictions based on the file system's existing mount targets.</li>
     *         <li>SubnetNotFoundException Returned if there is no subnet with ID <code>SubnetId</code> provided in the
     *         request.</li>
     *         <li>NoFreeAddressesInSubnetException Returned if <code>IpAddress</code> was not specified in the request
     *         and there are no free IP addresses in the subnet.</li>
     *         <li>IpAddressInUseException Returned if the request specified an <code>IpAddress</code> that is already
     *         in use in the subnet.</li>
     *         <li>NetworkInterfaceLimitExceededException The calling account has reached the limit for elastic network
     *         interfaces for the specific Amazon Web Services Region. Either delete some network interfaces or request
     *         that the account quota be raised. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/VPC_Appendix_Limits.html">Amazon VPC
     *         Quotas</a> in the <i>Amazon VPC User Guide</i> (see the <b>Network interfaces per Region</b> entry in the
     *         <b>Network interfaces</b> table).</li>
     *         <li>SecurityGroupLimitExceededException Returned if the size of <code>SecurityGroups</code> specified in
     *         the request is greater than five.</li>
     *         <li>SecurityGroupNotFoundException Returned if one of the specified security groups doesn't exist in the
     *         subnet's virtual private cloud (VPC).</li>
     *         <li>UnsupportedAvailabilityZoneException Returned if the requested Amazon EFS functionality is not
     *         available in the specified Availability Zone.</li>
     *         <li>AvailabilityZonesMismatchException Returned if the Availability Zone that was specified for a mount
     *         target is different from the Availability Zone that was specified for One Zone storage. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/efs/latest/ug/availability-durability.html">Regional and One Zone
     *         storage redundancy</a>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.CreateMountTarget
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/CreateMountTarget"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<CreateMountTargetResponse> createMountTarget(
            Consumer<CreateMountTargetRequest.Builder> createMountTargetRequest) {
        return createMountTarget(CreateMountTargetRequest.builder().applyMutation(createMountTargetRequest).build());
    }

    /**
     * <p>
     * Creates a replication configuration that replicates an existing EFS file system to a new, read-only file system.
     * For more information, see <a href="https://docs.aws.amazon.com/efs/latest/ug/efs-replication.html">Amazon EFS
     * replication</a> in the <i>Amazon EFS User Guide</i>. The replication configuration specifies the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Source file system</b> – The EFS file system that you want replicated. The source file system cannot be a
     * destination file system in an existing replication configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Amazon Web Services Region</b> – The Amazon Web Services Region in which the destination file system is
     * created. Amazon EFS replication is available in all Amazon Web Services Regions in which EFS is available. The
     * Region must be enabled. For more information, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/rande-manage.html#rande-manage-enable">Managing Amazon Web
     * Services Regions</a> in the <i>Amazon Web Services General Reference Reference Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Destination file system configuration</b> – The configuration of the destination file system to which the
     * source file system will be replicated. There can only be one destination file system in a replication
     * configuration.
     * </p>
     * <p>
     * Parameters for the replication configuration include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>File system ID</b> – The ID of the destination file system for the replication. If no ID is provided, then EFS
     * creates a new file system with the default settings. For existing file systems, the file system's replication
     * overwrite protection must be disabled. For more information, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/efs-replication#replicate-existing-destination"> Replicating to
     * an existing file system</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Availability Zone</b> – If you want the destination file system to use One Zone storage, you must specify the
     * Availability Zone to create the file system in. For more information, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/storage-classes.html"> EFS file system types</a> in the <i>Amazon
     * EFS User Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Encryption</b> – All destination file systems are created with encryption at rest enabled. You can specify the
     * Key Management Service (KMS) key that is used to encrypt the destination file system. If you don't specify a KMS
     * key, your service-managed KMS key for Amazon EFS is used.
     * </p>
     * <note>
     * <p>
     * After the file system is created, you cannot change the KMS key.
     * </p>
     * </note></li>
     * </ul>
     * </li>
     * </ul>
     * <note>
     * <p>
     * After the file system is created, you cannot change the KMS key.
     * </p>
     * </note>
     * <p>
     * For new destination file systems, the following properties are set by default:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Performance mode</b> - The destination file system's performance mode matches that of the source file system,
     * unless the destination file system uses EFS One Zone storage. In that case, the General Purpose performance mode
     * is used. The performance mode cannot be changed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Throughput mode</b> - The destination file system's throughput mode matches that of the source file system.
     * After the file system is created, you can modify the throughput mode.
     * </p>
     * </li>
     * </ul>
     * <ul>
     * <li>
     * <p>
     * <b>Lifecycle management</b> – Lifecycle management is not enabled on the destination file system. After the
     * destination file system is created, you can enable lifecycle management.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Automatic backups</b> – Automatic daily backups are enabled on the destination file system. After the file
     * system is created, you can change this setting.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/efs/latest/ug/efs-replication.html">Amazon EFS
     * replication</a> in the <i>Amazon EFS User Guide</i>.
     * </p>
     *
     * @param createReplicationConfigurationRequest
     * @return A Java Future containing the result of the CreateReplicationConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>IncorrectFileSystemLifeCycleStateException Returned if the file system's lifecycle state is not
     *         "available".</li>
     *         <li>ValidationException Returned if the Backup service is not available in the Amazon Web Services Region
     *         in which the request was made.</li>
     *         <li>ReplicationNotFoundException Returned if the specified file system does not have a replication
     *         configuration.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>UnsupportedAvailabilityZoneException Returned if the requested Amazon EFS functionality is not
     *         available in the specified Availability Zone.</li>
     *         <li>FileSystemLimitExceededException Returned if the Amazon Web Services account has already created the
     *         maximum number of file systems allowed per account.</li>
     *         <li>InsufficientThroughputCapacityException Returned if there's not enough capacity to provision
     *         additional throughput. This value might be returned when you try to create a file system in provisioned
     *         throughput mode, when you attempt to increase the provisioned throughput of an existing file system, or
     *         when you attempt to change an existing file system from Bursting Throughput to Provisioned Throughput
     *         mode. Try again later.</li>
     *         <li>ThroughputLimitExceededException Returned if the throughput mode or amount of provisioned throughput
     *         can't be changed because the throughput limit of 1024 MiB/s has been reached.</li>
     *         <li>ConflictException Returned if the source file system in a replication is encrypted but the
     *         destination file system is unencrypted.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.CreateReplicationConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/CreateReplicationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<CreateReplicationConfigurationResponse> createReplicationConfiguration(
            CreateReplicationConfigurationRequest createReplicationConfigurationRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a replication configuration that replicates an existing EFS file system to a new, read-only file system.
     * For more information, see <a href="https://docs.aws.amazon.com/efs/latest/ug/efs-replication.html">Amazon EFS
     * replication</a> in the <i>Amazon EFS User Guide</i>. The replication configuration specifies the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Source file system</b> – The EFS file system that you want replicated. The source file system cannot be a
     * destination file system in an existing replication configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Amazon Web Services Region</b> – The Amazon Web Services Region in which the destination file system is
     * created. Amazon EFS replication is available in all Amazon Web Services Regions in which EFS is available. The
     * Region must be enabled. For more information, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/rande-manage.html#rande-manage-enable">Managing Amazon Web
     * Services Regions</a> in the <i>Amazon Web Services General Reference Reference Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Destination file system configuration</b> – The configuration of the destination file system to which the
     * source file system will be replicated. There can only be one destination file system in a replication
     * configuration.
     * </p>
     * <p>
     * Parameters for the replication configuration include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>File system ID</b> – The ID of the destination file system for the replication. If no ID is provided, then EFS
     * creates a new file system with the default settings. For existing file systems, the file system's replication
     * overwrite protection must be disabled. For more information, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/efs-replication#replicate-existing-destination"> Replicating to
     * an existing file system</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Availability Zone</b> – If you want the destination file system to use One Zone storage, you must specify the
     * Availability Zone to create the file system in. For more information, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/storage-classes.html"> EFS file system types</a> in the <i>Amazon
     * EFS User Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Encryption</b> – All destination file systems are created with encryption at rest enabled. You can specify the
     * Key Management Service (KMS) key that is used to encrypt the destination file system. If you don't specify a KMS
     * key, your service-managed KMS key for Amazon EFS is used.
     * </p>
     * <note>
     * <p>
     * After the file system is created, you cannot change the KMS key.
     * </p>
     * </note></li>
     * </ul>
     * </li>
     * </ul>
     * <note>
     * <p>
     * After the file system is created, you cannot change the KMS key.
     * </p>
     * </note>
     * <p>
     * For new destination file systems, the following properties are set by default:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Performance mode</b> - The destination file system's performance mode matches that of the source file system,
     * unless the destination file system uses EFS One Zone storage. In that case, the General Purpose performance mode
     * is used. The performance mode cannot be changed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Throughput mode</b> - The destination file system's throughput mode matches that of the source file system.
     * After the file system is created, you can modify the throughput mode.
     * </p>
     * </li>
     * </ul>
     * <ul>
     * <li>
     * <p>
     * <b>Lifecycle management</b> – Lifecycle management is not enabled on the destination file system. After the
     * destination file system is created, you can enable lifecycle management.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Automatic backups</b> – Automatic daily backups are enabled on the destination file system. After the file
     * system is created, you can change this setting.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/efs/latest/ug/efs-replication.html">Amazon EFS
     * replication</a> in the <i>Amazon EFS User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateReplicationConfigurationRequest.Builder}
     * avoiding the need to create one manually via {@link CreateReplicationConfigurationRequest#builder()}
     * </p>
     *
     * @param createReplicationConfigurationRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.CreateReplicationConfigurationRequest.Builder} to create
     *        a request.
     * @return A Java Future containing the result of the CreateReplicationConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>IncorrectFileSystemLifeCycleStateException Returned if the file system's lifecycle state is not
     *         "available".</li>
     *         <li>ValidationException Returned if the Backup service is not available in the Amazon Web Services Region
     *         in which the request was made.</li>
     *         <li>ReplicationNotFoundException Returned if the specified file system does not have a replication
     *         configuration.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>UnsupportedAvailabilityZoneException Returned if the requested Amazon EFS functionality is not
     *         available in the specified Availability Zone.</li>
     *         <li>FileSystemLimitExceededException Returned if the Amazon Web Services account has already created the
     *         maximum number of file systems allowed per account.</li>
     *         <li>InsufficientThroughputCapacityException Returned if there's not enough capacity to provision
     *         additional throughput. This value might be returned when you try to create a file system in provisioned
     *         throughput mode, when you attempt to increase the provisioned throughput of an existing file system, or
     *         when you attempt to change an existing file system from Bursting Throughput to Provisioned Throughput
     *         mode. Try again later.</li>
     *         <li>ThroughputLimitExceededException Returned if the throughput mode or amount of provisioned throughput
     *         can't be changed because the throughput limit of 1024 MiB/s has been reached.</li>
     *         <li>ConflictException Returned if the source file system in a replication is encrypted but the
     *         destination file system is unencrypted.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.CreateReplicationConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/CreateReplicationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<CreateReplicationConfigurationResponse> createReplicationConfiguration(
            Consumer<CreateReplicationConfigurationRequest.Builder> createReplicationConfigurationRequest) {
        return createReplicationConfiguration(CreateReplicationConfigurationRequest.builder()
                .applyMutation(createReplicationConfigurationRequest).build());
    }

    /**
     * <note>
     * <p>
     * DEPRECATED - <code>CreateTags</code> is deprecated and not maintained. To create tags for EFS resources, use the
     * API action.
     * </p>
     * </note>
     * <p>
     * Creates or overwrites tags associated with a file system. Each tag is a key-value pair. If a tag key specified in
     * the request already exists on the file system, this operation overwrites its value with the value provided in the
     * request. If you add the <code>Name</code> tag to your file system, Amazon EFS returns it in the response to the
     * <a>DescribeFileSystems</a> operation.
     * </p>
     * <p>
     * This operation requires permission for the <code>elasticfilesystem:CreateTags</code> action.
     * </p>
     *
     * @param createTagsRequest
     * @return A Java Future containing the result of the CreateTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.CreateTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/CreateTags" target="_top">AWS
     *      API Documentation</a>
     *
     * @deprecated Use TagResource.
     */
    @Deprecated
    default CompletableFuture<CreateTagsResponse> createTags(CreateTagsRequest createTagsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <note>
     * <p>
     * DEPRECATED - <code>CreateTags</code> is deprecated and not maintained. To create tags for EFS resources, use the
     * API action.
     * </p>
     * </note>
     * <p>
     * Creates or overwrites tags associated with a file system. Each tag is a key-value pair. If a tag key specified in
     * the request already exists on the file system, this operation overwrites its value with the value provided in the
     * request. If you add the <code>Name</code> tag to your file system, Amazon EFS returns it in the response to the
     * <a>DescribeFileSystems</a> operation.
     * </p>
     * <p>
     * This operation requires permission for the <code>elasticfilesystem:CreateTags</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateTagsRequest.Builder} avoiding the need to
     * create one manually via {@link CreateTagsRequest#builder()}
     * </p>
     *
     * @param createTagsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.CreateTagsRequest.Builder} to create a request.
     * @return A Java Future containing the result of the CreateTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.CreateTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/CreateTags" target="_top">AWS
     *      API Documentation</a>
     *
     * @deprecated Use TagResource.
     */
    @Deprecated
    default CompletableFuture<CreateTagsResponse> createTags(Consumer<CreateTagsRequest.Builder> createTagsRequest) {
        return createTags(CreateTagsRequest.builder().applyMutation(createTagsRequest).build());
    }

    /**
     * <p>
     * Deletes the specified access point. After deletion is complete, new clients can no longer connect to the access
     * points. Clients connected to the access point at the time of deletion will continue to function until they
     * terminate their connection.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DeleteAccessPoint</code> action.
     * </p>
     *
     * @param deleteAccessPointRequest
     * @return A Java Future containing the result of the DeleteAccessPoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>AccessPointNotFoundException Returned if the specified <code>AccessPointId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DeleteAccessPoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DeleteAccessPoint"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteAccessPointResponse> deleteAccessPoint(DeleteAccessPointRequest deleteAccessPointRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the specified access point. After deletion is complete, new clients can no longer connect to the access
     * points. Clients connected to the access point at the time of deletion will continue to function until they
     * terminate their connection.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DeleteAccessPoint</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteAccessPointRequest.Builder} avoiding the need
     * to create one manually via {@link DeleteAccessPointRequest#builder()}
     * </p>
     *
     * @param deleteAccessPointRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.DeleteAccessPointRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DeleteAccessPoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>AccessPointNotFoundException Returned if the specified <code>AccessPointId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DeleteAccessPoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DeleteAccessPoint"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteAccessPointResponse> deleteAccessPoint(
            Consumer<DeleteAccessPointRequest.Builder> deleteAccessPointRequest) {
        return deleteAccessPoint(DeleteAccessPointRequest.builder().applyMutation(deleteAccessPointRequest).build());
    }

    /**
     * <p>
     * Deletes a file system, permanently severing access to its contents. Upon return, the file system no longer exists
     * and you can't access any contents of the deleted file system.
     * </p>
     * <p>
     * You need to manually delete mount targets attached to a file system before you can delete an EFS file system.
     * This step is performed for you when you use the Amazon Web Services console to delete a file system.
     * </p>
     * <note>
     * <p>
     * You cannot delete a file system that is part of an EFS Replication configuration. You need to delete the
     * replication configuration first.
     * </p>
     * </note>
     * <p>
     * You can't delete a file system that is in use. That is, if the file system has any mount targets, you must first
     * delete them. For more information, see <a>DescribeMountTargets</a> and <a>DeleteMountTarget</a>.
     * </p>
     * <note>
     * <p>
     * The <code>DeleteFileSystem</code> call returns while the file system state is still <code>deleting</code>. You
     * can check the file system deletion status by calling the <a>DescribeFileSystems</a> operation, which returns a
     * list of file systems in your account. If you pass file system ID or creation token for the deleted file system,
     * the <a>DescribeFileSystems</a> returns a <code>404 FileSystemNotFound</code> error.
     * </p>
     * </note>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DeleteFileSystem</code> action.
     * </p>
     *
     * @param deleteFileSystemRequest
     * @return A Java Future containing the result of the DeleteFileSystem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>FileSystemInUseException Returned if a file system has mount targets.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DeleteFileSystem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DeleteFileSystem"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteFileSystemResponse> deleteFileSystem(DeleteFileSystemRequest deleteFileSystemRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a file system, permanently severing access to its contents. Upon return, the file system no longer exists
     * and you can't access any contents of the deleted file system.
     * </p>
     * <p>
     * You need to manually delete mount targets attached to a file system before you can delete an EFS file system.
     * This step is performed for you when you use the Amazon Web Services console to delete a file system.
     * </p>
     * <note>
     * <p>
     * You cannot delete a file system that is part of an EFS Replication configuration. You need to delete the
     * replication configuration first.
     * </p>
     * </note>
     * <p>
     * You can't delete a file system that is in use. That is, if the file system has any mount targets, you must first
     * delete them. For more information, see <a>DescribeMountTargets</a> and <a>DeleteMountTarget</a>.
     * </p>
     * <note>
     * <p>
     * The <code>DeleteFileSystem</code> call returns while the file system state is still <code>deleting</code>. You
     * can check the file system deletion status by calling the <a>DescribeFileSystems</a> operation, which returns a
     * list of file systems in your account. If you pass file system ID or creation token for the deleted file system,
     * the <a>DescribeFileSystems</a> returns a <code>404 FileSystemNotFound</code> error.
     * </p>
     * </note>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DeleteFileSystem</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteFileSystemRequest.Builder} avoiding the need
     * to create one manually via {@link DeleteFileSystemRequest#builder()}
     * </p>
     *
     * @param deleteFileSystemRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.DeleteFileSystemRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DeleteFileSystem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>FileSystemInUseException Returned if a file system has mount targets.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DeleteFileSystem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DeleteFileSystem"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteFileSystemResponse> deleteFileSystem(
            Consumer<DeleteFileSystemRequest.Builder> deleteFileSystemRequest) {
        return deleteFileSystem(DeleteFileSystemRequest.builder().applyMutation(deleteFileSystemRequest).build());
    }

    /**
     * <p>
     * Deletes the <code>FileSystemPolicy</code> for the specified file system. The default
     * <code>FileSystemPolicy</code> goes into effect once the existing policy is deleted. For more information about
     * the default file system policy, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/res-based-policies-efs.html">Using Resource-based Policies with
     * EFS</a>.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DeleteFileSystemPolicy</code> action.
     * </p>
     *
     * @param deleteFileSystemPolicyRequest
     * @return A Java Future containing the result of the DeleteFileSystemPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>IncorrectFileSystemLifeCycleStateException Returned if the file system's lifecycle state is not
     *         "available".</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DeleteFileSystemPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DeleteFileSystemPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteFileSystemPolicyResponse> deleteFileSystemPolicy(
            DeleteFileSystemPolicyRequest deleteFileSystemPolicyRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the <code>FileSystemPolicy</code> for the specified file system. The default
     * <code>FileSystemPolicy</code> goes into effect once the existing policy is deleted. For more information about
     * the default file system policy, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/res-based-policies-efs.html">Using Resource-based Policies with
     * EFS</a>.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DeleteFileSystemPolicy</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteFileSystemPolicyRequest.Builder} avoiding the
     * need to create one manually via {@link DeleteFileSystemPolicyRequest#builder()}
     * </p>
     *
     * @param deleteFileSystemPolicyRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.DeleteFileSystemPolicyRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the DeleteFileSystemPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>IncorrectFileSystemLifeCycleStateException Returned if the file system's lifecycle state is not
     *         "available".</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DeleteFileSystemPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DeleteFileSystemPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteFileSystemPolicyResponse> deleteFileSystemPolicy(
            Consumer<DeleteFileSystemPolicyRequest.Builder> deleteFileSystemPolicyRequest) {
        return deleteFileSystemPolicy(DeleteFileSystemPolicyRequest.builder().applyMutation(deleteFileSystemPolicyRequest)
                .build());
    }

    /**
     * <p>
     * Deletes the specified mount target.
     * </p>
     * <p>
     * This operation forcibly breaks any mounts of the file system by using the mount target that is being deleted,
     * which might disrupt instances or applications using those mounts. To avoid applications getting cut off abruptly,
     * you might consider unmounting any mounts of the mount target, if feasible. The operation also deletes the
     * associated network interface. Uncommitted writes might be lost, but breaking a mount target using this operation
     * does not corrupt the file system itself. The file system you created remains. You can mount an EC2 instance in
     * your VPC by using another mount target.
     * </p>
     * <p>
     * This operation requires permissions for the following action on the file system:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>elasticfilesystem:DeleteMountTarget</code>
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The <code>DeleteMountTarget</code> call returns while the mount target state is still <code>deleting</code>. You
     * can check the mount target deletion by calling the <a>DescribeMountTargets</a> operation, which returns a list of
     * mount target descriptions for the given file system.
     * </p>
     * </note>
     * <p>
     * The operation also requires permissions for the following Amazon EC2 action on the mount target's network
     * interface:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ec2:DeleteNetworkInterface</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteMountTargetRequest
     * @return A Java Future containing the result of the DeleteMountTarget operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>DependencyTimeoutException The service timed out trying to fulfill the request, and the client should
     *         try the call again.</li>
     *         <li>MountTargetNotFoundException Returned if there is no mount target with the specified ID found in the
     *         caller's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DeleteMountTarget
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DeleteMountTarget"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteMountTargetResponse> deleteMountTarget(DeleteMountTargetRequest deleteMountTargetRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the specified mount target.
     * </p>
     * <p>
     * This operation forcibly breaks any mounts of the file system by using the mount target that is being deleted,
     * which might disrupt instances or applications using those mounts. To avoid applications getting cut off abruptly,
     * you might consider unmounting any mounts of the mount target, if feasible. The operation also deletes the
     * associated network interface. Uncommitted writes might be lost, but breaking a mount target using this operation
     * does not corrupt the file system itself. The file system you created remains. You can mount an EC2 instance in
     * your VPC by using another mount target.
     * </p>
     * <p>
     * This operation requires permissions for the following action on the file system:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>elasticfilesystem:DeleteMountTarget</code>
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The <code>DeleteMountTarget</code> call returns while the mount target state is still <code>deleting</code>. You
     * can check the mount target deletion by calling the <a>DescribeMountTargets</a> operation, which returns a list of
     * mount target descriptions for the given file system.
     * </p>
     * </note>
     * <p>
     * The operation also requires permissions for the following Amazon EC2 action on the mount target's network
     * interface:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ec2:DeleteNetworkInterface</code>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteMountTargetRequest.Builder} avoiding the need
     * to create one manually via {@link DeleteMountTargetRequest#builder()}
     * </p>
     *
     * @param deleteMountTargetRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.DeleteMountTargetRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DeleteMountTarget operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>DependencyTimeoutException The service timed out trying to fulfill the request, and the client should
     *         try the call again.</li>
     *         <li>MountTargetNotFoundException Returned if there is no mount target with the specified ID found in the
     *         caller's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DeleteMountTarget
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DeleteMountTarget"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteMountTargetResponse> deleteMountTarget(
            Consumer<DeleteMountTargetRequest.Builder> deleteMountTargetRequest) {
        return deleteMountTarget(DeleteMountTargetRequest.builder().applyMutation(deleteMountTargetRequest).build());
    }

    /**
     * <p>
     * Deletes a replication configuration. Deleting a replication configuration ends the replication process. After a
     * replication configuration is deleted, the destination file system becomes <code>Writeable</code> and its
     * replication overwrite protection is re-enabled. For more information, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/delete-replications.html">Delete a replication configuration</a>.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DeleteReplicationConfiguration</code> action.
     * </p>
     *
     * @param deleteReplicationConfigurationRequest
     * @return A Java Future containing the result of the DeleteReplicationConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>ReplicationNotFoundException Returned if the specified file system does not have a replication
     *         configuration.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DeleteReplicationConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DeleteReplicationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteReplicationConfigurationResponse> deleteReplicationConfiguration(
            DeleteReplicationConfigurationRequest deleteReplicationConfigurationRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a replication configuration. Deleting a replication configuration ends the replication process. After a
     * replication configuration is deleted, the destination file system becomes <code>Writeable</code> and its
     * replication overwrite protection is re-enabled. For more information, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/delete-replications.html">Delete a replication configuration</a>.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DeleteReplicationConfiguration</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteReplicationConfigurationRequest.Builder}
     * avoiding the need to create one manually via {@link DeleteReplicationConfigurationRequest#builder()}
     * </p>
     *
     * @param deleteReplicationConfigurationRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.DeleteReplicationConfigurationRequest.Builder} to create
     *        a request.
     * @return A Java Future containing the result of the DeleteReplicationConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>ReplicationNotFoundException Returned if the specified file system does not have a replication
     *         configuration.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DeleteReplicationConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DeleteReplicationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteReplicationConfigurationResponse> deleteReplicationConfiguration(
            Consumer<DeleteReplicationConfigurationRequest.Builder> deleteReplicationConfigurationRequest) {
        return deleteReplicationConfiguration(DeleteReplicationConfigurationRequest.builder()
                .applyMutation(deleteReplicationConfigurationRequest).build());
    }

    /**
     * <note>
     * <p>
     * DEPRECATED - <code>DeleteTags</code> is deprecated and not maintained. To remove tags from EFS resources, use the
     * API action.
     * </p>
     * </note>
     * <p>
     * Deletes the specified tags from a file system. If the <code>DeleteTags</code> request includes a tag key that
     * doesn't exist, Amazon EFS ignores it and doesn't cause an error. For more information about tags and related
     * restrictions, see <a href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html">Tag
     * restrictions</a> in the <i>Billing and Cost Management User Guide</i>.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DeleteTags</code> action.
     * </p>
     *
     * @param deleteTagsRequest
     * @return A Java Future containing the result of the DeleteTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DeleteTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DeleteTags" target="_top">AWS
     *      API Documentation</a>
     *
     * @deprecated Use UntagResource.
     */
    @Deprecated
    default CompletableFuture<DeleteTagsResponse> deleteTags(DeleteTagsRequest deleteTagsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <note>
     * <p>
     * DEPRECATED - <code>DeleteTags</code> is deprecated and not maintained. To remove tags from EFS resources, use the
     * API action.
     * </p>
     * </note>
     * <p>
     * Deletes the specified tags from a file system. If the <code>DeleteTags</code> request includes a tag key that
     * doesn't exist, Amazon EFS ignores it and doesn't cause an error. For more information about tags and related
     * restrictions, see <a href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html">Tag
     * restrictions</a> in the <i>Billing and Cost Management User Guide</i>.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DeleteTags</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteTagsRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteTagsRequest#builder()}
     * </p>
     *
     * @param deleteTagsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.DeleteTagsRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DeleteTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DeleteTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DeleteTags" target="_top">AWS
     *      API Documentation</a>
     *
     * @deprecated Use UntagResource.
     */
    @Deprecated
    default CompletableFuture<DeleteTagsResponse> deleteTags(Consumer<DeleteTagsRequest.Builder> deleteTagsRequest) {
        return deleteTags(DeleteTagsRequest.builder().applyMutation(deleteTagsRequest).build());
    }

    /**
     * <p>
     * Returns the description of a specific Amazon EFS access point if the <code>AccessPointId</code> is provided. If
     * you provide an EFS <code>FileSystemId</code>, it returns descriptions of all access points for that file system.
     * You can provide either an <code>AccessPointId</code> or a <code>FileSystemId</code> in the request, but not both.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeAccessPoints</code> action.
     * </p>
     *
     * @param describeAccessPointsRequest
     * @return A Java Future containing the result of the DescribeAccessPoints operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>AccessPointNotFoundException Returned if the specified <code>AccessPointId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeAccessPoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeAccessPoints"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeAccessPointsResponse> describeAccessPoints(
            DescribeAccessPointsRequest describeAccessPointsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the description of a specific Amazon EFS access point if the <code>AccessPointId</code> is provided. If
     * you provide an EFS <code>FileSystemId</code>, it returns descriptions of all access points for that file system.
     * You can provide either an <code>AccessPointId</code> or a <code>FileSystemId</code> in the request, but not both.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeAccessPoints</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeAccessPointsRequest.Builder} avoiding the
     * need to create one manually via {@link DescribeAccessPointsRequest#builder()}
     * </p>
     *
     * @param describeAccessPointsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.DescribeAccessPointsRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DescribeAccessPoints operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>AccessPointNotFoundException Returned if the specified <code>AccessPointId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeAccessPoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeAccessPoints"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeAccessPointsResponse> describeAccessPoints(
            Consumer<DescribeAccessPointsRequest.Builder> describeAccessPointsRequest) {
        return describeAccessPoints(DescribeAccessPointsRequest.builder().applyMutation(describeAccessPointsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #describeAccessPoints(software.amazon.awssdk.services.efs.model.DescribeAccessPointsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeAccessPointsPublisher publisher = client.describeAccessPointsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeAccessPointsPublisher publisher = client.describeAccessPointsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.efs.model.DescribeAccessPointsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.efs.model.DescribeAccessPointsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeAccessPoints(software.amazon.awssdk.services.efs.model.DescribeAccessPointsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeAccessPointsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>AccessPointNotFoundException Returned if the specified <code>AccessPointId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeAccessPoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeAccessPoints"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeAccessPointsPublisher describeAccessPointsPaginator(DescribeAccessPointsRequest describeAccessPointsRequest) {
        return new DescribeAccessPointsPublisher(this, describeAccessPointsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #describeAccessPoints(software.amazon.awssdk.services.efs.model.DescribeAccessPointsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeAccessPointsPublisher publisher = client.describeAccessPointsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeAccessPointsPublisher publisher = client.describeAccessPointsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.efs.model.DescribeAccessPointsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.efs.model.DescribeAccessPointsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeAccessPoints(software.amazon.awssdk.services.efs.model.DescribeAccessPointsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeAccessPointsRequest.Builder} avoiding the
     * need to create one manually via {@link DescribeAccessPointsRequest#builder()}
     * </p>
     *
     * @param describeAccessPointsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.DescribeAccessPointsRequest.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>AccessPointNotFoundException Returned if the specified <code>AccessPointId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeAccessPoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeAccessPoints"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeAccessPointsPublisher describeAccessPointsPaginator(
            Consumer<DescribeAccessPointsRequest.Builder> describeAccessPointsRequest) {
        return describeAccessPointsPaginator(DescribeAccessPointsRequest.builder().applyMutation(describeAccessPointsRequest)
                .build());
    }

    /**
     * <p>
     * Returns the account preferences settings for the Amazon Web Services account associated with the user making the
     * request, in the current Amazon Web Services Region.
     * </p>
     *
     * @param describeAccountPreferencesRequest
     * @return A Java Future containing the result of the DescribeAccountPreferences operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeAccountPreferences
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeAccountPreferences"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeAccountPreferencesResponse> describeAccountPreferences(
            DescribeAccountPreferencesRequest describeAccountPreferencesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the account preferences settings for the Amazon Web Services account associated with the user making the
     * request, in the current Amazon Web Services Region.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeAccountPreferencesRequest.Builder} avoiding
     * the need to create one manually via {@link DescribeAccountPreferencesRequest#builder()}
     * </p>
     *
     * @param describeAccountPreferencesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.DescribeAccountPreferencesRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the DescribeAccountPreferences operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeAccountPreferences
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeAccountPreferences"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeAccountPreferencesResponse> describeAccountPreferences(
            Consumer<DescribeAccountPreferencesRequest.Builder> describeAccountPreferencesRequest) {
        return describeAccountPreferences(DescribeAccountPreferencesRequest.builder()
                .applyMutation(describeAccountPreferencesRequest).build());
    }

    /**
     * <p>
     * Returns the backup policy for the specified EFS file system.
     * </p>
     *
     * @param describeBackupPolicyRequest
     * @return A Java Future containing the result of the DescribeBackupPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>PolicyNotFoundException Returned if the default file system policy is in effect for the EFS file
     *         system specified.</li>
     *         <li>ValidationException Returned if the Backup service is not available in the Amazon Web Services Region
     *         in which the request was made.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeBackupPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeBackupPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeBackupPolicyResponse> describeBackupPolicy(
            DescribeBackupPolicyRequest describeBackupPolicyRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the backup policy for the specified EFS file system.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeBackupPolicyRequest.Builder} avoiding the
     * need to create one manually via {@link DescribeBackupPolicyRequest#builder()}
     * </p>
     *
     * @param describeBackupPolicyRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.DescribeBackupPolicyRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DescribeBackupPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>PolicyNotFoundException Returned if the default file system policy is in effect for the EFS file
     *         system specified.</li>
     *         <li>ValidationException Returned if the Backup service is not available in the Amazon Web Services Region
     *         in which the request was made.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeBackupPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeBackupPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeBackupPolicyResponse> describeBackupPolicy(
            Consumer<DescribeBackupPolicyRequest.Builder> describeBackupPolicyRequest) {
        return describeBackupPolicy(DescribeBackupPolicyRequest.builder().applyMutation(describeBackupPolicyRequest).build());
    }

    /**
     * <p>
     * Returns the <code>FileSystemPolicy</code> for the specified EFS file system.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeFileSystemPolicy</code> action.
     * </p>
     *
     * @param describeFileSystemPolicyRequest
     * @return A Java Future containing the result of the DescribeFileSystemPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>PolicyNotFoundException Returned if the default file system policy is in effect for the EFS file
     *         system specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeFileSystemPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeFileSystemPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeFileSystemPolicyResponse> describeFileSystemPolicy(
            DescribeFileSystemPolicyRequest describeFileSystemPolicyRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the <code>FileSystemPolicy</code> for the specified EFS file system.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeFileSystemPolicy</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeFileSystemPolicyRequest.Builder} avoiding
     * the need to create one manually via {@link DescribeFileSystemPolicyRequest#builder()}
     * </p>
     *
     * @param describeFileSystemPolicyRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.DescribeFileSystemPolicyRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the DescribeFileSystemPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>PolicyNotFoundException Returned if the default file system policy is in effect for the EFS file
     *         system specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeFileSystemPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeFileSystemPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeFileSystemPolicyResponse> describeFileSystemPolicy(
            Consumer<DescribeFileSystemPolicyRequest.Builder> describeFileSystemPolicyRequest) {
        return describeFileSystemPolicy(DescribeFileSystemPolicyRequest.builder().applyMutation(describeFileSystemPolicyRequest)
                .build());
    }

    /**
     * <p>
     * Returns the description of a specific Amazon EFS file system if either the file system <code>CreationToken</code>
     * or the <code>FileSystemId</code> is provided. Otherwise, it returns descriptions of all file systems owned by the
     * caller's Amazon Web Services account in the Amazon Web Services Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all file system descriptions, you can optionally specify the <code>MaxItems</code> parameter to
     * limit the number of descriptions in a response. This number is automatically set to 100. If more file system
     * descriptions remain, Amazon EFS returns a <code>NextMarker</code>, an opaque token, in the response. In this
     * case, you should send a subsequent request with the <code>Marker</code> request parameter set to the value of
     * <code>NextMarker</code>.
     * </p>
     * <p>
     * To retrieve a list of your file system descriptions, this operation is used in an iterative process, where
     * <code>DescribeFileSystems</code> is called first without the <code>Marker</code> and then the operation continues
     * to call it with the <code>Marker</code> parameter set to the value of the <code>NextMarker</code> from the
     * previous response until the response has no <code>NextMarker</code>.
     * </p>
     * <p>
     * The order of file systems returned in the response of one <code>DescribeFileSystems</code> call and the order of
     * file systems returned across the responses of a multi-call iteration is unspecified.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeFileSystems</code> action.
     * </p>
     *
     * @param describeFileSystemsRequest
     * @return A Java Future containing the result of the DescribeFileSystems operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeFileSystems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeFileSystems"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeFileSystemsResponse> describeFileSystems(
            DescribeFileSystemsRequest describeFileSystemsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the description of a specific Amazon EFS file system if either the file system <code>CreationToken</code>
     * or the <code>FileSystemId</code> is provided. Otherwise, it returns descriptions of all file systems owned by the
     * caller's Amazon Web Services account in the Amazon Web Services Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all file system descriptions, you can optionally specify the <code>MaxItems</code> parameter to
     * limit the number of descriptions in a response. This number is automatically set to 100. If more file system
     * descriptions remain, Amazon EFS returns a <code>NextMarker</code>, an opaque token, in the response. In this
     * case, you should send a subsequent request with the <code>Marker</code> request parameter set to the value of
     * <code>NextMarker</code>.
     * </p>
     * <p>
     * To retrieve a list of your file system descriptions, this operation is used in an iterative process, where
     * <code>DescribeFileSystems</code> is called first without the <code>Marker</code> and then the operation continues
     * to call it with the <code>Marker</code> parameter set to the value of the <code>NextMarker</code> from the
     * previous response until the response has no <code>NextMarker</code>.
     * </p>
     * <p>
     * The order of file systems returned in the response of one <code>DescribeFileSystems</code> call and the order of
     * file systems returned across the responses of a multi-call iteration is unspecified.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeFileSystems</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeFileSystemsRequest.Builder} avoiding the
     * need to create one manually via {@link DescribeFileSystemsRequest#builder()}
     * </p>
     *
     * @param describeFileSystemsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.DescribeFileSystemsRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DescribeFileSystems operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeFileSystems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeFileSystems"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeFileSystemsResponse> describeFileSystems(
            Consumer<DescribeFileSystemsRequest.Builder> describeFileSystemsRequest) {
        return describeFileSystems(DescribeFileSystemsRequest.builder().applyMutation(describeFileSystemsRequest).build());
    }

    /**
     * <p>
     * Returns the description of a specific Amazon EFS file system if either the file system <code>CreationToken</code>
     * or the <code>FileSystemId</code> is provided. Otherwise, it returns descriptions of all file systems owned by the
     * caller's Amazon Web Services account in the Amazon Web Services Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all file system descriptions, you can optionally specify the <code>MaxItems</code> parameter to
     * limit the number of descriptions in a response. This number is automatically set to 100. If more file system
     * descriptions remain, Amazon EFS returns a <code>NextMarker</code>, an opaque token, in the response. In this
     * case, you should send a subsequent request with the <code>Marker</code> request parameter set to the value of
     * <code>NextMarker</code>.
     * </p>
     * <p>
     * To retrieve a list of your file system descriptions, this operation is used in an iterative process, where
     * <code>DescribeFileSystems</code> is called first without the <code>Marker</code> and then the operation continues
     * to call it with the <code>Marker</code> parameter set to the value of the <code>NextMarker</code> from the
     * previous response until the response has no <code>NextMarker</code>.
     * </p>
     * <p>
     * The order of file systems returned in the response of one <code>DescribeFileSystems</code> call and the order of
     * file systems returned across the responses of a multi-call iteration is unspecified.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeFileSystems</code> action.
     * </p>
     *
     * @return A Java Future containing the result of the DescribeFileSystems operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeFileSystems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeFileSystems"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeFileSystemsResponse> describeFileSystems() {
        return describeFileSystems(DescribeFileSystemsRequest.builder().build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #describeFileSystems(software.amazon.awssdk.services.efs.model.DescribeFileSystemsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeFileSystemsPublisher publisher = client.describeFileSystemsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeFileSystemsPublisher publisher = client.describeFileSystemsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.efs.model.DescribeFileSystemsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.efs.model.DescribeFileSystemsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxItems won't limit the number of results you get with the paginator.
     * It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeFileSystems(software.amazon.awssdk.services.efs.model.DescribeFileSystemsRequest)} operation.</b>
     * </p>
     *
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeFileSystems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeFileSystems"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeFileSystemsPublisher describeFileSystemsPaginator() {
        return describeFileSystemsPaginator(DescribeFileSystemsRequest.builder().build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #describeFileSystems(software.amazon.awssdk.services.efs.model.DescribeFileSystemsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeFileSystemsPublisher publisher = client.describeFileSystemsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeFileSystemsPublisher publisher = client.describeFileSystemsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.efs.model.DescribeFileSystemsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.efs.model.DescribeFileSystemsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxItems won't limit the number of results you get with the paginator.
     * It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeFileSystems(software.amazon.awssdk.services.efs.model.DescribeFileSystemsRequest)} operation.</b>
     * </p>
     *
     * @param describeFileSystemsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeFileSystems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeFileSystems"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeFileSystemsPublisher describeFileSystemsPaginator(DescribeFileSystemsRequest describeFileSystemsRequest) {
        return new DescribeFileSystemsPublisher(this, describeFileSystemsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #describeFileSystems(software.amazon.awssdk.services.efs.model.DescribeFileSystemsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeFileSystemsPublisher publisher = client.describeFileSystemsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeFileSystemsPublisher publisher = client.describeFileSystemsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.efs.model.DescribeFileSystemsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.efs.model.DescribeFileSystemsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxItems won't limit the number of results you get with the paginator.
     * It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeFileSystems(software.amazon.awssdk.services.efs.model.DescribeFileSystemsRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeFileSystemsRequest.Builder} avoiding the
     * need to create one manually via {@link DescribeFileSystemsRequest#builder()}
     * </p>
     *
     * @param describeFileSystemsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.DescribeFileSystemsRequest.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeFileSystems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeFileSystems"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeFileSystemsPublisher describeFileSystemsPaginator(
            Consumer<DescribeFileSystemsRequest.Builder> describeFileSystemsRequest) {
        return describeFileSystemsPaginator(DescribeFileSystemsRequest.builder().applyMutation(describeFileSystemsRequest)
                .build());
    }

    /**
     * <p>
     * Returns the current <code>LifecycleConfiguration</code> object for the specified Amazon EFS file system.
     * Lifecycle management uses the <code>LifecycleConfiguration</code> object to identify when to move files between
     * storage classes. For a file system without a <code>LifecycleConfiguration</code> object, the call returns an
     * empty array in the response.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeLifecycleConfiguration</code>
     * operation.
     * </p>
     *
     * @param describeLifecycleConfigurationRequest
     * @return A Java Future containing the result of the DescribeLifecycleConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeLifecycleConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeLifecycleConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeLifecycleConfigurationResponse> describeLifecycleConfiguration(
            DescribeLifecycleConfigurationRequest describeLifecycleConfigurationRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the current <code>LifecycleConfiguration</code> object for the specified Amazon EFS file system.
     * Lifecycle management uses the <code>LifecycleConfiguration</code> object to identify when to move files between
     * storage classes. For a file system without a <code>LifecycleConfiguration</code> object, the call returns an
     * empty array in the response.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeLifecycleConfiguration</code>
     * operation.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeLifecycleConfigurationRequest.Builder}
     * avoiding the need to create one manually via {@link DescribeLifecycleConfigurationRequest#builder()}
     * </p>
     *
     * @param describeLifecycleConfigurationRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.DescribeLifecycleConfigurationRequest.Builder} to create
     *        a request.
     * @return A Java Future containing the result of the DescribeLifecycleConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeLifecycleConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeLifecycleConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeLifecycleConfigurationResponse> describeLifecycleConfiguration(
            Consumer<DescribeLifecycleConfigurationRequest.Builder> describeLifecycleConfigurationRequest) {
        return describeLifecycleConfiguration(DescribeLifecycleConfigurationRequest.builder()
                .applyMutation(describeLifecycleConfigurationRequest).build());
    }

    /**
     * <p>
     * Returns the security groups currently in effect for a mount target. This operation requires that the network
     * interface of the mount target has been created and the lifecycle state of the mount target is not
     * <code>deleted</code>.
     * </p>
     * <p>
     * This operation requires permissions for the following actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>elasticfilesystem:DescribeMountTargetSecurityGroups</code> action on the mount target's file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:DescribeNetworkInterfaceAttribute</code> action on the mount target's network interface.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeMountTargetSecurityGroupsRequest
     * @return A Java Future containing the result of the DescribeMountTargetSecurityGroups operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>MountTargetNotFoundException Returned if there is no mount target with the specified ID found in the
     *         caller's Amazon Web Services account.</li>
     *         <li>IncorrectMountTargetStateException Returned if the mount target is not in the correct state for the
     *         operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeMountTargetSecurityGroups
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeMountTargetSecurityGroups"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeMountTargetSecurityGroupsResponse> describeMountTargetSecurityGroups(
            DescribeMountTargetSecurityGroupsRequest describeMountTargetSecurityGroupsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the security groups currently in effect for a mount target. This operation requires that the network
     * interface of the mount target has been created and the lifecycle state of the mount target is not
     * <code>deleted</code>.
     * </p>
     * <p>
     * This operation requires permissions for the following actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>elasticfilesystem:DescribeMountTargetSecurityGroups</code> action on the mount target's file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:DescribeNetworkInterfaceAttribute</code> action on the mount target's network interface.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeMountTargetSecurityGroupsRequest.Builder}
     * avoiding the need to create one manually via {@link DescribeMountTargetSecurityGroupsRequest#builder()}
     * </p>
     *
     * @param describeMountTargetSecurityGroupsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.DescribeMountTargetSecurityGroupsRequest.Builder} to
     *        create a request.
     * @return A Java Future containing the result of the DescribeMountTargetSecurityGroups operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>MountTargetNotFoundException Returned if there is no mount target with the specified ID found in the
     *         caller's Amazon Web Services account.</li>
     *         <li>IncorrectMountTargetStateException Returned if the mount target is not in the correct state for the
     *         operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeMountTargetSecurityGroups
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeMountTargetSecurityGroups"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeMountTargetSecurityGroupsResponse> describeMountTargetSecurityGroups(
            Consumer<DescribeMountTargetSecurityGroupsRequest.Builder> describeMountTargetSecurityGroupsRequest) {
        return describeMountTargetSecurityGroups(DescribeMountTargetSecurityGroupsRequest.builder()
                .applyMutation(describeMountTargetSecurityGroupsRequest).build());
    }

    /**
     * <p>
     * Returns the descriptions of all the current mount targets, or a specific mount target, for a file system. When
     * requesting all of the current mount targets, the order of mount targets returned in the response is unspecified.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeMountTargets</code> action, on either
     * the file system ID that you specify in <code>FileSystemId</code>, or on the file system of the mount target that
     * you specify in <code>MountTargetId</code>.
     * </p>
     *
     * @param describeMountTargetsRequest
     * @return A Java Future containing the result of the DescribeMountTargets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>MountTargetNotFoundException Returned if there is no mount target with the specified ID found in the
     *         caller's Amazon Web Services account.</li>
     *         <li>AccessPointNotFoundException Returned if the specified <code>AccessPointId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeMountTargets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeMountTargets"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeMountTargetsResponse> describeMountTargets(
            DescribeMountTargetsRequest describeMountTargetsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the descriptions of all the current mount targets, or a specific mount target, for a file system. When
     * requesting all of the current mount targets, the order of mount targets returned in the response is unspecified.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeMountTargets</code> action, on either
     * the file system ID that you specify in <code>FileSystemId</code>, or on the file system of the mount target that
     * you specify in <code>MountTargetId</code>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeMountTargetsRequest.Builder} avoiding the
     * need to create one manually via {@link DescribeMountTargetsRequest#builder()}
     * </p>
     *
     * @param describeMountTargetsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.DescribeMountTargetsRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DescribeMountTargets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>MountTargetNotFoundException Returned if there is no mount target with the specified ID found in the
     *         caller's Amazon Web Services account.</li>
     *         <li>AccessPointNotFoundException Returned if the specified <code>AccessPointId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeMountTargets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeMountTargets"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeMountTargetsResponse> describeMountTargets(
            Consumer<DescribeMountTargetsRequest.Builder> describeMountTargetsRequest) {
        return describeMountTargets(DescribeMountTargetsRequest.builder().applyMutation(describeMountTargetsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #describeMountTargets(software.amazon.awssdk.services.efs.model.DescribeMountTargetsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeMountTargetsPublisher publisher = client.describeMountTargetsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeMountTargetsPublisher publisher = client.describeMountTargetsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.efs.model.DescribeMountTargetsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.efs.model.DescribeMountTargetsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxItems won't limit the number of results you get with the paginator.
     * It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeMountTargets(software.amazon.awssdk.services.efs.model.DescribeMountTargetsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeMountTargetsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>MountTargetNotFoundException Returned if there is no mount target with the specified ID found in the
     *         caller's Amazon Web Services account.</li>
     *         <li>AccessPointNotFoundException Returned if the specified <code>AccessPointId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeMountTargets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeMountTargets"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeMountTargetsPublisher describeMountTargetsPaginator(DescribeMountTargetsRequest describeMountTargetsRequest) {
        return new DescribeMountTargetsPublisher(this, describeMountTargetsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #describeMountTargets(software.amazon.awssdk.services.efs.model.DescribeMountTargetsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeMountTargetsPublisher publisher = client.describeMountTargetsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeMountTargetsPublisher publisher = client.describeMountTargetsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.efs.model.DescribeMountTargetsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.efs.model.DescribeMountTargetsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxItems won't limit the number of results you get with the paginator.
     * It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeMountTargets(software.amazon.awssdk.services.efs.model.DescribeMountTargetsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeMountTargetsRequest.Builder} avoiding the
     * need to create one manually via {@link DescribeMountTargetsRequest#builder()}
     * </p>
     *
     * @param describeMountTargetsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.DescribeMountTargetsRequest.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>MountTargetNotFoundException Returned if there is no mount target with the specified ID found in the
     *         caller's Amazon Web Services account.</li>
     *         <li>AccessPointNotFoundException Returned if the specified <code>AccessPointId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeMountTargets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeMountTargets"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeMountTargetsPublisher describeMountTargetsPaginator(
            Consumer<DescribeMountTargetsRequest.Builder> describeMountTargetsRequest) {
        return describeMountTargetsPaginator(DescribeMountTargetsRequest.builder().applyMutation(describeMountTargetsRequest)
                .build());
    }

    /**
     * <p>
     * Retrieves the replication configuration for a specific file system. If a file system is not specified, all of the
     * replication configurations for the Amazon Web Services account in an Amazon Web Services Region are retrieved.
     * </p>
     *
     * @param describeReplicationConfigurationsRequest
     * @return A Java Future containing the result of the DescribeReplicationConfigurations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>ReplicationNotFoundException Returned if the specified file system does not have a replication
     *         configuration.</li>
     *         <li>ValidationException Returned if the Backup service is not available in the Amazon Web Services Region
     *         in which the request was made.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeReplicationConfigurations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeReplicationConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeReplicationConfigurationsResponse> describeReplicationConfigurations(
            DescribeReplicationConfigurationsRequest describeReplicationConfigurationsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the replication configuration for a specific file system. If a file system is not specified, all of the
     * replication configurations for the Amazon Web Services account in an Amazon Web Services Region are retrieved.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeReplicationConfigurationsRequest.Builder}
     * avoiding the need to create one manually via {@link DescribeReplicationConfigurationsRequest#builder()}
     * </p>
     *
     * @param describeReplicationConfigurationsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.DescribeReplicationConfigurationsRequest.Builder} to
     *        create a request.
     * @return A Java Future containing the result of the DescribeReplicationConfigurations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>ReplicationNotFoundException Returned if the specified file system does not have a replication
     *         configuration.</li>
     *         <li>ValidationException Returned if the Backup service is not available in the Amazon Web Services Region
     *         in which the request was made.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeReplicationConfigurations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeReplicationConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeReplicationConfigurationsResponse> describeReplicationConfigurations(
            Consumer<DescribeReplicationConfigurationsRequest.Builder> describeReplicationConfigurationsRequest) {
        return describeReplicationConfigurations(DescribeReplicationConfigurationsRequest.builder()
                .applyMutation(describeReplicationConfigurationsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #describeReplicationConfigurations(software.amazon.awssdk.services.efs.model.DescribeReplicationConfigurationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeReplicationConfigurationsPublisher publisher = client.describeReplicationConfigurationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeReplicationConfigurationsPublisher publisher = client.describeReplicationConfigurationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.efs.model.DescribeReplicationConfigurationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.efs.model.DescribeReplicationConfigurationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeReplicationConfigurations(software.amazon.awssdk.services.efs.model.DescribeReplicationConfigurationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeReplicationConfigurationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>ReplicationNotFoundException Returned if the specified file system does not have a replication
     *         configuration.</li>
     *         <li>ValidationException Returned if the Backup service is not available in the Amazon Web Services Region
     *         in which the request was made.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeReplicationConfigurations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeReplicationConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeReplicationConfigurationsPublisher describeReplicationConfigurationsPaginator(
            DescribeReplicationConfigurationsRequest describeReplicationConfigurationsRequest) {
        return new DescribeReplicationConfigurationsPublisher(this, describeReplicationConfigurationsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #describeReplicationConfigurations(software.amazon.awssdk.services.efs.model.DescribeReplicationConfigurationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeReplicationConfigurationsPublisher publisher = client.describeReplicationConfigurationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeReplicationConfigurationsPublisher publisher = client.describeReplicationConfigurationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.efs.model.DescribeReplicationConfigurationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.efs.model.DescribeReplicationConfigurationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeReplicationConfigurations(software.amazon.awssdk.services.efs.model.DescribeReplicationConfigurationsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeReplicationConfigurationsRequest.Builder}
     * avoiding the need to create one manually via {@link DescribeReplicationConfigurationsRequest#builder()}
     * </p>
     *
     * @param describeReplicationConfigurationsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.DescribeReplicationConfigurationsRequest.Builder} to
     *        create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>ReplicationNotFoundException Returned if the specified file system does not have a replication
     *         configuration.</li>
     *         <li>ValidationException Returned if the Backup service is not available in the Amazon Web Services Region
     *         in which the request was made.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeReplicationConfigurations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeReplicationConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeReplicationConfigurationsPublisher describeReplicationConfigurationsPaginator(
            Consumer<DescribeReplicationConfigurationsRequest.Builder> describeReplicationConfigurationsRequest) {
        return describeReplicationConfigurationsPaginator(DescribeReplicationConfigurationsRequest.builder()
                .applyMutation(describeReplicationConfigurationsRequest).build());
    }

    /**
     * <note>
     * <p>
     * DEPRECATED - The <code>DescribeTags</code> action is deprecated and not maintained. To view tags associated with
     * EFS resources, use the <code>ListTagsForResource</code> API action.
     * </p>
     * </note>
     * <p>
     * Returns the tags associated with a file system. The order of tags returned in the response of one
     * <code>DescribeTags</code> call and the order of tags returned across the responses of a multiple-call iteration
     * (when using pagination) is unspecified.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeTags</code> action.
     * </p>
     *
     * @param describeTagsRequest
     * @return A Java Future containing the result of the DescribeTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeTags"
     *      target="_top">AWS API Documentation</a>
     *
     * @deprecated Use ListTagsForResource.
     */
    @Deprecated
    default CompletableFuture<DescribeTagsResponse> describeTags(DescribeTagsRequest describeTagsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <note>
     * <p>
     * DEPRECATED - The <code>DescribeTags</code> action is deprecated and not maintained. To view tags associated with
     * EFS resources, use the <code>ListTagsForResource</code> API action.
     * </p>
     * </note>
     * <p>
     * Returns the tags associated with a file system. The order of tags returned in the response of one
     * <code>DescribeTags</code> call and the order of tags returned across the responses of a multiple-call iteration
     * (when using pagination) is unspecified.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeTags</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeTagsRequest.Builder} avoiding the need to
     * create one manually via {@link DescribeTagsRequest#builder()}
     * </p>
     *
     * @param describeTagsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.DescribeTagsRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DescribeTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeTags"
     *      target="_top">AWS API Documentation</a>
     *
     * @deprecated Use ListTagsForResource.
     */
    @Deprecated
    default CompletableFuture<DescribeTagsResponse> describeTags(Consumer<DescribeTagsRequest.Builder> describeTagsRequest) {
        return describeTags(DescribeTagsRequest.builder().applyMutation(describeTagsRequest).build());
    }

    /**
     * <p>
     * This is a variant of {@link #describeTags(software.amazon.awssdk.services.efs.model.DescribeTagsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeTagsPublisher publisher = client.describeTagsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeTagsPublisher publisher = client.describeTagsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.efs.model.DescribeTagsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.efs.model.DescribeTagsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxItems won't limit the number of results you get with the paginator.
     * It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeTags(software.amazon.awssdk.services.efs.model.DescribeTagsRequest)} operation.</b>
     * </p>
     *
     * @param describeTagsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeTags"
     *      target="_top">AWS API Documentation</a>
     *
     * @deprecated Use ListTagsForResource.
     */
    @Deprecated
    default DescribeTagsPublisher describeTagsPaginator(DescribeTagsRequest describeTagsRequest) {
        return new DescribeTagsPublisher(this, describeTagsRequest);
    }

    /**
     * <p>
     * This is a variant of {@link #describeTags(software.amazon.awssdk.services.efs.model.DescribeTagsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeTagsPublisher publisher = client.describeTagsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.DescribeTagsPublisher publisher = client.describeTagsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.efs.model.DescribeTagsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.efs.model.DescribeTagsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxItems won't limit the number of results you get with the paginator.
     * It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeTags(software.amazon.awssdk.services.efs.model.DescribeTagsRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeTagsRequest.Builder} avoiding the need to
     * create one manually via {@link DescribeTagsRequest#builder()}
     * </p>
     *
     * @param describeTagsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.DescribeTagsRequest.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.DescribeTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/DescribeTags"
     *      target="_top">AWS API Documentation</a>
     *
     * @deprecated Use ListTagsForResource.
     */
    @Deprecated
    default DescribeTagsPublisher describeTagsPaginator(Consumer<DescribeTagsRequest.Builder> describeTagsRequest) {
        return describeTagsPaginator(DescribeTagsRequest.builder().applyMutation(describeTagsRequest).build());
    }

    /**
     * <p>
     * Lists all tags for a top-level EFS resource. You must provide the ID of the resource that you want to retrieve
     * the tags for.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeAccessPoints</code> action.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>AccessPointNotFoundException Returned if the specified <code>AccessPointId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists all tags for a top-level EFS resource. You must provide the ID of the resource that you want to retrieve
     * the tags for.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:DescribeAccessPoints</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.ListTagsForResourceRequest.Builder} to create a request.
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>AccessPointNotFoundException Returned if the specified <code>AccessPointId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listTagsForResource(software.amazon.awssdk.services.efs.model.ListTagsForResourceRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.ListTagsForResourcePublisher publisher = client.listTagsForResourcePaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.ListTagsForResourcePublisher publisher = client.listTagsForResourcePaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.efs.model.ListTagsForResourceResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.efs.model.ListTagsForResourceResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTagsForResource(software.amazon.awssdk.services.efs.model.ListTagsForResourceRequest)} operation.</b>
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>AccessPointNotFoundException Returned if the specified <code>AccessPointId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default ListTagsForResourcePublisher listTagsForResourcePaginator(ListTagsForResourceRequest listTagsForResourceRequest) {
        return new ListTagsForResourcePublisher(this, listTagsForResourceRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listTagsForResource(software.amazon.awssdk.services.efs.model.ListTagsForResourceRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.ListTagsForResourcePublisher publisher = client.listTagsForResourcePaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.efs.paginators.ListTagsForResourcePublisher publisher = client.listTagsForResourcePaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.efs.model.ListTagsForResourceResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.efs.model.ListTagsForResourceResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTagsForResource(software.amazon.awssdk.services.efs.model.ListTagsForResourceRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.ListTagsForResourceRequest.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>AccessPointNotFoundException Returned if the specified <code>AccessPointId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default ListTagsForResourcePublisher listTagsForResourcePaginator(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) {
        return listTagsForResourcePaginator(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest)
                .build());
    }

    /**
     * <p>
     * Modifies the set of security groups in effect for a mount target.
     * </p>
     * <p>
     * When you create a mount target, Amazon EFS also creates a new network interface. For more information, see
     * <a>CreateMountTarget</a>. This operation replaces the security groups in effect for the network interface
     * associated with a mount target, with the <code>SecurityGroups</code> provided in the request. This operation
     * requires that the network interface of the mount target has been created and the lifecycle state of the mount
     * target is not <code>deleted</code>.
     * </p>
     * <p>
     * The operation requires permissions for the following actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>elasticfilesystem:ModifyMountTargetSecurityGroups</code> action on the mount target's file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:ModifyNetworkInterfaceAttribute</code> action on the mount target's network interface.
     * </p>
     * </li>
     * </ul>
     *
     * @param modifyMountTargetSecurityGroupsRequest
     * @return A Java Future containing the result of the ModifyMountTargetSecurityGroups operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>MountTargetNotFoundException Returned if there is no mount target with the specified ID found in the
     *         caller's Amazon Web Services account.</li>
     *         <li>IncorrectMountTargetStateException Returned if the mount target is not in the correct state for the
     *         operation.</li>
     *         <li>SecurityGroupLimitExceededException Returned if the size of <code>SecurityGroups</code> specified in
     *         the request is greater than five.</li>
     *         <li>SecurityGroupNotFoundException Returned if one of the specified security groups doesn't exist in the
     *         subnet's virtual private cloud (VPC).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.ModifyMountTargetSecurityGroups
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/ModifyMountTargetSecurityGroups"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ModifyMountTargetSecurityGroupsResponse> modifyMountTargetSecurityGroups(
            ModifyMountTargetSecurityGroupsRequest modifyMountTargetSecurityGroupsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Modifies the set of security groups in effect for a mount target.
     * </p>
     * <p>
     * When you create a mount target, Amazon EFS also creates a new network interface. For more information, see
     * <a>CreateMountTarget</a>. This operation replaces the security groups in effect for the network interface
     * associated with a mount target, with the <code>SecurityGroups</code> provided in the request. This operation
     * requires that the network interface of the mount target has been created and the lifecycle state of the mount
     * target is not <code>deleted</code>.
     * </p>
     * <p>
     * The operation requires permissions for the following actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>elasticfilesystem:ModifyMountTargetSecurityGroups</code> action on the mount target's file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:ModifyNetworkInterfaceAttribute</code> action on the mount target's network interface.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ModifyMountTargetSecurityGroupsRequest.Builder}
     * avoiding the need to create one manually via {@link ModifyMountTargetSecurityGroupsRequest#builder()}
     * </p>
     *
     * @param modifyMountTargetSecurityGroupsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.ModifyMountTargetSecurityGroupsRequest.Builder} to create
     *        a request.
     * @return A Java Future containing the result of the ModifyMountTargetSecurityGroups operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>MountTargetNotFoundException Returned if there is no mount target with the specified ID found in the
     *         caller's Amazon Web Services account.</li>
     *         <li>IncorrectMountTargetStateException Returned if the mount target is not in the correct state for the
     *         operation.</li>
     *         <li>SecurityGroupLimitExceededException Returned if the size of <code>SecurityGroups</code> specified in
     *         the request is greater than five.</li>
     *         <li>SecurityGroupNotFoundException Returned if one of the specified security groups doesn't exist in the
     *         subnet's virtual private cloud (VPC).</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.ModifyMountTargetSecurityGroups
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/ModifyMountTargetSecurityGroups"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ModifyMountTargetSecurityGroupsResponse> modifyMountTargetSecurityGroups(
            Consumer<ModifyMountTargetSecurityGroupsRequest.Builder> modifyMountTargetSecurityGroupsRequest) {
        return modifyMountTargetSecurityGroups(ModifyMountTargetSecurityGroupsRequest.builder()
                .applyMutation(modifyMountTargetSecurityGroupsRequest).build());
    }

    /**
     * <p>
     * Use this operation to set the account preference in the current Amazon Web Services Region to use long 17
     * character (63 bit) or short 8 character (32 bit) resource IDs for new EFS file system and mount target resources.
     * All existing resource IDs are not affected by any changes you make. You can set the ID preference during the
     * opt-in period as EFS transitions to long resource IDs. For more information, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/manage-efs-resource-ids.html">Managing Amazon EFS resource
     * IDs</a>.
     * </p>
     * <note>
     * <p>
     * Starting in October, 2021, you will receive an error if you try to set the account preference to use the short 8
     * character format resource ID. Contact Amazon Web Services support if you receive an error and must use short IDs
     * for file system and mount target resources.
     * </p>
     * </note>
     *
     * @param putAccountPreferencesRequest
     * @return A Java Future containing the result of the PutAccountPreferences operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.PutAccountPreferences
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/PutAccountPreferences"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<PutAccountPreferencesResponse> putAccountPreferences(
            PutAccountPreferencesRequest putAccountPreferencesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Use this operation to set the account preference in the current Amazon Web Services Region to use long 17
     * character (63 bit) or short 8 character (32 bit) resource IDs for new EFS file system and mount target resources.
     * All existing resource IDs are not affected by any changes you make. You can set the ID preference during the
     * opt-in period as EFS transitions to long resource IDs. For more information, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/manage-efs-resource-ids.html">Managing Amazon EFS resource
     * IDs</a>.
     * </p>
     * <note>
     * <p>
     * Starting in October, 2021, you will receive an error if you try to set the account preference to use the short 8
     * character format resource ID. Contact Amazon Web Services support if you receive an error and must use short IDs
     * for file system and mount target resources.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutAccountPreferencesRequest.Builder} avoiding the
     * need to create one manually via {@link PutAccountPreferencesRequest#builder()}
     * </p>
     *
     * @param putAccountPreferencesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.PutAccountPreferencesRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the PutAccountPreferences operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.PutAccountPreferences
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/PutAccountPreferences"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<PutAccountPreferencesResponse> putAccountPreferences(
            Consumer<PutAccountPreferencesRequest.Builder> putAccountPreferencesRequest) {
        return putAccountPreferences(PutAccountPreferencesRequest.builder().applyMutation(putAccountPreferencesRequest).build());
    }

    /**
     * <p>
     * Updates the file system's backup policy. Use this action to start or stop automatic backups of the file system.
     * </p>
     *
     * @param putBackupPolicyRequest
     * @return A Java Future containing the result of the PutBackupPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>IncorrectFileSystemLifeCycleStateException Returned if the file system's lifecycle state is not
     *         "available".</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>ValidationException Returned if the Backup service is not available in the Amazon Web Services Region
     *         in which the request was made.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.PutBackupPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/PutBackupPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<PutBackupPolicyResponse> putBackupPolicy(PutBackupPolicyRequest putBackupPolicyRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates the file system's backup policy. Use this action to start or stop automatic backups of the file system.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutBackupPolicyRequest.Builder} avoiding the need
     * to create one manually via {@link PutBackupPolicyRequest#builder()}
     * </p>
     *
     * @param putBackupPolicyRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.PutBackupPolicyRequest.Builder} to create a request.
     * @return A Java Future containing the result of the PutBackupPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>IncorrectFileSystemLifeCycleStateException Returned if the file system's lifecycle state is not
     *         "available".</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>ValidationException Returned if the Backup service is not available in the Amazon Web Services Region
     *         in which the request was made.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.PutBackupPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/PutBackupPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<PutBackupPolicyResponse> putBackupPolicy(
            Consumer<PutBackupPolicyRequest.Builder> putBackupPolicyRequest) {
        return putBackupPolicy(PutBackupPolicyRequest.builder().applyMutation(putBackupPolicyRequest).build());
    }

    /**
     * <p>
     * Applies an Amazon EFS <code>FileSystemPolicy</code> to an Amazon EFS file system. A file system policy is an IAM
     * resource-based policy and can contain multiple policy statements. A file system always has exactly one file
     * system policy, which can be the default policy or an explicit policy set or updated using this API operation. EFS
     * file system policies have a 20,000 character limit. When an explicit policy is set, it overrides the default
     * policy. For more information about the default file system policy, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/iam-access-control-nfs-efs.html#default-filesystempolicy">Default
     * EFS File System Policy</a>.
     * </p>
     * <note>
     * <p>
     * EFS file system policies have a 20,000 character limit.
     * </p>
     * </note>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:PutFileSystemPolicy</code> action.
     * </p>
     *
     * @param putFileSystemPolicyRequest
     * @return A Java Future containing the result of the PutFileSystemPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>InvalidPolicyException Returned if the <code>FileSystemPolicy</code> is malformed or contains an
     *         error such as a parameter value that is not valid or a missing required parameter. Returned in the case
     *         of a policy lockout safety check error.</li>
     *         <li>IncorrectFileSystemLifeCycleStateException Returned if the file system's lifecycle state is not
     *         "available".</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.PutFileSystemPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/PutFileSystemPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<PutFileSystemPolicyResponse> putFileSystemPolicy(
            PutFileSystemPolicyRequest putFileSystemPolicyRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Applies an Amazon EFS <code>FileSystemPolicy</code> to an Amazon EFS file system. A file system policy is an IAM
     * resource-based policy and can contain multiple policy statements. A file system always has exactly one file
     * system policy, which can be the default policy or an explicit policy set or updated using this API operation. EFS
     * file system policies have a 20,000 character limit. When an explicit policy is set, it overrides the default
     * policy. For more information about the default file system policy, see <a
     * href="https://docs.aws.amazon.com/efs/latest/ug/iam-access-control-nfs-efs.html#default-filesystempolicy">Default
     * EFS File System Policy</a>.
     * </p>
     * <note>
     * <p>
     * EFS file system policies have a 20,000 character limit.
     * </p>
     * </note>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:PutFileSystemPolicy</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutFileSystemPolicyRequest.Builder} avoiding the
     * need to create one manually via {@link PutFileSystemPolicyRequest#builder()}
     * </p>
     *
     * @param putFileSystemPolicyRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.PutFileSystemPolicyRequest.Builder} to create a request.
     * @return A Java Future containing the result of the PutFileSystemPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>InvalidPolicyException Returned if the <code>FileSystemPolicy</code> is malformed or contains an
     *         error such as a parameter value that is not valid or a missing required parameter. Returned in the case
     *         of a policy lockout safety check error.</li>
     *         <li>IncorrectFileSystemLifeCycleStateException Returned if the file system's lifecycle state is not
     *         "available".</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.PutFileSystemPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/PutFileSystemPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<PutFileSystemPolicyResponse> putFileSystemPolicy(
            Consumer<PutFileSystemPolicyRequest.Builder> putFileSystemPolicyRequest) {
        return putFileSystemPolicy(PutFileSystemPolicyRequest.builder().applyMutation(putFileSystemPolicyRequest).build());
    }

    /**
     * <p>
     * Use this action to manage storage for your file system. A <code>LifecycleConfiguration</code> consists of one or
     * more <code>LifecyclePolicy</code> objects that define the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>TransitionToIA</code> </b> – When to move files in the file system from primary storage (Standard
     * storage class) into the Infrequent Access (IA) storage.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>TransitionToArchive</code> </b> – When to move files in the file system from their current storage
     * class (either IA or Standard storage) into the Archive storage.
     * </p>
     * <p>
     * File systems cannot transition into Archive storage before transitioning into IA storage. Therefore,
     * TransitionToArchive must either not be set or must be later than TransitionToIA.
     * </p>
     * <note>
     * <p>
     * The Archive storage class is available only for file systems that use the Elastic Throughput mode and the General
     * Purpose Performance mode.
     * </p>
     * </note></li>
     * </ul>
     * <ul>
     * <li>
     * <p>
     * <b> <code>TransitionToPrimaryStorageClass</code> </b> – Whether to move files in the file system back to primary
     * storage (Standard storage class) after they are accessed in IA or Archive storage.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/efs/latest/ug/lifecycle-management-efs.html">
     * Managing file system storage</a>.
     * </p>
     * <p>
     * Each Amazon EFS file system supports one lifecycle configuration, which applies to all files in the file system.
     * If a <code>LifecycleConfiguration</code> object already exists for the specified file system, a
     * <code>PutLifecycleConfiguration</code> call modifies the existing configuration. A
     * <code>PutLifecycleConfiguration</code> call with an empty <code>LifecyclePolicies</code> array in the request
     * body deletes any existing <code>LifecycleConfiguration</code>. In the request, specify the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The ID for the file system for which you are enabling, disabling, or modifying Lifecycle management.
     * </p>
     * </li>
     * <li>
     * <p>
     * A <code>LifecyclePolicies</code> array of <code>LifecyclePolicy</code> objects that define when to move files to
     * IA storage, to Archive storage, and back to primary storage.
     * </p>
     * <note>
     * <p>
     * Amazon EFS requires that each <code>LifecyclePolicy</code> object have only have a single transition, so the
     * <code>LifecyclePolicies</code> array needs to be structured with separate <code>LifecyclePolicy</code> objects.
     * See the example requests in the following section for more information.
     * </p>
     * </note></li>
     * </ul>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:PutLifecycleConfiguration</code> operation.
     * </p>
     * <p>
     * To apply a <code>LifecycleConfiguration</code> object to an encrypted file system, you need the same Key
     * Management Service permissions as when you created the encrypted file system.
     * </p>
     *
     * @param putLifecycleConfigurationRequest
     * @return A Java Future containing the result of the PutLifecycleConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>IncorrectFileSystemLifeCycleStateException Returned if the file system's lifecycle state is not
     *         "available".</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.PutLifecycleConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/PutLifecycleConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<PutLifecycleConfigurationResponse> putLifecycleConfiguration(
            PutLifecycleConfigurationRequest putLifecycleConfigurationRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Use this action to manage storage for your file system. A <code>LifecycleConfiguration</code> consists of one or
     * more <code>LifecyclePolicy</code> objects that define the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>TransitionToIA</code> </b> – When to move files in the file system from primary storage (Standard
     * storage class) into the Infrequent Access (IA) storage.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>TransitionToArchive</code> </b> – When to move files in the file system from their current storage
     * class (either IA or Standard storage) into the Archive storage.
     * </p>
     * <p>
     * File systems cannot transition into Archive storage before transitioning into IA storage. Therefore,
     * TransitionToArchive must either not be set or must be later than TransitionToIA.
     * </p>
     * <note>
     * <p>
     * The Archive storage class is available only for file systems that use the Elastic Throughput mode and the General
     * Purpose Performance mode.
     * </p>
     * </note></li>
     * </ul>
     * <ul>
     * <li>
     * <p>
     * <b> <code>TransitionToPrimaryStorageClass</code> </b> – Whether to move files in the file system back to primary
     * storage (Standard storage class) after they are accessed in IA or Archive storage.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/efs/latest/ug/lifecycle-management-efs.html">
     * Managing file system storage</a>.
     * </p>
     * <p>
     * Each Amazon EFS file system supports one lifecycle configuration, which applies to all files in the file system.
     * If a <code>LifecycleConfiguration</code> object already exists for the specified file system, a
     * <code>PutLifecycleConfiguration</code> call modifies the existing configuration. A
     * <code>PutLifecycleConfiguration</code> call with an empty <code>LifecyclePolicies</code> array in the request
     * body deletes any existing <code>LifecycleConfiguration</code>. In the request, specify the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The ID for the file system for which you are enabling, disabling, or modifying Lifecycle management.
     * </p>
     * </li>
     * <li>
     * <p>
     * A <code>LifecyclePolicies</code> array of <code>LifecyclePolicy</code> objects that define when to move files to
     * IA storage, to Archive storage, and back to primary storage.
     * </p>
     * <note>
     * <p>
     * Amazon EFS requires that each <code>LifecyclePolicy</code> object have only have a single transition, so the
     * <code>LifecyclePolicies</code> array needs to be structured with separate <code>LifecyclePolicy</code> objects.
     * See the example requests in the following section for more information.
     * </p>
     * </note></li>
     * </ul>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:PutLifecycleConfiguration</code> operation.
     * </p>
     * <p>
     * To apply a <code>LifecycleConfiguration</code> object to an encrypted file system, you need the same Key
     * Management Service permissions as when you created the encrypted file system.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutLifecycleConfigurationRequest.Builder} avoiding
     * the need to create one manually via {@link PutLifecycleConfigurationRequest#builder()}
     * </p>
     *
     * @param putLifecycleConfigurationRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.PutLifecycleConfigurationRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the PutLifecycleConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>IncorrectFileSystemLifeCycleStateException Returned if the file system's lifecycle state is not
     *         "available".</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.PutLifecycleConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/PutLifecycleConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<PutLifecycleConfigurationResponse> putLifecycleConfiguration(
            Consumer<PutLifecycleConfigurationRequest.Builder> putLifecycleConfigurationRequest) {
        return putLifecycleConfiguration(PutLifecycleConfigurationRequest.builder()
                .applyMutation(putLifecycleConfigurationRequest).build());
    }

    /**
     * <p>
     * Creates a tag for an EFS resource. You can create tags for EFS file systems and access points using this API
     * operation.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:TagResource</code> action.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>AccessPointNotFoundException Returned if the specified <code>AccessPointId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a tag for an EFS resource. You can create tags for EFS file systems and access points using this API
     * operation.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:TagResource</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.TagResourceRequest.Builder} to create a request.
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>AccessPointNotFoundException Returned if the specified <code>AccessPointId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest) {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Removes tags from an EFS resource. You can remove tags from EFS file systems and access points using this API
     * operation.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:UntagResource</code> action.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>AccessPointNotFoundException Returned if the specified <code>AccessPointId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes tags from an EFS resource. You can remove tags from EFS file systems and access points using this API
     * operation.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:UntagResource</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.UntagResourceRequest.Builder} to create a request.
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>AccessPointNotFoundException Returned if the specified <code>AccessPointId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest) {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Updates the throughput mode or the amount of provisioned throughput of an existing file system.
     * </p>
     *
     * @param updateFileSystemRequest
     * @return A Java Future containing the result of the UpdateFileSystem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>IncorrectFileSystemLifeCycleStateException Returned if the file system's lifecycle state is not
     *         "available".</li>
     *         <li>InsufficientThroughputCapacityException Returned if there's not enough capacity to provision
     *         additional throughput. This value might be returned when you try to create a file system in provisioned
     *         throughput mode, when you attempt to increase the provisioned throughput of an existing file system, or
     *         when you attempt to change an existing file system from Bursting Throughput to Provisioned Throughput
     *         mode. Try again later.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>ThroughputLimitExceededException Returned if the throughput mode or amount of provisioned throughput
     *         can't be changed because the throughput limit of 1024 MiB/s has been reached.</li>
     *         <li>TooManyRequestsException Returned if you don’t wait at least 24 hours before either changing the
     *         throughput mode, or decreasing the Provisioned Throughput value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.UpdateFileSystem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/UpdateFileSystem"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<UpdateFileSystemResponse> updateFileSystem(UpdateFileSystemRequest updateFileSystemRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates the throughput mode or the amount of provisioned throughput of an existing file system.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateFileSystemRequest.Builder} avoiding the need
     * to create one manually via {@link UpdateFileSystemRequest#builder()}
     * </p>
     *
     * @param updateFileSystemRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.UpdateFileSystemRequest.Builder} to create a request.
     * @return A Java Future containing the result of the UpdateFileSystem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>IncorrectFileSystemLifeCycleStateException Returned if the file system's lifecycle state is not
     *         "available".</li>
     *         <li>InsufficientThroughputCapacityException Returned if there's not enough capacity to provision
     *         additional throughput. This value might be returned when you try to create a file system in provisioned
     *         throughput mode, when you attempt to increase the provisioned throughput of an existing file system, or
     *         when you attempt to change an existing file system from Bursting Throughput to Provisioned Throughput
     *         mode. Try again later.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>ThroughputLimitExceededException Returned if the throughput mode or amount of provisioned throughput
     *         can't be changed because the throughput limit of 1024 MiB/s has been reached.</li>
     *         <li>TooManyRequestsException Returned if you don’t wait at least 24 hours before either changing the
     *         throughput mode, or decreasing the Provisioned Throughput value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.UpdateFileSystem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/UpdateFileSystem"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<UpdateFileSystemResponse> updateFileSystem(
            Consumer<UpdateFileSystemRequest.Builder> updateFileSystemRequest) {
        return updateFileSystem(UpdateFileSystemRequest.builder().applyMutation(updateFileSystemRequest).build());
    }

    /**
     * <p>
     * Updates protection on the file system.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:UpdateFileSystemProtection</code> action.
     * </p>
     *
     * @param updateFileSystemProtectionRequest
     * @return A Java Future containing the result of the UpdateFileSystemProtection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>IncorrectFileSystemLifeCycleStateException Returned if the file system's lifecycle state is not
     *         "available".</li>
     *         <li>InsufficientThroughputCapacityException Returned if there's not enough capacity to provision
     *         additional throughput. This value might be returned when you try to create a file system in provisioned
     *         throughput mode, when you attempt to increase the provisioned throughput of an existing file system, or
     *         when you attempt to change an existing file system from Bursting Throughput to Provisioned Throughput
     *         mode. Try again later.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>ThroughputLimitExceededException Returned if the throughput mode or amount of provisioned throughput
     *         can't be changed because the throughput limit of 1024 MiB/s has been reached.</li>
     *         <li>ReplicationAlreadyExistsException Returned if the file system is already included in a replication
     *         configuration.&gt;</li>
     *         <li>TooManyRequestsException Returned if you don’t wait at least 24 hours before either changing the
     *         throughput mode, or decreasing the Provisioned Throughput value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.UpdateFileSystemProtection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/UpdateFileSystemProtection"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<UpdateFileSystemProtectionResponse> updateFileSystemProtection(
            UpdateFileSystemProtectionRequest updateFileSystemProtectionRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates protection on the file system.
     * </p>
     * <p>
     * This operation requires permissions for the <code>elasticfilesystem:UpdateFileSystemProtection</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateFileSystemProtectionRequest.Builder} avoiding
     * the need to create one manually via {@link UpdateFileSystemProtectionRequest#builder()}
     * </p>
     *
     * @param updateFileSystemProtectionRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.efs.model.UpdateFileSystemProtectionRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the UpdateFileSystemProtection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException Returned if the request is malformed or contains an error such as an invalid
     *         parameter value or a missing required parameter.</li>
     *         <li>FileSystemNotFoundException Returned if the specified <code>FileSystemId</code> value doesn't exist
     *         in the requester's Amazon Web Services account.</li>
     *         <li>IncorrectFileSystemLifeCycleStateException Returned if the file system's lifecycle state is not
     *         "available".</li>
     *         <li>InsufficientThroughputCapacityException Returned if there's not enough capacity to provision
     *         additional throughput. This value might be returned when you try to create a file system in provisioned
     *         throughput mode, when you attempt to increase the provisioned throughput of an existing file system, or
     *         when you attempt to change an existing file system from Bursting Throughput to Provisioned Throughput
     *         mode. Try again later.</li>
     *         <li>InternalServerErrorException Returned if an error occurred on the server side.</li>
     *         <li>ThroughputLimitExceededException Returned if the throughput mode or amount of provisioned throughput
     *         can't be changed because the throughput limit of 1024 MiB/s has been reached.</li>
     *         <li>ReplicationAlreadyExistsException Returned if the file system is already included in a replication
     *         configuration.&gt;</li>
     *         <li>TooManyRequestsException Returned if you don’t wait at least 24 hours before either changing the
     *         throughput mode, or decreasing the Provisioned Throughput value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>EfsException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample EfsAsyncClient.UpdateFileSystemProtection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/elasticfilesystem-2015-02-01/UpdateFileSystemProtection"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<UpdateFileSystemProtectionResponse> updateFileSystemProtection(
            Consumer<UpdateFileSystemProtectionRequest.Builder> updateFileSystemProtectionRequest) {
        return updateFileSystemProtection(UpdateFileSystemProtectionRequest.builder()
                .applyMutation(updateFileSystemProtectionRequest).build());
    }

    @Override
    default EfsServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }

    /**
     * Create a {@link EfsAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static EfsAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link EfsAsyncClient}.
     */
    static EfsAsyncClientBuilder builder() {
        return new DefaultEfsAsyncClientBuilder();
    }
}
